#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
/*
 * BlobWorker.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/ClientBooleanParams.h"
#include "fdbclient/BlobCipher.h"
#include "fdbclient/BlobGranuleFiles.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/GetEncryptCipherKeys.h"
#include "fdbclient/KeyRangeMap.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/BackupContainerFileSystem.h"
#include "fdbclient/BlobConnectionProvider.h"
#include "fdbclient/BlobGranuleCommon.h"
#include "fdbclient/BlobGranuleReader.actor.h"
#include "fdbclient/BlobMetadataUtils.h"
#include "fdbclient/BlobWorkerCommon.h"
#include "fdbclient/BlobWorkerInterface.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/Notified.h"

#include "fdbserver/BlobWorker.h"
#include "fdbserver/BlobGranuleServerCommon.actor.h"
#include "fdbclient/GetEncryptCipherKeys.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/MutationTracking.h"
#include "fdbserver/ServerDBInfo.actor.h"
#include "fdbserver/ServerDBInfo.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/IKeyValueStore.h"

#include "flow/Arena.h"
#include "flow/CompressionUtils.h"
#include "flow/EncryptUtils.h"
#include "flow/Error.h"
#include "flow/flow.h"
#include "flow/IRandom.h"
#include "flow/network.h"
#include "flow/Trace.h"
#include "flow/xxhash.h"

#include "fmt/format.h"

#include <limits>
#include <tuple>
#include <utility>
#include <vector>

#include "flow/actorcompiler.h" // has to be last include

#define BW_DEBUG false
#define BW_HISTORY_DEBUG false
#define BW_REQUEST_DEBUG false

void GranuleMetadata::resume() {
	if (resumeSnapshot.canBeSet()) {
		resumeSnapshot.send(Void());
	}
}

void GranuleMetadata::resetReadStats() {
	rdcCandidate = false;
	readStats.reset();
	runRDC.reset();
}

double GranuleMetadata::weightRDC() {
	// ratio of read amp to write amp that would be incurred by re-snapshotting now
	int64_t lastSnapshotSize = (files.snapshotFiles.empty()) ? 0 : files.snapshotFiles.back().length;
	int64_t minSnapshotSize = SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES / 2;
	lastSnapshotSize = std::max(minSnapshotSize, lastSnapshotSize);

	int64_t writeAmp = lastSnapshotSize + bufferedDeltaBytes + bytesInNewDeltaFiles;
	// read amp is deltaBytesRead. Read amp must be READ_FACTOR times larger than write amp
	return (1.0 * readStats.deltaBytesRead) / (writeAmp * SERVER_KNOBS->BG_RDC_READ_FACTOR);
}

bool GranuleMetadata::isEligibleRDC() const {
	// granule should be reasonably read-hot to be eligible
	int64_t bytesWritten = bufferedDeltaBytes + bytesInNewDeltaFiles;
	return bytesWritten > 0 && bytesWritten * SERVER_KNOBS->BG_RDC_READ_FACTOR < readStats.deltaBytesRead;
}

bool GranuleMetadata::updateReadStats(Version readVersion, const BlobGranuleChunkRef& chunk) {
	// Only update stats for re-compacting for at-latest reads that have to do snapshot + delta merge
	if (!SERVER_KNOBS->BG_ENABLE_READ_DRIVEN_COMPACTION || !chunk.snapshotFile.present() ||
	    pendingSnapshotVersion != durableSnapshotVersion.get() || readVersion <= pendingSnapshotVersion) {
		return false;
	}

	if (chunk.newDeltas.empty() && chunk.deltaFiles.empty()) {
		return false;
	}

	// any memory deltas must be newer than snapshot
	readStats.deltaBytesRead += chunk.newDeltas.expectedSize();
	for (auto& it : chunk.deltaFiles) {
		// some races where you get previous snapshot + deltas instead of new snapshot for read, don't count those
		// as we already re-snapshotted after them
		if (it.fileVersion > pendingSnapshotVersion) {
			// TODO: should this be logical size instead?
			readStats.deltaBytesRead += it.length;
		}
	}

	if (rdcCandidate) {
		return false;
	}

	if (isEligibleRDC() && weightRDC() > 1.0) {
		rdcCandidate = true;
		CODE_PROBE(true, "Granule read triggering read-driven compaction");
		if (BW_DEBUG) {
			fmt::print("Triggering read-driven compaction of [{0} - {1})\n",
			           keyRange.begin.printable(),
			           keyRange.end.printable());
		}
		return true;
	}
	return false;
}

void GranuleRangeMetadata::cancel() {
	if (activeMetadata->cancelled.canBeSet()) {
		activeMetadata->cancelled.send(Void());
	}
	activeMetadata.clear();
	assignFuture.cancel();
	historyLoaderFuture.cancel();
	fileUpdaterFuture.cancel();
}
/*
 * The Blob Worker is a stateless role assigned a set of granules by the Blob Manager.
 * It is responsible for managing the change feeds for those granules, and for consuming the mutations from
 * those change feeds and writing them out as files to blob storage.
 */
bool BlobWorkerData::managerEpochOk(int64_t epoch) {
	if (epoch < currentManagerEpoch) {
		if (BW_DEBUG) {
			fmt::print(
			    "BW {0} got request from old epoch {1}, notifying them they are out of date\n", id.toString(), epoch);
		}
		return false;
	} else {
		if (epoch > currentManagerEpoch) {
			currentManagerEpoch = epoch;
			if (BW_DEBUG) {
				fmt::print("BW {0} found new manager epoch {1}\n", id.toString(), currentManagerEpoch);
			}
			TraceEvent(SevDebug, "BlobWorkerFoundNewManager", id).detail("Epoch", epoch);
		}

		return true;
	}
}

bool BlobWorkerData::isFull() {
	if (!SERVER_KNOBS->BLOB_WORKER_DO_REJECT_WHEN_FULL) {
		return false;
	}
	if (g_network->isSimulated()) {
		if (g_simulator->speedUpSimulation) {
			return false;
		}
		return buggifyFull;
	}

	// TODO knob?
	if (now() >= 1.0 + lastResidentMemoryCheckTime) {
		// fdb as of 7.1 limits on resident memory instead of virtual memory
		stats.lastResidentMemory = getResidentMemoryUsage();
		lastResidentMemoryCheckTime = now();
	}

	// if we are already over threshold, no need to estimate extra memory
	if (stats.lastResidentMemory >= memoryFullThreshold) {
		return true;
	}

	// FIXME: since this isn't tested in simulation, could unit test this
	// Try to model how much memory we *could* use given the already existing assignments and workload on this blob
	// worker, before agreeing to take on a new assignment, given that several large sources of memory can grow and
	// change post-assignment

	// FIXME: change these to be byte counts
	// FIXME: buggify an extra multiplication factor for short periods of time to hopefully trigger this logic more
	// often? estimate slack in bytes buffered as max(0, assignments * (delta file size / 2) - bytesBuffered)
	// FIXME: this doesn't take increased delta file size for heavy write amp cases into account
	int64_t expectedExtraBytesBuffered = std::max<int64_t>(
	    0, stats.numRangesAssigned * (SERVER_KNOBS->BG_DELTA_FILE_TARGET_BYTES / 2) - stats.mutationBytesBuffered);
	// estimate slack in potential pending resnapshot
	int64_t totalExtra = expectedExtraBytesBuffered + deltaWritesBudget->available() + resnapshotBudget->available();
	// assumes initial snapshot parallelism is small enough and uncommon enough to not add it to this computation
	stats.estimatedMaxResidentMemory = stats.lastResidentMemory + totalExtra;

	return stats.estimatedMaxResidentMemory >= memoryFullThreshold;
}

void BlobWorkerData::triggerReadDrivenCompaction() {
	Promise<Void> doRDC = doReadDrivenCompaction;
	if (doRDC.canBeSet()) {
		doRDC.send(Void());
	}
}

void BlobWorkerData::addGRVHistory(Version readVersion) {
	if (grvVersion.get() < readVersion) {
		// We use GRVs from grv checker loop, plus other common BW transactions. To prevent the deque size from
		// exploding or the effective version window from getting too small, only put GRVs in the deque if they are
		// at least some small distance apart
		if (prevGRVVersions.empty() ||
		    prevGRVVersions.back() + SERVER_KNOBS->BLOB_WORKER_GRV_HISTORY_MIN_VERSION_GRANULARITY <= readVersion) {
			prevGRVVersions.push_back(readVersion);
			while (prevGRVVersions.size() > SERVER_KNOBS->BLOB_WORKER_GRV_HISTORY_MAX_SIZE) {
				prevGRVVersions.pop_front();
			}
		}

		// set notified version last, so that all triggered waiters have prevGRVVersions populated too
		grvVersion.set(readVersion);
	}
}
bool BlobWorkerData::maybeInjectTargetedRestart() {
	// inject a BW restart at most once per test
	if (g_network->isSimulated() && !g_simulator->speedUpSimulation &&
	    now() > g_simulator->injectTargetedBWRestartTime) {
		CODE_PROBE(true, "Injecting BW targeted restart");
		TraceEvent("SimBWInjectTargetedRestart", id);
		g_simulator->injectTargetedBWRestartTime = std::numeric_limits<double>::max();
		simInjectFailure.send(Void());
		return true;
	}
	return false;
}

namespace {

Optional<CompressionFilter> getBlobFileCompressFilter() {
	Optional<CompressionFilter> compFilter;
	if (SERVER_KNOBS->ENABLE_BLOB_GRANULE_COMPRESSION) {
		compFilter = CompressionUtils::fromFilterString(SERVER_KNOBS->BLOB_GRANULE_COMPRESSION_FILTER);
		if (BUGGIFY_WITH_PROB(0.1)) {
			compFilter = CompressionUtils::getRandomFilter();
		}
	}
	return compFilter;
}

// returns true if we can acquire it
void acquireGranuleLock(int64_t epoch, int64_t seqno, int64_t prevOwnerEpoch, int64_t prevOwnerSeqno) {
	// returns true if our lock (E, S) >= (Eprev, Sprev)
	if (epoch < prevOwnerEpoch || (epoch == prevOwnerEpoch && seqno < prevOwnerSeqno)) {
		if (BW_DEBUG) {
			fmt::print("Lock acquire check failed. Proposed ({0}, {1}) < previous ({2}, {3})\n",
			           epoch,
			           seqno,
			           prevOwnerEpoch,
			           prevOwnerSeqno);
		}
		throw granule_assignment_conflict();
	}
}

void checkGranuleLock(int64_t epoch, int64_t seqno, int64_t ownerEpoch, int64_t ownerSeqno) {
	// sanity check - lock value should never go backwards because of acquireGranuleLock
	ASSERT(epoch <= ownerEpoch);
	ASSERT(epoch < ownerEpoch || (epoch == ownerEpoch && seqno <= ownerSeqno));

	// returns true if we still own the lock, false if someone else does
	if (epoch != ownerEpoch || seqno != ownerSeqno) {
		if (BW_DEBUG) {
			fmt::print("Lock assignment check failed. Expected ({0}, {1}), got ({2}, {3})\n",
			           epoch,
			           seqno,
			           ownerEpoch,
			           ownerSeqno);
		}
		throw granule_assignment_conflict();
	}
}

} // namespace

// Below actors assist in fetching/lookup desired encryption keys. Following steps are done for an encryption key
// lookup:
// 1. Lookup process local in-memory cache `BlobCipherKeyCache` to check if desired EK is 'present' and 'valid'. Given
//    FDB supports 'revocable' & 'non-revocable' EKs; a cached EK can also be 'invalid'.
// 2. Local cache miss will follow with a RPC call to EncryptKeyProxy process (EKP), EKP maintain an in-memory cache of
//    KMS BaseCipher details with KMS defined TTL if applicable. The lookup call can either to serviced by EKP or would
//    lead to desired KMS endpoint invocation.
//
// In most of the cases, the EK lookup should be satisfied by process local in-memory cache and/or EKP in-memory cache,
// unless cluster and/or a process crash/restart.

															#line 315 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via getLatestGranuleCipherKeys()
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class GetLatestGranuleCipherKeysActor>
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class GetLatestGranuleCipherKeysActorState {
															#line 322 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GetLatestGranuleCipherKeysActorState(Reference<BlobWorkerData> const& bwData,KeyRange const& keyRange,Arena* const& arena) 
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   keyRange(keyRange),
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   arena(arena),
															#line 316 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   cipherKeysCtx(),
															#line 317 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   domainId(FDB_DEFAULT_ENCRYPT_DOMAIN_ID),
															#line 318 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   tenantData(),
															#line 319 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   retryCount(0)
															#line 341 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this));

	}
	~GetLatestGranuleCipherKeysActorState() 
	{
		fdb_probe_actor_destroy("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 320 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (bwData->encryptMode.mode == EncryptionAtRestMode::DOMAIN_AWARE)
															#line 356 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 321 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				;
															#line 360 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetLatestGranuleCipherKeysActorState();
		static_cast<GetLatestGranuleCipherKeysActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 339 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		std::unordered_set<EncryptCipherDomainId> domainIds;
															#line 340 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		domainIds.emplace(domainId);
															#line 341 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>> __when_expr_2 = GetEncryptCipherKeys<ServerDBInfo>::getLatestEncryptCipherKeys( bwData->dbInfo, domainIds, BlobCipherMetrics::BLOB_GRANULE);
															#line 341 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 394 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state = 3;
															#line 341 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetLatestGranuleCipherKeysActor, 2, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*>(static_cast<GetLatestGranuleCipherKeysActor*>(this)));
															#line 399 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 336 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		domainId = tenantData->entry.id;
															#line 408 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 322 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_0 = store(tenantData, bwData->tenantData.getDataForGranule(keyRange));
															#line 322 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 426 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state = 1;
															#line 322 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetLatestGranuleCipherKeysActor, 0, Void >*>(static_cast<GetLatestGranuleCipherKeysActor*>(this)));
															#line 431 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 323 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (tenantData.isValid())
															#line 453 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		else
		{
															#line 326 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			CODE_PROBE(true, "cipher keys for unknown tenant");
															#line 330 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			retryCount++;
															#line 331 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent(retryCount <= 10 ? SevDebug : SevWarn, "BlobWorkerUnknownTenantForCipherKeys", bwData->id) .detail("KeyRange", keyRange);
															#line 333 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0.1);
															#line 333 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 469 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state = 2;
															#line 333 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetLatestGranuleCipherKeysActor, 1, Void >*>(static_cast<GetLatestGranuleCipherKeysActor*>(this)));
															#line 474 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 323 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (tenantData.isValid())
															#line 484 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		else
		{
															#line 326 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			CODE_PROBE(true, "cipher keys for unknown tenant");
															#line 330 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			retryCount++;
															#line 331 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent(retryCount <= 10 ? SevDebug : SevWarn, "BlobWorkerUnknownTenantForCipherKeys", bwData->id) .detail("KeyRange", keyRange);
															#line 333 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0.1);
															#line 333 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 500 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state = 2;
															#line 333 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetLatestGranuleCipherKeysActor, 1, Void >*>(static_cast<GetLatestGranuleCipherKeysActor*>(this)));
															#line 505 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state > 0) static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state = 0;
		static_cast<GetLatestGranuleCipherKeysActor*>(this)->ActorCallback< GetLatestGranuleCipherKeysActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetLatestGranuleCipherKeysActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetLatestGranuleCipherKeysActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetLatestGranuleCipherKeysActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state > 0) static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state = 0;
		static_cast<GetLatestGranuleCipherKeysActor*>(this)->ActorCallback< GetLatestGranuleCipherKeysActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetLatestGranuleCipherKeysActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetLatestGranuleCipherKeysActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetLatestGranuleCipherKeysActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& domainKeyMap,int loopDepth) 
	{
															#line 345 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		auto domainKeyItr = domainKeyMap.find(domainId);
															#line 346 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(domainKeyItr != domainKeyMap.end());
															#line 347 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.textCipherKey = BlobGranuleCipherKey::fromBlobCipherKey(domainKeyItr->second, *arena);
															#line 349 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<TextAndHeaderCipherKeys> __when_expr_3 = GetEncryptCipherKeys<ServerDBInfo>::getLatestSystemEncryptCipherKeys(bwData->dbInfo, BlobCipherMetrics::BLOB_GRANULE);
															#line 349 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 691 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state = 4;
															#line 349 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetLatestGranuleCipherKeysActor, 3, TextAndHeaderCipherKeys >*>(static_cast<GetLatestGranuleCipherKeysActor*>(this)));
															#line 696 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && domainKeyMap,int loopDepth) 
	{
															#line 345 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		auto domainKeyItr = domainKeyMap.find(domainId);
															#line 346 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(domainKeyItr != domainKeyMap.end());
															#line 347 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.textCipherKey = BlobGranuleCipherKey::fromBlobCipherKey(domainKeyItr->second, *arena);
															#line 349 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<TextAndHeaderCipherKeys> __when_expr_3 = GetEncryptCipherKeys<ServerDBInfo>::getLatestSystemEncryptCipherKeys(bwData->dbInfo, BlobCipherMetrics::BLOB_GRANULE);
															#line 349 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 713 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state = 4;
															#line 349 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetLatestGranuleCipherKeysActor, 3, TextAndHeaderCipherKeys >*>(static_cast<GetLatestGranuleCipherKeysActor*>(this)));
															#line 718 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& domainKeyMap,int loopDepth) 
	{
		loopDepth = a_body1cont4(domainKeyMap, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && domainKeyMap,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(domainKeyMap), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state > 0) static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state = 0;
		static_cast<GetLatestGranuleCipherKeysActor*>(this)->ActorCallback< GetLatestGranuleCipherKeysActor, 2, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >::remove();

	}
	void a_callback_fire(ActorCallback< GetLatestGranuleCipherKeysActor, 2, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& value) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetLatestGranuleCipherKeysActor, 2, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && value) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetLatestGranuleCipherKeysActor, 2, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,Error err) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont5(TextAndHeaderCipherKeys const& systemCipherKeys,int loopDepth) 
	{
															#line 352 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(systemCipherKeys.cipherHeaderKey.isValid());
															#line 353 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.headerCipherKey = BlobGranuleCipherKey::fromBlobCipherKey(systemCipherKeys.cipherHeaderKey, *arena);
															#line 355 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.ivRef = makeString(AES_256_IV_LENGTH, *arena);
															#line 356 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		deterministicRandom()->randomBytes(mutateString(cipherKeysCtx.ivRef), AES_256_IV_LENGTH);
															#line 358 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BG_ENCRYPT_COMPRESS_DEBUG)
															#line 810 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 359 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent(SevDebug, "GetLatestGranuleCipherKey") .detail("TextDomainId", cipherKeysCtx.textCipherKey.encryptDomainId) .detail("TextBaseCipherId", cipherKeysCtx.textCipherKey.baseCipherId) .detail("TextSalt", cipherKeysCtx.textCipherKey.salt) .detail("HeaderDomainId", cipherKeysCtx.textCipherKey.encryptDomainId) .detail("HeaderBaseCipherId", cipherKeysCtx.textCipherKey.baseCipherId) .detail("HeaderSalt", cipherKeysCtx.textCipherKey.salt) .detail("IVChksum", XXH3_64bits(cipherKeysCtx.ivRef.begin(), cipherKeysCtx.ivRef.size()));
															#line 814 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 369 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<GetLatestGranuleCipherKeysActor*>(this)->SAV<BlobGranuleCipherKeysCtx>::futures) { (void)(cipherKeysCtx); this->~GetLatestGranuleCipherKeysActorState(); static_cast<GetLatestGranuleCipherKeysActor*>(this)->destroy(); return 0; }
															#line 818 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<GetLatestGranuleCipherKeysActor*>(this)->SAV< BlobGranuleCipherKeysCtx >::value()) BlobGranuleCipherKeysCtx(std::move(cipherKeysCtx)); // state_var_RVO
		this->~GetLatestGranuleCipherKeysActorState();
		static_cast<GetLatestGranuleCipherKeysActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(TextAndHeaderCipherKeys && systemCipherKeys,int loopDepth) 
	{
															#line 352 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(systemCipherKeys.cipherHeaderKey.isValid());
															#line 353 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.headerCipherKey = BlobGranuleCipherKey::fromBlobCipherKey(systemCipherKeys.cipherHeaderKey, *arena);
															#line 355 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.ivRef = makeString(AES_256_IV_LENGTH, *arena);
															#line 356 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		deterministicRandom()->randomBytes(mutateString(cipherKeysCtx.ivRef), AES_256_IV_LENGTH);
															#line 358 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BG_ENCRYPT_COMPRESS_DEBUG)
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 359 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent(SevDebug, "GetLatestGranuleCipherKey") .detail("TextDomainId", cipherKeysCtx.textCipherKey.encryptDomainId) .detail("TextBaseCipherId", cipherKeysCtx.textCipherKey.baseCipherId) .detail("TextSalt", cipherKeysCtx.textCipherKey.salt) .detail("HeaderDomainId", cipherKeysCtx.textCipherKey.encryptDomainId) .detail("HeaderBaseCipherId", cipherKeysCtx.textCipherKey.baseCipherId) .detail("HeaderSalt", cipherKeysCtx.textCipherKey.salt) .detail("IVChksum", XXH3_64bits(cipherKeysCtx.ivRef.begin(), cipherKeysCtx.ivRef.size()));
															#line 842 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 369 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<GetLatestGranuleCipherKeysActor*>(this)->SAV<BlobGranuleCipherKeysCtx>::futures) { (void)(cipherKeysCtx); this->~GetLatestGranuleCipherKeysActorState(); static_cast<GetLatestGranuleCipherKeysActor*>(this)->destroy(); return 0; }
															#line 846 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<GetLatestGranuleCipherKeysActor*>(this)->SAV< BlobGranuleCipherKeysCtx >::value()) BlobGranuleCipherKeysCtx(std::move(cipherKeysCtx)); // state_var_RVO
		this->~GetLatestGranuleCipherKeysActorState();
		static_cast<GetLatestGranuleCipherKeysActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4when1(TextAndHeaderCipherKeys const& systemCipherKeys,int loopDepth) 
	{
		loopDepth = a_body1cont5(systemCipherKeys, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(TextAndHeaderCipherKeys && systemCipherKeys,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(systemCipherKeys), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state > 0) static_cast<GetLatestGranuleCipherKeysActor*>(this)->actor_wait_state = 0;
		static_cast<GetLatestGranuleCipherKeysActor*>(this)->ActorCallback< GetLatestGranuleCipherKeysActor, 3, TextAndHeaderCipherKeys >::remove();

	}
	void a_callback_fire(ActorCallback< GetLatestGranuleCipherKeysActor, 3, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys const& value) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetLatestGranuleCipherKeysActor, 3, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys && value) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetLatestGranuleCipherKeysActor, 3, TextAndHeaderCipherKeys >*,Error err) 
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	KeyRange keyRange;
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Arena* arena;
															#line 316 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	BlobGranuleCipherKeysCtx cipherKeysCtx;
															#line 317 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	EncryptCipherDomainId domainId;
															#line 318 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<GranuleTenantData> tenantData;
															#line 319 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int retryCount;
															#line 943 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via getLatestGranuleCipherKeys()
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class GetLatestGranuleCipherKeysActor final : public Actor<BlobGranuleCipherKeysCtx>, public ActorCallback< GetLatestGranuleCipherKeysActor, 0, Void >, public ActorCallback< GetLatestGranuleCipherKeysActor, 1, Void >, public ActorCallback< GetLatestGranuleCipherKeysActor, 2, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >, public ActorCallback< GetLatestGranuleCipherKeysActor, 3, TextAndHeaderCipherKeys >, public FastAllocated<GetLatestGranuleCipherKeysActor>, public GetLatestGranuleCipherKeysActorState<GetLatestGranuleCipherKeysActor> {
															#line 948 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<GetLatestGranuleCipherKeysActor>::operator new;
	using FastAllocated<GetLatestGranuleCipherKeysActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5726619812797877760UL, 13878682436776748544UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobGranuleCipherKeysCtx>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetLatestGranuleCipherKeysActor, 0, Void >;
friend struct ActorCallback< GetLatestGranuleCipherKeysActor, 1, Void >;
friend struct ActorCallback< GetLatestGranuleCipherKeysActor, 2, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >;
friend struct ActorCallback< GetLatestGranuleCipherKeysActor, 3, TextAndHeaderCipherKeys >;
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GetLatestGranuleCipherKeysActor(Reference<BlobWorkerData> const& bwData,KeyRange const& keyRange,Arena* const& arena) 
															#line 968 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<BlobGranuleCipherKeysCtx>(),
		   GetLatestGranuleCipherKeysActorState<GetLatestGranuleCipherKeysActor>(bwData, keyRange, arena),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(454837814922370560UL, 2122910825243938816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestGranuleCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getLatestGranuleCipherKeys");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getLatestGranuleCipherKeys", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetLatestGranuleCipherKeysActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetLatestGranuleCipherKeysActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetLatestGranuleCipherKeysActor, 2, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetLatestGranuleCipherKeysActor, 3, TextAndHeaderCipherKeys >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<BlobGranuleCipherKeysCtx> getLatestGranuleCipherKeys( Reference<BlobWorkerData> const& bwData, KeyRange const& keyRange, Arena* const& arena ) {
															#line 313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<BlobGranuleCipherKeysCtx>(new GetLatestGranuleCipherKeysActor(bwData, keyRange, arena));
															#line 1004 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 371 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

															#line 1009 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via lookupCipherKey()
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class LookupCipherKeyActor>
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class LookupCipherKeyActorState {
															#line 1016 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	LookupCipherKeyActorState(Reference<AsyncVar<ServerDBInfo> const> const& dbInfo,BlobCipherDetails const& cipherDetails,Arena* const& arena) 
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : dbInfo(dbInfo),
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   cipherDetails(cipherDetails),
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   arena(arena)
															#line 1027 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("lookupCipherKey", reinterpret_cast<unsigned long>(this));

	}
	~LookupCipherKeyActorState() 
	{
		fdb_probe_actor_destroy("lookupCipherKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 375 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			std::unordered_set<BlobCipherDetails> cipherDetailsSet;
															#line 376 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			cipherDetailsSet.emplace(cipherDetails);
															#line 377 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>>> __when_expr_0 = GetEncryptCipherKeys<ServerDBInfo>::getEncryptCipherKeys( dbInfo, cipherDetailsSet, BlobCipherMetrics::BLOB_GRANULE);
															#line 377 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<LookupCipherKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1048 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LookupCipherKeyActor*>(this)->actor_wait_state = 1;
															#line 377 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LookupCipherKeyActor, 0, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*>(static_cast<LookupCipherKeyActor*>(this)));
															#line 1053 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LookupCipherKeyActorState();
		static_cast<LookupCipherKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 381 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(cipherKeyMap.size() == 1);
															#line 383 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		auto cipherKeyMapItr = cipherKeyMap.find(cipherDetails);
															#line 384 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (cipherKeyMapItr == cipherKeyMap.end())
															#line 1080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 385 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent(SevError, "CipherKeyLookup_Failure") .detail("EncryptDomainId", cipherDetails.encryptDomainId) .detail("BaseCipherId", cipherDetails.baseCipherId) .detail("Salt", cipherDetails.salt);
															#line 389 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			return a_body1Catch1(encrypt_keys_fetch_failed(), loopDepth);
															#line 1086 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 392 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<LookupCipherKeyActor*>(this)->SAV<BlobGranuleCipherKey>::futures) { (void)(BlobGranuleCipherKey::fromBlobCipherKey(cipherKeyMapItr->second, *arena)); this->~LookupCipherKeyActorState(); static_cast<LookupCipherKeyActor*>(this)->destroy(); return 0; }
															#line 1090 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<LookupCipherKeyActor*>(this)->SAV< BlobGranuleCipherKey >::value()) BlobGranuleCipherKey(BlobGranuleCipherKey::fromBlobCipherKey(cipherKeyMapItr->second, *arena));
		this->~LookupCipherKeyActorState();
		static_cast<LookupCipherKeyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> const& __cipherKeyMap,int loopDepth) 
	{
															#line 377 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeyMap = __cipherKeyMap;
															#line 1102 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> && __cipherKeyMap,int loopDepth) 
	{
		cipherKeyMap = std::move(__cipherKeyMap);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LookupCipherKeyActor*>(this)->actor_wait_state > 0) static_cast<LookupCipherKeyActor*>(this)->actor_wait_state = 0;
		static_cast<LookupCipherKeyActor*>(this)->ActorCallback< LookupCipherKeyActor, 0, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >::remove();

	}
	void a_callback_fire(ActorCallback< LookupCipherKeyActor, 0, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*,std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> const& value) 
	{
		fdb_probe_actor_enter("lookupCipherKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LookupCipherKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupCipherKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LookupCipherKeyActor, 0, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*,std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> && value) 
	{
		fdb_probe_actor_enter("lookupCipherKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LookupCipherKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupCipherKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LookupCipherKeyActor, 0, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*,Error err) 
	{
		fdb_probe_actor_enter("lookupCipherKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LookupCipherKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("lookupCipherKey", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> dbInfo;
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	BlobCipherDetails cipherDetails;
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Arena* arena;
															#line 377 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> cipherKeyMap;
															#line 1185 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via lookupCipherKey()
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class LookupCipherKeyActor final : public Actor<BlobGranuleCipherKey>, public ActorCallback< LookupCipherKeyActor, 0, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >, public FastAllocated<LookupCipherKeyActor>, public LookupCipherKeyActorState<LookupCipherKeyActor> {
															#line 1190 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<LookupCipherKeyActor>::operator new;
	using FastAllocated<LookupCipherKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6077149728586990592UL, 6416926724994834944UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobGranuleCipherKey>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LookupCipherKeyActor, 0, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >;
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	LookupCipherKeyActor(Reference<AsyncVar<ServerDBInfo> const> const& dbInfo,BlobCipherDetails const& cipherDetails,Arena* const& arena) 
															#line 1207 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<BlobGranuleCipherKey>(),
		   LookupCipherKeyActorState<LookupCipherKeyActor>(dbInfo, cipherDetails, arena),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("lookupCipherKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18445237761726279168UL, 17528444550329885696UL);
		ActorExecutionContextHelper __helper(static_cast<LookupCipherKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("lookupCipherKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("lookupCipherKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LookupCipherKeyActor, 0, std::unordered_map<BlobCipherDetails, Reference<BlobCipherKey>> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<BlobGranuleCipherKey> lookupCipherKey( Reference<AsyncVar<ServerDBInfo> const> const& dbInfo, BlobCipherDetails const& cipherDetails, Arena* const& arena ) {
															#line 372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<BlobGranuleCipherKey>(new LookupCipherKeyActor(dbInfo, cipherDetails, arena));
															#line 1240 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 394 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

															#line 1245 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via getGranuleCipherKeysImpl()
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class GetGranuleCipherKeysImplActor>
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class GetGranuleCipherKeysImplActorState {
															#line 1252 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GetGranuleCipherKeysImplActorState(Reference<AsyncVar<ServerDBInfo> const> const& dbInfo,BlobCipherDetails const& textCipherDetails,BlobCipherDetails const& headerCipherDetails,StringRef const& ivRef,Arena* const& arena) 
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : dbInfo(dbInfo),
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   textCipherDetails(textCipherDetails),
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   headerCipherDetails(headerCipherDetails),
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   ivRef(ivRef),
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   arena(arena),
															#line 400 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   cipherKeysCtx()
															#line 1269 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this));

	}
	~GetGranuleCipherKeysImplActorState() 
	{
		fdb_probe_actor_destroy("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 403 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<BlobGranuleCipherKey> __when_expr_0 = lookupCipherKey(dbInfo, textCipherDetails, arena);
															#line 403 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<GetGranuleCipherKeysImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1286 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetGranuleCipherKeysImplActor*>(this)->actor_wait_state = 1;
															#line 403 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetGranuleCipherKeysImplActor, 0, BlobGranuleCipherKey >*>(static_cast<GetGranuleCipherKeysImplActor*>(this)));
															#line 1291 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetGranuleCipherKeysImplActorState();
		static_cast<GetGranuleCipherKeysImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(BlobGranuleCipherKey const& textCipherKey,int loopDepth) 
	{
															#line 404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.textCipherKey = textCipherKey;
															#line 407 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<BlobGranuleCipherKey> __when_expr_1 = lookupCipherKey(dbInfo, headerCipherDetails, arena);
															#line 407 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GetGranuleCipherKeysImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1318 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetGranuleCipherKeysImplActor*>(this)->actor_wait_state = 2;
															#line 407 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetGranuleCipherKeysImplActor, 1, BlobGranuleCipherKey >*>(static_cast<GetGranuleCipherKeysImplActor*>(this)));
															#line 1323 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(BlobGranuleCipherKey && textCipherKey,int loopDepth) 
	{
															#line 404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.textCipherKey = textCipherKey;
															#line 407 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<BlobGranuleCipherKey> __when_expr_1 = lookupCipherKey(dbInfo, headerCipherDetails, arena);
															#line 407 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GetGranuleCipherKeysImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1336 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetGranuleCipherKeysImplActor*>(this)->actor_wait_state = 2;
															#line 407 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetGranuleCipherKeysImplActor, 1, BlobGranuleCipherKey >*>(static_cast<GetGranuleCipherKeysImplActor*>(this)));
															#line 1341 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(BlobGranuleCipherKey const& textCipherKey,int loopDepth) 
	{
		loopDepth = a_body1cont1(textCipherKey, loopDepth);

		return loopDepth;
	}
	int a_body1when1(BlobGranuleCipherKey && textCipherKey,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(textCipherKey), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetGranuleCipherKeysImplActor*>(this)->actor_wait_state > 0) static_cast<GetGranuleCipherKeysImplActor*>(this)->actor_wait_state = 0;
		static_cast<GetGranuleCipherKeysImplActor*>(this)->ActorCallback< GetGranuleCipherKeysImplActor, 0, BlobGranuleCipherKey >::remove();

	}
	void a_callback_fire(ActorCallback< GetGranuleCipherKeysImplActor, 0, BlobGranuleCipherKey >*,BlobGranuleCipherKey const& value) 
	{
		fdb_probe_actor_enter("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleCipherKeysImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetGranuleCipherKeysImplActor, 0, BlobGranuleCipherKey >*,BlobGranuleCipherKey && value) 
	{
		fdb_probe_actor_enter("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleCipherKeysImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetGranuleCipherKeysImplActor, 0, BlobGranuleCipherKey >*,Error err) 
	{
		fdb_probe_actor_enter("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleCipherKeysImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(BlobGranuleCipherKey const& headerCipherKey,int loopDepth) 
	{
															#line 408 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.headerCipherKey = headerCipherKey;
															#line 411 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT_EQ(ivRef.size(), AES_256_IV_LENGTH);
															#line 412 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.ivRef = StringRef(*arena, ivRef);
															#line 414 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BG_ENCRYPT_COMPRESS_DEBUG)
															#line 1431 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 415 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent(SevDebug, "GetGranuleCipherKey") .detail("TextDomainId", cipherKeysCtx.textCipherKey.encryptDomainId) .detail("TextBaseCipherId", cipherKeysCtx.textCipherKey.baseCipherId) .detail("TextSalt", cipherKeysCtx.textCipherKey.salt) .detail("HeaderDomainId", cipherKeysCtx.textCipherKey.encryptDomainId) .detail("HeaderBaseCipherId", cipherKeysCtx.textCipherKey.baseCipherId) .detail("HeaderSalt", cipherKeysCtx.textCipherKey.salt) .detail("IVChksum", XXH3_64bits(cipherKeysCtx.ivRef.begin(), cipherKeysCtx.ivRef.size()));
															#line 1435 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 425 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<GetGranuleCipherKeysImplActor*>(this)->SAV<BlobGranuleCipherKeysCtx>::futures) { (void)(cipherKeysCtx); this->~GetGranuleCipherKeysImplActorState(); static_cast<GetGranuleCipherKeysImplActor*>(this)->destroy(); return 0; }
															#line 1439 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<GetGranuleCipherKeysImplActor*>(this)->SAV< BlobGranuleCipherKeysCtx >::value()) BlobGranuleCipherKeysCtx(std::move(cipherKeysCtx)); // state_var_RVO
		this->~GetGranuleCipherKeysImplActorState();
		static_cast<GetGranuleCipherKeysImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(BlobGranuleCipherKey && headerCipherKey,int loopDepth) 
	{
															#line 408 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.headerCipherKey = headerCipherKey;
															#line 411 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT_EQ(ivRef.size(), AES_256_IV_LENGTH);
															#line 412 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx.ivRef = StringRef(*arena, ivRef);
															#line 414 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BG_ENCRYPT_COMPRESS_DEBUG)
															#line 1457 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 415 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent(SevDebug, "GetGranuleCipherKey") .detail("TextDomainId", cipherKeysCtx.textCipherKey.encryptDomainId) .detail("TextBaseCipherId", cipherKeysCtx.textCipherKey.baseCipherId) .detail("TextSalt", cipherKeysCtx.textCipherKey.salt) .detail("HeaderDomainId", cipherKeysCtx.textCipherKey.encryptDomainId) .detail("HeaderBaseCipherId", cipherKeysCtx.textCipherKey.baseCipherId) .detail("HeaderSalt", cipherKeysCtx.textCipherKey.salt) .detail("IVChksum", XXH3_64bits(cipherKeysCtx.ivRef.begin(), cipherKeysCtx.ivRef.size()));
															#line 1461 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 425 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<GetGranuleCipherKeysImplActor*>(this)->SAV<BlobGranuleCipherKeysCtx>::futures) { (void)(cipherKeysCtx); this->~GetGranuleCipherKeysImplActorState(); static_cast<GetGranuleCipherKeysImplActor*>(this)->destroy(); return 0; }
															#line 1465 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<GetGranuleCipherKeysImplActor*>(this)->SAV< BlobGranuleCipherKeysCtx >::value()) BlobGranuleCipherKeysCtx(std::move(cipherKeysCtx)); // state_var_RVO
		this->~GetGranuleCipherKeysImplActorState();
		static_cast<GetGranuleCipherKeysImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(BlobGranuleCipherKey const& headerCipherKey,int loopDepth) 
	{
		loopDepth = a_body1cont2(headerCipherKey, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(BlobGranuleCipherKey && headerCipherKey,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(headerCipherKey), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetGranuleCipherKeysImplActor*>(this)->actor_wait_state > 0) static_cast<GetGranuleCipherKeysImplActor*>(this)->actor_wait_state = 0;
		static_cast<GetGranuleCipherKeysImplActor*>(this)->ActorCallback< GetGranuleCipherKeysImplActor, 1, BlobGranuleCipherKey >::remove();

	}
	void a_callback_fire(ActorCallback< GetGranuleCipherKeysImplActor, 1, BlobGranuleCipherKey >*,BlobGranuleCipherKey const& value) 
	{
		fdb_probe_actor_enter("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleCipherKeysImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetGranuleCipherKeysImplActor, 1, BlobGranuleCipherKey >*,BlobGranuleCipherKey && value) 
	{
		fdb_probe_actor_enter("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleCipherKeysImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetGranuleCipherKeysImplActor, 1, BlobGranuleCipherKey >*,Error err) 
	{
		fdb_probe_actor_enter("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleCipherKeysImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> dbInfo;
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	BlobCipherDetails textCipherDetails;
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	BlobCipherDetails headerCipherDetails;
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	StringRef ivRef;
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Arena* arena;
															#line 400 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	BlobGranuleCipherKeysCtx cipherKeysCtx;
															#line 1560 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via getGranuleCipherKeysImpl()
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class GetGranuleCipherKeysImplActor final : public Actor<BlobGranuleCipherKeysCtx>, public ActorCallback< GetGranuleCipherKeysImplActor, 0, BlobGranuleCipherKey >, public ActorCallback< GetGranuleCipherKeysImplActor, 1, BlobGranuleCipherKey >, public FastAllocated<GetGranuleCipherKeysImplActor>, public GetGranuleCipherKeysImplActorState<GetGranuleCipherKeysImplActor> {
															#line 1565 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<GetGranuleCipherKeysImplActor>::operator new;
	using FastAllocated<GetGranuleCipherKeysImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15905010661829856256UL, 5806756501339085824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobGranuleCipherKeysCtx>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetGranuleCipherKeysImplActor, 0, BlobGranuleCipherKey >;
friend struct ActorCallback< GetGranuleCipherKeysImplActor, 1, BlobGranuleCipherKey >;
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GetGranuleCipherKeysImplActor(Reference<AsyncVar<ServerDBInfo> const> const& dbInfo,BlobCipherDetails const& textCipherDetails,BlobCipherDetails const& headerCipherDetails,StringRef const& ivRef,Arena* const& arena) 
															#line 1583 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<BlobGranuleCipherKeysCtx>(),
		   GetGranuleCipherKeysImplActorState<GetGranuleCipherKeysImplActor>(dbInfo, textCipherDetails, headerCipherDetails, ivRef, arena),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7908145831042676480UL, 16145471069075575296UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleCipherKeysImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getGranuleCipherKeysImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getGranuleCipherKeysImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetGranuleCipherKeysImplActor, 0, BlobGranuleCipherKey >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetGranuleCipherKeysImplActor, 1, BlobGranuleCipherKey >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<BlobGranuleCipherKeysCtx> getGranuleCipherKeysImpl( Reference<AsyncVar<ServerDBInfo> const> const& dbInfo, BlobCipherDetails const& textCipherDetails, BlobCipherDetails const& headerCipherDetails, StringRef const& ivRef, Arena* const& arena ) {
															#line 395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<BlobGranuleCipherKeysCtx>(new GetGranuleCipherKeysImplActor(dbInfo, textCipherDetails, headerCipherDetails, ivRef, arena));
															#line 1617 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 427 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

Future<BlobGranuleCipherKeysCtx> getGranuleCipherKeysFromKeysMeta(Reference<AsyncVar<ServerDBInfo> const> dbInfo,
                                                                  BlobGranuleCipherKeysMeta cipherKeysMeta,
                                                                  Arena* arena) {
	BlobCipherDetails textCipherDetails(
	    cipherKeysMeta.textDomainId, cipherKeysMeta.textBaseCipherId, cipherKeysMeta.textSalt);

	BlobCipherDetails headerCipherDetails(
	    cipherKeysMeta.headerDomainId, cipherKeysMeta.headerBaseCipherId, cipherKeysMeta.headerSalt);

	StringRef ivRef = StringRef(*arena, cipherKeysMeta.ivStr);

	return getGranuleCipherKeysImpl(dbInfo, textCipherDetails, headerCipherDetails, ivRef, arena);
}

Future<BlobGranuleCipherKeysCtx> getGranuleCipherKeysFromKeysMetaRef(Reference<AsyncVar<ServerDBInfo> const> dbInfo,
                                                                     BlobGranuleCipherKeysMetaRef cipherKeysMetaRef,
                                                                     Arena* arena) {
	BlobCipherDetails textCipherDetails(
	    cipherKeysMetaRef.textDomainId, cipherKeysMetaRef.textBaseCipherId, cipherKeysMetaRef.textSalt);

	BlobCipherDetails headerCipherDetails(
	    cipherKeysMetaRef.headerDomainId, cipherKeysMetaRef.headerBaseCipherId, cipherKeysMetaRef.headerSalt);

	return getGranuleCipherKeysImpl(dbInfo, textCipherDetails, headerCipherDetails, cipherKeysMetaRef.ivRef, arena);
}

															#line 1648 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via readAndCheckGranuleLock()
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class ReadAndCheckGranuleLockActor>
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class ReadAndCheckGranuleLockActorState {
															#line 1655 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	ReadAndCheckGranuleLockActorState(Reference<ReadYourWritesTransaction> const& tr,KeyRange const& granuleRange,int64_t const& epoch,int64_t const& seqno) 
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : tr(tr),
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleRange(granuleRange),
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   epoch(epoch),
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   seqno(seqno),
															#line 458 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   lockKey(blobGranuleLockKeyFor(granuleRange))
															#line 1670 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("readAndCheckGranuleLock", reinterpret_cast<unsigned long>(this));

	}
	~ReadAndCheckGranuleLockActorState() 
	{
		fdb_probe_actor_destroy("readAndCheckGranuleLock", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 459 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr->get(lockKey);
															#line 459 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<ReadAndCheckGranuleLockActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1687 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadAndCheckGranuleLockActor*>(this)->actor_wait_state = 1;
															#line 459 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadAndCheckGranuleLockActor, 0, Optional<Value> >*>(static_cast<ReadAndCheckGranuleLockActor*>(this)));
															#line 1692 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadAndCheckGranuleLockActorState();
		static_cast<ReadAndCheckGranuleLockActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> const& lockValue,int loopDepth) 
	{
															#line 461 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!lockValue.present())
															#line 1715 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 464 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			return a_body1Catch1(granule_assignment_conflict(), loopDepth);
															#line 1719 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 467 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		std::tuple<int64_t, int64_t, UID> currentOwner = decodeBlobGranuleLockValue(lockValue.get());
															#line 468 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		checkGranuleLock(epoch, seqno, std::get<0>(currentOwner), std::get<1>(currentOwner));
															#line 472 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->addReadConflictRange(singleKeyRange(lockKey));
															#line 474 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<ReadAndCheckGranuleLockActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadAndCheckGranuleLockActorState(); static_cast<ReadAndCheckGranuleLockActor*>(this)->destroy(); return 0; }
															#line 1729 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<ReadAndCheckGranuleLockActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ReadAndCheckGranuleLockActorState();
		static_cast<ReadAndCheckGranuleLockActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> && lockValue,int loopDepth) 
	{
															#line 461 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!lockValue.present())
															#line 1741 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 464 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			return a_body1Catch1(granule_assignment_conflict(), loopDepth);
															#line 1745 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 467 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		std::tuple<int64_t, int64_t, UID> currentOwner = decodeBlobGranuleLockValue(lockValue.get());
															#line 468 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		checkGranuleLock(epoch, seqno, std::get<0>(currentOwner), std::get<1>(currentOwner));
															#line 472 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->addReadConflictRange(singleKeyRange(lockKey));
															#line 474 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<ReadAndCheckGranuleLockActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadAndCheckGranuleLockActorState(); static_cast<ReadAndCheckGranuleLockActor*>(this)->destroy(); return 0; }
															#line 1755 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<ReadAndCheckGranuleLockActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ReadAndCheckGranuleLockActorState();
		static_cast<ReadAndCheckGranuleLockActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<Value> const& lockValue,int loopDepth) 
	{
		loopDepth = a_body1cont1(lockValue, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> && lockValue,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(lockValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadAndCheckGranuleLockActor*>(this)->actor_wait_state > 0) static_cast<ReadAndCheckGranuleLockActor*>(this)->actor_wait_state = 0;
		static_cast<ReadAndCheckGranuleLockActor*>(this)->ActorCallback< ReadAndCheckGranuleLockActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadAndCheckGranuleLockActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("readAndCheckGranuleLock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadAndCheckGranuleLockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readAndCheckGranuleLock", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadAndCheckGranuleLockActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("readAndCheckGranuleLock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadAndCheckGranuleLockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readAndCheckGranuleLock", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadAndCheckGranuleLockActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("readAndCheckGranuleLock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadAndCheckGranuleLockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readAndCheckGranuleLock", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	KeyRange granuleRange;
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t epoch;
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t seqno;
															#line 458 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Key lockKey;
															#line 1848 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via readAndCheckGranuleLock()
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class ReadAndCheckGranuleLockActor final : public Actor<Void>, public ActorCallback< ReadAndCheckGranuleLockActor, 0, Optional<Value> >, public FastAllocated<ReadAndCheckGranuleLockActor>, public ReadAndCheckGranuleLockActorState<ReadAndCheckGranuleLockActor> {
															#line 1853 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<ReadAndCheckGranuleLockActor>::operator new;
	using FastAllocated<ReadAndCheckGranuleLockActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11929591658361997056UL, 17266785283799288576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadAndCheckGranuleLockActor, 0, Optional<Value> >;
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	ReadAndCheckGranuleLockActor(Reference<ReadYourWritesTransaction> const& tr,KeyRange const& granuleRange,int64_t const& epoch,int64_t const& seqno) 
															#line 1870 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<Void>(),
		   ReadAndCheckGranuleLockActorState<ReadAndCheckGranuleLockActor>(tr, granuleRange, epoch, seqno),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readAndCheckGranuleLock", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8364198591602703872UL, 18058843997897424128UL);
		ActorExecutionContextHelper __helper(static_cast<ReadAndCheckGranuleLockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readAndCheckGranuleLock");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readAndCheckGranuleLock", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadAndCheckGranuleLockActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<Void> readAndCheckGranuleLock( Reference<ReadYourWritesTransaction> const& tr, KeyRange const& granuleRange, int64_t const& epoch, int64_t const& seqno ) {
															#line 454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<Void>(new ReadAndCheckGranuleLockActor(tr, granuleRange, epoch, seqno));
															#line 1903 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 476 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

// Read snapshot and delta files for granule history, for completed granule
// Retries on error local to this function
															#line 1910 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via loadHistoryFiles()
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class LoadHistoryFilesActor>
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class LoadHistoryFilesActorState {
															#line 1917 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	LoadHistoryFilesActorState(Reference<BlobWorkerData> const& bwData,UID const& granuleID) 
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleID(granuleID),
															#line 480 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   tr(bwData->db),
															#line 481 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   range(blobGranuleFileKeyRangeFor(granuleID)),
															#line 482 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   startKey(range.begin),
															#line 483 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   files()
															#line 1934 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("loadHistoryFiles", reinterpret_cast<unsigned long>(this));

	}
	~LoadHistoryFilesActorState() 
	{
		fdb_probe_actor_destroy("loadHistoryFiles", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 484 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			;
															#line 1949 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadHistoryFilesActorState();
		static_cast<LoadHistoryFilesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 486 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 487 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 488 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readGranuleFiles(&tr, &startKey, range.end, &files, granuleID);
															#line 488 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<LoadHistoryFilesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1986 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadHistoryFilesActor*>(this)->actor_wait_state = 1;
															#line 488 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadHistoryFilesActor, 0, Void >*>(static_cast<LoadHistoryFilesActor*>(this)));
															#line 1991 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 491 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 491 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<LoadHistoryFilesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2015 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<LoadHistoryFilesActor*>(this)->actor_wait_state = 2;
															#line 491 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadHistoryFilesActor, 1, Void >*>(static_cast<LoadHistoryFilesActor*>(this)));
															#line 2020 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 489 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<LoadHistoryFilesActor*>(this)->SAV<GranuleFiles>::futures) { (void)(files); this->~LoadHistoryFilesActorState(); static_cast<LoadHistoryFilesActor*>(this)->destroy(); return 0; }
															#line 2035 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<LoadHistoryFilesActor*>(this)->SAV< GranuleFiles >::value()) GranuleFiles(std::move(files)); // state_var_RVO
		this->~LoadHistoryFilesActorState();
		static_cast<LoadHistoryFilesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 489 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<LoadHistoryFilesActor*>(this)->SAV<GranuleFiles>::futures) { (void)(files); this->~LoadHistoryFilesActorState(); static_cast<LoadHistoryFilesActor*>(this)->destroy(); return 0; }
															#line 2047 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<LoadHistoryFilesActor*>(this)->SAV< GranuleFiles >::value()) GranuleFiles(std::move(files)); // state_var_RVO
		this->~LoadHistoryFilesActorState();
		static_cast<LoadHistoryFilesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadHistoryFilesActor*>(this)->actor_wait_state > 0) static_cast<LoadHistoryFilesActor*>(this)->actor_wait_state = 0;
		static_cast<LoadHistoryFilesActor*>(this)->ActorCallback< LoadHistoryFilesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadHistoryFilesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadHistoryFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadHistoryFilesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadHistoryFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadHistoryFilesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadHistoryFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadHistoryFilesActor*>(this)->actor_wait_state > 0) static_cast<LoadHistoryFilesActor*>(this)->actor_wait_state = 0;
		static_cast<LoadHistoryFilesActor*>(this)->ActorCallback< LoadHistoryFilesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadHistoryFilesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadHistoryFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadHistoryFilesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadHistoryFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadHistoryFilesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadHistoryFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadHistoryFiles", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID granuleID;
															#line 480 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Transaction tr;
															#line 481 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	KeyRange range;
															#line 482 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Key startKey;
															#line 483 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GranuleFiles files;
															#line 2229 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via loadHistoryFiles()
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class LoadHistoryFilesActor final : public Actor<GranuleFiles>, public ActorCallback< LoadHistoryFilesActor, 0, Void >, public ActorCallback< LoadHistoryFilesActor, 1, Void >, public FastAllocated<LoadHistoryFilesActor>, public LoadHistoryFilesActorState<LoadHistoryFilesActor> {
															#line 2234 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<LoadHistoryFilesActor>::operator new;
	using FastAllocated<LoadHistoryFilesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13246179602492198656UL, 519987870125808128UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GranuleFiles>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadHistoryFilesActor, 0, Void >;
friend struct ActorCallback< LoadHistoryFilesActor, 1, Void >;
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	LoadHistoryFilesActor(Reference<BlobWorkerData> const& bwData,UID const& granuleID) 
															#line 2252 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<GranuleFiles>(),
		   LoadHistoryFilesActorState<LoadHistoryFilesActor>(bwData, granuleID),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadHistoryFiles", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3853450806983765504UL, 8569546073562004992UL);
		ActorExecutionContextHelper __helper(static_cast<LoadHistoryFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadHistoryFiles");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadHistoryFiles", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadHistoryFilesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadHistoryFilesActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<GranuleFiles> loadHistoryFiles( Reference<BlobWorkerData> const& bwData, UID const& granuleID ) {
															#line 479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<GranuleFiles>(new LoadHistoryFilesActor(bwData, granuleID));
															#line 2286 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 495 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

// read snapshot and delta files from previous owner of the active granule
// This is separated out from above because this is done as part of granule open transaction
															#line 2293 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via loadPreviousFiles()
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class LoadPreviousFilesActor>
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class LoadPreviousFilesActorState {
															#line 2300 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	LoadPreviousFilesActorState(Transaction* const& tr,UID const& granuleID) 
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : tr(tr),
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleID(granuleID),
															#line 499 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   range(blobGranuleFileKeyRangeFor(granuleID)),
															#line 501 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   startKey(range.begin),
															#line 502 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   files()
															#line 2315 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("loadPreviousFiles", reinterpret_cast<unsigned long>(this));

	}
	~LoadPreviousFilesActorState() 
	{
		fdb_probe_actor_destroy("loadPreviousFiles", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 503 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_0 = readGranuleFiles(tr, &startKey, range.end, &files, granuleID);
															#line 503 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<LoadPreviousFilesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2332 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadPreviousFilesActor*>(this)->actor_wait_state = 1;
															#line 503 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadPreviousFilesActor, 0, Void >*>(static_cast<LoadPreviousFilesActor*>(this)));
															#line 2337 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadPreviousFilesActorState();
		static_cast<LoadPreviousFilesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 504 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<LoadPreviousFilesActor*>(this)->SAV<GranuleFiles>::futures) { (void)(files); this->~LoadPreviousFilesActorState(); static_cast<LoadPreviousFilesActor*>(this)->destroy(); return 0; }
															#line 2360 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<LoadPreviousFilesActor*>(this)->SAV< GranuleFiles >::value()) GranuleFiles(std::move(files)); // state_var_RVO
		this->~LoadPreviousFilesActorState();
		static_cast<LoadPreviousFilesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 504 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<LoadPreviousFilesActor*>(this)->SAV<GranuleFiles>::futures) { (void)(files); this->~LoadPreviousFilesActorState(); static_cast<LoadPreviousFilesActor*>(this)->destroy(); return 0; }
															#line 2372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<LoadPreviousFilesActor*>(this)->SAV< GranuleFiles >::value()) GranuleFiles(std::move(files)); // state_var_RVO
		this->~LoadPreviousFilesActorState();
		static_cast<LoadPreviousFilesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadPreviousFilesActor*>(this)->actor_wait_state > 0) static_cast<LoadPreviousFilesActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPreviousFilesActor*>(this)->ActorCallback< LoadPreviousFilesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPreviousFilesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadPreviousFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadPreviousFilesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadPreviousFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadPreviousFilesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadPreviousFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPreviousFiles", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Transaction* tr;
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID granuleID;
															#line 499 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	KeyRange range;
															#line 501 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Key startKey;
															#line 502 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GranuleFiles files;
															#line 2465 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via loadPreviousFiles()
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class LoadPreviousFilesActor final : public Actor<GranuleFiles>, public ActorCallback< LoadPreviousFilesActor, 0, Void >, public FastAllocated<LoadPreviousFilesActor>, public LoadPreviousFilesActorState<LoadPreviousFilesActor> {
															#line 2470 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<LoadPreviousFilesActor>::operator new;
	using FastAllocated<LoadPreviousFilesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5501727810334325504UL, 7861793731405145344UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GranuleFiles>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadPreviousFilesActor, 0, Void >;
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	LoadPreviousFilesActor(Transaction* const& tr,UID const& granuleID) 
															#line 2487 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<GranuleFiles>(),
		   LoadPreviousFilesActorState<LoadPreviousFilesActor>(tr, granuleID),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadPreviousFiles", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2218662552644758528UL, 5903518421879972864UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPreviousFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadPreviousFiles");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadPreviousFiles", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadPreviousFilesActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<GranuleFiles> loadPreviousFiles( Transaction* const& tr, UID const& granuleID ) {
															#line 498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<GranuleFiles>(new LoadPreviousFilesActor(tr, granuleID));
															#line 2520 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 506 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

// To cleanup of the old change feed for the old granule range, all new sub-granules split from the old range must
// update shared state to coordinate when it is safe to clean up the old change feed.
//  his goes through 3 phases for each new sub-granule:
//  1. Starting - the blob manager writes all sub-granules with this state as a durable intent to split the range
//  2. Assigned - a worker that is assigned a sub-granule updates that granule's state here. This means that the
//  worker
//     has started a new change feed for the new sub-granule, but still needs to consume from the old change feed.
//  3. Done - the worker that is assigned this sub-granule has persisted all of the data from its part of the old
//  change
//     feed in delta files. From this granule's perspective, it is safe to clean up the old change feed.

// Once all sub-granules have reached step 2 (Assigned), the change feed can be safely "stopped" - it needs to
// continue to serve the mutations it has seen so far, but will not need any new mutations after this version. The
// last sub-granule to reach this step is responsible for committing the change feed stop as part of its transaction.
// Because this change feed stops commits in the same transaction as the worker's new change feed start, it is
// guaranteed that no versions are missed between the old and new change feed.
//
// Once all sub-granules have reached step 3 (Done), the change feed can be safely destroyed, as all of the
// mutations in the old change feed are guaranteed to be persisted in delta files. The last sub-granule to reach
// this step is responsible for committing the change feed destroy, and for cleaning up the split state for all
// sub-granules as part of its transaction.

															#line 2547 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via updateGranuleSplitState()
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class UpdateGranuleSplitStateActor>
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class UpdateGranuleSplitStateActorState {
															#line 2554 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UpdateGranuleSplitStateActorState(Transaction* const& tr,KeyRange const& parentGranuleRange,UID const& parentGranuleID,UID const& currentGranuleID,BlobGranuleSplitState const& newState) 
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : tr(tr),
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   parentGranuleRange(parentGranuleRange),
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   parentGranuleID(parentGranuleID),
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   currentGranuleID(currentGranuleID),
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   newState(newState),
															#line 534 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   currentRange(blobGranuleSplitKeyRangeFor(parentGranuleID))
															#line 2571 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("updateGranuleSplitState", reinterpret_cast<unsigned long>(this));

	}
	~UpdateGranuleSplitStateActorState() 
	{
		fdb_probe_actor_destroy("updateGranuleSplitState", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 536 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(currentRange, SERVER_KNOBS->BG_MAX_SPLIT_FANOUT + 1);
															#line 536 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2588 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state = 1;
															#line 536 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateGranuleSplitStateActor, 0, RangeResult >*>(static_cast<UpdateGranuleSplitStateActor*>(this)));
															#line 2593 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateGranuleSplitStateActorState();
		static_cast<UpdateGranuleSplitStateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 538 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->addWriteConflictRange(currentRange);
															#line 540 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (totalState.more || totalState.size() > SERVER_KNOBS->BG_MAX_SPLIT_FANOUT)
															#line 2618 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 541 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<RangeResult> __when_expr_1 = tr->getRange(currentRange, 10000);
															#line 541 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2624 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state = 2;
															#line 541 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< UpdateGranuleSplitStateActor, 1, RangeResult >*>(static_cast<UpdateGranuleSplitStateActor*>(this)));
															#line 2629 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(RangeResult const& __totalState,int loopDepth) 
	{
															#line 536 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		totalState = __totalState;
															#line 2643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && __totalState,int loopDepth) 
	{
		totalState = std::move(__totalState);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state > 0) static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateGranuleSplitStateActor*>(this)->ActorCallback< UpdateGranuleSplitStateActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateGranuleSplitStateActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateGranuleSplitStateActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateGranuleSplitStateActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 546 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (totalState.empty())
															#line 2722 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 547 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			ASSERT(newState == BlobGranuleSplitState::Done);
															#line 548 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (BW_DEBUG)
															#line 2728 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 549 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				fmt::print("Found empty split state for parent granule {0}\n", parentGranuleID.toString());
															#line 2732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 552 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (!static_cast<UpdateGranuleSplitStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateGranuleSplitStateActorState(); static_cast<UpdateGranuleSplitStateActor*>(this)->destroy(); return 0; }
															#line 2736 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			new (&static_cast<UpdateGranuleSplitStateActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~UpdateGranuleSplitStateActorState();
			static_cast<UpdateGranuleSplitStateActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 554 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(totalState.size() >= 2);
															#line 556 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		int total = totalState.size();
															#line 557 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		int totalStarted = 0;
															#line 558 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		int totalDone = 0;
															#line 559 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		BlobGranuleSplitState currentState = BlobGranuleSplitState::Unknown;
															#line 560 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		for( auto& it : totalState ) {
															#line 561 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			UID pid;
															#line 562 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			UID cid;
															#line 563 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			std::pair<UID, UID> k = decodeBlobGranuleSplitKey(it.key);
															#line 564 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			pid = k.first;
															#line 565 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			cid = k.second;
															#line 566 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			ASSERT(pid == parentGranuleID);
															#line 568 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			BlobGranuleSplitState st = decodeBlobGranuleSplitValue(it.value).first;
															#line 569 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			ASSERT(st != BlobGranuleSplitState::Unknown);
															#line 570 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (st == BlobGranuleSplitState::Initialized)
															#line 2772 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 571 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				totalStarted++;
															#line 2776 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
			else
			{
															#line 572 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (st == BlobGranuleSplitState::Done)
															#line 2782 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				{
															#line 573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					totalDone++;
															#line 2786 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				}
			}
															#line 575 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (cid == currentGranuleID)
															#line 2791 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 576 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				ASSERT(currentState == BlobGranuleSplitState::Unknown);
															#line 577 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				currentState = st;
															#line 2797 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
		}
															#line 581 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(currentState != BlobGranuleSplitState::Unknown);
															#line 583 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (currentState < newState)
															#line 2804 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 584 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (BW_DEBUG)
															#line 2808 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 585 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				fmt::print("Updating granule {0} split state from {1} {2} -> {3}\n", currentGranuleID.toString(), parentGranuleID.toString(), static_cast<int>(currentState), static_cast<int>(newState));
															#line 2812 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 592 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			Key myStateKey = blobGranuleSplitKeyFor(parentGranuleID, currentGranuleID);
															#line 593 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (newState == BlobGranuleSplitState::Done && currentState == BlobGranuleSplitState::Assigned && totalDone == total - 1)
															#line 2818 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 597 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (BW_DEBUG)
															#line 2822 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				{
															#line 598 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					fmt::print("{0} destroying old granule {1}\n", currentGranuleID.toString(), parentGranuleID.toString());
															#line 2826 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				}
															#line 601 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<Void> __when_expr_2 = updateChangeFeed(tr, granuleIDToCFKey(parentGranuleID), ChangeFeedStatus::CHANGE_FEED_DESTROY);
															#line 601 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2832 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
				static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state = 3;
															#line 601 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< UpdateGranuleSplitStateActor, 2, Void >*>(static_cast<UpdateGranuleSplitStateActor*>(this)));
															#line 2837 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 614 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				tr->atomicOp(myStateKey, blobGranuleSplitValueFor(newState), MutationRef::SetVersionstampedValue);
															#line 615 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (newState == BlobGranuleSplitState::Assigned && currentState == BlobGranuleSplitState::Initialized && totalStarted == 1)
															#line 2846 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				{
															#line 618 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					if (BW_DEBUG)
															#line 2850 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
					{
															#line 619 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
						fmt::print("{0} stopping change feed for old granule {1}\n", currentGranuleID.toString().c_str(), parentGranuleID.toString().c_str());
															#line 2854 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
					}
															#line 624 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					StrictFuture<Void> __when_expr_3 = updateChangeFeed( tr, KeyRef(granuleIDToCFKey(parentGranuleID)), ChangeFeedStatus::CHANGE_FEED_STOP);
															#line 624 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					if (static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2860 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
					if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont2when2(__when_expr_3.get(), loopDepth); };
					static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state = 4;
															#line 624 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< UpdateGranuleSplitStateActor, 3, Void >*>(static_cast<UpdateGranuleSplitStateActor*>(this)));
															#line 2865 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont17(loopDepth);
				}
			}
		}
		else
		{
															#line 629 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (BW_DEBUG)
															#line 2878 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 630 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				CODE_PROBE(true, "Out of order granule split state updates ignored", probe::decoration::rare);
															#line 631 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				fmt::print("Ignoring granule {0} split state from {1} {2} -> {3}\n", currentGranuleID.toString(), parentGranuleID.toString(), static_cast<int>(currentState), static_cast<int>(newState));
															#line 2884 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
			loopDepth = a_body1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(RangeResult const& tryAgain,int loopDepth) 
	{
															#line 542 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(!tryAgain.more);
															#line 543 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		totalState = tryAgain;
															#line 2897 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(RangeResult && tryAgain,int loopDepth) 
	{
															#line 542 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(!tryAgain.more);
															#line 543 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		totalState = tryAgain;
															#line 2908 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult const& tryAgain,int loopDepth) 
	{
		loopDepth = a_body1cont3(tryAgain, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult && tryAgain,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(tryAgain), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state > 0) static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateGranuleSplitStateActor*>(this)->ActorCallback< UpdateGranuleSplitStateActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateGranuleSplitStateActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< UpdateGranuleSplitStateActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< UpdateGranuleSplitStateActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 638 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<UpdateGranuleSplitStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateGranuleSplitStateActorState(); static_cast<UpdateGranuleSplitStateActor*>(this)->destroy(); return 0; }
															#line 2992 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<UpdateGranuleSplitStateActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateGranuleSplitStateActorState();
		static_cast<UpdateGranuleSplitStateActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont13(int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont15(Void const& _,int loopDepth) 
	{
															#line 603 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Key oldGranuleLockKey = blobGranuleLockKeyFor(parentGranuleRange);
															#line 611 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->clear(currentRange);
															#line 612 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "Granule split cleanup on last delta file persisted");
															#line 3014 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont15(Void && _,int loopDepth) 
	{
															#line 603 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Key oldGranuleLockKey = blobGranuleLockKeyFor(parentGranuleRange);
															#line 611 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->clear(currentRange);
															#line 612 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "Granule split cleanup on last delta file persisted");
															#line 3027 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont15(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont15(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state > 0) static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateGranuleSplitStateActor*>(this)->ActorCallback< UpdateGranuleSplitStateActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateGranuleSplitStateActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< UpdateGranuleSplitStateActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< UpdateGranuleSplitStateActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont17(int loopDepth) 
	{
															#line 627 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "Granule split stopping change feed");
															#line 3111 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont18(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1cont18(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont17(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont18(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont18(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state > 0) static_cast<UpdateGranuleSplitStateActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateGranuleSplitStateActor*>(this)->ActorCallback< UpdateGranuleSplitStateActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateGranuleSplitStateActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< UpdateGranuleSplitStateActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< UpdateGranuleSplitStateActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Transaction* tr;
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	KeyRange parentGranuleRange;
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID parentGranuleID;
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID currentGranuleID;
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	BlobGranuleSplitState newState;
															#line 534 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	KeyRange currentRange;
															#line 536 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	RangeResult totalState;
															#line 3217 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via updateGranuleSplitState()
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class UpdateGranuleSplitStateActor final : public Actor<Void>, public ActorCallback< UpdateGranuleSplitStateActor, 0, RangeResult >, public ActorCallback< UpdateGranuleSplitStateActor, 1, RangeResult >, public ActorCallback< UpdateGranuleSplitStateActor, 2, Void >, public ActorCallback< UpdateGranuleSplitStateActor, 3, Void >, public FastAllocated<UpdateGranuleSplitStateActor>, public UpdateGranuleSplitStateActorState<UpdateGranuleSplitStateActor> {
															#line 3222 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<UpdateGranuleSplitStateActor>::operator new;
	using FastAllocated<UpdateGranuleSplitStateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1749095295002271488UL, 15347457930845728000UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateGranuleSplitStateActor, 0, RangeResult >;
friend struct ActorCallback< UpdateGranuleSplitStateActor, 1, RangeResult >;
friend struct ActorCallback< UpdateGranuleSplitStateActor, 2, Void >;
friend struct ActorCallback< UpdateGranuleSplitStateActor, 3, Void >;
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UpdateGranuleSplitStateActor(Transaction* const& tr,KeyRange const& parentGranuleRange,UID const& parentGranuleID,UID const& currentGranuleID,BlobGranuleSplitState const& newState) 
															#line 3242 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<Void>(),
		   UpdateGranuleSplitStateActorState<UpdateGranuleSplitStateActor>(tr, parentGranuleRange, parentGranuleID, currentGranuleID, newState),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1416020089856908800UL, 10264545448137153280UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateGranuleSplitState");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateGranuleSplitState", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateGranuleSplitStateActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< UpdateGranuleSplitStateActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< UpdateGranuleSplitStateActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< UpdateGranuleSplitStateActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<Void> updateGranuleSplitState( Transaction* const& tr, KeyRange const& parentGranuleRange, UID const& parentGranuleID, UID const& currentGranuleID, BlobGranuleSplitState const& newState ) {
															#line 529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<Void>(new UpdateGranuleSplitStateActor(tr, parentGranuleRange, parentGranuleID, currentGranuleID, newState));
															#line 3278 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

// Returns the split state for a given granule on granule reassignment, or unknown if it doesn't exist (meaning the
// granule splitting finished)
															#line 3285 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via getGranuleSplitState()
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class GetGranuleSplitStateActor>
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class GetGranuleSplitStateActorState {
															#line 3292 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GetGranuleSplitStateActorState(Transaction* const& tr,UID const& parentGranuleID,UID const& currentGranuleID) 
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : tr(tr),
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   parentGranuleID(parentGranuleID),
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   currentGranuleID(currentGranuleID)
															#line 3303 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("getGranuleSplitState", reinterpret_cast<unsigned long>(this));

	}
	~GetGranuleSplitStateActorState() 
	{
		fdb_probe_actor_destroy("getGranuleSplitState", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 646 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			Key myStateKey = blobGranuleSplitKeyFor(parentGranuleID, currentGranuleID);
															#line 648 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr->get(myStateKey);
															#line 648 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<GetGranuleSplitStateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3322 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetGranuleSplitStateActor*>(this)->actor_wait_state = 1;
															#line 648 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetGranuleSplitStateActor, 0, Optional<Value> >*>(static_cast<GetGranuleSplitStateActor*>(this)));
															#line 3327 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetGranuleSplitStateActorState();
		static_cast<GetGranuleSplitStateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> const& st,int loopDepth) 
	{
															#line 649 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (st.present())
															#line 3350 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 650 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (!static_cast<GetGranuleSplitStateActor*>(this)->SAV<std::pair<BlobGranuleSplitState, Version>>::futures) { (void)(decodeBlobGranuleSplitValue(st.get())); this->~GetGranuleSplitStateActorState(); static_cast<GetGranuleSplitStateActor*>(this)->destroy(); return 0; }
															#line 3354 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			new (&static_cast<GetGranuleSplitStateActor*>(this)->SAV< std::pair<BlobGranuleSplitState, Version> >::value()) std::pair<BlobGranuleSplitState, Version>(decodeBlobGranuleSplitValue(st.get()));
			this->~GetGranuleSplitStateActorState();
			static_cast<GetGranuleSplitStateActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 652 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (!static_cast<GetGranuleSplitStateActor*>(this)->SAV<std::pair<BlobGranuleSplitState, Version>>::futures) { (void)(std::pair(BlobGranuleSplitState::Unknown, invalidVersion)); this->~GetGranuleSplitStateActorState(); static_cast<GetGranuleSplitStateActor*>(this)->destroy(); return 0; }
															#line 3364 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			new (&static_cast<GetGranuleSplitStateActor*>(this)->SAV< std::pair<BlobGranuleSplitState, Version> >::value()) std::pair<BlobGranuleSplitState, Version>(std::pair(BlobGranuleSplitState::Unknown, invalidVersion));
			this->~GetGranuleSplitStateActorState();
			static_cast<GetGranuleSplitStateActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> && st,int loopDepth) 
	{
															#line 649 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (st.present())
															#line 3377 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 650 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (!static_cast<GetGranuleSplitStateActor*>(this)->SAV<std::pair<BlobGranuleSplitState, Version>>::futures) { (void)(decodeBlobGranuleSplitValue(st.get())); this->~GetGranuleSplitStateActorState(); static_cast<GetGranuleSplitStateActor*>(this)->destroy(); return 0; }
															#line 3381 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			new (&static_cast<GetGranuleSplitStateActor*>(this)->SAV< std::pair<BlobGranuleSplitState, Version> >::value()) std::pair<BlobGranuleSplitState, Version>(decodeBlobGranuleSplitValue(st.get()));
			this->~GetGranuleSplitStateActorState();
			static_cast<GetGranuleSplitStateActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 652 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (!static_cast<GetGranuleSplitStateActor*>(this)->SAV<std::pair<BlobGranuleSplitState, Version>>::futures) { (void)(std::pair(BlobGranuleSplitState::Unknown, invalidVersion)); this->~GetGranuleSplitStateActorState(); static_cast<GetGranuleSplitStateActor*>(this)->destroy(); return 0; }
															#line 3391 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			new (&static_cast<GetGranuleSplitStateActor*>(this)->SAV< std::pair<BlobGranuleSplitState, Version> >::value()) std::pair<BlobGranuleSplitState, Version>(std::pair(BlobGranuleSplitState::Unknown, invalidVersion));
			this->~GetGranuleSplitStateActorState();
			static_cast<GetGranuleSplitStateActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}

		return loopDepth;
	}
	int a_body1when1(Optional<Value> const& st,int loopDepth) 
	{
		loopDepth = a_body1cont1(st, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> && st,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(st), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetGranuleSplitStateActor*>(this)->actor_wait_state > 0) static_cast<GetGranuleSplitStateActor*>(this)->actor_wait_state = 0;
		static_cast<GetGranuleSplitStateActor*>(this)->ActorCallback< GetGranuleSplitStateActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GetGranuleSplitStateActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("getGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetGranuleSplitStateActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("getGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetGranuleSplitStateActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("getGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getGranuleSplitState", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Transaction* tr;
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID parentGranuleID;
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID currentGranuleID;
															#line 3481 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via getGranuleSplitState()
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class GetGranuleSplitStateActor final : public Actor<std::pair<BlobGranuleSplitState, Version>>, public ActorCallback< GetGranuleSplitStateActor, 0, Optional<Value> >, public FastAllocated<GetGranuleSplitStateActor>, public GetGranuleSplitStateActorState<GetGranuleSplitStateActor> {
															#line 3486 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<GetGranuleSplitStateActor>::operator new;
	using FastAllocated<GetGranuleSplitStateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8470841308605408512UL, 18307875415743057664UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::pair<BlobGranuleSplitState, Version>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetGranuleSplitStateActor, 0, Optional<Value> >;
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GetGranuleSplitStateActor(Transaction* const& tr,UID const& parentGranuleID,UID const& currentGranuleID) 
															#line 3503 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<std::pair<BlobGranuleSplitState, Version>>(),
		   GetGranuleSplitStateActorState<GetGranuleSplitStateActor>(tr, parentGranuleID, currentGranuleID),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getGranuleSplitState", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11682202558274289408UL, 7084910482377861120UL);
		ActorExecutionContextHelper __helper(static_cast<GetGranuleSplitStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getGranuleSplitState");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getGranuleSplitState", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetGranuleSplitStateActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<std::pair<BlobGranuleSplitState, Version>> getGranuleSplitState( Transaction* const& tr, UID const& parentGranuleID, UID const& currentGranuleID ) {
															#line 643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<std::pair<BlobGranuleSplitState, Version>>(new GetGranuleSplitStateActor(tr, parentGranuleID, currentGranuleID));
															#line 3536 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 655 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

// tries to use writeEntireFile if possible, but if too big falls back to multi-part upload
															#line 3542 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via writeFile()
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class WriteFileActor>
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WriteFileActorState {
															#line 3549 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WriteFileActorState(Reference<BackupContainerFileSystem> const& writeBStore,std::string const& fname,Value const& serialized) 
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : writeBStore(writeBStore),
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   fname(fname),
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   serialized(serialized)
															#line 3560 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("writeFile", reinterpret_cast<unsigned long>(this));

	}
	~WriteFileActorState() 
	{
		fdb_probe_actor_destroy("writeFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 658 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (!SERVER_KNOBS->BG_WRITE_MULTIPART)
															#line 3575 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
				try {
															#line 660 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					fileContents = serialized.toString();
															#line 661 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					StrictFuture<Void> __when_expr_0 = writeBStore->writeEntireFile(fname, fileContents);
															#line 661 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					if (static_cast<WriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3584 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<WriteFileActor*>(this)->actor_wait_state = 1;
															#line 661 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteFileActor, 0, Void >*>(static_cast<WriteFileActor*>(this)));
															#line 3589 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
					loopDepth = 0;
				}
				catch (Error& error) {
					loopDepth = a_body1Catch2(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1Catch2(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteFileActorState();
		static_cast<WriteFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 671 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Reference<IBackupFile>> __when_expr_1 = writeBStore->writeFile(fname);
															#line 671 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3625 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteFileActor*>(this)->actor_wait_state = 2;
															#line 671 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteFileActor, 1, Reference<IBackupFile> >*>(static_cast<WriteFileActor*>(this)));
															#line 3630 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 665 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (e.code() != error_code_file_too_large)
															#line 3646 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 666 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3650 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 662 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WriteFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteFileActorState(); static_cast<WriteFileActor*>(this)->destroy(); return 0; }
															#line 3666 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WriteFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteFileActorState();
		static_cast<WriteFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 662 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WriteFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteFileActorState(); static_cast<WriteFileActor*>(this)->destroy(); return 0; }
															#line 3678 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WriteFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteFileActorState();
		static_cast<WriteFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteFileActor*>(this)->actor_wait_state > 0) static_cast<WriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteFileActor*>(this)->ActorCallback< WriteFileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteFileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteFileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteFileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 672 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_2 = objectFile->append(serialized.begin(), serialized.size());
															#line 672 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3767 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont5when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteFileActor*>(this)->actor_wait_state = 3;
															#line 672 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteFileActor, 2, Void >*>(static_cast<WriteFileActor*>(this)));
															#line 3772 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Reference<IBackupFile> const& __objectFile,int loopDepth) 
	{
															#line 671 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		objectFile = __objectFile;
															#line 3781 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Reference<IBackupFile> && __objectFile,int loopDepth) 
	{
		objectFile = std::move(__objectFile);
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteFileActor*>(this)->actor_wait_state > 0) static_cast<WriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteFileActor*>(this)->ActorCallback< WriteFileActor, 1, Reference<IBackupFile> >::remove();

	}
	void a_callback_fire(ActorCallback< WriteFileActor, 1, Reference<IBackupFile> >*,Reference<IBackupFile> const& value) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteFileActor, 1, Reference<IBackupFile> >*,Reference<IBackupFile> && value) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteFileActor, 1, Reference<IBackupFile> >*,Error err) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_3 = objectFile->finish();
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3862 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont6when1(__when_expr_3.get(), loopDepth); };
		static_cast<WriteFileActor*>(this)->actor_wait_state = 4;
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WriteFileActor, 3, Void >*>(static_cast<WriteFileActor*>(this)));
															#line 3867 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_3 = objectFile->finish();
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3878 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont6when1(__when_expr_3.get(), loopDepth); };
		static_cast<WriteFileActor*>(this)->actor_wait_state = 4;
															#line 673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WriteFileActor, 3, Void >*>(static_cast<WriteFileActor*>(this)));
															#line 3883 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WriteFileActor*>(this)->actor_wait_state > 0) static_cast<WriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteFileActor*>(this)->ActorCallback< WriteFileActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteFileActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WriteFileActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WriteFileActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 675 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WriteFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteFileActorState(); static_cast<WriteFileActor*>(this)->destroy(); return 0; }
															#line 3967 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WriteFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteFileActorState();
		static_cast<WriteFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 675 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WriteFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteFileActorState(); static_cast<WriteFileActor*>(this)->destroy(); return 0; }
															#line 3979 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WriteFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteFileActorState();
		static_cast<WriteFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WriteFileActor*>(this)->actor_wait_state > 0) static_cast<WriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteFileActor*>(this)->ActorCallback< WriteFileActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteFileActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WriteFileActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WriteFileActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BackupContainerFileSystem> writeBStore;
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	std::string fname;
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Value serialized;
															#line 660 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	std::string fileContents;
															#line 671 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<IBackupFile> objectFile;
															#line 4072 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via writeFile()
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WriteFileActor final : public Actor<Void>, public ActorCallback< WriteFileActor, 0, Void >, public ActorCallback< WriteFileActor, 1, Reference<IBackupFile> >, public ActorCallback< WriteFileActor, 2, Void >, public ActorCallback< WriteFileActor, 3, Void >, public FastAllocated<WriteFileActor>, public WriteFileActorState<WriteFileActor> {
															#line 4077 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<WriteFileActor>::operator new;
	using FastAllocated<WriteFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5237775581923705088UL, 11993283299958805248UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteFileActor, 0, Void >;
friend struct ActorCallback< WriteFileActor, 1, Reference<IBackupFile> >;
friend struct ActorCallback< WriteFileActor, 2, Void >;
friend struct ActorCallback< WriteFileActor, 3, Void >;
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WriteFileActor(Reference<BackupContainerFileSystem> const& writeBStore,std::string const& fname,Value const& serialized) 
															#line 4097 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<Void>(),
		   WriteFileActorState<WriteFileActor>(writeBStore, fname, serialized),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(464077129954895360UL, 14623206835353640448UL);
		ActorExecutionContextHelper __helper(static_cast<WriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteFileActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteFileActor, 1, Reference<IBackupFile> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WriteFileActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WriteFileActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<Void> writeFile( Reference<BackupContainerFileSystem> const& writeBStore, std::string const& fname, Value const& serialized ) {
															#line 657 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<Void>(new WriteFileActor(writeBStore, fname, serialized));
															#line 4133 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 677 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

// writeDelta file writes speculatively in the common case to optimize throughput. It creates the s3 object even though
// the data in it may not yet be committed, and even though previous delta files with lower versioned data may still be
// in flight. The synchronization happens after the s3 file is written, but before we update the FDB index of what files
// exist. Before updating FDB, we ensure the version is committed and all previous delta files have updated FDB.
															#line 4142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via writeDeltaFile()
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class WriteDeltaFileActor>
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WriteDeltaFileActorState {
															#line 4149 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WriteDeltaFileActorState(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,KeyRange const& keyRange,UID const& granuleID,int64_t const& epoch,int64_t const& seqno,Standalone<GranuleDeltas> const& deltasToWrite,Version const& currentDeltaVersion,Future<BlobFileIndex> const& previousDeltaFileFuture,Future<Void> const& waitCommitted,Optional<std::pair<KeyRange, UID>> const& oldGranuleComplete,Future<Void> const& startDeltaFileWrite,int64_t const& deltaFileBudget) 
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   bstore(bstore),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   keyRange(keyRange),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleID(granuleID),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   epoch(epoch),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   seqno(seqno),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   deltasToWrite(deltasToWrite),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   currentDeltaVersion(currentDeltaVersion),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   previousDeltaFileFuture(previousDeltaFileFuture),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   waitCommitted(waitCommitted),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   oldGranuleComplete(oldGranuleComplete),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   startDeltaFileWrite(startDeltaFileWrite),
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   deltaFileBudget(deltaFileBudget)
															#line 4180 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("writeDeltaFile", reinterpret_cast<unsigned long>(this));

	}
	~WriteDeltaFileActorState() 
	{
		fdb_probe_actor_destroy("writeDeltaFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 695 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_0 = startDeltaFileWrite;
															#line 695 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4197 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 1;
															#line 695 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 0, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4202 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteDeltaFileActorState();
		static_cast<WriteDeltaFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 696 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		holdingLock = FlowLock::Releaser(*bwData->deltaWritesBudget, deltaFileBudget);
															#line 698 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::BlobWorkerUpdateStorage);
															#line 698 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4229 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 2;
															#line 698 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 1, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4234 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 696 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		holdingLock = FlowLock::Releaser(*bwData->deltaWritesBudget, deltaFileBudget);
															#line 698 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::BlobWorkerUpdateStorage);
															#line 698 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4247 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 2;
															#line 698 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 1, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4252 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 700 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		fileName = randomBGFilename(bwData->id, granuleID, currentDeltaVersion, ".delta");
															#line 702 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx = Optional<BlobGranuleCipherKeysCtx>();
															#line 703 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysMeta = Optional<BlobGranuleCipherKeysMeta>();
															#line 704 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		arena = Arena();
															#line 706 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (bwData->encryptMode.isEncryptionEnabled())
															#line 4344 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 707 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<BlobGranuleCipherKeysCtx> __when_expr_2 = getLatestGranuleCipherKeys(bwData, keyRange, &arena);
															#line 707 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4350 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 3;
															#line 707 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 2, BlobGranuleCipherKeysCtx >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4355 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 700 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		fileName = randomBGFilename(bwData->id, granuleID, currentDeltaVersion, ".delta");
															#line 702 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx = Optional<BlobGranuleCipherKeysCtx>();
															#line 703 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysMeta = Optional<BlobGranuleCipherKeysMeta>();
															#line 704 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		arena = Arena();
															#line 706 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (bwData->encryptMode.isEncryptionEnabled())
															#line 4377 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 707 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<BlobGranuleCipherKeysCtx> __when_expr_2 = getLatestGranuleCipherKeys(bwData, keyRange, &arena);
															#line 707 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4383 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 3;
															#line 707 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 2, BlobGranuleCipherKeysCtx >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4388 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 712 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		compressFilter = getBlobFileCompressFilter();
															#line 713 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(!bwData->encryptMode.isEncryptionEnabled() || cipherKeysCtx.present());
															#line 714 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		serialized = serializeChunkedDeltaFile(StringRef(fileName), deltasToWrite, keyRange, SERVER_KNOBS->BG_DELTA_FILE_TARGET_CHUNK_BYTES, compressFilter, cipherKeysCtx);
															#line 720 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		logicalSize = deltasToWrite.expectedSize();
															#line 721 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		serializedSize = serialized.size();
															#line 722 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.compressionBytesRaw += logicalSize;
															#line 723 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.compressionBytesFinal += serializedSize;
															#line 726 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		deltasToWrite = Standalone<GranuleDeltas>();
															#line 728 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		writeBStore = Reference<BackupContainerFileSystem>();
															#line 729 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		fname = std::string();
															#line 730 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		std::tie(writeBStore, fname) = bstore->createForWrite(fileName);
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		startTimer = g_network->timer();
															#line 734 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_3 = writeFile(writeBStore, fname, serialized);
															#line 734 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4503 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 4;
															#line 734 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 3, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4508 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(BlobGranuleCipherKeysCtx const& ciphKeysCtx,int loopDepth) 
	{
															#line 708 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx = std::move(ciphKeysCtx);
															#line 709 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysMeta = BlobGranuleCipherKeysCtx::toCipherKeysMeta(cipherKeysCtx.get());
															#line 4519 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(BlobGranuleCipherKeysCtx && ciphKeysCtx,int loopDepth) 
	{
															#line 708 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx = std::move(ciphKeysCtx);
															#line 709 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysMeta = BlobGranuleCipherKeysCtx::toCipherKeysMeta(cipherKeysCtx.get());
															#line 4530 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(BlobGranuleCipherKeysCtx const& ciphKeysCtx,int loopDepth) 
	{
		loopDepth = a_body1cont4(ciphKeysCtx, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(BlobGranuleCipherKeysCtx && ciphKeysCtx,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(ciphKeysCtx), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 2, BlobGranuleCipherKeysCtx >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 2, BlobGranuleCipherKeysCtx >*,BlobGranuleCipherKeysCtx const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 2, BlobGranuleCipherKeysCtx >*,BlobGranuleCipherKeysCtx && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 2, BlobGranuleCipherKeysCtx >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 736 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		++bwData->stats.s3PutReqs;
															#line 737 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		++bwData->stats.deltaFilesWritten;
															#line 738 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.deltaBytesWritten += serializedSize;
															#line 739 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		double duration = g_network->timer() - startTimer;
															#line 740 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.deltaBlobWriteLatencySample.addMeasurement(duration);
															#line 743 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		serialized = Value();
															#line 747 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		holdingLock.release();
															#line 749 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		numIterations = 0;
															#line 750 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		startTimer = g_network->timer();
															#line 4630 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		try {
															#line 753 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_4 = waitCommitted;
															#line 753 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont6Catch1(actor_cancelled(), loopDepth);
															#line 4636 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont6Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 5;
															#line 753 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 4, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4641 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont6Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont6Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 736 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		++bwData->stats.s3PutReqs;
															#line 737 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		++bwData->stats.deltaFilesWritten;
															#line 738 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.deltaBytesWritten += serializedSize;
															#line 739 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		double duration = g_network->timer() - startTimer;
															#line 740 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.deltaBlobWriteLatencySample.addMeasurement(duration);
															#line 743 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		serialized = Value();
															#line 747 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		holdingLock.release();
															#line 749 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		numIterations = 0;
															#line 750 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		startTimer = g_network->timer();
															#line 4672 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		try {
															#line 753 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_4 = waitCommitted;
															#line 753 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont6Catch1(actor_cancelled(), loopDepth);
															#line 4678 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont6Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 5;
															#line 753 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 4, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4683 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont6Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont6Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont6Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 824 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (numIterations > 0)
															#line 4774 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 825 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				CODE_PROBE(true, "Granule potentially leaving orphaned delta file");
															#line 826 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 4780 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 828 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (BW_DEBUG)
															#line 4784 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 829 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				fmt::print("deleting delta file {0} after error {1}\n", fname, e.name());
															#line 4788 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 831 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			CODE_PROBE(true, "Granule cleaning up delta file after error");
															#line 832 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			++bwData->stats.s3DeleteReqs;
															#line 833 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			bwData->addActor.send(writeBStore->deleteFile(fname));
															#line 834 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 4798 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
															#line 754 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<BlobFileIndex> __when_expr_5 = previousDeltaFileFuture;
															#line 754 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont6Catch1(actor_cancelled(), loopDepth);
															#line 4814 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont6Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 6;
															#line 754 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 5, BlobFileIndex >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4819 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
															#line 754 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<BlobFileIndex> __when_expr_5 = previousDeltaFileFuture;
															#line 754 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont6Catch1(actor_cancelled(), loopDepth);
															#line 4830 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont6Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 6;
															#line 754 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 5, BlobFileIndex >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4835 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont9(BlobFileIndex const& prev,int loopDepth) 
	{
															#line 755 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(0, TaskPriority::BlobWorkerUpdateFDB);
															#line 755 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont6Catch1(actor_cancelled(), loopDepth);
															#line 4921 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont6Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont9when1(__when_expr_6.get(), loopDepth); };
		static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 7;
															#line 755 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 6, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4926 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont9(BlobFileIndex && prev,int loopDepth) 
	{
															#line 755 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(0, TaskPriority::BlobWorkerUpdateFDB);
															#line 755 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont6Catch1(actor_cancelled(), loopDepth);
															#line 4937 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont6Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont9when1(__when_expr_6.get(), loopDepth); };
		static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 7;
															#line 755 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 6, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 4942 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont8when1(BlobFileIndex const& prev,int loopDepth) 
	{
		loopDepth = a_body1cont9(prev, loopDepth);

		return loopDepth;
	}
	int a_body1cont8when1(BlobFileIndex && prev,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(prev), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 5, BlobFileIndex >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 5, BlobFileIndex >*,BlobFileIndex const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 5, BlobFileIndex >*,BlobFileIndex && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 5, BlobFileIndex >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont6Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>(bwData->db);
															#line 759 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 5028 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont10loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>(bwData->db);
															#line 759 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 5039 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont10loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont6Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont10loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont10loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1(int loopDepth) 
	{
															#line 760 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 761 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 762 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 5134 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		try {
															#line 764 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_7 = readAndCheckGranuleLock(tr, keyRange, epoch, seqno);
															#line 764 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont10loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5140 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont10loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont10loopBody1when1(__when_expr_7.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 8;
															#line 764 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 7, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 5145 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont10loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont10loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont10loopHead1(0);

		return loopDepth;
	}
	int a_body1cont10loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 817 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_12 = tr->onError(e);
															#line 817 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont6Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5169 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1cont6Catch1(__when_expr_12.getError(), std::max(0, loopDepth - 1)); else return a_body1cont10loopBody1Catch1when1(__when_expr_12.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 13;
															#line 817 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 12, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 5174 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont6Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont6Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont10loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 765 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		numIterations++;
															#line 767 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Key dfKey = blobGranuleFileKeyFor(granuleID, currentDeltaVersion, 'D');
															#line 769 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Value dfValue = blobGranuleFileValueFor(fname, 0, serializedSize, serializedSize, logicalSize, cipherKeysMeta);
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->set(dfKey, dfValue);
															#line 773 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (oldGranuleComplete.present())
															#line 5197 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 774 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_8 = updateGranuleSplitState(&tr->getTransaction(), oldGranuleComplete.get().first, oldGranuleComplete.get().second, granuleID, BlobGranuleSplitState::Done);
															#line 774 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont10loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5203 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont10loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont10loopBody1cont2when1(__when_expr_8.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 9;
															#line 774 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 8, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 5208 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont10loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 765 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		numIterations++;
															#line 767 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Key dfKey = blobGranuleFileKeyFor(granuleID, currentDeltaVersion, 'D');
															#line 769 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Value dfValue = blobGranuleFileValueFor(fname, 0, serializedSize, serializedSize, logicalSize, cipherKeysMeta);
															#line 771 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->set(dfKey, dfValue);
															#line 773 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (oldGranuleComplete.present())
															#line 5230 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 774 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_8 = updateGranuleSplitState(&tr->getTransaction(), oldGranuleComplete.get().first, oldGranuleComplete.get().second, granuleID, BlobGranuleSplitState::Done);
															#line 774 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont10loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5236 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont10loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont10loopBody1cont2when1(__when_expr_8.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 9;
															#line 774 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 8, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 5241 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont10loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont10loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont10loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont10loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont10loopBody1cont3(int loopDepth) 
	{
															#line 781 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_9 = tr->commit();
															#line 781 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont10loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5332 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont10loopBody1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont10loopBody1cont3when1(__when_expr_9.get(), loopDepth); };
		static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 10;
															#line 781 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 9, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 5337 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont10loopBody1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont10loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont10loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont10loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont10loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 782 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->addGRVHistory(tr->getReadVersion().get());
															#line 783 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 5435 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 784 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print( "Granule {0} [{1} - {2}) updated fdb with delta file {3} of size {4} at version {5}, cv={6}\n", granuleID.toString(), keyRange.begin.printable(), keyRange.end.printable(), fname, serializedSize, currentDeltaVersion, tr->getCommittedVersion());
															#line 5439 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 795 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BUGGIFY && bwData->maybeInjectTargetedRestart())
															#line 5443 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 796 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_10 = Never();
															#line 796 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont10loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5449 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont10loopBody1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont10loopBody1cont5when1(__when_expr_10.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 11;
															#line 796 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 10, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 5454 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont10loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 782 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->addGRVHistory(tr->getReadVersion().get());
															#line 783 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 5470 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 784 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print( "Granule {0} [{1} - {2}) updated fdb with delta file {3} of size {4} at version {5}, cv={6}\n", granuleID.toString(), keyRange.begin.printable(), keyRange.end.printable(), fname, serializedSize, currentDeltaVersion, tr->getCommittedVersion());
															#line 5474 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 795 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BUGGIFY && bwData->maybeInjectTargetedRestart())
															#line 5478 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 796 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_10 = Never();
															#line 796 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont10loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5484 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont10loopBody1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont10loopBody1cont5when1(__when_expr_10.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 11;
															#line 796 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 10, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 5489 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont10loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont10loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont10loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont10loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont10loopBody1cont6(int loopDepth) 
	{
															#line 799 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BUGGIFY_WITH_PROB(0.01))
															#line 5578 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 800 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_11 = delay(deterministicRandom()->random01());
															#line 800 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont10loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5584 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1cont10loopBody1Catch1(__when_expr_11.getError(), loopDepth); else return a_body1cont10loopBody1cont6when1(__when_expr_11.get(), loopDepth); };
			static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 12;
															#line 800 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< WriteDeltaFileActor, 11, Void >*>(static_cast<WriteDeltaFileActor*>(this)));
															#line 5589 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont10loopBody1cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10loopBody1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont10loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont10loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont10loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont10loopBody1cont9(int loopDepth) 
	{
															#line 803 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 5690 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 804 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent(SevDebug, "DeltaFileWritten") .detail("FileName", fname) .detail("Encrypted", cipherKeysCtx.present()) .detail("Compressed", compressFilter.present());
															#line 5694 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 810 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		double duration = g_network->timer() - startTimer;
															#line 811 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.deltaUpdateSample.addMeasurement(duration);
															#line 814 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WriteDeltaFileActor*>(this)->SAV<BlobFileIndex>::futures) { (void)(BlobFileIndex( currentDeltaVersion, fname, 0, serializedSize, serializedSize, logicalSize, cipherKeysMeta)); this->~WriteDeltaFileActorState(); static_cast<WriteDeltaFileActor*>(this)->destroy(); return 0; }
															#line 5702 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WriteDeltaFileActor*>(this)->SAV< BlobFileIndex >::value()) BlobFileIndex(BlobFileIndex( currentDeltaVersion, fname, 0, serializedSize, serializedSize, logicalSize, cipherKeysMeta));
		this->~WriteDeltaFileActorState();
		static_cast<WriteDeltaFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont10loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont10loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont10loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont10loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont10loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont10loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<WriteDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteDeltaFileActor*>(this)->ActorCallback< WriteDeltaFileActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont10loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< WriteDeltaFileActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont10loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< WriteDeltaFileActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont6Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont6Catch1(error, 0);
		} catch (...) {
			a_body1cont6Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), 12);

	}
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobConnectionProvider> bstore;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	KeyRange keyRange;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID granuleID;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t epoch;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t seqno;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Standalone<GranuleDeltas> deltasToWrite;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version currentDeltaVersion;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Future<BlobFileIndex> previousDeltaFileFuture;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Future<Void> waitCommitted;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Optional<std::pair<KeyRange, UID>> oldGranuleComplete;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Future<Void> startDeltaFileWrite;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t deltaFileBudget;
															#line 696 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	FlowLock::Releaser holdingLock;
															#line 700 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	std::string fileName;
															#line 702 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Optional<BlobGranuleCipherKeysCtx> cipherKeysCtx;
															#line 703 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Optional<BlobGranuleCipherKeysMeta> cipherKeysMeta;
															#line 704 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Arena arena;
															#line 712 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Optional<CompressionFilter> compressFilter;
															#line 714 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Value serialized;
															#line 720 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	size_t logicalSize;
															#line 721 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	size_t serializedSize;
															#line 728 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BackupContainerFileSystem> writeBStore;
															#line 729 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	std::string fname;
															#line 732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	double startTimer;
															#line 749 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int numIterations;
															#line 758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 5938 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via writeDeltaFile()
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WriteDeltaFileActor final : public Actor<BlobFileIndex>, public ActorCallback< WriteDeltaFileActor, 0, Void >, public ActorCallback< WriteDeltaFileActor, 1, Void >, public ActorCallback< WriteDeltaFileActor, 2, BlobGranuleCipherKeysCtx >, public ActorCallback< WriteDeltaFileActor, 3, Void >, public ActorCallback< WriteDeltaFileActor, 4, Void >, public ActorCallback< WriteDeltaFileActor, 5, BlobFileIndex >, public ActorCallback< WriteDeltaFileActor, 6, Void >, public ActorCallback< WriteDeltaFileActor, 7, Void >, public ActorCallback< WriteDeltaFileActor, 8, Void >, public ActorCallback< WriteDeltaFileActor, 9, Void >, public ActorCallback< WriteDeltaFileActor, 10, Void >, public ActorCallback< WriteDeltaFileActor, 11, Void >, public ActorCallback< WriteDeltaFileActor, 12, Void >, public FastAllocated<WriteDeltaFileActor>, public WriteDeltaFileActorState<WriteDeltaFileActor> {
															#line 5943 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<WriteDeltaFileActor>::operator new;
	using FastAllocated<WriteDeltaFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9681548082620715008UL, 15903440664577159424UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobFileIndex>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteDeltaFileActor, 0, Void >;
friend struct ActorCallback< WriteDeltaFileActor, 1, Void >;
friend struct ActorCallback< WriteDeltaFileActor, 2, BlobGranuleCipherKeysCtx >;
friend struct ActorCallback< WriteDeltaFileActor, 3, Void >;
friend struct ActorCallback< WriteDeltaFileActor, 4, Void >;
friend struct ActorCallback< WriteDeltaFileActor, 5, BlobFileIndex >;
friend struct ActorCallback< WriteDeltaFileActor, 6, Void >;
friend struct ActorCallback< WriteDeltaFileActor, 7, Void >;
friend struct ActorCallback< WriteDeltaFileActor, 8, Void >;
friend struct ActorCallback< WriteDeltaFileActor, 9, Void >;
friend struct ActorCallback< WriteDeltaFileActor, 10, Void >;
friend struct ActorCallback< WriteDeltaFileActor, 11, Void >;
friend struct ActorCallback< WriteDeltaFileActor, 12, Void >;
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WriteDeltaFileActor(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,KeyRange const& keyRange,UID const& granuleID,int64_t const& epoch,int64_t const& seqno,Standalone<GranuleDeltas> const& deltasToWrite,Version const& currentDeltaVersion,Future<BlobFileIndex> const& previousDeltaFileFuture,Future<Void> const& waitCommitted,Optional<std::pair<KeyRange, UID>> const& oldGranuleComplete,Future<Void> const& startDeltaFileWrite,int64_t const& deltaFileBudget) 
															#line 5972 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<BlobFileIndex>(),
		   WriteDeltaFileActorState<WriteDeltaFileActor>(bwData, bstore, keyRange, granuleID, epoch, seqno, deltasToWrite, currentDeltaVersion, previousDeltaFileFuture, waitCommitted, oldGranuleComplete, startDeltaFileWrite, deltaFileBudget),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeDeltaFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(771967562555551488UL, 5204659431395714560UL);
		ActorExecutionContextHelper __helper(static_cast<WriteDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeDeltaFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeDeltaFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 2, BlobGranuleCipherKeysCtx >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 5, BlobFileIndex >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 10, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 11, Void >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< WriteDeltaFileActor, 12, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<BlobFileIndex> writeDeltaFile( Reference<BlobWorkerData> const& bwData, Reference<BlobConnectionProvider> const& bstore, KeyRange const& keyRange, UID const& granuleID, int64_t const& epoch, int64_t const& seqno, Standalone<GranuleDeltas> const& deltasToWrite, Version const& currentDeltaVersion, Future<BlobFileIndex> const& previousDeltaFileFuture, Future<Void> const& waitCommitted, Optional<std::pair<KeyRange, UID>> const& oldGranuleComplete, Future<Void> const& startDeltaFileWrite, int64_t const& deltaFileBudget ) {
															#line 682 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<BlobFileIndex>(new WriteDeltaFileActor(bwData, bstore, keyRange, granuleID, epoch, seqno, deltasToWrite, currentDeltaVersion, previousDeltaFileFuture, waitCommitted, oldGranuleComplete, startDeltaFileWrite, deltaFileBudget));
															#line 6017 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 837 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

															#line 6022 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via writeEmptyDeltaFile()
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class WriteEmptyDeltaFileActor>
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WriteEmptyDeltaFileActorState {
															#line 6029 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WriteEmptyDeltaFileActorState(Reference<BlobWorkerData> const& bwData,KeyRange const& keyRange,UID const& granuleID,int64_t const& epoch,int64_t const& seqno,Version const& previousVersion,Version const& currentDeltaVersion,Future<BlobFileIndex> const& previousDeltaFileFuture,Future<Void> const& waitCommitted,Optional<std::pair<KeyRange, UID>> const& oldGranuleComplete) 
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   keyRange(keyRange),
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleID(granuleID),
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   epoch(epoch),
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   seqno(seqno),
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   previousVersion(previousVersion),
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   currentDeltaVersion(currentDeltaVersion),
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   previousDeltaFileFuture(previousDeltaFileFuture),
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   waitCommitted(waitCommitted),
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   oldGranuleComplete(oldGranuleComplete)
															#line 6054 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this));

	}
	~WriteEmptyDeltaFileActorState() 
	{
		fdb_probe_actor_destroy("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 848 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			ASSERT(previousVersion < currentDeltaVersion);
															#line 849 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0, TaskPriority::BlobWorkerUpdateStorage);
															#line 849 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6073 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 1;
															#line 849 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 0, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6078 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteEmptyDeltaFileActorState();
		static_cast<WriteEmptyDeltaFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 852 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = waitCommitted;
															#line 852 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6103 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 2;
															#line 852 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 1, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6108 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 852 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = waitCommitted;
															#line 852 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6119 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 2;
															#line 852 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 1, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6124 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEmptyDeltaFileActor*>(this)->ActorCallback< WriteEmptyDeltaFileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteEmptyDeltaFileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 853 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<BlobFileIndex> __when_expr_2 = previousDeltaFileFuture;
															#line 853 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6210 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 3;
															#line 853 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 2, BlobFileIndex >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6215 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 853 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<BlobFileIndex> __when_expr_2 = previousDeltaFileFuture;
															#line 853 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6226 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 3;
															#line 853 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 2, BlobFileIndex >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6231 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEmptyDeltaFileActor*>(this)->ActorCallback< WriteEmptyDeltaFileActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteEmptyDeltaFileActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(BlobFileIndex const& prev,int loopDepth) 
	{
															#line 854 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(0, TaskPriority::BlobWorkerUpdateFDB);
															#line 854 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6317 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 4;
															#line 854 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 3, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6322 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(BlobFileIndex && prev,int loopDepth) 
	{
															#line 854 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(0, TaskPriority::BlobWorkerUpdateFDB);
															#line 854 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6333 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 4;
															#line 854 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 3, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6338 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(BlobFileIndex const& prev,int loopDepth) 
	{
		loopDepth = a_body1cont3(prev, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(BlobFileIndex && prev,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(prev), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEmptyDeltaFileActor*>(this)->ActorCallback< WriteEmptyDeltaFileActor, 2, BlobFileIndex >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 2, BlobFileIndex >*,BlobFileIndex const& value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 2, BlobFileIndex >*,BlobFileIndex && value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WriteEmptyDeltaFileActor, 2, BlobFileIndex >*,Error err) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 857 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		oldDFKey = blobGranuleFileKeyFor(granuleID, previousVersion, 'D');
															#line 858 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		newDFKey = blobGranuleFileKeyFor(granuleID, currentDeltaVersion, 'D');
															#line 859 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>(bwData->db);
															#line 860 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		dfValue = Optional<Value>();
															#line 861 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 6430 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 857 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		oldDFKey = blobGranuleFileKeyFor(granuleID, previousVersion, 'D');
															#line 858 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		newDFKey = blobGranuleFileKeyFor(granuleID, currentDeltaVersion, 'D');
															#line 859 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>(bwData->db);
															#line 860 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		dfValue = Optional<Value>();
															#line 861 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 6447 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEmptyDeltaFileActor*>(this)->ActorCallback< WriteEmptyDeltaFileActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WriteEmptyDeltaFileActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1(int loopDepth) 
	{
															#line 862 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 863 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 864 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 6542 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		try {
															#line 866 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_4 = readAndCheckGranuleLock(tr, keyRange, epoch, seqno);
															#line 866 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont4loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6548 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont4loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont4loopBody1when1(__when_expr_4.get(), loopDepth); };
			static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 5;
															#line 866 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 4, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6553 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont4loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont4loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1cont4loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 920 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_10 = tr->onError(e);
															#line 920 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6577 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch1(__when_expr_10.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1Catch1when1(__when_expr_10.get(), loopDepth); };
			static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 11;
															#line 920 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 10, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6582 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 870 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!dfValue.present())
															#line 6597 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 872 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_5 = store(dfValue, tr->get(oldDFKey));
															#line 872 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont4loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6603 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont4loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont4loopBody1cont2when1(__when_expr_5.get(), loopDepth); };
			static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 6;
															#line 872 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 5, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6608 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 883 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			tr->addReadConflictRange(singleKeyRange(oldDFKey));
															#line 6615 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1cont4loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 870 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!dfValue.present())
															#line 6625 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 872 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_5 = store(dfValue, tr->get(oldDFKey));
															#line 872 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont4loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6631 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont4loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont4loopBody1cont2when1(__when_expr_5.get(), loopDepth); };
			static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 6;
															#line 872 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 5, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6636 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 883 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			tr->addReadConflictRange(singleKeyRange(oldDFKey));
															#line 6643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1cont4loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEmptyDeltaFileActor*>(this)->ActorCallback< WriteEmptyDeltaFileActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< WriteEmptyDeltaFileActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont4loopBody1cont3(int loopDepth) 
	{
															#line 886 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->clear(oldDFKey);
															#line 887 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->set(newDFKey, dfValue.get());
															#line 889 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (oldGranuleComplete.present())
															#line 6732 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 890 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_6 = updateGranuleSplitState(&tr->getTransaction(), oldGranuleComplete.get().first, oldGranuleComplete.get().second, granuleID, BlobGranuleSplitState::Done);
															#line 890 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont4loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6738 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont4loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont4loopBody1cont3when1(__when_expr_6.get(), loopDepth); };
			static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 7;
															#line 890 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 6, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6743 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4loopBody1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 873 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!dfValue.present())
															#line 6757 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 874 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent("MissingFileEmptyWrite", bwData->id) .detail("Granule", keyRange) .detail("PrevVersion", previousVersion) .detail("CurrentVersion", currentDeltaVersion) .detail("PrevKey", oldDFKey) .detail("NewKey", newDFKey);
															#line 6761 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 881 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(dfValue.present());
															#line 6765 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 873 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!dfValue.present())
															#line 6774 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 874 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent("MissingFileEmptyWrite", bwData->id) .detail("Granule", keyRange) .detail("PrevVersion", previousVersion) .detail("CurrentVersion", currentDeltaVersion) .detail("PrevKey", oldDFKey) .detail("NewKey", newDFKey);
															#line 6778 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 881 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(dfValue.present());
															#line 6782 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEmptyDeltaFileActor*>(this)->ActorCallback< WriteEmptyDeltaFileActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont4loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont4loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< WriteEmptyDeltaFileActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont4loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont4loopBody1cont8(int loopDepth) 
	{
															#line 897 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_7 = tr->commit();
															#line 897 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont4loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6868 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont4loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont4loopBody1cont8when1(__when_expr_7.get(), loopDepth); };
		static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 8;
															#line 897 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 7, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6873 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEmptyDeltaFileActor*>(this)->ActorCallback< WriteEmptyDeltaFileActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont4loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont4loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< WriteEmptyDeltaFileActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont4loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont4loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 898 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->addGRVHistory(tr->getReadVersion().get());
															#line 899 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 6971 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 900 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print( "Granule {0} [{1} - {2}) empty delta file bumped version last delta file from {3} -> {4}, cv={5}\n", granuleID.toString(), keyRange.begin.printable(), keyRange.end.printable(), previousVersion, currentDeltaVersion, tr->getCommittedVersion());
															#line 6975 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 910 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BUGGIFY && bwData->maybeInjectTargetedRestart())
															#line 6979 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 911 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_8 = Never();
															#line 911 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont4loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6985 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont4loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont4loopBody1cont10when1(__when_expr_8.get(), loopDepth); };
			static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 9;
															#line 911 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 8, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 6990 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4loopBody1cont10cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 898 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->addGRVHistory(tr->getReadVersion().get());
															#line 899 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 7006 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 900 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print( "Granule {0} [{1} - {2}) empty delta file bumped version last delta file from {3} -> {4}, cv={5}\n", granuleID.toString(), keyRange.begin.printable(), keyRange.end.printable(), previousVersion, currentDeltaVersion, tr->getCommittedVersion());
															#line 7010 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 910 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BUGGIFY && bwData->maybeInjectTargetedRestart())
															#line 7014 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 911 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_8 = Never();
															#line 911 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont4loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7020 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont4loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont4loopBody1cont10when1(__when_expr_8.get(), loopDepth); };
			static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 9;
															#line 911 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 8, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 7025 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4loopBody1cont10cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEmptyDeltaFileActor*>(this)->ActorCallback< WriteEmptyDeltaFileActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont4loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont4loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< WriteEmptyDeltaFileActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont4loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont4loopBody1cont10cont1(int loopDepth) 
	{
															#line 914 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BUGGIFY_WITH_PROB(0.01))
															#line 7114 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 915 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_9 = delay(deterministicRandom()->random01());
															#line 915 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state < 0) return a_body1cont4loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7120 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont4loopBody1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont4loopBody1cont10cont1when1(__when_expr_9.get(), loopDepth); };
			static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 10;
															#line 915 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< WriteEmptyDeltaFileActor, 9, Void >*>(static_cast<WriteEmptyDeltaFileActor*>(this)));
															#line 7125 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4loopBody1cont10cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont10cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont10cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont10cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont10cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont10cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont10cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEmptyDeltaFileActor*>(this)->ActorCallback< WriteEmptyDeltaFileActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont4loopBody1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont4loopBody1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< WriteEmptyDeltaFileActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont4loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont4loopBody1cont10cont4(int loopDepth) 
	{
															#line 918 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WriteEmptyDeltaFileActor*>(this)->SAV<BlobFileIndex>::futures) { (void)(BlobFileIndex(currentDeltaVersion, "", 0, 0, 0, 0, {})); this->~WriteEmptyDeltaFileActorState(); static_cast<WriteEmptyDeltaFileActor*>(this)->destroy(); return 0; }
															#line 7226 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WriteEmptyDeltaFileActor*>(this)->SAV< BlobFileIndex >::value()) BlobFileIndex(BlobFileIndex(currentDeltaVersion, "", 0, 0, 0, 0, {}));
		this->~WriteEmptyDeltaFileActorState();
		static_cast<WriteEmptyDeltaFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4loopBody1cont10cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont10cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont10cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont10cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont10cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont10cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont10cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont10cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEmptyDeltaFileActor*>(this)->ActorCallback< WriteEmptyDeltaFileActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont4loopBody1cont10cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont4loopBody1cont10cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< WriteEmptyDeltaFileActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont4loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont4loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont4loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont4loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state > 0) static_cast<WriteEmptyDeltaFileActor*>(this)->actor_wait_state = 0;
		static_cast<WriteEmptyDeltaFileActor*>(this)->ActorCallback< WriteEmptyDeltaFileActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont4loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< WriteEmptyDeltaFileActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont4loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< WriteEmptyDeltaFileActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), 10);

	}
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	KeyRange keyRange;
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID granuleID;
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t epoch;
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t seqno;
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version previousVersion;
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version currentDeltaVersion;
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Future<BlobFileIndex> previousDeltaFileFuture;
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Future<Void> waitCommitted;
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Optional<std::pair<KeyRange, UID>> oldGranuleComplete;
															#line 857 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Key oldDFKey;
															#line 858 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Key newDFKey;
															#line 859 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 860 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Optional<Value> dfValue;
															#line 7436 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via writeEmptyDeltaFile()
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WriteEmptyDeltaFileActor final : public Actor<BlobFileIndex>, public ActorCallback< WriteEmptyDeltaFileActor, 0, Void >, public ActorCallback< WriteEmptyDeltaFileActor, 1, Void >, public ActorCallback< WriteEmptyDeltaFileActor, 2, BlobFileIndex >, public ActorCallback< WriteEmptyDeltaFileActor, 3, Void >, public ActorCallback< WriteEmptyDeltaFileActor, 4, Void >, public ActorCallback< WriteEmptyDeltaFileActor, 5, Void >, public ActorCallback< WriteEmptyDeltaFileActor, 6, Void >, public ActorCallback< WriteEmptyDeltaFileActor, 7, Void >, public ActorCallback< WriteEmptyDeltaFileActor, 8, Void >, public ActorCallback< WriteEmptyDeltaFileActor, 9, Void >, public ActorCallback< WriteEmptyDeltaFileActor, 10, Void >, public FastAllocated<WriteEmptyDeltaFileActor>, public WriteEmptyDeltaFileActorState<WriteEmptyDeltaFileActor> {
															#line 7441 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<WriteEmptyDeltaFileActor>::operator new;
	using FastAllocated<WriteEmptyDeltaFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13894014740460605184UL, 7624915815085214208UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobFileIndex>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteEmptyDeltaFileActor, 0, Void >;
friend struct ActorCallback< WriteEmptyDeltaFileActor, 1, Void >;
friend struct ActorCallback< WriteEmptyDeltaFileActor, 2, BlobFileIndex >;
friend struct ActorCallback< WriteEmptyDeltaFileActor, 3, Void >;
friend struct ActorCallback< WriteEmptyDeltaFileActor, 4, Void >;
friend struct ActorCallback< WriteEmptyDeltaFileActor, 5, Void >;
friend struct ActorCallback< WriteEmptyDeltaFileActor, 6, Void >;
friend struct ActorCallback< WriteEmptyDeltaFileActor, 7, Void >;
friend struct ActorCallback< WriteEmptyDeltaFileActor, 8, Void >;
friend struct ActorCallback< WriteEmptyDeltaFileActor, 9, Void >;
friend struct ActorCallback< WriteEmptyDeltaFileActor, 10, Void >;
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WriteEmptyDeltaFileActor(Reference<BlobWorkerData> const& bwData,KeyRange const& keyRange,UID const& granuleID,int64_t const& epoch,int64_t const& seqno,Version const& previousVersion,Version const& currentDeltaVersion,Future<BlobFileIndex> const& previousDeltaFileFuture,Future<Void> const& waitCommitted,Optional<std::pair<KeyRange, UID>> const& oldGranuleComplete) 
															#line 7468 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<BlobFileIndex>(),
		   WriteEmptyDeltaFileActorState<WriteEmptyDeltaFileActor>(bwData, keyRange, granuleID, epoch, seqno, previousVersion, currentDeltaVersion, previousDeltaFileFuture, waitCommitted, oldGranuleComplete),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11679681201696667904UL, 18392139448050962944UL);
		ActorExecutionContextHelper __helper(static_cast<WriteEmptyDeltaFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeEmptyDeltaFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeEmptyDeltaFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteEmptyDeltaFileActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteEmptyDeltaFileActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WriteEmptyDeltaFileActor, 2, BlobFileIndex >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WriteEmptyDeltaFileActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< WriteEmptyDeltaFileActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< WriteEmptyDeltaFileActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< WriteEmptyDeltaFileActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< WriteEmptyDeltaFileActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< WriteEmptyDeltaFileActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< WriteEmptyDeltaFileActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< WriteEmptyDeltaFileActor, 10, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<BlobFileIndex> writeEmptyDeltaFile( Reference<BlobWorkerData> const& bwData, KeyRange const& keyRange, UID const& granuleID, int64_t const& epoch, int64_t const& seqno, Version const& previousVersion, Version const& currentDeltaVersion, Future<BlobFileIndex> const& previousDeltaFileFuture, Future<Void> const& waitCommitted, Optional<std::pair<KeyRange, UID>> const& oldGranuleComplete ) {
															#line 838 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<BlobFileIndex>(new WriteEmptyDeltaFileActor(bwData, keyRange, granuleID, epoch, seqno, previousVersion, currentDeltaVersion, previousDeltaFileFuture, waitCommitted, oldGranuleComplete));
															#line 7511 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 924 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

															#line 7516 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
[[nodiscard]] Future<Void> reevaluateInitialSplit( Reference<BlobWorkerData> const& bwData, UID const& granuleID, KeyRange const& keyRange, int64_t const& epoch, int64_t const& seqno, Key const& proposedSplitKey );

#line 931 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

															#line 7521 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via writeSnapshot()
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class WriteSnapshotActor>
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WriteSnapshotActorState {
															#line 7528 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WriteSnapshotActorState(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,KeyRange const& keyRange,UID const& granuleID,int64_t const& epoch,int64_t const& seqno,Version const& version,PromiseStream<RangeResult> const& rows,bool const& initialSnapshot) 
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   bstore(bstore),
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   keyRange(keyRange),
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleID(granuleID),
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   epoch(epoch),
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   seqno(seqno),
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   version(version),
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   rows(rows),
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   initialSnapshot(initialSnapshot),
															#line 941 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   fileName(randomBGFilename(bwData->id, granuleID, version, ".snapshot")),
															#line 942 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   snapshot(),
															#line 943 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   bytesRead(0),
															#line 944 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   canStopEarly((SERVER_KNOBS->BG_KEY_TUPLE_TRUNCATE_OFFSET == 0 || SERVER_KNOBS->BG_ENABLE_SPLIT_TRUNCATED)),
															#line 946 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   injectTooBig(initialSnapshot && g_network->isSimulated() && BUGGIFY_WITH_PROB(0.1))
															#line 7561 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("writeSnapshot", reinterpret_cast<unsigned long>(this));

	}
	~WriteSnapshotActorState() 
	{
		fdb_probe_actor_destroy("writeSnapshot", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 948 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0, TaskPriority::BlobWorkerUpdateStorage);
															#line 948 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7578 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 1;
															#line 948 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 0, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 7583 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteSnapshotActorState();
		static_cast<WriteSnapshotActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 950 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 7606 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 950 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 7615 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 987 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_4 = delay(0, TaskPriority::BlobWorkerUpdateStorage);
															#line 987 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7701 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont2when1(__when_expr_4.get(), loopDepth); };
		static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 5;
															#line 987 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 4, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 7706 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 952 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (initialSnapshot && snapshot.size() > 3 && canStopEarly && (injectTooBig || bytesRead >= 3 * SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES))
															#line 7723 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 955 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				return a_body1cont1loopBody1Catch1(transaction_too_old(), loopDepth);
															#line 7727 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 957 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			auto __when_expr_1 = rows.getFuture();
															#line 7731 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			static_assert(std::is_same<decltype(__when_expr_1), FutureStream<RangeResult>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<RangeResult>>::value, "invalid type");
															#line 957 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7735 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.pop(), loopDepth); };
			static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 2;
															#line 957 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< WriteSnapshotActor, 1, RangeResult >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 7740 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 963 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (e.code() == error_code_end_of_stream)
															#line 7775 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
				return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 967 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (initialSnapshot && snapshot.size() > 3 && e.code() == error_code_transaction_too_old && (injectTooBig || bytesRead >= 2 * SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES))
															#line 7781 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 970 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (BW_DEBUG)
															#line 7785 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				{
															#line 971 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					fmt::print("Granule [{0} - {1}) re-evaluating snapshot after {2} bytes ({3} limit) {4}\n", keyRange.begin.printable(), keyRange.end.printable(), bytesRead, SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES, injectTooBig ? "(injected)" : "");
															#line 7789 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				}
															#line 978 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<Void> __when_expr_3 = reevaluateInitialSplit( bwData, granuleID, keyRange, epoch, seqno, snapshot[snapshot.size() / 2].key);
															#line 978 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7795 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
				static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 4;
															#line 978 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 3, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 7800 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 982 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 7807 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(RangeResult const& res,int loopDepth) 
	{
															#line 958 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		snapshot.arena().dependsOn(res.arena());
															#line 959 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		snapshot.append(snapshot.arena(), res.begin(), res.size());
															#line 960 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bytesRead += res.expectedSize();
															#line 961 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_2 = yield(TaskPriority::BlobWorkerUpdateStorage);
															#line 961 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7830 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 3;
															#line 961 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 2, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 7835 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(RangeResult && res,int loopDepth) 
	{
															#line 958 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		snapshot.arena().dependsOn(res.arena());
															#line 959 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		snapshot.append(snapshot.arena(), res.begin(), res.size());
															#line 960 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bytesRead += res.expectedSize();
															#line 961 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_2 = yield(TaskPriority::BlobWorkerUpdateStorage);
															#line 961 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7852 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 3;
															#line 961 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 2, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 7857 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(RangeResult const& res,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(res, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(RangeResult && res,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorSingleCallback< WriteSnapshotActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorSingleCallback< WriteSnapshotActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< WriteSnapshotActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< WriteSnapshotActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont3(Void const& _,int loopDepth) 
	{
															#line 980 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(false);
															#line 8047 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont3(Void && _,int loopDepth) 
	{
															#line 980 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(false);
															#line 8056 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 989 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 8140 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 990 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Granule [{0} - {1}) read {2} snapshot rows ({3} bytes)\n", keyRange.begin.printable(), keyRange.end.printable(), snapshot.size(), bytesRead);
															#line 8144 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 997 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (g_network->isSimulated())
															#line 8148 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 998 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (snapshot.size() > 0)
															#line 8152 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 999 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				ASSERT(keyRange.begin <= snapshot[0].key);
															#line 1000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				ASSERT(keyRange.end > snapshot[snapshot.size() - 1].key);
															#line 8158 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1002 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			for(int i = 0;i < snapshot.size() - 1;i++) {
															#line 1003 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (snapshot[i].key >= snapshot[i + 1].key)
															#line 8164 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				{
															#line 1004 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					fmt::print("SORT ORDER VIOLATION IN SNAPSHOT FILE: {0}, {1}\n", snapshot[i].key.printable(), snapshot[i + 1].key.printable());
															#line 8168 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				}
															#line 1008 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				ASSERT(snapshot[i].key < snapshot[i + 1].key);
															#line 8172 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
		}
															#line 1012 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx = Optional<BlobGranuleCipherKeysCtx>();
															#line 1013 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysMeta = Optional<BlobGranuleCipherKeysMeta>();
															#line 1014 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		arena = Arena();
															#line 1016 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (bwData->encryptMode.isEncryptionEnabled())
															#line 8183 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1017 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<BlobGranuleCipherKeysCtx> __when_expr_5 = getLatestGranuleCipherKeys(bwData, keyRange, &arena);
															#line 1017 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8189 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont3when1(__when_expr_5.get(), loopDepth); };
			static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 6;
															#line 1017 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 5, BlobGranuleCipherKeysCtx >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 8194 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 989 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 8208 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 990 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Granule [{0} - {1}) read {2} snapshot rows ({3} bytes)\n", keyRange.begin.printable(), keyRange.end.printable(), snapshot.size(), bytesRead);
															#line 8212 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 997 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (g_network->isSimulated())
															#line 8216 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 998 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (snapshot.size() > 0)
															#line 8220 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 999 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				ASSERT(keyRange.begin <= snapshot[0].key);
															#line 1000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				ASSERT(keyRange.end > snapshot[snapshot.size() - 1].key);
															#line 8226 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1002 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			for(int i = 0;i < snapshot.size() - 1;i++) {
															#line 1003 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (snapshot[i].key >= snapshot[i + 1].key)
															#line 8232 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				{
															#line 1004 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					fmt::print("SORT ORDER VIOLATION IN SNAPSHOT FILE: {0}, {1}\n", snapshot[i].key.printable(), snapshot[i + 1].key.printable());
															#line 8236 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				}
															#line 1008 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				ASSERT(snapshot[i].key < snapshot[i + 1].key);
															#line 8240 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
		}
															#line 1012 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx = Optional<BlobGranuleCipherKeysCtx>();
															#line 1013 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysMeta = Optional<BlobGranuleCipherKeysMeta>();
															#line 1014 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		arena = Arena();
															#line 1016 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (bwData->encryptMode.isEncryptionEnabled())
															#line 8251 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1017 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<BlobGranuleCipherKeysCtx> __when_expr_5 = getLatestGranuleCipherKeys(bwData, keyRange, &arena);
															#line 1017 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8257 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont3when1(__when_expr_5.get(), loopDepth); };
			static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 6;
															#line 1017 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 5, BlobGranuleCipherKeysCtx >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 8262 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 1022 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		compressFilter = getBlobFileCompressFilter();
															#line 1023 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(!bwData->encryptMode.isEncryptionEnabled() || cipherKeysCtx.present());
															#line 1024 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		serialized = serializeChunkedSnapshot(StringRef(fileName), snapshot, SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_CHUNK_BYTES, compressFilter, cipherKeysCtx);
															#line 1029 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		logicalSize = snapshot.expectedSize();
															#line 1030 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		serializedSize = serialized.size();
															#line 1031 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.compressionBytesRaw += logicalSize;
															#line 1032 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.compressionBytesFinal += serializedSize;
															#line 1035 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		snapshot = Standalone<GranuleSnapshot>();
															#line 1037 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (serializedSize >= 5 * SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES)
															#line 8367 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1039 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent(SevWarn, "BGSnapshotTooBig", bwData->id) .suppressFor(60) .detail("GranuleID", granuleID) .detail("Granule", keyRange) .detail("Version", version) .detail("Size", serializedSize);
															#line 8371 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1048 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		writeBStore = Reference<BackupContainerFileSystem>();
															#line 1049 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		fname = std::string();
															#line 1050 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		std::tie(writeBStore, fname) = bstore->createForWrite(fileName);
															#line 1052 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		writeStartTimer = g_network->timer();
															#line 1054 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_6 = writeFile(writeBStore, fname, serialized);
															#line 1054 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8385 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont4when1(__when_expr_6.get(), loopDepth); };
		static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 7;
															#line 1054 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 6, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 8390 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont10(BlobGranuleCipherKeysCtx const& ciphKeysCtx,int loopDepth) 
	{
															#line 1018 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx = std::move(ciphKeysCtx);
															#line 1019 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysMeta = BlobGranuleCipherKeysCtx::toCipherKeysMeta(cipherKeysCtx.get());
															#line 8401 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(BlobGranuleCipherKeysCtx && ciphKeysCtx,int loopDepth) 
	{
															#line 1018 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysCtx = std::move(ciphKeysCtx);
															#line 1019 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		cipherKeysMeta = BlobGranuleCipherKeysCtx::toCipherKeysMeta(cipherKeysCtx.get());
															#line 8412 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(BlobGranuleCipherKeysCtx const& ciphKeysCtx,int loopDepth) 
	{
		loopDepth = a_body1cont10(ciphKeysCtx, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(BlobGranuleCipherKeysCtx && ciphKeysCtx,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(ciphKeysCtx), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 5, BlobGranuleCipherKeysCtx >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 5, BlobGranuleCipherKeysCtx >*,BlobGranuleCipherKeysCtx const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 5, BlobGranuleCipherKeysCtx >*,BlobGranuleCipherKeysCtx && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 5, BlobGranuleCipherKeysCtx >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont11(Void const& _,int loopDepth) 
	{
															#line 1056 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		++bwData->stats.s3PutReqs;
															#line 1057 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		++bwData->stats.snapshotFilesWritten;
															#line 1058 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.snapshotBytesWritten += serializedSize;
															#line 1059 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		double duration = g_network->timer() - writeStartTimer;
															#line 1060 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.snapshotBlobWriteLatencySample.addMeasurement(duration);
															#line 1063 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		serialized = Value();
															#line 1065 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_7 = delay(0, TaskPriority::BlobWorkerUpdateFDB);
															#line 1065 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8510 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont11when1(__when_expr_7.get(), loopDepth); };
		static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 8;
															#line 1065 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 7, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 8515 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont11(Void && _,int loopDepth) 
	{
															#line 1056 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		++bwData->stats.s3PutReqs;
															#line 1057 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		++bwData->stats.snapshotFilesWritten;
															#line 1058 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.snapshotBytesWritten += serializedSize;
															#line 1059 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		double duration = g_network->timer() - writeStartTimer;
															#line 1060 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.snapshotBlobWriteLatencySample.addMeasurement(duration);
															#line 1063 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		serialized = Value();
															#line 1065 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_7 = delay(0, TaskPriority::BlobWorkerUpdateFDB);
															#line 1065 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8538 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont11when1(__when_expr_7.get(), loopDepth); };
		static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 8;
															#line 1065 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 7, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 8543 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont11cont1(Void const& _,int loopDepth) 
	{
															#line 1068 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>(bwData->db);
															#line 1069 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		numIterations = 0;
															#line 8629 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		try {
															#line 1072 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			;
															#line 8633 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1cont11cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont11cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont11cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont1(Void && _,int loopDepth) 
	{
															#line 1068 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>(bwData->db);
															#line 1069 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		numIterations = 0;
															#line 8650 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		try {
															#line 1072 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			;
															#line 8654 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1cont11cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont11cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont11cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont11cont2(int loopDepth) 
	{
															#line 1115 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 8744 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1116 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Granule [{0} - {1}) committed new snapshot file {2} with {3} bytes\n\n", keyRange.begin.printable(), keyRange.end.printable(), fname, serializedSize);
															#line 8748 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1123 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BUGGIFY_WITH_PROB(0.1))
															#line 8752 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1124 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_11 = delay(deterministicRandom()->random01());
															#line 1124 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8758 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1Catch1(__when_expr_11.getError(), loopDepth); else return a_body1cont11cont2when1(__when_expr_11.get(), loopDepth); };
			static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 12;
															#line 1124 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 11, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 8763 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont11cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1102 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (numIterations > 0)
															#line 8778 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1103 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				CODE_PROBE(true, "Granule potentially leaving orphaned snapshot file");
															#line 1104 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 8784 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1106 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (BW_DEBUG)
															#line 8788 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1107 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				fmt::print("deleting snapshot file {0} after error {1}\n", fname, e.name());
															#line 8792 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1109 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			CODE_PROBE(true, "Granule deleting snapshot file after error");
															#line 1110 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			++bwData->stats.s3DeleteReqs;
															#line 1111 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			bwData->addActor.send(writeBStore->deleteFile(fname));
															#line 1112 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 8802 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont3(int loopDepth) 
	{
		loopDepth = a_body1cont11cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont11cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1(int loopDepth) 
	{
															#line 1073 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1074 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1075 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 8833 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		try {
															#line 1077 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_8 = readAndCheckGranuleLock(tr, keyRange, epoch, seqno);
															#line 1077 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1cont11cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8839 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont11cont1loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont11cont1loopBody1when1(__when_expr_8.get(), loopDepth); };
			static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 9;
															#line 1077 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 8, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 8844 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont11cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont11cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont11cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont11cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont11cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont11cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1095 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_10 = tr->onError(e);
															#line 1095 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1cont11cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8881 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont11cont1Catch1(__when_expr_10.getError(), std::max(0, loopDepth - 1)); else return a_body1cont11cont1loopBody1Catch1when1(__when_expr_10.get(), loopDepth); };
			static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 11;
															#line 1095 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 10, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 8886 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont11cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont11cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1078 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		numIterations++;
															#line 1079 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Key snapshotFileKey = blobGranuleFileKeyFor(granuleID, version, 'S');
															#line 1081 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Key snapshotFileValue = blobGranuleFileValueFor(fname, 0, serializedSize, serializedSize, logicalSize, cipherKeysMeta);
															#line 1083 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->set(snapshotFileKey, snapshotFileValue);
															#line 1085 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (initialSnapshot)
															#line 8909 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1086 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			Key historyKey = blobGranuleHistoryKeyFor(keyRange, version);
															#line 1087 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			Standalone<BlobGranuleHistoryValue> historyValue;
															#line 1088 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			historyValue.granuleID = granuleID;
															#line 1089 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			tr->set(historyKey, blobGranuleHistoryValueFor(historyValue));
															#line 8919 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1091 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_9 = tr->commit();
															#line 1091 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1cont11cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8925 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont11cont1loopBody1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont11cont1loopBody1cont2when1(__when_expr_9.get(), loopDepth); };
		static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 10;
															#line 1091 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 9, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 8930 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1078 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		numIterations++;
															#line 1079 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Key snapshotFileKey = blobGranuleFileKeyFor(granuleID, version, 'S');
															#line 1081 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Key snapshotFileValue = blobGranuleFileValueFor(fname, 0, serializedSize, serializedSize, logicalSize, cipherKeysMeta);
															#line 1083 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->set(snapshotFileKey, snapshotFileValue);
															#line 1085 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (initialSnapshot)
															#line 8947 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1086 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			Key historyKey = blobGranuleHistoryKeyFor(keyRange, version);
															#line 1087 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			Standalone<BlobGranuleHistoryValue> historyValue;
															#line 1088 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			historyValue.granuleID = granuleID;
															#line 1089 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			tr->set(historyKey, blobGranuleHistoryValueFor(historyValue));
															#line 8957 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1091 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_9 = tr->commit();
															#line 1091 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1cont11cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8963 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont11cont1loopBody1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont11cont1loopBody1cont2when1(__when_expr_9.get(), loopDepth); };
		static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 10;
															#line 1091 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 9, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 8968 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont11cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont11cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont11cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont11cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont11cont1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1092 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->addGRVHistory(tr->getReadVersion().get());
															#line 9052 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		return a_body1cont11cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1092 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->addGRVHistory(tr->getReadVersion().get());
															#line 9061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		return a_body1cont11cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont11cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont11cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont11cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont11cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont11cont1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont11cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont11cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont11cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont11cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont11cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont5(int loopDepth) 
	{
															#line 1127 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 9245 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1128 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			TraceEvent(SevDebug, "SnapshotFileWritten") .detail("FileName", fileName) .detail("Encrypted", cipherKeysCtx.present()) .detail("Compressed", compressFilter.present());
															#line 9249 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1134 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BUGGIFY && bwData->maybeInjectTargetedRestart())
															#line 9253 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1135 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_12 = Never();
															#line 1135 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9259 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1Catch1(__when_expr_12.getError(), loopDepth); else return a_body1cont11cont5when1(__when_expr_12.get(), loopDepth); };
			static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 13;
															#line 1135 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< WriteSnapshotActor, 12, Void >*>(static_cast<WriteSnapshotActor*>(this)));
															#line 9264 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont11cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont11cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont11cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont11cont8(int loopDepth) 
	{
															#line 1139 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WriteSnapshotActor*>(this)->SAV<BlobFileIndex>::futures) { (void)(BlobFileIndex(version, fname, 0, serializedSize, serializedSize, logicalSize, cipherKeysMeta)); this->~WriteSnapshotActorState(); static_cast<WriteSnapshotActor*>(this)->destroy(); return 0; }
															#line 9365 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WriteSnapshotActor*>(this)->SAV< BlobFileIndex >::value()) BlobFileIndex(BlobFileIndex(version, fname, 0, serializedSize, serializedSize, logicalSize, cipherKeysMeta));
		this->~WriteSnapshotActorState();
		static_cast<WriteSnapshotActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<WriteSnapshotActor*>(this)->actor_wait_state > 0) static_cast<WriteSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<WriteSnapshotActor*>(this)->ActorCallback< WriteSnapshotActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont11cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< WriteSnapshotActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont11cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< WriteSnapshotActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), 12);

	}
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobConnectionProvider> bstore;
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	KeyRange keyRange;
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID granuleID;
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t epoch;
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t seqno;
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version version;
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	PromiseStream<RangeResult> rows;
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	bool initialSnapshot;
															#line 941 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	std::string fileName;
															#line 942 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Standalone<GranuleSnapshot> snapshot;
															#line 943 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t bytesRead;
															#line 944 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	bool canStopEarly;
															#line 946 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	bool injectTooBig;
															#line 1012 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Optional<BlobGranuleCipherKeysCtx> cipherKeysCtx;
															#line 1013 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Optional<BlobGranuleCipherKeysMeta> cipherKeysMeta;
															#line 1014 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Arena arena;
															#line 1022 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Optional<CompressionFilter> compressFilter;
															#line 1024 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Value serialized;
															#line 1029 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	size_t logicalSize;
															#line 1030 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	size_t serializedSize;
															#line 1048 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BackupContainerFileSystem> writeBStore;
															#line 1049 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	std::string fname;
															#line 1052 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	double writeStartTimer;
															#line 1068 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1069 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int numIterations;
															#line 9512 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via writeSnapshot()
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WriteSnapshotActor final : public Actor<BlobFileIndex>, public ActorCallback< WriteSnapshotActor, 0, Void >, public ActorSingleCallback< WriteSnapshotActor, 1, RangeResult >, public ActorCallback< WriteSnapshotActor, 2, Void >, public ActorCallback< WriteSnapshotActor, 3, Void >, public ActorCallback< WriteSnapshotActor, 4, Void >, public ActorCallback< WriteSnapshotActor, 5, BlobGranuleCipherKeysCtx >, public ActorCallback< WriteSnapshotActor, 6, Void >, public ActorCallback< WriteSnapshotActor, 7, Void >, public ActorCallback< WriteSnapshotActor, 8, Void >, public ActorCallback< WriteSnapshotActor, 9, Void >, public ActorCallback< WriteSnapshotActor, 10, Void >, public ActorCallback< WriteSnapshotActor, 11, Void >, public ActorCallback< WriteSnapshotActor, 12, Void >, public FastAllocated<WriteSnapshotActor>, public WriteSnapshotActorState<WriteSnapshotActor> {
															#line 9517 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<WriteSnapshotActor>::operator new;
	using FastAllocated<WriteSnapshotActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5689035011833172224UL, 8179065185935685632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobFileIndex>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteSnapshotActor, 0, Void >;
friend struct ActorSingleCallback< WriteSnapshotActor, 1, RangeResult >;
friend struct ActorCallback< WriteSnapshotActor, 2, Void >;
friend struct ActorCallback< WriteSnapshotActor, 3, Void >;
friend struct ActorCallback< WriteSnapshotActor, 4, Void >;
friend struct ActorCallback< WriteSnapshotActor, 5, BlobGranuleCipherKeysCtx >;
friend struct ActorCallback< WriteSnapshotActor, 6, Void >;
friend struct ActorCallback< WriteSnapshotActor, 7, Void >;
friend struct ActorCallback< WriteSnapshotActor, 8, Void >;
friend struct ActorCallback< WriteSnapshotActor, 9, Void >;
friend struct ActorCallback< WriteSnapshotActor, 10, Void >;
friend struct ActorCallback< WriteSnapshotActor, 11, Void >;
friend struct ActorCallback< WriteSnapshotActor, 12, Void >;
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WriteSnapshotActor(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,KeyRange const& keyRange,UID const& granuleID,int64_t const& epoch,int64_t const& seqno,Version const& version,PromiseStream<RangeResult> const& rows,bool const& initialSnapshot) 
															#line 9546 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<BlobFileIndex>(),
		   WriteSnapshotActorState<WriteSnapshotActor>(bwData, bstore, keyRange, granuleID, epoch, seqno, version, rows, initialSnapshot),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeSnapshot", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11404649709667446784UL, 9660011112791750656UL);
		ActorExecutionContextHelper __helper(static_cast<WriteSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeSnapshot");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeSnapshot", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteSnapshotActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorSingleCallback< WriteSnapshotActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WriteSnapshotActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WriteSnapshotActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< WriteSnapshotActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< WriteSnapshotActor, 5, BlobGranuleCipherKeysCtx >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< WriteSnapshotActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< WriteSnapshotActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< WriteSnapshotActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< WriteSnapshotActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< WriteSnapshotActor, 10, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< WriteSnapshotActor, 11, Void >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< WriteSnapshotActor, 12, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<BlobFileIndex> writeSnapshot( Reference<BlobWorkerData> const& bwData, Reference<BlobConnectionProvider> const& bstore, KeyRange const& keyRange, UID const& granuleID, int64_t const& epoch, int64_t const& seqno, Version const& version, PromiseStream<RangeResult> const& rows, bool const& initialSnapshot ) {
															#line 932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<BlobFileIndex>(new WriteSnapshotActor(bwData, bstore, keyRange, granuleID, epoch, seqno, version, rows, initialSnapshot));
															#line 9591 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 1141 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

															#line 9596 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via dumpInitialSnapshotFromFDB()
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class DumpInitialSnapshotFromFDBActor>
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class DumpInitialSnapshotFromFDBActorState {
															#line 9603 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	DumpInitialSnapshotFromFDBActorState(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,Reference<GranuleMetadata> const& metadata,UID const& granuleID,Key const& cfKey,std::deque<Future<Void>>* const& inFlightPops) 
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   bstore(bstore),
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   metadata(metadata),
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleID(granuleID),
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   cfKey(cfKey),
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   inFlightPops(inFlightPops)
															#line 9620 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this));

	}
	~DumpInitialSnapshotFromFDBActorState() 
	{
		fdb_probe_actor_destroy("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1148 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (BW_DEBUG)
															#line 9635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1149 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				fmt::print("Dumping snapshot from FDB for [{0} - {1})\n", metadata->keyRange.begin.printable(), metadata->keyRange.end.printable());
															#line 9639 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1153 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bwData->initialSnapshotLock->take();
															#line 1153 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9645 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 1;
															#line 1153 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DumpInitialSnapshotFromFDBActor, 0, Void >*>(static_cast<DumpInitialSnapshotFromFDBActor*>(this)));
															#line 9650 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DumpInitialSnapshotFromFDBActorState();
		static_cast<DumpInitialSnapshotFromFDBActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1154 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		holdingLock = FlowLock::Releaser(*bwData->initialSnapshotLock);
															#line 1156 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>(bwData->db);
															#line 1157 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		retries = 0;
															#line 1158 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		lastReadVersion = invalidVersion;
															#line 1159 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		readVersion = invalidVersion;
															#line 1161 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 9683 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1154 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		holdingLock = FlowLock::Releaser(*bwData->initialSnapshotLock);
															#line 1156 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>(bwData->db);
															#line 1157 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		retries = 0;
															#line 1158 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		lastReadVersion = invalidVersion;
															#line 1159 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		readVersion = invalidVersion;
															#line 1161 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 9702 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state > 0) static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 0;
		static_cast<DumpInitialSnapshotFromFDBActor*>(this)->ActorCallback< DumpInitialSnapshotFromFDBActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DumpInitialSnapshotFromFDBActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 1162 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1163 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::RAW_ACCESS);
															#line 1164 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1165 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 9799 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		try {
															#line 1167 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Version> __when_expr_1 = tr->getReadVersion();
															#line 1167 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9805 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 2;
															#line 1167 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DumpInitialSnapshotFromFDBActor, 1, Version >*>(static_cast<DumpInitialSnapshotFromFDBActor*>(this)));
															#line 9810 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1198 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 9832 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1199 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 9836 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1201 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (BW_DEBUG)
															#line 9840 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1202 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				fmt::print("Dumping snapshot {0} from FDB for [{1} - {2}) got error {3}\n", retries + 1, metadata->keyRange.begin.printable(), metadata->keyRange.end.printable(), e.name());
															#line 9844 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1208 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			err = e;
															#line 1209 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (e.code() == error_code_server_overloaded)
															#line 9850 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1210 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<Void> __when_expr_4 = delay(FLOW_KNOBS->PREVENT_FAST_SPIN_DELAY);
															#line 1210 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9856 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
				static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 5;
															#line 1210 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< DumpInitialSnapshotFromFDBActor, 4, Void >*>(static_cast<DumpInitialSnapshotFromFDBActor*>(this)));
															#line 9861 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1212 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<Void> __when_expr_5 = tr->onError(e);
															#line 1212 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9870 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when2(__when_expr_5.get(), loopDepth); };
				static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 6;
															#line 1212 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< DumpInitialSnapshotFromFDBActor, 5, Void >*>(static_cast<DumpInitialSnapshotFromFDBActor*>(this)));
															#line 9875 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Version const& rv,int loopDepth) 
	{
															#line 1168 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->addGRVHistory(rv);
															#line 1170 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		readVersion = rv;
															#line 1171 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(lastReadVersion <= readVersion);
															#line 1172 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		rowsStream = PromiseStream<RangeResult>();
															#line 1173 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		snapshotWriter = writeSnapshot(bwData, bstore, metadata->keyRange, granuleID, metadata->originalEpoch, metadata->originalSeqno, readVersion, rowsStream, true);
															#line 1182 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Future<Void> streamFuture = tr->getTransaction().getRangeStream(rowsStream, metadata->keyRange, GetRangeLimits(), Snapshot::True);
															#line 1184 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_2 = streamFuture && success(snapshotWriter);
															#line 1184 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9905 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 3;
															#line 1184 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DumpInitialSnapshotFromFDBActor, 2, Void >*>(static_cast<DumpInitialSnapshotFromFDBActor*>(this)));
															#line 9910 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Version && rv,int loopDepth) 
	{
															#line 1168 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->addGRVHistory(rv);
															#line 1170 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		readVersion = rv;
															#line 1171 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(lastReadVersion <= readVersion);
															#line 1172 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		rowsStream = PromiseStream<RangeResult>();
															#line 1173 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		snapshotWriter = writeSnapshot(bwData, bstore, metadata->keyRange, granuleID, metadata->originalEpoch, metadata->originalSeqno, readVersion, rowsStream, true);
															#line 1182 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Future<Void> streamFuture = tr->getTransaction().getRangeStream(rowsStream, metadata->keyRange, GetRangeLimits(), Snapshot::True);
															#line 1184 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_2 = streamFuture && success(snapshotWriter);
															#line 1184 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9933 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 3;
															#line 1184 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DumpInitialSnapshotFromFDBActor, 2, Void >*>(static_cast<DumpInitialSnapshotFromFDBActor*>(this)));
															#line 9938 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Version const& rv,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(rv, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Version && rv,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(rv), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state > 0) static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 0;
		static_cast<DumpInitialSnapshotFromFDBActor*>(this)->ActorCallback< DumpInitialSnapshotFromFDBActor, 1, Version >::remove();

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 1, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 1, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DumpInitialSnapshotFromFDBActor, 1, Version >*,Error err) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1185 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		TraceEvent(SevDebug, "BlobGranuleSnapshotFile", bwData->id) .detail("Granule", metadata->keyRange) .detail("Version", readVersion);
															#line 1188 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		DEBUG_KEY_RANGE("BlobWorkerFDBSnapshot", readVersion, metadata->keyRange, bwData->id);
															#line 1190 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BUGGIFY && bwData->maybeInjectTargetedRestart())
															#line 10026 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1191 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_3 = Never();
															#line 1191 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10032 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 4;
															#line 1191 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DumpInitialSnapshotFromFDBActor, 3, Void >*>(static_cast<DumpInitialSnapshotFromFDBActor*>(this)));
															#line 10037 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1185 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		TraceEvent(SevDebug, "BlobGranuleSnapshotFile", bwData->id) .detail("Granule", metadata->keyRange) .detail("Version", readVersion);
															#line 1188 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		DEBUG_KEY_RANGE("BlobWorkerFDBSnapshot", readVersion, metadata->keyRange, bwData->id);
															#line 1190 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BUGGIFY && bwData->maybeInjectTargetedRestart())
															#line 10055 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1191 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_3 = Never();
															#line 1191 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 4;
															#line 1191 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DumpInitialSnapshotFromFDBActor, 3, Void >*>(static_cast<DumpInitialSnapshotFromFDBActor*>(this)));
															#line 10066 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state > 0) static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 0;
		static_cast<DumpInitialSnapshotFromFDBActor*>(this)->ActorCallback< DumpInitialSnapshotFromFDBActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DumpInitialSnapshotFromFDBActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont4(int loopDepth) 
	{
															#line 1195 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		inFlightPops->push_back(bwData->db->popChangeFeedMutations(cfKey, readVersion + 1));
															#line 1196 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<DumpInitialSnapshotFromFDBActor*>(this)->SAV<BlobFileIndex>::futures) { (void)(snapshotWriter.get()); this->~DumpInitialSnapshotFromFDBActorState(); static_cast<DumpInitialSnapshotFromFDBActor*>(this)->destroy(); return 0; }
															#line 10157 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<DumpInitialSnapshotFromFDBActor*>(this)->SAV< BlobFileIndex >::value()) BlobFileIndex(snapshotWriter.get());
		this->~DumpInitialSnapshotFromFDBActorState();
		static_cast<DumpInitialSnapshotFromFDBActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state > 0) static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 0;
		static_cast<DumpInitialSnapshotFromFDBActor*>(this)->ActorCallback< DumpInitialSnapshotFromFDBActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DumpInitialSnapshotFromFDBActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1Catch1cont1(int loopDepth) 
	{
															#line 1214 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		retries++;
															#line 1215 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "Granule initial snapshot failed");
															#line 1216 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		TraceEvent(retries < 10 ? SevDebug : SevWarn, "BlobGranuleInitialSnapshotRetry", bwData->id) .error(err) .detail("Granule", metadata->keyRange) .detail("Count", retries);
															#line 1220 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		lastReadVersion = readVersion;
															#line 10262 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state > 0) static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 0;
		static_cast<DumpInitialSnapshotFromFDBActor*>(this)->ActorCallback< DumpInitialSnapshotFromFDBActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< DumpInitialSnapshotFromFDBActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1Catch1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state > 0) static_cast<DumpInitialSnapshotFromFDBActor*>(this)->actor_wait_state = 0;
		static_cast<DumpInitialSnapshotFromFDBActor*>(this)->ActorCallback< DumpInitialSnapshotFromFDBActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1Catch1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< DumpInitialSnapshotFromFDBActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1Catch1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< DumpInitialSnapshotFromFDBActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobConnectionProvider> bstore;
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<GranuleMetadata> metadata;
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID granuleID;
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Key cfKey;
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	std::deque<Future<Void>>* inFlightPops;
															#line 1154 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	FlowLock::Releaser holdingLock;
															#line 1156 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1157 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int retries;
															#line 1158 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version lastReadVersion;
															#line 1159 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version readVersion;
															#line 1172 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	PromiseStream<RangeResult> rowsStream;
															#line 1173 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Future<BlobFileIndex> snapshotWriter;
															#line 1208 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Error err;
															#line 10469 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via dumpInitialSnapshotFromFDB()
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class DumpInitialSnapshotFromFDBActor final : public Actor<BlobFileIndex>, public ActorCallback< DumpInitialSnapshotFromFDBActor, 0, Void >, public ActorCallback< DumpInitialSnapshotFromFDBActor, 1, Version >, public ActorCallback< DumpInitialSnapshotFromFDBActor, 2, Void >, public ActorCallback< DumpInitialSnapshotFromFDBActor, 3, Void >, public ActorCallback< DumpInitialSnapshotFromFDBActor, 4, Void >, public ActorCallback< DumpInitialSnapshotFromFDBActor, 5, Void >, public FastAllocated<DumpInitialSnapshotFromFDBActor>, public DumpInitialSnapshotFromFDBActorState<DumpInitialSnapshotFromFDBActor> {
															#line 10474 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<DumpInitialSnapshotFromFDBActor>::operator new;
	using FastAllocated<DumpInitialSnapshotFromFDBActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8655810934742224640UL, 13082125416521542656UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobFileIndex>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DumpInitialSnapshotFromFDBActor, 0, Void >;
friend struct ActorCallback< DumpInitialSnapshotFromFDBActor, 1, Version >;
friend struct ActorCallback< DumpInitialSnapshotFromFDBActor, 2, Void >;
friend struct ActorCallback< DumpInitialSnapshotFromFDBActor, 3, Void >;
friend struct ActorCallback< DumpInitialSnapshotFromFDBActor, 4, Void >;
friend struct ActorCallback< DumpInitialSnapshotFromFDBActor, 5, Void >;
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	DumpInitialSnapshotFromFDBActor(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,Reference<GranuleMetadata> const& metadata,UID const& granuleID,Key const& cfKey,std::deque<Future<Void>>* const& inFlightPops) 
															#line 10496 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<BlobFileIndex>(),
		   DumpInitialSnapshotFromFDBActorState<DumpInitialSnapshotFromFDBActor>(bwData, bstore, metadata, granuleID, cfKey, inFlightPops),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6032507104114808320UL, 8492076330784876288UL);
		ActorExecutionContextHelper __helper(static_cast<DumpInitialSnapshotFromFDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("dumpInitialSnapshotFromFDB");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("dumpInitialSnapshotFromFDB", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DumpInitialSnapshotFromFDBActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DumpInitialSnapshotFromFDBActor, 1, Version >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DumpInitialSnapshotFromFDBActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DumpInitialSnapshotFromFDBActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< DumpInitialSnapshotFromFDBActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< DumpInitialSnapshotFromFDBActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<BlobFileIndex> dumpInitialSnapshotFromFDB( Reference<BlobWorkerData> const& bwData, Reference<BlobConnectionProvider> const& bstore, Reference<GranuleMetadata> const& metadata, UID const& granuleID, Key const& cfKey, std::deque<Future<Void>>* const& inFlightPops ) {
															#line 1142 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<BlobFileIndex>(new DumpInitialSnapshotFromFDBActor(bwData, bstore, metadata, granuleID, cfKey, inFlightPops));
															#line 10534 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 1232 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

// files might not be the current set of files in metadata, in the case of doing the initial snapshot of a granule that
// was split.
															#line 10541 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via compactFromBlob()
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class CompactFromBlobActor>
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class CompactFromBlobActorState {
															#line 10548 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	CompactFromBlobActorState(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,Reference<GranuleMetadata> const& metadata,UID const& granuleID,std::vector<GranuleFiles> const& fileSet,Version const& version) 
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   bstore(bstore),
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   metadata(metadata),
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleID(granuleID),
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   fileSet(fileSet),
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   version(version),
															#line 1241 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   requiredBudget(0)
															#line 10567 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("compactFromBlob", reinterpret_cast<unsigned long>(this));

	}
	~CompactFromBlobActorState() 
	{
		fdb_probe_actor_destroy("compactFromBlob", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1242 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			for( auto& f : fileSet ) {
															#line 1243 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				ASSERT(!f.snapshotFiles.empty());
															#line 1244 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				ASSERT(!f.deltaFiles.empty());
															#line 1245 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				Version snapshotVersion = f.snapshotFiles.back().version;
															#line 1246 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				requiredBudget += f.snapshotFiles.back().length;
															#line 1247 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				int deltaIdx = f.deltaFiles.size() - 1;
															#line 1248 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				for(;deltaIdx >= 0 && f.deltaFiles[deltaIdx].version > snapshotVersion;) {
															#line 1249 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					requiredBudget += f.deltaFiles[deltaIdx].length;
															#line 1250 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					deltaIdx--;
															#line 10598 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				}
			}
															#line 1254 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			requiredBudget *= 2;
															#line 1256 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			requiredBudget = std::min(requiredBudget, SERVER_KNOBS->BLOB_WORKER_RESNAPSHOT_BUDGET_BYTES);
															#line 1257 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bwData->resnapshotBudget->take(TaskPriority::DefaultYield, requiredBudget);
															#line 1257 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10609 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 1;
															#line 1257 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CompactFromBlobActor, 0, Void >*>(static_cast<CompactFromBlobActor*>(this)));
															#line 10614 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CompactFromBlobActorState();
		static_cast<CompactFromBlobActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1258 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		holdingLock = FlowLock::Releaser(*bwData->resnapshotBudget, requiredBudget);
															#line 1259 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::BlobWorkerUpdateStorage);
															#line 1259 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10641 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 2;
															#line 1259 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CompactFromBlobActor, 1, Void >*>(static_cast<CompactFromBlobActor*>(this)));
															#line 10646 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1258 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		holdingLock = FlowLock::Releaser(*bwData->resnapshotBudget, requiredBudget);
															#line 1259 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::BlobWorkerUpdateStorage);
															#line 1259 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10659 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 2;
															#line 1259 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CompactFromBlobActor, 1, Void >*>(static_cast<CompactFromBlobActor*>(this)));
															#line 10664 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state > 0) static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 0;
		static_cast<CompactFromBlobActor*>(this)->ActorCallback< CompactFromBlobActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CompactFromBlobActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1260 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 10748 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1261 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Compacting snapshot from blob for [{0} - {1}) @ {2}\n", metadata->keyRange.begin.printable().c_str(), metadata->keyRange.end.printable().c_str(), version);
															#line 10752 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1267 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		filenameArena = Arena();
															#line 1268 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		chunksToRead = std::vector<Future<RangeResult>>();
															#line 1269 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		compactBytesRead = 0;
															#line 1270 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		resnapshotStartTimer = g_network->timer();
															#line 1272 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		RangeForbody1cont4Iterator0 = std::begin(fileSet);
															#line 10764 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1260 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 10773 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1261 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Compacting snapshot from blob for [{0} - {1}) @ {2}\n", metadata->keyRange.begin.printable().c_str(), metadata->keyRange.end.printable().c_str(), version);
															#line 10777 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1267 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		filenameArena = Arena();
															#line 1268 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		chunksToRead = std::vector<Future<RangeResult>>();
															#line 1269 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		compactBytesRead = 0;
															#line 1270 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		resnapshotStartTimer = g_network->timer();
															#line 1272 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		RangeForbody1cont4Iterator0 = std::begin(fileSet);
															#line 10789 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state > 0) static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 0;
		static_cast<CompactFromBlobActor*>(this)->ActorCallback< CompactFromBlobActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CompactFromBlobActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 1344 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 10873 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1345 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Re-snapshotting [{0} - {1}) @ {2} from blob\n", metadata->keyRange.begin.printable(), metadata->keyRange.end.printable(), version);
															#line 10877 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
		try {
															#line 1352 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			rowsStream = PromiseStream<RangeResult>();
															#line 1353 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			snapshotWriter = writeSnapshot(bwData, bstore, metadata->keyRange, granuleID, metadata->originalEpoch, metadata->originalSeqno, version, rowsStream, false);
															#line 1362 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			resultIdx = int();
															#line 1363 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			resultIdx = 0;
															#line 10888 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1cont5loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont5Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont5Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1(int loopDepth) 
	{
															#line 1272 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!(RangeForbody1cont4Iterator0 != std::end(fileSet)))
															#line 10910 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
			return a_body1cont4break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1272 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		auto& f = *RangeForbody1cont4Iterator0;
															#line 10916 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1273 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			chunk = BlobGranuleChunkRef();
															#line 1274 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			files = f;
															#line 1275 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			snapshotVersion = files.snapshotFiles.back().version;
															#line 1276 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			snapshotF = files.snapshotFiles.back();
															#line 1278 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (snapshotVersion >= version)
															#line 10928 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1279 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				fmt::print("Chunk snapshot version [{0} - {1}) @ {2} >= compact version {3}\n", chunk.keyRange.begin.printable().c_str(), chunk.keyRange.end.printable().c_str(), snapshotVersion, version);
															#line 10932 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1285 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			ASSERT(snapshotVersion < version);
															#line 1287 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			snapCipherKeysCtx = Optional<BlobGranuleCipherKeysCtx>();
															#line 1288 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			ASSERT(!(bwData->encryptMode.isEncryptionEnabled() && !snapshotF.cipherKeysMeta.present()));
															#line 1289 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (snapshotF.cipherKeysMeta.present())
															#line 10942 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1290 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				ASSERT(bwData->encryptMode.isEncryptionEnabled());
															#line 1291 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				CODE_PROBE(true, "fetching cipher keys for blob snapshot file");
															#line 1292 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<BlobGranuleCipherKeysCtx> __when_expr_2 = getGranuleCipherKeysFromKeysMeta(bwData->dbInfo, snapshotF.cipherKeysMeta.get(), &filenameArena);
															#line 1292 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10952 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1when1(__when_expr_2.get(), loopDepth); };
				static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 3;
															#line 1292 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CompactFromBlobActor, 2, BlobGranuleCipherKeysCtx >*>(static_cast<CompactFromBlobActor*>(this)));
															#line 10957 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont4loopBody1cont3(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1cont4break1(int loopDepth) 
	{
		try {
			return a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1(int loopDepth) 
	{
															#line 1272 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		++RangeForbody1cont4Iterator0;
															#line 10985 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3(int loopDepth) 
	{
															#line 1297 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		chunk.snapshotFile = BlobFilePointerRef(filenameArena, snapshotF.filename, snapshotF.offset, snapshotF.length, snapshotF.fullFileLength, snapshotF.version, snapCipherKeysCtx);
															#line 1305 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		compactBytesRead += snapshotF.length;
															#line 1306 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		deltaIdx = files.deltaFiles.size() - 1;
															#line 1307 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		for(;deltaIdx >= 0 && files.deltaFiles[deltaIdx].version > snapshotVersion;) {
															#line 1308 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			deltaIdx--;
															#line 11002 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1310 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		deltaIdx++;
															#line 1311 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		lastDeltaVersion = snapshotVersion;
															#line 1312 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		deltaF = BlobFileIndex();
															#line 1313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 11012 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont5(BlobGranuleCipherKeysCtx const& keysCtx,int loopDepth) 
	{
															#line 1294 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		snapCipherKeysCtx = std::move(keysCtx);
															#line 11021 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont5(BlobGranuleCipherKeysCtx && keysCtx,int loopDepth) 
	{
															#line 1294 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		snapCipherKeysCtx = std::move(keysCtx);
															#line 11030 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(BlobGranuleCipherKeysCtx const& keysCtx,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont5(keysCtx, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(BlobGranuleCipherKeysCtx && keysCtx,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont5(std::move(keysCtx), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state > 0) static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 0;
		static_cast<CompactFromBlobActor*>(this)->ActorCallback< CompactFromBlobActor, 2, BlobGranuleCipherKeysCtx >::remove();

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 2, BlobGranuleCipherKeysCtx >*,BlobGranuleCipherKeysCtx const& value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 2, BlobGranuleCipherKeysCtx >*,BlobGranuleCipherKeysCtx && value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CompactFromBlobActor, 2, BlobGranuleCipherKeysCtx >*,Error err) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4loopBody1cont7(int loopDepth) 
	{
															#line 1339 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(lastDeltaVersion >= version);
															#line 1340 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		chunk.includedVersion = version;
															#line 1341 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		chunksToRead.push_back(readBlobGranule(chunk, metadata->keyRange, 0, version, bstore, &bwData->stats));
															#line 11118 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont4loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 1313 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!(deltaIdx < files.deltaFiles.size() && lastDeltaVersion < version))
															#line 11134 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
			return a_body1cont4loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1314 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		deltaCipherKeysCtx = Optional<BlobGranuleCipherKeysCtx>();
															#line 1316 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		deltaF = files.deltaFiles[deltaIdx];
															#line 1318 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(!(bwData->encryptMode.isEncryptionEnabled() && !deltaF.cipherKeysMeta.present()));
															#line 1320 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (deltaF.cipherKeysMeta.present())
															#line 11146 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1321 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			ASSERT(bwData->encryptMode.isEncryptionEnabled());
															#line 1322 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			CODE_PROBE(true, "fetching cipher keys for delta file");
															#line 1323 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<BlobGranuleCipherKeysCtx> __when_expr_3 = getGranuleCipherKeysFromKeysMeta(bwData->dbInfo, deltaF.cipherKeysMeta.get(), &filenameArena);
															#line 1323 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 11156 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1cont4loopBody1cont3loopBody1when1(__when_expr_3.get(), loopDepth); };
			static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 4;
															#line 1323 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CompactFromBlobActor, 3, BlobGranuleCipherKeysCtx >*>(static_cast<CompactFromBlobActor*>(this)));
															#line 11161 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4loopBody1cont3loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1cont4loopBody1cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3loopBody1cont1(int loopDepth) 
	{
															#line 1328 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		chunk.deltaFiles.emplace_back_deep(filenameArena, deltaF.filename, deltaF.offset, deltaF.length, deltaF.fullFileLength, deltaF.version, deltaCipherKeysCtx);
															#line 1335 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		compactBytesRead += deltaF.length;
															#line 1336 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		lastDeltaVersion = files.deltaFiles[deltaIdx].version;
															#line 1337 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		deltaIdx++;
															#line 11194 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont4loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3loopBody1cont3(BlobGranuleCipherKeysCtx const& keysCtx,int loopDepth) 
	{
															#line 1325 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		deltaCipherKeysCtx = std::move(keysCtx);
															#line 11203 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3loopBody1cont3(BlobGranuleCipherKeysCtx && keysCtx,int loopDepth) 
	{
															#line 1325 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		deltaCipherKeysCtx = std::move(keysCtx);
															#line 11212 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3loopBody1when1(BlobGranuleCipherKeysCtx const& keysCtx,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont3loopBody1cont3(keysCtx, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont3loopBody1when1(BlobGranuleCipherKeysCtx && keysCtx,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont3loopBody1cont3(std::move(keysCtx), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state > 0) static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 0;
		static_cast<CompactFromBlobActor*>(this)->ActorCallback< CompactFromBlobActor, 3, BlobGranuleCipherKeysCtx >::remove();

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 3, BlobGranuleCipherKeysCtx >*,BlobGranuleCipherKeysCtx const& value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 3, BlobGranuleCipherKeysCtx >*,BlobGranuleCipherKeysCtx && value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< CompactFromBlobActor, 3, BlobGranuleCipherKeysCtx >*,Error err) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont5Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1377 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (BW_DEBUG)
															#line 11297 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1378 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				fmt::print("Compacting snapshot from blob for [{0} - {1}) got error {2}\n", metadata->keyRange.begin.printable(), metadata->keyRange.end.printable(), e.name());
															#line 11301 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1383 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 11305 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(int loopDepth) 
	{
															#line 1368 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.bytesReadFromS3ForCompaction += compactBytesRead;
															#line 1369 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		rowsStream.sendError(end_of_stream());
															#line 1371 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<BlobFileIndex> __when_expr_5 = snapshotWriter;
															#line 1371 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state < 0) return a_body1cont5Catch1(actor_cancelled(), loopDepth);
															#line 11325 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont5Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont9when1(__when_expr_5.get(), loopDepth); };
		static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 6;
															#line 1371 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< CompactFromBlobActor, 5, BlobFileIndex >*>(static_cast<CompactFromBlobActor*>(this)));
															#line 11330 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont5loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1(int loopDepth) 
	{
															#line 1363 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!(resultIdx < chunksToRead.size()))
															#line 11346 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
			return a_body1cont5break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1364 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<RangeResult> __when_expr_4 = chunksToRead[resultIdx];
															#line 1364 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state < 0) return a_body1cont5Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11354 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont5Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont5loopBody1when1(__when_expr_4.get(), loopDepth); };
		static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 5;
															#line 1364 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< CompactFromBlobActor, 4, RangeResult >*>(static_cast<CompactFromBlobActor*>(this)));
															#line 11359 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5break1(int loopDepth) 
	{
		try {
			return a_body1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont5Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont5Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1(RangeResult const& newGranuleChunk,int loopDepth) 
	{
															#line 1365 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		rowsStream.send(std::move(newGranuleChunk));
															#line 1363 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		resultIdx++;
															#line 11383 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont5loopHead1(0);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1(RangeResult && newGranuleChunk,int loopDepth) 
	{
															#line 1365 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		rowsStream.send(std::move(newGranuleChunk));
															#line 1363 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		resultIdx++;
															#line 11394 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont5loopHead1(0);

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(RangeResult const& newGranuleChunk,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont1(newGranuleChunk, loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(RangeResult && newGranuleChunk,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont1(std::move(newGranuleChunk), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state > 0) static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 0;
		static_cast<CompactFromBlobActor*>(this)->ActorCallback< CompactFromBlobActor, 4, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 4, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont5loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 4, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont5loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< CompactFromBlobActor, 4, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont5Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont10(BlobFileIndex const& f,int loopDepth) 
	{
															#line 1372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		DEBUG_KEY_RANGE("BlobWorkerBlobSnapshot", version, metadata->keyRange, bwData->id);
															#line 1373 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		double duration = g_network->timer() - resnapshotStartTimer;
															#line 1374 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.reSnapshotLatencySample.addMeasurement(duration);
															#line 1375 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<CompactFromBlobActor*>(this)->SAV<BlobFileIndex>::futures) { (void)(f); this->~CompactFromBlobActorState(); static_cast<CompactFromBlobActor*>(this)->destroy(); return 0; }
															#line 11484 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<CompactFromBlobActor*>(this)->SAV< BlobFileIndex >::value()) BlobFileIndex(f);
		this->~CompactFromBlobActorState();
		static_cast<CompactFromBlobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10(BlobFileIndex && f,int loopDepth) 
	{
															#line 1372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		DEBUG_KEY_RANGE("BlobWorkerBlobSnapshot", version, metadata->keyRange, bwData->id);
															#line 1373 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		double duration = g_network->timer() - resnapshotStartTimer;
															#line 1374 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->stats.reSnapshotLatencySample.addMeasurement(duration);
															#line 1375 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<CompactFromBlobActor*>(this)->SAV<BlobFileIndex>::futures) { (void)(f); this->~CompactFromBlobActorState(); static_cast<CompactFromBlobActor*>(this)->destroy(); return 0; }
															#line 11502 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<CompactFromBlobActor*>(this)->SAV< BlobFileIndex >::value()) BlobFileIndex(f);
		this->~CompactFromBlobActorState();
		static_cast<CompactFromBlobActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9when1(BlobFileIndex const& f,int loopDepth) 
	{
		loopDepth = a_body1cont10(f, loopDepth);

		return loopDepth;
	}
	int a_body1cont9when1(BlobFileIndex && f,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(f), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<CompactFromBlobActor*>(this)->actor_wait_state > 0) static_cast<CompactFromBlobActor*>(this)->actor_wait_state = 0;
		static_cast<CompactFromBlobActor*>(this)->ActorCallback< CompactFromBlobActor, 5, BlobFileIndex >::remove();

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 5, BlobFileIndex >*,BlobFileIndex const& value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< CompactFromBlobActor, 5, BlobFileIndex >*,BlobFileIndex && value) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< CompactFromBlobActor, 5, BlobFileIndex >*,Error err) 
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobConnectionProvider> bstore;
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<GranuleMetadata> metadata;
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID granuleID;
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	std::vector<GranuleFiles> fileSet;
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version version;
															#line 1241 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t requiredBudget;
															#line 1258 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	FlowLock::Releaser holdingLock;
															#line 1267 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Arena filenameArena;
															#line 1268 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	std::vector<Future<RangeResult>> chunksToRead;
															#line 1269 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t compactBytesRead;
															#line 1270 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	double resnapshotStartTimer;
															#line 1272 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	decltype(std::begin(std::declval<std::vector<GranuleFiles>>())) RangeForbody1cont4Iterator0;
															#line 1273 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	BlobGranuleChunkRef chunk;
															#line 1274 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GranuleFiles files;
															#line 1275 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version snapshotVersion;
															#line 1276 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	BlobFileIndex snapshotF;
															#line 1287 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Optional<BlobGranuleCipherKeysCtx> snapCipherKeysCtx;
															#line 1306 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int deltaIdx;
															#line 1311 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version lastDeltaVersion;
															#line 1312 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	BlobFileIndex deltaF;
															#line 1314 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Optional<BlobGranuleCipherKeysCtx> deltaCipherKeysCtx;
															#line 1352 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	PromiseStream<RangeResult> rowsStream;
															#line 1353 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Future<BlobFileIndex> snapshotWriter;
															#line 1362 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int resultIdx;
															#line 11635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via compactFromBlob()
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class CompactFromBlobActor final : public Actor<BlobFileIndex>, public ActorCallback< CompactFromBlobActor, 0, Void >, public ActorCallback< CompactFromBlobActor, 1, Void >, public ActorCallback< CompactFromBlobActor, 2, BlobGranuleCipherKeysCtx >, public ActorCallback< CompactFromBlobActor, 3, BlobGranuleCipherKeysCtx >, public ActorCallback< CompactFromBlobActor, 4, RangeResult >, public ActorCallback< CompactFromBlobActor, 5, BlobFileIndex >, public FastAllocated<CompactFromBlobActor>, public CompactFromBlobActorState<CompactFromBlobActor> {
															#line 11640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<CompactFromBlobActor>::operator new;
	using FastAllocated<CompactFromBlobActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15752811246982214656UL, 9572321061577044480UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobFileIndex>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CompactFromBlobActor, 0, Void >;
friend struct ActorCallback< CompactFromBlobActor, 1, Void >;
friend struct ActorCallback< CompactFromBlobActor, 2, BlobGranuleCipherKeysCtx >;
friend struct ActorCallback< CompactFromBlobActor, 3, BlobGranuleCipherKeysCtx >;
friend struct ActorCallback< CompactFromBlobActor, 4, RangeResult >;
friend struct ActorCallback< CompactFromBlobActor, 5, BlobFileIndex >;
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	CompactFromBlobActor(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,Reference<GranuleMetadata> const& metadata,UID const& granuleID,std::vector<GranuleFiles> const& fileSet,Version const& version) 
															#line 11662 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<BlobFileIndex>(),
		   CompactFromBlobActorState<CompactFromBlobActor>(bwData, bstore, metadata, granuleID, fileSet, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("compactFromBlob", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1053877112963097856UL, 10226069191783040768UL);
		ActorExecutionContextHelper __helper(static_cast<CompactFromBlobActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("compactFromBlob");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("compactFromBlob", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CompactFromBlobActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CompactFromBlobActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CompactFromBlobActor, 2, BlobGranuleCipherKeysCtx >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< CompactFromBlobActor, 3, BlobGranuleCipherKeysCtx >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< CompactFromBlobActor, 4, RangeResult >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< CompactFromBlobActor, 5, BlobFileIndex >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<BlobFileIndex> compactFromBlob( Reference<BlobWorkerData> const& bwData, Reference<BlobConnectionProvider> const& bstore, Reference<GranuleMetadata> const& metadata, UID const& granuleID, std::vector<GranuleFiles> const& fileSet, Version const& version ) {
															#line 1235 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<BlobFileIndex>(new CompactFromBlobActor(bwData, bstore, metadata, granuleID, fileSet, version));
															#line 11700 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 1386 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

struct CounterHolder {
	int* counter;
	bool completed;

	CounterHolder() : counter(nullptr), completed(true) {}
	CounterHolder(int* counter) : counter(counter), completed(false) { (*counter)++; }

	void complete() {
		if (!completed) {
			completed = true;
			(*counter)--;
		}
	}

	~CounterHolder() { complete(); }
};

															#line 11722 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via checkSplitAndReSnapshot()
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class CheckSplitAndReSnapshotActor>
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class CheckSplitAndReSnapshotActorState {
															#line 11729 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	CheckSplitAndReSnapshotActorState(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,Reference<GranuleMetadata> const& metadata,UID const& granuleID,int64_t const& bytesInNewDeltaFiles,Future<BlobFileIndex> const& lastDeltaBeforeSnapshot,int64_t const& versionsSinceLastSnapshot) 
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   bstore(bstore),
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   metadata(metadata),
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleID(granuleID),
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   bytesInNewDeltaFiles(bytesInNewDeltaFiles),
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   lastDeltaBeforeSnapshot(lastDeltaBeforeSnapshot),
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   versionsSinceLastSnapshot(versionsSinceLastSnapshot)
															#line 11748 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this));

	}
	~CheckSplitAndReSnapshotActorState() 
	{
		fdb_probe_actor_destroy("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1412 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<BlobFileIndex> __when_expr_0 = lastDeltaBeforeSnapshot;
															#line 1412 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11765 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 1;
															#line 1412 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 0, BlobFileIndex >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 11770 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckSplitAndReSnapshotActorState();
		static_cast<CheckSplitAndReSnapshotActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(BlobFileIndex const& lastDeltaIdx,int loopDepth) 
	{
															#line 1413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		reSnapshotVersion = lastDeltaIdx.version;
															#line 1414 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 11795 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(BlobFileIndex && lastDeltaIdx,int loopDepth) 
	{
															#line 1413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		reSnapshotVersion = lastDeltaIdx.version;
															#line 1414 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 11806 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(BlobFileIndex const& lastDeltaIdx,int loopDepth) 
	{
		loopDepth = a_body1cont1(lastDeltaIdx, loopDepth);

		return loopDepth;
	}
	int a_body1when1(BlobFileIndex && lastDeltaIdx,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(lastDeltaIdx), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state > 0) static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 0, BlobFileIndex >::remove();

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 0, BlobFileIndex >*,BlobFileIndex const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 0, BlobFileIndex >*,BlobFileIndex && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 0, BlobFileIndex >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1418 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(0, TaskPriority::BlobWorkerUpdateFDB);
															#line 1418 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11892 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 3;
															#line 1418 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 2, Void >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 11897 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 1414 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!(!bwData->statusStreamInitialized))
															#line 11913 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1415 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bwData->currentManagerStatusStream.onChange();
															#line 1415 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11921 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 2;
															#line 1415 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 1, Void >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 11926 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state > 0) static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1420 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		pendingCounter = CounterHolder(&bwData->stats.granulesPendingSplitCheck);
															#line 1422 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 12037 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1423 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Granule [{0} - {1}) checking with BM for re-snapshot after {2} bytes\n", metadata->keyRange.begin.printable(), metadata->keyRange.end.printable(), metadata->bytesInNewDeltaFiles);
															#line 12041 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1429 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		TraceEvent(SevDebug, "BlobGranuleSnapshotCheck", bwData->id) .detail("Granule", metadata->keyRange) .detail("Version", reSnapshotVersion);
															#line 1437 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		metadata->resumeSnapshot.reset();
															#line 1438 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		statusEpoch = metadata->continueEpoch;
															#line 1439 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		statusSeqno = metadata->continueSeqno;
															#line 1444 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		writeHot = versionsSinceLastSnapshot <= SERVER_KNOBS->BG_HOT_SNAPSHOT_VERSIONS;
															#line 1447 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 12055 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1420 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		pendingCounter = CounterHolder(&bwData->stats.granulesPendingSplitCheck);
															#line 1422 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 12066 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1423 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Granule [{0} - {1}) checking with BM for re-snapshot after {2} bytes\n", metadata->keyRange.begin.printable(), metadata->keyRange.end.printable(), metadata->bytesInNewDeltaFiles);
															#line 12070 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1429 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		TraceEvent(SevDebug, "BlobGranuleSnapshotCheck", bwData->id) .detail("Granule", metadata->keyRange) .detail("Version", reSnapshotVersion);
															#line 1437 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		metadata->resumeSnapshot.reset();
															#line 1438 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		statusEpoch = metadata->continueEpoch;
															#line 1439 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		statusSeqno = metadata->continueSeqno;
															#line 1444 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		writeHot = versionsSinceLastSnapshot <= SERVER_KNOBS->BG_HOT_SNAPSHOT_VERSIONS;
															#line 1447 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 12084 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state > 0) static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 1512 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		pendingCounter.complete();
															#line 1514 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 12170 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1515 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Granule [{0} - {1}) re-snapshotting after {2} bytes\n", metadata->keyRange.begin.printable(), metadata->keyRange.end.printable(), bytesInNewDeltaFiles);
															#line 12174 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1520 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		TraceEvent(SevDebug, "BlobGranuleSnapshotFile", bwData->id) .detail("Granule", metadata->keyRange) .detail("Version", metadata->durableDeltaVersion.get());
															#line 1525 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 12180 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1(int loopDepth) 
	{
															#line 1448 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 12196 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont3loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(int loopDepth) 
	{
															#line 1497 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_8 = bwData->currentManagerStatusStream.onChange();
															#line 1496 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12220 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when1(__when_expr_8.get(), loopDepth); };
															#line 1498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_9 = metadata->resumeSnapshot.getFuture();
															#line 12224 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when2(__when_expr_9.get(), loopDepth); };
															#line 1501 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_10 = delay(1.0);
															#line 12228 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch1(__when_expr_10.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when3(__when_expr_10.get(), loopDepth); };
		static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 7;
															#line 1497 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 8, Void >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 1498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 9, Void >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 1501 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 10, Void >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 12237 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1(int loopDepth) 
	{
		try {
															#line 1451 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			;
															#line 12254 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1cont3loopBody1loopBody1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont3loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont3loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1cont3loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont3loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1480 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 12289 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1481 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 2));
															#line 12293 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1483 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			CODE_PROBE(true, "Blob worker re-sending split evaluation to manager after not error/not hearing back");
															#line 1486 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (e.code() == error_code_broken_promise)
															#line 12299 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1487 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<Void> __when_expr_6 = delay(FLOW_KNOBS->PREVENT_FAST_SPIN_DELAY);
															#line 1487 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 12305 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 2)); else return a_body1cont3loopBody1loopBody1Catch1when1(__when_expr_6.get(), loopDepth); };
				static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 5;
															#line 1487 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 6, Void >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 12310 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1489 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<Void> __when_expr_7 = bwData->currentManagerStatusStream.onChange();
															#line 1489 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 12319 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 2)); else return a_body1cont3loopBody1loopBody1Catch1when2(__when_expr_7.get(), loopDepth); };
				static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 6;
															#line 1489 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 7, Void >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 12324 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1cont2(int loopDepth) 
	{
															#line 1462 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (metadata->resumeSnapshot.isSet())
															#line 12340 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
			return a_body1cont3loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1466 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->currentManagerStatusStream.get().send(GranuleStatusReply(metadata->keyRange, true, writeHot, false, statusEpoch, statusSeqno, granuleID, metadata->historyVersion, reSnapshotVersion, false, metadata->originalEpoch, metadata->originalSeqno));
															#line 12346 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		return a_body1cont3loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1loopBody1(int loopDepth) 
	{
															#line 1453 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_3 = bwData->currentManagerStatusStream.get().onReady();
															#line 1452 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state < 0) return a_body1cont3loopBody1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12364 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont3loopBody1loopBody1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1loopBody1loopBody1when1(__when_expr_3.get(), loopDepth); };
															#line 1456 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_4 = bwData->currentManagerStatusStream.onChange();
															#line 12368 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont3loopBody1loopBody1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1loopBody1loopBody1when2(__when_expr_4.get(), loopDepth); };
															#line 1457 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_5 = metadata->resumeSnapshot.getFuture();
															#line 12372 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont3loopBody1loopBody1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1loopBody1loopBody1when3(__when_expr_5.get(), loopDepth); };
		static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 4;
															#line 1453 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 3, Void >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 1456 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 4, Void >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 1457 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 5, Void >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 12381 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1cont3loopBody1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont3loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont3loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont3loopBody1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		return a_body1cont3loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		return a_body1cont3loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1loopBody1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1loopBody1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1loopBody1when3(Void const& _,int loopDepth) 
	{
		return a_body1cont3loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1loopBody1when3(Void && _,int loopDepth) 
	{
		return a_body1cont3loopBody1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state > 0) static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 3, Void >::remove();
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 4, Void >::remove();
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1loopBody1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1loopBody1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1loopBody1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1loopBody1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont3loopBody1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1Catch1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1Catch1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1loopBody1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1loopBody1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state > 0) static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3loopBody1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3loopBody1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont3loopBody1loopBody1Catch1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1Catch1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1Catch1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1loopBody1Catch1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1loopBody1Catch1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1loopBody1Catch1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state > 0) static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3loopBody1loopBody1Catch1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3loopBody1loopBody1Catch1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont3loopBody1cont2(int loopDepth) 
	{
															#line 1504 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 12804 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1505 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Granule [{0} - {1}), hasn't heard back from BM in BW {2}, re-sending status\n", metadata->keyRange.begin.printable(), metadata->keyRange.end.printable(), bwData->id.toString());
															#line 12808 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when2(Void const& _,int loopDepth) 
	{
		return a_body1cont3break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when2(Void && _,int loopDepth) 
	{
		return a_body1cont3break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state > 0) static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 8, Void >::remove();
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 9, Void >::remove();
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont3loopBody1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont3loopBody1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont3loopBody1cont1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont3loopBody1cont1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 1528 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		std::vector<GranuleFiles> toSnapshot;
															#line 1529 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		toSnapshot.push_back(metadata->files);
															#line 1530 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<BlobFileIndex> __when_expr_12 = compactFromBlob(bwData, bstore, metadata, granuleID, toSnapshot, reSnapshotVersion);
															#line 1530 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13039 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1Catch1(__when_expr_12.getError(), loopDepth); else return a_body1cont6when1(__when_expr_12.get(), loopDepth); };
		static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 9;
															#line 1530 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 12, BlobFileIndex >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 13044 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1(int loopDepth) 
	{
															#line 1525 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!(metadata->files.deltaFiles.empty() || metadata->files.deltaFiles.back().version < reSnapshotVersion))
															#line 13060 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
			return a_body1cont4break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1526 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_11 = delay(FLOW_KNOBS->PREVENT_FAST_SPIN_DELAY);
															#line 1526 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13068 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1Catch1(__when_expr_11.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1when1(__when_expr_11.get(), loopDepth); };
		static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 8;
															#line 1526 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< CheckSplitAndReSnapshotActor, 11, Void >*>(static_cast<CheckSplitAndReSnapshotActor*>(this)));
															#line 13073 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4break1(int loopDepth) 
	{
		try {
			return a_body1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state > 0) static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont8(BlobFileIndex const& reSnapshotIdx,int loopDepth) 
	{
															#line 1532 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<CheckSplitAndReSnapshotActor*>(this)->SAV<BlobFileIndex>::futures) { (void)(reSnapshotIdx); this->~CheckSplitAndReSnapshotActorState(); static_cast<CheckSplitAndReSnapshotActor*>(this)->destroy(); return 0; }
															#line 13182 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<CheckSplitAndReSnapshotActor*>(this)->SAV< BlobFileIndex >::value()) BlobFileIndex(reSnapshotIdx);
		this->~CheckSplitAndReSnapshotActorState();
		static_cast<CheckSplitAndReSnapshotActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont8(BlobFileIndex && reSnapshotIdx,int loopDepth) 
	{
															#line 1532 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<CheckSplitAndReSnapshotActor*>(this)->SAV<BlobFileIndex>::futures) { (void)(reSnapshotIdx); this->~CheckSplitAndReSnapshotActorState(); static_cast<CheckSplitAndReSnapshotActor*>(this)->destroy(); return 0; }
															#line 13194 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<CheckSplitAndReSnapshotActor*>(this)->SAV< BlobFileIndex >::value()) BlobFileIndex(reSnapshotIdx);
		this->~CheckSplitAndReSnapshotActorState();
		static_cast<CheckSplitAndReSnapshotActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6when1(BlobFileIndex const& reSnapshotIdx,int loopDepth) 
	{
		loopDepth = a_body1cont8(reSnapshotIdx, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(BlobFileIndex && reSnapshotIdx,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(reSnapshotIdx), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state > 0) static_cast<CheckSplitAndReSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<CheckSplitAndReSnapshotActor*>(this)->ActorCallback< CheckSplitAndReSnapshotActor, 12, BlobFileIndex >::remove();

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 12, BlobFileIndex >*,BlobFileIndex const& value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< CheckSplitAndReSnapshotActor, 12, BlobFileIndex >*,BlobFileIndex && value) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< CheckSplitAndReSnapshotActor, 12, BlobFileIndex >*,Error err) 
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), 12);

	}
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobConnectionProvider> bstore;
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<GranuleMetadata> metadata;
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID granuleID;
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t bytesInNewDeltaFiles;
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Future<BlobFileIndex> lastDeltaBeforeSnapshot;
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t versionsSinceLastSnapshot;
															#line 1413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version reSnapshotVersion;
															#line 1420 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	CounterHolder pendingCounter;
															#line 1438 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t statusEpoch;
															#line 1439 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t statusSeqno;
															#line 1444 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	bool writeHot;
															#line 13301 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via checkSplitAndReSnapshot()
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class CheckSplitAndReSnapshotActor final : public Actor<BlobFileIndex>, public ActorCallback< CheckSplitAndReSnapshotActor, 0, BlobFileIndex >, public ActorCallback< CheckSplitAndReSnapshotActor, 1, Void >, public ActorCallback< CheckSplitAndReSnapshotActor, 2, Void >, public ActorCallback< CheckSplitAndReSnapshotActor, 3, Void >, public ActorCallback< CheckSplitAndReSnapshotActor, 4, Void >, public ActorCallback< CheckSplitAndReSnapshotActor, 5, Void >, public ActorCallback< CheckSplitAndReSnapshotActor, 6, Void >, public ActorCallback< CheckSplitAndReSnapshotActor, 7, Void >, public ActorCallback< CheckSplitAndReSnapshotActor, 8, Void >, public ActorCallback< CheckSplitAndReSnapshotActor, 9, Void >, public ActorCallback< CheckSplitAndReSnapshotActor, 10, Void >, public ActorCallback< CheckSplitAndReSnapshotActor, 11, Void >, public ActorCallback< CheckSplitAndReSnapshotActor, 12, BlobFileIndex >, public FastAllocated<CheckSplitAndReSnapshotActor>, public CheckSplitAndReSnapshotActorState<CheckSplitAndReSnapshotActor> {
															#line 13306 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<CheckSplitAndReSnapshotActor>::operator new;
	using FastAllocated<CheckSplitAndReSnapshotActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(40981224639724800UL, 1653186159206989824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobFileIndex>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 0, BlobFileIndex >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 1, Void >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 2, Void >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 3, Void >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 4, Void >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 5, Void >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 6, Void >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 7, Void >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 8, Void >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 9, Void >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 10, Void >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 11, Void >;
friend struct ActorCallback< CheckSplitAndReSnapshotActor, 12, BlobFileIndex >;
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	CheckSplitAndReSnapshotActor(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,Reference<GranuleMetadata> const& metadata,UID const& granuleID,int64_t const& bytesInNewDeltaFiles,Future<BlobFileIndex> const& lastDeltaBeforeSnapshot,int64_t const& versionsSinceLastSnapshot) 
															#line 13335 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<BlobFileIndex>(),
		   CheckSplitAndReSnapshotActorState<CheckSplitAndReSnapshotActor>(bwData, bstore, metadata, granuleID, bytesInNewDeltaFiles, lastDeltaBeforeSnapshot, versionsSinceLastSnapshot),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2362867189839348480UL, 6093274178592357888UL);
		ActorExecutionContextHelper __helper(static_cast<CheckSplitAndReSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkSplitAndReSnapshot");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkSplitAndReSnapshot", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckSplitAndReSnapshotActor, 0, BlobFileIndex >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckSplitAndReSnapshotActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CheckSplitAndReSnapshotActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< CheckSplitAndReSnapshotActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< CheckSplitAndReSnapshotActor, 6, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< CheckSplitAndReSnapshotActor, 7, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< CheckSplitAndReSnapshotActor, 8, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< CheckSplitAndReSnapshotActor, 11, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< CheckSplitAndReSnapshotActor, 12, BlobFileIndex >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<BlobFileIndex> checkSplitAndReSnapshot( Reference<BlobWorkerData> const& bwData, Reference<BlobConnectionProvider> const& bstore, Reference<GranuleMetadata> const& metadata, UID const& granuleID, int64_t const& bytesInNewDeltaFiles, Future<BlobFileIndex> const& lastDeltaBeforeSnapshot, int64_t const& versionsSinceLastSnapshot ) {
															#line 1404 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<BlobFileIndex>(new CheckSplitAndReSnapshotActor(bwData, bstore, metadata, granuleID, bytesInNewDeltaFiles, lastDeltaBeforeSnapshot, versionsSinceLastSnapshot));
															#line 13376 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 1534 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

															#line 13381 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via reSnapshotNoCheck()
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class ReSnapshotNoCheckActor>
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class ReSnapshotNoCheckActorState {
															#line 13388 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	ReSnapshotNoCheckActorState(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,Reference<GranuleMetadata> const& metadata,UID const& granuleID,Future<BlobFileIndex> const& lastDeltaBeforeSnapshot) 
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   bstore(bstore),
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   metadata(metadata),
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleID(granuleID),
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   lastDeltaBeforeSnapshot(lastDeltaBeforeSnapshot)
															#line 13403 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this));

	}
	~ReSnapshotNoCheckActorState() 
	{
		fdb_probe_actor_destroy("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1540 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<BlobFileIndex> __when_expr_0 = lastDeltaBeforeSnapshot;
															#line 1540 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13420 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state = 1;
															#line 1540 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReSnapshotNoCheckActor, 0, BlobFileIndex >*>(static_cast<ReSnapshotNoCheckActor*>(this)));
															#line 13425 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReSnapshotNoCheckActorState();
		static_cast<ReSnapshotNoCheckActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(BlobFileIndex const& lastDeltaIdx,int loopDepth) 
	{
															#line 1541 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		reSnapshotVersion = lastDeltaIdx.version;
															#line 1542 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::BlobWorkerUpdateFDB);
															#line 1542 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13452 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state = 2;
															#line 1542 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReSnapshotNoCheckActor, 1, Void >*>(static_cast<ReSnapshotNoCheckActor*>(this)));
															#line 13457 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(BlobFileIndex && lastDeltaIdx,int loopDepth) 
	{
															#line 1541 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		reSnapshotVersion = lastDeltaIdx.version;
															#line 1542 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0, TaskPriority::BlobWorkerUpdateFDB);
															#line 1542 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13470 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state = 2;
															#line 1542 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReSnapshotNoCheckActor, 1, Void >*>(static_cast<ReSnapshotNoCheckActor*>(this)));
															#line 13475 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(BlobFileIndex const& lastDeltaIdx,int loopDepth) 
	{
		loopDepth = a_body1cont1(lastDeltaIdx, loopDepth);

		return loopDepth;
	}
	int a_body1when1(BlobFileIndex && lastDeltaIdx,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(lastDeltaIdx), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state > 0) static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state = 0;
		static_cast<ReSnapshotNoCheckActor*>(this)->ActorCallback< ReSnapshotNoCheckActor, 0, BlobFileIndex >::remove();

	}
	void a_callback_fire(ActorCallback< ReSnapshotNoCheckActor, 0, BlobFileIndex >*,BlobFileIndex const& value) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReSnapshotNoCheckActor, 0, BlobFileIndex >*,BlobFileIndex && value) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReSnapshotNoCheckActor, 0, BlobFileIndex >*,Error err) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1544 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "re-snapshotting without BM check because still on old change feed!");
															#line 1546 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 13561 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1547 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Granule [{0} - {1}) re-snapshotting @ {2} WITHOUT checking with BM, because it is still on old " "change feed!\n", metadata->keyRange.begin.printable(), metadata->keyRange.end.printable(), reSnapshotVersion);
															#line 13565 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1554 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		TraceEvent(SevDebug, "BlobGranuleReSnapshotOldFeed", bwData->id) .detail("Granule", metadata->keyRange) .detail("Version", reSnapshotVersion);
															#line 1557 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		++bwData->stats.oldFeedSnapshots;
															#line 1560 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 13573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1544 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "re-snapshotting without BM check because still on old change feed!");
															#line 1546 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BW_DEBUG)
															#line 13584 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1547 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			fmt::print("Granule [{0} - {1}) re-snapshotting @ {2} WITHOUT checking with BM, because it is still on old " "change feed!\n", metadata->keyRange.begin.printable(), metadata->keyRange.end.printable(), reSnapshotVersion);
															#line 13588 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 1554 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		TraceEvent(SevDebug, "BlobGranuleReSnapshotOldFeed", bwData->id) .detail("Granule", metadata->keyRange) .detail("Version", reSnapshotVersion);
															#line 1557 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		++bwData->stats.oldFeedSnapshots;
															#line 1560 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 13596 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state > 0) static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state = 0;
		static_cast<ReSnapshotNoCheckActor*>(this)->ActorCallback< ReSnapshotNoCheckActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReSnapshotNoCheckActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReSnapshotNoCheckActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReSnapshotNoCheckActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1564 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		std::vector<GranuleFiles> toSnapshot;
															#line 1565 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		toSnapshot.push_back(metadata->files);
															#line 1566 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<BlobFileIndex> __when_expr_3 = compactFromBlob(bwData, bstore, metadata, granuleID, toSnapshot, reSnapshotVersion);
															#line 1566 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13686 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state = 4;
															#line 1566 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReSnapshotNoCheckActor, 3, BlobFileIndex >*>(static_cast<ReSnapshotNoCheckActor*>(this)));
															#line 13691 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 1560 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!(metadata->files.deltaFiles.empty() || metadata->files.deltaFiles.back().version < reSnapshotVersion))
															#line 13707 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
			return a_body1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1561 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(FLOW_KNOBS->PREVENT_FAST_SPIN_DELAY);
															#line 1561 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13715 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state = 3;
															#line 1561 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReSnapshotNoCheckActor, 2, Void >*>(static_cast<ReSnapshotNoCheckActor*>(this)));
															#line 13720 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state > 0) static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state = 0;
		static_cast<ReSnapshotNoCheckActor*>(this)->ActorCallback< ReSnapshotNoCheckActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReSnapshotNoCheckActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReSnapshotNoCheckActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReSnapshotNoCheckActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont5(BlobFileIndex const& reSnapshotIdx,int loopDepth) 
	{
															#line 1569 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<ReSnapshotNoCheckActor*>(this)->SAV<BlobFileIndex>::futures) { (void)(reSnapshotIdx); this->~ReSnapshotNoCheckActorState(); static_cast<ReSnapshotNoCheckActor*>(this)->destroy(); return 0; }
															#line 13829 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<ReSnapshotNoCheckActor*>(this)->SAV< BlobFileIndex >::value()) BlobFileIndex(reSnapshotIdx);
		this->~ReSnapshotNoCheckActorState();
		static_cast<ReSnapshotNoCheckActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(BlobFileIndex && reSnapshotIdx,int loopDepth) 
	{
															#line 1569 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<ReSnapshotNoCheckActor*>(this)->SAV<BlobFileIndex>::futures) { (void)(reSnapshotIdx); this->~ReSnapshotNoCheckActorState(); static_cast<ReSnapshotNoCheckActor*>(this)->destroy(); return 0; }
															#line 13841 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<ReSnapshotNoCheckActor*>(this)->SAV< BlobFileIndex >::value()) BlobFileIndex(reSnapshotIdx);
		this->~ReSnapshotNoCheckActorState();
		static_cast<ReSnapshotNoCheckActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3when1(BlobFileIndex const& reSnapshotIdx,int loopDepth) 
	{
		loopDepth = a_body1cont5(reSnapshotIdx, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(BlobFileIndex && reSnapshotIdx,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(reSnapshotIdx), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state > 0) static_cast<ReSnapshotNoCheckActor*>(this)->actor_wait_state = 0;
		static_cast<ReSnapshotNoCheckActor*>(this)->ActorCallback< ReSnapshotNoCheckActor, 3, BlobFileIndex >::remove();

	}
	void a_callback_fire(ActorCallback< ReSnapshotNoCheckActor, 3, BlobFileIndex >*,BlobFileIndex const& value) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ReSnapshotNoCheckActor, 3, BlobFileIndex >*,BlobFileIndex && value) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ReSnapshotNoCheckActor, 3, BlobFileIndex >*,Error err) 
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobConnectionProvider> bstore;
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<GranuleMetadata> metadata;
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID granuleID;
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Future<BlobFileIndex> lastDeltaBeforeSnapshot;
															#line 1541 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version reSnapshotVersion;
															#line 13936 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via reSnapshotNoCheck()
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class ReSnapshotNoCheckActor final : public Actor<BlobFileIndex>, public ActorCallback< ReSnapshotNoCheckActor, 0, BlobFileIndex >, public ActorCallback< ReSnapshotNoCheckActor, 1, Void >, public ActorCallback< ReSnapshotNoCheckActor, 2, Void >, public ActorCallback< ReSnapshotNoCheckActor, 3, BlobFileIndex >, public FastAllocated<ReSnapshotNoCheckActor>, public ReSnapshotNoCheckActorState<ReSnapshotNoCheckActor> {
															#line 13941 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<ReSnapshotNoCheckActor>::operator new;
	using FastAllocated<ReSnapshotNoCheckActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(660401465280343808UL, 3685853752014624768UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobFileIndex>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReSnapshotNoCheckActor, 0, BlobFileIndex >;
friend struct ActorCallback< ReSnapshotNoCheckActor, 1, Void >;
friend struct ActorCallback< ReSnapshotNoCheckActor, 2, Void >;
friend struct ActorCallback< ReSnapshotNoCheckActor, 3, BlobFileIndex >;
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	ReSnapshotNoCheckActor(Reference<BlobWorkerData> const& bwData,Reference<BlobConnectionProvider> const& bstore,Reference<GranuleMetadata> const& metadata,UID const& granuleID,Future<BlobFileIndex> const& lastDeltaBeforeSnapshot) 
															#line 13961 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<BlobFileIndex>(),
		   ReSnapshotNoCheckActorState<ReSnapshotNoCheckActor>(bwData, bstore, metadata, granuleID, lastDeltaBeforeSnapshot),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2529696302917022208UL, 12256695065979655168UL);
		ActorExecutionContextHelper __helper(static_cast<ReSnapshotNoCheckActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("reSnapshotNoCheck");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("reSnapshotNoCheck", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReSnapshotNoCheckActor, 0, BlobFileIndex >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReSnapshotNoCheckActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReSnapshotNoCheckActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ReSnapshotNoCheckActor, 3, BlobFileIndex >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<BlobFileIndex> reSnapshotNoCheck( Reference<BlobWorkerData> const& bwData, Reference<BlobConnectionProvider> const& bstore, Reference<GranuleMetadata> const& metadata, UID const& granuleID, Future<BlobFileIndex> const& lastDeltaBeforeSnapshot ) {
															#line 1535 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<BlobFileIndex>(new ReSnapshotNoCheckActor(bwData, bstore, metadata, granuleID, lastDeltaBeforeSnapshot));
															#line 13997 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 1571 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

// wait indefinitely to tell manager to re-evaluate this split, until the granule is revoked
															#line 14003 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via reevaluateInitialSplit()
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class ReevaluateInitialSplitActor>
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class ReevaluateInitialSplitActorState {
															#line 14010 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	ReevaluateInitialSplitActorState(Reference<BlobWorkerData> const& bwData,UID const& granuleID,KeyRange const& keyRange,int64_t const& epoch,int64_t const& seqno,Key const& proposedSplitKey) 
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleID(granuleID),
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   keyRange(keyRange),
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   epoch(epoch),
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   seqno(seqno),
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   proposedSplitKey(proposedSplitKey)
															#line 14027 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this));

	}
	~ReevaluateInitialSplitActorState() 
	{
		fdb_probe_actor_destroy("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1580 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			;
															#line 14042 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReevaluateInitialSplitActorState();
		static_cast<ReevaluateInitialSplitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1583 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 14065 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1580 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!(!bwData->statusStreamInitialized))
															#line 14081 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1581 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_0 = bwData->currentManagerStatusStream.onChange();
															#line 1581 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14089 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 1;
															#line 1581 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 0, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 14094 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 1586 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			;
															#line 14211 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1618 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 14233 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1619 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 14237 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1622 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			CODE_PROBE(true, "Blob worker re-sending merge candidate to manager after not error/not hearing back");
															#line 1626 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (e.code() == error_code_broken_promise)
															#line 14243 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1627 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<Void> __when_expr_6 = delay(FLOW_KNOBS->PREVENT_FAST_SPIN_DELAY);
															#line 1627 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14249 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_6.get(), loopDepth); };
				static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 6;
															#line 1627 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 6, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 14254 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1629 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<Void> __when_expr_7 = bwData->currentManagerStatusStream.onChange();
															#line 1629 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14263 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when2(__when_expr_7.get(), loopDepth); };
				static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 7;
															#line 1629 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 7, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 14268 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(int loopDepth) 
	{
															#line 1595 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		GranuleStatusReply reply(keyRange, true, false, true, epoch, seqno, granuleID, invalidVersion, invalidVersion, false, epoch, seqno);
															#line 1607 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		reply.proposedSplitKey = proposedSplitKey;
															#line 1608 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->currentManagerStatusStream.get().send(reply);
															#line 1609 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (BUGGIFY && bwData->maybeInjectTargetedRestart())
															#line 14290 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1610 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_3 = Never();
															#line 1610 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14296 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_3.get(), loopDepth); };
			static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 3;
															#line 1610 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 3, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 14301 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1(int loopDepth) 
	{
															#line 1588 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bwData->currentManagerStatusStream.get().onReady();
															#line 1587 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14324 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1loopBody1when1(__when_expr_1.get(), loopDepth); };
															#line 1591 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_2 = bwData->currentManagerStatusStream.onChange();
															#line 14328 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1loopBody1when2(__when_expr_2.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 2;
															#line 1588 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 1, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 1591 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 2, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 14335 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1cont1loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		return a_body1cont1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		return a_body1cont1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1loopBody1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 1, Void >::remove();
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont3(int loopDepth) 
	{
															#line 1614 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_4 = success(timeout(bwData->currentManagerStatusStream.onChange(), 10.0));
															#line 1614 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14510 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_4.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 4;
															#line 1614 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 4, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 14515 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1615 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_5 = delay(0);
															#line 1615 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14613 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_5.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 5;
															#line 1615 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 5, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 14618 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1615 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_5 = delay(0);
															#line 1615 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14629 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_5.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 5;
															#line 1615 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 5, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 14634 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1616 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "Blob worker re-sending initialsplit too big");
															#line 14718 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1616 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "Blob worker re-sending initialsplit too big");
															#line 14727 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont1loopBody1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont1loopBody1Catch1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);

	}
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID granuleID;
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	KeyRange keyRange;
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t epoch;
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	int64_t seqno;
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Key proposedSplitKey;
															#line 15012 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via reevaluateInitialSplit()
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class ReevaluateInitialSplitActor final : public Actor<Void>, public ActorCallback< ReevaluateInitialSplitActor, 0, Void >, public ActorCallback< ReevaluateInitialSplitActor, 1, Void >, public ActorCallback< ReevaluateInitialSplitActor, 2, Void >, public ActorCallback< ReevaluateInitialSplitActor, 3, Void >, public ActorCallback< ReevaluateInitialSplitActor, 4, Void >, public ActorCallback< ReevaluateInitialSplitActor, 5, Void >, public ActorCallback< ReevaluateInitialSplitActor, 6, Void >, public ActorCallback< ReevaluateInitialSplitActor, 7, Void >, public FastAllocated<ReevaluateInitialSplitActor>, public ReevaluateInitialSplitActorState<ReevaluateInitialSplitActor> {
															#line 15017 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<ReevaluateInitialSplitActor>::operator new;
	using FastAllocated<ReevaluateInitialSplitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6693355065706049280UL, 14525879253646763520UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReevaluateInitialSplitActor, 0, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 1, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 2, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 3, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 4, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 5, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 6, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 7, Void >;
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	ReevaluateInitialSplitActor(Reference<BlobWorkerData> const& bwData,UID const& granuleID,KeyRange const& keyRange,int64_t const& epoch,int64_t const& seqno,Key const& proposedSplitKey) 
															#line 15041 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<Void>(),
		   ReevaluateInitialSplitActorState<ReevaluateInitialSplitActor>(bwData, granuleID, keyRange, epoch, seqno, proposedSplitKey),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12431337998140563968UL, 17512528355319016704UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("reevaluateInitialSplit");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 3, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 4, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 5, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 6, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<Void> reevaluateInitialSplit( Reference<BlobWorkerData> const& bwData, UID const& granuleID, KeyRange const& keyRange, int64_t const& epoch, int64_t const& seqno, Key const& proposedSplitKey ) {
															#line 1573 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<Void>(new ReevaluateInitialSplitActor(bwData, granuleID, keyRange, epoch, seqno, proposedSplitKey));
															#line 15080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 1634 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

															#line 15085 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via granuleCheckMergeCandidate()
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class GranuleCheckMergeCandidateActor>
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class GranuleCheckMergeCandidateActorState {
															#line 15092 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GranuleCheckMergeCandidateActorState(Reference<BlobWorkerData> const& bwData,Reference<GranuleMetadata> const& metadata,UID const& granuleID,Future<Void> const& waitStart) 
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   metadata(metadata),
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   granuleID(granuleID),
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   waitStart(waitStart)
															#line 15105 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this));

	}
	~GranuleCheckMergeCandidateActorState() 
	{
		fdb_probe_actor_destroy("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1639 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (!SERVER_KNOBS->BG_ENABLE_MERGING)
															#line 15120 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1640 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (!static_cast<GranuleCheckMergeCandidateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GranuleCheckMergeCandidateActorState(); static_cast<GranuleCheckMergeCandidateActor*>(this)->destroy(); return 0; }
															#line 15124 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				new (&static_cast<GranuleCheckMergeCandidateActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~GranuleCheckMergeCandidateActorState();
				static_cast<GranuleCheckMergeCandidateActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_0 = waitStart;
															#line 1643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15134 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 1;
															#line 1643 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 0, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 15139 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GranuleCheckMergeCandidateActorState();
		static_cast<GranuleCheckMergeCandidateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1644 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		double jitter = deterministicRandom()->random01() * 0.8 * SERVER_KNOBS->BG_MERGE_CANDIDATE_DELAY_SECONDS;
															#line 1645 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->BG_MERGE_CANDIDATE_THRESHOLD_SECONDS + jitter);
															#line 1645 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15166 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 2;
															#line 1645 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 1, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 15171 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1644 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		double jitter = deterministicRandom()->random01() * 0.8 * SERVER_KNOBS->BG_MERGE_CANDIDATE_DELAY_SECONDS;
															#line 1645 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->BG_MERGE_CANDIDATE_THRESHOLD_SECONDS + jitter);
															#line 1645 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15184 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 2;
															#line 1645 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 1, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 15189 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state > 0) static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 0;
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1646 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 15273 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1646 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 15282 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state > 0) static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 0;
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1(int loopDepth) 
	{
															#line 1650 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<StorageMetrics> __when_expr_2 = bwData->db->getStorageMetrics(metadata->keyRange, CLIENT_KNOBS->TOO_MANY);
															#line 1650 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15375 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 3;
															#line 1650 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 2, StorageMetrics >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 15380 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(StorageMetrics const& currentMetrics,int loopDepth) 
	{
															#line 1653 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (currentMetrics.bytes >= SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES / 2 || currentMetrics.bytesWrittenPerKSecond >= SERVER_KNOBS->SHARD_MIN_BYTES_PER_KSEC)
															#line 15389 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1655 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delayJittered(SERVER_KNOBS->BG_MERGE_CANDIDATE_THRESHOLD_SECONDS / 2.0);
															#line 1655 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15395 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
			static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 4;
															#line 1655 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 3, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 15400 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont3loopBody1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(StorageMetrics && currentMetrics,int loopDepth) 
	{
															#line 1653 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (currentMetrics.bytes >= SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES / 2 || currentMetrics.bytesWrittenPerKSecond >= SERVER_KNOBS->SHARD_MIN_BYTES_PER_KSEC)
															#line 15414 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1655 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delayJittered(SERVER_KNOBS->BG_MERGE_CANDIDATE_THRESHOLD_SECONDS / 2.0);
															#line 1655 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15420 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
			static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 4;
															#line 1655 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 3, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 15425 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont3loopBody1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(StorageMetrics const& currentMetrics,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont1(currentMetrics, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(StorageMetrics && currentMetrics,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont1(std::move(currentMetrics), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state > 0) static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 0;
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 2, StorageMetrics >::remove();

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 2, StorageMetrics >*,StorageMetrics const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 2, StorageMetrics >*,StorageMetrics && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 2, StorageMetrics >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont3loopBody1cont2(int loopDepth) 
	{
															#line 1660 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "Blob Worker identified merge candidate granule");
															#line 1663 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 15516 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont3loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "wait and check later to see if granule got smaller or colder");
															#line 15525 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		return a_body1cont3loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont3loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1656 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "wait and check later to see if granule got smaller or colder");
															#line 15534 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		return a_body1cont3loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state > 0) static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 0;
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont3loopBody1cont5(int loopDepth) 
	{
															#line 1667 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		sendTimeGiveUp = now() + SERVER_KNOBS->BG_MERGE_CANDIDATE_THRESHOLD_SECONDS / 2.0;
															#line 1668 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		;
															#line 15620 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont3loopBody1cont5loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 1663 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!(!bwData->statusStreamInitialized))
															#line 15636 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
			return a_body1cont3loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1664 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_4 = bwData->currentManagerStatusStream.onChange();
															#line 1664 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 15644 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 2)); else return a_body1cont3loopBody1cont2loopBody1when1(__when_expr_4.get(), loopDepth); };
		static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 5;
															#line 1664 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 4, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 15649 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont3loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont3loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont3loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state > 0) static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 0;
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont3loopBody1cont6(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1cont5loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1(int loopDepth) 
	{
		try {
															#line 1671 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			;
															#line 15772 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1cont3loopBody1cont5loopBody1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5break1(int loopDepth) 
	{
		try {
			return a_body1cont3loopBody1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont3loopBody1cont5loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1706 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 15807 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1707 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 2));
															#line 15811 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1710 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			CODE_PROBE(true, "Blob worker re-sending merge candidate to manager after not error/not hearing back");
															#line 1714 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (e.code() == error_code_broken_promise)
															#line 15817 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1715 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<Void> __when_expr_10 = delay(FLOW_KNOBS->PREVENT_FAST_SPIN_DELAY);
															#line 1715 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 15823 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch1(__when_expr_10.getError(), std::max(0, loopDepth - 2)); else return a_body1cont3loopBody1cont5loopBody1Catch1when1(__when_expr_10.get(), loopDepth); };
				static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 9;
															#line 1715 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 10, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 15828 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1717 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<Void> __when_expr_11 = bwData->currentManagerStatusStream.onChange();
															#line 1717 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 15837 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1Catch1(__when_expr_11.getError(), std::max(0, loopDepth - 2)); else return a_body1cont3loopBody1cont5loopBody1Catch1when2(__when_expr_11.get(), loopDepth); };
				static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 10;
															#line 1717 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 11, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 15842 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1cont2(int loopDepth) 
	{
															#line 1683 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (now() >= sendTimeGiveUp)
															#line 15858 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 1684 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			CODE_PROBE(true, "Blob worker could not send merge candidate in time, re-checking status");
															#line 15862 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			return a_body1cont3loopBody1cont5break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1688 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		bwData->currentManagerStatusStream.get().send(GranuleStatusReply(metadata->keyRange, false, false, false, metadata->continueEpoch, metadata->continueSeqno, granuleID, metadata->historyVersion, invalidVersion, true, metadata->originalEpoch, metadata->originalSeqno));
															#line 1702 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_8 = bwData->currentManagerStatusStream.onChange();
															#line 1702 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1cont3loopBody1cont5loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15871 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont3loopBody1cont5loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont3loopBody1cont5loopBody1cont2when1(__when_expr_8.get(), loopDepth); };
		static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 7;
															#line 1702 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 8, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 15876 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1cont5loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1loopBody1(int loopDepth) 
	{
															#line 1673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_5 = delay(std::max(0.0, sendTimeGiveUp - now()));
															#line 1672 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1cont3loopBody1cont5loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15894 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont3loopBody1cont5loopBody1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont5loopBody1loopBody1when1(__when_expr_5.get(), loopDepth); };
															#line 1676 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_6 = bwData->currentManagerStatusStream.get().onReady();
															#line 15898 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont3loopBody1cont5loopBody1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont5loopBody1loopBody1when2(__when_expr_6.get(), loopDepth); };
															#line 1679 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_7 = bwData->currentManagerStatusStream.onChange();
															#line 15902 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont3loopBody1cont5loopBody1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont5loopBody1loopBody1when3(__when_expr_7.get(), loopDepth); };
		static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 6;
															#line 1673 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 5, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 1676 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 6, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 1679 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 7, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 15911 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1cont3loopBody1cont5loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont3loopBody1cont5loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		return a_body1cont3loopBody1cont5loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		return a_body1cont3loopBody1cont5loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1loopBody1when2(Void const& _,int loopDepth) 
	{
		return a_body1cont3loopBody1cont5loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1loopBody1when2(Void && _,int loopDepth) 
	{
		return a_body1cont3loopBody1cont5loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1loopBody1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1loopBody1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state > 0) static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 0;
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 5, Void >::remove();
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 6, Void >::remove();
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3loopBody1cont5loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3loopBody1cont5loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3loopBody1cont5loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3loopBody1cont5loopBody1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3loopBody1cont5loopBody1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3loopBody1cont5loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3loopBody1cont5loopBody1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3loopBody1cont5loopBody1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3loopBody1cont5loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont3loopBody1cont5loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1703 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_9 = delay(0);
															#line 1703 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1cont3loopBody1cont5loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16156 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont3loopBody1cont5loopBody1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont3loopBody1cont5loopBody1cont3when1(__when_expr_9.get(), loopDepth); };
		static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 8;
															#line 1703 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 9, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 16161 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1703 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_9 = delay(0);
															#line 1703 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state < 0) return a_body1cont3loopBody1cont5loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16172 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont3loopBody1cont5loopBody1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont3loopBody1cont5loopBody1cont3when1(__when_expr_9.get(), loopDepth); };
		static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 8;
															#line 1703 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< GranuleCheckMergeCandidateActor, 9, Void >*>(static_cast<GranuleCheckMergeCandidateActor*>(this)));
															#line 16177 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state > 0) static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 0;
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont3loopBody1cont5loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont3loopBody1cont5loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont3loopBody1cont5loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont3loopBody1cont5loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1704 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "Blob worker re-sending merge candidate to new manager");
															#line 16261 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont3loopBody1cont5loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1704 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		CODE_PROBE(true, "Blob worker re-sending merge candidate to new manager");
															#line 16270 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont3loopBody1cont5loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state > 0) static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 0;
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont3loopBody1cont5loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont3loopBody1cont5loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont3loopBody1cont5loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont3loopBody1cont5loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont3loopBody1cont5loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1Catch1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1Catch1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state > 0) static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 0;
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont3loopBody1cont5loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont3loopBody1cont5loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont3loopBody1cont5loopBody1Catch1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1Catch1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1Catch1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont5loopBody1Catch1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont5loopBody1Catch1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state > 0) static_cast<GranuleCheckMergeCandidateActor*>(this)->actor_wait_state = 0;
		static_cast<GranuleCheckMergeCandidateActor*>(this)->ActorCallback< GranuleCheckMergeCandidateActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont3loopBody1cont5loopBody1Catch1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< GranuleCheckMergeCandidateActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont3loopBody1cont5loopBody1Catch1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< GranuleCheckMergeCandidateActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), 11);

	}
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<GranuleMetadata> metadata;
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	UID granuleID;
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Future<Void> waitStart;
															#line 1667 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	double sendTimeGiveUp;
															#line 16553 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via granuleCheckMergeCandidate()
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class GranuleCheckMergeCandidateActor final : public Actor<Void>, public ActorCallback< GranuleCheckMergeCandidateActor, 0, Void >, public ActorCallback< GranuleCheckMergeCandidateActor, 1, Void >, public ActorCallback< GranuleCheckMergeCandidateActor, 2, StorageMetrics >, public ActorCallback< GranuleCheckMergeCandidateActor, 3, Void >, public ActorCallback< GranuleCheckMergeCandidateActor, 4, Void >, public ActorCallback< GranuleCheckMergeCandidateActor, 5, Void >, public ActorCallback< GranuleCheckMergeCandidateActor, 6, Void >, public ActorCallback< GranuleCheckMergeCandidateActor, 7, Void >, public ActorCallback< GranuleCheckMergeCandidateActor, 8, Void >, public ActorCallback< GranuleCheckMergeCandidateActor, 9, Void >, public ActorCallback< GranuleCheckMergeCandidateActor, 10, Void >, public ActorCallback< GranuleCheckMergeCandidateActor, 11, Void >, public FastAllocated<GranuleCheckMergeCandidateActor>, public GranuleCheckMergeCandidateActorState<GranuleCheckMergeCandidateActor> {
															#line 16558 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<GranuleCheckMergeCandidateActor>::operator new;
	using FastAllocated<GranuleCheckMergeCandidateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3376322086110249728UL, 7044528858930070784UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 0, Void >;
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 1, Void >;
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 2, StorageMetrics >;
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 3, Void >;
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 4, Void >;
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 5, Void >;
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 6, Void >;
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 7, Void >;
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 8, Void >;
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 9, Void >;
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 10, Void >;
friend struct ActorCallback< GranuleCheckMergeCandidateActor, 11, Void >;
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GranuleCheckMergeCandidateActor(Reference<BlobWorkerData> const& bwData,Reference<GranuleMetadata> const& metadata,UID const& granuleID,Future<Void> const& waitStart) 
															#line 16586 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<Void>(),
		   GranuleCheckMergeCandidateActorState<GranuleCheckMergeCandidateActor>(bwData, metadata, granuleID, waitStart),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6707685142648200192UL, 9646091180970554880UL);
		ActorExecutionContextHelper __helper(static_cast<GranuleCheckMergeCandidateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("granuleCheckMergeCandidate");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("granuleCheckMergeCandidate", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GranuleCheckMergeCandidateActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GranuleCheckMergeCandidateActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GranuleCheckMergeCandidateActor, 2, StorageMetrics >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GranuleCheckMergeCandidateActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< GranuleCheckMergeCandidateActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< GranuleCheckMergeCandidateActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< GranuleCheckMergeCandidateActor, 8, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< GranuleCheckMergeCandidateActor, 9, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< GranuleCheckMergeCandidateActor, 10, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< GranuleCheckMergeCandidateActor, 11, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<Void> granuleCheckMergeCandidate( Reference<BlobWorkerData> const& bwData, Reference<GranuleMetadata> const& metadata, UID const& granuleID, Future<Void> const& waitStart ) {
															#line 1635 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<Void>(new GranuleCheckMergeCandidateActor(bwData, metadata, granuleID, waitStart));
															#line 16628 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 1723 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

namespace {
void handleCompletedDeltaFile(Reference<BlobWorkerData> bwData,
                              Reference<GranuleMetadata> metadata,
                              BlobFileIndex completedDeltaFile,
                              Key cfKey,
                              Version cfStartVersion,
                              std::deque<std::pair<Version, Version>>* rollbacksCompleted,
                              std::deque<Future<Void>>& inFlightPops,
                              bool emptyDeltaFile) {
	if (emptyDeltaFile) {
		ASSERT(!metadata->files.deltaFiles.empty());
		ASSERT(completedDeltaFile.length == 0);
		ASSERT(metadata->files.deltaFiles.back().version < completedDeltaFile.version);
		metadata->files.deltaFiles.back().version = completedDeltaFile.version;
	} else {
		metadata->files.deltaFiles.push_back(completedDeltaFile);
	}
	ASSERT(metadata->durableDeltaVersion.get() < completedDeltaFile.version);
	metadata->durableDeltaVersion.set(completedDeltaFile.version);

	if (completedDeltaFile.version > cfStartVersion) {
		if (BW_DEBUG) {
			fmt::print("Popping change feed {0} at {1}\n",
			           cfKeyToGranuleID(cfKey).toString().c_str(),
			           completedDeltaFile.version);
		}
		// FIXME: for a write-hot shard, we could potentially batch these and only pop the largest one after
		// several have completed
		// FIXME: since this is async, and worker could die, new blob worker that opens granule should probably
		// kick off an async pop at its previousDurableVersion after opening the granule to guarantee it is
		// eventually popped?
		Future<Void> popFuture = bwData->db->popChangeFeedMutations(cfKey, completedDeltaFile.version + 1);
		// Do pop asynchronously
		inFlightPops.push_back(popFuture);
	}
	while (!rollbacksCompleted->empty() && completedDeltaFile.version >= rollbacksCompleted->front().second) {
		if (BW_DEBUG) {
			fmt::print("Granule [{0} - {1}) on BW {2} completed rollback {3} -> {4} with delta file {5}\n",
			           metadata->keyRange.begin.printable().c_str(),
			           metadata->keyRange.end.printable().c_str(),
			           bwData->id.toString().substr(0, 5).c_str(),
			           rollbacksCompleted->front().second,
			           rollbacksCompleted->front().first,
			           completedDeltaFile.version);
		}
		rollbacksCompleted->pop_front();
	}
}

// if we get an i/o error updating files, or a rollback, reassign the granule to ourselves and start fresh
bool granuleCanRetry(const Error& e) {
	switch (e.code()) {
	case error_code_io_error:
	case error_code_io_timeout:
	// FIXME: handle connection errors in tighter retry loop around individual files.
	// FIXME: if these requests fail at a high enough rate, the whole worker should be marked as unhealthy and
	// its granules should be moved away, as there may be some problem with this host contacting blob storage
	case error_code_http_request_failed:
	case error_code_connection_failed:
	case error_code_lookup_failed: // dns
	case error_code_platform_error: // injected faults
		return true;
	default:
		return false;
	};
}
} // namespace

struct InFlightFile {
	Future<BlobFileIndex> future;
	Version version;
	uint64_t bytes;
	bool snapshot;
	bool emptyDeltaFile;

	InFlightFile(Future<BlobFileIndex> future, Version version, uint64_t bytes, bool snapshot, bool emptyDeltaFile)
	  : future(future), version(version), bytes(bytes), snapshot(snapshot), emptyDeltaFile(emptyDeltaFile) {}
};

namespace {
Version doGranuleRollback(Reference<GranuleMetadata> metadata,
                          Version mutationVersion,
                          Version rollbackVersion,
                          std::deque<InFlightFile>& inFlightFiles,
                          std::deque<std::pair<Version, Version>>& rollbacksInProgress,
                          std::deque<std::pair<Version, Version>>& rollbacksCompleted) {
	Version cfRollbackVersion;
	if (metadata->pendingDeltaVersion > rollbackVersion) {
		// if we already started writing mutations to a delta or snapshot file with version > rollbackVersion,
		// we need to rescind those delta file writes
		ASSERT(!inFlightFiles.empty());
		cfRollbackVersion = metadata->durableDeltaVersion.get();
		metadata->pendingSnapshotVersion = metadata->durableSnapshotVersion.get();
		int toPop = 0;
		// keep bytes in delta files pending here, then add back already durable delta files at end
		metadata->bytesInNewDeltaFiles = 0;
		metadata->newDeltaFileCount = 0;

		for (auto& f : inFlightFiles) {
			if (f.snapshot) {
				if (f.version > rollbackVersion) {
					CODE_PROBE(true, "Granule rollback cancelling snapshot file");
					if (BW_DEBUG) {
						fmt::print("[{0} - {1}) rollback cancelling snapshot file @ {2}\n",
						           metadata->keyRange.begin.printable(),
						           metadata->keyRange.end.printable(),
						           f.version);
					}
					f.future.cancel();
					toPop++;
				} else {
					metadata->pendingSnapshotVersion = f.version;
					metadata->bytesInNewDeltaFiles = 0;
					metadata->newDeltaFileCount = 0;
				}
			} else {
				if (f.version > rollbackVersion) {
					f.future.cancel();
					toPop++;
					CODE_PROBE(true, "Granule rollback cancelling delta file");
					if (BW_DEBUG) {
						fmt::print("[{0} - {1}) rollback cancelling delta file @ {2}\n",
						           metadata->keyRange.begin.printable(),
						           metadata->keyRange.end.printable(),
						           f.version);
					}
				} else {
					ASSERT(f.version > cfRollbackVersion);
					cfRollbackVersion = f.version;
					metadata->bytesInNewDeltaFiles += f.bytes;
					metadata->newDeltaFileCount++;
				}
			}
		}
		ASSERT(toPop > 0);
		while (toPop > 0) {
			inFlightFiles.pop_back();
			toPop--;
		}
		metadata->pendingDeltaVersion = cfRollbackVersion;
		if (BW_DEBUG) {
			fmt::print("[{0} - {1}) rollback discarding all {2} in-memory mutations",
			           metadata->keyRange.begin.printable(),
			           metadata->keyRange.end.printable(),
			           metadata->currentDeltas.size());
			if (metadata->currentDeltas.size()) {
				fmt::print(
				    " {0} - {1}", metadata->currentDeltas.front().version, metadata->currentDeltas.back().version);
			}
			fmt::print("\n");
		}

		// discard all in-memory mutations
		metadata->currentDeltas = Standalone<GranuleDeltas>();
		metadata->bufferedDeltaBytes = 0;
		metadata->bufferedDeltaVersion = cfRollbackVersion;

		// calculate number of bytes in durable delta files after last snapshot
		for (int i = metadata->files.deltaFiles.size() - 1;
		     i >= 0 && metadata->files.deltaFiles[i].version > metadata->pendingSnapshotVersion;
		     i--) {
			metadata->bytesInNewDeltaFiles += metadata->files.deltaFiles[i].logicalSize;
			metadata->newDeltaFileCount++;
		}

		// Track that this rollback happened, since we have to re-read mutations up to the rollback
		// Add this rollback to in progress, and put all completed ones back in progress
		rollbacksInProgress.push_back(std::pair(rollbackVersion, mutationVersion));
		while (!rollbacksCompleted.empty()) {
			if (rollbacksCompleted.back().first >= cfRollbackVersion) {
				rollbacksInProgress.push_front(rollbacksCompleted.back());
				rollbacksCompleted.pop_back();
			} else {
				// some rollbacks in completed could still have a delta file in flight after this rollback, they
				// should remain in completed
				break;
			}
		}

	} else {
		// No pending delta files to discard, just in-memory mutations
		CODE_PROBE(true, "Granule rollback discarding in memory mutations");

		// FIXME: could binary search?
		int mIdx = metadata->currentDeltas.size() - 1;
		Version firstDiscarded = invalidVersion;
		Version lastDiscarded = invalidVersion;
		while (mIdx >= 0) {
			if (metadata->currentDeltas[mIdx].version <= rollbackVersion) {
				break;
			}
			for (auto& m : metadata->currentDeltas[mIdx].mutations) {
				metadata->bufferedDeltaBytes -= m.totalSize();
			}
			if (firstDiscarded == invalidVersion) {
				firstDiscarded = metadata->currentDeltas[mIdx].version;
			}
			lastDiscarded = metadata->currentDeltas[mIdx].version;
			mIdx--;
		}

		if (BW_DEBUG) {
			fmt::print("[{0} - {1}) rollback discarding {2} in-memory mutations",
			           metadata->keyRange.begin.printable(),
			           metadata->keyRange.end.printable(),
			           metadata->currentDeltas.size() - mIdx - 1);

			if (firstDiscarded != invalidVersion) {
				fmt::print(" {0} - {1}", lastDiscarded, firstDiscarded);
			}

			fmt::print(", {0} mutations", mIdx);
			if (mIdx >= 0) {
				fmt::print(
				    " ({0} - {1})", metadata->currentDeltas.front().version, metadata->currentDeltas[mIdx].version);
			}
			fmt::print(" and {0} bytes left\n", metadata->bufferedDeltaBytes);
		}

		if (mIdx < 0) {
			metadata->currentDeltas = Standalone<GranuleDeltas>();
			metadata->bufferedDeltaBytes = 0;
		} else {
			metadata->currentDeltas.resize(metadata->currentDeltas.arena(), mIdx + 1);
		}

		// delete all deltas in rollback range, but we can optimize here to just skip the uncommitted mutations
		// directly and immediately pop the rollback out of inProgress to completed

		metadata->bufferedDeltaVersion = rollbackVersion;
		cfRollbackVersion = mutationVersion;
		rollbacksCompleted.push_back(std::pair(rollbackVersion, mutationVersion));
	}

	if (BW_DEBUG) {
		fmt::print("[{0} - {1}) finishing rollback to {2}\n",
		           metadata->keyRange.begin.printable(),
		           metadata->keyRange.end.printable(),
		           cfRollbackVersion);
	}

	return cfRollbackVersion;
}
} // namespace

															#line 16878 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via waitOnCFVersion()
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class WaitOnCFVersionActor>
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WaitOnCFVersionActorState {
															#line 16885 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WaitOnCFVersionActorState(Reference<GranuleMetadata> const& metadata,Version const& waitVersion) 
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : metadata(metadata),
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   waitVersion(waitVersion)
															#line 16894 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("waitOnCFVersion", reinterpret_cast<unsigned long>(this));

	}
	~WaitOnCFVersionActorState() 
	{
		fdb_probe_actor_destroy("waitOnCFVersion", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1970 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			;
															#line 16909 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitOnCFVersionActorState();
		static_cast<WaitOnCFVersionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1995 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(0, TaskPriority::BlobWorkerReadChangeFeed);
															#line 1995 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 16934 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state = 3;
															#line 1995 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WaitOnCFVersionActor, 3, Void >*>(static_cast<WaitOnCFVersionActor*>(this)));
															#line 16939 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1973 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			atLeast = metadata->activeCFData.get().isValid() ? metadata->activeCFData.get()->whenAtLeast(waitVersion) : Never();
															#line 1977 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_0 = atLeast;
															#line 1976 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16960 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 1980 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_1 = metadata->activeCFData.onChange();
															#line 16964 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state = 1;
															#line 1977 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitOnCFVersionActor, 0, Void >*>(static_cast<WaitOnCFVersionActor*>(this)));
															#line 1980 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitOnCFVersionActor, 1, Void >*>(static_cast<WaitOnCFVersionActor*>(this)));
															#line 16971 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1983 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (e.code() == error_code_operation_cancelled || e.code() == error_code_change_feed_popped)
															#line 17006 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 1984 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 17010 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
															#line 1990 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(0.05);
															#line 1990 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 17016 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state = 2;
															#line 1990 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WaitOnCFVersionActor, 2, Void >*>(static_cast<WaitOnCFVersionActor*>(this)));
															#line 17021 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state > 0) static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state = 0;
		static_cast<WaitOnCFVersionActor*>(this)->ActorCallback< WaitOnCFVersionActor, 0, Void >::remove();
		static_cast<WaitOnCFVersionActor*>(this)->ActorCallback< WaitOnCFVersionActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitOnCFVersionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitOnCFVersionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitOnCFVersionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitOnCFVersionActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitOnCFVersionActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitOnCFVersionActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state > 0) static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state = 0;
		static_cast<WaitOnCFVersionActor*>(this)->ActorCallback< WaitOnCFVersionActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitOnCFVersionActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WaitOnCFVersionActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WaitOnCFVersionActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1997 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WaitOnCFVersionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitOnCFVersionActorState(); static_cast<WaitOnCFVersionActor*>(this)->destroy(); return 0; }
															#line 17287 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WaitOnCFVersionActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitOnCFVersionActorState();
		static_cast<WaitOnCFVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1997 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WaitOnCFVersionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitOnCFVersionActorState(); static_cast<WaitOnCFVersionActor*>(this)->destroy(); return 0; }
															#line 17299 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WaitOnCFVersionActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitOnCFVersionActorState();
		static_cast<WaitOnCFVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state > 0) static_cast<WaitOnCFVersionActor*>(this)->actor_wait_state = 0;
		static_cast<WaitOnCFVersionActor*>(this)->ActorCallback< WaitOnCFVersionActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitOnCFVersionActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WaitOnCFVersionActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WaitOnCFVersionActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<GranuleMetadata> metadata;
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version waitVersion;
															#line 1973 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Future<Void> atLeast;
															#line 17388 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via waitOnCFVersion()
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WaitOnCFVersionActor final : public Actor<Void>, public ActorCallback< WaitOnCFVersionActor, 0, Void >, public ActorCallback< WaitOnCFVersionActor, 1, Void >, public ActorCallback< WaitOnCFVersionActor, 2, Void >, public ActorCallback< WaitOnCFVersionActor, 3, Void >, public FastAllocated<WaitOnCFVersionActor>, public WaitOnCFVersionActorState<WaitOnCFVersionActor> {
															#line 17393 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<WaitOnCFVersionActor>::operator new;
	using FastAllocated<WaitOnCFVersionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3920979010901223680UL, 9655714283518020352UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitOnCFVersionActor, 0, Void >;
friend struct ActorCallback< WaitOnCFVersionActor, 1, Void >;
friend struct ActorCallback< WaitOnCFVersionActor, 2, Void >;
friend struct ActorCallback< WaitOnCFVersionActor, 3, Void >;
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WaitOnCFVersionActor(Reference<GranuleMetadata> const& metadata,Version const& waitVersion) 
															#line 17413 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<Void>(),
		   WaitOnCFVersionActorState<WaitOnCFVersionActor>(metadata, waitVersion),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitOnCFVersion", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11824750818098250496UL, 7096571583594184448UL);
		ActorExecutionContextHelper __helper(static_cast<WaitOnCFVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitOnCFVersion");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitOnCFVersion", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitOnCFVersionActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitOnCFVersionActor, 2, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WaitOnCFVersionActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<Void> waitOnCFVersion( Reference<GranuleMetadata> const& metadata, Version const& waitVersion ) {
															#line 1969 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<Void>(new WaitOnCFVersionActor(metadata, waitVersion));
															#line 17448 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 1999 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

															#line 17453 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via waitCommittedGrv()
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class WaitCommittedGrvActor>
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WaitCommittedGrvActorState {
															#line 17460 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WaitCommittedGrvActorState(Reference<BlobWorkerData> const& bwData,Reference<GranuleMetadata> const& metadata,Version const& version) 
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   metadata(metadata),
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   version(version),
															#line 2003 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   grvVersion()
															#line 17473 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("waitCommittedGrv", reinterpret_cast<unsigned long>(this));

	}
	~WaitCommittedGrvActorState() 
	{
		fdb_probe_actor_destroy("waitCommittedGrv", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2004 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (version > bwData->grvVersion.get())
															#line 17488 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 2005 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				CODE_PROBE(true, "Using new GRV for delta file committed version");
															#line 2008 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				Future<Void> grvAtLeast = bwData->grvVersion.whenAtLeast(version);
															#line 2009 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				Promise<Void> doGrvCheck = bwData->doGRVCheck;
															#line 2010 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (doGrvCheck.canBeSet())
															#line 17498 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				{
															#line 2011 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					doGrvCheck.send(Void());
															#line 17502 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				}
															#line 2013 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				StrictFuture<Void> __when_expr_0 = grvAtLeast;
															#line 2013 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (static_cast<WaitCommittedGrvActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17508 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<WaitCommittedGrvActor*>(this)->actor_wait_state = 1;
															#line 2013 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitCommittedGrvActor, 0, Void >*>(static_cast<WaitCommittedGrvActor*>(this)));
															#line 17513 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 2016 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				CODE_PROBE(true, "Using previous GRV for delta file committed version");
															#line 2018 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				ASSERT(!bwData->prevGRVVersions.empty());
															#line 2019 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				auto nextLargestGRV = std::lower_bound(bwData->prevGRVVersions.begin(), bwData->prevGRVVersions.end(), version);
															#line 2021 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (nextLargestGRV == bwData->prevGRVVersions.end())
															#line 17526 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				{
															#line 2024 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					CODE_PROBE(true, "GRV difference less than min granularity");
															#line 2025 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					grvVersion = bwData->grvVersion.get();
															#line 17532 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				}
				else
				{
															#line 2027 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					grvVersion = *nextLargestGRV;
															#line 2028 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					if (nextLargestGRV != bwData->prevGRVVersions.begin())
															#line 17540 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
					{
															#line 2029 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
						ASSERT(*(nextLargestGRV - 1) < version);
															#line 17544 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
					}
				}
															#line 2035 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				Promise<Void> doGrvCheck = bwData->doGRVCheck;
															#line 2036 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (doGrvCheck.canBeSet())
															#line 17551 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				{
															#line 2037 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
					doGrvCheck.send(Void());
															#line 17555 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				}
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitCommittedGrvActorState();
		static_cast<WaitCommittedGrvActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2041 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		ASSERT(grvVersion >= version);
															#line 2045 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		Version waitVersion = std::min(grvVersion, version + SERVER_KNOBS->MAX_READ_TRANSACTION_LIFE_VERSIONS);
															#line 2047 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		StrictFuture<Void> __when_expr_1 = waitOnCFVersion(metadata, waitVersion);
															#line 2047 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (static_cast<WaitCommittedGrvActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17586 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WaitCommittedGrvActor*>(this)->actor_wait_state = 2;
															#line 2047 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitCommittedGrvActor, 1, Void >*>(static_cast<WaitCommittedGrvActor*>(this)));
															#line 17591 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2014 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		grvVersion = bwData->grvVersion.get();
															#line 17600 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2014 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		grvVersion = bwData->grvVersion.get();
															#line 17609 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitCommittedGrvActor*>(this)->actor_wait_state > 0) static_cast<WaitCommittedGrvActor*>(this)->actor_wait_state = 0;
		static_cast<WaitCommittedGrvActor*>(this)->ActorCallback< WaitCommittedGrvActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitCommittedGrvActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitCommittedGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitCommittedGrvActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitCommittedGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitCommittedGrvActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitCommittedGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
															#line 2048 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WaitCommittedGrvActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitCommittedGrvActorState(); static_cast<WaitCommittedGrvActor*>(this)->destroy(); return 0; }
															#line 17693 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WaitCommittedGrvActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitCommittedGrvActorState();
		static_cast<WaitCommittedGrvActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
															#line 2048 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WaitCommittedGrvActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitCommittedGrvActorState(); static_cast<WaitCommittedGrvActor*>(this)->destroy(); return 0; }
															#line 17705 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WaitCommittedGrvActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitCommittedGrvActorState();
		static_cast<WaitCommittedGrvActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WaitCommittedGrvActor*>(this)->actor_wait_state > 0) static_cast<WaitCommittedGrvActor*>(this)->actor_wait_state = 0;
		static_cast<WaitCommittedGrvActor*>(this)->ActorCallback< WaitCommittedGrvActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitCommittedGrvActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitCommittedGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitCommittedGrvActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitCommittedGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitCommittedGrvActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitCommittedGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitCommittedGrv", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<GranuleMetadata> metadata;
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version version;
															#line 2003 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version grvVersion;
															#line 17796 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via waitCommittedGrv()
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WaitCommittedGrvActor final : public Actor<Void>, public ActorCallback< WaitCommittedGrvActor, 0, Void >, public ActorCallback< WaitCommittedGrvActor, 1, Void >, public FastAllocated<WaitCommittedGrvActor>, public WaitCommittedGrvActorState<WaitCommittedGrvActor> {
															#line 17801 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<WaitCommittedGrvActor>::operator new;
	using FastAllocated<WaitCommittedGrvActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17276741985180396032UL, 11890950064771378432UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitCommittedGrvActor, 0, Void >;
friend struct ActorCallback< WaitCommittedGrvActor, 1, Void >;
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WaitCommittedGrvActor(Reference<BlobWorkerData> const& bwData,Reference<GranuleMetadata> const& metadata,Version const& version) 
															#line 17819 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<Void>(),
		   WaitCommittedGrvActorState<WaitCommittedGrvActor>(bwData, metadata, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitCommittedGrv", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4029656559626629632UL, 2863118724301117696UL);
		ActorExecutionContextHelper __helper(static_cast<WaitCommittedGrvActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitCommittedGrv");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitCommittedGrv", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitCommittedGrvActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WaitCommittedGrvActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<Void> waitCommittedGrv( Reference<BlobWorkerData> const& bwData, Reference<GranuleMetadata> const& metadata, Version const& version ) {
															#line 2000 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<Void>(new WaitCommittedGrvActor(bwData, metadata, version));
															#line 17853 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 2050 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

															#line 17858 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via waitVersionCommitted()
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class WaitVersionCommittedActor>
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WaitVersionCommittedActorState {
															#line 17865 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WaitVersionCommittedActorState(Reference<BlobWorkerData> const& bwData,Reference<GranuleMetadata> const& metadata,Version const& version) 
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   metadata(metadata),
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   version(version)
															#line 17876 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("waitVersionCommitted", reinterpret_cast<unsigned long>(this));

	}
	~WaitVersionCommittedActorState() 
	{
		fdb_probe_actor_destroy("waitVersionCommitted", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2054 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_0 = waitCommittedGrv(bwData, metadata, version);
															#line 2054 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<WaitVersionCommittedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17893 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WaitVersionCommittedActor*>(this)->actor_wait_state = 1;
															#line 2054 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitVersionCommittedActor, 0, Void >*>(static_cast<WaitVersionCommittedActor*>(this)));
															#line 17898 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitVersionCommittedActorState();
		static_cast<WaitVersionCommittedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2055 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (version > metadata->knownCommittedVersion)
															#line 17921 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 2056 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			metadata->knownCommittedVersion = version;
															#line 17925 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 2058 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WaitVersionCommittedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitVersionCommittedActorState(); static_cast<WaitVersionCommittedActor*>(this)->destroy(); return 0; }
															#line 17929 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WaitVersionCommittedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitVersionCommittedActorState();
		static_cast<WaitVersionCommittedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2055 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (version > metadata->knownCommittedVersion)
															#line 17941 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 2056 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			metadata->knownCommittedVersion = version;
															#line 17945 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		}
															#line 2058 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<WaitVersionCommittedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WaitVersionCommittedActorState(); static_cast<WaitVersionCommittedActor*>(this)->destroy(); return 0; }
															#line 17949 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<WaitVersionCommittedActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WaitVersionCommittedActorState();
		static_cast<WaitVersionCommittedActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitVersionCommittedActor*>(this)->actor_wait_state > 0) static_cast<WaitVersionCommittedActor*>(this)->actor_wait_state = 0;
		static_cast<WaitVersionCommittedActor*>(this)->ActorCallback< WaitVersionCommittedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitVersionCommittedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitVersionCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitVersionCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitVersionCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitVersionCommittedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitVersionCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitVersionCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitVersionCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitVersionCommittedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitVersionCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitVersionCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitVersionCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<GranuleMetadata> metadata;
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version version;
															#line 18038 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via waitVersionCommitted()
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class WaitVersionCommittedActor final : public Actor<Void>, public ActorCallback< WaitVersionCommittedActor, 0, Void >, public FastAllocated<WaitVersionCommittedActor>, public WaitVersionCommittedActorState<WaitVersionCommittedActor> {
															#line 18043 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<WaitVersionCommittedActor>::operator new;
	using FastAllocated<WaitVersionCommittedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8231676468722324480UL, 14000238334340047616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitVersionCommittedActor, 0, Void >;
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	WaitVersionCommittedActor(Reference<BlobWorkerData> const& bwData,Reference<GranuleMetadata> const& metadata,Version const& version) 
															#line 18060 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<Void>(),
		   WaitVersionCommittedActorState<WaitVersionCommittedActor>(bwData, metadata, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitVersionCommitted", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12490808501786351616UL, 852123515776179200UL);
		ActorExecutionContextHelper __helper(static_cast<WaitVersionCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitVersionCommitted");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitVersionCommitted", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitVersionCommittedActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<Void> waitVersionCommitted( Reference<BlobWorkerData> const& bwData, Reference<GranuleMetadata> const& metadata, Version const& version ) {
															#line 2051 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<Void>(new WaitVersionCommittedActor(bwData, metadata, version));
															#line 18093 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 2060 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

															#line 18098 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via checkFileNotFoundForcePurgeRace()
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class CheckFileNotFoundForcePurgeRaceActor>
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class CheckFileNotFoundForcePurgeRaceActorState {
															#line 18105 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	CheckFileNotFoundForcePurgeRaceActorState(Reference<BlobWorkerData> const& bwData,KeyRange const& range) 
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   range(range),
															#line 2062 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   tr(bwData->db)
															#line 18116 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this));

	}
	~CheckFileNotFoundForcePurgeRaceActorState() 
	{
		fdb_probe_actor_destroy("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2063 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			;
															#line 18131 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckFileNotFoundForcePurgeRaceActorState();
		static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2065 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2066 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2067 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<ForcedPurgeState> __when_expr_0 = getForcePurgedState(&tr, range);
															#line 2067 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 18168 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->actor_wait_state = 1;
															#line 2067 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 0, ForcedPurgeState >*>(static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)));
															#line 18173 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2070 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 2070 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 18197 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->actor_wait_state = 2;
															#line 2070 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 1, Void >*>(static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)));
															#line 18202 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(ForcedPurgeState const& purgeState,int loopDepth) 
	{
															#line 2068 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->SAV<bool>::futures) { (void)(purgeState != ForcedPurgeState::NonePurged); this->~CheckFileNotFoundForcePurgeRaceActorState(); static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->destroy(); return 0; }
															#line 18217 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->SAV< bool >::value()) bool(purgeState != ForcedPurgeState::NonePurged);
		this->~CheckFileNotFoundForcePurgeRaceActorState();
		static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(ForcedPurgeState && purgeState,int loopDepth) 
	{
															#line 2068 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->SAV<bool>::futures) { (void)(purgeState != ForcedPurgeState::NonePurged); this->~CheckFileNotFoundForcePurgeRaceActorState(); static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->destroy(); return 0; }
															#line 18229 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->SAV< bool >::value()) bool(purgeState != ForcedPurgeState::NonePurged);
		this->~CheckFileNotFoundForcePurgeRaceActorState();
		static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(ForcedPurgeState const& purgeState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(purgeState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(ForcedPurgeState && purgeState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(purgeState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->actor_wait_state > 0) static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->actor_wait_state = 0;
		static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 0, ForcedPurgeState >::remove();

	}
	void a_callback_fire(ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 0, ForcedPurgeState >*,ForcedPurgeState const& value) 
	{
		fdb_probe_actor_enter("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 0, ForcedPurgeState >*,ForcedPurgeState && value) 
	{
		fdb_probe_actor_enter("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 0, ForcedPurgeState >*,Error err) 
	{
		fdb_probe_actor_enter("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->actor_wait_state > 0) static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->actor_wait_state = 0;
		static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	KeyRange range;
															#line 2062 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Transaction tr;
															#line 18405 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via checkFileNotFoundForcePurgeRace()
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class CheckFileNotFoundForcePurgeRaceActor final : public Actor<bool>, public ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 0, ForcedPurgeState >, public ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 1, Void >, public FastAllocated<CheckFileNotFoundForcePurgeRaceActor>, public CheckFileNotFoundForcePurgeRaceActorState<CheckFileNotFoundForcePurgeRaceActor> {
															#line 18410 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<CheckFileNotFoundForcePurgeRaceActor>::operator new;
	using FastAllocated<CheckFileNotFoundForcePurgeRaceActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7572801710499968256UL, 3921608244231326208UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 0, ForcedPurgeState >;
friend struct ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 1, Void >;
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	CheckFileNotFoundForcePurgeRaceActor(Reference<BlobWorkerData> const& bwData,KeyRange const& range) 
															#line 18428 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<bool>(),
		   CheckFileNotFoundForcePurgeRaceActorState<CheckFileNotFoundForcePurgeRaceActor>(bwData, range),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5569177106452273408UL, 3698191113853988864UL);
		ActorExecutionContextHelper __helper(static_cast<CheckFileNotFoundForcePurgeRaceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkFileNotFoundForcePurgeRace");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkFileNotFoundForcePurgeRace", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 0, ForcedPurgeState >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckFileNotFoundForcePurgeRaceActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<bool> checkFileNotFoundForcePurgeRace( Reference<BlobWorkerData> const& bwData, KeyRange const& range ) {
															#line 2061 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<bool>(new CheckFileNotFoundForcePurgeRaceActor(bwData, range));
															#line 18462 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 2074 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

// Does a force flush after consuming all data post-split from the parent granule's old change feed.
// This guarantees that all of the data from the old change feed gets persisted to blob storage, and the old feed can be
// cleaned up. This is particularly necessary for sequential workloads, where only one child granule after the split has
// new writes. Adds a delay so that, if the granule is not write-cold and would have written a delta file soon anyway,
// this does not add any extra overhead.
															#line 18472 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via forceFlushCleanup()
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class ForceFlushCleanupActor>
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class ForceFlushCleanupActorState {
															#line 18479 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	ForceFlushCleanupActorState(Reference<BlobWorkerData> const& bwData,Reference<GranuleMetadata> const& metadata,Version const& v) 
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   metadata(metadata),
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   v(v)
															#line 18490 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("forceFlushCleanup", reinterpret_cast<unsigned long>(this));

	}
	~ForceFlushCleanupActorState() 
	{
		fdb_probe_actor_destroy("forceFlushCleanup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2081 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			double cleanupDelay = SERVER_KNOBS->BLOB_WORKER_FORCE_FLUSH_CLEANUP_DELAY;
															#line 2082 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (cleanupDelay < 0)
															#line 18507 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 2083 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (!static_cast<ForceFlushCleanupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ForceFlushCleanupActorState(); static_cast<ForceFlushCleanupActor*>(this)->destroy(); return 0; }
															#line 18511 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
				new (&static_cast<ForceFlushCleanupActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~ForceFlushCleanupActorState();
				static_cast<ForceFlushCleanupActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2085 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(cleanupDelay);
															#line 2085 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (static_cast<ForceFlushCleanupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18521 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ForceFlushCleanupActor*>(this)->actor_wait_state = 1;
															#line 2085 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ForceFlushCleanupActor, 0, Void >*>(static_cast<ForceFlushCleanupActor*>(this)));
															#line 18526 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ForceFlushCleanupActorState();
		static_cast<ForceFlushCleanupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2086 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (metadata->forceFlushVersion.get() < v && metadata->pendingDeltaVersion < v)
															#line 18549 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 2087 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			metadata->forceFlushVersion.set(v);
															#line 2088 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			++bwData->stats.forceFlushCleanups;
															#line 2089 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (BW_DEBUG)
															#line 18557 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 2090 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				fmt::print("Granule [{0} - {1}) forcing flush cleanup @ {2}\n", metadata->keyRange.begin.printable(), metadata->keyRange.end.printable(), v);
															#line 18561 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
		}
															#line 2096 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<ForceFlushCleanupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ForceFlushCleanupActorState(); static_cast<ForceFlushCleanupActor*>(this)->destroy(); return 0; }
															#line 18566 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<ForceFlushCleanupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ForceFlushCleanupActorState();
		static_cast<ForceFlushCleanupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2086 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (metadata->forceFlushVersion.get() < v && metadata->pendingDeltaVersion < v)
															#line 18578 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		{
															#line 2087 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			metadata->forceFlushVersion.set(v);
															#line 2088 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			++bwData->stats.forceFlushCleanups;
															#line 2089 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (BW_DEBUG)
															#line 18586 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 2090 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				fmt::print("Granule [{0} - {1}) forcing flush cleanup @ {2}\n", metadata->keyRange.begin.printable(), metadata->keyRange.end.printable(), v);
															#line 18590 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			}
		}
															#line 2096 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		if (!static_cast<ForceFlushCleanupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ForceFlushCleanupActorState(); static_cast<ForceFlushCleanupActor*>(this)->destroy(); return 0; }
															#line 18595 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		new (&static_cast<ForceFlushCleanupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ForceFlushCleanupActorState();
		static_cast<ForceFlushCleanupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ForceFlushCleanupActor*>(this)->actor_wait_state > 0) static_cast<ForceFlushCleanupActor*>(this)->actor_wait_state = 0;
		static_cast<ForceFlushCleanupActor*>(this)->ActorCallback< ForceFlushCleanupActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ForceFlushCleanupActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("forceFlushCleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceFlushCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceFlushCleanup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ForceFlushCleanupActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("forceFlushCleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceFlushCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceFlushCleanup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ForceFlushCleanupActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("forceFlushCleanup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ForceFlushCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceFlushCleanup", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<BlobWorkerData> bwData;
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Reference<GranuleMetadata> metadata;
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	Version v;
															#line 18684 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
};
// This generated class is to be used only via forceFlushCleanup()
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class ForceFlushCleanupActor final : public Actor<Void>, public ActorCallback< ForceFlushCleanupActor, 0, Void >, public FastAllocated<ForceFlushCleanupActor>, public ForceFlushCleanupActorState<ForceFlushCleanupActor> {
															#line 18689 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
	using FastAllocated<ForceFlushCleanupActor>::operator new;
	using FastAllocated<ForceFlushCleanupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7176298412597376512UL, 6818130259282557696UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ForceFlushCleanupActor, 0, Void >;
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	ForceFlushCleanupActor(Reference<BlobWorkerData> const& bwData,Reference<GranuleMetadata> const& metadata,Version const& v) 
															#line 18706 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
		 : Actor<Void>(),
		   ForceFlushCleanupActorState<ForceFlushCleanupActor>(bwData, metadata, v),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("forceFlushCleanup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4037263273277538304UL, 7432629704297548544UL);
		ActorExecutionContextHelper __helper(static_cast<ForceFlushCleanupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("forceFlushCleanup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("forceFlushCleanup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ForceFlushCleanupActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
[[nodiscard]] Future<Void> forceFlushCleanup( Reference<BlobWorkerData> const& bwData, Reference<GranuleMetadata> const& metadata, Version const& v ) {
															#line 2080 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	return Future<Void>(new ForceFlushCleanupActor(bwData, metadata, v));
															#line 18739 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
}

#line 2098 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"

// We have a normal target snapshot/delta ratio, but in write-heavy cases where are behind and need to catch up,
// we want to change this target to get more efficient. To maintain a consistent write amp if the snapshot size is
// growing without the ability to split, we scale the bytes before recompact to reach the target write amp. We also
// decrease the target write amp the further behind we are to be more efficient.
struct WriteAmpTarget {
	double targetWriteAmp;
	int bytesBeforeCompact;
	int targetSnapshotBytes;
	bool catchingUp;

	WriteAmpTarget() { reset(); }

	void reset() {
		catchingUp = false;
		bytesBeforeCompact = SERVER_KNOBS->BG_DELTA_BYTES_BEFORE_COMPACT;
		targetSnapshotBytes = SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES;
		targetWriteAmp = 1.0 * (bytesBeforeCompact + targetSnapshotBytes) / bytesBeforeCompact;
	}

	void decrease(int deltaBytes) {
		if (SERVER_KNOBS->BG_ENABLE_DYNAMIC_WRITE_AMP) {
			catchingUp = true;
			double minWriteAmp =
			    SERVER_KNOBS->BG_DYNAMIC_WRITE_AMP_MIN_FACTOR *
			    (SERVER_KNOBS->BG_DELTA_BYTES_BEFORE_COMPACT + SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES) /
			    SERVER_KNOBS->BG_DELTA_BYTES_BEFORE_COMPACT;
			targetWriteAmp = std::max(targetWriteAmp * SERVER_KNOBS->BG_DYNAMIC_WRITE_AMP_DECREASE_FACTOR, minWriteAmp);
			// To not wait for next re-snapshot, pessimistically assume that all deltas are inserts and new snapshot
			// will be bigger. This is usually the case in write-heavy catchup cases, but helps us catch up faster
			// regardless
			targetSnapshotBytes += deltaBytes;
			bytesBeforeCompact = targetSnapshotBytes / (targetWriteAmp - 1.0);
		}
	}

	void newSnapshotSize(int snapshotSize) {
		if (SERVER_KNOBS->BG_ENABLE_DYNAMIC_WRITE_AMP) {
			snapshotSize = std::max(snapshotSize, SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES);
			if (catchingUp) {
				// if catching up, always aim for larger snapshot cycles to help catch up faster
				snapshotSize = std::max(targetSnapshotBytes, snapshotSize);
			}
			targetSnapshotBytes = snapshotSize;
			bytesBeforeCompact = targetSnapshotBytes / (targetWriteAmp - 1.0);
		}
	}

	int getDeltaFileBytes() { return getBytesBeforeCompact() / 10; }

	int getBytesBeforeCompact() { return bytesBeforeCompact; }
};

															#line 18796 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
namespace {
// This generated class is to be used only via getTenantPrefix()
															#line 2151 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
template <class GetTenantPrefixActor>
															#line 2151 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
class GetTenantPrefixActorState {
															#line 18803 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
public:
															#line 2151 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
	GetTenantPrefixActorState(Reference<BlobWorkerData> const& bwData,KeyRange const& keyRange) 
															#line 2151 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
															#line 2151 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		 : bwData(bwData),
															#line 2151 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   keyRange(keyRange),
															#line 2152 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
		   retryCount(0)
															#line 18814 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
	{
		fdb_probe_actor_create("getTenantPrefix", reinterpret_cast<unsigned long>(this));

	}
	~GetTenantPrefixActorState() 
	{
		fdb_probe_actor_destroy("getTenantPrefix", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2153 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
			if (SERVER_KNOBS->BG_METADATA_SOURCE != "tenant")
															#line 18829 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
			{
															#line 2154 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/fdbserver/BlobWorker.actor.cpp"
				if (!static_cast<GetTenantPrefixActor*>(this)->SAV<Key>::futures) { (void)(Key()); this->~GetTenantPrefixActorState(); static_cast<GetTenantPrefixActor*>(this)->destroy(); return 0; }
															#line 18833 "/codebuild/output/src4221181136/src/github.com/apple/foundationdb/build_output/fdbserver/BlobWorker.actor.g.cpp"
