#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
/*
 * GrvTransactionRateInfo.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbserver/GrvTransactionRateInfo.h"

#include "fdbserver/Knobs.h"
#include "flow/UnitTest.h"
#include "flow/actorcompiler.h" // must be last include

GrvTransactionRateInfo::GrvTransactionRateInfo(double rateWindow, double maxEmptyQueueBudget, double rate)
  : rateWindow(rateWindow), maxEmptyQueueBudget(maxEmptyQueueBudget), rate(rate), smoothRate(rateWindow),
    smoothReleased(rateWindow) {
	smoothRate.setTotal(rate);
}

bool GrvTransactionRateInfo::canStart(int64_t numAlreadyStarted, int64_t count) const {
	return numAlreadyStarted + count <=
	       std::min(limit + budget, SERVER_KNOBS->START_TRANSACTION_MAX_TRANSACTIONS_TO_START);
}

void GrvTransactionRateInfo::endReleaseWindow(int64_t numStarted, bool queueEmpty, double elapsed) {
	// Update the budget to accumulate any extra capacity available or remove any excess that was used.
	// The actual delta is the portion of the limit we didn't use multiplied by the fraction of the rate window that
	// elapsed.
	//
	// We may have exceeded our limit due to the budget or because of higher priority transactions, in which case
	// this delta will be negative. The delta can also be negative in the event that our limit was negative, which
	// can happen if we had already started more transactions in our rate window than our rate would have allowed.
	//
	// This budget has the property that when the budget is required to start transactions (because batches are
	// big), the sum limit+budget will increase linearly from 0 to the batch size over time and decrease by the
	// batch size upon starting a batch. In other words, this works equivalently to a model where we linearly
	// accumulate budget over time in the case that our batches are too big to take advantage of the rate window based
	// limits.
	//
	// Note that "rate window" here indicates a period of rateWindow seconds,
	// whereas "release window" is the period between wait statements, with duration indicated by "elapsed."
	budget = std::max(0.0, budget + elapsed * (limit - numStarted) / rateWindow);

	// If we are emptying out the queue of requests, then we don't need to carry much budget forward
	// If we did keep accumulating budget, then our responsiveness to changes in workflow could be compromised
	if (queueEmpty) {
		budget = std::min(budget, maxEmptyQueueBudget);
	}

	smoothReleased.addDelta(numStarted);
}

void GrvTransactionRateInfo::disable() {
	disabled = true;
	// Use smoothRate.setTotal(0) instead of setting rate to 0 so txns will not be throttled immediately.
	smoothRate.setTotal(0);
}

void GrvTransactionRateInfo::setRate(double rate) {
	ASSERT(rate >= 0 && rate != std::numeric_limits<double>::infinity() && !std::isnan(rate));

	this->rate = rate;
	if (disabled) {
		smoothRate.reset(rate);
		disabled = false;
	} else {
		smoothRate.setTotal(rate);
	}
}

void GrvTransactionRateInfo::startReleaseWindow() {
	// Determine the number of transactions that this proxy is allowed to release
	// Roughly speaking, this is done by computing the number of transactions over some historical window that we
	// could have started but didn't, and making that our limit. More precisely, we track a smoothed rate limit and
	// release rate, the difference of which is the rate of additional transactions that we could have released
	// based on that window. Then we multiply by the window size to get a number of transactions.
	//
	// Limit can be negative in the event that we are releasing more transactions than we are allowed (due to the
	// use of our budget or because of higher priority transactions).
	double releaseRate = smoothRate.smoothTotal() - smoothReleased.smoothRate();
	limit = rateWindow * releaseRate;
}

static bool isNear(double desired, int64_t actual) {
	return std::abs(desired - actual) * 10 < desired;
}

															#line 103 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
namespace {
// This generated class is to be used only via mockClient()
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
template <class MockClientActor>
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
class MockClientActorState {
															#line 110 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
public:
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	MockClientActorState(GrvTransactionRateInfo* const& rateInfo,double const& desiredRate,int64_t* const& counter) 
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		 : rateInfo(rateInfo),
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		   desiredRate(desiredRate),
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		   counter(counter)
															#line 121 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
	{
		fdb_probe_actor_create("mockClient", reinterpret_cast<unsigned long>(this));

	}
	~MockClientActorState() 
	{
		fdb_probe_actor_destroy("mockClient", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 102 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
			;
															#line 136 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MockClientActorState();
		static_cast<MockClientActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 103 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		elapsed = (0.9 + 0.2 * deterministicRandom()->random01()) / desiredRate;
															#line 104 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(elapsed);
															#line 104 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		if (static_cast<MockClientActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 170 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MockClientActor*>(this)->actor_wait_state = 1;
															#line 104 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MockClientActor, 0, Void >*>(static_cast<MockClientActor*>(this)));
															#line 175 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 105 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		rateInfo->startReleaseWindow();
															#line 106 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		int started = rateInfo->canStart(0, 1) ? 1 : 0;
															#line 107 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		*counter += started;
															#line 108 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		rateInfo->endReleaseWindow(started, false, elapsed);
															#line 190 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 105 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		rateInfo->startReleaseWindow();
															#line 106 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		int started = rateInfo->canStart(0, 1) ? 1 : 0;
															#line 107 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		*counter += started;
															#line 108 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		rateInfo->endReleaseWindow(started, false, elapsed);
															#line 205 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MockClientActor*>(this)->actor_wait_state > 0) static_cast<MockClientActor*>(this)->actor_wait_state = 0;
		static_cast<MockClientActor*>(this)->ActorCallback< MockClientActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MockClientActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("mockClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MockClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MockClientActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("mockClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MockClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MockClientActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("mockClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MockClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockClient", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	GrvTransactionRateInfo* rateInfo;
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	double desiredRate;
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	int64_t* counter;
															#line 103 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	double elapsed;
															#line 293 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
};
// This generated class is to be used only via mockClient()
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
class MockClientActor final : public Actor<Void>, public ActorCallback< MockClientActor, 0, Void >, public FastAllocated<MockClientActor>, public MockClientActorState<MockClientActor> {
															#line 298 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
public:
	using FastAllocated<MockClientActor>::operator new;
	using FastAllocated<MockClientActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16974455123693436928UL, 16666352862627273472UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MockClientActor, 0, Void >;
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	MockClientActor(GrvTransactionRateInfo* const& rateInfo,double const& desiredRate,int64_t* const& counter) 
															#line 315 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
		 : Actor<Void>(),
		   MockClientActorState<MockClientActor>(rateInfo, desiredRate, counter),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("mockClient", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7285660657920109568UL, 11812854688363359744UL);
		ActorExecutionContextHelper __helper(static_cast<MockClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("mockClient");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("mockClient", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MockClientActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
[[nodiscard]] static Future<Void> mockClient( GrvTransactionRateInfo* const& rateInfo, double const& desiredRate, int64_t* const& counter ) {
															#line 101 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	return Future<Void>(new MockClientActor(rateInfo, desiredRate, counter));
															#line 348 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
}

#line 111 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"

// Rate limit set at 10, but client attempts 20 transactions per second.
// Client should be throttled to only 10 transactions per second.
															#line 355 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase114()
															#line 114 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
template <class FlowTestCase114Actor>
															#line 114 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
class FlowTestCase114ActorState {
															#line 362 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
public:
															#line 114 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	FlowTestCase114ActorState(UnitTestParameters const& params) 
															#line 114 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
															#line 114 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		 : params(params),
															#line 115 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		   rateInfo(2.0, 100, 10),
															#line 116 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		   counter()
															#line 373 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase114", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase114ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase114", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 117 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
			rateInfo.setRate(10.0);
															#line 118 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeout(mockClient(&rateInfo, 20.0, &counter), 60.0, Void());
															#line 118 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
			if (static_cast<FlowTestCase114Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 392 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase114Actor*>(this)->actor_wait_state = 1;
															#line 118 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase114Actor, 0, Void >*>(static_cast<FlowTestCase114Actor*>(this)));
															#line 397 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase114ActorState();
		static_cast<FlowTestCase114Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 119 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		TraceEvent("GrvTransactionRateInfoTest").detail("Counter", counter);
															#line 120 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		ASSERT(isNear(60.0 * 10.0, counter));
															#line 121 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		if (!static_cast<FlowTestCase114Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase114ActorState(); static_cast<FlowTestCase114Actor*>(this)->destroy(); return 0; }
															#line 424 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
		new (&static_cast<FlowTestCase114Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase114ActorState();
		static_cast<FlowTestCase114Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 119 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		TraceEvent("GrvTransactionRateInfoTest").detail("Counter", counter);
															#line 120 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		ASSERT(isNear(60.0 * 10.0, counter));
															#line 121 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
		if (!static_cast<FlowTestCase114Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase114ActorState(); static_cast<FlowTestCase114Actor*>(this)->destroy(); return 0; }
															#line 440 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
		new (&static_cast<FlowTestCase114Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase114ActorState();
		static_cast<FlowTestCase114Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase114Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase114Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase114Actor*>(this)->ActorCallback< FlowTestCase114Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase114Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase114", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase114Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase114", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase114Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase114", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase114Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase114", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase114Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase114", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase114Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase114", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 114 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	UnitTestParameters params;
															#line 115 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	GrvTransactionRateInfo rateInfo;
															#line 116 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	int64_t counter;
															#line 529 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase114()
															#line 114 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
class FlowTestCase114Actor final : public Actor<Void>, public ActorCallback< FlowTestCase114Actor, 0, Void >, public FastAllocated<FlowTestCase114Actor>, public FlowTestCase114ActorState<FlowTestCase114Actor> {
															#line 534 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase114Actor>::operator new;
	using FastAllocated<FlowTestCase114Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17124822930661256448UL, 4207475078422787328UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase114Actor, 0, Void >;
															#line 114 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	FlowTestCase114Actor(UnitTestParameters const& params) 
															#line 551 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase114ActorState<FlowTestCase114Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase114", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10495338519304545024UL, 11630361558155599360UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase114Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase114");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase114", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase114Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 114 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
static Future<Void> flowTestCase114( UnitTestParameters const& params ) {
															#line 114 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
	return Future<Void>(new FlowTestCase114Actor(params));
															#line 584 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/build_output/fdbserver/GrvTransactionRateInfo.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase114, "/GrvTransactionRateInfo/Simple")

#line 123 "/codebuild/output/src517104744/src/github.com/apple/foundationdb/fdbserver/GrvTransactionRateInfo.actor.cpp"
