#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
/*
 * ClientTransactionProfileCorrectness.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbserver/workloads/workloads.actor.h"
#include "fdbserver/ServerDBInfo.h"
#include "fdbclient/GlobalConfig.actor.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/RunRYWTransaction.actor.h"
#include "fdbclient/Tuple.h"
#include "flow/actorcompiler.h" // has to be last include

static const Key CLIENT_LATENCY_INFO_PREFIX = "client_latency/"_sr;
static const Key CLIENT_LATENCY_INFO_CTR_PREFIX = "client_latency_counter/"_sr;

/*
FF               - 2 bytes \xff\x02
SSSSSSSSSS       - 10 bytes Version Stamp
RRRRRRRRRRRRRRRR - 16 bytes Transaction id
NNNN             - 4 Bytes Chunk number (Big Endian)
TTTT             - 4 Bytes Total number of chunks (Big Endian)
XXXX             - Variable length user provided transaction identifier
*/
StringRef sampleTrInfoKey = "\xff\x02/fdbClientInfo/client_latency/SSSSSSSSSS/RRRRRRRRRRRRRRRR/NNNNTTTT/XXXX/"_sr;
static const auto chunkNumStartIndex = sampleTrInfoKey.toString().find('N');
static const auto numChunksStartIndex = sampleTrInfoKey.toString().find('T');
static const int chunkFormatSize = 4;
static const auto trIdStartIndex = sampleTrInfoKey.toString().find('R');
static const int trIdFormatSize = 16;

namespace ClientLogEventsParser {

void parseEventGetVersion(BinaryReader& reader) {
	FdbClientLogEvents::EventGetVersion gv;
	reader >> gv;
	ASSERT(gv.latency < 10000);
}

void parseEventGetVersion_V2(BinaryReader& reader) {
	FdbClientLogEvents::EventGetVersion_V2 gv;
	reader >> gv;
	ASSERT(gv.latency < 10000);
}

void parseEventGetVersion_V3(BinaryReader& reader) {
	FdbClientLogEvents::EventGetVersion_V3 gv;
	reader >> gv;
	ASSERT(gv.latency < 10000);
	ASSERT(gv.readVersion > 0);
}

void parseEventGet(BinaryReader& reader) {
	FdbClientLogEvents::EventGet g;
	reader >> g;
	ASSERT(g.latency < 10000 && g.valueSize < CLIENT_KNOBS->VALUE_SIZE_LIMIT &&
	       g.key.size() < CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
}

void parseEventGetRange(BinaryReader& reader) {
	FdbClientLogEvents::EventGetRange gr;
	reader >> gr;
	ASSERT(gr.latency < 10000 && gr.rangeSize < 1000000000 &&
	       gr.startKey.size() < CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT &&
	       gr.endKey.size() < CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
}

void parseEventCommit(BinaryReader& reader) {
	FdbClientLogEvents::EventCommit c;
	reader >> c;
	ASSERT(c.latency < 10000 && c.commitBytes < CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT && c.numMutations < 1000000);
}

void parseEventCommit_V2(BinaryReader& reader) {
	FdbClientLogEvents::EventCommit_V2 c;
	reader >> c;
	ASSERT(c.latency < 10000 && c.commitBytes < CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT && c.numMutations < 1000000 &&
	       c.commitVersion > 0);
}

void parseEventErrorGet(BinaryReader& reader) {
	FdbClientLogEvents::EventGetError ge;
	reader >> ge;
	ASSERT(ge.errCode < 10000 && ge.key.size() < CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
}

void parseEventErrorGetRange(BinaryReader& reader) {
	FdbClientLogEvents::EventGetRangeError gre;
	reader >> gre;
	ASSERT(gre.errCode < 10000 && gre.startKey.size() < CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT &&
	       gre.endKey.size() < CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
}

void parseEventErrorCommit(BinaryReader& reader) {
	FdbClientLogEvents::EventCommitError ce;
	reader >> ce;
	ASSERT(ce.errCode < 10000);
}

struct ParserBase {
	std::function<void(BinaryReader&)> parseGetVersion = parseEventGetVersion;
	std::function<void(BinaryReader&)> parseGet = parseEventGet;
	std::function<void(BinaryReader&)> parseGetRange = parseEventGetRange;
	std::function<void(BinaryReader&)> parseCommit = parseEventCommit;
	std::function<void(BinaryReader&)> parseErrorGet = parseEventErrorGet;
	std::function<void(BinaryReader&)> parseErrorGetRange = parseEventErrorGetRange;
	std::function<void(BinaryReader&)> parseErrorCommit = parseEventErrorCommit;
	virtual ~ParserBase() = 0;
};
ParserBase::~ParserBase() {}

struct Parser_V1 : ParserBase {
	~Parser_V1() override {}
};
struct Parser_V2 : ParserBase {
	Parser_V2() { parseGetVersion = parseEventGetVersion_V2; }
	~Parser_V2() override {}
};
struct Parser_V3 : ParserBase {
	Parser_V3() {
		parseGetVersion = parseEventGetVersion_V3;
		parseCommit = parseEventCommit_V2;
	}
	~Parser_V3() override {}
};

struct ParserFactory {
	static std::unique_ptr<ParserBase> getParser(ProtocolVersion version) {
		if (version.version() >= (uint64_t)0x0FDB00B063010001LL) {
			return std::unique_ptr<ParserBase>(new Parser_V3());
		} else if (version.version() >= (uint64_t)0x0FDB00B062000001LL) {
			return std::unique_ptr<ParserBase>(new Parser_V2());
		} else {
			return std::unique_ptr<ParserBase>(new Parser_V1());
		}
	}
};
}; // namespace ClientLogEventsParser

// Checks TransactionInfo format
bool checkTxInfoEntryFormat(BinaryReader& reader) {
	// Check protocol version
	ProtocolVersion protocolVersion;
	reader >> protocolVersion;
	reader.setProtocolVersion(protocolVersion);
	std::unique_ptr<ClientLogEventsParser::ParserBase> parser =
	    ClientLogEventsParser::ParserFactory::getParser(protocolVersion);

	while (!reader.empty()) {
		// Get EventType and timestamp
		FdbClientLogEvents::Event event;
		reader >> event;
		switch (event.type) {
		case FdbClientLogEvents::EventType::GET_VERSION_LATENCY:
			parser->parseGetVersion(reader);
			break;
		case FdbClientLogEvents::EventType::GET_LATENCY:
			parser->parseGet(reader);
			break;
		case FdbClientLogEvents::EventType::GET_RANGE_LATENCY:
			parser->parseGetRange(reader);
			break;
		case FdbClientLogEvents::EventType::COMMIT_LATENCY:
			parser->parseCommit(reader);
			break;
		case FdbClientLogEvents::EventType::ERROR_GET:
			parser->parseErrorGet(reader);
			break;
		case FdbClientLogEvents::EventType::ERROR_GET_RANGE:
			parser->parseErrorGetRange(reader);
			break;
		case FdbClientLogEvents::EventType::ERROR_COMMIT:
			parser->parseErrorCommit(reader);
			break;
		default:
			TraceEvent(SevError, "ClientTransactionProfilingUnknownEvent").detail("EventType", event.type);
			return false;
		}
	}

	return true;
}

struct ClientTransactionProfileCorrectnessWorkload : TestWorkload {
	static constexpr auto NAME = "ClientTransactionProfileCorrectness";
	double samplingProbability;
	int64_t trInfoSizeLimit;

	ClientTransactionProfileCorrectnessWorkload(WorkloadContext const& wcx) : TestWorkload(wcx) {
		samplingProbability = getOption(options,
		                                "samplingProbability"_sr,
		                                deterministicRandom()->random01() / 10); // rand range 0 - 0.1
		trInfoSizeLimit = getOption(options,
		                            "trInfoSizeLimit"_sr,
		                            deterministicRandom()->randomInt(100 * 1024, 10 * 1024 * 1024)); // 100 KB - 10 MB
		TraceEvent(SevInfo, "ClientTransactionProfilingSetup")
		    .detail("ClientId", clientId)
		    .detail("SamplingProbability", samplingProbability)
		    .detail("TrInfoSizeLimit", trInfoSizeLimit);
	}

	Future<Void> setup(Database const& cx) override {
		if (clientId == 0) {
			IKnobCollection::getMutableGlobalKnobCollection().setKnob("csi_status_delay",
			                                                          KnobValueRef::create(double{ 2.0 })); // 2 seconds
			return changeProfilingParameters(cx, trInfoSizeLimit, samplingProbability);
		}
		return Void();
	}

	Future<Void> start(Database const& cx) override { return Void(); }

	int getNumChunks(KeyRef key) const {
		return bigEndian32(
		    BinaryReader::fromStringRef<int>(key.substr(numChunksStartIndex, chunkFormatSize), Unversioned()));
	}

	int getChunkNum(KeyRef key) const {
		return bigEndian32(
		    BinaryReader::fromStringRef<int>(key.substr(chunkNumStartIndex, chunkFormatSize), Unversioned()));
	}

	std::string getTrId(KeyRef key) const { return key.substr(trIdStartIndex, trIdFormatSize).toString(); }

	bool checkTxInfoEntriesFormat(const RangeResult& txInfoEntries) {
		std::string val;
		std::map<std::string, std::vector<ValueRef>> trInfoChunks;
		for (auto kv : txInfoEntries) {
			int numChunks = getNumChunks(kv.key);
			int chunkNum = getChunkNum(kv.key);
			std::string trId = getTrId(kv.key);

			if (numChunks == 1) {
				ASSERT(chunkNum == 1);
				BinaryReader reader(kv.value, Unversioned());
				if (!checkTxInfoEntryFormat(reader))
					return false;
			} else {
				if (chunkNum == 1) { // First chunk
					// Remove any entry if already present. There are scenarios (eg., commit_unknown_result) where a
					// transaction info may be logged multiple times
					trInfoChunks.erase(trId);
					trInfoChunks.insert(std::pair<std::string, std::vector<ValueRef>>(trId, { kv.value }));
				} else {
					if (trInfoChunks.find(trId) == trInfoChunks.end()) {
						// Some of the earlier chunks for this trId should have been deleted.
						// Discard this chunk as it is of not much use
						TraceEvent(SevInfo, "ClientTransactionProfilingSomeChunksMissing").detail("TrId", trId);
					} else {
						// Check if it is the expected chunk. Otherwise discard the whole transaction entry.
						// There are scenarios (eg., when deletion is happening) where some chunks get missed.
						if (chunkNum != trInfoChunks.find(trId)->second.size() + 1) {
							TraceEvent(SevInfo, "ClientTransactionProfilingChunksMissing").detail("TrId", trId);
							trInfoChunks.erase(trId);
						} else {
							trInfoChunks.find(trId)->second.push_back(kv.value);
						}
					}
				}
				if (chunkNum == numChunks && trInfoChunks.find(trId) != trInfoChunks.end()) {
					auto iter = trInfoChunks.find(trId);
					BinaryWriter bw(Unversioned());
					for (auto val : iter->second)
						bw.serializeBytes(val.begin(), val.size());
					BinaryReader reader(bw.getData(), bw.getLength(), Unversioned());
					if (!checkTxInfoEntryFormat(reader))
						return false;
					trInfoChunks.erase(iter);
				}
			}
		}
		return true;
	}

																#line 292 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
// This generated class is to be used only via changeProfilingParameters()
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
template <class ChangeProfilingParametersActor>
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
class ChangeProfilingParametersActorState {
															#line 298 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
public:
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	ChangeProfilingParametersActorState(Database const& cx,int64_t const& sizeLimit,double const& sampleProbability) 
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		 : cx(cx),
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		   sizeLimit(sizeLimit),
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		   sampleProbability(sampleProbability)
															#line 309 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("changeProfilingParameters", reinterpret_cast<unsigned long>(this));

	}
	~ChangeProfilingParametersActorState() 
	{
		fdb_probe_actor_destroy("changeProfilingParameters", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 292 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_0 = runRYWTransaction(cx, [=](Reference<ReadYourWritesTransaction> tr) -> Future<Void> { tr->setOption(FDBTransactionOptions::SPECIAL_KEY_SPACE_ENABLE_WRITES); tr->setOption(FDBTransactionOptions::RAW_ACCESS); tr->setOption(FDBTransactionOptions::LOCK_AWARE); Tuple rate = Tuple::makeTuple(sampleProbability); Tuple size = Tuple::makeTuple(sizeLimit); tr->set(GlobalConfig::prefixedKey(fdbClientInfoTxnSampleRate), rate.pack()); tr->set(GlobalConfig::prefixedKey(fdbClientInfoTxnSizeLimit), size.pack()); return Void(); });
															#line 292 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			if (static_cast<ChangeProfilingParametersActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 326 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ChangeProfilingParametersActor*>(this)->actor_wait_state = 1;
															#line 292 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ChangeProfilingParametersActor, 0, Void >*>(static_cast<ChangeProfilingParametersActor*>(this)));
															#line 331 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ChangeProfilingParametersActorState();
		static_cast<ChangeProfilingParametersActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 302 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		if (!static_cast<ChangeProfilingParametersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ChangeProfilingParametersActorState(); static_cast<ChangeProfilingParametersActor*>(this)->destroy(); return 0; }
															#line 354 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		new (&static_cast<ChangeProfilingParametersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ChangeProfilingParametersActorState();
		static_cast<ChangeProfilingParametersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 302 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		if (!static_cast<ChangeProfilingParametersActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ChangeProfilingParametersActorState(); static_cast<ChangeProfilingParametersActor*>(this)->destroy(); return 0; }
															#line 366 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		new (&static_cast<ChangeProfilingParametersActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ChangeProfilingParametersActorState();
		static_cast<ChangeProfilingParametersActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ChangeProfilingParametersActor*>(this)->actor_wait_state > 0) static_cast<ChangeProfilingParametersActor*>(this)->actor_wait_state = 0;
		static_cast<ChangeProfilingParametersActor*>(this)->ActorCallback< ChangeProfilingParametersActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangeProfilingParametersActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changeProfilingParameters", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeProfilingParametersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeProfilingParameters", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ChangeProfilingParametersActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changeProfilingParameters", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeProfilingParametersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeProfilingParameters", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ChangeProfilingParametersActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changeProfilingParameters", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeProfilingParametersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changeProfilingParameters", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	Database cx;
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	int64_t sizeLimit;
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	double sampleProbability;
															#line 455 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
};
// This generated class is to be used only via changeProfilingParameters()
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
class ChangeProfilingParametersActor final : public Actor<Void>, public ActorCallback< ChangeProfilingParametersActor, 0, Void >, public FastAllocated<ChangeProfilingParametersActor>, public ChangeProfilingParametersActorState<ChangeProfilingParametersActor> {
															#line 460 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
public:
	using FastAllocated<ChangeProfilingParametersActor>::operator new;
	using FastAllocated<ChangeProfilingParametersActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10499520206333812736UL, 16667834808860550400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ChangeProfilingParametersActor, 0, Void >;
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	ChangeProfilingParametersActor(Database const& cx,int64_t const& sizeLimit,double const& sampleProbability) 
															#line 477 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		 : Actor<Void>(),
		   ChangeProfilingParametersActorState<ChangeProfilingParametersActor>(cx, sizeLimit, sampleProbability),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("changeProfilingParameters", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9786684525058980096UL, 13704156376182779136UL);
		ActorExecutionContextHelper __helper(static_cast<ChangeProfilingParametersActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("changeProfilingParameters");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("changeProfilingParameters", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ChangeProfilingParametersActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
[[nodiscard]] Future<Void> changeProfilingParameters( Database const& cx, int64_t const& sizeLimit, double const& sampleProbability ) {
															#line 290 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	return Future<Void>(new ChangeProfilingParametersActor(cx, sizeLimit, sampleProbability));
															#line 509 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
}

#line 304 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"

																#line 514 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
// This generated class is to be used only via _check()
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
template <class _checkActor>
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
class _checkActorState {
															#line 520 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
public:
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	_checkActorState(Database const& cx,ClientTransactionProfileCorrectnessWorkload* const& self) 
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		 : cx(cx),
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		   self(self)
															#line 529 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("_check", reinterpret_cast<unsigned long>(this));

	}
	~_checkActorState() 
	{
		fdb_probe_actor_destroy("_check", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 306 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->changeProfilingParameters(cx, self->trInfoSizeLimit, 0);
															#line 306 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 546 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<_checkActor*>(this)->actor_wait_state = 1;
															#line 306 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 0, Void >*>(static_cast<_checkActor*>(this)));
															#line 551 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_checkActorState();
		static_cast<_checkActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 308 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(CLIENT_KNOBS->CSI_STATUS_DELAY);
															#line 308 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 576 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_checkActor*>(this)->actor_wait_state = 2;
															#line 308 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 1, Void >*>(static_cast<_checkActor*>(this)));
															#line 581 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 308 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(CLIENT_KNOBS->CSI_STATUS_DELAY);
															#line 308 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 592 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_checkActor*>(this)->actor_wait_state = 2;
															#line 308 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 1, Void >*>(static_cast<_checkActor*>(this)));
															#line 597 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _checkActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _checkActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 310 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		clientLatencyAtomicCtr = CLIENT_LATENCY_INFO_CTR_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin);
															#line 311 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		counter = int64_t();
															#line 312 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		txInfoEntries = RangeResult();
															#line 313 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_2 = runRYWTransaction(cx, [=](Reference<ReadYourWritesTransaction> tr) -> Future<Optional<Value>> { tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS); tr->setOption(FDBTransactionOptions::LOCK_AWARE); return tr->get(clientLatencyAtomicCtr); });
															#line 313 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 689 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<_checkActor*>(this)->actor_wait_state = 3;
															#line 313 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 2, Optional<Value> >*>(static_cast<_checkActor*>(this)));
															#line 694 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 310 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		clientLatencyAtomicCtr = CLIENT_LATENCY_INFO_CTR_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin);
															#line 311 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		counter = int64_t();
															#line 312 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		txInfoEntries = RangeResult();
															#line 313 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_2 = runRYWTransaction(cx, [=](Reference<ReadYourWritesTransaction> tr) -> Future<Optional<Value>> { tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS); tr->setOption(FDBTransactionOptions::LOCK_AWARE); return tr->get(clientLatencyAtomicCtr); });
															#line 313 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 711 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<_checkActor*>(this)->actor_wait_state = 3;
															#line 313 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 2, Optional<Value> >*>(static_cast<_checkActor*>(this)));
															#line 716 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _checkActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _checkActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Optional<Value> const& ctrValue,int loopDepth) 
	{
															#line 319 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		counter = ctrValue.present() ? BinaryReader::fromStringRef<int64_t>(ctrValue.get(), Unversioned()) : 0;
															#line 320 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		clientLatencyName = CLIENT_LATENCY_INFO_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin);
															#line 322 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		begin = firstGreaterOrEqual(CLIENT_LATENCY_INFO_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin));
															#line 324 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		end = firstGreaterOrEqual(strinc(begin.getKey()));
															#line 325 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		keysLimit = 10;
															#line 326 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		tr = Transaction(cx);
															#line 327 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		;
															#line 812 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Optional<Value> && ctrValue,int loopDepth) 
	{
															#line 319 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		counter = ctrValue.present() ? BinaryReader::fromStringRef<int64_t>(ctrValue.get(), Unversioned()) : 0;
															#line 320 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		clientLatencyName = CLIENT_LATENCY_INFO_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin);
															#line 322 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		begin = firstGreaterOrEqual(CLIENT_LATENCY_INFO_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin));
															#line 324 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		end = firstGreaterOrEqual(strinc(begin.getKey()));
															#line 325 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		keysLimit = 10;
															#line 326 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		tr = Transaction(cx);
															#line 327 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		;
															#line 833 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> const& ctrValue,int loopDepth) 
	{
		loopDepth = a_body1cont3(ctrValue, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> && ctrValue,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(ctrValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 2, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 2, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< _checkActor, 2, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< _checkActor, 2, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 346 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		int64_t contentsSize = 0;
															#line 347 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		for( auto& kv : txInfoEntries ) {
															#line 348 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			contentsSize += kv.key.size() + kv.value.size();
															#line 921 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		}
															#line 355 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		TraceEvent(SevInfo, "ClientTransactionProfilingCtrval").detail("Counter", counter);
															#line 356 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		TraceEvent(SevInfo, "ClientTransactionProfilingContentsSize").detail("ContentsSize", contentsSize);
															#line 359 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		if (!static_cast<_checkActor*>(this)->SAV<bool>::futures) { (void)(self->checkTxInfoEntriesFormat(txInfoEntries)); this->~_checkActorState(); static_cast<_checkActor*>(this)->destroy(); return 0; }
															#line 929 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		new (&static_cast<_checkActor*>(this)->SAV< bool >::value()) bool(self->checkTxInfoEntriesFormat(txInfoEntries));
		this->~_checkActorState();
		static_cast<_checkActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1(int loopDepth) 
	{
		try {
															#line 329 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 330 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 331 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			StrictFuture<RangeResult> __when_expr_3 = tr.getRange(begin, end, keysLimit);
															#line 331 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 955 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont3loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3loopBody1when1(__when_expr_3.get(), loopDepth); };
			static_cast<_checkActor*>(this)->actor_wait_state = 4;
															#line 331 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 3, RangeResult >*>(static_cast<_checkActor*>(this)));
															#line 960 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont3loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont3loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 339 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			if (e.code() == error_code_transaction_too_old)
															#line 995 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
			{
															#line 340 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
				keysLimit = std::max(1, keysLimit / 2);
															#line 999 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
			}
															#line 341 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 341 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1005 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<_checkActor*>(this)->actor_wait_state = 5;
															#line 341 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 4, Void >*>(static_cast<_checkActor*>(this)));
															#line 1010 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2(int loopDepth) 
	{
															#line 332 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		if (kvRange.empty())
															#line 1025 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		{
			return a_body1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 334 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		txInfoEntries.arena().dependsOn(kvRange.arena());
															#line 335 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		txInfoEntries.append(txInfoEntries.arena(), kvRange.begin(), kvRange.size());
															#line 336 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		begin = firstGreaterThan(kvRange.back().key);
															#line 337 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		tr.reset();
															#line 1037 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		loopDepth = a_body1cont3loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(RangeResult const& __kvRange,int loopDepth) 
	{
															#line 331 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
		kvRange = __kvRange;
															#line 1046 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		loopDepth = a_body1cont3loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(RangeResult && __kvRange,int loopDepth) 
	{
		kvRange = std::move(__kvRange);
		loopDepth = a_body1cont3loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 3, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 3, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< _checkActor, 3, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< _checkActor, 3, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont3loopBody1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont3loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont3loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< _checkActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< _checkActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	Database cx;
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	ClientTransactionProfileCorrectnessWorkload* self;
															#line 310 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	Key clientLatencyAtomicCtr;
															#line 311 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	int64_t counter;
															#line 312 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	RangeResult txInfoEntries;
															#line 320 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	Key clientLatencyName;
															#line 322 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	KeySelector begin;
															#line 324 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	KeySelector end;
															#line 325 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	int keysLimit;
															#line 326 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	Transaction tr;
															#line 331 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	RangeResult kvRange;
															#line 1243 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
};
// This generated class is to be used only via _check()
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
class _checkActor final : public Actor<bool>, public ActorCallback< _checkActor, 0, Void >, public ActorCallback< _checkActor, 1, Void >, public ActorCallback< _checkActor, 2, Optional<Value> >, public ActorCallback< _checkActor, 3, RangeResult >, public ActorCallback< _checkActor, 4, Void >, public FastAllocated<_checkActor>, public _checkActorState<_checkActor> {
															#line 1248 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
public:
	using FastAllocated<_checkActor>::operator new;
	using FastAllocated<_checkActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6898635192380970496UL, 11587519569977872896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _checkActor, 0, Void >;
friend struct ActorCallback< _checkActor, 1, Void >;
friend struct ActorCallback< _checkActor, 2, Optional<Value> >;
friend struct ActorCallback< _checkActor, 3, RangeResult >;
friend struct ActorCallback< _checkActor, 4, Void >;
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	_checkActor(Database const& cx,ClientTransactionProfileCorrectnessWorkload* const& self) 
															#line 1269 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
		 : Actor<bool>(),
		   _checkActorState<_checkActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18414574916690025728UL, 14115196402046968064UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_check");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _checkActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _checkActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< _checkActor, 2, Optional<Value> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< _checkActor, 3, RangeResult >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< _checkActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
[[nodiscard]] Future<bool> _check( Database const& cx, ClientTransactionProfileCorrectnessWorkload* const& self ) {
															#line 305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"
	return Future<bool>(new _checkActor(cx, self));
															#line 1305 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.g.cpp"
}

#line 361 "/codebuild/output/src1900875502/src/github.com/apple/foundationdb/fdbserver/workloads/ClientTransactionProfileCorrectness.actor.cpp"

	Future<bool> check(Database const& cx) override {
		if (clientId != 0)
			return true;
		return _check(cx, this);
	}

	void getMetrics(std::vector<PerfMetric>& m) override {}
};

WorkloadFactory<ClientTransactionProfileCorrectnessWorkload> ClientTransactionProfileCorrectnessWorkloadFactory;
