package com.apple.foundationdb;

/**
 * A set of operations that can be performed atomically on a database. These are used as parameters to {@link Transaction#mutate(MutationType, byte[], byte[])}.
 */
public enum MutationType {
	/**
	 * Performs an addition of little-endian integers. If the existing value in the database is not present or shorter than {@code param}, it is first extended to the length of {@code param} with zero bytes.  If {@code param} is shorter than the existing value in the database, the existing value is truncated to match the length of {@code param}. The integers to be added must be stored in a little-endian representation.  They can be signed in two's complement representation or unsigned. You can add to an integer at a known offset in the value by prepending the appropriate number of zero bytes to {@code param} and padding with zero bytes to match the length of the value. However, this offset technique requires that you know the addition will not cause the integer field within the value to overflow.
	 */
	ADD(2),

	/**
	 * Deprecated.
	 */
	@Deprecated
	AND(6),

	/**
	 * Performs a bitwise {@code and} operation.  If the existing value in the database is not present, then {@code param} is stored in the database. If the existing value in the database is shorter than {@code param}, it is first extended to the length of {@code param} with zero bytes.  If {@code param} is shorter than the existing value in the database, the existing value is truncated to match the length of {@code param}.
	 */
	BIT_AND(6),

	/**
	 * Deprecated.
	 */
	@Deprecated
	OR(7),

	/**
	 * Performs a bitwise {@code or} operation.  If the existing value in the database is not present or shorter than {@code param}, it is first extended to the length of {@code param} with zero bytes.  If {@code param} is shorter than the existing value in the database, the existing value is truncated to match the length of {@code param}.
	 */
	BIT_OR(7),

	/**
	 * Deprecated.
	 */
	@Deprecated
	XOR(8),

	/**
	 * Performs a bitwise {@code xor} operation.  If the existing value in the database is not present or shorter than {@code param}, it is first extended to the length of {@code param} with zero bytes.  If {@code param} is shorter than the existing value in the database, the existing value is truncated to match the length of {@code param}.
	 */
	BIT_XOR(8),

	/**
	 * Appends {@code param} to the end of the existing value already in the database at the given key (or creates the key and sets the value to {@code param} if the key is empty). This will only append the value if the final concatenated value size is less than or equal to the maximum value size (i.e., if it fits). WARNING: No error is surfaced back to the user if the final value is too large because the mutation will not be applied until after the transaction has been committed. Therefore, it is only safe to use this mutation type if one can guarantee that one will keep the total value size under the maximum size.
	 */
	APPEND_IF_FITS(9),

	/**
	 * Performs a little-endian comparison of byte strings. If the existing value in the database is not present or shorter than {@code param}, it is first extended to the length of {@code param} with zero bytes.  If {@code param} is shorter than the existing value in the database, the existing value is truncated to match the length of {@code param}. The larger of the two values is then stored in the database.
	 */
	MAX(12),

	/**
	 * Performs a little-endian comparison of byte strings. If the existing value in the database is not present, then {@code param} is stored in the database. If the existing value in the database is shorter than {@code param}, it is first extended to the length of {@code param} with zero bytes.  If {@code param} is shorter than the existing value in the database, the existing value is truncated to match the length of {@code param}. The smaller of the two values is then stored in the database.
	 */
	MIN(13),

	/**
	 * Transforms {@code key} using a versionstamp for the transaction. Sets the transformed key in the database to {@code param}. The key is transformed by removing the final four bytes from the key and reading those as a little-Endian 32-bit integer to get a position {@code pos}. The 10 bytes of the key from {@code pos} to {@code pos + 10} are replaced with the versionstamp of the transaction used. The first byte of the key is position 0. A versionstamp is a 10 byte, unique, monotonically (but not sequentially) increasing value for each committed transaction. The first 8 bytes are the committed version of the database (serialized in big-Endian order). The last 2 bytes are monotonic in the serialization order for transactions. WARNING: At this time, versionstamps are compatible with the Tuple layer only in the Java, Python, and Go bindings. Also, note that prior to API version 520, the offset was computed from only the final two bytes rather than the final four bytes.
	 */
	SET_VERSIONSTAMPED_KEY(14),

	/**
	 * Transforms {@code param} using a versionstamp for the transaction. Sets the {@code key} given to the transformed {@code param}. The parameter is transformed by removing the final four bytes from {@code param} and reading those as a little-Endian 32-bit integer to get a position {@code pos}. The 10 bytes of the parameter from {@code pos} to {@code pos + 10} are replaced with the versionstamp of the transaction used. The first byte of the parameter is position 0. A versionstamp is a 10 byte, unique, monotonically (but not sequentially) increasing value for each committed transaction. The first 8 bytes are the committed version of the database (serialized in big-Endian order). The last 2 bytes are monotonic in the serialization order for transactions. WARNING: At this time, versionstamps are compatible with the Tuple layer only in the Java, Python, and Go bindings. Also, note that prior to API version 520, the versionstamp was always placed at the beginning of the parameter rather than computing an offset.
	 */
	SET_VERSIONSTAMPED_VALUE(15),

	/**
	 * Performs lexicographic comparison of byte strings. If the existing value in the database is not present, then {@code param} is stored. Otherwise the smaller of the two values is then stored in the database.
	 */
	BYTE_MIN(16),

	/**
	 * Performs lexicographic comparison of byte strings. If the existing value in the database is not present, then {@code param} is stored. Otherwise the larger of the two values is then stored in the database.
	 */
	BYTE_MAX(17),

	/**
	 * Performs an atomic {@code compare and clear} operation. If the existing value in the database is equal to the given value, then given key is cleared.
	 */
	COMPARE_AND_CLEAR(20);
    private final int code;

    MutationType(int code) {
        this.code = code;
    }

    /**
     * Gets the FoundationDB native-level constant code for a {@code MutationType}.
     *
     * @return the native code for a FoundationDB {@code MutationType} constant.
     */
    public int code() {
        return this.code;
    }
}
