#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
/*
 * ConsistencyScanInterface.h
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/SystemData.h"
#include "fdbclient/json_spirit/json_spirit_value.h"
#include "flow/serialize.h"
#include "fmt/core.h"
#if defined(NO_INTELLISENSE) && !defined(FDBCLIENT_CONSISTENCYSCANINTERFACE_ACTOR_G_H)
#define FDBCLIENT_CONSISTENCYSCANINTERFACE_ACTOR_G_H
#include "fdbclient/ConsistencyScanInterface.actor.g.h"
#elif !defined(FDBCLIENT_CONSISTENCYSCANINTERFACE_ACTOR_H)
#define FDBCLIENT_CONSISTENCYSCANINTERFACE_ACTOR_H

#include "fdbclient/CommitProxyInterface.h"
#include "fdbclient/DatabaseConfiguration.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/RunRYWTransaction.actor.h"
#include "fdbrpc/fdbrpc.h"
#include "fdbrpc/Locality.h"
#include "fdbclient/KeyBackedTypes.actor.h"
#include "fdbclient/KeyBackedRangeMap.actor.h"

#include "flow/actorcompiler.h" // must be last include

struct ConsistencyScanInterface {
	constexpr static FileIdentifier file_identifier = 4983265;
	RequestStream<ReplyPromise<Void>> waitFailure;
	RequestStream<struct HaltConsistencyScanRequest> haltConsistencyScan;
	struct LocalityData locality;
	UID myId;

	ConsistencyScanInterface() {}
	explicit ConsistencyScanInterface(const struct LocalityData& l, UID id) : locality(l), myId(id) {}

	void initEndpoints() {}
	UID id() const { return myId; }
	NetworkAddress address() const { return waitFailure.getEndpoint().getPrimaryAddress(); }
	bool operator==(const ConsistencyScanInterface& r) const { return id() == r.id(); }
	bool operator!=(const ConsistencyScanInterface& r) const { return !(*this == r); }

	template <class Archive>
	void serialize(Archive& ar) {
		serializer(ar, waitFailure, haltConsistencyScan, locality, myId);
	}
};

struct HaltConsistencyScanRequest {
	constexpr static FileIdentifier file_identifier = 2323417;
	UID requesterID;
	ReplyPromise<Void> reply;

	HaltConsistencyScanRequest() {}
	explicit HaltConsistencyScanRequest(UID uid) : requesterID(uid) {}

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, requesterID, reply);
	}
};

// Consistency Scan State
// This class provides access to the Consistency Scan's state stored in the database.
// The state is divided into these components
//   Config - Tells the scan whether and how to run.  Written by user, read by scan.
//   RangeConfig - Tells the scan what ranges to operate on or ignore
//
//   CurrentRoundStats - Execution state and stats for the current round
//   RoundHistory - History of RoundInfo's by start version
// 	 LifetimeStats - Accumulated lifetime counts for the cluster
//
// The class trigger will only be fired by changes to Config or RangeConfig
struct ConsistencyScanState : public KeyBackedClass {
	ConsistencyScanState(Key prefix = SystemKey("\xff/consistencyScanState"_sr)) : KeyBackedClass(prefix) {}

	struct Config {
		constexpr static FileIdentifier file_identifier = 23123;

		bool enabled = false;

		// The values below are NOT being initialized from knobs because once the scan is enabled
		// changing the knobs does nothing.  The consistency check knobs are for the consistency
		// check workload, which is different from the Consistency Scan feature

		// Max byte read bandwidth allowed, default 50 MB/s
		int64_t maxReadByteRate = 50e6;
		// Target time in seconds for completion of one full scan of the database, default 30 days.
		int64_t targetRoundTimeSeconds = 60 * 60 * 24 * 30;
		// Minimum time in seconds a round should take.

		// If a round completes faster than this, the scanner will delay afterwards, though if the
		// scan role is restarted it will start a new scan.
		int64_t minRoundTimeSeconds = 60 * 60 * 24 * 30;

		// The minimum start version allowed for the current round.  If the round started before this
		// it will be ended without completion, moved to history, and a new round will begin.
		Version minStartVersion = 0;

		// Number of days of history to keep, this is enforced using CORE_VERSIONSPERSECOND
		int64_t roundHistoryDays = 90;

		json_spirit::mObject toJSON() const {
			json_spirit::mObject doc;
			doc["enabled"] = enabled;
			doc["max_rate_bytes_per_second"] = maxReadByteRate;
			doc["target_interval_seconds"] = targetRoundTimeSeconds;
			doc["min_interval_seconds"] = minRoundTimeSeconds;
			doc["min_start_version"] = minStartVersion;
			doc["round_history_days"] = roundHistoryDays;
			return doc;
		}

		template <class Ar>
		void serialize(Ar& ar) {
			serializer(ar,
			           enabled,
			           maxReadByteRate,
			           targetRoundTimeSeconds,
			           minRoundTimeSeconds,
			           minStartVersion,
			           roundHistoryDays);
		}
	};

	// Configuration value in a range map for a key range
	struct RangeConfig {
		constexpr static FileIdentifier file_identifier = 846323;

		// Whether the range is included as a configured target for the scan
		// This should normally be set by the user
		Optional<bool> included;

		// Whether the range should be currently even though it remains a scan target
		// This should be set by operations on the cluster that would make shards temporarily inconsistent.
		Optional<bool> skip;

		RangeConfig apply(RangeConfig const& rhs) const {
			RangeConfig result = *this;
			if (rhs.included.present()) {
				result.included = rhs.included;
			}
			if (rhs.skip.present()) {
				result.skip = rhs.skip;
			}
			return result;
		}

		template <class Ar>
		void serialize(Ar& ar) {
			serializer(ar, included, skip);
		}

		std::string toString() const { return fmt::format("included={} skip={}", included, skip); }
		json_spirit::mObject toJSON() const {
			json_spirit::mObject doc;
			if (included.present()) {
				doc["included"] = *included;
			}
			if (skip.present()) {
				doc["skip"] = *skip;
			}
			return doc;
		}
	};

	struct LifetimeStats {
		constexpr static FileIdentifier file_identifier = 7897646;

		// Amount of FDB keyspace read, regardless of replication
		int64_t logicalBytesScanned = 0;
		// Actual amount of data read from shard replicas
		int64_t replicatedBytesRead = 0;
		int64_t errorCount = 0;

		template <class Ar>
		void serialize(Ar& ar) {
			serializer(ar, logicalBytesScanned, replicatedBytesRead, errorCount);
		}

		json_spirit::mObject toJSON() const {
			json_spirit::mObject doc;
			doc["logical_bytes_scanned"] = logicalBytesScanned;
			doc["replicated_bytes_scanned"] = replicatedBytesRead;
			doc["errors"] = errorCount;
			return doc;
		}
	};

	struct RoundStats {
		constexpr static FileIdentifier file_identifier = 23126;

		Version startVersion = 0;
		double startTime = 0;
		Version endVersion = 0;
		double endTime = 0;
		Version lastProgressVersion = 0;
		double lastProgressTime = 0;

		// Whether the scan finished, useful for history round stats.
		bool complete = false;

		// Amount of FDB keyspace read, regardless of replication
		int64_t logicalBytesScanned = 0;
		// Actual amount of data read from shard replicas
		int64_t replicatedBytesRead = 0;
		int64_t errorCount = 0;
		int64_t skippedRanges = 0;
		// FIXME: add failed request count that we periodically save even if no progress too?

		Key lastEndKey = ""_sr;

		/*
		// TODO:  Ideas of more things to track:
		int64_t shardsScanned = 0;
		int64_t replicasScanned = 0;
		// Shards or replicas can be skipped due to being offline or locked
		int64_t shardsSkipped = 0;
		int64_t replicasSkipped = 0;
		*/

		template <class Ar>
		void serialize(Ar& ar) {
			serializer(ar,
			           startVersion,
			           startTime,
			           endVersion,
			           endTime,
			           lastProgressVersion,
			           lastProgressTime,
			           complete,
			           logicalBytesScanned,
			           replicatedBytesRead,
			           errorCount,
			           skippedRanges,
			           lastEndKey);
		}

		json_spirit::mObject toJSON() const {
			json_spirit::mObject doc;
			doc["complete"] = complete;
			doc["start_version"] = startVersion;
			if (startTime != 0) {
				doc["start_timestamp"] = startTime;
				doc["start_datetime"] = epochsToGMTString(startTime);
			}

			doc["end_version"] = endVersion;
			if (endTime != 0) {
				doc["end_timestamp"] = endTime;
				doc["end_datetime"] = epochsToGMTString(endTime);
			}

			doc["last_progress_version"] = lastProgressVersion;
			if (lastProgressTime != 0) {
				doc["last_progress_timestamp"] = lastProgressTime;
				doc["last_progress_datetime"] = epochsToGMTString(lastProgressTime);
			}

			doc["logical_bytes_scanned"] = logicalBytesScanned;
			doc["replicated_bytes_scanned"] = replicatedBytesRead;
			doc["errors"] = errorCount;
			doc["last_end_key"] = lastEndKey.toString();
			doc["skippedRanges"] = skippedRanges;
			return doc;
		}
	};

	// Range map for configuring key range options.  By default, all ranges are scanned.
	typedef KeyBackedRangeMap<Key, RangeConfig, TupleCodec<Key>, ObjectCodec<RangeConfig, _IncludeVersion>>
	    RangeConfigMap;

	// Map of scan start version to its stats so a history can be maintained.
	typedef KeyBackedObjectMap<Version, RoundStats, _IncludeVersion> StatsHistoryMap;

	RangeConfigMap rangeConfig() {
		// Updating rangeConfig updates the class trigger
		return { subspace.pack(__FUNCTION__sr), trigger, IncludeVersion() };
	}

	KeyBackedObjectProperty<Config, _IncludeVersion> config() {
		// Updating rangeConfig updates the class trigger
		return { subspace.pack(__FUNCTION__sr), IncludeVersion(), trigger };
	}

	// Updating the lifetime stats does not update the class trigger because the stats are constantly updated, but when
	// resetting them the same transaction that sets the stats value must also call trigger.update(tr) so that the scan
	// loop will restart and not overwrite the reset value with a stale copy.
	KeyBackedObjectProperty<LifetimeStats, _IncludeVersion> lifetimeStats() {
		return { subspace.pack(__FUNCTION__sr), IncludeVersion() };
	}

	KeyBackedObjectProperty<RoundStats, _IncludeVersion> currentRoundStats() {
		return { subspace.pack(__FUNCTION__sr), IncludeVersion() };
	}

	// History of scan round stats stored by their start version
	StatsHistoryMap roundStatsHistory() { return { subspace.pack(__FUNCTION__sr), IncludeVersion() }; }

																#line 317 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
// This generated class is to be used only via clearStatsActor()
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
template <class ClearStatsActorActor>
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
class ClearStatsActorActorState {
															#line 323 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
public:
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
	ClearStatsActorActorState(ConsistencyScanState* const& self,Reference<ReadYourWritesTransaction> const& tr) 
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
		 : self(self),
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
		   tr(tr)
															#line 332 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
	{
		fdb_probe_actor_create("clearStatsActor", reinterpret_cast<unsigned long>(this));

	}
	~ClearStatsActorActorState() 
	{
		fdb_probe_actor_destroy("clearStatsActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 319 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
			StrictFuture<Void> __when_expr_0 = success(self->currentRoundStats().getD(tr)) && success(self->lifetimeStats().getD(tr)) && success(self->roundStatsHistory().getRange(tr, {}, {}, 1, Snapshot::False, Reverse::False));
															#line 319 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
			if (static_cast<ClearStatsActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 349 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ClearStatsActorActor*>(this)->actor_wait_state = 1;
															#line 319 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ClearStatsActorActor, 0, Void >*>(static_cast<ClearStatsActorActor*>(this)));
															#line 354 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ClearStatsActorActorState();
		static_cast<ClearStatsActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 323 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
		self->currentRoundStats().set(tr, ConsistencyScanState::RoundStats());
															#line 324 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
		self->lifetimeStats().set(tr, ConsistencyScanState::LifetimeStats());
															#line 325 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
		self->roundStatsHistory().erase(tr, 0, MAX_VERSION);
															#line 326 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
		if (!static_cast<ClearStatsActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ClearStatsActorActorState(); static_cast<ClearStatsActorActor*>(this)->destroy(); return 0; }
															#line 383 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
		new (&static_cast<ClearStatsActorActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ClearStatsActorActorState();
		static_cast<ClearStatsActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 323 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
		self->currentRoundStats().set(tr, ConsistencyScanState::RoundStats());
															#line 324 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
		self->lifetimeStats().set(tr, ConsistencyScanState::LifetimeStats());
															#line 325 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
		self->roundStatsHistory().erase(tr, 0, MAX_VERSION);
															#line 326 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
		if (!static_cast<ClearStatsActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ClearStatsActorActorState(); static_cast<ClearStatsActorActor*>(this)->destroy(); return 0; }
															#line 401 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
		new (&static_cast<ClearStatsActorActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ClearStatsActorActorState();
		static_cast<ClearStatsActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ClearStatsActorActor*>(this)->actor_wait_state > 0) static_cast<ClearStatsActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClearStatsActorActor*>(this)->ActorCallback< ClearStatsActorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClearStatsActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clearStatsActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearStatsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearStatsActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ClearStatsActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clearStatsActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClearStatsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearStatsActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ClearStatsActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clearStatsActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClearStatsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clearStatsActor", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
	ConsistencyScanState* self;
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
	Reference<ReadYourWritesTransaction> tr;
															#line 488 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
};
// This generated class is to be used only via clearStatsActor()
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
class ClearStatsActorActor final : public Actor<Void>, public ActorCallback< ClearStatsActorActor, 0, Void >, public FastAllocated<ClearStatsActorActor>, public ClearStatsActorActorState<ClearStatsActorActor> {
															#line 493 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
public:
	using FastAllocated<ClearStatsActorActor>::operator new;
	using FastAllocated<ClearStatsActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1901308111900003072UL, 17738189030914806272UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ClearStatsActorActor, 0, Void >;
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
	ClearStatsActorActor(ConsistencyScanState* const& self,Reference<ReadYourWritesTransaction> const& tr) 
															#line 510 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
		 : Actor<Void>(),
		   ClearStatsActorActorState<ClearStatsActorActor>(self, tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("clearStatsActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9049068141488380672UL, 12535793511331615744UL);
		ActorExecutionContextHelper __helper(static_cast<ClearStatsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("clearStatsActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("clearStatsActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ClearStatsActorActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
[[nodiscard]] static Future<Void> clearStatsActor( ConsistencyScanState* const& self, Reference<ReadYourWritesTransaction> const& tr ) {
															#line 315 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
	return Future<Void>(new ClearStatsActorActor(self, tr));
															#line 542 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
}

#line 328 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"

	Future<Void> clearStats(Reference<ReadYourWritesTransaction> tr) { return clearStatsActor(this, tr); }
};

/////////////////////
// Code below this line is not used by the Consistency Scan Role, only the ConsistencyCheck Workload.

															#line 553 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
[[nodiscard]] Future<Version> getVersion( Database const& cx );

#line 336 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
															#line 557 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
[[nodiscard]] Future<bool> getKeyServers( Database const& cx, Promise<std::vector<std::pair<KeyRange, std::vector<StorageServerInterface>>>> const& keyServersPromise, KeyRangeRef const& kr, bool const& performQuiescentChecks, bool const& failureIsError, bool* const& success );

#line 343 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
															#line 561 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
[[nodiscard]] Future<bool> getKeyLocations( Database const& cx, std::vector<std::pair<KeyRange, std::vector<StorageServerInterface>>> const& shards, Promise<Standalone<VectorRef<KeyValueRef>>> const& keyLocationPromise, bool const& performQuiescentChecks, bool* const& success );

#line 348 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"
															#line 565 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.g.h"
[[nodiscard]] Future<Void> checkDataConsistency( Database const& cx, VectorRef<KeyValueRef> const& keyLocations, DatabaseConfiguration const& configuration, std::map<UID, StorageServerInterface> const& tssMapping, bool const& performQuiescentChecks, bool const& performTSSCheck, bool const& firstClient, bool const& failureIsError, int const& clientId, int const& clientCount, bool const& distributed, bool const& shuffleShards, int const& shardSampleFactor, int64_t const& sharedRandomNumber, int64_t const& repetitions, int64_t* const& bytesReadInPreviousRound, int const& restart, int64_t const& maxRate, int64_t const& targetInterval, bool* const& success );

#line 368 "/codebuild/output/src3112615453/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/ConsistencyScanInterface.actor.h"

#include "flow/unactorcompiler.h"

#endif // FDBCLIENT_CONSISTENCYSCANINTERFACE_H