#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
/*
 * storageserver.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <cinttypes>
#include <functional>
#include <iterator>
#include <limits>
#include <memory>
#include <type_traits>
#include <unordered_map>

#include "fdbclient/BlobCipher.h"
#include "fdbclient/BulkLoading.h"
#include "fdbclient/Knobs.h"
#include "fdbrpc/TenantInfo.h"
#include "flow/ApiVersion.h"
#include "flow/Buggify.h"
#include "flow/Platform.h"
#include "flow/network.h"
#include "fmt/format.h"
#include "fdbclient/Audit.h"
#include "fdbclient/CommitTransaction.h"
#include "fdbclient/FDBTypes.h"
#include "fdbrpc/fdbrpc.h"
#include "fdbrpc/LoadBalance.h"
#include "fdbserver/OTELSpanContextMessage.h"
#include "flow/ActorCollection.h"
#include "flow/Arena.h"
#include "flow/Error.h"
#include "flow/Hash3.h"
#include "flow/Histogram.h"
#include "flow/PriorityMultiLock.actor.h"
#include "flow/IRandom.h"
#include "flow/IndexedSet.h"
#include "flow/SystemMonitor.h"
#include "flow/Trace.h"
#include "fdbclient/Tracing.h"
#include "flow/Util.h"
#include "fdbclient/Atomic.h"
#include "fdbclient/AuditUtils.actor.h"
#include "fdbclient/CommitProxyInterface.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/KeyBackedTypes.actor.h"
#include "fdbclient/KeyRangeMap.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/Notified.h"
#include "fdbclient/StatusClient.h"
#include "fdbclient/StorageServerShard.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/Tenant.h"
#include "fdbclient/TransactionLineage.h"
#include "fdbclient/Tuple.h"
#include "fdbclient/VersionedMap.h"
#include "fdbrpc/sim_validation.h"
#include "fdbrpc/Smoother.h"
#include "fdbrpc/Stats.h"
#include "fdbserver/AccumulativeChecksumUtil.h"
#include "fdbserver/BulkDumpUtil.actor.h"
#include "fdbserver/BulkLoadUtil.actor.h"
#include "fdbserver/DataDistribution.actor.h"
#include "fdbserver/FDBExecHelper.actor.h"
#include "fdbclient/GetEncryptCipherKeys.h"
#include "fdbserver/IKeyValueStore.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/LatencyBandConfig.h"
#include "fdbserver/LogProtocolMessage.h"
#include "fdbserver/LogSystem.h"
#include "fdbserver/MoveKeys.actor.h"
#include "fdbserver/MutationTracking.h"
#include "fdbserver/OTELSpanContextMessage.h"
#include "fdbserver/Ratekeeper.h"
#include "fdbserver/RecoveryState.h"
#include "fdbserver/RocksDBCheckpointUtils.actor.h"
#include "fdbserver/ServerCheckpoint.actor.h"
#include "fdbserver/ServerDBInfo.h"
#include "fdbserver/SpanContextMessage.h"
#include "fdbserver/StorageMetrics.actor.h"
#include "fdbserver/TLogInterface.h"
#include "fdbserver/TransactionTagCounter.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "fdbserver/StorageCorruptionBug.h"
#include "fdbserver/StorageServerUtils.h"
#include "flow/ActorCollection.h"
#include "flow/Arena.h"
#include "flow/Error.h"
#include "flow/Hash3.h"
#include "flow/Histogram.h"
#include "flow/IRandom.h"
#include "flow/IndexedSet.h"
#include "flow/SystemMonitor.h"
#include "flow/TDMetric.actor.h"
#include "flow/Trace.h"
#include "flow/Util.h"
#include "flow/genericactors.actor.h"
#include "fdbserver/FDBRocksDBVersion.h"

#include "flow/actorcompiler.h" // This must be the last #include.

#ifndef __INTEL_COMPILER
#pragma region Data Structures
#endif

#define SHORT_CIRCUT_ACTUAL_STORAGE 0

namespace {
enum ChangeServerKeysContext { CSK_UPDATE, CSK_RESTORE, CSK_ASSIGN_EMPTY, CSK_FALL_BACK };

std::string changeServerKeysContextName(const ChangeServerKeysContext& context) {
	switch (context) {
	case CSK_UPDATE:
		return "Update";
	case CSK_RESTORE:
		return "Restore";
	case CSK_ASSIGN_EMPTY:
		return "AssignEmpty";
	case CSK_FALL_BACK:
		return "FallBackToFetchKeys";
	default:
		ASSERT(false);
	}
	return "UnknownContext";
}

bool canReplyWith(Error e) {
	switch (e.code()) {
	case error_code_transaction_too_old:
	case error_code_future_version:
	case error_code_wrong_shard_server:
	case error_code_process_behind:
	case error_code_watch_cancelled:
	case error_code_server_overloaded:
	case error_code_tenant_name_required:
	case error_code_tenant_removed:
	case error_code_tenant_not_found:
	case error_code_tenant_locked:
	// getMappedRange related exceptions that are not retriable:
	case error_code_mapper_bad_index:
	case error_code_mapper_no_such_key:
	case error_code_mapper_bad_range_decriptor:
	case error_code_quick_get_key_values_has_more:
	case error_code_quick_get_value_miss:
	case error_code_quick_get_key_values_miss:
	case error_code_get_mapped_key_values_has_more:
	case error_code_key_not_tuple:
	case error_code_value_not_tuple:
	case error_code_mapper_not_tuple:
		// case error_code_all_alternatives_failed:
		return true;
	default:
		return false;
	}
}

} // namespace

#define PERSIST_PREFIX "\xff\xff"

FDB_BOOLEAN_PARAM(UnlimitedCommitBytes);
FDB_BOOLEAN_PARAM(MoveInFailed);
FDB_BOOLEAN_PARAM(MoveInUpdatesSpilled);

// Immutable
static const KeyValueRef persistFormat(PERSIST_PREFIX "Format"_sr, "FoundationDB/StorageServer/1/4"_sr);
static const KeyValueRef persistShardAwareFormat(PERSIST_PREFIX "Format"_sr, "FoundationDB/StorageServer/1/5"_sr);
static const KeyRangeRef persistFormatReadableRange("FoundationDB/StorageServer/1/2"_sr,
                                                    "FoundationDB/StorageServer/1/6"_sr);
static const KeyRef persistID = PERSIST_PREFIX "ID"_sr;
static const KeyRef persistTssPairID = PERSIST_PREFIX "tssPairID"_sr;
static const KeyRef persistSSPairID = PERSIST_PREFIX "ssWithTSSPairID"_sr;
static const KeyRef persistTssQuarantine = PERSIST_PREFIX "tssQ"_sr;

// (Potentially) change with the durable version or when fetchKeys completes
static const KeyRef persistVersion = PERSIST_PREFIX "Version"_sr;
static const KeyRangeRef persistShardAssignedKeys =
    KeyRangeRef(PERSIST_PREFIX "ShardAssigned/"_sr, PERSIST_PREFIX "ShardAssigned0"_sr);
static const KeyRangeRef persistShardAvailableKeys =
    KeyRangeRef(PERSIST_PREFIX "ShardAvailable/"_sr, PERSIST_PREFIX "ShardAvailable0"_sr);
static const KeyRangeRef persistByteSampleKeys = KeyRangeRef(PERSIST_PREFIX "BS/"_sr, PERSIST_PREFIX "BS0"_sr);
static const KeyRangeRef persistByteSampleSampleKeys =
    KeyRangeRef(PERSIST_PREFIX "BS/"_sr PERSIST_PREFIX "BS/"_sr, PERSIST_PREFIX "BS/"_sr PERSIST_PREFIX "BS0"_sr);
static const KeyRef persistLogProtocol = PERSIST_PREFIX "LogProtocol"_sr;
static const KeyRef persistPrimaryLocality = PERSIST_PREFIX "PrimaryLocality"_sr;
static const KeyRangeRef persistTenantMapKeys = KeyRangeRef(PERSIST_PREFIX "TM/"_sr, PERSIST_PREFIX "TM0"_sr);
// data keys are unmangled (but never start with PERSIST_PREFIX because they are always in allKeys)

static const KeyRangeRef persistStorageServerShardKeys =
    KeyRangeRef(PERSIST_PREFIX "StorageServerShard/"_sr, PERSIST_PREFIX "StorageServerShard0"_sr);

// Checkpoint related prefixes.
static const KeyRangeRef persistCheckpointKeys =
    KeyRangeRef(PERSIST_PREFIX "Checkpoint/"_sr, PERSIST_PREFIX "Checkpoint0"_sr);
static const KeyRangeRef persistPendingCheckpointKeys =
    KeyRangeRef(PERSIST_PREFIX "PendingCheckpoint/"_sr, PERSIST_PREFIX "PendingCheckpoint0"_sr);
static const std::string serverCheckpointFolder = "serverCheckpoints";
static const std::string checkpointBytesSampleTempFolder = "/metadata_temp";
static const std::string fetchedCheckpointFolder = "fetchedCheckpoints";
static const std::string serverBulkDumpFolder = "bulkDumpFiles";
static const std::string serverBulkLoadFolder = "bulkLoadFiles";
static const std::string invalidTeamId = "InvalidTeam";

static const KeyRangeRef persistBulkLoadTaskKeys =
    KeyRangeRef(PERSIST_PREFIX "BulkLoadTask/"_sr, PERSIST_PREFIX "BulkLoadTask0"_sr);

// Accumulative checksum related prefix
static const KeyRangeRef persistAccumulativeChecksumKeys =
    KeyRangeRef(PERSIST_PREFIX "AccumulativeChecksum/"_sr, PERSIST_PREFIX "AccumulativeChecksum0"_sr);

inline Key encodePersistAccumulativeChecksumKey(uint16_t acsIndex) {
	BinaryWriter wr(Unversioned());
	wr.serializeBytes(persistAccumulativeChecksumKeys.begin);
	wr << bigEndian16(acsIndex);
	return wr.toValue();
}

inline uint16_t decodePersistAccumulativeChecksumKey(const Key& key) {
	uint16_t acsIndex;
	BinaryReader rd(key.removePrefix(persistAccumulativeChecksumKeys.begin), Unversioned());
	rd >> acsIndex;
	return bigEndian16(acsIndex);
}

// MoveInUpdates caches new updates of a move-in shard, before that shard is ready to accept writes.
struct MoveInUpdates {
	MoveInUpdates() : spilled(MoveInUpdatesSpilled::False) {}
	MoveInUpdates(UID id,
	              Version version,
	              struct StorageServer* data,
	              IKeyValueStore* store,
	              MoveInUpdatesSpilled spilled);

	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation,
	                 bool allowSpill);

	bool hasNext() const;

	std::vector<Standalone<VerUpdateRef>> next(const int byteLimit);
	const std::deque<Standalone<VerUpdateRef>>& getUpdatesQueue() const { return this->updates; }

	UID id;
	Version lastRepliedVersion;
	std::deque<Standalone<VerUpdateRef>> updates;
	std::vector<Standalone<VerUpdateRef>> spillBuffer;
	struct StorageServer* data;
	IKeyValueStore* store;
	KeyRange range;
	bool fail;
	MoveInUpdatesSpilled spilled;
	size_t size;
	Future<Void> loadFuture;
	Severity logSev;

private:
																#line 277 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
[[nodiscard]] static Future<Void> loadUpdates( MoveInUpdates* const& self, Version const& begin, Version const& end );
template <class> friend class MoveInUpdates_LoadUpdatesActorState;

#line 276 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

	Key getPersistKey(const Version version, const int idx) const;
};

															#line 286 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
// This generated class is to be used only via loadUpdates()
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class MoveInUpdates_LoadUpdatesActor>
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class MoveInUpdates_LoadUpdatesActorState {
															#line 292 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	MoveInUpdates_LoadUpdatesActorState(MoveInUpdates* const& self,Version const& begin,Version const& end) 
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   begin(begin),
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   end(end)
															#line 303 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("loadUpdates", reinterpret_cast<unsigned long>(this));

	}
	~MoveInUpdates_LoadUpdatesActorState() 
	{
		fdb_probe_actor_destroy("loadUpdates", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 281 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(self->spilled);
															#line 282 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (begin >= end)
															#line 320 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 283 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				self->spilled = MoveInUpdatesSpilled::False;
															#line 284 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveInUpdates_LoadUpdatesActorState(); static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->destroy(); return 0; }
															#line 326 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~MoveInUpdates_LoadUpdatesActorState();
				static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 287 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const Key beginKey = persistUpdatesKey(self->id, begin), endKey = persistUpdatesKey(self->id, end);
															#line 288 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(self->logSev, "MoveInUpdatesLoadBegin", self->id) .detail("BeginVersion", begin) .detail("EndVersion", end) .detail("BeginKey", beginKey) .detail("EndKey", endKey);
															#line 293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(beginKey < endKey);
															#line 294 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = self->store->readRange(KeyRangeRef(beginKey, endKey), SERVER_KNOBS->FETCH_SHARD_UPDATES_BYTE_LIMIT, SERVER_KNOBS->FETCH_SHARD_UPDATES_BYTE_LIMIT);
															#line 294 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 342 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state = 1;
															#line 294 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*>(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)));
															#line 347 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MoveInUpdates_LoadUpdatesActorState();
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult const& res,int loopDepth) 
	{
															#line 297 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		std::vector<Standalone<VerUpdateRef>> restored;
															#line 298 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 299 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const Version version = decodePersistUpdateVersion(res[i].key.removePrefix(self->range.begin));
															#line 300 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Standalone<VerUpdateRef> vur = BinaryReader::fromStringRef<Standalone<VerUpdateRef>>(res[i].value, IncludeVersion());
															#line 302 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(version == vur.version);
															#line 303 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(self->logSev, "MoveInUpdatesLoadedMutations", self->id) .detail("Version", version) .detail("Mutations", vur.mutations.size());
															#line 306 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			restored.push_back(std::move(vur));
															#line 382 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 309 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!res.more)
															#line 386 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 310 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = restored.size() - 1;i >= 0;--i) {
															#line 311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (self->updates.empty() || restored[i].version < self->updates.front().version)
															#line 392 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 312 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					self->updates.push_front(std::move(restored[i]));
															#line 396 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 315 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			self->spilled = MoveInUpdatesSpilled::False;
															#line 401 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 317 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(self->spillBuffer.empty());
															#line 318 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::swap(self->spillBuffer, restored);
															#line 409 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 321 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->loadFuture = Future<Void>();
															#line 322 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(self->logSev, "MoveInUpdatesLoadEnd", self->id) .detail("MinVersion", restored.empty() ? invalidVersion : restored.front().version) .detail("MaxVersion", restored.empty() ? invalidVersion : restored.back().version) .detail("VersionCount", restored.size()) .detail("LastBatch", !res.more);
															#line 328 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveInUpdates_LoadUpdatesActorState(); static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->destroy(); return 0; }
															#line 417 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MoveInUpdates_LoadUpdatesActorState();
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult && res,int loopDepth) 
	{
															#line 297 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		std::vector<Standalone<VerUpdateRef>> restored;
															#line 298 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 299 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const Version version = decodePersistUpdateVersion(res[i].key.removePrefix(self->range.begin));
															#line 300 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Standalone<VerUpdateRef> vur = BinaryReader::fromStringRef<Standalone<VerUpdateRef>>(res[i].value, IncludeVersion());
															#line 302 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(version == vur.version);
															#line 303 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(self->logSev, "MoveInUpdatesLoadedMutations", self->id) .detail("Version", version) .detail("Mutations", vur.mutations.size());
															#line 306 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			restored.push_back(std::move(vur));
															#line 441 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 309 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!res.more)
															#line 445 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 310 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = restored.size() - 1;i >= 0;--i) {
															#line 311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (self->updates.empty() || restored[i].version < self->updates.front().version)
															#line 451 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 312 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					self->updates.push_front(std::move(restored[i]));
															#line 455 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 315 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			self->spilled = MoveInUpdatesSpilled::False;
															#line 460 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 317 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(self->spillBuffer.empty());
															#line 318 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::swap(self->spillBuffer, restored);
															#line 468 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 321 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->loadFuture = Future<Void>();
															#line 322 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(self->logSev, "MoveInUpdatesLoadEnd", self->id) .detail("MinVersion", restored.empty() ? invalidVersion : restored.front().version) .detail("MaxVersion", restored.empty() ? invalidVersion : restored.back().version) .detail("VersionCount", restored.size()) .detail("LastBatch", !res.more);
															#line 328 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MoveInUpdates_LoadUpdatesActorState(); static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->destroy(); return 0; }
															#line 476 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MoveInUpdates_LoadUpdatesActorState();
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(RangeResult const& res,int loopDepth) 
	{
		loopDepth = a_body1cont1(res, loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && res,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state > 0) static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->actor_wait_state = 0;
		static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	MoveInUpdates* self;
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version begin;
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version end;
															#line 565 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via loadUpdates()
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class MoveInUpdates_LoadUpdatesActor final : public Actor<Void>, public ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >, public FastAllocated<MoveInUpdates_LoadUpdatesActor>, public MoveInUpdates_LoadUpdatesActorState<MoveInUpdates_LoadUpdatesActor> {
															#line 570 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<MoveInUpdates_LoadUpdatesActor>::operator new;
	using FastAllocated<MoveInUpdates_LoadUpdatesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8402420762850946304UL, 4048829975540609792UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >;
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	MoveInUpdates_LoadUpdatesActor(MoveInUpdates* const& self,Version const& begin,Version const& end) 
															#line 587 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   MoveInUpdates_LoadUpdatesActorState<MoveInUpdates_LoadUpdatesActor>(self, begin, end),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadUpdates", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5561495993262577152UL, 4925366652637659904UL);
		ActorExecutionContextHelper __helper(static_cast<MoveInUpdates_LoadUpdatesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadUpdates");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadUpdates", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MoveInUpdates_LoadUpdatesActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> MoveInUpdates::loadUpdates( MoveInUpdates* const& self, Version const& begin, Version const& end ) {
															#line 280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new MoveInUpdates_LoadUpdatesActor(self, begin, end));
															#line 619 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 330 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

bool MoveInUpdates::hasNext() const {
	return this->spilled || (!this->updates.empty() && this->updates.back().version > this->lastRepliedVersion);
}

// MoveInShard corresponds to a move-in physical shard, a class representation of MoveInShardMetaData.
struct MoveInShard {
	std::shared_ptr<MoveInShardMetaData> meta;
	struct StorageServer* server;
	std::shared_ptr<MoveInUpdates> updates;
	bool isRestored;
	Version transferredVersion;
	ConductBulkLoad conductBulkLoad = ConductBulkLoad::False;

	Future<Void> fetchClient; // holds FetchShard() actor
	Promise<Void> fetchComplete;
	Promise<Void> readWrite;

	Severity logSev = static_cast<Severity>(SERVER_KNOBS->PHYSICAL_SHARD_MOVE_LOG_SEVERITY);

	MoveInShard() = default;
	MoveInShard(StorageServer* server,
	            const UID& id,
	            const UID& dataMoveId,
	            const Version version,
	            const ConductBulkLoad conductBulkLoad,
	            MoveInPhase phase);
	MoveInShard(StorageServer* server,
	            const UID& id,
	            const UID& dataMoveId,
	            const Version version,
	            const ConductBulkLoad conductBulkLoad);
	MoveInShard(StorageServer* server, MoveInShardMetaData meta);
	~MoveInShard();

	UID id() const { return this->meta->id; }
	UID dataMoveId() const { return this->meta->dataMoveId; }
	void setPhase(const MoveInPhase& phase) { this->meta->setPhase(phase); }
	MoveInPhase getPhase() const { return this->meta->getPhase(); }
	const std::vector<KeyRange>& ranges() const { return this->meta->ranges; }
	const std::vector<CheckpointMetaData>& checkpoints() const { return this->meta->checkpoints; }
	std::string destShardIdString() const { return this->meta->destShardIdString(); }
	void addRange(const KeyRangeRef range);
	void removeRange(const KeyRangeRef range);
	void cancel(const MoveInFailed failed = MoveInFailed::False);
	bool isDataTransferred() const { return meta->getPhase() >= MoveInPhase::ApplyingUpdates; }
	bool isDataAndCFTransferred() const { throw not_implemented(); }
	bool failed() const { return this->getPhase() == MoveInPhase::Cancel || this->getPhase() == MoveInPhase::Error; }
	void setHighWatermark(const Version version) { this->meta->highWatermark = version; }
	Version getHighWatermark() const { return this->meta->highWatermark; }

	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation);

	KeyRangeRef getAffectedRange(const MutationRef& mutation) const;

	std::string toString() const { return meta != nullptr ? meta->toString() : "Empty"; }
};

struct AddingShard : NonCopyable {
	KeyRange keys;
	Future<Void> fetchClient; // holds FetchKeys() actor
	Promise<Void> fetchComplete;
	Promise<Void> readWrite;
	DataMovementReason reason;
	SSBulkLoadMetadata ssBulkLoadMetadata;
	std::string teamId;

	// During the Fetching phase, it saves newer mutations whose version is greater or equal to fetchClient's
	// fetchVersion, while the shard is still busy catching up with fetchClient. It applies these updates after fetching
	// completes.
	std::deque<Standalone<VerUpdateRef>> updates;

	struct StorageServer* server;
	Version transferredVersion;
	Version fetchVersion;

	// To learn more details of the phase transitions, see function fetchKeys(). The phases below are sorted in
	// chronological order and do not go back.
	enum Phase {
		WaitPrevious,
		// During Fetching phase, it fetches data before fetchVersion and write it to storage, then let updater know it
		// is ready to update the deferred updates` (see the comment of member variable `updates` above).
		Fetching,
		// During the FetchingCF phase, the shard data is transferred but the remaining change feed data is still being
		// transferred. This is equivalent to the waiting phase for non-changefeed data.
		// TODO(gglass): remove FetchingCF.  Probably requires some refactoring of permanent logic,
		// not just flat out removal of CF-specific logic, so come back to this.
		FetchingCF,
		// During Waiting phase, it sends updater the deferred updates, and wait until they are durable.
		Waiting
		// The shard's state is changed from adding to readWrite then.
	};

	Phase phase;

	AddingShard(StorageServer* server,
	            KeyRangeRef const& keys,
	            DataMovementReason reason,
	            const SSBulkLoadMetadata& ssBulkLoadMetadata);

	// When fetchKeys "partially completes" (splits an adding shard in two), this is used to construct the left half
	AddingShard(AddingShard* prev, KeyRange const& keys)
	  : keys(keys), fetchClient(prev->fetchClient), server(prev->server), transferredVersion(prev->transferredVersion),
	    fetchVersion(prev->fetchVersion), phase(prev->phase), reason(prev->reason),
	    ssBulkLoadMetadata(prev->ssBulkLoadMetadata) {}
	~AddingShard() {
		if (!fetchComplete.isSet())
			fetchComplete.send(Void());
		if (!readWrite.isSet())
			readWrite.send(Void());
	}

	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation);

	bool isDataTransferred() const { return phase >= FetchingCF; }
	bool isDataAndCFTransferred() const { return phase >= Waiting; }

	SSBulkLoadMetadata getSSBulkLoadMetadata() const { return ssBulkLoadMetadata; }
};

class ShardInfo : public ReferenceCounted<ShardInfo>, NonCopyable {
private:
	ShardInfo(KeyRange keys, std::unique_ptr<AddingShard>&& adding, StorageServer* readWrite)
	  : adding(std::move(adding)), readWrite(readWrite), keys(keys), shardId(0LL), desiredShardId(0LL), version(0) {}
	ShardInfo(KeyRange keys, std::shared_ptr<MoveInShard> moveInShard)
	  : adding(nullptr), readWrite(nullptr), moveInShard(moveInShard), keys(keys),
	    shardId(moveInShard->meta->destShardId()), desiredShardId(moveInShard->meta->destShardId()),
	    version(moveInShard->meta->createVersion) {}

	// A shard has 4 mutual exclusive states: adding, moveInShard, readWrite and notAssigned.
	std::unique_ptr<AddingShard> adding;
	struct StorageServer* readWrite;
	std::shared_ptr<MoveInShard> moveInShard; // The shard is being moved in via physical-shard-move.
	KeyRange keys;
	uint64_t changeCounter;
	uint64_t shardId;
	uint64_t desiredShardId;
	std::string teamId = invalidTeamId;
	Version version;

public:
	static ShardInfo* newNotAssigned(KeyRange keys) { return new ShardInfo(keys, nullptr, nullptr); }
	static ShardInfo* newReadWrite(KeyRange keys, StorageServer* data) { return new ShardInfo(keys, nullptr, data); }
	static ShardInfo* newAdding(StorageServer* data,
	                            KeyRange keys,
	                            DataMovementReason reason,
	                            const SSBulkLoadMetadata& ssBulkLoadMetadata) {
		return new ShardInfo(keys, std::make_unique<AddingShard>(data, keys, reason, ssBulkLoadMetadata), nullptr);
	}
	static ShardInfo* addingSplitLeft(KeyRange keys, AddingShard* oldShard) {
		return new ShardInfo(keys, std::make_unique<AddingShard>(oldShard, keys), nullptr);
	}

	static ShardInfo* newShard(StorageServer* data, const StorageServerShard& shard);

	static bool canMerge(const ShardInfo* l, const ShardInfo* r) {
		if (l == nullptr || r == nullptr || l->keys.end != r->keys.begin || l->version == invalidVersion ||
		    r->version == invalidVersion) {
			return false;
		}
		if (l->getShardId() != r->getShardId() || l->getDesiredShardId() != r->getDesiredShardId()) {
			return false;
		}
		return (l->isReadable() && r->isReadable()) || (!l->assigned() && !r->assigned());
	}

	StorageServerShard toStorageServerShard() const {
		StorageServerShard::ShardState st = StorageServerShard::NotAssigned;
		Optional<UID> moveInShardId;
		if (this->isReadable()) {
			st = StorageServerShard::ReadWrite;
		} else if (!this->assigned()) {
			st = StorageServerShard::NotAssigned;
		} else if (this->getAddingShard()) {
			st = this->getAddingShard()->phase == AddingShard::Waiting ? StorageServerShard::ReadWritePending
			                                                           : StorageServerShard::Adding;
		} else {
			ASSERT(this->getMoveInShard());
			const MoveInPhase phase = this->getMoveInShard()->getPhase();
			if (phase < MoveInPhase::ReadWritePending) {
				st = StorageServerShard::MovingIn;
			} else if (phase == MoveInPhase::ReadWritePending) {
				st = StorageServerShard::ReadWritePending;
			} else if (phase == MoveInPhase::Complete) {
				st = StorageServerShard::ReadWrite;
			} else {
				st = StorageServerShard::MovingIn;
			}
			// Clear moveInShardId if the data move is complete.
			if (phase != MoveInPhase::ReadWritePending && phase != MoveInPhase::Complete) {
				moveInShardId = this->getMoveInShard()->id();
			}
		}
		return StorageServerShard(this->keys, this->version, this->shardId, this->desiredShardId, st, moveInShardId);
	}

	// Copies necessary information from `shard`.
	void populateShard(const StorageServerShard& shard) {
		this->version = shard.version;
		this->shardId = shard.id;
		this->desiredShardId = shard.desiredId;
	}

	// Returns true if the current shard is merged with `other`.
	bool mergeWith(const ShardInfo* other) {
		if (!canMerge(this, other)) {
			return false;
		}
		this->keys = KeyRangeRef(this->keys.begin, other->range().end);
		this->version = std::max(this->version, other->getVersion());
		return true;
	}

	void validate() const {
		// TODO: Complete this.
	}

	bool isReadable() const { return readWrite != nullptr; }
	bool notAssigned() const { return !readWrite && !adding && !moveInShard; }
	bool assigned() const { return readWrite || adding || moveInShard; }

	KeyRange range() const { return keys; }
	uint64_t getShardId() const { return shardId; }
	uint64_t getDesiredShardId() const { return desiredShardId; }
	uint64_t getChangeCounter() const { return changeCounter; }
	AddingShard* getAddingShard() const { return adding.get(); }
	std::shared_ptr<MoveInShard> getMoveInShard() const { return moveInShard; }
	Version getVersion() const { return version; }
	std::string getTeamId() const { return teamId; }

	void setChangeCounter(uint64_t shardChangeCounter) { changeCounter = shardChangeCounter; }
	void setShardId(uint64_t id) { shardId = id; }
	void setDesiredShardId(uint64_t id) { desiredShardId = id; }

	bool isInVersionedData() const {
		return readWrite || (adding && adding->isDataTransferred()) ||
		       (moveInShard && moveInShard->isDataTransferred());
	}
	bool isCFInVersionedData() const { return readWrite || (adding && adding->isDataAndCFTransferred()); }
	bool isReadWritePending() const {
		return isCFInVersionedData() || (moveInShard && (moveInShard->getPhase() == MoveInPhase::ReadWritePending ||
		                                                 moveInShard->getPhase() == MoveInPhase::Complete));
	}
	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation);
	bool isFetched() const {
		return readWrite || (adding && adding->fetchComplete.isSet()) ||
		       (moveInShard && moveInShard->fetchComplete.isSet());
	}

	bool isFetching() const {
		if (adding) {
			return !adding->fetchComplete.isSet();
		}
		if (moveInShard) {
			return !moveInShard->fetchComplete.isSet();
		}
		return false;
	}

	std::string debugDescribeState() const {
		if (notAssigned()) {
			return "NotAssigned";
		} else if (adding && !adding->isDataAndCFTransferred()) {
			return "AddingFetchingCF";
		} else if (adding && !adding->isDataTransferred()) {
			return "AddingFetching";
		} else if (adding) {
			return "AddingTransferred";
		} else if (moveInShard) {
			return moveInShard->meta->toString();
		} else {
			return "ReadWrite";
		}
	}
};

struct StorageServerDisk {
	explicit StorageServerDisk(struct StorageServer* data, IKeyValueStore* storage) : data(data), storage(storage) {}

	IKeyValueStore* getKeyValueStore() const { return this->storage; }

	void makeNewStorageServerDurable(const bool shardAware);
	bool makeVersionMutationsDurable(Version& prevStorageVersion,
	                                 Version newStorageVersion,
	                                 int64_t& bytesLeft,
	                                 UnlimitedCommitBytes unlimitedCommitBytes,
	                                 int64_t& clearRangesLeft,
	                                 const UID& ssId,
	                                 bool verbose = false);
	void makeVersionDurable(Version version);
	void makeAccumulativeChecksumDurable(const AccumulativeChecksumState& acsState);
	void clearAccumulativeChecksumState(const AccumulativeChecksumState& acsState);
	void makeTssQuarantineDurable();
	Future<bool> restoreDurableState();

	void changeLogProtocol(Version version, ProtocolVersion protocol);

	void writeMutation(MutationRef mutation);
	void writeKeyValue(KeyValueRef kv);
	void clearRange(KeyRangeRef keys);

	Future<Void> addRange(KeyRangeRef range, std::string id) {
		return storage->addRange(range, id, !SERVER_KNOBS->SHARDED_ROCKSDB_DELAY_COMPACTION_FOR_DATA_MOVE);
	}

	std::vector<std::string> removeRange(KeyRangeRef range) { return storage->removeRange(range); }

	void markRangeAsActive(KeyRangeRef range) { storage->markRangeAsActive(range); }

	Future<Void> replaceRange(KeyRange range, Standalone<VectorRef<KeyValueRef>> data) {
		return storage->replaceRange(range, data);
	}

	void persistRangeMapping(KeyRangeRef range, bool isAdd) { storage->persistRangeMapping(range, isAdd); }

	CoalescedKeyRangeMap<std::string> getExistingRanges() { return storage->getExistingRanges(); }

	Future<Void> getError() { return storage->getError(); }
	Future<Void> init() { return storage->init(); }
	Future<Void> canCommit() { return storage->canCommit(); }
	Future<Void> commit() { return storage->commit(); }

	void logRecentRocksDBBackgroundWorkStats(UID ssId, std::string logReason) {
		return storage->logRecentRocksDBBackgroundWorkStats(ssId, logReason);
	}

	// SOMEDAY: Put readNextKeyInclusive in IKeyValueStore
	// Read the key that is equal or greater then 'key' from the storage engine.
	// For example, readNextKeyInclusive("a") should return:
	//  - "a", if key "a" exist
	//  - "b", if key "a" doesn't exist, and "b" is the next existing key in total order
	//  - allKeys.end, if keyrange [a, allKeys.end) is empty
	Future<Key> readNextKeyInclusive(KeyRef key, Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvScans);
		return readFirstKey(storage, KeyRangeRef(key, allKeys.end), options);
	}
	Future<Optional<Value>> readValue(KeyRef key, Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvGets);
		return storage->readValue(key, options);
	}
	Future<Optional<Value>> readValuePrefix(KeyRef key,
	                                        int maxLength,
	                                        Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvGets);
		return storage->readValuePrefix(key, maxLength, options);
	}
	Future<RangeResult> readRange(KeyRangeRef keys,
	                              int rowLimit = 1 << 30,
	                              int byteLimit = 1 << 30,
	                              Optional<ReadOptions> options = Optional<ReadOptions>()) {
		++(*kvScans);
		return storage->readRange(keys, rowLimit, byteLimit, options);
	}

	Future<CheckpointMetaData> checkpoint(const CheckpointRequest& request) { return storage->checkpoint(request); }

	Future<Void> restore(const std::vector<CheckpointMetaData>& checkpoints) { return storage->restore(checkpoints); }

	Future<Void> restore(const std::string& shardId,
	                     const std::vector<KeyRange>& ranges,
	                     const std::vector<CheckpointMetaData>& checkpoints) {
		return storage->restore(shardId, ranges, checkpoints);
	}

	Future<Void> deleteCheckpoint(const CheckpointMetaData& checkpoint) {
		return storage->deleteCheckpoint(checkpoint);
	}

	KeyValueStoreType getKeyValueStoreType() const { return storage->getType(); }
	StorageBytes getStorageBytes() const { return storage->getStorageBytes(); }
	std::tuple<size_t, size_t, size_t> getSize() const { return storage->getSize(); }

	Future<EncryptionAtRestMode> encryptionMode() { return storage->encryptionMode(); }

	// The following are pointers to the Counters in StorageServer::counters of the same names.
	Counter* kvCommitLogicalBytes;
	Counter* kvClearRanges;
	Counter* kvClearSingleKey;
	Counter* kvGets;
	Counter* kvScans;
	Counter* kvCommits;

private:
	struct StorageServer* data;
	IKeyValueStore* storage;
	void writeMutations(const VectorRef<MutationRef>& mutations, Version debugVersion, const char* debugContext);
	void writeMutationsBuggy(const VectorRef<MutationRef>& mutations, Version debugVersion, const char* debugContext);

																#line 1020 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
// This generated class is to be used only via readFirstKey()
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class ReadFirstKeyActor>
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadFirstKeyActorState {
															#line 1026 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadFirstKeyActorState(IKeyValueStore* const& storage,KeyRangeRef const& range,Optional<ReadOptions> const& options) 
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : storage(storage),
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   range(range),
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   options(options)
															#line 1037 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("readFirstKey", reinterpret_cast<unsigned long>(this));

	}
	~ReadFirstKeyActorState() 
	{
		fdb_probe_actor_destroy("readFirstKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 728 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = storage->readRange(range, 1, 1 << 30, options);
															#line 728 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<ReadFirstKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1054 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadFirstKeyActor*>(this)->actor_wait_state = 1;
															#line 728 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadFirstKeyActor, 0, RangeResult >*>(static_cast<ReadFirstKeyActor*>(this)));
															#line 1059 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadFirstKeyActorState();
		static_cast<ReadFirstKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(RangeResult const& r,int loopDepth) 
	{
															#line 729 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.size())
															#line 1082 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 730 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(r[0].key); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1086 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(r[0].key);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 732 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(range.end); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1096 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(range.end);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}

		return loopDepth;
	}
	int a_body1cont1(RangeResult && r,int loopDepth) 
	{
															#line 729 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.size())
															#line 1109 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 730 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(r[0].key); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1113 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(r[0].key);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 732 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<ReadFirstKeyActor*>(this)->SAV<Key>::futures) { (void)(range.end); this->~ReadFirstKeyActorState(); static_cast<ReadFirstKeyActor*>(this)->destroy(); return 0; }
															#line 1123 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<ReadFirstKeyActor*>(this)->SAV< Key >::value()) Key(range.end);
			this->~ReadFirstKeyActorState();
			static_cast<ReadFirstKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}

		return loopDepth;
	}
	int a_body1when1(RangeResult const& r,int loopDepth) 
	{
		loopDepth = a_body1cont1(r, loopDepth);

		return loopDepth;
	}
	int a_body1when1(RangeResult && r,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadFirstKeyActor*>(this)->actor_wait_state > 0) static_cast<ReadFirstKeyActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFirstKeyActor*>(this)->ActorCallback< ReadFirstKeyActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFirstKeyActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadFirstKeyActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadFirstKeyActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	IKeyValueStore* storage;
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRangeRef range;
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<ReadOptions> options;
															#line 1213 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via readFirstKey()
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadFirstKeyActor final : public Actor<Key>, public ActorCallback< ReadFirstKeyActor, 0, RangeResult >, public FastAllocated<ReadFirstKeyActor>, public ReadFirstKeyActorState<ReadFirstKeyActor> {
															#line 1218 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<ReadFirstKeyActor>::operator new;
	using FastAllocated<ReadFirstKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1459044102044457216UL, 571999524660955648UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Key>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadFirstKeyActor, 0, RangeResult >;
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadFirstKeyActor(IKeyValueStore* const& storage,KeyRangeRef const& range,Optional<ReadOptions> const& options) 
															#line 1235 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Key>(),
		   ReadFirstKeyActorState<ReadFirstKeyActor>(storage, range, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readFirstKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4418144802222492160UL, 5296783460543853568UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFirstKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readFirstKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readFirstKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadFirstKeyActor, 0, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] static Future<Key> readFirstKey( IKeyValueStore* const& storage, KeyRangeRef const& range, Optional<ReadOptions> const& options ) {
															#line 727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Key>(new ReadFirstKeyActor(storage, range, options));
															#line 1267 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 734 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
};

struct UpdateEagerReadInfo {
	std::vector<KeyRef> keyBegin;
	std::vector<Key> keyEnd; // these are for ClearRange

	std::vector<std::pair<KeyRef, int>> keys;
	std::vector<Optional<Value>> value;

	Arena arena;
	bool enableClearRangeEagerReads;

	UpdateEagerReadInfo(bool enableClearRangeEagerReads) : enableClearRangeEagerReads(enableClearRangeEagerReads) {}

	void addMutations(VectorRef<MutationRef> const& mutations) {
		for (auto& m : mutations)
			addMutation(m);
	}

	void addMutation(MutationRef const& m) {
		// SOMEDAY: Theoretically we can avoid a read if there is an earlier overlapping ClearRange
		if (m.type == MutationRef::ClearRange && !m.param2.startsWith(systemKeys.end) && enableClearRangeEagerReads)
			keyBegin.push_back(m.param2);
		else if (m.type == MutationRef::CompareAndClear) {
			if (enableClearRangeEagerReads)
				keyBegin.push_back(keyAfter(m.param1, arena));
			if (keys.size() > 0 && keys.back().first == m.param1) {
				// Don't issue a second read, if the last read was equal to the current key.
				// CompareAndClear is likely to be used after another atomic operation on same key.
				keys.back().second = std::max(keys.back().second, m.param2.size() + 1);
			} else {
				keys.emplace_back(m.param1, m.param2.size() + 1);
			}
		} else if ((m.type == MutationRef::AppendIfFits) || (m.type == MutationRef::ByteMin) ||
		           (m.type == MutationRef::ByteMax))
			keys.emplace_back(m.param1, CLIENT_KNOBS->VALUE_SIZE_LIMIT);
		else if (isAtomicOp((MutationRef::Type)m.type))
			keys.emplace_back(m.param1, m.param2.size());
	}

	void finishKeyBegin() {
		if (enableClearRangeEagerReads) {
			std::sort(keyBegin.begin(), keyBegin.end());
			keyBegin.resize(std::unique(keyBegin.begin(), keyBegin.end()) - keyBegin.begin());
		}
		std::sort(keys.begin(), keys.end(), [](const std::pair<KeyRef, int>& lhs, const std::pair<KeyRef, int>& rhs) {
			return (lhs.first < rhs.first) || (lhs.first == rhs.first && lhs.second > rhs.second);
		});
		keys.resize(std::unique(keys.begin(),
		                        keys.end(),
		                        [](const std::pair<KeyRef, int>& lhs, const std::pair<KeyRef, int>& rhs) {
			                        return lhs.first == rhs.first;
		                        }) -
		            keys.begin());
		// value gets populated in doEagerReads
	}

	Optional<Value>& getValue(KeyRef key) {
		int i = std::lower_bound(keys.begin(),
		                         keys.end(),
		                         std::pair<KeyRef, int>(key, 0),
		                         [](const std::pair<KeyRef, int>& lhs, const std::pair<KeyRef, int>& rhs) {
			                         return lhs.first < rhs.first;
		                         }) -
		        keys.begin();
		ASSERT(i < keys.size() && keys[i].first == key);
		return value[i];
	}

	KeyRef getKeyEnd(KeyRef key) {
		int i = std::lower_bound(keyBegin.begin(), keyBegin.end(), key) - keyBegin.begin();
		ASSERT(i < keyBegin.size() && keyBegin[i] == key);
		return keyEnd[i];
	}
};

const int VERSION_OVERHEAD =
    64 + sizeof(Version) + sizeof(Standalone<VerUpdateRef>) + // mutationLog, 64b overhead for map
    2 * (64 + sizeof(Version) +
         sizeof(Reference<VersionedMap<KeyRef, ValueOrClearToRef>::PTreeT>)); // versioned map [ x2 for
                                                                              // createNewVersion(version+1) ], 64b
                                                                              // overhead for map

static int mvccStorageBytes(MutationRef const& m) {
	return mvccStorageBytes(m.param1.size() + m.param2.size());
}

struct FetchInjectionInfo {
	Arena arena;
	std::vector<VerUpdateRef> changes;
};

class ServerWatchMetadata : public ReferenceCounted<ServerWatchMetadata> {
public:
	Key key;
	Optional<Value> value;
	Version version;
	Future<Version> watch_impl;
	Promise<Version> versionPromise;
	Optional<TagSet> tags;
	Optional<UID> debugID;
	int64_t tenantId;

	ServerWatchMetadata(Key key,
	                    Optional<Value> value,
	                    Version version,
	                    Optional<TagSet> tags,
	                    Optional<UID> debugID,
	                    int64_t tenantId)
	  : key(key), value(value), version(version), tags(tags), debugID(debugID), tenantId(tenantId) {}
};

struct BusiestWriteTagContext {
	const std::string busiestWriteTagTrackingKey;
	UID ratekeeperID;
	Reference<EventCacheHolder> busiestWriteTagEventHolder;
	double lastUpdateTime;

	BusiestWriteTagContext(const UID& thisServerID)
	  : busiestWriteTagTrackingKey(thisServerID.toString() + "/BusiestWriteTag"), ratekeeperID(UID()),
	    busiestWriteTagEventHolder(makeReference<EventCacheHolder>(busiestWriteTagTrackingKey)), lastUpdateTime(-1) {}
};

struct TenantSSInfo {
	constexpr static FileIdentifier file_identifier = 3253114;
	TenantAPI::TenantLockState lockState;

	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, lockState);
	}
};

struct SSBulkLoadMetrics {
public:
	SSBulkLoadMetrics() : ongoingTasks(0) {}
	void addTask() { ongoingTasks++; }
	void removeTask() { ongoingTasks--; }
	int getOngoingTasks() { return ongoingTasks; }

private:
	int ongoingTasks = 0;
};

struct StorageServer : public IStorageMetricsService {
	typedef VersionedMap<KeyRef, ValueOrClearToRef> VersionedData;

private:
	// versionedData contains sets and clears.

	// * Nonoverlapping: No clear overlaps a set or another clear, or adjoins another clear.
	// ~ Clears are maximal: If versionedData.at(v) contains a clear [b,e) then
	//      there is a key data[e]@v, or e==allKeys.end, or a shard boundary or former boundary at e

	// * Reads are possible: When k is in a readable shard, for any v in [storageVersion, version.get()],
	//      storage[k] + versionedData.at(v)[k] = database[k] @ v    (storage[k] might be @ any version in
	//      [durableVersion, storageVersion])

	// * Transferred shards are partially readable: When k is in an adding, transferred shard, for any v in
	// [transferredVersion, version.get()],
	//      storage[k] + versionedData.at(v)[k] = database[k] @ v

	// * versionedData contains versions [storageVersion(), version.get()].  It might also contain version
	// (version.get()+1), in which changeDurableVersion may be deleting ghosts, and/or it might
	//      contain later versions if applyUpdate is on the stack.

	// * Old shards are erased: versionedData.atLatest() has entries (sets or intersecting clears) only for keys in
	// readable or adding,transferred shards.
	//   Earlier versions may have extra entries for shards that *were* readable or adding,transferred when those
	//   versions were the latest, but they eventually are forgotten.

	// * Old mutations are erased: All items in versionedData.atLatest() have insertVersion() > durableVersion(), but
	// views
	//   at older versions may contain older items which are also in storage (this is OK because of idempotency)

	VersionedData versionedData;
	std::map<Version, Standalone<VerUpdateRef>> mutationLog; // versions (durableVersion, version]

	using WatchMapKey = std::pair<int64_t, Key>;
	using WatchMapKeyHasher = boost::hash<WatchMapKey>;
	using WatchMapValue = Reference<ServerWatchMetadata>;
	using WatchMap_t = std::unordered_map<WatchMapKey, WatchMapValue, WatchMapKeyHasher>;
	WatchMap_t watchMap; // keep track of server watches

public:
	struct PendingNewShard {
		PendingNewShard(uint64_t shardId, KeyRangeRef range) : shardId(format("%016llx", shardId)), range(range) {}

		std::string toString() const {
			return fmt::format("PendingNewShard: [ShardID]: {} [Range]: {}",
			                   this->shardId,
			                   Traceable<KeyRangeRef>::toString(this->range));
		}

		std::string shardId;
		KeyRange range;
	};

	std::map<Version, std::vector<CheckpointMetaData>> pendingCheckpoints; // Pending checkpoint requests
	std::unordered_map<UID, CheckpointMetaData> checkpoints; // Existing and deleting checkpoints
	std::unordered_map<UID, ICheckpointReader*> liveCheckpointReaders; // Active checkpoint readers
	VersionedMap<int64_t, TenantSSInfo> tenantMap;
	std::map<Version, std::vector<PendingNewShard>>
	    pendingAddRanges; // Pending requests to add ranges to physical shards
	std::map<Version, std::vector<KeyRange>>
	    pendingRemoveRanges; // Pending requests to remove ranges from physical shards
	std::deque<std::pair<Standalone<StringRef>, Standalone<StringRef>>> constructedData;

	bool shardAware; // True if the storage server is aware of the physical shards.

	LocalityData locality; // Storage server's locality information.

	// Histograms
	struct FetchKeysHistograms {
		const Reference<Histogram> latency;
		const Reference<Histogram> bytes;
		const Reference<Histogram> bandwidth;
		const Reference<Histogram> bytesPerCommit;

		FetchKeysHistograms()
		  : latency(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                    FETCH_KEYS_LATENCY_HISTOGRAM,
		                                    Histogram::Unit::milliseconds)),
		    bytes(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                  FETCH_KEYS_BYTES_HISTOGRAM,
		                                  Histogram::Unit::bytes)),
		    bandwidth(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                      FETCH_KEYS_BYTES_PER_SECOND_HISTOGRAM,
		                                      Histogram::Unit::bytes_per_second)),
		    bytesPerCommit(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
		                                           FETCH_KEYS_BYTES_PER_COMMIT_HISTOGRAM,
		                                           Histogram::Unit::bytes)) {}
	} fetchKeysHistograms;

	Reference<Histogram> tlogCursorReadsLatencyHistogram;
	Reference<Histogram> ssVersionLockLatencyHistogram;
	Reference<Histogram> eagerReadsLatencyHistogram;
	Reference<Histogram> fetchKeysPTreeUpdatesLatencyHistogram;
	Reference<Histogram> tLogMsgsPTreeUpdatesLatencyHistogram;
	Reference<Histogram> storageUpdatesDurableLatencyHistogram;
	Reference<Histogram> storageCommitLatencyHistogram;
	Reference<Histogram> ssDurableVersionUpdateLatencyHistogram;
	// Histograms of requests sent to KVS.
	Reference<Histogram> readRangeBytesReturnedHistogram;
	Reference<Histogram> readRangeBytesLimitHistogram;
	Reference<Histogram> readRangeKVPairsReturnedHistogram;

	// watch map operations
	Reference<ServerWatchMetadata> getWatchMetadata(KeyRef key, int64_t tenantId) const;
	KeyRef setWatchMetadata(Reference<ServerWatchMetadata> metadata);
	void deleteWatchMetadata(KeyRef key, int64_t tenantId);
	void clearWatchMetadata();

	// tenant map operations
	void insertTenant(TenantMapEntry const& tenant, Version version, bool persist);
	void clearTenants(StringRef startTenant, StringRef endTenant, Version version);

	void checkTenantEntry(Version version, TenantInfo tenant, bool lockAware);

	std::vector<StorageServerShard> getStorageServerShards(KeyRangeRef range);
	std::shared_ptr<MoveInShard> getMoveInShard(const UID& dataMoveId,
	                                            const Version version,
	                                            const ConductBulkLoad conductBulkLoad);

	class CurrentRunningFetchKeys {
		std::unordered_map<UID, double> startTimeMap;
		std::unordered_map<UID, KeyRange> keyRangeMap;

		static const StringRef emptyString;
		static const KeyRangeRef emptyKeyRange;

	public:
		void recordStart(const UID id, const KeyRange& keyRange) {
			startTimeMap[id] = now();
			keyRangeMap[id] = keyRange;
		}

		void recordFinish(const UID id) {
			startTimeMap.erase(id);
			keyRangeMap.erase(id);
		}

		std::pair<double, KeyRange> longestTime() const {
			if (numRunning() == 0) {
				return { -1, emptyKeyRange };
			}

			const double currentTime = now();
			double longest = 0;
			UID UIDofLongest;
			for (const auto& kv : startTimeMap) {
				const double currentRunningTime = currentTime - kv.second;
				if (longest <= currentRunningTime) {
					longest = currentRunningTime;
					UIDofLongest = kv.first;
				}
			}
			if (BUGGIFY) {
				UIDofLongest = deterministicRandom()->randomUniqueID();
			}
			auto it = keyRangeMap.find(UIDofLongest);
			if (it != keyRangeMap.end()) {
				return { longest, it->second };
			}
			return { -1, emptyKeyRange };
		}

		int numRunning() const { return startTimeMap.size(); }
	} currentRunningFetchKeys;

	Tag tag;
	std::vector<std::pair<Version, Tag>> history;
	std::vector<std::pair<Version, Tag>> allHistory;
	Version poppedAllAfter;
	std::map<Version, Arena>
	    freeable; // for each version, an Arena that must be held until that version is < oldestVersion
	Arena lastArena;
	double cpuUsage;
	double diskUsage;

	std::map<Version, Standalone<VerUpdateRef>> const& getMutationLog() const { return mutationLog; }
	std::map<Version, Standalone<VerUpdateRef>>& getMutableMutationLog() { return mutationLog; }
	VersionedData const& data() const { return versionedData; }
	VersionedData& mutableData() { return versionedData; }

	mutable double old_rate = 1.0;
	double currentRate() const {
		auto versionLag = version.get() - durableVersion.get();
		double res;
		if (versionLag >= SERVER_KNOBS->STORAGE_DURABILITY_LAG_HARD_MAX) {
			res = 0.0;
		} else if (versionLag > SERVER_KNOBS->STORAGE_DURABILITY_LAG_SOFT_MAX) {
			res =
			    1.0 -
			    (double(versionLag - SERVER_KNOBS->STORAGE_DURABILITY_LAG_SOFT_MAX) /
			     double(SERVER_KNOBS->STORAGE_DURABILITY_LAG_HARD_MAX - SERVER_KNOBS->STORAGE_DURABILITY_LAG_SOFT_MAX));
		} else {
			res = 1.0;
		}
		if (res != old_rate) {
			TraceEvent(SevDebug, "LocalRatekeeperChange", thisServerID)
			    .detail("Old", old_rate)
			    .detail("New", res)
			    .detail("NonDurableVersions", versionLag);
			old_rate = res;
		}
		return res;
	}

	void addMutationToMutationLogOrStorage(
	    Version ver,
	    MutationRef m); // Appends m to mutationLog@ver, or to storage if ver==invalidVersion

	// Update the byteSample, and write the updates to the mutation log@ver, or to storage if ver==invalidVersion
	void byteSampleApplyMutation(MutationRef const& m, Version ver);
	void byteSampleApplySet(KeyValueRef kv, Version ver);
	void byteSampleApplyClear(KeyRangeRef range, Version ver);

	void popVersion(Version v, bool popAllTags = false) {
		if (logSystem && !isTss()) {
			if (v > poppedAllAfter) {
				popAllTags = true;
				poppedAllAfter = std::numeric_limits<Version>::max();
			}

			std::vector<std::pair<Version, Tag>>* hist = &history;
			std::vector<std::pair<Version, Tag>> allHistoryCopy;
			if (popAllTags) {
				allHistoryCopy = allHistory;
				hist = &allHistoryCopy;
			}

			while (hist->size() && v > hist->back().first) {
				logSystem->pop(v, hist->back().second);
				hist->pop_back();
			}
			if (hist->size()) {
				logSystem->pop(v, hist->back().second);
			} else {
				logSystem->pop(v, tag);
			}
		}
	}

	Standalone<VerUpdateRef>& addVersionToMutationLog(Version v) {
		// return existing version...
		auto m = mutationLog.find(v);
		if (m != mutationLog.end())
			return m->second;

		// ...or create a new one
		auto& u = mutationLog[v];
		u.version = v;
		if (lastArena.getSize() >= 65536)
			lastArena = Arena(4096);
		u.arena() = lastArena;
		counters.bytesInput += VERSION_OVERHEAD;
		return u;
	}

	MutationRef addMutationToMutationLog(Standalone<VerUpdateRef>& mLV, MutationRef const& m) {
		byteSampleApplyMutation(m, mLV.version);
		counters.bytesInput += mvccStorageBytes(m);
		return mLV.push_back_deep(mLV.arena(), m);
	}

	void setTssPair(UID pairId) {
		tssPairID = Optional<UID>(pairId);

		// Set up tss fault injection here, only if we are in simulated mode and with fault injection.
		// With fault injection enabled, the tss will start acting normal for a bit, then after the specified delay
		// start behaving incorrectly.
		if (g_network->isSimulated() && !g_simulator->speedUpSimulation &&
		    g_simulator->tssMode >= ISimulator::TSSMode::EnabledAddDelay) {
			tssFaultInjectTime = now() + deterministicRandom()->randomInt(60, 300);
			TraceEvent(SevWarnAlways, "TSSInjectFaultEnabled", thisServerID)
			    .detail("Mode", g_simulator->tssMode)
			    .detail("At", tssFaultInjectTime.get());
		}
	}

	// If a TSS is "in quarantine", it means it has incorrect data. It is effectively in a "zombie" state where it
	// rejects all read requests and ignores all non-private mutations and data movements, but otherwise is still part
	// of the cluster. The purpose of this state is to "freeze" the TSS state after a mismatch so a human operator can
	// investigate, but preventing a new storage process from replacing the TSS on the worker. It will still get removed
	// from the cluster if it falls behind on the mutation stream, or if its tss pair gets removed and its tag is no
	// longer valid.
	bool isTSSInQuarantine() const { return tssPairID.present() && tssInQuarantine; }

	void startTssQuarantine() {
		if (!tssInQuarantine) {
			// persist quarantine so it's still quarantined if rebooted
			storage.makeTssQuarantineDurable();
		}
		tssInQuarantine = true;
	}

	StorageServerDisk storage;

	KeyRangeMap<Reference<ShardInfo>> shards;
	KeyRangeMap<SSBulkLoadMetadata> ssBulkLoadMetadataMap; // store the latest bulkload task on ranges
	uint64_t shardChangeCounter; // max( shards->changecounter )

	KeyRangeMap<bool> cachedRangeMap; // indicates if a key-range is being cached

	// newestAvailableVersion[k]
	//   == invalidVersion -> k is unavailable at all versions
	//   <= storageVersion -> k is unavailable at all versions (but might be read anyway from storage if we are in the
	//   process of committing makeShardDurable)
	//   == v              -> k is readable (from storage+versionedData) @ [storageVersion,v], and not being updated
	//   when version increases
	//   == latestVersion  -> k is readable (from storage+versionedData) @ [storageVersion,version.get()], and thus
	//   stays available when version increases
	CoalescedKeyRangeMap<Version> newestAvailableVersion;

	CoalescedKeyRangeMap<Version> newestDirtyVersion; // Similar to newestAvailableVersion, but includes (only) keys
	                                                  // that were only partly available (due to cancelled fetchKeys)

	// The following are in rough order from newest to oldest
	Version lastTLogVersion, lastVersionWithData, restoredVersion, prevVersion;
	NotifiedVersion version;
	NotifiedVersion desiredOldestVersion; // We can increase oldestVersion (and then durableVersion) to this version
	                                      // when the disk permits
	NotifiedVersion oldestVersion; // See also storageVersion()
	NotifiedVersion durableVersion; // At least this version will be readable from storage after a power failure
	// In the event of the disk corruption, sqlite and redwood will either not recover, recover to durableVersion
	// but be unable to read some data, or they could lose the last commit. If we lose the last commit, the storage
	// might not be able to peek from the tlog (depending on when it sent the last pop). So this version just keeps
	// track of the version we committed to the storage engine before we did commit durableVersion.
	Version storageMinRecoverVersion = 0;
	Version rebootAfterDurableVersion;
	int8_t primaryLocality;
	NotifiedVersion knownCommittedVersion;

	Deque<std::pair<Version, Version>> recoveryVersionSkips;
	int64_t versionLag; // An estimate for how many versions it takes for the data to move from the logs to this storage
	                    // server

	Optional<UID> sourceTLogID; // the tLog from which the latest batch of versions were fetched

	ProtocolVersion logProtocol;

	Reference<ILogSystem> logSystem;
	Reference<ILogSystem::IPeekCursor> logCursor;

	// The version the cluster starts on. This value is not persisted and may
	// not be valid after a recovery.
	Version initialClusterVersion = 1;
	UID thisServerID;
	Optional<UID> tssPairID; // if this server is a tss, this is the id of its (ss) pair
	Optional<UID> ssPairID; // if this server is an ss, this is the id of its (tss) pair
	Optional<double> tssFaultInjectTime;
	bool tssInQuarantine;

	Key sk;
	Reference<AsyncVar<ServerDBInfo> const> db;
	Database cx;
	ActorCollection actors;

	CoalescedKeyRangeMap<bool, int64_t, KeyBytesMetric<int64_t>> byteSampleClears;
	AsyncVar<bool> byteSampleClearsTooLarge;
	Future<Void> byteSampleRecovery;
	Future<Void> durableInProgress;

	AsyncMap<Key, bool> watches;
	AsyncMap<int64_t, bool> tenantWatches;
	int64_t watchBytes;
	int64_t numWatches;
	AsyncVar<bool> noRecentUpdates;
	double lastUpdate;

	std::string folder;
	std::string checkpointFolder;
	std::string fetchedCheckpointFolder;
	std::string bulkDumpFolder;
	std::string bulkLoadFolder;

	// defined only during splitMutations()/addMutation()
	UpdateEagerReadInfo* updateEagerReads;

	FlowLock durableVersionLock;
	FlowLock fetchKeysParallelismLock;
	int64_t fetchKeysBytesBudget;
	AsyncVar<bool> fetchKeysBudgetUsed;
	int64_t fetchKeysTotalCommitBytes;
	std::vector<Promise<FetchInjectionInfo*>> readyFetchKeys;

	ThroughputLimiter fetchKeysLimiter;

	FlowLock serveFetchCheckpointParallelismLock;

	std::unordered_map<UID, std::shared_ptr<MoveInShard>> moveInShards;

	Reference<PriorityMultiLock> ssLock;
	std::vector<int> readPriorityRanks;

	Future<PriorityMultiLock::Lock> getReadLock(const Optional<ReadOptions>& options) {
		int readType = (int)(options.present() ? options.get().type : ReadType::NORMAL);
		readType = std::clamp<int>(readType, 0, readPriorityRanks.size() - 1);
		return ssLock->lock(readPriorityRanks[readType]);
	}

	FlowLock serveAuditStorageParallelismLock;

	FlowLock serveBulkDumpParallelismLock;

	int64_t instanceID;

	Promise<Void> otherError;
	Promise<Void> coreStarted;
	bool shuttingDown;

	Promise<Void> registerInterfaceAcceptingRequests;
	Future<Void> interfaceRegistered;

	bool behind;
	bool versionBehind;

	bool debug_inApplyUpdate;
	double debug_lastValidateTime;

	int64_t lastBytesInputEBrake;
	Version lastDurableVersionEBrake;

	int maxQueryQueue;
	int getAndResetMaxQueryQueueSize() {
		int val = maxQueryQueue;
		maxQueryQueue = 0;
		return val;
	}

	TransactionTagCounter transactionTagCounter;
	BusiestWriteTagContext busiestWriteTagContext;

	Optional<LatencyBandConfig> latencyBandConfig;

	Optional<EncryptionAtRestMode> encryptionMode;
	Reference<GetEncryptCipherKeysMonitor> getEncryptCipherKeysMonitor;

	struct Counters : CommonStorageCounters {

		Counter allQueries, systemKeyQueries, getKeyQueries, getValueQueries, getRangeQueries, getRangeSystemKeyQueries,
		    getRangeStreamQueries, lowPriorityQueries, rowsQueried, watchQueries, emptyQueries;

		// counters related to getMappedRange queries
		Counter getMappedRangeBytesQueried, finishedGetMappedRangeSecondaryQueries, getMappedRangeQueries,
		    finishedGetMappedRangeQueries;

		// Bytes pulled from TLogs, it counts the size of the key value pairs, e.g., key-value pair ("a", "b") is
		// counted as 2 Bytes.
		Counter logicalBytesInput;
		// Bytes pulled from TLogs for moving-in shards, it counts the mutations sent to the moving-in shard during
		// Fetching and Waiting phases.
		Counter logicalBytesMoveInOverhead;
		// Bytes committed to the underlying storage engine by SS, it counts the size of key value pairs.
		Counter kvCommitLogicalBytes;
		// Count of all clearRange operations to the storage engine.
		Counter kvClearRanges;
		// Count of all clearRange operations on a singlekeyRange(key delete) to the storage engine.
		Counter kvClearSingleKey;
		// ClearRange operations issued by FDB, instead of from users, e.g., ClearRange operations to remove a shard
		// from a storage server, as in removeDataRange().
		Counter kvSystemClearRanges;
		// Bytes of the mutations that have been removed from memory because they durable. The counting is same as
		// bytesInput, instead of the actual bytes taken in the storages, so that (bytesInput - bytesDurable) can
		// reflect the current memory footprint of MVCC.
		Counter bytesDurable;
		// Count of all fetchKey clearRange operations to the storage engine.
		Counter kvClearRangesInFetchKeys;

		// Bytes fetched by fetchChangeFeed for data movements.
		Counter feedBytesFetched;

		Counter sampledBytesCleared;
		Counter atomicMutations, changeFeedMutations, changeFeedMutationsDurable;
		Counter updateBatches, updateVersions;
		Counter loops;
		Counter fetchWaitingMS, fetchWaitingCount, fetchExecutingMS, fetchExecutingCount;
		Counter readsRejected;
		Counter wrongShardServer;
		Counter fetchedVersions;
		Counter fetchesFromLogs;
		// The following counters measure how many of lookups in the getMappedRangeQueries are effective. "Miss"
		// means fallback if fallback is enabled, otherwise means failure (so that another layer could implement
		// fallback).
		Counter quickGetValueHit, quickGetValueMiss, quickGetKeyValuesHit, quickGetKeyValuesMiss;

		// The number of logical bytes returned from storage engine, in response to readRange operations.
		Counter kvScanBytes;
		// The number of logical bytes returned from storage engine, in response to readValue operations.
		Counter kvGetBytes;
		// The number of keys read from storage engine by eagerReads.
		Counter eagerReadsKeys;
		// The count of readValue operation to the storage engine.
		Counter kvGets;
		// The count of readValue operation to the storage engine.
		Counter kvScans;
		// The count of commit operation to the storage engine.
		Counter kvCommits;
		// The count of change feed reads that hit disk
		Counter changeFeedDiskReads;
		// The count of ChangeServerKeys actions.
		Counter changeServerKeysAssigned;
		Counter changeServerKeysUnassigned;

		// The count of 'set' inserted to pTree. The actual ptree.insert() number could be higher, because of the range
		// clear split, see metric pTreeClearSplits.
		Counter pTreeSets;
		// The count of clear range inserted to pTree
		Counter pTreeClears;
		// If set is within a range of clear, the clear is split. It's tracking the number of splits, the split could be
		// expensive.
		Counter pTreeClearSplits;

		std::unique_ptr<LatencySample> readLatencySample;
		std::unique_ptr<LatencySample> readKeyLatencySample;
		std::unique_ptr<LatencySample> readValueLatencySample;
		std::unique_ptr<LatencySample> readRangeLatencySample;
		std::unique_ptr<LatencySample> readVersionWaitSample;
		std::unique_ptr<LatencySample> readQueueWaitSample;
		std::unique_ptr<LatencySample> kvReadRangeLatencySample;
		std::unique_ptr<LatencySample> updateLatencySample;
		std::unique_ptr<LatencySample> updateEncryptionLatencySample;
		LatencyBands readLatencyBands;
		std::unique_ptr<LatencySample> mappedRangeSample; // Samples getMappedRange latency
		std::unique_ptr<LatencySample> mappedRangeRemoteSample; // Samples getMappedRange remote subquery latency
		std::unique_ptr<LatencySample> mappedRangeLocalSample; // Samples getMappedRange local subquery latency
		std::unique_ptr<LatencySample> ingestDurationLatencySample;

		explicit Counters(StorageServer* self)
		  : CommonStorageCounters("StorageServer", self->thisServerID.toString(), &self->metrics),
		    allQueries("QueryQueue", cc), systemKeyQueries("SystemKeyQueries", cc), getKeyQueries("GetKeyQueries", cc),
		    getValueQueries("GetValueQueries", cc), getRangeQueries("GetRangeQueries", cc),
		    getRangeSystemKeyQueries("GetRangeSystemKeyQueries", cc),
		    getMappedRangeQueries("GetMappedRangeQueries", cc), getRangeStreamQueries("GetRangeStreamQueries", cc),
		    lowPriorityQueries("LowPriorityQueries", cc), rowsQueried("RowsQueried", cc),
		    watchQueries("WatchQueries", cc), emptyQueries("EmptyQueries", cc),
		    logicalBytesInput("LogicalBytesInput", cc), logicalBytesMoveInOverhead("LogicalBytesMoveInOverhead", cc),
		    kvCommitLogicalBytes("KVCommitLogicalBytes", cc), kvClearRanges("KVClearRanges", cc),
		    kvClearSingleKey("KVClearSingleKey", cc), kvSystemClearRanges("KVSystemClearRanges", cc),
		    bytesDurable("BytesDurable", cc), feedBytesFetched("FeedBytesFetched", cc),
		    sampledBytesCleared("SampledBytesCleared", cc), atomicMutations("AtomicMutations", cc),
		    changeFeedMutations("ChangeFeedMutations", cc),
		    changeFeedMutationsDurable("ChangeFeedMutationsDurable", cc), updateBatches("UpdateBatches", cc),
		    updateVersions("UpdateVersions", cc), loops("Loops", cc), fetchWaitingMS("FetchWaitingMS", cc),
		    fetchWaitingCount("FetchWaitingCount", cc), fetchExecutingMS("FetchExecutingMS", cc),
		    fetchExecutingCount("FetchExecutingCount", cc), readsRejected("ReadsRejected", cc),
		    wrongShardServer("WrongShardServer", cc), fetchedVersions("FetchedVersions", cc),
		    fetchesFromLogs("FetchesFromLogs", cc), quickGetValueHit("QuickGetValueHit", cc),
		    quickGetValueMiss("QuickGetValueMiss", cc), quickGetKeyValuesHit("QuickGetKeyValuesHit", cc),
		    quickGetKeyValuesMiss("QuickGetKeyValuesMiss", cc), kvScanBytes("KVScanBytes", cc),
		    kvGetBytes("KVGetBytes", cc), eagerReadsKeys("EagerReadsKeys", cc), kvGets("KVGets", cc),
		    kvScans("KVScans", cc), kvCommits("KVCommits", cc), changeFeedDiskReads("ChangeFeedDiskReads", cc),
		    getMappedRangeBytesQueried("GetMappedRangeBytesQueried", cc),
		    finishedGetMappedRangeQueries("FinishedGetMappedRangeQueries", cc),
		    finishedGetMappedRangeSecondaryQueries("FinishedGetMappedRangeSecondaryQueries", cc),
		    pTreeSets("PTreeSets", cc), pTreeClears("PTreeClears", cc), pTreeClearSplits("PTreeClearSplits", cc),
		    changeServerKeysAssigned("ChangeServerKeysAssigned", cc),
		    changeServerKeysUnassigned("ChangeServerKeysUnassigned", cc),
		    kvClearRangesInFetchKeys("KvClearRangesInFetchKeys", cc),
		    readLatencySample(std::make_unique<LatencySample>("ReadLatencyMetrics",
		                                                      self->thisServerID,
		                                                      SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                      SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readKeyLatencySample(std::make_unique<LatencySample>("GetKeyMetrics",
		                                                         self->thisServerID,
		                                                         SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                         SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readValueLatencySample(std::make_unique<LatencySample>("GetValueMetrics",
		                                                           self->thisServerID,
		                                                           SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                           SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readRangeLatencySample(std::make_unique<LatencySample>("GetRangeMetrics",
		                                                           self->thisServerID,
		                                                           SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                           SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readVersionWaitSample(std::make_unique<LatencySample>("ReadVersionWaitMetrics",
		                                                          self->thisServerID,
		                                                          SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                          SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readQueueWaitSample(std::make_unique<LatencySample>("ReadQueueWaitMetrics",
		                                                        self->thisServerID,
		                                                        SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                        SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    kvReadRangeLatencySample(std::make_unique<LatencySample>("KVGetRangeMetrics",
		                                                             self->thisServerID,
		                                                             SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                             SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    updateLatencySample(std::make_unique<LatencySample>("UpdateLatencyMetrics",
		                                                        self->thisServerID,
		                                                        SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                        SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    updateEncryptionLatencySample(
		        std::make_unique<LatencySample>("UpdateEncryptionLatencyMetrics",
		                                        self->thisServerID,
		                                        SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                        SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    readLatencyBands("ReadLatencyBands", self->thisServerID, SERVER_KNOBS->STORAGE_LOGGING_DELAY),
		    mappedRangeSample(std::make_unique<LatencySample>("GetMappedRangeMetrics",
		                                                      self->thisServerID,
		                                                      SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                      SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    mappedRangeRemoteSample(std::make_unique<LatencySample>("GetMappedRangeRemoteMetrics",
		                                                            self->thisServerID,
		                                                            SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                            SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    mappedRangeLocalSample(std::make_unique<LatencySample>("GetMappedRangeLocalMetrics",
		                                                           self->thisServerID,
		                                                           SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                           SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)),
		    ingestDurationLatencySample(std::make_unique<LatencySample>("IngestDurationMetrics",
		                                                                self->thisServerID,
		                                                                SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
		                                                                SERVER_KNOBS->LATENCY_SKETCH_ACCURACY)) {
			specialCounter(cc, "LastTLogVersion", [self]() { return self->lastTLogVersion; });
			specialCounter(cc, "Version", [self]() { return self->version.get(); });
			specialCounter(cc, "StorageVersion", [self]() { return self->storageVersion(); });
			specialCounter(cc, "DurableVersion", [self]() { return self->durableVersion.get(); });
			specialCounter(cc, "DesiredOldestVersion", [self]() { return self->desiredOldestVersion.get(); });
			specialCounter(cc, "VersionLag", [self]() { return self->versionLag; });
			specialCounter(cc, "LocalRate", [self] { return int64_t(self->currentRate() * 100); });

			specialCounter(
			    cc, "FetchKeysFetchActive", [self]() { return self->fetchKeysParallelismLock.activePermits(); });
			specialCounter(cc, "FetchKeysWaiting", [self]() { return self->fetchKeysParallelismLock.waiters(); });
			specialCounter(cc, "ServeFetchCheckpointActive", [self]() {
				return self->serveFetchCheckpointParallelismLock.activePermits();
			});
			specialCounter(cc, "ServeFetchCheckpointWaiting", [self]() {
				return self->serveFetchCheckpointParallelismLock.waiters();
			});
			specialCounter(cc, "ServeValidateStorageActive", [self]() {
				return self->serveAuditStorageParallelismLock.activePermits();
			});
			specialCounter(cc, "ServeValidateStorageWaiting", [self]() {
				return self->serveAuditStorageParallelismLock.waiters();
			});
			specialCounter(
			    cc, "ServerBulkDumpActive", [self]() { return self->serveBulkDumpParallelismLock.activePermits(); });
			specialCounter(
			    cc, "ServerBulkDumpWaiting", [self]() { return self->serveBulkDumpParallelismLock.waiters(); });
			specialCounter(cc, "QueryQueueMax", [self]() { return self->getAndResetMaxQueryQueueSize(); });
			specialCounter(cc, "ActiveWatches", [self]() { return self->numWatches; });
			specialCounter(cc, "WatchBytes", [self]() { return self->watchBytes; });
			specialCounter(cc, "KvstoreSizeTotal", [self]() { return std::get<0>(self->storage.getSize()); });
			specialCounter(cc, "KvstoreNodeTotal", [self]() { return std::get<1>(self->storage.getSize()); });
			specialCounter(cc, "KvstoreInlineKey", [self]() { return std::get<2>(self->storage.getSize()); });
		}
	} counters;

	// Bytes read from storage engine when a storage server starts.
	int64_t bytesRestored = 0;

	Reference<EventCacheHolder> storageServerSourceTLogIDEventHolder;

	std::shared_ptr<AccumulativeChecksumValidator> acsValidator = nullptr;

	std::shared_ptr<SSBulkLoadMetrics> bulkLoadMetrics = nullptr;

	StorageServer(IKeyValueStore* storage,
	              Reference<AsyncVar<ServerDBInfo> const> const& db,
	              StorageServerInterface const& ssi,
	              Reference<GetEncryptCipherKeysMonitor> encryptionMonitor)
	  : shardAware(false), locality(ssi.locality),
	    tlogCursorReadsLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                            TLOG_CURSOR_READS_LATENCY_HISTOGRAM,
	                                                            Histogram::Unit::milliseconds)),
	    ssVersionLockLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                          SS_VERSION_LOCK_LATENCY_HISTOGRAM,
	                                                          Histogram::Unit::milliseconds)),
	    eagerReadsLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                       EAGER_READS_LATENCY_HISTOGRAM,
	                                                       Histogram::Unit::milliseconds)),
	    fetchKeysPTreeUpdatesLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                  FETCH_KEYS_PTREE_UPDATES_LATENCY_HISTOGRAM,
	                                                                  Histogram::Unit::milliseconds)),
	    tLogMsgsPTreeUpdatesLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                 TLOG_MSGS_PTREE_UPDATES_LATENCY_HISTOGRAM,
	                                                                 Histogram::Unit::milliseconds)),
	    storageUpdatesDurableLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                  STORAGE_UPDATES_DURABLE_LATENCY_HISTOGRAM,
	                                                                  Histogram::Unit::milliseconds)),
	    storageCommitLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                          STORAGE_COMMIT_LATENCY_HISTOGRAM,
	                                                          Histogram::Unit::milliseconds)),
	    ssDurableVersionUpdateLatencyHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                                   SS_DURABLE_VERSION_UPDATE_LATENCY_HISTOGRAM,
	                                                                   Histogram::Unit::milliseconds)),
	    readRangeBytesReturnedHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                            SS_READ_RANGE_BYTES_RETURNED_HISTOGRAM,
	                                                            Histogram::Unit::bytes)),
	    readRangeBytesLimitHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                         SS_READ_RANGE_BYTES_LIMIT_HISTOGRAM,
	                                                         Histogram::Unit::bytes)),
	    readRangeKVPairsReturnedHistogram(Histogram::getHistogram(STORAGESERVER_HISTOGRAM_GROUP,
	                                                              SS_READ_RANGE_KV_PAIRS_RETURNED_HISTOGRAM,
	                                                              Histogram::Unit::countLinear)),
	    tag(invalidTag), poppedAllAfter(std::numeric_limits<Version>::max()), cpuUsage(0.0), diskUsage(0.0),
	    storage(this, storage), shardChangeCounter(0), lastTLogVersion(0), lastVersionWithData(0), restoredVersion(0),
	    prevVersion(0), rebootAfterDurableVersion(std::numeric_limits<Version>::max()),
	    primaryLocality(tagLocalityInvalid), knownCommittedVersion(0), versionLag(0), logProtocol(0),
	    thisServerID(ssi.id()), tssInQuarantine(false), db(db), actors(false),
	    trackShardAssignmentMinVersion(invalidVersion), byteSampleClears(false, "\xff\xff\xff"_sr),
	    durableInProgress(Void()), watchBytes(0), numWatches(0), noRecentUpdates(false), lastUpdate(now()),
	    updateEagerReads(nullptr), fetchKeysParallelismLock(SERVER_KNOBS->FETCH_KEYS_PARALLELISM),
	    fetchKeysBytesBudget(SERVER_KNOBS->STORAGE_FETCH_BYTES), fetchKeysBudgetUsed(false),
	    fetchKeysTotalCommitBytes(0), fetchKeysLimiter(SERVER_KNOBS->STORAGE_FETCH_KEYS_RATE_LIMIT),
	    serveFetchCheckpointParallelismLock(SERVER_KNOBS->SERVE_FETCH_CHECKPOINT_PARALLELISM),
	    ssLock(makeReference<PriorityMultiLock>(SERVER_KNOBS->STORAGE_SERVER_READ_CONCURRENCY,
	                                            SERVER_KNOBS->STORAGESERVER_READ_PRIORITIES)),
	    serveAuditStorageParallelismLock(SERVER_KNOBS->SERVE_AUDIT_STORAGE_PARALLELISM),
	    serveBulkDumpParallelismLock(SERVER_KNOBS->SS_SERVE_BULKDUMP_PARALLELISM),
	    instanceID(deterministicRandom()->randomUniqueID().first()), shuttingDown(false), behind(false),
	    versionBehind(false), debug_inApplyUpdate(false), debug_lastValidateTime(0), lastBytesInputEBrake(0),
	    lastDurableVersionEBrake(0), maxQueryQueue(0),
	    transactionTagCounter(ssi.id(),
	                          /*maxTagsTracked=*/SERVER_KNOBS->SS_THROTTLE_TAGS_TRACKED,
	                          /*minRateTracked=*/SERVER_KNOBS->MIN_TAG_READ_PAGES_RATE *
	                              CLIENT_KNOBS->TAG_THROTTLING_PAGE_SIZE),
	    busiestWriteTagContext(ssi.id()), getEncryptCipherKeysMonitor(encryptionMonitor), counters(this),
	    storageServerSourceTLogIDEventHolder(
	        makeReference<EventCacheHolder>(ssi.id().toString() + "/StorageServerSourceTLogID")),
	    acsValidator(CLIENT_KNOBS->ENABLE_MUTATION_CHECKSUM && CLIENT_KNOBS->ENABLE_ACCUMULATIVE_CHECKSUM &&
	                         !SERVER_KNOBS->ENABLE_VERSION_VECTOR && !SERVER_KNOBS->ENABLE_VERSION_VECTOR_TLOG_UNICAST
	                     ? std::make_shared<AccumulativeChecksumValidator>()
	                     : nullptr),
	    bulkLoadMetrics(std::make_shared<SSBulkLoadMetrics>()) {
		readPriorityRanks = parseStringToVector<int>(SERVER_KNOBS->STORAGESERVER_READTYPE_PRIORITY_MAP, ',');
		ASSERT(readPriorityRanks.size() > (int)ReadType::MAX);
		version.initMetric("StorageServer.Version"_sr, counters.cc.getId());
		oldestVersion.initMetric("StorageServer.OldestVersion"_sr, counters.cc.getId());
		durableVersion.initMetric("StorageServer.DurableVersion"_sr, counters.cc.getId());
		desiredOldestVersion.initMetric("StorageServer.DesiredOldestVersion"_sr, counters.cc.getId());

		newestAvailableVersion.insert(allKeys, invalidVersion);
		newestDirtyVersion.insert(allKeys, invalidVersion);
		if (storage->shardAware()) {
			addShard(ShardInfo::newShard(this, StorageServerShard::notAssigned(allKeys)));
		} else {
			addShard(ShardInfo::newNotAssigned(allKeys));
		}

		cx = openDBOnServer(db, TaskPriority::DefaultEndpoint, LockAware::True);

		this->storage.kvCommitLogicalBytes = &counters.kvCommitLogicalBytes;
		this->storage.kvClearRanges = &counters.kvClearRanges;
		this->storage.kvClearSingleKey = &counters.kvClearSingleKey;
		this->storage.kvGets = &counters.kvGets;
		this->storage.kvScans = &counters.kvScans;
		this->storage.kvCommits = &counters.kvCommits;
	}

	//~StorageServer() { fclose(log); }

	// Puts the given shard into shards.  The caller is responsible for adding shards
	//   for all ranges in shards.getAffectedRangesAfterInsertion(newShard->keys)), because these
	//   shards are invalidated by the call.
	void addShard(ShardInfo* newShard) {
		ASSERT(!newShard->range().empty());
		newShard->setChangeCounter(++shardChangeCounter);
		// TraceEvent("AddShard", this->thisServerID).detail("KeyBegin", newShard->keys.begin).detail("KeyEnd", newShard->keys.end).detail("State",newShard->isReadable() ? "Readable" : newShard->notAssigned() ? "NotAssigned" : "Adding").detail("Version", this->version.get());
		/*auto affected = shards.getAffectedRangesAfterInsertion( newShard->keys, Reference<ShardInfo>() );
		for(auto i = affected.begin(); i != affected.end(); ++i)
		    shards.insert( *i, Reference<ShardInfo>() );*/

		if (shardAware && newShard->notAssigned()) {
			auto sh = shards.intersectingRanges(newShard->range());
			for (auto it = sh.begin(); it != sh.end(); ++it) {
				if (it->value().isValid() && !it->value()->notAssigned()) {
					TraceEvent(SevVerbose, "StorageServerAddShardClear")
					    .detail("NewShardRange", newShard->range())
					    .detail("Range", it->value()->range())
					    .detail("ShardID", format("%016llx", it->value()->getShardId()))
					    .detail("NewShardID", format("%016llx", newShard->getDesiredShardId()))
					    .detail("NewShardActualID", format("%016llx", newShard->getShardId()));
				}
			}
		}

		Reference<ShardInfo> rShard(newShard);
		shards.insert(newShard->range(), rShard);
	}
	void addMutation(Version version,
	                 bool fromFetch,
	                 MutationRef const& mutation,
	                 MutationRefAndCipherKeys const& encryptedMutation,
	                 KeyRangeRef const& shard,
	                 UpdateEagerReadInfo* eagerReads);
	void setInitialVersion(Version ver) {
		version = ver;
		desiredOldestVersion = ver;
		oldestVersion = ver;
		durableVersion = ver;
		storageMinRecoverVersion = ver;
		lastVersionWithData = ver;
		restoredVersion = ver;

		mutableData().createNewVersion(ver);
		mutableData().forgetVersionsBefore(ver);
	}

	bool isTss() const { return tssPairID.present(); }

	bool isSSWithTSSPair() const { return ssPairID.present(); }

	void setSSWithTssPair(UID idOfTSS) { ssPairID = Optional<UID>(idOfTSS); }

	void clearSSWithTssPair() { ssPairID = Optional<UID>(); }

	// This is the maximum version that might be read from storage (the minimum version is durableVersion)
	Version storageVersion() const { return oldestVersion.get(); }

	bool isReadable(KeyRangeRef const& keys) const override {
		auto sh = shards.intersectingRanges(keys);
		for (auto i = sh.begin(); i != sh.end(); ++i)
			if (!i->value()->isReadable())
				return false;
		return true;
	}

	void checkChangeCounter(uint64_t oldShardChangeCounter, KeyRef const& key) {
		if (oldShardChangeCounter != shardChangeCounter && shards[key]->getChangeCounter() > oldShardChangeCounter) {
			CODE_PROBE(true, "shard change during getValueQ");
			throw wrong_shard_server();
		}
	}

	void checkChangeCounter(uint64_t oldShardChangeCounter, KeyRangeRef const& keys) {
		if (oldShardChangeCounter != shardChangeCounter) {
			auto sh = shards.intersectingRanges(keys);
			for (auto i = sh.begin(); i != sh.end(); ++i)
				if (i->value()->getChangeCounter() > oldShardChangeCounter) {
					CODE_PROBE(true, "shard change during range operation");
					throw wrong_shard_server();
				}
		}
	}

	Counter::Value queueSize() const { return counters.bytesInput.getValue() - counters.bytesDurable.getValue(); }

	// penalty used by loadBalance() to balance requests among SSes. We prefer SS with less write queue size.
	double getPenalty() const override {
		return std::max(std::max(1.0,
		                         (queueSize() - (SERVER_KNOBS->TARGET_BYTES_PER_STORAGE_SERVER -
		                                         2.0 * SERVER_KNOBS->SPRING_BYTES_STORAGE_SERVER)) /
		                             SERVER_KNOBS->SPRING_BYTES_STORAGE_SERVER),
		                (currentRate() < 1e-6 ? 1e6 : 1.0 / currentRate()));
	}

	// Normally the storage server prefers to serve read requests over making mutations
	// durable to disk. However, when the storage server falls to far behind on
	// making mutations durable, this function will change the priority to prefer writes.
	Future<Void> getQueryDelay() {
		if ((version.get() - durableVersion.get() > SERVER_KNOBS->LOW_PRIORITY_DURABILITY_LAG) ||
		    (queueSize() > SERVER_KNOBS->LOW_PRIORITY_STORAGE_QUEUE_BYTES)) {
			++counters.lowPriorityQueries;
			return delay(0, TaskPriority::LowPriorityRead);
		}
		return delay(0, TaskPriority::DefaultEndpoint);
	}

	template <class Reply>
	using isLoadBalancedReply = std::is_base_of<LoadBalancedReply, Reply>;

	template <class Reply>
	typename std::enable_if<isLoadBalancedReply<Reply>::value, void>::type
	sendErrorWithPenalty(const ReplyPromise<Reply>& promise, const Error& err, double penalty) {
		if (err.code() == error_code_wrong_shard_server) {
			++counters.wrongShardServer;
		}
		Reply reply;
		reply.error = err;
		reply.penalty = penalty;
		promise.send(reply);
	}

	template <class Reply>
	typename std::enable_if<!isLoadBalancedReply<Reply>::value, void>::type
	sendErrorWithPenalty(const ReplyPromise<Reply>& promise, const Error& err, double) {
		if (err.code() == error_code_wrong_shard_server) {
			++counters.wrongShardServer;
		}
		promise.sendError(err);
	}

	template <class Request>
	bool shouldRead(const Request& request) {
		auto rate = currentRate();
		if (isTSSInQuarantine() || (rate < SERVER_KNOBS->STORAGE_DURABILITY_LAG_REJECT_THRESHOLD &&
		                            deterministicRandom()->random01() >
		                                std::max(SERVER_KNOBS->STORAGE_DURABILITY_LAG_MIN_RATE,
		                                         rate / SERVER_KNOBS->STORAGE_DURABILITY_LAG_REJECT_THRESHOLD))) {
			sendErrorWithPenalty(request.reply, server_overloaded(), getPenalty());
			++counters.readsRejected;
			return false;
		}
		return true;
	}

	template <class Request, class HandleFunction>
	Future<Void> readGuard(const Request& request, const HandleFunction& fun) {
		bool read = shouldRead(request);
		if (!read) {
			return Void();
		}
		return fun(this, request);
	}

	void getSplitPoints(SplitRangeRequest const& req) override {
		try {
			checkTenantEntry(version.get(), req.tenantInfo, true);
			metrics.getSplitPoints(req, req.tenantInfo.prefix);
		} catch (Error& e) {
			req.reply.sendError(e);
		}
	}

	void maybeInjectTargetedRestart(Version v) {
		// inject an SS restart at most once per test
		if (g_network->isSimulated() && !g_simulator->speedUpSimulation &&
		    now() > g_simulator->injectTargetedSSRestartTime &&
		    rebootAfterDurableVersion == std::numeric_limits<Version>::max()) {
			CODE_PROBE(true, "Injecting SS targeted restart");
			TraceEvent("SimSSInjectTargetedRestart", thisServerID).detail("Version", v);
			rebootAfterDurableVersion = v;
			g_simulator->injectTargetedSSRestartTime = std::numeric_limits<double>::max();
		}
	}

	bool maybeInjectDelay() {
		if (g_network->isSimulated() && !g_simulator->speedUpSimulation && now() > g_simulator->injectSSDelayTime) {
			CODE_PROBE(true, "Injecting SS targeted delay");
			TraceEvent("SimSSInjectDelay", thisServerID).log();
			g_simulator->injectSSDelayTime = std::numeric_limits<double>::max();
			return true;
		}
		return false;
	}

	Future<Void> waitMetricsTenantAware(const WaitMetricsRequest& req) override;

	void addActor(Future<Void> future) override { actors.add(future); }

	void getStorageMetrics(const GetStorageMetricsRequest& req) override {
		StorageBytes sb = storage.getStorageBytes();
		metrics.getStorageMetrics(req,
		                          sb,
		                          counters.bytesInput.getRate(),
		                          versionLag,
		                          lastUpdate,
		                          counters.bytesDurable.getValue(),
		                          counters.bytesInput.getValue(),
		                          bulkLoadMetrics->getOngoingTasks());
	}

	void getSplitMetrics(const SplitMetricsRequest& req) override { this->metrics.splitMetrics(req); }

	void getHotRangeMetrics(const ReadHotSubRangeRequest& req) override { this->metrics.getReadHotRanges(req); }

	int64_t getHotShardsMetrics(const KeyRange& range) override { return this->metrics.getHotShards(range); }

	// Used for recording shard assignment history for auditStorage
	std::vector<std::pair<Version, KeyRange>> shardAssignmentHistory;
	Version trackShardAssignmentMinVersion; // == invalidVersion means tracking stopped

	std::string printShardAssignmentHistory() {
		std::string toPrint = "";
		for (const auto& [version, range] : shardAssignmentHistory) {
			toPrint = toPrint + std::to_string(version) + " ";
		}
		return toPrint;
	}

	void startTrackShardAssignment(Version startVersion) {
		ASSERT(startVersion != invalidVersion);
		ASSERT(trackShardAssignmentMinVersion == invalidVersion);
		trackShardAssignmentMinVersion = startVersion;
		return;
	}

	void stopTrackShardAssignment() { trackShardAssignmentMinVersion = invalidVersion; }

	std::vector<std::pair<Version, KeyRangeRef>> getShardAssignmentHistory(Version early, Version later) {
		std::vector<std::pair<Version, KeyRangeRef>> res;
		for (const auto& shardAssignment : shardAssignmentHistory) {
			if (shardAssignment.first >= early && shardAssignment.first <= later) {
				TraceEvent(SevVerbose, "ShardAssignmentHistoryGetOne", thisServerID)
				    .detail("Keys", shardAssignment.second)
				    .detail("Version", shardAssignment.first);
				res.push_back(shardAssignment);
			} else {
				TraceEvent(SevVerbose, "ShardAssignmentHistoryGetSkip", thisServerID)
				    .detail("Keys", shardAssignment.second)
				    .detail("Version", shardAssignment.first)
				    .detail("EarlyVersion", early)
				    .detail("LaterVersion", later);
			}
		}
		TraceEvent(SevVerbose, "ShardAssignmentHistoryGetDone", thisServerID)
		    .detail("EarlyVersion", early)
		    .detail("LaterVersion", later)
		    .detail("HistoryTotalSize", shardAssignmentHistory.size())
		    .detail("HistoryTotal", printShardAssignmentHistory());
		return res;
	}
};

const StringRef StorageServer::CurrentRunningFetchKeys::emptyString = ""_sr;
const KeyRangeRef StorageServer::CurrentRunningFetchKeys::emptyKeyRange =
    KeyRangeRef(StorageServer::CurrentRunningFetchKeys::emptyString,
                StorageServer::CurrentRunningFetchKeys::emptyString);

// If and only if key:=value is in (storage+versionedData),    // NOT ACTUALLY: and key < allKeys.end,
//   and H(key) < |key+value|/bytesPerSample,
//     let sampledSize = max(|key+value|,bytesPerSample)
//     persistByteSampleKeys.begin()+key := sampledSize is in storage
//     (key,sampledSize) is in byteSample

// So P(key is sampled) * sampledSize == |key+value|

void StorageServer::byteSampleApplyMutation(MutationRef const& m, Version ver) {
	if (m.type == MutationRef::ClearRange)
		byteSampleApplyClear(KeyRangeRef(m.param1, m.param2), ver);
	else if (m.type == MutationRef::SetValue)
		byteSampleApplySet(KeyValueRef(m.param1, m.param2), ver);
	else
		ASSERT(false); // Mutation of unknown type modifying byte sample
}

// watchMap Operations
Reference<ServerWatchMetadata> StorageServer::getWatchMetadata(KeyRef key, int64_t tenantId) const {
	const WatchMapKey mapKey(tenantId, key);
	const auto it = watchMap.find(mapKey);
	if (it == watchMap.end())
		return Reference<ServerWatchMetadata>();
	return it->second;
}

KeyRef StorageServer::setWatchMetadata(Reference<ServerWatchMetadata> metadata) {
	KeyRef keyRef = metadata->key.contents();
	int64_t tenantId = metadata->tenantId;
	const WatchMapKey mapKey(tenantId, keyRef);

	watchMap[mapKey] = metadata;
	return keyRef;
}

void StorageServer::deleteWatchMetadata(KeyRef key, int64_t tenantId) {
	const WatchMapKey mapKey(tenantId, key);
	watchMap.erase(mapKey);
}

void StorageServer::clearWatchMetadata() {
	watchMap.clear();
}

#ifndef __INTEL_COMPILER
#pragma endregion
#endif

/////////////////////////////////// Validation ///////////////////////////////////////
#ifndef __INTEL_COMPILER
#pragma region Validation
#endif
bool validateRange(StorageServer::VersionedData::ViewAtVersion const& view,
                   KeyRangeRef range,
                   Version version,
                   UID id,
                   Version minInsertVersion) {
	// * Nonoverlapping: No clear overlaps a set or another clear, or adjoins another clear.
	// * Old mutations are erased: All items in versionedData.atLatest() have insertVersion() > durableVersion()

	//TraceEvent("ValidateRange", id).detail("KeyBegin", range.begin).detail("KeyEnd", range.end).detail("Version", version);
	KeyRef k;
	bool ok = true;
	bool kIsClear = false;
	auto i = view.lower_bound(range.begin);
	if (i != view.begin())
		--i;
	for (; i != view.end() && i.key() < range.end; ++i) {
		ASSERT(i.insertVersion() > minInsertVersion);
		if (kIsClear && i->isClearTo() ? i.key() <= k : i.key() < k) {
			TraceEvent(SevError, "InvalidRange", id)
			    .detail("Key1", k)
			    .detail("Key2", i.key())
			    .detail("Version", version);
			ok = false;
		}
		// ASSERT( i.key() >= k );
		kIsClear = i->isClearTo();
		k = kIsClear ? i->getEndKey() : i.key();
	}
	return ok;
}

void validate(StorageServer* data, bool force = false) {
	try {
		if (!data->shuttingDown && (force || (EXPENSIVE_VALIDATION))) {
			data->newestAvailableVersion.validateCoalesced();
			data->newestDirtyVersion.validateCoalesced();

			for (auto s = data->shards.ranges().begin(); s != data->shards.ranges().end(); ++s) {
				TraceEvent(SevVerbose, "ValidateShard", data->thisServerID)
				    .detail("Range", s->range())
				    .detail("ShardID", format("%016llx", s->value()->getShardId()))
				    .detail("DesiredShardID", format("%016llx", s->value()->getDesiredShardId()))
				    .detail("ShardRange", s->value()->range())
				    .detail("ShardState", s->value()->debugDescribeState())
				    .log();
				ASSERT(s->value()->range() == s->range());
				ASSERT(!s->value()->range().empty());
				if (data->shardAware) {
					s->value()->validate();
				}
			}

			for (auto s = data->shards.ranges().begin(); s != data->shards.ranges().end(); ++s) {
				if (s->value()->isReadable()) {
					auto ar = data->newestAvailableVersion.intersectingRanges(s->range());
					for (auto a = ar.begin(); a != ar.end(); ++a) {
						TraceEvent(SevVerbose, "ValidateShardReadable", data->thisServerID)
						    .detail("Range", s->range())
						    .detail("ShardRange", s->value()->range())
						    .detail("ShardState", s->value()->debugDescribeState())
						    .detail("AvailableRange", a->range())
						    .detail("AvailableVersion", a->value())
						    .log();
						ASSERT(a->value() == latestVersion);
					}
				}
			}

			// * versionedData contains versions [storageVersion(), version.get()].  It might also contain version
			// (version.get()+1), in which changeDurableVersion may be deleting ghosts, and/or it might
			//      contain later versions if applyUpdate is on the stack.
			ASSERT(data->data().getOldestVersion() == data->storageVersion());
			ASSERT(data->data().getLatestVersion() == data->version.get() ||
			       data->data().getLatestVersion() == data->version.get() + 1 ||
			       (data->debug_inApplyUpdate && data->data().getLatestVersion() > data->version.get()));

			auto latest = data->data().atLatest();

			// * Old shards are erased: versionedData.atLatest() has entries (sets or clear *begins*) only for keys in
			// readable or adding,transferred shards.
			for (auto s = data->shards.ranges().begin(); s != data->shards.ranges().end(); ++s) {
				ShardInfo* shard = s->value().getPtr();
				if (!shard->isInVersionedData()) {
					auto beginNext = latest.lower_bound(s->begin());
					auto endNext = latest.lower_bound(s->end());
					if (beginNext != endNext) {
						TraceEvent(SevError, "VF", data->thisServerID)
						    .detail("LastValidTime", data->debug_lastValidateTime)
						    .detail("KeyBegin", s->begin())
						    .detail("KeyEnd", s->end())
						    .detail("DbgState", shard->debugDescribeState())
						    .detail("FirstKey", beginNext.key())
						    .detail("LastKey", endNext != latest.end() ? endNext.key() : "End"_sr)
						    .detail("FirstInsertV", beginNext.insertVersion())
						    .detail("LastInsertV", endNext != latest.end() ? endNext.insertVersion() : invalidVersion);
					}
					ASSERT(beginNext == endNext);
				}

				if (shard->assigned() && data->shardAware) {
					TraceEvent(SevVerbose, "ValidateAssignedShard", data->thisServerID)
					    .detail("Range", shard->range())
					    .detailf("ShardID", "%016llx", shard->getShardId())
					    .detailf("DesiredShardID", "%016llx", shard->getDesiredShardId())
					    .detail("State", shard->debugDescribeState());
					ASSERT(shard->getShardId() != 0UL && shard->getDesiredShardId() != 0UL);
				}
			}

			// FIXME: do some change feed validation?

			latest.validate();
			validateRange(latest, allKeys, data->version.get(), data->thisServerID, data->durableVersion.get());

			data->debug_lastValidateTime = now();
		}
	} catch (...) {
		TraceEvent(SevError, "ValidationFailure", data->thisServerID)
		    .detail("LastValidTime", data->debug_lastValidateTime);
		throw;
	}
}
#ifndef __INTEL_COMPILER
#pragma endregion
#endif

void updateProcessStats(StorageServer* self) {
	if (g_network->isSimulated()) {
		// diskUsage and cpuUsage are not relevant in the simulator,
		// and relying on the actual values could break seed determinism
		self->cpuUsage = 100.0;
		self->diskUsage = 100.0;
		return;
	}

	SystemStatistics sysStats = getSystemStatistics();
	if (sysStats.initialized) {
		self->cpuUsage = 100 * sysStats.processCPUSeconds / sysStats.elapsed;
		self->diskUsage = 100 * std::max(0.0, (sysStats.elapsed - sysStats.processDiskIdleSeconds) / sysStats.elapsed);
	}
}

///////////////////////////////////// Queries /////////////////////////////////
#ifndef __INTEL_COMPILER
#pragma region Queries
#endif

															#line 2622 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via waitForVersionActor()
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WaitForVersionActorActor>
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionActorActorState {
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionActorActorState(StorageServer* const& data,Version const& version,SpanContext const& spanContext) 
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   spanContext(spanContext),
															#line 2086 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:WaitForVersion"_loc, spanContext)
															#line 2642 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("waitForVersionActor", reinterpret_cast<unsigned long>(this));

	}
	~WaitForVersionActorActorState() 
	{
		fdb_probe_actor_destroy("waitForVersionActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2088 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(version);
															#line 2087 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WaitForVersionActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2659 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2096 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->FUTURE_VERSION_DELAY);
															#line 2663 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForVersionActorActor*>(this)->actor_wait_state = 1;
															#line 2088 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionActorActor, 0, Void >*>(static_cast<WaitForVersionActorActor*>(this)));
															#line 2096 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionActorActor, 1, Void >*>(static_cast<WaitForVersionActorActor*>(this)));
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForVersionActorActorState();
		static_cast<WaitForVersionActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
															#line 2091 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->oldestVersion.get())
															#line 2693 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2092 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), loopDepth);
															#line 2697 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2094 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionActorActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionActorActorState(); static_cast<WaitForVersionActorActor*>(this)->destroy(); return 0; }
															#line 2701 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForVersionActorActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionActorActorState();
		static_cast<WaitForVersionActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
															#line 2091 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->oldestVersion.get())
															#line 2713 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2092 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), loopDepth);
															#line 2717 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2094 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionActorActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionActorActorState(); static_cast<WaitForVersionActorActor*>(this)->destroy(); return 0; }
															#line 2721 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForVersionActorActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionActorActorState();
		static_cast<WaitForVersionActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 2097 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 2733 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2098 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 2737 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2102 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 2741 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 2097 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 2749 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2098 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 2753 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2102 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 2757 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForVersionActorActor*>(this)->actor_wait_state > 0) static_cast<WaitForVersionActorActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForVersionActorActor*>(this)->ActorCallback< WaitForVersionActorActor, 0, Void >::remove();
		static_cast<WaitForVersionActorActor*>(this)->ActorCallback< WaitForVersionActorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForVersionActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForVersionActorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForVersionActorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext spanContext;
															#line 2086 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 2890 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via waitForVersionActor()
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionActorActor final : public Actor<Version>, public ActorCallback< WaitForVersionActorActor, 0, Void >, public ActorCallback< WaitForVersionActorActor, 1, Void >, public FastAllocated<WaitForVersionActorActor>, public WaitForVersionActorActorState<WaitForVersionActorActor> {
															#line 2895 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<WaitForVersionActorActor>::operator new;
	using FastAllocated<WaitForVersionActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10960824064526162176UL, 11598922573575787008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForVersionActorActor, 0, Void >;
friend struct ActorCallback< WaitForVersionActorActor, 1, Void >;
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionActorActor(StorageServer* const& data,Version const& version,SpanContext const& spanContext) 
															#line 2913 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Version>(),
		   WaitForVersionActorActorState<WaitForVersionActorActor>(data, version, spanContext),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForVersionActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17177306746367876352UL, 6989725333754748672UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForVersionActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForVersionActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForVersionActorActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> waitForVersionActor( StorageServer* const& data, Version const& version, SpanContext const& spanContext ) {
															#line 2085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WaitForVersionActorActor(data, version, spanContext));
															#line 2946 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2106 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// If the latest commit version that mutated the shard(s) being served by the specified storage
// server is below the client specified read version then do a read at the latest commit version
// of the storage server.
Version getRealReadVersion(VersionVector& ssLatestCommitVersions, Tag& tag, Version specifiedReadVersion) {
	Version realReadVersion =
	    ssLatestCommitVersions.hasVersion(tag) ? ssLatestCommitVersions.getVersion(tag) : specifiedReadVersion;
	ASSERT(realReadVersion <= specifiedReadVersion);
	return realReadVersion;
}

// Find the latest commit version of the given tag.
Version getLatestCommitVersion(VersionVector& ssLatestCommitVersions, Tag& tag) {
	Version commitVersion =
	    ssLatestCommitVersions.hasVersion(tag) ? ssLatestCommitVersions.getVersion(tag) : invalidVersion;
	return commitVersion;
}

Future<Version> waitForVersion(StorageServer* data, Version version, SpanContext spanContext) {
	if (version == latestVersion) {
		version = std::max(Version(1), data->version.get());
	}

	if (version < data->oldestVersion.get() || version <= 0) {
		// TraceEvent(SevDebug, "WFVThrow", data->thisServerID).detail("Version", version).detail("OldestVersion", data->oldestVersion.get());
		return transaction_too_old();
	} else if (version <= data->version.get()) {
		return version;
	}

	if ((data->behind || data->versionBehind) && version > data->version.get()) {
		return process_behind();
	}

	if (deterministicRandom()->random01() < 0.001) {
		TraceEvent("WaitForVersion1000x").log();
	}
	return waitForVersionActor(data, version, spanContext);
}

Future<Version> waitForVersion(StorageServer* data,
                               Version commitVersion,
                               Version readVersion,
                               SpanContext spanContext) {
	ASSERT(commitVersion == invalidVersion || commitVersion < readVersion);

	if (commitVersion == invalidVersion) {
		return waitForVersion(data, readVersion, spanContext);
	}

	if (readVersion == latestVersion) {
		readVersion = std::max(Version(1), data->version.get());
	}

	if (readVersion < data->oldestVersion.get() || readVersion <= 0) {
		return transaction_too_old();
	} else {
		// It is correct to read any version between [commitVersion, readVersion],
		// because version vector guarantees no mutations between them.
		if (commitVersion < data->oldestVersion.get()) {
			if (data->version.get() < readVersion) {
				// Majority of the case, try using higher version to avoid
				// transaction_too_old error when oldestVersion advances.
				// BTW, any version in the range [oldestVersion, data->version.get()] is valid in this case.
				return data->version.get();
			} else {
				ASSERT(readVersion >= data->oldestVersion.get());
				return readVersion;
			}
		} else if (commitVersion <= data->version.get()) {
			return commitVersion;
		}
	}

	if ((data->behind || data->versionBehind) && commitVersion > data->version.get()) {
		return process_behind();
	}

	if (deterministicRandom()->random01() < 0.001) {
		TraceEvent("WaitForVersion1000x");
	}
	return waitForVersionActor(data, std::max(commitVersion, data->oldestVersion.get()), spanContext);
}

															#line 3034 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via waitForVersionNoTooOld()
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WaitForVersionNoTooOldActor>
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionNoTooOldActorState {
															#line 3041 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionNoTooOldActorState(StorageServer* const& data,Version const& version) 
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version)
															#line 3050 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this));

	}
	~WaitForVersionNoTooOldActorState() 
	{
		fdb_probe_actor_destroy("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2192 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version == latestVersion)
															#line 3065 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2193 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				version = std::max(Version(1), data->version.get());
															#line 3069 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2194 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version <= data->version.get())
															#line 3073 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2195 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 3077 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
				this->~WaitForVersionNoTooOldActorState();
				static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2197 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(version);
															#line 2196 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3087 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2200 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->FUTURE_VERSION_DELAY);
															#line 3091 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state = 1;
															#line 2197 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*>(static_cast<WaitForVersionNoTooOldActor*>(this)));
															#line 2200 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*>(static_cast<WaitForVersionNoTooOldActor*>(this)));
															#line 3098 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
															#line 2198 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 3121 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
															#line 2198 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForVersionNoTooOldActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForVersionNoTooOldActorState(); static_cast<WaitForVersionNoTooOldActor*>(this)->destroy(); return 0; }
															#line 3133 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForVersionNoTooOldActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForVersionNoTooOldActorState();
		static_cast<WaitForVersionNoTooOldActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 2201 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3145 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2202 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3149 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2206 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3153 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 2201 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3161 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2202 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3165 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2206 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3169 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state > 0) static_cast<WaitForVersionNoTooOldActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForVersionNoTooOldActor*>(this)->ActorCallback< WaitForVersionNoTooOldActor, 0, Void >::remove();
		static_cast<WaitForVersionNoTooOldActor*>(this)->ActorCallback< WaitForVersionNoTooOldActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForVersionNoTooOldActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3298 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via waitForVersionNoTooOld()
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForVersionNoTooOldActor final : public Actor<Version>, public ActorCallback< WaitForVersionNoTooOldActor, 0, Void >, public ActorCallback< WaitForVersionNoTooOldActor, 1, Void >, public FastAllocated<WaitForVersionNoTooOldActor>, public WaitForVersionNoTooOldActorState<WaitForVersionNoTooOldActor> {
															#line 3303 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<WaitForVersionNoTooOldActor>::operator new;
	using FastAllocated<WaitForVersionNoTooOldActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3664734090615808256UL, 9646845847410991360UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForVersionNoTooOldActor, 0, Void >;
friend struct ActorCallback< WaitForVersionNoTooOldActor, 1, Void >;
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForVersionNoTooOldActor(StorageServer* const& data,Version const& version) 
															#line 3321 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Version>(),
		   WaitForVersionNoTooOldActorState<WaitForVersionNoTooOldActor>(data, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14315589494631704320UL, 1805358750176616448UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForVersionNoTooOldActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForVersionNoTooOld");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForVersionNoTooOld", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForVersionNoTooOldActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> waitForVersionNoTooOld( StorageServer* const& data, Version const& version ) {
															#line 2190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WaitForVersionNoTooOldActor(data, version));
															#line 3354 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2210 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 3359 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via waitForMinVersion()
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WaitForMinVersionActor>
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForMinVersionActorState {
															#line 3366 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForMinVersionActorState(StorageServer* const& data,Version const& version) 
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version)
															#line 3375 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("waitForMinVersion", reinterpret_cast<unsigned long>(this));

	}
	~WaitForMinVersionActorState() 
	{
		fdb_probe_actor_destroy("waitForMinVersion", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2213 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version == latestVersion)
															#line 3390 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2214 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				version = std::max(Version(1), data->version.get());
															#line 3394 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2215 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (version < data->oldestVersion.get() || version <= 0)
															#line 3398 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2216 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(data->oldestVersion.get()); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3402 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(data->oldestVersion.get());
				this->~WaitForMinVersionActorState();
				static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 2217 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (version <= data->version.get())
															#line 3412 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2218 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3416 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
					this->~WaitForMinVersionActorState();
					static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
															#line 2221 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->version.whenAtLeast(version);
															#line 2220 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WaitForMinVersionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3427 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2224 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(SERVER_KNOBS->FUTURE_VERSION_DELAY);
															#line 3431 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<WaitForMinVersionActor*>(this)->actor_wait_state = 1;
															#line 2221 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WaitForMinVersionActor, 0, Void >*>(static_cast<WaitForMinVersionActor*>(this)));
															#line 2224 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WaitForMinVersionActor, 1, Void >*>(static_cast<WaitForMinVersionActor*>(this)));
															#line 3438 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WaitForMinVersionActorState();
		static_cast<WaitForMinVersionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
															#line 2222 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3461 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForMinVersionActorState();
		static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
															#line 2222 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WaitForMinVersionActor*>(this)->SAV<Version>::futures) { (void)(version); this->~WaitForMinVersionActorState(); static_cast<WaitForMinVersionActor*>(this)->destroy(); return 0; }
															#line 3473 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WaitForMinVersionActor*>(this)->SAV< Version >::value()) Version(std::move(version)); // state_var_RVO
		this->~WaitForMinVersionActorState();
		static_cast<WaitForMinVersionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 2225 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3485 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2226 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3489 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2230 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3493 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 2225 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 3501 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2226 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "ShardServerFutureVersion1000x", data->thisServerID) .detail("Version", version) .detail("MyVersion", data->version.get()) .detail("ServerID", data->thisServerID);
															#line 3505 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2230 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(future_version(), loopDepth);
															#line 3509 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WaitForMinVersionActor*>(this)->actor_wait_state > 0) static_cast<WaitForMinVersionActor*>(this)->actor_wait_state = 0;
		static_cast<WaitForMinVersionActor*>(this)->ActorCallback< WaitForMinVersionActor, 0, Void >::remove();
		static_cast<WaitForMinVersionActor*>(this)->ActorCallback< WaitForMinVersionActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WaitForMinVersionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WaitForMinVersionActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WaitForMinVersionActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3638 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via waitForMinVersion()
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WaitForMinVersionActor final : public Actor<Version>, public ActorCallback< WaitForMinVersionActor, 0, Void >, public ActorCallback< WaitForMinVersionActor, 1, Void >, public FastAllocated<WaitForMinVersionActor>, public WaitForMinVersionActorState<WaitForMinVersionActor> {
															#line 3643 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<WaitForMinVersionActor>::operator new;
	using FastAllocated<WaitForMinVersionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13826398196640195072UL, 1247622536751223552UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WaitForMinVersionActor, 0, Void >;
friend struct ActorCallback< WaitForMinVersionActor, 1, Void >;
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WaitForMinVersionActor(StorageServer* const& data,Version const& version) 
															#line 3661 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Version>(),
		   WaitForMinVersionActorState<WaitForMinVersionActor>(data, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("waitForMinVersion", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3358323868325720064UL, 6852897307082981376UL);
		ActorExecutionContextHelper __helper(static_cast<WaitForMinVersionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("waitForMinVersion");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("waitForMinVersion", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WaitForMinVersionActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> waitForMinVersion( StorageServer* const& data, Version const& version ) {
															#line 2211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WaitForMinVersionActor(data, version));
															#line 3694 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2234 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

void StorageServer::checkTenantEntry(Version version, TenantInfo tenantInfo, bool lockAware) {
	if (tenantInfo.hasTenant()) {
		ASSERT(version == latestVersion || (version >= tenantMap.oldestVersion && version <= this->version.get()));
		auto view = tenantMap.at(version);
		auto itr = view.find(tenantInfo.tenantId);
		if (itr == view.end()) {
			TraceEvent(SevWarn, "StorageTenantNotFound", thisServerID)
			    .detail("Tenant", tenantInfo.tenantId)
			    .backtrace();
			CODE_PROBE(true, "Storage server tenant not found");
			throw tenant_not_found();
		} else if (!lockAware && itr->lockState == TenantAPI::TenantLockState::LOCKED) {
			CODE_PROBE(true, "Storage server access locked tenant without lock awareness");
			throw tenant_locked();
		}
	}
}

std::vector<StorageServerShard> StorageServer::getStorageServerShards(KeyRangeRef range) {
	std::vector<StorageServerShard> res;
	for (auto t : this->shards.intersectingRanges(range)) {
		res.push_back(t.value()->toStorageServerShard());
	}
	return res;
}

static Error dataMoveConflictError(const bool isTss) {
	if (isTss && g_network->isSimulated()) {
		// TSS data move conflicts can happen in both sim and prod, but in sim,
		// the sev40s cause failures of Joshua tests. We have been using please_reboot
		// as means to avoid sev40s, but semantically this is undesired because rebooting
		// will not fix/heal the TSS.
		// TODO: think of a proper TSS move conflict error that does not trigger
		// reboot but also avoids sev40. And throw that error regardless of sim or prod.
		return please_reboot();
	}
	return data_move_conflict();
}

std::shared_ptr<MoveInShard> StorageServer::getMoveInShard(const UID& dataMoveId,
                                                           const Version version,
                                                           const ConductBulkLoad conductBulkLoad) {
	for (auto& [id, moveInShard] : this->moveInShards) {
		if (moveInShard->dataMoveId() == dataMoveId && moveInShard->meta->createVersion == version) {
			return moveInShard;
		}
	}

	const UID id = deterministicRandom()->randomUniqueID();
	std::shared_ptr<MoveInShard> shard = std::make_shared<MoveInShard>(this, id, dataMoveId, version, conductBulkLoad);
	auto [it, inserted] = this->moveInShards.emplace(id, shard);
	ASSERT(inserted);
	TraceEvent(SevDebug, "SSNewMoveInShard", this->thisServerID)
	    .detail("MoveInShard", shard->toString())
	    .detail("ConductBulkLoad", conductBulkLoad);
	return shard;
}

															#line 3757 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getValueQ()
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetValueQActor>
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetValueQActorState {
															#line 3764 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueQActorState(StorageServer* const& data,GetValueRequest const& req) 
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req),
															#line 2294 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultSize(0)
															#line 3775 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("getValueQ", reinterpret_cast<unsigned long>(this));

	}
	~GetValueQActorState() 
	{
		fdb_probe_actor_destroy("getValueQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2295 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Span span("SS:getValue"_loc, req.spanContext);
															#line 3790 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			try {
															#line 2300 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.getValueQueries;
															#line 2301 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.allQueries;
															#line 2302 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (req.key.startsWith(systemKeys.begin))
															#line 3798 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2303 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					++data->counters.systemKeyQueries;
															#line 3802 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 2305 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				data->maxQueryQueue = std::max<int>( data->maxQueryQueue, data->counters.allQueries.getValue() - data->counters.finishedQueries.getValue());
															#line 2310 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_0 = data->getQueryDelay();
															#line 2310 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3810 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetValueQActor*>(this)->actor_wait_state = 1;
															#line 2310 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 0, Void >*>(static_cast<GetValueQActor*>(this)));
															#line 3815 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetValueQActorState();
		static_cast<GetValueQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2418 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->transactionTagCounter.addRequest(req.tags, req.key.size() + resultSize);
															#line 2420 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.finishedQueries;
															#line 2422 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - req.requestTime();
															#line 2423 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySample->addMeasurement(duration);
															#line 2424 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readValueLatencySample->addMeasurement(duration);
															#line 2425 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->latencyBandConfig.present())
															#line 3854 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2426 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int maxReadBytes = data->latencyBandConfig.get().readConfig.maxReadBytes.orDefault(std::numeric_limits<int>::max());
															#line 2428 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.readLatencyBands.addMeasurement(duration, 1, Filtered(resultSize > maxReadBytes));
															#line 3860 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2431 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetValueQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetValueQActorState(); static_cast<GetValueQActor*>(this)->destroy(); return 0; }
															#line 3864 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<GetValueQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetValueQActorState();
		static_cast<GetValueQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2411 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 3877 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2412 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3881 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2413 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->sendErrorWithPenalty(req.reply, e, data->getPenalty());
															#line 3885 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 2311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3902 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetValueQActor*>(this)->actor_wait_state = 2;
															#line 2311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetValueQActor*>(this)));
															#line 3907 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 2311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3918 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetValueQActor*>(this)->actor_wait_state = 2;
															#line 2311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetValueQActor*>(this)));
															#line 3923 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 2314 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		queueWaitEnd = g_network->timer();
															#line 2315 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readQueueWaitSample->addMeasurement(queueWaitEnd - req.requestTime());
															#line 2317 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 4011 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2318 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.DoRead");
															#line 4015 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2322 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		v = Optional<Value>();
															#line 2323 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version commitVersion = getLatestCommitVersion(req.ssLatestCommitVersions, data->tag);
															#line 2324 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Version> __when_expr_2 = waitForVersion(data, commitVersion, req.version, req.spanContext);
															#line 2324 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4025 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetValueQActor*>(this)->actor_wait_state = 3;
															#line 2324 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 2, Version >*>(static_cast<GetValueQActor*>(this)));
															#line 4030 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(PriorityMultiLock::Lock const& __readLock,int loopDepth) 
	{
															#line 2311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readLock = __readLock;
															#line 4039 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(PriorityMultiLock::Lock && __readLock,int loopDepth) 
	{
		readLock = std::move(__readLock);
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 2325 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readVersionWaitSample->addMeasurement(g_network->timer() - queueWaitEnd);
															#line 2327 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 4120 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2328 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.AfterVersion");
															#line 4124 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2332 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkTenantEntry(version, req.tenantInfo, req.options.present() ? req.options.get().lockAware : false);
															#line 2333 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.tenantInfo.hasTenant())
															#line 4130 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2334 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.key = req.key.withPrefix(req.tenantInfo.prefix.get());
															#line 4134 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2336 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		changeCounter = data->shardChangeCounter;
															#line 2338 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!data->shards[req.key]->isReadable())
															#line 4140 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2340 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(wrong_shard_server(), loopDepth);
															#line 4144 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2343 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		path = 0;
															#line 2344 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto i = data->data().at(version).lastLessOrEqual(req.key);
															#line 2345 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (i && i->isValue() && i.key() == req.key)
															#line 4152 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2346 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			v = (Value)i->getValue();
															#line 2347 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			path = 1;
															#line 4158 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont7(loopDepth);
		}
		else
		{
															#line 2348 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!i || !i->isClearTo() || i->getEndKey() <= req.key)
															#line 4165 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2349 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				path = 2;
															#line 2350 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Optional<Value>> __when_expr_3 = data->storage.readValue(req.key, req.options);
															#line 2350 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<GetValueQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4173 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
				static_cast<GetValueQActor*>(this)->actor_wait_state = 4;
															#line 2350 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetValueQActor, 3, Optional<Value> >*>(static_cast<GetValueQActor*>(this)));
															#line 4178 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont12(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1cont4when1(Version const& __version,int loopDepth) 
	{
															#line 2324 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		version = __version;
															#line 4193 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Version && __version,int loopDepth) 
	{
		version = std::move(__version);
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 2, Version >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 2, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 2, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 2, Version >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 2361 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardGetValue", version, MutationRef(MutationRef::DebugKey, req.key, v.present() ? v.get() : "<null>"_sr), data->thisServerID);
															#line 2365 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardGetPath", version, MutationRef(MutationRef::DebugKey, req.key, path == 0 ? "0"_sr : path == 1 ? "1"_sr : "2"_sr), data->thisServerID);
															#line 2381 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (v.present())
															#line 4276 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2382 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.rowsQueried;
															#line 2383 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			resultSize = v.get().size();
															#line 2384 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.bytesQueried += resultSize;
															#line 4284 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 2386 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.emptyQueries;
															#line 4290 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2389 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 4294 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2391 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int64_t bytesReadPerKSecond = v.present() ? std::max((int64_t)(req.key.size() + v.get().size()), SERVER_KNOBS->EMPTY_READ_PENALTY) : SERVER_KNOBS->EMPTY_READ_PENALTY;
															#line 2394 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond(req.key, bytesReadPerKSecond);
															#line 4300 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2397 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 4304 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2398 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("GetValueDebug", req.options.get().debugID.get().first(), "getValueQ.AfterRead");
															#line 4308 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2403 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto cached = data->cachedRangeMap[req.key];
															#line 2407 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		GetValueReply reply(v, cached);
															#line 2408 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.penalty = data->getPenalty();
															#line 2409 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 4318 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont19(loopDepth);

		return loopDepth;
	}
	int a_body1cont12(int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(Optional<Value> const& vv,int loopDepth) 
	{
															#line 2351 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvGetBytes += vv.expectedSize();
															#line 2353 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->storageVersion())
															#line 4335 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2354 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "transaction_too_old after readValue");
															#line 2355 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(transaction_too_old(), loopDepth);
															#line 4341 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2357 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter(changeCounter, req.key);
															#line 2358 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		v = vv;
															#line 4347 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(Optional<Value> && vv,int loopDepth) 
	{
															#line 2351 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvGetBytes += vv.expectedSize();
															#line 2353 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (version < data->storageVersion())
															#line 4358 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2354 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "transaction_too_old after readValue");
															#line 2355 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch2(transaction_too_old(), loopDepth);
															#line 4364 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2357 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter(changeCounter, req.key);
															#line 2358 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		v = vv;
															#line 4370 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Optional<Value> const& vv,int loopDepth) 
	{
		loopDepth = a_body1cont13(vv, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Optional<Value> && vv,int loopDepth) 
	{
		loopDepth = a_body1cont13(std::move(vv), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetValueQActor*>(this)->actor_wait_state > 0) static_cast<GetValueQActor*>(this)->actor_wait_state = 0;
		static_cast<GetValueQActor*>(this)->ActorCallback< GetValueQActor, 3, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 3, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetValueQActor, 3, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetValueQActor, 3, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont19(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueRequest req;
															#line 2294 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t resultSize;
															#line 2311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	PriorityMultiLock::Lock readLock;
															#line 2314 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double queueWaitEnd;
															#line 2322 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<Value> v;
															#line 2324 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 2336 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	uint64_t changeCounter;
															#line 2343 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int path;
															#line 4481 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via getValueQ()
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetValueQActor final : public Actor<Void>, public ActorCallback< GetValueQActor, 0, Void >, public ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >, public ActorCallback< GetValueQActor, 2, Version >, public ActorCallback< GetValueQActor, 3, Optional<Value> >, public FastAllocated<GetValueQActor>, public GetValueQActorState<GetValueQActor> {
															#line 4486 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<GetValueQActor>::operator new;
	using FastAllocated<GetValueQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6375139201164447488UL, 5017849053673497600UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetValueQActor, 0, Void >;
friend struct ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >;
friend struct ActorCallback< GetValueQActor, 2, Version >;
friend struct ActorCallback< GetValueQActor, 3, Optional<Value> >;
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueQActor(StorageServer* const& data,GetValueRequest const& req) 
															#line 4506 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   GetValueQActorState<GetValueQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getValueQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17920039446934625280UL, 7083594275336435200UL);
		ActorExecutionContextHelper __helper(static_cast<GetValueQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getValueQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getValueQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetValueQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetValueQActor, 1, PriorityMultiLock::Lock >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetValueQActor, 2, Version >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetValueQActor, 3, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getValueQ( StorageServer* const& data, GetValueRequest const& req ) {
															#line 2293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetValueQActor(data, req));
															#line 4542 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2433 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// Pessimistic estimate the number of overhead bytes used by each
// watch. Watch key references are stored in an AsyncMap<Key,bool>, and actors
// must be kept alive until the watch is finished.
extern size_t WATCH_OVERHEAD_WATCHQ, WATCH_OVERHEAD_WATCHIMPL;

															#line 4552 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via watchWaitForValueChange()
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WatchWaitForValueChangeActor>
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchWaitForValueChangeActorState {
															#line 4559 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchWaitForValueChangeActorState(StorageServer* const& data,SpanContext const& parent,KeyRef const& key,int64_t const& tenantId) 
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   parent(parent),
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   key(key),
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   tenantId(tenantId),
															#line 2440 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   spanLocation("SS:watchWaitForValueChange"_loc),
															#line 2441 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   span(spanLocation, parent),
															#line 2442 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   metadata(data->getWatchMetadata(key, tenantId))
															#line 4578 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("watchWaitForValueChange", reinterpret_cast<unsigned long>(this));

	}
	~WatchWaitForValueChangeActorState() 
	{
		fdb_probe_actor_destroy("watchWaitForValueChange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2443 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (metadata->debugID.present())
															#line 4593 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2444 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.Before");
															#line 4597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2448 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			originalMetadataVersion = metadata->version;
															#line 2449 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(waitForVersionNoTooOld(data, metadata->version));
															#line 2449 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4605 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 1;
															#line 2449 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 0, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4610 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WatchWaitForValueChangeActorState();
		static_cast<WatchWaitForValueChangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2450 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 4633 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2451 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterVersion");
															#line 4637 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2455 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		minVersion = data->data().latestVersion;
															#line 2456 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		watchFuture = data->watches.onChange(metadata->key);
															#line 2457 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (tenantId != TenantInfo::INVALID_TENANT)
															#line 4645 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2458 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			watchFuture = watchFuture || data->tenantWatches.onChange(tenantId);
															#line 4649 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2460 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		options = ReadOptions();
															#line 2461 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 4655 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2450 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 4664 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2451 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterVersion");
															#line 4668 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2455 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		minVersion = data->data().latestVersion;
															#line 2456 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		watchFuture = data->watches.onChange(metadata->key);
															#line 2457 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (tenantId != TenantInfo::INVALID_TENANT)
															#line 4676 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2458 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			watchFuture = watchFuture || data->tenantWatches.onChange(tenantId);
															#line 4680 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2460 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		options = ReadOptions();
															#line 2461 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 4686 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 2463 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (tenantId != TenantInfo::INVALID_TENANT)
															#line 4778 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2464 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				auto view = data->tenantMap.at(latestVersion);
															#line 2465 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (view.find(tenantId) == view.end())
															#line 4784 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2466 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					CODE_PROBE(true, "Watched tenant removed");
															#line 2467 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					return a_body1cont1loopBody1Catch1(tenant_removed(), loopDepth);
															#line 4790 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 2470 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			metadata = data->getWatchMetadata(key, tenantId);
															#line 2471 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			latest = data->version.get();
															#line 2472 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			options.debugID = metadata->debugID;
															#line 2474 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(latest >= minVersion && latest < data->data().latestVersion, "Starting watch loop with latestVersion > data->version", probe::decoration::rare);
															#line 2477 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			GetValueRequest getReq( span.context, TenantInfo(), metadata->key, latest, metadata->tags, options, VersionVector());
															#line 2479 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getValue = getValueQ( data, getReq);
															#line 2481 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetValueReply> __when_expr_1 = getReq.reply.getFuture();
															#line 2481 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4809 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 2;
															#line 2481 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4814 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 2559 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		watchFuture = data->watches.onChange(metadata->key);
															#line 2560 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (tenantId != TenantInfo::INVALID_TENANT)
															#line 4831 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2561 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			watchFuture = watchFuture || data->tenantWatches.onChange(tenantId);
															#line 4835 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2564 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = data->version.whenAtLeast(data->data().latestVersion);
															#line 2564 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4841 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 4;
															#line 2564 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 3, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4846 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2552 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() != error_code_transaction_too_old)
															#line 4856 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2553 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 4860 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2556 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Reading a watched key failed with transaction_too_old");
															#line 4864 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(GetValueReply const& reply,int loopDepth) 
	{
															#line 2482 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		span = Span(spanLocation, parent);
															#line 2484 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.error.present())
															#line 4881 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2485 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(reply.error.get().code() != error_code_future_version);
															#line 2486 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(reply.error.get(), loopDepth);
															#line 4887 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (BUGGIFY)
															#line 4891 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2489 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(transaction_too_old(), loopDepth);
															#line 4895 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2492 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardWatchValue", latest, MutationRef(MutationRef::DebugKey, metadata->key, reply.value.present() ? StringRef(reply.value.get()) : "<null>"_sr), data->thisServerID);
															#line 2499 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 4901 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2500 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterRead");
															#line 4905 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2508 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version waitVersion = minVersion;
															#line 2509 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.value != metadata->value)
															#line 4911 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2510 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest >= metadata->version)
															#line 4915 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2511 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WatchWaitForValueChangeActor*>(this)->SAV<Version>::futures) { (void)(latest); this->~WatchWaitForValueChangeActorState(); static_cast<WatchWaitForValueChangeActor*>(this)->destroy(); return 0; }
															#line 4919 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<WatchWaitForValueChangeActor*>(this)->SAV< Version >::value()) Version(std::move(latest)); // state_var_RVO
				this->~WatchWaitForValueChangeActorState();
				static_cast<WatchWaitForValueChangeActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 2512 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (metadata->version > originalMetadataVersion)
															#line 4929 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2515 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					CODE_PROBE(true, "racing watches for same value at different versions", probe::decoration::rare);
															#line 2516 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (metadata->version > waitVersion)
															#line 4935 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 2517 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						waitVersion = metadata->version;
															#line 4939 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
			}
		}
															#line 2522 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->watchBytes > SERVER_KNOBS->MAX_STORAGE_SERVER_WATCH_BYTES)
															#line 4946 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2523 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Too many watches, reverting to polling");
															#line 2524 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(watch_cancelled(), loopDepth);
															#line 4952 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2527 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		watchBytes = (metadata->key.expectedSize() + metadata->value.expectedSize() + key.expectedSize() + sizeof(Reference<ServerWatchMetadata>) + sizeof(ServerWatchMetadata) + WATCH_OVERHEAD_WATCHIMPL);
															#line 2531 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes += watchBytes;
															#line 4958 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2533 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest < waitVersion)
															#line 4962 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2535 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || data->version.whenAtLeast(waitVersion);
															#line 4966 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2537 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (BUGGIFY)
															#line 4970 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2539 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || delay(deterministicRandom()->random01());
															#line 4974 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2542 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (metadata->debugID.present())
															#line 4978 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2543 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.WaitChange");
															#line 4982 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2545 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = watchFuture;
															#line 2545 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 4988 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 3;
															#line 2545 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 2, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 4993 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(GetValueReply && reply,int loopDepth) 
	{
															#line 2482 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		span = Span(spanLocation, parent);
															#line 2484 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.error.present())
															#line 5010 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2485 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(reply.error.get().code() != error_code_future_version);
															#line 2486 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(reply.error.get(), loopDepth);
															#line 5016 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (BUGGIFY)
															#line 5020 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2489 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(transaction_too_old(), loopDepth);
															#line 5024 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2492 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		DEBUG_MUTATION("ShardWatchValue", latest, MutationRef(MutationRef::DebugKey, metadata->key, reply.value.present() ? StringRef(reply.value.get()) : "<null>"_sr), data->thisServerID);
															#line 2499 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (metadata->debugID.present())
															#line 5030 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2500 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.AfterRead");
															#line 5034 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2508 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Version waitVersion = minVersion;
															#line 2509 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reply.value != metadata->value)
															#line 5040 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2510 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest >= metadata->version)
															#line 5044 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2511 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<WatchWaitForValueChangeActor*>(this)->SAV<Version>::futures) { (void)(latest); this->~WatchWaitForValueChangeActorState(); static_cast<WatchWaitForValueChangeActor*>(this)->destroy(); return 0; }
															#line 5048 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<WatchWaitForValueChangeActor*>(this)->SAV< Version >::value()) Version(std::move(latest)); // state_var_RVO
				this->~WatchWaitForValueChangeActorState();
				static_cast<WatchWaitForValueChangeActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 2512 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (metadata->version > originalMetadataVersion)
															#line 5058 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2515 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					CODE_PROBE(true, "racing watches for same value at different versions", probe::decoration::rare);
															#line 2516 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (metadata->version > waitVersion)
															#line 5064 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 2517 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						waitVersion = metadata->version;
															#line 5068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
			}
		}
															#line 2522 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->watchBytes > SERVER_KNOBS->MAX_STORAGE_SERVER_WATCH_BYTES)
															#line 5075 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2523 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Too many watches, reverting to polling");
															#line 2524 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(watch_cancelled(), loopDepth);
															#line 5081 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2527 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		watchBytes = (metadata->key.expectedSize() + metadata->value.expectedSize() + key.expectedSize() + sizeof(Reference<ServerWatchMetadata>) + sizeof(ServerWatchMetadata) + WATCH_OVERHEAD_WATCHIMPL);
															#line 2531 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes += watchBytes;
															#line 5087 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2533 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (latest < waitVersion)
															#line 5091 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2535 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || data->version.whenAtLeast(waitVersion);
															#line 5095 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2537 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (BUGGIFY)
															#line 5099 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2539 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				watchFuture = watchFuture || delay(deterministicRandom()->random01());
															#line 5103 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2542 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (metadata->debugID.present())
															#line 5107 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2543 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "WatchValueDebug", metadata->debugID.get().first(), "watchValueSendReply.WaitChange");
															#line 5111 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2545 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = watchFuture;
															#line 2545 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 5117 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 3;
															#line 2545 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchWaitForValueChangeActor, 2, Void >*>(static_cast<WatchWaitForValueChangeActor*>(this)));
															#line 5122 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(GetValueReply const& reply,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(reply, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(GetValueReply && reply,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*,GetValueReply const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*,GetValueReply && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont19(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2548 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->watchBytes -= watchBytes;
															#line 2549 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(e, loopDepth);
															#line 5221 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont15(Void const& _,int loopDepth) 
	{
															#line 2546 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= watchBytes;
															#line 5235 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont15cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont15(Void && _,int loopDepth) 
	{
															#line 2546 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= watchBytes;
															#line 5244 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont15cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont15(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont15(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont15cont2(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont19(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont20(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont20(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont20(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont20(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state > 0) static_cast<WatchWaitForValueChangeActor*>(this)->actor_wait_state = 0;
		static_cast<WatchWaitForValueChangeActor*>(this)->ActorCallback< WatchWaitForValueChangeActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WatchWaitForValueChangeActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WatchWaitForValueChangeActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext parent;
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRef key;
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t tenantId;
															#line 2440 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Location spanLocation;
															#line 2441 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 2442 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Reference<ServerWatchMetadata> metadata;
															#line 2448 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version originalMetadataVersion;
															#line 2455 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version minVersion;
															#line 2456 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Void> watchFuture;
															#line 2460 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadOptions options;
															#line 2471 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version latest;
															#line 2479 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Void> getValue;
															#line 2527 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t watchBytes;
															#line 5465 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via watchWaitForValueChange()
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchWaitForValueChangeActor final : public Actor<Version>, public ActorCallback< WatchWaitForValueChangeActor, 0, Void >, public ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >, public ActorCallback< WatchWaitForValueChangeActor, 2, Void >, public ActorCallback< WatchWaitForValueChangeActor, 3, Void >, public FastAllocated<WatchWaitForValueChangeActor>, public WatchWaitForValueChangeActorState<WatchWaitForValueChangeActor> {
															#line 5470 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<WatchWaitForValueChangeActor>::operator new;
	using FastAllocated<WatchWaitForValueChangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11930950175516195328UL, 3393915456275492096UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Version>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WatchWaitForValueChangeActor, 0, Void >;
friend struct ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >;
friend struct ActorCallback< WatchWaitForValueChangeActor, 2, Void >;
friend struct ActorCallback< WatchWaitForValueChangeActor, 3, Void >;
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchWaitForValueChangeActor(StorageServer* const& data,SpanContext const& parent,KeyRef const& key,int64_t const& tenantId) 
															#line 5490 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Version>(),
		   WatchWaitForValueChangeActorState<WatchWaitForValueChangeActor>(data, parent, key, tenantId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(510625173218191360UL, 2956297564950870528UL);
		ActorExecutionContextHelper __helper(static_cast<WatchWaitForValueChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("watchWaitForValueChange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("watchWaitForValueChange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 1, GetValueReply >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WatchWaitForValueChangeActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Version> watchWaitForValueChange( StorageServer* const& data, SpanContext const& parent, KeyRef const& key, int64_t const& tenantId ) {
															#line 2439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Version>(new WatchWaitForValueChangeActor(data, parent, key, tenantId));
															#line 5526 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2567 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

void checkCancelWatchImpl(StorageServer* data, WatchValueRequest req) {
	Reference<ServerWatchMetadata> metadata = data->getWatchMetadata(req.key.contents(), req.tenantInfo.tenantId);
	if (metadata.isValid() && metadata->versionPromise.getFutureReferenceCount() == 1) {
		// last watch timed out so cancel watch_impl and delete key from the map
		data->deleteWatchMetadata(req.key.contents(), req.tenantInfo.tenantId);
		metadata->watch_impl.cancel();
	}
}

															#line 5540 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via watchValueSendReply()
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class WatchValueSendReplyActor>
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchValueSendReplyActorState {
															#line 5547 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchValueSendReplyActorState(StorageServer* const& data,WatchValueRequest const& req,Future<Version> const& resp,SpanContext const& spanContext) 
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req),
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   resp(resp),
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   spanContext(spanContext),
															#line 2581 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:watchValue"_loc, spanContext),
															#line 2582 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   startTime(now())
															#line 5564 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("watchValueSendReply", reinterpret_cast<unsigned long>(this));

	}
	~WatchValueSendReplyActorState() 
	{
		fdb_probe_actor_destroy("watchValueSendReply", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2583 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.watchQueries;
															#line 2584 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->numWatches;
															#line 2585 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->watchBytes += WATCH_OVERHEAD_WATCHQ;
															#line 2587 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 5585 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2588 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		double timeoutDelay = -1;
															#line 2589 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->noRecentUpdates.get())
															#line 5617 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2590 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			timeoutDelay = std::max(CLIENT_KNOBS->FAST_WATCH_TIMEOUT - (now() - startTime), 0.0);
															#line 5621 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 2591 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!BUGGIFY)
															#line 5627 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2592 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				timeoutDelay = std::max(CLIENT_KNOBS->WATCH_TIMEOUT - (now() - startTime), 0.0);
															#line 5631 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
		try {
															#line 2597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Version> __when_expr_0 = resp;
															#line 2596 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5639 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 2605 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = timeoutDelay < 0 ? Never() : delay(timeoutDelay);
															#line 5643 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
															#line 2613 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = data->noRecentUpdates.onChange();
															#line 5647 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1when3(__when_expr_2.get(), loopDepth); };
			static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state = 1;
															#line 2597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WatchValueSendReplyActor, 0, Version >*>(static_cast<WatchValueSendReplyActor*>(this)));
															#line 2605 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WatchValueSendReplyActor, 1, Void >*>(static_cast<WatchValueSendReplyActor*>(this)));
															#line 2613 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WatchValueSendReplyActor, 2, Void >*>(static_cast<WatchValueSendReplyActor*>(this)));
															#line 5656 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2616 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2617 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			checkCancelWatchImpl(data, req);
															#line 2618 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			--data->numWatches;
															#line 2620 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 5684 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2621 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 5688 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2622 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->sendErrorWithPenalty(req.reply, e, data->getPenalty());
															#line 2623 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5694 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~WatchValueSendReplyActorState();
			static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Version const& ver,int loopDepth) 
	{
															#line 2599 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(WatchValueReply{ ver });
															#line 2600 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2601 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2602 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2603 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5726 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Version && ver,int loopDepth) 
	{
															#line 2599 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(WatchValueReply{ ver });
															#line 2600 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2601 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2602 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2603 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5746 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 2607 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 2608 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2609 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2610 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2611 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5766 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 2607 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 2608 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		checkCancelWatchImpl(data, req);
															#line 2609 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		--data->numWatches;
															#line 2610 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->watchBytes -= WATCH_OVERHEAD_WATCHQ;
															#line 2611 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<WatchValueSendReplyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WatchValueSendReplyActorState(); static_cast<WatchValueSendReplyActor*>(this)->destroy(); return 0; }
															#line 5786 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<WatchValueSendReplyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WatchValueSendReplyActorState();
		static_cast<WatchValueSendReplyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state > 0) static_cast<WatchValueSendReplyActor*>(this)->actor_wait_state = 0;
		static_cast<WatchValueSendReplyActor*>(this)->ActorCallback< WatchValueSendReplyActor, 0, Version >::remove();
		static_cast<WatchValueSendReplyActor*>(this)->ActorCallback< WatchValueSendReplyActor, 1, Void >::remove();
		static_cast<WatchValueSendReplyActor*>(this)->ActorCallback< WatchValueSendReplyActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 0, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 0, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WatchValueSendReplyActor, 0, Version >*,Error err) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WatchValueSendReplyActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WatchValueSendReplyActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WatchValueSendReplyActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchValueRequest req;
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Version> resp;
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext spanContext;
															#line 2581 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 2582 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double startTime;
															#line 6010 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via watchValueSendReply()
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class WatchValueSendReplyActor final : public Actor<Void>, public ActorCallback< WatchValueSendReplyActor, 0, Version >, public ActorCallback< WatchValueSendReplyActor, 1, Void >, public ActorCallback< WatchValueSendReplyActor, 2, Void >, public FastAllocated<WatchValueSendReplyActor>, public WatchValueSendReplyActorState<WatchValueSendReplyActor> {
															#line 6015 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<WatchValueSendReplyActor>::operator new;
	using FastAllocated<WatchValueSendReplyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13947502085738939136UL, 18164697969206351360UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WatchValueSendReplyActor, 0, Version >;
friend struct ActorCallback< WatchValueSendReplyActor, 1, Void >;
friend struct ActorCallback< WatchValueSendReplyActor, 2, Void >;
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	WatchValueSendReplyActor(StorageServer* const& data,WatchValueRequest const& req,Future<Version> const& resp,SpanContext const& spanContext) 
															#line 6034 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   WatchValueSendReplyActorState<WatchValueSendReplyActor>(data, req, resp, spanContext),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("watchValueSendReply", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3738460070669966080UL, 521894726446841856UL);
		ActorExecutionContextHelper __helper(static_cast<WatchValueSendReplyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("watchValueSendReply");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("watchValueSendReply", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WatchValueSendReplyActor, 0, Version >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> watchValueSendReply( StorageServer* const& data, WatchValueRequest const& req, Future<Version> const& resp, SpanContext const& spanContext ) {
															#line 2577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new WatchValueSendReplyActor(data, req, resp, spanContext));
															#line 6067 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2627 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// Finds a checkpoint.
															#line 6073 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getCheckpointQ()
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetCheckpointQActor>
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetCheckpointQActorState {
															#line 6080 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetCheckpointQActorState(StorageServer* const& self,GetCheckpointRequest const& req) 
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 6089 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("getCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	~GetCheckpointQActorState() 
	{
		fdb_probe_actor_destroy("getCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2631 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->durableVersion.whenAtLeast(req.version + 1);
															#line 2631 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6106 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetCheckpointQActor*>(this)->actor_wait_state = 1;
															#line 2631 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetCheckpointQActor, 0, Void >*>(static_cast<GetCheckpointQActor*>(this)));
															#line 6111 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetCheckpointQActorState();
		static_cast<GetCheckpointQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2633 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevDebug, "ServeGetCheckpointVersionSatisfied", self->thisServerID) .detail("Version", req.version) .detail("Ranges", describe(req.ranges)) .detail("Format", static_cast<int>(req.format));
															#line 2637 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(req.ranges.size() == 1);
															#line 2638 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( const auto& range : req.ranges ) {
															#line 2639 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!self->isReadable(range))
															#line 6140 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2640 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(wrong_shard_server());
															#line 2641 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<GetCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetCheckpointQActorState(); static_cast<GetCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6146 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<GetCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~GetCheckpointQActorState();
				static_cast<GetCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
		try {
															#line 2646 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::unordered_map<UID, CheckpointMetaData>::iterator it = self->checkpoints.begin();
															#line 2647 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;it != self->checkpoints.end();++it) {
															#line 2648 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				const CheckpointMetaData& md = it->second;
															#line 2649 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (md.version == req.version && md.format == req.format && req.actionId == md.actionId && md.hasRanges(req.ranges) && md.getState() == CheckpointMetaData::Complete)
															#line 6162 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2651 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.send(md);
															#line 2652 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevDebug, "ServeGetCheckpointEnd", self->thisServerID).detail("Checkpoint", md.toString());
															#line 6168 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					break;
				}
			}
															#line 2657 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (it == self->checkpoints.end())
															#line 6174 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2658 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(checkpoint_not_found());
															#line 6178 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			loopDepth = a_body1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2633 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevDebug, "ServeGetCheckpointVersionSatisfied", self->thisServerID) .detail("Version", req.version) .detail("Ranges", describe(req.ranges)) .detail("Format", static_cast<int>(req.format));
															#line 2637 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(req.ranges.size() == 1);
															#line 2638 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( const auto& range : req.ranges ) {
															#line 2639 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!self->isReadable(range))
															#line 6200 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2640 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(wrong_shard_server());
															#line 2641 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<GetCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetCheckpointQActorState(); static_cast<GetCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6206 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<GetCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~GetCheckpointQActorState();
				static_cast<GetCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}
		try {
															#line 2646 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::unordered_map<UID, CheckpointMetaData>::iterator it = self->checkpoints.begin();
															#line 2647 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;it != self->checkpoints.end();++it) {
															#line 2648 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				const CheckpointMetaData& md = it->second;
															#line 2649 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (md.version == req.version && md.format == req.format && req.actionId == md.actionId && md.hasRanges(req.ranges) && md.getState() == CheckpointMetaData::Complete)
															#line 6222 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2651 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.send(md);
															#line 2652 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevDebug, "ServeGetCheckpointEnd", self->thisServerID).detail("Checkpoint", md.toString());
															#line 6228 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					break;
				}
			}
															#line 2657 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (it == self->checkpoints.end())
															#line 6234 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2658 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(checkpoint_not_found());
															#line 6238 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			loopDepth = a_body1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<GetCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<GetCheckpointQActor*>(this)->ActorCallback< GetCheckpointQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetCheckpointQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetCheckpointQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetCheckpointQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2666 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetCheckpointQActorState(); static_cast<GetCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6329 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<GetCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetCheckpointQActorState();
		static_cast<GetCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2661 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 6342 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2662 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 6346 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2664 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(e);
															#line 6350 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetCheckpointRequest req;
															#line 6378 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via getCheckpointQ()
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetCheckpointQActor final : public Actor<Void>, public ActorCallback< GetCheckpointQActor, 0, Void >, public FastAllocated<GetCheckpointQActor>, public GetCheckpointQActorState<GetCheckpointQActor> {
															#line 6383 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<GetCheckpointQActor>::operator new;
	using FastAllocated<GetCheckpointQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10207878807151111168UL, 16691871543586161408UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetCheckpointQActor, 0, Void >;
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetCheckpointQActor(StorageServer* const& self,GetCheckpointRequest const& req) 
															#line 6400 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   GetCheckpointQActorState<GetCheckpointQActor>(self, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getCheckpointQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7171777638048396032UL, 5824655659692947712UL);
		ActorExecutionContextHelper __helper(static_cast<GetCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getCheckpointQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getCheckpointQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetCheckpointQActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getCheckpointQ( StorageServer* const& self, GetCheckpointRequest const& req ) {
															#line 2629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetCheckpointQActor(self, req));
															#line 6433 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2668 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// Delete the checkpoint from disk, as well as all related persisted meta data.
															#line 6439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via deleteCheckpointQ()
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class DeleteCheckpointQActor>
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class DeleteCheckpointQActorState {
															#line 6446 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	DeleteCheckpointQActorState(StorageServer* const& self,Version const& version,CheckpointMetaData const& checkpoint) 
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   checkpoint(checkpoint)
															#line 6457 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("deleteCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	~DeleteCheckpointQActorState() 
	{
		fdb_probe_actor_destroy("deleteCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2671 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(0, TaskPriority::Low);
															#line 2671 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6474 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 1;
															#line 2671 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 0, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6479 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeleteCheckpointQActorState();
		static_cast<DeleteCheckpointQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2673 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->durableVersion.whenAtLeast(version);
															#line 2673 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6504 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 2;
															#line 2673 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 1, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6509 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2673 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->durableVersion.whenAtLeast(version);
															#line 2673 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6520 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 2;
															#line 2673 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 1, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6525 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteCheckpointQActor*>(this)->ActorCallback< DeleteCheckpointQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeleteCheckpointQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2675 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "DeleteCheckpointBegin", self->thisServerID).detail("Checkpoint", checkpoint.toString());
															#line 2677 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->checkpoints.erase(checkpoint.checkpointID);
															#line 6611 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2680 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = deleteCheckpoint(checkpoint);
															#line 2680 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1cont2Catch1(actor_cancelled(), loopDepth);
															#line 6617 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 3;
															#line 2680 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 2, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6622 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2675 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "DeleteCheckpointBegin", self->thisServerID).detail("Checkpoint", checkpoint.toString());
															#line 2677 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->checkpoints.erase(checkpoint.checkpointID);
															#line 6639 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2680 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_2 = deleteCheckpoint(checkpoint);
															#line 2680 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1cont2Catch1(actor_cancelled(), loopDepth);
															#line 6645 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont2Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 3;
															#line 2680 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeleteCheckpointQActor, 2, Void >*>(static_cast<DeleteCheckpointQActor*>(this)));
															#line 6650 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteCheckpointQActor*>(this)->ActorCallback< DeleteCheckpointQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeleteCheckpointQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2686 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		persistCheckpointKey = Key(persistCheckpointKeys.begin.toString() + checkpoint.checkpointID.toString());
															#line 2687 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		pendingCheckpointKey = Key(persistPendingCheckpointKeys.begin.toString() + checkpoint.checkpointID.toString());
															#line 2688 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto& mLV = self->addVersionToMutationLog(self->data().getLatestVersion());
															#line 2689 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->addMutationToMutationLog( mLV, MutationRef(MutationRef::ClearRange, pendingCheckpointKey, keyAfter(pendingCheckpointKey)));
															#line 2691 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		self->addMutationToMutationLog( mLV, MutationRef(MutationRef::ClearRange, persistCheckpointKey, keyAfter(persistCheckpointKey)));
															#line 2693 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "DeleteCheckpointEnd", self->thisServerID).detail("Checkpoint", checkpoint.toString());
															#line 2695 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<DeleteCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeleteCheckpointQActorState(); static_cast<DeleteCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6752 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<DeleteCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeleteCheckpointQActorState();
		static_cast<DeleteCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2683 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 6765 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<DeleteCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteCheckpointQActor*>(this)->ActorCallback< DeleteCheckpointQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DeleteCheckpointQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DeleteCheckpointQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont2Catch1(error, 0);
		} catch (...) {
			a_body1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	CheckpointMetaData checkpoint;
															#line 2686 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key persistCheckpointKey;
															#line 2687 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key pendingCheckpointKey;
															#line 6885 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via deleteCheckpointQ()
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class DeleteCheckpointQActor final : public Actor<Void>, public ActorCallback< DeleteCheckpointQActor, 0, Void >, public ActorCallback< DeleteCheckpointQActor, 1, Void >, public ActorCallback< DeleteCheckpointQActor, 2, Void >, public FastAllocated<DeleteCheckpointQActor>, public DeleteCheckpointQActorState<DeleteCheckpointQActor> {
															#line 6890 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<DeleteCheckpointQActor>::operator new;
	using FastAllocated<DeleteCheckpointQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10558666196178192640UL, 15625543762024560384UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeleteCheckpointQActor, 0, Void >;
friend struct ActorCallback< DeleteCheckpointQActor, 1, Void >;
friend struct ActorCallback< DeleteCheckpointQActor, 2, Void >;
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	DeleteCheckpointQActor(StorageServer* const& self,Version const& version,CheckpointMetaData const& checkpoint) 
															#line 6909 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   DeleteCheckpointQActorState<DeleteCheckpointQActor>(self, version, checkpoint),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7967604653483254528UL, 11896452891661470208UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deleteCheckpointQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deleteCheckpointQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeleteCheckpointQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeleteCheckpointQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DeleteCheckpointQActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> deleteCheckpointQ( StorageServer* const& self, Version const& version, CheckpointMetaData const& checkpoint ) {
															#line 2670 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new DeleteCheckpointQActor(self, version, checkpoint));
															#line 6944 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2697 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// Serves FetchCheckpointRequests.
															#line 6950 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via fetchCheckpointQ()
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class FetchCheckpointQActor>
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointQActorState {
															#line 6957 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointQActorState(StorageServer* const& self,FetchCheckpointRequest const& req) 
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 6966 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("fetchCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	~FetchCheckpointQActorState() 
	{
		fdb_probe_actor_destroy("fetchCheckpointQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2700 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent("ServeFetchCheckpointBegin", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("Token", req.token);
															#line 2704 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = nullptr;
															#line 2705 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			totalSize = 0;
															#line 2707 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.setByteLimit(SERVER_KNOBS->CHECKPOINT_TRANSFER_BLOCK_BYTES);
															#line 2710 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const auto it = self->checkpoints.find(req.checkpointID);
															#line 2711 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (it == self->checkpoints.end())
															#line 6991 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2712 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(checkpoint_not_found());
															#line 2713 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent("ServeFetchCheckpointNotFound", self->thisServerID).detail("CheckpointID", req.checkpointID);
															#line 2714 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<FetchCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointQActorState(); static_cast<FetchCheckpointQActor*>(this)->destroy(); return 0; }
															#line 6999 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<FetchCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~FetchCheckpointQActorState();
				static_cast<FetchCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 2718 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				reader = newCheckpointReader(it->second, CheckpointAsKeyValues::False, deterministicRandom()->randomUniqueID());
															#line 2719 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_0 = reader->init(req.token);
															#line 2719 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7012 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 1;
															#line 2719 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 0, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7017 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchCheckpointQActorState();
		static_cast<FetchCheckpointQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2753 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_4 = reader->close();
															#line 2753 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7048 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1when1(__when_expr_4.get(), loopDepth); };
		static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 5;
															#line 2753 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 4, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7053 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2730 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_end_of_stream || e.code() == error_code_checkpoint_not_found)
															#line 7063 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2731 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(e);
															#line 2732 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent("ServeFetchCheckpointEnd", self->thisServerID) .error(e) .detail("CheckpointID", req.checkpointID) .detail("TotalSize", totalSize) .detail("Token", req.token);
															#line 7069 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = a_body1Catch2cont1(loopDepth);
			}
			else
			{
															#line 2737 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (e.code() != error_code_operation_obsolete)
															#line 7076 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2738 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevWarnAlways, "ServerFetchCheckpointFailure") .errorUnsuppressed(e) .detail("CheckpointID", req.checkpointID) .detail("Token", req.token);
															#line 2742 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (canReplyWith(e))
															#line 7082 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 2743 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						req.reply.sendError(e);
															#line 7086 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
															#line 2745 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					err = e;
															#line 2746 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (reader != nullptr)
															#line 7092 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 2747 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						StrictFuture<Void> __when_expr_3 = reader->close();
															#line 2747 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7098 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1Catch2when1(__when_expr_3.get(), loopDepth); };
						static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 4;
															#line 2747 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 3, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7103 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
						loopDepth = a_body1Catch2cont4(loopDepth);
					}
				}
				else
				{
					loopDepth = a_body1Catch2cont3(loopDepth);
				}
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 2721 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 7129 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 2721 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 7138 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1(int loopDepth) 
	{
															#line 2722 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Standalone<StringRef>> __when_expr_1 = reader->nextChunk(CLIENT_KNOBS->REPLY_BYTE_LIMIT);
															#line 2722 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7231 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 2;
															#line 2722 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7236 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1(int loopDepth) 
	{
															#line 2723 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = req.reply.onReady();
															#line 2723 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7247 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 3;
															#line 2723 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointQActor, 2, Void >*>(static_cast<FetchCheckpointQActor*>(this)));
															#line 7252 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Standalone<StringRef> const& __data,int loopDepth) 
	{
															#line 2722 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data = __data;
															#line 7261 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1when1(Standalone<StringRef> && __data,int loopDepth) 
	{
		data = std::move(__data);
		loopDepth = a_body1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*,Standalone<StringRef> const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*,Standalone<StringRef> && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 2724 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointReply reply(req.token);
															#line 2725 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.data = data;
															#line 2726 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 2727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		totalSize += data.size();
															#line 7346 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 2724 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointReply reply(req.token);
															#line 2725 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.data = data;
															#line 2726 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 2727 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		totalSize += data.size();
															#line 7361 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1Catch2cont1(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2cont3(int loopDepth) 
	{
		loopDepth = a_body1Catch2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2cont4(int loopDepth) 
	{
															#line 2749 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		return a_body1Catch1(err, loopDepth);
															#line 7457 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"

		return loopDepth;
	}
	int a_body1Catch2cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont4(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont4(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1Catch2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1Catch2cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 2754 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<FetchCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointQActorState(); static_cast<FetchCheckpointQActor*>(this)->destroy(); return 0; }
															#line 7552 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<FetchCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchCheckpointQActorState();
		static_cast<FetchCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 2754 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<FetchCheckpointQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointQActorState(); static_cast<FetchCheckpointQActor*>(this)->destroy(); return 0; }
															#line 7564 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<FetchCheckpointQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchCheckpointQActorState();
		static_cast<FetchCheckpointQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FetchCheckpointQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointQActor*>(this)->ActorCallback< FetchCheckpointQActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FetchCheckpointQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointRequest req;
															#line 2704 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ICheckpointReader* reader;
															#line 2705 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t totalSize;
															#line 2722 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Standalone<StringRef> data;
															#line 2745 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Error err;
															#line 7659 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via fetchCheckpointQ()
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointQActor final : public Actor<Void>, public ActorCallback< FetchCheckpointQActor, 0, Void >, public ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >, public ActorCallback< FetchCheckpointQActor, 2, Void >, public ActorCallback< FetchCheckpointQActor, 3, Void >, public ActorCallback< FetchCheckpointQActor, 4, Void >, public FastAllocated<FetchCheckpointQActor>, public FetchCheckpointQActorState<FetchCheckpointQActor> {
															#line 7664 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<FetchCheckpointQActor>::operator new;
	using FastAllocated<FetchCheckpointQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6232357106229481984UL, 3568680178005981440UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchCheckpointQActor, 0, Void >;
friend struct ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >;
friend struct ActorCallback< FetchCheckpointQActor, 2, Void >;
friend struct ActorCallback< FetchCheckpointQActor, 3, Void >;
friend struct ActorCallback< FetchCheckpointQActor, 4, Void >;
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointQActor(StorageServer* const& self,FetchCheckpointRequest const& req) 
															#line 7685 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   FetchCheckpointQActorState<FetchCheckpointQActor>(self, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13937557724080409344UL, 5112112820403248640UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchCheckpointQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchCheckpointQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 1, Standalone<StringRef> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FetchCheckpointQActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> fetchCheckpointQ( StorageServer* const& self, FetchCheckpointRequest const& req ) {
															#line 2699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new FetchCheckpointQActor(self, req));
															#line 7722 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2756 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// Serves FetchCheckpointKeyValuesRequest, reads local checkpoint and sends it to the client over wire.
															#line 7728 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via fetchCheckpointKeyValuesQ()
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class FetchCheckpointKeyValuesQActor>
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointKeyValuesQActorState {
															#line 7735 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointKeyValuesQActorState(StorageServer* const& self,FetchCheckpointKeyValuesRequest const& req) 
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : self(self),
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 7744 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	~FetchCheckpointKeyValuesQActorState() 
	{
		fdb_probe_actor_destroy("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2759 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->serveFetchCheckpointParallelismLock.take(TaskPriority::DefaultYield);
															#line 2759 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7761 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 1;
															#line 2759 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 7766 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchCheckpointKeyValuesQActorState();
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2760 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(self->serveFetchCheckpointParallelismLock);
															#line 2762 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent("ServeFetchCheckpointKeyValuesBegin", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 2766 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.setByteLimit(SERVER_KNOBS->CHECKPOINT_TRANSFER_BLOCK_BYTES);
															#line 2769 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const auto it = self->checkpoints.find(req.checkpointID);
															#line 2770 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (it == self->checkpoints.end())
															#line 7797 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2771 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(checkpoint_not_found());
															#line 2772 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent("ServeFetchCheckpointNotFound", self->thisServerID).detail("CheckpointID", req.checkpointID);
															#line 2773 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointKeyValuesQActorState(); static_cast<FetchCheckpointKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 7805 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FetchCheckpointKeyValuesQActorState();
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2776 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reader = nullptr;
															#line 2777 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto crIt = self->liveCheckpointReaders.find(req.checkpointID);
															#line 2778 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (crIt != self->liveCheckpointReaders.end())
															#line 7817 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2779 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = crIt->second;
															#line 7821 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 2781 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = newCheckpointReader(it->second, CheckpointAsKeyValues::True, deterministicRandom()->randomUniqueID());
															#line 2782 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			self->liveCheckpointReaders[req.checkpointID] = reader;
															#line 7829 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2785 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = std::unique_ptr<ICheckpointIterator>();
															#line 7833 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2787 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = reader->init(BinaryWriter::toValue(req.range, IncludeVersion()));
															#line 2787 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7839 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 2787 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 7844 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2760 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(self->serveFetchCheckpointParallelismLock);
															#line 2762 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent("ServeFetchCheckpointKeyValuesBegin", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 2766 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.setByteLimit(SERVER_KNOBS->CHECKPOINT_TRANSFER_BLOCK_BYTES);
															#line 2769 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const auto it = self->checkpoints.find(req.checkpointID);
															#line 2770 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (it == self->checkpoints.end())
															#line 7867 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2771 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(checkpoint_not_found());
															#line 2772 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent("ServeFetchCheckpointNotFound", self->thisServerID).detail("CheckpointID", req.checkpointID);
															#line 2773 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointKeyValuesQActorState(); static_cast<FetchCheckpointKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 7875 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FetchCheckpointKeyValuesQActorState();
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2776 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reader = nullptr;
															#line 2777 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		auto crIt = self->liveCheckpointReaders.find(req.checkpointID);
															#line 2778 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (crIt != self->liveCheckpointReaders.end())
															#line 7887 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2779 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = crIt->second;
															#line 7891 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 2781 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reader = newCheckpointReader(it->second, CheckpointAsKeyValues::True, deterministicRandom()->randomUniqueID());
															#line 2782 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			self->liveCheckpointReaders[req.checkpointID] = reader;
															#line 7899 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2785 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = std::unique_ptr<ICheckpointIterator>();
															#line 7903 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 2787 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = reader->init(BinaryWriter::toValue(req.range, IncludeVersion()));
															#line 2787 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 7909 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 2787 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 7914 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 2831 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		iter.reset();
															#line 2832 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reader->inUse())
															#line 8006 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2833 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			self->liveCheckpointReaders.erase(req.checkpointID);
															#line 2834 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_4 = reader->close();
															#line 2834 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8014 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont2when1(__when_expr_4.get(), loopDepth); };
			static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 5;
															#line 2834 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 8019 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2814 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_end_of_stream || e.code() == error_code_checkpoint_not_found)
															#line 8034 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2815 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(e);
															#line 2816 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevInfo, "ServeFetchCheckpointKeyValuesEnd", self->thisServerID) .error(e) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 8040 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 2821 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarnAlways, "ServerFetchCheckpointKeyValuesFailure") .errorUnsuppressed(e) .detail("CheckpointID", req.checkpointID) .detail("Range", req.range);
															#line 2825 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (canReplyWith(e))
															#line 8048 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2826 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(e);
															#line 8052 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 2788 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = reader->getIterator(req.range);
															#line 2790 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 8071 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont6loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 2788 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		iter = reader->getIterator(req.range);
															#line 2790 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 8082 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont6loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont6loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1(int loopDepth) 
	{
															#line 2791 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<RangeResult> __when_expr_2 = iter->nextBatch(CLIENT_KNOBS->REPLY_BYTE_LIMIT, CLIENT_KNOBS->REPLY_BYTE_LIMIT);
															#line 2791 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8175 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 3;
															#line 2791 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 8180 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1(int loopDepth) 
	{
															#line 2793 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!res.empty())
															#line 8189 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2794 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "FetchCheckpontKeyValuesReadRange", self->thisServerID) .detail("CheckpointID", req.checkpointID) .detail("FirstReturnedKey", res.front().key) .detail("LastReturnedKey", res.back().key) .detail("Size", res.size());
															#line 8193 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 2800 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "FetchCheckpontKeyValuesEmptyRange", self->thisServerID) .detail("CheckpointID", req.checkpointID);
															#line 8199 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2804 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = req.reply.onReady();
															#line 2804 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8205 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont6loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 4;
															#line 2804 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*>(static_cast<FetchCheckpointKeyValuesQActor*>(this)));
															#line 8210 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6loopBody1when1(RangeResult const& __res,int loopDepth) 
	{
															#line 2791 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		res = __res;
															#line 8219 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont6loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1when1(RangeResult && __res,int loopDepth) 
	{
		res = std::move(__res);
		loopDepth = a_body1cont6loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 2805 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointKeyValuesStreamReply reply;
															#line 2806 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.arena.dependsOn(res.arena());
															#line 2807 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 2808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reply.data.push_back(reply.arena, res[i]);
															#line 8304 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2811 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 8308 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 2805 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		FetchCheckpointKeyValuesStreamReply reply;
															#line 2806 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		reply.arena.dependsOn(res.arena());
															#line 2807 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < res.size();++i) {
															#line 2808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			reply.data.push_back(reply.arena, res[i]);
															#line 8323 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 2811 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(reply);
															#line 8327 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont8(int loopDepth) 
	{
															#line 2836 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchCheckpointKeyValuesQActorState(); static_cast<FetchCheckpointKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 8411 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<FetchCheckpointKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchCheckpointKeyValuesQActorState();
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<FetchCheckpointKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<FetchCheckpointKeyValuesQActor*>(this)->ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* self;
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointKeyValuesRequest req;
															#line 2760 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FlowLock::Releaser holder;
															#line 2776 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ICheckpointReader* reader;
															#line 2785 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::unique_ptr<ICheckpointIterator> iter;
															#line 2791 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	RangeResult res;
															#line 8518 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via fetchCheckpointKeyValuesQ()
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FetchCheckpointKeyValuesQActor final : public Actor<Void>, public ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >, public ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >, public ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >, public ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >, public ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >, public FastAllocated<FetchCheckpointKeyValuesQActor>, public FetchCheckpointKeyValuesQActorState<FetchCheckpointKeyValuesQActor> {
															#line 8523 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<FetchCheckpointKeyValuesQActor>::operator new;
	using FastAllocated<FetchCheckpointKeyValuesQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10090861299242754304UL, 6859997095327505920UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >;
friend struct ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >;
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FetchCheckpointKeyValuesQActor(StorageServer* const& self,FetchCheckpointKeyValuesRequest const& req) 
															#line 8544 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   FetchCheckpointKeyValuesQActorState<FetchCheckpointKeyValuesQActor>(self, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12412750992388461056UL, 6903825642391152640UL);
		ActorExecutionContextHelper __helper(static_cast<FetchCheckpointKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchCheckpointKeyValuesQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchCheckpointKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 2, RangeResult >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FetchCheckpointKeyValuesQActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> fetchCheckpointKeyValuesQ( StorageServer* const& self, FetchCheckpointKeyValuesRequest const& req ) {
															#line 2758 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new FetchCheckpointKeyValuesQActor(self, req));
															#line 8581 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2838 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

MutationsAndVersionRef filterMutations(Arena& arena,
                                       EncryptedMutationsAndVersionRef const& m,
                                       KeyRange const& range,
                                       bool encrypted,
                                       int commonPrefixLength) {
	if (m.mutations.size() == 1 && m.mutations.back().param1 == lastEpochEndPrivateKey) {
		return MutationsAndVersionRef(m.mutations, m.version, m.knownCommittedVersion);
	}

	Optional<VectorRef<MutationRef>> modifiedMutations;
	for (int i = 0; i < m.mutations.size(); i++) {
		if (m.mutations[i].type == MutationRef::SetValue) {
			bool inRange = range.begin.compareSuffix(m.mutations[i].param1, commonPrefixLength) <= 0 &&
			               m.mutations[i].param1.compareSuffix(range.end, commonPrefixLength) < 0;
			if (modifiedMutations.present() && inRange) {
				modifiedMutations.get().push_back(
				    arena, encrypted && m.encrypted.present() ? m.encrypted.get()[i] : m.mutations[i]);
			}
			if (!modifiedMutations.present() && !inRange) {
				if (encrypted && m.encrypted.present()) {
					modifiedMutations = m.encrypted.get().slice(0, i);
				} else {
					modifiedMutations = m.mutations.slice(0, i);
				}
				arena.dependsOn(range.arena());
			}
		} else {
			ASSERT(m.mutations[i].type == MutationRef::ClearRange);
			// param1 < range.begin || param2 > range.end
			if (!modifiedMutations.present() &&
			    (m.mutations[i].param1.compareSuffix(range.begin, commonPrefixLength) < 0 ||
			     m.mutations[i].param2.compareSuffix(range.end, commonPrefixLength) > 0)) {
				if (encrypted && m.encrypted.present()) {
					modifiedMutations = m.encrypted.get().slice(0, i);
				} else {
					modifiedMutations = m.mutations.slice(0, i);
				}
				arena.dependsOn(range.arena());
			}
			if (modifiedMutations.present()) {
				// param1 < range.end && range.begin < param2
				if (m.mutations[i].param1.compareSuffix(range.end, commonPrefixLength) < 0 &&
				    range.begin.compareSuffix(m.mutations[i].param2, commonPrefixLength) < 0) {
					StringRef clearBegin = m.mutations[i].param1;
					StringRef clearEnd = m.mutations[i].param2;
					bool modified = false;
					if (clearBegin.compareSuffix(range.begin, commonPrefixLength) < 0) {
						clearBegin = range.begin;
						modified = true;
					}
					if (range.end.compareSuffix(clearEnd, commonPrefixLength) < 0) {
						clearEnd = range.end;
						modified = true;
					}
					if (modified) {
						MutationRef clearMutation = MutationRef(MutationRef::ClearRange, clearBegin, clearEnd);
						if (encrypted && m.encrypted.present() && m.encrypted.get()[i].isEncrypted()) {
							clearMutation = clearMutation.encrypt(m.cipherKeys[i], arena, BlobCipherMetrics::TLOG);
						}
						modifiedMutations.get().push_back(arena, clearMutation);
					} else {
						modifiedMutations.get().push_back(
						    arena, encrypted && m.encrypted.present() ? m.encrypted.get()[i] : m.mutations[i]);
					}
				}
			}
		}
	}
	if (modifiedMutations.present()) {
		return MutationsAndVersionRef(modifiedMutations.get(), m.version, m.knownCommittedVersion);
	}
	if (!encrypted || !m.encrypted.present()) {
		return MutationsAndVersionRef(m.mutations, m.version, m.knownCommittedVersion);
	}
	return MutationsAndVersionRef(m.encrypted.get(), m.version, m.knownCommittedVersion);
}

#ifdef NO_INTELLISENSE
size_t WATCH_OVERHEAD_WATCHQ =
    sizeof(WatchValueSendReplyActorState<WatchValueSendReplyActor>) + sizeof(WatchValueSendReplyActor);
size_t WATCH_OVERHEAD_WATCHIMPL =
    sizeof(WatchWaitForValueChangeActorState<WatchWaitForValueChangeActor>) + sizeof(WatchWaitForValueChangeActor);
#else
size_t WATCH_OVERHEAD_WATCHQ = 0; // only used in IDE so value is irrelevant
size_t WATCH_OVERHEAD_WATCHIMPL = 0;
#endif

															#line 8673 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getShardState_impl()
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetShardState_implActor>
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardState_implActorState {
															#line 8680 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardState_implActorState(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 8689 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("getShardState_impl", reinterpret_cast<unsigned long>(this));

	}
	~GetShardState_implActorState() 
	{
		fdb_probe_actor_destroy("getShardState_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2927 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(req.mode != GetShardStateRequest::NO_WAIT);
															#line 2929 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 8706 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetShardState_implActorState();
		static_cast<GetShardState_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2930 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		std::vector<Future<Void>> onChange;
															#line 2932 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto t : data->shards.intersectingRanges(req.keys) ) {
															#line 2933 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!t.value()->assigned())
															#line 8740 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2934 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				onChange.push_back(delay(SERVER_KNOBS->SHARD_READY_DELAY));
															#line 8744 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				break;
			}
															#line 2938 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.mode == GetShardStateRequest::READABLE && !t.value()->isReadable())
															#line 8749 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2939 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (t.value()->getAddingShard())
															#line 8753 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2940 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getAddingShard()->readWrite.getFuture());
															#line 8757 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 2942 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(t.value()->getMoveInShard());
															#line 2943 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getMoveInShard()->readWrite.getFuture());
															#line 8765 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 2947 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.mode == GetShardStateRequest::FETCHING && !t.value()->isFetched())
															#line 8770 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2948 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (t.value()->getAddingShard())
															#line 8774 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 2949 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getAddingShard()->fetchComplete.getFuture());
															#line 8778 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 2951 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(t.value()->getMoveInShard());
															#line 2952 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					onChange.push_back(t.value()->getMoveInShard()->fetchComplete.getFuture());
															#line 8786 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
		}
															#line 2957 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!onChange.size())
															#line 8792 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 2958 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			GetShardStateReply rep(data->version.get(), data->durableVersion.get());
															#line 2959 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.includePhysicalShard)
															#line 8798 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 2960 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				rep.shards = data->getStorageServerShards(req.keys);
															#line 8802 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 2962 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(rep);
															#line 2963 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<GetShardState_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetShardState_implActorState(); static_cast<GetShardState_implActor*>(this)->destroy(); return 0; }
															#line 8808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<GetShardState_implActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~GetShardState_implActorState();
			static_cast<GetShardState_implActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2966 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = waitForAll(onChange);
															#line 2966 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8818 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<GetShardState_implActor*>(this)->actor_wait_state = 1;
															#line 2966 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetShardState_implActor, 0, Void >*>(static_cast<GetShardState_implActor*>(this)));
															#line 8823 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 2967 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0);
															#line 2967 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8834 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetShardState_implActor*>(this)->actor_wait_state = 2;
															#line 2967 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetShardState_implActor, 1, Void >*>(static_cast<GetShardState_implActor*>(this)));
															#line 8839 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 2967 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(0);
															#line 2967 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8850 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetShardState_implActor*>(this)->actor_wait_state = 2;
															#line 2967 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetShardState_implActor, 1, Void >*>(static_cast<GetShardState_implActor*>(this)));
															#line 8855 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state > 0) static_cast<GetShardState_implActor*>(this)->actor_wait_state = 0;
		static_cast<GetShardState_implActor*>(this)->ActorCallback< GetShardState_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetShardState_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont12(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont12(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetShardState_implActor*>(this)->actor_wait_state > 0) static_cast<GetShardState_implActor*>(this)->actor_wait_state = 0;
		static_cast<GetShardState_implActor*>(this)->ActorCallback< GetShardState_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetShardState_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetShardState_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateRequest req;
															#line 9026 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via getShardState_impl()
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardState_implActor final : public Actor<Void>, public ActorCallback< GetShardState_implActor, 0, Void >, public ActorCallback< GetShardState_implActor, 1, Void >, public FastAllocated<GetShardState_implActor>, public GetShardState_implActorState<GetShardState_implActor> {
															#line 9031 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<GetShardState_implActor>::operator new;
	using FastAllocated<GetShardState_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8700970531701272064UL, 11221400556161005824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetShardState_implActor, 0, Void >;
friend struct ActorCallback< GetShardState_implActor, 1, Void >;
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardState_implActor(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 9049 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   GetShardState_implActorState<GetShardState_implActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getShardState_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3570919601400010496UL, 14753609829273848064UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardState_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getShardState_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getShardState_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetShardState_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetShardState_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getShardState_impl( StorageServer* const& data, GetShardStateRequest const& req ) {
															#line 2926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetShardState_implActor(data, req));
															#line 9083 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2970 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 9088 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getShardStateQ()
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetShardStateQActor>
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardStateQActorState {
															#line 9095 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateQActorState(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 9104 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("getShardStateQ", reinterpret_cast<unsigned long>(this));

	}
	~GetShardStateQActorState() 
	{
		fdb_probe_actor_destroy("getShardStateQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2973 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = getShardState_impl(data, req);
															#line 2972 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetShardStateQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9121 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 2974 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(g_network->isSimulated() ? 10 : 60);
															#line 9125 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<GetShardStateQActor*>(this)->actor_wait_state = 1;
															#line 2973 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetShardStateQActor, 0, Void >*>(static_cast<GetShardStateQActor*>(this)));
															#line 2974 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetShardStateQActor, 1, Void >*>(static_cast<GetShardStateQActor*>(this)));
															#line 9132 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetShardStateQActorState();
		static_cast<GetShardStateQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2978 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetShardStateQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetShardStateQActorState(); static_cast<GetShardStateQActor*>(this)->destroy(); return 0; }
															#line 9155 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<GetShardStateQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetShardStateQActorState();
		static_cast<GetShardStateQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 2975 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 9179 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 2975 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->sendErrorWithPenalty(req.reply, timed_out(), data->getPenalty());
															#line 9188 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetShardStateQActor*>(this)->actor_wait_state > 0) static_cast<GetShardStateQActor*>(this)->actor_wait_state = 0;
		static_cast<GetShardStateQActor*>(this)->ActorCallback< GetShardStateQActor, 0, Void >::remove();
		static_cast<GetShardStateQActor*>(this)->ActorCallback< GetShardStateQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetShardStateQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetShardStateQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetShardStateQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateRequest req;
															#line 9318 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via getShardStateQ()
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetShardStateQActor final : public Actor<Void>, public ActorCallback< GetShardStateQActor, 0, Void >, public ActorCallback< GetShardStateQActor, 1, Void >, public FastAllocated<GetShardStateQActor>, public GetShardStateQActorState<GetShardStateQActor> {
															#line 9323 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<GetShardStateQActor>::operator new;
	using FastAllocated<GetShardStateQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12961947696458916608UL, 11632320440331639552UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetShardStateQActor, 0, Void >;
friend struct ActorCallback< GetShardStateQActor, 1, Void >;
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetShardStateQActor(StorageServer* const& data,GetShardStateRequest const& req) 
															#line 9341 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   GetShardStateQActorState<GetShardStateQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getShardStateQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8443565892279057664UL, 11087380315085298688UL);
		ActorExecutionContextHelper __helper(static_cast<GetShardStateQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getShardStateQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getShardStateQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetShardStateQActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getShardStateQ( StorageServer* const& data, GetShardStateRequest const& req ) {
															#line 2971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetShardStateQActor(data, req));
															#line 9374 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 2980 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

KeyRef addPrefix(KeyRef const& key, Optional<KeyRef> prefix, Arena& arena) {
	if (prefix.present()) {
		return key.withPrefix(prefix.get(), arena);
	} else {
		return key;
	}
}

KeyValueRef removePrefix(KeyValueRef const& src, Optional<KeyRef> prefix) {
	if (prefix.present()) {
		return KeyValueRef(src.key.removePrefix(prefix.get()), src.value);
	} else {
		return src;
	}
}

void merge(Arena& arena,
           VectorRef<KeyValueRef, VecSerStrategy::String>& output,
           VectorRef<KeyValueRef> const& vm_output,
           RangeResult const& base,
           int& vCount,
           int limit,
           bool stopAtEndOfBase,
           int& pos,
           int limitBytes,
           Optional<KeyRef> tenantPrefix)
// Combines data from base (at an older version) with sets from newer versions in [start, end) and appends the first (up
// to) |limit| rows to output If limit<0, base and output are in descending order, and start->key()>end->key(), but
// start is still inclusive and end is exclusive
{
	ASSERT(limit != 0);
	// Add a dependency of the new arena on the result from the KVS so that we don't have to copy any of the KVS
	// results.
	arena.dependsOn(base.arena());

	bool forward = limit > 0;
	if (!forward)
		limit = -limit;
	int adjustedLimit = limit + output.size();
	int accumulatedBytes = 0;
	KeyValueRef const* baseStart = base.begin();
	KeyValueRef const* baseEnd = base.end();
	while (baseStart != baseEnd && vCount > 0 && output.size() < adjustedLimit && accumulatedBytes < limitBytes) {
		if (forward ? baseStart->key < vm_output[pos].key : baseStart->key > vm_output[pos].key) {
			output.push_back(arena, removePrefix(*baseStart++, tenantPrefix));
		} else {
			output.push_back_deep(arena, removePrefix(vm_output[pos], tenantPrefix));
			if (baseStart->key == vm_output[pos].key)
				++baseStart;
			++pos;
			vCount--;
		}
		accumulatedBytes += sizeof(KeyValueRef) + output.end()[-1].expectedSize();
	}
	while (baseStart != baseEnd && output.size() < adjustedLimit && accumulatedBytes < limitBytes) {
		output.push_back(arena, removePrefix(*baseStart++, tenantPrefix));
		accumulatedBytes += sizeof(KeyValueRef) + output.end()[-1].expectedSize();
	}
	if (!stopAtEndOfBase) {
		while (vCount > 0 && output.size() < adjustedLimit && accumulatedBytes < limitBytes) {
			output.push_back_deep(arena, removePrefix(vm_output[pos], tenantPrefix));
			accumulatedBytes += sizeof(KeyValueRef) + output.end()[-1].expectedSize();
			++pos;
			vCount--;
		}
	}
}

static inline void copyOptionalValue(Arena* a,
                                     GetValueReqAndResultRef& getValue,
                                     const Optional<Value>& optionalValue) {
	getValue.result = optionalValue.castTo<ValueRef>();
	if (optionalValue.present()) {
		a->dependsOn(optionalValue.get().arena());
	}
}
															#line 9455 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via quickGetValue()
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class QuickGetValueActor>
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetValueActorState {
															#line 9462 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetValueActorState(StorageServer* const& data,StringRef const& key,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   key(key),
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   a(a),
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   pOriginalReq(pOriginalReq),
															#line 3063 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   getValue(),
															#line 3064 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   getValueStart(g_network->timer())
															#line 9481 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("quickGetValue", reinterpret_cast<unsigned long>(this));

	}
	~QuickGetValueActorState() 
	{
		fdb_probe_actor_destroy("quickGetValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3065 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getValue.key = key;
															#line 3067 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (data->shards[key]->isReadable())
															#line 9498 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
				try {
															#line 3070 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					GetValueRequest req(pOriginalReq->spanContext, pOriginalReq->tenantInfo, key, version, pOriginalReq->tags, pOriginalReq->options, VersionVector());
															#line 3080 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					data->actors.add(getValueQ(data, req));
															#line 3081 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					StrictFuture<GetValueReply> __when_expr_0 = req.reply.getFuture();
															#line 3081 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (static_cast<QuickGetValueActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9509 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<QuickGetValueActor*>(this)->actor_wait_state = 1;
															#line 3081 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< QuickGetValueActor, 0, GetValueReply >*>(static_cast<QuickGetValueActor*>(this)));
															#line 9514 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					loopDepth = 0;
				}
				catch (Error& error) {
					loopDepth = a_body1Catch2(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1Catch2(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~QuickGetValueActorState();
		static_cast<QuickGetValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3096 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.quickGetValueMiss;
															#line 3097 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (SERVER_KNOBS->QUICK_GET_VALUE_FALLBACK)
															#line 9550 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3098 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Optional<Reference<Tenant>> tenant = pOriginalReq->tenantInfo.hasTenant() ? makeReference<Tenant>(pOriginalReq->tenantInfo.tenantId) : Optional<Reference<Tenant>>();
															#line 3101 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr = Transaction(data->cx, tenant);
															#line 3102 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setVersion(version);
															#line 3104 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.trState->taskID = TaskPriority::DefaultPromiseEndpoint;
															#line 3105 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Future<Optional<Value>> valueFuture = tr.get(key, Snapshot::True);
															#line 3107 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_1 = valueFuture;
															#line 3107 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<QuickGetValueActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9566 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<QuickGetValueActor*>(this)->actor_wait_state = 2;
															#line 3107 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< QuickGetValueActor, 1, Optional<Value> >*>(static_cast<QuickGetValueActor*>(this)));
															#line 9571 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 3113 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(quick_get_value_miss(), loopDepth);
															#line 9578 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(GetValueReply const& reply,int loopDepth) 
	{
															#line 3082 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 9606 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3083 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetValueHit;
															#line 3084 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			copyOptionalValue(a, getValue, reply.value);
															#line 3085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValueStart;
															#line 3086 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.mappedRangeLocalSample->addMeasurement(duration);
															#line 3087 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 9618 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
			this->~QuickGetValueActorState();
			static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(GetValueReply && reply,int loopDepth) 
	{
															#line 3082 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 9632 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3083 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetValueHit;
															#line 3084 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			copyOptionalValue(a, getValue, reply.value);
															#line 3085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValueStart;
															#line 3086 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.mappedRangeLocalSample->addMeasurement(duration);
															#line 3087 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 9644 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
			this->~QuickGetValueActorState();
			static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetValueReply const& reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(reply, loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetValueReply && reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<QuickGetValueActor*>(this)->actor_wait_state > 0) static_cast<QuickGetValueActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetValueActor*>(this)->ActorCallback< QuickGetValueActor, 0, GetValueReply >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 0, GetValueReply >*,GetValueReply const& value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 0, GetValueReply >*,GetValueReply && value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< QuickGetValueActor, 0, GetValueReply >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> const& valueOption,int loopDepth) 
	{
															#line 3108 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		copyOptionalValue(a, getValue, valueOption);
															#line 3109 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - getValueStart;
															#line 3110 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.mappedRangeRemoteSample->addMeasurement(duration);
															#line 3111 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 9752 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
		this->~QuickGetValueActorState();
		static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> && valueOption,int loopDepth) 
	{
															#line 3108 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		copyOptionalValue(a, getValue, valueOption);
															#line 3109 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - getValueStart;
															#line 3110 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.mappedRangeRemoteSample->addMeasurement(duration);
															#line 3111 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetValueActor*>(this)->SAV<GetValueReqAndResultRef>::futures) { (void)(getValue); this->~QuickGetValueActorState(); static_cast<QuickGetValueActor*>(this)->destroy(); return 0; }
															#line 9770 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<QuickGetValueActor*>(this)->SAV< GetValueReqAndResultRef >::value()) GetValueReqAndResultRef(std::move(getValue)); // state_var_RVO
		this->~QuickGetValueActorState();
		static_cast<QuickGetValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Value> const& valueOption,int loopDepth) 
	{
		loopDepth = a_body1cont8(valueOption, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Value> && valueOption,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(valueOption), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<QuickGetValueActor*>(this)->actor_wait_state > 0) static_cast<QuickGetValueActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetValueActor*>(this)->ActorCallback< QuickGetValueActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< QuickGetValueActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< QuickGetValueActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StringRef key;
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Arena* a;
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetMappedKeyValuesRequest* pOriginalReq;
															#line 3063 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetValueReqAndResultRef getValue;
															#line 3064 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double getValueStart;
															#line 3101 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Transaction tr;
															#line 9869 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via quickGetValue()
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetValueActor final : public Actor<GetValueReqAndResultRef>, public ActorCallback< QuickGetValueActor, 0, GetValueReply >, public ActorCallback< QuickGetValueActor, 1, Optional<Value> >, public FastAllocated<QuickGetValueActor>, public QuickGetValueActorState<QuickGetValueActor> {
															#line 9874 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<QuickGetValueActor>::operator new;
	using FastAllocated<QuickGetValueActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13370224133406375680UL, 6936956499149022720UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GetValueReqAndResultRef>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< QuickGetValueActor, 0, GetValueReply >;
friend struct ActorCallback< QuickGetValueActor, 1, Optional<Value> >;
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetValueActor(StorageServer* const& data,StringRef const& key,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 9892 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<GetValueReqAndResultRef>(),
		   QuickGetValueActorState<QuickGetValueActor>(data, key, version, a, pOriginalReq),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("quickGetValue", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13229342185383356416UL, 6029856400378886912UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("quickGetValue");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("quickGetValue", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< QuickGetValueActor, 0, GetValueReply >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< QuickGetValueActor, 1, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<GetValueReqAndResultRef> quickGetValue( StorageServer* const& data, StringRef const& key, Version const& version, Arena* const& a, GetMappedKeyValuesRequest* const& pOriginalReq ) {
															#line 3057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<GetValueReqAndResultRef>(new QuickGetValueActor(data, key, version, a, pOriginalReq));
															#line 9926 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 3116 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

// If limit>=0, it returns the first rows in the range (sorted ascending), otherwise the last rows (sorted descending).
// readRange has O(|result|) + O(log |data|) cost
															#line 9933 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via readRange()
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class ReadRangeActor>
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadRangeActorState {
															#line 9940 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadRangeActorState(StorageServer* const& data,Version const& version,KeyRange const& range,int const& limit,int* const& pLimitBytes,SpanContext const& parentSpan,Optional<ReadOptions> const& options,Optional<KeyRef> const& tenantPrefix) 
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   range(range),
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   limit(limit),
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   pLimitBytes(pLimitBytes),
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   parentSpan(parentSpan),
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   options(options),
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   tenantPrefix(tenantPrefix),
															#line 3127 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   result(),
															#line 3128 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   view(data->data().at(version)),
															#line 3129 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   vCurrent(view.end()),
															#line 3130 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   readBegin(),
															#line 3131 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   readEnd(),
															#line 3132 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   readBeginTemp(),
															#line 3133 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   vCount(0),
															#line 3134 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:readRange"_loc, parentSpan),
															#line 3135 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultLogicalSize(0),
															#line 3136 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   logicalSize(0),
															#line 3139 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultCache(),
															#line 3142 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   pos(0)
															#line 9985 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("readRange", reinterpret_cast<unsigned long>(this));

	}
	~ReadRangeActorState() 
	{
		fdb_probe_actor_destroy("readRange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3145 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			auto containingRange = data->cachedRangeMap.rangeContaining(range.begin);
															#line 3146 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (containingRange.value() && containingRange->range().end >= range.end)
															#line 10002 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3149 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				result.cached = true;
															#line 10006 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 3151 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				result.cached = false;
															#line 10012 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3154 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (limit >= 0)
															#line 10016 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3157 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				vCurrent = view.lastLessOrEqual(range.begin);
															#line 3158 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (vCurrent && vCurrent->isClearTo() && vCurrent->getEndKey() > range.begin)
															#line 10022 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3159 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					readBegin = vCurrent->getEndKey();
															#line 10026 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 3161 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					readBegin = range.begin;
															#line 10032 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 3164 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (vCurrent)
															#line 10036 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3165 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (vCurrent.key() != readBegin)
															#line 10040 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 3166 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						++vCurrent;
															#line 10044 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
				else
				{
															#line 3172 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					vCurrent = view.begin();
															#line 10051 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 3174 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (EXPENSIVE_VALIDATION)
															#line 10055 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3175 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(vCurrent == view.lower_bound(readBegin));
															#line 10059 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 3178 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				;
															#line 10063 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
															#line 3270 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				vCurrent = view.lastLess(range.end);
															#line 3273 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (vCurrent && vCurrent->isClearTo() && vCurrent->getEndKey() >= range.end)
															#line 10072 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3274 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					readEnd = vCurrent.key();
															#line 3275 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					--vCurrent;
															#line 10078 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 3277 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					readEnd = range.end;
															#line 10084 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 3280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				;
															#line 10088 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = a_body1loopHead2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadRangeActorState();
		static_cast<ReadRangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3362 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesReturnedHistogram->sample(resultLogicalSize);
															#line 3363 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeKVPairsReturnedHistogram->sample(result.data.size());
															#line 3366 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(result.data.size() == 0 || *pLimitBytes + result.data.end()[-1].expectedSize() + sizeof(KeyValueRef) > 0);
															#line 3367 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		result.more = limit == 0 || *pLimitBytes <= 0;
															#line 3368 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		result.version = version;
															#line 3369 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<ReadRangeActor*>(this)->SAV<GetKeyValuesReply>::futures) { (void)(result); this->~ReadRangeActorState(); static_cast<ReadRangeActor*>(this)->destroy(); return 0; }
															#line 10122 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<ReadRangeActor*>(this)->SAV< GetKeyValuesReply >::value()) GetKeyValuesReply(std::move(result)); // state_var_RVO
		this->~ReadRangeActorState();
		static_cast<ReadRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 3178 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!(limit > 0 && *pLimitBytes > 0 && readBegin < range.end))
															#line 10147 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3179 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(!vCurrent || vCurrent.key() >= readBegin);
															#line 3180 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(data->storageVersion() <= version);
															#line 3183 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pos == resultCache.size())
															#line 10157 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3184 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent)
															#line 10161 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3185 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				auto b = vCurrent;
															#line 3186 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				--b;
															#line 3187 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(!b || b.key() < readBegin);
															#line 10169 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3191 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int vSize = 0;
															#line 3192 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;vCurrent && vCurrent.key() < range.end && !vCurrent->isClearTo() && vCount < limit && vSize < *pLimitBytes;) {
															#line 3195 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				resultCache.emplace_back(result.arena, vCurrent.key(), vCurrent->getValue());
															#line 3196 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				vSize += sizeof(KeyValueRef) + resultCache.cback().expectedSize() - (tenantPrefix.present() ? tenantPrefix.get().size() : 0);
															#line 3198 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCount;
															#line 3199 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCurrent;
															#line 10183 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 3204 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readEnd = vCurrent ? std::min(vCurrent.key(), range.end) : range.end;
															#line 3205 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<RangeResult> __when_expr_0 = data->storage.readRange(KeyRangeRef(readBegin, readEnd), limit, *pLimitBytes, options);
															#line 3205 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10192 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ReadRangeActor*>(this)->actor_wait_state = 1;
															#line 3205 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadRangeActor, 0, RangeResult >*>(static_cast<ReadRangeActor*>(this)));
															#line 10197 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(RangeResult const& atStorageVersion,int loopDepth) 
	{
															#line 3207 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 3208 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 3209 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 3210 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 3212 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= limit);
															#line 3213 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 10229 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3214 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 3219 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 10235 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3224 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 3225 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes, tenantPrefix);
															#line 3235 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limit -= result.data.size() - prevSize;
															#line 3237 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 3238 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 10247 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3241 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit <= 0 || *pLimitBytes <= 0)
															#line 10251 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3249 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 10257 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3250 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() + tenantPrefix.orDefault(""_sr).size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key) && atStorageVersion.end()[-1].key.startsWith(tenantPrefix.orDefault(""_sr)));
															#line 3255 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			readBegin = readBeginTemp = keyAfter(atStorageVersion.end()[-1].key);
															#line 10263 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 3259 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 10269 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3260 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent->getEndKey() > readBegin);
															#line 3262 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				readBegin = vCurrent->getEndKey();
															#line 3263 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCurrent;
															#line 10277 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 3265 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readEnd == range.end);
															#line 10283 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(RangeResult && atStorageVersion,int loopDepth) 
	{
															#line 3207 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 3208 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 3209 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 3210 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 3212 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= limit);
															#line 3213 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 10305 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3214 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 3219 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 10311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3224 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 3225 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes, tenantPrefix);
															#line 3235 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limit -= result.data.size() - prevSize;
															#line 3237 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 3238 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 10323 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3241 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit <= 0 || *pLimitBytes <= 0)
															#line 10327 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3249 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 10333 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3250 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() + tenantPrefix.orDefault(""_sr).size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key) && atStorageVersion.end()[-1].key.startsWith(tenantPrefix.orDefault(""_sr)));
															#line 3255 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			readBegin = readBeginTemp = keyAfter(atStorageVersion.end()[-1].key);
															#line 10339 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 3259 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 10345 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3260 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent->getEndKey() > readBegin);
															#line 3262 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				readBegin = vCurrent->getEndKey();
															#line 3263 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCurrent;
															#line 10353 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 3265 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readEnd == range.end);
															#line 10359 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				return a_body1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(atStorageVersion, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(atStorageVersion), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state > 0) static_cast<ReadRangeActor*>(this)->actor_wait_state = 0;
		static_cast<ReadRangeActor*>(this)->ActorCallback< ReadRangeActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadRangeActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont11(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead2(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2(int loopDepth) 
	{
															#line 3280 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!(limit < 0 && *pLimitBytes > 0 && readEnd > range.begin))
															#line 10459 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break2(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3281 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(!vCurrent || vCurrent.key() < readEnd);
															#line 3282 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(data->storageVersion() <= version);
															#line 3285 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pos == resultCache.size())
															#line 10469 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3286 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent)
															#line 10473 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3287 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				auto b = vCurrent;
															#line 3288 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++b;
															#line 3289 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(!b || b.key() >= readEnd);
															#line 10481 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3292 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			vCount = 0;
															#line 3293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int vSize = 0;
															#line 3294 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;vCurrent && vCurrent.key() >= range.begin && !vCurrent->isClearTo() && vCount < -limit && vSize < *pLimitBytes;) {
															#line 3297 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				resultCache.emplace_back(result.arena, vCurrent.key(), vCurrent->getValue());
															#line 3298 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				vSize += sizeof(KeyValueRef) + resultCache.cback().expectedSize() - (tenantPrefix.present() ? tenantPrefix.get().size() : 0);
															#line 3300 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++vCount;
															#line 3301 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				--vCurrent;
															#line 10497 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 3305 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readBegin = vCurrent ? std::max(vCurrent->isClearTo() ? vCurrent->getEndKey() : vCurrent.key(), range.begin) : range.begin;
															#line 3307 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = data->storage.readRange(KeyRangeRef(readBegin, readEnd), limit, *pLimitBytes, options);
															#line 3307 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10506 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadRangeActor*>(this)->actor_wait_state = 2;
															#line 3307 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadRangeActor, 1, RangeResult >*>(static_cast<ReadRangeActor*>(this)));
															#line 10511 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break2(int loopDepth) 
	{
		try {
			return a_body1cont11(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody2cont1(RangeResult const& atStorageVersion,int loopDepth) 
	{
															#line 3309 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 3310 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 3311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 3312 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 3314 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= -limit);
															#line 3315 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 10543 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3316 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 3321 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 10549 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3324 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 3325 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes, tenantPrefix);
															#line 3335 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limit += result.data.size() - prevSize;
															#line 3337 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 3338 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 10561 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3341 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit >= 0 || *pLimitBytes <= 0)
															#line 10565 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break2(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3345 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 10571 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3346 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() + tenantPrefix.orDefault(""_sr).size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key) && atStorageVersion.end()[-1].key.startsWith(tenantPrefix.orDefault(""_sr)));
															#line 3351 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			readEnd = atStorageVersion.end()[-1].key;
															#line 10577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 3352 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 10583 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3353 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent.key() < readEnd);
															#line 3354 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				readEnd = vCurrent.key();
															#line 3355 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				--vCurrent;
															#line 10591 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 3357 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readBegin == range.begin);
															#line 10597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				return a_body1break2(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2cont1(RangeResult && atStorageVersion,int loopDepth) 
	{
															#line 3309 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		logicalSize = atStorageVersion.logicalSize();
															#line 3310 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvScanBytes += logicalSize;
															#line 3311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultLogicalSize += logicalSize;
															#line 3312 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->readRangeBytesLimitHistogram->sample(*pLimitBytes);
															#line 3314 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(atStorageVersion.size() <= -limit);
															#line 3315 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->storageVersion() > version)
															#line 10619 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3316 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			DisabledTraceEvent("SS_TTO", data->thisServerID) .detail("StorageVersion", data->storageVersion()) .detail("Oldest", data->oldestVersion.get()) .detail("Version", version) .detail("Range", range);
															#line 3321 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(transaction_too_old(), std::max(0, loopDepth - 1));
															#line 10625 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3324 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int prevSize = result.data.size();
															#line 3325 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		merge(result.arena, result.data, resultCache, atStorageVersion, vCount, limit, atStorageVersion.more, pos, *pLimitBytes, tenantPrefix);
															#line 3335 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limit += result.data.size() - prevSize;
															#line 3337 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(auto i = result.data.begin() + prevSize;i != result.data.end();i++) {
															#line 3338 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pLimitBytes -= sizeof(KeyValueRef) + i->expectedSize();
															#line 10637 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3341 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (limit >= 0 || *pLimitBytes <= 0)
															#line 10641 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1break2(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3345 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (atStorageVersion.more)
															#line 10647 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3346 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(atStorageVersion.end()[-1].key.size() == result.data.end()[-1].key.size() + tenantPrefix.orDefault(""_sr).size() && atStorageVersion.end()[-1].key.endsWith(result.data.end()[-1].key) && atStorageVersion.end()[-1].key.startsWith(tenantPrefix.orDefault(""_sr)));
															#line 3351 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			readEnd = atStorageVersion.end()[-1].key;
															#line 10653 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 3352 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (vCurrent && vCurrent->isClearTo())
															#line 10659 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3353 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(vCurrent.key() < readEnd);
															#line 3354 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				readEnd = vCurrent.key();
															#line 3355 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				--vCurrent;
															#line 10667 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 3357 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(readBegin == range.begin);
															#line 10673 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				return a_body1break2(loopDepth==0?0:loopDepth-1); // break
			}
		}
		if (loopDepth == 0) return a_body1loopHead2(0);

		return loopDepth;
	}
	int a_body1loopBody2when1(RangeResult const& atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(atStorageVersion, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody2when1(RangeResult && atStorageVersion,int loopDepth) 
	{
		loopDepth = a_body1loopBody2cont1(std::move(atStorageVersion), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadRangeActor*>(this)->actor_wait_state > 0) static_cast<ReadRangeActor*>(this)->actor_wait_state = 0;
		static_cast<ReadRangeActor*>(this)->ActorCallback< ReadRangeActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadRangeActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadRangeActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange range;
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int limit;
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int* pLimitBytes;
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext parentSpan;
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<ReadOptions> options;
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<KeyRef> tenantPrefix;
															#line 3127 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesReply result;
															#line 3128 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer::VersionedData::ViewAtVersion view;
															#line 3129 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer::VersionedData::iterator vCurrent;
															#line 3130 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRef readBegin;
															#line 3131 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRef readEnd;
															#line 3132 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key readBeginTemp;
															#line 3133 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int vCount;
															#line 3134 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 3135 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int resultLogicalSize;
															#line 3136 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int logicalSize;
															#line 3139 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	VectorRef<KeyValueRef> resultCache;
															#line 3142 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int pos;
															#line 10796 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via readRange()
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class ReadRangeActor final : public Actor<GetKeyValuesReply>, public ActorCallback< ReadRangeActor, 0, RangeResult >, public ActorCallback< ReadRangeActor, 1, RangeResult >, public FastAllocated<ReadRangeActor>, public ReadRangeActorState<ReadRangeActor> {
															#line 10801 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<ReadRangeActor>::operator new;
	using FastAllocated<ReadRangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8437752693718140160UL, 2317434142485431552UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GetKeyValuesReply>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadRangeActor, 0, RangeResult >;
friend struct ActorCallback< ReadRangeActor, 1, RangeResult >;
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	ReadRangeActor(StorageServer* const& data,Version const& version,KeyRange const& range,int const& limit,int* const& pLimitBytes,SpanContext const& parentSpan,Optional<ReadOptions> const& options,Optional<KeyRef> const& tenantPrefix) 
															#line 10819 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<GetKeyValuesReply>(),
		   ReadRangeActorState<ReadRangeActor>(data, version, range, limit, pLimitBytes, parentSpan, options, tenantPrefix),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readRange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3635396024098275328UL, 3211122134102601216UL);
		ActorExecutionContextHelper __helper(static_cast<ReadRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readRange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readRange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadRangeActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadRangeActor, 1, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<GetKeyValuesReply> readRange( StorageServer* const& data, Version const& version, KeyRange const& range, int const& limit, int* const& pLimitBytes, SpanContext const& parentSpan, Optional<ReadOptions> const& options, Optional<KeyRef> const& tenantPrefix ) {
															#line 3119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<GetKeyValuesReply>(new ReadRangeActor(data, version, range, limit, pLimitBytes, parentSpan, options, tenantPrefix));
															#line 10853 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 3371 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 10858 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via findKey()
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class FindKeyActor>
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FindKeyActorState {
															#line 10865 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FindKeyActorState(StorageServer* const& data,KeySelectorRef const& sel,Version const& version,KeyRange const& range,int* const& pOffset,SpanContext const& parentSpan,Optional<ReadOptions> const& options) 
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   sel(sel),
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   range(range),
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   pOffset(pOffset),
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   parentSpan(parentSpan),
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   options(options)
															#line 10884 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("findKey", reinterpret_cast<unsigned long>(this));

	}
	~FindKeyActorState() 
	{
		fdb_probe_actor_destroy("findKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3390 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(version != latestVersion);
															#line 3391 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(selectorInRange(sel, range) && version >= data->oldestVersion.get());
															#line 3394 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			forward = sel.offset > 0;
															#line 3395 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			sign = forward ? +1 : -1;
															#line 3396 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			skipEqualKey = sel.orEqual == forward;
															#line 3397 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			distance = forward ? sel.offset : 1 - sel.offset;
															#line 3398 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			span = Span("SS.findKey"_loc, parentSpan);
															#line 3402 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			maxBytes = int();
															#line 3403 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (sel.offset <= 1 && sel.offset >= 0)
															#line 10915 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3404 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				maxBytes = std::numeric_limits<int>::max();
															#line 10919 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 3406 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				maxBytes = (g_network->isSimulated() && g_simulator->tssMode == ISimulator::TSSMode::Disabled && BUGGIFY) ? SERVER_KNOBS->BUGGIFY_LIMIT_BYTES : SERVER_KNOBS->STORAGE_LIMIT_BYTES;
															#line 10925 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3410 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetKeyValuesReply> __when_expr_0 = readRange(data, version, forward ? KeyRangeRef(sel.getKey(), range.end) : KeyRangeRef(range.begin, keyAfter(sel.getKey())), (distance + skipEqualKey) * sign, &maxBytes, span.context, options, {});
															#line 3410 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FindKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10931 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FindKeyActor*>(this)->actor_wait_state = 1;
															#line 3410 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*>(static_cast<FindKeyActor*>(this)));
															#line 10936 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FindKeyActorState();
		static_cast<FindKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3419 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		more = rep.more && rep.data.size() != distance + skipEqualKey;
															#line 3423 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (more && !forward && rep.data.size() == 1)
															#line 10961 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3424 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "Reverse key selector returned only one result in range read");
															#line 3425 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			maxBytes = std::numeric_limits<int>::max();
															#line 3426 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetKeyValuesReply> __when_expr_1 = readRange( data, version, KeyRangeRef(range.begin, keyAfter(sel.getKey())), -2, &maxBytes, span.context, options, {});
															#line 3426 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<FindKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10971 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<FindKeyActor*>(this)->actor_wait_state = 2;
															#line 3426 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*>(static_cast<FindKeyActor*>(this)));
															#line 10976 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply const& __rep,int loopDepth) 
	{
															#line 3410 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rep = __rep;
															#line 10990 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply && __rep,int loopDepth) 
	{
		rep = std::move(__rep);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FindKeyActor*>(this)->actor_wait_state > 0) static_cast<FindKeyActor*>(this)->actor_wait_state = 0;
		static_cast<FindKeyActor*>(this)->ActorCallback< FindKeyActor, 0, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 3433 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int index = distance - 1;
															#line 3434 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (skipEqualKey && rep.data.size() && rep.data[0].key == sel.getKey())
															#line 11071 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3435 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++index;
															#line 11075 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3437 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (index < rep.data.size())
															#line 11079 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3438 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pOffset = 0;
															#line 3440 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 11085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3441 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				int64_t bytesReadPerKSecond = std::max((int64_t)rep.data[index].key.size(), SERVER_KNOBS->EMPTY_READ_PENALTY);
															#line 3443 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				data->metrics.notifyBytesReadPerKSecond(sel.getKey(), bytesReadPerKSecond);
															#line 11091 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3446 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<FindKeyActor*>(this)->SAV<Key>::futures) { (void)(rep.data[index].key); this->~FindKeyActorState(); static_cast<FindKeyActor*>(this)->destroy(); return 0; }
															#line 11095 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<FindKeyActor*>(this)->SAV< Key >::value()) Key(rep.data[index].key);
			this->~FindKeyActorState();
			static_cast<FindKeyActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 3448 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 11105 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3449 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				int64_t bytesReadPerKSecond = SERVER_KNOBS->EMPTY_READ_PENALTY;
															#line 3450 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				data->metrics.notifyBytesReadPerKSecond(sel.getKey(), bytesReadPerKSecond);
															#line 11111 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3454 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			*pOffset = index - rep.data.size() + 1;
															#line 3455 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!forward)
															#line 11117 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3456 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				*pOffset = -*pOffset;
															#line 11121 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3458 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (more)
															#line 11125 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3459 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				CODE_PROBE(true, "Key selector read range had more results");
															#line 3461 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(rep.data.size());
															#line 3462 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				Key returnKey = forward ? keyAfter(rep.data.back().key) : rep.data.back().key;
															#line 3466 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ASSERT(returnKey != sel.getKey());
															#line 3467 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<FindKeyActor*>(this)->SAV<Key>::futures) { (void)(returnKey); this->~FindKeyActorState(); static_cast<FindKeyActor*>(this)->destroy(); return 0; }
															#line 11137 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<FindKeyActor*>(this)->SAV< Key >::value()) Key(returnKey);
				this->~FindKeyActorState();
				static_cast<FindKeyActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 3469 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<FindKeyActor*>(this)->SAV<Key>::futures) { (void)(forward ? range.end : range.begin); this->~FindKeyActorState(); static_cast<FindKeyActor*>(this)->destroy(); return 0; }
															#line 11147 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<FindKeyActor*>(this)->SAV< Key >::value()) Key(forward ? range.end : range.begin);
				this->~FindKeyActorState();
				static_cast<FindKeyActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
		}

		return loopDepth;
	}
	int a_body1cont5(GetKeyValuesReply const& rep2,int loopDepth) 
	{
															#line 3428 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rep = rep2;
															#line 3429 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		more = rep.more && rep.data.size() != distance + skipEqualKey;
															#line 3430 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(rep.data.size() == 2 || !more);
															#line 11165 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(GetKeyValuesReply && rep2,int loopDepth) 
	{
															#line 3428 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rep = rep2;
															#line 3429 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		more = rep.more && rep.data.size() != distance + skipEqualKey;
															#line 3430 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ASSERT(rep.data.size() == 2 || !more);
															#line 11178 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(GetKeyValuesReply const& rep2,int loopDepth) 
	{
		loopDepth = a_body1cont5(rep2, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(GetKeyValuesReply && rep2,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(rep2), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FindKeyActor*>(this)->actor_wait_state > 0) static_cast<FindKeyActor*>(this)->actor_wait_state = 0;
		static_cast<FindKeyActor*>(this)->ActorCallback< FindKeyActor, 1, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeySelectorRef sel;
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange range;
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int* pOffset;
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	SpanContext parentSpan;
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Optional<ReadOptions> options;
															#line 3394 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool forward;
															#line 3395 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int sign;
															#line 3396 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool skipEqualKey;
															#line 3397 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int distance;
															#line 3398 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 3402 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int maxBytes;
															#line 3410 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesReply rep;
															#line 3419 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool more;
															#line 11288 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via findKey()
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class FindKeyActor final : public Actor<Key>, public ActorCallback< FindKeyActor, 0, GetKeyValuesReply >, public ActorCallback< FindKeyActor, 1, GetKeyValuesReply >, public FastAllocated<FindKeyActor>, public FindKeyActorState<FindKeyActor> {
															#line 11293 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<FindKeyActor>::operator new;
	using FastAllocated<FindKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18425706085405398272UL, 14833325181917898240UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Key>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FindKeyActor, 0, GetKeyValuesReply >;
friend struct ActorCallback< FindKeyActor, 1, GetKeyValuesReply >;
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FindKeyActor(StorageServer* const& data,KeySelectorRef const& sel,Version const& version,KeyRange const& range,int* const& pOffset,SpanContext const& parentSpan,Optional<ReadOptions> const& options) 
															#line 11311 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Key>(),
		   FindKeyActorState<FindKeyActor>(data, sel, version, range, pOffset, parentSpan, options),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("findKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11946933499123123200UL, 11635880071562820096UL);
		ActorExecutionContextHelper __helper(static_cast<FindKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("findKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("findKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FindKeyActor, 0, GetKeyValuesReply >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FindKeyActor, 1, GetKeyValuesReply >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Key> findKey( StorageServer* const& data, KeySelectorRef const& sel, Version const& version, KeyRange const& range, int* const& pOffset, SpanContext const& parentSpan, Optional<ReadOptions> const& options ) {
															#line 3372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Key>(new FindKeyActor(data, sel, version, range, pOffset, parentSpan, options));
															#line 11345 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 3473 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

KeyRange getShardKeyRange(StorageServer* data, const KeySelectorRef& sel)
// Returns largest range such that the shard state isReadable and selectorInRange(sel, range) or wrong_shard_server if
// no such range exists
{
	auto i = sel.isBackward() ? data->shards.rangeContainingKeyBefore(sel.getKey())
	                          : data->shards.rangeContaining(sel.getKey());
	auto fullRange = data->shards.ranges();
	if (!i->value()->isReadable())
		throw wrong_shard_server();
	ASSERT(selectorInRange(sel, i->range()));
	Key begin, end;
	if (sel.isBackward()) {
		end = i->range().end;
		while (i != fullRange.begin() && i.value()->isReadable()) {
			begin = i->range().begin;
			--i;
		}
		if (i.value()->isReadable()) {
			begin = i->range().begin;
		}
	} else {
		begin = i->range().begin;
		while (i != fullRange.end() && i.value()->isReadable()) {
			end = i->range().end;
			++i;
		}
	}
	return KeyRangeRef(begin, end);
}

void maybeInjectConsistencyScanCorruption(UID thisServerID, GetKeyValuesRequest const& req, GetKeyValuesReply& reply) {
	if (g_simulator->consistencyScanState != ISimulator::SimConsistencyScanState::Enabled_InjectCorruption ||
	    !req.options.present() || !req.options.get().consistencyCheckStartVersion.present() ||
	    !g_simulator->consistencyScanCorruptRequestKey.present()) {
		return;
	}

	UID destination = req.reply.getEndpoint().token;

	ASSERT(g_simulator->consistencyScanInjectedCorruptionType.present() ==
	       g_simulator->consistencyScanInjectedCorruptionDestination.present());
	// if we already injected a corruption, reinject it if this request was a retransmit of the same one we corrupted
	// could also check that this storage sent the corruption but the reply endpoints should be globally unique so this
	// covers it
	if (g_simulator->consistencyScanInjectedCorruptionDestination.present() &&
	    (g_simulator->consistencyScanInjectedCorruptionDestination.get() != destination)) {
		return;
	}

	CODE_PROBE(true, "consistency check injecting corruption");
	CODE_PROBE(g_simulator->consistencyScanInjectedCorruptionDestination.present() &&
	               g_simulator->consistencyScanInjectedCorruptionDestination.get() == destination,
	           "consistency check re-injecting corruption after retransmit",
	           probe::decoration::rare);

	g_simulator->consistencyScanInjectedCorruptionDestination = destination;
	// FIXME: reinject same type of corruption once we enable other types

	// FIXME: code probe for each type?

	if (true /*deterministicRandom()->random01() < 0.3*/) {
		// flip more flag
		reply.more = !reply.more;
		g_simulator->consistencyScanInjectedCorruptionType = ISimulator::SimConsistencyScanCorruptionType::FlipMoreFlag;
	} else {
		// FIXME: weird memory issues when messing with actual response data, enable and figure out later
		ASSERT(false);
		// make deep copy of request, since some of the underlying memory can reference storage engine data directly
		GetKeyValuesReply copy = reply;
		reply = GetKeyValuesReply();
		reply.more = copy.more;
		reply.cached = copy.cached;
		reply.version = copy.version;
		reply.data.append_deep(reply.arena, copy.data.begin(), copy.data.size());

		if (reply.data.empty()) {
			// add row to empty response
			g_simulator->consistencyScanInjectedCorruptionType =
			    ISimulator::SimConsistencyScanCorruptionType::AddToEmpty;
			reply.data.push_back_deep(
			    reply.arena,
			    KeyValueRef(g_simulator->consistencyScanCorruptRequestKey.get(), "consistencyCheckCorruptValue"_sr));
		} else if (deterministicRandom()->coinflip() || reply.data.back().value.empty()) {
			// change value in non-empty response
			g_simulator->consistencyScanInjectedCorruptionType =
			    ISimulator::SimConsistencyScanCorruptionType::RemoveLastRow;
			reply.data.pop_back();
		} else {
			// chop off last byte of first value
			g_simulator->consistencyScanInjectedCorruptionType =
			    ISimulator::SimConsistencyScanCorruptionType::ChangeFirstValue;

			reply.data[0].value = reply.data[0].value.substr(0, reply.data[0].value.size() - 1);
		}
	}

	TraceEvent(SevWarnAlways, "InjectedConsistencyScanCorruption", thisServerID)
	    .detail("CorruptionType", g_simulator->consistencyScanInjectedCorruptionType.get())
	    .detail("Version", req.version)
	    .detail("Count", reply.data.size());
}

															#line 11452 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getKeyValuesQ()
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class GetKeyValuesQActor>
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetKeyValuesQActorState {
															#line 11459 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesQActorState(StorageServer* const& data,GetKeyValuesRequest const& req) 
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req),
															#line 3580 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   span("SS:getKeyValues"_loc, req.spanContext),
															#line 3581 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   resultSize(0)
															#line 11472 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("getKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	~GetKeyValuesQActorState() 
	{
		fdb_probe_actor_destroy("getKeyValuesQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3583 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getCurrentLineage()->modify(&TransactionLineage::txID) = req.spanContext.traceID;
															#line 3585 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.getRangeQueries;
															#line 3586 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.allQueries;
															#line 3587 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.begin.getKey().startsWith(systemKeys.begin))
															#line 11493 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3588 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.systemKeyQueries;
															#line 3589 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++data->counters.getRangeSystemKeyQueries;
															#line 11499 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3591 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->maxQueryQueue = std::max<int>( data->maxQueryQueue, data->counters.allQueries.getValue() - data->counters.finishedQueries.getValue());
															#line 3596 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->getQueryDelay();
															#line 3596 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11507 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 1;
															#line 3596 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 0, Void >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11512 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetKeyValuesQActorState();
		static_cast<GetKeyValuesQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 3597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11537 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 3597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11542 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<PriorityMultiLock::Lock> __when_expr_1 = data->getReadLock(req.options);
															#line 3597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11553 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 2;
															#line 3597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11558 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 3600 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		queueWaitEnd = g_network->timer();
															#line 3601 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readQueueWaitSample->addMeasurement(queueWaitEnd - req.requestTime());
															#line 11644 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 3604 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.options.present() && req.options.get().debugID.present())
															#line 11648 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3605 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.Before");
															#line 11652 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3608 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Version commitVersion = getLatestCommitVersion(req.ssLatestCommitVersions, data->tag);
															#line 3609 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Version> __when_expr_2 = waitForVersion(data, commitVersion, req.version, span.context);
															#line 3609 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 11660 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont3Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 3;
															#line 3609 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 2, Version >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11665 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(PriorityMultiLock::Lock const& __readLock,int loopDepth) 
	{
															#line 3597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readLock = __readLock;
															#line 11680 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(PriorityMultiLock::Lock && __readLock,int loopDepth) 
	{
		readLock = std::move(__readLock);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*,PriorityMultiLock::Lock && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 3787 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->transactionTagCounter.addRequest(req.tags, resultSize);
															#line 3788 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.finishedQueries;
															#line 3790 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		double duration = g_network->timer() - req.requestTime();
															#line 3791 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readLatencySample->addMeasurement(duration);
															#line 3792 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readRangeLatencySample->addMeasurement(duration);
															#line 3793 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->latencyBandConfig.present())
															#line 11769 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3794 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int maxReadBytes = data->latencyBandConfig.get().readConfig.maxReadBytes.orDefault(std::numeric_limits<int>::max());
															#line 3796 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int maxSelectorOffset = data->latencyBandConfig.get().readConfig.maxKeySelectorOffset.orDefault(std::numeric_limits<int>::max());
															#line 3798 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.readLatencyBands.addMeasurement(duration, 1, Filtered(resultSize > maxReadBytes || abs(req.begin.offset) > maxSelectorOffset || abs(req.end.offset) > maxSelectorOffset));
															#line 11777 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3805 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<GetKeyValuesQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetKeyValuesQActorState(); static_cast<GetKeyValuesQActor*>(this)->destroy(); return 0; }
															#line 11781 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<GetKeyValuesQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetKeyValuesQActorState();
		static_cast<GetKeyValuesQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3782 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!canReplyWith(e))
															#line 11794 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3783 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 11798 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3784 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->sendErrorWithPenalty(req.reply, e, data->getPenalty());
															#line 11802 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
															#line 3610 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		DisabledTraceEvent("VVV", data->thisServerID) .detail("Version", version) .detail("ReqVersion", req.version) .detail("Oldest", data->oldestVersion.get()) .detail("VV", req.ssLatestCommitVersions.toString()) .detail("DebugID", req.options.present() && req.options.get().debugID.present() ? req.options.get().debugID.get() : UID());
															#line 3618 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.readVersionWaitSample->addMeasurement(g_network->timer() - queueWaitEnd);
															#line 3620 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkTenantEntry(version, req.tenantInfo, req.options.present() ? req.options.get().lockAware : false);
															#line 3621 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.tenantInfo.hasTenant())
															#line 11823 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3622 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.begin.setKeyUnlimited(req.begin.getKey().withPrefix(req.tenantInfo.prefix.get(), req.arena));
															#line 3623 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.end.setKeyUnlimited(req.end.getKey().withPrefix(req.tenantInfo.prefix.get(), req.arena));
															#line 11829 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3626 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		changeCounter = data->shardChangeCounter;
															#line 3628 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		shard = getShardKeyRange(data, req.begin);
															#line 3630 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 11837 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3631 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterVersion");
															#line 11841 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3638 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!selectorInRange(req.end, shard) && !(req.end.isFirstGreaterOrEqual() && req.end.getKey() == shard.end))
															#line 11845 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3646 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont3Catch1(wrong_shard_server(), loopDepth);
															#line 11849 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3649 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		KeyRangeRef searchRange = TenantAPI::clampRangeToTenant(shard, req.tenantInfo, req.arena);
															#line 3651 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		offset1 = 0;
															#line 3652 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		offset2 = int();
															#line 3653 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fBegin = req.begin.isFirstGreaterOrEqual() ? Future<Key>(req.begin.getKey()) : findKey(data, req.begin, version, searchRange, &offset1, span.context, req.options);
															#line 3657 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fEnd = req.end.isFirstGreaterOrEqual() ? Future<Key>(req.end.getKey()) : findKey(data, req.end, version, searchRange, &offset2, span.context, req.options);
															#line 3661 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Key> __when_expr_3 = fBegin;
															#line 3661 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 11865 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont3Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 4;
															#line 3661 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 3, Key >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11870 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3when1(Version const& __version,int loopDepth) 
	{
															#line 3609 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		version = __version;
															#line 11879 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Version && __version,int loopDepth) 
	{
		version = std::move(__version);
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 2, Version >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 2, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 2, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 2, Version >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 3662 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Key> __when_expr_4 = fEnd;
															#line 3662 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 11960 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont3Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont7when1(__when_expr_4.get(), loopDepth); };
		static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 5;
															#line 3662 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 4, Key >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 11965 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5when1(Key const& __begin,int loopDepth) 
	{
															#line 3661 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		begin = __begin;
															#line 11974 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Key && __begin,int loopDepth) 
	{
		begin = std::move(__begin);
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 3, Key >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 3, Key >*,Key const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 3, Key >*,Key && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 3, Key >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont11(int loopDepth) 
	{
															#line 3664 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 12053 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3665 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterKeys");
															#line 12057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3673 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if ((offset1 && offset1 != 1) || (offset2 && offset2 != 1))
															#line 12061 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3674 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			CODE_PROBE(true, "wrong_shard_server due to offset");
															#line 3690 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont3Catch1(wrong_shard_server(), loopDepth);
															#line 12067 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3693 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (begin >= end)
															#line 12071 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3694 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (req.options.present() && req.options.get().debugID.present())
															#line 12075 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3695 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent( "TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.Send");
															#line 12079 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			GetKeyValuesReply none;
															#line 3700 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			none.version = version;
															#line 3701 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			none.more = false;
															#line 3702 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			none.penalty = data->getPenalty();
															#line 3704 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->checkChangeCounter(changeCounter, KeyRangeRef(std::min<KeyRef>(req.begin.getKey(), req.end.getKey()), std::max<KeyRef>(req.begin.getKey(), req.end.getKey())));
															#line 3708 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (g_network->isSimulated())
															#line 12093 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3709 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				maybeInjectConsistencyScanCorruption(data->thisServerID, req, none);
															#line 12097 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3711 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(none);
															#line 12101 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont11cont1(loopDepth);
		}
		else
		{
															#line 3713 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			remainingLimitBytes = req.limitBytes;
															#line 3715 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			kvReadRange = g_network->timer();
															#line 3716 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<GetKeyValuesReply> __when_expr_5 = readRange(data, version, KeyRangeRef(begin, end), req.limit, &remainingLimitBytes, span.context, req.options, req.tenantInfo.prefix);
															#line 3716 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 12114 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont3Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont11when1(__when_expr_5.get(), loopDepth); };
			static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 6;
															#line 3716 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*>(static_cast<GetKeyValuesQActor*>(this)));
															#line 12119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont7when1(Key const& __end,int loopDepth) 
	{
															#line 3662 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		end = __end;
															#line 12129 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Key && __end,int loopDepth) 
	{
		end = std::move(__end);
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 4, Key >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 4, Key >*,Key const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 4, Key >*,Key && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 4, Key >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont11cont1(int loopDepth) 
	{
		loopDepth = a_body1cont11cont19(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont7(GetKeyValuesReply const& _r,int loopDepth) 
	{
															#line 3724 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - kvReadRange;
															#line 3725 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvReadRangeLatencySample->addMeasurement(duration);
															#line 3726 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		GetKeyValuesReply r = _r;
															#line 3728 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 12220 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3729 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterReadRange");
															#line 12224 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3733 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter( changeCounter, KeyRangeRef(std::min<KeyRef>(begin, std::min<KeyRef>(req.begin.getKey(), req.end.getKey())), std::max<KeyRef>(end, std::max<KeyRef>(req.begin.getKey(), req.end.getKey()))));
															#line 3737 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (EXPENSIVE_VALIDATION)
															#line 12230 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3738 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = 0;i < r.data.size();i++) {
															#line 3739 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (req.tenantInfo.prefix.present())
															#line 12236 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3740 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(r.data[i].key >= begin.removePrefix(req.tenantInfo.prefix.get()) && r.data[i].key < end.removePrefix(req.tenantInfo.prefix.get()));
															#line 12240 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 3743 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(r.data[i].key >= begin && r.data[i].key < end);
															#line 12246 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 3746 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(r.data.size() <= std::abs(req.limit));
															#line 12251 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3750 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int64_t totalByteSize = 0;
															#line 3751 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < r.data.size();i++) {
															#line 3752 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			totalByteSize += r.data[i].expectedSize();
															#line 12259 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3755 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.taskID.present() && req.taskID.get() == TaskPriority::FetchKeys)
															#line 12263 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3756 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchServed += r.data.size();
															#line 3757 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchBytesServed += (totalByteSize + (8 - (int)sizeof(KeyValueRef)) * r.data.size());
															#line 12269 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3760 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (totalByteSize > 0 && SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 12273 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3761 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int64_t bytesReadPerKSecond = std::max(totalByteSize, SERVER_KNOBS->EMPTY_READ_PENALTY) / 2;
															#line 3762 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond(addPrefix(r.data[0].key, req.tenantInfo.prefix, req.arena), bytesReadPerKSecond);
															#line 3764 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond( addPrefix(r.data[r.data.size() - 1].key, req.tenantInfo.prefix, req.arena), bytesReadPerKSecond);
															#line 12281 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3768 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		r.penalty = data->getPenalty();
															#line 3769 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (g_network->isSimulated())
															#line 12287 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3770 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			maybeInjectConsistencyScanCorruption(data->thisServerID, req, r);
															#line 12291 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3772 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(r);
															#line 3774 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultSize = req.limitBytes - remainingLimitBytes;
															#line 3775 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.bytesQueried += resultSize;
															#line 3776 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.rowsQueried += r.data.size();
															#line 3777 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.data.size() == 0)
															#line 12303 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3778 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.emptyQueries;
															#line 12307 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		loopDepth = a_body1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont7(GetKeyValuesReply && _r,int loopDepth) 
	{
															#line 3724 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - kvReadRange;
															#line 3725 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.kvReadRangeLatencySample->addMeasurement(duration);
															#line 3726 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		GetKeyValuesReply r = _r;
															#line 3728 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.options.present() && req.options.get().debugID.present())
															#line 12323 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3729 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", req.options.get().debugID.get().first(), "storageserver.getKeyValues.AfterReadRange");
															#line 12327 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3733 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->checkChangeCounter( changeCounter, KeyRangeRef(std::min<KeyRef>(begin, std::min<KeyRef>(req.begin.getKey(), req.end.getKey())), std::max<KeyRef>(end, std::max<KeyRef>(req.begin.getKey(), req.end.getKey()))));
															#line 3737 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (EXPENSIVE_VALIDATION)
															#line 12333 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3738 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(int i = 0;i < r.data.size();i++) {
															#line 3739 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (req.tenantInfo.prefix.present())
															#line 12339 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 3740 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(r.data[i].key >= begin.removePrefix(req.tenantInfo.prefix.get()) && r.data[i].key < end.removePrefix(req.tenantInfo.prefix.get()));
															#line 12343 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 3743 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					ASSERT(r.data[i].key >= begin && r.data[i].key < end);
															#line 12349 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 3746 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(r.data.size() <= std::abs(req.limit));
															#line 12354 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3750 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		int64_t totalByteSize = 0;
															#line 3751 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < r.data.size();i++) {
															#line 3752 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			totalByteSize += r.data[i].expectedSize();
															#line 12362 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3755 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (req.taskID.present() && req.taskID.get() == TaskPriority::FetchKeys)
															#line 12366 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3756 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchServed += r.data.size();
															#line 3757 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.kvFetchBytesServed += (totalByteSize + (8 - (int)sizeof(KeyValueRef)) * r.data.size());
															#line 12372 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3760 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (totalByteSize > 0 && SERVER_KNOBS->READ_SAMPLING_ENABLED)
															#line 12376 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3761 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int64_t bytesReadPerKSecond = std::max(totalByteSize, SERVER_KNOBS->EMPTY_READ_PENALTY) / 2;
															#line 3762 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond(addPrefix(r.data[0].key, req.tenantInfo.prefix, req.arena), bytesReadPerKSecond);
															#line 3764 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->metrics.notifyBytesReadPerKSecond( addPrefix(r.data[r.data.size() - 1].key, req.tenantInfo.prefix, req.arena), bytesReadPerKSecond);
															#line 12384 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3768 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		r.penalty = data->getPenalty();
															#line 3769 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (g_network->isSimulated())
															#line 12390 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3770 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			maybeInjectConsistencyScanCorruption(data->thisServerID, req, r);
															#line 12394 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3772 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.send(r);
															#line 3774 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		resultSize = req.limitBytes - remainingLimitBytes;
															#line 3775 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.bytesQueried += resultSize;
															#line 3776 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.rowsQueried += r.data.size();
															#line 3777 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (r.data.size() == 0)
															#line 12406 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3778 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.emptyQueries;
															#line 12410 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		loopDepth = a_body1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(GetKeyValuesReply const& _r,int loopDepth) 
	{
		loopDepth = a_body1cont11cont7(_r, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(GetKeyValuesReply && _r,int loopDepth) 
	{
		loopDepth = a_body1cont11cont7(std::move(_r), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<GetKeyValuesQActor*>(this)->actor_wait_state > 0) static_cast<GetKeyValuesQActor*>(this)->actor_wait_state = 0;
		static_cast<GetKeyValuesQActor*>(this)->ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont11cont19(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesRequest req;
															#line 3580 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Span span;
															#line 3581 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t resultSize;
															#line 3597 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	PriorityMultiLock::Lock readLock;
															#line 3600 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double queueWaitEnd;
															#line 3609 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3626 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	uint64_t changeCounter;
															#line 3628 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange shard;
															#line 3651 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int offset1;
															#line 3652 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int offset2;
															#line 3653 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Key> fBegin;
															#line 3657 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Future<Key> fEnd;
															#line 3661 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key begin;
															#line 3662 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key end;
															#line 3713 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int remainingLimitBytes;
															#line 3715 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double kvReadRange;
															#line 12538 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via getKeyValuesQ()
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class GetKeyValuesQActor final : public Actor<Void>, public ActorCallback< GetKeyValuesQActor, 0, Void >, public ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >, public ActorCallback< GetKeyValuesQActor, 2, Version >, public ActorCallback< GetKeyValuesQActor, 3, Key >, public ActorCallback< GetKeyValuesQActor, 4, Key >, public ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >, public FastAllocated<GetKeyValuesQActor>, public GetKeyValuesQActorState<GetKeyValuesQActor> {
															#line 12543 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<GetKeyValuesQActor>::operator new;
	using FastAllocated<GetKeyValuesQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7871000478408148480UL, 9406043148795705600UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetKeyValuesQActor, 0, Void >;
friend struct ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >;
friend struct ActorCallback< GetKeyValuesQActor, 2, Version >;
friend struct ActorCallback< GetKeyValuesQActor, 3, Key >;
friend struct ActorCallback< GetKeyValuesQActor, 4, Key >;
friend struct ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >;
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetKeyValuesQActor(StorageServer* const& data,GetKeyValuesRequest const& req) 
															#line 12565 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   GetKeyValuesQActorState<GetKeyValuesQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15085295619130978816UL, 4562439640475096832UL);
		ActorExecutionContextHelper __helper(static_cast<GetKeyValuesQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getKeyValuesQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getKeyValuesQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 1, PriorityMultiLock::Lock >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 2, Version >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 3, Key >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 4, Key >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< GetKeyValuesQActor, 5, GetKeyValuesReply >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> getKeyValuesQ( StorageServer* const& data, GetKeyValuesRequest const& req ) {
															#line 3576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new GetKeyValuesQActor(data, req));
															#line 12603 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 3807 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 12608 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via quickGetKeyValues()
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class QuickGetKeyValuesActor>
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetKeyValuesActorState {
															#line 12615 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetKeyValuesActorState(StorageServer* const& data,StringRef const& prefix,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   prefix(prefix),
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   version(version),
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   a(a),
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   pOriginalReq(pOriginalReq),
															#line 3815 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   getRange(),
															#line 3816 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   getValuesStart(g_network->timer())
															#line 12634 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("quickGetKeyValues", reinterpret_cast<unsigned long>(this));

	}
	~QuickGetKeyValuesActorState() 
	{
		fdb_probe_actor_destroy("quickGetKeyValues", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3817 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.begin = firstGreaterOrEqual(KeyRef(*a, prefix));
															#line 3818 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.end = firstGreaterOrEqual(strinc(prefix, *a));
															#line 3819 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12653 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3820 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.Before");
															#line 12657 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			try {
															#line 3825 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				GetKeyValuesRequest req;
															#line 3826 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.spanContext = pOriginalReq->spanContext;
															#line 3827 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.options = pOriginalReq->options;
															#line 3828 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.arena = *a;
															#line 3829 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.begin = getRange.begin;
															#line 3830 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.end = getRange.end;
															#line 3831 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.version = version;
															#line 3832 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.tenantInfo = pOriginalReq->tenantInfo;
															#line 3835 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.limit = SERVER_KNOBS->QUICK_GET_KEY_VALUES_LIMIT;
															#line 3836 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.limitBytes = SERVER_KNOBS->QUICK_GET_KEY_VALUES_LIMIT_BYTES;
															#line 3837 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.options = pOriginalReq->options;
															#line 3839 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.tags = pOriginalReq->tags;
															#line 3840 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.ssLatestCommitVersions = VersionVector();
															#line 3845 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				data->actors.add(getKeyValuesQ(data, req));
															#line 3846 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<GetKeyValuesReply> __when_expr_0 = req.reply.getFuture();
															#line 3846 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 12692 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 1;
															#line 3846 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*>(static_cast<QuickGetKeyValuesActor*>(this)));
															#line 12697 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~QuickGetKeyValuesActorState();
		static_cast<QuickGetKeyValuesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 3865 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		++data->counters.quickGetKeyValuesMiss;
															#line 3866 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (SERVER_KNOBS->QUICK_GET_KEY_VALUES_FALLBACK)
															#line 12728 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3867 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Optional<Reference<Tenant>> tenant = pOriginalReq->tenantInfo.hasTenant() ? makeReference<Tenant>(pOriginalReq->tenantInfo.tenantId) : Optional<Reference<Tenant>>();
															#line 3870 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr = Transaction(data->cx, tenant);
															#line 3871 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setVersion(version);
															#line 3872 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12738 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3873 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				tr.debugTransaction(pOriginalReq->options.get().debugID.get());
															#line 12742 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3876 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.trState->taskID = TaskPriority::DefaultPromiseEndpoint;
															#line 3877 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Future<RangeResult> rangeResultFuture = tr.getRange(prefixRange(prefix), GetRangeLimits::ROW_LIMIT_UNLIMITED, Snapshot::True);
															#line 3880 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<RangeResult> __when_expr_1 = rangeResultFuture;
															#line 3880 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12752 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 2;
															#line 3880 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*>(static_cast<QuickGetKeyValuesActor*>(this)));
															#line 12757 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 3891 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1Catch1(quick_get_key_values_miss(), loopDepth);
															#line 12764 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(GetKeyValuesReply const& reply,int loopDepth) 
	{
															#line 3847 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 12786 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3848 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetKeyValuesHit;
															#line 3850 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			a->dependsOn(reply.arena);
															#line 3851 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.result = RangeResultRef(reply.data, reply.more);
															#line 3852 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValuesStart;
															#line 3853 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.mappedRangeLocalSample->addMeasurement(duration);
															#line 3854 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12800 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3855 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterLocalFetch");
															#line 12804 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3858 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 12808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
			this->~QuickGetKeyValuesActorState();
			static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(GetKeyValuesReply && reply,int loopDepth) 
	{
															#line 3847 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!reply.error.present())
															#line 12822 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3848 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			++data->counters.quickGetKeyValuesHit;
															#line 3850 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			a->dependsOn(reply.arena);
															#line 3851 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			getRange.result = RangeResultRef(reply.data, reply.more);
															#line 3852 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const double duration = g_network->timer() - getValuesStart;
															#line 3853 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->counters.mappedRangeLocalSample->addMeasurement(duration);
															#line 3854 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12836 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 3855 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterLocalFetch");
															#line 12840 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 3858 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 12844 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
			this->~QuickGetKeyValuesActorState();
			static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply const& reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(reply, loopDepth);

		return loopDepth;
	}
	int a_body1when1(GetKeyValuesReply && reply,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(reply), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state > 0) static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetKeyValuesActor*>(this)->ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*,GetKeyValuesReply const& value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*,GetKeyValuesReply && value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(RangeResult const& rangeResult,int loopDepth) 
	{
															#line 3881 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		a->dependsOn(rangeResult.arena());
															#line 3882 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		getRange.result = rangeResult;
															#line 3883 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - getValuesStart;
															#line 3884 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.mappedRangeRemoteSample->addMeasurement(duration);
															#line 3885 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12954 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3886 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterRemoteFetch");
															#line 12958 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3889 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 12962 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
		this->~QuickGetKeyValuesActorState();
		static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9(RangeResult && rangeResult,int loopDepth) 
	{
															#line 3881 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		a->dependsOn(rangeResult.arena());
															#line 3882 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		getRange.result = rangeResult;
															#line 3883 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const double duration = g_network->timer() - getValuesStart;
															#line 3884 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->counters.mappedRangeRemoteSample->addMeasurement(duration);
															#line 3885 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (pOriginalReq->options.present() && pOriginalReq->options.get().debugID.present())
															#line 12982 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 3886 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			g_traceBatch.addEvent("TransactionDebug", pOriginalReq->options.get().debugID.get().first(), "storageserver.quickGetKeyValues.AfterRemoteFetch");
															#line 12986 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 3889 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<QuickGetKeyValuesActor*>(this)->SAV<GetRangeReqAndResultRef>::futures) { (void)(getRange); this->~QuickGetKeyValuesActorState(); static_cast<QuickGetKeyValuesActor*>(this)->destroy(); return 0; }
															#line 12990 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<QuickGetKeyValuesActor*>(this)->SAV< GetRangeReqAndResultRef >::value()) GetRangeReqAndResultRef(std::move(getRange)); // state_var_RVO
		this->~QuickGetKeyValuesActorState();
		static_cast<QuickGetKeyValuesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult const& rangeResult,int loopDepth) 
	{
		loopDepth = a_body1cont9(rangeResult, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult && rangeResult,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(rangeResult), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state > 0) static_cast<QuickGetKeyValuesActor*>(this)->actor_wait_state = 0;
		static_cast<QuickGetKeyValuesActor*>(this)->ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StringRef prefix;
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Arena* a;
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetMappedKeyValuesRequest* pOriginalReq;
															#line 3815 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	GetRangeReqAndResultRef getRange;
															#line 3816 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double getValuesStart;
															#line 3870 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Transaction tr;
															#line 13089 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via quickGetKeyValues()
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class QuickGetKeyValuesActor final : public Actor<GetRangeReqAndResultRef>, public ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >, public ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >, public FastAllocated<QuickGetKeyValuesActor>, public QuickGetKeyValuesActorState<QuickGetKeyValuesActor> {
															#line 13094 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<QuickGetKeyValuesActor>::operator new;
	using FastAllocated<QuickGetKeyValuesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1946304717113292288UL, 16883351579704500480UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<GetRangeReqAndResultRef>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >;
friend struct ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >;
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	QuickGetKeyValuesActor(StorageServer* const& data,StringRef const& prefix,Version const& version,Arena* const& a,GetMappedKeyValuesRequest* const& pOriginalReq) 
															#line 13112 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<GetRangeReqAndResultRef>(),
		   QuickGetKeyValuesActorState<QuickGetKeyValuesActor>(data, prefix, version, a, pOriginalReq),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("quickGetKeyValues", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10745188868789411328UL, 14092707623558697728UL);
		ActorExecutionContextHelper __helper(static_cast<QuickGetKeyValuesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("quickGetKeyValues");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("quickGetKeyValues", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< QuickGetKeyValuesActor, 0, GetKeyValuesReply >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< QuickGetKeyValuesActor, 1, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<GetRangeReqAndResultRef> quickGetKeyValues( StorageServer* const& data, StringRef const& prefix, Version const& version, Arena* const& a, GetMappedKeyValuesRequest* const& pOriginalReq ) {
															#line 3808 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<GetRangeReqAndResultRef>(new QuickGetKeyValuesActor(data, prefix, version, a, pOriginalReq));
															#line 13146 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 3894 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

void unpackKeyTuple(Tuple** referenceTuple, Optional<Tuple>& keyTuple, KeyValueRef* keyValue) {
	if (!keyTuple.present()) {
		// May throw exception if the key is not parsable as a tuple.
		try {
			keyTuple = Tuple::unpack(keyValue->key);
		} catch (Error& e) {
			TraceEvent("KeyNotTuple").error(e).detail("Key", keyValue->key.printable());
			throw key_not_tuple();
		}
	}
	*referenceTuple = &keyTuple.get();
}

void unpackValueTuple(Tuple** referenceTuple, Optional<Tuple>& valueTuple, KeyValueRef* keyValue) {
	if (!valueTuple.present()) {
		// May throw exception if the value is not parsable as a tuple.
		try {
			valueTuple = Tuple::unpack(keyValue->value);
		} catch (Error& e) {
			TraceEvent("ValueNotTuple").error(e).detail("Value", keyValue->value.printable());
			throw value_not_tuple();
		}
	}
	*referenceTuple = &valueTuple.get();
}

bool unescapeLiterals(std::string& s, std::string before, std::string after) {
	bool escaped = false;
	size_t p = 0;
	while (true) {
		size_t found = s.find(before, p);
		if (found == std::string::npos) {
			break;
		}
		s.replace(found, before.length(), after);
		p = found + after.length();
		escaped = true;
	}
	return escaped;
}

bool singleKeyOrValue(const std::string& s, size_t sz) {
	// format would be {K[??]} or {V[??]}
	return sz > 5 && s[0] == '{' && (s[1] == 'K' || s[1] == 'V') && s[2] == '[' && s[sz - 2] == ']' && s[sz - 1] == '}';
}

bool rangeQuery(const std::string& s) {
	return s == "{...}";
}

// create a vector of Optional<Tuple>
// in case of a singleKeyOrValue, insert an empty Tuple to vector as placeholder
// in case of a rangeQuery, insert Optional.empty as placeholder
// in other cases, insert the correct Tuple to be used.
void preprocessMappedKey(Tuple& mappedKeyFormatTuple, std::vector<Optional<Tuple>>& vt, bool& isRangeQuery) {
	vt.reserve(mappedKeyFormatTuple.size());

	for (int i = 0; i < mappedKeyFormatTuple.size(); i++) {
		Tuple::ElementType type = mappedKeyFormatTuple.getType(i);
		if (type == Tuple::BYTES || type == Tuple::UTF8) {
			std::string s = mappedKeyFormatTuple.getString(i).toString();
			auto sz = s.size();
			bool escaped = unescapeLiterals(s, "{{", "{");
			escaped = unescapeLiterals(s, "}}", "}") || escaped;
			if (escaped) {
				vt.emplace_back(Tuple::makeTuple(s));
			} else if (singleKeyOrValue(s, sz)) {
				// when it is SingleKeyOrValue, insert an empty Tuple to vector as placeholder
				vt.emplace_back(Tuple());
			} else if (rangeQuery(s)) {
				if (i != mappedKeyFormatTuple.size() - 1) {
					// It must be the last element of the mapper tuple
					throw mapper_bad_range_decriptor();
				}
				// when it is rangeQuery, insert Optional.empty as placeholder
				vt.emplace_back(Optional<Tuple>());
				isRangeQuery = true;
			} else {
				Tuple t;
				t.appendRaw(mappedKeyFormatTuple.subTupleRawString(i));
				vt.emplace_back(t);
			}
		} else {
			Tuple t;
			t.appendRaw(mappedKeyFormatTuple.subTupleRawString(i));
			vt.emplace_back(t);
		}
	}
}

Key constructMappedKey(KeyValueRef* keyValue, std::vector<Optional<Tuple>>& vec, Tuple& mappedKeyFormatTuple) {
	// Lazily parse key and/or value to tuple because they may not need to be a tuple if not used.
	Optional<Tuple> keyTuple;
	Optional<Tuple> valueTuple;
	Tuple mappedKeyTuple;

	mappedKeyTuple.reserve(vec.size());

	for (int i = 0; i < vec.size(); i++) {
		if (!vec[i].present()) {
			// rangeQuery
			continue;
		}
		if (vec[i].get().size()) {
			mappedKeyTuple.append(vec[i].get());
		} else {
			// singleKeyOrValue is true
			std::string s = mappedKeyFormatTuple.getString(i).toString();
			auto sz = s.size();
			int idx;
			Tuple* referenceTuple;
			try {
				idx = std::stoi(s.substr(3, sz - 5));
			} catch (std::exception& e) {
				throw mapper_bad_index();
			}
			if (s[1] == 'K') {
				unpackKeyTuple(&referenceTuple, keyTuple, keyValue);
			} else if (s[1] == 'V') {
				unpackValueTuple(&referenceTuple, valueTuple, keyValue);
			} else {
				ASSERT(false);
				throw internal_error();
			}
			if (idx < 0 || idx >= referenceTuple->size()) {
				throw mapper_bad_index();
			}
			mappedKeyTuple.appendRaw(referenceTuple->subTupleRawString(idx));
		}
	}

	return mappedKeyTuple.pack();
}

struct AuditGetShardInfoRes {
	Version readAtVersion;
	UID serverId;
	std::vector<KeyRange> ownRanges;
	AuditGetShardInfoRes() = default;
	AuditGetShardInfoRes(Version readAtVersion, UID serverId, std::vector<KeyRange> ownRanges)
	  : readAtVersion(readAtVersion), serverId(serverId), ownRanges(ownRanges) {}
};

// Given an input server, get ranges with in the input range
// from the perspective of SS->shardInfo
// Input: (1) SS ID; (2) within range
// Return AuditGetShardInfoRes including: (1) version of the read; (2) ranges of the SS
AuditGetShardInfoRes getThisServerShardInfo(StorageServer* data, KeyRange range) {
	std::vector<KeyRange> ownRange;
	for (auto& t : data->shards.intersectingRanges(range)) {
		KeyRange alignedRange = t.value()->range() & range;
		if (alignedRange.empty()) {
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways,
			           "SSAuditStorageReadShardInfoEmptyAlignedRange",
			           data->thisServerID)
			    .detail("Range", range);
			throw audit_storage_cancelled();
		}
		TraceEvent(SevVerbose, "SSAuditStorageGetThisServerShardInfo", data->thisServerID)
		    .detail("AlignedRange", alignedRange)
		    .detail("Range", t.value()->range())
		    .detail("AtVersion", data->version.get())
		    .detail("AuditServer", data->thisServerID)
		    .detail("ReadWrite", t.value()->isReadable() ? "True" : "False")
		    .detail("Adding", t.value()->isFetching() ? "True" : "False");
		if (t.value()->assigned()) {
			ownRange.push_back(alignedRange);
		}
	}
	return AuditGetShardInfoRes(data->version.get(), data->thisServerID, ownRange);
}

// Check consistency between StorageServer->shardInfo and ServerKeys system key space
															#line 13324 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via auditStorageServerShardQ()
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class AuditStorageServerShardQActor>
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class AuditStorageServerShardQActorState {
															#line 13331 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageServerShardQActorState(StorageServer* const& data,AuditStorageRequest const& req) 
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 13340 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this));

	}
	~AuditStorageServerShardQActorState() 
	{
		fdb_probe_actor_destroy("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 4069 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(req.getType() == AuditType::ValidateStorageServerShard);
															#line 4070 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->serveAuditStorageParallelismLock.take(TaskPriority::DefaultYield);
															#line 4070 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13359 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 1;
															#line 4070 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 0, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13364 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AuditStorageServerShardQActorState();
		static_cast<AuditStorageServerShardQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 4075 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->trackShardAssignmentMinVersion != invalidVersion)
															#line 13387 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4077 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(audit_storage_cancelled());
															#line 4078 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "ExistStorageServerShardAuditExit") .detail("NewAuditId", req.id) .detail("NewAuditType", req.getType());
															#line 4082 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<AuditStorageServerShardQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageServerShardQActorState(); static_cast<AuditStorageServerShardQActor*>(this)->destroy(); return 0; }
															#line 13395 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<AuditStorageServerShardQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~AuditStorageServerShardQActorState();
			static_cast<AuditStorageServerShardQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 4084 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(data->serveAuditStorageParallelismLock);
															#line 4085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageSsShardBegin", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range);
															#line 4088 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		res = AuditStorageState(req.id, data->thisServerID, req.getType());
															#line 4089 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		errors = std::vector<std::string>();
															#line 4090 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs = std::vector<Future<Void>>();
															#line 4091 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		tr = Transaction(data->cx);
															#line 4092 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyRes = AuditGetServerKeysRes();
															#line 4093 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = Version();
															#line 4094 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = KeyRange();
															#line 4095 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerRes = AuditGetKeyServersRes();
															#line 4096 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = Version();
															#line 4097 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = KeyRange();
															#line 4098 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesLocalViewRes = AuditGetShardInfoRes();
															#line 4099 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localShardInfoReadAtVersion = Version();
															#line 4102 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByKeyServerMap = std::unordered_map<UID, std::vector<KeyRange>>();
															#line 4103 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByServerKey = std::vector<KeyRange>();
															#line 4104 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByKeyServer = std::vector<KeyRange>();
															#line 4105 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesLocalView = std::vector<KeyRange>();
															#line 4106 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		failureReason = std::string();
															#line 4112 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = KeyRange();
															#line 4113 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToReadBegin = req.range.begin;
															#line 4114 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToRead = KeyRangeRef();
															#line 4115 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		retryCount = 0;
															#line 4116 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedLocalShardsNum = 0;
															#line 4117 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedServerKeysNum = 0;
															#line 4118 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiter = Reference<IRateControl>(new SpeedLimit(SERVER_KNOBS->AUDIT_STORAGE_RATE_PER_SERVER_MAX, 1));
															#line 4120 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = 0;
															#line 4121 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		startTime = now();
															#line 4122 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = 0;
															#line 4123 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = 0;
															#line 4124 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = 0;
															#line 13463 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 4127 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 13467 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 4075 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (data->trackShardAssignmentMinVersion != invalidVersion)
															#line 13482 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4077 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.sendError(audit_storage_cancelled());
															#line 4078 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "ExistStorageServerShardAuditExit") .detail("NewAuditId", req.id) .detail("NewAuditType", req.getType());
															#line 4082 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!static_cast<AuditStorageServerShardQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageServerShardQActorState(); static_cast<AuditStorageServerShardQActor*>(this)->destroy(); return 0; }
															#line 13490 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			new (&static_cast<AuditStorageServerShardQActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~AuditStorageServerShardQActorState();
			static_cast<AuditStorageServerShardQActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 4084 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(data->serveAuditStorageParallelismLock);
															#line 4085 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageSsShardBegin", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range);
															#line 4088 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		res = AuditStorageState(req.id, data->thisServerID, req.getType());
															#line 4089 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		errors = std::vector<std::string>();
															#line 4090 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs = std::vector<Future<Void>>();
															#line 4091 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		tr = Transaction(data->cx);
															#line 4092 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyRes = AuditGetServerKeysRes();
															#line 4093 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = Version();
															#line 4094 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = KeyRange();
															#line 4095 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerRes = AuditGetKeyServersRes();
															#line 4096 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = Version();
															#line 4097 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = KeyRange();
															#line 4098 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesLocalViewRes = AuditGetShardInfoRes();
															#line 4099 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localShardInfoReadAtVersion = Version();
															#line 4102 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByKeyServerMap = std::unordered_map<UID, std::vector<KeyRange>>();
															#line 4103 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByServerKey = std::vector<KeyRange>();
															#line 4104 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesSeenByKeyServer = std::vector<KeyRange>();
															#line 4105 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		ownRangesLocalView = std::vector<KeyRange>();
															#line 4106 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		failureReason = std::string();
															#line 4112 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = KeyRange();
															#line 4113 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToReadBegin = req.range.begin;
															#line 4114 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToRead = KeyRangeRef();
															#line 4115 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		retryCount = 0;
															#line 4116 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedLocalShardsNum = 0;
															#line 4117 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedServerKeysNum = 0;
															#line 4118 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiter = Reference<IRateControl>(new SpeedLimit(SERVER_KNOBS->AUDIT_STORAGE_RATE_PER_SERVER_MAX, 1));
															#line 4120 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = 0;
															#line 4121 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		startTime = now();
															#line 4122 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = 0;
															#line 4123 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = 0;
															#line 4124 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = 0;
															#line 13558 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 4127 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 13562 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 4482 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->stopTrackShardAssignment();
															#line 4483 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevVerbose, "SSShardAssignmentHistoryRecordStopWhenExit", data->thisServerID).detail("AuditID", req.id);
															#line 4485 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<AuditStorageServerShardQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageServerShardQActorState(); static_cast<AuditStorageServerShardQActor*>(this)->destroy(); return 0; }
															#line 13656 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<AuditStorageServerShardQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AuditStorageServerShardQActorState();
		static_cast<AuditStorageServerShardQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4456 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 13669 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4457 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<AuditStorageServerShardQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageServerShardQActorState(); static_cast<AuditStorageServerShardQActor*>(this)->destroy(); return 0; }
															#line 13673 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<AuditStorageServerShardQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~AuditStorageServerShardQActorState();
				static_cast<AuditStorageServerShardQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 4459 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageSsShardFailed", data->thisServerID) .errorUnsuppressed(e) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("Reason", failureReason) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime) .detail("TotalTime", now() - startTime);
															#line 4468 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->stopTrackShardAssignment();
															#line 4469 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevVerbose, "SSShardAssignmentHistoryRecordStopWhenError", data->thisServerID) .detail("AuditID", req.id);
															#line 4472 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_audit_storage_cancelled)
															#line 13687 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4473 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(audit_storage_cancelled());
															#line 13691 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 4474 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (e.code() == error_code_audit_storage_task_outdated)
															#line 13697 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4475 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(audit_storage_task_outdated());
															#line 13701 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 4477 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(audit_storage_failed());
															#line 13707 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 4129 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (data->version.get() == 0)
															#line 13738 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4130 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				failureReason = "SS version is 0";
															#line 4131 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_failed(), loopDepth);
															#line 13744 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4135 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			errors.clear();
															#line 4137 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesLocalView.clear();
															#line 4138 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesSeenByServerKey.clear();
															#line 4139 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesSeenByKeyServer.clear();
															#line 4140 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesSeenByKeyServerMap.clear();
															#line 4141 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeToRead = KeyRangeRef(rangeToReadBegin, req.range.end);
															#line 4143 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(!rangeToRead.empty());
															#line 4147 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesLocalViewRes = getThisServerShardInfo(data, rangeToRead);
															#line 4148 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			localShardInfoReadAtVersion = ownRangesLocalViewRes.readAtVersion;
															#line 4149 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (localShardInfoReadAtVersion != data->version.get())
															#line 13766 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4150 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardGRVMismatchError", data->thisServerID);
															#line 4153 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 13772 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4157 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->startTrackShardAssignment(localShardInfoReadAtVersion);
															#line 4158 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevVerbose, "SSShardAssignmentHistoryRecordStart", data->thisServerID) .detail("AuditID", req.id);
															#line 4162 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 4163 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 4164 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.clear();
															#line 4165 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.push_back( store(serverKeyRes, getThisServerKeysFromServerKeys(data->thisServerID, &tr, rangeToRead)));
															#line 4167 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.push_back(store(keyServerRes, getShardMapFromKeyServers(data->thisServerID, &tr, rangeToRead)));
															#line 4168 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = waitForAll(fs);
															#line 4168 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13792 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 2;
															#line 4168 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 1, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13797 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 4450 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = now();
															#line 4451 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_9 = rateLimiter->getAllowance(remoteReadBytes);
															#line 4451 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13829 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont1Catch1(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_9.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 10;
															#line 4451 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 9, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13834 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4439 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 13844 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4444 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1Catch1(e, std::max(0, loopDepth - 1));
															#line 13848 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4446 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			data->stopTrackShardAssignment();
															#line 4447 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_8 = tr.onError(e);
															#line 4447 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13856 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_8.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 9;
															#line 4447 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 8, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 13861 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 4170 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = serverKeyRes.completeRange;
															#line 4171 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = serverKeyRes.readAtVersion;
															#line 4173 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = keyServerRes.completeRange;
															#line 4174 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = keyServerRes.readAtVersion;
															#line 4176 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = keyServerRes.readBytes + serverKeyRes.readBytes;
															#line 4178 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 13886 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 4170 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = serverKeyRes.completeRange;
															#line 4171 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = serverKeyRes.readAtVersion;
															#line 4173 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = keyServerRes.completeRange;
															#line 4174 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = keyServerRes.readAtVersion;
															#line 4176 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = keyServerRes.readBytes + serverKeyRes.readBytes;
															#line 4178 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		;
															#line 13905 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
															#line 4204 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (serverKeyReadAtVersion < localShardInfoReadAtVersion)
															#line 13989 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4205 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardComparedVersionError", data->thisServerID);
															#line 4208 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 13995 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4210 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (keyServerReadAtVersion != serverKeyReadAtVersion)
															#line 13999 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4211 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardKSVersionMismatchError", data->thisServerID);
															#line 4214 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 14005 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		try {
															#line 4217 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_5 = timeoutError(data->version.whenAtLeast(serverKeyReadAtVersion), 30);
															#line 4217 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1cont5Catch1(actor_cancelled(), loopDepth);
															#line 14012 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1cont5Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_5.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 6;
															#line 4217 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 5, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14017 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1cont5Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1cont5Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 4178 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!(serverKeyReadAtVersion < localShardInfoReadAtVersion))
															#line 14039 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
			return a_body1cont1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 4179 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (retryCount >= SERVER_KNOBS->AUDIT_RETRY_COUNT_MAX)
															#line 14045 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4180 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			failureReason = "Read serverKeys retry count exceeds the max";
															#line 4181 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_failed(), std::max(0, loopDepth - 1));
															#line 14051 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4183 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = rateLimiter->getAllowance(remoteReadBytes);
															#line 4183 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14057 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 3;
															#line 4183 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 2, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14062 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 4184 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		retryCount++;
															#line 4185 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(0.5);
															#line 4185 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14088 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 4;
															#line 4185 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 3, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14093 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 4184 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		retryCount++;
															#line 4185 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(0.5);
															#line 4185 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14106 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2loopBody1cont1when1(__when_expr_3.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 4;
															#line 4185 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 3, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14111 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont2loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 4186 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.reset();
															#line 4187 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 4188 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 4189 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.clear();
															#line 4190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back( store(serverKeyRes, getThisServerKeysFromServerKeys(data->thisServerID, &tr, rangeToRead)));
															#line 4192 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back(store(keyServerRes, getShardMapFromKeyServers(data->thisServerID, &tr, rangeToRead)));
															#line 4193 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_4 = waitForAll(fs);
															#line 4193 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14209 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 5;
															#line 4193 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 4, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14214 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 4186 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.reset();
															#line 4187 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 4188 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 4189 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.clear();
															#line 4190 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back( store(serverKeyRes, getThisServerKeysFromServerKeys(data->thisServerID, &tr, rangeToRead)));
															#line 4192 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back(store(keyServerRes, getShardMapFromKeyServers(data->thisServerID, &tr, rangeToRead)));
															#line 4193 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_4 = waitForAll(fs);
															#line 4193 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14237 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 5;
															#line 4193 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 4, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14242 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont2loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 4195 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = serverKeyRes.completeRange;
															#line 4196 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = serverKeyRes.readAtVersion;
															#line 4198 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = keyServerRes.completeRange;
															#line 4199 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = keyServerRes.readAtVersion;
															#line 4201 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = keyServerRes.readBytes + serverKeyRes.readBytes;
															#line 14334 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 4195 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyCompleteRange = serverKeyRes.completeRange;
															#line 4196 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverKeyReadAtVersion = serverKeyRes.readAtVersion;
															#line 4198 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerCompleteRange = keyServerRes.completeRange;
															#line 4199 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		keyServerReadAtVersion = keyServerRes.readAtVersion;
															#line 4201 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		remoteReadBytes = keyServerRes.readBytes + serverKeyRes.readBytes;
															#line 14351 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont2loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont2loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont6(int loopDepth) 
	{
															#line 4229 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->stopTrackShardAssignment();
															#line 4230 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevVerbose, "ShardAssignmentHistoryRecordStop", data->thisServerID) .detail("AuditID", req.id);
															#line 4234 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		std::vector<std::pair<Version, KeyRangeRef>> shardAssignments = data->getShardAssignmentHistory(localShardInfoReadAtVersion, serverKeyReadAtVersion);
															#line 4236 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageSsShardGetHistory", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ServerKeyAtVersion", serverKeyReadAtVersion) .detail("LocalShardInfoAtVersion", localShardInfoReadAtVersion) .detail("ShardAssignmentsCount", shardAssignments.size());
															#line 4244 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!shardAssignments.empty())
															#line 14443 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4245 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			failureReason = "Shard assignment history is not empty";
															#line 4246 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_failed(), loopDepth);
															#line 14449 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4249 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		KeyRange claimRange = rangeToRead;
															#line 4250 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = claimRange & serverKeyCompleteRange;
															#line 4251 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (claimRange.empty())
															#line 14457 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4252 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardOverlapRangeEmpty", data->thisServerID);
															#line 4255 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 14463 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4257 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = claimRange & keyServerCompleteRange;
															#line 4258 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (claimRange.empty())
															#line 14469 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4259 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardOverlapRangeEmpty", data->thisServerID);
															#line 4262 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 14475 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4266 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto& range : ownRangesLocalViewRes.ownRanges ) {
															#line 4267 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange overlappingRange = range & claimRange;
															#line 4268 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (overlappingRange.empty())
															#line 14483 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
				continue;
			}
															#line 4271 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesLocalView.push_back(overlappingRange);
															#line 14489 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4274 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( auto& range : serverKeyRes.ownRanges ) {
															#line 4275 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange overlappingRange = range & claimRange;
															#line 4276 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (overlappingRange.empty())
															#line 14497 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
				continue;
			}
															#line 4279 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ownRangesSeenByServerKey.push_back(overlappingRange);
															#line 14503 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4282 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (keyServerRes.rangeOwnershipMap.contains(data->thisServerID))
															#line 14507 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4283 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::vector mergedRanges = coalesceRangeList(keyServerRes.rangeOwnershipMap[data->thisServerID]);
															#line 4284 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for( auto& range : mergedRanges ) {
															#line 4285 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyRange overlappingRange = range & claimRange;
															#line 4286 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (overlappingRange.empty())
															#line 14517 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
					continue;
				}
															#line 4289 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				ownRangesSeenByKeyServer.push_back(overlappingRange);
															#line 14523 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 4292 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageSsShardReadDone", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ClaimRange", claimRange) .detail("ServerKeyAtVersion", serverKeyReadAtVersion) .detail("ShardInfoAtVersion", data->version.get());
															#line 4299 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedLocalShardsNum = cumulatedValidatedLocalShardsNum + ownRangesLocalView.size();
															#line 4300 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		cumulatedValidatedServerKeysNum = cumulatedValidatedServerKeysNum + ownRangesSeenByServerKey.size();
															#line 4301 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageStatisticShardInfo", data->thisServerID) .suppressFor(30.0) .detail("AuditType", req.getType()) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("CurrentValidatedLocalShardsNum", ownRangesLocalView.size()) .detail("CurrentValidatedServerKeysNum", ownRangesSeenByServerKey.size()) .detail("CurrentValidatedInclusiveRange", claimRange) .detail("CumulatedValidatedLocalShardsNum", cumulatedValidatedLocalShardsNum) .detail("CumulatedValidatedServerKeysNum", cumulatedValidatedServerKeysNum) .detail("CumulatedValidatedInclusiveRange", KeyRangeRef(req.range.begin, claimRange.end));
															#line 4315 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (ownRangesSeenByKeyServer.empty())
															#line 14536 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4316 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!ownRangesSeenByServerKey.empty())
															#line 14540 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4317 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				std::string error = format("ServerKeys shows %zu ranges that not appear on keyServers for Server(%s): ", ownRangesSeenByServerKey.size(), data->thisServerID.toString().c_str(), describe(ownRangesSeenByServerKey).c_str());
															#line 4322 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevError, "SSAuditStorageSsShardError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ClaimRange", claimRange) .detail("ErrorMessage", error) .detail("MismatchedRangeByLocalView", describe(ownRangesSeenByServerKey)) .detail("AuditServer", data->thisServerID);
															#line 14546 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
		else
		{
															#line 4333 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Optional<std::pair<KeyRange, KeyRange>> anyMismatch = rangesSame(ownRangesSeenByServerKey, ownRangesSeenByKeyServer);
															#line 4335 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (anyMismatch.present())
															#line 14555 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4336 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyRange mismatchedRangeByServerKey = anyMismatch.get().first;
															#line 4337 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyRange mismatchedRangeByKeyServer = anyMismatch.get().second;
															#line 4338 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				std::string error = format("KeyServers and serverKeys mismatch on Server(%s): ServerKey: %s; KeyServer: %s", data->thisServerID.toString().c_str(), mismatchedRangeByServerKey.toString().c_str(), mismatchedRangeByKeyServer.toString().c_str());
															#line 4343 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevError, "SSAuditStorageSsShardError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ClaimRange", claimRange) .detail("ErrorMessage", error) .detail("MismatchedRangeByKeyServer", mismatchedRangeByKeyServer) .detail("MismatchedRangeByServerKey", mismatchedRangeByServerKey) .detail("AuditServer", data->thisServerID);
															#line 4353 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				errors.push_back(error);
															#line 14567 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 4358 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		Optional<std::pair<KeyRange, KeyRange>> anyMismatch = rangesSame(ownRangesSeenByServerKey, ownRangesLocalView);
															#line 4360 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (anyMismatch.present())
															#line 14574 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4361 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange mismatchedRangeByServerKey = anyMismatch.get().first;
															#line 4362 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange mismatchedRangeByLocalView = anyMismatch.get().second;
															#line 4363 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::string error = format("Storage server shard info mismatch on Server(%s): ServerKey: %s; ServerShardInfo: %s", data->thisServerID.toString().c_str(), mismatchedRangeByServerKey.toString().c_str(), mismatchedRangeByLocalView.toString().c_str());
															#line 4368 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevError, "SSAuditStorageSsShardError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ClaimRange", claimRange) .detail("ErrorMessage", error) .detail("MismatchedRangeByLocalView", mismatchedRangeByLocalView) .detail("MismatchedRangeByServerKey", mismatchedRangeByServerKey) .detail("AuditServer", data->thisServerID);
															#line 4378 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			errors.push_back(error);
															#line 14586 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4382 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!errors.empty())
															#line 14590 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4383 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevVerbose, "SSAuditStorageSsShardErrorEnd", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID);
															#line 4387 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.range = claimRange;
															#line 4388 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.setPhase(AuditPhase::Error);
															#line 4389 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!req.ddId.isValid())
															#line 14600 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4390 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardDDIdInvalid", data->thisServerID);
															#line 4393 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 14606 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4395 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.ddId = req.ddId;
															#line 4396 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_6 = persistAuditStateByServer(data->cx, res);
															#line 4396 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14614 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1loopBody1cont6when1(__when_expr_6.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 7;
															#line 4396 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 6, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14619 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 4401 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.range = Standalone(KeyRangeRef(req.range.begin, claimRange.end));
															#line 4402 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.setPhase(AuditPhase::Complete);
															#line 4403 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!req.ddId.isValid())
															#line 14630 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4404 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageSsShardDDIdInvalid", data->thisServerID);
															#line 4407 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 14636 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4409 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.ddId = req.ddId;
															#line 4410 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_7 = persistAuditStateByServer(data->cx, res);
															#line 4410 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14644 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont1loopBody1cont6when2(__when_expr_7.get(), loopDepth); };
			static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 8;
															#line 4410 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< AuditStorageServerShardQActor, 7, Void >*>(static_cast<AuditStorageServerShardQActor*>(this)));
															#line 14649 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4219 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "SSAuditStorageSsShardWaitSSVersionTooLong", data->thisServerID) .detail("ServerKeyReadAtVersion", serverKeyReadAtVersion) .detail("SSVersion", data->version.get());
															#line 4222 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			failureReason = "SS version takes long time to catch up with serverKeyReadAtVersion";
															#line 4223 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_failed(), loopDepth);
															#line 14664 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1cont5Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1cont5Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont1loopBody1cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont27(Void const& _,int loopDepth) 
	{
															#line 4397 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.sendError(audit_storage_error());
															#line 14784 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont27(Void && _,int loopDepth) 
	{
															#line 4397 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.sendError(audit_storage_error());
															#line 14793 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont27(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont27(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont1loopBody1cont29(Void const& _,int loopDepth) 
	{
															#line 4411 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (res.range.end < req.range.end)
															#line 14877 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4412 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageSsShardPartialDone", data->thisServerID) .suppressFor(10.0) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("CompleteRange", res.range) .detail("ClaimRange", claimRange) .detail("RangeToReadEnd", req.range.end) .detail("LastRateLimiterWaitTime", lastRateLimiterWaitTime) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime);
															#line 4422 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeToReadBegin = res.range.end;
															#line 14883 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 4424 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(res);
															#line 4425 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageSsShardComplete", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("ClaimRange", claimRange) .detail("CompleteRange", res.range) .detail("NumValidatedLocalShards", cumulatedValidatedLocalShardsNum) .detail("NumValidatedServerKeys", cumulatedValidatedServerKeysNum) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime) .detail("TotalTime", now() - startTime);
															#line 14891 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1cont1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont29(Void && _,int loopDepth) 
	{
															#line 4411 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (res.range.end < req.range.end)
															#line 14902 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4412 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageSsShardPartialDone", data->thisServerID) .suppressFor(10.0) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("CompleteRange", res.range) .detail("ClaimRange", claimRange) .detail("RangeToReadEnd", req.range.end) .detail("LastRateLimiterWaitTime", lastRateLimiterWaitTime) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime);
															#line 4422 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeToReadBegin = res.range.end;
															#line 14908 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		else
		{
															#line 4424 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(res);
															#line 4425 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageSsShardComplete", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("ClaimRange", claimRange) .detail("CompleteRange", res.range) .detail("NumValidatedLocalShards", cumulatedValidatedLocalShardsNum) .detail("NumValidatedServerKeys", cumulatedValidatedServerKeysNum) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime) .detail("TotalTime", now() - startTime);
															#line 14916 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1cont1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont29(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont29(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont6when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont6when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont1loopBody1cont11cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont1loopBody1cont31(Void const& _,int loopDepth) 
	{
															#line 4452 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = now() - rateLimiterBeforeWaitTime;
															#line 4453 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = rateLimiterTotalWaitTime + lastRateLimiterWaitTime;
															#line 15104 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont31(Void && _,int loopDepth) 
	{
															#line 4452 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = now() - rateLimiterBeforeWaitTime;
															#line 4453 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = rateLimiterTotalWaitTime + lastRateLimiterWaitTime;
															#line 15115 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont31(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont31(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageServerShardQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageServerShardQActor*>(this)->ActorCallback< AuditStorageServerShardQActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< AuditStorageServerShardQActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< AuditStorageServerShardQActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageRequest req;
															#line 4084 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FlowLock::Releaser holder;
															#line 4088 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageState res;
															#line 4089 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<std::string> errors;
															#line 4090 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<Future<Void>> fs;
															#line 4091 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Transaction tr;
															#line 4092 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditGetServerKeysRes serverKeyRes;
															#line 4093 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version serverKeyReadAtVersion;
															#line 4094 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange serverKeyCompleteRange;
															#line 4095 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditGetKeyServersRes keyServerRes;
															#line 4096 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version keyServerReadAtVersion;
															#line 4097 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange keyServerCompleteRange;
															#line 4098 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditGetShardInfoRes ownRangesLocalViewRes;
															#line 4099 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version localShardInfoReadAtVersion;
															#line 4102 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::unordered_map<UID, std::vector<KeyRange>> ownRangesSeenByKeyServerMap;
															#line 4103 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<KeyRange> ownRangesSeenByServerKey;
															#line 4104 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<KeyRange> ownRangesSeenByKeyServer;
															#line 4105 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<KeyRange> ownRangesLocalView;
															#line 4106 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::string failureReason;
															#line 4112 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange claimRange;
															#line 4113 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key rangeToReadBegin;
															#line 4114 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRangeRef rangeToRead;
															#line 4115 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int retryCount;
															#line 4116 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t cumulatedValidatedLocalShardsNum;
															#line 4117 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t cumulatedValidatedServerKeysNum;
															#line 4118 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Reference<IRateControl> rateLimiter;
															#line 4120 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t remoteReadBytes;
															#line 4121 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double startTime;
															#line 4122 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double lastRateLimiterWaitTime;
															#line 4123 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double rateLimiterBeforeWaitTime;
															#line 4124 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double rateLimiterTotalWaitTime;
															#line 15272 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via auditStorageServerShardQ()
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class AuditStorageServerShardQActor final : public Actor<Void>, public ActorCallback< AuditStorageServerShardQActor, 0, Void >, public ActorCallback< AuditStorageServerShardQActor, 1, Void >, public ActorCallback< AuditStorageServerShardQActor, 2, Void >, public ActorCallback< AuditStorageServerShardQActor, 3, Void >, public ActorCallback< AuditStorageServerShardQActor, 4, Void >, public ActorCallback< AuditStorageServerShardQActor, 5, Void >, public ActorCallback< AuditStorageServerShardQActor, 6, Void >, public ActorCallback< AuditStorageServerShardQActor, 7, Void >, public ActorCallback< AuditStorageServerShardQActor, 8, Void >, public ActorCallback< AuditStorageServerShardQActor, 9, Void >, public FastAllocated<AuditStorageServerShardQActor>, public AuditStorageServerShardQActorState<AuditStorageServerShardQActor> {
															#line 15277 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<AuditStorageServerShardQActor>::operator new;
	using FastAllocated<AuditStorageServerShardQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5948118628840067072UL, 3548405170864472832UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AuditStorageServerShardQActor, 0, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 1, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 2, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 3, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 4, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 5, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 6, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 7, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 8, Void >;
friend struct ActorCallback< AuditStorageServerShardQActor, 9, Void >;
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageServerShardQActor(StorageServer* const& data,AuditStorageRequest const& req) 
															#line 15303 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   AuditStorageServerShardQActorState<AuditStorageServerShardQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10868456401308189184UL, 5217512031670601728UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageServerShardQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("auditStorageServerShardQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("auditStorageServerShardQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< AuditStorageServerShardQActor, 9, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> auditStorageServerShardQ( StorageServer* const& data, AuditStorageRequest const& req ) {
															#line 4068 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new AuditStorageServerShardQActor(data, req));
															#line 15345 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 4487 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 15350 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via auditStorageShardReplicaQ()
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
template <class AuditStorageShardReplicaQActor>
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class AuditStorageShardReplicaQActorState {
															#line 15357 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageShardReplicaQActorState(StorageServer* const& data,AuditStorageRequest const& req) 
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		 : data(data),
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		   req(req)
															#line 15366 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
	{
		fdb_probe_actor_create("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this));

	}
	~AuditStorageShardReplicaQActorState() 
	{
		fdb_probe_actor_destroy("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 4489 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(req.getType() == AuditType::ValidateHA || req.getType() == AuditType::ValidateReplica);
															#line 4490 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_0 = data->serveAuditStorageParallelismLock.take(TaskPriority::DefaultYield);
															#line 4490 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 15385 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 1;
															#line 4490 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 0, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15390 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AuditStorageShardReplicaQActorState();
		static_cast<AuditStorageShardReplicaQActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 4491 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(data->serveAuditStorageParallelismLock);
															#line 4493 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageShardReplicaBegin", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("TargetServers", describe(req.targetServers));
															#line 4499 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		res = AuditStorageState(req.id, req.getType());
															#line 4500 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverListValues = std::vector<Optional<Value>>();
															#line 4501 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs = std::vector<Future<ErrorOr<GetKeyValuesReply>>>();
															#line 4502 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		errors = std::vector<std::string>();
															#line 4503 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		version = Version();
															#line 4504 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		tr = Transaction(data->cx);
															#line 4505 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToRead = req.range;
															#line 4506 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToReadBegin = req.range.begin;
															#line 4507 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = KeyRange();
															#line 4508 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limit = 1e4;
															#line 4509 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limitBytes = CLIENT_KNOBS->REPLY_BYTE_LIMIT;
															#line 4510 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readBytes = 0;
															#line 4511 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		numValidatedKeys = 0;
															#line 4512 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		validatedBytes = 0;
															#line 4513 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		complete = false;
															#line 4514 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		checkTimes = 0;
															#line 4515 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		startTime = now();
															#line 4516 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = 0;
															#line 4517 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = 0;
															#line 4518 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = 0;
															#line 4519 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiter = Reference<IRateControl>(new SpeedLimit(SERVER_KNOBS->AUDIT_STORAGE_RATE_PER_SERVER_MAX, 1));
															#line 15457 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 4522 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 15461 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 4491 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		holder = FlowLock::Releaser(data->serveAuditStorageParallelismLock);
															#line 4493 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageShardReplicaBegin", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("TargetServers", describe(req.targetServers));
															#line 4499 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		res = AuditStorageState(req.id, req.getType());
															#line 4500 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		serverListValues = std::vector<Optional<Value>>();
															#line 4501 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs = std::vector<Future<ErrorOr<GetKeyValuesReply>>>();
															#line 4502 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		errors = std::vector<std::string>();
															#line 4503 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		version = Version();
															#line 4504 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		tr = Transaction(data->cx);
															#line 4505 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToRead = req.range;
															#line 4506 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rangeToReadBegin = req.range.begin;
															#line 4507 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = KeyRange();
															#line 4508 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limit = 1e4;
															#line 4509 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		limitBytes = CLIENT_KNOBS->REPLY_BYTE_LIMIT;
															#line 4510 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		readBytes = 0;
															#line 4511 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		numValidatedKeys = 0;
															#line 4512 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		validatedBytes = 0;
															#line 4513 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		complete = false;
															#line 4514 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		checkTimes = 0;
															#line 4515 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		startTime = now();
															#line 4516 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = 0;
															#line 4517 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = 0;
															#line 4518 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = 0;
															#line 4519 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiter = Reference<IRateControl>(new SpeedLimit(SERVER_KNOBS->AUDIT_STORAGE_RATE_PER_SERVER_MAX, 1));
															#line 15520 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		try {
															#line 4522 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			;
															#line 15524 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 4897 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!static_cast<AuditStorageShardReplicaQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageShardReplicaQActorState(); static_cast<AuditStorageShardReplicaQActor*>(this)->destroy(); return 0; }
															#line 15614 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		new (&static_cast<AuditStorageShardReplicaQActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AuditStorageShardReplicaQActorState();
		static_cast<AuditStorageShardReplicaQActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4878 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 15627 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4879 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!static_cast<AuditStorageShardReplicaQActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AuditStorageShardReplicaQActorState(); static_cast<AuditStorageShardReplicaQActor*>(this)->destroy(); return 0; }
															#line 15631 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				new (&static_cast<AuditStorageShardReplicaQActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~AuditStorageShardReplicaQActorState();
				static_cast<AuditStorageShardReplicaQActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 4881 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageShardReplicaFailed", data->thisServerID) .errorUnsuppressed(e) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime) .detail("TotalTime", now() - startTime);
															#line 4888 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (e.code() == error_code_audit_storage_cancelled)
															#line 15641 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4889 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				req.reply.sendError(audit_storage_cancelled());
															#line 15645 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
			else
			{
															#line 4890 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (e.code() == error_code_audit_storage_task_outdated)
															#line 15651 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4891 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(audit_storage_task_outdated());
															#line 15655 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 4893 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					req.reply.sendError(audit_storage_failed());
															#line 15661 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 4524 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			readBytes = 0;
															#line 4525 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeToRead = KeyRangeRef(rangeToReadBegin, req.range.end);
															#line 4526 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			ASSERT(!rangeToRead.empty());
															#line 4528 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevDebug, "SSAuditStorageShardReplicaNewRoundBegin", data->thisServerID) .suppressFor(10.0) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("ReadRangeBegin", rangeToReadBegin) .detail("ReadRangeEnd", req.range.end);
															#line 4535 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			serverListValues.clear();
															#line 4536 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			errors.clear();
															#line 4537 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.clear();
															#line 4538 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.reset();
															#line 4539 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 4540 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 4541 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 4544 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::vector<Future<Optional<Value>>> serverListEntries;
															#line 4545 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for( const UID& id : req.targetServers ) {
															#line 4546 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (id != data->thisServerID)
															#line 15718 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4547 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					serverListEntries.push_back(tr.get(serverListKeyFor(id)));
															#line 15722 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
			}
															#line 4550 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = store(serverListValues, getAll(serverListEntries));
															#line 4550 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15729 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 2;
															#line 4550 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 1, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15734 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 4870 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterBeforeWaitTime = now();
															#line 4871 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_7 = rateLimiter->getAllowance(readBytes);
															#line 4871 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15766 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_7.get(), loopDepth); };
		static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 8;
															#line 4871 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 7, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15771 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 4867 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_6 = tr.onError(e);
															#line 4867 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15783 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_6.get(), loopDepth); };
			static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 7;
															#line 4867 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 6, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15788 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 4553 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(version, tr.getReadVersion());
															#line 4553 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15805 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 3;
															#line 4553 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15810 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 4553 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(version, tr.getReadVersion());
															#line 4553 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15821 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 3;
															#line 4553 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15826 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 4556 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( const auto& v : serverListValues ) {
															#line 4557 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!v.present())
															#line 15912 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4558 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaRemoteServerNotFound", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type);
															#line 4562 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_failed(), loopDepth);
															#line 15918 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4564 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StorageServerInterface remoteServer = decodeServerListValue(v.get());
															#line 4565 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			GetKeyValuesRequest req;
															#line 4566 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.begin = firstGreaterOrEqual(rangeToRead.begin);
															#line 4567 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.end = firstGreaterOrEqual(rangeToRead.end);
															#line 4568 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.limit = limit;
															#line 4569 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.limitBytes = limitBytes;
															#line 4570 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.version = version;
															#line 4571 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.tags = TagSet();
															#line 4572 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.push_back(remoteServer.getKeyValues.getReplyUnlessFailedFor(req, 2, 0));
															#line 15938 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		GetKeyValuesRequest localReq;
															#line 4577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.begin = firstGreaterOrEqual(rangeToRead.begin);
															#line 4578 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.end = firstGreaterOrEqual(rangeToRead.end);
															#line 4579 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.limit = limit;
															#line 4580 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.limitBytes = limitBytes;
															#line 4581 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.version = version;
															#line 4582 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.tags = TagSet();
															#line 4583 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->actors.add(getKeyValuesQ(data, localReq));
															#line 4584 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back(errorOr(localReq.reply.getFuture()));
															#line 4585 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<std::vector<ErrorOr<GetKeyValuesReply>>> __when_expr_3 = getAll(fs);
															#line 4585 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15962 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 4;
															#line 4585 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 15967 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 4556 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for( const auto& v : serverListValues ) {
															#line 4557 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!v.present())
															#line 15978 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4558 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaRemoteServerNotFound", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type);
															#line 4562 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_failed(), loopDepth);
															#line 15984 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4564 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StorageServerInterface remoteServer = decodeServerListValue(v.get());
															#line 4565 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			GetKeyValuesRequest req;
															#line 4566 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.begin = firstGreaterOrEqual(rangeToRead.begin);
															#line 4567 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.end = firstGreaterOrEqual(rangeToRead.end);
															#line 4568 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.limit = limit;
															#line 4569 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.limitBytes = limitBytes;
															#line 4570 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.version = version;
															#line 4571 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.tags = TagSet();
															#line 4572 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			fs.push_back(remoteServer.getKeyValues.getReplyUnlessFailedFor(req, 2, 0));
															#line 16004 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4576 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		GetKeyValuesRequest localReq;
															#line 4577 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.begin = firstGreaterOrEqual(rangeToRead.begin);
															#line 4578 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.end = firstGreaterOrEqual(rangeToRead.end);
															#line 4579 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.limit = limit;
															#line 4580 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.limitBytes = limitBytes;
															#line 4581 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.version = version;
															#line 4582 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		localReq.tags = TagSet();
															#line 4583 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		data->actors.add(getKeyValuesQ(data, localReq));
															#line 4584 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		fs.push_back(errorOr(localReq.reply.getFuture()));
															#line 4585 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		StrictFuture<std::vector<ErrorOr<GetKeyValuesReply>>> __when_expr_3 = getAll(fs);
															#line 4585 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16028 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 4;
															#line 4585 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 16033 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont6(std::vector<ErrorOr<GetKeyValuesReply>> const& reps,int loopDepth) 
	{
															#line 4589 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < reps.size();++i) {
															#line 4590 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (reps[i].isError())
															#line 16119 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4591 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaGetKeyValuesError", data->thisServerID) .errorUnsuppressed(reps[i].getError()) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("ReplyIndex", i) .detail("RangeRead", rangeToRead) .detail("TargetServers", describe(req.targetServers));
															#line 4599 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(reps[i].getError(), loopDepth);
															#line 16125 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4601 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (reps[i].get().error.present())
															#line 16129 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4602 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaGetKeyValuesError", data->thisServerID) .errorUnsuppressed(reps[i].get().error.get()) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("ReplyIndex", i) .detail("RangeRead", rangeToRead) .detail("TargetServers", describe(req.targetServers));
															#line 4610 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(reps[i].get().error.get(), loopDepth);
															#line 16135 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4612 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			readBytes = readBytes + reps[i].get().data.expectedSize();
															#line 4613 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			validatedBytes = validatedBytes + reps[i].get().data.expectedSize();
															#line 4618 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!reps[i].get().more)
															#line 16143 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4619 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				complete = true;
															#line 16147 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 4624 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = rangeToRead;
															#line 4625 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const GetKeyValuesReply& local = reps.back().get();
															#line 4626 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (serverListValues.size() != reps.size() - 1)
															#line 16156 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4627 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaRepsLengthWrong", data->thisServerID) .detail("ServerListValuesSize", serverListValues.size()) .detail("RepsSize", reps.size());
															#line 4632 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16162 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4634 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reps.size() == 1)
															#line 16166 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4636 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "SSAuditStorageShardReplicaNothingToCompare", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("TargetServers", describe(req.targetServers));
															#line 4641 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			complete = true;
															#line 16172 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4645 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int repIdx = 0;repIdx < reps.size() - 1;repIdx++) {
															#line 4646 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const GetKeyValuesReply& remote = reps[repIdx].get();
															#line 4648 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!serverListValues[repIdx].present())
															#line 16180 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4649 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaRepIdxNotPresent", data->thisServerID) .detail("RepIdx", repIdx);
															#line 4653 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16186 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4655 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const StorageServerInterface& remoteServer = decodeServerListValue(serverListValues[repIdx].get());
															#line 4656 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Key lastKey = rangeToRead.begin;
															#line 4657 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const int end = std::min(local.data.size(), remote.data.size());
															#line 4658 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			bool missingKey = local.data.size() != remote.data.size();
															#line 4660 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::string error;
															#line 4661 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int i = 0;
															#line 4662 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;i < end;++i) {
															#line 4663 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyValueRef remoteKV = remote.data[i];
															#line 4664 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyValueRef localKV = local.data[i];
															#line 4665 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!req.range.contains(remoteKV.key) || !req.range.contains(localKV.key))
															#line 16208 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4666 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevWarn, "SSAuditStorageShardReplicaKeyOutOfRange", data->thisServerID) .detail("AuditRange", req.range) .detail("RemoteServer", remoteServer.toString()) .detail("LocalKey", localKV.key) .detail("RemoteKey", remoteKV.key);
															#line 4671 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					return a_body1cont1loopBody1Catch1(wrong_shard_server(), loopDepth);
															#line 16214 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 4674 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (remoteKV.key != localKV.key)
															#line 16218 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4675 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					error = format("Key Mismatch: local server (%016llx): %s, remote server(%016llx) %s", data->thisServerID.first(), Traceable<StringRef>::toString(localKV.key).c_str(), remoteServer.uniqueID.first(), Traceable<StringRef>::toString(remoteKV.key).c_str());
															#line 4680 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4688 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					errors.push_back(error);
															#line 16226 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					break;
				}
				else
				{
															#line 4690 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (remoteKV.value != localKV.value)
															#line 16233 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 4691 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						error = format( "Value Mismatch for Key %s: local server (%016llx): %s, remote server(%016llx) %s", Traceable<StringRef>::toString(localKV.key).c_str(), data->thisServerID.first(), Traceable<StringRef>::toString(localKV.value).c_str(), remoteServer.uniqueID.first(), Traceable<StringRef>::toString(remoteKV.value).c_str());
															#line 4698 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4706 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						errors.push_back(error);
															#line 16241 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						break;
					}
					else
					{
															#line 4709 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevVerbose, "SSAuditStorageShardReplicaValidatedKey", data->thisServerID) .detail("Key", localKV.key);
															#line 16248 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
															#line 4712 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++numValidatedKeys;
															#line 4713 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				lastKey = localKV.key;
															#line 16255 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4715 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange completeRange = Standalone(KeyRangeRef(rangeToRead.begin, keyAfter(lastKey)));
															#line 4716 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (completeRange.empty() || claimRange.begin != completeRange.begin)
															#line 16261 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4717 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaCompleteRangeUnexpected", data->thisServerID) .detail("ClaimRange", claimRange) .detail("CompleteRange", completeRange);
															#line 4722 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16267 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4724 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			claimRange = claimRange & completeRange;
															#line 4725 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!error.empty())
															#line 16273 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
				continue;
			}
															#line 4728 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!local.more && !remote.more && local.data.size() == remote.data.size())
															#line 16279 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
				continue;
			}
			else
			{
															#line 4730 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (i >= local.data.size() && !local.more && i < remote.data.size())
															#line 16287 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4731 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (!missingKey)
															#line 16291 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 4732 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaMissingKeyUnexpected", data->thisServerID);
															#line 16295 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
															#line 4736 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					std::string error = format("Missing key(s) form local server (%lld), next key: %s, remote server(%016llx) ", data->thisServerID.first(), Traceable<StringRef>::toString(remote.data[i].key).c_str(), remoteServer.uniqueID.first());
															#line 4741 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4749 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					errors.push_back(error);
															#line 16303 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					continue;
				}
				else
				{
															#line 4751 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (i >= remote.data.size() && !remote.more && i < local.data.size())
															#line 16310 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 4752 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						if (!missingKey)
															#line 16314 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						{
															#line 4753 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
							TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaMissingKeyUnexpected", data->thisServerID);
															#line 16318 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						}
															#line 4757 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						std::string error = format("Missing key(s) form remote server (%lld), next local server(%016llx) key: %s", remoteServer.uniqueID.first(), data->thisServerID.first(), Traceable<StringRef>::toString(local.data[i].key).c_str());
															#line 4762 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4770 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						errors.push_back(error);
															#line 16326 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						continue;
					}
				}
			}
		}
															#line 4775 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (claimRange.end >= req.range.end)
															#line 16334 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4776 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			complete = true;
															#line 16338 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4779 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageStatisticValidateReplica", data->thisServerID) .suppressFor(30.0) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("AuditServer", data->thisServerID) .detail("ReplicaServers", req.targetServers) .detail("CheckTimes", checkTimes) .detail("NumValidatedKeys", numValidatedKeys) .detail("CurrentValidatedInclusiveRange", claimRange) .detail("CumulatedValidatedInclusiveRange", KeyRangeRef(req.range.begin, claimRange.end));
															#line 4792 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!errors.empty())
															#line 16344 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4793 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorCount", errors.size()) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4801 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.range = claimRange;
															#line 4802 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.setPhase(AuditPhase::Error);
															#line 4803 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!req.ddId.isValid())
															#line 16354 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4804 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaDDIdInvalid", data->thisServerID);
															#line 4807 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16360 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4809 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.ddId = req.ddId;
															#line 4810 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_4 = persistAuditStateByRange(data->cx, res);
															#line 4810 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16368 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 5;
															#line 4810 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 16373 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 4814 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (complete || checkTimes % 100 == 0)
															#line 16380 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4815 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (complete)
															#line 16384 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4816 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					res.range = req.range;
															#line 16388 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 4818 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					res.range = Standalone(KeyRangeRef(req.range.begin, claimRange.end));
															#line 16394 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 4820 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				res.setPhase(AuditPhase::Complete);
															#line 4821 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!req.ddId.isValid())
															#line 16400 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4822 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaDDIdInvalid", data->thisServerID);
															#line 4825 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16406 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 4827 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				res.ddId = req.ddId;
															#line 4828 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevInfo, "SSAuditStorageShardReplicaProgressPersist", data->thisServerID) .suppressFor(10.0) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("Progress", res.toString());
															#line 4834 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_5 = persistAuditStateByRange(data->cx, res);
															#line 4834 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16416 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont6when2(__when_expr_5.get(), loopDepth); };
				static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 6;
															#line 4834 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 16421 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1loopBody1cont36(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6(std::vector<ErrorOr<GetKeyValuesReply>> && reps,int loopDepth) 
	{
															#line 4589 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int i = 0;i < reps.size();++i) {
															#line 4590 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (reps[i].isError())
															#line 16438 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4591 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaGetKeyValuesError", data->thisServerID) .errorUnsuppressed(reps[i].getError()) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("ReplyIndex", i) .detail("RangeRead", rangeToRead) .detail("TargetServers", describe(req.targetServers));
															#line 4599 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(reps[i].getError(), loopDepth);
															#line 16444 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4601 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (reps[i].get().error.present())
															#line 16448 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4602 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevWarn, "SSAuditStorageShardReplicaGetKeyValuesError", data->thisServerID) .errorUnsuppressed(reps[i].get().error.get()) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("ReplyIndex", i) .detail("RangeRead", rangeToRead) .detail("TargetServers", describe(req.targetServers));
															#line 4610 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(reps[i].get().error.get(), loopDepth);
															#line 16454 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4612 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			readBytes = readBytes + reps[i].get().data.expectedSize();
															#line 4613 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			validatedBytes = validatedBytes + reps[i].get().data.expectedSize();
															#line 4618 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!reps[i].get().more)
															#line 16462 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4619 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				complete = true;
															#line 16466 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
		}
															#line 4624 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		claimRange = rangeToRead;
															#line 4625 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		const GetKeyValuesReply& local = reps.back().get();
															#line 4626 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (serverListValues.size() != reps.size() - 1)
															#line 16475 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4627 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaRepsLengthWrong", data->thisServerID) .detail("ServerListValuesSize", serverListValues.size()) .detail("RepsSize", reps.size());
															#line 4632 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16481 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4634 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (reps.size() == 1)
															#line 16485 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4636 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevWarn, "SSAuditStorageShardReplicaNothingToCompare", data->thisServerID) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("TargetServers", describe(req.targetServers));
															#line 4641 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			complete = true;
															#line 16491 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4645 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		for(int repIdx = 0;repIdx < reps.size() - 1;repIdx++) {
															#line 4646 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const GetKeyValuesReply& remote = reps[repIdx].get();
															#line 4648 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!serverListValues[repIdx].present())
															#line 16499 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4649 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaRepIdxNotPresent", data->thisServerID) .detail("RepIdx", repIdx);
															#line 4653 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16505 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4655 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const StorageServerInterface& remoteServer = decodeServerListValue(serverListValues[repIdx].get());
															#line 4656 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			Key lastKey = rangeToRead.begin;
															#line 4657 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			const int end = std::min(local.data.size(), remote.data.size());
															#line 4658 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			bool missingKey = local.data.size() != remote.data.size();
															#line 4660 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			std::string error;
															#line 4661 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			int i = 0;
															#line 4662 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			for(;i < end;++i) {
															#line 4663 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyValueRef remoteKV = remote.data[i];
															#line 4664 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				KeyValueRef localKV = local.data[i];
															#line 4665 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!req.range.contains(remoteKV.key) || !req.range.contains(localKV.key))
															#line 16527 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4666 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevWarn, "SSAuditStorageShardReplicaKeyOutOfRange", data->thisServerID) .detail("AuditRange", req.range) .detail("RemoteServer", remoteServer.toString()) .detail("LocalKey", localKV.key) .detail("RemoteKey", remoteKV.key);
															#line 4671 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					return a_body1cont1loopBody1Catch1(wrong_shard_server(), loopDepth);
															#line 16533 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 4674 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (remoteKV.key != localKV.key)
															#line 16537 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4675 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					error = format("Key Mismatch: local server (%016llx): %s, remote server(%016llx) %s", data->thisServerID.first(), Traceable<StringRef>::toString(localKV.key).c_str(), remoteServer.uniqueID.first(), Traceable<StringRef>::toString(remoteKV.key).c_str());
															#line 4680 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4688 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					errors.push_back(error);
															#line 16545 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					break;
				}
				else
				{
															#line 4690 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (remoteKV.value != localKV.value)
															#line 16552 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 4691 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						error = format( "Value Mismatch for Key %s: local server (%016llx): %s, remote server(%016llx) %s", Traceable<StringRef>::toString(localKV.key).c_str(), data->thisServerID.first(), Traceable<StringRef>::toString(localKV.value).c_str(), remoteServer.uniqueID.first(), Traceable<StringRef>::toString(remoteKV.value).c_str());
															#line 4698 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4706 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						errors.push_back(error);
															#line 16560 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						break;
					}
					else
					{
															#line 4709 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevVerbose, "SSAuditStorageShardReplicaValidatedKey", data->thisServerID) .detail("Key", localKV.key);
															#line 16567 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
				}
															#line 4712 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				++numValidatedKeys;
															#line 4713 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				lastKey = localKV.key;
															#line 16574 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4715 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			KeyRange completeRange = Standalone(KeyRangeRef(rangeToRead.begin, keyAfter(lastKey)));
															#line 4716 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (completeRange.empty() || claimRange.begin != completeRange.begin)
															#line 16580 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4717 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaCompleteRangeUnexpected", data->thisServerID) .detail("ClaimRange", claimRange) .detail("CompleteRange", completeRange);
															#line 4722 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16586 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4724 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			claimRange = claimRange & completeRange;
															#line 4725 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!error.empty())
															#line 16592 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
				continue;
			}
															#line 4728 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!local.more && !remote.more && local.data.size() == remote.data.size())
															#line 16598 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
				continue;
			}
			else
			{
															#line 4730 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (i >= local.data.size() && !local.more && i < remote.data.size())
															#line 16606 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4731 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (!missingKey)
															#line 16610 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 4732 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaMissingKeyUnexpected", data->thisServerID);
															#line 16614 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					}
															#line 4736 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					std::string error = format("Missing key(s) form local server (%lld), next key: %s, remote server(%016llx) ", data->thisServerID.first(), Traceable<StringRef>::toString(remote.data[i].key).c_str(), remoteServer.uniqueID.first());
															#line 4741 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4749 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					errors.push_back(error);
															#line 16622 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					continue;
				}
				else
				{
															#line 4751 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					if (i >= remote.data.size() && !remote.more && i < local.data.size())
															#line 16629 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
					{
															#line 4752 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						if (!missingKey)
															#line 16633 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						{
															#line 4753 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
							TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaMissingKeyUnexpected", data->thisServerID);
															#line 16637 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						}
															#line 4757 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						std::string error = format("Missing key(s) form remote server (%lld), next local server(%016llx) key: %s", remoteServer.uniqueID.first(), data->thisServerID.first(), Traceable<StringRef>::toString(local.data[i].key).c_str());
															#line 4762 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorMessage", error) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4770 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
						errors.push_back(error);
															#line 16645 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
						continue;
					}
				}
			}
		}
															#line 4775 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (claimRange.end >= req.range.end)
															#line 16653 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4776 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			complete = true;
															#line 16657 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
															#line 4779 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		TraceEvent(SevInfo, "SSAuditStorageStatisticValidateReplica", data->thisServerID) .suppressFor(30.0) .detail("AuditID", req.id) .detail("AuditRange", req.range) .detail("AuditType", req.type) .detail("AuditServer", data->thisServerID) .detail("ReplicaServers", req.targetServers) .detail("CheckTimes", checkTimes) .detail("NumValidatedKeys", numValidatedKeys) .detail("CurrentValidatedInclusiveRange", claimRange) .detail("CumulatedValidatedInclusiveRange", KeyRangeRef(req.range.begin, claimRange.end));
															#line 4792 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (!errors.empty())
															#line 16663 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4793 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevError, "SSAuditStorageShardReplicaError", data->thisServerID) .setMaxFieldLength(-1) .setMaxEventLength(-1) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("ErrorCount", errors.size()) .detail("Version", version) .detail("ClaimRange", claimRange);
															#line 4801 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.range = claimRange;
															#line 4802 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.setPhase(AuditPhase::Error);
															#line 4803 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (!req.ddId.isValid())
															#line 16673 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4804 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaDDIdInvalid", data->thisServerID);
															#line 4807 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16679 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			}
															#line 4809 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			res.ddId = req.ddId;
															#line 4810 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			StrictFuture<Void> __when_expr_4 = persistAuditStateByRange(data->cx, res);
															#line 4810 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16687 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 5;
															#line 4810 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 16692 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 4814 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			if (complete || checkTimes % 100 == 0)
															#line 16699 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			{
															#line 4815 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (complete)
															#line 16703 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4816 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					res.range = req.range;
															#line 16707 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
				else
				{
															#line 4818 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					res.range = Standalone(KeyRangeRef(req.range.begin, claimRange.end));
															#line 16713 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 4820 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				res.setPhase(AuditPhase::Complete);
															#line 4821 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (!req.ddId.isValid())
															#line 16719 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				{
															#line 4822 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					TraceEvent(g_network->isSimulated() ? SevError : SevWarnAlways, "SSAuditStorageShardReplicaDDIdInvalid", data->thisServerID);
															#line 4825 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
					return a_body1cont1loopBody1Catch1(audit_storage_cancelled(), loopDepth);
															#line 16725 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				}
															#line 4827 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				res.ddId = req.ddId;
															#line 4828 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				TraceEvent(SevInfo, "SSAuditStorageShardReplicaProgressPersist", data->thisServerID) .suppressFor(10.0) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("Progress", res.toString());
															#line 4834 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				StrictFuture<Void> __when_expr_5 = persistAuditStateByRange(data->cx, res);
															#line 4834 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16735 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont6when2(__when_expr_5.get(), loopDepth); };
				static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 6;
															#line 4834 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
				__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*>(static_cast<AuditStorageShardReplicaQActor*>(this)));
															#line 16740 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1loopBody1cont36(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(std::vector<ErrorOr<GetKeyValuesReply>> const& reps,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont6(reps, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(std::vector<ErrorOr<GetKeyValuesReply>> && reps,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont6(std::move(reps), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*,std::vector<ErrorOr<GetKeyValuesReply>> const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*,std::vector<ErrorOr<GetKeyValuesReply>> && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont41(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont34(Void const& _,int loopDepth) 
	{
															#line 4811 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.sendError(audit_storage_error());
															#line 16836 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont34(Void && _,int loopDepth) 
	{
															#line 4811 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		req.reply.sendError(audit_storage_error());
															#line 16845 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont34(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont34(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont36(int loopDepth) 
	{
															#line 4837 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		if (complete)
															#line 16929 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		{
															#line 4838 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			req.reply.send(res);
															#line 4839 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageShardReplicaComplete", data->thisServerID) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("ReplicaServers", req.targetServers) .detail("ClaimRange", claimRange) .detail("CompleteRange", res.range) .detail("CheckTimes", checkTimes) .detail("NumValidatedKeys", numValidatedKeys) .detail("ValidatedBytes", validatedBytes) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime) .detail("TotalTime", now() - startTime);
															#line 16935 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
		else
		{
															#line 4853 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			TraceEvent(SevInfo, "SSAuditStorageShardReplicaPartialDone", data->thisServerID) .suppressFor(10.0) .detail("AuditId", req.id) .detail("AuditRange", req.range) .detail("AuditServer", data->thisServerID) .detail("ReplicaServers", req.targetServers) .detail("ClaimRange", claimRange) .detail("CompleteRange", res.range) .detail("LastRateLimiterWaitTime", lastRateLimiterWaitTime) .detail("RateLimiterTotalWaitTime", rateLimiterTotalWaitTime);
															#line 4863 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
			rangeToReadBegin = claimRange.end;
															#line 16944 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		}
		loopDepth = a_body1cont1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont37(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont36(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont37(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont36(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont37(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont37(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont6when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont6when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont1loopBody1cont41(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont1loopBody1cont42(Void const& _,int loopDepth) 
	{
															#line 4872 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = now() - rateLimiterBeforeWaitTime;
															#line 4873 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = rateLimiterTotalWaitTime + lastRateLimiterWaitTime;
															#line 4874 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		++checkTimes;
															#line 17145 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont42(Void && _,int loopDepth) 
	{
															#line 4872 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		lastRateLimiterWaitTime = now() - rateLimiterBeforeWaitTime;
															#line 4873 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		rateLimiterTotalWaitTime = rateLimiterTotalWaitTime + lastRateLimiterWaitTime;
															#line 4874 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
		++checkTimes;
															#line 17158 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont42(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont42(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state > 0) static_cast<AuditStorageShardReplicaQActor*>(this)->actor_wait_state = 0;
		static_cast<AuditStorageShardReplicaQActor*>(this)->ActorCallback< AuditStorageShardReplicaQActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< AuditStorageShardReplicaQActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< AuditStorageShardReplicaQActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	StorageServer* data;
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageRequest req;
															#line 4491 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	FlowLock::Releaser holder;
															#line 4499 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageState res;
															#line 4500 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<Optional<Value>> serverListValues;
															#line 4501 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<Future<ErrorOr<GetKeyValuesReply>>> fs;
															#line 4502 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	std::vector<std::string> errors;
															#line 4503 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Version version;
															#line 4504 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Transaction tr;
															#line 4505 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange rangeToRead;
															#line 4506 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Key rangeToReadBegin;
															#line 4507 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	KeyRange claimRange;
															#line 4508 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int limit;
															#line 4509 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int limitBytes;
															#line 4510 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t readBytes;
															#line 4511 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t numValidatedKeys;
															#line 4512 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t validatedBytes;
															#line 4513 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	bool complete;
															#line 4514 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	int64_t checkTimes;
															#line 4515 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double startTime;
															#line 4516 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double lastRateLimiterWaitTime;
															#line 4517 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double rateLimiterBeforeWaitTime;
															#line 4518 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	double rateLimiterTotalWaitTime;
															#line 4519 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	Reference<IRateControl> rateLimiter;
															#line 17299 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
};
// This generated class is to be used only via auditStorageShardReplicaQ()
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
class AuditStorageShardReplicaQActor final : public Actor<Void>, public ActorCallback< AuditStorageShardReplicaQActor, 0, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 1, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 2, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >, public ActorCallback< AuditStorageShardReplicaQActor, 4, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 5, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 6, Void >, public ActorCallback< AuditStorageShardReplicaQActor, 7, Void >, public FastAllocated<AuditStorageShardReplicaQActor>, public AuditStorageShardReplicaQActorState<AuditStorageShardReplicaQActor> {
															#line 17304 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
public:
	using FastAllocated<AuditStorageShardReplicaQActor>::operator new;
	using FastAllocated<AuditStorageShardReplicaQActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7508418109553033728UL, 15159729713238481664UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AuditStorageShardReplicaQActor, 0, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 1, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 2, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 4, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 5, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 6, Void >;
friend struct ActorCallback< AuditStorageShardReplicaQActor, 7, Void >;
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	AuditStorageShardReplicaQActor(StorageServer* const& data,AuditStorageRequest const& req) 
															#line 17328 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
		 : Actor<Void>(),
		   AuditStorageShardReplicaQActorState<AuditStorageShardReplicaQActor>(data, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16484260831731161088UL, 9346140896814425600UL);
		ActorExecutionContextHelper __helper(static_cast<AuditStorageShardReplicaQActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("auditStorageShardReplicaQ");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("auditStorageShardReplicaQ", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 3, std::vector<ErrorOr<GetKeyValuesReply>> >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< AuditStorageShardReplicaQActor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
[[nodiscard]] Future<Void> auditStorageShardReplicaQ( StorageServer* const& data, AuditStorageRequest const& req ) {
															#line 4488 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"
	return Future<Void>(new AuditStorageShardReplicaQActor(data, req));
															#line 17368 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
}

#line 4899 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/fdbserver/storageserver.actor.cpp"

															#line 17373 "/codebuild/output/src1580915050/src/github.com/apple/foundationdb/build_output/fdbserver/storageserver.actor.g.cpp"
namespace {
// This generated class is to be used only via getRangeDataToDump()
															#line 4900 "/codebuild/output/src1580915050/src/github