#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
/*
 * IPageEncryptionKeyProvider.actor.h
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/Knobs.h"
#include "fdbclient/TenantManagement.actor.h"
#include "fdbrpc/TenantInfo.h"
#if defined(NO_INTELLISENSE) && !defined(FDBSERVER_IPAGEENCRYPTIONKEYPROVIDER_ACTOR_G_H)
#define FDBSERVER_IPAGEENCRYPTIONKEYPROVIDER_ACTOR_G_H
#include "fdbserver/IPageEncryptionKeyProvider.actor.g.h"
#elif !defined(FDBSERVER_IPAGEENCRYPTIONKEYPROVIDER_ACTOR_H)
#define FDBSERVER_IPAGEENCRYPTIONKEYPROVIDER_ACTOR_H

#include "fdbclient/BlobCipher.h"
#include "fdbclient/GetEncryptCipherKeys.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/Tenant.h"

#include "fdbserver/IPager.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/ServerDBInfo.h"

#include "flow/Arena.h"
#include "flow/EncryptUtils.h"
#define XXH_INLINE_ALL
#include "flow/xxhash.h"

#include <functional>
#include <limits>
#include <tuple>
#include <type_traits>

#include "flow/actorcompiler.h" // This must be the last #include.

// Interface used by pager to get encryption keys reading pages from disk
// and by the BTree to get encryption keys to use for new pages.
//
// Cipher key rotation:
// The key provider can rotate encryption keys, potentially per encryption domain (see below). Each of the new pages
// are encrypted using the latest encryption keys.
//
// Encryption domains:
// The key provider can specify how the page split the full key range into encryption domains by key prefixes.
// Encryption domains are expected to have their own set of encryption keys, which is managed by the key provider.
// The pager will ensure that data from different encryption domain won't fall in the same page, to make
// sure it can use one single encryption key to encrypt the whole page.
// The key provider needs to provide a default encryption domain, which is used to encrypt pages contain only
// full or partial encryption domain prefixes.
class IPageEncryptionKeyProvider : public ReferenceCounted<IPageEncryptionKeyProvider> {
public:
	using EncryptionKey = ArenaPage::EncryptionKey;

	virtual ~IPageEncryptionKeyProvider() = default;

	// Expected encoding type being used with the encryption key provider.
	virtual EncodingType expectedEncodingType() const = 0;

	// Whether encryption domain is enabled.
	virtual bool enableEncryptionDomain() const = 0;

	// Get an encryption key from given encoding header.
	virtual Future<EncryptionKey> getEncryptionKey(const void* encodingHeader) { throw not_implemented(); }

	// Get latest encryption key. If encryption domain is enabled, get encryption key for the default domain.
	virtual Future<EncryptionKey> getLatestDefaultEncryptionKey() {
		return getLatestEncryptionKey(getDefaultEncryptionDomainId());
	}

	// Get latest encryption key for data in given encryption domain.
	virtual Future<EncryptionKey> getLatestEncryptionKey(int64_t domainId) { throw not_implemented(); }

	// Return the default encryption domain.
	virtual int64_t getDefaultEncryptionDomainId() const { throw not_implemented(); }

	// Get encryption domain from a key. Return the domain id, and the size of the encryption domain prefix.
	virtual std::tuple<int64_t, size_t> getEncryptionDomain(const KeyRef& key) { throw not_implemented(); }

	// Get encryption domain of a page given encoding header.
	virtual int64_t getEncryptionDomainIdFromHeader(const void* encodingHeader) { throw not_implemented(); }

	// Helper methods.

	// Check if a key fits in an encryption domain.
	bool keyFitsInDomain(int64_t domainId, const KeyRef& key, bool canUseDefaultDomain) {
		ASSERT(enableEncryptionDomain());
		int64_t keyDomainId;
		size_t prefixLength;
		std::tie(keyDomainId, prefixLength) = getEncryptionDomain(key);
		return keyDomainId == domainId ||
		       (canUseDefaultDomain && (domainId == getDefaultEncryptionDomainId() && key.size() == prefixLength));
	}
};

// The null key provider is useful to simplify page decoding.
// It throws an error for any key info requested.
class NullEncryptionKeyProvider : public IPageEncryptionKeyProvider {
public:
	virtual ~NullEncryptionKeyProvider() {}
	EncodingType expectedEncodingType() const override { return EncodingType::XXHash64; }
	bool enableEncryptionDomain() const override { return false; }
};

// Key provider for dummy XOR encryption scheme
class XOREncryptionKeyProvider_TestOnly : public IPageEncryptionKeyProvider {
public:
	using EncodingHeader = ArenaPage::XOREncryptionEncoder::Header;

	XOREncryptionKeyProvider_TestOnly(std::string filename) {
		ASSERT(g_network->isSimulated());

		// Choose a deterministic random filename (without path) byte for secret generation
		// Remove any leading directory names
		size_t lastSlash = filename.find_last_of("\\/");
		if (lastSlash != filename.npos) {
			filename.erase(0, lastSlash);
		}
		xorWith = filename.empty() ? 0x5e
		                           : (uint8_t)filename[XXH3_64bits(filename.data(), filename.size()) % filename.size()];
	}

	virtual ~XOREncryptionKeyProvider_TestOnly() {}

	EncodingType expectedEncodingType() const override { return EncodingType::XOREncryption_TestOnly; }

	bool enableEncryptionDomain() const override { return false; }

	Future<EncryptionKey> getEncryptionKey(const void* encodingHeader) override {

		const EncodingHeader* h = reinterpret_cast<const EncodingHeader*>(encodingHeader);
		EncryptionKey s;
		s.xorKey = h->xorKey;
		s.xorWith = xorWith;
		return s;
	}

	Future<EncryptionKey> getLatestDefaultEncryptionKey() override {
		EncryptionKey s;
		s.xorKey = static_cast<uint8_t>(deterministicRandom()->randomInt(0, std::numeric_limits<uint8_t>::max() + 1));
		s.xorWith = xorWith;
		return s;
	}

	uint8_t xorWith;
};

namespace {
template <EncodingType encodingType>
int64_t getEncryptionDomainIdFromAesEncryptionHeader(const void* encodingHeader) {
	using Encoder = typename ArenaPage::AESEncryptionEncoder<encodingType>;
	ASSERT(encodingHeader != nullptr);
	BlobCipherEncryptHeaderRef headerRef = Encoder::getEncryptionHeaderRef(encodingHeader);
	return headerRef.getCipherDetails().textCipherDetails.encryptDomainId;
}
} // anonymous namespace

// Key provider to provider cipher keys randomly from a pre-generated pool. It does not maintain encryption domains.
// Use for testing.
template <EncodingType encodingType,
          typename std::enable_if<encodingType == AESEncryption || encodingType == AESEncryptionWithAuth, bool>::type =
              true>
class RandomEncryptionKeyProvider : public IPageEncryptionKeyProvider {
public:
	enum EncryptionDomainMode : unsigned int {
		DISABLED = 0, // disable encryption domain
		RANDOM, // for each key prefix, deterministic randomly decide if there's an encryption domain for it.
		ALL, // all key prefixes has an encryption domain assigned to it.
		MAX,
	};

	explicit RandomEncryptionKeyProvider(EncryptionDomainMode mode = DISABLED) : mode(mode) {
		ASSERT(mode < EncryptionDomainMode::MAX);
		for (unsigned i = 0; i < NUM_CIPHER; i++) {
			BlobCipherDetails cipherDetails;
			cipherDetails.encryptDomainId = 0;
			cipherDetails.baseCipherId = i;
			cipherDetails.salt = deterministicRandom()->randomUInt64();
			cipherKeys[i] = generateCipherKey(cipherDetails);
		}
	}
	virtual ~RandomEncryptionKeyProvider() = default;

	EncodingType expectedEncodingType() const override { return encodingType; }

	bool enableEncryptionDomain() const override { return mode > 0; }

	Future<EncryptionKey> getEncryptionKey(const void* encodingHeader) override {
		using Encoder = typename ArenaPage::AESEncryptionEncoder<encodingType>;
		EncryptionKey s;
		const BlobCipherEncryptHeaderRef headerRef = Encoder::getEncryptionHeaderRef(encodingHeader);
		EncryptHeaderCipherDetails details = headerRef.getCipherDetails();
		ASSERT(details.textCipherDetails.isValid());
		s.aesKey.cipherTextKey =
		    getCipherKey(details.textCipherDetails.encryptDomainId, details.textCipherDetails.baseCipherId);
		if (details.headerCipherDetails.present()) {
			ASSERT(details.headerCipherDetails.get().isValid());
			s.aesKey.cipherHeaderKey = getCipherKey(details.headerCipherDetails.get().encryptDomainId,
			                                        details.headerCipherDetails.get().baseCipherId);
		}
		return s;
	}

	Future<EncryptionKey> getLatestEncryptionKey(int64_t domainId) override {
		domainId = checkDomainId(domainId);
		EncryptionKey s;
		s.aesKey.cipherTextKey = getCipherKey(domainId, deterministicRandom()->randomInt(1, NUM_CIPHER + 1));
		s.aesKey.cipherHeaderKey =
		    getCipherKey(ENCRYPT_HEADER_DOMAIN_ID, deterministicRandom()->randomInt(1, NUM_CIPHER + 1));
		return s;
	}

	int64_t getDefaultEncryptionDomainId() const override { return FDB_DEFAULT_ENCRYPT_DOMAIN_ID; }

	std::tuple<int64_t, size_t> getEncryptionDomain(const KeyRef& key) override {
		int64_t domainId;
		if (key.size() < PREFIX_LENGTH) {
			domainId = getDefaultEncryptionDomainId();
		} else {
			// Use first 4 bytes as a 32-bit int for the domain id.
			domainId = checkDomainId(static_cast<int64_t>(*reinterpret_cast<const int32_t*>(key.begin())));
		}
		return { domainId, (domainId == getDefaultEncryptionDomainId() ? 0 : PREFIX_LENGTH) };
	}

	int64_t getEncryptionDomainIdFromHeader(const void* encodingHeader) override {
		return getEncryptionDomainIdFromAesEncryptionHeader<encodingType>(encodingHeader);
	}

private:
	Reference<BlobCipherKey> generateCipherKey(const BlobCipherDetails& cipherDetails) {
		static unsigned char SHA_KEY[] = "3ab9570b44b8315fdb261da6b1b6c13b";
		Arena arena;
		uint8_t digest[AUTH_TOKEN_HMAC_SHA_SIZE];
		computeAuthToken(
		    { { reinterpret_cast<const uint8_t*>(&cipherDetails.baseCipherId), sizeof(EncryptCipherBaseKeyId) } },
		    SHA_KEY,
		    AES_256_KEY_LENGTH,
		    &digest[0],
		    EncryptAuthTokenAlgo::ENCRYPT_HEADER_AUTH_TOKEN_ALGO_HMAC_SHA,
		    AUTH_TOKEN_HMAC_SHA_SIZE);
		ASSERT_EQ(AUTH_TOKEN_HMAC_SHA_SIZE, AES_256_KEY_LENGTH);
		const EncryptCipherKeyCheckValue kcv = Sha256KCV().computeKCV(&digest[0], AES_256_KEY_LENGTH);
		return makeReference<BlobCipherKey>(cipherDetails.encryptDomainId,
		                                    cipherDetails.baseCipherId,
		                                    &digest[0],
		                                    AES_256_KEY_LENGTH,
		                                    kcv,
		                                    cipherDetails.salt,
		                                    std::numeric_limits<int64_t>::max() /* refreshAt */,
		                                    std::numeric_limits<int64_t>::max() /* expireAt */);
	}

	int64_t checkDomainId(int64_t domainId) {
		std::hash<int64_t> hasher;
		if (mode == DISABLED || (mode == RANDOM && hasher(domainId) % 2 == 0)) {
			return getDefaultEncryptionDomainId();
		}
		return domainId;
	}

	Reference<BlobCipherKey> getCipherKey(EncryptCipherDomainId domainId, EncryptCipherBaseKeyId cipherId) {
		// Create a new cipher key by replacing the domain id.
		ASSERT(cipherId > 0 && cipherId <= NUM_CIPHER);
		return makeReference<BlobCipherKey>(domainId,
		                                    cipherId,
		                                    cipherKeys[cipherId - 1]->rawBaseCipher(),
		                                    AES_256_KEY_LENGTH,
		                                    cipherKeys[cipherId - 1]->getBaseCipherKCV(),
		                                    cipherKeys[cipherId - 1]->getSalt(),
		                                    std::numeric_limits<int64_t>::max() /* refreshAt */,
		                                    std::numeric_limits<int64_t>::max() /* expireAt */);
	}

	static constexpr int NUM_CIPHER = 1000;
	static constexpr size_t PREFIX_LENGTH = 4;
	EncryptionDomainMode mode;
	Reference<BlobCipherKey> cipherKeys[NUM_CIPHER];
};

// Key provider which extract tenant id from range key prefixes, and fetch tenant specific encryption keys from
// EncryptKeyProxy.
template <EncodingType encodingType,
          typename std::enable_if<encodingType == AESEncryption || encodingType == AESEncryptionWithAuth, bool>::type =
              true>
class AESEncryptionKeyProvider : public IPageEncryptionKeyProvider {
public:
	using Encoder = typename ArenaPage::AESEncryptionEncoder<encodingType>;
	using EncodingHeader = typename Encoder::Header;

	const StringRef systemKeysPrefix = systemKeys.begin;

	AESEncryptionKeyProvider(Reference<AsyncVar<ServerDBInfo> const> db,
	                         EncryptionAtRestMode encryptionMode,
	                         Reference<GetEncryptCipherKeysMonitor> monitor)
	  : db(db), encryptionMode(encryptionMode), monitor(monitor) {
		ASSERT(encryptionMode != EncryptionAtRestMode::DISABLED);
		ASSERT(db.isValid());
	}

	virtual ~AESEncryptionKeyProvider() = default;

	EncodingType expectedEncodingType() const override { return encodingType; }

	bool enableEncryptionDomain() const override {
		// Regardless of encryption mode, system keys always encrypted using system key space domain.
		// Because of this, AESEncryptionKeyProvider always appears to be domain-aware.
		return true;
	}

																#line 327 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
// This generated class is to be used only via getEncryptionKey()
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
template <class GetEncryptionKeyActor>
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
class GetEncryptionKeyActorState {
															#line 333 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
public:
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
	GetEncryptionKeyActorState(AESEncryptionKeyProvider* const& self,const void* const& encodingHeader) 
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		 : self(self),
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		   encodingHeader(encodingHeader),
															#line 326 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		   cipherKeys()
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
	{
		fdb_probe_actor_create("getEncryptionKey", reinterpret_cast<unsigned long>(this));

	}
	~GetEncryptionKeyActorState() 
	{
		fdb_probe_actor_destroy("getEncryptionKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 327 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
			BlobCipherEncryptHeaderRef headerRef = Encoder::getEncryptionHeaderRef(encodingHeader);
															#line 328 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
			StrictFuture<TextAndHeaderCipherKeys> __when_expr_0 = GetEncryptCipherKeys<ServerDBInfo>::getEncryptCipherKeys( self->db, headerRef, BlobCipherMetrics::KV_REDWOOD, self->monitor);
															#line 328 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
			if (static_cast<GetEncryptionKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 363 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetEncryptionKeyActor*>(this)->actor_wait_state = 1;
															#line 328 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetEncryptionKeyActor, 0, TextAndHeaderCipherKeys >*>(static_cast<GetEncryptionKeyActor*>(this)));
															#line 368 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetEncryptionKeyActorState();
		static_cast<GetEncryptionKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(TextAndHeaderCipherKeys const& cks,int loopDepth) 
	{
															#line 330 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		cipherKeys = cks;
															#line 331 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		EncryptionKey encryptionKey;
															#line 332 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		encryptionKey.aesKey = cipherKeys;
															#line 333 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		if (!static_cast<GetEncryptionKeyActor*>(this)->SAV<EncryptionKey>::futures) { (void)(encryptionKey); this->~GetEncryptionKeyActorState(); static_cast<GetEncryptionKeyActor*>(this)->destroy(); return 0; }
															#line 397 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
		new (&static_cast<GetEncryptionKeyActor*>(this)->SAV< EncryptionKey >::value()) EncryptionKey(encryptionKey);
		this->~GetEncryptionKeyActorState();
		static_cast<GetEncryptionKeyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(TextAndHeaderCipherKeys && cks,int loopDepth) 
	{
															#line 330 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		cipherKeys = cks;
															#line 331 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		EncryptionKey encryptionKey;
															#line 332 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		encryptionKey.aesKey = cipherKeys;
															#line 333 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		if (!static_cast<GetEncryptionKeyActor*>(this)->SAV<EncryptionKey>::futures) { (void)(encryptionKey); this->~GetEncryptionKeyActorState(); static_cast<GetEncryptionKeyActor*>(this)->destroy(); return 0; }
															#line 415 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
		new (&static_cast<GetEncryptionKeyActor*>(this)->SAV< EncryptionKey >::value()) EncryptionKey(encryptionKey);
		this->~GetEncryptionKeyActorState();
		static_cast<GetEncryptionKeyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(TextAndHeaderCipherKeys const& cks,int loopDepth) 
	{
		loopDepth = a_body1cont1(cks, loopDepth);

		return loopDepth;
	}
	int a_body1when1(TextAndHeaderCipherKeys && cks,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(cks), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetEncryptionKeyActor*>(this)->actor_wait_state > 0) static_cast<GetEncryptionKeyActor*>(this)->actor_wait_state = 0;
		static_cast<GetEncryptionKeyActor*>(this)->ActorCallback< GetEncryptionKeyActor, 0, TextAndHeaderCipherKeys >::remove();

	}
	void a_callback_fire(ActorCallback< GetEncryptionKeyActor, 0, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys const& value) 
	{
		fdb_probe_actor_enter("getEncryptionKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetEncryptionKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getEncryptionKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetEncryptionKeyActor, 0, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys && value) 
	{
		fdb_probe_actor_enter("getEncryptionKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetEncryptionKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getEncryptionKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetEncryptionKeyActor, 0, TextAndHeaderCipherKeys >*,Error err) 
	{
		fdb_probe_actor_enter("getEncryptionKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetEncryptionKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getEncryptionKey", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
	AESEncryptionKeyProvider* self;
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
	const void* encodingHeader;
															#line 326 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
	TextAndHeaderCipherKeys cipherKeys;
															#line 504 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
};
// This generated class is to be used only via getEncryptionKey()
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
class GetEncryptionKeyActor final : public Actor<EncryptionKey>, public ActorCallback< GetEncryptionKeyActor, 0, TextAndHeaderCipherKeys >, public FastAllocated<GetEncryptionKeyActor>, public GetEncryptionKeyActorState<GetEncryptionKeyActor> {
															#line 509 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
public:
	using FastAllocated<GetEncryptionKeyActor>::operator new;
	using FastAllocated<GetEncryptionKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10208098288222338304UL, 9957387168856325632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<EncryptionKey>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetEncryptionKeyActor, 0, TextAndHeaderCipherKeys >;
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
	GetEncryptionKeyActor(AESEncryptionKeyProvider* const& self,const void* const& encodingHeader) 
															#line 526 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
		 : Actor<EncryptionKey>(),
		   GetEncryptionKeyActorState<GetEncryptionKeyActor>(self, encodingHeader),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getEncryptionKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3590999031006583808UL, 8442963300777650944UL);
		ActorExecutionContextHelper __helper(static_cast<GetEncryptionKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getEncryptionKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getEncryptionKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetEncryptionKeyActor, 0, TextAndHeaderCipherKeys >*)0, actor_cancelled()); break;
		}

	}
};
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
[[nodiscard]] static Future<EncryptionKey> getEncryptionKey( AESEncryptionKeyProvider* const& self, const void* const& encodingHeader ) {
															#line 325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
	return Future<EncryptionKey>(new GetEncryptionKeyActor(self, encodingHeader));
															#line 558 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
}

#line 335 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"

	Future<EncryptionKey> getEncryptionKey(const void* encodingHeader) override {
		return getEncryptionKey(this, encodingHeader);
	}

	Future<EncryptionKey> getLatestDefaultEncryptionKey() override {
		return getLatestEncryptionKey(getDefaultEncryptionDomainId());
	}

																#line 571 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
// This generated class is to be used only via getLatestEncryptionKey()
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
template <class GetLatestEncryptionKeyActor>
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
class GetLatestEncryptionKeyActorState {
															#line 577 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
public:
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
	GetLatestEncryptionKeyActorState(AESEncryptionKeyProvider* const& self,int64_t const& domainId) 
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		 : self(self),
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		   domainId(domainId)
															#line 586 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
	{
		fdb_probe_actor_create("getLatestEncryptionKey", reinterpret_cast<unsigned long>(this));

	}
	~GetLatestEncryptionKeyActorState() 
	{
		fdb_probe_actor_destroy("getLatestEncryptionKey", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 345 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
			ASSERT(self->encryptionMode == EncryptionAtRestMode::DOMAIN_AWARE || domainId < 0);
															#line 346 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
			StrictFuture<TextAndHeaderCipherKeys> __when_expr_0 = GetEncryptCipherKeys<ServerDBInfo>::getLatestEncryptCipherKeysForDomain( self->db, domainId, BlobCipherMetrics::KV_REDWOOD, self->monitor);
															#line 346 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
			if (static_cast<GetLatestEncryptionKeyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 605 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetLatestEncryptionKeyActor*>(this)->actor_wait_state = 1;
															#line 346 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetLatestEncryptionKeyActor, 0, TextAndHeaderCipherKeys >*>(static_cast<GetLatestEncryptionKeyActor*>(this)));
															#line 610 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetLatestEncryptionKeyActorState();
		static_cast<GetLatestEncryptionKeyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(TextAndHeaderCipherKeys const& cipherKeys,int loopDepth) 
	{
															#line 349 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		EncryptionKey encryptionKey;
															#line 350 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		encryptionKey.aesKey = cipherKeys;
															#line 351 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		if (!static_cast<GetLatestEncryptionKeyActor*>(this)->SAV<EncryptionKey>::futures) { (void)(encryptionKey); this->~GetLatestEncryptionKeyActorState(); static_cast<GetLatestEncryptionKeyActor*>(this)->destroy(); return 0; }
															#line 637 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
		new (&static_cast<GetLatestEncryptionKeyActor*>(this)->SAV< EncryptionKey >::value()) EncryptionKey(encryptionKey);
		this->~GetLatestEncryptionKeyActorState();
		static_cast<GetLatestEncryptionKeyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(TextAndHeaderCipherKeys && cipherKeys,int loopDepth) 
	{
															#line 349 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		EncryptionKey encryptionKey;
															#line 350 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		encryptionKey.aesKey = cipherKeys;
															#line 351 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
		if (!static_cast<GetLatestEncryptionKeyActor*>(this)->SAV<EncryptionKey>::futures) { (void)(encryptionKey); this->~GetLatestEncryptionKeyActorState(); static_cast<GetLatestEncryptionKeyActor*>(this)->destroy(); return 0; }
															#line 653 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
		new (&static_cast<GetLatestEncryptionKeyActor*>(this)->SAV< EncryptionKey >::value()) EncryptionKey(encryptionKey);
		this->~GetLatestEncryptionKeyActorState();
		static_cast<GetLatestEncryptionKeyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(TextAndHeaderCipherKeys const& cipherKeys,int loopDepth) 
	{
		loopDepth = a_body1cont1(cipherKeys, loopDepth);

		return loopDepth;
	}
	int a_body1when1(TextAndHeaderCipherKeys && cipherKeys,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(cipherKeys), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetLatestEncryptionKeyActor*>(this)->actor_wait_state > 0) static_cast<GetLatestEncryptionKeyActor*>(this)->actor_wait_state = 0;
		static_cast<GetLatestEncryptionKeyActor*>(this)->ActorCallback< GetLatestEncryptionKeyActor, 0, TextAndHeaderCipherKeys >::remove();

	}
	void a_callback_fire(ActorCallback< GetLatestEncryptionKeyActor, 0, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys const& value) 
	{
		fdb_probe_actor_enter("getLatestEncryptionKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestEncryptionKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestEncryptionKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetLatestEncryptionKeyActor, 0, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys && value) 
	{
		fdb_probe_actor_enter("getLatestEncryptionKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestEncryptionKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestEncryptionKey", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetLatestEncryptionKeyActor, 0, TextAndHeaderCipherKeys >*,Error err) 
	{
		fdb_probe_actor_enter("getLatestEncryptionKey", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestEncryptionKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestEncryptionKey", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
	AESEncryptionKeyProvider* self;
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
	int64_t domainId;
															#line 740 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
};
// This generated class is to be used only via getLatestEncryptionKey()
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
class GetLatestEncryptionKeyActor final : public Actor<EncryptionKey>, public ActorCallback< GetLatestEncryptionKeyActor, 0, TextAndHeaderCipherKeys >, public FastAllocated<GetLatestEncryptionKeyActor>, public GetLatestEncryptionKeyActorState<GetLatestEncryptionKeyActor> {
															#line 745 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
public:
	using FastAllocated<GetLatestEncryptionKeyActor>::operator new;
	using FastAllocated<GetLatestEncryptionKeyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12770068908184369920UL, 3584935072104210432UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<EncryptionKey>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetLatestEncryptionKeyActor, 0, TextAndHeaderCipherKeys >;
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
	GetLatestEncryptionKeyActor(AESEncryptionKeyProvider* const& self,int64_t const& domainId) 
															#line 762 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
		 : Actor<EncryptionKey>(),
		   GetLatestEncryptionKeyActorState<GetLatestEncryptionKeyActor>(self, domainId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getLatestEncryptionKey", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5985122012808392448UL, 6090826688484248832UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestEncryptionKeyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getLatestEncryptionKey");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getLatestEncryptionKey", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetLatestEncryptionKeyActor, 0, TextAndHeaderCipherKeys >*)0, actor_cancelled()); break;
		}

	}
};
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
[[nodiscard]] static Future<EncryptionKey> getLatestEncryptionKey( AESEncryptionKeyProvider* const& self, int64_t const& domainId ) {
															#line 344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"
	return Future<EncryptionKey>(new GetLatestEncryptionKeyActor(self, domainId));
															#line 794 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.g.h"
}

#line 353 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/include/fdbserver/IPageEncryptionKeyProvider.actor.h"

	Future<EncryptionKey> getLatestEncryptionKey(int64_t domainId) override {
		return getLatestEncryptionKey(this, domainId);
	}

	int64_t getDefaultEncryptionDomainId() const override { return FDB_DEFAULT_ENCRYPT_DOMAIN_ID; }

	std::tuple<int64_t, size_t> getEncryptionDomain(const KeyRef& key) override {
		// System key.
		if (key.startsWith(systemKeysPrefix)) {
			return { SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID, systemKeysPrefix.size() };
		}
		// Cluster-aware encryption.
		if (encryptionMode == EncryptionAtRestMode::CLUSTER_AWARE) {
			return { FDB_DEFAULT_ENCRYPT_DOMAIN_ID, 0 };
		}
		// Key smaller than tenant prefix in size belongs to the default domain.
		if (key.size() < TenantAPI::PREFIX_SIZE) {
			return { FDB_DEFAULT_ENCRYPT_DOMAIN_ID, 0 };
		}
		int64_t tenantId = TenantAPI::extractTenantIdFromKeyRef(key);
		if (tenantId == TenantInfo::INVALID_TENANT) {
			return { FDB_DEFAULT_ENCRYPT_DOMAIN_ID, 0 };
		}
		return { tenantId, TenantAPI::PREFIX_SIZE };
	}

	int64_t getEncryptionDomainIdFromHeader(const void* encodingHeader) override {
		return getEncryptionDomainIdFromAesEncryptionHeader<encodingType>(encodingHeader);
	}

private:
	Reference<AsyncVar<ServerDBInfo> const> db;
	EncryptionAtRestMode encryptionMode;
	Reference<GetEncryptCipherKeysMonitor> monitor;
};

#include "flow/unactorcompiler.h"
#endif