#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
/*
 * BlobManager.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <algorithm>
#include <ctime>
#include <limits>
#include <sstream>
#include <queue>
#include <vector>
#include <unordered_map>

#include "fdbclient/BackupContainer.h"
#include "fdbclient/ClientBooleanParams.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/KeyBackedTypes.actor.h"
#include "fdbclient/ServerKnobs.h"
#include "fdbrpc/simulator.h"
#include "flow/CodeProbe.h"
#include "flow/FastRef.h"
#include "flow/genericactors.actor.h"
#include "flow/serialize.h"
#include "fmt/format.h"
#include "fdbclient/BackupContainerFileSystem.h"
#include "fdbclient/BlobGranuleCommon.h"
#include "fdbclient/BlobGranuleRequest.actor.h"
#include "fdbclient/BlobWorkerInterface.h"
#include "fdbclient/KeyRangeMap.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/ReadYourWrites.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/Tuple.h"
#include "fdbclient/BackupAgent.actor.h"
#include "fdbclient/BlobRestoreCommon.h"
#include "fdbserver/BlobManagerInterface.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/BlobGranuleValidation.actor.h"
#include "fdbserver/BlobGranuleServerCommon.actor.h"
#include "fdbserver/ExclusionTracker.actor.h"
#include "fdbserver/QuietDatabase.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "flow/Error.h"
#include "flow/IRandom.h"
#include "flow/UnitTest.h"
#include "flow/actorcompiler.h" // has to be last include

/*
 * The Blob Manager is responsible for managing range granules, and recruiting and monitoring Blob Workers.
 */

#define BM_DEBUG false
#define BM_PURGE_DEBUG false

void handleClientBlobRange(int64_t epoch,
                           KeyRangeMap<bool>* knownBlobRanges,
                           Arena& ar,
                           VectorRef<KeyRangeRef>* rangesToAdd,
                           VectorRef<KeyRangeRef>* rangesToRemove,
                           KeyRef rangeStart,
                           KeyRef rangeEnd,
                           bool rangeActive) {
	if (BM_DEBUG) {
		fmt::print(
		    "db range [{0} - {1}): {2}\n", rangeStart.printable(), rangeEnd.printable(), rangeActive ? "T" : "F");
	}
	KeyRange keyRange(KeyRangeRef(rangeStart, rangeEnd));
	auto allRanges = knownBlobRanges->intersectingRanges(keyRange);
	for (auto& r : allRanges) {
		if (r.value() != rangeActive) {
			KeyRef overlapStart = (r.begin() > keyRange.begin) ? r.begin() : keyRange.begin;
			KeyRef overlapEnd = (keyRange.end < r.end()) ? keyRange.end : r.end();
			KeyRangeRef overlap(overlapStart, overlapEnd);
			if (rangeActive) {
				if (BM_DEBUG) {
					fmt::print("BM {0} Adding client range [{1} - {2})\n",
					           epoch,
					           overlapStart.printable().c_str(),
					           overlapEnd.printable().c_str());
				}
				rangesToAdd->push_back_deep(ar, overlap);
			} else {
				if (BM_DEBUG) {
					fmt::print("BM {0} Removing client range [{1} - {2})\n",
					           epoch,
					           overlapStart.printable().c_str(),
					           overlapEnd.printable().c_str());
				}
				rangesToRemove->push_back_deep(ar, overlap);
			}
		}
	}
	knownBlobRanges->insert(keyRange, rangeActive);
}

void updateClientBlobRanges(int64_t epoch,
                            KeyRangeMap<bool>* knownBlobRanges,
                            RangeResult dbBlobRanges,
                            Arena& ar,
                            VectorRef<KeyRangeRef>* rangesToAdd,
                            VectorRef<KeyRangeRef>* rangesToRemove) {
	if (BM_DEBUG) {
		fmt::print("Updating {0} client blob ranges", dbBlobRanges.size() / 2);
		for (int i = 0; i < dbBlobRanges.size() - 1; i += 2) {
			fmt::print("  [{0} - {1})", dbBlobRanges[i].key.printable(), dbBlobRanges[i + 1].key.printable());
		}
		printf("\n");
	}
	// essentially do merge diff of current known blob ranges and new ranges, to assign new ranges to
	// workers and revoke old ranges from workers

	// basically, for any range that is set in results that isn't set in ranges, assign the range to the
	// worker. for any range that isn't set in results that is set in ranges, revoke the range from the
	// worker. and, update ranges to match results as you go

	// SOMEDAY: could change this to O(N) instead of O(NLogN) by doing a sorted merge instead of requesting the
	// intersection for each insert, but this operation is pretty infrequent so it's probably not necessary
	if (dbBlobRanges.size() == 0) {
		// special case. Nothing in the DB, reset knownBlobRanges and revoke all existing ranges from workers
		handleClientBlobRange(
		    epoch, knownBlobRanges, ar, rangesToAdd, rangesToRemove, normalKeys.begin, normalKeys.end, false);
	} else {
		if (dbBlobRanges[0].key > normalKeys.begin) {
			handleClientBlobRange(
			    epoch, knownBlobRanges, ar, rangesToAdd, rangesToRemove, normalKeys.begin, dbBlobRanges[0].key, false);
		}
		for (int i = 0; i < dbBlobRanges.size() - 1; i++) {
			if (dbBlobRanges[i].key >= normalKeys.end) {
				if (BM_DEBUG) {
					fmt::print("Found invalid blob range start {0}\n", dbBlobRanges[i].key.printable());
				}
				break;
			}
			bool active = isBlobRangeActive(dbBlobRanges[i].value);
			if (active) {
				if (BM_DEBUG) {
					fmt::print("BM {0} sees client range [{1} - {2})\n",
					           epoch,
					           dbBlobRanges[i].key.printable(),
					           dbBlobRanges[i + 1].key.printable());
				}
			}
			KeyRef endKey = dbBlobRanges[i + 1].key;
			if (endKey > normalKeys.end) {
				if (BM_DEBUG) {
					fmt::print("Removing system keyspace from blob range [{0} - {1})\n",
					           dbBlobRanges[i].key.printable(),
					           endKey.printable());
				}
				endKey = normalKeys.end;
			}
			handleClientBlobRange(
			    epoch, knownBlobRanges, ar, rangesToAdd, rangesToRemove, dbBlobRanges[i].key, endKey, active);
		}
		if (dbBlobRanges[dbBlobRanges.size() - 1].key < normalKeys.end) {
			handleClientBlobRange(epoch,
			                      knownBlobRanges,
			                      ar,
			                      rangesToAdd,
			                      rangesToRemove,
			                      dbBlobRanges[dbBlobRanges.size() - 1].key,
			                      normalKeys.end,
			                      false);
		}
	}
	knownBlobRanges->coalesce(normalKeys);
}

void getRanges(std::vector<std::pair<KeyRangeRef, bool>>& results, KeyRangeMap<bool>& knownBlobRanges) {
	if (BM_DEBUG) {
		printf("Getting ranges:\n");
	}
	auto allRanges = knownBlobRanges.ranges();
	for (auto& r : allRanges) {
		results.emplace_back(r.range(), r.value());
		if (BM_DEBUG) {
			fmt::print("  [{0} - {1}): {2}\n", r.begin().printable(), r.end().printable(), r.value() ? "T" : "F");
		}
	}
}

struct RangeAssignmentData {
	AssignRequestType type;

	RangeAssignmentData() : type(AssignRequestType::Normal) {}
	RangeAssignmentData(AssignRequestType type) : type(type) {}
};

struct RangeRevokeData {
	bool dispose;

	RangeRevokeData() {}
	RangeRevokeData(bool dispose) : dispose(dispose) {}
};

struct RangeAssignment {
	bool isAssign;
	KeyRange keyRange;
	Optional<UID> worker;
	Optional<std::pair<UID, Error>> previousFailure;

	// I tried doing this with a union and it was just kind of messy
	Optional<RangeAssignmentData> assign;
	Optional<RangeRevokeData> revoke;
};

// SOMEDAY: track worker's reads/writes eventually
// FIXME: namespace?
struct BlobWorkerInfo {
	int numGranulesAssigned;
	int recentGranulesAssigned;

	BlobWorkerInfo(int numGranulesAssigned = 0, int recentGranulesAssigned = 0)
	  : numGranulesAssigned(numGranulesAssigned), recentGranulesAssigned(recentGranulesAssigned) {}
};

// recover is when the BM assigns an ambiguously owned range on recovery
// merge is when the BM initiated a merge candidate for the range
// split is when the BM initiated a split check for the range
enum BoundaryEvalType { UNKNOWN, RECOVER, MERGE, SPLIT };

struct BoundaryEvaluation {
	int64_t epoch;
	int64_t seqno;
	BoundaryEvalType type;
	Future<Void> inProgress;
	int64_t originalEpoch;
	int64_t originalSeqno;

	BoundaryEvaluation() : epoch(0), seqno(0), type(UNKNOWN), originalEpoch(0), originalSeqno(0) {}
	BoundaryEvaluation(int64_t epoch,
	                   int64_t seqno,
	                   BoundaryEvalType type,
	                   int64_t originalEpoch,
	                   int64_t originalSeqno)
	  : epoch(epoch), seqno(seqno), type(type), originalEpoch(originalEpoch), originalSeqno(originalSeqno) {
		ASSERT(type != UNKNOWN);
	}

	bool operator==(const BoundaryEvaluation& other) const {
		return epoch == other.epoch && seqno == other.seqno && type == other.type;
	}

	bool operator<(const BoundaryEvaluation& other) {
		// if (epoch, seqno) don't match, go by (epoch, seqno)
		if (epoch == other.epoch && seqno == other.seqno) {
			return type < other.type;
		}
		return epoch < other.epoch || (epoch == other.epoch && seqno < other.seqno);
	}

	bool isOlderThanOriginal(const BoundaryEvaluation& other) {
		return originalEpoch < other.originalEpoch ||
		       (originalEpoch == other.originalEpoch && originalSeqno < other.originalSeqno);
	}

	std::string toString() const {
		return fmt::format("{0} @ ({1}, {2})",
		                   type == BoundaryEvalType::UNKNOWN ? "unknown"
		                                                     : (type == BoundaryEvalType::MERGE ? "merge" : "split"),
		                   epoch,
		                   seqno);
	}
};

struct BlobManagerStats {
	CounterCollection cc;

	Counter granuleSplits;
	Counter granuleWriteHotSplits;
	Counter granuleMerges;
	Counter ccGranulesChecked;
	Counter ccRowsChecked;
	Counter ccBytesChecked;
	Counter ccMismatches;
	Counter ccTimeouts;
	Counter ccErrors;
	Counter purgesProcessed;
	Counter granulesFullyPurged;
	Counter granulesPartiallyPurged;
	Counter filesPurged;
	Counter granulesHitMedianLimit;

	Future<Void> logger;
	int64_t activeMerges;
	int64_t blockedAssignments;

	Version lastFlushVersion;
	Version lastMLogTruncationVersion;
	int64_t lastManifestSeqNo;
	int64_t lastManifestDumpTs;
	int64_t manifestSizeInBytes;

	// Current stats maintained for a given blob worker process
	explicit BlobManagerStats(UID id,
	                          double interval,
	                          int64_t epoch,
	                          std::unordered_map<UID, BlobWorkerInterface>* workers,
	                          std::map<Key, bool>* mergeHardBoundaries,
	                          std::unordered_map<Key, BlobGranuleMergeBoundary>* mergeBoundaries)
	  : cc("BlobManagerStats", id.toString()), granuleSplits("GranuleSplits", cc),
	    granuleWriteHotSplits("GranuleWriteHotSplits", cc), granuleMerges("GranuleMerges", cc),
	    ccGranulesChecked("CCGranulesChecked", cc), ccRowsChecked("CCRowsChecked", cc),
	    ccBytesChecked("CCBytesChecked", cc), ccMismatches("CCMismatches", cc), ccTimeouts("CCTimeouts", cc),
	    ccErrors("CCErrors", cc), purgesProcessed("PurgesProcessed", cc),
	    granulesFullyPurged("GranulesFullyPurged", cc), granulesPartiallyPurged("GranulesPartiallyPurged", cc),
	    filesPurged("FilesPurged", cc), granulesHitMedianLimit("GranulesHitMedianLimit", cc), activeMerges(0),
	    blockedAssignments(0), lastFlushVersion(0), lastMLogTruncationVersion(0), lastManifestSeqNo(0),
	    lastManifestDumpTs(0), manifestSizeInBytes(0) {
		specialCounter(cc, "WorkerCount", [workers]() { return workers->size(); });
		specialCounter(cc, "Epoch", [epoch]() { return epoch; });
		specialCounter(cc, "ActiveMerges", [this]() { return this->activeMerges; });
		specialCounter(cc, "HardBoundaries", [mergeHardBoundaries]() { return mergeHardBoundaries->size(); });
		specialCounter(cc, "SoftBoundaries", [mergeBoundaries]() { return mergeBoundaries->size(); });
		specialCounter(cc, "BlockedAssignments", [this]() { return this->blockedAssignments; });
		specialCounter(cc, "LastFlushVersion", [this]() { return this->lastFlushVersion; });
		specialCounter(cc, "LastMLogTruncationVersion", [this]() { return this->lastMLogTruncationVersion; });
		specialCounter(cc, "LastManifestSeqNo", [this]() { return this->lastManifestSeqNo; });
		specialCounter(cc, "LastManifestDumpTs", [this]() { return this->lastManifestDumpTs; });
		specialCounter(cc, "ManifestSizeInBytes", [this]() { return this->manifestSizeInBytes; });
		logger = cc.traceCounters("BlobManagerMetrics", id, interval, "BlobManagerMetrics");
	}
};

enum MergeCandidateState {
	MergeCandidateUnknown,
	MergeCandidateCannotMerge,
	MergeCandidateCanMerge,
	MergeCandidateMerging
};

// The current merge algorithm, skipping just granules that will be merge-eligible on the next pass, but not
// their neighbors, is optimal for guaranteeing merges to make progress where possible, with decently
// optimal but not globally optimal merge behavior.
// Alternative algorithms include not doing a two-pass consideration at all and immediately considering
// all merge candidates, which guarantees the most progress but pretty much guarantees undesirably
// suboptimal merge decisions, because of the time variance of granules becoming merge candidates. Or,
// also skipping adjacent eligible granules in addition to the one that will be eligible next pass,
// which ensures optimally large merges in a future pass, but adds decent delay to doing the merge. Or,
// smarter considering of merge candidates adjacent to the one that will be eligible next pass
// (depending on whether potential future merges with adjacent ones could include this candidate), which
// would be the best of both worlds, but would add a decent amount of code complexity.
struct MergeCandidateInfo {
	MergeCandidateState st;
	UID granuleID;
	Version startVersion;
	// This is if this candidate has been seen by the merge checker before.
	bool seen;

	MergeCandidateInfo() : st(MergeCandidateUnknown), startVersion(invalidVersion), seen(false) {}

	MergeCandidateInfo(MergeCandidateState st) : st(st), startVersion(invalidVersion), seen(false) {
		ASSERT(st != MergeCandidateCanMerge);
	}
	MergeCandidateInfo(UID granuleID, Version startVersion)
	  : st(MergeCandidateCanMerge), granuleID(granuleID), startVersion(startVersion), seen(false) {}

	bool canMerge() const { return st == MergeCandidateCanMerge; }

	bool mergeEligible() const { return st == MergeCandidateCanMerge && seen; }
};

struct BlobGranuleSplitPoints {
	Standalone<VectorRef<KeyRef>> keys;
	std::unordered_map<Key, BlobGranuleMergeBoundary> boundaries;
};

struct BlobManagerData : NonCopyable, ReferenceCounted<BlobManagerData> {
	UID id;
	Database db;
	Optional<Key> dcId;
	PromiseStream<Future<Void>> addActor;
	Promise<Void> doLockCheck;

	BlobManagerStats stats;

	Reference<BlobConnectionProvider> bstore;

	std::unordered_map<UID, BlobWorkerInterface> workersById;
	std::unordered_map<UID, BlobWorkerInfo> workerStats; // mapping between workerID -> workerStats
	std::deque<UID> recentBWAssignments;
	std::unordered_set<NetworkAddress> workerAddresses;
	std::unordered_set<UID> deadWorkers;
	std::unordered_map<UID, UID> workerAffinities;
	KeyRangeMap<UID> workerAssignments;
	KeyRangeActorMap assignsInProgress;
	KeyRangeMap<BoundaryEvaluation> boundaryEvaluations;
	KeyRangeMap<bool> knownBlobRanges;
	BGTenantMap tenantData;
	KeyRangeMap<MergeCandidateInfo> mergeCandidates; // granule range to granule id + start version.
	KeyRangeMap<Version> activeGranuleMerges; // range map of active granule merges, because range in boundaryEval
	                                          // doesn't correspond to merge range. invalidVersion is no merge,
	                                          // 0 is no merge version determined yet
	// TODO: consider switching to an iterator approach.
	std::map<Key, bool> mergeHardBoundaries;
	std::unordered_map<Key, BlobGranuleMergeBoundary> mergeBoundaries;
	CoalescedKeyRangeMap<bool> forcePurgingRanges;

	FlowLock concurrentMergeChecks;

	AsyncTrigger startRecruiting;
	Debouncer restartRecruiting;
	std::set<NetworkAddress> recruitingLocalities; // the addrs of the workers being recruited on
	AsyncVar<int> recruitingStream;
	ExclusionTracker exclusionTracker;

	Promise<Void> foundBlobWorkers;
	Promise<Void> doneRecovering;
	Promise<Void> loadedClientRanges;

	int64_t epoch;
	int64_t seqNo = 1;
	int64_t manifestDumperSeqNo = 1;
	bool enableManifestEncryption = false;
	AsyncTrigger backupTrigger;
	AsyncTrigger manifestCompletitionTrigger;

	Promise<Void> iAmReplaced;

	bool isFullRestoreMode = false;
	Reference<AsyncVar<ServerDBInfo> const> dbInfo;

	BlobManagerData(UID id,
	                Reference<AsyncVar<ServerDBInfo> const> dbInfo,
	                Database db,
	                Optional<Key> dcId,
	                int64_t epoch)
	  : id(id), db(db), dcId(dcId),
	    stats(id, SERVER_KNOBS->WORKER_LOGGING_INTERVAL, epoch, &workersById, &mergeHardBoundaries, &mergeBoundaries),
	    knownBlobRanges(false, normalKeys.end), tenantData(BGTenantMap(dbInfo)),
	    mergeCandidates(MergeCandidateInfo(MergeCandidateUnknown), normalKeys.end),
	    activeGranuleMerges(invalidVersion, normalKeys.end), forcePurgingRanges(false, normalKeys.end),
	    concurrentMergeChecks(SERVER_KNOBS->BLOB_MANAGER_CONCURRENT_MERGE_CHECKS),
	    restartRecruiting(SERVER_KNOBS->DEBOUNCE_RECRUITING_DELAY), recruitingStream(0), exclusionTracker(db),
	    epoch(epoch), dbInfo(dbInfo) {}

	// only initialize blob store if actually needed
	void initBStore() {
		if (!bstore.isValid() && SERVER_KNOBS->BG_METADATA_SOURCE != "tenant") {
			if (BM_DEBUG) {
				fmt::print("BM {} constructing backup container from {}\n", epoch, SERVER_KNOBS->BG_URL.c_str());
			}
			bstore = BlobConnectionProvider::newBlobConnectionProvider(SERVER_KNOBS->BG_URL);
			if (BM_DEBUG) {
				fmt::print("BM {} constructed backup container\n", epoch);
			}
		}
	}

	bool isMergeActive(const KeyRangeRef& range) {
		auto ranges = activeGranuleMerges.intersectingRanges(range);
		for (auto& it : ranges) {
			if (it.value() != invalidVersion) {
				return true;
			}
		}
		return false;
	}

	Version activeMergeVersion(const KeyRangeRef& range) {
		auto ranges = activeGranuleMerges.intersectingRanges(range);
		Version v = invalidVersion;
		for (auto& it : ranges) {
			v = std::max(v, it.cvalue());
		}
		return v;
	}

	// FIXME: is it possible for merge/split/re-merge to call this with same range but a different granule id or
	// startVersion? Unlikely but could cause weird history problems
	void setMergeCandidate(const KeyRangeRef& range, UID granuleID, Version startVersion) {
		// if this granule is not an active granule, it can't be merged
		auto gIt = workerAssignments.rangeContaining(range.begin);
		if (gIt->begin() != range.begin || gIt->end() != range.end) {
			CODE_PROBE(true, "non-active granule reported merge eligible, ignoring", probe::decoration::rare);
			if (BM_DEBUG) {
				fmt::print(
				    "BM {0} Ignoring Merge Candidate [{1} - {2}): range mismatch with active granule [{3} - {4})\n",
				    epoch,
				    range.begin.printable(),
				    range.end.printable(),
				    gIt->begin().printable(),
				    gIt->end().printable());
			}
			return;
		}
		// Want this to be idempotent. If a granule was already reported as merge-eligible, we want to use the existing
		// merge and mergeNow state.
		auto it = mergeCandidates.rangeContaining(range.begin);

		if (it->begin() == range.begin && it.end() == range.end) {
			if (it->cvalue().st != MergeCandidateCanMerge) {
				// same range, just update
				it->value() = MergeCandidateInfo(granuleID, startVersion);
			} else {
				// else no-op, but validate data
				ASSERT(granuleID == it->cvalue().granuleID);
				ASSERT(startVersion == it->cvalue().startVersion);
			}
		} else if (it->cvalue().st != MergeCandidateMerging) {
			mergeCandidates.insert(range, MergeCandidateInfo(granuleID, startVersion));
		}
	}

	void setMergeCandidate(const KeyRangeRef& range, MergeCandidateState st) {
		ASSERT(st != MergeCandidateCanMerge);
		mergeCandidates.insert(range, MergeCandidateInfo(st));
	}

	void clearMergeCandidate(const KeyRangeRef& range) { setMergeCandidate(range, MergeCandidateCannotMerge); }

	bool isForcePurging(const KeyRangeRef& range) {
		auto ranges = forcePurgingRanges.intersectingRanges(range);
		for (auto& it : ranges) {
			if (it.value()) {
				return true;
			}
		}
		return false;
	}

	bool maybeInjectTargetedRestart() {
		// inject a BW restart at most once per test
		if (g_network->isSimulated() && !g_simulator->speedUpSimulation &&
		    now() > g_simulator->injectTargetedBMRestartTime && iAmReplaced.canBeSet()) {
			CODE_PROBE(true, "Injecting BM targeted restart");
			TraceEvent("SimBMInjectTargetedRestart", id).log();
			g_simulator->injectTargetedBMRestartTime = std::numeric_limits<double>::max();
			iAmReplaced.send(Void());
			return true;
		}
		return false;
	}
};

// Helper function for alignKeys().
// This attempts to do truncation and compares with the last key in splitPoints.keys.
static void alignKeyBoundary(Reference<BlobManagerData> bmData,
                             Reference<GranuleTenantData> tenantData,
                             KeyRef key,
                             int offset,
                             BlobGranuleSplitPoints& splitPoints) {
	Standalone<VectorRef<KeyRef>>& keys = splitPoints.keys;
	std::unordered_map<Key, BlobGranuleMergeBoundary>& boundaries = splitPoints.boundaries;
	KeyRef alignedKey = key;
	Tuple t, t2;

	if (!offset) {
		keys.push_back_deep(keys.arena(), alignedKey);
		return;
	}

	// If this is tenant aware code.
	if (tenantData.isValid()) {
		alignedKey = alignedKey.removePrefix(tenantData->entry.prefix);
	}
	try {
		t = Tuple::unpackUserType(alignedKey, true);
		if (t.size() > offset) {
			t2 = t.subTuple(0, t.size() - offset);
			alignedKey = t2.pack();
		}
	} catch (Error& e) {
		if (e.code() != error_code_invalid_tuple_data_type) {
			throw;
		}
	}
	if (tenantData.isValid()) {
		alignedKey = alignedKey.withPrefix(tenantData->entry.prefix, keys.arena());
	}

	// Only add the alignedKey if it's larger than the last key. If it's the same, drop the split if not allowed.
	if (alignedKey <= keys.back()) {
		if (SERVER_KNOBS->BG_ENABLE_SPLIT_TRUNCATED) {
			// Set split boundary.
			BlobGranuleMergeBoundary boundary = { /*buddy=*/true };
			boundaries[key] = boundary;
			keys.push_back_deep(keys.arena(), key);
		} // else drop the split
	} else {
		keys.push_back_deep(keys.arena(), alignedKey);
	}
}

															#line 604 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via alignKeys()
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class AlignKeysActor>
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class AlignKeysActorState {
															#line 611 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	AlignKeysActorState(Reference<BlobManagerData> const& bmData,KeyRange const& granuleRange,Standalone<VectorRef<KeyRef>> const& splits) 
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   granuleRange(granuleRange),
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   splits(splits),
															#line 605 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   splitPoints(),
															#line 607 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   offset(SERVER_KNOBS->BG_KEY_TUPLE_TRUNCATE_OFFSET)
															#line 626 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("alignKeys", reinterpret_cast<unsigned long>(this));

	}
	~AlignKeysActorState() 
	{
		fdb_probe_actor_destroy("alignKeys", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 608 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (offset <= 0)
															#line 641 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 609 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				splitPoints.keys = splits;
															#line 610 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (!static_cast<AlignKeysActor*>(this)->SAV<BlobGranuleSplitPoints>::futures) { (void)(splitPoints); this->~AlignKeysActorState(); static_cast<AlignKeysActor*>(this)->destroy(); return 0; }
															#line 647 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				new (&static_cast<AlignKeysActor*>(this)->SAV< BlobGranuleSplitPoints >::value()) BlobGranuleSplitPoints(std::move(splitPoints)); // state_var_RVO
				this->~AlignKeysActorState();
				static_cast<AlignKeysActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 613 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitPoints.keys.push_back_deep(splitPoints.keys.arena(), splits.front());
															#line 615 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			idx = 1;
															#line 616 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tenantData = Reference<GranuleTenantData>();
															#line 617 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(tenantData, bmData->tenantData.getDataForGranule(granuleRange));
															#line 617 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<AlignKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 663 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<AlignKeysActor*>(this)->actor_wait_state = 1;
															#line 617 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AlignKeysActor, 0, Void >*>(static_cast<AlignKeysActor*>(this)));
															#line 668 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AlignKeysActorState();
		static_cast<AlignKeysActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 618 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		;
															#line 691 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 618 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		;
															#line 700 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AlignKeysActor*>(this)->actor_wait_state > 0) static_cast<AlignKeysActor*>(this)->actor_wait_state = 0;
		static_cast<AlignKeysActor*>(this)->ActorCallback< AlignKeysActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AlignKeysActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("alignKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AlignKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("alignKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AlignKeysActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("alignKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AlignKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("alignKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AlignKeysActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("alignKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AlignKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("alignKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 623 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(;idx < splits.size() - 1;idx++) {
															#line 624 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			alignKeyBoundary(bmData, tenantData, splits[idx], offset, splitPoints);
															#line 786 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 627 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitPoints.keys.push_back_deep(splitPoints.keys.arena(), splits.back());
															#line 629 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<AlignKeysActor*>(this)->SAV<BlobGranuleSplitPoints>::futures) { (void)(splitPoints); this->~AlignKeysActorState(); static_cast<AlignKeysActor*>(this)->destroy(); return 0; }
															#line 792 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<AlignKeysActor*>(this)->SAV< BlobGranuleSplitPoints >::value()) BlobGranuleSplitPoints(std::move(splitPoints)); // state_var_RVO
		this->~AlignKeysActorState();
		static_cast<AlignKeysActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 618 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!(SERVER_KNOBS->BG_METADATA_SOURCE == "tenant" && !tenantData.isValid()))
															#line 811 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 620 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(1.0);
															#line 620 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<AlignKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 819 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<AlignKeysActor*>(this)->actor_wait_state = 2;
															#line 620 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AlignKeysActor, 1, Void >*>(static_cast<AlignKeysActor*>(this)));
															#line 824 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 621 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(tenantData, bmData->tenantData.getDataForGranule(granuleRange));
															#line 621 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<AlignKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 848 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<AlignKeysActor*>(this)->actor_wait_state = 3;
															#line 621 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AlignKeysActor, 2, Void >*>(static_cast<AlignKeysActor*>(this)));
															#line 853 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 621 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_2 = store(tenantData, bmData->tenantData.getDataForGranule(granuleRange));
															#line 621 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<AlignKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 864 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<AlignKeysActor*>(this)->actor_wait_state = 3;
															#line 621 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AlignKeysActor, 2, Void >*>(static_cast<AlignKeysActor*>(this)));
															#line 869 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AlignKeysActor*>(this)->actor_wait_state > 0) static_cast<AlignKeysActor*>(this)->actor_wait_state = 0;
		static_cast<AlignKeysActor*>(this)->ActorCallback< AlignKeysActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AlignKeysActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("alignKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AlignKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("alignKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AlignKeysActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("alignKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AlignKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("alignKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AlignKeysActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("alignKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AlignKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("alignKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont3(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<AlignKeysActor*>(this)->actor_wait_state > 0) static_cast<AlignKeysActor*>(this)->actor_wait_state = 0;
		static_cast<AlignKeysActor*>(this)->ActorCallback< AlignKeysActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AlignKeysActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("alignKeys", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AlignKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("alignKeys", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< AlignKeysActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("alignKeys", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AlignKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("alignKeys", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< AlignKeysActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("alignKeys", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AlignKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("alignKeys", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	KeyRange granuleRange;
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Standalone<VectorRef<KeyRef>> splits;
															#line 605 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	BlobGranuleSplitPoints splitPoints;
															#line 607 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int offset;
															#line 615 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int idx;
															#line 616 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<GranuleTenantData> tenantData;
															#line 1050 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via alignKeys()
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class AlignKeysActor final : public Actor<BlobGranuleSplitPoints>, public ActorCallback< AlignKeysActor, 0, Void >, public ActorCallback< AlignKeysActor, 1, Void >, public ActorCallback< AlignKeysActor, 2, Void >, public FastAllocated<AlignKeysActor>, public AlignKeysActorState<AlignKeysActor> {
															#line 1055 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<AlignKeysActor>::operator new;
	using FastAllocated<AlignKeysActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12977031365470394880UL, 7893803799967002112UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobGranuleSplitPoints>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AlignKeysActor, 0, Void >;
friend struct ActorCallback< AlignKeysActor, 1, Void >;
friend struct ActorCallback< AlignKeysActor, 2, Void >;
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	AlignKeysActor(Reference<BlobManagerData> const& bmData,KeyRange const& granuleRange,Standalone<VectorRef<KeyRef>> const& splits) 
															#line 1074 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<BlobGranuleSplitPoints>(),
		   AlignKeysActorState<AlignKeysActor>(bmData, granuleRange, splits),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("alignKeys", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17645636835980548096UL, 12392887057365423616UL);
		ActorExecutionContextHelper __helper(static_cast<AlignKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("alignKeys");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("alignKeys", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AlignKeysActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AlignKeysActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< AlignKeysActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<BlobGranuleSplitPoints> alignKeys( Reference<BlobManagerData> const& bmData, KeyRange const& granuleRange, Standalone<VectorRef<KeyRef>> const& splits ) {
															#line 602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<BlobGranuleSplitPoints>(new AlignKeysActor(bmData, granuleRange, splits));
															#line 1109 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 631 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

// Find the next full db key for each split point
															#line 1115 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via nextFullKeys()
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class NextFullKeysActor>
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class NextFullKeysActorState {
															#line 1122 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	NextFullKeysActorState(Reference<BlobManagerData> const& bmData,Standalone<VectorRef<KeyRef>> const& keys) 
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   keys(keys),
															#line 635 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   result()
															#line 1133 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("nextFullKeys", reinterpret_cast<unsigned long>(this));

	}
	~NextFullKeysActorState() 
	{
		fdb_probe_actor_destroy("nextFullKeys", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 636 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			result.push_back_deep(result.arena(), keys.front());
															#line 638 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr = Transaction(bmData->db);
															#line 639 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			i = 1;
															#line 640 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			;
															#line 1154 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~NextFullKeysActorState();
		static_cast<NextFullKeysActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 657 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		result.push_back_deep(result.arena(), keys.back());
															#line 658 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<NextFullKeysActor*>(this)->SAV<Standalone<VectorRef<KeyRef>>>::futures) { (void)(result); this->~NextFullKeysActorState(); static_cast<NextFullKeysActor*>(this)->destroy(); return 0; }
															#line 1179 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<NextFullKeysActor*>(this)->SAV< Standalone<VectorRef<KeyRef>> >::value()) Standalone<VectorRef<KeyRef>>(std::move(result)); // state_var_RVO
		this->~NextFullKeysActorState();
		static_cast<NextFullKeysActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 640 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!(i < keys.size() - 1))
															#line 1198 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 641 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		;
															#line 1204 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 640 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		++i;
															#line 1226 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 642 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 643 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1244 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		try {
															#line 645 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr.getRange(firstGreaterOrEqual(keys[i]), lastLessThan(keys[i + 1]), GetRangeLimits(1));
															#line 645 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<NextFullKeysActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1250 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<NextFullKeysActor*>(this)->actor_wait_state = 1;
															#line 645 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< NextFullKeysActor, 0, RangeResult >*>(static_cast<NextFullKeysActor*>(this)));
															#line 1255 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 653 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr.onError(e);
															#line 653 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<NextFullKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 1292 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<NextFullKeysActor*>(this)->actor_wait_state = 2;
															#line 653 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< NextFullKeysActor, 1, Void >*>(static_cast<NextFullKeysActor*>(this)));
															#line 1297 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(RangeResult const& nextKeyRes,int loopDepth) 
	{
															#line 647 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (nextKeyRes.size() == 0)
															#line 1312 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 650 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		result.push_back_deep(result.arena(), nextKeyRes[0].key);
															#line 1318 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(RangeResult && nextKeyRes,int loopDepth) 
	{
															#line 647 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (nextKeyRes.size() == 0)
															#line 1327 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 650 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		result.push_back_deep(result.arena(), nextKeyRes[0].key);
															#line 1333 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(RangeResult const& nextKeyRes,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(nextKeyRes, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(RangeResult && nextKeyRes,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(std::move(nextKeyRes), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<NextFullKeysActor*>(this)->actor_wait_state > 0) static_cast<NextFullKeysActor*>(this)->actor_wait_state = 0;
		static_cast<NextFullKeysActor*>(this)->ActorCallback< NextFullKeysActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< NextFullKeysActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("nextFullKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<NextFullKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("nextFullKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< NextFullKeysActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("nextFullKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<NextFullKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("nextFullKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< NextFullKeysActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("nextFullKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<NextFullKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("nextFullKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<NextFullKeysActor*>(this)->actor_wait_state > 0) static_cast<NextFullKeysActor*>(this)->actor_wait_state = 0;
		static_cast<NextFullKeysActor*>(this)->ActorCallback< NextFullKeysActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< NextFullKeysActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("nextFullKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<NextFullKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("nextFullKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< NextFullKeysActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("nextFullKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<NextFullKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("nextFullKeys", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< NextFullKeysActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("nextFullKeys", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<NextFullKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("nextFullKeys", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Standalone<VectorRef<KeyRef>> keys;
															#line 635 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Standalone<VectorRef<KeyRef>> result;
															#line 638 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Transaction tr;
															#line 639 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int i;
															#line 1510 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via nextFullKeys()
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class NextFullKeysActor final : public Actor<Standalone<VectorRef<KeyRef>>>, public ActorCallback< NextFullKeysActor, 0, RangeResult >, public ActorCallback< NextFullKeysActor, 1, Void >, public FastAllocated<NextFullKeysActor>, public NextFullKeysActorState<NextFullKeysActor> {
															#line 1515 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<NextFullKeysActor>::operator new;
	using FastAllocated<NextFullKeysActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7364315279395928832UL, 3386525292575488512UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Standalone<VectorRef<KeyRef>>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< NextFullKeysActor, 0, RangeResult >;
friend struct ActorCallback< NextFullKeysActor, 1, Void >;
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	NextFullKeysActor(Reference<BlobManagerData> const& bmData,Standalone<VectorRef<KeyRef>> const& keys) 
															#line 1533 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<Standalone<VectorRef<KeyRef>>>(),
		   NextFullKeysActorState<NextFullKeysActor>(bmData, keys),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("nextFullKeys", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(526255464919385600UL, 15694378122349672192UL);
		ActorExecutionContextHelper __helper(static_cast<NextFullKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("nextFullKeys");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("nextFullKeys", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< NextFullKeysActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< NextFullKeysActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<Standalone<VectorRef<KeyRef>>> nextFullKeys( Reference<BlobManagerData> const& bmData, Standalone<VectorRef<KeyRef>> const& keys ) {
															#line 633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<Standalone<VectorRef<KeyRef>>>(new NextFullKeysActor(bmData, keys));
															#line 1567 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 660 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

// split recursively in the middle to guarantee roughly equal splits across different parts of key space
static void downsampleSplit(const Standalone<VectorRef<KeyRef>>& splits,
                            Standalone<VectorRef<KeyRef>>& out,
                            int startIdx,
                            int endIdx,
                            int remaining) {
	ASSERT(endIdx - startIdx >= remaining);
	ASSERT(remaining >= 0);
	if (remaining == 0) {
		return;
	}
	if (endIdx - startIdx == remaining) {
		out.append(out.arena(), splits.begin() + startIdx, remaining);
	} else {
		int mid = (startIdx + endIdx) / 2;
		int startCount = (remaining - 1) / 2;
		int endCount = remaining - startCount - 1;
		// ensure no infinite recursion
		ASSERT(mid != endIdx);
		ASSERT(mid + 1 != startIdx);
		downsampleSplit(splits, out, startIdx, mid, startCount);
		out.push_back(out.arena(), splits[mid]);
		downsampleSplit(splits, out, mid + 1, endIdx, endCount);
	}
}

															#line 1598 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via splitRange()
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class SplitRangeActor>
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class SplitRangeActorState {
															#line 1605 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	SplitRangeActorState(Reference<BlobManagerData> const& bmData,KeyRange const& range,bool const& writeHot,bool const& initialSplit,Optional<int> const& maxSplitPoints = Optional<int>()) 
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   range(range),
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   writeHot(writeHot),
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   initialSplit(initialSplit),
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   maxSplitPoints(maxSplitPoints),
															#line 692 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   splitPoints()
															#line 1622 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("splitRange", reinterpret_cast<unsigned long>(this));

	}
	~SplitRangeActorState() 
	{
		fdb_probe_actor_destroy("splitRange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 694 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (BM_DEBUG)
															#line 1638 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 695 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					fmt::print("Splitting new range [{0} - {1}): {2}\n", range.begin.printable(), range.end.printable(), writeHot ? "hot" : "normal");
															#line 1642 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
															#line 700 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				StrictFuture<StorageMetrics> __when_expr_0 = bmData->db->getStorageMetrics(range, CLIENT_KNOBS->TOO_MANY);
															#line 700 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (static_cast<SplitRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1648 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<SplitRangeActor*>(this)->actor_wait_state = 1;
															#line 700 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SplitRangeActor, 0, StorageMetrics >*>(static_cast<SplitRangeActor*>(this)));
															#line 1653 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SplitRangeActorState();
		static_cast<SplitRangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 809 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 1683 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 810 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 1687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 813 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent(SevError, "BlobManagerUnexpectedErrorSplitRange", bmData->id) .error(e) .detail("Epoch", bmData->epoch);
															#line 816 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ASSERT_WE_THINK(false);
															#line 819 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (bmData->iAmReplaced.canBeSet())
															#line 1695 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 820 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->iAmReplaced.sendError(e);
															#line 1699 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 822 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 1703 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 702 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 1717 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 703 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Estimated bytes for [{0} - {1}): {2}\n", range.begin.printable(), range.end.printable(), estimated.bytes);
															#line 1721 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 709 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int64_t splitThreshold = SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES;
															#line 710 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!initialSplit)
															#line 1727 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 716 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitThreshold = (splitThreshold * 4) / 3;
															#line 1731 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 722 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (writeHot)
															#line 1735 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 723 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitThreshold /= 3;
															#line 1739 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 725 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		CODE_PROBE(writeHot, "Change feed write hot split");
															#line 726 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (estimated.bytes > splitThreshold)
															#line 1745 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 728 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitMetrics = StorageMetrics();
															#line 729 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitMetrics.bytes = SERVER_KNOBS->BG_SNAPSHOT_FILE_TARGET_BYTES;
															#line 730 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitMetrics.bytesWrittenPerKSecond = SERVER_KNOBS->SHARD_SPLIT_BYTES_PER_KSEC;
															#line 731 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (writeHot)
															#line 1755 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 732 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				splitMetrics.bytesWrittenPerKSecond = std::min(splitMetrics.bytesWrittenPerKSecond, estimated.bytesWrittenPerKSecond / 2);
															#line 734 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				splitMetrics.bytesWrittenPerKSecond = std::max(splitMetrics.bytesWrittenPerKSecond, SERVER_KNOBS->SHARD_MIN_BYTES_PER_KSEC);
															#line 1761 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 737 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitMetrics.iosPerKSecond = splitMetrics.infinity;
															#line 738 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitMetrics.bytesReadPerKSecond = splitMetrics.infinity;
															#line 740 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			resultStream = PromiseStream<Key>();
															#line 741 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			keys = Standalone<VectorRef<KeyRef>>();
															#line 746 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			streamFuture = bmData->db->splitStorageMetricsStream( resultStream, range, splitMetrics, StorageMetrics(), splitMetrics.bytes / 3);
															#line 748 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			;
															#line 1775 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = a_body1cont2loopHead1(loopDepth);
		}
		else
		{
															#line 800 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(writeHot, "Not splitting write-hot because granules would be too small");
															#line 801 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 1784 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 802 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				printf("Not splitting range\n");
															#line 1788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 804 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitPoints.keys.push_back_deep(splitPoints.keys.arena(), range.begin);
															#line 805 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitPoints.keys.push_back_deep(splitPoints.keys.arena(), range.end);
															#line 806 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<SplitRangeActor*>(this)->SAV<BlobGranuleSplitPoints>::futures) { (void)(splitPoints); this->~SplitRangeActorState(); static_cast<SplitRangeActor*>(this)->destroy(); return 0; }
															#line 1796 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<SplitRangeActor*>(this)->SAV< BlobGranuleSplitPoints >::value()) BlobGranuleSplitPoints(std::move(splitPoints)); // state_var_RVO
			this->~SplitRangeActorState();
			static_cast<SplitRangeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}

		return loopDepth;
	}
	int a_body1when1(StorageMetrics const& __estimated,int loopDepth) 
	{
															#line 700 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		estimated = __estimated;
															#line 1809 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(StorageMetrics && __estimated,int loopDepth) 
	{
		estimated = std::move(__estimated);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SplitRangeActor*>(this)->actor_wait_state > 0) static_cast<SplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<SplitRangeActor*>(this)->ActorCallback< SplitRangeActor, 0, StorageMetrics >::remove();

	}
	void a_callback_fire(ActorCallback< SplitRangeActor, 0, StorageMetrics >*,StorageMetrics const& value) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SplitRangeActor, 0, StorageMetrics >*,StorageMetrics && value) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SplitRangeActor, 0, StorageMetrics >*,Error err) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont8(int loopDepth) 
	{
															#line 760 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (keys.size() > 2)
															#line 1888 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 762 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Standalone<VectorRef<KeyRef>>> __when_expr_2 = nextFullKeys(bmData, keys);
															#line 762 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<SplitRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1894 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont8when1(__when_expr_2.get(), loopDepth); };
			static_cast<SplitRangeActor*>(this)->actor_wait_state = 3;
															#line 762 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SplitRangeActor, 2, Standalone<VectorRef<KeyRef>> >*>(static_cast<SplitRangeActor*>(this)));
															#line 1899 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 792 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitPoints.keys = keys;
															#line 1906 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = a_body1cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
		try {
															#line 750 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			auto __when_expr_1 = resultStream.getFuture();
															#line 1924 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			static_assert(std::is_same<decltype(__when_expr_1), FutureStream<Key>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<Key>>::value, "invalid type");
															#line 750 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<SplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont2loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1928 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont2loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont2loopBody1when1(__when_expr_1.pop(), loopDepth); };
			static_cast<SplitRangeActor*>(this)->actor_wait_state = 2;
															#line 750 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< SplitRangeActor, 1, Key >*>(static_cast<SplitRangeActor*>(this)));
															#line 1933 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont2loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont2loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 753 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (e.code() != error_code_end_of_stream)
															#line 1968 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 754 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				return a_body1Catch2(e, std::max(0, loopDepth - 1));
															#line 1972 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
			return a_body1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Key const& k,int loopDepth) 
	{
															#line 751 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		keys.push_back_deep(keys.arena(), k);
															#line 1988 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Key && k,int loopDepth) 
	{
															#line 751 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		keys.push_back_deep(keys.arena(), k);
															#line 1997 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Key const& k,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(k, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Key && k,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(std::move(k), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SplitRangeActor*>(this)->actor_wait_state > 0) static_cast<SplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<SplitRangeActor*>(this)->ActorSingleCallback< SplitRangeActor, 1, Key >::remove();

	}
	void a_callback_fire(ActorSingleCallback< SplitRangeActor, 1, Key >*,Key const& value) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< SplitRangeActor, 1, Key >*,Key && value) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< SplitRangeActor, 1, Key >*,Error err) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopBody1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont10(int loopDepth) 
	{
															#line 795 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitPoints.keys.size() >= 2);
															#line 796 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitPoints.keys.front() == range.begin);
															#line 797 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitPoints.keys.back() == range.end);
															#line 798 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<SplitRangeActor*>(this)->SAV<BlobGranuleSplitPoints>::futures) { (void)(splitPoints); this->~SplitRangeActorState(); static_cast<SplitRangeActor*>(this)->destroy(); return 0; }
															#line 2100 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<SplitRangeActor*>(this)->SAV< BlobGranuleSplitPoints >::value()) BlobGranuleSplitPoints(std::move(splitPoints)); // state_var_RVO
		this->~SplitRangeActorState();
		static_cast<SplitRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11(int loopDepth) 
	{
															#line 763 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<BlobGranuleSplitPoints> __when_expr_3 = alignKeys(bmData, range, fullKeys);
															#line 763 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<SplitRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2114 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1cont11when1(__when_expr_3.get(), loopDepth); };
		static_cast<SplitRangeActor*>(this)->actor_wait_state = 4;
															#line 763 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< SplitRangeActor, 3, BlobGranuleSplitPoints >*>(static_cast<SplitRangeActor*>(this)));
															#line 2119 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont8when1(Standalone<VectorRef<KeyRef>> const& __fullKeys,int loopDepth) 
	{
															#line 762 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		fullKeys = __fullKeys;
															#line 2128 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont8when1(Standalone<VectorRef<KeyRef>> && __fullKeys,int loopDepth) 
	{
		fullKeys = std::move(__fullKeys);
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<SplitRangeActor*>(this)->actor_wait_state > 0) static_cast<SplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<SplitRangeActor*>(this)->ActorCallback< SplitRangeActor, 2, Standalone<VectorRef<KeyRef>> >::remove();

	}
	void a_callback_fire(ActorCallback< SplitRangeActor, 2, Standalone<VectorRef<KeyRef>> >*,Standalone<VectorRef<KeyRef>> const& value) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< SplitRangeActor, 2, Standalone<VectorRef<KeyRef>> >*,Standalone<VectorRef<KeyRef>> && value) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< SplitRangeActor, 2, Standalone<VectorRef<KeyRef>> >*,Error err) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont11cont1(BlobGranuleSplitPoints const& _splitPoints,int loopDepth) 
	{
															#line 764 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitPoints = _splitPoints;
															#line 767 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (maxSplitPoints.present() && splitPoints.keys.size() > maxSplitPoints.get())
															#line 2209 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 769 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "downsampling granule split because fanout too high");
															#line 770 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			Standalone<VectorRef<KeyRef>> downsampledKeys;
															#line 771 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			downsampledKeys.arena().dependsOn(fullKeys.arena());
															#line 772 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			downsampledKeys.push_back(downsampledKeys.arena(), fullKeys.front());
															#line 776 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			downsampleSplit(fullKeys, downsampledKeys, 1, fullKeys.size() - 1, maxSplitPoints.get() - 2);
															#line 778 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			downsampledKeys.push_back(downsampledKeys.arena(), fullKeys.back());
															#line 779 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ASSERT(downsampledKeys.size() == maxSplitPoints.get());
															#line 780 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 2227 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 781 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("Downsampled split [{0} - {1}) from {2} -> {3} granules\n", range.begin.printable(), range.end.printable(), splitPoints.keys.size() - 1, maxSplitPoints.get() - 1);
															#line 2231 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_4 = store(splitPoints, alignKeys(bmData, range, downsampledKeys));
															#line 788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<SplitRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2237 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), loopDepth); else return a_body1cont11cont1when1(__when_expr_4.get(), loopDepth); };
			static_cast<SplitRangeActor*>(this)->actor_wait_state = 5;
															#line 788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< SplitRangeActor, 4, Void >*>(static_cast<SplitRangeActor*>(this)));
															#line 2242 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont11cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont1(BlobGranuleSplitPoints && _splitPoints,int loopDepth) 
	{
															#line 764 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitPoints = _splitPoints;
															#line 767 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (maxSplitPoints.present() && splitPoints.keys.size() > maxSplitPoints.get())
															#line 2258 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 769 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "downsampling granule split because fanout too high");
															#line 770 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			Standalone<VectorRef<KeyRef>> downsampledKeys;
															#line 771 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			downsampledKeys.arena().dependsOn(fullKeys.arena());
															#line 772 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			downsampledKeys.push_back(downsampledKeys.arena(), fullKeys.front());
															#line 776 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			downsampleSplit(fullKeys, downsampledKeys, 1, fullKeys.size() - 1, maxSplitPoints.get() - 2);
															#line 778 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			downsampledKeys.push_back(downsampledKeys.arena(), fullKeys.back());
															#line 779 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ASSERT(downsampledKeys.size() == maxSplitPoints.get());
															#line 780 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 2276 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 781 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("Downsampled split [{0} - {1}) from {2} -> {3} granules\n", range.begin.printable(), range.end.printable(), splitPoints.keys.size() - 1, maxSplitPoints.get() - 1);
															#line 2280 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_4 = store(splitPoints, alignKeys(bmData, range, downsampledKeys));
															#line 788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<SplitRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2286 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), loopDepth); else return a_body1cont11cont1when1(__when_expr_4.get(), loopDepth); };
			static_cast<SplitRangeActor*>(this)->actor_wait_state = 5;
															#line 788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< SplitRangeActor, 4, Void >*>(static_cast<SplitRangeActor*>(this)));
															#line 2291 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont11cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11when1(BlobGranuleSplitPoints const& _splitPoints,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(_splitPoints, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(BlobGranuleSplitPoints && _splitPoints,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(std::move(_splitPoints), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<SplitRangeActor*>(this)->actor_wait_state > 0) static_cast<SplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<SplitRangeActor*>(this)->ActorCallback< SplitRangeActor, 3, BlobGranuleSplitPoints >::remove();

	}
	void a_callback_fire(ActorCallback< SplitRangeActor, 3, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints const& value) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< SplitRangeActor, 3, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints && value) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< SplitRangeActor, 3, BlobGranuleSplitPoints >*,Error err) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont11cont2(int loopDepth) 
	{
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont3(Void const& _,int loopDepth) 
	{
															#line 789 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitPoints.keys.size() <= maxSplitPoints.get());
															#line 2386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont11cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont3(Void && _,int loopDepth) 
	{
															#line 789 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitPoints.keys.size() <= maxSplitPoints.get());
															#line 2395 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont11cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<SplitRangeActor*>(this)->actor_wait_state > 0) static_cast<SplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<SplitRangeActor*>(this)->ActorCallback< SplitRangeActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SplitRangeActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont11cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< SplitRangeActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont11cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< SplitRangeActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	KeyRange range;
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	bool writeHot;
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	bool initialSplit;
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Optional<int> maxSplitPoints;
															#line 692 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	BlobGranuleSplitPoints splitPoints;
															#line 700 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	StorageMetrics estimated;
															#line 728 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	StorageMetrics splitMetrics;
															#line 740 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	PromiseStream<Key> resultStream;
															#line 741 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Standalone<VectorRef<KeyRef>> keys;
															#line 746 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Future<Void> streamFuture;
															#line 762 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Standalone<VectorRef<KeyRef>> fullKeys;
															#line 2499 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via splitRange()
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class SplitRangeActor final : public Actor<BlobGranuleSplitPoints>, public ActorCallback< SplitRangeActor, 0, StorageMetrics >, public ActorSingleCallback< SplitRangeActor, 1, Key >, public ActorCallback< SplitRangeActor, 2, Standalone<VectorRef<KeyRef>> >, public ActorCallback< SplitRangeActor, 3, BlobGranuleSplitPoints >, public ActorCallback< SplitRangeActor, 4, Void >, public FastAllocated<SplitRangeActor>, public SplitRangeActorState<SplitRangeActor> {
															#line 2504 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<SplitRangeActor>::operator new;
	using FastAllocated<SplitRangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10977157294749860096UL, 8204767612727399424UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<BlobGranuleSplitPoints>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SplitRangeActor, 0, StorageMetrics >;
friend struct ActorSingleCallback< SplitRangeActor, 1, Key >;
friend struct ActorCallback< SplitRangeActor, 2, Standalone<VectorRef<KeyRef>> >;
friend struct ActorCallback< SplitRangeActor, 3, BlobGranuleSplitPoints >;
friend struct ActorCallback< SplitRangeActor, 4, Void >;
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	SplitRangeActor(Reference<BlobManagerData> const& bmData,KeyRange const& range,bool const& writeHot,bool const& initialSplit,Optional<int> const& maxSplitPoints = Optional<int>()) 
															#line 2525 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<BlobGranuleSplitPoints>(),
		   SplitRangeActorState<SplitRangeActor>(bmData, range, writeHot, initialSplit, maxSplitPoints),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("splitRange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4779449971748862976UL, 15620764825018788864UL);
		ActorExecutionContextHelper __helper(static_cast<SplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("splitRange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("splitRange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SplitRangeActor, 0, StorageMetrics >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorSingleCallback< SplitRangeActor, 1, Key >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< SplitRangeActor, 2, Standalone<VectorRef<KeyRef>> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< SplitRangeActor, 3, BlobGranuleSplitPoints >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< SplitRangeActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<BlobGranuleSplitPoints> splitRange( Reference<BlobManagerData> const& bmData, KeyRange const& range, bool const& writeHot, bool const& initialSplit, Optional<int> const& maxSplitPoints = Optional<int>() ) {
															#line 687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<BlobGranuleSplitPoints>(new SplitRangeActor(bmData, range, writeHot, initialSplit, maxSplitPoints));
															#line 2562 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 825 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

// Picks a worker with the fewest number of already assigned ranges.
// If there is a tie, picks one such worker at random.
															#line 2569 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via pickWorkerForAssign()
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class PickWorkerForAssignActor>
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class PickWorkerForAssignActorState {
															#line 2576 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	PickWorkerForAssignActorState(Reference<BlobManagerData> const& bmData,Optional<std::pair<UID, Error>> const& previousFailure) 
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   previousFailure(previousFailure)
															#line 2585 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("pickWorkerForAssign", reinterpret_cast<unsigned long>(this));

	}
	~PickWorkerForAssignActorState() 
	{
		fdb_probe_actor_destroy("pickWorkerForAssign", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 831 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			;
															#line 2600 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PickWorkerForAssignActorState();
		static_cast<PickWorkerForAssignActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 853 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int minGranulesAssigned = INT_MAX;
															#line 858 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (bmData->workerStats.size() >= 2 && previousFailure.present() && previousFailure.get().second.code() == error_code_blob_worker_full && deterministicRandom()->coinflip())
															#line 2625 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 860 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "randomly picking worker due to blob_worker_full");
															#line 861 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			std::vector<UID> randomWorkers;
															#line 862 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			randomWorkers.reserve(bmData->workerStats.size());
															#line 863 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for( auto& it : bmData->workerStats ) {
															#line 864 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (it.first != previousFailure.get().first)
															#line 2637 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 865 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					randomWorkers.push_back(it.first);
															#line 2641 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
			}
															#line 868 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ASSERT(!randomWorkers.empty());
															#line 869 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			int randomIdx = deterministicRandom()->randomInt(0, randomWorkers.size());
															#line 870 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 2650 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 871 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("picked worker {0} randomly since previous attempt got blob_worker_full\n", randomWorkers[randomIdx].toString().substr(0, 5));
															#line 2654 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 875 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<PickWorkerForAssignActor*>(this)->SAV<UID>::futures) { (void)(randomWorkers[randomIdx]); this->~PickWorkerForAssignActorState(); static_cast<PickWorkerForAssignActor*>(this)->destroy(); return 0; }
															#line 2658 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<PickWorkerForAssignActor*>(this)->SAV< UID >::value()) UID(randomWorkers[randomIdx]);
			this->~PickWorkerForAssignActorState();
			static_cast<PickWorkerForAssignActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 879 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		std::vector<int> medianCalc;
															#line 881 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		std::vector<std::tuple<UID, int, int>> eligibleWorkers;
															#line 883 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		Optional<int> excludeIfRecentOver;
															#line 884 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for( auto const& worker : bmData->workerStats ) {
															#line 885 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			UID currId = worker.first;
															#line 888 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (bmData->workerStats.size() >= 2 && previousFailure.present() && previousFailure.get().first == currId)
															#line 2676 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
				continue;
			}
															#line 892 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			eligibleWorkers.push_back({ currId, worker.second.numGranulesAssigned, worker.second.recentGranulesAssigned });
															#line 893 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			medianCalc.push_back(worker.second.recentGranulesAssigned);
															#line 2684 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 896 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (SERVER_KNOBS->BLOB_MANAGER_ENABLE_MEDIAN_ASSIGNMENT_LIMITING && medianCalc.size() > 1 && bmData->recentBWAssignments.size() >= SERVER_KNOBS->BLOB_MANAGER_MEDIAN_ASSIGNMENT_MIN_SAMPLES_PER_WORKER * bmData->workerStats.size())
															#line 2688 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 899 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "blob manager enabling median assignment limiting");
															#line 901 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			std::sort(medianCalc.begin(), medianCalc.end());
															#line 903 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			int medianIdx = (medianCalc.size() - 1) / 2;
															#line 905 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			double multiplyFactor = std::max(1.0, SERVER_KNOBS->BLOB_MANAGER_MEDIAN_ASSIGNMENT_ALLOWANCE);
															#line 906 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			excludeIfRecentOver = std::max(1, (int)(multiplyFactor * medianCalc[medianIdx]));
															#line 2700 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 909 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		std::vector<UID> finalEligibleWorkers;
															#line 910 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bool anyOverLimit = false;
															#line 911 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for( auto& it : eligibleWorkers ) {
															#line 912 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			UID currId = std::get<0>(it);
															#line 913 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			int granulesAssigned = std::get<1>(it);
															#line 914 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			int recentGranulesAssigned = std::get<2>(it);
															#line 916 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (excludeIfRecentOver.present() && recentGranulesAssigned > excludeIfRecentOver.get())
															#line 2716 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 917 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				anyOverLimit = true;
															#line 2720 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				continue;
			}
															#line 921 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (granulesAssigned <= minGranulesAssigned)
															#line 2725 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 922 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (granulesAssigned < minGranulesAssigned)
															#line 2729 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 923 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					finalEligibleWorkers.clear();
															#line 924 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					minGranulesAssigned = granulesAssigned;
															#line 2735 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
															#line 926 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				finalEligibleWorkers.emplace_back(currId);
															#line 2739 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
															#line 930 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (anyOverLimit)
															#line 2744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 931 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "BM excluding BW due to median assignment algorithm");
															#line 932 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			++bmData->stats.granulesHitMedianLimit;
															#line 2750 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 936 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(finalEligibleWorkers.size() > 0);
															#line 937 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int idx = deterministicRandom()->randomInt(0, finalEligibleWorkers.size());
															#line 938 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 2758 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 939 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("picked worker {0}, which has a minimal number ({1}) of granules assigned\n", finalEligibleWorkers[idx].toString().substr(0, 5), minGranulesAssigned);
															#line 2762 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 944 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<PickWorkerForAssignActor*>(this)->SAV<UID>::futures) { (void)(finalEligibleWorkers[idx]); this->~PickWorkerForAssignActorState(); static_cast<PickWorkerForAssignActor*>(this)->destroy(); return 0; }
															#line 2766 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<PickWorkerForAssignActor*>(this)->SAV< UID >::value()) UID(finalEligibleWorkers[idx]);
		this->~PickWorkerForAssignActorState();
		static_cast<PickWorkerForAssignActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 832 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		wasZeroWorkers = false;
															#line 833 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		;
															#line 2787 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 842 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (wasZeroWorkers)
															#line 2809 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 845 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_1 = delay(0.1);
															#line 845 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PickWorkerForAssignActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2815 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<PickWorkerForAssignActor*>(this)->actor_wait_state = 2;
															#line 845 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PickWorkerForAssignActor, 1, Void >*>(static_cast<PickWorkerForAssignActor*>(this)));
															#line 2820 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 833 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!(bmData->workerStats.size() == 0))
															#line 2841 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 834 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		wasZeroWorkers = true;
															#line 835 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		CODE_PROBE(true, "BM wants to assign range, but no workers available");
															#line 836 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 2851 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 837 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("BM {0} waiting for blob workers before assigning granules\n", bmData->epoch);
															#line 2855 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 839 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->restartRecruiting.trigger();
															#line 840 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_0 = bmData->recruitingStream.onChange() || bmData->foundBlobWorkers.getFuture();
															#line 840 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PickWorkerForAssignActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 2863 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<PickWorkerForAssignActor*>(this)->actor_wait_state = 1;
															#line 840 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PickWorkerForAssignActor, 0, Void >*>(static_cast<PickWorkerForAssignActor*>(this)));
															#line 2868 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PickWorkerForAssignActor*>(this)->actor_wait_state > 0) static_cast<PickWorkerForAssignActor*>(this)->actor_wait_state = 0;
		static_cast<PickWorkerForAssignActor*>(this)->ActorCallback< PickWorkerForAssignActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PickWorkerForAssignActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PickWorkerForAssignActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PickWorkerForAssignActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PickWorkerForAssignActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PickWorkerForAssignActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PickWorkerForAssignActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 847 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (bmData->workerStats.size() != 0)
															#line 2977 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PickWorkerForAssignActor*>(this)->actor_wait_state > 0) static_cast<PickWorkerForAssignActor*>(this)->actor_wait_state = 0;
		static_cast<PickWorkerForAssignActor*>(this)->ActorCallback< PickWorkerForAssignActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PickWorkerForAssignActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PickWorkerForAssignActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PickWorkerForAssignActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PickWorkerForAssignActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PickWorkerForAssignActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PickWorkerForAssignActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Optional<std::pair<UID, Error>> previousFailure;
															#line 832 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	bool wasZeroWorkers;
															#line 3078 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via pickWorkerForAssign()
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class PickWorkerForAssignActor final : public Actor<UID>, public ActorCallback< PickWorkerForAssignActor, 0, Void >, public ActorCallback< PickWorkerForAssignActor, 1, Void >, public FastAllocated<PickWorkerForAssignActor>, public PickWorkerForAssignActorState<PickWorkerForAssignActor> {
															#line 3083 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<PickWorkerForAssignActor>::operator new;
	using FastAllocated<PickWorkerForAssignActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1208650691039039744UL, 3205423406337623296UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<UID>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PickWorkerForAssignActor, 0, Void >;
friend struct ActorCallback< PickWorkerForAssignActor, 1, Void >;
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	PickWorkerForAssignActor(Reference<BlobManagerData> const& bmData,Optional<std::pair<UID, Error>> const& previousFailure) 
															#line 3101 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<UID>(),
		   PickWorkerForAssignActorState<PickWorkerForAssignActor>(bmData, previousFailure),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15544713192263436032UL, 15939660962383161344UL);
		ActorExecutionContextHelper __helper(static_cast<PickWorkerForAssignActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("pickWorkerForAssign");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("pickWorkerForAssign", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PickWorkerForAssignActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PickWorkerForAssignActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<UID> pickWorkerForAssign( Reference<BlobManagerData> const& bmData, Optional<std::pair<UID, Error>> const& previousFailure ) {
															#line 828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<UID>(new PickWorkerForAssignActor(bmData, previousFailure));
															#line 3135 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 946 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

// circular dependency between handleRangeAssign and doRangeAssignment
static bool handleRangeAssign(Reference<BlobManagerData> bmData, RangeAssignment assignment);

															#line 3143 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via doRangeAssignment()
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class DoRangeAssignmentActor>
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class DoRangeAssignmentActorState {
															#line 3150 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	DoRangeAssignmentActorState(Reference<BlobManagerData> const& bmData,RangeAssignment const& assignment,Optional<UID> const& workerID,int64_t const& epoch,int64_t const& seqNo) 
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   assignment(assignment),
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   workerID(workerID),
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   epoch(epoch),
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   seqNo(seqNo),
															#line 955 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   blockedWaitingForWorker(false)
															#line 3167 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("doRangeAssignment", reinterpret_cast<unsigned long>(this));

	}
	~DoRangeAssignmentActorState() 
	{
		fdb_probe_actor_destroy("doRangeAssignment", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 959 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (BUGGIFY_WITH_PROB(0.05))
															#line 3183 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 960 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					StrictFuture<Void> __when_expr_0 = delay(deterministicRandom()->random01());
															#line 960 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3189 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 1;
															#line 960 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoRangeAssignmentActor, 0, Void >*>(static_cast<DoRangeAssignmentActor*>(this)));
															#line 3194 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
															#line 964 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					StrictFuture<Void> __when_expr_1 = delay(0.0);
															#line 964 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3203 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
					static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 2;
															#line 964 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoRangeAssignmentActor, 1, Void >*>(static_cast<DoRangeAssignmentActor*>(this)));
															#line 3208 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
					loopDepth = 0;
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoRangeAssignmentActorState();
		static_cast<DoRangeAssignmentActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1019 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 3238 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1020 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("BM {0} {1} range [{2} - {3}) @ ({4}, {5}) to {6}\n", bmData->epoch, assignment.isAssign ? "assigning" : "revoking", assignment.keyRange.begin.printable(), assignment.keyRange.end.printable(), epoch, seqNo, workerID.get().toString());
															#line 3242 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		try {
															#line 1031 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (assignment.isAssign)
															#line 3247 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1032 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				ASSERT(assignment.assign.present());
															#line 1033 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				ASSERT(!assignment.revoke.present());
															#line 1035 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				AssignBlobRangeRequest req;
															#line 1036 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				req.keyRange = KeyRangeRef(StringRef(req.arena, assignment.keyRange.begin), StringRef(req.arena, assignment.keyRange.end));
															#line 1038 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				req.managerEpoch = epoch;
															#line 1039 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				req.managerSeqno = seqNo;
															#line 1040 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				req.type = assignment.assign.get().type;
															#line 1043 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (!bmData->workersById.contains(workerID.get()))
															#line 3265 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 1044 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					return a_body1cont1Catch1(no_more_servers(), loopDepth);
															#line 3269 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
															#line 1046 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				assignFuture = bmData->workersById[workerID.get()].assignBlobRangeRequest.getReply(req);
															#line 1048 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (BUGGIFY)
															#line 3275 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 1050 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					StrictFuture<Void> __when_expr_3 = delay(0);
															#line 1050 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3281 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
					if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1when1(__when_expr_3.get(), loopDepth); };
					static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 4;
															#line 1050 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DoRangeAssignmentActor, 3, Void >*>(static_cast<DoRangeAssignmentActor*>(this)));
															#line 3286 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont19(loopDepth);
				}
			}
			else
			{
															#line 1065 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				ASSERT(!assignment.assign.present());
															#line 1066 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				ASSERT(assignment.revoke.present());
															#line 1068 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				RevokeBlobRangeRequest req;
															#line 1069 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				req.keyRange = KeyRangeRef(StringRef(req.arena, assignment.keyRange.begin), StringRef(req.arena, assignment.keyRange.end));
															#line 1071 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				req.managerEpoch = epoch;
															#line 1072 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				req.managerSeqno = seqNo;
															#line 1073 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				req.dispose = assignment.revoke.get().dispose;
															#line 1076 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (bmData->workersById.contains(workerID.get()))
															#line 3312 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 1077 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					StrictFuture<Void> __when_expr_5 = bmData->workersById[workerID.get()].revokeBlobRangeRequest.getReply(req);
															#line 1077 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3318 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
					if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1when2(__when_expr_5.get(), loopDepth); };
					static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 6;
															#line 1077 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< DoRangeAssignmentActor, 5, Void >*>(static_cast<DoRangeAssignmentActor*>(this)));
															#line 3323 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
															#line 1079 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					if (!static_cast<DoRangeAssignmentActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoRangeAssignmentActorState(); static_cast<DoRangeAssignmentActor*>(this)->destroy(); return 0; }
															#line 3330 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
					new (&static_cast<DoRangeAssignmentActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~DoRangeAssignmentActorState();
					static_cast<DoRangeAssignmentActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1013 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (assignment.previousFailure.present() || blockedWaitingForWorker)
															#line 3351 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1014 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				--bmData->stats.blockedAssignments;
															#line 3355 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1016 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 3359 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 966 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!workerID.present())
															#line 3373 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 967 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ASSERT(assignment.isAssign && assignment.assign.get().type != AssignRequestType::Continue);
															#line 969 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			blockedWaitingForWorker = true;
															#line 970 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!assignment.previousFailure.present())
															#line 3381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 972 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				++bmData->stats.blockedAssignments;
															#line 3385 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 975 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<UID> __when_expr_2 = pickWorkerForAssign(bmData, assignment.previousFailure);
															#line 975 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3391 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 3;
															#line 975 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoRangeAssignmentActor, 2, UID >*>(static_cast<DoRangeAssignmentActor*>(this)));
															#line 3396 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state > 0) static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 0;
		static_cast<DoRangeAssignmentActor*>(this)->ActorCallback< DoRangeAssignmentActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoRangeAssignmentActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state > 0) static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 0;
		static_cast<DoRangeAssignmentActor*>(this)->ActorCallback< DoRangeAssignmentActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoRangeAssignmentActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(UID const& _workerId,int loopDepth) 
	{
															#line 976 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 3590 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 977 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Chose BW {0} for seqno {1} in BM {2}\n", _workerId.toString(), seqNo, bmData->epoch);
															#line 3594 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 979 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		workerID = _workerId;
															#line 981 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (SERVER_KNOBS->BLOB_MANAGER_ENABLE_MEDIAN_ASSIGNMENT_LIMITING)
															#line 3600 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 984 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->recentBWAssignments.push_back(workerID.get());
															#line 985 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->workerStats[workerID.get()].recentGranulesAssigned++;
															#line 987 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (bmData->recentBWAssignments.size() >= SERVER_KNOBS->BLOB_MANAGER_MEDIAN_ASSIGNMENT_MAX_SAMPLES_PER_WORKER * bmData->workerStats.size())
															#line 3608 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 989 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				UID workerIdToPop = bmData->recentBWAssignments.front();
															#line 990 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->recentBWAssignments.pop_front();
															#line 992 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				auto it = bmData->workerStats.find(workerIdToPop);
															#line 993 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (it != bmData->workerStats.end())
															#line 3618 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					it->second.recentGranulesAssigned--;
															#line 3622 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
			}
		}
															#line 1001 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->workerAssignments.insert(assignment.keyRange, workerID.get());
															#line 1003 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (bmData->workerStats.contains(workerID.get()))
															#line 3630 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1004 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->workerStats[workerID.get()].numGranulesAssigned += 1;
															#line 3634 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1007 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!assignment.previousFailure.present())
															#line 3638 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1009 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			--bmData->stats.blockedAssignments;
															#line 3642 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(UID && _workerId,int loopDepth) 
	{
															#line 976 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 3652 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 977 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Chose BW {0} for seqno {1} in BM {2}\n", _workerId.toString(), seqNo, bmData->epoch);
															#line 3656 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 979 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		workerID = _workerId;
															#line 981 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (SERVER_KNOBS->BLOB_MANAGER_ENABLE_MEDIAN_ASSIGNMENT_LIMITING)
															#line 3662 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 984 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->recentBWAssignments.push_back(workerID.get());
															#line 985 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->workerStats[workerID.get()].recentGranulesAssigned++;
															#line 987 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (bmData->recentBWAssignments.size() >= SERVER_KNOBS->BLOB_MANAGER_MEDIAN_ASSIGNMENT_MAX_SAMPLES_PER_WORKER * bmData->workerStats.size())
															#line 3670 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 989 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				UID workerIdToPop = bmData->recentBWAssignments.front();
															#line 990 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->recentBWAssignments.pop_front();
															#line 992 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				auto it = bmData->workerStats.find(workerIdToPop);
															#line 993 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (it != bmData->workerStats.end())
															#line 3680 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					it->second.recentGranulesAssigned--;
															#line 3684 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
			}
		}
															#line 1001 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->workerAssignments.insert(assignment.keyRange, workerID.get());
															#line 1003 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (bmData->workerStats.contains(workerID.get()))
															#line 3692 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1004 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->workerStats[workerID.get()].numGranulesAssigned += 1;
															#line 3696 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1007 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!assignment.previousFailure.present())
															#line 3700 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1009 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			--bmData->stats.blockedAssignments;
															#line 3704 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(UID const& _workerId,int loopDepth) 
	{
		loopDepth = a_body1cont6(_workerId, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(UID && _workerId,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_workerId), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state > 0) static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 0;
		static_cast<DoRangeAssignmentActor*>(this)->ActorCallback< DoRangeAssignmentActor, 2, UID >::remove();

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 2, UID >*,UID const& value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 2, UID >*,UID && value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DoRangeAssignmentActor, 2, UID >*,Error err) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont15(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont16(int loopDepth) 
	{
															#line 1215 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<DoRangeAssignmentActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoRangeAssignmentActorState(); static_cast<DoRangeAssignmentActor*>(this)->destroy(); return 0; }
															#line 3802 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<DoRangeAssignmentActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DoRangeAssignmentActorState();
		static_cast<DoRangeAssignmentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1083 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (assignment.previousFailure.present())
															#line 3815 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1085 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				--bmData->stats.blockedAssignments;
															#line 3819 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1087 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			e2 = e;
															#line 1088 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 3825 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1089 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3829 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1091 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (e.code() == error_code_blob_manager_replaced)
															#line 3833 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1092 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (bmData->iAmReplaced.canBeSet())
															#line 3837 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 1093 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					bmData->iAmReplaced.send(Void());
															#line 3841 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
															#line 1095 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (!static_cast<DoRangeAssignmentActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoRangeAssignmentActorState(); static_cast<DoRangeAssignmentActor*>(this)->destroy(); return 0; }
															#line 3845 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				new (&static_cast<DoRangeAssignmentActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~DoRangeAssignmentActorState();
				static_cast<DoRangeAssignmentActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1097 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (e.code() == error_code_granule_assignment_conflict)
															#line 3853 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1102 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (BM_DEBUG)
															#line 3857 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 1103 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					fmt::print("BM {0} got conflict assigning [{1} - {2}) to worker {3}, ignoring\n", bmData->epoch, assignment.keyRange.begin.printable(), assignment.keyRange.end.printable(), workerID.get().toString());
															#line 3861 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
															#line 1109 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (bmData->doLockCheck.canBeSet())
															#line 3865 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 1110 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					bmData->doLockCheck.send(Void());
															#line 3869 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
															#line 1112 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (!static_cast<DoRangeAssignmentActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoRangeAssignmentActorState(); static_cast<DoRangeAssignmentActor*>(this)->destroy(); return 0; }
															#line 3873 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				new (&static_cast<DoRangeAssignmentActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~DoRangeAssignmentActorState();
				static_cast<DoRangeAssignmentActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1115 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (e.code() != error_code_broken_promise && e.code() != error_code_no_more_servers && e.code() != error_code_blob_worker_full)
															#line 3881 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1117 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				TraceEvent(SevWarn, "BlobManagerUnexpectedErrorDoRangeAssignment", bmData->id) .error(e) .detail("Epoch", bmData->epoch);
															#line 1120 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				ASSERT_WE_THINK(false);
															#line 1121 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (bmData->iAmReplaced.canBeSet())
															#line 3889 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 1122 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					bmData->iAmReplaced.sendError(e);
															#line 3893 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
															#line 1124 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3897 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1127 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (e.code() == error_code_no_more_servers && assignment.isAssign && assignment.assign.get().type == AssignRequestType::Continue)
															#line 3901 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1131 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				TraceEvent("BMDroppingContinueAssignment", bmData->id) .detail("Epoch", bmData->epoch) .detail("Seqno", seqNo);
															#line 1134 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				CODE_PROBE(true, "BM no more servers on continue", probe::decoration::rare);
															#line 1135 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (!static_cast<DoRangeAssignmentActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoRangeAssignmentActorState(); static_cast<DoRangeAssignmentActor*>(this)->destroy(); return 0; }
															#line 3909 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				new (&static_cast<DoRangeAssignmentActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~DoRangeAssignmentActorState();
				static_cast<DoRangeAssignmentActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1139 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (assignment.isAssign)
															#line 3917 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1140 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				++bmData->stats.blockedAssignments;
															#line 3921 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1143 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (e.code() == error_code_blob_worker_full)
															#line 3925 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1144 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				CODE_PROBE(true, "blob worker too full");
															#line 1145 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				ASSERT(assignment.isAssign);
															#line 3931 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				try {
															#line 1147 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					if (assignment.previousFailure.present() && assignment.previousFailure.get().second.code() == error_code_blob_worker_full)
															#line 3935 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
					{
															#line 1151 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
						CODE_PROBE(true, "multiple blob workers too full");
															#line 1152 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
						StrictFuture<Void> __when_expr_6 = delayJittered(10.0);
															#line 1152 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
						if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1Catch1(actor_cancelled(), loopDepth);
															#line 3943 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
						if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1Catch1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1Catch1when1(__when_expr_6.get(), loopDepth); };
						static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 7;
															#line 1152 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
						__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< DoRangeAssignmentActor, 6, Void >*>(static_cast<DoRangeAssignmentActor*>(this)));
															#line 3948 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
															#line 1154 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
						StrictFuture<Void> __when_expr_7 = delayJittered(1.0);
															#line 1154 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
						if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1Catch1(actor_cancelled(), loopDepth);
															#line 3957 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
						if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont1Catch1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont1Catch1when2(__when_expr_7.get(), loopDepth); };
						static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 8;
															#line 1154 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
						__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< DoRangeAssignmentActor, 7, Void >*>(static_cast<DoRangeAssignmentActor*>(this)));
															#line 3962 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
						loopDepth = 0;
					}
				}
				catch (Error& error) {
					loopDepth = a_body1cont1Catch1Catch1(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1cont1Catch1Catch1(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont1Catch1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont18(int loopDepth) 
	{
		loopDepth = a_body1cont18cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont19(int loopDepth) 
	{
															#line 1056 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_4 = assignFuture;
															#line 1056 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 3997 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont19when1(__when_expr_4.get(), loopDepth); };
		static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 5;
															#line 1056 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< DoRangeAssignmentActor, 4, Void >*>(static_cast<DoRangeAssignmentActor*>(this)));
															#line 4002 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont21(Void const& _,int loopDepth) 
	{
															#line 1051 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (bmData->maybeInjectTargetedRestart())
															#line 4011 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1052 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			return a_body1cont1Catch1(blob_manager_replaced(), loopDepth);
															#line 4015 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		loopDepth = a_body1cont19(loopDepth);

		return loopDepth;
	}
	int a_body1cont21(Void && _,int loopDepth) 
	{
															#line 1051 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (bmData->maybeInjectTargetedRestart())
															#line 4025 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1052 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			return a_body1cont1Catch1(blob_manager_replaced(), loopDepth);
															#line 4029 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		loopDepth = a_body1cont19(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont21(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont21(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state > 0) static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 0;
		static_cast<DoRangeAssignmentActor*>(this)->ActorCallback< DoRangeAssignmentActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DoRangeAssignmentActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont19cont1(Void const& _,int loopDepth) 
	{
															#line 1058 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (assignment.previousFailure.present())
															#line 4114 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1060 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			--bmData->stats.blockedAssignments;
															#line 4118 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1063 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<DoRangeAssignmentActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoRangeAssignmentActorState(); static_cast<DoRangeAssignmentActor*>(this)->destroy(); return 0; }
															#line 4122 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<DoRangeAssignmentActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DoRangeAssignmentActorState();
		static_cast<DoRangeAssignmentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont19cont1(Void && _,int loopDepth) 
	{
															#line 1058 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (assignment.previousFailure.present())
															#line 4134 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1060 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			--bmData->stats.blockedAssignments;
															#line 4138 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1063 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<DoRangeAssignmentActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoRangeAssignmentActorState(); static_cast<DoRangeAssignmentActor*>(this)->destroy(); return 0; }
															#line 4142 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<DoRangeAssignmentActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DoRangeAssignmentActorState();
		static_cast<DoRangeAssignmentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont19when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont19cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont19when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont19cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state > 0) static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 0;
		static_cast<DoRangeAssignmentActor*>(this)->ActorCallback< DoRangeAssignmentActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont19when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont19when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< DoRangeAssignmentActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont22(int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont23(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont22(loopDepth);

		return loopDepth;
	}
	int a_body1cont23(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont22(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont23(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont23(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state > 0) static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 0;
		static_cast<DoRangeAssignmentActor*>(this)->ActorCallback< DoRangeAssignmentActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< DoRangeAssignmentActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont18cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont16(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1Catch1cont1(int loopDepth) 
	{
															#line 1162 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		CODE_PROBE(true, "BM retrying range assign");
															#line 1166 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (assignment.isAssign)
															#line 4337 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1167 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 4341 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1168 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM got error {0} assigning range [{1} - {2}) to worker {3}, requeueing\n", e2.name(), assignment.keyRange.begin.printable(), assignment.keyRange.end.printable(), workerID.get().toString());
															#line 4345 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1176 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			RangeAssignment revokeOld;
															#line 1177 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			revokeOld.isAssign = false;
															#line 1178 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			revokeOld.worker = workerID;
															#line 1179 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			revokeOld.keyRange = assignment.keyRange;
															#line 1180 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			revokeOld.revoke = RangeRevokeData(false);
															#line 1182 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			handleRangeAssign(bmData, revokeOld);
															#line 1187 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ASSERT(assignment.assign.present());
															#line 1188 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			assignment.assign.get().type = AssignRequestType::Normal;
															#line 1189 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			assignment.worker.reset();
															#line 1190 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			std::pair<UID, Error> failure = { workerID.get(), e2 };
															#line 1191 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			assignment.previousFailure = failure;
															#line 1192 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			handleRangeAssign(bmData, assignment);
															#line 4371 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		else
		{
															#line 1194 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 4377 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1195 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM got error revoking range [{0} - {1}) from worker", assignment.keyRange.begin.printable(), assignment.keyRange.end.printable());
															#line 4381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1200 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (assignment.revoke.get().dispose)
															#line 4385 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1201 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (BM_DEBUG)
															#line 4389 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 1202 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					printf(", retrying for dispose\n");
															#line 4393 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
															#line 1205 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				assignment.worker.reset();
															#line 1206 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				handleRangeAssign(bmData, assignment);
															#line 4399 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
			else
			{
															#line 1209 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (BM_DEBUG)
															#line 4405 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 1210 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					printf(", ignoring\n");
															#line 4409 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
			}
		}
		loopDepth = a_body1cont16(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1cont13(int loopDepth) 
	{
		loopDepth = a_body1cont1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1157 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			--bmData->stats.blockedAssignments;
															#line 1158 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 4430 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1Catch1cont14(int loopDepth) 
	{
		loopDepth = a_body1cont1Catch1cont14cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1cont15(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1cont15(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch1cont15(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch1cont15(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state > 0) static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 0;
		static_cast<DoRangeAssignmentActor*>(this)->ActorCallback< DoRangeAssignmentActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< DoRangeAssignmentActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1Catch1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont1Catch1cont16(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1cont16(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch1cont16(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch1cont16(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state > 0) static_cast<DoRangeAssignmentActor*>(this)->actor_wait_state = 0;
		static_cast<DoRangeAssignmentActor*>(this)->ActorCallback< DoRangeAssignmentActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1Catch1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< DoRangeAssignmentActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1Catch1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< DoRangeAssignmentActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1Catch1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont1Catch1cont14cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1Catch1cont13(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	RangeAssignment assignment;
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Optional<UID> workerID;
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int64_t epoch;
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int64_t seqNo;
															#line 955 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	bool blockedWaitingForWorker;
															#line 1046 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Future<Void> assignFuture;
															#line 1087 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Error e2;
															#line 4649 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via doRangeAssignment()
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class DoRangeAssignmentActor final : public Actor<Void>, public ActorCallback< DoRangeAssignmentActor, 0, Void >, public ActorCallback< DoRangeAssignmentActor, 1, Void >, public ActorCallback< DoRangeAssignmentActor, 2, UID >, public ActorCallback< DoRangeAssignmentActor, 3, Void >, public ActorCallback< DoRangeAssignmentActor, 4, Void >, public ActorCallback< DoRangeAssignmentActor, 5, Void >, public ActorCallback< DoRangeAssignmentActor, 6, Void >, public ActorCallback< DoRangeAssignmentActor, 7, Void >, public FastAllocated<DoRangeAssignmentActor>, public DoRangeAssignmentActorState<DoRangeAssignmentActor> {
															#line 4654 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<DoRangeAssignmentActor>::operator new;
	using FastAllocated<DoRangeAssignmentActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6402113829836634112UL, 17534501850123167232UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoRangeAssignmentActor, 0, Void >;
friend struct ActorCallback< DoRangeAssignmentActor, 1, Void >;
friend struct ActorCallback< DoRangeAssignmentActor, 2, UID >;
friend struct ActorCallback< DoRangeAssignmentActor, 3, Void >;
friend struct ActorCallback< DoRangeAssignmentActor, 4, Void >;
friend struct ActorCallback< DoRangeAssignmentActor, 5, Void >;
friend struct ActorCallback< DoRangeAssignmentActor, 6, Void >;
friend struct ActorCallback< DoRangeAssignmentActor, 7, Void >;
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	DoRangeAssignmentActor(Reference<BlobManagerData> const& bmData,RangeAssignment const& assignment,Optional<UID> const& workerID,int64_t const& epoch,int64_t const& seqNo) 
															#line 4678 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<Void>(),
		   DoRangeAssignmentActorState<DoRangeAssignmentActor>(bmData, assignment, workerID, epoch, seqNo),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doRangeAssignment", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12805347850781785088UL, 10593969368853654272UL);
		ActorExecutionContextHelper __helper(static_cast<DoRangeAssignmentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doRangeAssignment");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doRangeAssignment", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoRangeAssignmentActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DoRangeAssignmentActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DoRangeAssignmentActor, 2, UID >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DoRangeAssignmentActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< DoRangeAssignmentActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< DoRangeAssignmentActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< DoRangeAssignmentActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< DoRangeAssignmentActor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<Void> doRangeAssignment( Reference<BlobManagerData> const& bmData, RangeAssignment const& assignment, Optional<UID> const& workerID, int64_t const& epoch, int64_t const& seqNo ) {
															#line 950 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<Void>(new DoRangeAssignmentActor(bmData, assignment, workerID, epoch, seqNo));
															#line 4718 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 1217 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

static bool handleRangeIsAssign(Reference<BlobManagerData> bmData, RangeAssignment assignment, int64_t seqNo) {
	// Ensure range isn't currently assigned anywhere, and there is only 1 intersecting range
	auto currentAssignments = bmData->workerAssignments.intersectingRanges(assignment.keyRange);
	int count = 0;
	UID workerId;
	for (auto i = currentAssignments.begin(); i != currentAssignments.end(); ++i) {
		if (assignment.assign.get().type == AssignRequestType::Continue) {
			ASSERT(assignment.worker.present());
			if (i.range() != assignment.keyRange || i.cvalue() != assignment.worker.get()) {
				CODE_PROBE(true, "BM assignment out of date", probe::decoration::rare);
				if (BM_DEBUG) {
					fmt::print("Out of date re-assign for ({0}, {1}). Assignment must have changed while "
					           "checking split.\n  Reassign: [{2} - {3}): {4}\n  Existing: [{5} - {6}): {7}\n",
					           bmData->epoch,
					           seqNo,
					           assignment.keyRange.begin.printable(),
					           assignment.keyRange.end.printable(),
					           assignment.worker.get().toString().substr(0, 5),
					           i.begin().printable(),
					           i.end().printable(),
					           i.cvalue().toString().substr(0, 5));
				}
				return false;
			}
		}
		count++;
	}
	ASSERT(count == 1);

	bool forcePurging = bmData->isForcePurging(assignment.keyRange);

	if (forcePurging && assignment.previousFailure.present()) {
		--bmData->stats.blockedAssignments;
	}
	if (assignment.worker.present() && assignment.worker.get().isValid()) {
		if (BM_DEBUG) {
			fmt::print("BW {0} already chosen for seqno {1} in BM {2}\n",
			           assignment.worker.get().toString(),
			           seqNo,
			           bmData->id.toString());
		}
		workerId = assignment.worker.get();

		bmData->workerAssignments.insert(assignment.keyRange, workerId);

		// If we know about the worker and this is not a continue, then this is a new range for the worker
		if (assignment.assign.get().type == AssignRequestType::Continue) {
			// if it is a continue, don't cancel an in-flight re-assignment. Send to actor collection instead of
			// assignsInProgress
			bmData->addActor.send(doRangeAssignment(bmData, assignment, workerId, bmData->epoch, seqNo));
		} else {
			if (!forcePurging) {
				// FIXME: not handling error propagation correctly here
				bmData->assignsInProgress.insert(assignment.keyRange,
				                                 doRangeAssignment(bmData, assignment, workerId, bmData->epoch, seqNo));
			}
			if (bmData->workerStats.contains(workerId)) {
				bmData->workerStats[workerId].numGranulesAssigned += 1;
			}
		}
	} else {
		// Ensure the key boundaries are updated before we pick a worker
		bmData->workerAssignments.insert(assignment.keyRange, UID());
		ASSERT(assignment.assign.get().type != AssignRequestType::Continue);
		if (!forcePurging) {
			// FIXME: not handling error propagation correctly here
			bmData->assignsInProgress.insert(
			    assignment.keyRange, doRangeAssignment(bmData, assignment, Optional<UID>(), bmData->epoch, seqNo));
		}
	}
	return true;
}

static bool handleRangeIsRevoke(Reference<BlobManagerData> bmData, RangeAssignment assignment, int64_t seqNo) {
	if (assignment.worker.present()) {
		// revoke this specific range from this specific worker. Either part of recovery or failing a worker
		if (bmData->workerStats.contains(assignment.worker.get())) {
			bmData->workerStats[assignment.worker.get()].numGranulesAssigned -= 1;
		}
		// if this revoke matches the worker assignment state, mark the range as unassigned
		auto existingRange = bmData->workerAssignments.rangeContaining(assignment.keyRange.begin);
		if (existingRange.range() == assignment.keyRange && existingRange.cvalue() == assignment.worker.get()) {
			bmData->workerAssignments.insert(assignment.keyRange, UID());
		}
		bmData->addActor.send(doRangeAssignment(bmData, assignment, assignment.worker.get(), bmData->epoch, seqNo));
	} else {
		auto currentAssignments = bmData->workerAssignments.intersectingRanges(assignment.keyRange);
		for (auto& it : currentAssignments) {
			// ensure range doesn't truncate existing ranges
			if (it.begin() < assignment.keyRange.begin || it.end() > assignment.keyRange.end) {
				// the only case where this is ok is on startup when a BM is revoking old granules after reading
				// knownBlobRanges and seeing that some are no longer present.
				auto knownRanges = bmData->knownBlobRanges.intersectingRanges(it.range());
				bool inKnownBlobRanges = false;
				for (auto& r : knownRanges) {
					if (r.value()) {
						inKnownBlobRanges = true;
						break;
					}
				}
				bool forcePurging = bmData->isForcePurging(it.range());
				if (it.cvalue() != UID() || (inKnownBlobRanges && !forcePurging)) {
					fmt::print("Assignment [{0} - {1}): {2} truncates range [{3} - {4}) ({5}, {6})\n",
					           assignment.keyRange.begin.printable(),
					           assignment.keyRange.end.printable(),
					           it.cvalue().toString().substr(0, 5),
					           it.begin().printable(),
					           it.end().printable(),
					           inKnownBlobRanges,
					           forcePurging);
					// assert on condition again to make assertion failure better than "false"
					ASSERT(it.cvalue() == UID() && (!inKnownBlobRanges || forcePurging));
				}
			}

			// It is fine for multiple disjoint sub-ranges to have the same sequence number since they were part
			// of the same logical change

			if (bmData->workerStats.contains(it.value())) {
				bmData->workerStats[it.value()].numGranulesAssigned -= 1;
			}

			// revoke the range for the worker that owns it, not the worker specified in the revoke
			bmData->addActor.send(doRangeAssignment(bmData, assignment, it.value(), bmData->epoch, seqNo));
		}
		bmData->workerAssignments.insert(assignment.keyRange, UID());
	}

	bmData->assignsInProgress.cancel(assignment.keyRange);
	return true;
}

static bool handleRangeAssign(Reference<BlobManagerData> bmData, RangeAssignment assignment) {
	int64_t seqNo = bmData->seqNo;
	bmData->seqNo++;

	// modify the in-memory assignment data structures, and send request off to worker
	if (assignment.isAssign) {
		return handleRangeIsAssign(bmData, assignment, seqNo);
	} else {
		return handleRangeIsRevoke(bmData, assignment, seqNo);
	}
}

															#line 4867 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via checkManagerLock()
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class CheckManagerLockActor>
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class CheckManagerLockActorState {
															#line 4874 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	CheckManagerLockActorState(Transaction* const& tr,Reference<BlobManagerData> const& bmData) 
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : tr(tr),
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   bmData(bmData)
															#line 4883 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("checkManagerLock", reinterpret_cast<unsigned long>(this));

	}
	~CheckManagerLockActorState() 
	{
		fdb_probe_actor_destroy("checkManagerLock", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1363 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr->get(blobManagerEpochKey);
															#line 1363 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<CheckManagerLockActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4900 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckManagerLockActor*>(this)->actor_wait_state = 1;
															#line 1363 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckManagerLockActor, 0, Optional<Value> >*>(static_cast<CheckManagerLockActor*>(this)));
															#line 4905 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckManagerLockActorState();
		static_cast<CheckManagerLockActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> const& currentLockValue,int loopDepth) 
	{
															#line 1364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(currentLockValue.present());
															#line 1365 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int64_t currentEpoch = decodeBlobManagerEpochValue(currentLockValue.get());
															#line 1366 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (currentEpoch != bmData->epoch)
															#line 4932 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1367 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 4936 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1368 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print( "BM {0} found new epoch {1} > {2} in lock check\n", bmData->id.toString(), currentEpoch, bmData->epoch);
															#line 4940 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1371 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (bmData->iAmReplaced.canBeSet())
															#line 4944 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1372 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->iAmReplaced.send(Void());
															#line 4948 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1375 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			return a_body1Catch1(blob_manager_replaced(), loopDepth);
															#line 4952 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1378 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<CheckManagerLockActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckManagerLockActorState(); static_cast<CheckManagerLockActor*>(this)->destroy(); return 0; }
															#line 4956 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<CheckManagerLockActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckManagerLockActorState();
		static_cast<CheckManagerLockActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> && currentLockValue,int loopDepth) 
	{
															#line 1364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(currentLockValue.present());
															#line 1365 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int64_t currentEpoch = decodeBlobManagerEpochValue(currentLockValue.get());
															#line 1366 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (currentEpoch != bmData->epoch)
															#line 4972 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1367 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 4976 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1368 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print( "BM {0} found new epoch {1} > {2} in lock check\n", bmData->id.toString(), currentEpoch, bmData->epoch);
															#line 4980 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1371 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (bmData->iAmReplaced.canBeSet())
															#line 4984 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1372 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->iAmReplaced.send(Void());
															#line 4988 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1375 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			return a_body1Catch1(blob_manager_replaced(), loopDepth);
															#line 4992 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1378 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<CheckManagerLockActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckManagerLockActorState(); static_cast<CheckManagerLockActor*>(this)->destroy(); return 0; }
															#line 4996 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<CheckManagerLockActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckManagerLockActorState();
		static_cast<CheckManagerLockActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Optional<Value> const& currentLockValue,int loopDepth) 
	{
		loopDepth = a_body1cont1(currentLockValue, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> && currentLockValue,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(currentLockValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckManagerLockActor*>(this)->actor_wait_state > 0) static_cast<CheckManagerLockActor*>(this)->actor_wait_state = 0;
		static_cast<CheckManagerLockActor*>(this)->ActorCallback< CheckManagerLockActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< CheckManagerLockActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckManagerLockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckManagerLockActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckManagerLockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckManagerLockActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckManagerLockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Transaction* tr;
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 5083 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via checkManagerLock()
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class CheckManagerLockActor final : public Actor<Void>, public ActorCallback< CheckManagerLockActor, 0, Optional<Value> >, public FastAllocated<CheckManagerLockActor>, public CheckManagerLockActorState<CheckManagerLockActor> {
															#line 5088 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<CheckManagerLockActor>::operator new;
	using FastAllocated<CheckManagerLockActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10215626936449470976UL, 11142264939282213632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckManagerLockActor, 0, Optional<Value> >;
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	CheckManagerLockActor(Transaction* const& tr,Reference<BlobManagerData> const& bmData) 
															#line 5105 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<Void>(),
		   CheckManagerLockActorState<CheckManagerLockActor>(tr, bmData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkManagerLock", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(409268149449470720UL, 6591103125374685952UL);
		ActorExecutionContextHelper __helper(static_cast<CheckManagerLockActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkManagerLock");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkManagerLock", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckManagerLockActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<Void> checkManagerLock( Transaction* const& tr, Reference<BlobManagerData> const& bmData ) {
															#line 1362 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<Void>(new CheckManagerLockActor(tr, bmData));
															#line 5138 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 1380 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

															#line 5143 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via checkManagerLock()
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class CheckManagerLockActor1>
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class CheckManagerLockActor1State {
															#line 5150 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	CheckManagerLockActor1State(Reference<ReadYourWritesTransaction> const& tr,Reference<BlobManagerData> const& bmData) 
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : tr(tr),
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   bmData(bmData)
															#line 5159 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("checkManagerLock", reinterpret_cast<unsigned long>(this));

	}
	~CheckManagerLockActor1State() 
	{
		fdb_probe_actor_destroy("checkManagerLock", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1382 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkManagerLock(&(tr->getTransaction()), bmData);
															#line 1382 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<CheckManagerLockActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5176 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckManagerLockActor1*>(this)->actor_wait_state = 1;
															#line 1382 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckManagerLockActor1, 0, Void >*>(static_cast<CheckManagerLockActor1*>(this)));
															#line 5181 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckManagerLockActor1State();
		static_cast<CheckManagerLockActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1383 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<CheckManagerLockActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckManagerLockActor1State(); static_cast<CheckManagerLockActor1*>(this)->destroy(); return 0; }
															#line 5204 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<CheckManagerLockActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckManagerLockActor1State();
		static_cast<CheckManagerLockActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1383 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<CheckManagerLockActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckManagerLockActor1State(); static_cast<CheckManagerLockActor1*>(this)->destroy(); return 0; }
															#line 5216 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<CheckManagerLockActor1*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckManagerLockActor1State();
		static_cast<CheckManagerLockActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckManagerLockActor1*>(this)->actor_wait_state > 0) static_cast<CheckManagerLockActor1*>(this)->actor_wait_state = 0;
		static_cast<CheckManagerLockActor1*>(this)->ActorCallback< CheckManagerLockActor1, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckManagerLockActor1, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckManagerLockActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckManagerLockActor1, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckManagerLockActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckManagerLockActor1, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckManagerLockActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkManagerLock", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 5303 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via checkManagerLock()
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class CheckManagerLockActor1 final : public Actor<Void>, public ActorCallback< CheckManagerLockActor1, 0, Void >, public FastAllocated<CheckManagerLockActor1>, public CheckManagerLockActor1State<CheckManagerLockActor1> {
															#line 5308 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<CheckManagerLockActor1>::operator new;
	using FastAllocated<CheckManagerLockActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10215626936449470976UL, 11142264939282213632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckManagerLockActor1, 0, Void >;
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	CheckManagerLockActor1(Reference<ReadYourWritesTransaction> const& tr,Reference<BlobManagerData> const& bmData) 
															#line 5325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<Void>(),
		   CheckManagerLockActor1State<CheckManagerLockActor1>(tr, bmData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkManagerLock", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5288769477241580800UL, 6483429838404427264UL);
		ActorExecutionContextHelper __helper(static_cast<CheckManagerLockActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkManagerLock");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkManagerLock", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckManagerLockActor1, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<Void> checkManagerLock( Reference<ReadYourWritesTransaction> const& tr, Reference<BlobManagerData> const& bmData ) {
															#line 1381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<Void>(new CheckManagerLockActor1(tr, bmData));
															#line 5358 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 1385 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

															#line 5363 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via writeInitialGranuleMapping()
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class WriteInitialGranuleMappingActor>
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class WriteInitialGranuleMappingActorState {
															#line 5370 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	WriteInitialGranuleMappingActorState(Reference<BlobManagerData> const& bmData,BlobGranuleSplitPoints const& splitPoints) 
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   splitPoints(splitPoints),
															#line 1387 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   tr(makeReference<ReadYourWritesTransaction>(bmData->db)),
															#line 1389 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   i(0),
															#line 1390 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   transactionChunkSize(BUGGIFY ? deterministicRandom()->randomInt(2, 5) : 1000)
															#line 5385 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this));

	}
	~WriteInitialGranuleMappingActorState() 
	{
		fdb_probe_actor_destroy("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1391 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			;
															#line 5400 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WriteInitialGranuleMappingActorState();
		static_cast<WriteInitialGranuleMappingActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1434 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BUGGIFY && bmData->maybeInjectTargetedRestart())
															#line 5423 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1435 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(0);
															#line 1435 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5429 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1when1(__when_expr_4.get(), loopDepth); };
			static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 5;
															#line 1435 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< WriteInitialGranuleMappingActor, 4, Void >*>(static_cast<WriteInitialGranuleMappingActor*>(this)));
															#line 5434 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1391 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!(i < splitPoints.keys.size() - 1))
															#line 5455 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1392 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		CODE_PROBE(i > 0, "multiple transactions for large granule split");
															#line 1393 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->reset();
															#line 1394 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		j = 0;
															#line 1395 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		;
															#line 5467 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1432 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i += j;
															#line 5489 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
		try {
															#line 1397 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1398 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1399 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1400 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkManagerLock(tr, bmData);
															#line 1400 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5514 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 1;
															#line 1400 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WriteInitialGranuleMappingActor, 0, Void >*>(static_cast<WriteInitialGranuleMappingActor*>(this)));
															#line 5519 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1428 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr->onError(e);
															#line 1428 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 5556 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 4;
															#line 1428 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WriteInitialGranuleMappingActor, 3, Void >*>(static_cast<WriteInitialGranuleMappingActor*>(this)));
															#line 5561 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1404 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		endIdx = std::min(i + transactionChunkSize, (int)(splitPoints.keys.size() - 1));
															#line 1405 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(tr, blobGranuleMappingKeys.begin, KeyRangeRef(splitPoints.keys[i], splitPoints.keys[endIdx]), blobGranuleMappingValueFor(UID()));
															#line 1405 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5580 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 2;
															#line 1405 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteInitialGranuleMappingActor, 1, Void >*>(static_cast<WriteInitialGranuleMappingActor*>(this)));
															#line 5585 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1404 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		endIdx = std::min(i + transactionChunkSize, (int)(splitPoints.keys.size() - 1));
															#line 1405 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_1 = krmSetRange(tr, blobGranuleMappingKeys.begin, KeyRangeRef(splitPoints.keys[i], splitPoints.keys[endIdx]), blobGranuleMappingValueFor(UID()));
															#line 1405 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5598 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 2;
															#line 1405 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WriteInitialGranuleMappingActor, 1, Void >*>(static_cast<WriteInitialGranuleMappingActor*>(this)));
															#line 5603 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state > 0) static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 0;
		static_cast<WriteInitialGranuleMappingActor*>(this)->ActorCallback< WriteInitialGranuleMappingActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteInitialGranuleMappingActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WriteInitialGranuleMappingActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WriteInitialGranuleMappingActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1409 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(j = 0;i + j < endIdx;j++) {
															#line 1410 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (splitPoints.boundaries.contains(splitPoints.keys[i + j]))
															#line 5689 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1411 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				tr->set(blobGranuleMergeBoundaryKeyFor(splitPoints.keys[i + j]), blobGranuleMergeBoundaryValueFor(splitPoints.boundaries[splitPoints.keys[i + j]]));
															#line 5693 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1414 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->set(splitPoints.keys[i + j].withPrefix(blobGranuleMappingKeys.begin), blobGranuleMappingValueFor(UID()));
															#line 5697 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1417 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr->commit();
															#line 1417 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5703 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 3;
															#line 1417 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteInitialGranuleMappingActor, 2, Void >*>(static_cast<WriteInitialGranuleMappingActor*>(this)));
															#line 5708 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1409 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(j = 0;i + j < endIdx;j++) {
															#line 1410 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (splitPoints.boundaries.contains(splitPoints.keys[i + j]))
															#line 5719 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1411 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				tr->set(blobGranuleMergeBoundaryKeyFor(splitPoints.keys[i + j]), blobGranuleMergeBoundaryValueFor(splitPoints.boundaries[splitPoints.keys[i + j]]));
															#line 5723 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1414 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->set(splitPoints.keys[i + j].withPrefix(blobGranuleMappingKeys.begin), blobGranuleMappingValueFor(UID()));
															#line 5727 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1417 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr->commit();
															#line 1417 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5733 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 3;
															#line 1417 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WriteInitialGranuleMappingActor, 2, Void >*>(static_cast<WriteInitialGranuleMappingActor*>(this)));
															#line 5738 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state > 0) static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 0;
		static_cast<WriteInitialGranuleMappingActor*>(this)->ActorCallback< WriteInitialGranuleMappingActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteInitialGranuleMappingActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WriteInitialGranuleMappingActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WriteInitialGranuleMappingActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1420 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(int k = i;k < i + j;k++) {
															#line 1421 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			KeyRef beginKey = splitPoints.keys[k];
															#line 1422 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (splitPoints.boundaries.contains(beginKey))
															#line 5826 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1423 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->mergeBoundaries[beginKey] = splitPoints.boundaries[beginKey];
															#line 5830 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1420 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(int k = i;k < i + j;k++) {
															#line 1421 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			KeyRef beginKey = splitPoints.keys[k];
															#line 1422 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (splitPoints.boundaries.contains(beginKey))
															#line 5845 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1423 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->mergeBoundaries[beginKey] = splitPoints.boundaries[beginKey];
															#line 5849 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state > 0) static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 0;
		static_cast<WriteInitialGranuleMappingActor*>(this)->ActorCallback< WriteInitialGranuleMappingActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteInitialGranuleMappingActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WriteInitialGranuleMappingActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WriteInitialGranuleMappingActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
															#line 1429 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		j = 0;
															#line 5935 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
															#line 1429 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		j = 0;
															#line 5944 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state > 0) static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 0;
		static_cast<WriteInitialGranuleMappingActor*>(this)->ActorCallback< WriteInitialGranuleMappingActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteInitialGranuleMappingActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WriteInitialGranuleMappingActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WriteInitialGranuleMappingActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 1438 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<WriteInitialGranuleMappingActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WriteInitialGranuleMappingActorState(); static_cast<WriteInitialGranuleMappingActor*>(this)->destroy(); return 0; }
															#line 6028 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<WriteInitialGranuleMappingActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WriteInitialGranuleMappingActorState();
		static_cast<WriteInitialGranuleMappingActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1436 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(false);
															#line 6040 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1436 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(false);
															#line 6049 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state > 0) static_cast<WriteInitialGranuleMappingActor*>(this)->actor_wait_state = 0;
		static_cast<WriteInitialGranuleMappingActor*>(this)->ActorCallback< WriteInitialGranuleMappingActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WriteInitialGranuleMappingActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< WriteInitialGranuleMappingActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< WriteInitialGranuleMappingActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	BlobGranuleSplitPoints splitPoints;
															#line 1387 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1389 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int i;
															#line 1390 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int transactionChunkSize;
															#line 1394 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int j;
															#line 1404 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int endIdx;
															#line 6143 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via writeInitialGranuleMapping()
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class WriteInitialGranuleMappingActor final : public Actor<Void>, public ActorCallback< WriteInitialGranuleMappingActor, 0, Void >, public ActorCallback< WriteInitialGranuleMappingActor, 1, Void >, public ActorCallback< WriteInitialGranuleMappingActor, 2, Void >, public ActorCallback< WriteInitialGranuleMappingActor, 3, Void >, public ActorCallback< WriteInitialGranuleMappingActor, 4, Void >, public FastAllocated<WriteInitialGranuleMappingActor>, public WriteInitialGranuleMappingActorState<WriteInitialGranuleMappingActor> {
															#line 6148 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<WriteInitialGranuleMappingActor>::operator new;
	using FastAllocated<WriteInitialGranuleMappingActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16248457632806944000UL, 13190573398486380032UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WriteInitialGranuleMappingActor, 0, Void >;
friend struct ActorCallback< WriteInitialGranuleMappingActor, 1, Void >;
friend struct ActorCallback< WriteInitialGranuleMappingActor, 2, Void >;
friend struct ActorCallback< WriteInitialGranuleMappingActor, 3, Void >;
friend struct ActorCallback< WriteInitialGranuleMappingActor, 4, Void >;
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	WriteInitialGranuleMappingActor(Reference<BlobManagerData> const& bmData,BlobGranuleSplitPoints const& splitPoints) 
															#line 6169 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<Void>(),
		   WriteInitialGranuleMappingActorState<WriteInitialGranuleMappingActor>(bmData, splitPoints),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17969257437310571776UL, 1583862476135348224UL);
		ActorExecutionContextHelper __helper(static_cast<WriteInitialGranuleMappingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("writeInitialGranuleMapping");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("writeInitialGranuleMapping", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WriteInitialGranuleMappingActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WriteInitialGranuleMappingActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WriteInitialGranuleMappingActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WriteInitialGranuleMappingActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< WriteInitialGranuleMappingActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<Void> writeInitialGranuleMapping( Reference<BlobManagerData> const& bmData, BlobGranuleSplitPoints const& splitPoints ) {
															#line 1386 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<Void>(new WriteInitialGranuleMappingActor(bmData, splitPoints));
															#line 6206 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 1440 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

															#line 6211 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via loadTenantMap()
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class LoadTenantMapActor>
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class LoadTenantMapActorState {
															#line 6218 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	LoadTenantMapActorState(Reference<ReadYourWritesTransaction> const& tr,Reference<BlobManagerData> const& bmData) 
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : tr(tr),
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   bmData(bmData),
															#line 1442 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   tenantResults()
															#line 6229 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("loadTenantMap", reinterpret_cast<unsigned long>(this));

	}
	~LoadTenantMapActorState() 
	{
		fdb_probe_actor_destroy("loadTenantMap", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1443 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(tenantResults, TenantMetadata::tenantMap().getRange(tr, {}, {}, CLIENT_KNOBS->MAX_TENANTS_PER_CLUSTER + 1));
															#line 1443 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<LoadTenantMapActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6246 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadTenantMapActor*>(this)->actor_wait_state = 1;
															#line 1443 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadTenantMapActor, 0, Void >*>(static_cast<LoadTenantMapActor*>(this)));
															#line 6251 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadTenantMapActorState();
		static_cast<LoadTenantMapActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1445 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(tenantResults.results.size() <= CLIENT_KNOBS->MAX_TENANTS_PER_CLUSTER && !tenantResults.more);
															#line 1447 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->tenantData.addTenants(tenantResults.results);
															#line 1449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<LoadTenantMapActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadTenantMapActorState(); static_cast<LoadTenantMapActor*>(this)->destroy(); return 0; }
															#line 6278 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<LoadTenantMapActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadTenantMapActorState();
		static_cast<LoadTenantMapActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1445 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(tenantResults.results.size() <= CLIENT_KNOBS->MAX_TENANTS_PER_CLUSTER && !tenantResults.more);
															#line 1447 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->tenantData.addTenants(tenantResults.results);
															#line 1449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<LoadTenantMapActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadTenantMapActorState(); static_cast<LoadTenantMapActor*>(this)->destroy(); return 0; }
															#line 6294 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<LoadTenantMapActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadTenantMapActorState();
		static_cast<LoadTenantMapActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadTenantMapActor*>(this)->actor_wait_state > 0) static_cast<LoadTenantMapActor*>(this)->actor_wait_state = 0;
		static_cast<LoadTenantMapActor*>(this)->ActorCallback< LoadTenantMapActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadTenantMapActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadTenantMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadTenantMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadTenantMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadTenantMapActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadTenantMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadTenantMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadTenantMap", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadTenantMapActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadTenantMap", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadTenantMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadTenantMap", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 1442 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	KeyBackedRangeResult<std::pair<int64_t, TenantMapEntry>> tenantResults;
															#line 6383 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via loadTenantMap()
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class LoadTenantMapActor final : public Actor<Void>, public ActorCallback< LoadTenantMapActor, 0, Void >, public FastAllocated<LoadTenantMapActor>, public LoadTenantMapActorState<LoadTenantMapActor> {
															#line 6388 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<LoadTenantMapActor>::operator new;
	using FastAllocated<LoadTenantMapActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1649884709236979200UL, 7647813648438420480UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadTenantMapActor, 0, Void >;
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	LoadTenantMapActor(Reference<ReadYourWritesTransaction> const& tr,Reference<BlobManagerData> const& bmData) 
															#line 6405 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<Void>(),
		   LoadTenantMapActorState<LoadTenantMapActor>(tr, bmData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadTenantMap", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15441272082253051904UL, 15021541507267853568UL);
		ActorExecutionContextHelper __helper(static_cast<LoadTenantMapActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadTenantMap");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadTenantMap", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadTenantMapActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<Void> loadTenantMap( Reference<ReadYourWritesTransaction> const& tr, Reference<BlobManagerData> const& bmData ) {
															#line 1441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<Void>(new LoadTenantMapActor(tr, bmData));
															#line 6438 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 1451 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

															#line 6443 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via monitorTenants()
															#line 1452 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class MonitorTenantsActor>
															#line 1452 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class MonitorTenantsActorState {
															#line 6450 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 1452 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	MonitorTenantsActorState(Reference<BlobManagerData> const& bmData) 
															#line 1452 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 1452 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData)
															#line 6457 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorTenants", reinterpret_cast<unsigned long>(this));

	}
	~MonitorTenantsActorState() 
	{
		fdb_probe_actor_destroy("monitorTenants", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1453 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			;
															#line 6472 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorTenantsActorState();
		static_cast<MonitorTenantsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1454 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>(bmData->db);
															#line 1455 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		;
															#line 6504 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
		try {
															#line 1457 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1458 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1459 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1460 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_0 = loadTenantMap(tr, bmData);
															#line 1460 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorTenantsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6529 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MonitorTenantsActor*>(this)->actor_wait_state = 1;
															#line 1460 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorTenantsActor, 0, Void >*>(static_cast<MonitorTenantsActor*>(this)));
															#line 6534 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1467 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr->onError(e);
															#line 1467 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorTenantsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 6558 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<MonitorTenantsActor*>(this)->actor_wait_state = 4;
															#line 1467 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MonitorTenantsActor, 3, Void >*>(static_cast<MonitorTenantsActor*>(this)));
															#line 6563 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1462 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		watchChange = TenantMetadata::lastTenantModification().watch(tr);
															#line 1463 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_1 = tr->commit();
															#line 1463 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MonitorTenantsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6582 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MonitorTenantsActor*>(this)->actor_wait_state = 2;
															#line 1463 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MonitorTenantsActor, 1, Void >*>(static_cast<MonitorTenantsActor*>(this)));
															#line 6587 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1462 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		watchChange = TenantMetadata::lastTenantModification().watch(tr);
															#line 1463 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_1 = tr->commit();
															#line 1463 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MonitorTenantsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6600 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<MonitorTenantsActor*>(this)->actor_wait_state = 2;
															#line 1463 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MonitorTenantsActor, 1, Void >*>(static_cast<MonitorTenantsActor*>(this)));
															#line 6605 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorTenantsActor*>(this)->actor_wait_state > 0) static_cast<MonitorTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorTenantsActor*>(this)->ActorCallback< MonitorTenantsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorTenantsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorTenantsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorTenantsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 1464 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_2 = watchChange;
															#line 1464 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MonitorTenantsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6691 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MonitorTenantsActor*>(this)->actor_wait_state = 3;
															#line 1464 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MonitorTenantsActor, 2, Void >*>(static_cast<MonitorTenantsActor*>(this)));
															#line 6696 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 1464 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_2 = watchChange;
															#line 1464 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MonitorTenantsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6707 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<MonitorTenantsActor*>(this)->actor_wait_state = 3;
															#line 1464 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MonitorTenantsActor, 2, Void >*>(static_cast<MonitorTenantsActor*>(this)));
															#line 6712 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<MonitorTenantsActor*>(this)->actor_wait_state > 0) static_cast<MonitorTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorTenantsActor*>(this)->ActorCallback< MonitorTenantsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorTenantsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MonitorTenantsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MonitorTenantsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1465 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->reset();
															#line 6796 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1465 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->reset();
															#line 6805 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<MonitorTenantsActor*>(this)->actor_wait_state > 0) static_cast<MonitorTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorTenantsActor*>(this)->ActorCallback< MonitorTenantsActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorTenantsActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MonitorTenantsActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MonitorTenantsActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<MonitorTenantsActor*>(this)->actor_wait_state > 0) static_cast<MonitorTenantsActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorTenantsActor*>(this)->ActorCallback< MonitorTenantsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorTenantsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< MonitorTenantsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< MonitorTenantsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 1452 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 1454 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1462 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Future<Void> watchChange;
															#line 6991 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via monitorTenants()
															#line 1452 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class MonitorTenantsActor final : public Actor<Void>, public ActorCallback< MonitorTenantsActor, 0, Void >, public ActorCallback< MonitorTenantsActor, 1, Void >, public ActorCallback< MonitorTenantsActor, 2, Void >, public ActorCallback< MonitorTenantsActor, 3, Void >, public FastAllocated<MonitorTenantsActor>, public MonitorTenantsActorState<MonitorTenantsActor> {
															#line 6996 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<MonitorTenantsActor>::operator new;
	using FastAllocated<MonitorTenantsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12350535313029884672UL, 6336563111128909056UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorTenantsActor, 0, Void >;
friend struct ActorCallback< MonitorTenantsActor, 1, Void >;
friend struct ActorCallback< MonitorTenantsActor, 2, Void >;
friend struct ActorCallback< MonitorTenantsActor, 3, Void >;
															#line 1452 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	MonitorTenantsActor(Reference<BlobManagerData> const& bmData) 
															#line 7016 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorTenantsActorState<MonitorTenantsActor>(bmData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorTenants", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17385235540494092544UL, 13179936150937425152UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorTenantsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorTenants");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorTenants", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorTenantsActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< MonitorTenantsActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< MonitorTenantsActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< MonitorTenantsActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1452 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<Void> monitorTenants( Reference<BlobManagerData> const& bmData ) {
															#line 1452 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<Void>(new MonitorTenantsActor(bmData));
															#line 7052 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 1472 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

// FIXME: better way to load tenant mapping?
															#line 7058 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via monitorClientRanges()
															#line 1474 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class MonitorClientRangesActor>
															#line 1474 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class MonitorClientRangesActorState {
															#line 7065 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 1474 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	MonitorClientRangesActorState(Reference<BlobManagerData> const& bmData) 
															#line 1474 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 1474 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 1475 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   lastChangeKeyValue(),
															#line 1476 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   needToCoalesce(bmData->epoch > 1),
															#line 1477 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   firstLoad(true),
															#line 1478 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   lastChangeLogVersion(-1)
															#line 7080 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("monitorClientRanges", reinterpret_cast<unsigned long>(this));

	}
	~MonitorClientRangesActorState() 
	{
		fdb_probe_actor_destroy("monitorClientRanges", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1480 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			;
															#line 7095 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorClientRangesActorState();
		static_cast<MonitorClientRangesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1481 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>(bmData->db);
															#line 1483 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 7127 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1484 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			printf("Blob manager checking for range updates\n");
															#line 7131 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1486 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		rangesToAdd = VectorRef<KeyRangeRef>();
															#line 1487 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		rangesToRemove = VectorRef<KeyRangeRef>();
															#line 1488 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ar = Arena();
															#line 1489 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ckvBegin = Optional<Value>();
															#line 1490 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		;
															#line 7143 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1612 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitFutures = std::vector<Future<BlobGranuleSplitPoints>>();
															#line 1614 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for( KeyRangeRef range : rangesToAdd ) {
															#line 1615 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("ClientBlobRangeAdded", bmData->id).detail("Range", range);
															#line 1620 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			auto r = bmData->workerAssignments.intersectingRanges(range);
															#line 1621 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for( auto& it : r ) {
															#line 1622 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				ASSERT(it.cvalue() == UID());
															#line 7162 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1624 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->workerAssignments.insert(range, UID());
															#line 1627 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitFutures.push_back(splitRange(bmData, range, false, true));
															#line 7168 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1630 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for( KeyRangeRef range : rangesToRemove ) {
															#line 1631 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("ClientBlobRangeRemoved", bmData->id).detail("Range", range);
															#line 1632 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 7176 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1633 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM Got range to revoke [{0} - {1})\n", range.begin.printable(), range.end.printable());
															#line 7180 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1636 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			RangeAssignment ra;
															#line 1637 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ra.isAssign = false;
															#line 1638 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ra.keyRange = range;
															#line 1639 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ra.revoke = RangeRevokeData(true);
															#line 1640 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			handleRangeAssign(bmData, ra);
															#line 7192 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1643 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (firstLoad)
															#line 7196 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1644 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->loadedClientRanges.send(Void());
															#line 1645 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			firstLoad = false;
															#line 7202 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1648 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		RangeForbody1loopBody1cont1Iterator0 = std::begin(splitFutures);
															#line 7206 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
		try {
															#line 1492 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1493 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1494 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1497 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_0 = store(ckvBegin, tr->get(blobRangeChangeKey));
															#line 1497 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7231 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 1;
															#line 1497 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 0, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 7236 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1605 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 7271 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1606 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("Blob manager got error looking for range updates {}\n", e.name());
															#line 7275 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1608 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_6 = tr->onError(e);
															#line 1608 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 7281 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1Catch1when1(__when_expr_6.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 7;
															#line 1608 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 6, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 7286 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1499 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (firstLoad || !SERVER_KNOBS->BG_USE_BLOB_RANGE_CHANGE_LOG)
															#line 7301 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1501 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(tr, blobRangeKeys.begin, KeyRange(normalKeys), CLIENT_KNOBS->TOO_MANY, GetRangeLimits::BYTE_LIMIT_UNLIMITED);
															#line 1501 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7307 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 2;
															#line 1501 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 1, RangeResult >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 7312 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1548 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			KeyRange readRange = KeyRangeRef(blobRangeChangeLogReadKeyFor(lastChangeLogVersion + 1), blobRangeChangeLogKeys.end);
															#line 1550 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<RangeResult> __when_expr_3 = tr->getRange(readRange, CLIENT_KNOBS->TOO_MANY);
															#line 1550 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7323 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when2(__when_expr_3.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 4;
															#line 1550 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 3, RangeResult >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 7328 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1499 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (firstLoad || !SERVER_KNOBS->BG_USE_BLOB_RANGE_CHANGE_LOG)
															#line 7338 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1501 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<RangeResult> __when_expr_1 = krmGetRanges(tr, blobRangeKeys.begin, KeyRange(normalKeys), CLIENT_KNOBS->TOO_MANY, GetRangeLimits::BYTE_LIMIT_UNLIMITED);
															#line 1501 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7344 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 2;
															#line 1501 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 1, RangeResult >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 7349 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1548 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			KeyRange readRange = KeyRangeRef(blobRangeChangeLogReadKeyFor(lastChangeLogVersion + 1), blobRangeChangeLogKeys.end);
															#line 1550 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<RangeResult> __when_expr_3 = tr->getRange(readRange, CLIENT_KNOBS->TOO_MANY);
															#line 1550 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7360 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when2(__when_expr_3.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 4;
															#line 1550 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 3, RangeResult >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 7365 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont3(int loopDepth) 
	{
		return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4(int loopDepth) 
	{
															#line 1506 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT_WE_THINK(!results.more && results.size() < CLIENT_KNOBS->TOO_MANY);
															#line 1507 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (results.more || results.size() >= CLIENT_KNOBS->TOO_MANY)
															#line 7458 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1508 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent(SevError, "BlobManagerTooManyClientRanges", bmData->id) .detail("Epoch", bmData->epoch) .detail("ClientRanges", results.size() - 1);
															#line 1511 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_2 = delay(600);
															#line 1511 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7466 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1loopBody1cont4when1(__when_expr_2.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 3;
															#line 1511 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 2, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 7471 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1loopBody1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(RangeResult const& __results,int loopDepth) 
	{
															#line 1501 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		results = __results;
															#line 7485 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(RangeResult && __results,int loopDepth) 
	{
		results = std::move(__results);
		loopDepth = a_body1loopBody1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1loopBody1cont5(int loopDepth) 
	{
															#line 1519 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->mergeHardBoundaries.clear();
															#line 1520 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for( auto& it : results ) {
															#line 1521 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->mergeHardBoundaries[it.key] = true;
															#line 7568 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1523 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ar.dependsOn(results.arena());
															#line 1525 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		updateClientBlobRanges( bmData->epoch, &bmData->knownBlobRanges, results, ar, &rangesToAdd, &rangesToRemove);
															#line 1528 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (needToCoalesce)
															#line 7576 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1533 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			needToCoalesce = false;
															#line 1535 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for(int i = 0;i < results.size() - 1;i++) {
															#line 1536 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bool active = isBlobRangeActive(results[i].value);
															#line 1537 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->knownBlobRanges.insert(KeyRangeRef(results[i].key, results[i + 1].key), active);
															#line 7586 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
															#line 1542 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (SERVER_KNOBS->BG_USE_BLOB_RANGE_CHANGE_LOG)
															#line 7591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1543 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			lastChangeLogVersion = tr->getReadVersion().get();
															#line 7595 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1512 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (bmData->iAmReplaced.canBeSet())
															#line 7605 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1513 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->iAmReplaced.sendError(internal_error());
															#line 7609 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1515 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		return a_body1loopBody1loopBody1Catch1(internal_error(), loopDepth);
															#line 7613 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1512 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (bmData->iAmReplaced.canBeSet())
															#line 7621 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1513 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->iAmReplaced.sendError(internal_error());
															#line 7625 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1515 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		return a_body1loopBody1loopBody1Catch1(internal_error(), loopDepth);
															#line 7629 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1loopBody1cont14(int loopDepth) 
	{
															#line 1552 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT_WE_THINK(!changeLog.more && changeLog.size() < CLIENT_KNOBS->TOO_MANY);
															#line 1553 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (changeLog.more || changeLog.size() >= CLIENT_KNOBS->TOO_MANY)
															#line 7714 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1554 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent(SevError, "BlobManagerTooManyClientRangeChanges", bmData->id) .detail("Epoch", bmData->epoch) .detail("ClientRangeChanges", changeLog.size());
															#line 1557 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(600);
															#line 1557 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 7722 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1loopBody1cont14when1(__when_expr_4.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 5;
															#line 1557 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 4, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 7727 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1loopBody1cont14cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when2(RangeResult const& __changeLog,int loopDepth) 
	{
															#line 1550 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		changeLog = __changeLog;
															#line 7741 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when2(RangeResult && __changeLog,int loopDepth) 
	{
		changeLog = std::move(__changeLog);
		loopDepth = a_body1loopBody1loopBody1cont14(loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 3, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 3, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1loopBody1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 3, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1loopBody1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 3, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1loopBody1cont14cont1(int loopDepth) 
	{
															#line 1564 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ar.dependsOn(changeLog.arena());
															#line 1566 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i = int();
															#line 1567 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i = 0;
															#line 7824 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont14cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont14cont2(Void const& _,int loopDepth) 
	{
															#line 1558 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (bmData->iAmReplaced.canBeSet())
															#line 7833 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1559 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->iAmReplaced.sendError(internal_error());
															#line 7837 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1561 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		return a_body1loopBody1loopBody1Catch1(internal_error(), loopDepth);
															#line 7841 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont14cont2(Void && _,int loopDepth) 
	{
															#line 1558 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (bmData->iAmReplaced.canBeSet())
															#line 7849 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1559 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->iAmReplaced.sendError(internal_error());
															#line 7853 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1561 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		return a_body1loopBody1loopBody1Catch1(internal_error(), loopDepth);
															#line 7857 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont14when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont14cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont14when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont14cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1loopBody1cont14when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1loopBody1cont14when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1loopBody1cont14cont5(int loopDepth) 
	{
															#line 1601 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		lastChangeLogVersion = tr->getReadVersion().get();
															#line 7940 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont14cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1cont14cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont14cont1loopBody1(int loopDepth) 
	{
															#line 1567 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!(i < changeLog.size()))
															#line 7956 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1loopBody1loopBody1cont14cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1568 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		Standalone<BlobRangeChangeLogRef> v = decodeBlobRangeChangeLogValue(changeLog[i].value);
															#line 1569 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		KeyRangeRef range = v.range & normalKeys;
															#line 1570 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bool active = isBlobRangeActive(v.value);
															#line 1571 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 7968 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1572 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("DBG: BM {0} got range [{1} - {2}): {3} from change log\n", bmData->epoch, range.begin.printable(), range.end.printable(), active ? "T" : "F");
															#line 7972 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1578 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		handleClientBlobRange(bmData->epoch, &bmData->knownBlobRanges, ar, &rangesToAdd, &rangesToRemove, range.begin, range.end, active);
															#line 1587 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (active)
															#line 7978 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1588 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->mergeHardBoundaries[range.begin] = true;
															#line 1589 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->mergeHardBoundaries[range.end] = true;
															#line 7984 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		else
		{
															#line 1592 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			auto it = bmData->mergeHardBoundaries.lower_bound(keyAfter(range.begin));
															#line 1593 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for(;it != bmData->mergeHardBoundaries.end() && it->first < range.end;) {
															#line 1594 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				it = bmData->mergeHardBoundaries.erase(it);
															#line 7994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
															#line 1598 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_5 = yield();
															#line 1598 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8001 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1loopBody1cont14cont1loopBody1when1(__when_expr_5.get(), loopDepth); };
		static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 6;
															#line 1598 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 5, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8006 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont14cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1loopBody1cont14cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont14cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1567 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i++;
															#line 8028 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopBody1cont14cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont14cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1567 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i++;
															#line 8037 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopBody1cont14cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont14cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont14cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont14cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont14cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1loopBody1cont14cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1loopBody1cont14cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1686 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		lastChangeKeyValue = ckvBegin;
															#line 1687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		watchFuture = Future<Void>();
															#line 1689 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		;
															#line 8212 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 1648 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!(RangeForbody1loopBody1cont1Iterator0 != std::end(splitFutures)))
															#line 8228 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1648 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		auto f = *RangeForbody1loopBody1cont1Iterator0;
															#line 8234 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1649 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<BlobGranuleSplitPoints> __when_expr_7 = f;
															#line 1649 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8240 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1when1(__when_expr_7.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 8;
															#line 1649 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 7, BlobGranuleSplitPoints >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8245 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont1(int loopDepth) 
	{
															#line 1648 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		++RangeForbody1loopBody1cont1Iterator0;
															#line 8268 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont3(int loopDepth) 
	{
															#line 1650 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 8277 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1651 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("BM {0} Splitting client range [{1} - {2}) into {3} ranges.\n", bmData->epoch, splitPoints.keys[0].printable(), splitPoints.keys[splitPoints.keys.size() - 1].printable(), splitPoints.keys.size() - 1);
															#line 8281 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1660 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_8 = writeInitialGranuleMapping(bmData, splitPoints);
															#line 1660 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8287 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont1loopBody1cont3when1(__when_expr_8.get(), loopDepth); };
		static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 9;
															#line 1660 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 8, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8292 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(BlobGranuleSplitPoints const& __splitPoints,int loopDepth) 
	{
															#line 1649 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitPoints = __splitPoints;
															#line 8301 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1when1(BlobGranuleSplitPoints && __splitPoints,int loopDepth) 
	{
		splitPoints = std::move(__splitPoints);
		loopDepth = a_body1loopBody1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 7, BlobGranuleSplitPoints >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 7, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 7, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 7, BlobGranuleSplitPoints >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1loopBody1cont1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1662 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 8380 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1663 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("BM {0} Split client range [{1} - {2}) into {3} ranges:\n", bmData->epoch, splitPoints.keys[0].printable(), splitPoints.keys[splitPoints.keys.size() - 1].printable(), splitPoints.keys.size() - 1);
															#line 8384 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1670 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(int i = 0;i < splitPoints.keys.size() - 1;i++) {
															#line 1671 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			KeyRange range = KeyRange(KeyRangeRef(splitPoints.keys[i], splitPoints.keys[i + 1]));
															#line 1673 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 8392 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1674 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("    [{0} - {1})\n", range.begin.printable().c_str(), range.end.printable().c_str());
															#line 8396 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1677 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			RangeAssignment ra;
															#line 1678 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ra.isAssign = true;
															#line 1679 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ra.keyRange = range;
															#line 1680 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ra.assign = RangeAssignmentData();
															#line 1681 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			handleRangeAssign(bmData, ra);
															#line 8408 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1662 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 8418 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1663 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("BM {0} Split client range [{1} - {2}) into {3} ranges:\n", bmData->epoch, splitPoints.keys[0].printable(), splitPoints.keys[splitPoints.keys.size() - 1].printable(), splitPoints.keys.size() - 1);
															#line 8422 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1670 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(int i = 0;i < splitPoints.keys.size() - 1;i++) {
															#line 1671 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			KeyRange range = KeyRange(KeyRangeRef(splitPoints.keys[i], splitPoints.keys[i + 1]));
															#line 1673 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 8430 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1674 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("    [{0} - {1})\n", range.begin.printable().c_str(), range.end.printable().c_str());
															#line 8434 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1677 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			RangeAssignment ra;
															#line 1678 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ra.isAssign = true;
															#line 1679 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ra.keyRange = range;
															#line 1680 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ra.assign = RangeAssignmentData();
															#line 1681 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			handleRangeAssign(bmData, ra);
															#line 8446 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1cont1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1cont1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1(int loopDepth) 
	{
		try {
															#line 1692 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->reset();
															#line 1693 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1694 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1695 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1697 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_9 = tr->get(blobRangeChangeKey);
															#line 1697 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8555 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1loopBody1cont3loopBody1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1loopBody1cont3loopBody1when1(__when_expr_9.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 10;
															#line 1697 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 9, Optional<Value> >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8560 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont3loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont3loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1735 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 8595 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1736 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("Blob manager got error waiting for new range updates {}\n", e.name());
															#line 8599 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1738 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_15 = tr->onError(e);
															#line 1738 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8605 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_15.isReady()) { if (__when_expr_15.isError()) return a_body1Catch1(__when_expr_15.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1Catch1when1(__when_expr_15.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 16;
															#line 1738 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_15.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 15, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8610 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont2(Optional<Value> const& ckvEnd,int loopDepth) 
	{
															#line 1700 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		Version changeLogCleanupVersion = tr->getReadVersion().get() - 2 * SERVER_KNOBS->MAX_READ_TRANSACTION_LIFE_VERSIONS;
															#line 1704 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		changeLogCleanupVersion = std::max<Version>(0, std::min(lastChangeLogVersion, changeLogCleanupVersion));
															#line 1705 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		KeyRange cleanupKeyRange = KeyRangeRef(blobRangeChangeLogKeys.begin, blobRangeChangeLogReadKeyFor(changeLogCleanupVersion));
															#line 1707 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (ckvEnd == lastChangeKeyValue)
															#line 8631 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1708 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			watchFuture = tr->watch(blobRangeChangeKey);
															#line 1710 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->clear(cleanupKeyRange);
															#line 1711 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_10 = checkManagerLock(tr, bmData);
															#line 1711 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8641 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1loopBody1cont3loopBody1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1loopBody1cont3loopBody1cont2when1(__when_expr_10.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 11;
															#line 1711 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 10, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8646 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1720 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (deterministicRandom()->random01() < 0.2)
															#line 8653 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1721 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				tr->clear(cleanupKeyRange);
															#line 1722 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				StrictFuture<Void> __when_expr_12 = checkManagerLock(tr, bmData);
															#line 1722 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8661 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1loopBody1cont3loopBody1Catch1(__when_expr_12.getError(), loopDepth); else return a_body1loopBody1cont3loopBody1cont2when2(__when_expr_12.get(), loopDepth); };
				static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 13;
															#line 1722 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 12, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8666 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont3loopBody1cont8(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont2(Optional<Value> && ckvEnd,int loopDepth) 
	{
															#line 1700 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		Version changeLogCleanupVersion = tr->getReadVersion().get() - 2 * SERVER_KNOBS->MAX_READ_TRANSACTION_LIFE_VERSIONS;
															#line 1704 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		changeLogCleanupVersion = std::max<Version>(0, std::min(lastChangeLogVersion, changeLogCleanupVersion));
															#line 1705 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		KeyRange cleanupKeyRange = KeyRangeRef(blobRangeChangeLogKeys.begin, blobRangeChangeLogReadKeyFor(changeLogCleanupVersion));
															#line 1707 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (ckvEnd == lastChangeKeyValue)
															#line 8687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1708 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			watchFuture = tr->watch(blobRangeChangeKey);
															#line 1710 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->clear(cleanupKeyRange);
															#line 1711 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_10 = checkManagerLock(tr, bmData);
															#line 1711 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8697 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1loopBody1cont3loopBody1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1loopBody1cont3loopBody1cont2when1(__when_expr_10.get(), loopDepth); };
			static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 11;
															#line 1711 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 10, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8702 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1720 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (deterministicRandom()->random01() < 0.2)
															#line 8709 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1721 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				tr->clear(cleanupKeyRange);
															#line 1722 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				StrictFuture<Void> __when_expr_12 = checkManagerLock(tr, bmData);
															#line 1722 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8717 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1loopBody1cont3loopBody1Catch1(__when_expr_12.getError(), loopDepth); else return a_body1loopBody1cont3loopBody1cont2when2(__when_expr_12.get(), loopDepth); };
				static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 13;
															#line 1722 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 12, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8722 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont3loopBody1cont8(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Optional<Value> const& ckvEnd,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont2(ckvEnd, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(Optional<Value> && ckvEnd,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont2(std::move(ckvEnd), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 9, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 9, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 9, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 9, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1cont3loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1loopBody1cont3loopBody1cont3(int loopDepth) 
	{
															#line 1732 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_14 = watchFuture;
															#line 1732 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8814 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_14.isReady()) { if (__when_expr_14.isError()) return a_body1loopBody1cont3loopBody1Catch1(__when_expr_14.getError(), loopDepth); else return a_body1loopBody1cont3loopBody1cont3when1(__when_expr_14.get(), loopDepth); };
		static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 15;
															#line 1732 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_14.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 14, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8819 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1712 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_11 = tr->commit();
															#line 1712 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8830 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1loopBody1cont3loopBody1Catch1(__when_expr_11.getError(), loopDepth); else return a_body1loopBody1cont3loopBody1cont4when1(__when_expr_11.get(), loopDepth); };
		static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 12;
															#line 1712 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 11, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8835 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1712 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_11 = tr->commit();
															#line 1712 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 8846 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1loopBody1cont3loopBody1Catch1(__when_expr_11.getError(), loopDepth); else return a_body1loopBody1cont3loopBody1cont4when1(__when_expr_11.get(), loopDepth); };
		static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 12;
															#line 1712 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 11, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 8851 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1loopBody1cont3loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1loopBody1cont3loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1loopBody1cont3loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1loopBody1cont3loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1713 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 8935 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1714 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Blob manager done processing client ranges @ {0}, awaiting update\n", tr->getCommittedVersion());
															#line 8939 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont3loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1713 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 8949 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1714 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Blob manager done processing client ranges @ {0}, awaiting update\n", tr->getCommittedVersion());
															#line 8953 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont3loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1loopBody1cont3loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1loopBody1cont3loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1loopBody1cont3loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1loopBody1cont3loopBody1cont8(int loopDepth) 
	{
															#line 1729 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		watchFuture = Future<Void>(Void());
															#line 9038 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont9(Void const& _,int loopDepth) 
	{
															#line 1723 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_13 = tr->commit();
															#line 1723 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9049 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_13.isReady()) { if (__when_expr_13.isError()) return a_body1loopBody1cont3loopBody1Catch1(__when_expr_13.getError(), loopDepth); else return a_body1loopBody1cont3loopBody1cont9when1(__when_expr_13.get(), loopDepth); };
		static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 14;
															#line 1723 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_13.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 13, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 9054 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont9(Void && _,int loopDepth) 
	{
															#line 1723 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_13 = tr->commit();
															#line 1723 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont3loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 9065 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_13.isReady()) { if (__when_expr_13.isError()) return a_body1loopBody1cont3loopBody1Catch1(__when_expr_13.getError(), loopDepth); else return a_body1loopBody1cont3loopBody1cont9when1(__when_expr_13.get(), loopDepth); };
		static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 14;
															#line 1723 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_13.addCallbackAndClear(static_cast<ActorCallback< MonitorClientRangesActor, 13, Void >*>(static_cast<MonitorClientRangesActor*>(this)));
															#line 9070 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont2when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont2when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1loopBody1cont3loopBody1cont2when2(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1loopBody1cont3loopBody1cont2when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1loopBody1cont3loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 12);

	}
	int a_body1loopBody1cont3loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 1724 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 9154 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1725 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Blob manager done processing client ranges @ {0}, retrying\n", tr->getCommittedVersion());
															#line 9158 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont3loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 1724 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 9168 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1725 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Blob manager done processing client ranges @ {0}, retrying\n", tr->getCommittedVersion());
															#line 9172 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont3loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose14() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 13, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 13, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1loopBody1cont3loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 13, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1loopBody1cont3loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 13, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1loopBody1cont3loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 13);

	}
	int a_body1loopBody1cont3loopBody1cont12(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont12(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose15() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 14, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 14, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose15();
		try {
			a_body1loopBody1cont3loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 14, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose15();
		try {
			a_body1loopBody1cont3loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 14, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose15();
		try {
			a_body1loopBody1cont3loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont3loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont3loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 14);

	}
	int a_body1loopBody1cont3loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose16() 
	{
		if (static_cast<MonitorClientRangesActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientRangesActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientRangesActor*>(this)->ActorCallback< MonitorClientRangesActor, 15, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 15, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose16();
		try {
			a_body1loopBody1cont3loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 15);

	}
	void a_callback_fire(ActorCallback< MonitorClientRangesActor, 15, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose16();
		try {
			a_body1loopBody1cont3loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 15);

	}
	void a_callback_error(ActorCallback< MonitorClientRangesActor, 15, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose16();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), 15);

	}
															#line 1474 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 1475 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Optional<Value> lastChangeKeyValue;
															#line 1476 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	bool needToCoalesce;
															#line 1477 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	bool firstLoad;
															#line 1478 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Version lastChangeLogVersion;
															#line 1481 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1486 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	VectorRef<KeyRangeRef> rangesToAdd;
															#line 1487 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	VectorRef<KeyRangeRef> rangesToRemove;
															#line 1488 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Arena ar;
															#line 1489 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Optional<Value> ckvBegin;
															#line 1501 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	RangeResult results;
															#line 1550 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	RangeResult changeLog;
															#line 1566 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int i;
															#line 1612 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	std::vector<Future<BlobGranuleSplitPoints>> splitFutures;
															#line 1648 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	decltype(std::begin(std::declval<std::vector<Future<BlobGranuleSplitPoints>>>())) RangeForbody1loopBody1cont1Iterator0;
															#line 1649 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	BlobGranuleSplitPoints splitPoints;
															#line 1687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Future<Void> watchFuture;
															#line 9461 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via monitorClientRanges()
															#line 1474 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class MonitorClientRangesActor final : public Actor<Void>, public ActorCallback< MonitorClientRangesActor, 0, Void >, public ActorCallback< MonitorClientRangesActor, 1, RangeResult >, public ActorCallback< MonitorClientRangesActor, 2, Void >, public ActorCallback< MonitorClientRangesActor, 3, RangeResult >, public ActorCallback< MonitorClientRangesActor, 4, Void >, public ActorCallback< MonitorClientRangesActor, 5, Void >, public ActorCallback< MonitorClientRangesActor, 6, Void >, public ActorCallback< MonitorClientRangesActor, 7, BlobGranuleSplitPoints >, public ActorCallback< MonitorClientRangesActor, 8, Void >, public ActorCallback< MonitorClientRangesActor, 9, Optional<Value> >, public ActorCallback< MonitorClientRangesActor, 10, Void >, public ActorCallback< MonitorClientRangesActor, 11, Void >, public ActorCallback< MonitorClientRangesActor, 12, Void >, public ActorCallback< MonitorClientRangesActor, 13, Void >, public ActorCallback< MonitorClientRangesActor, 14, Void >, public ActorCallback< MonitorClientRangesActor, 15, Void >, public FastAllocated<MonitorClientRangesActor>, public MonitorClientRangesActorState<MonitorClientRangesActor> {
															#line 9466 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<MonitorClientRangesActor>::operator new;
	using FastAllocated<MonitorClientRangesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17336565991003897856UL, 3622488350116815104UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorClientRangesActor, 0, Void >;
friend struct ActorCallback< MonitorClientRangesActor, 1, RangeResult >;
friend struct ActorCallback< MonitorClientRangesActor, 2, Void >;
friend struct ActorCallback< MonitorClientRangesActor, 3, RangeResult >;
friend struct ActorCallback< MonitorClientRangesActor, 4, Void >;
friend struct ActorCallback< MonitorClientRangesActor, 5, Void >;
friend struct ActorCallback< MonitorClientRangesActor, 6, Void >;
friend struct ActorCallback< MonitorClientRangesActor, 7, BlobGranuleSplitPoints >;
friend struct ActorCallback< MonitorClientRangesActor, 8, Void >;
friend struct ActorCallback< MonitorClientRangesActor, 9, Optional<Value> >;
friend struct ActorCallback< MonitorClientRangesActor, 10, Void >;
friend struct ActorCallback< MonitorClientRangesActor, 11, Void >;
friend struct ActorCallback< MonitorClientRangesActor, 12, Void >;
friend struct ActorCallback< MonitorClientRangesActor, 13, Void >;
friend struct ActorCallback< MonitorClientRangesActor, 14, Void >;
friend struct ActorCallback< MonitorClientRangesActor, 15, Void >;
															#line 1474 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	MonitorClientRangesActor(Reference<BlobManagerData> const& bmData) 
															#line 9498 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<Void>(),
		   MonitorClientRangesActorState<MonitorClientRangesActor>(bmData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorClientRanges", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15686057068050585344UL, 4938376407833360128UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientRangesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorClientRanges");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorClientRanges", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 3, RangeResult >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 7, BlobGranuleSplitPoints >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 9, Optional<Value> >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 10, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 11, Void >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 12, Void >*)0, actor_cancelled()); break;
		case 14: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 13, Void >*)0, actor_cancelled()); break;
		case 15: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 14, Void >*)0, actor_cancelled()); break;
		case 16: this->a_callback_error((ActorCallback< MonitorClientRangesActor, 15, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1474 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<Void> monitorClientRanges( Reference<BlobManagerData> const& bmData ) {
															#line 1474 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<Void>(new MonitorClientRangesActor(bmData));
															#line 9546 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 1743 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

															#line 9551 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via reevaluateInitialSplit()
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class ReevaluateInitialSplitActor>
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class ReevaluateInitialSplitActorState {
															#line 9558 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	ReevaluateInitialSplitActorState(Reference<BlobManagerData> const& bmData,UID const& currentWorkerId,KeyRange const& granuleRange,UID const& granuleID,int64_t const& epoch,int64_t const& seqno,Key const& proposedSplitKey) 
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   currentWorkerId(currentWorkerId),
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   granuleRange(granuleRange),
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   granuleID(granuleID),
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   epoch(epoch),
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   seqno(seqno),
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   proposedSplitKey(proposedSplitKey)
															#line 9577 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this));

	}
	~ReevaluateInitialSplitActorState() 
	{
		fdb_probe_actor_destroy("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1751 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "BM re-evaluating initial split too big");
															#line 1752 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 9594 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1753 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} re-evaluating initial split [{1} - {2}) too big from {3} @ ({4}, {5})\n", bmData->epoch, granuleRange.begin.printable(), granuleRange.end.printable(), currentWorkerId.toString().substr(0, 5), epoch, seqno);
															#line 1760 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("Proposed split (2):\n");
															#line 1761 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("    {0}\n", granuleRange.begin.printable());
															#line 1762 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("    {0}\n", proposedSplitKey.printable());
															#line 1763 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("    {0}\n", granuleRange.end.printable());
															#line 9606 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1765 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("BMCheckInitialSplitTooBig", bmData->id) .detail("Epoch", bmData->epoch) .detail("Granule", granuleRange) .detail("ProposedSplitKey", proposedSplitKey);
															#line 1770 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ASSERT(granuleRange.begin < proposedSplitKey);
															#line 1771 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ASSERT(proposedSplitKey < granuleRange.end);
															#line 1772 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fSplitFirst = splitRange(bmData, KeyRangeRef(granuleRange.begin, proposedSplitKey), false, true);
															#line 1774 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fSplitSecond = splitRange(bmData, KeyRangeRef(proposedSplitKey, granuleRange.end), false, true);
															#line 1777 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			newRanges = Standalone<VectorRef<KeyRef>>();
															#line 1779 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<BlobGranuleSplitPoints> __when_expr_0 = fSplitFirst;
															#line 1779 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9624 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 1;
															#line 1779 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 0, BlobGranuleSplitPoints >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 9629 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReevaluateInitialSplitActorState();
		static_cast<ReevaluateInitialSplitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(BlobGranuleSplitPoints const& splitFirst,int loopDepth) 
	{
															#line 1780 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitFirst.keys.size() >= 2);
															#line 1781 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitFirst.keys.front() == granuleRange.begin);
															#line 1782 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitFirst.keys.back() == proposedSplitKey);
															#line 1783 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Standalone<VectorRef<KeyRef>>> __when_expr_1 = nextFullKeys(bmData, splitFirst.keys);
															#line 1783 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9660 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 2;
															#line 1783 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 1, Standalone<VectorRef<KeyRef>> >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 9665 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(BlobGranuleSplitPoints && splitFirst,int loopDepth) 
	{
															#line 1780 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitFirst.keys.size() >= 2);
															#line 1781 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitFirst.keys.front() == granuleRange.begin);
															#line 1782 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitFirst.keys.back() == proposedSplitKey);
															#line 1783 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Standalone<VectorRef<KeyRef>>> __when_expr_1 = nextFullKeys(bmData, splitFirst.keys);
															#line 1783 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9682 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 2;
															#line 1783 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 1, Standalone<VectorRef<KeyRef>> >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 9687 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(BlobGranuleSplitPoints const& splitFirst,int loopDepth) 
	{
		loopDepth = a_body1cont1(splitFirst, loopDepth);

		return loopDepth;
	}
	int a_body1when1(BlobGranuleSplitPoints && splitFirst,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(splitFirst), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 0, BlobGranuleSplitPoints >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 0, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 0, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 0, BlobGranuleSplitPoints >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Standalone<VectorRef<KeyRef>> const& splitFirstFullKeys,int loopDepth) 
	{
															#line 1784 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(int i = 0;i < splitFirstFullKeys.size();i++) {
															#line 1785 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			newRanges.push_back_deep(newRanges.arena(), splitFirstFullKeys[i]);
															#line 9773 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<BlobGranuleSplitPoints> __when_expr_2 = fSplitSecond;
															#line 1788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9779 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 3;
															#line 1788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 2, BlobGranuleSplitPoints >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 9784 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Standalone<VectorRef<KeyRef>> && splitFirstFullKeys,int loopDepth) 
	{
															#line 1784 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(int i = 0;i < splitFirstFullKeys.size();i++) {
															#line 1785 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			newRanges.push_back_deep(newRanges.arena(), splitFirstFullKeys[i]);
															#line 9795 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<BlobGranuleSplitPoints> __when_expr_2 = fSplitSecond;
															#line 1788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9801 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 3;
															#line 1788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 2, BlobGranuleSplitPoints >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 9806 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Standalone<VectorRef<KeyRef>> const& splitFirstFullKeys,int loopDepth) 
	{
		loopDepth = a_body1cont3(splitFirstFullKeys, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Standalone<VectorRef<KeyRef>> && splitFirstFullKeys,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(splitFirstFullKeys), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 1, Standalone<VectorRef<KeyRef>> >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 1, Standalone<VectorRef<KeyRef>> >*,Standalone<VectorRef<KeyRef>> const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 1, Standalone<VectorRef<KeyRef>> >*,Standalone<VectorRef<KeyRef>> && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 1, Standalone<VectorRef<KeyRef>> >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(BlobGranuleSplitPoints const& splitSecond,int loopDepth) 
	{
															#line 1789 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitSecond.keys.size() >= 2);
															#line 1790 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitSecond.keys.front() == proposedSplitKey);
															#line 1791 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitSecond.keys.back() == granuleRange.end);
															#line 1792 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Standalone<VectorRef<KeyRef>>> __when_expr_3 = nextFullKeys(bmData, splitSecond.keys);
															#line 1792 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9898 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 4;
															#line 1792 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 3, Standalone<VectorRef<KeyRef>> >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 9903 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(BlobGranuleSplitPoints && splitSecond,int loopDepth) 
	{
															#line 1789 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitSecond.keys.size() >= 2);
															#line 1790 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitSecond.keys.front() == proposedSplitKey);
															#line 1791 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitSecond.keys.back() == granuleRange.end);
															#line 1792 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Standalone<VectorRef<KeyRef>>> __when_expr_3 = nextFullKeys(bmData, splitSecond.keys);
															#line 1792 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9920 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 4;
															#line 1792 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 3, Standalone<VectorRef<KeyRef>> >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 9925 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3when1(BlobGranuleSplitPoints const& splitSecond,int loopDepth) 
	{
		loopDepth = a_body1cont4(splitSecond, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(BlobGranuleSplitPoints && splitSecond,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(splitSecond), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 2, BlobGranuleSplitPoints >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 2, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 2, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 2, BlobGranuleSplitPoints >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(Standalone<VectorRef<KeyRef>> const& splitSecondFullKeys,int loopDepth) 
	{
															#line 1794 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(int i = 1;i < splitSecondFullKeys.size();i++) {
															#line 1795 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			newRanges.push_back_deep(newRanges.arena(), splitSecondFullKeys[i]);
															#line 10011 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1798 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 10015 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1799 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Re-evaluated split ({0}):\n", newRanges.size());
															#line 1800 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for( auto& it : newRanges ) {
															#line 1801 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("    {0}\n", it.printable());
															#line 10023 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
															#line 1807 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<BlobGranuleSplitPoints> __when_expr_4 = alignKeys(bmData, granuleRange, newRanges);
															#line 1807 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10030 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont6when1(__when_expr_4.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 5;
															#line 1807 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 4, BlobGranuleSplitPoints >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10035 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6(Standalone<VectorRef<KeyRef>> && splitSecondFullKeys,int loopDepth) 
	{
															#line 1794 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(int i = 1;i < splitSecondFullKeys.size();i++) {
															#line 1795 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			newRanges.push_back_deep(newRanges.arena(), splitSecondFullKeys[i]);
															#line 10046 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1798 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 10050 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1799 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Re-evaluated split ({0}):\n", newRanges.size());
															#line 1800 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for( auto& it : newRanges ) {
															#line 1801 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("    {0}\n", it.printable());
															#line 10058 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
															#line 1807 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<BlobGranuleSplitPoints> __when_expr_4 = alignKeys(bmData, granuleRange, newRanges);
															#line 1807 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10065 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont6when1(__when_expr_4.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 5;
															#line 1807 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 4, BlobGranuleSplitPoints >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10070 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4when1(Standalone<VectorRef<KeyRef>> const& splitSecondFullKeys,int loopDepth) 
	{
		loopDepth = a_body1cont6(splitSecondFullKeys, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Standalone<VectorRef<KeyRef>> && splitSecondFullKeys,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(splitSecondFullKeys), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 3, Standalone<VectorRef<KeyRef>> >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 3, Standalone<VectorRef<KeyRef>> >*,Standalone<VectorRef<KeyRef>> const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 3, Standalone<VectorRef<KeyRef>> >*,Standalone<VectorRef<KeyRef>> && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 3, Standalone<VectorRef<KeyRef>> >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 1809 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 10154 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1810 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Aligned split ({0}):\n", finalSplit.keys.size());
															#line 1811 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for( auto& it : finalSplit.keys ) {
															#line 1812 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("    {0}{1}\n", it.printable(), finalSplit.boundaries.contains(it) ? " *" : "");
															#line 10162 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
															#line 1816 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(finalSplit.keys.size() > 2);
															#line 1821 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>(bmData->db);
															#line 1822 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		lockKey = blobGranuleLockKeyFor(granuleRange);
															#line 1823 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		retried = false;
															#line 1824 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		;
															#line 10175 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont7loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(BlobGranuleSplitPoints const& __finalSplit,int loopDepth) 
	{
															#line 1807 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		finalSplit = __finalSplit;
															#line 10184 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(BlobGranuleSplitPoints && __finalSplit,int loopDepth) 
	{
		finalSplit = std::move(__finalSplit);
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 4, BlobGranuleSplitPoints >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 4, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 4, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 4, BlobGranuleSplitPoints >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont11(int loopDepth) 
	{
															#line 1959 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BUGGIFY && bmData->maybeInjectTargetedRestart())
															#line 10263 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1960 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_16 = delay(0);
															#line 1960 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10269 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_16.isReady()) { if (__when_expr_16.isError()) return a_body1Catch1(__when_expr_16.getError(), loopDepth); else return a_body1cont11when1(__when_expr_16.get(), loopDepth); };
			static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 17;
															#line 1960 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_16.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 16, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10274 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont11cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont7loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1(int loopDepth) 
	{
		try {
															#line 1826 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1827 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1828 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1830 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_5 = checkManagerLock(tr, bmData);
															#line 1830 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10304 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont7loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont7loopBody1when1(__when_expr_5.get(), loopDepth); };
			static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 6;
															#line 1830 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 5, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10309 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont7loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont7loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7break1(int loopDepth) 
	{
		try {
			return a_body1cont11(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont7loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1955 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_15 = tr->onError(e);
															#line 1955 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10346 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_15.isReady()) { if (__when_expr_15.isError()) return a_body1Catch1(__when_expr_15.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1Catch1when1(__when_expr_15.get(), loopDepth); };
			static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 16;
															#line 1955 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_15.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 15, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10351 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 1832 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<ForcedPurgeState> __when_expr_6 = getForcePurgedState(&tr->getTransaction(), granuleRange);
															#line 1832 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10368 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont7loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont7loopBody1cont2when1(__when_expr_6.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 7;
															#line 1832 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 6, ForcedPurgeState >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10373 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 1832 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<ForcedPurgeState> __when_expr_6 = getForcePurgedState(&tr->getTransaction(), granuleRange);
															#line 1832 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10384 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont7loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont7loopBody1cont2when1(__when_expr_6.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 7;
															#line 1832 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 6, ForcedPurgeState >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10389 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont7loopBody1cont3(ForcedPurgeState const& purgeState,int loopDepth) 
	{
															#line 1833 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (purgeState != ForcedPurgeState::NonePurged)
															#line 10473 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1834 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "Initial Split Re-evaluate stopped because of force purge", probe::decoration::rare);
															#line 1835 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("GranuleSplitReEvalCancelledForcePurge", bmData->id) .detail("Epoch", bmData->epoch) .detail("GranuleRange", granuleRange);
															#line 1840 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_7 = updateChangeFeed(&tr->getTransaction(), granuleIDToCFKey(granuleID), ChangeFeedStatus::CHANGE_FEED_DESTROY, granuleRange);
															#line 1840 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10483 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont7loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont7loopBody1cont3when1(__when_expr_7.get(), loopDepth); };
			static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 8;
															#line 1840 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 7, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10488 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont3(ForcedPurgeState && purgeState,int loopDepth) 
	{
															#line 1833 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (purgeState != ForcedPurgeState::NonePurged)
															#line 10502 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1834 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "Initial Split Re-evaluate stopped because of force purge", probe::decoration::rare);
															#line 1835 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("GranuleSplitReEvalCancelledForcePurge", bmData->id) .detail("Epoch", bmData->epoch) .detail("GranuleRange", granuleRange);
															#line 1840 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_7 = updateChangeFeed(&tr->getTransaction(), granuleIDToCFKey(granuleID), ChangeFeedStatus::CHANGE_FEED_DESTROY, granuleRange);
															#line 1840 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10512 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont7loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont7loopBody1cont3when1(__when_expr_7.get(), loopDepth); };
			static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 8;
															#line 1840 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 7, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10517 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont2when1(ForcedPurgeState const& purgeState,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont3(purgeState, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont2when1(ForcedPurgeState && purgeState,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont3(std::move(purgeState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 6, ForcedPurgeState >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 6, ForcedPurgeState >*,ForcedPurgeState const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont7loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 6, ForcedPurgeState >*,ForcedPurgeState && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont7loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 6, ForcedPurgeState >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont7loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont7loopBody1cont4(int loopDepth) 
	{
															#line 1852 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		KeyRange range = blobGranuleFileKeyRangeFor(granuleID);
															#line 1853 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<RangeResult> __when_expr_9 = tr->getRange(range, 1);
															#line 1853 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10610 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont7loopBody1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont7loopBody1cont4when1(__when_expr_9.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 10;
															#line 1853 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 9, RangeResult >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10615 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 1845 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_8 = tr->commit();
															#line 1845 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10626 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont7loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont7loopBody1cont5when1(__when_expr_8.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 9;
															#line 1845 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 8, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10631 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 1845 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_8 = tr->commit();
															#line 1845 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10642 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont7loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont7loopBody1cont5when1(__when_expr_8.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 9;
															#line 1845 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 8, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10647 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont7loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont7loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont7loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont7loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1847 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<ReevaluateInitialSplitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReevaluateInitialSplitActorState(); static_cast<ReevaluateInitialSplitActor*>(this)->destroy(); return 0; }
															#line 10731 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<ReevaluateInitialSplitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ReevaluateInitialSplitActorState();
		static_cast<ReevaluateInitialSplitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1847 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<ReevaluateInitialSplitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReevaluateInitialSplitActorState(); static_cast<ReevaluateInitialSplitActor*>(this)->destroy(); return 0; }
															#line 10743 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<ReevaluateInitialSplitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ReevaluateInitialSplitActorState();
		static_cast<ReevaluateInitialSplitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont7loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont7loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont7loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont7loopBody1cont8(RangeResult const& granuleFiles,int loopDepth) 
	{
															#line 1854 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!granuleFiles.empty())
															#line 10830 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1855 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "split too big was eventually solved by another worker", probe::decoration::rare);
															#line 1856 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 10836 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1857 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} re-evaluating initial split [{1} - {2}) too big: solved by another worker\n", bmData->epoch, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 10840 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1862 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<ReevaluateInitialSplitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReevaluateInitialSplitActorState(); static_cast<ReevaluateInitialSplitActor*>(this)->destroy(); return 0; }
															#line 10844 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<ReevaluateInitialSplitActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReevaluateInitialSplitActorState();
			static_cast<ReevaluateInitialSplitActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1865 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_10 = tr->get(lockKey);
															#line 1865 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10854 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont7loopBody1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont7loopBody1cont8when1(__when_expr_10.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 11;
															#line 1865 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 10, Optional<Value> >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10859 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont8(RangeResult && granuleFiles,int loopDepth) 
	{
															#line 1854 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!granuleFiles.empty())
															#line 10868 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1855 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "split too big was eventually solved by another worker", probe::decoration::rare);
															#line 1856 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 10874 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1857 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} re-evaluating initial split [{1} - {2}) too big: solved by another worker\n", bmData->epoch, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 10878 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1862 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<ReevaluateInitialSplitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReevaluateInitialSplitActorState(); static_cast<ReevaluateInitialSplitActor*>(this)->destroy(); return 0; }
															#line 10882 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<ReevaluateInitialSplitActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReevaluateInitialSplitActorState();
			static_cast<ReevaluateInitialSplitActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1865 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_10 = tr->get(lockKey);
															#line 1865 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10892 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont7loopBody1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont7loopBody1cont8when1(__when_expr_10.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 11;
															#line 1865 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 10, Optional<Value> >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 10897 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont4when1(RangeResult const& granuleFiles,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont8(granuleFiles, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont4when1(RangeResult && granuleFiles,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont8(std::move(granuleFiles), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 9, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 9, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont7loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 9, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont7loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 9, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont7loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont7loopBody1cont9(Optional<Value> const& prevLockValue,int loopDepth) 
	{
															#line 1866 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(prevLockValue.present());
															#line 1867 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		std::tuple<int64_t, int64_t, UID> prevOwner = decodeBlobGranuleLockValue(prevLockValue.get());
															#line 1868 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int64_t prevOwnerEpoch = std::get<0>(prevOwner);
															#line 1869 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int64_t prevOwnerSeqno = std::get<1>(prevOwner);
															#line 1870 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		UID prevGranuleID = std::get<2>(prevOwner);
															#line 1871 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (prevOwnerEpoch != epoch || prevOwnerSeqno != seqno || prevGranuleID != granuleID)
															#line 10991 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1872 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (retried && prevOwnerEpoch == bmData->epoch && prevGranuleID == granuleID && prevOwnerSeqno == std::numeric_limits<int64_t>::max())
															#line 10995 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1875 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				CODE_PROBE(true, "split too big adjustment succeeded after retry", probe::decoration::rare);
															#line 10999 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				return a_body1cont7break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 1878 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "split too big was since moved to another worker", probe::decoration::rare);
															#line 1879 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 11006 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1880 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} re-evaluating initial split [{1} - {2}) too big: moved to another worker\n", bmData->epoch, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 1884 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("Epoch: Prev {0}, Cur {1}\n", prevOwnerEpoch, epoch);
															#line 1885 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("Seqno: Prev {0}, Cur {1}\n", prevOwnerSeqno, seqno);
															#line 1886 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("GranuleID: Prev {0}, Cur {1}\n", prevGranuleID.toString().substr(0, 6), granuleID.toString().substr(0, 6));
															#line 11016 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1890 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<ReevaluateInitialSplitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReevaluateInitialSplitActorState(); static_cast<ReevaluateInitialSplitActor*>(this)->destroy(); return 0; }
															#line 11020 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<ReevaluateInitialSplitActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReevaluateInitialSplitActorState();
			static_cast<ReevaluateInitialSplitActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1893 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (prevOwnerEpoch > bmData->epoch)
															#line 11028 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1894 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 11032 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1895 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} found a higher epoch {1} for granule lock of [{2} - {3})\n", bmData->epoch, prevOwnerEpoch, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 11036 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1902 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (bmData->iAmReplaced.canBeSet())
															#line 11040 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1903 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->iAmReplaced.send(Void());
															#line 11044 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1905 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<ReevaluateInitialSplitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReevaluateInitialSplitActorState(); static_cast<ReevaluateInitialSplitActor*>(this)->destroy(); return 0; }
															#line 11048 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<ReevaluateInitialSplitActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReevaluateInitialSplitActorState();
			static_cast<ReevaluateInitialSplitActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1910 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<RangeResult> __when_expr_11 = krmGetRanges(tr, blobGranuleMappingKeys.begin, granuleRange, 3, GetRangeLimits::BYTE_LIMIT_UNLIMITED);
															#line 1910 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11058 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1cont7loopBody1Catch1(__when_expr_11.getError(), loopDepth); else return a_body1cont7loopBody1cont9when1(__when_expr_11.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 12;
															#line 1910 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 11, RangeResult >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 11063 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont9(Optional<Value> && prevLockValue,int loopDepth) 
	{
															#line 1866 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(prevLockValue.present());
															#line 1867 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		std::tuple<int64_t, int64_t, UID> prevOwner = decodeBlobGranuleLockValue(prevLockValue.get());
															#line 1868 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int64_t prevOwnerEpoch = std::get<0>(prevOwner);
															#line 1869 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int64_t prevOwnerSeqno = std::get<1>(prevOwner);
															#line 1870 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		UID prevGranuleID = std::get<2>(prevOwner);
															#line 1871 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (prevOwnerEpoch != epoch || prevOwnerSeqno != seqno || prevGranuleID != granuleID)
															#line 11082 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1872 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (retried && prevOwnerEpoch == bmData->epoch && prevGranuleID == granuleID && prevOwnerSeqno == std::numeric_limits<int64_t>::max())
															#line 11086 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1875 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				CODE_PROBE(true, "split too big adjustment succeeded after retry", probe::decoration::rare);
															#line 11090 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				return a_body1cont7break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 1878 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "split too big was since moved to another worker", probe::decoration::rare);
															#line 1879 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 11097 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1880 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} re-evaluating initial split [{1} - {2}) too big: moved to another worker\n", bmData->epoch, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 1884 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("Epoch: Prev {0}, Cur {1}\n", prevOwnerEpoch, epoch);
															#line 1885 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("Seqno: Prev {0}, Cur {1}\n", prevOwnerSeqno, seqno);
															#line 1886 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("GranuleID: Prev {0}, Cur {1}\n", prevGranuleID.toString().substr(0, 6), granuleID.toString().substr(0, 6));
															#line 11107 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1890 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<ReevaluateInitialSplitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReevaluateInitialSplitActorState(); static_cast<ReevaluateInitialSplitActor*>(this)->destroy(); return 0; }
															#line 11111 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<ReevaluateInitialSplitActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReevaluateInitialSplitActorState();
			static_cast<ReevaluateInitialSplitActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1893 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (prevOwnerEpoch > bmData->epoch)
															#line 11119 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1894 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 11123 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1895 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} found a higher epoch {1} for granule lock of [{2} - {3})\n", bmData->epoch, prevOwnerEpoch, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 11127 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1902 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (bmData->iAmReplaced.canBeSet())
															#line 11131 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1903 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->iAmReplaced.send(Void());
															#line 11135 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 1905 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<ReevaluateInitialSplitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReevaluateInitialSplitActorState(); static_cast<ReevaluateInitialSplitActor*>(this)->destroy(); return 0; }
															#line 11139 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<ReevaluateInitialSplitActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReevaluateInitialSplitActorState();
			static_cast<ReevaluateInitialSplitActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1910 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<RangeResult> __when_expr_11 = krmGetRanges(tr, blobGranuleMappingKeys.begin, granuleRange, 3, GetRangeLimits::BYTE_LIMIT_UNLIMITED);
															#line 1910 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11149 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1cont7loopBody1Catch1(__when_expr_11.getError(), loopDepth); else return a_body1cont7loopBody1cont9when1(__when_expr_11.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 12;
															#line 1910 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 11, RangeResult >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 11154 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont8when1(Optional<Value> const& prevLockValue,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont9(prevLockValue, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont8when1(Optional<Value> && prevLockValue,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont9(std::move(prevLockValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 10, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 10, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont7loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 10, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont7loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 10, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont7loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont7loopBody1cont12(RangeResult const& existingRanges,int loopDepth) 
	{
															#line 1912 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (existingRanges.size() > 2 || existingRanges.more)
															#line 11238 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1913 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "split too big was already re-split", probe::decoration::rare);
															#line 1914 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 11244 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1915 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} re-evaluating initial split [{1} - {2}) too big: already split\n", bmData->epoch, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 1919 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				for( auto& it : existingRanges ) {
															#line 1920 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					fmt::print("  {0}\n", it.key.printable());
															#line 11252 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
			}
															#line 1923 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<ReevaluateInitialSplitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReevaluateInitialSplitActorState(); static_cast<ReevaluateInitialSplitActor*>(this)->destroy(); return 0; }
															#line 11257 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<ReevaluateInitialSplitActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReevaluateInitialSplitActorState();
			static_cast<ReevaluateInitialSplitActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1928 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->set(lockKey, blobGranuleLockValueFor(bmData->epoch, std::numeric_limits<int64_t>::max(), granuleID));
															#line 1931 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i = int();
															#line 1932 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i = 0;
															#line 11269 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont7loopBody1cont12loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12(RangeResult && existingRanges,int loopDepth) 
	{
															#line 1912 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (existingRanges.size() > 2 || existingRanges.more)
															#line 11278 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1913 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "split too big was already re-split", probe::decoration::rare);
															#line 1914 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 11284 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 1915 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} re-evaluating initial split [{1} - {2}) too big: already split\n", bmData->epoch, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 1919 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				for( auto& it : existingRanges ) {
															#line 1920 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					fmt::print("  {0}\n", it.key.printable());
															#line 11292 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
			}
															#line 1923 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<ReevaluateInitialSplitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReevaluateInitialSplitActorState(); static_cast<ReevaluateInitialSplitActor*>(this)->destroy(); return 0; }
															#line 11297 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<ReevaluateInitialSplitActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReevaluateInitialSplitActorState();
			static_cast<ReevaluateInitialSplitActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1928 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->set(lockKey, blobGranuleLockValueFor(bmData->epoch, std::numeric_limits<int64_t>::max(), granuleID));
															#line 1931 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i = int();
															#line 1932 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i = 0;
															#line 11309 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont7loopBody1cont12loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont9when1(RangeResult const& existingRanges,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont12(existingRanges, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont9when1(RangeResult && existingRanges,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont12(std::move(existingRanges), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 11, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 11, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont7loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 11, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont7loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 11, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont7loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont7loopBody1cont12cont1(int loopDepth) 
	{
															#line 1946 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_13 = updateChangeFeed(&tr->getTransaction(), granuleIDToCFKey(granuleID), ChangeFeedStatus::CHANGE_FEED_DESTROY, granuleRange);
															#line 1946 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11395 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_13.isReady()) { if (__when_expr_13.isError()) return a_body1cont7loopBody1Catch1(__when_expr_13.getError(), loopDepth); else return a_body1cont7loopBody1cont12cont1when1(__when_expr_13.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 14;
															#line 1946 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_13.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 13, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 11400 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont7loopBody1cont12loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12loopBody1(int loopDepth) 
	{
															#line 1932 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!(i < finalSplit.keys.size() - 1))
															#line 11416 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1cont7loopBody1cont12break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1933 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_12 = krmSetRange(tr, blobGranuleMappingKeys.begin, KeyRangeRef(finalSplit.keys[i], finalSplit.keys[i + 1]), blobGranuleMappingValueFor(UID()));
															#line 1933 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11424 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1cont7loopBody1Catch1(__when_expr_12.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont12loopBody1when1(__when_expr_12.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 13;
															#line 1933 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 12, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 11429 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12break1(int loopDepth) 
	{
		try {
			return a_body1cont7loopBody1cont12cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont7loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont7loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1937 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (finalSplit.boundaries.contains(finalSplit.keys[i]))
															#line 11451 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1938 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->set(blobGranuleMergeBoundaryKeyFor(finalSplit.keys[i]), blobGranuleMergeBoundaryValueFor(finalSplit.boundaries[finalSplit.keys[i]]));
															#line 11455 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1932 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i++;
															#line 11459 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont7loopBody1cont12loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1937 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (finalSplit.boundaries.contains(finalSplit.keys[i]))
															#line 11468 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1938 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->set(blobGranuleMergeBoundaryKeyFor(finalSplit.keys[i]), blobGranuleMergeBoundaryValueFor(finalSplit.boundaries[finalSplit.keys[i]]));
															#line 11472 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1932 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i++;
															#line 11476 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont7loopBody1cont12loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont12loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont12loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont7loopBody1cont12loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont7loopBody1cont12loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont7loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 12);

	}
	int a_body1cont7loopBody1cont12cont5(Void const& _,int loopDepth) 
	{
															#line 1951 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		retried = true;
															#line 1952 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_14 = tr->commit();
															#line 1952 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11564 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_14.isReady()) { if (__when_expr_14.isError()) return a_body1cont7loopBody1Catch1(__when_expr_14.getError(), loopDepth); else return a_body1cont7loopBody1cont12cont5when1(__when_expr_14.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 15;
															#line 1952 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_14.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 14, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 11569 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12cont5(Void && _,int loopDepth) 
	{
															#line 1951 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		retried = true;
															#line 1952 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_14 = tr->commit();
															#line 1952 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 11582 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_14.isReady()) { if (__when_expr_14.isError()) return a_body1cont7loopBody1Catch1(__when_expr_14.getError(), loopDepth); else return a_body1cont7loopBody1cont12cont5when1(__when_expr_14.get(), loopDepth); };
		static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 15;
															#line 1952 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_14.addCallbackAndClear(static_cast<ActorCallback< ReevaluateInitialSplitActor, 14, Void >*>(static_cast<ReevaluateInitialSplitActor*>(this)));
															#line 11587 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont12cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont12cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose14() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 13, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 13, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1cont7loopBody1cont12cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 13, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1cont7loopBody1cont12cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 13, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1cont7loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 13);

	}
	int a_body1cont7loopBody1cont12cont6(Void const& _,int loopDepth) 
	{
		return a_body1cont7break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12cont6(Void && _,int loopDepth) 
	{
		return a_body1cont7break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont12cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont12cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont12cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose15() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 14, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 14, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose15();
		try {
			a_body1cont7loopBody1cont12cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 14, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose15();
		try {
			a_body1cont7loopBody1cont12cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 14, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose15();
		try {
			a_body1cont7loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 14);

	}
	int a_body1cont7loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose16() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 15, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 15, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose16();
		try {
			a_body1cont7loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 15);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 15, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose16();
		try {
			a_body1cont7loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 15);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 15, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose16();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 15);

	}
	int a_body1cont11cont1(int loopDepth) 
	{
															#line 1966 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		CODE_PROBE(true, "BM successfully changed initial split too big");
															#line 1967 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		RangeAssignment raRevoke;
															#line 1968 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		raRevoke.isAssign = false;
															#line 1969 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		raRevoke.keyRange = granuleRange;
															#line 1970 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		raRevoke.revoke = RangeRevokeData(false);
															#line 1971 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		handleRangeAssign(bmData, raRevoke);
															#line 1973 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(int i = 0;i < finalSplit.keys.size() - 1;i++) {
															#line 1975 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			RangeAssignment raAssignSplit;
															#line 1976 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			raAssignSplit.isAssign = true;
															#line 1977 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			raAssignSplit.keyRange = KeyRangeRef(finalSplit.keys[i], finalSplit.keys[i + 1]);
															#line 1978 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			raAssignSplit.assign = RangeAssignmentData();
															#line 1980 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			handleRangeAssign(bmData, raAssignSplit);
															#line 11867 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1983 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 11871 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 1984 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("BM {0} Re-splitting initial range [{1} - {2}) into {3} granules done\n", bmData->epoch, granuleRange.begin.printable(), granuleRange.end.printable(), finalSplit.keys.size() - 1);
															#line 11875 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 1991 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<ReevaluateInitialSplitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReevaluateInitialSplitActorState(); static_cast<ReevaluateInitialSplitActor*>(this)->destroy(); return 0; }
															#line 11879 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<ReevaluateInitialSplitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ReevaluateInitialSplitActorState();
		static_cast<ReevaluateInitialSplitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11cont2(Void const& _,int loopDepth) 
	{
															#line 1961 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(false);
															#line 11891 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont2(Void && _,int loopDepth) 
	{
															#line 1961 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(false);
															#line 11900 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose17() 
	{
		if (static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state > 0) static_cast<ReevaluateInitialSplitActor*>(this)->actor_wait_state = 0;
		static_cast<ReevaluateInitialSplitActor*>(this)->ActorCallback< ReevaluateInitialSplitActor, 16, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 16, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 16);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose17();
		try {
			a_body1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 16);

	}
	void a_callback_fire(ActorCallback< ReevaluateInitialSplitActor, 16, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 16);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose17();
		try {
			a_body1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 16);

	}
	void a_callback_error(ActorCallback< ReevaluateInitialSplitActor, 16, Void >*,Error err) 
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 16);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose17();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), 16);

	}
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	UID currentWorkerId;
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	KeyRange granuleRange;
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	UID granuleID;
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int64_t epoch;
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int64_t seqno;
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Key proposedSplitKey;
															#line 1772 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Future<BlobGranuleSplitPoints> fSplitFirst;
															#line 1774 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Future<BlobGranuleSplitPoints> fSplitSecond;
															#line 1777 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Standalone<VectorRef<KeyRef>> newRanges;
															#line 1807 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	BlobGranuleSplitPoints finalSplit;
															#line 1821 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1822 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Key lockKey;
															#line 1823 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	bool retried;
															#line 1931 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int i;
															#line 12010 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via reevaluateInitialSplit()
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class ReevaluateInitialSplitActor final : public Actor<Void>, public ActorCallback< ReevaluateInitialSplitActor, 0, BlobGranuleSplitPoints >, public ActorCallback< ReevaluateInitialSplitActor, 1, Standalone<VectorRef<KeyRef>> >, public ActorCallback< ReevaluateInitialSplitActor, 2, BlobGranuleSplitPoints >, public ActorCallback< ReevaluateInitialSplitActor, 3, Standalone<VectorRef<KeyRef>> >, public ActorCallback< ReevaluateInitialSplitActor, 4, BlobGranuleSplitPoints >, public ActorCallback< ReevaluateInitialSplitActor, 5, Void >, public ActorCallback< ReevaluateInitialSplitActor, 6, ForcedPurgeState >, public ActorCallback< ReevaluateInitialSplitActor, 7, Void >, public ActorCallback< ReevaluateInitialSplitActor, 8, Void >, public ActorCallback< ReevaluateInitialSplitActor, 9, RangeResult >, public ActorCallback< ReevaluateInitialSplitActor, 10, Optional<Value> >, public ActorCallback< ReevaluateInitialSplitActor, 11, RangeResult >, public ActorCallback< ReevaluateInitialSplitActor, 12, Void >, public ActorCallback< ReevaluateInitialSplitActor, 13, Void >, public ActorCallback< ReevaluateInitialSplitActor, 14, Void >, public ActorCallback< ReevaluateInitialSplitActor, 15, Void >, public ActorCallback< ReevaluateInitialSplitActor, 16, Void >, public FastAllocated<ReevaluateInitialSplitActor>, public ReevaluateInitialSplitActorState<ReevaluateInitialSplitActor> {
															#line 12015 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<ReevaluateInitialSplitActor>::operator new;
	using FastAllocated<ReevaluateInitialSplitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3220384201462359552UL, 9858422917159159552UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReevaluateInitialSplitActor, 0, BlobGranuleSplitPoints >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 1, Standalone<VectorRef<KeyRef>> >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 2, BlobGranuleSplitPoints >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 3, Standalone<VectorRef<KeyRef>> >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 4, BlobGranuleSplitPoints >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 5, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 6, ForcedPurgeState >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 7, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 8, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 9, RangeResult >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 10, Optional<Value> >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 11, RangeResult >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 12, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 13, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 14, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 15, Void >;
friend struct ActorCallback< ReevaluateInitialSplitActor, 16, Void >;
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	ReevaluateInitialSplitActor(Reference<BlobManagerData> const& bmData,UID const& currentWorkerId,KeyRange const& granuleRange,UID const& granuleID,int64_t const& epoch,int64_t const& seqno,Key const& proposedSplitKey) 
															#line 12048 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<Void>(),
		   ReevaluateInitialSplitActorState<ReevaluateInitialSplitActor>(bmData, currentWorkerId, granuleRange, granuleID, epoch, seqno, proposedSplitKey),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12431337998140563968UL, 17512528355319016704UL);
		ActorExecutionContextHelper __helper(static_cast<ReevaluateInitialSplitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("reevaluateInitialSplit");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("reevaluateInitialSplit", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 0, BlobGranuleSplitPoints >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 1, Standalone<VectorRef<KeyRef>> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 2, BlobGranuleSplitPoints >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 3, Standalone<VectorRef<KeyRef>> >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 4, BlobGranuleSplitPoints >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 6, ForcedPurgeState >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 9, RangeResult >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 10, Optional<Value> >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 11, RangeResult >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 12, Void >*)0, actor_cancelled()); break;
		case 14: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 13, Void >*)0, actor_cancelled()); break;
		case 15: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 14, Void >*)0, actor_cancelled()); break;
		case 16: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 15, Void >*)0, actor_cancelled()); break;
		case 17: this->a_callback_error((ActorCallback< ReevaluateInitialSplitActor, 16, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<Void> reevaluateInitialSplit( Reference<BlobManagerData> const& bmData, UID const& currentWorkerId, KeyRange const& granuleRange, UID const& granuleID, int64_t const& epoch, int64_t const& seqno, Key const& proposedSplitKey ) {
															#line 1744 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<Void>(new ReevaluateInitialSplitActor(bmData, currentWorkerId, granuleRange, granuleID, epoch, seqno, proposedSplitKey));
															#line 12097 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 1993 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

															#line 12102 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via maybeSplitRange()
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class MaybeSplitRangeActor>
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class MaybeSplitRangeActorState {
															#line 12109 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	MaybeSplitRangeActorState(Reference<BlobManagerData> const& bmData,UID const& currentWorkerId,KeyRange const& granuleRange,UID const& granuleID,Version const& granuleStartVersion,bool const& writeHot,int64_t const& originalEpoch,int64_t const& originalSeqno) 
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   currentWorkerId(currentWorkerId),
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   granuleRange(granuleRange),
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   granuleID(granuleID),
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   granuleStartVersion(granuleStartVersion),
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   writeHot(writeHot),
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   originalEpoch(originalEpoch),
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   originalSeqno(originalSeqno)
															#line 12130 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("maybeSplitRange", reinterpret_cast<unsigned long>(this));

	}
	~MaybeSplitRangeActorState() 
	{
		fdb_probe_actor_destroy("maybeSplitRange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2002 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (bmData->isForcePurging(granuleRange))
															#line 12145 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2004 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (!static_cast<MaybeSplitRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MaybeSplitRangeActorState(); static_cast<MaybeSplitRangeActor*>(this)->destroy(); return 0; }
															#line 12149 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				new (&static_cast<MaybeSplitRangeActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~MaybeSplitRangeActorState();
				static_cast<MaybeSplitRangeActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2006 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr = makeReference<ReadYourWritesTransaction>(bmData->db);
															#line 2010 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<BlobGranuleSplitPoints> __when_expr_0 = splitRange(bmData, granuleRange, writeHot, false, SERVER_KNOBS->BG_MAX_SPLIT_FANOUT + 1);
															#line 2010 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12161 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 1;
															#line 2010 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 0, BlobGranuleSplitPoints >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 12166 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MaybeSplitRangeActorState();
		static_cast<MaybeSplitRangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2013 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitPoints.keys.size() >= 2);
															#line 2014 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (splitPoints.keys.size() == 2)
															#line 12191 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2016 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 12195 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2017 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("Not splitting existing range [{0} - {1}). Continuing assignment to {2}\n", granuleRange.begin.printable(), granuleRange.end.printable(), currentWorkerId.toString());
															#line 12199 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 2025 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			int64_t seqnoForEval = bmData->seqNo - 1;
															#line 2027 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			RangeAssignment raContinue;
															#line 2028 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			raContinue.isAssign = true;
															#line 2029 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			raContinue.worker = currentWorkerId;
															#line 2030 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			raContinue.keyRange = granuleRange;
															#line 2031 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			raContinue.assign = RangeAssignmentData(AssignRequestType::Continue);
															#line 2032 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bool reassignSuccess = handleRangeAssign(bmData, raContinue);
															#line 2036 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (reassignSuccess)
															#line 12217 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2037 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->boundaryEvaluations.insert( granuleRange, BoundaryEvaluation(bmData->epoch, seqnoForEval, BoundaryEvalType::SPLIT, originalEpoch, originalSeqno));
															#line 12221 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 2042 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<MaybeSplitRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MaybeSplitRangeActorState(); static_cast<MaybeSplitRangeActor*>(this)->destroy(); return 0; }
															#line 12225 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<MaybeSplitRangeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~MaybeSplitRangeActorState();
			static_cast<MaybeSplitRangeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2045 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(granuleRange.begin == splitPoints.keys.front());
															#line 2046 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(granuleRange.end == splitPoints.keys.back());
															#line 2051 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		newGranuleIDs = std::vector<UID>();
															#line 2052 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		newGranuleIDs.reserve(splitPoints.keys.size() - 1);
															#line 2053 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(int i = 0;i < splitPoints.keys.size() - 1;i++) {
															#line 2054 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			newGranuleIDs.push_back(deterministicRandom()->randomUniqueID());
															#line 12243 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2057 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 12247 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2058 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Splitting range [{0} - {1}) into {2} granules:\n", granuleRange.begin.printable(), granuleRange.end.printable(), splitPoints.keys.size() - 1);
															#line 2062 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for(int i = 0;i < splitPoints.keys.size();i++) {
															#line 2063 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("    {0}:{1}{2}\n", (i < newGranuleIDs.size() ? newGranuleIDs[i] : UID()).toString().substr(0, 6).c_str(), splitPoints.keys[i].printable(), splitPoints.boundaries.contains(splitPoints.keys[i]) ? " *" : "");
															#line 12255 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
															#line 2070 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitVersion = Version();
															#line 2074 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		;
															#line 12262 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(BlobGranuleSplitPoints const& __splitPoints,int loopDepth) 
	{
															#line 2010 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitPoints = __splitPoints;
															#line 12271 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(BlobGranuleSplitPoints && __splitPoints,int loopDepth) 
	{
		splitPoints = std::move(__splitPoints);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 0, BlobGranuleSplitPoints >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 0, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 0, BlobGranuleSplitPoints >*,BlobGranuleSplitPoints && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 0, BlobGranuleSplitPoints >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2253 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 12350 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2254 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Splitting range [{0} - {1}) into {2} granules @ {3} done, sending assignments:\n", granuleRange.begin.printable(), granuleRange.end.printable(), splitPoints.keys.size() - 1, splitVersion);
															#line 12354 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2261 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		++bmData->stats.granuleSplits;
															#line 2262 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (writeHot)
															#line 12360 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2263 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			++bmData->stats.granuleWriteHotSplits;
															#line 12364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2266 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int64_t seqnoForEval = bmData->seqNo;
															#line 2270 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		RangeAssignment raRevoke;
															#line 2271 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		raRevoke.isAssign = false;
															#line 2272 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		raRevoke.keyRange = granuleRange;
															#line 2273 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		raRevoke.revoke = RangeRevokeData(false);
															#line 2274 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		handleRangeAssign(bmData, raRevoke);
															#line 2276 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(int i = 0;i < splitPoints.keys.size() - 1;i++) {
															#line 2278 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			RangeAssignment raAssignSplit;
															#line 2279 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			raAssignSplit.isAssign = true;
															#line 2280 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			raAssignSplit.keyRange = KeyRangeRef(splitPoints.keys[i], splitPoints.keys[i + 1]);
															#line 2281 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			raAssignSplit.assign = RangeAssignmentData();
															#line 2283 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			handleRangeAssign(bmData, raAssignSplit);
															#line 12390 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2286 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 12394 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2287 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Splitting range [{0} - {1}) into {2} granules @ {3} got assignments processed\n", granuleRange.begin.printable(), granuleRange.end.printable(), splitPoints.keys.size() - 1, splitVersion);
															#line 12398 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2295 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->boundaryEvaluations.insert( granuleRange, BoundaryEvaluation(bmData->epoch, seqnoForEval, BoundaryEvalType::SPLIT, originalEpoch, originalSeqno));
															#line 2299 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<MaybeSplitRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MaybeSplitRangeActorState(); static_cast<MaybeSplitRangeActor*>(this)->destroy(); return 0; }
															#line 12404 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<MaybeSplitRangeActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MaybeSplitRangeActorState();
		static_cast<MaybeSplitRangeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 2076 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->reset();
															#line 2077 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2078 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2079 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2080 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			ASSERT(splitPoints.keys.size() > 2);
															#line 2083 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_1 = checkManagerLock(tr, bmData);
															#line 2083 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12436 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 2;
															#line 2083 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 1, Void >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 12441 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2237 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 12476 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2238 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 12480 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 2240 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 12484 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2241 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} Persisting granule split got error {1}\n", bmData->epoch, e.name());
															#line 12488 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 2243 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (e.code() == error_code_granule_assignment_conflict)
															#line 12492 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2244 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (bmData->iAmReplaced.canBeSet())
															#line 12496 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 2245 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					bmData->iAmReplaced.send(Void());
															#line 12500 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				}
															#line 2247 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (!static_cast<MaybeSplitRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MaybeSplitRangeActorState(); static_cast<MaybeSplitRangeActor*>(this)->destroy(); return 0; }
															#line 12504 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				new (&static_cast<MaybeSplitRangeActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~MaybeSplitRangeActorState();
				static_cast<MaybeSplitRangeActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2249 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_11 = tr->onError(e);
															#line 2249 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12514 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1Catch1(__when_expr_11.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_11.get(), loopDepth); };
			static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 12;
															#line 2249 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 11, Void >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 12519 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 2084 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<ForcedPurgeState> __when_expr_2 = getForcePurgedState(&tr->getTransaction(), granuleRange);
															#line 2084 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12536 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 3;
															#line 2084 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 2, ForcedPurgeState >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 12541 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 2084 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<ForcedPurgeState> __when_expr_2 = getForcePurgedState(&tr->getTransaction(), granuleRange);
															#line 2084 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12552 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 3;
															#line 2084 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 2, ForcedPurgeState >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 12557 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont3(ForcedPurgeState const& purgeState,int loopDepth) 
	{
															#line 2085 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (purgeState != ForcedPurgeState::NonePurged)
															#line 12641 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2086 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "Split stopped because of force purge", probe::decoration::rare);
															#line 2087 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("GranuleSplitCancelledForcePurge", bmData->id) .detail("Epoch", bmData->epoch) .detail("GranuleRange", granuleRange);
															#line 2090 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<MaybeSplitRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MaybeSplitRangeActorState(); static_cast<MaybeSplitRangeActor*>(this)->destroy(); return 0; }
															#line 12649 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<MaybeSplitRangeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~MaybeSplitRangeActorState();
			static_cast<MaybeSplitRangeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2096 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<RangeResult> __when_expr_3 = tr->getRange(blobGranuleSplitKeyRangeFor(granuleID), SERVER_KNOBS->BG_MAX_SPLIT_FANOUT + 2);
															#line 2096 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12659 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 4;
															#line 2096 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 3, RangeResult >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 12664 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(ForcedPurgeState && purgeState,int loopDepth) 
	{
															#line 2085 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (purgeState != ForcedPurgeState::NonePurged)
															#line 12673 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2086 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "Split stopped because of force purge", probe::decoration::rare);
															#line 2087 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("GranuleSplitCancelledForcePurge", bmData->id) .detail("Epoch", bmData->epoch) .detail("GranuleRange", granuleRange);
															#line 2090 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<MaybeSplitRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MaybeSplitRangeActorState(); static_cast<MaybeSplitRangeActor*>(this)->destroy(); return 0; }
															#line 12681 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<MaybeSplitRangeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~MaybeSplitRangeActorState();
			static_cast<MaybeSplitRangeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2096 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<RangeResult> __when_expr_3 = tr->getRange(blobGranuleSplitKeyRangeFor(granuleID), SERVER_KNOBS->BG_MAX_SPLIT_FANOUT + 2);
															#line 2096 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12691 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 4;
															#line 2096 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 3, RangeResult >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 12696 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(ForcedPurgeState const& purgeState,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(purgeState, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(ForcedPurgeState && purgeState,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(std::move(purgeState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 2, ForcedPurgeState >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 2, ForcedPurgeState >*,ForcedPurgeState const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 2, ForcedPurgeState >*,ForcedPurgeState && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 2, ForcedPurgeState >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont4(int loopDepth) 
	{
															#line 2098 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT_WE_THINK(!existingState.more && existingState.size() <= SERVER_KNOBS->BG_MAX_SPLIT_FANOUT + 1);
															#line 2100 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (existingState.more || existingState.size() > SERVER_KNOBS->BG_MAX_SPLIT_FANOUT)
															#line 12782 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2101 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<RangeResult> __when_expr_4 = tr->getRange(blobGranuleSplitKeyRangeFor(granuleID), 10000);
															#line 2101 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
			static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 5;
															#line 2101 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 4, RangeResult >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 12793 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(RangeResult const& __existingState,int loopDepth) 
	{
															#line 2096 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		existingState = __existingState;
															#line 12807 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(RangeResult && __existingState,int loopDepth) 
	{
		existingState = std::move(__existingState);
		loopDepth = a_body1cont1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 3, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 3, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 3, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 3, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont6(int loopDepth) 
	{
															#line 2105 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!existingState.empty())
															#line 12886 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2108 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "Overriding split ranges with existing ones from DB");
															#line 2109 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<RangeResult> __when_expr_5 = tr->getRange(KeyRangeRef(granuleRange.begin.withPrefix(blobGranuleMappingKeys.begin), keyAfter(granuleRange.end).withPrefix(blobGranuleMappingKeys.begin)), existingState.size() + 2);
															#line 2109 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 12894 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont1loopBody1cont6when1(__when_expr_5.get(), loopDepth); };
			static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 6;
															#line 2109 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 5, RangeResult >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 12899 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(RangeResult const& tryAgain,int loopDepth) 
	{
															#line 2102 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(!tryAgain.more);
															#line 2103 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		existingState = tryAgain;
															#line 12915 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont7(RangeResult && tryAgain,int loopDepth) 
	{
															#line 2102 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(!tryAgain.more);
															#line 2103 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		existingState = tryAgain;
															#line 12926 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4when1(RangeResult const& tryAgain,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(tryAgain, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4when1(RangeResult && tryAgain,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont7(std::move(tryAgain), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 4, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 4, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 4, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 4, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont9(int loopDepth) 
	{
															#line 2139 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		lockKey = blobGranuleLockKeyFor(granuleRange);
															#line 2140 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_6 = tr->get(lockKey);
															#line 2140 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13014 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont1loopBody1cont9when1(__when_expr_6.get(), loopDepth); };
		static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 7;
															#line 2140 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 6, Optional<Value> >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 13019 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(RangeResult const& existingBoundaries,int loopDepth) 
	{
															#line 2114 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(!existingBoundaries.more);
															#line 2115 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(existingBoundaries.size() == existingState.size() + 1);
															#line 2116 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitPoints.keys.clear();
															#line 2117 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitPoints.keys.arena().dependsOn(existingBoundaries.arena());
															#line 2118 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for( auto& it : existingBoundaries ) {
															#line 2119 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitPoints.keys.push_back(splitPoints.keys.arena(), it.key.removePrefix(blobGranuleMappingKeys.begin));
															#line 13038 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2123 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitPoints.boundaries.clear();
															#line 2124 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitPoints.keys.front() == granuleRange.begin);
															#line 2125 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitPoints.keys.back() == granuleRange.end);
															#line 2126 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 13048 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2127 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Replaced old range splits for [{0} - {1}) with {2}:\n", granuleRange.begin.printable(), granuleRange.end.printable(), splitPoints.keys.size() - 1);
															#line 2131 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for(int i = 0;i < splitPoints.keys.size();i++) {
															#line 2132 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("    {}\n", splitPoints.keys[i].printable());
															#line 13056 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont10(RangeResult && existingBoundaries,int loopDepth) 
	{
															#line 2114 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(!existingBoundaries.more);
															#line 2115 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(existingBoundaries.size() == existingState.size() + 1);
															#line 2116 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitPoints.keys.clear();
															#line 2117 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitPoints.keys.arena().dependsOn(existingBoundaries.arena());
															#line 2118 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for( auto& it : existingBoundaries ) {
															#line 2119 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			splitPoints.keys.push_back(splitPoints.keys.arena(), it.key.removePrefix(blobGranuleMappingKeys.begin));
															#line 13077 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2123 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitPoints.boundaries.clear();
															#line 2124 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitPoints.keys.front() == granuleRange.begin);
															#line 2125 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(splitPoints.keys.back() == granuleRange.end);
															#line 2126 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 13087 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2127 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Replaced old range splits for [{0} - {1}) with {2}:\n", granuleRange.begin.printable(), granuleRange.end.printable(), splitPoints.keys.size() - 1);
															#line 2131 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for(int i = 0;i < splitPoints.keys.size();i++) {
															#line 2132 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("    {}\n", splitPoints.keys[i].printable());
															#line 13095 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when1(RangeResult const& existingBoundaries,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(existingBoundaries, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6when1(RangeResult && existingBoundaries,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont10(std::move(existingBoundaries), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 5, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 5, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 5, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 5, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont1loopBody1cont11(Optional<Value> const& lockValue,int loopDepth) 
	{
															#line 2141 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(lockValue.present());
															#line 2142 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		std::tuple<int64_t, int64_t, UID> prevGranuleLock = decodeBlobGranuleLockValue(lockValue.get());
															#line 2143 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int64_t ownerEpoch = std::get<0>(prevGranuleLock);
															#line 2145 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (ownerEpoch > bmData->epoch)
															#line 13187 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2146 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 13191 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2147 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} found a higher epoch {1} than {2} for granule lock of [{3} - {4})\n", bmData->epoch, ownerEpoch, bmData->epoch, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 13195 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 2155 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (bmData->iAmReplaced.canBeSet())
															#line 13199 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2156 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->iAmReplaced.send(Void());
															#line 13203 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 2158 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<MaybeSplitRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MaybeSplitRangeActorState(); static_cast<MaybeSplitRangeActor*>(this)->destroy(); return 0; }
															#line 13207 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<MaybeSplitRangeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~MaybeSplitRangeActorState();
			static_cast<MaybeSplitRangeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2164 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->set(lockKey, blobGranuleLockValueFor( bmData->epoch, std::numeric_limits<int64_t>::max(), std::get<2>(prevGranuleLock)));
															#line 2169 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<RangeResult> __when_expr_7 = tr->getRange(blobGranuleFileKeyRangeFor(granuleID), 1, Snapshot::False, Reverse::True);
															#line 2169 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13219 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont1loopBody1cont11when1(__when_expr_7.get(), loopDepth); };
		static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 8;
															#line 2169 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 7, RangeResult >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 13224 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11(Optional<Value> && lockValue,int loopDepth) 
	{
															#line 2141 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(lockValue.present());
															#line 2142 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		std::tuple<int64_t, int64_t, UID> prevGranuleLock = decodeBlobGranuleLockValue(lockValue.get());
															#line 2143 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int64_t ownerEpoch = std::get<0>(prevGranuleLock);
															#line 2145 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (ownerEpoch > bmData->epoch)
															#line 13239 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2146 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 13243 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2147 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("BM {0} found a higher epoch {1} than {2} for granule lock of [{3} - {4})\n", bmData->epoch, ownerEpoch, bmData->epoch, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 13247 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 2155 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (bmData->iAmReplaced.canBeSet())
															#line 13251 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2156 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				bmData->iAmReplaced.send(Void());
															#line 13255 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 2158 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (!static_cast<MaybeSplitRangeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MaybeSplitRangeActorState(); static_cast<MaybeSplitRangeActor*>(this)->destroy(); return 0; }
															#line 13259 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			new (&static_cast<MaybeSplitRangeActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~MaybeSplitRangeActorState();
			static_cast<MaybeSplitRangeActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 2164 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->set(lockKey, blobGranuleLockValueFor( bmData->epoch, std::numeric_limits<int64_t>::max(), std::get<2>(prevGranuleLock)));
															#line 2169 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<RangeResult> __when_expr_7 = tr->getRange(blobGranuleFileKeyRangeFor(granuleID), 1, Snapshot::False, Reverse::True);
															#line 2169 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13271 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont1loopBody1cont11when1(__when_expr_7.get(), loopDepth); };
		static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 8;
															#line 2169 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 7, RangeResult >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 13276 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9when1(Optional<Value> const& lockValue,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11(lockValue, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9when1(Optional<Value> && lockValue,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11(std::move(lockValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 6, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 6, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 6, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 6, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont1loopBody1cont11cont1(RangeResult const& lastDeltaFile,int loopDepth) 
	{
															#line 2171 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(lastDeltaFile.size() == 1);
															#line 2172 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		std::tuple<UID, Version, uint8_t> k = decodeBlobGranuleFileKey(lastDeltaFile[0].key);
															#line 2173 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(std::get<0>(k) == granuleID);
															#line 2174 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(std::get<2>(k) == 'D');
															#line 2175 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitVersion = std::get<1>(k);
															#line 2177 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 13370 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2178 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("BM {0} found version {1} for splitting [{2} - {3})\n", bmData->epoch, splitVersion, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 13374 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2186 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i = int();
															#line 2187 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i = 0;
															#line 13380 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont11cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont1(RangeResult && lastDeltaFile,int loopDepth) 
	{
															#line 2171 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(lastDeltaFile.size() == 1);
															#line 2172 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		std::tuple<UID, Version, uint8_t> k = decodeBlobGranuleFileKey(lastDeltaFile[0].key);
															#line 2173 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(std::get<0>(k) == granuleID);
															#line 2174 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(std::get<2>(k) == 'D');
															#line 2175 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		splitVersion = std::get<1>(k);
															#line 2177 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 13399 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2178 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("BM {0} found version {1} for splitting [{2} - {3})\n", bmData->epoch, splitVersion, granuleRange.begin.printable(), granuleRange.end.printable());
															#line 13403 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2186 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i = int();
															#line 2187 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i = 0;
															#line 13409 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont11cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11when1(RangeResult const& lastDeltaFile,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont1(lastDeltaFile, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11when1(RangeResult && lastDeltaFile,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont1(std::move(lastDeltaFile), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 7, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 7, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 7, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 7, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont1loopBody1cont11cont5(int loopDepth) 
	{
															#line 2223 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_9 = tr->commit();
															#line 2223 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13495 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont1loopBody1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont1loopBody1cont11cont5when1(__when_expr_9.get(), loopDepth); };
		static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 10;
															#line 2223 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 9, Void >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 13500 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1cont11cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont1loopBody1(int loopDepth) 
	{
															#line 2187 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!(i < splitPoints.keys.size() - 1))
															#line 13516 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1cont1loopBody1cont11cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2188 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		KeyRangeRef splitRange(splitPoints.keys[i], splitPoints.keys[i + 1]);
															#line 2191 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		Key splitKey = blobGranuleSplitKeyFor(granuleID, newGranuleIDs[i]);
															#line 2193 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->atomicOp(splitKey, blobGranuleSplitValueFor(BlobGranuleSplitState::Initialized), MutationRef::SetVersionstampedValue);
															#line 2198 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (splitPoints.boundaries.contains(splitRange.begin))
															#line 13528 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2199 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->set(blobGranuleMergeBoundaryKeyFor(splitRange.begin), blobGranuleMergeBoundaryValueFor(splitPoints.boundaries[splitRange.begin]));
															#line 13532 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2204 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		Key historyKey = blobGranuleHistoryKeyFor(splitRange, splitVersion);
															#line 2206 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		Standalone<BlobGranuleHistoryValue> historyValue;
															#line 2207 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		historyValue.granuleID = newGranuleIDs[i];
															#line 2208 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		historyValue.parentBoundaries.push_back(historyValue.arena(), granuleRange.begin);
															#line 2209 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		historyValue.parentBoundaries.push_back(historyValue.arena(), granuleRange.end);
															#line 2210 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		historyValue.parentVersions.push_back(historyValue.arena(), granuleStartVersion);
															#line 2212 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->set(historyKey, blobGranuleHistoryValueFor(historyValue));
															#line 2217 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_8 = krmSetRange(tr, blobGranuleMappingKeys.begin, KeyRangeRef(splitPoints.keys[i], splitPoints.keys[i + 1]), blobGranuleMappingValueFor(currentWorkerId));
															#line 2217 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13552 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont1loopBody1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont11cont1loopBody1when1(__when_expr_8.get(), loopDepth); };
		static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 9;
															#line 2217 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 8, Void >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 13557 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont1loopBody1cont11cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 2187 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i++;
															#line 13579 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopBody1cont11cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 2187 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		i++;
															#line 13588 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopBody1cont11cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1cont11cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1cont11cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont1loopBody1cont11cont7(Void const& _,int loopDepth) 
	{
															#line 2225 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BUGGIFY && bmData->maybeInjectTargetedRestart())
															#line 13672 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2226 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_10 = delay(0);
															#line 2226 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13678 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont1loopBody1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont1loopBody1cont11cont7when1(__when_expr_10.get(), loopDepth); };
			static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 11;
															#line 2226 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 10, Void >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 13683 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont11cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont7(Void && _,int loopDepth) 
	{
															#line 2225 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BUGGIFY && bmData->maybeInjectTargetedRestart())
															#line 13697 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2226 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_10 = delay(0);
															#line 2226 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 13703 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont1loopBody1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont1loopBody1cont11cont7when1(__when_expr_10.get(), loopDepth); };
			static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 11;
															#line 2226 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< MaybeSplitRangeActor, 10, Void >*>(static_cast<MaybeSplitRangeActor*>(this)));
															#line 13708 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont11cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1loopBody1cont11cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1loopBody1cont11cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont1loopBody1cont11cont8(int loopDepth) 
	{
															#line 2231 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(auto it = splitPoints.boundaries.begin();it != splitPoints.boundaries.end();it++) {
															#line 2232 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->mergeBoundaries[it->first] = it->second;
															#line 13799 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont9(Void const& _,int loopDepth) 
	{
															#line 2227 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(false);
															#line 13809 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont11cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont9(Void && _,int loopDepth) 
	{
															#line 2227 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(false);
															#line 13818 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont11cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont1loopBody1cont11cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont1loopBody1cont11cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state > 0) static_cast<MaybeSplitRangeActor*>(this)->actor_wait_state = 0;
		static_cast<MaybeSplitRangeActor*>(this)->ActorCallback< MaybeSplitRangeActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< MaybeSplitRangeActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< MaybeSplitRangeActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), 11);

	}
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	UID currentWorkerId;
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	KeyRange granuleRange;
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	UID granuleID;
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Version granuleStartVersion;
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	bool writeHot;
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int64_t originalEpoch;
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int64_t originalSeqno;
															#line 2006 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 2010 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	BlobGranuleSplitPoints splitPoints;
															#line 2051 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	std::vector<UID> newGranuleIDs;
															#line 2070 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Version splitVersion;
															#line 2096 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	RangeResult existingState;
															#line 2139 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Key lockKey;
															#line 2186 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int i;
															#line 14015 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via maybeSplitRange()
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class MaybeSplitRangeActor final : public Actor<Void>, public ActorCallback< MaybeSplitRangeActor, 0, BlobGranuleSplitPoints >, public ActorCallback< MaybeSplitRangeActor, 1, Void >, public ActorCallback< MaybeSplitRangeActor, 2, ForcedPurgeState >, public ActorCallback< MaybeSplitRangeActor, 3, RangeResult >, public ActorCallback< MaybeSplitRangeActor, 4, RangeResult >, public ActorCallback< MaybeSplitRangeActor, 5, RangeResult >, public ActorCallback< MaybeSplitRangeActor, 6, Optional<Value> >, public ActorCallback< MaybeSplitRangeActor, 7, RangeResult >, public ActorCallback< MaybeSplitRangeActor, 8, Void >, public ActorCallback< MaybeSplitRangeActor, 9, Void >, public ActorCallback< MaybeSplitRangeActor, 10, Void >, public ActorCallback< MaybeSplitRangeActor, 11, Void >, public FastAllocated<MaybeSplitRangeActor>, public MaybeSplitRangeActorState<MaybeSplitRangeActor> {
															#line 14020 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<MaybeSplitRangeActor>::operator new;
	using FastAllocated<MaybeSplitRangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16753900763115192064UL, 7884941113235477760UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MaybeSplitRangeActor, 0, BlobGranuleSplitPoints >;
friend struct ActorCallback< MaybeSplitRangeActor, 1, Void >;
friend struct ActorCallback< MaybeSplitRangeActor, 2, ForcedPurgeState >;
friend struct ActorCallback< MaybeSplitRangeActor, 3, RangeResult >;
friend struct ActorCallback< MaybeSplitRangeActor, 4, RangeResult >;
friend struct ActorCallback< MaybeSplitRangeActor, 5, RangeResult >;
friend struct ActorCallback< MaybeSplitRangeActor, 6, Optional<Value> >;
friend struct ActorCallback< MaybeSplitRangeActor, 7, RangeResult >;
friend struct ActorCallback< MaybeSplitRangeActor, 8, Void >;
friend struct ActorCallback< MaybeSplitRangeActor, 9, Void >;
friend struct ActorCallback< MaybeSplitRangeActor, 10, Void >;
friend struct ActorCallback< MaybeSplitRangeActor, 11, Void >;
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	MaybeSplitRangeActor(Reference<BlobManagerData> const& bmData,UID const& currentWorkerId,KeyRange const& granuleRange,UID const& granuleID,Version const& granuleStartVersion,bool const& writeHot,int64_t const& originalEpoch,int64_t const& originalSeqno) 
															#line 14048 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<Void>(),
		   MaybeSplitRangeActorState<MaybeSplitRangeActor>(bmData, currentWorkerId, granuleRange, granuleID, granuleStartVersion, writeHot, originalEpoch, originalSeqno),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("maybeSplitRange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1978140127858620416UL, 5463554423642924800UL);
		ActorExecutionContextHelper __helper(static_cast<MaybeSplitRangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("maybeSplitRange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("maybeSplitRange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 0, BlobGranuleSplitPoints >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 2, ForcedPurgeState >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 3, RangeResult >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 4, RangeResult >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 5, RangeResult >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 6, Optional<Value> >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 7, RangeResult >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 10, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< MaybeSplitRangeActor, 11, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<Void> maybeSplitRange( Reference<BlobManagerData> const& bmData, UID const& currentWorkerId, KeyRange const& granuleRange, UID const& granuleID, Version const& granuleStartVersion, bool const& writeHot, int64_t const& originalEpoch, int64_t const& originalSeqno ) {
															#line 1994 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<Void>(new MaybeSplitRangeActor(bmData, currentWorkerId, granuleRange, granuleID, granuleStartVersion, writeHot, originalEpoch, originalSeqno));
															#line 14092 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 2301 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

// read mapping from db to handle any in flight granules or other issues
// Forces all granules in the specified key range to flush data to blob up to the specified version. This is required
// for executing a merge.
															#line 14100 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via forceGranuleFlush()
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class ForceGranuleFlushActor>
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class ForceGranuleFlushActorState {
															#line 14107 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	ForceGranuleFlushActorState(Reference<BlobManagerData> const& bmData,UID const& mergeGranuleID,KeyRange const& keyRange,Version const& version) 
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   mergeGranuleID(mergeGranuleID),
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   keyRange(keyRange),
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   version(version),
															#line 2309 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   tr(bmData->db),
															#line 2310 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   beginKey(keyRange.begin),
															#line 2311 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   endKey(keyRange.end)
															#line 14126 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("forceGranuleFlush", reinterpret_cast<unsigned long>(this));

	}
	~ForceGranuleFlushActorState() 
	{
		fdb_probe_actor_destroy("forceGranuleFlush", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2312 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			int64_t epoch = bmData->epoch;
															#line 2313 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			Version v = version;
															#line 2314 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			req = FlushGranuleRequest(epoch, keyRange, v, false);
															#line 2316 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (BM_DEBUG)
															#line 14147 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2317 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				fmt::print("Flushing Granules [{0} - {1}) @ {2}\n", beginKey.printable(), endKey.printable(), version);
															#line 14151 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
															#line 2320 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			;
															#line 14155 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ForceGranuleFlushActorState();
		static_cast<ForceGranuleFlushActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2351 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 14178 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2353 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Flushing Granules [{0} - {1}) @ {2} Complete!\n", keyRange.begin.printable(), keyRange.end.printable(), version);
															#line 14182 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2359 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<ForceGranuleFlushActor*>(this)->SAV<bool>::futures) { (void)(true); this->~ForceGranuleFlushActorState(); static_cast<ForceGranuleFlushActor*>(this)->destroy(); return 0; }
															#line 14186 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<ForceGranuleFlushActor*>(this)->SAV< bool >::value()) bool(true);
		this->~ForceGranuleFlushActorState();
		static_cast<ForceGranuleFlushActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2321 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (beginKey >= endKey)
															#line 14205 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
		try {
															#line 2325 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2326 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr.setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2327 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2328 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<ForcedPurgeState> __when_expr_0 = getForcePurgedState(&tr, keyRange);
															#line 2328 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14220 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 1;
															#line 2328 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ForceGranuleFlushActor, 0, ForcedPurgeState >*>(static_cast<ForceGranuleFlushActor*>(this)));
															#line 14225 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2347 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 2347 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 14262 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 5;
															#line 2347 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ForceGranuleFlushActor, 4, Void >*>(static_cast<ForceGranuleFlushActor*>(this)));
															#line 14267 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(ForcedPurgeState const& purgeState,int loopDepth) 
	{
															#line 2329 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (purgeState != ForcedPurgeState::NonePurged)
															#line 14282 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2330 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "Granule flush stopped because of force purge", probe::decoration::rare);
															#line 2331 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("GranuleFlushCancelledForcePurge", bmData->id) .detail("Epoch", bmData->epoch) .detail("KeyRange", keyRange);
															#line 2336 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_1 = updateChangeFeed( &tr, granuleIDToCFKey(mergeGranuleID), ChangeFeedStatus::CHANGE_FEED_DESTROY, keyRange);
															#line 2336 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14292 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
			static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 2;
															#line 2336 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ForceGranuleFlushActor, 1, Void >*>(static_cast<ForceGranuleFlushActor*>(this)));
															#line 14297 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(ForcedPurgeState && purgeState,int loopDepth) 
	{
															#line 2329 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (purgeState != ForcedPurgeState::NonePurged)
															#line 14311 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2330 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "Granule flush stopped because of force purge", probe::decoration::rare);
															#line 2331 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("GranuleFlushCancelledForcePurge", bmData->id) .detail("Epoch", bmData->epoch) .detail("KeyRange", keyRange);
															#line 2336 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_1 = updateChangeFeed( &tr, granuleIDToCFKey(mergeGranuleID), ChangeFeedStatus::CHANGE_FEED_DESTROY, keyRange);
															#line 2336 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14321 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
			static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 2;
															#line 2336 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ForceGranuleFlushActor, 1, Void >*>(static_cast<ForceGranuleFlushActor*>(this)));
															#line 14326 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(ForcedPurgeState const& purgeState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(purgeState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(ForcedPurgeState && purgeState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(purgeState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state > 0) static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 0;
		static_cast<ForceGranuleFlushActor*>(this)->ActorCallback< ForceGranuleFlushActor, 0, ForcedPurgeState >::remove();

	}
	void a_callback_fire(ActorCallback< ForceGranuleFlushActor, 0, ForcedPurgeState >*,ForcedPurgeState const& value) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ForceGranuleFlushActor, 0, ForcedPurgeState >*,ForcedPurgeState && value) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ForceGranuleFlushActor, 0, ForcedPurgeState >*,Error err) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 2343 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_3 = success( txnDoBlobGranuleRequests(&tr, &beginKey, endKey, req, &BlobWorkerInterface::flushGranuleRequest));
															#line 2343 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14417 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 4;
															#line 2343 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ForceGranuleFlushActor, 3, Void >*>(static_cast<ForceGranuleFlushActor*>(this)));
															#line 14422 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 2339 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 2339 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14433 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_2.get(), loopDepth); };
		static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 3;
															#line 2339 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ForceGranuleFlushActor, 2, Void >*>(static_cast<ForceGranuleFlushActor*>(this)));
															#line 14438 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 2339 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_2 = tr.commit();
															#line 2339 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_2.get(), loopDepth); };
		static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 3;
															#line 2339 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ForceGranuleFlushActor, 2, Void >*>(static_cast<ForceGranuleFlushActor*>(this)));
															#line 14454 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state > 0) static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 0;
		static_cast<ForceGranuleFlushActor*>(this)->ActorCallback< ForceGranuleFlushActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ForceGranuleFlushActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ForceGranuleFlushActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ForceGranuleFlushActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2340 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<ForceGranuleFlushActor*>(this)->SAV<bool>::futures) { (void)(false); this->~ForceGranuleFlushActorState(); static_cast<ForceGranuleFlushActor*>(this)->destroy(); return 0; }
															#line 14538 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<ForceGranuleFlushActor*>(this)->SAV< bool >::value()) bool(false);
		this->~ForceGranuleFlushActorState();
		static_cast<ForceGranuleFlushActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2340 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<ForceGranuleFlushActor*>(this)->SAV<bool>::futures) { (void)(false); this->~ForceGranuleFlushActorState(); static_cast<ForceGranuleFlushActor*>(this)->destroy(); return 0; }
															#line 14550 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<ForceGranuleFlushActor*>(this)->SAV< bool >::value()) bool(false);
		this->~ForceGranuleFlushActorState();
		static_cast<ForceGranuleFlushActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state > 0) static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 0;
		static_cast<ForceGranuleFlushActor*>(this)->ActorCallback< ForceGranuleFlushActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ForceGranuleFlushActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ForceGranuleFlushActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ForceGranuleFlushActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state > 0) static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 0;
		static_cast<ForceGranuleFlushActor*>(this)->ActorCallback< ForceGranuleFlushActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ForceGranuleFlushActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ForceGranuleFlushActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ForceGranuleFlushActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state > 0) static_cast<ForceGranuleFlushActor*>(this)->actor_wait_state = 0;
		static_cast<ForceGranuleFlushActor*>(this)->ActorCallback< ForceGranuleFlushActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ForceGranuleFlushActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ForceGranuleFlushActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ForceGranuleFlushActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	UID mergeGranuleID;
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	KeyRange keyRange;
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Version version;
															#line 2309 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Transaction tr;
															#line 2310 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Key beginKey;
															#line 2311 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Key endKey;
															#line 2314 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	FlushGranuleRequest req;
															#line 14836 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via forceGranuleFlush()
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class ForceGranuleFlushActor final : public Actor<bool>, public ActorCallback< ForceGranuleFlushActor, 0, ForcedPurgeState >, public ActorCallback< ForceGranuleFlushActor, 1, Void >, public ActorCallback< ForceGranuleFlushActor, 2, Void >, public ActorCallback< ForceGranuleFlushActor, 3, Void >, public ActorCallback< ForceGranuleFlushActor, 4, Void >, public FastAllocated<ForceGranuleFlushActor>, public ForceGranuleFlushActorState<ForceGranuleFlushActor> {
															#line 14841 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<ForceGranuleFlushActor>::operator new;
	using FastAllocated<ForceGranuleFlushActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8754682828950778112UL, 786995825675435520UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ForceGranuleFlushActor, 0, ForcedPurgeState >;
friend struct ActorCallback< ForceGranuleFlushActor, 1, Void >;
friend struct ActorCallback< ForceGranuleFlushActor, 2, Void >;
friend struct ActorCallback< ForceGranuleFlushActor, 3, Void >;
friend struct ActorCallback< ForceGranuleFlushActor, 4, Void >;
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	ForceGranuleFlushActor(Reference<BlobManagerData> const& bmData,UID const& mergeGranuleID,KeyRange const& keyRange,Version const& version) 
															#line 14862 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<bool>(),
		   ForceGranuleFlushActorState<ForceGranuleFlushActor>(bmData, mergeGranuleID, keyRange, version),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("forceGranuleFlush", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10954189611990226432UL, 17717251051683356672UL);
		ActorExecutionContextHelper __helper(static_cast<ForceGranuleFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("forceGranuleFlush");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("forceGranuleFlush", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ForceGranuleFlushActor, 0, ForcedPurgeState >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ForceGranuleFlushActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ForceGranuleFlushActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ForceGranuleFlushActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ForceGranuleFlushActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<bool> forceGranuleFlush( Reference<BlobManagerData> const& bmData, UID const& mergeGranuleID, KeyRange const& keyRange, Version const& version ) {
															#line 2305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<bool>(new ForceGranuleFlushActor(bmData, mergeGranuleID, keyRange, version));
															#line 14899 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 2361 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

// Persist the merge intent for this merge in the database. Once this transaction commits, the merge is in progress. It
// cannot be aborted, and must be completed.
															#line 14906 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via persistMergeGranulesStart()
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class PersistMergeGranulesStartActor>
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class PersistMergeGranulesStartActorState {
															#line 14913 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	PersistMergeGranulesStartActorState(Reference<BlobManagerData> const& bmData,KeyRange const& mergeRange,std::vector<UID> const& parentGranuleIDs,std::vector<Key> const& parentGranuleRanges,std::vector<Version> const& parentGranuleStartVersions) 
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   mergeRange(mergeRange),
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   parentGranuleIDs(parentGranuleIDs),
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   parentGranuleRanges(parentGranuleRanges),
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   parentGranuleStartVersions(parentGranuleStartVersions),
															#line 2369 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   mergeGranuleID(deterministicRandom()->randomUniqueID()),
															#line 2370 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   tr(makeReference<ReadYourWritesTransaction>(bmData->db))
															#line 14932 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this));

	}
	~PersistMergeGranulesStartActorState() 
	{
		fdb_probe_actor_destroy("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2371 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			;
															#line 14947 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistMergeGranulesStartActorState();
		static_cast<PersistMergeGranulesStartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 2373 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2374 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2375 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2377 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_0 = checkManagerLock(tr, bmData);
															#line 2377 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 14986 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 1;
															#line 2377 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 0, Void >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 14991 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2442 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_7 = tr->onError(e);
															#line 2442 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 15015 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_7.get(), loopDepth); };
			static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 8;
															#line 2442 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 7, Void >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15020 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 2379 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<ForcedPurgeState> __when_expr_1 = getForcePurgedState(&tr->getTransaction(), mergeRange);
															#line 2379 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15037 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 2;
															#line 2379 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 1, ForcedPurgeState >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15042 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 2379 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<ForcedPurgeState> __when_expr_1 = getForcePurgedState(&tr->getTransaction(), mergeRange);
															#line 2379 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15053 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 2;
															#line 2379 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 1, ForcedPurgeState >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15058 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesStartActor*>(this)->ActorCallback< PersistMergeGranulesStartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesStartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(ForcedPurgeState const& purgeState,int loopDepth) 
	{
															#line 2380 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (purgeState != ForcedPurgeState::NonePurged)
															#line 15142 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "Merge start stopped because of force purge", probe::decoration::rare);
															#line 2382 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("GranuleMergeStartCancelledForcePurge", bmData->id) .detail("Epoch", bmData->epoch) .detail("GranuleRange", mergeRange);
															#line 2387 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_2 = updateChangeFeed(&tr->getTransaction(), granuleIDToCFKey(mergeGranuleID), ChangeFeedStatus::CHANGE_FEED_DESTROY, mergeRange);
															#line 2387 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15152 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 3;
															#line 2387 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 2, Void >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15157 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(ForcedPurgeState && purgeState,int loopDepth) 
	{
															#line 2380 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (purgeState != ForcedPurgeState::NonePurged)
															#line 15171 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2381 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "Merge start stopped because of force purge", probe::decoration::rare);
															#line 2382 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("GranuleMergeStartCancelledForcePurge", bmData->id) .detail("Epoch", bmData->epoch) .detail("GranuleRange", mergeRange);
															#line 2387 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_2 = updateChangeFeed(&tr->getTransaction(), granuleIDToCFKey(mergeGranuleID), ChangeFeedStatus::CHANGE_FEED_DESTROY, mergeRange);
															#line 2387 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15181 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 3;
															#line 2387 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 2, Void >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15186 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(ForcedPurgeState const& purgeState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(purgeState, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(ForcedPurgeState && purgeState,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(purgeState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesStartActor*>(this)->ActorCallback< PersistMergeGranulesStartActor, 1, ForcedPurgeState >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 1, ForcedPurgeState >*,ForcedPurgeState const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 1, ForcedPurgeState >*,ForcedPurgeState && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesStartActor, 1, ForcedPurgeState >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 2403 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->atomicOp( blobGranuleMergeKeyFor(mergeGranuleID), blobGranuleMergeValueFor(mergeRange, parentGranuleIDs, parentGranuleRanges, parentGranuleStartVersions), MutationRef::SetVersionstampedValue);
															#line 2408 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_4 = updateChangeFeed( tr, granuleIDToCFKey(mergeGranuleID), ChangeFeedStatus::CHANGE_FEED_CREATE, mergeRange);
															#line 2408 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15279 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 5;
															#line 2408 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 4, Void >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15284 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 2392 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr->commit();
															#line 2392 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15295 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 4;
															#line 2392 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 3, Void >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15300 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 2392 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr->commit();
															#line 2392 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15311 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 4;
															#line 2392 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 3, Void >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15316 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesStartActor*>(this)->ActorCallback< PersistMergeGranulesStartActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesStartActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2394 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->activeGranuleMerges.insert(mergeRange, invalidVersion);
															#line 2395 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->activeGranuleMerges.coalesce(mergeRange.begin);
															#line 2398 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<PersistMergeGranulesStartActor*>(this)->SAV<std::pair<UID, Version>>::futures) { (void)(std::pair(UID(), invalidVersion)); this->~PersistMergeGranulesStartActorState(); static_cast<PersistMergeGranulesStartActor*>(this)->destroy(); return 0; }
															#line 15404 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<PersistMergeGranulesStartActor*>(this)->SAV< std::pair<UID, Version> >::value()) std::pair<UID, Version>(std::pair(UID(), invalidVersion));
		this->~PersistMergeGranulesStartActorState();
		static_cast<PersistMergeGranulesStartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2394 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->activeGranuleMerges.insert(mergeRange, invalidVersion);
															#line 2395 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->activeGranuleMerges.coalesce(mergeRange.begin);
															#line 2398 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<PersistMergeGranulesStartActor*>(this)->SAV<std::pair<UID, Version>>::futures) { (void)(std::pair(UID(), invalidVersion)); this->~PersistMergeGranulesStartActorState(); static_cast<PersistMergeGranulesStartActor*>(this)->destroy(); return 0; }
															#line 15420 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<PersistMergeGranulesStartActor*>(this)->SAV< std::pair<UID, Version> >::value()) std::pair<UID, Version>(std::pair(UID(), invalidVersion));
		this->~PersistMergeGranulesStartActorState();
		static_cast<PersistMergeGranulesStartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesStartActor*>(this)->ActorCallback< PersistMergeGranulesStartActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesStartActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
															#line 2411 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_5 = tr->commit();
															#line 2411 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15509 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 6;
															#line 2411 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 5, Void >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15514 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
															#line 2411 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_5 = tr->commit();
															#line 2411 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15525 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont8when1(__when_expr_5.get(), loopDepth); };
		static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 6;
															#line 2411 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 5, Void >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15530 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesStartActor*>(this)->ActorCallback< PersistMergeGranulesStartActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesStartActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
															#line 2413 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BUGGIFY && bmData->maybeInjectTargetedRestart())
															#line 15614 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2414 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_6 = delay(0);
															#line 2414 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15620 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_6.get(), loopDepth); };
			static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 7;
															#line 2414 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 6, Void >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15625 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
															#line 2413 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BUGGIFY && bmData->maybeInjectTargetedRestart())
															#line 15639 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2414 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_6 = delay(0);
															#line 2414 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 15645 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_6.get(), loopDepth); };
			static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 7;
															#line 2414 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesStartActor, 6, Void >*>(static_cast<PersistMergeGranulesStartActor*>(this)));
															#line 15650 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesStartActor*>(this)->ActorCallback< PersistMergeGranulesStartActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesStartActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1cont10(int loopDepth) 
	{
															#line 2418 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		Version mergeVersion = tr->getCommittedVersion();
															#line 2419 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 15741 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2420 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Granule merge intent persisted [{0} - {1}): {2} @ {3}!\n", mergeRange.begin.printable(), mergeRange.end.printable(), mergeGranuleID.shortString().substr(0, 6), mergeVersion);
															#line 15745 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2428 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		auto mergeInProgress = bmData->activeGranuleMerges.rangeContaining(mergeRange.begin);
															#line 2429 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 15751 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2430 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Updating merge in progress [{0} - {1}) to merge version {2}!\n", mergeInProgress.begin().printable(), mergeInProgress.end().printable(), mergeVersion);
															#line 15755 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2435 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(mergeInProgress.begin() == mergeRange.begin);
															#line 2436 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(mergeInProgress.end() == mergeRange.end);
															#line 2437 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(mergeInProgress.cvalue() == 0);
															#line 2438 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		mergeInProgress.value() = mergeVersion;
															#line 2440 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<PersistMergeGranulesStartActor*>(this)->SAV<std::pair<UID, Version>>::futures) { (void)(std::pair(mergeGranuleID, mergeVersion)); this->~PersistMergeGranulesStartActorState(); static_cast<PersistMergeGranulesStartActor*>(this)->destroy(); return 0; }
															#line 15767 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<PersistMergeGranulesStartActor*>(this)->SAV< std::pair<UID, Version> >::value()) std::pair<UID, Version>(std::pair(mergeGranuleID, mergeVersion));
		this->~PersistMergeGranulesStartActorState();
		static_cast<PersistMergeGranulesStartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void const& _,int loopDepth) 
	{
															#line 2415 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(false);
															#line 15779 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void && _,int loopDepth) 
	{
															#line 2415 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(false);
															#line 15788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesStartActor*>(this)->ActorCallback< PersistMergeGranulesStartActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesStartActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesStartActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesStartActor*>(this)->ActorCallback< PersistMergeGranulesStartActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesStartActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesStartActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), 7);

	}
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	KeyRange mergeRange;
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	std::vector<UID> parentGranuleIDs;
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	std::vector<Key> parentGranuleRanges;
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	std::vector<Version> parentGranuleStartVersions;
															#line 2369 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	UID mergeGranuleID;
															#line 2370 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 15969 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via persistMergeGranulesStart()
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class PersistMergeGranulesStartActor final : public Actor<std::pair<UID, Version>>, public ActorCallback< PersistMergeGranulesStartActor, 0, Void >, public ActorCallback< PersistMergeGranulesStartActor, 1, ForcedPurgeState >, public ActorCallback< PersistMergeGranulesStartActor, 2, Void >, public ActorCallback< PersistMergeGranulesStartActor, 3, Void >, public ActorCallback< PersistMergeGranulesStartActor, 4, Void >, public ActorCallback< PersistMergeGranulesStartActor, 5, Void >, public ActorCallback< PersistMergeGranulesStartActor, 6, Void >, public ActorCallback< PersistMergeGranulesStartActor, 7, Void >, public FastAllocated<PersistMergeGranulesStartActor>, public PersistMergeGranulesStartActorState<PersistMergeGranulesStartActor> {
															#line 15974 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<PersistMergeGranulesStartActor>::operator new;
	using FastAllocated<PersistMergeGranulesStartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14041819710844732672UL, 7020435590575456768UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::pair<UID, Version>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistMergeGranulesStartActor, 0, Void >;
friend struct ActorCallback< PersistMergeGranulesStartActor, 1, ForcedPurgeState >;
friend struct ActorCallback< PersistMergeGranulesStartActor, 2, Void >;
friend struct ActorCallback< PersistMergeGranulesStartActor, 3, Void >;
friend struct ActorCallback< PersistMergeGranulesStartActor, 4, Void >;
friend struct ActorCallback< PersistMergeGranulesStartActor, 5, Void >;
friend struct ActorCallback< PersistMergeGranulesStartActor, 6, Void >;
friend struct ActorCallback< PersistMergeGranulesStartActor, 7, Void >;
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	PersistMergeGranulesStartActor(Reference<BlobManagerData> const& bmData,KeyRange const& mergeRange,std::vector<UID> const& parentGranuleIDs,std::vector<Key> const& parentGranuleRanges,std::vector<Version> const& parentGranuleStartVersions) 
															#line 15998 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<std::pair<UID, Version>>(),
		   PersistMergeGranulesStartActorState<PersistMergeGranulesStartActor>(bmData, mergeRange, parentGranuleIDs, parentGranuleRanges, parentGranuleStartVersions),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15308229457407910144UL, 4194998597180649728UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistMergeGranulesStart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistMergeGranulesStart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistMergeGranulesStartActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistMergeGranulesStartActor, 1, ForcedPurgeState >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PersistMergeGranulesStartActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< PersistMergeGranulesStartActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< PersistMergeGranulesStartActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< PersistMergeGranulesStartActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< PersistMergeGranulesStartActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< PersistMergeGranulesStartActor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<std::pair<UID, Version>> persistMergeGranulesStart( Reference<BlobManagerData> const& bmData, KeyRange const& mergeRange, std::vector<UID> const& parentGranuleIDs, std::vector<Key> const& parentGranuleRanges, std::vector<Version> const& parentGranuleStartVersions ) {
															#line 2364 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<std::pair<UID, Version>>(new PersistMergeGranulesStartActor(bmData, mergeRange, parentGranuleIDs, parentGranuleRanges, parentGranuleStartVersions));
															#line 16038 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 2446 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

// Persists the merge being complete in the database by clearing the merge intent. Once this transaction commits, the
// merge is considered completed.
															#line 16045 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via persistMergeGranulesDone()
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class PersistMergeGranulesDoneActor>
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class PersistMergeGranulesDoneActorState {
															#line 16052 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	PersistMergeGranulesDoneActorState(Reference<BlobManagerData> const& bmData,UID const& mergeGranuleID,KeyRange const& mergeRange,Version const& mergeVersion,std::vector<UID> const& parentGranuleIDs,std::vector<Key> const& parentGranuleRanges,std::vector<Version> const& parentGranuleStartVersions) 
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   mergeGranuleID(mergeGranuleID),
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   mergeRange(mergeRange),
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   mergeVersion(mergeVersion),
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   parentGranuleIDs(parentGranuleIDs),
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   parentGranuleRanges(parentGranuleRanges),
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   parentGranuleStartVersions(parentGranuleStartVersions),
															#line 2456 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   tr(makeReference<ReadYourWritesTransaction>(bmData->db)),
															#line 2458 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   tmpWorkerId()
															#line 16075 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this));

	}
	~PersistMergeGranulesDoneActorState() 
	{
		fdb_probe_actor_destroy("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2459 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			auto ranges = bmData->workerAssignments.intersectingRanges(mergeRange);
															#line 2460 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for( auto& it : ranges ) {
															#line 2461 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				if (it.cvalue() != UID())
															#line 16094 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				{
															#line 2462 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
					tmpWorkerId = it.cvalue();
															#line 16098 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
					break;
				}
			}
															#line 2466 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (tmpWorkerId == UID())
															#line 16104 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			{
															#line 2467 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				CODE_PROBE(true, "All workers dead right now", probe::decoration::rare);
															#line 2468 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				;
															#line 16110 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistMergeGranulesDoneActorState();
		static_cast<PersistMergeGranulesDoneActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2473 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		;
															#line 16138 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
															#line 2471 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tmpWorkerId = bmData->workersById.begin()->first;
															#line 16147 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2468 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!(bmData->workersById.empty()))
															#line 16163 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2469 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_0 = bmData->recruitingStream.onChange() || bmData->foundBlobWorkers.getFuture();
															#line 2469 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16171 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 1;
															#line 2469 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 0, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16176 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesDoneActor*>(this)->ActorCallback< PersistMergeGranulesDoneActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesDoneActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 2475 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2476 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 2477 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 2479 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_1 = checkManagerLock(tr, bmData);
															#line 2479 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16301 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 2;
															#line 2479 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 1, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16306 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2585 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_10 = tr->onError(e);
															#line 2585 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16330 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch1(__when_expr_10.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_10.get(), loopDepth); };
			static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 11;
															#line 2585 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 10, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16335 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 2481 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<ForcedPurgeState> __when_expr_2 = getForcePurgedState(&tr->getTransaction(), mergeRange);
															#line 2481 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16352 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 3;
															#line 2481 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 2, ForcedPurgeState >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16357 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 2481 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<ForcedPurgeState> __when_expr_2 = getForcePurgedState(&tr->getTransaction(), mergeRange);
															#line 2481 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16368 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 3;
															#line 2481 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 2, ForcedPurgeState >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16373 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesDoneActor*>(this)->ActorCallback< PersistMergeGranulesDoneActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesDoneActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont3(ForcedPurgeState const& purgeState,int loopDepth) 
	{
															#line 2482 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (purgeState != ForcedPurgeState::NonePurged)
															#line 16457 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2483 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "Merge finish stopped because of force purge", probe::decoration::rare);
															#line 2484 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("GranuleMergeCancelledForcePurge", bmData->id) .detail("Epoch", bmData->epoch) .detail("GranuleRange", mergeRange);
															#line 2489 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_3 = updateChangeFeed(&tr->getTransaction(), granuleIDToCFKey(mergeGranuleID), ChangeFeedStatus::CHANGE_FEED_DESTROY, mergeRange);
															#line 2489 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16467 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 4;
															#line 2489 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 3, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16472 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(ForcedPurgeState && purgeState,int loopDepth) 
	{
															#line 2482 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (purgeState != ForcedPurgeState::NonePurged)
															#line 16486 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2483 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			CODE_PROBE(true, "Merge finish stopped because of force purge", probe::decoration::rare);
															#line 2484 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			TraceEvent("GranuleMergeCancelledForcePurge", bmData->id) .detail("Epoch", bmData->epoch) .detail("GranuleRange", mergeRange);
															#line 2489 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_3 = updateChangeFeed(&tr->getTransaction(), granuleIDToCFKey(mergeGranuleID), ChangeFeedStatus::CHANGE_FEED_DESTROY, mergeRange);
															#line 2489 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16496 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 4;
															#line 2489 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 3, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16501 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(ForcedPurgeState const& purgeState,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(purgeState, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(ForcedPurgeState && purgeState,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(std::move(purgeState), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesDoneActor*>(this)->ActorCallback< PersistMergeGranulesDoneActor, 2, ForcedPurgeState >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 2, ForcedPurgeState >*,ForcedPurgeState const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 2, ForcedPurgeState >*,ForcedPurgeState && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesDoneActor, 2, ForcedPurgeState >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont4(int loopDepth) 
	{
															#line 2504 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->clear(blobGranuleMergeKeyFor(mergeGranuleID));
															#line 2506 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		parentIdx = int();
															#line 2508 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		parentIdx = 0;
															#line 16594 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont4loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 2496 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_4 = tr->commit();
															#line 2496 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16605 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 5;
															#line 2496 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 4, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16610 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 2496 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_4 = tr->commit();
															#line 2496 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16621 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1loopBody1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 5;
															#line 2496 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 4, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16626 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesDoneActor*>(this)->ActorCallback< PersistMergeGranulesDoneActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesDoneActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 2498 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->activeGranuleMerges.insert(mergeRange, invalidVersion);
															#line 2499 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->activeGranuleMerges.coalesce(mergeRange.begin);
															#line 2501 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<PersistMergeGranulesDoneActor*>(this)->SAV<bool>::futures) { (void)(false); this->~PersistMergeGranulesDoneActorState(); static_cast<PersistMergeGranulesDoneActor*>(this)->destroy(); return 0; }
															#line 16714 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<PersistMergeGranulesDoneActor*>(this)->SAV< bool >::value()) bool(false);
		this->~PersistMergeGranulesDoneActorState();
		static_cast<PersistMergeGranulesDoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 2498 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->activeGranuleMerges.insert(mergeRange, invalidVersion);
															#line 2499 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		bmData->activeGranuleMerges.coalesce(mergeRange.begin);
															#line 2501 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<PersistMergeGranulesDoneActor*>(this)->SAV<bool>::futures) { (void)(false); this->~PersistMergeGranulesDoneActorState(); static_cast<PersistMergeGranulesDoneActor*>(this)->destroy(); return 0; }
															#line 16730 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<PersistMergeGranulesDoneActor*>(this)->SAV< bool >::value()) bool(false);
		this->~PersistMergeGranulesDoneActorState();
		static_cast<PersistMergeGranulesDoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesDoneActor*>(this)->ActorCallback< PersistMergeGranulesDoneActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesDoneActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont8(int loopDepth) 
	{
															#line 2540 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->clear(KeyRangeRef(keyAfter(blobGranuleMergeBoundaryKeyFor(mergeRange.begin)), blobGranuleMergeBoundaryKeyFor(mergeRange.end)));
															#line 2544 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		int64_t seqNo = bmData->seqNo++;
															#line 2545 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->set(blobGranuleLockKeyFor(mergeRange), blobGranuleLockValueFor(bmData->epoch, seqNo, mergeGranuleID));
															#line 2548 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		Key historyKey = blobGranuleHistoryKeyFor(mergeRange, mergeVersion);
															#line 2550 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		Standalone<BlobGranuleHistoryValue> historyValue;
															#line 2551 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		historyValue.granuleID = mergeGranuleID;
															#line 2552 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(parentIdx = 0;parentIdx < parentGranuleIDs.size();parentIdx++) {
															#line 2553 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			historyValue.parentBoundaries.push_back(historyValue.arena(), parentGranuleRanges[parentIdx]);
															#line 2554 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			historyValue.parentVersions.push_back(historyValue.arena(), parentGranuleStartVersions[parentIdx]);
															#line 16833 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2556 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		historyValue.parentBoundaries.push_back(historyValue.arena(), parentGranuleRanges.back());
															#line 2558 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->set(historyKey, blobGranuleHistoryValueFor(historyValue));
															#line 2560 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_7 = krmSetRange(tr, blobGranuleMappingKeys.begin, mergeRange, blobGranuleMappingValueFor(tmpWorkerId));
															#line 2560 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 16843 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont1loopBody1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont1loopBody1cont8when1(__when_expr_7.get(), loopDepth); };
		static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 8;
															#line 2560 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 7, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16848 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4loopBody1(int loopDepth) 
	{
															#line 2508 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!(parentIdx < parentGranuleIDs.size()))
															#line 16864 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
			return a_body1cont1loopBody1cont4break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2509 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		KeyRange parentRange(KeyRangeRef(parentGranuleRanges[parentIdx], parentGranuleRanges[parentIdx + 1]));
															#line 2510 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		lockKey = blobGranuleLockKeyFor(parentRange);
															#line 2511 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		oldLockFuture = tr->get(lockKey);
															#line 2514 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->clear(blobGranuleMergeBoundaryKeyFor(parentRange.begin));
															#line 2518 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_5 = updateChangeFeed(&tr->getTransaction(), granuleIDToCFKey(parentGranuleIDs[parentIdx]), ChangeFeedStatus::CHANGE_FEED_DESTROY, parentRange);
															#line 2518 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16880 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1loopBody1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont4loopBody1when1(__when_expr_5.get(), loopDepth); };
		static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 6;
															#line 2518 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 5, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16885 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4break1(int loopDepth) 
	{
		try {
			return a_body1cont1loopBody1cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 2522 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 16907 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2523 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Granule merge destroying CF {0} ({1})!\n", parentGranuleIDs[parentIdx].shortString().substr(0, 6), granuleIDToCFKey(parentGranuleIDs[parentIdx]).printable());
															#line 16911 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2528 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_6 = oldLockFuture;
															#line 2528 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16917 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1loopBody1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont4loopBody1cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 7;
															#line 2528 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 6, Optional<Value> >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16922 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 2522 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 16931 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2523 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Granule merge destroying CF {0} ({1})!\n", parentGranuleIDs[parentIdx].shortString().substr(0, 6), granuleIDToCFKey(parentGranuleIDs[parentIdx]).printable());
															#line 16935 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2528 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_6 = oldLockFuture;
															#line 2528 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 16941 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont1loopBody1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont4loopBody1cont1when1(__when_expr_6.get(), loopDepth); };
		static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 7;
															#line 2528 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 6, Optional<Value> >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 16946 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesDoneActor*>(this)->ActorCallback< PersistMergeGranulesDoneActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesDoneActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont1loopBody1cont4loopBody1cont3(Optional<Value> const& oldLock,int loopDepth) 
	{
															#line 2529 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(oldLock.present());
															#line 2530 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		auto prevLock = decodeBlobGranuleLockValue(oldLock.get());
															#line 2535 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->set( lockKey, blobGranuleLockValueFor(bmData->epoch, std::numeric_limits<int64_t>::max(), std::get<2>(prevLock)));
															#line 2508 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		parentIdx++;
															#line 17036 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopBody1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4loopBody1cont3(Optional<Value> && oldLock,int loopDepth) 
	{
															#line 2529 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(oldLock.present());
															#line 2530 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		auto prevLock = decodeBlobGranuleLockValue(oldLock.get());
															#line 2535 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		tr->set( lockKey, blobGranuleLockValueFor(bmData->epoch, std::numeric_limits<int64_t>::max(), std::get<2>(prevLock)));
															#line 2508 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		parentIdx++;
															#line 17051 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopBody1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4loopBody1cont1when1(Optional<Value> const& oldLock,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4loopBody1cont3(oldLock, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4loopBody1cont1when1(Optional<Value> && oldLock,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4loopBody1cont3(std::move(oldLock), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesDoneActor*>(this)->ActorCallback< PersistMergeGranulesDoneActor, 6, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 6, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont4loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 6, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1cont4loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesDoneActor, 6, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont1loopBody1cont9(Void const& _,int loopDepth) 
	{
															#line 2561 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_8 = tr->commit();
															#line 2561 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17137 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont1loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont1loopBody1cont9when1(__when_expr_8.get(), loopDepth); };
		static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 9;
															#line 2561 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 8, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 17142 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9(Void && _,int loopDepth) 
	{
															#line 2561 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_8 = tr->commit();
															#line 2561 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17153 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont1loopBody1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont1loopBody1cont9when1(__when_expr_8.get(), loopDepth); };
		static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 9;
															#line 2561 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 8, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 17158 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesDoneActor*>(this)->ActorCallback< PersistMergeGranulesDoneActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesDoneActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont1loopBody1cont11(Void const& _,int loopDepth) 
	{
															#line 2564 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(parentIdx = 1;parentIdx < parentGranuleIDs.size();parentIdx++) {
															#line 2565 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->mergeBoundaries.erase(parentGranuleRanges[parentIdx]);
															#line 17244 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2568 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 17248 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2569 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Granule merge intent cleared [{0} - {1}): {2} @ {3} (cv={4})\n", mergeRange.begin.printable(), mergeRange.end.printable(), mergeGranuleID.shortString().substr(0, 6), mergeVersion, tr->getCommittedVersion());
															#line 17252 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2576 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		CODE_PROBE(true, "Granule merge complete");
															#line 2578 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BUGGIFY && bmData->maybeInjectTargetedRestart())
															#line 17258 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2579 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_9 = delay(0);
															#line 2579 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17264 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont1loopBody1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont1loopBody1cont11when1(__when_expr_9.get(), loopDepth); };
			static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 10;
															#line 2579 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 9, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 17269 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont11cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11(Void && _,int loopDepth) 
	{
															#line 2564 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		for(parentIdx = 1;parentIdx < parentGranuleIDs.size();parentIdx++) {
															#line 2565 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->mergeBoundaries.erase(parentGranuleRanges[parentIdx]);
															#line 17285 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2568 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BM_DEBUG)
															#line 17289 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2569 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			fmt::print("Granule merge intent cleared [{0} - {1}): {2} @ {3} (cv={4})\n", mergeRange.begin.printable(), mergeRange.end.printable(), mergeGranuleID.shortString().substr(0, 6), mergeVersion, tr->getCommittedVersion());
															#line 17293 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		}
															#line 2576 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		CODE_PROBE(true, "Granule merge complete");
															#line 2578 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (BUGGIFY && bmData->maybeInjectTargetedRestart())
															#line 17299 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2579 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_9 = delay(0);
															#line 2579 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 17305 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont1loopBody1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont1loopBody1cont11when1(__when_expr_9.get(), loopDepth); };
			static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 10;
															#line 2579 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< PersistMergeGranulesDoneActor, 9, Void >*>(static_cast<PersistMergeGranulesDoneActor*>(this)));
															#line 17310 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont11cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesDoneActor*>(this)->ActorCallback< PersistMergeGranulesDoneActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesDoneActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont1loopBody1cont11cont1(int loopDepth) 
	{
															#line 2583 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!static_cast<PersistMergeGranulesDoneActor*>(this)->SAV<bool>::futures) { (void)(true); this->~PersistMergeGranulesDoneActorState(); static_cast<PersistMergeGranulesDoneActor*>(this)->destroy(); return 0; }
															#line 17399 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		new (&static_cast<PersistMergeGranulesDoneActor*>(this)->SAV< bool >::value()) bool(true);
		this->~PersistMergeGranulesDoneActorState();
		static_cast<PersistMergeGranulesDoneActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont4(Void const& _,int loopDepth) 
	{
															#line 2580 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(false);
															#line 17411 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11cont4(Void && _,int loopDepth) 
	{
															#line 2580 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		ASSERT(false);
															#line 17420 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont11cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesDoneActor*>(this)->ActorCallback< PersistMergeGranulesDoneActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1loopBody1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1loopBody1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesDoneActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state > 0) static_cast<PersistMergeGranulesDoneActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMergeGranulesDoneActor*>(this)->ActorCallback< PersistMergeGranulesDoneActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< PersistMergeGranulesDoneActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< PersistMergeGranulesDoneActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), 10);

	}
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<BlobManagerData> bmData;
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	UID mergeGranuleID;
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	KeyRange mergeRange;
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Version mergeVersion;
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	std::vector<UID> parentGranuleIDs;
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	std::vector<Key> parentGranuleRanges;
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	std::vector<Version> parentGranuleStartVersions;
															#line 2456 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 2458 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	UID tmpWorkerId;
															#line 2506 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	int parentIdx;
															#line 2510 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Key lockKey;
															#line 2511 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	Future<Optional<Value>> oldLockFuture;
															#line 17611 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
};
// This generated class is to be used only via persistMergeGranulesDone()
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class PersistMergeGranulesDoneActor final : public Actor<bool>, public ActorCallback< PersistMergeGranulesDoneActor, 0, Void >, public ActorCallback< PersistMergeGranulesDoneActor, 1, Void >, public ActorCallback< PersistMergeGranulesDoneActor, 2, ForcedPurgeState >, public ActorCallback< PersistMergeGranulesDoneActor, 3, Void >, public ActorCallback< PersistMergeGranulesDoneActor, 4, Void >, public ActorCallback< PersistMergeGranulesDoneActor, 5, Void >, public ActorCallback< PersistMergeGranulesDoneActor, 6, Optional<Value> >, public ActorCallback< PersistMergeGranulesDoneActor, 7, Void >, public ActorCallback< PersistMergeGranulesDoneActor, 8, Void >, public ActorCallback< PersistMergeGranulesDoneActor, 9, Void >, public ActorCallback< PersistMergeGranulesDoneActor, 10, Void >, public FastAllocated<PersistMergeGranulesDoneActor>, public PersistMergeGranulesDoneActorState<PersistMergeGranulesDoneActor> {
															#line 17616 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
	using FastAllocated<PersistMergeGranulesDoneActor>::operator new;
	using FastAllocated<PersistMergeGranulesDoneActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4002398582777317120UL, 13281228174697935872UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistMergeGranulesDoneActor, 0, Void >;
friend struct ActorCallback< PersistMergeGranulesDoneActor, 1, Void >;
friend struct ActorCallback< PersistMergeGranulesDoneActor, 2, ForcedPurgeState >;
friend struct ActorCallback< PersistMergeGranulesDoneActor, 3, Void >;
friend struct ActorCallback< PersistMergeGranulesDoneActor, 4, Void >;
friend struct ActorCallback< PersistMergeGranulesDoneActor, 5, Void >;
friend struct ActorCallback< PersistMergeGranulesDoneActor, 6, Optional<Value> >;
friend struct ActorCallback< PersistMergeGranulesDoneActor, 7, Void >;
friend struct ActorCallback< PersistMergeGranulesDoneActor, 8, Void >;
friend struct ActorCallback< PersistMergeGranulesDoneActor, 9, Void >;
friend struct ActorCallback< PersistMergeGranulesDoneActor, 10, Void >;
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	PersistMergeGranulesDoneActor(Reference<BlobManagerData> const& bmData,UID const& mergeGranuleID,KeyRange const& mergeRange,Version const& mergeVersion,std::vector<UID> const& parentGranuleIDs,std::vector<Key> const& parentGranuleRanges,std::vector<Version> const& parentGranuleStartVersions) 
															#line 17643 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		 : Actor<bool>(),
		   PersistMergeGranulesDoneActorState<PersistMergeGranulesDoneActor>(bmData, mergeGranuleID, mergeRange, mergeVersion, parentGranuleIDs, parentGranuleRanges, parentGranuleStartVersions),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14360226622964786944UL, 6751847166575670272UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMergeGranulesDoneActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistMergeGranulesDone");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistMergeGranulesDone", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistMergeGranulesDoneActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistMergeGranulesDoneActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PersistMergeGranulesDoneActor, 2, ForcedPurgeState >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< PersistMergeGranulesDoneActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< PersistMergeGranulesDoneActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< PersistMergeGranulesDoneActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< PersistMergeGranulesDoneActor, 6, Optional<Value> >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< PersistMergeGranulesDoneActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< PersistMergeGranulesDoneActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< PersistMergeGranulesDoneActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< PersistMergeGranulesDoneActor, 10, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
[[nodiscard]] Future<bool> persistMergeGranulesDone( Reference<BlobManagerData> const& bmData, UID const& mergeGranuleID, KeyRange const& mergeRange, Version const& mergeVersion, std::vector<UID> const& parentGranuleIDs, std::vector<Key> const& parentGranuleRanges, std::vector<Version> const& parentGranuleStartVersions ) {
															#line 2449 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	return Future<bool>(new PersistMergeGranulesDoneActor(bmData, mergeGranuleID, mergeRange, mergeVersion, parentGranuleIDs, parentGranuleRanges, parentGranuleStartVersions));
															#line 17686 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
}

#line 2589 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"

// This is the idempotent function that executes a granule merge once the initial merge intent has been persisted.
															#line 17692 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
namespace {
// This generated class is to be used only via finishMergeGranules()
															#line 2591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
template <class FinishMergeGranulesActor>
															#line 2591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
class FinishMergeGranulesActorState {
															#line 17699 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
public:
															#line 2591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
	FinishMergeGranulesActorState(Reference<BlobManagerData> const& bmData,UID const& mergeGranuleID,KeyRange const& mergeRange,Version const& mergeVersion,std::vector<UID> const& parentGranuleIDs,std::vector<Key> const& parentGranuleRanges,std::vector<Version> const& parentGranuleStartVersions) 
															#line 2591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
															#line 2591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		 : bmData(bmData),
															#line 2591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   mergeGranuleID(mergeGranuleID),
															#line 2591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   mergeRange(mergeRange),
															#line 2591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   mergeVersion(mergeVersion),
															#line 2591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   parentGranuleIDs(parentGranuleIDs),
															#line 2591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   parentGranuleRanges(parentGranuleRanges),
															#line 2591 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		   parentGranuleStartVersions(parentGranuleStartVersions)
															#line 17718 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
	{
		fdb_probe_actor_create("finishMergeGranules", reinterpret_cast<unsigned long>(this));

	}
	~FinishMergeGranulesActorState() 
	{
		fdb_probe_actor_destroy("finishMergeGranules", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2598 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			++bmData->stats.activeMerges;
															#line 2601 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			StrictFuture<Void> __when_expr_0 = bmData->doneRecovering.getFuture();
															#line 2601 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			if (static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17737 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state = 1;
															#line 2601 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FinishMergeGranulesActor, 0, Void >*>(static_cast<FinishMergeGranulesActor*>(this)));
															#line 17742 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FinishMergeGranulesActorState();
		static_cast<FinishMergeGranulesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bmData->loadedClientRanges.getFuture();
															#line 2602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17767 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state = 2;
															#line 2602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinishMergeGranulesActor, 1, Void >*>(static_cast<FinishMergeGranulesActor*>(this)));
															#line 17772 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_1 = bmData->loadedClientRanges.getFuture();
															#line 2602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17783 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state = 2;
															#line 2602 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FinishMergeGranulesActor, 1, Void >*>(static_cast<FinishMergeGranulesActor*>(this)));
															#line 17788 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state > 0) static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state = 0;
		static_cast<FinishMergeGranulesActor*>(this)->ActorCallback< FinishMergeGranulesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinishMergeGranulesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finishMergeGranules", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMergeGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMergeGranules", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FinishMergeGranulesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finishMergeGranules", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMergeGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMergeGranules", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FinishMergeGranulesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finishMergeGranules", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMergeGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMergeGranules", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2603 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(0);
															#line 2603 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17874 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state = 3;
															#line 2603 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FinishMergeGranulesActor, 2, Void >*>(static_cast<FinishMergeGranulesActor*>(this)));
															#line 17879 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2603 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(0);
															#line 2603 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17890 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state = 3;
															#line 2603 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FinishMergeGranulesActor, 2, Void >*>(static_cast<FinishMergeGranulesActor*>(this)));
															#line 17895 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state > 0) static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state = 0;
		static_cast<FinishMergeGranulesActor*>(this)->ActorCallback< FinishMergeGranulesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinishMergeGranulesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finishMergeGranules", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMergeGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMergeGranules", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FinishMergeGranulesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finishMergeGranules", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMergeGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMergeGranules", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FinishMergeGranulesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finishMergeGranules", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMergeGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMergeGranules", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 2606 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (g_network->isSimulated())
															#line 17979 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2607 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for(int i = 1;i < parentGranuleRanges.size() - 1;i++) {
															#line 2608 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				ASSERT(!bmData->mergeHardBoundaries.contains(parentGranuleRanges[i]));
															#line 17985 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
															#line 2613 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<bool> __when_expr_3 = forceGranuleFlush(bmData, mergeGranuleID, mergeRange, mergeVersion);
															#line 2613 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 17992 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state = 4;
															#line 2613 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FinishMergeGranulesActor, 3, bool >*>(static_cast<FinishMergeGranulesActor*>(this)));
															#line 17997 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 2606 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (g_network->isSimulated())
															#line 18006 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2607 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			for(int i = 1;i < parentGranuleRanges.size() - 1;i++) {
															#line 2608 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
				ASSERT(!bmData->mergeHardBoundaries.contains(parentGranuleRanges[i]));
															#line 18012 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
			}
		}
															#line 2613 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		StrictFuture<bool> __when_expr_3 = forceGranuleFlush(bmData, mergeGranuleID, mergeRange, mergeVersion);
															#line 2613 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 18019 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state = 4;
															#line 2613 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FinishMergeGranulesActor, 3, bool >*>(static_cast<FinishMergeGranulesActor*>(this)));
															#line 18024 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state > 0) static_cast<FinishMergeGranulesActor*>(this)->actor_wait_state = 0;
		static_cast<FinishMergeGranulesActor*>(this)->ActorCallback< FinishMergeGranulesActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FinishMergeGranulesActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("finishMergeGranules", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMergeGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMergeGranules", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FinishMergeGranulesActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("finishMergeGranules", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMergeGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMergeGranules", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FinishMergeGranulesActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("finishMergeGranules", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FinishMergeGranulesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("finishMergeGranules", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(bool const& successFlush,int loopDepth) 
	{
															#line 2614 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
		if (!successFlush)
															#line 18108 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/build_output/fdbserver/BlobManager.actor.g.cpp"
		{
															#line 2615 "/codebuild/output/src1051786411/src/github.com/apple/foundationdb/fdbserver/BlobManager.actor.cpp"
			bmData->activeGranuleMerges.insert(mergeRange, invalidVersion);
												