#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
/*
 * AsyncFileRead.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <utility>
#include <vector>

#include "fdbserver/workloads/workloads.actor.h"
#include "flow/ActorCollection.h"
#include "flow/SystemMonitor.h"
#include "flow/IAsyncFile.h"
#include "fdbserver/workloads/AsyncFile.actor.h"
#include "flow/DeterministicRandom.h"
#include "flow/actorcompiler.h" // This must be the last #include.

static const double ROLL_TIME = 5.0;

struct IOLog {
	struct ProcessLog {
		double startTime;
		double lastTime;
		double sumSq, sum, max, count;

		bool logLatency; // when false, log times and compute elapsed, else, interpret the log'd time as a latency

		ProcessLog() : logLatency(false) {};

		virtual void reset() {
			startTime = now();
			lastTime = startTime;
			sumSq = 0.0;
			sum = 0.0;
			max = 0.0;
			count = 0.0;
		}

		void log(double time) {
			count++;
			auto l = logLatency ? time : (time - lastTime); // see logLatency comment above
			sum += l;
			sumSq += l * l;
			max = std::max<double>(max, l);
			lastTime = time;
		}

		void dumpMetrics(std::string name) {
			double elapsed = now() - startTime;
			TraceEvent("ProcessLog")
			    .detail("Name", name)
			    .detail("Hz", count / elapsed)
			    .detail("Latency", sumSq / elapsed / 2.0)
			    .detail("AvgLatency", sum / count)
			    .detail("MaxLatency", max)
			    .detail("StartTime", startTime)
			    .detail("Elapsed", elapsed);
		}
	};

	double lastRoll;

	ProcessLog issue, completion, duration;
	ProcessLog issueR, completionR, durationR;
	ProcessLog issueW, completionW, durationW;

	std::vector<std::pair<std::string, ProcessLog*>> logs;

	IOLog() {
		logs.emplace_back("issue", &issue);
		logs.emplace_back("completion", &completion);
		logs.emplace_back("duration", &duration);
		logs.emplace_back("issueR", &issueR);
		logs.emplace_back("completionR", &completionR);
		logs.emplace_back("durationR", &durationR);
		logs.emplace_back("issueW", &issueW);
		logs.emplace_back("completionW", &completionW);
		logs.emplace_back("durationW", &durationW);

		duration.logLatency = true;
		durationR.logLatency = true;
		durationW.logLatency = true;

		lastRoll = now();
		for (int i = 0; i < logs.size(); i++)
			logs[i].second->reset();
	}

	~IOLog() { roll(); }

	void roll() {
		for (int i = 0; i < logs.size(); i++) {
			logs[i].second->dumpMetrics(logs[i].first);
			logs[i].second->reset();
		}
	}

	void checkRoll(double time) {
		if (time - lastRoll > ROLL_TIME) {
			roll();
			lastRoll = time;
		}
	}

	void logIOIssue(bool isWrite, double issueTime) {
		issue.log(issueTime);
		if (isWrite)
			issueW.log(issueTime);
		else
			issueR.log(issueTime);

		checkRoll(issueTime);
	}

	void logIOCompletion(bool isWrite, double start, double end) {
		completion.log(end);
		if (isWrite)
			completionW.log(end);
		else
			completionR.log(end);

		auto elapsed = end - start;
		duration.log(elapsed);
		if (isWrite)
			durationW.log(elapsed);
		else
			durationR.log(elapsed);

		checkRoll(end);
	}
};

struct AsyncFileReadWorkload : public AsyncFileWorkload {
	constexpr static auto NAME = "AsyncFileRead";
	// Buffers used to store what is being read or written
	std::vector<Reference<AsyncFileBuffer>> readBuffers;

	// The futures for the asynchronous read operations
	std::vector<Future<int>> readFutures;

	// Number of reads to perform in parallel.  Read tests are performed only if this is greater than zero
	int numParallelReads;

	// The number of bytes read in each call of read
	int readSize;

	// Whether or not I/O should be performed sequentially
	bool sequential;

	bool randomData; // if true, randomize the data in writes

	bool unbatched; // If true, issue reads continuously instead of waiting for all numParallelReads reads to complete
	double writeFraction;
	double fixedRate;

	double averageCpuUtilization;
	PerfIntCounter bytesRead;

	IOLog* ioLog;
	RandomByteGenerator rbg;

	AsyncFileReadWorkload(WorkloadContext const& wcx) : AsyncFileWorkload(wcx), bytesRead("Bytes Read"), ioLog(0) {
		// Only run on one client
		numParallelReads = getOption(options, "numParallelReads"_sr, 0);
		readSize = getOption(options, "readSize"_sr, _PAGE_SIZE);
		fileSize = getOption(options, "fileSize"_sr, (int64_t)0); // 0 = use existing, else, change file size
		unbatched = getOption(options, "unbatched"_sr, false);
		sequential = getOption(options, "sequential"_sr, true);
		writeFraction = getOption(options, "writeFraction"_sr, 0.0);
		randomData = getOption(options, "randomData"_sr, true);
		fixedRate = getOption(options, "fixedRate"_sr, 0.0);
	}

	~AsyncFileReadWorkload() override {}

	Future<Void> setup(Database const& cx) override {
		if (enabled)
			return _setup(this);

		return Void();
	}

																#line 198 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
// This generated class is to be used only via _setup()
															#line 196 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
template <class _setupActor>
															#line 196 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
class _setupActorState {
															#line 204 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
public:
															#line 196 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	_setupActorState(AsyncFileReadWorkload* const& self) 
															#line 196 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
															#line 196 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		 : self(self)
															#line 211 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("_setup", reinterpret_cast<unsigned long>(this));

	}
	~_setupActorState() 
	{
		fdb_probe_actor_destroy("_setup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 198 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			if (self->unbufferedIO && self->readSize % AsyncFileWorkload::_PAGE_SIZE != 0)
															#line 226 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			{
															#line 199 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				self->readSize = std::max(AsyncFileWorkload::_PAGE_SIZE, self->readSize - self->readSize % AsyncFileWorkload::_PAGE_SIZE);
															#line 230 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			}
															#line 203 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			for(int i = 0;i < self->numParallelReads;i++) {
															#line 204 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				self->readBuffers.push_back(self->allocateBuffer(self->readSize));
															#line 236 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			}
															#line 206 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->openFile( self, IAsyncFile::OPEN_CREATE | IAsyncFile::OPEN_READWRITE, 0666, self->fileSize, self->fileSize != 0);
															#line 206 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 242 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<_setupActor*>(this)->actor_wait_state = 1;
															#line 206 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 0, Void >*>(static_cast<_setupActor*>(this)));
															#line 247 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 209 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		StrictFuture<int64_t> __when_expr_1 = self->fileHandle->file->size();
															#line 209 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 272 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 2;
															#line 209 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 1, int64_t >*>(static_cast<_setupActor*>(this)));
															#line 277 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 209 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		StrictFuture<int64_t> __when_expr_1 = self->fileHandle->file->size();
															#line 209 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 288 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_setupActor*>(this)->actor_wait_state = 2;
															#line 209 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 1, int64_t >*>(static_cast<_setupActor*>(this)));
															#line 293 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _setupActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int64_t const& fileSize,int loopDepth) 
	{
															#line 210 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		self->fileSize = fileSize;
															#line 212 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (!static_cast<_setupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_setupActorState(); static_cast<_setupActor*>(this)->destroy(); return 0; }
															#line 379 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		new (&static_cast<_setupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(int64_t && fileSize,int loopDepth) 
	{
															#line 210 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		self->fileSize = fileSize;
															#line 212 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (!static_cast<_setupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_setupActorState(); static_cast<_setupActor*>(this)->destroy(); return 0; }
															#line 393 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		new (&static_cast<_setupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(int64_t const& fileSize,int loopDepth) 
	{
		loopDepth = a_body1cont4(fileSize, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(int64_t && fileSize,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(fileSize), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 1, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 1, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _setupActor, 1, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _setupActor, 1, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 196 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	AsyncFileReadWorkload* self;
															#line 478 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
};
// This generated class is to be used only via _setup()
															#line 196 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
class _setupActor final : public Actor<Void>, public ActorCallback< _setupActor, 0, Void >, public ActorCallback< _setupActor, 1, int64_t >, public FastAllocated<_setupActor>, public _setupActorState<_setupActor> {
															#line 483 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
public:
	using FastAllocated<_setupActor>::operator new;
	using FastAllocated<_setupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11016333054934873344UL, 14529058968295032576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _setupActor, 0, Void >;
friend struct ActorCallback< _setupActor, 1, int64_t >;
															#line 196 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	_setupActor(AsyncFileReadWorkload* const& self) 
															#line 501 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   _setupActorState<_setupActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3044502327521131520UL, 13387266639422932480UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_setup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _setupActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _setupActor, 1, int64_t >*)0, actor_cancelled()); break;
		}

	}
};
															#line 196 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
[[nodiscard]] Future<Void> _setup( AsyncFileReadWorkload* const& self ) {
															#line 196 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	return Future<Void>(new _setupActor(self));
															#line 534 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
}

#line 214 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"

	Future<Void> start(Database const& cx) override {
		if (enabled)
			return _start(this);

		return Void();
	}

																#line 546 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
// This generated class is to be used only via _start()
															#line 222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
template <class _startActor>
															#line 222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
class _startActorState {
															#line 552 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
public:
															#line 222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	_startActorState(AsyncFileReadWorkload* const& self) 
															#line 222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
															#line 222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		 : self(self),
															#line 223 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		   statState()
															#line 561 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("_start", reinterpret_cast<unsigned long>(this));

	}
	~_startActorState() 
	{
		fdb_probe_actor_destroy("_start", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 224 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			customSystemMonitor("AsyncFile Metrics", &statState);
															#line 226 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeout(self->runReadTest(self), self->testDuration, Void());
															#line 226 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 580 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<_startActor*>(this)->actor_wait_state = 1;
															#line 226 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _startActor, 0, Void >*>(static_cast<_startActor*>(this)));
															#line 585 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_startActorState();
		static_cast<_startActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 228 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		SystemStatistics stats = customSystemMonitor("AsyncFile Metrics", &statState);
															#line 229 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		self->averageCpuUtilization = stats.processCPUSeconds / stats.elapsed;
															#line 232 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		StrictFuture<Void> __when_expr_1 = timeout(waitForAll(self->readFutures), 10, Void());
															#line 232 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 614 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_startActor*>(this)->actor_wait_state = 2;
															#line 232 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _startActor, 1, Void >*>(static_cast<_startActor*>(this)));
															#line 619 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 228 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		SystemStatistics stats = customSystemMonitor("AsyncFile Metrics", &statState);
															#line 229 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		self->averageCpuUtilization = stats.processCPUSeconds / stats.elapsed;
															#line 232 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		StrictFuture<Void> __when_expr_1 = timeout(waitForAll(self->readFutures), 10, Void());
															#line 232 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 634 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_startActor*>(this)->actor_wait_state = 2;
															#line 232 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _startActor, 1, Void >*>(static_cast<_startActor*>(this)));
															#line 639 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _startActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _startActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 234 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (!static_cast<_startActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_startActorState(); static_cast<_startActor*>(this)->destroy(); return 0; }
															#line 723 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		new (&static_cast<_startActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_startActorState();
		static_cast<_startActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 234 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (!static_cast<_startActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_startActorState(); static_cast<_startActor*>(this)->destroy(); return 0; }
															#line 735 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		new (&static_cast<_startActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_startActorState();
		static_cast<_startActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _startActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _startActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	AsyncFileReadWorkload* self;
															#line 223 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	StatisticsState statState;
															#line 822 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
};
// This generated class is to be used only via _start()
															#line 222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
class _startActor final : public Actor<Void>, public ActorCallback< _startActor, 0, Void >, public ActorCallback< _startActor, 1, Void >, public FastAllocated<_startActor>, public _startActorState<_startActor> {
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
public:
	using FastAllocated<_startActor>::operator new;
	using FastAllocated<_startActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11462119031217882368UL, 5402737592775685376UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _startActor, 0, Void >;
friend struct ActorCallback< _startActor, 1, Void >;
															#line 222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	_startActor(AsyncFileReadWorkload* const& self) 
															#line 845 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   _startActorState<_startActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16516870674810565632UL, 12235738791954228224UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_start");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _startActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _startActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
[[nodiscard]] Future<Void> _start( AsyncFileReadWorkload* const& self ) {
															#line 222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	return Future<Void>(new _startActor(self));
															#line 878 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
}

#line 236 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"

																#line 883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
// This generated class is to be used only via readLoop()
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
template <class ReadLoopActor>
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
class ReadLoopActorState {
															#line 889 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
public:
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	ReadLoopActorState(AsyncFileReadWorkload* const& self,int const& bufferIndex,double const& fixedRate) 
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		 : self(self),
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		   bufferIndex(bufferIndex),
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		   fixedRate(fixedRate),
															#line 238 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		   writeFlag(false),
															#line 239 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		   begin(0.0),
															#line 240 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		   lastTime(now())
															#line 906 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("readLoop", reinterpret_cast<unsigned long>(this));

	}
	~ReadLoopActorState() 
	{
		fdb_probe_actor_destroy("readLoop", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 241 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			;
															#line 921 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadLoopActorState();
		static_cast<ReadLoopActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 242 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (fixedRate)
															#line 951 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		{
															#line 243 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			StrictFuture<Void> __when_expr_0 = poisson(&lastTime, 1.0 / fixedRate);
															#line 243 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			if (static_cast<ReadLoopActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 957 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadLoopActor*>(this)->actor_wait_state = 1;
															#line 243 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadLoopActor, 0, Void >*>(static_cast<ReadLoopActor*>(this)));
															#line 962 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 246 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		int64_t offset;
															#line 247 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (self->unbufferedIO)
															#line 978 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		{
															#line 248 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			offset = (int64_t)(deterministicRandom()->random01() * (self->fileSize - 1) / AsyncFileWorkload::_PAGE_SIZE) * AsyncFileWorkload::_PAGE_SIZE;
															#line 982 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		}
		else
		{
															#line 252 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			offset = (int64_t)(deterministicRandom()->random01() * (self->fileSize - 1));
															#line 988 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		}
															#line 254 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		writeFlag = deterministicRandom()->random01() < self->writeFraction;
															#line 255 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (writeFlag)
															#line 994 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		{
															#line 256 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			self->rbg.writeRandomBytesToBuffer((char*)self->readBuffers[bufferIndex]->buffer, self->readSize);
															#line 998 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		}
															#line 258 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		auto r = writeFlag ? tag(self->fileHandle->file->write(self->readBuffers[bufferIndex]->buffer, self->readSize, offset), self->readSize) : self->fileHandle->file->read(self->readBuffers[bufferIndex]->buffer, self->readSize, offset);
															#line 263 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		begin = now();
															#line 264 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (self->ioLog)
															#line 1006 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		{
															#line 265 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			self->ioLog->logIOIssue(writeFlag, begin);
															#line 1010 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		}
															#line 266 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		StrictFuture<Void> __when_expr_1 = success(uncancellable(holdWhile(self->fileHandle, holdWhile(self->readBuffers[bufferIndex], r))));
															#line 266 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (static_cast<ReadLoopActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1016 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadLoopActor*>(this)->actor_wait_state = 2;
															#line 266 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadLoopActor, 1, Void >*>(static_cast<ReadLoopActor*>(this)));
															#line 1021 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadLoopActor*>(this)->actor_wait_state > 0) static_cast<ReadLoopActor*>(this)->actor_wait_state = 0;
		static_cast<ReadLoopActor*>(this)->ActorCallback< ReadLoopActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadLoopActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readLoop", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readLoop", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadLoopActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readLoop", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readLoop", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadLoopActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readLoop", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readLoop", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 267 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (self->ioLog)
															#line 1117 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		{
															#line 268 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			self->ioLog->logIOCompletion(writeFlag, begin, now());
															#line 1121 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		}
															#line 269 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		self->bytesRead += self->readSize;
															#line 1125 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 267 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (self->ioLog)
															#line 1134 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		{
															#line 268 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			self->ioLog->logIOCompletion(writeFlag, begin, now());
															#line 1138 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		}
															#line 269 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		self->bytesRead += self->readSize;
															#line 1142 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadLoopActor*>(this)->actor_wait_state > 0) static_cast<ReadLoopActor*>(this)->actor_wait_state = 0;
		static_cast<ReadLoopActor*>(this)->ActorCallback< ReadLoopActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadLoopActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readLoop", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readLoop", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadLoopActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readLoop", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readLoop", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadLoopActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readLoop", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readLoop", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	AsyncFileReadWorkload* self;
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	int bufferIndex;
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	double fixedRate;
															#line 238 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	bool writeFlag;
															#line 239 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	double begin;
															#line 240 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	double lastTime;
															#line 1234 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
};
// This generated class is to be used only via readLoop()
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
class ReadLoopActor final : public Actor<Void>, public ActorCallback< ReadLoopActor, 0, Void >, public ActorCallback< ReadLoopActor, 1, Void >, public FastAllocated<ReadLoopActor>, public ReadLoopActorState<ReadLoopActor> {
															#line 1239 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
public:
	using FastAllocated<ReadLoopActor>::operator new;
	using FastAllocated<ReadLoopActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4313902015896791552UL, 3606021969131302400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadLoopActor, 0, Void >;
friend struct ActorCallback< ReadLoopActor, 1, Void >;
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	ReadLoopActor(AsyncFileReadWorkload* const& self,int const& bufferIndex,double const& fixedRate) 
															#line 1257 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ReadLoopActorState<ReadLoopActor>(self, bufferIndex, fixedRate),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readLoop", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1788455649453472768UL, 16390995398306349568UL);
		ActorExecutionContextHelper __helper(static_cast<ReadLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readLoop");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readLoop", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadLoopActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadLoopActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
[[nodiscard]] static Future<Void> readLoop( AsyncFileReadWorkload* const& self, int const& bufferIndex, double const& fixedRate ) {
															#line 237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	return Future<Void>(new ReadLoopActor(self, bufferIndex, fixedRate));
															#line 1290 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
}

#line 273 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"

																#line 1295 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
// This generated class is to be used only via runReadTest()
															#line 274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
template <class RunReadTestActor>
															#line 274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
class RunReadTestActorState {
															#line 1301 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
public:
															#line 274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	RunReadTestActorState(AsyncFileReadWorkload* const& self) 
															#line 274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
															#line 274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		 : self(self)
															#line 1308 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("runReadTest", reinterpret_cast<unsigned long>(this));

	}
	~RunReadTestActorState() 
	{
		fdb_probe_actor_destroy("runReadTest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 275 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			if (self->unbatched)
															#line 1323 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			{
															#line 276 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				self->ioLog = new IOLog();
															#line 278 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				std::vector<Future<Void>> readers;
															#line 280 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				readers.reserve(self->numParallelReads);
															#line 281 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				for(int i = 0;i < self->numParallelReads;i++) {
															#line 282 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
					readers.push_back(readLoop(self, i, self->fixedRate / self->numParallelReads));
															#line 1335 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
				}
															#line 283 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				StrictFuture<Void> __when_expr_0 = waitForAll(readers);
															#line 283 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				if (static_cast<RunReadTestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1341 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<RunReadTestActor*>(this)->actor_wait_state = 1;
															#line 283 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RunReadTestActor, 0, Void >*>(static_cast<RunReadTestActor*>(this)));
															#line 1346 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunReadTestActorState();
		static_cast<RunReadTestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 289 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		offset = self->fileSize;
															#line 290 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		;
															#line 1376 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 285 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		delete self->ioLog;
															#line 286 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (!static_cast<RunReadTestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RunReadTestActorState(); static_cast<RunReadTestActor*>(this)->destroy(); return 0; }
															#line 1387 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		new (&static_cast<RunReadTestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RunReadTestActorState();
		static_cast<RunReadTestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 285 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		delete self->ioLog;
															#line 286 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (!static_cast<RunReadTestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RunReadTestActorState(); static_cast<RunReadTestActor*>(this)->destroy(); return 0; }
															#line 1401 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		new (&static_cast<RunReadTestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RunReadTestActorState();
		static_cast<RunReadTestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunReadTestActor*>(this)->actor_wait_state > 0) static_cast<RunReadTestActor*>(this)->actor_wait_state = 0;
		static_cast<RunReadTestActor*>(this)->ActorCallback< RunReadTestActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunReadTestActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runReadTest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunReadTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runReadTest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunReadTestActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runReadTest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunReadTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runReadTest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RunReadTestActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runReadTest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunReadTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runReadTest", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 292 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		for(int i = 0;i < self->numParallelReads;i++) {
															#line 293 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			if (self->sequential)
															#line 1497 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
			{
															#line 294 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				offset += self->readSize;
															#line 297 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				if (offset >= self->fileSize)
															#line 1503 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
				{
															#line 298 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
					offset = 0;
															#line 1507 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
				}
			}
			else
			{
															#line 299 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
				if (self->unbufferedIO)
															#line 1514 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
				{
															#line 300 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
					offset = (int64_t)(deterministicRandom()->random01() * (self->fileSize - 1) / AsyncFileWorkload::_PAGE_SIZE) * AsyncFileWorkload::_PAGE_SIZE;
															#line 1518 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
				}
				else
				{
															#line 304 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
					offset = (int64_t)(deterministicRandom()->random01() * (self->fileSize - 1));
															#line 1524 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
				}
			}
															#line 308 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
			self->readFutures.push_back(uncancellable(holdWhile( self->fileHandle, holdWhile(self->readBuffers[i], self->fileHandle->file->read(self->readBuffers[i]->buffer, self->readSize, offset)))));
															#line 1529 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		}
															#line 314 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		StrictFuture<Void> __when_expr_1 = waitForAll(self->readFutures);
															#line 314 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (static_cast<RunReadTestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1535 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RunReadTestActor*>(this)->actor_wait_state = 2;
															#line 314 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RunReadTestActor, 1, Void >*>(static_cast<RunReadTestActor*>(this)));
															#line 1540 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 315 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		self->bytesRead += self->readSize * self->numParallelReads;
															#line 317 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		self->readFutures.clear();
															#line 319 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(0);
															#line 319 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (static_cast<RunReadTestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<RunReadTestActor*>(this)->actor_wait_state = 3;
															#line 319 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RunReadTestActor, 2, Void >*>(static_cast<RunReadTestActor*>(this)));
															#line 1560 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 315 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		self->bytesRead += self->readSize * self->numParallelReads;
															#line 317 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		self->readFutures.clear();
															#line 319 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(0);
															#line 319 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		if (static_cast<RunReadTestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1575 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<RunReadTestActor*>(this)->actor_wait_state = 3;
															#line 319 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RunReadTestActor, 2, Void >*>(static_cast<RunReadTestActor*>(this)));
															#line 1580 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RunReadTestActor*>(this)->actor_wait_state > 0) static_cast<RunReadTestActor*>(this)->actor_wait_state = 0;
		static_cast<RunReadTestActor*>(this)->ActorCallback< RunReadTestActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunReadTestActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runReadTest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunReadTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runReadTest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RunReadTestActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runReadTest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunReadTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runReadTest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RunReadTestActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runReadTest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunReadTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runReadTest", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont8(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont8(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RunReadTestActor*>(this)->actor_wait_state > 0) static_cast<RunReadTestActor*>(this)->actor_wait_state = 0;
		static_cast<RunReadTestActor*>(this)->ActorCallback< RunReadTestActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunReadTestActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runReadTest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunReadTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runReadTest", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RunReadTestActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runReadTest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunReadTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runReadTest", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RunReadTestActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runReadTest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunReadTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runReadTest", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	AsyncFileReadWorkload* self;
															#line 289 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	int64_t offset;
															#line 1751 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
};
// This generated class is to be used only via runReadTest()
															#line 274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
class RunReadTestActor final : public Actor<Void>, public ActorCallback< RunReadTestActor, 0, Void >, public ActorCallback< RunReadTestActor, 1, Void >, public ActorCallback< RunReadTestActor, 2, Void >, public FastAllocated<RunReadTestActor>, public RunReadTestActorState<RunReadTestActor> {
															#line 1756 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
public:
	using FastAllocated<RunReadTestActor>::operator new;
	using FastAllocated<RunReadTestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5400951308479590656UL, 17365466537205651456UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RunReadTestActor, 0, Void >;
friend struct ActorCallback< RunReadTestActor, 1, Void >;
friend struct ActorCallback< RunReadTestActor, 2, Void >;
															#line 274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	RunReadTestActor(AsyncFileReadWorkload* const& self) 
															#line 1775 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RunReadTestActorState<RunReadTestActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("runReadTest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9898814479209417472UL, 4026280813362670080UL);
		ActorExecutionContextHelper __helper(static_cast<RunReadTestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("runReadTest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("runReadTest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RunReadTestActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RunReadTestActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RunReadTestActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
[[nodiscard]] Future<Void> runReadTest( AsyncFileReadWorkload* const& self ) {
															#line 274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"
	return Future<Void>(new RunReadTestActor(self));
															#line 1809 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/AsyncFileRead.actor.g.cpp.py_gen"
}

#line 322 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/workloads/AsyncFileRead.actor.cpp"

	void getMetrics(std::vector<PerfMetric>& m) override {
		if (enabled) {
			m.emplace_back("Bytes read/sec", bytesRead.getValue() / testDuration, Averaged::False);
			m.emplace_back("Average CPU Utilization (Percentage)", averageCpuUtilization * 100, Averaged::False);
		}
	}
};

WorkloadFactory<AsyncFileReadWorkload> AsyncFileReadWorkloadFactory;
