#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
/*
 * MockS3Server.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2025 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbserver/MockS3Server.h"

#include "fdbrpc/HTTP.h"
#include "fdbrpc/simulator.h"
#include "flow/Trace.h"
#include "flow/IRandom.h"
#include "flow/serialize.h"

#include <string>
#include <map>
#include <sstream>
#include <regex>
#include <utility>
#include <iostream>
#include <algorithm>

#include "flow/IAsyncFile.h" // For IAsyncFileSystem
#include "flow/Platform.h" // For platform::createDirectory

#include "rapidjson/document.h"
#include "rapidjson/writer.h"
#include "rapidjson/stringbuffer.h"
#include "rapidjson/prettywriter.h"

#include "flow/actorcompiler.h" // This must be the last #include.

/*
 * ACTOR STATE VARIABLE INITIALIZATION REQUIREMENT
 *
 * ACTORs with early returns (before any wait()) crash with canBeSet() assertion if no state
 * variable is declared before the return. The actor compiler generates a member initialization
 * list (": member(value)") in the state class constructor only when it sees state variables.
 * This initialization list ensures the Actor<T> base class and its internal Promise are fully
 * initialized before any code runs. Without it, early returns try to use an uninitialized Promise.
 *
 * FIX: Declare at least one state variable BEFORE any early return. Declaration alone is enough.
 *
 * CORRECT:
 *   ACTOR Future<Void> someActor(...) {
 *       state std::string data;                // Triggers member init list (requires default ctor)
 *       if (earlyExitCondition) return Void(); // Safe - Promise is initialized
 *       data = computeValue();                 // Can initialize later
 *   }
 *
 *   // Or if no default constructor: state MyType x(params); and initialize at declaration
 *
 * WRONG (canBeSet() crash):
 *   ACTOR Future<Void> someActor(...) {
 *       if (earlyExitCondition) return Void(); // CRASH - no member init list generated yet
 *       state std::string data;                // Too late - compiler didn't see it early enough
 *   }
 */

// MockS3 persistence file extensions and constants
namespace {
constexpr const char* DEFAULT_MOCKS3_PERSISTENCE_DIR = "simfdb/mocks3";
constexpr const char* OBJECT_DATA_SUFFIX = ".data";
constexpr const char* OBJECT_META_SUFFIX = ".meta.json";
constexpr const char* MULTIPART_STATE_SUFFIX = ".state.json";
constexpr size_t OBJECT_META_SUFFIX_LEN = 10; // strlen(".meta.json")
constexpr size_t MULTIPART_STATE_SUFFIX_LEN = 11; // strlen(".state.json")
} // namespace

// Global storage for MockS3 (shared across all simulated processes)
struct MockS3GlobalStorage {
	struct ObjectData {
		std::string content;
		HTTP::Headers headers;
		std::map<std::string, std::string> tags;
		std::string etag;
		double lastModified;

		ObjectData() : lastModified(now()) {}
		ObjectData(const std::string& data) : content(data), lastModified(now()) { etag = generateETag(data); }

		static std::string generateETag(const std::string& content) {
			return "\"" + HTTP::computeMD5Sum(content) + "\"";
		}
	};

	struct MultipartUpload {
		std::string uploadId;
		std::string bucket;
		std::string object;
		std::map<int, std::pair<std::string, std::string>> parts; // partNum -> {etag, content}
		HTTP::Headers metadata;
		double initiated;

		MultipartUpload() = default;
		MultipartUpload(const std::string& b, const std::string& o) : bucket(b), object(o), initiated(now()) {
			uploadId = deterministicRandom()->randomUniqueID().toString();
		}
	};

	std::map<std::string, std::map<std::string, ObjectData>> buckets;
	std::map<std::string, MultipartUpload> multipartUploads;

	// Persistence configuration
	std::string persistenceDir;
	bool persistenceEnabled = false;
	bool persistenceLoaded = false;

	// Note: In FDB simulation, function-local statics are SHARED across all simulated processes
	// because they all run on the same OS thread. This is exactly what we want for MockS3 storage.
	MockS3GlobalStorage() {
		// Note: Don't use TraceEvent in constructor - may be called before GlobalConfig is initialized
	}

	// Clear all stored data - called at the start of each simulation test to prevent
	// data accumulation across multiple tests
	void clearStorage() {
		buckets.clear();
		multipartUploads.clear();
		TraceEvent("MockS3GlobalStorageCleared").detail("Address", format("%p", this));
	}

	// Enable persistence to specified directory
	// Note: When using simulation filesystem, directories are created automatically by file open()
	// and each simulated machine has its own isolated directory structure
	void enablePersistence(const std::string& dir) {
		persistenceDir = dir;
		persistenceEnabled = true;
		persistenceLoaded = false;

		// Note: Don't use TraceEvent here - may be called before GlobalConfig is initialized
		// TraceEvent during explicit registration (initializeMockS3Persistence) is safe
	}

	// Get paths for persistence files
	// Note: Object names with slashes map directly to filesystem directory structure
	std::string getObjectDataPath(const std::string& bucket, const std::string& object) const {
		return persistenceDir + "/objects/" + bucket + "/" + object + OBJECT_DATA_SUFFIX;
	}

	std::string getObjectMetaPath(const std::string& bucket, const std::string& object) const {
		return persistenceDir + "/objects/" + bucket + "/" + object + OBJECT_META_SUFFIX;
	}

	std::string getMultipartStatePath(const std::string& uploadId) const {
		return persistenceDir + "/multipart/" + uploadId + MULTIPART_STATE_SUFFIX;
	}

	std::string getMultipartPartPath(const std::string& uploadId, int partNum) const {
		return persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
	}

	std::string getMultipartPartMetaPath(const std::string& uploadId, int partNum) const {
		return persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum) + OBJECT_META_SUFFIX;
	}
};

// Accessor function - uses function-local static for lazy initialization
// In simulation, this static is shared across all simulated processes (same OS thread)
static MockS3GlobalStorage& getGlobalStorage() {
	static MockS3GlobalStorage storage;
	return storage;
}

// Helper: Create all parent directories for a file path
// Uses platform::createDirectory which handles recursive creation and EEXIST errors
static void createParentDirectories(const std::string& filePath) {
	size_t lastSlash = filePath.find_last_of('/');
	if (lastSlash != std::string::npos && lastSlash > 0) {
		std::string parentDir = filePath.substr(0, lastSlash);
		platform::createDirectory(parentDir); // Handles recursive creation and EEXIST
	}
}

// ACTOR: Atomic file write using simulation filesystem without chaos injection
// Chaos-free because AsyncFileChaos only affects files with "storage-" in the name
// (see AsyncFileChaos.h:40). OPEN_NO_AIO controls AsyncFileNonDurable behavior.
															#line 194 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via atomicWriteFile()
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class AtomicWriteFileActor>
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class AtomicWriteFileActorState {
															#line 201 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	AtomicWriteFileActorState(std::string const& path,std::string const& content) 
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path),
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   content(content)
															#line 210 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("atomicWriteFile", reinterpret_cast<unsigned long>(this));

	}
	~AtomicWriteFileActorState() 
	{
		fdb_probe_actor_destroy("atomicWriteFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 195 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				createParentDirectories(path);
															#line 199 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( path, IAsyncFile::OPEN_ATOMIC_WRITE_AND_CREATE | IAsyncFile::OPEN_CREATE | IAsyncFile::OPEN_READWRITE | IAsyncFile::OPEN_UNCACHED | IAsyncFile::OPEN_NO_AIO, 0644);
															#line 199 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 230 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 1;
															#line 199 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 235 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AtomicWriteFileActorState();
		static_cast<AtomicWriteFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 213 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<AtomicWriteFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AtomicWriteFileActorState(); static_cast<AtomicWriteFileActor*>(this)->destroy(); return 0; }
															#line 264 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<AtomicWriteFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AtomicWriteFileActorState();
		static_cast<AtomicWriteFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 211 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceWriteException").error(e).detail("Path", path);
															#line 277 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 205 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = file->write(content.data(), content.size(), 0);
															#line 205 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 294 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 2;
															#line 205 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 1, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 299 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& __file,int loopDepth) 
	{
															#line 199 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = __file;
															#line 308 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && __file,int loopDepth) 
	{
		file = std::move(__file);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 206 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = file->sync();
															#line 206 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 389 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 3;
															#line 206 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 2, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 394 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 206 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = file->sync();
															#line 206 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 405 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 3;
															#line 206 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 2, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 410 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 207 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 209 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceWriteSuccess").detail("Path", path).detail("Size", content.size());
															#line 496 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 207 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 209 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceWriteSuccess").detail("Path", path).detail("Size", content.size());
															#line 507 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 199 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<IAsyncFile> file;
															#line 606 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via atomicWriteFile()
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class AtomicWriteFileActor final : public Actor<Void>, public ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >, public ActorCallback< AtomicWriteFileActor, 1, Void >, public ActorCallback< AtomicWriteFileActor, 2, Void >, public FastAllocated<AtomicWriteFileActor>, public AtomicWriteFileActorState<AtomicWriteFileActor> {
															#line 611 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<AtomicWriteFileActor>::operator new;
	using FastAllocated<AtomicWriteFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(441850151131920896UL, 10829873217072752384UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< AtomicWriteFileActor, 1, Void >;
friend struct ActorCallback< AtomicWriteFileActor, 2, Void >;
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	AtomicWriteFileActor(std::string const& path,std::string const& content) 
															#line 630 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   AtomicWriteFileActorState<AtomicWriteFileActor>(path, content),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1551474338982279168UL, 5682969234323159552UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("atomicWriteFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> atomicWriteFile( std::string const& path, std::string const& content ) {
															#line 192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new AtomicWriteFileActor(path, content));
															#line 665 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 215 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Read file content using simulation filesystem without chaos
// Chaos-free because AsyncFileChaos only affects files with "storage-" in the name
															#line 672 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via readFileContent()
															#line 218 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class ReadFileContentActor>
															#line 218 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class ReadFileContentActorState {
															#line 679 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 218 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	ReadFileContentActorState(std::string const& path) 
															#line 218 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 218 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path),
															#line 219 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   exists(fileExists(path))
															#line 688 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("readFileContent", reinterpret_cast<unsigned long>(this));

	}
	~ReadFileContentActorState() 
	{
		fdb_probe_actor_destroy("readFileContent", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!exists)
															#line 704 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				{
															#line 223 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(std::string()); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 708 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::string());
					this->~ReadFileContentActorState();
					static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 226 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( path, IAsyncFile::OPEN_READONLY | IAsyncFile::OPEN_UNCACHED | IAsyncFile::OPEN_NO_AIO, 0644);
															#line 226 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 718 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ReadFileContentActor*>(this)->actor_wait_state = 1;
															#line 226 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*>(static_cast<ReadFileContentActor*>(this)));
															#line 723 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 244 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadException").error(e).detail("Path", path);
															#line 245 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(std::string()); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 755 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::string());
			this->~ReadFileContentActorState();
			static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 228 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<int64_t> __when_expr_1 = file->size();
															#line 228 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 775 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadFileContentActor*>(this)->actor_wait_state = 2;
															#line 228 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 1, int64_t >*>(static_cast<ReadFileContentActor*>(this)));
															#line 780 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& __file,int loopDepth) 
	{
															#line 226 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = __file;
															#line 789 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && __file,int loopDepth) 
	{
		file = std::move(__file);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 230 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content = std::string();
															#line 231 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content.resize(fileSize);
															#line 232 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<int> __when_expr_2 = file->read((uint8_t*)content.data(), fileSize, 0);
															#line 232 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 874 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReadFileContentActor*>(this)->actor_wait_state = 3;
															#line 232 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 2, int >*>(static_cast<ReadFileContentActor*>(this)));
															#line 879 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(int64_t const& __fileSize,int loopDepth) 
	{
															#line 228 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileSize = __fileSize;
															#line 888 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(int64_t && __fileSize,int loopDepth) 
	{
		fileSize = std::move(__fileSize);
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 1, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 1, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 1, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 1, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int const& bytesRead,int loopDepth) 
	{
															#line 233 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 235 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bytesRead != fileSize)
															#line 969 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
															#line 236 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadShort") .detail("Path", path) .detail("Expected", fileSize) .detail("Actual", bytesRead);
															#line 973 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		}
															#line 242 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(content); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 977 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::move(content)); // state_var_RVO
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(int && bytesRead,int loopDepth) 
	{
															#line 233 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 235 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bytesRead != fileSize)
															#line 991 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
															#line 236 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadShort") .detail("Path", path) .detail("Expected", fileSize) .detail("Actual", bytesRead);
															#line 995 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		}
															#line 242 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(content); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 999 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::move(content)); // state_var_RVO
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4when1(int const& bytesRead,int loopDepth) 
	{
		loopDepth = a_body1cont5(bytesRead, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(int && bytesRead,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(bytesRead), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 2, int >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 2, int >*,int const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 2, int >*,int && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 2, int >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 218 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 219 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	bool exists;
															#line 226 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<IAsyncFile> file;
															#line 228 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int64_t fileSize;
															#line 230 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 1092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via readFileContent()
															#line 218 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class ReadFileContentActor final : public Actor<std::string>, public ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >, public ActorCallback< ReadFileContentActor, 1, int64_t >, public ActorCallback< ReadFileContentActor, 2, int >, public FastAllocated<ReadFileContentActor>, public ReadFileContentActorState<ReadFileContentActor> {
															#line 1097 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<ReadFileContentActor>::operator new;
	using FastAllocated<ReadFileContentActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17720353732269189888UL, 17034817112521762816UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::string>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< ReadFileContentActor, 1, int64_t >;
friend struct ActorCallback< ReadFileContentActor, 2, int >;
															#line 218 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	ReadFileContentActor(std::string const& path) 
															#line 1116 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<std::string>(),
		   ReadFileContentActorState<ReadFileContentActor>(path),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7623102229524567296UL, 7126613595367428096UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readFileContent");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadFileContentActor, 1, int64_t >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadFileContentActor, 2, int >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 218 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<std::string> readFileContent( std::string const& path ) {
															#line 218 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<std::string>(new ReadFileContentActor(path));
															#line 1151 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 248 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete file using simulation filesystem
// Wraps deleteFile with trace events and error handling for MockS3 persistence cleanup
															#line 1158 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via deletePersistedFile()
															#line 251 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedFileActor>
															#line 251 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedFileActorState {
															#line 1165 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 251 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedFileActorState(std::string const& path) 
															#line 251 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 251 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path)
															#line 1172 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("deletePersistedFile", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedFileActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 253 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = IAsyncFileSystem::filesystem()->deleteFile(path, true);
															#line 253 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1190 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedFileActor*>(this)->actor_wait_state = 1;
															#line 253 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedFileActor, 0, Void >*>(static_cast<DeletePersistedFileActor*>(this)));
															#line 1195 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedFileActorState();
		static_cast<DeletePersistedFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 258 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedFileActorState(); static_cast<DeletePersistedFileActor*>(this)->destroy(); return 0; }
															#line 1224 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<DeletePersistedFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedFileActorState();
		static_cast<DeletePersistedFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 256 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceDeleteException").error(e).detail("Path", path);
															#line 1237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 254 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceDelete").detail("Path", path);
															#line 1252 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 254 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceDelete").detail("Path", path);
															#line 1261 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedFileActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedFileActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedFileActor*>(this)->ActorCallback< DeletePersistedFileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedFileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedFileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedFileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 251 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 1356 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via deletePersistedFile()
															#line 251 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedFileActor final : public Actor<Void>, public ActorCallback< DeletePersistedFileActor, 0, Void >, public FastAllocated<DeletePersistedFileActor>, public DeletePersistedFileActorState<DeletePersistedFileActor> {
															#line 1361 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<DeletePersistedFileActor>::operator new;
	using FastAllocated<DeletePersistedFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11377622672091701248UL, 14817211112060854784UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedFileActor, 0, Void >;
															#line 251 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedFileActor(std::string const& path) 
															#line 1378 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   DeletePersistedFileActorState<DeletePersistedFileActor>(path),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8878611606913210624UL, 16427746229923233536UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedFileActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 251 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedFile( std::string const& path ) {
															#line 251 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedFileActor(path));
															#line 1411 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 260 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// JSON Serialization using rapidjson
static std::string serializeObjectMeta(const MockS3GlobalStorage::ObjectData& obj) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();

	doc.AddMember("etag", Value(obj.etag.c_str(), allocator), allocator);
	doc.AddMember("lastModified", obj.lastModified, allocator);

	Value tagsObj(kObjectType);
	for (const auto& tag : obj.tags) {
		tagsObj.AddMember(Value(tag.first.c_str(), allocator), Value(tag.second.c_str(), allocator), allocator);
	}
	doc.AddMember("tags", tagsObj, allocator);

	StringBuffer buffer;
	PrettyWriter<StringBuffer> writer(buffer);
	doc.Accept(writer);
	return buffer.GetString();
}

// JSON Deserialization using rapidjson
static void deserializeObjectMeta(const std::string& jsonStr, MockS3GlobalStorage::ObjectData& obj) {
	using namespace rapidjson;
	Document doc;
	doc.Parse(jsonStr.c_str());

	if (doc.HasMember("etag") && doc["etag"].IsString())
		obj.etag = doc["etag"].GetString();
	if (doc.HasMember("lastModified") && doc["lastModified"].IsNumber())
		obj.lastModified = doc["lastModified"].GetDouble();
	if (doc.HasMember("tags") && doc["tags"].IsObject()) {
		for (auto& m : doc["tags"].GetObject()) {
			if (m.value.IsString())
				obj.tags[m.name.GetString()] = m.value.GetString();
		}
	}
}

static std::string serializeMultipartState(const MockS3GlobalStorage::MultipartUpload& upload) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();

	doc.AddMember("uploadId", Value(upload.uploadId.c_str(), allocator), allocator);
	doc.AddMember("bucket", Value(upload.bucket.c_str(), allocator), allocator);
	doc.AddMember("object", Value(upload.object.c_str(), allocator), allocator);
	doc.AddMember("initiated", upload.initiated, allocator);

	Value partsArray(kArrayType);
	for (const auto& part : upload.parts) {
		Value partObj(kObjectType);
		partObj.AddMember("partNum", part.first, allocator);
		partObj.AddMember("etag", Value(part.second.first.c_str(), allocator), allocator);
		partsArray.PushBack(partObj, allocator);
	}
	doc.AddMember("parts", partsArray, allocator);

	StringBuffer buffer;
	PrettyWriter<StringBuffer> writer(buffer);
	doc.Accept(writer);
	return buffer.GetString();
}

static void deserializeMultipartState(const std::string& jsonStr, MockS3GlobalStorage::MultipartUpload& upload) {
	using namespace rapidjson;
	Document doc;
	doc.Parse(jsonStr.c_str());

	if (doc.HasMember("uploadId") && doc["uploadId"].IsString())
		upload.uploadId = doc["uploadId"].GetString();
	if (doc.HasMember("bucket") && doc["bucket"].IsString())
		upload.bucket = doc["bucket"].GetString();
	if (doc.HasMember("object") && doc["object"].IsString())
		upload.object = doc["object"].GetString();
	if (doc.HasMember("initiated") && doc["initiated"].IsNumber())
		upload.initiated = doc["initiated"].GetDouble();
	if (doc.HasMember("parts") && doc["parts"].IsArray()) {
		for (auto& partVal : doc["parts"].GetArray()) {
			if (partVal.HasMember("partNum") && partVal["partNum"].IsInt() && partVal.HasMember("etag") &&
			    partVal["etag"].IsString()) {
				int partNum = partVal["partNum"].GetInt();
				std::string etag = partVal["etag"].GetString();
				upload.parts[partNum] = { etag, "" }; // Content loaded separately from .part.N files
			}
		}
	}
}

// Forward declarations for state loading functions
															#line 1508 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
[[nodiscard]] static Future<Void> loadPersistedObjects( std::string const& persistenceDir );

#line 354 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1512 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
[[nodiscard]] static Future<Void> loadPersistedMultipartUploads( std::string const& persistenceDir );

#line 355 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1516 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
[[nodiscard]] static Future<Void> loadMockS3PersistedStateImpl(  );

#line 356 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
Future<Void> loadMockS3PersistedStateFuture();

static std::string serializePartMeta(const std::string& etag) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();
	doc.AddMember("etag", Value(etag.c_str(), allocator), allocator);

	StringBuffer buffer;
	Writer<StringBuffer> writer(buffer); // Use Writer instead of PrettyWriter for compact output
	doc.Accept(writer);
	return buffer.GetString();
}

// ACTOR: Persist object data and metadata
															#line 1536 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via persistObject()
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class PersistObjectActor>
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistObjectActorState {
															#line 1543 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistObjectActorState(std::string const& bucket,std::string const& object) 
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : bucket(bucket),
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 1552 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("persistObject", reinterpret_cast<unsigned long>(this));

	}
	~PersistObjectActorState() 
	{
		fdb_probe_actor_destroy("persistObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 373 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 376 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!storage.persistenceEnabled && g_network->isSimulated())
															#line 1569 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 377 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				storage.enablePersistence(DEFAULT_MOCKS3_PERSISTENCE_DIR);
															#line 1573 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			}
															#line 381 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!storage.persistenceEnabled)
															#line 1577 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 382 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1581 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistObjectActorState();
				static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 385 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto bucketIter = storage.buckets.find(bucket);
															#line 386 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (bucketIter == storage.buckets.end())
															#line 1591 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 387 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistObjectActorState();
				static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 390 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto objectIter = bucketIter->second.find(object);
															#line 391 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (objectIter == bucketIter->second.end())
															#line 1605 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 392 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1609 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistObjectActorState();
				static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 396 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			content = objectIter->second.content;
															#line 397 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaJson = serializeObjectMeta(objectIter->second);
															#line 1619 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			try {
															#line 402 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				dataPath = storage.getObjectDataPath(bucket, object);
															#line 403 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				metaPath = storage.getObjectMetaPath(bucket, object);
															#line 407 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (fileExists(dataPath) || fileExists(dataPath + ".part"))
															#line 1627 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				{
															#line 408 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					TraceEvent("MockS3ObjectAlreadyPersisted") .detail("Bucket", bucket) .detail("Object", object) .detail("DataPath", dataPath);
															#line 412 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1633 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~PersistObjectActorState();
					static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 415 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3PersistingObject") .detail("Bucket", bucket) .detail("Object", object) .detail("DataPath", dataPath) .detail("Size", content.size());
															#line 422 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = atomicWriteFile(dataPath, content);
															#line 422 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1645 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PersistObjectActor*>(this)->actor_wait_state = 1;
															#line 422 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 0, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1650 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistObjectActorState();
		static_cast<PersistObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 435 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1679 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistObjectActorState();
		static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 432 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3PersistObjectFailed").error(e).detail("Bucket", bucket).detail("Object", object);
															#line 1692 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 425 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(metaPath, metaJson);
															#line 425 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1709 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont6when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistObjectActor*>(this)->actor_wait_state = 2;
															#line 425 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 1, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1714 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 425 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(metaPath, metaJson);
															#line 425 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1725 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont6when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistObjectActor*>(this)->actor_wait_state = 2;
															#line 425 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 1, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1730 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state > 0) static_cast<PersistObjectActor*>(this)->actor_wait_state = 0;
		static_cast<PersistObjectActor*>(this)->ActorCallback< PersistObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
															#line 427 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectPersisted") .detail("Bucket", bucket) .detail("Object", object) .detail("Size", content.size());
															#line 1814 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
															#line 427 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectPersisted") .detail("Bucket", bucket) .detail("Object", object) .detail("Size", content.size());
															#line 1823 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state > 0) static_cast<PersistObjectActor*>(this)->actor_wait_state = 0;
		static_cast<PersistObjectActor*>(this)->ActorCallback< PersistObjectActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistObjectActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 396 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 397 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaJson;
															#line 402 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 403 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 1928 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via persistObject()
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistObjectActor final : public Actor<Void>, public ActorCallback< PersistObjectActor, 0, Void >, public ActorCallback< PersistObjectActor, 1, Void >, public FastAllocated<PersistObjectActor>, public PersistObjectActorState<PersistObjectActor> {
															#line 1933 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<PersistObjectActor>::operator new;
	using FastAllocated<PersistObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2014277493725906432UL, 788613063876194560UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistObjectActor, 0, Void >;
friend struct ActorCallback< PersistObjectActor, 1, Void >;
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistObjectActor(std::string const& bucket,std::string const& object) 
															#line 1951 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   PersistObjectActorState<PersistObjectActor>(bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1247695763033469440UL, 1096140585632004864UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistObjectActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistObjectActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> persistObject( std::string const& bucket, std::string const& object ) {
															#line 372 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new PersistObjectActor(bucket, object));
															#line 1985 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 437 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Persist multipart upload state
															#line 1991 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via persistMultipartState()
															#line 439 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class PersistMultipartStateActor>
															#line 439 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistMultipartStateActorState {
															#line 1998 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 439 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistMultipartStateActorState(std::string const& uploadId) 
															#line 439 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 439 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : uploadId(uploadId),
															#line 440 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(),
															#line 441 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   parts()
															#line 2009 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("persistMultipartState", reinterpret_cast<unsigned long>(this));

	}
	~PersistMultipartStateActorState() 
	{
		fdb_probe_actor_destroy("persistMultipartState", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 443 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 446 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!storage.persistenceEnabled && g_network->isSimulated())
															#line 2026 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 447 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				storage.enablePersistence(DEFAULT_MOCKS3_PERSISTENCE_DIR);
															#line 2030 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			}
															#line 451 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!storage.persistenceEnabled)
															#line 2034 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 452 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2038 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistMultipartStateActorState();
				static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 455 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = storage.multipartUploads.find(uploadId);
															#line 456 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == storage.multipartUploads.end())
															#line 2048 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 457 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2052 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistMultipartStateActorState();
				static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 460 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			const auto& upload = uploadIter->second;
															#line 461 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			persistenceDir = storage.persistenceDir;
															#line 462 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			parts = upload.parts;
															#line 2064 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			try {
															#line 467 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string statePath = persistenceDir + "/multipart/" + uploadId + ".state.json";
															#line 471 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (fileExists(statePath) || fileExists(statePath + ".part"))
															#line 2070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				{
															#line 472 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					TraceEvent("MockS3MultipartAlreadyPersisted").detail("UploadId", uploadId).detail("StatePath", statePath);
															#line 473 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2076 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~PersistMultipartStateActorState();
					static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 476 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string stateJson = serializeMultipartState(upload);
															#line 477 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = atomicWriteFile(statePath, stateJson);
															#line 477 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2088 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 1;
															#line 477 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 0, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2093 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistMultipartStateActorState();
		static_cast<PersistMultipartStateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 501 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2122 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistMultipartStateActorState();
		static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 498 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistMultipartFailed").error(e).detail("UploadId", uploadId);
															#line 2135 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 480 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter = parts.begin();
															#line 481 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 2152 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 480 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter = parts.begin();
															#line 481 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 2163 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 496 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartPersisted").detail("UploadId", uploadId).detail("PartsCount", parts.size());
															#line 2247 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont5loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1(int loopDepth) 
	{
															#line 481 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partIter != parts.end()))
															#line 2263 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
			return a_body1cont5break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 482 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = partIter->first;
															#line 483 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		etag = partIter->second.first;
															#line 484 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partData = partIter->second.second;
															#line 486 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
															#line 487 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(partPath, partData);
															#line 487 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2279 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont5loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 2;
															#line 487 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 1, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2284 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5break1(int loopDepth) 
	{
		try {
			return a_body1cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 489 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 490 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = serializePartMeta(etag);
															#line 491 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = atomicWriteFile(partMetaPath, partMetaJson);
															#line 491 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2312 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont5loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 3;
															#line 491 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 2, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2317 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 489 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 490 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = serializePartMeta(etag);
															#line 491 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = atomicWriteFile(partMetaPath, partMetaJson);
															#line 491 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2332 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont5loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 3;
															#line 491 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 2, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2337 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont5loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont5loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 493 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter++;
															#line 2421 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont5loopHead1(0);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 493 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter++;
															#line 2430 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont5loopHead1(0);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 439 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 440 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 441 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<int, std::pair<std::string, std::string>> parts;
															#line 480 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<int, std::pair<std::string, std::string>>::iterator partIter;
															#line 482 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 483 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 484 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partData;
															#line 486 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 489 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 490 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaJson;
															#line 2543 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via persistMultipartState()
															#line 439 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistMultipartStateActor final : public Actor<Void>, public ActorCallback< PersistMultipartStateActor, 0, Void >, public ActorCallback< PersistMultipartStateActor, 1, Void >, public ActorCallback< PersistMultipartStateActor, 2, Void >, public FastAllocated<PersistMultipartStateActor>, public PersistMultipartStateActorState<PersistMultipartStateActor> {
															#line 2548 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<PersistMultipartStateActor>::operator new;
	using FastAllocated<PersistMultipartStateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(745669800516219648UL, 9643976065052025088UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistMultipartStateActor, 0, Void >;
friend struct ActorCallback< PersistMultipartStateActor, 1, Void >;
friend struct ActorCallback< PersistMultipartStateActor, 2, Void >;
															#line 439 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistMultipartStateActor(std::string const& uploadId) 
															#line 2567 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   PersistMultipartStateActorState<PersistMultipartStateActor>(uploadId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9066204211401481728UL, 7216261254142226944UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistMultipartState");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 439 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> persistMultipartState( std::string const& uploadId ) {
															#line 439 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new PersistMultipartStateActor(uploadId));
															#line 2602 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 503 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete persisted object
															#line 2608 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via deletePersistedObject()
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedObjectActor>
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedObjectActorState {
															#line 2615 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedObjectActorState(std::string const& bucket,std::string const& object) 
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : bucket(bucket),
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 506 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   dataPath(),
															#line 507 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   metaPath()
															#line 2628 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("deletePersistedObject", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedObjectActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 509 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 510 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 512 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			dataPath = storage.getObjectDataPath(bucket, object);
															#line 513 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaPath = storage.getObjectMetaPath(bucket, object);
															#line 2649 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			try {
															#line 516 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedFile(dataPath);
															#line 516 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2655 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 1;
															#line 516 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 0, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2660 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedObjectActorState();
		static_cast<DeletePersistedObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 527 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedObjectActorState(); static_cast<DeletePersistedObjectActor*>(this)->destroy(); return 0; }
															#line 2689 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<DeletePersistedObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedObjectActorState();
		static_cast<DeletePersistedObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 521 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3DeletePersistedObjectFailed") .error(e) .detail("Bucket", bucket) .detail("Object", object);
															#line 2702 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 517 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(metaPath);
															#line 517 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2719 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 2;
															#line 517 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 1, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2724 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 517 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(metaPath);
															#line 517 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2735 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 2;
															#line 517 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 1, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2740 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedObjectActor*>(this)->ActorCallback< DeletePersistedObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 519 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 2824 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 519 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 2833 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedObjectActor*>(this)->ActorCallback< DeletePersistedObjectActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 506 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 507 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 2934 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via deletePersistedObject()
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedObjectActor final : public Actor<Void>, public ActorCallback< DeletePersistedObjectActor, 0, Void >, public ActorCallback< DeletePersistedObjectActor, 1, Void >, public FastAllocated<DeletePersistedObjectActor>, public DeletePersistedObjectActorState<DeletePersistedObjectActor> {
															#line 2939 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<DeletePersistedObjectActor>::operator new;
	using FastAllocated<DeletePersistedObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17001562103346210560UL, 8699930955194233088UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedObjectActor, 0, Void >;
friend struct ActorCallback< DeletePersistedObjectActor, 1, Void >;
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedObjectActor(std::string const& bucket,std::string const& object) 
															#line 2957 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   DeletePersistedObjectActorState<DeletePersistedObjectActor>(bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5402173951988437248UL, 4607022702540543488UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedObjectActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeletePersistedObjectActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedObject( std::string const& bucket, std::string const& object ) {
															#line 505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedObjectActor(bucket, object));
															#line 2991 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 529 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete persisted multipart upload
															#line 2997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via deletePersistedMultipart()
															#line 531 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedMultipartActor>
															#line 531 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedMultipartActorState {
															#line 3004 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 531 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedMultipartActorState(std::string const& uploadId) 
															#line 531 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 531 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : uploadId(uploadId),
															#line 532 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   maxPart(),
															#line 533 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(),
															#line 534 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partNum(),
															#line 535 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partPath(),
															#line 536 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partMetaPath()
															#line 3021 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("deletePersistedMultipart", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedMultipartActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedMultipart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 538 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 539 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 3038 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			try {
															#line 543 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				auto uploadIter = storage.multipartUploads.find(uploadId);
															#line 544 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				maxPart = 100;
															#line 545 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (uploadIter != storage.multipartUploads.end())
															#line 3046 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				{
															#line 546 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					for( const auto& part : uploadIter->second.parts ) {
															#line 547 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						maxPart = std::max(maxPart, part.first);
															#line 3052 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					}
				}
															#line 552 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				persistenceDir = storage.persistenceDir;
															#line 555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string statePath = persistenceDir + "/multipart/" + uploadId + ".state.json";
															#line 556 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedFile(statePath);
															#line 556 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3063 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 1;
															#line 556 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 0, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3068 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedMultipartActorState();
		static_cast<DeletePersistedMultipartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 580 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedMultipartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedMultipartActorState(); static_cast<DeletePersistedMultipartActor*>(this)->destroy(); return 0; }
															#line 3097 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<DeletePersistedMultipartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedMultipartActorState();
		static_cast<DeletePersistedMultipartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 577 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3DeletePersistedMultipartFailed").error(e).detail("UploadId", uploadId);
															#line 3110 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 560 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 561 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 3127 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 560 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 561 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 3138 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 575 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartDeleted").detail("UploadId", uploadId);
															#line 3222 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 561 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partNum <= maxPart + 10))
															#line 3238 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
			return a_body1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 562 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
															#line 563 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 564 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(partPath);
															#line 564 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3250 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 2;
															#line 564 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 1, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3255 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 565 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = deletePersistedFile(partMetaPath);
															#line 565 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3279 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 3;
															#line 565 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 2, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3284 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 565 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = deletePersistedFile(partMetaPath);
															#line 565 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3295 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 3;
															#line 565 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 2, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3300 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 566 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 570 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (partNum % 2 == 0)
															#line 3386 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
															#line 571 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0.02);
															#line 571 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3392 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 4;
															#line 571 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 3, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3397 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 566 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 570 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (partNum % 2 == 0)
															#line 3413 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
															#line 571 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0.02);
															#line 571 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3419 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 4;
															#line 571 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 3, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3424 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2loopBody1cont4(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 531 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 532 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int maxPart;
															#line 533 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 534 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 535 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 536 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 3627 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via deletePersistedMultipart()
															#line 531 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedMultipartActor final : public Actor<Void>, public ActorCallback< DeletePersistedMultipartActor, 0, Void >, public ActorCallback< DeletePersistedMultipartActor, 1, Void >, public ActorCallback< DeletePersistedMultipartActor, 2, Void >, public ActorCallback< DeletePersistedMultipartActor, 3, Void >, public FastAllocated<DeletePersistedMultipartActor>, public DeletePersistedMultipartActorState<DeletePersistedMultipartActor> {
															#line 3632 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<DeletePersistedMultipartActor>::operator new;
	using FastAllocated<DeletePersistedMultipartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10546240417433320704UL, 9831948138691496960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedMultipartActor, 0, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 1, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 2, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 3, Void >;
															#line 531 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedMultipartActor(std::string const& uploadId) 
															#line 3652 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   DeletePersistedMultipartActorState<DeletePersistedMultipartActor>(uploadId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15695025802244908800UL, 1209722574274508288UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedMultipart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 531 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedMultipart( std::string const& uploadId ) {
															#line 531 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedMultipartActor(uploadId));
															#line 3688 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 582 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Mock S3 Server Implementation for deterministic testing
class MockS3ServerImpl {
public:
	using ObjectData = MockS3GlobalStorage::ObjectData;
	using MultipartUpload = MockS3GlobalStorage::MultipartUpload;

	// Note: No TraceEvent here - constructor called before GlobalConfig is initialized
	MockS3ServerImpl() {}

	~MockS3ServerImpl() {}

	// S3 Operation Handlers
																#line 3705 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
// This generated class is to be used only via handleRequest()
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleRequestActor>
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleRequestActorState {
															#line 3711 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleRequestActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response) 
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response)
															#line 3722 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("handleRequest", reinterpret_cast<unsigned long>(this));

	}
	~HandleRequestActorState() 
	{
		fdb_probe_actor_destroy("handleRequest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 599 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3Request") .detail("Method", req->verb) .detail("Resource", req->resource) .detail("ContentLength", req->data.contentLen) .detail("Headers", req->data.headers.size()) .detail("UserAgent", req->data.headers.find("User-Agent") != req->data.headers.end() ? req->data.headers.at("User-Agent") : "N/A") .detail("Host", req->data.headers.find("Host") != req->data.headers.end() ? req->data.headers.at("Host") : "N/A");
															#line 3737 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			try {
															#line 612 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string bucket, object;
															#line 613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::map<std::string, std::string> queryParams;
															#line 614 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->parseS3Request(req->resource, bucket, object, queryParams);
															#line 616 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ParsedRequest") .detail("Bucket", bucket) .detail("Object", object) .detail("QueryParamCount", queryParams.size());
															#line 622 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (queryParams.count("uploads"))
															#line 3749 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				{
															#line 623 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					StrictFuture<Void> __when_expr_0 = self->handleMultipartStart(self, req, response, bucket, object);
															#line 623 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3755 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<HandleRequestActor*>(this)->actor_wait_state = 1;
															#line 623 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 0, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3760 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					loopDepth = 0;
				}
				else
				{
															#line 624 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (queryParams.count("uploadId"))
															#line 3767 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					{
															#line 625 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						if (queryParams.count("partNumber"))
															#line 3771 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
						{
															#line 626 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							StrictFuture<Void> __when_expr_1 = self->handleUploadPart(self, req, response, bucket, object, queryParams);
															#line 626 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3777 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
							if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
							static_cast<HandleRequestActor*>(this)->actor_wait_state = 2;
															#line 626 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 1, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3782 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
							loopDepth = 0;
						}
						else
						{
															#line 627 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (req->verb == "POST")
															#line 3789 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
							{
															#line 628 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_2 = self->handleMultipartComplete(self, req, response, bucket, object, queryParams);
															#line 628 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3795 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
								if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1when3(__when_expr_2.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 3;
															#line 628 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 2, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3800 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
								loopDepth = 0;
							}
							else
							{
															#line 629 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (req->verb == "DELETE")
															#line 3807 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
								{
															#line 630 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									StrictFuture<Void> __when_expr_3 = self->handleMultipartAbort(self, req, response, bucket, object, queryParams);
															#line 630 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3813 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
									if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1when4(__when_expr_3.get(), loopDepth); };
									static_cast<HandleRequestActor*>(this)->actor_wait_state = 4;
															#line 630 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 3, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3818 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
									loopDepth = 0;
								}
								else
								{
															#line 632 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									self->sendError( response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "InvalidRequest", "Unknown multipart operation");
															#line 3825 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
									loopDepth = a_body1cont9(loopDepth);
								}
							}
						}
					}
					else
					{
															#line 635 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						if (queryParams.count("tagging"))
															#line 3835 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
						{
															#line 636 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (req->verb == "PUT")
															#line 3839 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
							{
															#line 637 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_4 = self->handlePutObjectTags(self, req, response, bucket, object);
															#line 637 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3845 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
								if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), loopDepth); else return a_body1when5(__when_expr_4.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 5;
															#line 637 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 4, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3850 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
								loopDepth = 0;
							}
							else
							{
															#line 638 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (req->verb == "GET")
															#line 3857 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
								{
															#line 639 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									StrictFuture<Void> __when_expr_5 = self->handleGetObjectTags(self, req, response, bucket, object);
															#line 639 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3863 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
									if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch2(__when_expr_5.getError(), loopDepth); else return a_body1when6(__when_expr_5.get(), loopDepth); };
									static_cast<HandleRequestActor*>(this)->actor_wait_state = 6;
															#line 639 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 5, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3868 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
									loopDepth = 0;
								}
								else
								{
															#line 641 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									self->sendError(response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Method not allowed for tagging");
															#line 3875 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
									loopDepth = a_body1cont15(loopDepth);
								}
							}
						}
						else
						{
															#line 646 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (queryParams.count("list-type") || (req->verb == "GET" && object.empty()))
															#line 3884 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
							{
															#line 648 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_6 = self->handleListObjects(self, req, response, bucket, queryParams);
															#line 648 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3890 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
								if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch2(__when_expr_6.getError(), loopDepth); else return a_body1when7(__when_expr_6.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 7;
															#line 648 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 6, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3895 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
								loopDepth = 0;
							}
							else
							{
															#line 649 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (object.empty())
															#line 3902 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
								{
															#line 651 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (req->verb == "HEAD")
															#line 3906 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
									{
															#line 652 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										StrictFuture<Void> __when_expr_7 = self->handleHeadBucket(self, req, response, bucket);
															#line 652 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3912 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
										if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch2(__when_expr_7.getError(), loopDepth); else return a_body1when8(__when_expr_7.get(), loopDepth); };
										static_cast<HandleRequestActor*>(this)->actor_wait_state = 8;
															#line 652 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 7, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3917 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
										loopDepth = 0;
									}
									else
									{
															#line 653 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (req->verb == "PUT")
															#line 3924 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
										{
															#line 654 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											StrictFuture<Void> __when_expr_8 = self->handlePutBucket(self, req, response, bucket);
															#line 654 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3930 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
											if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch2(__when_expr_8.getError(), loopDepth); else return a_body1when9(__when_expr_8.get(), loopDepth); };
											static_cast<HandleRequestActor*>(this)->actor_wait_state = 9;
															#line 654 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 8, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3935 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
											loopDepth = 0;
										}
										else
										{
															#line 656 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											self->sendError(response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Bucket operation not supported");
															#line 3942 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
											loopDepth = a_body1cont23(loopDepth);
										}
									}
								}
								else
								{
															#line 663 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (req->verb == "PUT")
															#line 3951 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
									{
															#line 664 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										StrictFuture<Void> __when_expr_9 = self->handlePutObject(self, req, response, bucket, object);
															#line 664 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3957 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
										if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch2(__when_expr_9.getError(), loopDepth); else return a_body1when10(__when_expr_9.get(), loopDepth); };
										static_cast<HandleRequestActor*>(this)->actor_wait_state = 10;
															#line 664 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 9, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3962 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
										loopDepth = 0;
									}
									else
									{
															#line 665 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (req->verb == "GET")
															#line 3969 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
										{
															#line 666 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											StrictFuture<Void> __when_expr_10 = self->handleGetObject(self, req, response, bucket, object);
															#line 666 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3975 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
											if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch2(__when_expr_10.getError(), loopDepth); else return a_body1when11(__when_expr_10.get(), loopDepth); };
											static_cast<HandleRequestActor*>(this)->actor_wait_state = 11;
															#line 666 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 10, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3980 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
											loopDepth = 0;
										}
										else
										{
															#line 667 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (req->verb == "DELETE")
															#line 3987 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
											{
															#line 668 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												StrictFuture<Void> __when_expr_11 = self->handleDeleteObject(self, req, response, bucket, object);
															#line 668 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3993 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
												if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1Catch2(__when_expr_11.getError(), loopDepth); else return a_body1when12(__when_expr_11.get(), loopDepth); };
												static_cast<HandleRequestActor*>(this)->actor_wait_state = 12;
															#line 668 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 11, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3998 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
												loopDepth = 0;
											}
											else
											{
															#line 669 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												if (req->verb == "HEAD")
															#line 4005 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
												{
															#line 670 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													StrictFuture<Void> __when_expr_12 = self->handleHeadObject(self, req, response, bucket, object);
															#line 670 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4011 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
													if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1Catch2(__when_expr_12.getError(), loopDepth); else return a_body1when13(__when_expr_12.get(), loopDepth); };
													static_cast<HandleRequestActor*>(this)->actor_wait_state = 13;
															#line 670 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 12, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 4016 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
													loopDepth = 0;
												}
												else
												{
															#line 672 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													self->sendError( response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Method not supported");
															#line 4023 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
													loopDepth = a_body1cont32(loopDepth);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleRequestActorState();
		static_cast<HandleRequestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 682 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleRequestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleRequestActorState(); static_cast<HandleRequestActor*>(this)->destroy(); return 0; }
															#line 4061 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<HandleRequestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleRequestActorState();
		static_cast<HandleRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 678 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3RequestError").error(e).detail("Resource", req->resource);
															#line 679 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			self->sendError(response, 500, "InternalError", "Internal server error");
															#line 4076 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont35(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1when4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1when4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont12(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont14(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont14(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1when5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14(_, loopDepth);

		return loopDepth;
	}
	int a_body1when5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont15(int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1when6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont16(_, loopDepth);

		return loopDepth;
	}
	int a_body1when6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont16(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1when6(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont18(int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont19(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont19(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1when7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont19(_, loopDepth);

		return loopDepth;
	}
	int a_body1when7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont19(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1when7(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont20(int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont21(int loopDepth) 
	{
		loopDepth = a_body1cont20(loopDepth);

		return loopDepth;
	}
	int a_body1cont22(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1cont22(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1when8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont22(_, loopDepth);

		return loopDepth;
	}
	int a_body1when8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont22(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1when8(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1when8(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont23(int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1cont24(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont23(loopDepth);

		return loopDepth;
	}
	int a_body1cont24(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont23(loopDepth);

		return loopDepth;
	}
	int a_body1when9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont24(_, loopDepth);

		return loopDepth;
	}
	int a_body1when9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont24(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1when9(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1when9(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont26(int loopDepth) 
	{
		loopDepth = a_body1cont20(loopDepth);

		return loopDepth;
	}
	int a_body1cont27(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1cont27(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1when10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont27(_, loopDepth);

		return loopDepth;
	}
	int a_body1when10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont27(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1when10(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1when10(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont28(int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1cont29(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1cont29(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1when11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont29(_, loopDepth);

		return loopDepth;
	}
	int a_body1when11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont29(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1when11(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1when11(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont30(int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1cont31(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont31(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1when12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont31(_, loopDepth);

		return loopDepth;
	}
	int a_body1when12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont31(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1when12(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1when12(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont32(int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont33(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont32(loopDepth);

		return loopDepth;
	}
	int a_body1cont33(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont32(loopDepth);

		return loopDepth;
	}
	int a_body1when13(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont33(_, loopDepth);

		return loopDepth;
	}
	int a_body1when13(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont33(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1when13(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1when13(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	int a_body1cont35(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 5333 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via handleRequest()
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleRequestActor final : public Actor<Void>, public ActorCallback< HandleRequestActor, 0, Void >, public ActorCallback< HandleRequestActor, 1, Void >, public ActorCallback< HandleRequestActor, 2, Void >, public ActorCallback< HandleRequestActor, 3, Void >, public ActorCallback< HandleRequestActor, 4, Void >, public ActorCallback< HandleRequestActor, 5, Void >, public ActorCallback< HandleRequestActor, 6, Void >, public ActorCallback< HandleRequestActor, 7, Void >, public ActorCallback< HandleRequestActor, 8, Void >, public ActorCallback< HandleRequestActor, 9, Void >, public ActorCallback< HandleRequestActor, 10, Void >, public ActorCallback< HandleRequestActor, 11, Void >, public ActorCallback< HandleRequestActor, 12, Void >, public FastAllocated<HandleRequestActor>, public HandleRequestActorState<HandleRequestActor> {
															#line 5338 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<HandleRequestActor>::operator new;
	using FastAllocated<HandleRequestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1553703081749143552UL, 751938499498529280UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleRequestActor, 0, Void >;
friend struct ActorCallback< HandleRequestActor, 1, Void >;
friend struct ActorCallback< HandleRequestActor, 2, Void >;
friend struct ActorCallback< HandleRequestActor, 3, Void >;
friend struct ActorCallback< HandleRequestActor, 4, Void >;
friend struct ActorCallback< HandleRequestActor, 5, Void >;
friend struct ActorCallback< HandleRequestActor, 6, Void >;
friend struct ActorCallback< HandleRequestActor, 7, Void >;
friend struct ActorCallback< HandleRequestActor, 8, Void >;
friend struct ActorCallback< HandleRequestActor, 9, Void >;
friend struct ActorCallback< HandleRequestActor, 10, Void >;
friend struct ActorCallback< HandleRequestActor, 11, Void >;
friend struct ActorCallback< HandleRequestActor, 12, Void >;
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleRequestActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response) 
															#line 5367 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   HandleRequestActorState<HandleRequestActor>(self, req, response),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3085149730912622080UL, 127431907239638784UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleRequest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleRequestActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< HandleRequestActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< HandleRequestActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< HandleRequestActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< HandleRequestActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< HandleRequestActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< HandleRequestActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< HandleRequestActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< HandleRequestActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< HandleRequestActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< HandleRequestActor, 10, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< HandleRequestActor, 11, Void >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< HandleRequestActor, 12, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleRequest( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response ) {
															#line 595 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleRequestActor(self, req, response));
															#line 5411 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 684 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	void parseS3Request(const std::string& resource,
	                    std::string& bucket,
	                    std::string& object,
	                    std::map<std::string, std::string>& queryParams) {

		// Split resource into path and query string
		size_t queryPos = resource.find('?');
		std::string path = (queryPos != std::string::npos) ? resource.substr(0, queryPos) : resource;
		std::string query = (queryPos != std::string::npos) ? resource.substr(queryPos + 1) : "";

		// Parse path: /bucket/object (like real S3)
		if (path.size() > 1) {
			path = path.substr(1); // Remove leading /
			size_t slashPos = path.find('/');
			if (slashPos != std::string::npos) {
				bucket = path.substr(0, slashPos);
				object = path.substr(slashPos + 1);
			} else {
				bucket = path;
				object = "";
			}
		}

		// Parse query parameters
		if (!query.empty()) {
			std::regex paramRegex("([^&=]+)=?([^&]*)");
			std::sregex_iterator iter(query.begin(), query.end(), paramRegex);
			std::sregex_iterator end;

			for (; iter != end; ++iter) {
				std::string key = iter->str(1);
				std::string value = iter->str(2);
				// URL decode the parameter value
				queryParams[key] = HTTP::urlDecode(value);
			}
		}

		// MockS3Server handles S3 HTTP requests where bucket is always the first path component
		// For bucket operations: HEAD /bucket_name
		// For object operations: HEAD /bucket_name/object_path
		if (bucket.empty()) {
			TraceEvent(SevWarn, "MockS3MissingBucketInPath").detail("Resource", resource).detail("QueryString", query);
			throw backup_invalid_url();
		}

		TraceEvent("MockS3ParsedPath")
		    .detail("OriginalResource", resource)
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("QueryString", query);
	}

	// Parse HTTP Range header: "bytes=start-end"
	// Returns true if parsing succeeded, false otherwise
	// Sets rangeStart and rangeEnd to the parsed values
	static bool parseRangeHeader(const std::string& rangeHeader, int64_t& rangeStart, int64_t& rangeEnd) {
		if (rangeHeader.empty()) {
			return false;
		}

		// Check for "bytes=" prefix
		if (rangeHeader.substr(0, 6) != "bytes=") {
			return false;
		}

		std::string range = rangeHeader.substr(6);
		size_t dashPos = range.find('-');
		if (dashPos == std::string::npos) {
			return false;
		}

		try {
			rangeStart = std::stoll(range.substr(0, dashPos));
			std::string endStr = range.substr(dashPos + 1);
			if (endStr.empty()) {
				// Open-ended range (e.g., "bytes=100-")
				rangeEnd = -1; // Indicates open-ended
			} else {
				rangeEnd = std::stoll(endStr);
			}
			return true;
		} catch (...) {
			return false;
		}
	}

	// Multipart Upload Operations
																#line 5503 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
// This generated class is to be used only via handleMultipartStart()
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartStartActor>
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartStartActorState {
															#line 5509 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartStartActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 5524 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("handleMultipartStart", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartStartActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartStart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 778 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartStart").detail("Bucket", bucket).detail("Object", object);
															#line 783 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string existingUploadId;
															#line 784 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			for( const auto& pair : getGlobalStorage().multipartUploads ) {
															#line 785 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (pair.second.bucket == bucket && pair.second.object == object)
															#line 5545 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				{
															#line 786 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					existingUploadId = pair.first;
															#line 787 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					TraceEvent("MockS3MultipartStartIdempotent") .detail("Bucket", bucket) .detail("Object", object) .detail("ExistingUploadId", existingUploadId);
															#line 5551 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					break;
				}
			}
															#line 795 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadId = std::string();
															#line 796 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!existingUploadId.empty())
															#line 5559 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 797 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				uploadId = existingUploadId;
															#line 5563 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = a_body1cont1(loopDepth);
			}
			else
			{
															#line 800 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				MultipartUpload upload(bucket, object);
															#line 801 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				uploadId = upload.uploadId;
															#line 802 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				getGlobalStorage().multipartUploads[uploadId] = std::move(upload);
															#line 803 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3MultipartStarted").detail("UploadId", uploadId);
															#line 806 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (getGlobalStorage().persistenceEnabled)
															#line 5578 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				{
															#line 807 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					StrictFuture<Void> __when_expr_0 = persistMultipartState(uploadId);
															#line 807 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (static_cast<HandleMultipartStartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5584 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<HandleMultipartStartActor*>(this)->actor_wait_state = 1;
															#line 807 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartStartActor, 0, Void >*>(static_cast<HandleMultipartStartActor*>(this)));
															#line 5589 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont5(loopDepth);
				}
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartStartActorState();
		static_cast<HandleMultipartStartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 812 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<InitiateMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <UploadId>%s</UploadId>\n" "</InitiateMultipartUploadResult>", bucket.c_str(), object.c_str(), uploadId.c_str());
															#line 822 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 824 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartStartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartStartActorState(); static_cast<HandleMultipartStartActor*>(this)->destroy(); return 0; }
															#line 5622 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<HandleMultipartStartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartStartActorState();
		static_cast<HandleMultipartStartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartStartActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartStartActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartStartActor*>(this)->ActorCallback< HandleMultipartStartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartStartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartStartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartStartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 795 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 5735 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via handleMultipartStart()
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartStartActor final : public Actor<Void>, public ActorCallback< HandleMultipartStartActor, 0, Void >, public FastAllocated<HandleMultipartStartActor>, public HandleMultipartStartActorState<HandleMultipartStartActor> {
															#line 5740 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<HandleMultipartStartActor>::operator new;
	using FastAllocated<HandleMultipartStartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3270833457051938816UL, 4344215680519964416UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartStartActor, 0, Void >;
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartStartActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 5757 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   HandleMultipartStartActorState<HandleMultipartStartActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10763588338010117376UL, 16830117502434504448UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartStart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartStartActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartStart( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartStartActor(self, req, response, bucket, object));
															#line 5789 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 826 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 5794 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
// This generated class is to be used only via handleUploadPart()
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleUploadPartActor>
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleUploadPartActorState {
															#line 5800 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleUploadPartActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 834 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId")),
															#line 835 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partNumber(std::stoi(queryParams.at("partNumber")))
															#line 5821 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("handleUploadPart", reinterpret_cast<unsigned long>(this));

	}
	~HandleUploadPartActorState() 
	{
		fdb_probe_actor_destroy("handleUploadPart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 837 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3UploadPart") .detail("UploadId", uploadId) .detail("PartNumber", partNumber) .detail("ContentLength", req->data.contentLen) .detail("ActualContentSize", req->data.content.size()) .detail("ContentPreview", req->data.content.size() > 0 ? req->data.content.substr(0, std::min((size_t)20, req->data.content.size())) : "EMPTY");
															#line 847 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 848 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 5840 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 849 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 850 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleUploadPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleUploadPartActorState(); static_cast<HandleUploadPartActor*>(this)->destroy(); return 0; }
															#line 5846 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<HandleUploadPartActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleUploadPartActorState();
				static_cast<HandleUploadPartActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 854 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			etag = ObjectData::generateETag(req->data.content);
															#line 855 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadIter->second.parts[partNumber] = { etag, req->data.content };
															#line 858 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (getGlobalStorage().persistenceEnabled)
															#line 5858 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 859 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = persistMultipartState(uploadId);
															#line 859 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<HandleUploadPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5864 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<HandleUploadPartActor*>(this)->actor_wait_state = 1;
															#line 859 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleUploadPartActor, 0, Void >*>(static_cast<HandleUploadPartActor*>(this)));
															#line 5869 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleUploadPartActorState();
		static_cast<HandleUploadPartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 863 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 864 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 865 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 866 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 868 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PartUploaded") .detail("UploadId", uploadId) .detail("PartNumber", partNumber) .detail("ETag", etag);
															#line 873 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleUploadPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleUploadPartActorState(); static_cast<HandleUploadPartActor*>(this)->destroy(); return 0; }
															#line 5907 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<HandleUploadPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleUploadPartActorState();
		static_cast<HandleUploadPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleUploadPartActor*>(this)->actor_wait_state > 0) static_cast<HandleUploadPartActor*>(this)->actor_wait_state = 0;
		static_cast<HandleUploadPartActor*>(this)->ActorCallback< HandleUploadPartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleUploadPartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleUploadPartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleUploadPartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 834 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 835 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNumber;
															#line 854 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 6020 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via handleUploadPart()
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleUploadPartActor final : public Actor<Void>, public ActorCallback< HandleUploadPartActor, 0, Void >, public FastAllocated<HandleUploadPartActor>, public HandleUploadPartActorState<HandleUploadPartActor> {
															#line 6025 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<HandleUploadPartActor>::operator new;
	using FastAllocated<HandleUploadPartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3579236463229318912UL, 1338038367828427520UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleUploadPartActor, 0, Void >;
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleUploadPartActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 6042 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   HandleUploadPartActorState<HandleUploadPartActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2867735766649073408UL, 15512396953593662720UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleUploadPart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleUploadPartActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleUploadPart( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleUploadPartActor(self, req, response, bucket, object, queryParams));
															#line 6074 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 875 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 6079 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
// This generated class is to be used only via handleMultipartComplete()
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartCompleteActor>
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartCompleteActorState {
															#line 6085 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartCompleteActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId"))
															#line 6104 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("handleMultipartComplete", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartCompleteActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartComplete", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 885 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartComplete").detail("UploadId", uploadId);
															#line 887 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 888 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 6123 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 889 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 890 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleMultipartCompleteActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartCompleteActorState(); static_cast<HandleMultipartCompleteActor*>(this)->destroy(); return 0; }
															#line 6129 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<HandleMultipartCompleteActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleMultipartCompleteActorState();
				static_cast<HandleMultipartCompleteActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 894 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			combinedContent = std::string();
															#line 895 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			for( auto& part : uploadIter->second.parts ) {
															#line 896 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				combinedContent += part.second.second;
															#line 6141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			}
															#line 899 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartDebug") .detail("UploadId", uploadId) .detail("PartsCount", uploadIter->second.parts.size()) .detail("CombinedSize", combinedContent.size()) .detail("CombinedPreview", combinedContent.size() > 0 ? combinedContent.substr(0, std::min((size_t)20, combinedContent.size())) : "EMPTY");
															#line 908 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ObjectData obj(combinedContent);
															#line 909 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().buckets[bucket][object] = std::move(obj);
															#line 911 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartFinalObject") .detail("UploadId", uploadId) .detail("StoredSize", getGlobalStorage().buckets[bucket][object].content.size()) .detail("StoredPreview", getGlobalStorage().buckets[bucket][object].content.size() > 0 ? getGlobalStorage().buckets[bucket][object].content.substr( 0, std::min((size_t)20, getGlobalStorage().buckets[bucket][object].content.size())) : "EMPTY");
															#line 921 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (getGlobalStorage().persistenceEnabled)
															#line 6153 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 922 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = persistObject(bucket, object);
															#line 922 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6159 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 1;
															#line 922 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartCompleteActor, 0, Void >*>(static_cast<HandleMultipartCompleteActor*>(this)));
															#line 6164 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartCompleteActorState();
		static_cast<HandleMultipartCompleteActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 926 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().multipartUploads.erase(uploadId);
															#line 927 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (getGlobalStorage().persistenceEnabled)
															#line 6194 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
															#line 928 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_1 = deletePersistedMultipart(uploadId);
															#line 928 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6200 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 2;
															#line 928 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartCompleteActor, 1, Void >*>(static_cast<HandleMultipartCompleteActor*>(this)));
															#line 6205 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartCompleteActor*>(this)->ActorCallback< HandleMultipartCompleteActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 932 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<CompleteMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <ETag>%s</ETag>\n" "</CompleteMultipartUploadResult>", bucket.c_str(), object.c_str(), getGlobalStorage().buckets[bucket][object].etag.c_str());
															#line 942 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 944 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartCompleted").detail("UploadId", uploadId).detail("FinalSize", combinedContent.size());
															#line 946 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartCompleteActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartCompleteActorState(); static_cast<HandleMultipartCompleteActor*>(this)->destroy(); return 0; }
															#line 6312 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<HandleMultipartCompleteActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartCompleteActorState();
		static_cast<HandleMultipartCompleteActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartCompleteActor*>(this)->ActorCallback< HandleMultipartCompleteActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 894 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string combinedContent;
															#line 6423 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via handleMultipartComplete()
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartCompleteActor final : public Actor<Void>, public ActorCallback< HandleMultipartCompleteActor, 0, Void >, public ActorCallback< HandleMultipartCompleteActor, 1, Void >, public FastAllocated<HandleMultipartCompleteActor>, public HandleMultipartCompleteActorState<HandleMultipartCompleteActor> {
															#line 6428 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<HandleMultipartCompleteActor>::operator new;
	using FastAllocated<HandleMultipartCompleteActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15430869604805798656UL, 10036215779918659072UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartCompleteActor, 0, Void >;
friend struct ActorCallback< HandleMultipartCompleteActor, 1, Void >;
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartCompleteActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 6446 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   HandleMultipartCompleteActorState<HandleMultipartCompleteActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7470605948036193280UL, 16274542347142593024UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartComplete");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartCompleteActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< HandleMultipartCompleteActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartComplete( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartCompleteActor(self, req, response, bucket, object, queryParams));
															#line 6479 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 948 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 6484 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
// This generated class is to be used only via handleMultipartAbort()
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartAbortActor>
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartAbortActorState {
															#line 6490 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartAbortActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 956 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId"))
															#line 6509 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("handleMultipartAbort", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartAbortActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartAbort", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 958 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartAbort").detail("UploadId", uploadId);
															#line 960 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 961 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 6528 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 962 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 963 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleMultipartAbortActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartAbortActorState(); static_cast<HandleMultipartAbortActor*>(this)->destroy(); return 0; }
															#line 6534 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<HandleMultipartAbortActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleMultipartAbortActorState();
				static_cast<HandleMultipartAbortActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 967 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().multipartUploads.erase(uploadId);
															#line 968 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (getGlobalStorage().persistenceEnabled)
															#line 6544 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 969 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedMultipart(uploadId);
															#line 969 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6550 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state = 1;
															#line 969 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartAbortActor, 0, Void >*>(static_cast<HandleMultipartAbortActor*>(this)));
															#line 6555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartAbortActorState();
		static_cast<HandleMultipartAbortActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 972 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 973 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 974 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 976 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartAborted").detail("UploadId", uploadId);
															#line 978 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartAbortActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartAbortActorState(); static_cast<HandleMultipartAbortActor*>(this)->destroy(); return 0; }
															#line 6591 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<HandleMultipartAbortActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartAbortActorState();
		static_cast<HandleMultipartAbortActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartAbortActor*>(this)->ActorCallback< HandleMultipartAbortActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 956 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 6700 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via handleMultipartAbort()
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartAbortActor final : public Actor<Void>, public ActorCallback< HandleMultipartAbortActor, 0, Void >, public FastAllocated<HandleMultipartAbortActor>, public HandleMultipartAbortActorState<HandleMultipartAbortActor> {
															#line 6705 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<HandleMultipartAbortActor>::operator new;
	using FastAllocated<HandleMultipartAbortActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9705080281817526272UL, 13541105442238880512UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartAbortActor, 0, Void >;
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartAbortActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 6722 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   HandleMultipartAbortActorState<HandleMultipartAbortActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3855017746569260032UL, 6136799802664629248UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartAbort");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartAbortActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartAbort( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartAbortActor(self, req, response, bucket, object, queryParams));
															#line 6754 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 980 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	// Object Tagging Operations
	static Future<Void> handlePutObjectTags(MockS3ServerImpl* self,
	                                        Reference<HTTP::IncomingRequest> req,
	                                        Reference<HTTP::OutgoingResponse> response,
	                                        std::string bucket,
	                                        std::string object) {

		TraceEvent("MockS3PutObjectTags")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagsXML", req->data.content);

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		// Parse tags XML (simplified parser)
		std::map<std::string, std::string> tags = self->parseTagsXML(req->data.content);
		objectIter->second.tags = tags;

		response->code = 200;
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3ObjectTagsSet")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagCount", tags.size());

		return Void();
	}

	static Future<Void> handleGetObjectTags(MockS3ServerImpl* self,
	                                        Reference<HTTP::IncomingRequest> req,
	                                        Reference<HTTP::OutgoingResponse> response,
	                                        std::string bucket,
	                                        std::string object) {

		TraceEvent("MockS3GetObjectTags").detail("Bucket", bucket).detail("Object", object);

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		// Generate tags XML response
		std::string xml = self->generateTagsXML(objectIter->second.tags);
		self->sendXMLResponse(response, 200, xml);

		TraceEvent("MockS3ObjectTagsRetrieved")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagCount", objectIter->second.tags.size());

		return Void();
	}

	// Basic Object Operations
																#line 6832 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
// This generated class is to be used only via handlePutObject()
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandlePutObjectActor>
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandlePutObjectActorState {
															#line 6838 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandlePutObjectActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 6853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("handlePutObject", reinterpret_cast<unsigned long>(this));

	}
	~HandlePutObjectActorState() 
	{
		fdb_probe_actor_destroy("handlePutObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1060 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3PutObject_Debug") .detail("Bucket", bucket) .detail("Object", object) .detail("ContentLength", req->data.contentLen) .detail("ContentSize", req->data.content.size()) .detail("ContentPreview", req->data.content.substr(0, std::min(100, (int)req->data.content.size())));
															#line 1067 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ObjectData obj(req->data.content);
															#line 1068 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			etag = obj.etag;
															#line 1069 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().buckets[bucket][object] = std::move(obj);
															#line 1071 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3PutObject_Stored") .detail("Bucket", bucket) .detail("Object", object) .detail("ETag", etag) .detail("StoredSize", getGlobalStorage().buckets[bucket][object].content.size());
															#line 1078 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (getGlobalStorage().persistenceEnabled)
															#line 6878 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 1079 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = persistObject(bucket, object);
															#line 1079 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<HandlePutObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6884 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<HandlePutObjectActor*>(this)->actor_wait_state = 1;
															#line 1079 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandlePutObjectActor, 0, Void >*>(static_cast<HandlePutObjectActor*>(this)));
															#line 6889 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandlePutObjectActorState();
		static_cast<HandlePutObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1082 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 1083 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 1084 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1087 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PutObject_Response") .detail("Bucket", bucket) .detail("Object", object) .detail("ResponseCode", response->code) .detail("ContentLen", response->data.contentLen) .detail("HasContent", response->data.content != nullptr);
															#line 1094 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandlePutObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandlePutObjectActorState(); static_cast<HandlePutObjectActor*>(this)->destroy(); return 0; }
															#line 6925 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<HandlePutObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandlePutObjectActorState();
		static_cast<HandlePutObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandlePutObjectActor*>(this)->actor_wait_state > 0) static_cast<HandlePutObjectActor*>(this)->actor_wait_state = 0;
		static_cast<HandlePutObjectActor*>(this)->ActorCallback< HandlePutObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandlePutObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandlePutObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandlePutObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 1068 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 7032 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via handlePutObject()
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandlePutObjectActor final : public Actor<Void>, public ActorCallback< HandlePutObjectActor, 0, Void >, public FastAllocated<HandlePutObjectActor>, public HandlePutObjectActorState<HandlePutObjectActor> {
															#line 7037 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<HandlePutObjectActor>::operator new;
	using FastAllocated<HandlePutObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15352640389553059328UL, 15879656028821951232UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandlePutObjectActor, 0, Void >;
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandlePutObjectActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 7054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   HandlePutObjectActorState<HandlePutObjectActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4560658774791891968UL, 15471708591759102464UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handlePutObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandlePutObjectActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handlePutObject( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 1054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandlePutObjectActor(self, req, response, bucket, object));
															#line 7086 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 1096 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	static Future<Void> handleGetObject(MockS3ServerImpl* self,
	                                    Reference<HTTP::IncomingRequest> req,
	                                    Reference<HTTP::OutgoingResponse> response,
	                                    std::string bucket,
	                                    std::string object) {

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		std::string content = objectIter->second.content;
		std::string etag = objectIter->second.etag;
		std::string contentMD5 = HTTP::computeMD5Sum(content);

		// Handle HTTP Range header for partial content requests
		// This is essential for AsyncFileEncrypted to read encrypted blocks correctly
		int64_t rangeStart = 0;
		int64_t rangeEnd = static_cast<int64_t>(content.size() - 1);
		bool isRangeRequest = false;

		auto rangeHeader = req->data.headers.find("Range");
		if (rangeHeader != req->data.headers.end()) {
			int64_t parsedStart, parsedEnd;
			if (parseRangeHeader(rangeHeader->second, parsedStart, parsedEnd)) {
				rangeStart = parsedStart;
				if (parsedEnd == -1) {
					// Open-ended range (e.g., "bytes=100-")
					rangeEnd = static_cast<int64_t>(content.size() - 1);
				} else {
					rangeEnd = parsedEnd;
				}
				// Clamp range to actual content size
				int64_t contentSize = static_cast<int64_t>(content.size() - 1);
				rangeEnd = std::min(rangeEnd, contentSize);
				rangeStart = std::min(rangeStart, contentSize);
				if (rangeStart <= rangeEnd) {
					isRangeRequest = true;
				}
			}
		}

		// Extract the requested range
		std::string responseContent;
		if (isRangeRequest && rangeStart <= rangeEnd) {
			responseContent =
			    content.substr(static_cast<size_t>(rangeStart), static_cast<size_t>(rangeEnd - rangeStart + 1));
			response->code = 206; // Partial Content
			response->data.headers["Content-Range"] =
			    format("bytes %lld-%lld/%zu", rangeStart, rangeEnd, content.size());
			// For range requests, calculate MD5 of the partial content, not full content
			contentMD5 = HTTP::computeMD5Sum(responseContent);
		} else {
			responseContent = content;
			response->code = 200;
		}

		response->data.headers["ETag"] = etag;
		response->data.headers["Content-Type"] = "binary/octet-stream";
		response->data.headers["Content-MD5"] = contentMD5;

		// Write content to response
		response->data.contentLen = responseContent.size();
		response->data.headers["Content-Length"] = std::to_string(responseContent.size());
		response->data.content->discardAll(); // Clear existing content

		if (!responseContent.empty()) {
			// Use the correct approach: getWriteBuffer from the UnsentPacketQueue
			PacketBuffer* buffer = response->data.content->getWriteBuffer(responseContent.size());
			PacketWriter pw(buffer, nullptr, Unversioned());
			pw.serializeBytes(responseContent);
			pw.finish();
		}

		TraceEvent("MockS3GetObjectComplete")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("ResponseCode", response->code)
		    .detail("ResponseSize", responseContent.size());

		return Void();
	}

																#line 7181 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
// This generated class is to be used only via handleDeleteObject()
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleDeleteObjectActor>
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleDeleteObjectActorState {
															#line 7187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleDeleteObjectActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 7202 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("handleDeleteObject", reinterpret_cast<unsigned long>(this));

	}
	~HandleDeleteObjectActorState() 
	{
		fdb_probe_actor_destroy("handleDeleteObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1193 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3DeleteObject").detail("Bucket", bucket).detail("Object", object);
															#line 1195 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto bucketIter = getGlobalStorage().buckets.find(bucket);
															#line 1196 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (bucketIter != getGlobalStorage().buckets.end())
															#line 7221 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 1197 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIter->second.erase(object);
															#line 7225 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			}
															#line 1201 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (getGlobalStorage().persistenceEnabled)
															#line 7229 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 1202 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedObject(bucket, object);
															#line 1202 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7235 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state = 1;
															#line 1202 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleDeleteObjectActor, 0, Void >*>(static_cast<HandleDeleteObjectActor*>(this)));
															#line 7240 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleDeleteObjectActorState();
		static_cast<HandleDeleteObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1205 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 1206 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1207 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 1209 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 1211 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleDeleteObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleDeleteObjectActorState(); static_cast<HandleDeleteObjectActor*>(this)->destroy(); return 0; }
															#line 7276 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<HandleDeleteObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleDeleteObjectActorState();
		static_cast<HandleDeleteObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state > 0) static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state = 0;
		static_cast<HandleDeleteObjectActor*>(this)->ActorCallback< HandleDeleteObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 7381 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via handleDeleteObject()
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleDeleteObjectActor final : public Actor<Void>, public ActorCallback< HandleDeleteObjectActor, 0, Void >, public FastAllocated<HandleDeleteObjectActor>, public HandleDeleteObjectActorState<HandleDeleteObjectActor> {
															#line 7386 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<HandleDeleteObjectActor>::operator new;
	using FastAllocated<HandleDeleteObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14625364977275660800UL, 6531309749781319168UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleDeleteObjectActor, 0, Void >;
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleDeleteObjectActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 7403 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   HandleDeleteObjectActorState<HandleDeleteObjectActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7788094060450111232UL, 5901935141690097664UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleDeleteObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleDeleteObjectActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleDeleteObject( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 1187 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleDeleteObjectActor(self, req, response, bucket, object));
															#line 7435 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 1213 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	static Future<Void> handleHeadObject(MockS3ServerImpl* self,
	                                     Reference<HTTP::IncomingRequest> req,
	                                     Reference<HTTP::OutgoingResponse> response,
	                                     std::string bucket,
	                                     std::string object) {

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			TraceEvent("MockS3HeadObjectNoBucket")
			    .detail("Bucket", bucket)
			    .detail("Object", object)
			    .detail("AvailableBuckets", getGlobalStorage().buckets.size());
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			TraceEvent("MockS3HeadObjectNoObject")
			    .detail("Bucket", bucket)
			    .detail("Object", object)
			    .detail("ObjectsInBucket", bucketIter->second.size());
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		const ObjectData& obj = objectIter->second;
		std::string etag = obj.etag;
		size_t contentSize = obj.content.size();
		std::string preview = contentSize > 0 ? obj.content.substr(0, std::min((size_t)20, contentSize)) : "EMPTY";

		TraceEvent("MockS3HeadObjectFound")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("Size", contentSize)
		    .detail("Preview", preview);

		response->code = 200;
		response->data.headers["ETag"] = etag;
		response->data.headers["Content-Length"] = std::to_string(contentSize);
		response->data.headers["Content-Type"] = "binary/octet-stream";
		// HEAD requests need contentLen set to actual size for headers
		response->data.contentLen = contentSize; // This controls ResponseContentSize in HTTP logs

		return Void();
	}

	// S3 ListObjects Operation
	static Future<Void> handleListObjects(MockS3ServerImpl* self,
	                                      Reference<HTTP::IncomingRequest> req,
	                                      Reference<HTTP::OutgoingResponse> response,
	                                      std::string bucket,
	                                      std::map<std::string, std::string> queryParams) {

		TraceEvent("MockS3ListObjects").detail("Bucket", bucket).detail("QueryParamCount", queryParams.size());

		// Get query parameters for listing
		std::string prefix = queryParams.count("prefix") ? queryParams.at("prefix") : "";
		std::string delimiter = queryParams.count("delimiter") ? queryParams.at("delimiter") : "";
		std::string marker = queryParams.count("marker") ? queryParams.at("marker") : "";
		std::string continuationToken =
		    queryParams.count("continuation-token") ? queryParams.at("continuation-token") : "";
		int maxKeys = queryParams.count("max-keys") ? std::stoi(queryParams.at("max-keys")) : 1000;

		TraceEvent("MockS3ListObjectsDebug")
		    .detail("Bucket", bucket)
		    .detail("Prefix", prefix)
		    .detail("Delimiter", delimiter)
		    .detail("Marker", marker)
		    .detail("ContinuationToken", continuationToken)
		    .detail("MaxKeys", maxKeys);

		// Find bucket
		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		// Collect all matching objects first
		std::vector<std::pair<std::string, const ObjectData*>> matchingObjects;
		for (const auto& objectPair : bucketIter->second) {
			const std::string& objectName = objectPair.first;
			const ObjectData& objectData = objectPair.second;

			// Apply prefix filter
			if (!prefix.empty() && objectName.find(prefix) != 0) {
				continue;
			}

			matchingObjects.push_back({ objectName, &objectData });
		}

		// Sort objects by name for consistent pagination
		std::sort(matchingObjects.begin(), matchingObjects.end());

		// Find starting point for pagination
		size_t startIndex = 0;
		if (!marker.empty()) {
			for (size_t i = 0; i < matchingObjects.size(); i++) {
				if (matchingObjects[i].first > marker) {
					startIndex = i;
					break;
				}
			}
		} else if (!continuationToken.empty()) {
			// Simple continuation token implementation (just use the last object name)
			for (size_t i = 0; i < matchingObjects.size(); i++) {
				if (matchingObjects[i].first > continuationToken) {
					startIndex = i;
					break;
				}
			}
		}

		// Build list of objects for this page
		std::string xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<ListBucketResult>\n";
		xml += "<Name>" + bucket + "</Name>\n";
		xml += "<Prefix>" + prefix + "</Prefix>\n";
		xml += "<MaxKeys>" + std::to_string(maxKeys) + "</MaxKeys>\n";

		if (!marker.empty()) {
			xml += "<Marker>" + marker + "</Marker>\n";
		}

		int count = 0;
		std::string lastKey;
		size_t totalMatching = matchingObjects.size();

		for (size_t i = startIndex; i < matchingObjects.size() && count < maxKeys; i++) {
			const std::string& objectName = matchingObjects[i].first;
			const ObjectData* objectData = matchingObjects[i].second;

			xml += "<Contents>\n";
			xml += "<Key>" + objectName + "</Key>\n";
			xml += "<LastModified>" + std::to_string((int64_t)objectData->lastModified) + "</LastModified>\n";
			xml += "<ETag>" + objectData->etag + "</ETag>\n";
			xml += "<Size>" + std::to_string(objectData->content.size()) + "</Size>\n";
			xml += "<StorageClass>STANDARD</StorageClass>\n";
			xml += "</Contents>\n";

			lastKey = objectName;
			count++;
		}

		// Determine if there are more results
		bool isTruncated = (startIndex + count) < totalMatching;
		xml += "<IsTruncated>" + std::string(isTruncated ? "true" : "false") + "</IsTruncated>\n";

		if (isTruncated && !lastKey.empty()) {
			xml += "<NextMarker>" + lastKey + "</NextMarker>\n";
		}

		xml += "</ListBucketResult>";

		self->sendXMLResponse(response, 200, xml);

		TraceEvent("MockS3ListObjectsCompleted")
		    .detail("Bucket", bucket)
		    .detail("Prefix", prefix)
		    .detail("ObjectCount", count)
		    .detail("StartIndex", startIndex)
		    .detail("TotalMatching", totalMatching)
		    .detail("IsTruncated", isTruncated)
		    .detail("NextMarker", isTruncated ? lastKey : "");

		return Void();
	}

	// S3 Bucket Operations
	static Future<Void> handleHeadBucket(MockS3ServerImpl* self,
	                                     Reference<HTTP::IncomingRequest> req,
	                                     Reference<HTTP::OutgoingResponse> response,
	                                     std::string bucket) {

		TraceEvent("MockS3HeadBucket").detail("Bucket", bucket);

		// Ensure bucket exists in our storage (implicit creation like real S3)
		if (getGlobalStorage().buckets.find(bucket) == getGlobalStorage().buckets.end()) {
			getGlobalStorage().buckets[bucket] = std::map<std::string, ObjectData>();
		}

		response->code = 200;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3BucketHead").detail("Bucket", bucket);

		return Void();
	}

	static Future<Void> handlePutBucket(MockS3ServerImpl* self,
	                                    Reference<HTTP::IncomingRequest> req,
	                                    Reference<HTTP::OutgoingResponse> response,
	                                    std::string bucket) {

		TraceEvent("MockS3PutBucket").detail("Bucket", bucket);

		// Ensure bucket exists in our storage (implicit creation)
		if (getGlobalStorage().buckets.find(bucket) == getGlobalStorage().buckets.end()) {
			getGlobalStorage().buckets[bucket] = std::map<std::string, ObjectData>();
		}

		response->code = 200;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3BucketCreated").detail("Bucket", bucket);

		return Void();
	}

	// Utility Methods

	void sendError(Reference<HTTP::OutgoingResponse> response,
	               int code,
	               const std::string& errorCode,
	               const std::string& message) {

		TraceEvent("MockS3Error").detail("Code", code).detail("ErrorCode", errorCode).detail("Message", message);

		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		                         "<Error>\n"
		                         "  <Code>%s</Code>\n"
		                         "  <Message>%s</Message>\n"
		                         "</Error>",
		                         errorCode.c_str(),
		                         message.c_str());

		sendXMLResponse(response, code, xml);
	}

	void sendXMLResponse(Reference<HTTP::OutgoingResponse> response, int code, const std::string& xml) {
		TraceEvent("MockS3SendXMLResponse_Start")
		    .detail("Code", code)
		    .detail("XMLSize", xml.size())
		    .detail("XMLPreview", xml.size() > 0 ? xml.substr(0, std::min((size_t)50, xml.size())) : "EMPTY");

		response->code = code;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.headers["Content-Length"] = std::to_string(xml.size());
		response->data.headers["Content-MD5"] = HTTP::computeMD5Sum(xml);

		// Actually put the XML content into the response
		if (xml.empty()) {
			response->data.contentLen = 0;
			TraceEvent("MockS3SendXMLResponse_Empty").detail("ResponseCode", response->code);
		} else {
			// Use the existing content queue instead of creating a new one
			// This prevents memory management issues and potential canBeSet() failures
			size_t contentSize = xml.size();
			response->data.contentLen = contentSize;

			// Clear any existing content and write the XML
			response->data.content->discardAll();
			PacketBuffer* buffer = response->data.content->getWriteBuffer(contentSize);
			PacketWriter pw(buffer, nullptr, Unversioned());
			pw.serializeBytes(xml);
			pw.finish();
		}

		TraceEvent("MockS3SendXMLResponse_Complete")
		    .detail("FinalCode", response->code)
		    .detail("FinalContentLen", response->data.contentLen)
		    .detail("XMLSize", xml.size());
	}

	std::map<std::string, std::string> parseTagsXML(const std::string& xml) {
		std::map<std::string, std::string> tags;

		// Simplified XML parsing for tags - this would need a proper XML parser in production
		std::regex tagRegex("<Tag><Key>([^<]+)</Key><Value>([^<]*)</Value></Tag>");
		std::sregex_iterator iter(xml.begin(), xml.end(), tagRegex);
		std::sregex_iterator end;

		for (; iter != end; ++iter) {
			std::string key = iter->str(1);
			std::string value = iter->str(2);
			tags[key] = value;

			TraceEvent("MockS3ParsedTag").detail("Key", key).detail("Value", value);
		}

		return tags;
	}

	std::string generateTagsXML(const std::map<std::string, std::string>& tags) {
		std::string xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<Tagging><TagSet>";

		for (const auto& tag : tags) {
			xml += "<Tag><Key>" + tag.first + "</Key><Value>" + tag.second + "</Value></Tag>";
		}

		xml += "</TagSet></Tagging>";
		return xml;
	}
};

// Global registry to track registered servers and avoid conflicts
static std::map<std::string, bool> registeredServers;

// Clear global storage state for clean test runs
static void clearSingletonState() {
	getGlobalStorage().buckets.clear();
	getGlobalStorage().multipartUploads.clear();
	TraceEvent("MockS3ServerImpl_StateCleared");
}

// Process a Mock S3 request directly (for wrapping/chaos injection)
Future<Void> processMockS3Request(Reference<HTTP::IncomingRequest> req, Reference<HTTP::OutgoingResponse> response) {
	static MockS3ServerImpl serverInstance;
	return MockS3ServerImpl::handleRequest(&serverInstance, req, response);
}

// Request Handler Implementation - Each handler instance works with global storage
Future<Void> MockS3RequestHandler::handleRequest(Reference<HTTP::IncomingRequest> req,
                                                 Reference<HTTP::OutgoingResponse> response) {
	// Guard against calling virtual functions during destruction
	if (destructing.load()) {
		TraceEvent(SevWarn, "MockS3RequestHandler_DestructingGuard")
		    .detail("Method", "handleRequest")
		    .detail("Resource", req->resource);
		return Void();
	}

	return processMockS3Request(req, response);
}

Reference<HTTP::IRequestHandler> MockS3RequestHandler::clone() {
	// Guard against calling virtual functions during destruction
	if (destructing.load()) {
		TraceEvent(SevWarn, "MockS3RequestHandler_DestructingGuard").detail("Method", "clone");
		// Return nullptr - caller must handle this gracefully
		return Reference<HTTP::IRequestHandler>();
	}
	return makeReference<MockS3RequestHandler>();
}

// Safe server registration that prevents conflicts (internal implementation)
															#line 7781 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via registerMockS3Server_impl()
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class RegisterMockS3Server_implActor>
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class RegisterMockS3Server_implActorState {
															#line 7788 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	RegisterMockS3Server_implActorState(std::string const& ip,std::string const& port) 
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : ip(ip),
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   port(port),
															#line 1556 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   serverKey(ip + ":" + port)
															#line 7799 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this));

	}
	~RegisterMockS3Server_implActorState() 
	{
		fdb_probe_actor_destroy("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1559 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Registration Start") .detail("IP", ip) .detail("Port", port) .detail("ServerKey", serverKey) .detail("IsSimulated", g_network->isSimulated()) .detail("AlreadyRegistered", registeredServers.count(serverKey) > 0);
															#line 1568 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (registeredServers.count(serverKey))
															#line 7816 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 1569 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent(SevWarn, "MockS3ServerAlreadyRegistered").detail("Address", serverKey);
															#line 1570 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<RegisterMockS3Server_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterMockS3Server_implActorState(); static_cast<RegisterMockS3Server_implActor*>(this)->destroy(); return 0; }
															#line 7822 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<RegisterMockS3Server_implActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~RegisterMockS3Server_implActorState();
				static_cast<RegisterMockS3Server_implActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1576 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!getGlobalStorage().persistenceEnabled)
															#line 7831 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				{
															#line 1577 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					std::string persistenceDir = "simfdb/mocks3";
															#line 1578 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					enableMockS3Persistence(persistenceDir);
															#line 1579 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					TraceEvent("MockS3ServerPersistenceEnabled") .detail("Address", serverKey) .detail("PersistenceDir", persistenceDir);
															#line 1584 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					StrictFuture<Void> __when_expr_0 = loadMockS3PersistedStateFuture();
															#line 1584 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7843 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 1;
															#line 1584 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RegisterMockS3Server_implActor, 0, Void >*>(static_cast<RegisterMockS3Server_implActor*>(this)));
															#line 7848 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont3(loopDepth);
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RegisterMockS3Server_implActorState();
		static_cast<RegisterMockS3Server_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1609 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<RegisterMockS3Server_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterMockS3Server_implActorState(); static_cast<RegisterMockS3Server_implActor*>(this)->destroy(); return 0; }
															#line 7882 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<RegisterMockS3Server_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RegisterMockS3Server_implActorState();
		static_cast<RegisterMockS3Server_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1601 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3ServerRegistrationFailed") .error(e) .detail("Address", serverKey) .detail("ErrorCode", e.code()) .detail("ErrorName", e.name());
															#line 1606 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 7897 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1587 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Calling registerSimHTTPServer") .detail("Address", serverKey);
															#line 1591 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = g_simulator->registerSimHTTPServer(ip, port, makeReference<MockS3RequestHandler>());
															#line 1591 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7915 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 2;
															#line 1591 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RegisterMockS3Server_implActor, 1, Void >*>(static_cast<RegisterMockS3Server_implActor*>(this)));
															#line 7920 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state > 0) static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterMockS3Server_implActor*>(this)->ActorCallback< RegisterMockS3Server_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 1592 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		registeredServers[serverKey] = true;
															#line 1594 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerRegistered").detail("Address", serverKey).detail("Success", true);
															#line 1596 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Registration Complete") .detail("Address", serverKey) .detail("TotalRegistered", registeredServers.size());
															#line 8020 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 1592 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		registeredServers[serverKey] = true;
															#line 1594 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerRegistered").detail("Address", serverKey).detail("Success", true);
															#line 1596 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Registration Complete") .detail("Address", serverKey) .detail("TotalRegistered", registeredServers.size());
															#line 8033 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state > 0) static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterMockS3Server_implActor*>(this)->ActorCallback< RegisterMockS3Server_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string ip;
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string port;
															#line 1556 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string serverKey;
															#line 8132 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via registerMockS3Server_impl()
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class RegisterMockS3Server_implActor final : public Actor<Void>, public ActorCallback< RegisterMockS3Server_implActor, 0, Void >, public ActorCallback< RegisterMockS3Server_implActor, 1, Void >, public FastAllocated<RegisterMockS3Server_implActor>, public RegisterMockS3Server_implActorState<RegisterMockS3Server_implActor> {
															#line 8137 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<RegisterMockS3Server_implActor>::operator new;
	using FastAllocated<RegisterMockS3Server_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13830534177566059776UL, 11192474355077445376UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RegisterMockS3Server_implActor, 0, Void >;
friend struct ActorCallback< RegisterMockS3Server_implActor, 1, Void >;
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	RegisterMockS3Server_implActor(std::string const& ip,std::string const& port) 
															#line 8155 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RegisterMockS3Server_implActorState<RegisterMockS3Server_implActor>(ip, port),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13948469852865528320UL, 9795983047084281344UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("registerMockS3Server_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RegisterMockS3Server_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RegisterMockS3Server_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> registerMockS3Server_impl( std::string const& ip, std::string const& port ) {
															#line 1555 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new RegisterMockS3Server_implActor(ip, port));
															#line 8189 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 1611 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Public Interface Implementation
															#line 8195 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via startMockS3Server()
															#line 1613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class StartMockS3ServerActor>
															#line 1613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerActorState {
															#line 8202 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerActorState(NetworkAddress const& listenAddress) 
															#line 1613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : listenAddress(listenAddress)
															#line 8209 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("startMockS3Server", reinterpret_cast<unsigned long>(this));

	}
	~StartMockS3ServerActorState() 
	{
		fdb_probe_actor_destroy("startMockS3Server", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1614 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerStarting").detail("ListenAddress", listenAddress.toString());
															#line 8224 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			try {
															#line 1617 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerRegistering") .detail("IP", listenAddress.ip.toString()) .detail("Port", std::to_string(listenAddress.port)) .detail("IsSimulated", g_network->isSimulated());
															#line 1623 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = registerMockS3Server_impl(listenAddress.ip.toString(), std::to_string(listenAddress.port));
															#line 1623 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<StartMockS3ServerActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 8232 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<StartMockS3ServerActor*>(this)->actor_wait_state = 1;
															#line 1623 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerActor, 0, Void >*>(static_cast<StartMockS3ServerActor*>(this)));
															#line 8237 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StartMockS3ServerActorState();
		static_cast<StartMockS3ServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1634 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerActorState(); static_cast<StartMockS3ServerActor*>(this)->destroy(); return 0; }
															#line 8266 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<StartMockS3ServerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerActorState();
		static_cast<StartMockS3ServerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1630 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3ServerStartError").error(e).detail("ListenAddress", listenAddress.toString());
															#line 1631 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 8281 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1625 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerStarted") .detail("ListenAddress", listenAddress.toString()) .detail("HandlerCreated", true);
															#line 8295 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1625 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerStarted") .detail("ListenAddress", listenAddress.toString()) .detail("HandlerCreated", true);
															#line 8304 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StartMockS3ServerActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerActor*>(this)->ActorCallback< StartMockS3ServerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	NetworkAddress listenAddress;
															#line 8399 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via startMockS3Server()
															#line 1613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerActor final : public Actor<Void>, public ActorCallback< StartMockS3ServerActor, 0, Void >, public FastAllocated<StartMockS3ServerActor>, public StartMockS3ServerActorState<StartMockS3ServerActor> {
															#line 8404 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<StartMockS3ServerActor>::operator new;
	using FastAllocated<StartMockS3ServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11024617619658476544UL, 5248664789294665728UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StartMockS3ServerActor, 0, Void >;
															#line 1613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerActor(NetworkAddress const& listenAddress) 
															#line 8421 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   StartMockS3ServerActorState<StartMockS3ServerActor>(listenAddress),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16442790091880884736UL, 12597676879247326208UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("startMockS3Server");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StartMockS3ServerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> startMockS3Server( NetworkAddress const& listenAddress ) {
															#line 1613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new StartMockS3ServerActor(listenAddress));
															#line 8454 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 1636 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Clear all MockS3 global storage - called at the start of each simulation test
void clearMockS3Storage() {
	getGlobalStorage().clearStorage();
	// Note: Do NOT clear chaos server registry here - it must persist across tests
	// like the simulator's httpHandlers map, to prevent duplicate registration attempts
}

// Enable persistence for MockS3 storage
void enableMockS3Persistence(const std::string& persistenceDir) {
	getGlobalStorage().enablePersistence(persistenceDir);
	TraceEvent("MockS3PersistenceConfigured").detail("Directory", persistenceDir);
}

// Check if MockS3 persistence is currently enabled
bool isMockS3PersistenceEnabled() {
	return getGlobalStorage().persistenceEnabled;
}

// ACTOR: Load persisted objects from disk
															#line 8478 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via loadPersistedObjects()
															#line 1656 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadPersistedObjectsActor>
															#line 1656 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedObjectsActorState {
															#line 8485 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1656 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedObjectsActorState(std::string const& persistenceDir) 
															#line 1656 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1656 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir(persistenceDir),
															#line 1657 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   objectsDir(persistenceDir + "/objects")
															#line 8494 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("loadPersistedObjects", reinterpret_cast<unsigned long>(this));

	}
	~LoadPersistedObjectsActorState() 
	{
		fdb_probe_actor_destroy("loadPersistedObjects", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1659 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(objectsDir))
															#line 8509 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 1660 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3LoadObjects").detail("Status", "NoObjectsDir");
															#line 1661 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadPersistedObjectsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedObjectsActorState(); static_cast<LoadPersistedObjectsActor*>(this)->destroy(); return 0; }
															#line 8515 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<LoadPersistedObjectsActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadPersistedObjectsActorState();
				static_cast<LoadPersistedObjectsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1666 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				buckets = platform::listFiles(objectsDir, "");
															#line 1668 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::sort(buckets.begin(), buckets.end());
															#line 1669 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIdx = 0;
															#line 1671 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIdx = 0;
															#line 8530 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadPersistedObjectsActorState();
		static_cast<LoadPersistedObjectsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1726 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadPersistedObjectsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedObjectsActorState(); static_cast<LoadPersistedObjectsActor*>(this)->destroy(); return 0; }
															#line 8559 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<LoadPersistedObjectsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadPersistedObjectsActorState();
		static_cast<LoadPersistedObjectsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1723 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadObjectsFailed").error(e);
															#line 8572 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1721 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectsLoaded").detail("BucketsCount", getGlobalStorage().buckets.size());
															#line 8587 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1671 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(bucketIdx < buckets.size()))
															#line 8603 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1672 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucket = buckets[bucketIdx];
															#line 1673 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bucket == "." || bucket == "..")
															#line 8611 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
			return a_body1continue1(loopDepth); // continue
		}
															#line 1676 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketDir = objectsDir + "/" + bucket;
															#line 1677 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!directoryExists(bucketDir))
															#line 8619 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
			return a_body1continue1(loopDepth); // continue
		}
															#line 1681 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		files = platform::listFiles(bucketDir, "");
															#line 1682 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::sort(files.begin(), files.end());
															#line 1683 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx = 0;
															#line 1685 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx = 0;
															#line 8631 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1continue1(int loopDepth) 
	{
															#line 1671 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketIdx++;
															#line 8653 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1671 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketIdx++;
															#line 8662 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 1685 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(fileIdx < files.size()))
															#line 8678 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1686 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileName = files[fileIdx];
															#line 1689 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (fileName.size() > OBJECT_META_SUFFIX_LEN && fileName.substr(fileName.size() - OBJECT_META_SUFFIX_LEN) == OBJECT_META_SUFFIX)
															#line 8686 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
															#line 1692 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			objectName = fileName.substr(0, fileName.size() - OBJECT_META_SUFFIX_LEN);
															#line 1693 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			dataPath = bucketDir + "/" + objectName + OBJECT_DATA_SUFFIX;
															#line 1694 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaPath = bucketDir + "/" + fileName;
															#line 1696 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(dataPath))
															#line 8696 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 1697 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent(SevWarn, "MockS3LoadObjectSkipped") .detail("Bucket", bucket) .detail("Object", objectName) .detail("Reason", "NoDataFile");
															#line 8700 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				return a_body1loopBody1continue1(loopDepth); // continue
			}
															#line 1705 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<std::string> __when_expr_0 = readFileContent(dataPath);
															#line 1705 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8707 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 1;
															#line 1705 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedObjectsActor, 0, std::string >*>(static_cast<LoadPersistedObjectsActor*>(this)));
															#line 8712 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1continue1(int loopDepth) 
	{
															#line 1685 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 8739 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 1685 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 8748 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(int loopDepth) 
	{
															#line 1706 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_1 = readFileContent(metaPath);
															#line 1706 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8759 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 2;
															#line 1706 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedObjectsActor, 1, std::string >*>(static_cast<LoadPersistedObjectsActor*>(this)));
															#line 8764 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(std::string const& __content,int loopDepth) 
	{
															#line 1705 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content = __content;
															#line 8773 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(std::string && __content,int loopDepth) 
	{
		content = std::move(__content);
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedObjectsActor*>(this)->ActorCallback< LoadPersistedObjectsActor, 0, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont5(int loopDepth) 
	{
															#line 1709 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		MockS3GlobalStorage::ObjectData obj(content);
															#line 1710 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		deserializeObjectMeta(metaJson, obj);
															#line 1711 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().buckets[bucket][objectName] = std::move(obj);
															#line 1713 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectRestored") .detail("Bucket", bucket) .detail("Object", objectName) .detail("Size", content.size());
															#line 8858 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(std::string const& __metaJson,int loopDepth) 
	{
															#line 1706 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		metaJson = __metaJson;
															#line 8867 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(std::string && __metaJson,int loopDepth) 
	{
		metaJson = std::move(__metaJson);
		loopDepth = a_body1loopBody1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedObjectsActor*>(this)->ActorCallback< LoadPersistedObjectsActor, 1, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1656 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 1657 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string objectsDir;
															#line 1666 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> buckets;
															#line 1669 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int bucketIdx;
															#line 1672 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1676 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucketDir;
															#line 1681 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> files;
															#line 1683 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int fileIdx;
															#line 1686 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string fileName;
															#line 1692 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string objectName;
															#line 1693 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 1694 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 1705 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 1706 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaJson;
															#line 8983 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via loadPersistedObjects()
															#line 1656 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedObjectsActor final : public Actor<Void>, public ActorCallback< LoadPersistedObjectsActor, 0, std::string >, public ActorCallback< LoadPersistedObjectsActor, 1, std::string >, public FastAllocated<LoadPersistedObjectsActor>, public LoadPersistedObjectsActorState<LoadPersistedObjectsActor> {
															#line 8988 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<LoadPersistedObjectsActor>::operator new;
	using FastAllocated<LoadPersistedObjectsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13593828343188012032UL, 12424158023382634240UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadPersistedObjectsActor, 0, std::string >;
friend struct ActorCallback< LoadPersistedObjectsActor, 1, std::string >;
															#line 1656 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedObjectsActor(std::string const& persistenceDir) 
															#line 9006 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   LoadPersistedObjectsActorState<LoadPersistedObjectsActor>(persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15447654064422368256UL, 7597335790883620352UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadPersistedObjects");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadPersistedObjectsActor, 0, std::string >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadPersistedObjectsActor, 1, std::string >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1656 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadPersistedObjects( std::string const& persistenceDir ) {
															#line 1656 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadPersistedObjectsActor(persistenceDir));
															#line 9040 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 1728 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Load persisted multipart uploads from disk
															#line 9046 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via loadPersistedMultipartUploads()
															#line 1730 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadPersistedMultipartUploadsActor>
															#line 1730 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedMultipartUploadsActorState {
															#line 9053 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1730 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedMultipartUploadsActorState(std::string const& persistenceDir) 
															#line 1730 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1730 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir(persistenceDir),
															#line 1731 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   multipartDir(persistenceDir + "/multipart")
															#line 9062 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this));

	}
	~LoadPersistedMultipartUploadsActorState() 
	{
		fdb_probe_actor_destroy("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1733 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(multipartDir))
															#line 9077 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 1734 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3LoadMultipart").detail("Status", "NoMultipartDir");
															#line 1735 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedMultipartUploadsActorState(); static_cast<LoadPersistedMultipartUploadsActor*>(this)->destroy(); return 0; }
															#line 9083 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadPersistedMultipartUploadsActorState();
				static_cast<LoadPersistedMultipartUploadsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1740 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				files = platform::listFiles(multipartDir, "");
															#line 1741 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::sort(files.begin(), files.end());
															#line 1742 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				fileIdx = 0;
															#line 1744 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				fileIdx = 0;
															#line 9098 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadPersistedMultipartUploadsActorState();
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1810 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedMultipartUploadsActorState(); static_cast<LoadPersistedMultipartUploadsActor*>(this)->destroy(); return 0; }
															#line 9127 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadPersistedMultipartUploadsActorState();
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1807 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadMultipartFailed").error(e);
															#line 9140 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1805 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartUploadsLoaded").detail("UploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9155 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1744 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(fileIdx < files.size()))
															#line 9171 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1745 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileName = files[fileIdx];
															#line 1748 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (fileName.size() > MULTIPART_STATE_SUFFIX_LEN && fileName.substr(fileName.size() - MULTIPART_STATE_SUFFIX_LEN) == MULTIPART_STATE_SUFFIX)
															#line 9179 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
															#line 1750 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadId = fileName.substr(0, fileName.size() - MULTIPART_STATE_SUFFIX_LEN);
															#line 1751 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			statePath = multipartDir + "/" + fileName;
															#line 1754 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<std::string> __when_expr_0 = readFileContent(statePath);
															#line 1754 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9189 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 1;
															#line 1754 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9194 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1continue1(int loopDepth) 
	{
															#line 1744 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 9221 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1744 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 9230 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1755 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (stateJson.empty())
															#line 9239 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
															#line 1756 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadMultipartSkipped") .detail("UploadId", uploadId) .detail("Reason", "EmptyStateFile");
															#line 9243 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			return a_body1continue1(loopDepth); // continue
		}
															#line 1763 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload = MockS3GlobalStorage::MultipartUpload("", "");
															#line 1764 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload.uploadId = uploadId;
															#line 1765 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		deserializeMultipartState(stateJson, upload);
															#line 1768 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 1769 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		maxAttempts = 10000;
															#line 1770 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 9258 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::string const& __stateJson,int loopDepth) 
	{
															#line 1754 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		stateJson = __stateJson;
															#line 9267 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::string && __stateJson,int loopDepth) 
	{
		stateJson = std::move(__stateJson);
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1795 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartUploadRestored") .detail("UploadId", uploadId) .detail("Bucket", upload.bucket) .detail("Object", upload.object) .detail("PartsCount", upload.parts.size());
															#line 1801 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().multipartUploads[uploadId] = std::move(upload);
															#line 9348 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 1770 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partNum <= maxAttempts))
															#line 9364 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1771 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = multipartDir + "/" + uploadId + ".part." + std::to_string(partNum);
															#line 1772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 1774 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!fileExists(partPath) || !fileExists(partMetaPath))
															#line 9374 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1778 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_1 = readFileContent(partPath);
															#line 1778 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 9382 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 2;
															#line 1778 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9387 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
															#line 1779 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_2 = readFileContent(partMetaPath);
															#line 1779 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 9411 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 3;
															#line 1779 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9416 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(std::string const& __partData,int loopDepth) 
	{
															#line 1778 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partData = __partData;
															#line 9425 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(std::string && __partData,int loopDepth) 
	{
		partData = std::move(__partData);
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3loopBody1cont4(int loopDepth) 
	{
															#line 1782 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		using namespace rapidjson;
															#line 1783 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		Document doc;
															#line 1784 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		doc.Parse(partMetaJson.c_str());
															#line 1785 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string etag = doc.HasMember("etag") && doc["etag"].IsString() ? doc["etag"].GetString() : "";
															#line 1786 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload.parts[partNum] = { etag, partData };
															#line 1788 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartPartRestored") .detail("UploadId", uploadId) .detail("PartNumber", partNum) .detail("Size", partData.size());
															#line 1770 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 9516 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(std::string const& __partMetaJson,int loopDepth) 
	{
															#line 1779 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = __partMetaJson;
															#line 9525 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont3loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(std::string && __partMetaJson,int loopDepth) 
	{
		partMetaJson = std::move(__partMetaJson);
		loopDepth = a_body1loopBody1cont3loopBody1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1730 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 1731 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string multipartDir;
															#line 1740 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> files;
															#line 1742 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int fileIdx;
															#line 1745 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string fileName;
															#line 1750 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 1751 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string statePath;
															#line 1754 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string stateJson;
															#line 1763 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3GlobalStorage::MultipartUpload upload;
															#line 1768 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 1769 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int maxAttempts;
															#line 1771 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 1772 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 1778 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partData;
															#line 1779 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaJson;
															#line 9643 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via loadPersistedMultipartUploads()
															#line 1730 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedMultipartUploadsActor final : public Actor<Void>, public ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >, public ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >, public ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >, public FastAllocated<LoadPersistedMultipartUploadsActor>, public LoadPersistedMultipartUploadsActorState<LoadPersistedMultipartUploadsActor> {
															#line 9648 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<LoadPersistedMultipartUploadsActor>::operator new;
	using FastAllocated<LoadPersistedMultipartUploadsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5942636701820736256UL, 7567940829663039232UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >;
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >;
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >;
															#line 1730 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedMultipartUploadsActor(std::string const& persistenceDir) 
															#line 9667 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   LoadPersistedMultipartUploadsActorState<LoadPersistedMultipartUploadsActor>(persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17523876871774881024UL, 15657059887496463360UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadPersistedMultipartUploads");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1730 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadPersistedMultipartUploads( std::string const& persistenceDir ) {
															#line 1730 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadPersistedMultipartUploadsActor(persistenceDir));
															#line 9702 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 1812 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Load all persisted state from disk
															#line 9708 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via loadMockS3PersistedStateImpl()
															#line 1814 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadMockS3PersistedStateImplActor>
															#line 1814 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadMockS3PersistedStateImplActorState {
															#line 9715 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1814 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadMockS3PersistedStateImplActorState() 
															#line 1814 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1815 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir()
															#line 9722 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this));

	}
	~LoadMockS3PersistedStateImplActorState() 
	{
		fdb_probe_actor_destroy("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1817 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!getGlobalStorage().persistenceEnabled || getGlobalStorage().persistenceLoaded)
															#line 9737 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 1818 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadMockS3PersistedStateImplActorState(); static_cast<LoadMockS3PersistedStateImplActor*>(this)->destroy(); return 0; }
															#line 9741 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				new (&static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadMockS3PersistedStateImplActorState();
				static_cast<LoadMockS3PersistedStateImplActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1821 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			persistenceDir = getGlobalStorage().persistenceDir;
															#line 1822 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3LoadPersistedStateStart").detail("PersistenceDir", persistenceDir);
															#line 9751 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			try {
															#line 1826 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = loadPersistedObjects(persistenceDir);
															#line 1826 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9757 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 1;
															#line 1826 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9762 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadMockS3PersistedStateImplActorState();
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1841 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadMockS3PersistedStateImplActorState(); static_cast<LoadMockS3PersistedStateImplActor*>(this)->destroy(); return 0; }
															#line 9791 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadMockS3PersistedStateImplActorState();
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1837 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3LoadPersistedStateFailed").error(e);
															#line 1838 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 9806 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1829 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = loadPersistedMultipartUploads(persistenceDir);
															#line 1829 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9822 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 2;
															#line 1829 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9827 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1829 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = loadPersistedMultipartUploads(persistenceDir);
															#line 1829 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9838 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 2;
															#line 1829 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9843 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state > 0) static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 0;
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1831 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().persistenceLoaded = true;
															#line 1833 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3LoadPersistedStateComplete") .detail("ObjectsCount", getGlobalStorage().buckets.size()) .detail("MultipartUploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9929 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1831 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().persistenceLoaded = true;
															#line 1833 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3LoadPersistedStateComplete") .detail("ObjectsCount", getGlobalStorage().buckets.size()) .detail("MultipartUploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9940 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state > 0) static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 0;
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1815 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 10035 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via loadMockS3PersistedStateImpl()
															#line 1814 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadMockS3PersistedStateImplActor final : public Actor<Void>, public ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >, public ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >, public FastAllocated<LoadMockS3PersistedStateImplActor>, public LoadMockS3PersistedStateImplActorState<LoadMockS3PersistedStateImplActor> {
															#line 10040 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<LoadMockS3PersistedStateImplActor>::operator new;
	using FastAllocated<LoadMockS3PersistedStateImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9810051022791715584UL, 7106898465531486976UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >;
friend struct ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >;
															#line 1814 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadMockS3PersistedStateImplActor() 
															#line 10058 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   LoadMockS3PersistedStateImplActorState<LoadMockS3PersistedStateImplActor>(),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10516119037031120128UL, 15944437332465256704UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadMockS3PersistedStateImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1814 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadMockS3PersistedStateImpl(  ) {
															#line 1814 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadMockS3PersistedStateImplActor());
															#line 10092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 1843 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Load persisted state from disk (called at server startup) - returns Future for use in ACTOR context
Future<Void> loadMockS3PersistedStateFuture() {
	if (getGlobalStorage().persistenceEnabled && !getGlobalStorage().persistenceLoaded) {
		return loadMockS3PersistedStateImpl();
	}
	return Void();
}

// Initialize MockS3 persistence for simulation tests (exported for MockS3ServerChaos)
															#line 10106 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via initializeMockS3Persistence()
															#line 1853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class InitializeMockS3PersistenceActor>
															#line 1853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class InitializeMockS3PersistenceActorState {
															#line 10113 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	InitializeMockS3PersistenceActorState(std::string const& serverKey) 
															#line 1853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : serverKey(serverKey)
															#line 10120 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this));

	}
	~InitializeMockS3PersistenceActorState() 
	{
		fdb_probe_actor_destroy("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1854 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!getGlobalStorage().persistenceEnabled)
															#line 10135 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 1855 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				enableMockS3Persistence(DEFAULT_MOCKS3_PERSISTENCE_DIR);
															#line 1856 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerPersistenceEnabled") .detail("Address", serverKey) .detail("PersistenceDir", DEFAULT_MOCKS3_PERSISTENCE_DIR);
															#line 1861 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = loadMockS3PersistedStateFuture();
															#line 1861 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<InitializeMockS3PersistenceActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10145 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<InitializeMockS3PersistenceActor*>(this)->actor_wait_state = 1;
															#line 1861 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitializeMockS3PersistenceActor, 0, Void >*>(static_cast<InitializeMockS3PersistenceActor*>(this)));
															#line 10150 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitializeMockS3PersistenceActorState();
		static_cast<InitializeMockS3PersistenceActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1863 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<InitializeMockS3PersistenceActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitializeMockS3PersistenceActorState(); static_cast<InitializeMockS3PersistenceActor*>(this)->destroy(); return 0; }
															#line 10178 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<InitializeMockS3PersistenceActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitializeMockS3PersistenceActorState();
		static_cast<InitializeMockS3PersistenceActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitializeMockS3PersistenceActor*>(this)->actor_wait_state > 0) static_cast<InitializeMockS3PersistenceActor*>(this)->actor_wait_state = 0;
		static_cast<InitializeMockS3PersistenceActor*>(this)->ActorCallback< InitializeMockS3PersistenceActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitializeMockS3PersistenceActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeMockS3PersistenceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitializeMockS3PersistenceActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeMockS3PersistenceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitializeMockS3PersistenceActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeMockS3PersistenceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string serverKey;
															#line 10275 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via initializeMockS3Persistence()
															#line 1853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class InitializeMockS3PersistenceActor final : public Actor<Void>, public ActorCallback< InitializeMockS3PersistenceActor, 0, Void >, public FastAllocated<InitializeMockS3PersistenceActor>, public InitializeMockS3PersistenceActorState<InitializeMockS3PersistenceActor> {
															#line 10280 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<InitializeMockS3PersistenceActor>::operator new;
	using FastAllocated<InitializeMockS3PersistenceActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16769525787949811456UL, 14104545679119034112UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitializeMockS3PersistenceActor, 0, Void >;
															#line 1853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	InitializeMockS3PersistenceActor(std::string const& serverKey) 
															#line 10297 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   InitializeMockS3PersistenceActorState<InitializeMockS3PersistenceActor>(serverKey),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15097143290471136000UL, 6435086413190992384UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeMockS3PersistenceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("initializeMockS3Persistence");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitializeMockS3PersistenceActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> initializeMockS3Persistence( std::string const& serverKey ) {
															#line 1853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new InitializeMockS3PersistenceActor(serverKey));
															#line 10330 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 1865 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Unit Tests for MockS3Server
															#line 10336 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1867()
															#line 1867 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1867Actor>
															#line 1867 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1867ActorState {
															#line 10343 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1867 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1867ActorState(UnitTestParameters const& params) 
															#line 1867 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1867 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10350 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1867", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1867ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1867", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1869 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1870 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1871 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1872 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1874 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1876 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1877 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1878 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1880 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1867Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1867ActorState(); static_cast<FlowTestCase1867Actor*>(this)->destroy(); return 0; }
															#line 10381 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase1867Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1867ActorState();
			static_cast<FlowTestCase1867Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1867ActorState();
		static_cast<FlowTestCase1867Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1867 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10405 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1867()
															#line 1867 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1867Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1867Actor>, public FlowTestCase1867ActorState<FlowTestCase1867Actor> {
															#line 10410 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1867Actor>::operator new;
	using FastAllocated<FlowTestCase1867Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13004101879599179008UL, 17797430423663073280UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1867 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1867Actor(UnitTestParameters const& params) 
															#line 10426 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1867ActorState<FlowTestCase1867Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1867", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7940739426933585152UL, 661026354547473920UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1867Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1867");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1867", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1867 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1867( UnitTestParameters const& params ) {
															#line 1867 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1867Actor(params));
															#line 10458 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1867, "/MockS3Server/parseS3Request/ValidBucketParameter")

#line 1882 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10464 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1883()
															#line 1883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1883Actor>
															#line 1883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1883ActorState {
															#line 10471 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1883ActorState(UnitTestParameters const& params) 
															#line 1883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10478 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1883", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1883ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1883", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1885 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1886 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/?region=us-east-1";
															#line 1887 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1888 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 10499 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			try {
															#line 1891 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				server.parseS3Request(resource, bucket, object, queryParams);
															#line 1892 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				ASSERT(false);
															#line 10505 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1883ActorState();
		static_cast<FlowTestCase1883Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1897 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<FlowTestCase1883Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1883ActorState(); static_cast<FlowTestCase1883Actor*>(this)->destroy(); return 0; }
															#line 10534 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase1883Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1883ActorState();
		static_cast<FlowTestCase1883Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1894 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(e.code() == error_code_backup_invalid_url);
															#line 10547 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10573 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1883()
															#line 1883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1883Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1883Actor>, public FlowTestCase1883ActorState<FlowTestCase1883Actor> {
															#line 10578 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1883Actor>::operator new;
	using FastAllocated<FlowTestCase1883Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11796888142462930176UL, 3526754607814218752UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1883Actor(UnitTestParameters const& params) 
															#line 10594 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1883ActorState<FlowTestCase1883Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1883", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2424204218361978624UL, 3853298329899812864UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1883Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1883");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1883", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1883( UnitTestParameters const& params ) {
															#line 1883 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1883Actor(params));
															#line 10626 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1883, "/MockS3Server/parseS3Request/MissingBucketParameter")

#line 1899 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10632 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1900()
															#line 1900 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1900Actor>
															#line 1900 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1900ActorState {
															#line 10639 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1900 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1900ActorState(UnitTestParameters const& params) 
															#line 1900 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1900 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10646 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1900", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1900ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1900", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1902 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1903 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/";
															#line 1904 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1905 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 10667 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			try {
															#line 1908 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				server.parseS3Request(resource, bucket, object, queryParams);
															#line 1909 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				ASSERT(false);
															#line 10673 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1900ActorState();
		static_cast<FlowTestCase1900Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1914 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<FlowTestCase1900Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1900ActorState(); static_cast<FlowTestCase1900Actor*>(this)->destroy(); return 0; }
															#line 10702 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase1900Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1900ActorState();
		static_cast<FlowTestCase1900Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1911 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(e.code() == error_code_backup_invalid_url);
															#line 10715 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1900 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10741 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1900()
															#line 1900 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1900Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1900Actor>, public FlowTestCase1900ActorState<FlowTestCase1900Actor> {
															#line 10746 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1900Actor>::operator new;
	using FastAllocated<FlowTestCase1900Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18382982821220502016UL, 11740410569754726912UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1900 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1900Actor(UnitTestParameters const& params) 
															#line 10762 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1900ActorState<FlowTestCase1900Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1900", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15878836570831953920UL, 12575881630478421760UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1900Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1900");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1900", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1900 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1900( UnitTestParameters const& params ) {
															#line 1900 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1900Actor(params));
															#line 10794 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1900, "/MockS3Server/parseS3Request/EmptyQueryString")

#line 1916 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10800 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1917()
															#line 1917 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1917Actor>
															#line 1917 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1917ActorState {
															#line 10807 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1917 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1917ActorState(UnitTestParameters const& params) 
															#line 1917 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1917 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10814 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1917", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1917ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1917", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1919 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1920 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket/testobject?region=us-east-1";
															#line 1921 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1922 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1924 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1926 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1927 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "testobject");
															#line 1928 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1930 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1917Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1917ActorState(); static_cast<FlowTestCase1917Actor*>(this)->destroy(); return 0; }
															#line 10845 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase1917Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1917ActorState();
			static_cast<FlowTestCase1917Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1917ActorState();
		static_cast<FlowTestCase1917Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1917 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10869 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1917()
															#line 1917 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1917Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1917Actor>, public FlowTestCase1917ActorState<FlowTestCase1917Actor> {
															#line 10874 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1917Actor>::operator new;
	using FastAllocated<FlowTestCase1917Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18342030025431904000UL, 55070175025074688UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1917 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1917Actor(UnitTestParameters const& params) 
															#line 10890 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1917ActorState<FlowTestCase1917Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1917", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3348035063840958720UL, 6068966871818414848UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1917Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1917");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1917", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1917 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1917( UnitTestParameters const& params ) {
															#line 1917 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1917Actor(params));
															#line 10922 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1917, "/MockS3Server/parseS3Request/BucketParameterOverride")

#line 1932 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10928 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1933()
															#line 1933 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1933Actor>
															#line 1933 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1933ActorState {
															#line 10935 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1933 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1933ActorState(UnitTestParameters const& params) 
															#line 1933 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1933 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10942 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1933", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1933ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1933", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1935 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1936 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket/folder/subfolder/file.txt?region=us-east-1";
															#line 1937 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1938 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1940 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1942 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1943 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "folder/subfolder/file.txt");
															#line 1944 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1946 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1933Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1933ActorState(); static_cast<FlowTestCase1933Actor*>(this)->destroy(); return 0; }
															#line 10973 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase1933Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1933ActorState();
			static_cast<FlowTestCase1933Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1933ActorState();
		static_cast<FlowTestCase1933Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1933 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1933()
															#line 1933 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1933Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1933Actor>, public FlowTestCase1933ActorState<FlowTestCase1933Actor> {
															#line 11002 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1933Actor>::operator new;
	using FastAllocated<FlowTestCase1933Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2884131290916407296UL, 7667268626630998528UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1933 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1933Actor(UnitTestParameters const& params) 
															#line 11018 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1933ActorState<FlowTestCase1933Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1933", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8044786960800708608UL, 8974789255812525312UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1933Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1933");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1933", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1933 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1933( UnitTestParameters const& params ) {
															#line 1933 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1933Actor(params));
															#line 11050 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1933, "/MockS3Server/parseS3Request/ComplexPath")

#line 1948 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11056 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1949()
															#line 1949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1949Actor>
															#line 1949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1949ActorState {
															#line 11063 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1949ActorState(UnitTestParameters const& params) 
															#line 1949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1949", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1949ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1949", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1951 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1952 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1&param=value%3Dtest";
															#line 1953 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1954 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1956 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1958 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1959 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1960 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["param"] == "value=test");
															#line 1962 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1949Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1949ActorState(); static_cast<FlowTestCase1949Actor*>(this)->destroy(); return 0; }
															#line 11101 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase1949Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1949ActorState();
			static_cast<FlowTestCase1949Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1949ActorState();
		static_cast<FlowTestCase1949Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11125 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1949()
															#line 1949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1949Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1949Actor>, public FlowTestCase1949ActorState<FlowTestCase1949Actor> {
															#line 11130 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1949Actor>::operator new;
	using FastAllocated<FlowTestCase1949Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2049837440796185600UL, 5410070837381538816UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1949Actor(UnitTestParameters const& params) 
															#line 11146 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1949ActorState<FlowTestCase1949Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1949", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16714075822183166464UL, 16605525840637074432UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1949Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1949");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1949", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1949( UnitTestParameters const& params ) {
															#line 1949 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1949Actor(params));
															#line 11178 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1949, "/MockS3Server/parseS3Request/URLEncodedParameters")

#line 1964 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11184 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1965()
															#line 1965 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1965Actor>
															#line 1965 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1965ActorState {
															#line 11191 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1965 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1965ActorState(UnitTestParameters const& params) 
															#line 1965 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1965 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11198 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1965", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1965ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1965", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1967 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1968 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1969 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1970 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1972 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1974 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1975 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1976 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1978 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1965Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1965ActorState(); static_cast<FlowTestCase1965Actor*>(this)->destroy(); return 0; }
															#line 11229 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase1965Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1965ActorState();
			static_cast<FlowTestCase1965Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1965ActorState();
		static_cast<FlowTestCase1965Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1965 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11253 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1965()
															#line 1965 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1965Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1965Actor>, public FlowTestCase1965ActorState<FlowTestCase1965Actor> {
															#line 11258 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1965Actor>::operator new;
	using FastAllocated<FlowTestCase1965Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14917307551263776768UL, 17543491662304935936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1965 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1965Actor(UnitTestParameters const& params) 
															#line 11274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1965ActorState<FlowTestCase1965Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1965", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9795474850132498432UL, 1804973063523656960UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1965Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1965");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1965", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1965 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1965( UnitTestParameters const& params ) {
															#line 1965 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1965Actor(params));
															#line 11306 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1965, "/MockS3Server/parseS3Request/EmptyPath")

#line 1980 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11312 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1981()
															#line 1981 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1981Actor>
															#line 1981 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1981ActorState {
															#line 11319 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1981 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1981ActorState(UnitTestParameters const& params) 
															#line 1981 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1981 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11326 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1981", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1981ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1981", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1983 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1984 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1985 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1986 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1988 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1990 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1991 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1992 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1994 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1981Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1981ActorState(); static_cast<FlowTestCase1981Actor*>(this)->destroy(); return 0; }
															#line 11357 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase1981Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1981ActorState();
			static_cast<FlowTestCase1981Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1981ActorState();
		static_cast<FlowTestCase1981Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1981 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11381 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1981()
															#line 1981 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1981Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1981Actor>, public FlowTestCase1981ActorState<FlowTestCase1981Actor> {
															#line 11386 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1981Actor>::operator new;
	using FastAllocated<FlowTestCase1981Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(248537672425468928UL, 11139769260476009472UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1981 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1981Actor(UnitTestParameters const& params) 
															#line 11402 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1981ActorState<FlowTestCase1981Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1981", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1116805334484101120UL, 510896995444785408UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1981Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1981");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1981", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1981 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1981( UnitTestParameters const& params ) {
															#line 1981 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1981Actor(params));
															#line 11434 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1981, "/MockS3Server/parseS3Request/OnlyBucketInPath")

#line 1996 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11440 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1997()
															#line 1997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1997Actor>
															#line 1997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1997ActorState {
															#line 11447 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 1997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1997ActorState(UnitTestParameters const& params) 
															#line 1997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11454 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1997", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1997ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1997", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1999 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 2000 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1&version=1&encoding=utf8";
															#line 2001 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 2002 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 2004 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 2006 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 2007 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 2008 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["version"] == "1");
															#line 2009 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["encoding"] == "utf8");
															#line 2010 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams.size() == 3);
															#line 2012 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1997Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1997ActorState(); static_cast<FlowTestCase1997Actor*>(this)->destroy(); return 0; }
															#line 11489 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase1997Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1997ActorState();
			static_cast<FlowTestCase1997Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1997ActorState();
		static_cast<FlowTestCase1997Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11513 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1997()
															#line 1997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1997Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1997Actor>, public FlowTestCase1997ActorState<FlowTestCase1997Actor> {
															#line 11518 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1997Actor>::operator new;
	using FastAllocated<FlowTestCase1997Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7259263404961394176UL, 16907723823667126528UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1997Actor(UnitTestParameters const& params) 
															#line 11534 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1997ActorState<FlowTestCase1997Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1997", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17786562727881281280UL, 10828723253955129600UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1997Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1997");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1997", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1997( UnitTestParameters const& params ) {
															#line 1997 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1997Actor(params));
															#line 11566 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1997, "/MockS3Server/parseS3Request/MultipleParameters")

#line 2014 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11572 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2015()
															#line 2015 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2015Actor>
															#line 2015 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2015ActorState {
															#line 11579 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 2015 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2015ActorState(UnitTestParameters const& params) 
															#line 2015 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2015 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11586 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2015", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2015ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2015", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2017 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 2018 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?flag&region=us-east-1";
															#line 2019 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 2020 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 2022 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 2024 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 2025 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["flag"] == "");
															#line 2026 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 2028 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2015Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2015ActorState(); static_cast<FlowTestCase2015Actor*>(this)->destroy(); return 0; }
															#line 11617 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2015Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2015ActorState();
			static_cast<FlowTestCase2015Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2015ActorState();
		static_cast<FlowTestCase2015Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2015 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11641 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2015()
															#line 2015 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2015Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2015Actor>, public FlowTestCase2015ActorState<FlowTestCase2015Actor> {
															#line 11646 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2015Actor>::operator new;
	using FastAllocated<FlowTestCase2015Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17456401042348967680UL, 2777567165678411264UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2015 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2015Actor(UnitTestParameters const& params) 
															#line 11662 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2015ActorState<FlowTestCase2015Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2015", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3222446662451621632UL, 15978232096606025984UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2015Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2015");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2015", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2015 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2015( UnitTestParameters const& params ) {
															#line 2015 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2015Actor(params));
															#line 11694 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2015, "/MockS3Server/parseS3Request/ParametersWithoutValues")

#line 2030 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11700 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2031()
															#line 2031 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2031Actor>
															#line 2031 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2031ActorState {
															#line 11707 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 2031 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2031ActorState(UnitTestParameters const& params) 
															#line 2031 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2031 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11714 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2031", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2031ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2031", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2032 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=0-99";
															#line 2033 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2035 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2037 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2038 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 0);
															#line 2039 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 99);
															#line 2041 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2031Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2031ActorState(); static_cast<FlowTestCase2031Actor*>(this)->destroy(); return 0; }
															#line 11741 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2031Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2031ActorState();
			static_cast<FlowTestCase2031Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2031ActorState();
		static_cast<FlowTestCase2031Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2031 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11765 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2031()
															#line 2031 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2031Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2031Actor>, public FlowTestCase2031ActorState<FlowTestCase2031Actor> {
															#line 11770 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2031Actor>::operator new;
	using FastAllocated<FlowTestCase2031Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5182242147866646528UL, 5140098561376863488UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2031 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2031Actor(UnitTestParameters const& params) 
															#line 11786 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2031ActorState<FlowTestCase2031Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2031", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9156885137961901056UL, 3636434032296420608UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2031Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2031");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2031", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2031 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2031( UnitTestParameters const& params ) {
															#line 2031 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2031Actor(params));
															#line 11818 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2031, "/MockS3Server/RangeHeader/SimpleByteRange")

#line 2043 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11824 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2044()
															#line 2044 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2044Actor>
															#line 2044 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2044ActorState {
															#line 11831 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 2044 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2044ActorState(UnitTestParameters const& params) 
															#line 2044 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2044 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11838 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2044", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2044ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2044", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2045 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=100-199";
															#line 2046 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2048 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2050 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2051 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 100);
															#line 2052 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 199);
															#line 2054 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2044Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2044ActorState(); static_cast<FlowTestCase2044Actor*>(this)->destroy(); return 0; }
															#line 11865 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2044Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2044ActorState();
			static_cast<FlowTestCase2044Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2044ActorState();
		static_cast<FlowTestCase2044Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2044 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11889 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2044()
															#line 2044 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2044Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2044Actor>, public FlowTestCase2044ActorState<FlowTestCase2044Actor> {
															#line 11894 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2044Actor>::operator new;
	using FastAllocated<FlowTestCase2044Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14349788779115335424UL, 2283550468621418240UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2044 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2044Actor(UnitTestParameters const& params) 
															#line 11910 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2044ActorState<FlowTestCase2044Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2044", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14430895883109110528UL, 11449251985658075136UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2044Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2044");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2044", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2044 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2044( UnitTestParameters const& params ) {
															#line 2044 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2044Actor(params));
															#line 11942 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2044, "/MockS3Server/RangeHeader/MiddleRange")

#line 2056 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11948 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2057()
															#line 2057 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2057Actor>
															#line 2057 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2057ActorState {
															#line 11955 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 2057 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2057ActorState(UnitTestParameters const& params) 
															#line 2057 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2057 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11962 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2057", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2057ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2057", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2058 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=1000000-1999999";
															#line 2059 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2061 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2063 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2064 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 1000000);
															#line 2065 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 1999999);
															#line 2067 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2057Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2057ActorState(); static_cast<FlowTestCase2057Actor*>(this)->destroy(); return 0; }
															#line 11989 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2057Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2057ActorState();
			static_cast<FlowTestCase2057Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2057ActorState();
		static_cast<FlowTestCase2057Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2057 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12013 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2057()
															#line 2057 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2057Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2057Actor>, public FlowTestCase2057ActorState<FlowTestCase2057Actor> {
															#line 12018 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2057Actor>::operator new;
	using FastAllocated<FlowTestCase2057Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6081221913022728192UL, 8269826911632849408UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2057 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2057Actor(UnitTestParameters const& params) 
															#line 12034 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2057ActorState<FlowTestCase2057Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2057", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7137483070859857152UL, 7540132617862297600UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2057Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2057");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2057", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2057 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2057( UnitTestParameters const& params ) {
															#line 2057 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2057Actor(params));
															#line 12066 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2057, "/MockS3Server/RangeHeader/LargeOffsets")

#line 2069 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12072 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2070()
															#line 2070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2070Actor>
															#line 2070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2070ActorState {
															#line 12079 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 2070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2070ActorState(UnitTestParameters const& params) 
															#line 2070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12086 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2070", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2070ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2070", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2071 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "invalid-range";
															#line 2072 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2074 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2076 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2078 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2070Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2070ActorState(); static_cast<FlowTestCase2070Actor*>(this)->destroy(); return 0; }
															#line 12109 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2070Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2070ActorState();
			static_cast<FlowTestCase2070Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2070ActorState();
		static_cast<FlowTestCase2070Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12133 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2070()
															#line 2070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2070Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2070Actor>, public FlowTestCase2070ActorState<FlowTestCase2070Actor> {
															#line 12138 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2070Actor>::operator new;
	using FastAllocated<FlowTestCase2070Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(887590145955516672UL, 11016469424320138752UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2070Actor(UnitTestParameters const& params) 
															#line 12154 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2070ActorState<FlowTestCase2070Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2070", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14816885981023497472UL, 242571789973605888UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2070Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2070");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2070", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2070( UnitTestParameters const& params ) {
															#line 2070 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2070Actor(params));
															#line 12186 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2070, "/MockS3Server/RangeHeader/InvalidFormat")

#line 2080 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12192 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2081()
															#line 2081 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2081Actor>
															#line 2081 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2081ActorState {
															#line 12199 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 2081 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2081ActorState(UnitTestParameters const& params) 
															#line 2081 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2081 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12206 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2081", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2081ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2081", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2082 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "0-99";
															#line 2083 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2085 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2087 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2089 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2081Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2081ActorState(); static_cast<FlowTestCase2081Actor*>(this)->destroy(); return 0; }
															#line 12229 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2081Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2081ActorState();
			static_cast<FlowTestCase2081Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2081ActorState();
		static_cast<FlowTestCase2081Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2081 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12253 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2081()
															#line 2081 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2081Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2081Actor>, public FlowTestCase2081ActorState<FlowTestCase2081Actor> {
															#line 12258 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2081Actor>::operator new;
	using FastAllocated<FlowTestCase2081Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7174832620371994624UL, 1548895977002307840UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2081 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2081Actor(UnitTestParameters const& params) 
															#line 12274 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2081ActorState<FlowTestCase2081Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2081", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3218867917415945984UL, 13851157547664296448UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2081Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2081");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2081", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2081 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2081( UnitTestParameters const& params ) {
															#line 2081 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2081Actor(params));
															#line 12306 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2081, "/MockS3Server/RangeHeader/MissingBytesPrefix")

#line 2091 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12312 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2092()
															#line 2092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2092Actor>
															#line 2092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2092ActorState {
															#line 12319 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 2092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2092ActorState(UnitTestParameters const& params) 
															#line 2092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12326 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2092", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2092ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2092", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2093 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=0";
															#line 2094 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2096 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2098 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2100 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2092Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2092ActorState(); static_cast<FlowTestCase2092Actor*>(this)->destroy(); return 0; }
															#line 12349 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2092Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2092ActorState();
			static_cast<FlowTestCase2092Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2092ActorState();
		static_cast<FlowTestCase2092Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12373 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2092()
															#line 2092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2092Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2092Actor>, public FlowTestCase2092ActorState<FlowTestCase2092Actor> {
															#line 12378 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2092Actor>::operator new;
	using FastAllocated<FlowTestCase2092Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5478577434837285632UL, 15792448005118566400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2092Actor(UnitTestParameters const& params) 
															#line 12394 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2092ActorState<FlowTestCase2092Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2092", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1618482209217406976UL, 5090510695447585792UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2092Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2092");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2092", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2092( UnitTestParameters const& params ) {
															#line 2092 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2092Actor(params));
															#line 12426 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2092, "/MockS3Server/RangeHeader/MissingDash")

#line 2102 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12432 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2103()
															#line 2103 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2103Actor>
															#line 2103 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2103ActorState {
															#line 12439 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 2103 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2103ActorState(UnitTestParameters const& params) 
															#line 2103 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2103 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12446 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2103", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2103ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2103", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2104 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "";
															#line 2105 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2107 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2109 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2111 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2103Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2103ActorState(); static_cast<FlowTestCase2103Actor*>(this)->destroy(); return 0; }
															#line 12469 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2103Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2103ActorState();
			static_cast<FlowTestCase2103Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2103ActorState();
		static_cast<FlowTestCase2103Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2103 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12493 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2103()
															#line 2103 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2103Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2103Actor>, public FlowTestCase2103ActorState<FlowTestCase2103Actor> {
															#line 12498 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2103Actor>::operator new;
	using FastAllocated<FlowTestCase2103Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18395829316662135552UL, 11856503923665217280UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2103 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2103Actor(UnitTestParameters const& params) 
															#line 12514 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2103ActorState<FlowTestCase2103Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2103", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(789473734077782528UL, 12995386529830013696UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2103Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2103");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2103", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2103 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2103( UnitTestParameters const& params ) {
															#line 2103 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2103Actor(params));
															#line 12546 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2103, "/MockS3Server/RangeHeader/EmptyString")

#line 2113 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12552 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2114()
															#line 2114 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2114Actor>
															#line 2114 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2114ActorState {
															#line 12559 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 2114 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2114ActorState(UnitTestParameters const& params) 
															#line 2114 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2114 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12566 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2114", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2114ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2114", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2115 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=-100-200";
															#line 2116 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2118 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2121 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2123 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2114Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2114ActorState(); static_cast<FlowTestCase2114Actor*>(this)->destroy(); return 0; }
															#line 12589 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2114Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2114ActorState();
			static_cast<FlowTestCase2114Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2114ActorState();
		static_cast<FlowTestCase2114Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2114 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12613 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2114()
															#line 2114 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2114Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2114Actor>, public FlowTestCase2114ActorState<FlowTestCase2114Actor> {
															#line 12618 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2114Actor>::operator new;
	using FastAllocated<FlowTestCase2114Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14873768753982911744UL, 3081089651707877888UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2114 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2114Actor(UnitTestParameters const& params) 
															#line 12634 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2114ActorState<FlowTestCase2114Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2114", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9767834669430648064UL, 16025103715635521280UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2114Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2114");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2114", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2114 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2114( UnitTestParameters const& params ) {
															#line 2114 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2114Actor(params));
															#line 12666 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2114, "/MockS3Server/RangeHeader/NegativeStart")

#line 2125 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12672 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2126()
															#line 2126 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2126Actor>
															#line 2126 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2126ActorState {
															#line 12679 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 2126 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2126ActorState(UnitTestParameters const& params) 
															#line 2126 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2126 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12686 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2126", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2126ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2126", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2127 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=200-100";
															#line 2128 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2130 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2133 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2134 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 200);
															#line 2135 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 100);
															#line 2137 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2126Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2126ActorState(); static_cast<FlowTestCase2126Actor*>(this)->destroy(); return 0; }
															#line 12713 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2126Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2126ActorState();
			static_cast<FlowTestCase2126Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2126ActorState();
		static_cast<FlowTestCase2126Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2126 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12737 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2126()
															#line 2126 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2126Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2126Actor>, public FlowTestCase2126ActorState<FlowTestCase2126Actor> {
															#line 12742 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2126Actor>::operator new;
	using FastAllocated<FlowTestCase2126Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5454670598706600704UL, 14900897506518977280UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2126 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2126Actor(UnitTestParameters const& params) 
															#line 12758 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2126ActorState<FlowTestCase2126Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2126", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5663599179642057728UL, 16123320737846329856UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2126Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2126");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2126", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2126 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2126( UnitTestParameters const& params ) {
															#line 2126 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2126Actor(params));
															#line 12790 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2126, "/MockS3Server/RangeHeader/StartGreaterThanEnd")

#line 2139 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Real HTTP Server Implementation for ctests
															#line 12797 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via startMockS3ServerReal_impl()
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class StartMockS3ServerReal_implActor>
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerReal_implActorState {
															#line 12804 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerReal_implActorState(NetworkAddress const& listenAddress,std::string const& persistenceDir) 
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : listenAddress(listenAddress),
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(persistenceDir)
															#line 12813 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this));

	}
	~StartMockS3ServerReal_implActorState() 
	{
		fdb_probe_actor_destroy("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2142 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerRealStarting").detail("ListenAddress", listenAddress.toString());
															#line 2145 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!getGlobalStorage().persistenceEnabled)
															#line 12830 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
			{
															#line 2147 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (persistenceDir.empty())
															#line 12834 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				{
															#line 2148 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					persistenceDir = DEFAULT_MOCKS3_PERSISTENCE_DIR;
															#line 12838 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				}
															#line 2150 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				enableMockS3Persistence(persistenceDir);
															#line 2151 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerRealPersistenceEnabled") .detail("ListenAddress", listenAddress.toString()) .detail("PersistenceDir", persistenceDir);
															#line 2156 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = loadMockS3PersistedStateFuture();
															#line 2156 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12848 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 1;
															#line 2156 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*>(static_cast<StartMockS3ServerReal_implActor*>(this)));
															#line 12853 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2159 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		server = makeReference<HTTP::SimServerContext>();
															#line 2160 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		server->registerNewServer(listenAddress, makeReference<MockS3RequestHandler>());
															#line 2162 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerRealStarted") .detail("ListenAddress", listenAddress.toString()) .detail("ServerPtr", format("%p", server.getPtr()));
															#line 2167 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = Never();
															#line 2167 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12889 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 2;
															#line 2167 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*>(static_cast<StartMockS3ServerReal_implActor*>(this)));
															#line 12894 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerReal_implActor*>(this)->ActorCallback< StartMockS3ServerReal_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 2168 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerReal_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerReal_implActorState(); static_cast<StartMockS3ServerReal_implActor*>(this)->destroy(); return 0; }
															#line 12990 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<StartMockS3ServerReal_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 2168 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerReal_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerReal_implActorState(); static_cast<StartMockS3ServerReal_implActor*>(this)->destroy(); return 0; }
															#line 13002 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		new (&static_cast<StartMockS3ServerReal_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerReal_implActor*>(this)->ActorCallback< StartMockS3ServerReal_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	NetworkAddress listenAddress;
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 2159 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::SimServerContext> server;
															#line 13091 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
};
// This generated class is to be used only via startMockS3ServerReal_impl()
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerReal_implActor final : public Actor<Void>, public ActorCallback< StartMockS3ServerReal_implActor, 0, Void >, public ActorCallback< StartMockS3ServerReal_implActor, 1, Void >, public FastAllocated<StartMockS3ServerReal_implActor>, public StartMockS3ServerReal_implActorState<StartMockS3ServerReal_implActor> {
															#line 13096 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
public:
	using FastAllocated<StartMockS3ServerReal_implActor>::operator new;
	using FastAllocated<StartMockS3ServerReal_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9667678463422971392UL, 3487386179579716608UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StartMockS3ServerReal_implActor, 0, Void >;
friend struct ActorCallback< StartMockS3ServerReal_implActor, 1, Void >;
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerReal_implActor(NetworkAddress const& listenAddress,std::string const& persistenceDir) 
															#line 13114 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   StartMockS3ServerReal_implActorState<StartMockS3ServerReal_implActor>(listenAddress, persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15792334911365886208UL, 3052106341508935936UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("startMockS3ServerReal_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> startMockS3ServerReal_impl( NetworkAddress const& listenAddress, std::string const& persistenceDir ) {
															#line 2141 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new StartMockS3ServerReal_implActor(listenAddress, persistenceDir));
															#line 13148 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp.py_gen"
}

#line 2170 "/codebuild/output/src3091928138/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

Future<Void> startMockS3ServerReal(const NetworkAddress& listenAddress, const std::string& persistenceDir) {
	return startMockS3ServerReal_impl(listenAddress, persistenceDir);
}

// Wrapper for registerMockS3Server (calls the ACTOR implementation)
Future<Void> registerMockS3Server(std::string ip, std::string port) {
	return registerMockS3Server_impl(ip, port);
}
