#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
/*
 * StorageMetrics.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "flow/UnitTest.h"
#include "fdbserver/StorageMetrics.actor.h"
#include "flow/actorcompiler.h" // This must be the last #include.

CommonStorageCounters::CommonStorageCounters(const std::string& name,
                                             const std::string& id,
                                             const StorageServerMetrics* metrics)
  : cc(name, id), finishedQueries("FinishedQueries", cc), bytesQueried("BytesQueried", cc),
    bytesFetched("BytesFetched", cc), bytesInput("BytesInput", cc), mutationBytes("MutationBytes", cc),
    kvFetched("KVFetched", cc), mutations("Mutations", cc), setMutations("SetMutations", cc),
    clearRangeMutations("ClearRangeMutations", cc), kvFetchServed("KVFetchServed", cc),
    kvFetchBytesServed("KVFetchBytesServed", cc), fetchKeyErrors("FetchKeyErrors", cc) {
	if (metrics) {
		specialCounter(cc, "BytesStored", [metrics]() { return metrics->byteSample.getEstimate(allKeys); });
		specialCounter(cc, "BytesReadSampleCount", [metrics]() { return metrics->bytesReadSample.queue.size(); });
		specialCounter(cc, "OpsReadSampleCount", [metrics]() { return metrics->opsReadSample.queue.size(); });
		specialCounter(cc, "BytesWriteSampleCount", [metrics]() { return metrics->bytesWriteSample.queue.size(); });
		specialCounter(cc, "IopsReadSampleCount", [metrics]() { return metrics->iopsSample.queue.size(); });
	}
}

// TODO: update the cost as bytesReadPerKSecond + opsReadPerKSecond * SERVER_KNOBS->EMPTY_READ_PENALTY. The source of
// this model is Redwood will have a constant cost of seek of each read ops then read the actual data.
// As by 71.2.8, bytesReadPerKSecond should be larger than opsReadPerKSecond * SERVER_KNOBS->EMPTY_READ_PENALTY because
// the bytes always round to EMPTY_READ_PENALTY when the returned result size is less than EMPTY_READ_PENALTY. This cost
// is different from what tag throttling use to produce throttling decision.
int64_t StorageMetrics::readLoadKSecond() const {
	return std::max(bytesReadPerKSecond, opsReadPerKSecond * SERVER_KNOBS->EMPTY_READ_PENALTY);
}

int64_t StorageMetricSample::getEstimate(KeyRangeRef keys) const {
	return sample.sumRange(keys.begin, keys.end);
}

KeyRef StorageMetricSample::splitEstimate(KeyRangeRef range, int64_t offset, bool front) const {
	auto fwd_split = sample.index(front ? sample.sumTo(sample.lower_bound(range.begin)) + offset
	                                    : sample.sumTo(sample.lower_bound(range.end)) - offset);

	if (fwd_split == sample.end() || *fwd_split >= range.end)
		return range.end;

	if (!front && *fwd_split <= range.begin)
		return range.begin;

	auto bck_split = fwd_split;

	// Butterfly search - start at midpoint then go in both directions.
	while ((fwd_split != sample.end() && *fwd_split < range.end) ||
	       (bck_split != sample.begin() && *bck_split > range.begin)) {
		if (bck_split != sample.begin() && *bck_split > range.begin) {
			auto it = bck_split;
			bck_split.decrementNonEnd();

			KeyRef split = keyBetween(KeyRangeRef(
			    bck_split != sample.begin() ? std::max<KeyRef>(*bck_split, range.begin) : range.begin, *it));
			if (!front || (getEstimate(KeyRangeRef(range.begin, split)) > 0 &&
			               split.size() <= CLIENT_KNOBS->SPLIT_KEY_SIZE_LIMIT))
				return split;
		}

		if (fwd_split != sample.end() && *fwd_split < range.end) {
			auto it = fwd_split;
			++it;

			KeyRef split =
			    keyBetween(KeyRangeRef(*fwd_split, it != sample.end() ? std::min<KeyRef>(*it, range.end) : range.end));
			if (front ||
			    (getEstimate(KeyRangeRef(split, range.end)) > 0 && split.size() <= CLIENT_KNOBS->SPLIT_KEY_SIZE_LIMIT))
				return split;

			fwd_split = it;
		}
	}

	// If we didn't return above, we didn't find anything.
	TraceEvent(SevWarn, "CannotSplitLastSampleKey").detail("Range", range).detail("Offset", offset);
	return front ? range.end : range.begin;
}

// Get the current estimated metrics for the given keys
StorageMetrics StorageServerMetrics::getMetrics(KeyRangeRef const& keys) const {
	StorageMetrics result;
	result.bytes = byteSample.getEstimate(keys);
	result.bytesWrittenPerKSecond =
	    bytesWriteSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	result.iosPerKSecond = iopsSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;

	result.bytesReadPerKSecond =
	    bytesReadSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	result.opsReadPerKSecond =
	    opsReadSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	return result;
}

// Called when metrics should change (IO for a given key)
// Notifies waiting WaitMetricsRequests through waitMetricsMap, and updates metricsAverageQueue and metricsSampleMap
void StorageServerMetrics::notify(const Key& key, StorageMetrics& metrics) {
	ASSERT(metrics.bytes == 0); // ShardNotifyMetrics
	if (g_network->isSimulated()) {
		CODE_PROBE(metrics.bytesWrittenPerKSecond != 0, "ShardNotifyMetrics bytes");
		CODE_PROBE(metrics.iosPerKSecond != 0, "ShardNotifyMetrics ios");
		CODE_PROBE(metrics.bytesReadPerKSecond != 0, "ShardNotifyMetrics bytesRead");
		CODE_PROBE(metrics.opsReadPerKSecond != 0, "ShardNotifyMetrics opsRead");
	}

	double expire = now() + SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL;

	StorageMetrics notifyMetrics;

	if (metrics.bytesWrittenPerKSecond)
		notifyMetrics.bytesWrittenPerKSecond =
		    bytesWriteSample.addAndExpire(key, metrics.bytesWrittenPerKSecond, expire) *
		    SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	if (metrics.iosPerKSecond)
		notifyMetrics.iosPerKSecond = iopsSample.addAndExpire(key, metrics.iosPerKSecond, expire) *
		                              SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	if (metrics.bytesReadPerKSecond)
		notifyMetrics.bytesReadPerKSecond = bytesReadSample.addAndExpire(key, metrics.bytesReadPerKSecond, expire) *
		                                    SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	if (metrics.opsReadPerKSecond) {
		notifyMetrics.opsReadPerKSecond = opsReadSample.addAndExpire(key, metrics.opsReadPerKSecond, expire) *
		                                  SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	}

	if (!notifyMetrics.allZero()) {
		auto& v = waitMetricsMap[key];
		for (int i = 0; i < v.size(); i++) {
			if (g_network->isSimulated()) {
				CODE_PROBE(true, "shard notify metrics");
			}
			// ShardNotifyMetrics
			v[i].send(notifyMetrics);
		}
	}
}

// Due to the fact that read sampling will be called on all reads, use this specialized function to avoid overhead
// around branch misses and unnecessary stack allocation which eventually adds up under heavy load.
void StorageServerMetrics::notifyBytesReadPerKSecond(const Key& key, int64_t in) {
	double expire = now() + SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL;
	int64_t bytesReadPerKSecond =
	    bytesReadSample.addAndExpire(key, in, expire) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	int64_t opsReadPerKSecond =
	    opsReadSample.addAndExpire(key, 1, expire) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;

	if (bytesReadPerKSecond > 0 || opsReadPerKSecond > 0) {
		StorageMetrics notifyMetrics;
		notifyMetrics.bytesReadPerKSecond = bytesReadPerKSecond;
		notifyMetrics.opsReadPerKSecond = opsReadPerKSecond;
		auto& v = waitMetricsMap[key];
		for (int i = 0; i < v.size(); i++) {
			CODE_PROBE(bytesReadPerKSecond > 0, "ShardNotifyMetrics bytesRead");
			CODE_PROBE(opsReadPerKSecond > 0, "ShardNotifyMetrics opsRead");
			v[i].send(notifyMetrics);
		}
	}
}

// Called by StorageServerDisk when the size of a key in byteSample changes, to notify WaitMetricsRequest
// Should not be called for keys past allKeys.end
void StorageServerMetrics::notifyBytes(
    RangeMap<Key, std::vector<PromiseStream<StorageMetrics>>, KeyRangeRef>::iterator shard,
    int64_t bytes) {
	ASSERT(shard.end() <= allKeys.end);

	StorageMetrics notifyMetrics;
	notifyMetrics.bytes = bytes;
	auto size = shard->cvalue().size();
	for (int i = 0; i < size; i++) {
		// fmt::print("NotifyBytes {} {}\n", shard->value().size(), shard->range().toString());
		CODE_PROBE(true, "notifyBytes");
		shard.value()[i].send(notifyMetrics);
	}
}

// Called by StorageServerDisk when the size of a key in byteSample changes, to notify WaitMetricsRequest
void StorageServerMetrics::notifyBytes(const KeyRef& key, int64_t bytes) {
	if (key >= allKeys.end) // Do not notify on changes to internal storage server state
		return;

	notifyBytes(waitMetricsMap.rangeContaining(key), bytes);
}

// Called when a range of keys becomes unassigned (and therefore not readable), to notify waiting
// WaitMetricsRequests (also other types of wait
//   requests in the future?)
void StorageServerMetrics::notifyNotReadable(KeyRangeRef keys) {
	auto rs = waitMetricsMap.intersectingRanges(keys);
	for (auto r = rs.begin(); r != rs.end(); ++r) {
		auto& v = r->value();
		CODE_PROBE(v.size(), "notifyNotReadable() sending errors to intersecting ranges");
		for (int n = 0; n < v.size(); n++)
			v[n].sendError(wrong_shard_server());
	}
}

// Called periodically (~1 sec intervals) to remove older IOs from the averages
// Removes old entries from metricsAverageQueue, updates metricsSampleMap accordingly, and notifies
//   WaitMetricsRequests through waitMetricsMap.
void StorageServerMetrics::poll() {
	{
		StorageMetrics m;
		m.bytesWrittenPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		bytesWriteSample.poll(waitMetricsMap, m);
	}
	{
		StorageMetrics m;
		m.iosPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		iopsSample.poll(waitMetricsMap, m);
	}
	{
		StorageMetrics m;
		m.bytesReadPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		bytesReadSample.poll(waitMetricsMap, m);
	}
	{
		StorageMetrics m;
		m.opsReadPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		opsReadSample.poll(waitMetricsMap, m);
	}
	// bytesSample doesn't need polling because we never call addExpire() on it
}

// This function can run on untrusted user data.  We must validate all divisions carefully.
KeyRef StorageServerMetrics::getSplitKey(int64_t remaining,
                                         int64_t estimated,
                                         int64_t limits,
                                         int64_t used,
                                         int64_t infinity,
                                         bool isLastShard,
                                         const StorageMetricSample& sample,
                                         double divisor,
                                         KeyRef const& lastKey,
                                         KeyRef const& key,
                                         bool hasUsed) const {
	ASSERT(remaining >= 0);
	ASSERT(limits > 0);
	ASSERT(divisor > 0);

	if (limits < infinity / 2) {
		int64_t expectedSize;
		if (isLastShard || remaining > estimated) {
			double remaining_divisor = (double(remaining) / limits) + 0.5;
			expectedSize = remaining / remaining_divisor;
		} else {
			// If we are here, then estimated >= remaining >= 0
			double estimated_divisor = (double(estimated) / limits) + 0.5;
			expectedSize = remaining / estimated_divisor;
		}

		if (remaining > expectedSize) {
			// This does the conversion from native units to bytes using the divisor.
			double offset = (expectedSize - used) / divisor;
			if (offset <= 0)
				return hasUsed ? lastKey : key;
			return sample.splitEstimate(
			    KeyRangeRef(lastKey, key),
			    offset * ((1.0 - SERVER_KNOBS->SPLIT_JITTER_AMOUNT) +
			              2 * deterministicRandom()->random01() * SERVER_KNOBS->SPLIT_JITTER_AMOUNT));
		}
	}

	return key;
}

void StorageServerMetrics::splitMetrics(SplitMetricsRequest req) const {
	int minSplitBytes = req.minSplitBytes.present() ? req.minSplitBytes.get() : SERVER_KNOBS->MIN_SHARD_BYTES;
	int minSplitWriteTraffic = SERVER_KNOBS->SHARD_SPLIT_BYTES_PER_KSEC;
	try {
		SplitMetricsReply reply;
		KeyRef lastKey = req.keys.begin;
		StorageMetrics used = req.used;
		StorageMetrics estimated = req.estimated;
		StorageMetrics remaining = getMetrics(req.keys) + used;

		//TraceEvent("SplitMetrics").detail("Begin", req.keys.begin).detail("End", req.keys.end).detail("Remaining", remaining.bytes).detail("Used", used.bytes).detail("MinSplitBytes", minSplitBytes);

		while (true) {
			if (remaining.bytes < 2 * minSplitBytes && (!SERVER_KNOBS->ENABLE_WRITE_BASED_SHARD_SPLIT ||
			                                            remaining.bytesWrittenPerKSecond < minSplitWriteTraffic))
				break;
			KeyRef key = req.keys.end;
			bool hasUsed = used.bytes != 0 || used.bytesWrittenPerKSecond != 0 || used.iosPerKSecond != 0;
			key = getSplitKey(remaining.bytes,
			                  estimated.bytes,
			                  req.limits.bytes,
			                  used.bytes,
			                  req.limits.infinity,
			                  req.isLastShard,
			                  byteSample,
			                  1,
			                  lastKey,
			                  key,
			                  hasUsed);
			if (used.bytes < minSplitBytes)
				key = std::max(
				    key, byteSample.splitEstimate(KeyRangeRef(lastKey, req.keys.end), minSplitBytes - used.bytes));
			key = getSplitKey(remaining.iosPerKSecond,
			                  estimated.iosPerKSecond,
			                  req.limits.iosPerKSecond,
			                  used.iosPerKSecond,
			                  req.limits.infinity,
			                  req.isLastShard,
			                  iopsSample,
			                  SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS,
			                  lastKey,
			                  key,
			                  hasUsed);
			key = getSplitKey(remaining.bytesWrittenPerKSecond,
			                  estimated.bytesWrittenPerKSecond,
			                  req.limits.bytesWrittenPerKSecond,
			                  used.bytesWrittenPerKSecond,
			                  req.limits.infinity,
			                  req.isLastShard,
			                  bytesWriteSample,
			                  SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS,
			                  lastKey,
			                  key,
			                  hasUsed);
			ASSERT(key != lastKey || hasUsed);
			if (key == req.keys.end)
				break;
			reply.splits.push_back_deep(reply.splits.arena(), key);
			if (reply.splits.size() > SERVER_KNOBS->SPLIT_METRICS_MAX_ROWS) {
				reply.more = true;
				break;
			}

			StorageMetrics diff = (getMetrics(KeyRangeRef(lastKey, key)) + used);
			remaining -= diff;
			estimated -= diff;

			used = StorageMetrics();
			lastKey = key;
		}

		reply.used = reply.more ? StorageMetrics() : getMetrics(KeyRangeRef(lastKey, req.keys.end)) + used;
		req.reply.send(reply);
	} catch (Error& e) {
		req.reply.sendError(e);
	}
}

void StorageServerMetrics::getStorageMetrics(GetStorageMetricsRequest req,
                                             StorageBytes sb,
                                             double bytesInputRate,
                                             int64_t versionLag,
                                             double lastUpdate,
                                             int64_t bytesDurable,
                                             int64_t bytesInput,
                                             int ongoingBulkLoadTaskCount) const {
	GetStorageMetricsReply rep;

	// SOMEDAY: make bytes dynamic with hard disk space
	rep.load = getMetrics(allKeys);

	if (sb.free < 1e9) {
		TraceEvent(SevWarn, "PhysicalDiskMetrics")
		    .suppressFor(60.0)
		    .detail("Free", sb.free)
		    .detail("Total", sb.total)
		    .detail("Available", sb.available)
		    .detail("Load", rep.load.bytes);
	}

	rep.available.bytes = sb.available;
	rep.available.iosPerKSecond = 10e6;
	rep.available.bytesWrittenPerKSecond = 100e9;
	rep.available.bytesReadPerKSecond = 100e9;

	rep.capacity.bytes = sb.total;
	rep.capacity.iosPerKSecond = 10e6;
	rep.capacity.bytesWrittenPerKSecond = 100e9;
	rep.capacity.bytesReadPerKSecond = 100e9;

	rep.bytesInputRate = bytesInputRate;

	rep.versionLag = versionLag;
	rep.lastUpdate = lastUpdate;

	rep.bytesDurable = bytesDurable;
	rep.bytesInput = bytesInput;

	rep.ongoingBulkLoadTaskCount = ongoingBulkLoadTaskCount;

	req.reply.send(rep);
}

// Equally split the metrics (specified by splitType) of parentRange into chunkCount and return all the sampled metrics
// (bytes, readBytes and readOps) of each chunk
// NOTE: update unit test "equalDivide" after change
std::vector<ReadHotRangeWithMetrics> StorageServerMetrics::getReadHotRanges(KeyRangeRef parentRange,
                                                                            int chunkCount,
                                                                            uint8_t splitType) const {
	const StorageMetricSample* sampler = nullptr;
	switch (splitType) {
	case ReadHotSubRangeRequest::SplitType::BYTES:
		sampler = &byteSample;
		break;
	case ReadHotSubRangeRequest::SplitType::READ_BYTES:
		sampler = &bytesReadSample;
		break;
	case ReadHotSubRangeRequest::SplitType::READ_OPS:
		sampler = &opsReadSample;
		break;
	default:
		ASSERT(false);
	}

	std::vector<ReadHotRangeWithMetrics> toReturn;
	if (sampler->sample.empty()) {
		return toReturn;
	}

	auto total = sampler->getEstimate(parentRange);
	double splitChunk = std::max(1.0, (double)total / chunkCount);

	KeyRef beginKey = parentRange.begin;
	while (parentRange.contains(beginKey)) {
		auto beginIt = sampler->sample.lower_bound(beginKey);
		if (beginIt == sampler->sample.end()) {
			break;
		}
		auto endIt = sampler->sample.index(sampler->sample.sumTo(beginIt) + splitChunk - 1);
		// because index return x where sumTo(x+1) (that including sample at x) > metrics, we have to forward endIt here
		if (endIt != sampler->sample.end())
			++endIt;

		if (endIt == sampler->sample.end()) {
			KeyRangeRef lastRange(beginKey, parentRange.end);
			toReturn.emplace_back(
			    lastRange,
			    byteSample.getEstimate(lastRange),
			    (double)bytesReadSample.getEstimate(lastRange) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL,
			    (double)opsReadSample.getEstimate(lastRange) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
			break;
		}

		ASSERT_LT(beginKey, *endIt);
		KeyRangeRef range(beginKey, *endIt);
		toReturn.emplace_back(
		    range,
		    byteSample.getEstimate(range),
		    (double)bytesReadSample.getEstimate(range) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL,
		    (double)opsReadSample.getEstimate(range) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);

		beginKey = *endIt;
	}
	return toReturn;
}

// Given a read hot shard, this function will divide the shard into chunks and find those chunks whose
// readBytes/sizeBytes exceeds the `readDensityRatio`. Please make sure to run unit tests
// `StorageMetricsSampleTests.txt` after change made.
std::vector<ReadHotRangeWithMetrics> StorageServerMetrics::_getReadHotRanges(
    KeyRangeRef shard,
    double readDensityRatio,
    int64_t baseChunkSize,
    int64_t minShardReadBandwidthPerKSeconds) const {
	std::vector<ReadHotRangeWithMetrics> toReturn;

	double shardSize = (double)byteSample.getEstimate(shard);
	int64_t shardReadBandwidth = bytesReadSample.getEstimate(shard);
	if (shardReadBandwidth * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS <=
	    minShardReadBandwidthPerKSeconds) {
		return toReturn;
	}
	if (shardSize <= baseChunkSize) {
		// Shard is small, use it as is
		if (bytesReadSample.getEstimate(shard) > (readDensityRatio * shardSize)) {
			toReturn.emplace_back(shard,
			                      bytesReadSample.getEstimate(shard) / shardSize,
			                      bytesReadSample.getEstimate(shard) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
		}
		return toReturn;
	}
	KeyRef beginKey = shard.begin;
	auto endKey =
	    byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + baseChunkSize);
	while (endKey != byteSample.sample.end()) {
		if (*endKey > shard.end) {
			endKey = byteSample.sample.lower_bound(shard.end);
			if (*endKey == beginKey) {
				// No need to increment endKey since otherwise it would stuck here forever.
				break;
			}
		}
		if (*endKey == beginKey) {
			++endKey;
			continue;
		}
		if (bytesReadSample.getEstimate(KeyRangeRef(beginKey, *endKey)) >
		    (readDensityRatio * std::max(baseChunkSize, byteSample.getEstimate(KeyRangeRef(beginKey, *endKey))))) {
			auto range = KeyRangeRef(beginKey, *endKey);
			if (!toReturn.empty() && toReturn.back().keys.end == range.begin) {
				// in case two consecutive chunks both are over the ratio, merge them.
				range = KeyRangeRef(toReturn.back().keys.begin, *endKey);
				toReturn.pop_back();
			}
			toReturn.emplace_back(range,
			                      (double)bytesReadSample.getEstimate(range) /
			                          std::max(baseChunkSize, byteSample.getEstimate(range)),
			                      bytesReadSample.getEstimate(range) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
		}
		beginKey = *endKey;
		endKey =
		    byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + baseChunkSize);
	}
	return toReturn;
}

int64_t StorageServerMetrics::getHotShards(const KeyRange& range) const {
	auto bytesWrittenPerKSecond =
	    bytesWriteSample.getEstimate(range) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	return bytesWrittenPerKSecond;
}

void StorageServerMetrics::getReadHotRanges(ReadHotSubRangeRequest req) const {
	ReadHotSubRangeReply reply;
	auto _ranges = getReadHotRanges(req.keys, req.chunkCount, req.type);
	reply.readHotRanges = VectorRef(_ranges.data(), _ranges.size());
	req.reply.send(reply);
}

void StorageServerMetrics::getSplitPoints(SplitRangeRequest req, Optional<KeyRef> prefix) const {
	SplitRangeReply reply;
	KeyRangeRef range = req.keys;
	if (prefix.present()) {
		range = range.withPrefix(prefix.get(), req.arena);
	}
	std::vector<KeyRef> points = getSplitPoints(range, req.chunkSize, prefix);

	reply.splitPoints.append_deep(reply.splitPoints.arena(), points.data(), points.size());
	req.reply.send(reply);
}

std::vector<KeyRef> StorageServerMetrics::getSplitPoints(KeyRangeRef range,
                                                         int64_t chunkSize,
                                                         Optional<KeyRef> prefixToRemove) const {
	std::vector<KeyRef> toReturn;
	KeyRef beginKey = range.begin;
	IndexedSet<Key, int64_t>::const_iterator endKey =
	    byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + chunkSize);
	while (endKey != byteSample.sample.end()) {
		if (*endKey > range.end) {
			break;
		}
		if (*endKey == beginKey) {
			++endKey;
			continue;
		}
		KeyRef splitPoint = *endKey;
		if (prefixToRemove.present()) {
			splitPoint = splitPoint.removePrefix(prefixToRemove.get());
		}
		toReturn.push_back(splitPoint);
		beginKey = *endKey;
		endKey = byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + chunkSize);
	}
	return toReturn;
}

void StorageServerMetrics::collapse(KeyRangeMap<int>& map, KeyRef const& key) {
	auto range = map.rangeContaining(key);
	if (range == map.ranges().begin() || range == map.ranges().end())
		return;
	int value = range->value();
	auto prev = range;
	--prev;
	if (prev->value() != value)
		return;
	KeyRange keys = KeyRangeRef(prev->begin(), range->end());
	map.insert(keys, value);
}

void StorageServerMetrics::add(KeyRangeMap<int>& map, KeyRangeRef const& keys, int delta) {
	auto rs = map.modify(keys);
	for (auto r = rs.begin(); r != rs.end(); ++r)
		r->value() += delta;
	collapse(map, keys.begin);
	collapse(map, keys.end);
}

// Returns the sampled metric value (possibly 0, possibly increased by the sampling factor)
int64_t TransientStorageMetricSample::addAndExpire(const Key& key, int64_t metric, double expiration) {
	auto x = add(key, metric);
	if (x)
		queue.emplace_back(expiration, std::make_pair(key, -x));
	return x;
}

// FIXME: both versions of erase are broken, because they do not remove items in the queue with will subtract a
// metric from the value sometime in the future
int64_t TransientStorageMetricSample::erase(KeyRef key) {
	auto it = sample.find(key);
	if (it == sample.end())
		return 0;
	int64_t x = sample.getMetric(it);
	sample.erase(it);
	return x;
}

void TransientStorageMetricSample::erase(KeyRangeRef keys) {
	sample.erase(keys.begin, keys.end);
}

bool TransientStorageMetricSample::roll(int64_t metric) const {
	return deterministicRandom()->random01() < (double)metric / metricUnitsPerSample; //< SOMEDAY: Better randomInt64?
}

void TransientStorageMetricSample::poll(KeyRangeMap<std::vector<PromiseStream<StorageMetrics>>>& waitMap,
                                        StorageMetrics metrics) {
	double now = ::now();
	while (queue.size() && queue.front().first <= now) {
		KeyRef key = queue.front().second.first;
		int64_t delta = queue.front().second.second;
		ASSERT(delta != 0);

		auto [m, it] = sample.addMetric(key, delta);
		if (m == 0)
			sample.erase(it);

		StorageMetrics deltaM = metrics * delta;
		auto v = waitMap[key];
		for (int i = 0; i < v.size(); i++) {
			CODE_PROBE(true, "TransientStorageMetricSample poll update");
			v[i].send(deltaM);
		}

		queue.pop_front();
	}
}

void TransientStorageMetricSample::poll() {
	double now = ::now();
	while (queue.size() && queue.front().first <= now) {
		KeyRef key = queue.front().second.first;
		int64_t delta = queue.front().second.second;
		ASSERT(delta != 0);

		auto [m, it] = sample.addMetric(key, delta);
		if (m == 0)
			sample.erase(it);

		queue.pop_front();
	}
}

int64_t TransientStorageMetricSample::add(const Key& key, int64_t metric) {
	if (!metric)
		return 0;
	int64_t mag = metric < 0 ? -metric : metric;

	if (mag < metricUnitsPerSample) {
		if (!roll(mag))
			return 0;
		metric = metric < 0 ? -metricUnitsPerSample : metricUnitsPerSample;
	}

	auto [m, it] = sample.addMetric(key, metric);
	if (m == 0)
		sample.erase(it);

	return metric;
}

															#line 689 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase687()
															#line 687 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase687Actor>
															#line 687 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase687ActorState {
															#line 696 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 687 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase687ActorState(UnitTestParameters const& params) 
															#line 687 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 687 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 703 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase687", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase687ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase687", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 688 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageMetricSample s(1000);
															#line 689 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Apple"_sr, 1000);
															#line 690 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Banana"_sr, 2000);
															#line 691 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Cat"_sr, 1000);
															#line 692 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Cathode"_sr, 1000);
															#line 693 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Dog"_sr, 1000);
															#line 695 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(s.getEstimate(KeyRangeRef("A"_sr, "D"_sr)) == 5000);
															#line 696 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(s.getEstimate(KeyRangeRef("A"_sr, "E"_sr)) == 6000);
															#line 697 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(s.getEstimate(KeyRangeRef("B"_sr, "C"_sr)) == 2000);
															#line 702 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase687Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase687ActorState(); static_cast<FlowTestCase687Actor*>(this)->destroy(); return 0; }
															#line 736 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase687Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase687ActorState();
			static_cast<FlowTestCase687Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase687ActorState();
		static_cast<FlowTestCase687Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 687 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 760 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase687()
															#line 687 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase687Actor final : public Actor<Void>, public FastAllocated<FlowTestCase687Actor>, public FlowTestCase687ActorState<FlowTestCase687Actor> {
															#line 765 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase687Actor>::operator new;
	using FastAllocated<FlowTestCase687Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14692192202049048576UL, 4942413521944935936UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 687 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase687Actor(UnitTestParameters const& params) 
															#line 781 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase687ActorState<FlowTestCase687Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase687", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11787494980133412608UL, 3635602780786964736UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase687Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase687");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase687", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 687 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase687( UnitTestParameters const& params ) {
															#line 687 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase687Actor(params));
															#line 813 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase687, "/fdbserver/StorageMetricSample/simple")

#line 704 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 819 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase705()
															#line 705 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase705Actor>
															#line 705 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase705ActorState {
															#line 826 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 705 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase705ActorState(UnitTestParameters const& params) 
															#line 705 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 705 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 833 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase705", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase705ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase705", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 707 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 708 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 710 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 200 * sampleUnit);
															#line 711 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 800 * sampleUnit);
															#line 712 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 713 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 714 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 715 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 716 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 717 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 719 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 2000 * sampleUnit, {});
															#line 721 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 1 && t[0] == "Bah"_sr);
															#line 723 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase705Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase705ActorState(); static_cast<FlowTestCase705Actor*>(this)->destroy(); return 0; }
															#line 872 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase705Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase705ActorState();
			static_cast<FlowTestCase705Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase705ActorState();
		static_cast<FlowTestCase705Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 705 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 896 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase705()
															#line 705 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase705Actor final : public Actor<Void>, public FastAllocated<FlowTestCase705Actor>, public FlowTestCase705ActorState<FlowTestCase705Actor> {
															#line 901 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase705Actor>::operator new;
	using FastAllocated<FlowTestCase705Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12617969036537335296UL, 8723123106767352576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 705 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase705Actor(UnitTestParameters const& params) 
															#line 917 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase705ActorState<FlowTestCase705Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase705", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11672349708822624000UL, 7238086034748890112UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase705Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase705");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase705", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 705 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase705( UnitTestParameters const& params ) {
															#line 705 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase705Actor(params));
															#line 949 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase705, "/fdbserver/StorageMetricSample/rangeSplitPoints/simple")

#line 725 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 955 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase726()
															#line 726 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase726Actor>
															#line 726 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase726ActorState {
															#line 962 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 726 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase726ActorState(UnitTestParameters const& params) 
															#line 726 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 726 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 969 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase726", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase726ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase726", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 728 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 729 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 731 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 200 * sampleUnit);
															#line 732 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 800 * sampleUnit);
															#line 733 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 734 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 735 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 736 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 737 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 738 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 740 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 600 * sampleUnit, {});
															#line 742 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 3 && t[0] == "Absolute"_sr && t[1] == "Apple"_sr && t[2] == "Bah"_sr);
															#line 744 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase726Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase726ActorState(); static_cast<FlowTestCase726Actor*>(this)->destroy(); return 0; }
															#line 1008 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase726Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase726ActorState();
			static_cast<FlowTestCase726Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase726ActorState();
		static_cast<FlowTestCase726Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 726 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1032 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase726()
															#line 726 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase726Actor final : public Actor<Void>, public FastAllocated<FlowTestCase726Actor>, public FlowTestCase726ActorState<FlowTestCase726Actor> {
															#line 1037 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase726Actor>::operator new;
	using FastAllocated<FlowTestCase726Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1830503280844335104UL, 12964074587290571008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 726 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase726Actor(UnitTestParameters const& params) 
															#line 1053 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase726ActorState<FlowTestCase726Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase726", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(182326355348364544UL, 12557845224503988992UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase726Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase726");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase726", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 726 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase726( UnitTestParameters const& params ) {
															#line 726 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase726Actor(params));
															#line 1085 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase726, "/fdbserver/StorageMetricSample/rangeSplitPoints/multipleReturnedPoints")

#line 746 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1091 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase747()
															#line 747 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase747Actor>
															#line 747 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase747ActorState {
															#line 1098 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 747 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase747ActorState(UnitTestParameters const& params) 
															#line 747 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 747 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1105 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase747", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase747ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase747", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 749 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 750 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 752 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 200 * sampleUnit);
															#line 753 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 800 * sampleUnit);
															#line 754 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 755 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 756 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 757 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 758 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 759 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 761 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 10000 * sampleUnit, {});
															#line 763 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 0);
															#line 765 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase747Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase747ActorState(); static_cast<FlowTestCase747Actor*>(this)->destroy(); return 0; }
															#line 1144 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase747Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase747ActorState();
			static_cast<FlowTestCase747Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase747ActorState();
		static_cast<FlowTestCase747Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 747 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1168 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase747()
															#line 747 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase747Actor final : public Actor<Void>, public FastAllocated<FlowTestCase747Actor>, public FlowTestCase747ActorState<FlowTestCase747Actor> {
															#line 1173 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase747Actor>::operator new;
	using FastAllocated<FlowTestCase747Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8474064490440917248UL, 2905086112027853568UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 747 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase747Actor(UnitTestParameters const& params) 
															#line 1189 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase747ActorState<FlowTestCase747Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase747", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7398373504158579968UL, 9257727052614553856UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase747Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase747");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase747", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 747 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase747( UnitTestParameters const& params ) {
															#line 747 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase747Actor(params));
															#line 1221 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase747, "/fdbserver/StorageMetricSample/rangeSplitPoints/noneSplitable")

#line 767 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1227 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase768()
															#line 768 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase768Actor>
															#line 768 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase768ActorState {
															#line 1234 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 768 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase768ActorState(UnitTestParameters const& params) 
															#line 768 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 768 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1241 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase768", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase768ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase768", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 770 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 771 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 773 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 774 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 775 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 10 * sampleUnit);
															#line 776 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 777 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 778 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 20 * sampleUnit);
															#line 779 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 10 * sampleUnit);
															#line 780 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 30 * sampleUnit);
															#line 782 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 1000 * sampleUnit, {});
															#line 784 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 0);
															#line 786 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase768Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase768ActorState(); static_cast<FlowTestCase768Actor*>(this)->destroy(); return 0; }
															#line 1280 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase768Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase768ActorState();
			static_cast<FlowTestCase768Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase768ActorState();
		static_cast<FlowTestCase768Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 768 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1304 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase768()
															#line 768 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase768Actor final : public Actor<Void>, public FastAllocated<FlowTestCase768Actor>, public FlowTestCase768ActorState<FlowTestCase768Actor> {
															#line 1309 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase768Actor>::operator new;
	using FastAllocated<FlowTestCase768Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6759239995469688064UL, 11195481115530158592UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 768 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase768Actor(UnitTestParameters const& params) 
															#line 1325 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase768ActorState<FlowTestCase768Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase768", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8182258776000434944UL, 9833904026492196608UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase768Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase768");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase768", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 768 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase768( UnitTestParameters const& params ) {
															#line 768 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase768Actor(params));
															#line 1357 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase768, "/fdbserver/StorageMetricSample/rangeSplitPoints/chunkTooLarge")

#line 788 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1363 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase789()
															#line 789 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase789Actor>
															#line 789 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase789ActorState {
															#line 1370 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 789 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase789ActorState(UnitTestParameters const& params) 
															#line 789 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 789 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1377 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase789", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase789ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase789", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 791 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 792 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 794 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 795 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 796 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 797 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 798 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 1000 * sampleUnit);
															#line 800 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 801 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 802 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 803 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 804 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 805 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 806 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 807 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 809 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm._getReadHotRanges(KeyRangeRef("A"_sr, "C"_sr), 2.0, 200 * sampleUnit, 0);
															#line 812 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 1 && (*t.begin()).keys.begin == "Bah"_sr && (*t.begin()).keys.end == "Bob"_sr);
															#line 814 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase789Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase789ActorState(); static_cast<FlowTestCase789Actor*>(this)->destroy(); return 0; }
															#line 1426 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase789Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase789ActorState();
			static_cast<FlowTestCase789Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase789ActorState();
		static_cast<FlowTestCase789Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 789 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1450 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase789()
															#line 789 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase789Actor final : public Actor<Void>, public FastAllocated<FlowTestCase789Actor>, public FlowTestCase789ActorState<FlowTestCase789Actor> {
															#line 1455 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase789Actor>::operator new;
	using FastAllocated<FlowTestCase789Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10492675692240086528UL, 5222015877851196416UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 789 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase789Actor(UnitTestParameters const& params) 
															#line 1471 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase789ActorState<FlowTestCase789Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase789", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15884184549003857408UL, 10377501721790070528UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase789Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase789");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase789", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 789 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase789( UnitTestParameters const& params ) {
															#line 789 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase789Actor(params));
															#line 1503 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase789, "/fdbserver/StorageMetricSample/readHotDetect/simple")

#line 816 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1509 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase817()
															#line 817 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase817Actor>
															#line 817 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase817ActorState {
															#line 1516 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 817 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase817ActorState(UnitTestParameters const& params) 
															#line 817 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 817 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1523 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase817", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase817ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase817", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 819 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 820 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 822 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 823 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 824 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 825 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 826 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 1000 * sampleUnit);
															#line 827 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Final"_sr, 2000 * sampleUnit);
															#line 829 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 830 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 831 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 832 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 833 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 834 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 835 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 836 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 837 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Dah"_sr, 300 * sampleUnit);
															#line 839 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm._getReadHotRanges(KeyRangeRef("A"_sr, "D"_sr), 2.0, 200 * sampleUnit, 0);
															#line 842 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 2 && (*t.begin()).keys.begin == "Bah"_sr && (*t.begin()).keys.end == "Bob"_sr);
															#line 843 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.at(1).keys.begin == "Cat"_sr && t.at(1).keys.end == "Dah"_sr);
															#line 845 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase817Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase817ActorState(); static_cast<FlowTestCase817Actor*>(this)->destroy(); return 0; }
															#line 1578 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase817Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase817ActorState();
			static_cast<FlowTestCase817Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase817ActorState();
		static_cast<FlowTestCase817Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 817 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1602 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase817()
															#line 817 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase817Actor final : public Actor<Void>, public FastAllocated<FlowTestCase817Actor>, public FlowTestCase817ActorState<FlowTestCase817Actor> {
															#line 1607 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase817Actor>::operator new;
	using FastAllocated<FlowTestCase817Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8515155807394510336UL, 729568171332054784UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 817 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase817Actor(UnitTestParameters const& params) 
															#line 1623 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase817ActorState<FlowTestCase817Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase817", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9501935286054431744UL, 8225223456356111872UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase817Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase817");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase817", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 817 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase817( UnitTestParameters const& params ) {
															#line 817 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase817Actor(params));
															#line 1655 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase817, "/fdbserver/StorageMetricSample/readHotDetect/moreThanOneRange")

#line 847 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1661 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase848()
															#line 848 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase848Actor>
															#line 848 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase848ActorState {
															#line 1668 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 848 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase848ActorState(UnitTestParameters const& params) 
															#line 848 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 848 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1675 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase848", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase848ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase848", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 850 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 851 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 853 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 854 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 855 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Bucket"_sr, 2000 * sampleUnit);
															#line 856 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 857 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 858 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 5000 * sampleUnit);
															#line 859 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Final"_sr, 2000 * sampleUnit);
															#line 861 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 862 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 863 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 864 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 865 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 866 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 867 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 868 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 869 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Dah"_sr, 300 * sampleUnit);
															#line 871 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm._getReadHotRanges(KeyRangeRef("A"_sr, "D"_sr), 2.0, 200 * sampleUnit, 0);
															#line 874 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 2 && (*t.begin()).keys.begin == "Bah"_sr && (*t.begin()).keys.end == "But"_sr);
															#line 875 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.at(1).keys.begin == "Cat"_sr && t.at(1).keys.end == "Dah"_sr);
															#line 877 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase848Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase848ActorState(); static_cast<FlowTestCase848Actor*>(this)->destroy(); return 0; }
															#line 1732 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase848Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase848ActorState();
			static_cast<FlowTestCase848Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase848ActorState();
		static_cast<FlowTestCase848Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 848 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1756 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase848()
															#line 848 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase848Actor final : public Actor<Void>, public FastAllocated<FlowTestCase848Actor>, public FlowTestCase848ActorState<FlowTestCase848Actor> {
															#line 1761 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase848Actor>::operator new;
	using FastAllocated<FlowTestCase848Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11064098723593236992UL, 5791401561328714240UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 848 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase848Actor(UnitTestParameters const& params) 
															#line 1777 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase848ActorState<FlowTestCase848Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase848", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15008045860339799040UL, 4945807545357661952UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase848Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase848");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase848", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 848 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase848( UnitTestParameters const& params ) {
															#line 848 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase848Actor(params));
															#line 1809 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase848, "/fdbserver/StorageMetricSample/readHotDetect/consecutiveRanges")

#line 879 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1815 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase880()
															#line 880 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase880Actor>
															#line 880 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase880ActorState {
															#line 1822 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 880 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase880ActorState(UnitTestParameters const& params) 
															#line 880 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 880 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1829 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase880", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase880ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase880", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 882 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 883 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 887 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 888 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 890 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Bucket"_sr, 2000 * sampleUnit);
															#line 892 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 893 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 895 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 5000 * sampleUnit);
															#line 897 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Final"_sr, 2000 * sampleUnit);
															#line 900 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20);
															#line 901 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80);
															#line 902 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000);
															#line 903 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20);
															#line 904 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80);
															#line 905 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200);
															#line 907 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100);
															#line 909 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300);
															#line 910 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Dah"_sr, 300);
															#line 913 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm.getReadHotRanges(KeyRangeRef("Y"_sr, "Z"_sr), 7, ReadHotSubRangeRequest::SplitType::READ_BYTES);
															#line 915 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.size(), 0);
															#line 918 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			t = ssm.getReadHotRanges(KeyRangeRef(""_sr, "\xff"_sr), 7, ReadHotSubRangeRequest::SplitType::READ_BYTES);
															#line 919 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.size(), 5);
															#line 923 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ((*t.begin()).keys.begin, ""_sr);
															#line 925 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ((*t.begin()).keys.end, "Bucket"_sr);
															#line 926 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t[0].bytes, 1400);
															#line 928 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(1).keys.begin, "Bucket"_sr);
															#line 929 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(1).keys.end, "Cat"_sr);
															#line 931 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(2).bytes, 600);
															#line 932 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(3).readBandwidthSec, 5000 * sampleUnit / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
															#line 933 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(3).bytes, 0);
															#line 934 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase880Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase880ActorState(); static_cast<FlowTestCase880Actor*>(this)->destroy(); return 0; }
															#line 1904 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase880Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase880ActorState();
			static_cast<FlowTestCase880Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase880ActorState();
		static_cast<FlowTestCase880Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 880 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1928 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase880()
															#line 880 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase880Actor final : public Actor<Void>, public FastAllocated<FlowTestCase880Actor>, public FlowTestCase880ActorState<FlowTestCase880Actor> {
															#line 1933 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase880Actor>::operator new;
	using FastAllocated<FlowTestCase880Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16639158019299579392UL, 14728931643226096896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 880 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase880Actor(UnitTestParameters const& params) 
															#line 1949 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase880ActorState<FlowTestCase880Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase880", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2593485873683401216UL, 11064098919435038720UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase880Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase880");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase880", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 880 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase880( UnitTestParameters const& params ) {
															#line 880 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase880Actor(params));
															#line 1981 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase880, "/fdbserver/StorageMetricSample/readHotDetect/equalDivide")

#line 936 "/codebuild/output/src121764822/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
