#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
/*
 * BackupS3BlobCorrectness.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * S3-SPECIFIC BACKUP CORRECTNESS WORKLOAD
 *
 * This workload is a specialized version of BackupAndRestoreCorrectness specifically
 * designed for testing S3/blobstore:// backup URLs. It differs from the original in
 * several key ways to handle S3's eventual consistency and timing characteristics:
 *
 * KEY DIFFERENCES FROM BackupAndRestoreCorrectness:
 *
 * 1. MockS3Server Registration:
 *    - Registers a MockS3Server for blobstore:// URLs in simulation
 *    - Only client 0 registers to avoid duplicate server instances
 *
 * 2. Encryption Defaults:
 *    - Defaults to NO encryption (encrypted=false) to simplify S3 testing
 *    - Original uses random encryption (50% chance)
 *    - Tests can still explicitly enable encryption via TOML config
 *
 * 3. Tenant Cleanup Handling:
 *    - Gracefully handles tenant_not_found errors during setup
 *    - Returns early if tenant was deleted during test cleanup
 *    - Original would fail the test on tenant_not_found
 *
 * 4. Status Loop Behavior:
 *    - Exits early when backup reaches "Completed" state or snapshot closes
 *    - Reduces unnecessary polling for S3 metadata that may be eventually consistent
 *    - Original polls continuously until external termination
 *
 * 5. Configurable Snapshot Intervals:
 *    - Accepts initSnapshotInterval and snapshotInterval parameters
 *    - Allows tests to control S3 backup timing characteristics
 *    - Original uses hardcoded random values
 *
 * 6. Configurable Backup URL:
 *    - Accepts backupURL parameter (defaults to file://simfdb/backups/)
 *    - Enables testing with blobstore:// URLs
 *    - Original hardcodes file:// URLs
 *
 * WHY A SEPARATE WORKLOAD?
 *
 * S3/blobstore backups have fundamentally different timing and consistency
 * characteristics than file-based backups. Modifying the shared BackupAndRestoreCorrectness
 * workload to handle both cases introduced subtle race conditions and timing changes
 * that caused flaky failures in file-based backup tests. By creating a separate workload,
 * we ensure:
 *
 * - File-based backup tests maintain their original, stable behavior
 * - S3-specific workarounds don't affect non-S3 tests
 * - S3 tests can be tuned for eventual consistency without impacting other tests
 * - Clear separation of concerns and easier maintenance
 *
 * USAGE:
 *
 * Use this workload in TOML files that test S3/blobstore:// backups:
 *
 *   [[test.workload]]
 *   testName = 'BackupS3BlobCorrectness'
 *   backupURL = 'blobstore://127.0.0.1:8080/bucket'
 *   encrypted = false
 *   initSnapshotInterval = 0
 *   snapshotInterval = 30
 *
 * For file-based backups, continue using the original BackupAndRestoreCorrectness workload.
 */

#include "fdbclient/DatabaseConfiguration.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/ReadYourWrites.h"
#include "fdbrpc/simulator.h"
#include "fdbclient/BackupAgent.actor.h"
#include "fdbclient/BackupContainer.h"
#include "fdbclient/BackupContainerFileSystem.h"
#include "fdbclient/TenantManagement.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "fdbserver/workloads/BulkSetup.actor.h"
#include "fdbserver/MockS3Server.h"
#include "flow/IRandom.h"
#include "flow/actorcompiler.h" // This must be the last #include.

// S3-specific backup correctness workload - see file header for differences from BackupAndRestoreCorrectness
struct BackupS3BlobCorrectnessWorkload : TestWorkload {
	static constexpr auto NAME = "BackupS3BlobCorrectness";
	double backupAfter, restoreAfter, abortAndRestartAfter;
	double minBackupAfter;
	double backupStartAt, restoreStartAfterBackupFinished, stopDifferentialAfter;
	Key backupTag;
	int backupRangesCount, backupRangeLengthMax;
	bool differentialBackup, performRestore, agentRequest;
	Standalone<VectorRef<KeyRangeRef>> backupRanges;
	std::vector<KeyRange> skippedRestoreRanges;
	Standalone<VectorRef<KeyRangeRef>> restoreRanges;
	static int backupAgentRequests;
	LockDB locked{ false };
	bool allowPauses;
	bool shareLogRange;
	bool shouldSkipRestoreRanges;
	bool defaultBackup;
	Optional<std::string> encryptionKeyFileName;

	// S3-specific additions
	std::string backupURL;
	bool skipDirtyRestore;
	int initSnapshotInterval;
	int snapshotInterval;

	// This workload is not compatible with RandomRangeLock workload because they will race in locked range
	void disableFailureInjectionWorkloads(std::set<std::string>& out) const override {
		out.insert({ "RandomRangeLock" });
	}

	BackupS3BlobCorrectnessWorkload(WorkloadContext const& wcx) : TestWorkload(wcx) {
		locked.set(sharedRandomNumber % 2);
		backupAfter = getOption(options, "backupAfter"_sr, 10.0);
		double minBackupAfter = getOption(options, "minBackupAfter"_sr, backupAfter);
		if (backupAfter > minBackupAfter) {
			backupAfter = deterministicRandom()->random01() * (backupAfter - minBackupAfter) + minBackupAfter;
		}
		restoreAfter = getOption(options, "restoreAfter"_sr, 35.0);
		performRestore = getOption(options, "performRestore"_sr, true);
		backupTag = getOption(options, "backupTag"_sr, BackupAgentBase::getDefaultTag());
		backupRangesCount = getOption(options, "backupRangesCount"_sr, 5);
		backupRangeLengthMax = getOption(options, "backupRangeLengthMax"_sr, 1);
		abortAndRestartAfter =
		    getOption(options,
		              "abortAndRestartAfter"_sr,
		              deterministicRandom()->random01() < 0.5
		                  ? deterministicRandom()->random01() * (restoreAfter - backupAfter) + backupAfter
		                  : 0.0);
		differentialBackup =
		    getOption(options, "differentialBackup"_sr, deterministicRandom()->random01() < 0.5 ? true : false);
		stopDifferentialAfter =
		    getOption(options,
		              "stopDifferentialAfter"_sr,
		              differentialBackup ? deterministicRandom()->random01() *
		                                           (restoreAfter - std::max(abortAndRestartAfter, backupAfter)) +
		                                       std::max(abortAndRestartAfter, backupAfter)
		                                 : 0.0);
		agentRequest = getOption(options, "simBackupAgents"_sr, true);
		allowPauses = getOption(options, "allowPauses"_sr, true);
		shareLogRange = getOption(options, "shareLogRange"_sr, false);
		defaultBackup = getOption(options, "defaultBackup"_sr, false);

		// S3-specific options
		backupURL = getOption(options, "backupURL"_sr, "file://simfdb/backups/"_sr).toString();
		skipDirtyRestore = getOption(options, "skipDirtyRestore"_sr, true);
		initSnapshotInterval = getOption(options, "initSnapshotInterval"_sr, 0);
		snapshotInterval = getOption(options, "snapshotInterval"_sr, 30);

		std::vector<std::string> restorePrefixesToInclude =
		    getOption(options, "restorePrefixesToInclude"_sr, std::vector<std::string>());

		shouldSkipRestoreRanges = deterministicRandom()->random01() < 0.3 ? true : false;

		// S3-specific: Default to no encryption to simplify S3 testing
		// Tests can explicitly enable encryption by setting encrypted=true in the toml file
		if (getOption(options, "encrypted"_sr, false)) {
			encryptionKeyFileName = "simfdb/" + getTestEncryptionFileName();
		}

		TraceEvent("BS3BCW_ClientId").detail("Id", wcx.clientId);

		if (backupRangesCount <= 0) {
			backupRanges.push_back_deep(backupRanges.arena(), normalKeys);
		} else {
			// Add backup ranges
			std::set<std::string> rangeEndpoints;
			while (rangeEndpoints.size() < backupRangesCount * 2) {
				rangeEndpoints.insert(deterministicRandom()->randomAlphaNumeric(
				    deterministicRandom()->randomInt(1, backupRangeLengthMax + 1)));
			}

			// Create ranges from the keys, in order, to prevent overlaps
			std::vector<std::string> sortedEndpoints(rangeEndpoints.begin(), rangeEndpoints.end());
			for (auto i = sortedEndpoints.begin(); i != sortedEndpoints.end(); ++i) {
				const std::string& start = *i++;
				backupRanges.push_back_deep(backupRanges.arena(), KeyRangeRef(start, *i));
			}
		}

		if (shouldSkipRestoreRanges && backupRangesCount > 1) {
			skippedRestoreRanges.push_back(backupRanges[deterministicRandom()->randomInt(0, backupRanges.size())]);
		}

		for (const auto& range : backupRanges) {
			if (std::find(skippedRestoreRanges.begin(), skippedRestoreRanges.end(), range) ==
			    skippedRestoreRanges.end()) {
				restoreRanges.push_back_deep(restoreRanges.arena(), range);
			}
		}

		if (!restorePrefixesToInclude.empty()) {
			Standalone<VectorRef<KeyRangeRef>> filteredRestoreRanges;
			for (const auto& range : restoreRanges) {
				for (const auto& prefix : restorePrefixesToInclude) {
					if (range.begin.startsWith(StringRef(prefix))) {
						filteredRestoreRanges.push_back_deep(filteredRestoreRanges.arena(), range);
						break;
					}
				}
			}
			restoreRanges = filteredRestoreRanges;
		}

		TraceEvent("BS3BCW_Ranges");
	}

	Future<Void> setup(Database const& cx) override {
		// Only client 0 performs setup operations (MockS3 registration, tenant adjustment)
		// Other clients just run the Cycle workload
		if (clientId != 0) {
			return Void();
		}
		return _setup(cx, this);
	}

																#line 239 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
// This generated class is to be used only via _setup()
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
template <class _setupActor>
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
class _setupActorState {
															#line 245 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
public:
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	_setupActorState(Database const& cx,BackupS3BlobCorrectnessWorkload* const& self) 
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		 : cx(cx),
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   self(self),
															#line 238 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   adjusted(false),
															#line 239 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   entry()
															#line 258 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("_setup", reinterpret_cast<unsigned long>(this));

	}
	~_setupActorState() 
	{
		fdb_probe_actor_destroy("_setup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 243 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (self->clientId == 0 && self->backupURL.rfind("blobstore://", 0) == 0 && (self->backupURL.find("127.0.0.1") != std::string::npos || self->backupURL.find("localhost") != std::string::npos) && g_network->isSimulated())
															#line 273 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 247 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				TraceEvent("BS3BCW_RegisterMockS3").detail("URL", self->backupURL).detail("ClientId", self->clientId);
															#line 248 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				StrictFuture<Void> __when_expr_0 = g_simulator->registerSimHTTPServer("127.0.0.1", "8080", makeReference<MockS3RequestHandler>());
															#line 248 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 281 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<_setupActor*>(this)->actor_wait_state = 1;
															#line 248 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 0, Void >*>(static_cast<_setupActor*>(this)));
															#line 286 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 254 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (!self->defaultBackup && (cx->defaultTenant.present() || BUGGIFY))
															#line 314 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 255 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (cx->defaultTenant.present())
															#line 318 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
				try {
															#line 257 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
					StrictFuture<Void> __when_expr_1 = store(entry, TenantAPI::getTenant(cx.getReference(), cx->defaultTenant.get()));
															#line 257 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
					if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 325 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
					if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
					static_cast<_setupActor*>(this)->actor_wait_state = 2;
															#line 257 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
					__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 1, Void >*>(static_cast<_setupActor*>(this)));
															#line 330 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
					loopDepth = 0;
				}
				catch (Error& error) {
					loopDepth = a_body1cont1Catch1(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
				}
			}
			else
			{
				try {
															#line 302 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
					StrictFuture<Optional<TenantMapEntry>> __when_expr_2 = TenantAPI::createTenant(cx.getReference(), "BARWTenant"_sr);
															#line 302 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
					if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch2(actor_cancelled(), loopDepth);
															#line 346 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
					if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont1when2(__when_expr_2.get(), loopDepth); };
					static_cast<_setupActor*>(this)->actor_wait_state = 3;
															#line 302 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
					__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 2, Optional<TenantMapEntry> >*>(static_cast<_setupActor*>(this)));
															#line 351 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
					loopDepth = 0;
				}
				catch (Error& error) {
					loopDepth = a_body1cont1Catch2(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1cont1Catch2(unknown_error(), loopDepth);
				}
			}
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 249 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_RegisteredMockS3") .detail("Address", "127.0.0.1:8080") .detail("ClientId", self->clientId);
															#line 372 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 249 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_RegisteredMockS3") .detail("Address", "127.0.0.1:8080") .detail("ClientId", self->clientId);
															#line 381 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _setupActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 314 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (!adjusted)
															#line 465 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 316 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			self->backupRanges.push_back_deep(self->backupRanges.arena(), normalKeys);
															#line 317 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			self->restoreRanges.push_back_deep(self->restoreRanges.arena(), normalKeys);
															#line 471 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		}
															#line 320 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (!static_cast<_setupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_setupActorState(); static_cast<_setupActor*>(this)->destroy(); return 0; }
															#line 475 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		new (&static_cast<_setupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(int loopDepth) 
	{
															#line 274 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (self->backupRangesCount > 0 || deterministicRandom()->coinflip())
															#line 493 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 275 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			Standalone<VectorRef<KeyRangeRef>> adjustedBackupRanges;
															#line 276 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			for( auto& range : self->backupRanges ) {
															#line 277 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				adjustedBackupRanges.push_back_deep( adjustedBackupRanges.arena(), range.withPrefix(entry.prefix, adjustedBackupRanges.arena()));
															#line 501 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
															#line 280 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			self->backupRanges = adjustedBackupRanges;
															#line 282 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			Standalone<VectorRef<KeyRangeRef>> adjustedRestoreRanges;
															#line 283 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			for( auto& range : self->restoreRanges ) {
															#line 284 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				adjustedRestoreRanges.push_back_deep( adjustedRestoreRanges.arena(), range.withPrefix(entry.prefix, adjustedRestoreRanges.arena()));
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
															#line 288 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			self->restoreRanges = adjustedRestoreRanges;
															#line 290 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			std::vector<KeyRange> adjustedSkippedRestoreRanges;
															#line 291 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			for( auto& range : self->skippedRestoreRanges ) {
															#line 292 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				adjustedSkippedRestoreRanges.push_back(range.withPrefix(entry.prefix));
															#line 521 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
															#line 294 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			self->skippedRestoreRanges = adjustedSkippedRestoreRanges;
															#line 296 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_AdjustedRanges");
															#line 298 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			adjusted = true;
															#line 529 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		}
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 261 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (e.code() == error_code_tenant_not_found)
															#line 540 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 262 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				TraceEvent(SevInfo, "BS3BCW_TenantCleanedUp") .detail("Reason", "Tenant was cleaned up during test, skipping backup operations") .detail("TenantName", cx->defaultTenant.get()) .detail("ErrorCode", e.code()) .detail("ErrorDescription", e.what());
															#line 267 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				if (!static_cast<_setupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_setupActorState(); static_cast<_setupActor*>(this)->destroy(); return 0; }
															#line 546 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				new (&static_cast<_setupActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~_setupActorState();
				static_cast<_setupActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 269 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 554 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _setupActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _setupActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont14(int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 305 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (e.code() == error_code_tenant_already_exists)
															#line 675 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 306 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				StrictFuture<Void> __when_expr_3 = store(entry, TenantAPI::getTenant(cx.getReference(), "BARWTenant"_sr));
															#line 306 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 681 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont1Catch2when1(__when_expr_3.get(), loopDepth); };
				static_cast<_setupActor*>(this)->actor_wait_state = 4;
															#line 306 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 3, Void >*>(static_cast<_setupActor*>(this)));
															#line 686 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 308 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 693 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont15(Optional<TenantMapEntry> const& entry_,int loopDepth) 
	{
															#line 303 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		entry = entry_.get();
															#line 708 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1cont15cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont15(Optional<TenantMapEntry> && entry_,int loopDepth) 
	{
															#line 303 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		entry = entry_.get();
															#line 717 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1cont15cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(Optional<TenantMapEntry> const& entry_,int loopDepth) 
	{
		loopDepth = a_body1cont15(entry_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(Optional<TenantMapEntry> && entry_,int loopDepth) 
	{
		loopDepth = a_body1cont15(std::move(entry_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 2, Optional<TenantMapEntry> >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 2, Optional<TenantMapEntry> >*,Optional<TenantMapEntry> const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< _setupActor, 2, Optional<TenantMapEntry> >*,Optional<TenantMapEntry> && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< _setupActor, 2, Optional<TenantMapEntry> >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch2(error, 0);
		} catch (...) {
			a_body1cont1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont15cont2(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont14(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1Catch2cont1(int loopDepth) 
	{
		loopDepth = a_body1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch2cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch2cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch2cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch2cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1Catch2cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1Catch2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< _setupActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1Catch2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< _setupActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Database cx;
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	BackupS3BlobCorrectnessWorkload* self;
															#line 238 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	bool adjusted;
															#line 239 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	TenantMapEntry entry;
															#line 911 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
};
// This generated class is to be used only via _setup()
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
class _setupActor final : public Actor<Void>, public ActorCallback< _setupActor, 0, Void >, public ActorCallback< _setupActor, 1, Void >, public ActorCallback< _setupActor, 2, Optional<TenantMapEntry> >, public ActorCallback< _setupActor, 3, Void >, public FastAllocated<_setupActor>, public _setupActorState<_setupActor> {
															#line 916 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
public:
	using FastAllocated<_setupActor>::operator new;
	using FastAllocated<_setupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12845478988812043520UL, 15499500118456192512UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _setupActor, 0, Void >;
friend struct ActorCallback< _setupActor, 1, Void >;
friend struct ActorCallback< _setupActor, 2, Optional<TenantMapEntry> >;
friend struct ActorCallback< _setupActor, 3, Void >;
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	_setupActor(Database const& cx,BackupS3BlobCorrectnessWorkload* const& self) 
															#line 936 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		 : Actor<Void>(),
		   _setupActorState<_setupActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3044502327521131520UL, 13387266639422932480UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_setup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _setupActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _setupActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< _setupActor, 2, Optional<TenantMapEntry> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< _setupActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
[[nodiscard]] Future<Void> _setup( Database const& cx, BackupS3BlobCorrectnessWorkload* const& self ) {
															#line 237 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	return Future<Void>(new _setupActor(cx, self));
															#line 971 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
}

#line 322 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"

	Future<Void> start(Database const& cx) override {
		// Only client 0 runs backup/restore operations
		// Other clients just run the Cycle workload
		if (clientId != 0) {
			return Void();
		}
		return _start(cx, this);
	}

	Future<bool> check(Database const& cx) override { return true; }

	void getMetrics(std::vector<PerfMetric>& m) override {}

																#line 989 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
// This generated class is to be used only via changePaused()
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
template <class ChangePausedActor>
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
class ChangePausedActorState {
															#line 995 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
public:
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	ChangePausedActorState(Database const& cx,FileBackupAgent* const& backupAgent) 
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		 : cx(cx),
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   backupAgent(backupAgent)
															#line 1004 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("changePaused", reinterpret_cast<unsigned long>(this));

	}
	~ChangePausedActorState() 
	{
		fdb_probe_actor_destroy("changePaused", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 337 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			;
															#line 1019 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ChangePausedActorState();
		static_cast<ChangePausedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 338 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_0 = backupAgent->taskBucket->changePause(cx, deterministicRandom()->coinflip());
															#line 338 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<ChangePausedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1051 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ChangePausedActor*>(this)->actor_wait_state = 1;
															#line 338 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ChangePausedActor, 0, Void >*>(static_cast<ChangePausedActor*>(this)));
															#line 1056 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 339 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(30 * deterministicRandom()->random01());
															#line 339 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<ChangePausedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1067 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ChangePausedActor*>(this)->actor_wait_state = 2;
															#line 339 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ChangePausedActor, 1, Void >*>(static_cast<ChangePausedActor*>(this)));
															#line 1072 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 339 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(30 * deterministicRandom()->random01());
															#line 339 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<ChangePausedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1083 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ChangePausedActor*>(this)->actor_wait_state = 2;
															#line 339 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ChangePausedActor, 1, Void >*>(static_cast<ChangePausedActor*>(this)));
															#line 1088 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ChangePausedActor*>(this)->actor_wait_state > 0) static_cast<ChangePausedActor*>(this)->actor_wait_state = 0;
		static_cast<ChangePausedActor*>(this)->ActorCallback< ChangePausedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangePausedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changePaused", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangePausedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changePaused", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ChangePausedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changePaused", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangePausedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changePaused", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ChangePausedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changePaused", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangePausedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changePaused", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ChangePausedActor*>(this)->actor_wait_state > 0) static_cast<ChangePausedActor*>(this)->actor_wait_state = 0;
		static_cast<ChangePausedActor*>(this)->ActorCallback< ChangePausedActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ChangePausedActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("changePaused", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangePausedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changePaused", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ChangePausedActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("changePaused", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ChangePausedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changePaused", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ChangePausedActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("changePaused", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ChangePausedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("changePaused", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Database cx;
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	FileBackupAgent* backupAgent;
															#line 1259 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
};
// This generated class is to be used only via changePaused()
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
class ChangePausedActor final : public Actor<Void>, public ActorCallback< ChangePausedActor, 0, Void >, public ActorCallback< ChangePausedActor, 1, Void >, public FastAllocated<ChangePausedActor>, public ChangePausedActorState<ChangePausedActor> {
															#line 1264 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
public:
	using FastAllocated<ChangePausedActor>::operator new;
	using FastAllocated<ChangePausedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8949463935851079680UL, 3786716894771595776UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ChangePausedActor, 0, Void >;
friend struct ActorCallback< ChangePausedActor, 1, Void >;
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	ChangePausedActor(Database const& cx,FileBackupAgent* const& backupAgent) 
															#line 1282 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		 : Actor<Void>(),
		   ChangePausedActorState<ChangePausedActor>(cx, backupAgent),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("changePaused", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5431177707453768448UL, 12966551901980677120UL);
		ActorExecutionContextHelper __helper(static_cast<ChangePausedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("changePaused");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("changePaused", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ChangePausedActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ChangePausedActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
[[nodiscard]] static Future<Void> changePaused( Database const& cx, FileBackupAgent* const& backupAgent ) {
															#line 336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	return Future<Void>(new ChangePausedActor(cx, backupAgent));
															#line 1315 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
}

#line 342 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"

																#line 1320 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
// This generated class is to be used only via statusLoop()
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
template <class StatusLoopActor>
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
class StatusLoopActorState {
															#line 1326 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
public:
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	StatusLoopActorState(Database const& cx,std::string const& tag) 
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		 : cx(cx),
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   tag(tag),
															#line 344 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   agent()
															#line 1337 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("statusLoop", reinterpret_cast<unsigned long>(this));

	}
	~StatusLoopActorState() 
	{
		fdb_probe_actor_destroy("statusLoop", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 345 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			;
															#line 1352 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StatusLoopActorState();
		static_cast<StatusLoopActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 346 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<bool> __when_expr_0 = agent.checkActive(cx);
															#line 346 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<StatusLoopActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1384 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<StatusLoopActor*>(this)->actor_wait_state = 1;
															#line 346 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StatusLoopActor, 0, bool >*>(static_cast<StatusLoopActor*>(this)));
															#line 1389 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(bool const& active,int loopDepth) 
	{
															#line 347 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_AgentActivityCheck").detail("IsActive", active);
															#line 348 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<std::string> __when_expr_1 = agent.getStatus(cx, ShowErrors::True, tag);
															#line 348 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<StatusLoopActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1402 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<StatusLoopActor*>(this)->actor_wait_state = 2;
															#line 348 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< StatusLoopActor, 1, std::string >*>(static_cast<StatusLoopActor*>(this)));
															#line 1407 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(bool && active,int loopDepth) 
	{
															#line 347 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_AgentActivityCheck").detail("IsActive", active);
															#line 348 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<std::string> __when_expr_1 = agent.getStatus(cx, ShowErrors::True, tag);
															#line 348 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<StatusLoopActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1420 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<StatusLoopActor*>(this)->actor_wait_state = 2;
															#line 348 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< StatusLoopActor, 1, std::string >*>(static_cast<StatusLoopActor*>(this)));
															#line 1425 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(bool const& active,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(active, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(bool && active,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(active), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StatusLoopActor*>(this)->actor_wait_state > 0) static_cast<StatusLoopActor*>(this)->actor_wait_state = 0;
		static_cast<StatusLoopActor*>(this)->ActorCallback< StatusLoopActor, 0, bool >::remove();

	}
	void a_callback_fire(ActorCallback< StatusLoopActor, 0, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StatusLoopActor, 0, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StatusLoopActor, 0, bool >*,Error err) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 351 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<std::string> __when_expr_2 = agent.getStatusJSON(cx, tag);
															#line 351 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<StatusLoopActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<StatusLoopActor*>(this)->actor_wait_state = 3;
															#line 351 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< StatusLoopActor, 2, std::string >*>(static_cast<StatusLoopActor*>(this)));
															#line 1516 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(std::string const& __statusText,int loopDepth) 
	{
															#line 348 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		statusText = __statusText;
															#line 1525 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(std::string && __statusText,int loopDepth) 
	{
		statusText = std::move(__statusText);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<StatusLoopActor*>(this)->actor_wait_state > 0) static_cast<StatusLoopActor*>(this)->actor_wait_state = 0;
		static_cast<StatusLoopActor*>(this)->ActorCallback< StatusLoopActor, 1, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< StatusLoopActor, 1, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< StatusLoopActor, 1, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< StatusLoopActor, 1, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(std::string const& statusJSON,int loopDepth) 
	{
															#line 356 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (statusText.find("\"Name\":\"Completed\"") != std::string::npos || (statusJSON.find("\"StopAfterSnapshot\":true") != std::string::npos && statusJSON.find("\"ExpectedProgress\":100") != std::string::npos))
															#line 1604 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 359 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_StatusLoopExit").detail("Reason", "CompletedOrSnapshotClosed");
															#line 360 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (!static_cast<StatusLoopActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StatusLoopActorState(); static_cast<StatusLoopActor*>(this)->destroy(); return 0; }
															#line 1610 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			new (&static_cast<StatusLoopActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~StatusLoopActorState();
			static_cast<StatusLoopActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 362 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(2.0);
															#line 362 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<StatusLoopActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1620 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<StatusLoopActor*>(this)->actor_wait_state = 4;
															#line 362 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< StatusLoopActor, 3, Void >*>(static_cast<StatusLoopActor*>(this)));
															#line 1625 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(std::string && statusJSON,int loopDepth) 
	{
															#line 356 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (statusText.find("\"Name\":\"Completed\"") != std::string::npos || (statusJSON.find("\"StopAfterSnapshot\":true") != std::string::npos && statusJSON.find("\"ExpectedProgress\":100") != std::string::npos))
															#line 1634 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 359 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_StatusLoopExit").detail("Reason", "CompletedOrSnapshotClosed");
															#line 360 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (!static_cast<StatusLoopActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StatusLoopActorState(); static_cast<StatusLoopActor*>(this)->destroy(); return 0; }
															#line 1640 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			new (&static_cast<StatusLoopActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~StatusLoopActorState();
			static_cast<StatusLoopActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 362 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(2.0);
															#line 362 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<StatusLoopActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1650 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<StatusLoopActor*>(this)->actor_wait_state = 4;
															#line 362 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< StatusLoopActor, 3, Void >*>(static_cast<StatusLoopActor*>(this)));
															#line 1655 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(std::string const& statusJSON,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(statusJSON, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(std::string && statusJSON,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(statusJSON), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<StatusLoopActor*>(this)->actor_wait_state > 0) static_cast<StatusLoopActor*>(this)->actor_wait_state = 0;
		static_cast<StatusLoopActor*>(this)->ActorCallback< StatusLoopActor, 2, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< StatusLoopActor, 2, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< StatusLoopActor, 2, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< StatusLoopActor, 2, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<StatusLoopActor*>(this)->actor_wait_state > 0) static_cast<StatusLoopActor*>(this)->actor_wait_state = 0;
		static_cast<StatusLoopActor*>(this)->ActorCallback< StatusLoopActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StatusLoopActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< StatusLoopActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< StatusLoopActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Database cx;
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	std::string tag;
															#line 344 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	FileBackupAgent agent;
															#line 348 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	std::string statusText;
															#line 1830 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
};
// This generated class is to be used only via statusLoop()
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
class StatusLoopActor final : public Actor<Void>, public ActorCallback< StatusLoopActor, 0, bool >, public ActorCallback< StatusLoopActor, 1, std::string >, public ActorCallback< StatusLoopActor, 2, std::string >, public ActorCallback< StatusLoopActor, 3, Void >, public FastAllocated<StatusLoopActor>, public StatusLoopActorState<StatusLoopActor> {
															#line 1835 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
public:
	using FastAllocated<StatusLoopActor>::operator new;
	using FastAllocated<StatusLoopActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17786802756810660352UL, 3439363034539254272UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StatusLoopActor, 0, bool >;
friend struct ActorCallback< StatusLoopActor, 1, std::string >;
friend struct ActorCallback< StatusLoopActor, 2, std::string >;
friend struct ActorCallback< StatusLoopActor, 3, Void >;
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	StatusLoopActor(Database const& cx,std::string const& tag) 
															#line 1855 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		 : Actor<Void>(),
		   StatusLoopActorState<StatusLoopActor>(cx, tag),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("statusLoop", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1858547908313483776UL, 2242404897950274816UL);
		ActorExecutionContextHelper __helper(static_cast<StatusLoopActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("statusLoop");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("statusLoop", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StatusLoopActor, 0, bool >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< StatusLoopActor, 1, std::string >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< StatusLoopActor, 2, std::string >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< StatusLoopActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
[[nodiscard]] static Future<Void> statusLoop( Database const& cx, std::string const& tag ) {
															#line 343 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	return Future<Void>(new StatusLoopActor(cx, tag));
															#line 1890 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
}

#line 365 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"

																#line 1895 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
// This generated class is to be used only via doBackup()
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
template <class DoBackupActor>
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
class DoBackupActorState {
															#line 1901 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
public:
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	DoBackupActorState(BackupS3BlobCorrectnessWorkload* const& self,double const& startDelay,FileBackupAgent* const& backupAgent,Database const& cx,Key const& tag,Standalone<VectorRef<KeyRangeRef>> const& backupRanges,double const& stopDifferentialDelay,Promise<Void> const& submitted) 
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		 : self(self),
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   startDelay(startDelay),
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   backupAgent(backupAgent),
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   cx(cx),
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   tag(tag),
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   backupRanges(backupRanges),
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   stopDifferentialDelay(stopDifferentialDelay),
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   submitted(submitted),
															#line 375 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   randomID(nondeterministicRandom()->randomUniqueID())
															#line 1924 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("doBackup", reinterpret_cast<unsigned long>(this));

	}
	~DoBackupActorState() 
	{
		fdb_probe_actor_destroy("doBackup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 378 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (self->agentRequest)
															#line 1939 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 379 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				BackupS3BlobCorrectnessWorkload::backupAgentRequests++;
															#line 1943 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
															#line 382 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			stopDifferentialFuture = delay(stopDifferentialDelay);
															#line 383 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(startDelay);
															#line 383 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (static_cast<DoBackupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1951 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DoBackupActor*>(this)->actor_wait_state = 1;
															#line 383 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoBackupActor, 0, Void >*>(static_cast<DoBackupActor*>(this)));
															#line 1956 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DoBackupActorState();
		static_cast<DoBackupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 388 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (startDelay || BUGGIFY)
															#line 1979 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 389 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_DoBackupAbortBackup1", randomID) .detail("Tag", printable(tag)) .detail("StartDelay", startDelay);
															#line 1983 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			try {
															#line 394 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				StrictFuture<Void> __when_expr_1 = backupAgent->abortBackup(cx, tag.toString());
															#line 394 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				if (static_cast<DoBackupActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 1989 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<DoBackupActor*>(this)->actor_wait_state = 2;
															#line 394 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBackupActor, 1, Void >*>(static_cast<DoBackupActor*>(this)));
															#line 1994 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont1Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 388 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (startDelay || BUGGIFY)
															#line 2014 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 389 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_DoBackupAbortBackup1", randomID) .detail("Tag", printable(tag)) .detail("StartDelay", startDelay);
															#line 2018 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			try {
															#line 394 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				StrictFuture<Void> __when_expr_1 = backupAgent->abortBackup(cx, tag.toString());
															#line 394 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				if (static_cast<DoBackupActor*>(this)->actor_wait_state < 0) return a_body1cont1Catch1(actor_cancelled(), loopDepth);
															#line 2024 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<DoBackupActor*>(this)->actor_wait_state = 2;
															#line 394 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoBackupActor, 1, Void >*>(static_cast<DoBackupActor*>(this)));
															#line 2029 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont1Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoBackupActor*>(this)->actor_wait_state > 0) static_cast<DoBackupActor*>(this)->actor_wait_state = 0;
		static_cast<DoBackupActor*>(this)->ActorCallback< DoBackupActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoBackupActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 402 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_DoBackupWaitBackup", randomID).detail("Tag", printable(tag));
															#line 404 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		statusValue = EBackupState();
															#line 2126 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		try {
															#line 406 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			StrictFuture<EBackupState> __when_expr_2 = backupAgent->waitBackup(cx, tag.toString(), StopWhenDone::False);
															#line 406 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (static_cast<DoBackupActor*>(this)->actor_wait_state < 0) return a_body1cont3Catch1(actor_cancelled(), loopDepth);
															#line 2132 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont3Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<DoBackupActor*>(this)->actor_wait_state = 3;
															#line 406 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoBackupActor, 2, EBackupState >*>(static_cast<DoBackupActor*>(this)));
															#line 2137 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 396 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_DoBackupAbortBackupException", randomID).error(e).detail("Tag", printable(tag));
															#line 397 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (e.code() != error_code_backup_unneeded)
															#line 2161 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 398 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 2165 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
			loopDepth = a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoBackupActor*>(this)->actor_wait_state > 0) static_cast<DoBackupActor*>(this)->actor_wait_state = 0;
		static_cast<DoBackupActor*>(this)->ActorCallback< DoBackupActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoBackupActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7(int loopDepth) 
	{
															#line 419 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_DoBackupWaitBackupStatus", randomID) .detail("Status", BackupAgentBase::getStateText(statusValue)) .detail("Tag", printable(tag));
															#line 423 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (statusValue == EBackupState::STATE_COMPLETED)
															#line 2283 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 424 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_DoBackupDiscontinued", randomID).detail("Tag", printable(tag));
															#line 425 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (!static_cast<DoBackupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoBackupActorState(); static_cast<DoBackupActor*>(this)->destroy(); return 0; }
															#line 2289 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			new (&static_cast<DoBackupActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DoBackupActorState();
			static_cast<DoBackupActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 428 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (statusValue != EBackupState::STATE_NEVERRAN)
															#line 2297 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 429 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_DoBackupAbortBackup2", randomID).detail("Tag", printable(tag));
															#line 2301 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			try {
															#line 432 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				StrictFuture<Void> __when_expr_3 = backupAgent->abortBackup(cx, tag.toString());
															#line 432 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				if (static_cast<DoBackupActor*>(this)->actor_wait_state < 0) return a_body1cont7Catch1(actor_cancelled(), loopDepth);
															#line 2307 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont7Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont7when1(__when_expr_3.get(), loopDepth); };
				static_cast<DoBackupActor*>(this)->actor_wait_state = 4;
															#line 432 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DoBackupActor, 3, Void >*>(static_cast<DoBackupActor*>(this)));
															#line 2312 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1cont7Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont7Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 411 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (e.code() == error_code_backup_unneeded)
															#line 2333 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 412 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				TraceEvent("BS3BCW_DoBackupWaitBackupUnneeded", randomID).detail("Tag", printable(tag));
															#line 413 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				statusValue = EBackupState::STATE_NEVERRAN;
															#line 2339 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
			else
			{
															#line 415 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 2345 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
			loopDepth = a_body1cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8(EBackupState const& _statusValue,int loopDepth) 
	{
															#line 407 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		statusValue = _statusValue;
															#line 2361 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(EBackupState && _statusValue,int loopDepth) 
	{
															#line 407 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		statusValue = _statusValue;
															#line 2370 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(EBackupState const& _statusValue,int loopDepth) 
	{
		loopDepth = a_body1cont8(_statusValue, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(EBackupState && _statusValue,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_statusValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DoBackupActor*>(this)->actor_wait_state > 0) static_cast<DoBackupActor*>(this)->actor_wait_state = 0;
		static_cast<DoBackupActor*>(this)->ActorCallback< DoBackupActor, 2, EBackupState >::remove();

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 2, EBackupState >*,EBackupState const& value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 2, EBackupState >*,EBackupState && value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DoBackupActor, 2, EBackupState >*,Error err) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont3Catch1(error, 0);
		} catch (...) {
			a_body1cont3Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11(int loopDepth) 
	{
															#line 440 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_DoBackupSubmitBackup", randomID) .detail("Tag", printable(tag)) .detail("StopWhenDone", stopDifferentialDelay ? "False" : "True");
															#line 445 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		backupContainer = self->backupURL;
															#line 446 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		status = statusLoop(cx, tag.toString());
															#line 2471 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		try {
															#line 448 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_4 = backupAgent->submitBackup(cx, StringRef(backupContainer), {}, self->initSnapshotInterval, self->snapshotInterval, tag.toString(), backupRanges, true, StopWhenDone{ !stopDifferentialDelay }, UsePartitionedLog::False, IncrementalBackupOnly::False, self->encryptionKeyFileName);
															#line 448 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (static_cast<DoBackupActor*>(this)->actor_wait_state < 0) return a_body1cont11Catch1(actor_cancelled(), loopDepth);
															#line 2477 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont11Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont11when1(__when_expr_4.get(), loopDepth); };
			static_cast<DoBackupActor*>(this)->actor_wait_state = 5;
															#line 448 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< DoBackupActor, 4, Void >*>(static_cast<DoBackupActor*>(this)));
															#line 2482 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont11Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont11Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont13(int loopDepth) 
	{
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont7Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 434 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_DoBackupAbortBackupException", randomID).error(e).detail("Tag", printable(tag));
															#line 435 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (e.code() != error_code_backup_unneeded)
															#line 2506 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 436 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 2510 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
			loopDepth = a_body1cont13(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont14(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont14(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DoBackupActor*>(this)->actor_wait_state > 0) static_cast<DoBackupActor*>(this)->actor_wait_state = 0;
		static_cast<DoBackupActor*>(this)->ActorCallback< DoBackupActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DoBackupActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont14cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont13(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont1(int loopDepth) 
	{
															#line 466 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		submitted.send(Void());
															#line 468 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_DoBackupWaitToDiscontinue", randomID) .detail("Tag", printable(tag)) .detail("DifferentialAfter", stopDifferentialDelay);
															#line 2628 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		try {
															#line 473 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_5 = success(backupAgent->waitBackup(cx, tag.toString(), StopWhenDone::True));
															#line 473 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (static_cast<DoBackupActor*>(this)->actor_wait_state < 0) return a_body1cont11cont1Catch1(actor_cancelled(), loopDepth);
															#line 2634 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont11cont1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont11cont1when1(__when_expr_5.get(), loopDepth); };
			static_cast<DoBackupActor*>(this)->actor_wait_state = 6;
															#line 473 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< DoBackupActor, 5, Void >*>(static_cast<DoBackupActor*>(this)));
															#line 2639 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont11cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont11cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 461 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_DoBackupSubmitBackupException", randomID).error(e).detail("Tag", printable(tag));
															#line 462 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (e.code() != error_code_backup_unneeded && e.code() != error_code_backup_duplicate)
															#line 2657 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 463 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 2661 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
			loopDepth = a_body1cont11cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<DoBackupActor*>(this)->actor_wait_state > 0) static_cast<DoBackupActor*>(this)->actor_wait_state = 0;
		static_cast<DoBackupActor*>(this)->ActorCallback< DoBackupActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont11Catch1(error, 0);
		} catch (...) {
			a_body1cont11Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont11Catch1(error, 0);
		} catch (...) {
			a_body1cont11Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< DoBackupActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont11Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont11Catch1(error, 0);
		} catch (...) {
			a_body1cont11Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont11cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont11cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont4(int loopDepth) 
	{
															#line 482 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_DoBackupDiscontinueBackup", randomID).detail("Tag", printable(tag));
															#line 2777 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		try {
															#line 485 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_6 = backupAgent->discontinueBackup(cx, tag);
															#line 485 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (static_cast<DoBackupActor*>(this)->actor_wait_state < 0) return a_body1cont11cont4Catch1(actor_cancelled(), loopDepth);
															#line 2783 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont11cont4Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont11cont4when1(__when_expr_6.get(), loopDepth); };
			static_cast<DoBackupActor*>(this)->actor_wait_state = 7;
															#line 485 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< DoBackupActor, 6, Void >*>(static_cast<DoBackupActor*>(this)));
															#line 2788 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont11cont4Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont11cont4Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 475 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (e.code() == error_code_backup_unneeded)
															#line 2804 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 476 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				TraceEvent("BS3BCW_DoBackupWaitToDiscontinueUnneeded", randomID).detail("Tag", printable(tag));
															#line 477 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				if (!static_cast<DoBackupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoBackupActorState(); static_cast<DoBackupActor*>(this)->destroy(); return 0; }
															#line 2810 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				new (&static_cast<DoBackupActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~DoBackupActorState();
				static_cast<DoBackupActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 479 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 2818 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<DoBackupActor*>(this)->actor_wait_state > 0) static_cast<DoBackupActor*>(this)->actor_wait_state = 0;
		static_cast<DoBackupActor*>(this)->ActorCallback< DoBackupActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< DoBackupActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont11cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont11cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont11cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont11cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont7(int loopDepth) 
	{
															#line 494 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_DoBackupWaitForDiscontinued", randomID).detail("Tag", printable(tag));
															#line 2932 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		try {
															#line 497 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_7 = success(backupAgent->waitBackup(cx, tag.toString(), StopWhenDone::True));
															#line 497 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (static_cast<DoBackupActor*>(this)->actor_wait_state < 0) return a_body1cont11cont7Catch1(actor_cancelled(), loopDepth);
															#line 2938 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont11cont7Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont11cont7when1(__when_expr_7.get(), loopDepth); };
			static_cast<DoBackupActor*>(this)->actor_wait_state = 8;
															#line 497 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< DoBackupActor, 7, Void >*>(static_cast<DoBackupActor*>(this)));
															#line 2943 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont11cont7Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont11cont7Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont4Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 487 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (e.code() == error_code_backup_unneeded)
															#line 2959 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 488 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				TraceEvent("BS3BCW_DoBackupDiscontinueBackupUnneeded", randomID).detail("Tag", printable(tag));
															#line 489 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				if (!static_cast<DoBackupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoBackupActorState(); static_cast<DoBackupActor*>(this)->destroy(); return 0; }
															#line 2965 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				new (&static_cast<DoBackupActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~DoBackupActorState();
				static_cast<DoBackupActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 491 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 2973 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<DoBackupActor*>(this)->actor_wait_state > 0) static_cast<DoBackupActor*>(this)->actor_wait_state = 0;
		static_cast<DoBackupActor*>(this)->ActorCallback< DoBackupActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont11cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont11cont4Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont11cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont11cont4Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< DoBackupActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont11cont4Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont11cont4Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont4Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont11cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont11cont7(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont10(int loopDepth) 
	{
															#line 506 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_DoBackupComplete", randomID).detail("Tag", printable(tag));
															#line 508 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (!static_cast<DoBackupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoBackupActorState(); static_cast<DoBackupActor*>(this)->destroy(); return 0; }
															#line 3089 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		new (&static_cast<DoBackupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DoBackupActorState();
		static_cast<DoBackupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11cont7Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 499 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (e.code() == error_code_backup_unneeded)
															#line 3102 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 500 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				TraceEvent("BS3BCW_DoBackupWaitForDiscontinuedUnneeded", randomID).detail("Tag", printable(tag));
															#line 501 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				if (!static_cast<DoBackupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DoBackupActorState(); static_cast<DoBackupActor*>(this)->destroy(); return 0; }
															#line 3108 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				new (&static_cast<DoBackupActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~DoBackupActorState();
				static_cast<DoBackupActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 503 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 3116 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<DoBackupActor*>(this)->actor_wait_state > 0) static_cast<DoBackupActor*>(this)->actor_wait_state = 0;
		static_cast<DoBackupActor*>(this)->ActorCallback< DoBackupActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont11cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont11cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< DoBackupActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont11cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont11cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< DoBackupActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont11cont7Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont11cont7Catch1(error, 0);
		} catch (...) {
			a_body1cont11cont7Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont11cont11cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont11cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	BackupS3BlobCorrectnessWorkload* self;
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	double startDelay;
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	FileBackupAgent* backupAgent;
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Database cx;
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Key tag;
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Standalone<VectorRef<KeyRangeRef>> backupRanges;
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	double stopDifferentialDelay;
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Promise<Void> submitted;
															#line 375 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	UID randomID;
															#line 382 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Future<Void> stopDifferentialFuture;
															#line 404 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	EBackupState statusValue;
															#line 445 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	std::string backupContainer;
															#line 446 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Future<Void> status;
															#line 3252 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
};
// This generated class is to be used only via doBackup()
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
class DoBackupActor final : public Actor<Void>, public ActorCallback< DoBackupActor, 0, Void >, public ActorCallback< DoBackupActor, 1, Void >, public ActorCallback< DoBackupActor, 2, EBackupState >, public ActorCallback< DoBackupActor, 3, Void >, public ActorCallback< DoBackupActor, 4, Void >, public ActorCallback< DoBackupActor, 5, Void >, public ActorCallback< DoBackupActor, 6, Void >, public ActorCallback< DoBackupActor, 7, Void >, public FastAllocated<DoBackupActor>, public DoBackupActorState<DoBackupActor> {
															#line 3257 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
public:
	using FastAllocated<DoBackupActor>::operator new;
	using FastAllocated<DoBackupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8804183518122887424UL, 380393455548093440UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DoBackupActor, 0, Void >;
friend struct ActorCallback< DoBackupActor, 1, Void >;
friend struct ActorCallback< DoBackupActor, 2, EBackupState >;
friend struct ActorCallback< DoBackupActor, 3, Void >;
friend struct ActorCallback< DoBackupActor, 4, Void >;
friend struct ActorCallback< DoBackupActor, 5, Void >;
friend struct ActorCallback< DoBackupActor, 6, Void >;
friend struct ActorCallback< DoBackupActor, 7, Void >;
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	DoBackupActor(BackupS3BlobCorrectnessWorkload* const& self,double const& startDelay,FileBackupAgent* const& backupAgent,Database const& cx,Key const& tag,Standalone<VectorRef<KeyRangeRef>> const& backupRanges,double const& stopDifferentialDelay,Promise<Void> const& submitted) 
															#line 3281 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		 : Actor<Void>(),
		   DoBackupActorState<DoBackupActor>(self, startDelay, backupAgent, cx, tag, backupRanges, stopDifferentialDelay, submitted),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doBackup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15925328432552866816UL, 11115638302355080704UL);
		ActorExecutionContextHelper __helper(static_cast<DoBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doBackup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doBackup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DoBackupActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DoBackupActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DoBackupActor, 2, EBackupState >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DoBackupActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< DoBackupActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< DoBackupActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< DoBackupActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< DoBackupActor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
[[nodiscard]] static Future<Void> doBackup( BackupS3BlobCorrectnessWorkload* const& self, double const& startDelay, FileBackupAgent* const& backupAgent, Database const& cx, Key const& tag, Standalone<VectorRef<KeyRangeRef>> const& backupRanges, double const& stopDifferentialDelay, Promise<Void> const& submitted ) {
															#line 366 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	return Future<Void>(new DoBackupActor(self, startDelay, backupAgent, cx, tag, backupRanges, stopDifferentialDelay, submitted));
															#line 3320 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
}

#line 510 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"

																#line 3325 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
// This generated class is to be used only via _start()
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
template <class _startActor>
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
class _startActorState {
															#line 3331 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
public:
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	_startActorState(Database const& cx,BackupS3BlobCorrectnessWorkload* const& self) 
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		 : cx(cx),
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   self(self),
															#line 512 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   backupAgent(),
															#line 513 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		   stopDifferentialBackup(delay(self->stopDifferentialAfter))
															#line 3344 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
	{
		fdb_probe_actor_create("_start", reinterpret_cast<unsigned long>(this));

	}
	~_startActorState() 
	{
		fdb_probe_actor_destroy("_start", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 515 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_Arguments") .detail("BackupAfter", self->backupAfter) .detail("RestoreAfter", self->restoreAfter) .detail("AbortAndRestartAfter", self->abortAndRestartAfter) .detail("DifferentialAfter", self->stopDifferentialAfter);
															#line 528 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (self->agentRequest)
															#line 3361 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 529 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				TraceEvent("BS3BCW_CleanupOnRestart").detail("Tag", printable(self->backupTag));
															#line 3365 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				try {
															#line 531 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
					StrictFuture<Void> __when_expr_0 = backupAgent.abortBackup(cx, self->backupTag.toString());
															#line 531 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
					if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3371 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<_startActor*>(this)->actor_wait_state = 1;
															#line 531 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _startActor, 0, Void >*>(static_cast<_startActor*>(this)));
															#line 3376 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
					loopDepth = 0;
				}
				catch (Error& error) {
					loopDepth = a_body1Catch2(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1Catch2(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_startActorState();
		static_cast<_startActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 539 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (g_network->isSimulated())
															#line 3410 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 540 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			clearMockS3Storage();
															#line 541 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			TraceEvent("BS3BCW_ClearedMockS3Storage");
															#line 3416 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		}
															#line 544 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (self->agentRequest)
															#line 3420 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 545 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			submitted = Promise<Void>();
															#line 546 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			b = doBackup(self, self->backupAfter, &backupAgent, cx, self->backupTag, self->backupRanges, self->stopDifferentialAfter, submitted);
															#line 555 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (self->abortAndRestartAfter)
															#line 3428 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 556 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				TraceEvent("BS3BCW_AbortAndRestartAfter").detail("AbortAndRestartAfter", self->abortAndRestartAfter);
															#line 557 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				StrictFuture<Void> __when_expr_1 = submitted.getFuture();
															#line 557 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3436 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<_startActor*>(this)->actor_wait_state = 2;
															#line 557 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _startActor, 1, Void >*>(static_cast<_startActor*>(this)));
															#line 3441 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont7(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 533 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (e.code() != error_code_backup_unneeded)
															#line 3467 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 534 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3471 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _startActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _startActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
															#line 633 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (!static_cast<_startActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_startActorState(); static_cast<_startActor*>(this)->destroy(); return 0; }
															#line 3587 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		new (&static_cast<_startActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_startActorState();
		static_cast<_startActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7(int loopDepth) 
	{
															#line 579 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (self->performRestore)
															#line 3599 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 580 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_5 = delay(self->restoreAfter);
															#line 580 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3605 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont7when1(__when_expr_5.get(), loopDepth); };
			static_cast<_startActor*>(this)->actor_wait_state = 6;
															#line 580 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< _startActor, 5, Void >*>(static_cast<_startActor*>(this)));
															#line 3610 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont12(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
															#line 558 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(self->abortAndRestartAfter - self->backupAfter);
															#line 558 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3626 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont8when1(__when_expr_2.get(), loopDepth); };
		static_cast<_startActor*>(this)->actor_wait_state = 3;
															#line 558 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _startActor, 2, Void >*>(static_cast<_startActor*>(this)));
															#line 3631 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
															#line 558 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(self->abortAndRestartAfter - self->backupAfter);
															#line 558 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3642 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont8when1(__when_expr_2.get(), loopDepth); };
		static_cast<_startActor*>(this)->actor_wait_state = 3;
															#line 558 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _startActor, 2, Void >*>(static_cast<_startActor*>(this)));
															#line 3647 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _startActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _startActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
															#line 559 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_AbortBackup").detail("Tag", printable(self->backupTag));
															#line 3731 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		try {
															#line 561 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_3 = backupAgent.abortBackup(cx, self->backupTag.toString());
															#line 561 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1cont9Catch1(actor_cancelled(), loopDepth);
															#line 3737 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont9Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont9when1(__when_expr_3.get(), loopDepth); };
			static_cast<_startActor*>(this)->actor_wait_state = 4;
															#line 561 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< _startActor, 3, Void >*>(static_cast<_startActor*>(this)));
															#line 3742 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont9Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont9Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
															#line 559 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_AbortBackup").detail("Tag", printable(self->backupTag));
															#line 3757 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		try {
															#line 561 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			StrictFuture<Void> __when_expr_3 = backupAgent.abortBackup(cx, self->backupTag.toString());
															#line 561 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1cont9Catch1(actor_cancelled(), loopDepth);
															#line 3763 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont9Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont9when1(__when_expr_3.get(), loopDepth); };
			static_cast<_startActor*>(this)->actor_wait_state = 4;
															#line 561 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< _startActor, 3, Void >*>(static_cast<_startActor*>(this)));
															#line 3768 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont9Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont9Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< _startActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< _startActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont10(int loopDepth) 
	{
															#line 566 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_AbortComplete").detail("Tag", printable(self->backupTag));
															#line 567 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_4 = b;
															#line 567 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3862 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont10when1(__when_expr_4.get(), loopDepth); };
		static_cast<_startActor*>(this)->actor_wait_state = 5;
															#line 567 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< _startActor, 4, Void >*>(static_cast<_startActor*>(this)));
															#line 3867 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont9Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 563 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (e.code() != error_code_backup_unneeded)
															#line 3877 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 564 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 3881 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			}
			loopDepth = a_body1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont9Catch1(error, 0);
		} catch (...) {
			a_body1cont9Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< _startActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont9Catch1(error, 0);
		} catch (...) {
			a_body1cont9Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< _startActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont9Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont9Catch1(error, 0);
		} catch (...) {
			a_body1cont9Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont11cont1(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont10cont1(Void const& _,int loopDepth) 
	{
															#line 568 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_RestartBackup").detail("Tag", printable(self->backupTag));
															#line 569 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		b = doBackup(self, 0, &backupAgent, cx, self->backupTag, self->backupRanges, self->stopDifferentialAfter - self->abortAndRestartAfter, Promise<Void>());
															#line 3999 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont1(Void && _,int loopDepth) 
	{
															#line 568 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_RestartBackup").detail("Tag", printable(self->backupTag));
															#line 569 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		b = doBackup(self, 0, &backupAgent, cx, self->backupTag, self->backupRanges, self->stopDifferentialAfter - self->abortAndRestartAfter, Promise<Void>());
															#line 4010 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< _startActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< _startActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont12(int loopDepth) 
	{
															#line 630 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Void> __when_expr_10 = b;
															#line 630 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4096 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont12when1(__when_expr_10.get(), loopDepth); };
		static_cast<_startActor*>(this)->actor_wait_state = 11;
															#line 630 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< _startActor, 10, Void >*>(static_cast<_startActor*>(this)));
															#line 4101 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont13(Void const& _,int loopDepth) 
	{
															#line 581 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_RestoreAfter").detail("RestoreAfter", self->restoreAfter);
															#line 584 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		keyBackedTag = makeBackupTag(self->backupTag.toString());
															#line 585 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<UidAndAbortedFlagT> __when_expr_6 = keyBackedTag.getOrThrow(cx.getReference());
															#line 585 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4116 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont13when1(__when_expr_6.get(), loopDepth); };
		static_cast<_startActor*>(this)->actor_wait_state = 7;
															#line 585 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< _startActor, 6, UidAndAbortedFlagT >*>(static_cast<_startActor*>(this)));
															#line 4121 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont13(Void && _,int loopDepth) 
	{
															#line 581 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_RestoreAfter").detail("RestoreAfter", self->restoreAfter);
															#line 584 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		keyBackedTag = makeBackupTag(self->backupTag.toString());
															#line 585 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<UidAndAbortedFlagT> __when_expr_6 = keyBackedTag.getOrThrow(cx.getReference());
															#line 585 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4136 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont13when1(__when_expr_6.get(), loopDepth); };
		static_cast<_startActor*>(this)->actor_wait_state = 7;
															#line 585 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< _startActor, 6, UidAndAbortedFlagT >*>(static_cast<_startActor*>(this)));
															#line 4141 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< _startActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< _startActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont13cont1(UidAndAbortedFlagT const& uidFlag,int loopDepth) 
	{
															#line 586 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		logUid = uidFlag.first;
															#line 587 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Reference<IBackupContainer>> __when_expr_7 = BackupConfig(logUid).backupContainer().getD(cx.getReference());
															#line 587 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4229 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont13cont1when1(__when_expr_7.get(), loopDepth); };
		static_cast<_startActor*>(this)->actor_wait_state = 8;
															#line 587 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< _startActor, 7, Reference<IBackupContainer> >*>(static_cast<_startActor*>(this)));
															#line 4234 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont13cont1(UidAndAbortedFlagT && uidFlag,int loopDepth) 
	{
															#line 586 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		logUid = uidFlag.first;
															#line 587 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Reference<IBackupContainer>> __when_expr_7 = BackupConfig(logUid).backupContainer().getD(cx.getReference());
															#line 587 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4247 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont13cont1when1(__when_expr_7.get(), loopDepth); };
		static_cast<_startActor*>(this)->actor_wait_state = 8;
															#line 587 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< _startActor, 7, Reference<IBackupContainer> >*>(static_cast<_startActor*>(this)));
															#line 4252 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont13when1(UidAndAbortedFlagT const& uidFlag,int loopDepth) 
	{
		loopDepth = a_body1cont13cont1(uidFlag, loopDepth);

		return loopDepth;
	}
	int a_body1cont13when1(UidAndAbortedFlagT && uidFlag,int loopDepth) 
	{
		loopDepth = a_body1cont13cont1(std::move(uidFlag), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 6, UidAndAbortedFlagT >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 6, UidAndAbortedFlagT >*,UidAndAbortedFlagT const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont13when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< _startActor, 6, UidAndAbortedFlagT >*,UidAndAbortedFlagT && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont13when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< _startActor, 6, UidAndAbortedFlagT >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont13cont2(int loopDepth) 
	{
															#line 590 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (lastBackupContainer)
															#line 4336 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		{
															#line 592 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
			if (!self->skipDirtyRestore)
															#line 4340 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
			{
															#line 593 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				StrictFuture<Void> __when_expr_8 = runRYWTransaction(cx, [=](Reference<ReadYourWritesTransaction> tr) -> Future<Void> { tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS); for (auto& kvrange : self->backupRanges) tr->clear(kvrange); return Void(); });
															#line 593 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4346 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont13cont2when1(__when_expr_8.get(), loopDepth); };
				static_cast<_startActor*>(this)->actor_wait_state = 9;
															#line 593 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
				__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< _startActor, 8, Void >*>(static_cast<_startActor*>(this)));
															#line 4351 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont13cont4(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont13cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont13cont1when1(Reference<IBackupContainer> const& __lastBackupContainer,int loopDepth) 
	{
															#line 587 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		lastBackupContainer = __lastBackupContainer;
															#line 4370 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1cont13cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont13cont1when1(Reference<IBackupContainer> && __lastBackupContainer,int loopDepth) 
	{
		lastBackupContainer = std::move(__lastBackupContainer);
		loopDepth = a_body1cont13cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 7, Reference<IBackupContainer> >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 7, Reference<IBackupContainer> >*,Reference<IBackupContainer> const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont13cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< _startActor, 7, Reference<IBackupContainer> >*,Reference<IBackupContainer> && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont13cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< _startActor, 7, Reference<IBackupContainer> >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont13cont3(int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont13cont4(int loopDepth) 
	{
															#line 602 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_Restore") .detail("LastBackupContainer", lastBackupContainer->getURL()) .detail("BackupTag", printable(self->backupTag)) .detail("SkipDirtyRestore", self->skipDirtyRestore);
															#line 607 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		Standalone<StringRef> restoreTag(self->backupTag.toString() + "_restore");
															#line 608 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		StrictFuture<Version> __when_expr_9 = backupAgent.restore(cx, cx, restoreTag, KeyRef(lastBackupContainer->getURL()), lastBackupContainer->getProxy(), self->restoreRanges, WaitForComplete::True, ::invalidVersion, Verbose::True, Key(), Key(), self->locked, UnlockDB::True, OnlyApplyMutationLogs::False, InconsistentSnapshotOnly::False, ::invalidVersion, lastBackupContainer->getEncryptionKeyFileName());
															#line 608 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4461 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont13cont4when1(__when_expr_9.get(), loopDepth); };
		static_cast<_startActor*>(this)->actor_wait_state = 10;
															#line 608 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< _startActor, 9, Version >*>(static_cast<_startActor*>(this)));
															#line 4466 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont13cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont13cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont13cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont13cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont13cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< _startActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont13cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< _startActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont13cont6(Version const& _,int loopDepth) 
	{
															#line 626 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_RestoreComplete").detail("BackupTag", printable(self->backupTag));
															#line 4562 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1cont13cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont13cont6(Version && _,int loopDepth) 
	{
															#line 626 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
		TraceEvent("BS3BCW_RestoreComplete").detail("BackupTag", printable(self->backupTag));
															#line 4571 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		loopDepth = a_body1cont13cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont13cont4when1(Version const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont13cont4when1(Version && _,int loopDepth) 
	{
		loopDepth = a_body1cont13cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 9, Version >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 9, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont13cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< _startActor, 9, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont13cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< _startActor, 9, Version >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont12cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont12cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont12when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont12when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont12when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< _startActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont12when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< _startActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 10);

	}
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Database cx;
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	BackupS3BlobCorrectnessWorkload* self;
															#line 512 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	FileBackupAgent backupAgent;
															#line 513 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Future<Void> stopDifferentialBackup;
															#line 545 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Promise<Void> submitted;
															#line 546 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Future<Void> b;
															#line 584 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	KeyBackedTag keyBackedTag;
															#line 586 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	UID logUid;
															#line 587 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	Reference<IBackupContainer> lastBackupContainer;
															#line 4756 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
};
// This generated class is to be used only via _start()
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
class _startActor final : public Actor<Void>, public ActorCallback< _startActor, 0, Void >, public ActorCallback< _startActor, 1, Void >, public ActorCallback< _startActor, 2, Void >, public ActorCallback< _startActor, 3, Void >, public ActorCallback< _startActor, 4, Void >, public ActorCallback< _startActor, 5, Void >, public ActorCallback< _startActor, 6, UidAndAbortedFlagT >, public ActorCallback< _startActor, 7, Reference<IBackupContainer> >, public ActorCallback< _startActor, 8, Void >, public ActorCallback< _startActor, 9, Version >, public ActorCallback< _startActor, 10, Void >, public FastAllocated<_startActor>, public _startActorState<_startActor> {
															#line 4761 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
public:
	using FastAllocated<_startActor>::operator new;
	using FastAllocated<_startActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12010082431922690304UL, 8056053311479004672UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _startActor, 0, Void >;
friend struct ActorCallback< _startActor, 1, Void >;
friend struct ActorCallback< _startActor, 2, Void >;
friend struct ActorCallback< _startActor, 3, Void >;
friend struct ActorCallback< _startActor, 4, Void >;
friend struct ActorCallback< _startActor, 5, Void >;
friend struct ActorCallback< _startActor, 6, UidAndAbortedFlagT >;
friend struct ActorCallback< _startActor, 7, Reference<IBackupContainer> >;
friend struct ActorCallback< _startActor, 8, Void >;
friend struct ActorCallback< _startActor, 9, Version >;
friend struct ActorCallback< _startActor, 10, Void >;
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	_startActor(Database const& cx,BackupS3BlobCorrectnessWorkload* const& self) 
															#line 4788 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
		 : Actor<Void>(),
		   _startActorState<_startActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16516870674810565632UL, 12235738791954228224UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_start");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _startActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _startActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< _startActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< _startActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< _startActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< _startActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< _startActor, 6, UidAndAbortedFlagT >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< _startActor, 7, Reference<IBackupContainer> >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< _startActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< _startActor, 9, Version >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< _startActor, 10, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
[[nodiscard]] static Future<Void> _start( Database const& cx, BackupS3BlobCorrectnessWorkload* const& self ) {
															#line 511 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
	return Future<Void>(new _startActor(cx, self));
															#line 4830 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/BackupS3BlobCorrectness.actor.g.cpp"
}

#line 635 "/codebuild/output/src2171772327/src/github.com/apple/foundationdb/fdbserver/workloads/BackupS3BlobCorrectness.actor.cpp"
};

int BackupS3BlobCorrectnessWorkload::backupAgentRequests = 0;

WorkloadFactory<BackupS3BlobCorrectnessWorkload> BackupS3BlobCorrectnessWorkloadFactory;
