#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
/*
 * TagThrottle.actor.h
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "flow/Arena.h"
#if defined(NO_INTELLISENSE) && !defined(FDBCLIENT_TAG_THROTTLE_ACTOR_G_H)
#define FDBCLIENT_TAG_THROTTLE_ACTOR_G_H
#include "fdbclient/TagThrottle.actor.g.h"
#elif !defined(FDBCLIENT_TAG_THROTTLE_ACTOR_H)
#define FDBCLIENT_TAG_THROTTLE_ACTOR_H

#pragma once

#include "fmt/format.h"
#include "flow/Error.h"
#include "flow/flow.h"
#include "flow/network.h"
#include "flow/ThreadHelper.actor.h"
#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/CommitTransaction.h"
#include "flow/actorcompiler.h" // This must be the last #include.

typedef StringRef TransactionTagRef;
typedef Standalone<TransactionTagRef> TransactionTag;

FDB_BOOLEAN_PARAM(ContainsRecommended);
FDB_BOOLEAN_PARAM(Capitalize);

class TagSet {
public:
	typedef std::vector<TransactionTagRef>::const_iterator const_iterator;

	TagSet() : bytes(0) {}

	void addTag(TransactionTagRef tag);
	size_t size() const;

	const_iterator begin() const { return tags.begin(); }

	const_iterator end() const { return tags.end(); }

	void clear() {
		tags.clear();
		bytes = 0;
	}

	template <class Context>
	void save(uint8_t* out, Context&) const {
		uint8_t* start = out;
		for (const auto& tag : *this) {
			*(out++) = (uint8_t)tag.size();

			std::copy(tag.begin(), tag.end(), out);
			out += tag.size();
		}

		ASSERT((size_t)(out - start) == size() + bytes);
	}

	template <class Context>
	void load(const uint8_t* data, size_t size, Context& context) {
		// const uint8_t *start = data;
		const uint8_t* end = data + size;
		while (data < end) {
			uint8_t len = *(data++);
			// Tags are already deduplicated
			const auto& tag = tags.emplace_back(context.tryReadZeroCopy(data, len), len);
			data += len;
			bytes += tag.size();
		}

		ASSERT(data == end);

		// Deserialized tag sets share the arena with the request that contained them
		// For this reason, persisting a TagSet that shares memory with other request
		// members should be done with caution.
		arena = context.arena();
	}

	size_t getBytes() const { return bytes; }

	const Arena& getArena() const { return arena; }

	// Used by fdbcli commands
	std::string toString(Capitalize = Capitalize::False) const;

private:
	size_t bytes;
	Arena arena;
	// Currently there are never >= 256 tags, so
	// std::vector is faster than std::set. This may
	// change if we allow more tags in the future.
	std::vector<TransactionTagRef> tags;
};

template <>
struct dynamic_size_traits<TagSet> : std::true_type {
	// May be called multiple times during one serialization
	template <class Context>
	static size_t size(const TagSet& t, Context&) {
		return t.size() + t.getBytes();
	}

	// Guaranteed to be called only once during serialization
	template <class Context>
	static void save(uint8_t* out, const TagSet& t, Context& c) {
		t.save(out, c);
	}

	// Context is an arbitrary type that is plumbed by reference throughout the
	// load call tree.
	template <class Context>
	static void load(const uint8_t* data, size_t size, TagSet& t, Context& context) {
		t.load(data, size, context);
	}
};

enum class TagThrottleType : uint8_t { MANUAL, AUTO };

enum class TagThrottledReason : uint8_t { UNSET = 0, MANUAL, BUSY_READ, BUSY_WRITE };

struct TagThrottleKey {
	TagSet tags;
	TagThrottleType throttleType;
	TransactionPriority priority;

	TagThrottleKey() : throttleType(TagThrottleType::MANUAL), priority(TransactionPriority::DEFAULT) {}
	TagThrottleKey(TagSet tags, TagThrottleType throttleType, TransactionPriority priority)
	  : tags(tags), throttleType(throttleType), priority(priority) {}

	Key toKey() const;
	static TagThrottleKey fromKey(const KeyRef& key);
};

struct TagThrottleValue {
	double tpsRate;
	double expirationTime;
	double initialDuration;
	TagThrottledReason reason;

	TagThrottleValue() : tpsRate(0), expirationTime(0), initialDuration(0), reason(TagThrottledReason::UNSET) {}
	TagThrottleValue(double tpsRate, double expirationTime, double initialDuration, TagThrottledReason reason)
	  : tpsRate(tpsRate), expirationTime(expirationTime), initialDuration(initialDuration), reason(reason) {}

	static TagThrottleValue fromValue(const ValueRef& value);

	// To change this serialization, ProtocolVersion::TagThrottleValue must be updated, and downgrades need to be
	// considered
	template <class Ar>
	void serialize(Ar& ar) {
		ASSERT_WE_THINK(ar.protocolVersion().hasTagThrottleValueReason());
		serializer(ar, tpsRate, expirationTime, initialDuration, reason);
	}
};

struct TagThrottleInfo {
	TransactionTag tag;
	TagThrottleType throttleType;
	TransactionPriority priority;
	double tpsRate;
	double expirationTime;
	double initialDuration;
	TagThrottledReason reason;

	TagThrottleInfo(TransactionTag tag,
	                TagThrottleType throttleType,
	                TransactionPriority priority,
	                double tpsRate,
	                double expirationTime,
	                double initialDuration,
	                TagThrottledReason reason = TagThrottledReason::UNSET)
	  : tag(tag), throttleType(throttleType), priority(priority), tpsRate(tpsRate), expirationTime(expirationTime),
	    initialDuration(initialDuration), reason(reason) {}

	TagThrottleInfo(TagThrottleKey key, TagThrottleValue value)
	  : throttleType(key.throttleType), priority(key.priority), tpsRate(value.tpsRate),
	    expirationTime(value.expirationTime), initialDuration(value.initialDuration), reason(value.reason) {
		ASSERT(key.tags.size() == 1); // Multiple tags per throttle is not currently supported
		tag = *key.tags.begin();
	}
};

struct ClientTagThrottleLimits {
	double tpsRate;
	double expiration;

	static double const NO_EXPIRATION;

	ClientTagThrottleLimits() : tpsRate(0), expiration(0) {}
	ClientTagThrottleLimits(double tpsRate, double expiration) : tpsRate(tpsRate), expiration(expiration) {}

	template <class Archive>
	void serialize(Archive& ar) {
		// Convert expiration time to a duration to avoid clock differences
		double duration = 0;
		if (!ar.isDeserializing) {
			duration = expiration - now();
		}

		serializer(ar, tpsRate, duration);

		if (ar.isDeserializing) {
			expiration = now() + duration;
		}
	}
};

struct ClientTrCommitCostEstimation {
	int opsCount = 0;
	uint64_t writeCosts = 0;
	std::deque<std::pair<int, uint64_t>> clearIdxCosts;
	uint32_t expensiveCostEstCount = 0;
	template <class Ar>
	void serialize(Ar& ar) {
		serializer(ar, opsCount, writeCosts, clearIdxCosts, expensiveCostEstCount);
	}
};

// Keys to view and control tag throttling
extern const KeyRangeRef tagThrottleKeys;
extern const KeyRef tagThrottleKeysPrefix;
extern const KeyRef tagThrottleAutoKeysPrefix;
extern const KeyRef tagThrottleSignalKey;
extern const KeyRef tagThrottleAutoEnabledKey;
extern const KeyRef tagThrottleLimitKey;
extern const KeyRef tagThrottleCountKey;

namespace ThrottleApi {

// The template functions can be called with Native API like DatabaseContext, Transaction/ReadYourWritesTransaction
// or using IClientAPI like IDatabase, ITransaction

															#line 252 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
// This generated class is to be used only via getValidAutoEnabled()
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class Tr, class GetValidAutoEnabledActor>
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class GetValidAutoEnabledActorState {
															#line 258 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	GetValidAutoEnabledActorState(Reference<Tr> const& tr) 
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		 : tr(tr),
															#line 253 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   valueF(tr->get(tagThrottleAutoEnabledKey))
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
	{
		fdb_probe_actor_create("getValidAutoEnabled", reinterpret_cast<unsigned long>(this));

	}
	~GetValidAutoEnabledActorState() 
	{
		fdb_probe_actor_destroy("getValidAutoEnabled", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 254 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Optional<Value>> __when_expr_0 = safeThreadFutureToFuture(valueF);
															#line 254 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<GetValidAutoEnabledActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 284 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetValidAutoEnabledActor*>(this)->actor_wait_state = 1;
															#line 254 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetValidAutoEnabledActor, 0, Optional<Value> >*>(static_cast<GetValidAutoEnabledActor*>(this)));
															#line 289 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetValidAutoEnabledActorState();
		static_cast<GetValidAutoEnabledActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> const& value,int loopDepth) 
	{
															#line 255 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!value.present())
															#line 312 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 256 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (!static_cast<GetValidAutoEnabledActor*>(this)->SAV<Optional<bool>>::futures) { (void)({}); this->~GetValidAutoEnabledActorState(); static_cast<GetValidAutoEnabledActor*>(this)->destroy(); return 0; }
															#line 316 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			new (&static_cast<GetValidAutoEnabledActor*>(this)->SAV< Optional<bool> >::value()) Optional<bool>({});
			this->~GetValidAutoEnabledActorState();
			static_cast<GetValidAutoEnabledActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 257 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (value.get() == "1"_sr)
															#line 326 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			{
															#line 258 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				if (!static_cast<GetValidAutoEnabledActor*>(this)->SAV<Optional<bool>>::futures) { (void)(true); this->~GetValidAutoEnabledActorState(); static_cast<GetValidAutoEnabledActor*>(this)->destroy(); return 0; }
															#line 330 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				new (&static_cast<GetValidAutoEnabledActor*>(this)->SAV< Optional<bool> >::value()) Optional<bool>(true);
				this->~GetValidAutoEnabledActorState();
				static_cast<GetValidAutoEnabledActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 259 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				if (value.get() == "0"_sr)
															#line 340 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				{
															#line 260 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
					if (!static_cast<GetValidAutoEnabledActor*>(this)->SAV<Optional<bool>>::futures) { (void)(false); this->~GetValidAutoEnabledActorState(); static_cast<GetValidAutoEnabledActor*>(this)->destroy(); return 0; }
															#line 344 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
					new (&static_cast<GetValidAutoEnabledActor*>(this)->SAV< Optional<bool> >::value()) Optional<bool>(false);
					this->~GetValidAutoEnabledActorState();
					static_cast<GetValidAutoEnabledActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
				else
				{
															#line 262 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
					TraceEvent(SevWarnAlways, "InvalidAutoTagThrottlingValue").detail("Value", value.get());
															#line 263 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
					if (!static_cast<GetValidAutoEnabledActor*>(this)->SAV<Optional<bool>>::futures) { (void)({}); this->~GetValidAutoEnabledActorState(); static_cast<GetValidAutoEnabledActor*>(this)->destroy(); return 0; }
															#line 356 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
					new (&static_cast<GetValidAutoEnabledActor*>(this)->SAV< Optional<bool> >::value()) Optional<bool>({});
					this->~GetValidAutoEnabledActorState();
					static_cast<GetValidAutoEnabledActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
		}

		return loopDepth;
	}
	int a_body1cont1(Optional<Value> && value,int loopDepth) 
	{
															#line 255 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!value.present())
															#line 371 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 256 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (!static_cast<GetValidAutoEnabledActor*>(this)->SAV<Optional<bool>>::futures) { (void)({}); this->~GetValidAutoEnabledActorState(); static_cast<GetValidAutoEnabledActor*>(this)->destroy(); return 0; }
															#line 375 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			new (&static_cast<GetValidAutoEnabledActor*>(this)->SAV< Optional<bool> >::value()) Optional<bool>({});
			this->~GetValidAutoEnabledActorState();
			static_cast<GetValidAutoEnabledActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		else
		{
															#line 257 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (value.get() == "1"_sr)
															#line 385 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			{
															#line 258 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				if (!static_cast<GetValidAutoEnabledActor*>(this)->SAV<Optional<bool>>::futures) { (void)(true); this->~GetValidAutoEnabledActorState(); static_cast<GetValidAutoEnabledActor*>(this)->destroy(); return 0; }
															#line 389 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				new (&static_cast<GetValidAutoEnabledActor*>(this)->SAV< Optional<bool> >::value()) Optional<bool>(true);
				this->~GetValidAutoEnabledActorState();
				static_cast<GetValidAutoEnabledActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			else
			{
															#line 259 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				if (value.get() == "0"_sr)
															#line 399 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				{
															#line 260 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
					if (!static_cast<GetValidAutoEnabledActor*>(this)->SAV<Optional<bool>>::futures) { (void)(false); this->~GetValidAutoEnabledActorState(); static_cast<GetValidAutoEnabledActor*>(this)->destroy(); return 0; }
															#line 403 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
					new (&static_cast<GetValidAutoEnabledActor*>(this)->SAV< Optional<bool> >::value()) Optional<bool>(false);
					this->~GetValidAutoEnabledActorState();
					static_cast<GetValidAutoEnabledActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
				else
				{
															#line 262 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
					TraceEvent(SevWarnAlways, "InvalidAutoTagThrottlingValue").detail("Value", value.get());
															#line 263 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
					if (!static_cast<GetValidAutoEnabledActor*>(this)->SAV<Optional<bool>>::futures) { (void)({}); this->~GetValidAutoEnabledActorState(); static_cast<GetValidAutoEnabledActor*>(this)->destroy(); return 0; }
															#line 415 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
					new (&static_cast<GetValidAutoEnabledActor*>(this)->SAV< Optional<bool> >::value()) Optional<bool>({});
					this->~GetValidAutoEnabledActorState();
					static_cast<GetValidAutoEnabledActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
		}

		return loopDepth;
	}
	int a_body1when1(Optional<Value> const& value,int loopDepth) 
	{
		loopDepth = a_body1cont1(value, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> && value,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(value), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetValidAutoEnabledActor*>(this)->actor_wait_state > 0) static_cast<GetValidAutoEnabledActor*>(this)->actor_wait_state = 0;
		static_cast<GetValidAutoEnabledActor*>(this)->ActorCallback< GetValidAutoEnabledActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< GetValidAutoEnabledActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("getValidAutoEnabled", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValidAutoEnabledActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValidAutoEnabled", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetValidAutoEnabledActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("getValidAutoEnabled", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetValidAutoEnabledActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValidAutoEnabled", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetValidAutoEnabledActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("getValidAutoEnabled", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetValidAutoEnabledActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getValidAutoEnabled", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<Tr> tr;
															#line 253 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	typename Tr::template FutureT<Optional<Value>> valueF;
															#line 505 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
};
// This generated class is to be used only via getValidAutoEnabled()
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class Tr>
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class GetValidAutoEnabledActor final : public Actor<Optional<bool>>, public ActorCallback< GetValidAutoEnabledActor<Tr>, 0, Optional<Value> >, public FastAllocated<GetValidAutoEnabledActor<Tr>>, public GetValidAutoEnabledActorState<Tr, GetValidAutoEnabledActor<Tr>> {
															#line 512 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
	using FastAllocated<GetValidAutoEnabledActor<Tr>>::operator new;
	using FastAllocated<GetValidAutoEnabledActor<Tr>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17783007406476419328UL, 3052099349888022528UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<bool>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetValidAutoEnabledActor<Tr>, 0, Optional<Value> >;
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	GetValidAutoEnabledActor(Reference<Tr> const& tr) 
															#line 529 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		 : Actor<Optional<bool>>(),
		   GetValidAutoEnabledActorState<Tr, GetValidAutoEnabledActor<Tr>>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getValidAutoEnabled", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9384394887517903872UL, 6221360926521865472UL);
		ActorExecutionContextHelper __helper(static_cast<GetValidAutoEnabledActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getValidAutoEnabled");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getValidAutoEnabled", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetValidAutoEnabledActor<Tr>, 0, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class Tr>
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
[[nodiscard]] Future<Optional<bool>> getValidAutoEnabled( Reference<Tr> const& tr ) {
															#line 250 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	return Future<Optional<bool>>(new GetValidAutoEnabledActor<Tr>(tr));
															#line 563 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
}

#line 266 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"

															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
// This generated class is to be used only via getRecommendedTags()
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB, class GetRecommendedTagsActor>
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class GetRecommendedTagsActorState {
															#line 574 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	GetRecommendedTagsActorState(Reference<DB> const& db,int const& limit) 
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		 : db(db),
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   limit(limit),
															#line 269 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   tr(db->createTransaction())
															#line 585 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
	{
		fdb_probe_actor_create("getRecommendedTags", reinterpret_cast<unsigned long>(this));

	}
	~GetRecommendedTagsActorState() 
	{
		fdb_probe_actor_destroy("getRecommendedTags", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 270 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			;
															#line 600 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetRecommendedTagsActorState();
		static_cast<GetRecommendedTagsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 272 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			tr->setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 273 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Optional<bool>> __when_expr_0 = getValidAutoEnabled(tr);
															#line 273 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 635 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state = 1;
															#line 273 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetRecommendedTagsActor, 0, Optional<bool> >*>(static_cast<GetRecommendedTagsActor*>(this)));
															#line 640 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 291 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_3 = safeThreadFutureToFuture(tr->onError(e));
															#line 291 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 664 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state = 4;
															#line 291 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetRecommendedTagsActor, 3, Void >*>(static_cast<GetRecommendedTagsActor*>(this)));
															#line 669 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<bool> const& enableAuto,int loopDepth) 
	{
															#line 274 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!enableAuto.present())
															#line 684 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 275 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			tr->reset();
															#line 276 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_1 = delay(CLIENT_KNOBS->DEFAULT_BACKOFF);
															#line 276 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 692 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state = 2;
															#line 276 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetRecommendedTagsActor, 1, Void >*>(static_cast<GetRecommendedTagsActor*>(this)));
															#line 697 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
															#line 278 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (enableAuto.get())
															#line 704 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			{
															#line 279 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				if (!static_cast<GetRecommendedTagsActor*>(this)->SAV<std::vector<TagThrottleInfo>>::futures) { (void)(std::vector<TagThrottleInfo>()); this->~GetRecommendedTagsActorState(); static_cast<GetRecommendedTagsActor*>(this)->destroy(); return 0; }
															#line 708 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				new (&static_cast<GetRecommendedTagsActor*>(this)->SAV< std::vector<TagThrottleInfo> >::value()) std::vector<TagThrottleInfo>(std::vector<TagThrottleInfo>());
				this->~GetRecommendedTagsActorState();
				static_cast<GetRecommendedTagsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<bool> && enableAuto,int loopDepth) 
	{
															#line 274 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!enableAuto.present())
															#line 723 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 275 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			tr->reset();
															#line 276 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_1 = delay(CLIENT_KNOBS->DEFAULT_BACKOFF);
															#line 276 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 731 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state = 2;
															#line 276 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetRecommendedTagsActor, 1, Void >*>(static_cast<GetRecommendedTagsActor*>(this)));
															#line 736 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
															#line 278 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (enableAuto.get())
															#line 743 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			{
															#line 279 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				if (!static_cast<GetRecommendedTagsActor*>(this)->SAV<std::vector<TagThrottleInfo>>::futures) { (void)(std::vector<TagThrottleInfo>()); this->~GetRecommendedTagsActorState(); static_cast<GetRecommendedTagsActor*>(this)->destroy(); return 0; }
															#line 747 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				new (&static_cast<GetRecommendedTagsActor*>(this)->SAV< std::vector<TagThrottleInfo> >::value()) std::vector<TagThrottleInfo>(std::vector<TagThrottleInfo>());
				this->~GetRecommendedTagsActorState();
				static_cast<GetRecommendedTagsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<bool> const& enableAuto,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(enableAuto, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<bool> && enableAuto,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(enableAuto), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state > 0) static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state = 0;
		static_cast<GetRecommendedTagsActor*>(this)->ActorCallback< GetRecommendedTagsActor, 0, Optional<bool> >::remove();

	}
	void a_callback_fire(ActorCallback< GetRecommendedTagsActor, 0, Optional<bool> >*,Optional<bool> const& value) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetRecommendedTagsActor, 0, Optional<bool> >*,Optional<bool> && value) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetRecommendedTagsActor, 0, Optional<bool> >*,Error err) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 281 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		f = tr->getRange(KeyRangeRef(tagThrottleAutoKeysPrefix, tagThrottleKeys.end), limit);
															#line 283 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		StrictFuture<RangeResult> __when_expr_2 = safeThreadFutureToFuture(f);
															#line 283 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 841 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state = 3;
															#line 283 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetRecommendedTagsActor, 2, RangeResult >*>(static_cast<GetRecommendedTagsActor*>(this)));
															#line 846 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state > 0) static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state = 0;
		static_cast<GetRecommendedTagsActor*>(this)->ActorCallback< GetRecommendedTagsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetRecommendedTagsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetRecommendedTagsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetRecommendedTagsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont8(RangeResult const& throttles,int loopDepth) 
	{
															#line 284 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		std::vector<TagThrottleInfo> results;
															#line 285 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		for( auto throttle : throttles ) {
															#line 286 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			results.push_back(TagThrottleInfo(TagThrottleKey::fromKey(throttle.key), TagThrottleValue::fromValue(throttle.value)));
															#line 946 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 289 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!static_cast<GetRecommendedTagsActor*>(this)->SAV<std::vector<TagThrottleInfo>>::futures) { (void)(results); this->~GetRecommendedTagsActorState(); static_cast<GetRecommendedTagsActor*>(this)->destroy(); return 0; }
															#line 950 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		new (&static_cast<GetRecommendedTagsActor*>(this)->SAV< std::vector<TagThrottleInfo> >::value()) std::vector<TagThrottleInfo>(results);
		this->~GetRecommendedTagsActorState();
		static_cast<GetRecommendedTagsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont8(RangeResult && throttles,int loopDepth) 
	{
															#line 284 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		std::vector<TagThrottleInfo> results;
															#line 285 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		for( auto throttle : throttles ) {
															#line 286 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			results.push_back(TagThrottleInfo(TagThrottleKey::fromKey(throttle.key), TagThrottleValue::fromValue(throttle.value)));
															#line 966 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 289 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!static_cast<GetRecommendedTagsActor*>(this)->SAV<std::vector<TagThrottleInfo>>::futures) { (void)(results); this->~GetRecommendedTagsActorState(); static_cast<GetRecommendedTagsActor*>(this)->destroy(); return 0; }
															#line 970 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		new (&static_cast<GetRecommendedTagsActor*>(this)->SAV< std::vector<TagThrottleInfo> >::value()) std::vector<TagThrottleInfo>(results);
		this->~GetRecommendedTagsActorState();
		static_cast<GetRecommendedTagsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(RangeResult const& throttles,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(throttles, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(RangeResult && throttles,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(throttles), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state > 0) static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state = 0;
		static_cast<GetRecommendedTagsActor*>(this)->ActorCallback< GetRecommendedTagsActor, 2, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetRecommendedTagsActor, 2, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetRecommendedTagsActor, 2, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetRecommendedTagsActor, 2, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state > 0) static_cast<GetRecommendedTagsActor*>(this)->actor_wait_state = 0;
		static_cast<GetRecommendedTagsActor*>(this)->ActorCallback< GetRecommendedTagsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetRecommendedTagsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetRecommendedTagsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetRecommendedTagsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<DB> db;
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	int limit;
															#line 269 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<typename DB::TransactionT> tr;
															#line 281 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	typename DB::TransactionT::template FutureT<RangeResult> f;
															#line 1148 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
};
// This generated class is to be used only via getRecommendedTags()
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class GetRecommendedTagsActor final : public Actor<std::vector<TagThrottleInfo>>, public ActorCallback< GetRecommendedTagsActor<DB>, 0, Optional<bool> >, public ActorCallback< GetRecommendedTagsActor<DB>, 1, Void >, public ActorCallback< GetRecommendedTagsActor<DB>, 2, RangeResult >, public ActorCallback< GetRecommendedTagsActor<DB>, 3, Void >, public FastAllocated<GetRecommendedTagsActor<DB>>, public GetRecommendedTagsActorState<DB, GetRecommendedTagsActor<DB>> {
															#line 1155 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
	using FastAllocated<GetRecommendedTagsActor<DB>>::operator new;
	using FastAllocated<GetRecommendedTagsActor<DB>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18331894523656505088UL, 8760485374114601984UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<TagThrottleInfo>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetRecommendedTagsActor<DB>, 0, Optional<bool> >;
friend struct ActorCallback< GetRecommendedTagsActor<DB>, 1, Void >;
friend struct ActorCallback< GetRecommendedTagsActor<DB>, 2, RangeResult >;
friend struct ActorCallback< GetRecommendedTagsActor<DB>, 3, Void >;
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	GetRecommendedTagsActor(Reference<DB> const& db,int const& limit) 
															#line 1175 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		 : Actor<std::vector<TagThrottleInfo>>(),
		   GetRecommendedTagsActorState<DB, GetRecommendedTagsActor<DB>>(db, limit),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getRecommendedTags", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12239974780417389056UL, 14839766086862942976UL);
		ActorExecutionContextHelper __helper(static_cast<GetRecommendedTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getRecommendedTags");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getRecommendedTags", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetRecommendedTagsActor<DB>, 0, Optional<bool> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetRecommendedTagsActor<DB>, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetRecommendedTagsActor<DB>, 2, RangeResult >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetRecommendedTagsActor<DB>, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
[[nodiscard]] Future<std::vector<TagThrottleInfo>> getRecommendedTags( Reference<DB> const& db, int const& limit ) {
															#line 267 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	return Future<std::vector<TagThrottleInfo>>(new GetRecommendedTagsActor<DB>(db, limit));
															#line 1212 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
}

#line 295 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"

															#line 1217 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
// This generated class is to be used only via getThrottledTags()
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB, class GetThrottledTagsActor>
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class GetThrottledTagsActorState {
															#line 1223 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	GetThrottledTagsActorState(Reference<DB> const& db,int const& limit,ContainsRecommended const& containsRecommended = ContainsRecommended::False) 
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		 : db(db),
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   limit(limit),
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   containsRecommended(containsRecommended),
															#line 299 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   tr(db->createTransaction()),
															#line 300 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   reportAuto()
															#line 1238 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
	{
		fdb_probe_actor_create("getThrottledTags", reinterpret_cast<unsigned long>(this));

	}
	~GetThrottledTagsActorState() 
	{
		fdb_probe_actor_destroy("getThrottledTags", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 301 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			;
															#line 1253 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetThrottledTagsActorState();
		static_cast<GetThrottledTagsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 303 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			tr->setOption(FDBTransactionOptions::READ_SYSTEM_KEYS);
															#line 304 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_0 = store(reportAuto, getValidAutoEnabled(tr));
															#line 304 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<GetThrottledTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1288 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<GetThrottledTagsActor*>(this)->actor_wait_state = 1;
															#line 304 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetThrottledTagsActor, 0, Void >*>(static_cast<GetThrottledTagsActor*>(this)));
															#line 1293 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 321 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_3 = safeThreadFutureToFuture(tr->onError(e));
															#line 321 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<GetThrottledTagsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1317 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<GetThrottledTagsActor*>(this)->actor_wait_state = 4;
															#line 321 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< GetThrottledTagsActor, 3, Void >*>(static_cast<GetThrottledTagsActor*>(this)));
															#line 1322 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 305 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!reportAuto.present())
															#line 1337 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 306 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			tr->reset();
															#line 307 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_1 = delay(CLIENT_KNOBS->DEFAULT_BACKOFF);
															#line 307 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<GetThrottledTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1345 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetThrottledTagsActor*>(this)->actor_wait_state = 2;
															#line 307 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetThrottledTagsActor, 1, Void >*>(static_cast<GetThrottledTagsActor*>(this)));
															#line 1350 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 305 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!reportAuto.present())
															#line 1364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 306 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			tr->reset();
															#line 307 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_1 = delay(CLIENT_KNOBS->DEFAULT_BACKOFF);
															#line 307 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<GetThrottledTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1372 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<GetThrottledTagsActor*>(this)->actor_wait_state = 2;
															#line 307 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetThrottledTagsActor, 1, Void >*>(static_cast<GetThrottledTagsActor*>(this)));
															#line 1377 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetThrottledTagsActor*>(this)->actor_wait_state > 0) static_cast<GetThrottledTagsActor*>(this)->actor_wait_state = 0;
		static_cast<GetThrottledTagsActor*>(this)->ActorCallback< GetThrottledTagsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetThrottledTagsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetThrottledTagsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetThrottledTagsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 310 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		f = tr->getRange( reportAuto.get() ? tagThrottleKeys : KeyRangeRef(tagThrottleKeysPrefix, tagThrottleAutoKeysPrefix), limit);
															#line 313 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		StrictFuture<RangeResult> __when_expr_2 = safeThreadFutureToFuture(f);
															#line 313 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (static_cast<GetThrottledTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1470 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<GetThrottledTagsActor*>(this)->actor_wait_state = 3;
															#line 313 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< GetThrottledTagsActor, 2, RangeResult >*>(static_cast<GetThrottledTagsActor*>(this)));
															#line 1475 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
		return a_body1loopHead1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<GetThrottledTagsActor*>(this)->actor_wait_state > 0) static_cast<GetThrottledTagsActor*>(this)->actor_wait_state = 0;
		static_cast<GetThrottledTagsActor*>(this)->ActorCallback< GetThrottledTagsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetThrottledTagsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetThrottledTagsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetThrottledTagsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(RangeResult const& throttles,int loopDepth) 
	{
															#line 314 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		std::vector<TagThrottleInfo> results;
															#line 315 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		for( auto throttle : throttles ) {
															#line 316 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			results.push_back(TagThrottleInfo(TagThrottleKey::fromKey(throttle.key), TagThrottleValue::fromValue(throttle.value)));
															#line 1575 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 319 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!static_cast<GetThrottledTagsActor*>(this)->SAV<std::vector<TagThrottleInfo>>::futures) { (void)(results); this->~GetThrottledTagsActorState(); static_cast<GetThrottledTagsActor*>(this)->destroy(); return 0; }
															#line 1579 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		new (&static_cast<GetThrottledTagsActor*>(this)->SAV< std::vector<TagThrottleInfo> >::value()) std::vector<TagThrottleInfo>(results);
		this->~GetThrottledTagsActorState();
		static_cast<GetThrottledTagsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(RangeResult && throttles,int loopDepth) 
	{
															#line 314 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		std::vector<TagThrottleInfo> results;
															#line 315 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		for( auto throttle : throttles ) {
															#line 316 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			results.push_back(TagThrottleInfo(TagThrottleKey::fromKey(throttle.key), TagThrottleValue::fromValue(throttle.value)));
															#line 1595 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 319 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!static_cast<GetThrottledTagsActor*>(this)->SAV<std::vector<TagThrottleInfo>>::futures) { (void)(results); this->~GetThrottledTagsActorState(); static_cast<GetThrottledTagsActor*>(this)->destroy(); return 0; }
															#line 1599 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		new (&static_cast<GetThrottledTagsActor*>(this)->SAV< std::vector<TagThrottleInfo> >::value()) std::vector<TagThrottleInfo>(results);
		this->~GetThrottledTagsActorState();
		static_cast<GetThrottledTagsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(RangeResult const& throttles,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(throttles, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(RangeResult && throttles,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(throttles), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<GetThrottledTagsActor*>(this)->actor_wait_state > 0) static_cast<GetThrottledTagsActor*>(this)->actor_wait_state = 0;
		static_cast<GetThrottledTagsActor*>(this)->ActorCallback< GetThrottledTagsActor, 2, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< GetThrottledTagsActor, 2, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< GetThrottledTagsActor, 2, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< GetThrottledTagsActor, 2, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<GetThrottledTagsActor*>(this)->actor_wait_state > 0) static_cast<GetThrottledTagsActor*>(this)->actor_wait_state = 0;
		static_cast<GetThrottledTagsActor*>(this)->ActorCallback< GetThrottledTagsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetThrottledTagsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< GetThrottledTagsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< GetThrottledTagsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<DB> db;
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	int limit;
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	ContainsRecommended containsRecommended;
															#line 299 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<typename DB::TransactionT> tr;
															#line 300 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Optional<bool> reportAuto;
															#line 310 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	typename DB::TransactionT::template FutureT<RangeResult> f;
															#line 1781 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
};
// This generated class is to be used only via getThrottledTags()
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class GetThrottledTagsActor final : public Actor<std::vector<TagThrottleInfo>>, public ActorCallback< GetThrottledTagsActor<DB>, 0, Void >, public ActorCallback< GetThrottledTagsActor<DB>, 1, Void >, public ActorCallback< GetThrottledTagsActor<DB>, 2, RangeResult >, public ActorCallback< GetThrottledTagsActor<DB>, 3, Void >, public FastAllocated<GetThrottledTagsActor<DB>>, public GetThrottledTagsActorState<DB, GetThrottledTagsActor<DB>> {
															#line 1788 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
	using FastAllocated<GetThrottledTagsActor<DB>>::operator new;
	using FastAllocated<GetThrottledTagsActor<DB>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6824524648477796864UL, 11377336964599046912UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::vector<TagThrottleInfo>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetThrottledTagsActor<DB>, 0, Void >;
friend struct ActorCallback< GetThrottledTagsActor<DB>, 1, Void >;
friend struct ActorCallback< GetThrottledTagsActor<DB>, 2, RangeResult >;
friend struct ActorCallback< GetThrottledTagsActor<DB>, 3, Void >;
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	GetThrottledTagsActor(Reference<DB> const& db,int const& limit,ContainsRecommended const& containsRecommended = ContainsRecommended::False) 
															#line 1808 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		 : Actor<std::vector<TagThrottleInfo>>(),
		   GetThrottledTagsActorState<DB, GetThrottledTagsActor<DB>>(db, limit, containsRecommended),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getThrottledTags", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10966917763343875072UL, 12730619737688705792UL);
		ActorExecutionContextHelper __helper(static_cast<GetThrottledTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getThrottledTags");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getThrottledTags", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetThrottledTagsActor<DB>, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< GetThrottledTagsActor<DB>, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< GetThrottledTagsActor<DB>, 2, RangeResult >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< GetThrottledTagsActor<DB>, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
[[nodiscard]] Future<std::vector<TagThrottleInfo>> getThrottledTags( Reference<DB> const& db, int const& limit, ContainsRecommended const& containsRecommended = ContainsRecommended::False ) {
															#line 296 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	return Future<std::vector<TagThrottleInfo>>(new GetThrottledTagsActor<DB>(db, limit, containsRecommended));
															#line 1845 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
}

#line 325 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"

template <class Tr>
void signalThrottleChange(Reference<Tr> tr) {
	tr->atomicOp(tagThrottleSignalKey, "XXXXXXXXXX\x00\x00\x00\x00"_sr, MutationRef::SetVersionstampedValue);
}

															#line 1855 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
// This generated class is to be used only via updateThrottleCount()
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class Tr, class UpdateThrottleCountActor>
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class UpdateThrottleCountActorState {
															#line 1861 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	UpdateThrottleCountActorState(Reference<Tr> const& tr,int64_t const& delta) 
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		 : tr(tr),
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   delta(delta),
															#line 333 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   countVal(tr->get(tagThrottleCountKey)),
															#line 334 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   limitVal(tr->get(tagThrottleLimitKey))
															#line 1874 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
	{
		fdb_probe_actor_create("updateThrottleCount", reinterpret_cast<unsigned long>(this));

	}
	~UpdateThrottleCountActorState() 
	{
		fdb_probe_actor_destroy("updateThrottleCount", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 336 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_0 = success(safeThreadFutureToFuture(countVal)) && success(safeThreadFutureToFuture(limitVal));
															#line 336 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<UpdateThrottleCountActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1891 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UpdateThrottleCountActor*>(this)->actor_wait_state = 1;
															#line 336 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateThrottleCountActor, 0, Void >*>(static_cast<UpdateThrottleCountActor*>(this)));
															#line 1896 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateThrottleCountActorState();
		static_cast<UpdateThrottleCountActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 338 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		int64_t count = 0;
															#line 339 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		int64_t limit = 0;
															#line 341 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (countVal.get().present())
															#line 1923 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 342 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			BinaryReader reader(countVal.get().get(), Unversioned());
															#line 343 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			reader >> count;
															#line 1929 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 346 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (limitVal.get().present())
															#line 1933 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 347 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			BinaryReader reader(limitVal.get().get(), Unversioned());
															#line 348 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			reader >> limit;
															#line 1939 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 351 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		count += delta;
															#line 353 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (count > limit)
															#line 1945 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 354 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			return a_body1Catch1(too_many_tag_throttles(), loopDepth);
															#line 1949 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 357 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		BinaryWriter writer(Unversioned());
															#line 358 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		writer << count;
															#line 360 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		tr->set(tagThrottleCountKey, writer.toValue());
															#line 361 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!static_cast<UpdateThrottleCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateThrottleCountActorState(); static_cast<UpdateThrottleCountActor*>(this)->destroy(); return 0; }
															#line 1959 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		new (&static_cast<UpdateThrottleCountActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateThrottleCountActorState();
		static_cast<UpdateThrottleCountActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 338 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		int64_t count = 0;
															#line 339 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		int64_t limit = 0;
															#line 341 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (countVal.get().present())
															#line 1975 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 342 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			BinaryReader reader(countVal.get().get(), Unversioned());
															#line 343 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			reader >> count;
															#line 1981 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 346 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (limitVal.get().present())
															#line 1985 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 347 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			BinaryReader reader(limitVal.get().get(), Unversioned());
															#line 348 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			reader >> limit;
															#line 1991 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 351 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		count += delta;
															#line 353 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (count > limit)
															#line 1997 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 354 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			return a_body1Catch1(too_many_tag_throttles(), loopDepth);
															#line 2001 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 357 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		BinaryWriter writer(Unversioned());
															#line 358 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		writer << count;
															#line 360 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		tr->set(tagThrottleCountKey, writer.toValue());
															#line 361 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!static_cast<UpdateThrottleCountActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateThrottleCountActorState(); static_cast<UpdateThrottleCountActor*>(this)->destroy(); return 0; }
															#line 2011 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		new (&static_cast<UpdateThrottleCountActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateThrottleCountActorState();
		static_cast<UpdateThrottleCountActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateThrottleCountActor*>(this)->actor_wait_state > 0) static_cast<UpdateThrottleCountActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateThrottleCountActor*>(this)->ActorCallback< UpdateThrottleCountActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateThrottleCountActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("updateThrottleCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateThrottleCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateThrottleCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateThrottleCountActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("updateThrottleCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateThrottleCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateThrottleCount", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateThrottleCountActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("updateThrottleCount", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateThrottleCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateThrottleCount", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<Tr> tr;
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	int64_t delta;
															#line 333 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	typename Tr::template FutureT<Optional<Value>> countVal;
															#line 334 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	typename Tr::template FutureT<Optional<Value>> limitVal;
															#line 2102 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
};
// This generated class is to be used only via updateThrottleCount()
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class Tr>
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class UpdateThrottleCountActor final : public Actor<Void>, public ActorCallback< UpdateThrottleCountActor<Tr>, 0, Void >, public FastAllocated<UpdateThrottleCountActor<Tr>>, public UpdateThrottleCountActorState<Tr, UpdateThrottleCountActor<Tr>> {
															#line 2109 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
	using FastAllocated<UpdateThrottleCountActor<Tr>>::operator new;
	using FastAllocated<UpdateThrottleCountActor<Tr>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7871380155323901440UL, 96735950689035264UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateThrottleCountActor<Tr>, 0, Void >;
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	UpdateThrottleCountActor(Reference<Tr> const& tr,int64_t const& delta) 
															#line 2126 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		 : Actor<Void>(),
		   UpdateThrottleCountActorState<Tr, UpdateThrottleCountActor<Tr>>(tr, delta),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateThrottleCount", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9379847179563386368UL, 6596548654355459328UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateThrottleCountActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateThrottleCount");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateThrottleCount", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateThrottleCountActor<Tr>, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class Tr>
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
[[nodiscard]] Future<Void> updateThrottleCount( Reference<Tr> const& tr, int64_t const& delta ) {
															#line 331 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	return Future<Void>(new UpdateThrottleCountActor<Tr>(tr, delta));
															#line 2160 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
}

#line 363 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"

															#line 2165 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
// This generated class is to be used only via unthrottleMatchingThrottles()
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB, class UnthrottleMatchingThrottlesActor>
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class UnthrottleMatchingThrottlesActorState {
															#line 2171 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	UnthrottleMatchingThrottlesActorState(Reference<DB> const& db,KeyRef const& beginKey,KeyRef const& endKey,Optional<TransactionPriority> const& priority,bool const& onlyExpiredThrottles) 
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		 : db(db),
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   beginKey(beginKey),
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   endKey(endKey),
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   priority(priority),
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   onlyExpiredThrottles(onlyExpiredThrottles),
															#line 370 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   tr(db->createTransaction()),
															#line 372 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   begin(firstGreaterOrEqual(beginKey)),
															#line 373 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   end(firstGreaterOrEqual(endKey)),
															#line 375 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   removed(false)
															#line 2194 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
	{
		fdb_probe_actor_create("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this));

	}
	~UnthrottleMatchingThrottlesActorState() 
	{
		fdb_probe_actor_destroy("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 377 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			;
															#line 2209 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UnthrottleMatchingThrottlesActorState();
		static_cast<UnthrottleMatchingThrottlesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 378 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2239 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		try {
															#line 381 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			f = tr->getRange(begin, end, 1000);
															#line 382 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<RangeResult> __when_expr_0 = safeThreadFutureToFuture(f);
															#line 382 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2247 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state = 1;
															#line 382 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UnthrottleMatchingThrottlesActor, 0, RangeResult >*>(static_cast<UnthrottleMatchingThrottlesActor*>(this)));
															#line 2252 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 424 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_3 = safeThreadFutureToFuture(tr->onError(e));
															#line 424 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2276 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state = 4;
															#line 424 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< UnthrottleMatchingThrottlesActor, 3, Void >*>(static_cast<UnthrottleMatchingThrottlesActor*>(this)));
															#line 2281 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 383 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		unthrottledTags = 0;
															#line 384 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		uint64_t manualUnthrottledTags = 0;
															#line 385 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		for( auto tag : tags ) {
															#line 386 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (onlyExpiredThrottles)
															#line 2302 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			{
															#line 387 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				double expirationTime = TagThrottleValue::fromValue(tag.value).expirationTime;
															#line 388 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				if (expirationTime == 0 || expirationTime > now())
															#line 2308 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				{
					continue;
				}
			}
															#line 393 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			TagThrottleKey key = TagThrottleKey::fromKey(tag.key);
															#line 394 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (priority.present() && key.priority != priority.get())
															#line 2317 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			{
				continue;
			}
															#line 398 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (key.throttleType == TagThrottleType::MANUAL)
															#line 2323 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			{
															#line 399 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				++manualUnthrottledTags;
															#line 2327 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			}
															#line 402 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			removed = true;
															#line 403 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			tr->clear(tag.key);
															#line 404 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			unthrottledTags++;
															#line 2335 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 407 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (manualUnthrottledTags > 0)
															#line 2339 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 408 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_1 = updateThrottleCount(tr, -manualUnthrottledTags);
															#line 408 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2345 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state = 2;
															#line 408 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< UnthrottleMatchingThrottlesActor, 1, Void >*>(static_cast<UnthrottleMatchingThrottlesActor*>(this)));
															#line 2350 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __tags,int loopDepth) 
	{
															#line 382 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		tags = __tags;
															#line 2364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __tags,int loopDepth) 
	{
		tags = std::move(__tags);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state > 0) static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state = 0;
		static_cast<UnthrottleMatchingThrottlesActor*>(this)->ActorCallback< UnthrottleMatchingThrottlesActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< UnthrottleMatchingThrottlesActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UnthrottleMatchingThrottlesActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UnthrottleMatchingThrottlesActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 411 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (unthrottledTags > 0)
															#line 2443 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 412 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			signalThrottleChange(tr);
															#line 2447 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 415 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		StrictFuture<Void> __when_expr_2 = safeThreadFutureToFuture(tr->commit());
															#line 415 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2453 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state = 3;
															#line 415 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< UnthrottleMatchingThrottlesActor, 2, Void >*>(static_cast<UnthrottleMatchingThrottlesActor*>(this)));
															#line 2458 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state > 0) static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state = 0;
		static_cast<UnthrottleMatchingThrottlesActor*>(this)->ActorCallback< UnthrottleMatchingThrottlesActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UnthrottleMatchingThrottlesActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< UnthrottleMatchingThrottlesActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< UnthrottleMatchingThrottlesActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 417 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!tags.more)
															#line 2554 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 418 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (!static_cast<UnthrottleMatchingThrottlesActor*>(this)->SAV<bool>::futures) { (void)(removed); this->~UnthrottleMatchingThrottlesActorState(); static_cast<UnthrottleMatchingThrottlesActor*>(this)->destroy(); return 0; }
															#line 2558 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			new (&static_cast<UnthrottleMatchingThrottlesActor*>(this)->SAV< bool >::value()) bool(std::move(removed)); // state_var_RVO
			this->~UnthrottleMatchingThrottlesActorState();
			static_cast<UnthrottleMatchingThrottlesActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 421 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		ASSERT(tags.size() > 0);
															#line 422 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		begin = KeySelector(firstGreaterThan(tags[tags.size() - 1].key), tags.arena());
															#line 2568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		loopDepth = a_body1loopBody1cont10cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 417 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!tags.more)
															#line 2577 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 418 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (!static_cast<UnthrottleMatchingThrottlesActor*>(this)->SAV<bool>::futures) { (void)(removed); this->~UnthrottleMatchingThrottlesActorState(); static_cast<UnthrottleMatchingThrottlesActor*>(this)->destroy(); return 0; }
															#line 2581 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			new (&static_cast<UnthrottleMatchingThrottlesActor*>(this)->SAV< bool >::value()) bool(std::move(removed)); // state_var_RVO
			this->~UnthrottleMatchingThrottlesActorState();
			static_cast<UnthrottleMatchingThrottlesActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 421 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		ASSERT(tags.size() > 0);
															#line 422 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		begin = KeySelector(firstGreaterThan(tags[tags.size() - 1].key), tags.arena());
															#line 2591 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		loopDepth = a_body1loopBody1cont10cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state > 0) static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state = 0;
		static_cast<UnthrottleMatchingThrottlesActor*>(this)->ActorCallback< UnthrottleMatchingThrottlesActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UnthrottleMatchingThrottlesActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< UnthrottleMatchingThrottlesActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< UnthrottleMatchingThrottlesActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont10cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state > 0) static_cast<UnthrottleMatchingThrottlesActor*>(this)->actor_wait_state = 0;
		static_cast<UnthrottleMatchingThrottlesActor*>(this)->ActorCallback< UnthrottleMatchingThrottlesActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UnthrottleMatchingThrottlesActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< UnthrottleMatchingThrottlesActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< UnthrottleMatchingThrottlesActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<DB> db;
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	KeyRef beginKey;
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	KeyRef endKey;
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Optional<TransactionPriority> priority;
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	bool onlyExpiredThrottles;
															#line 370 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<typename DB::TransactionT> tr;
															#line 372 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	KeySelector begin;
															#line 373 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	KeySelector end;
															#line 375 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	bool removed;
															#line 381 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	typename DB::TransactionT::template FutureT<RangeResult> f;
															#line 382 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	RangeResult tags;
															#line 383 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	uint64_t unthrottledTags;
															#line 2795 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
};
// This generated class is to be used only via unthrottleMatchingThrottles()
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class UnthrottleMatchingThrottlesActor final : public Actor<bool>, public ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 0, RangeResult >, public ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 1, Void >, public ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 2, Void >, public ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 3, Void >, public FastAllocated<UnthrottleMatchingThrottlesActor<DB>>, public UnthrottleMatchingThrottlesActorState<DB, UnthrottleMatchingThrottlesActor<DB>> {
															#line 2802 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
	using FastAllocated<UnthrottleMatchingThrottlesActor<DB>>::operator new;
	using FastAllocated<UnthrottleMatchingThrottlesActor<DB>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11103552997733105152UL, 7906290397463178752UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 0, RangeResult >;
friend struct ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 1, Void >;
friend struct ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 2, Void >;
friend struct ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 3, Void >;
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	UnthrottleMatchingThrottlesActor(Reference<DB> const& db,KeyRef const& beginKey,KeyRef const& endKey,Optional<TransactionPriority> const& priority,bool const& onlyExpiredThrottles) 
															#line 2822 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		 : Actor<bool>(),
		   UnthrottleMatchingThrottlesActorState<DB, UnthrottleMatchingThrottlesActor<DB>>(db, beginKey, endKey, priority, onlyExpiredThrottles),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6354176767819722240UL, 15398737564316821760UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleMatchingThrottlesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("unthrottleMatchingThrottles");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("unthrottleMatchingThrottles", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< UnthrottleMatchingThrottlesActor<DB>, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
[[nodiscard]] Future<bool> unthrottleMatchingThrottles( Reference<DB> const& db, KeyRef const& beginKey, KeyRef const& endKey, Optional<TransactionPriority> const& priority, bool const& onlyExpiredThrottles ) {
															#line 364 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	return Future<bool>(new UnthrottleMatchingThrottlesActor<DB>(db, beginKey, endKey, priority, onlyExpiredThrottles));
															#line 2859 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
}

#line 428 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"

template <class DB>
Future<bool> expire(DB db) {
	return unthrottleMatchingThrottles(
	    db, tagThrottleKeys.begin, tagThrottleKeys.end, Optional<TransactionPriority>(), true);
}

template <class DB>
Future<bool> unthrottleAll(Reference<DB> db,
                           Optional<TagThrottleType> tagThrottleType,
                           Optional<TransactionPriority> priority) {
	KeyRef begin = tagThrottleKeys.begin;
	KeyRef end = tagThrottleKeys.end;

	if (tagThrottleType.present() && tagThrottleType == TagThrottleType::AUTO) {
		begin = tagThrottleAutoKeysPrefix;
	} else if (tagThrottleType.present() && tagThrottleType == TagThrottleType::MANUAL) {
		end = tagThrottleAutoKeysPrefix;
	}

	return unthrottleMatchingThrottles(db, begin, end, priority, false);
}

															#line 2886 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
// This generated class is to be used only via unthrottleTags()
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB, class UnthrottleTagsActor>
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class UnthrottleTagsActorState {
															#line 2892 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	UnthrottleTagsActorState(Reference<DB> const& db,TagSet const& tags,Optional<TagThrottleType> const& throttleType,Optional<TransactionPriority> const& priority) 
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		 : db(db),
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   tags(tags),
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   throttleType(throttleType),
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   priority(priority),
															#line 456 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   tr(db->createTransaction()),
															#line 458 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   keys()
															#line 2909 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
	{
		fdb_probe_actor_create("unthrottleTags", reinterpret_cast<unsigned long>(this));

	}
	~UnthrottleTagsActorState() 
	{
		fdb_probe_actor_destroy("unthrottleTags", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 459 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			for( auto p : allTransactionPriorities ) {
															#line 460 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				if (!priority.present() || priority.get() == p)
															#line 2926 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				{
															#line 461 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
					if (!throttleType.present() || throttleType.get() == TagThrottleType::AUTO)
															#line 2930 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
					{
															#line 462 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
						keys.push_back(TagThrottleKey(tags, TagThrottleType::AUTO, p).toKey());
															#line 2934 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
					}
															#line 464 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
					if (!throttleType.present() || throttleType.get() == TagThrottleType::MANUAL)
															#line 2938 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
					{
															#line 465 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
						keys.push_back(TagThrottleKey(tags, TagThrottleType::MANUAL, p).toKey());
															#line 2942 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
					}
				}
			}
															#line 470 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			removed = false;
															#line 472 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			;
															#line 2950 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UnthrottleTagsActorState();
		static_cast<UnthrottleTagsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 473 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 2980 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		try {
															#line 475 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			valueFutures = std::vector<typename DB::TransactionT::template FutureT<Optional<Value>>>();
															#line 476 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			values = std::vector<Future<Optional<Value>>>();
															#line 477 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			values.reserve(keys.size());
															#line 478 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			for( auto key : keys ) {
															#line 479 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				valueFutures.push_back(tr->get(key));
															#line 480 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				values.push_back(safeThreadFutureToFuture(valueFutures.back()));
															#line 2994 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			}
															#line 483 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_0 = waitForAll(values);
															#line 483 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<UnthrottleTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3000 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UnthrottleTagsActor*>(this)->actor_wait_state = 1;
															#line 483 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UnthrottleTagsActor, 0, Void >*>(static_cast<UnthrottleTagsActor*>(this)));
															#line 3005 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 512 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_3 = safeThreadFutureToFuture(tr->onError(e));
															#line 512 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<UnthrottleTagsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3029 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<UnthrottleTagsActor*>(this)->actor_wait_state = 4;
															#line 512 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< UnthrottleTagsActor, 3, Void >*>(static_cast<UnthrottleTagsActor*>(this)));
															#line 3034 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 485 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		int delta = 0;
															#line 486 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		for(int i = 0;i < values.size();++i) {
															#line 487 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (values[i].get().present())
															#line 3053 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			{
															#line 488 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				if (TagThrottleKey::fromKey(keys[i]).throttleType == TagThrottleType::MANUAL)
															#line 3057 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				{
															#line 489 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
					delta -= 1;
															#line 3061 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				}
															#line 492 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				tr->clear(keys[i]);
															#line 498 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				removed = true;
															#line 3067 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			}
		}
															#line 502 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (delta != 0)
															#line 3072 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 503 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_1 = updateThrottleCount(tr, delta);
															#line 503 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<UnthrottleTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3078 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<UnthrottleTagsActor*>(this)->actor_wait_state = 2;
															#line 503 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< UnthrottleTagsActor, 1, Void >*>(static_cast<UnthrottleTagsActor*>(this)));
															#line 3083 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 485 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		int delta = 0;
															#line 486 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		for(int i = 0;i < values.size();++i) {
															#line 487 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (values[i].get().present())
															#line 3101 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			{
															#line 488 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				if (TagThrottleKey::fromKey(keys[i]).throttleType == TagThrottleType::MANUAL)
															#line 3105 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				{
															#line 489 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
					delta -= 1;
															#line 3109 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				}
															#line 492 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				tr->clear(keys[i]);
															#line 498 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				removed = true;
															#line 3115 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			}
		}
															#line 502 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (delta != 0)
															#line 3120 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 503 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_1 = updateThrottleCount(tr, delta);
															#line 503 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<UnthrottleTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3126 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<UnthrottleTagsActor*>(this)->actor_wait_state = 2;
															#line 503 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< UnthrottleTagsActor, 1, Void >*>(static_cast<UnthrottleTagsActor*>(this)));
															#line 3131 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UnthrottleTagsActor*>(this)->actor_wait_state > 0) static_cast<UnthrottleTagsActor*>(this)->actor_wait_state = 0;
		static_cast<UnthrottleTagsActor*>(this)->ActorCallback< UnthrottleTagsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UnthrottleTagsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UnthrottleTagsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UnthrottleTagsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 505 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (removed)
															#line 3220 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 506 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			signalThrottleChange(tr);
															#line 507 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_2 = safeThreadFutureToFuture(tr->commit());
															#line 507 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<UnthrottleTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3228 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_2.get(), loopDepth); };
			static_cast<UnthrottleTagsActor*>(this)->actor_wait_state = 3;
															#line 507 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< UnthrottleTagsActor, 2, Void >*>(static_cast<UnthrottleTagsActor*>(this)));
															#line 3233 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<UnthrottleTagsActor*>(this)->actor_wait_state > 0) static_cast<UnthrottleTagsActor*>(this)->actor_wait_state = 0;
		static_cast<UnthrottleTagsActor*>(this)->ActorCallback< UnthrottleTagsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UnthrottleTagsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< UnthrottleTagsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< UnthrottleTagsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
															#line 510 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!static_cast<UnthrottleTagsActor*>(this)->SAV<bool>::futures) { (void)(removed); this->~UnthrottleTagsActorState(); static_cast<UnthrottleTagsActor*>(this)->destroy(); return 0; }
															#line 3334 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		new (&static_cast<UnthrottleTagsActor*>(this)->SAV< bool >::value()) bool(std::move(removed)); // state_var_RVO
		this->~UnthrottleTagsActorState();
		static_cast<UnthrottleTagsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<UnthrottleTagsActor*>(this)->actor_wait_state > 0) static_cast<UnthrottleTagsActor*>(this)->actor_wait_state = 0;
		static_cast<UnthrottleTagsActor*>(this)->ActorCallback< UnthrottleTagsActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UnthrottleTagsActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< UnthrottleTagsActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< UnthrottleTagsActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<UnthrottleTagsActor*>(this)->actor_wait_state > 0) static_cast<UnthrottleTagsActor*>(this)->actor_wait_state = 0;
		static_cast<UnthrottleTagsActor*>(this)->ActorCallback< UnthrottleTagsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< UnthrottleTagsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< UnthrottleTagsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< UnthrottleTagsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<DB> db;
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	TagSet tags;
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Optional<TagThrottleType> throttleType;
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Optional<TransactionPriority> priority;
															#line 456 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<typename DB::TransactionT> tr;
															#line 458 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	std::vector<Key> keys;
															#line 470 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	bool removed;
															#line 475 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	std::vector<typename DB::TransactionT::template FutureT<Optional<Value>>> valueFutures;
															#line 476 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	std::vector<Future<Optional<Value>>> values;
															#line 3534 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
};
// This generated class is to be used only via unthrottleTags()
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class UnthrottleTagsActor final : public Actor<bool>, public ActorCallback< UnthrottleTagsActor<DB>, 0, Void >, public ActorCallback< UnthrottleTagsActor<DB>, 1, Void >, public ActorCallback< UnthrottleTagsActor<DB>, 2, Void >, public ActorCallback< UnthrottleTagsActor<DB>, 3, Void >, public FastAllocated<UnthrottleTagsActor<DB>>, public UnthrottleTagsActorState<DB, UnthrottleTagsActor<DB>> {
															#line 3541 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
	using FastAllocated<UnthrottleTagsActor<DB>>::operator new;
	using FastAllocated<UnthrottleTagsActor<DB>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12373717616170225152UL, 14071156526034902272UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UnthrottleTagsActor<DB>, 0, Void >;
friend struct ActorCallback< UnthrottleTagsActor<DB>, 1, Void >;
friend struct ActorCallback< UnthrottleTagsActor<DB>, 2, Void >;
friend struct ActorCallback< UnthrottleTagsActor<DB>, 3, Void >;
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	UnthrottleTagsActor(Reference<DB> const& db,TagSet const& tags,Optional<TagThrottleType> const& throttleType,Optional<TransactionPriority> const& priority) 
															#line 3561 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		 : Actor<bool>(),
		   UnthrottleTagsActorState<DB, UnthrottleTagsActor<DB>>(db, tags, throttleType, priority),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("unthrottleTags", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9343514251818156032UL, 11358661800978973184UL);
		ActorExecutionContextHelper __helper(static_cast<UnthrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("unthrottleTags");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("unthrottleTags", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UnthrottleTagsActor<DB>, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< UnthrottleTagsActor<DB>, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< UnthrottleTagsActor<DB>, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< UnthrottleTagsActor<DB>, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
[[nodiscard]] Future<bool> unthrottleTags( Reference<DB> const& db, TagSet const& tags, Optional<TagThrottleType> const& throttleType, Optional<TransactionPriority> const& priority ) {
															#line 451 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	return Future<bool>(new UnthrottleTagsActor<DB>(db, tags, throttleType, priority));
															#line 3598 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
}

#line 516 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"

															#line 3603 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
// This generated class is to be used only via throttleTags()
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB, class ThrottleTagsActor>
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class ThrottleTagsActorState {
															#line 3609 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	ThrottleTagsActorState(Reference<DB> const& db,TagSet const& tags,double const& tpsRate,double const& initialDuration,TagThrottleType const& throttleType,TransactionPriority const& priority,Optional<double> const& expirationTime = Optional<double>(),Optional<TagThrottledReason> const& reason = Optional<TagThrottledReason>()) 
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		 : db(db),
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   tags(tags),
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   tpsRate(tpsRate),
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   initialDuration(initialDuration),
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   throttleType(throttleType),
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   priority(priority),
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   expirationTime(expirationTime),
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   reason(reason),
															#line 526 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   tr(db->createTransaction()),
															#line 527 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   key(TagThrottleKey(tags, throttleType, priority).toKey())
															#line 3634 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
	{
		fdb_probe_actor_create("throttleTags", reinterpret_cast<unsigned long>(this));

	}
	~ThrottleTagsActorState() 
	{
		fdb_probe_actor_destroy("throttleTags", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 529 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			ASSERT(initialDuration > 0);
															#line 531 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (throttleType == TagThrottleType::MANUAL)
															#line 3651 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			{
															#line 532 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				reason = TagThrottledReason::MANUAL;
															#line 3655 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			}
															#line 534 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			TagThrottleValue throttle(tpsRate, expirationTime.present() ? expirationTime.get() : 0, initialDuration, reason.present() ? reason.get() : TagThrottledReason::UNSET);
															#line 538 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			BinaryWriter wr(IncludeVersion(ProtocolVersion::withTagThrottleValueReason()));
															#line 539 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			wr << throttle;
															#line 540 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			value = wr.toValue();
															#line 542 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			;
															#line 3667 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ThrottleTagsActorState();
		static_cast<ThrottleTagsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 543 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 3697 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		try {
															#line 545 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (throttleType == TagThrottleType::MANUAL)
															#line 3701 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			{
															#line 547 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				oldThrottleF = tr->get(key);
															#line 548 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				StrictFuture<Optional<Value>> __when_expr_0 = safeThreadFutureToFuture(oldThrottleF);
															#line 548 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				if (static_cast<ThrottleTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3709 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ThrottleTagsActor*>(this)->actor_wait_state = 1;
															#line 548 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ThrottleTagsActor, 0, Optional<Value> >*>(static_cast<ThrottleTagsActor*>(this)));
															#line 3714 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont2(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 563 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_3 = safeThreadFutureToFuture(tr->onError(e));
															#line 563 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<ThrottleTagsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3743 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<ThrottleTagsActor*>(this)->actor_wait_state = 4;
															#line 563 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ThrottleTagsActor, 3, Void >*>(static_cast<ThrottleTagsActor*>(this)));
															#line 3748 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 554 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		tr->set(key, value);
															#line 556 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (throttleType == TagThrottleType::MANUAL)
															#line 3765 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 557 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			signalThrottleChange(tr);
															#line 3769 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		}
															#line 560 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		StrictFuture<Void> __when_expr_2 = safeThreadFutureToFuture(tr->commit());
															#line 560 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (static_cast<ThrottleTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3775 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<ThrottleTagsActor*>(this)->actor_wait_state = 3;
															#line 560 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ThrottleTagsActor, 2, Void >*>(static_cast<ThrottleTagsActor*>(this)));
															#line 3780 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> const& oldThrottle,int loopDepth) 
	{
															#line 549 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!oldThrottle.present())
															#line 3789 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 550 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_1 = updateThrottleCount(tr, 1);
															#line 550 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<ThrottleTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3795 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
			static_cast<ThrottleTagsActor*>(this)->actor_wait_state = 2;
															#line 550 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ThrottleTagsActor, 1, Void >*>(static_cast<ThrottleTagsActor*>(this)));
															#line 3800 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(Optional<Value> && oldThrottle,int loopDepth) 
	{
															#line 549 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!oldThrottle.present())
															#line 3814 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 550 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_1 = updateThrottleCount(tr, 1);
															#line 550 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<ThrottleTagsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3820 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
			static_cast<ThrottleTagsActor*>(this)->actor_wait_state = 2;
															#line 550 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ThrottleTagsActor, 1, Void >*>(static_cast<ThrottleTagsActor*>(this)));
															#line 3825 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& oldThrottle,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(oldThrottle, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && oldThrottle,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(oldThrottle), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ThrottleTagsActor*>(this)->actor_wait_state > 0) static_cast<ThrottleTagsActor*>(this)->actor_wait_state = 0;
		static_cast<ThrottleTagsActor*>(this)->ActorCallback< ThrottleTagsActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< ThrottleTagsActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ThrottleTagsActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ThrottleTagsActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ThrottleTagsActor*>(this)->actor_wait_state > 0) static_cast<ThrottleTagsActor*>(this)->actor_wait_state = 0;
		static_cast<ThrottleTagsActor*>(this)->ActorCallback< ThrottleTagsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ThrottleTagsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ThrottleTagsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ThrottleTagsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 561 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!static_cast<ThrottleTagsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ThrottleTagsActorState(); static_cast<ThrottleTagsActor*>(this)->destroy(); return 0; }
															#line 4007 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		new (&static_cast<ThrottleTagsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ThrottleTagsActorState();
		static_cast<ThrottleTagsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 561 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!static_cast<ThrottleTagsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ThrottleTagsActorState(); static_cast<ThrottleTagsActor*>(this)->destroy(); return 0; }
															#line 4019 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		new (&static_cast<ThrottleTagsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ThrottleTagsActorState();
		static_cast<ThrottleTagsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ThrottleTagsActor*>(this)->actor_wait_state > 0) static_cast<ThrottleTagsActor*>(this)->actor_wait_state = 0;
		static_cast<ThrottleTagsActor*>(this)->ActorCallback< ThrottleTagsActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ThrottleTagsActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ThrottleTagsActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ThrottleTagsActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ThrottleTagsActor*>(this)->actor_wait_state > 0) static_cast<ThrottleTagsActor*>(this)->actor_wait_state = 0;
		static_cast<ThrottleTagsActor*>(this)->ActorCallback< ThrottleTagsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ThrottleTagsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ThrottleTagsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ThrottleTagsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<DB> db;
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	TagSet tags;
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	double tpsRate;
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	double initialDuration;
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	TagThrottleType throttleType;
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	TransactionPriority priority;
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Optional<double> expirationTime;
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Optional<TagThrottledReason> reason;
															#line 526 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<typename DB::TransactionT> tr;
															#line 527 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Key key;
															#line 540 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Value value;
															#line 547 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	typename DB::TransactionT::template FutureT<Optional<Value>> oldThrottleF;
															#line 4213 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
};
// This generated class is to be used only via throttleTags()
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class ThrottleTagsActor final : public Actor<Void>, public ActorCallback< ThrottleTagsActor<DB>, 0, Optional<Value> >, public ActorCallback< ThrottleTagsActor<DB>, 1, Void >, public ActorCallback< ThrottleTagsActor<DB>, 2, Void >, public ActorCallback< ThrottleTagsActor<DB>, 3, Void >, public FastAllocated<ThrottleTagsActor<DB>>, public ThrottleTagsActorState<DB, ThrottleTagsActor<DB>> {
															#line 4220 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
	using FastAllocated<ThrottleTagsActor<DB>>::operator new;
	using FastAllocated<ThrottleTagsActor<DB>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3595513631442186240UL, 2723868805277099776UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ThrottleTagsActor<DB>, 0, Optional<Value> >;
friend struct ActorCallback< ThrottleTagsActor<DB>, 1, Void >;
friend struct ActorCallback< ThrottleTagsActor<DB>, 2, Void >;
friend struct ActorCallback< ThrottleTagsActor<DB>, 3, Void >;
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	ThrottleTagsActor(Reference<DB> const& db,TagSet const& tags,double const& tpsRate,double const& initialDuration,TagThrottleType const& throttleType,TransactionPriority const& priority,Optional<double> const& expirationTime = Optional<double>(),Optional<TagThrottledReason> const& reason = Optional<TagThrottledReason>()) 
															#line 4240 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		 : Actor<Void>(),
		   ThrottleTagsActorState<DB, ThrottleTagsActor<DB>>(db, tags, tpsRate, initialDuration, throttleType, priority, expirationTime, reason),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("throttleTags", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3409772753682858240UL, 981046334634253568UL);
		ActorExecutionContextHelper __helper(static_cast<ThrottleTagsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("throttleTags");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("throttleTags", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ThrottleTagsActor<DB>, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ThrottleTagsActor<DB>, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ThrottleTagsActor<DB>, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ThrottleTagsActor<DB>, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
[[nodiscard]] Future<Void> throttleTags( Reference<DB> const& db, TagSet const& tags, double const& tpsRate, double const& initialDuration, TagThrottleType const& throttleType, TransactionPriority const& priority, Optional<double> const& expirationTime = Optional<double>(), Optional<TagThrottledReason> const& reason = Optional<TagThrottledReason>() ) {
															#line 517 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	return Future<Void>(new ThrottleTagsActor<DB>(db, tags, tpsRate, initialDuration, throttleType, priority, expirationTime, reason));
															#line 4277 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
}

#line 567 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"

															#line 4282 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
// This generated class is to be used only via enableAuto()
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB, class EnableAutoActor>
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class EnableAutoActorState {
															#line 4288 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	EnableAutoActorState(Reference<DB> const& db,bool const& enabled) 
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		 : db(db),
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   enabled(enabled),
															#line 570 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		   tr(db->createTransaction())
															#line 4299 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
	{
		fdb_probe_actor_create("enableAuto", reinterpret_cast<unsigned long>(this));

	}
	~EnableAutoActorState() 
	{
		fdb_probe_actor_destroy("enableAuto", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 572 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			;
															#line 4314 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~EnableAutoActorState();
		static_cast<EnableAutoActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 573 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 4344 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		try {
															#line 576 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			valueF = tr->get(tagThrottleAutoEnabledKey);
															#line 578 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Optional<Value>> __when_expr_0 = safeThreadFutureToFuture(valueF);
															#line 578 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<EnableAutoActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4352 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<EnableAutoActor*>(this)->actor_wait_state = 1;
															#line 578 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< EnableAutoActor, 0, Optional<Value> >*>(static_cast<EnableAutoActor*>(this)));
															#line 4357 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 587 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_2 = safeThreadFutureToFuture(tr->onError(e));
															#line 587 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<EnableAutoActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4381 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<EnableAutoActor*>(this)->actor_wait_state = 3;
															#line 587 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< EnableAutoActor, 2, Void >*>(static_cast<EnableAutoActor*>(this)));
															#line 4386 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& value,int loopDepth) 
	{
															#line 579 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!value.present() || (enabled && value.get() != "1"_sr) || (!enabled && value.get() != "0"_sr))
															#line 4401 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 580 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			tr->set(tagThrottleAutoEnabledKey, enabled ? "1"_sr : "0"_sr);
															#line 581 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			signalThrottleChange<typename DB::TransactionT>(tr);
															#line 583 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_1 = safeThreadFutureToFuture(tr->commit());
															#line 583 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<EnableAutoActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4411 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<EnableAutoActor*>(this)->actor_wait_state = 2;
															#line 583 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EnableAutoActor, 1, Void >*>(static_cast<EnableAutoActor*>(this)));
															#line 4416 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && value,int loopDepth) 
	{
															#line 579 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!value.present() || (enabled && value.get() != "1"_sr) || (!enabled && value.get() != "0"_sr))
															#line 4430 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		{
															#line 580 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			tr->set(tagThrottleAutoEnabledKey, enabled ? "1"_sr : "0"_sr);
															#line 581 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			signalThrottleChange<typename DB::TransactionT>(tr);
															#line 583 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			StrictFuture<Void> __when_expr_1 = safeThreadFutureToFuture(tr->commit());
															#line 583 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			if (static_cast<EnableAutoActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 4440 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
			static_cast<EnableAutoActor*>(this)->actor_wait_state = 2;
															#line 583 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< EnableAutoActor, 1, Void >*>(static_cast<EnableAutoActor*>(this)));
															#line 4445 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& value,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(value, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && value,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(value), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<EnableAutoActor*>(this)->actor_wait_state > 0) static_cast<EnableAutoActor*>(this)->actor_wait_state = 0;
		static_cast<EnableAutoActor*>(this)->ActorCallback< EnableAutoActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< EnableAutoActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("enableAuto", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableAutoActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableAuto", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< EnableAutoActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("enableAuto", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableAutoActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableAuto", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< EnableAutoActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("enableAuto", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EnableAutoActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableAuto", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 585 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
		if (!static_cast<EnableAutoActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EnableAutoActorState(); static_cast<EnableAutoActor*>(this)->destroy(); return 0; }
															#line 4534 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		new (&static_cast<EnableAutoActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~EnableAutoActorState();
		static_cast<EnableAutoActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<EnableAutoActor*>(this)->actor_wait_state > 0) static_cast<EnableAutoActor*>(this)->actor_wait_state = 0;
		static_cast<EnableAutoActor*>(this)->ActorCallback< EnableAutoActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EnableAutoActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("enableAuto", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableAutoActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableAuto", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< EnableAutoActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("enableAuto", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableAutoActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableAuto", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< EnableAutoActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("enableAuto", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EnableAutoActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableAuto", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<EnableAutoActor*>(this)->actor_wait_state > 0) static_cast<EnableAutoActor*>(this)->actor_wait_state = 0;
		static_cast<EnableAutoActor*>(this)->ActorCallback< EnableAutoActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< EnableAutoActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("enableAuto", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableAutoActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableAuto", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< EnableAutoActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("enableAuto", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EnableAutoActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableAuto", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< EnableAutoActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("enableAuto", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EnableAutoActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("enableAuto", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<DB> db;
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	bool enabled;
															#line 570 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	Reference<typename DB::TransactionT> tr;
															#line 576 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	typename DB::TransactionT::template FutureT<Optional<Value>> valueF;
															#line 4724 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
};
// This generated class is to be used only via enableAuto()
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
class EnableAutoActor final : public Actor<Void>, public ActorCallback< EnableAutoActor<DB>, 0, Optional<Value> >, public ActorCallback< EnableAutoActor<DB>, 1, Void >, public ActorCallback< EnableAutoActor<DB>, 2, Void >, public FastAllocated<EnableAutoActor<DB>>, public EnableAutoActorState<DB, EnableAutoActor<DB>> {
															#line 4731 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
public:
	using FastAllocated<EnableAutoActor<DB>>::operator new;
	using FastAllocated<EnableAutoActor<DB>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10254379008987330048UL, 16409569342284541184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< EnableAutoActor<DB>, 0, Optional<Value> >;
friend struct ActorCallback< EnableAutoActor<DB>, 1, Void >;
friend struct ActorCallback< EnableAutoActor<DB>, 2, Void >;
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	EnableAutoActor(Reference<DB> const& db,bool const& enabled) 
															#line 4750 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
		 : Actor<Void>(),
		   EnableAutoActorState<DB, EnableAutoActor<DB>>(db, enabled),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("enableAuto", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7275946659403600896UL, 8529435081498628608UL);
		ActorExecutionContextHelper __helper(static_cast<EnableAutoActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("enableAuto");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("enableAuto", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< EnableAutoActor<DB>, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< EnableAutoActor<DB>, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< EnableAutoActor<DB>, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
template <class DB>
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
[[nodiscard]] Future<Void> enableAuto( Reference<DB> const& db, bool const& enabled ) {
															#line 568 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"
	return Future<Void>(new EnableAutoActor<DB>(db, enabled));
															#line 4786 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/build_output/fdbclient/include/fdbclient/TagThrottle.actor.g.h"
}

#line 591 "/codebuild/output/src2725962233/src/github.com/apple/foundationdb/fdbclient/include/fdbclient/TagThrottle.actor.h"

class TagQuotaValue {
public:
	int64_t reservedQuota{ 0 };
	int64_t totalQuota{ 0 };
	bool isValid() const;
	Value toValue() const;
	static TagQuotaValue fromValue(ValueRef);
};

Key getTagQuotaKey(TransactionTagRef);

template <class Tr>
void setTagQuota(Reference<Tr> tr, TransactionTagRef tag, int64_t reservedQuota, int64_t totalQuota) {
	TagQuotaValue tagQuotaValue;
	tagQuotaValue.reservedQuota = reservedQuota;
	tagQuotaValue.totalQuota = totalQuota;
	if (!tagQuotaValue.isValid()) {
		throw invalid_throttle_quota_value();
	}
	tr->set(getTagQuotaKey(tag), tagQuotaValue.toValue());
	signalThrottleChange(tr);
}

}; // namespace ThrottleApi

template <class Value>
using TransactionTagMap = std::unordered_map<TransactionTag, Value, std::hash<TransactionTagRef>>;

template <class Value>
using PrioritizedTransactionTagMap = std::map<TransactionPriority, TransactionTagMap<Value>>;

template <class Value>
using UIDTransactionTagMap = std::unordered_map<UID, TransactionTagMap<Value>>;

#include "flow/unactorcompiler.h"
#endif
