#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
/*
 * ConfigureDatabase.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <algorithm>

#include "fdbclient/FDBTypes.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbserver/TesterInterface.actor.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/RunRYWTransaction.actor.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "fdbrpc/simulator.h"
#include "fdbserver/QuietDatabase.h"
#include "fdbserver/SimulatedCluster.h"
#include "flow/IRandom.h"
#include "flow/actorcompiler.h" // This must be the last #include.

static const char* storageMigrationTypes[] = { "perpetual_storage_wiggle=0 storage_migration_type=aggressive",
	                                           "perpetual_storage_wiggle=1",
	                                           "perpetual_storage_wiggle=1 storage_migration_type=gradual",
	                                           "storage_migration_type=aggressive" };
static const char* logTypes[] = { "log_engine:=1",
	                              "log_engine:=2",
	                              "log_spill:=1",
	                              "log_spill:=2",
	                              "log_version:=2",
	                              "log_version:=3",
	                              "log_version:=4",
	                              "log_version:=5",
	                              "log_version:=6",
	                              // downgrade incompatible log version
	                              "log_version:=7" };
static const char* redundancies[] = { "single", "double", "triple" };
static const char* backupTypes[] = { "backup_worker_enabled:=0", "backup_worker_enabled:=1" };

std::string generateRegions() {
	std::string result;
	if (g_simulator->physicalDatacenters == 1 ||
	    (g_simulator->physicalDatacenters == 2 && deterministicRandom()->random01() < 0.25) ||
	    g_simulator->physicalDatacenters == 3) {
		return " usable_regions=1 regions=\"\"";
	}

	if (deterministicRandom()->random01() < 0.25) {
		return format(" usable_regions=%d", deterministicRandom()->randomInt(1, 3));
	}

	int primaryPriority = 1;
	int remotePriority = -1;
	double priorityType = deterministicRandom()->random01();
	if (priorityType < 0.1) {
		primaryPriority = -1;
		remotePriority = 1;
	} else if (priorityType < 0.2) {
		remotePriority = 1;
		primaryPriority = 1;
	}

	StatusObject primaryObj;
	StatusObject primaryDcObj;
	primaryDcObj["id"] = "0";
	primaryDcObj["priority"] = primaryPriority;
	StatusArray primaryDcArr;
	primaryDcArr.push_back(primaryDcObj);

	StatusObject remoteObj;
	StatusObject remoteDcObj;
	remoteDcObj["id"] = "1";
	remoteDcObj["priority"] = remotePriority;
	StatusArray remoteDcArr;
	remoteDcArr.push_back(remoteDcObj);

	if (g_simulator->physicalDatacenters > 3 && deterministicRandom()->random01() < 0.5) {
		StatusObject primarySatelliteObj;
		primarySatelliteObj["id"] = "2";
		primarySatelliteObj["priority"] = 1;
		primarySatelliteObj["satellite"] = 1;
		if (deterministicRandom()->random01() < 0.25)
			primarySatelliteObj["satellite_logs"] = deterministicRandom()->randomInt(1, 7);
		primaryDcArr.push_back(primarySatelliteObj);

		StatusObject remoteSatelliteObj;
		remoteSatelliteObj["id"] = "3";
		remoteSatelliteObj["priority"] = 1;
		remoteSatelliteObj["satellite"] = 1;
		if (deterministicRandom()->random01() < 0.25)
			remoteSatelliteObj["satellite_logs"] = deterministicRandom()->randomInt(1, 7);
		remoteDcArr.push_back(remoteSatelliteObj);

		if (g_simulator->physicalDatacenters > 5 && deterministicRandom()->random01() < 0.5) {
			StatusObject primarySatelliteObjB;
			primarySatelliteObjB["id"] = "4";
			primarySatelliteObjB["priority"] = 1;
			primarySatelliteObjB["satellite"] = 1;
			if (deterministicRandom()->random01() < 0.25)
				primarySatelliteObjB["satellite_logs"] = deterministicRandom()->randomInt(1, 7);
			primaryDcArr.push_back(primarySatelliteObjB);

			StatusObject remoteSatelliteObjB;
			remoteSatelliteObjB["id"] = "5";
			remoteSatelliteObjB["priority"] = 1;
			remoteSatelliteObjB["satellite"] = 1;
			if (deterministicRandom()->random01() < 0.25)
				remoteSatelliteObjB["satellite_logs"] = deterministicRandom()->randomInt(1, 7);
			remoteDcArr.push_back(remoteSatelliteObjB);

			int satellite_replication_type = deterministicRandom()->randomInt(0, 3);
			switch (satellite_replication_type) {
			case 0: {
				CODE_PROBE(true, "Simulated cluster using no satellite redundancy mode");
				break;
			}
			case 1: {
				CODE_PROBE(true, "Simulated cluster using two satellite fast redundancy mode");
				primaryObj["satellite_redundancy_mode"] = "two_satellite_fast";
				remoteObj["satellite_redundancy_mode"] = "two_satellite_fast";
				break;
			}
			case 2: {
				CODE_PROBE(true, "Simulated cluster using two satellite safe redundancy mode");
				primaryObj["satellite_redundancy_mode"] = "two_satellite_safe";
				remoteObj["satellite_redundancy_mode"] = "two_satellite_safe";
				break;
			}
			default:
				ASSERT(false); // Programmer forgot to adjust cases.
			}
		} else {
			int satellite_replication_type = deterministicRandom()->randomInt(0, 4);
			switch (satellite_replication_type) {
			case 0: {
				// FIXME: implement
				CODE_PROBE(true, "Simulated cluster using custom satellite redundancy mode");
				break;
			}
			case 1: {
				CODE_PROBE(true, "Simulated cluster using no satellite redundancy mode (<5 datacenters)");
				break;
			}
			case 2: {
				CODE_PROBE(true, "Simulated cluster using single satellite redundancy mode");
				primaryObj["satellite_redundancy_mode"] = "one_satellite_single";
				remoteObj["satellite_redundancy_mode"] = "one_satellite_single";
				break;
			}
			case 3: {
				CODE_PROBE(true, "Simulated cluster using double satellite redundancy mode");
				primaryObj["satellite_redundancy_mode"] = "one_satellite_double";
				remoteObj["satellite_redundancy_mode"] = "one_satellite_double";
				break;
			}
			default:
				ASSERT(false); // Programmer forgot to adjust cases.
			}
		}

		if (deterministicRandom()->random01() < 0.25)
			primaryObj["satellite_logs"] = deterministicRandom()->randomInt(1, 7);
		if (deterministicRandom()->random01() < 0.25)
			remoteObj["satellite_logs"] = deterministicRandom()->randomInt(1, 7);

		int remote_replication_type = deterministicRandom()->randomInt(0, 4);
		switch (remote_replication_type) {
		case 0: {
			// FIXME: implement
			CODE_PROBE(true, "Simulated cluster using custom remote redundancy mode");
			break;
		}
		case 1: {
			CODE_PROBE(true, "Simulated cluster using default remote redundancy mode");
			break;
		}
		case 2: {
			CODE_PROBE(true, "Simulated cluster using single remote redundancy mode");
			result += " remote_single";
			break;
		}
		case 3: {
			CODE_PROBE(true, "Simulated cluster using double remote redundancy mode");
			result += " remote_double";
			break;
		}
		default:
			ASSERT(false); // Programmer forgot to adjust cases.
		}

		result += format(" log_routers=%d", deterministicRandom()->randomInt(1, 7));
		result += format(" remote_logs=%d", deterministicRandom()->randomInt(1, 7));
	}

	primaryObj["datacenters"] = primaryDcArr;
	remoteObj["datacenters"] = remoteDcArr;

	StatusArray regionArr;
	regionArr.push_back(primaryObj);

	if (deterministicRandom()->random01() < 0.8) {
		regionArr.push_back(remoteObj);
		if (deterministicRandom()->random01() < 0.25) {
			result += format(" usable_regions=%d", deterministicRandom()->randomInt(1, 3));
		}
	}

	result +=
	    " regions=" + json_spirit::write_string(json_spirit::mValue(regionArr), json_spirit::Output_options::none);
	return result;
}

struct ConfigureDatabaseWorkload : TestWorkload {
	static constexpr auto NAME = "ConfigureDatabase";
	double testDuration;
	int additionalDBs;
	bool allowDescriptorChange;
	bool allowTestStorageMigration; // allow change storage migration and perpetual wiggle conf
	bool storageMigrationCompatibleConf; // only allow generating configuration suitable for storage migration test
	bool waitStoreTypeCheck;
	bool downgradeTest1; // if this is true, don't pick up downgrade incompatible config
	std::vector<int> storageEngineExcludeTypes;
	std::vector<Future<Void>> clients;
	PerfIntCounter retries;

	ConfigureDatabaseWorkload(WorkloadContext const& wcx) : TestWorkload(wcx), retries("Retries") {
		testDuration = getOption(options, "testDuration"_sr, 200.0);
		allowDescriptorChange =
		    getOption(options, "allowDescriptorChange"_sr, SERVER_KNOBS->ENABLE_CROSS_CLUSTER_SUPPORT);
		allowTestStorageMigration =
		    getOption(options, "allowTestStorageMigration"_sr, false) && g_simulator->allowStorageMigrationTypeChange;
		storageMigrationCompatibleConf = getOption(options, "storageMigrationCompatibleConf"_sr, false);
		waitStoreTypeCheck = getOption(options, "waitStoreTypeCheck"_sr, false);
		downgradeTest1 = getOption(options, "downgradeTest1"_sr, false);
		storageEngineExcludeTypes = getOption(options, "storageEngineExcludeTypes"_sr);
		g_simulator->usableRegions = 1;
	}

	void disableFailureInjectionWorkloads(std::set<std::string>& out) const override { out.insert("Attrition"); }

	Future<Void> setup(Database const& cx) override { return _setup(cx, this); }

	Future<Void> start(Database const& cx) override { return _start(this, cx); }
	Future<bool> check(Database const& cx) override { return _check(this, cx); }

	void getMetrics(std::vector<PerfMetric>& m) override { m.push_back(retries.getMetric()); }

	static inline uint64_t valueToUInt64(const StringRef& v) {
		long long unsigned int x = 0;
		sscanf(v.toString().c_str(), "%llx", &x);
		return x;
	}

	static inline Standalone<StringRef> getDatabaseName(ConfigureDatabaseWorkload* self, int dbIndex) {
		return StringRef(format("DestroyDB%d", dbIndex));
	}

	static Future<ConfigurationResult> IssueConfigurationChange(Database cx, const std::string& config, bool force) {
		printf("Issuing configuration change: %s\n", config.c_str());
		return ManagementAPI::changeConfig(cx.getReference(), config, force);
	}

																#line 279 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
// This generated class is to be used only via _setup()
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
template <class _setupActor>
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
class _setupActorState {
															#line 285 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
public:
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	_setupActorState(Database const& cx,ConfigureDatabaseWorkload* const& self) 
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		 : cx(cx),
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		   self(self)
															#line 294 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
	{
		fdb_probe_actor_create("_setup", reinterpret_cast<unsigned long>(this));

	}
	~_setupActorState() 
	{
		fdb_probe_actor_destroy("_setup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 278 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success(ManagementAPI::changeConfig(cx.getReference(), "single storage_migration_type=aggressive", true));
															#line 278 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (static_cast<_setupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 311 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<_setupActor*>(this)->actor_wait_state = 1;
															#line 278 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _setupActor, 0, Void >*>(static_cast<_setupActor*>(this)));
															#line 316 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 279 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!static_cast<_setupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_setupActorState(); static_cast<_setupActor*>(this)->destroy(); return 0; }
															#line 339 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		new (&static_cast<_setupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 279 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!static_cast<_setupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_setupActorState(); static_cast<_setupActor*>(this)->destroy(); return 0; }
															#line 351 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		new (&static_cast<_setupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_setupActorState();
		static_cast<_setupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_setupActor*>(this)->actor_wait_state > 0) static_cast<_setupActor*>(this)->actor_wait_state = 0;
		static_cast<_setupActor*>(this)->ActorCallback< _setupActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _setupActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _setupActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	Database cx;
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	ConfigureDatabaseWorkload* self;
															#line 438 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
};
// This generated class is to be used only via _setup()
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
class _setupActor final : public Actor<Void>, public ActorCallback< _setupActor, 0, Void >, public FastAllocated<_setupActor>, public _setupActorState<_setupActor> {
															#line 443 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
public:
	using FastAllocated<_setupActor>::operator new;
	using FastAllocated<_setupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9566499226513179648UL, 17639382524072718848UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _setupActor, 0, Void >;
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	_setupActor(Database const& cx,ConfigureDatabaseWorkload* const& self) 
															#line 460 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		 : Actor<Void>(),
		   _setupActorState<_setupActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_setup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3044502327521131520UL, 13387266639422932480UL);
		ActorExecutionContextHelper __helper(static_cast<_setupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_setup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_setup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _setupActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
[[nodiscard]] Future<Void> _setup( Database const& cx, ConfigureDatabaseWorkload* const& self ) {
															#line 277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	return Future<Void>(new _setupActor(cx, self));
															#line 492 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
}

#line 281 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"

																#line 497 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
// This generated class is to be used only via _start()
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
template <class _startActor>
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
class _startActorState {
															#line 503 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
public:
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	_startActorState(ConfigureDatabaseWorkload* const& self,Database const& cx) 
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		 : self(self),
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		   cx(cx)
															#line 512 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
	{
		fdb_probe_actor_create("_start", reinterpret_cast<unsigned long>(this));

	}
	~_startActorState() 
	{
		fdb_probe_actor_destroy("_start", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 284 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			StrictFuture<DatabaseConfiguration> __when_expr_0 = getDatabaseConfiguration(cx);
															#line 284 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 529 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<_startActor*>(this)->actor_wait_state = 1;
															#line 284 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _startActor, 0, DatabaseConfiguration >*>(static_cast<_startActor*>(this)));
															#line 534 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_startActorState();
		static_cast<_startActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(DatabaseConfiguration const& config,int loopDepth) 
	{
															#line 285 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		TraceEvent("ConfigureDatabase_Config").detail("Config", config.toString());
															#line 286 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (config.encryptionAtRestMode.isEncryptionEnabled())
															#line 559 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 287 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			TraceEvent("ConfigureDatabase_EncryptionEnabled");
															#line 288 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			self->storageEngineExcludeTypes = { (int)SimulationStorageEngine::SSD, (int)SimulationStorageEngine::MEMORY, (int)SimulationStorageEngine::RADIX_TREE, (int)SimulationStorageEngine::ROCKSDB, (int)SimulationStorageEngine::SHARDED_ROCKSDB };
															#line 565 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		}
															#line 294 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 569 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 295 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			self->storageEngineExcludeTypes.push_back((int)SimulationStorageEngine::SHARDED_ROCKSDB);
															#line 573 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		}
															#line 297 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (self->clientId == 0)
															#line 577 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 298 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			self->clients.push_back(timeout(self->singleDB(self, cx), self->testDuration, Void()));
															#line 299 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			StrictFuture<Void> __when_expr_1 = waitForAll(self->clients);
															#line 299 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 585 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<_startActor*>(this)->actor_wait_state = 2;
															#line 299 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _startActor, 1, Void >*>(static_cast<_startActor*>(this)));
															#line 590 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(DatabaseConfiguration && config,int loopDepth) 
	{
															#line 285 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		TraceEvent("ConfigureDatabase_Config").detail("Config", config.toString());
															#line 286 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (config.encryptionAtRestMode.isEncryptionEnabled())
															#line 606 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 287 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			TraceEvent("ConfigureDatabase_EncryptionEnabled");
															#line 288 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			self->storageEngineExcludeTypes = { (int)SimulationStorageEngine::SSD, (int)SimulationStorageEngine::MEMORY, (int)SimulationStorageEngine::RADIX_TREE, (int)SimulationStorageEngine::ROCKSDB, (int)SimulationStorageEngine::SHARDED_ROCKSDB };
															#line 612 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		}
															#line 294 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA)
															#line 616 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 295 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			self->storageEngineExcludeTypes.push_back((int)SimulationStorageEngine::SHARDED_ROCKSDB);
															#line 620 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		}
															#line 297 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (self->clientId == 0)
															#line 624 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 298 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			self->clients.push_back(timeout(self->singleDB(self, cx), self->testDuration, Void()));
															#line 299 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			StrictFuture<Void> __when_expr_1 = waitForAll(self->clients);
															#line 299 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 632 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<_startActor*>(this)->actor_wait_state = 2;
															#line 299 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _startActor, 1, Void >*>(static_cast<_startActor*>(this)));
															#line 637 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(DatabaseConfiguration const& config,int loopDepth) 
	{
		loopDepth = a_body1cont1(config, loopDepth);

		return loopDepth;
	}
	int a_body1when1(DatabaseConfiguration && config,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(config), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 0, DatabaseConfiguration >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 0, DatabaseConfiguration >*,DatabaseConfiguration const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _startActor, 0, DatabaseConfiguration >*,DatabaseConfiguration && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _startActor, 0, DatabaseConfiguration >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 301 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!static_cast<_startActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_startActorState(); static_cast<_startActor*>(this)->destroy(); return 0; }
															#line 726 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		new (&static_cast<_startActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_startActorState();
		static_cast<_startActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _startActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _startActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	ConfigureDatabaseWorkload* self;
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	Database cx;
															#line 825 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
};
// This generated class is to be used only via _start()
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
class _startActor final : public Actor<Void>, public ActorCallback< _startActor, 0, DatabaseConfiguration >, public ActorCallback< _startActor, 1, Void >, public FastAllocated<_startActor>, public _startActorState<_startActor> {
															#line 830 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
public:
	using FastAllocated<_startActor>::operator new;
	using FastAllocated<_startActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7633164833241700864UL, 14374699718411579136UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _startActor, 0, DatabaseConfiguration >;
friend struct ActorCallback< _startActor, 1, Void >;
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	_startActor(ConfigureDatabaseWorkload* const& self,Database const& cx) 
															#line 848 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		 : Actor<Void>(),
		   _startActorState<_startActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16516870674810565632UL, 12235738791954228224UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_start");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _startActor, 0, DatabaseConfiguration >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _startActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
[[nodiscard]] Future<Void> _start( ConfigureDatabaseWorkload* const& self, Database const& cx ) {
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	return Future<Void>(new _startActor(self, cx));
															#line 881 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
}

#line 303 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"

	// Returns true iff aggressive migration was triggered.
	// This is needed because there are certain topologies where it's not possible for DD (perpetual wiggle) to migrate
	// storage servers to the new storage engine. As an example, if the DC has just 1 SS, DD will decide not to do
	// perpetual wiggle because there's no room. In such cases, we issue aggressive migration which can do in-place
	// migration.
																#line 891 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
// This generated class is to be used only via issueAggressiveMigrationIfNeeded()
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
template <class IssueAggressiveMigrationIfNeededActor>
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
class IssueAggressiveMigrationIfNeededActorState {
															#line 897 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
public:
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	IssueAggressiveMigrationIfNeededActorState(Database const& cx,DatabaseConfiguration const& conf,std::vector<StorageServerInterface> const& storageServers) 
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		 : cx(cx),
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		   conf(conf),
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		   storageServers(storageServers),
															#line 312 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		   dcIdToSSCount()
															#line 910 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
	{
		fdb_probe_actor_create("issueAggressiveMigrationIfNeeded", reinterpret_cast<unsigned long>(this));

	}
	~IssueAggressiveMigrationIfNeededActorState() 
	{
		fdb_probe_actor_destroy("issueAggressiveMigrationIfNeeded", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 313 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			for( const auto& ss : storageServers ) {
															#line 314 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				if (ss.locality.dcId().present())
															#line 927 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
				{
															#line 315 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					const auto& dcId = ss.locality.dcId().get().toString();
															#line 316 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					if (!dcIdToSSCount.contains(dcId))
															#line 933 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					{
															#line 317 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						dcIdToSSCount[dcId] = 0;
															#line 937 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					}
															#line 319 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					dcIdToSSCount[dcId] += 1;
															#line 941 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
				}
			}
															#line 323 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			RangeForbody1Iterator0 = std::begin(dcIdToSSCount);
															#line 946 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~IssueAggressiveMigrationIfNeededActorState();
		static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 330 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->SAV<bool>::futures) { (void)(false); this->~IssueAggressiveMigrationIfNeededActorState(); static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->destroy(); return 0; }
															#line 969 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		new (&static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->SAV< bool >::value()) bool(false);
		this->~IssueAggressiveMigrationIfNeededActorState();
		static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 323 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!(RangeForbody1Iterator0 != std::end(dcIdToSSCount)))
															#line 988 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 323 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		const auto& [_, ssCount] = *RangeForbody1Iterator0;
															#line 994 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 324 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (ssCount <= conf.storageTeamSize)
															#line 998 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			{
															#line 325 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				StrictFuture<Void> __when_expr_0 = success(IssueConfigurationChange(cx, "storage_migration_type=aggressive", false));
															#line 325 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				if (static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1004 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
				static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->actor_wait_state = 1;
															#line 325 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< IssueAggressiveMigrationIfNeededActor, 0, Void >*>(static_cast<IssueAggressiveMigrationIfNeededActor*>(this)));
															#line 1009 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont3(loopDepth);
			}
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 323 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		++RangeForbody1Iterator0;
															#line 1037 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 326 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->SAV<bool>::futures) { (void)(true); this->~IssueAggressiveMigrationIfNeededActorState(); static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->destroy(); return 0; }
															#line 1052 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		new (&static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->SAV< bool >::value()) bool(true);
		this->~IssueAggressiveMigrationIfNeededActorState();
		static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 326 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->SAV<bool>::futures) { (void)(true); this->~IssueAggressiveMigrationIfNeededActorState(); static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->destroy(); return 0; }
															#line 1064 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		new (&static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->SAV< bool >::value()) bool(true);
		this->~IssueAggressiveMigrationIfNeededActorState();
		static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->actor_wait_state > 0) static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->actor_wait_state = 0;
		static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->ActorCallback< IssueAggressiveMigrationIfNeededActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< IssueAggressiveMigrationIfNeededActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("issueAggressiveMigrationIfNeeded", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("issueAggressiveMigrationIfNeeded", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< IssueAggressiveMigrationIfNeededActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("issueAggressiveMigrationIfNeeded", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("issueAggressiveMigrationIfNeeded", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< IssueAggressiveMigrationIfNeededActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("issueAggressiveMigrationIfNeeded", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("issueAggressiveMigrationIfNeeded", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	Database cx;
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	DatabaseConfiguration conf;
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	std::vector<StorageServerInterface> storageServers;
															#line 312 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	std::unordered_map<std::string , int > dcIdToSSCount;
															#line 323 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	decltype(std::begin(std::declval<std::unordered_map<std::string , int >>())) RangeForbody1Iterator0;
															#line 1157 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
};
// This generated class is to be used only via issueAggressiveMigrationIfNeeded()
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
class IssueAggressiveMigrationIfNeededActor final : public Actor<bool>, public ActorCallback< IssueAggressiveMigrationIfNeededActor, 0, Void >, public FastAllocated<IssueAggressiveMigrationIfNeededActor>, public IssueAggressiveMigrationIfNeededActorState<IssueAggressiveMigrationIfNeededActor> {
															#line 1162 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
public:
	using FastAllocated<IssueAggressiveMigrationIfNeededActor>::operator new;
	using FastAllocated<IssueAggressiveMigrationIfNeededActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1778383941019957248UL, 15876801579777427712UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< IssueAggressiveMigrationIfNeededActor, 0, Void >;
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	IssueAggressiveMigrationIfNeededActor(Database const& cx,DatabaseConfiguration const& conf,std::vector<StorageServerInterface> const& storageServers) 
															#line 1179 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		 : Actor<bool>(),
		   IssueAggressiveMigrationIfNeededActorState<IssueAggressiveMigrationIfNeededActor>(cx, conf, storageServers),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("issueAggressiveMigrationIfNeeded", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7403175392092468224UL, 9323586042675653632UL);
		ActorExecutionContextHelper __helper(static_cast<IssueAggressiveMigrationIfNeededActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("issueAggressiveMigrationIfNeeded");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("issueAggressiveMigrationIfNeeded", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< IssueAggressiveMigrationIfNeededActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
[[nodiscard]] Future<bool> issueAggressiveMigrationIfNeeded( Database const& cx, DatabaseConfiguration const& conf, std::vector<StorageServerInterface> const& storageServers ) {
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	return Future<bool>(new IssueAggressiveMigrationIfNeededActor(cx, conf, storageServers));
															#line 1211 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
}

#line 332 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"

																#line 1216 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
// This generated class is to be used only via _check()
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
template <class _checkActor>
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
class _checkActorState {
															#line 1222 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
public:
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	_checkActorState(ConfigureDatabaseWorkload* const& self,Database const& cx) 
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		 : self(self),
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		   cx(cx)
															#line 1231 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
	{
		fdb_probe_actor_create("_check", reinterpret_cast<unsigned long>(this));

	}
	~_checkActorState() 
	{
		fdb_probe_actor_destroy("_check", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 334 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(30.0);
															#line 334 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<_checkActor*>(this)->actor_wait_state = 1;
															#line 334 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 0, Void >*>(static_cast<_checkActor*>(this)));
															#line 1253 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_checkActorState();
		static_cast<_checkActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 338 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (self->allowTestStorageMigration || self->waitStoreTypeCheck)
															#line 1276 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 339 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			aggressiveMigrationTriggered = false;
															#line 340 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			;
															#line 1282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 338 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (self->allowTestStorageMigration || self->waitStoreTypeCheck)
															#line 1296 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 339 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			aggressiveMigrationTriggered = false;
															#line 340 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			;
															#line 1302 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _checkActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _checkActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 386 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!static_cast<_checkActor*>(this)->SAV<bool>::futures) { (void)(true); this->~_checkActorState(); static_cast<_checkActor*>(this)->destroy(); return 0; }
															#line 1391 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		new (&static_cast<_checkActor*>(this)->SAV< bool >::value()) bool(true);
		this->~_checkActorState();
		static_cast<_checkActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 345 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		StrictFuture<DatabaseConfiguration> __when_expr_1 = getDatabaseConfiguration(cx);
															#line 345 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1418 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_checkActor*>(this)->actor_wait_state = 2;
															#line 345 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 1, DatabaseConfiguration >*>(static_cast<_checkActor*>(this)));
															#line 1423 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
															#line 347 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		wiggleLocalityKeyValue = conf.perpetualStorageWiggleLocality;
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		wiggleLocalityKeyValues = ParsePerpetualStorageWiggleLocality(wiggleLocalityKeyValue);
															#line 350 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		i = int();
															#line 352 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		pass = true;
															#line 353 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		StrictFuture<std::vector<StorageServerInterface>> __when_expr_2 = getStorageServers(cx);
															#line 353 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1455 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<_checkActor*>(this)->actor_wait_state = 3;
															#line 353 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 2, std::vector<StorageServerInterface> >*>(static_cast<_checkActor*>(this)));
															#line 1460 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(DatabaseConfiguration const& __conf,int loopDepth) 
	{
															#line 345 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		conf = __conf;
															#line 1469 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(DatabaseConfiguration && __conf,int loopDepth) 
	{
		conf = std::move(__conf);
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 1, DatabaseConfiguration >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 1, DatabaseConfiguration >*,DatabaseConfiguration const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _checkActor, 1, DatabaseConfiguration >*,DatabaseConfiguration && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _checkActor, 1, DatabaseConfiguration >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont2(int loopDepth) 
	{
															#line 355 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!aggressiveMigrationTriggered)
															#line 1548 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			StrictFuture<Void> __when_expr_3 = store(aggressiveMigrationTriggered, self->issueAggressiveMigrationIfNeeded(cx, conf, storageServers));
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1554 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont2when1(__when_expr_3.get(), loopDepth); };
			static_cast<_checkActor*>(this)->actor_wait_state = 4;
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 3, Void >*>(static_cast<_checkActor*>(this)));
															#line 1559 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(std::vector<StorageServerInterface> const& __storageServers,int loopDepth) 
	{
															#line 353 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		storageServers = __storageServers;
															#line 1573 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(std::vector<StorageServerInterface> && __storageServers,int loopDepth) 
	{
		storageServers = std::move(__storageServers);
		loopDepth = a_body1cont1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 2, std::vector<StorageServerInterface> >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 2, std::vector<StorageServerInterface> >*,std::vector<StorageServerInterface> const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< _checkActor, 2, std::vector<StorageServerInterface> >*,std::vector<StorageServerInterface> && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< _checkActor, 2, std::vector<StorageServerInterface> >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont1loopBody1cont3(int loopDepth) 
	{
															#line 360 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		i = 0;
															#line 1652 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		loopDepth = a_body1cont1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< _checkActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< _checkActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont1loopBody1cont5(int loopDepth) 
	{
															#line 381 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (pass)
															#line 1748 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 383 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		StrictFuture<Void> __when_expr_5 = delay(g_network->isSimulated() ? 2.0 : 30.0);
															#line 383 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1756 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont5when1(__when_expr_5.get(), loopDepth); };
		static_cast<_checkActor*>(this)->actor_wait_state = 6;
															#line 383 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 5, Void >*>(static_cast<_checkActor*>(this)));
															#line 1761 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 360 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!(i < storageServers.size()))
															#line 1777 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
			return a_body1cont1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 362 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (!storageServers[i].isTss() && (wiggleLocalityKeyValue == "0" || localityMatchInList(wiggleLocalityKeyValues, storageServers[i].locality)))
															#line 1783 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 365 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			ReplyPromise<KeyValueStoreType> typeReply;
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			StrictFuture<ErrorOr<KeyValueStoreType>> __when_expr_4 = storageServers[i].getKeyValueStoreType.getReplyUnlessFailedFor(typeReply, 2, 0);
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (static_cast<_checkActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 1791 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 2)); else return a_body1cont1loopBody1cont3loopBody1when1(__when_expr_4.get(), loopDepth); };
			static_cast<_checkActor*>(this)->actor_wait_state = 5;
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< _checkActor, 4, ErrorOr<KeyValueStoreType> >*>(static_cast<_checkActor*>(this)));
															#line 1796 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont1loopBody1cont3loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1cont1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
															#line 360 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		i++;
															#line 1823 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3loopBody1cont3(ErrorOr<KeyValueStoreType> const& keyValueStoreType,int loopDepth) 
	{
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (keyValueStoreType.present() && keyValueStoreType.get() != conf.storageServerStoreType)
															#line 1832 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 369 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			TraceEvent(SevWarn, "ConfigureDatabase_WrongStoreType") .suppressFor(5.0) .detail("ServerID", storageServers[i].id()) .detail("ProcessID", storageServers[i].locality.processId()) .detail("ServerStoreType", keyValueStoreType.present() ? keyValueStoreType.get().toString() : "?") .detail("ConfigStoreType", conf.storageServerStoreType.toString());
															#line 376 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			pass = false;
															#line 1838 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			return a_body1cont1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1cont1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3loopBody1cont3(ErrorOr<KeyValueStoreType> && keyValueStoreType,int loopDepth) 
	{
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (keyValueStoreType.present() && keyValueStoreType.get() != conf.storageServerStoreType)
															#line 1849 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 369 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			TraceEvent(SevWarn, "ConfigureDatabase_WrongStoreType") .suppressFor(5.0) .detail("ServerID", storageServers[i].id()) .detail("ProcessID", storageServers[i].locality.processId()) .detail("ServerStoreType", keyValueStoreType.present() ? keyValueStoreType.get().toString() : "?") .detail("ConfigStoreType", conf.storageServerStoreType.toString());
															#line 376 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			pass = false;
															#line 1855 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			return a_body1cont1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
		loopDepth = a_body1cont1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3loopBody1when1(ErrorOr<KeyValueStoreType> const& keyValueStoreType,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3loopBody1cont3(keyValueStoreType, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3loopBody1when1(ErrorOr<KeyValueStoreType> && keyValueStoreType,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3loopBody1cont3(std::move(keyValueStoreType), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 4, ErrorOr<KeyValueStoreType> >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 4, ErrorOr<KeyValueStoreType> >*,ErrorOr<KeyValueStoreType> const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< _checkActor, 4, ErrorOr<KeyValueStoreType> >*,ErrorOr<KeyValueStoreType> && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< _checkActor, 4, ErrorOr<KeyValueStoreType> >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont1loopBody1cont6(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont6(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<_checkActor*>(this)->actor_wait_state > 0) static_cast<_checkActor*>(this)->actor_wait_state = 0;
		static_cast<_checkActor*>(this)->ActorCallback< _checkActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _checkActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< _checkActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< _checkActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	ConfigureDatabaseWorkload* self;
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	Database cx;
															#line 339 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	bool aggressiveMigrationTriggered;
															#line 345 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	DatabaseConfiguration conf;
															#line 347 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	std::string wiggleLocalityKeyValue;
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	std::vector<std::pair<Optional<Value>, Optional<Value>>> wiggleLocalityKeyValues;
															#line 350 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	int i;
															#line 352 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	bool pass;
															#line 353 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	std::vector<StorageServerInterface> storageServers;
															#line 2042 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
};
// This generated class is to be used only via _check()
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
class _checkActor final : public Actor<bool>, public ActorCallback< _checkActor, 0, Void >, public ActorCallback< _checkActor, 1, DatabaseConfiguration >, public ActorCallback< _checkActor, 2, std::vector<StorageServerInterface> >, public ActorCallback< _checkActor, 3, Void >, public ActorCallback< _checkActor, 4, ErrorOr<KeyValueStoreType> >, public ActorCallback< _checkActor, 5, Void >, public FastAllocated<_checkActor>, public _checkActorState<_checkActor> {
															#line 2047 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
public:
	using FastAllocated<_checkActor>::operator new;
	using FastAllocated<_checkActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17431325110208957952UL, 8629446125792285440UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _checkActor, 0, Void >;
friend struct ActorCallback< _checkActor, 1, DatabaseConfiguration >;
friend struct ActorCallback< _checkActor, 2, std::vector<StorageServerInterface> >;
friend struct ActorCallback< _checkActor, 3, Void >;
friend struct ActorCallback< _checkActor, 4, ErrorOr<KeyValueStoreType> >;
friend struct ActorCallback< _checkActor, 5, Void >;
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	_checkActor(ConfigureDatabaseWorkload* const& self,Database const& cx) 
															#line 2069 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		 : Actor<bool>(),
		   _checkActorState<_checkActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_check", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18414574916690025728UL, 14115196402046968064UL);
		ActorExecutionContextHelper __helper(static_cast<_checkActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_check");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_check", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _checkActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _checkActor, 1, DatabaseConfiguration >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< _checkActor, 2, std::vector<StorageServerInterface> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< _checkActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< _checkActor, 4, ErrorOr<KeyValueStoreType> >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< _checkActor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
[[nodiscard]] Future<bool> _check( ConfigureDatabaseWorkload* const& self, Database const& cx ) {
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	return Future<bool>(new _checkActor(self, cx));
															#line 2106 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
}

#line 388 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"

	static int randomRoleNumber() {
		int i = deterministicRandom()->randomInt(0, 4);
		return i ? i : -1;
	}

																#line 2116 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
// This generated class is to be used only via singleDB()
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
template <class SingleDBActor>
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
class SingleDBActorState {
															#line 2122 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
public:
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	SingleDBActorState(ConfigureDatabaseWorkload* const& self,Database const& cx) 
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		 : self(self),
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		   cx(cx),
															#line 395 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		   tr()
															#line 2133 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
	{
		fdb_probe_actor_create("singleDB", reinterpret_cast<unsigned long>(this));

	}
	~SingleDBActorState() 
	{
		fdb_probe_actor_destroy("singleDB", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 396 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			;
															#line 2148 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SingleDBActorState();
		static_cast<SingleDBActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 397 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (g_simulator->speedUpSimulation)
															#line 2178 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 398 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (!static_cast<SingleDBActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SingleDBActorState(); static_cast<SingleDBActor*>(this)->destroy(); return 0; }
															#line 2182 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			new (&static_cast<SingleDBActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~SingleDBActorState();
			static_cast<SingleDBActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 400 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		randomChoice = int();
															#line 401 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (self->allowTestStorageMigration)
															#line 2192 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 402 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			randomChoice = (deterministicRandom()->random01() < 0.375) ? deterministicRandom()->randomInt(0, 3) : deterministicRandom()->randomInt(4, 9);
															#line 2196 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		}
		else
		{
															#line 404 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (self->storageMigrationCompatibleConf)
															#line 2202 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			{
															#line 405 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				randomChoice = (deterministicRandom()->random01() < 3.0 / 7) ? deterministicRandom()->randomInt(0, 3) : deterministicRandom()->randomInt(4, 8);
															#line 2206 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			}
			else
			{
															#line 408 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				randomChoice = deterministicRandom()->randomInt(0, 8);
															#line 2212 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			}
		}
															#line 410 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (randomChoice == 0)
															#line 2217 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 411 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			StrictFuture<Void> __when_expr_0 = success( runRYWTransaction(cx, [=](Reference<ReadYourWritesTransaction> tr) -> Future<Optional<Value>> { return tr->get("This read is only to ensure that the database recovered"_sr); }));
															#line 411 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (static_cast<SingleDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2223 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SingleDBActor*>(this)->actor_wait_state = 1;
															#line 411 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SingleDBActor, 0, Void >*>(static_cast<SingleDBActor*>(this)));
															#line 2228 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 416 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (randomChoice < 3)
															#line 2235 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			{
															#line 417 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				double waitDuration = 3.0 * deterministicRandom()->random01();
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				StrictFuture<Void> __when_expr_2 = delay(waitDuration);
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				if (static_cast<SingleDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2243 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_2.get(), loopDepth); };
				static_cast<SingleDBActor*>(this)->actor_wait_state = 3;
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SingleDBActor, 2, Void >*>(static_cast<SingleDBActor*>(this)));
															#line 2248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 420 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				if (randomChoice == 3)
															#line 2255 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
				{
															#line 422 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					int maxRedundancies = sizeof(redundancies) / sizeof(redundancies[0]);
															#line 423 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					if (g_simulator->physicalDatacenters == 2 || g_simulator->physicalDatacenters > 3)
															#line 2261 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					{
															#line 424 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						maxRedundancies--;
															#line 2265 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					}
															#line 427 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					int redundancy = deterministicRandom()->randomInt(0, maxRedundancies);
															#line 428 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					std::string config = redundancies[redundancy];
															#line 430 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					if (config == "triple" && g_simulator->physicalDatacenters == 3)
															#line 2273 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					{
															#line 431 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						config = "three_data_hall ";
															#line 2277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					}
															#line 434 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					config += generateRegions();
															#line 436 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					if (deterministicRandom()->random01() < 0.5)
															#line 2283 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					{
															#line 437 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						config += " logs=" + format("%d", randomRoleNumber());
															#line 2287 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					}
															#line 439 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					if (deterministicRandom()->random01() < 0.2)
															#line 2291 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					{
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						config += " proxies=" + format("%d", deterministicRandom()->randomInt(2, 5));
															#line 2295 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					}
					else
					{
															#line 442 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						if (deterministicRandom()->random01() < 0.5)
															#line 2301 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
						{
															#line 443 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							config += " commit_proxies=" + format("%d", randomRoleNumber());
															#line 2305 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
						}
															#line 444 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						if (deterministicRandom()->random01() < 0.5)
															#line 2309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
						{
															#line 445 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							config += " grv_proxies=" + format("%d", randomRoleNumber());
															#line 2313 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
						}
					}
															#line 447 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					if (deterministicRandom()->random01() < 0.5)
															#line 2318 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					{
															#line 448 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						config += " resolvers=" + format("%d", randomRoleNumber());
															#line 2322 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					}
															#line 450 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					StrictFuture<Void> __when_expr_3 = success(IssueConfigurationChange(cx, config, false));
															#line 450 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					if (static_cast<SingleDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2328 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when3(__when_expr_3.get(), loopDepth); };
					static_cast<SingleDBActor*>(this)->actor_wait_state = 4;
															#line 450 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< SingleDBActor, 3, Void >*>(static_cast<SingleDBActor*>(this)));
															#line 2333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
															#line 453 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					if (randomChoice == 4)
															#line 2340 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
					{
															#line 455 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						auto ch = autoQuorumChange();
															#line 456 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						std::string desiredClusterName = "NewName%d";
															#line 457 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						if (!self->allowDescriptorChange)
															#line 2348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
						{
															#line 459 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							desiredClusterName = "";
															#line 2352 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
						}
															#line 461 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						if (deterministicRandom()->randomInt(0, 2))
															#line 2356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
						{
															#line 462 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							ch = nameQuorumChange(format(desiredClusterName.c_str(), deterministicRandom()->randomInt(0, 100)), ch);
															#line 2360 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
						}
															#line 464 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						StrictFuture<Void> __when_expr_4 = success(changeQuorum(cx, ch));
															#line 464 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						if (static_cast<SingleDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
						if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when4(__when_expr_4.get(), loopDepth); };
						static_cast<SingleDBActor*>(this)->actor_wait_state = 5;
															#line 464 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< SingleDBActor, 4, Void >*>(static_cast<SingleDBActor*>(this)));
															#line 2371 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
															#line 466 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
						if (randomChoice == 5)
															#line 2378 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
						{
															#line 467 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							int storeType = 0;
															#line 468 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							for(;true;) {
															#line 469 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
								storeType = deterministicRandom()->randomInt(0, 6);
															#line 470 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
								if (std::find(self->storageEngineExcludeTypes.begin(), self->storageEngineExcludeTypes.end(), storeType) == self->storageEngineExcludeTypes.end())
															#line 2388 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
								{
									break;
								}
							}
															#line 476 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							constexpr std::array ssdTypes{ "ssd", "ssd-1", "ssd-2" };
															#line 477 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							constexpr std::array memoryTypes{ "memory", "memory-1", "memory-2" };
															#line 478 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							const char* storeTypeStr = nullptr;
															#line 479 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							switch (storeType) { case 0: storeTypeStr = ssdTypes[deterministicRandom()->randomInt(0, 3)]; break; case 1: storeTypeStr = memoryTypes[deterministicRandom()->randomInt(0, 3)]; break; case 2: storeTypeStr = "memory-radixtree"; break; case 3: storeTypeStr = "ssd-redwood-1"; break; case 4: storeTypeStr = "ssd-rocksdb-v1"; break; case 5: storeTypeStr = "ssd-sharded-rocksdb"; break; default: ASSERT(false); };
															#line 501 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							StrictFuture<Void> __when_expr_5 = success(IssueConfigurationChange(cx, storeTypeStr, true));
															#line 501 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							if (static_cast<SingleDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2405 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
							if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when5(__when_expr_5.get(), loopDepth); };
							static_cast<SingleDBActor*>(this)->actor_wait_state = 6;
															#line 501 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< SingleDBActor, 5, Void >*>(static_cast<SingleDBActor*>(this)));
															#line 2410 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
							loopDepth = 0;
						}
						else
						{
															#line 502 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
							if (randomChoice == 6)
															#line 2417 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
							{
															#line 504 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
								int length = sizeof(logTypes) / sizeof(logTypes[0]);
															#line 506 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
								if (self->downgradeTest1)
															#line 2423 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
								{
															#line 507 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
									length -= 1;
															#line 2427 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
								}
															#line 510 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
								StrictFuture<Void> __when_expr_6 = success( IssueConfigurationChange(cx, logTypes[deterministicRandom()->randomInt(0, length)], false));
															#line 510 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
								if (static_cast<SingleDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2433 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
								if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when6(__when_expr_6.get(), loopDepth); };
								static_cast<SingleDBActor*>(this)->actor_wait_state = 7;
															#line 510 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
								__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< SingleDBActor, 6, Void >*>(static_cast<SingleDBActor*>(this)));
															#line 2438 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
								loopDepth = 0;
							}
							else
							{
															#line 512 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
								if (randomChoice == 7)
															#line 2445 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
								{
															#line 513 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
									StrictFuture<Void> __when_expr_7 = success(IssueConfigurationChange( cx, backupTypes[deterministicRandom()->randomInt(0, sizeof(backupTypes) / sizeof(backupTypes[0]))], false));
															#line 513 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
									if (static_cast<SingleDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2451 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
									if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when7(__when_expr_7.get(), loopDepth); };
									static_cast<SingleDBActor*>(this)->actor_wait_state = 8;
															#line 513 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
									__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< SingleDBActor, 7, Void >*>(static_cast<SingleDBActor*>(this)));
															#line 2456 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
									loopDepth = 0;
								}
								else
								{
															#line 517 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
									if (randomChoice == 8)
															#line 2463 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
									{
															#line 518 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
										if (self->allowTestStorageMigration)
															#line 2467 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
										{
															#line 519 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
											CODE_PROBE(true, "storage migration type change");
															#line 522 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
											randomPerpetualWiggleLocality = std::string();
															#line 523 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
											if (deterministicRandom()->random01() < 0.25)
															#line 2475 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
											{
															#line 524 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
												StrictFuture<std::vector<StorageServerInterface>> __when_expr_8 = getStorageServers(cx);
															#line 524 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
												if (static_cast<SingleDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2481 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
												if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when8(__when_expr_8.get(), loopDepth); };
												static_cast<SingleDBActor*>(this)->actor_wait_state = 9;
															#line 524 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
												__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< SingleDBActor, 8, std::vector<StorageServerInterface> >*>(static_cast<SingleDBActor*>(this)));
															#line 2486 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
												loopDepth = 0;
											}
											else
											{
												loopDepth = a_body1loopBody1cont36(loopDepth);
											}
										}
										else
										{
											loopDepth = a_body1loopBody1cont35(loopDepth);
										}
									}
									else
									{
															#line 561 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
										ASSERT(false);
															#line 2503 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
										loopDepth = a_body1loopBody1cont34(loopDepth);
									}
								}
							}
						}
					}
				}
			}
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 415 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(20 + 10 * deterministicRandom()->random01());
															#line 415 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (static_cast<SingleDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2528 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_1.get(), loopDepth); };
		static_cast<SingleDBActor*>(this)->actor_wait_state = 2;
															#line 415 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SingleDBActor, 1, Void >*>(static_cast<SingleDBActor*>(this)));
															#line 2533 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 415 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(20 + 10 * deterministicRandom()->random01());
															#line 415 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (static_cast<SingleDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2544 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont7when1(__when_expr_1.get(), loopDepth); };
		static_cast<SingleDBActor*>(this)->actor_wait_state = 2;
															#line 415 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SingleDBActor, 1, Void >*>(static_cast<SingleDBActor*>(this)));
															#line 2549 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SingleDBActor*>(this)->actor_wait_state > 0) static_cast<SingleDBActor*>(this)->actor_wait_state = 0;
		static_cast<SingleDBActor*>(this)->ActorCallback< SingleDBActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SingleDBActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SingleDBActor*>(this)->actor_wait_state > 0) static_cast<SingleDBActor*>(this)->actor_wait_state = 0;
		static_cast<SingleDBActor*>(this)->ActorCallback< SingleDBActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SingleDBActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont9(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<SingleDBActor*>(this)->actor_wait_state > 0) static_cast<SingleDBActor*>(this)->actor_wait_state = 0;
		static_cast<SingleDBActor*>(this)->ActorCallback< SingleDBActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< SingleDBActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont11(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<SingleDBActor*>(this)->actor_wait_state > 0) static_cast<SingleDBActor*>(this)->actor_wait_state = 0;
		static_cast<SingleDBActor*>(this)->ActorCallback< SingleDBActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< SingleDBActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont21(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont22(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont22(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont22(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont22(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<SingleDBActor*>(this)->actor_wait_state > 0) static_cast<SingleDBActor*>(this)->actor_wait_state = 0;
		static_cast<SingleDBActor*>(this)->ActorCallback< SingleDBActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< SingleDBActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont25(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont26(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont25(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont26(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont25(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont26(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont26(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<SingleDBActor*>(this)->actor_wait_state > 0) static_cast<SingleDBActor*>(this)->actor_wait_state = 0;
		static_cast<SingleDBActor*>(this)->ActorCallback< SingleDBActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< SingleDBActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1cont29(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont25(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont30(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont29(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont30(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont29(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont30(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont30(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<SingleDBActor*>(this)->actor_wait_state > 0) static_cast<SingleDBActor*>(this)->actor_wait_state = 0;
		static_cast<SingleDBActor*>(this)->ActorCallback< SingleDBActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1when6(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< SingleDBActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1loopBody1cont32(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont29(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont33(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont32(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont33(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont32(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont33(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont33(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<SingleDBActor*>(this)->actor_wait_state > 0) static_cast<SingleDBActor*>(this)->actor_wait_state = 0;
		static_cast<SingleDBActor*>(this)->ActorCallback< SingleDBActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1when7(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< SingleDBActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1loopBody1cont34(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont32(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont35(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont34(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont36(int loopDepth) 
	{
															#line 553 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		StrictFuture<Void> __when_expr_9 = success(IssueConfigurationChange( cx, storageMigrationTypes[deterministicRandom()->randomInt( 0, sizeof(storageMigrationTypes) / sizeof(storageMigrationTypes[0]))] + randomPerpetualWiggleLocality, false));
															#line 553 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (static_cast<SingleDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3292 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont36when1(__when_expr_9.get(), loopDepth); };
		static_cast<SingleDBActor*>(this)->actor_wait_state = 10;
															#line 553 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< SingleDBActor, 9, Void >*>(static_cast<SingleDBActor*>(this)));
															#line 3297 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont37(int loopDepth) 
	{
															#line 525 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		std::string localityFilter;
															#line 526 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		int selectSSCount = deterministicRandom()->randomInt(1, std::min(4, (int)(storageServers.size() + 1)));
															#line 528 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		std::vector<StringRef> localityKeys = { LocalityData::keyDcId, LocalityData::keyDataHallId, LocalityData::keyZoneId, LocalityData::keyMachineId, LocalityData::keyProcessId };
															#line 533 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		for(int i = 0;i < selectSSCount;++i) {
															#line 534 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			StorageServerInterface randomSS = storageServers[deterministicRandom()->randomInt(0, storageServers.size())];
															#line 536 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			StringRef randomLocalityKey = localityKeys[deterministicRandom()->randomInt(0, localityKeys.size())];
															#line 538 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			if (randomSS.locality.isPresent(randomLocalityKey))
															#line 3318 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			{
															#line 539 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				if (localityFilter.size() > 0)
															#line 3322 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
				{
															#line 540 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
					localityFilter += ";";
															#line 3326 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
				}
															#line 542 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
				localityFilter += randomLocalityKey.toString() + ":" + randomSS.locality.get(randomLocalityKey).get().toString();
															#line 3330 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
			}
		}
															#line 547 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		if (localityFilter.size() > 0)
															#line 3335 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		{
															#line 548 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			TraceEvent("ConfigureTestSettingWiggleLocality").detail("LocalityFilter", localityFilter);
															#line 549 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
			randomPerpetualWiggleLocality = " perpetual_storage_wiggle_locality=" + localityFilter;
															#line 3341 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		}
		loopDepth = a_body1loopBody1cont36(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when8(std::vector<StorageServerInterface> const& __storageServers,int loopDepth) 
	{
															#line 524 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
		storageServers = __storageServers;
															#line 3351 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		loopDepth = a_body1loopBody1cont37(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when8(std::vector<StorageServerInterface> && __storageServers,int loopDepth) 
	{
		storageServers = std::move(__storageServers);
		loopDepth = a_body1loopBody1cont37(loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<SingleDBActor*>(this)->actor_wait_state > 0) static_cast<SingleDBActor*>(this)->actor_wait_state = 0;
		static_cast<SingleDBActor*>(this)->ActorCallback< SingleDBActor, 8, std::vector<StorageServerInterface> >::remove();

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 8, std::vector<StorageServerInterface> >*,std::vector<StorageServerInterface> const& value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1when8(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 8, std::vector<StorageServerInterface> >*,std::vector<StorageServerInterface> && value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1loopBody1when8(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< SingleDBActor, 8, std::vector<StorageServerInterface> >*,Error err) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1loopBody1cont36cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont35(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont36cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont35(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont36when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont36cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont36when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont36cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<SingleDBActor*>(this)->actor_wait_state > 0) static_cast<SingleDBActor*>(this)->actor_wait_state = 0;
		static_cast<SingleDBActor*>(this)->ActorCallback< SingleDBActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1cont36when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< SingleDBActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1loopBody1cont36when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< SingleDBActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), 9);

	}
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	ConfigureDatabaseWorkload* self;
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	Database cx;
															#line 395 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	Transaction tr;
															#line 400 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	int randomChoice;
															#line 522 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	std::string randomPerpetualWiggleLocality;
															#line 524 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	std::vector<StorageServerInterface> storageServers;
															#line 3525 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
};
// This generated class is to be used only via singleDB()
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
class SingleDBActor final : public Actor<Void>, public ActorCallback< SingleDBActor, 0, Void >, public ActorCallback< SingleDBActor, 1, Void >, public ActorCallback< SingleDBActor, 2, Void >, public ActorCallback< SingleDBActor, 3, Void >, public ActorCallback< SingleDBActor, 4, Void >, public ActorCallback< SingleDBActor, 5, Void >, public ActorCallback< SingleDBActor, 6, Void >, public ActorCallback< SingleDBActor, 7, Void >, public ActorCallback< SingleDBActor, 8, std::vector<StorageServerInterface> >, public ActorCallback< SingleDBActor, 9, Void >, public FastAllocated<SingleDBActor>, public SingleDBActorState<SingleDBActor> {
															#line 3530 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
public:
	using FastAllocated<SingleDBActor>::operator new;
	using FastAllocated<SingleDBActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5951728338480893696UL, 7459335304289650944UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SingleDBActor, 0, Void >;
friend struct ActorCallback< SingleDBActor, 1, Void >;
friend struct ActorCallback< SingleDBActor, 2, Void >;
friend struct ActorCallback< SingleDBActor, 3, Void >;
friend struct ActorCallback< SingleDBActor, 4, Void >;
friend struct ActorCallback< SingleDBActor, 5, Void >;
friend struct ActorCallback< SingleDBActor, 6, Void >;
friend struct ActorCallback< SingleDBActor, 7, Void >;
friend struct ActorCallback< SingleDBActor, 8, std::vector<StorageServerInterface> >;
friend struct ActorCallback< SingleDBActor, 9, Void >;
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	SingleDBActor(ConfigureDatabaseWorkload* const& self,Database const& cx) 
															#line 3556 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
		 : Actor<Void>(),
		   SingleDBActorState<SingleDBActor>(self, cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("singleDB", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(825236547584731136UL, 8116974491519717888UL);
		ActorExecutionContextHelper __helper(static_cast<SingleDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("singleDB");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("singleDB", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SingleDBActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SingleDBActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< SingleDBActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< SingleDBActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< SingleDBActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< SingleDBActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< SingleDBActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< SingleDBActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< SingleDBActor, 8, std::vector<StorageServerInterface> >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< SingleDBActor, 9, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
[[nodiscard]] Future<Void> singleDB( ConfigureDatabaseWorkload* const& self, Database const& cx ) {
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
	return Future<Void>(new SingleDBActor(self, cx));
															#line 3597 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/ConfigureDatabase.actor.g.cpp"
}

#line 565 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/workloads/ConfigureDatabase.actor.cpp"
};

WorkloadFactory<ConfigureDatabaseWorkload> DestroyDatabaseWorkloadFactory;
