#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
/*
 * GrvProxyTagThrottler.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/Knobs.h"
#include "fdbserver/GrvProxyTagThrottler.h"
#include "fdbserver/Knobs.h"
#include "flow/UnitTest.h"
#include "flow/actorcompiler.h" // must be last include

uint64_t GrvProxyTagThrottler::DelayedRequest::lastSequenceNumber = 0;

void GrvProxyTagThrottler::DelayedRequest::updateProxyTagThrottledDuration(LatencyBandsMap& latencyBandsMap) {
	req.proxyTagThrottledDuration = now() - startTime;
	auto const& [tag, count] = *req.tags.begin();
	latencyBandsMap.addMeasurement(tag, req.proxyTagThrottledDuration, count);
}

bool GrvProxyTagThrottler::DelayedRequest::isMaxThrottled(double maxThrottleDuration) const {
	return now() - startTime > maxThrottleDuration;
}

void GrvProxyTagThrottler::TagQueue::setRate(double rate) {
	if (rateInfo.present()) {
		rateInfo.get().setRate(rate);
	} else {
		rateInfo = GrvTransactionRateInfo(
		    SERVER_KNOBS->TAG_THROTTLE_RATE_WINDOW, SERVER_KNOBS->TAG_THROTTLE_MAX_EMPTY_QUEUE_BUDGET, rate);
	}
}

bool GrvProxyTagThrottler::TagQueue::isMaxThrottled(double maxThrottleDuration) const {
	return !requests.empty() && requests.front().isMaxThrottled(maxThrottleDuration);
}

void GrvProxyTagThrottler::TagQueue::rejectRequests(LatencyBandsMap& latencyBandsMap) {
	CODE_PROBE(true, "GrvProxyTagThrottler rejecting requests");
	while (!requests.empty()) {
		auto& delayedReq = requests.front();
		delayedReq.updateProxyTagThrottledDuration(latencyBandsMap);
		delayedReq.req.reply.sendError(proxy_tag_throttled());
		requests.pop_front();
	}
}

void GrvProxyTagThrottler::TagQueue::endReleaseWindow(int64_t numStarted, double elapsed) {
	if (rateInfo.present()) {
		CODE_PROBE(requests.empty(), "Tag queue ending release window with empty request queue");
		CODE_PROBE(!requests.empty(), "Tag queue ending release window with requests still queued");
		rateInfo.get().endReleaseWindow(numStarted, requests.empty(), elapsed);
	}
}

GrvProxyTagThrottler::GrvProxyTagThrottler(double maxThrottleDuration)
  : maxThrottleDuration(maxThrottleDuration),
    latencyBandsMap("GrvProxyTagThrottler",
                    deterministicRandom()->randomUniqueID(),
                    SERVER_KNOBS->GLOBAL_TAG_THROTTLING_PROXY_LOGGING_INTERVAL,
                    SERVER_KNOBS->GLOBAL_TAG_THROTTLING_MAX_TAGS_TRACKED) {}

void GrvProxyTagThrottler::updateRates(TransactionTagMap<double> const& newRates) {
	for (const auto& [tag, rate] : newRates) {
		auto it = queues.find(tag);
		if (it == queues.end()) {
			queues[tag] = TagQueue(rate);
		} else {
			it->second.setRate(rate);
		}
	}

	// Clean up tags that did not appear in newRates
	for (auto& [tag, queue] : queues) {
		if (newRates.find(tag) == newRates.end()) {
			queue.rateInfo.reset();
		}
	}

	// TODO: Use std::erase_if in C++20
	for (auto it = queues.begin(); it != queues.end();) {
		const auto& [tag, queue] = *it;
		if (queue.requests.empty() && !queue.rateInfo.present()) {
			it = queues.erase(it);
		} else {
			++it;
		}
	}
}

void GrvProxyTagThrottler::addRequest(GetReadVersionRequest const& req) {
	ASSERT(req.isTagged());
	auto const& tag = req.tags.begin()->first;
	if (req.tags.size() > 1) {
		// The GrvProxyTagThrottler assumes that each GetReadVersionRequest
		// has at most one tag. If a transaction uses multiple tags and
		// SERVER_KNOBS->ENFORCE_TAG_THROTTLING_ON_PROXIES is enabled, there may be
		// unexpected behaviour, because only one tag is used for throttling.
		TraceEvent(SevWarnAlways, "GrvProxyTagThrottler_MultipleTags")
		    .suppressFor(60.0)
		    .detail("NumTags", req.tags.size())
		    .detail("UsingTag", tag);
	}
	queues[tag].requests.emplace_back(req);
}

GrvProxyTagThrottler::ReleaseTransactionsResult GrvProxyTagThrottler::releaseTransactions(
    double elapsed,
    Deque<GetReadVersionRequest>& outBatchPriority,
    Deque<GetReadVersionRequest>& outDefaultPriority) {
	ReleaseTransactionsResult result;

	// Pointer to a TagQueue with some extra metadata stored alongside
	struct TagQueueHandle {
		// Store pointers here to avoid frequent std::unordered_map lookups
		TagQueue* queue;
		// Cannot be stored directly because we need to
		uint32_t* numReleased;
		// Sequence number of the first queued request
		int64_t nextSeqNo;
		bool operator>(TagQueueHandle const& rhs) const { return nextSeqNo > rhs.nextSeqNo; }
		explicit TagQueueHandle(TagQueue& queue, uint32_t& numReleased) : queue(&queue), numReleased(&numReleased) {
			ASSERT(!this->queue->requests.empty());
			nextSeqNo = this->queue->requests.front().sequenceNumber;
		}
	};

	// Priority queue of queues for each tag, ordered by the sequence number of the
	// next request to process in each queue
	std::priority_queue<TagQueueHandle, std::vector<TagQueueHandle>, std::greater<TagQueueHandle>> pqOfQueues;

	// Track transactions released for each tag
	std::vector<std::pair<TransactionTag, uint32_t>> transactionsReleased;
	transactionsReleased.reserve(queues.size());
	auto const transactionsReleasedInitialCapacity = transactionsReleased.capacity();

	for (auto& [tag, queue] : queues) {
		if (queue.rateInfo.present()) {
			queue.rateInfo.get().startReleaseWindow();
		}
		if (!queue.requests.empty()) {
			// First place the count in the transactionsReleased object,
			// then pass a reference to the count to the TagQueueHandle object
			// emplaced into pqOfQueues.
			//
			// Because we've reserved enough space in transactionsReleased
			// to avoid resizing, this reference should remain valid.
			// This allows each TagQueueHandle to update its number of
			// numReleased counter without incurring the cost of a std::unordered_map lookup.
			auto& [_, count] = transactionsReleased.emplace_back(tag, 0);
			pqOfQueues.emplace(queue, count);
		}
	}

	while (!pqOfQueues.empty()) {
		auto tagQueueHandle = pqOfQueues.top();
		pqOfQueues.pop();
		// Used to determine when it is time to start processing another tag
		auto const nextQueueSeqNo =
		    pqOfQueues.empty() ? std::numeric_limits<int64_t>::max() : pqOfQueues.top().nextSeqNo;

		while (!tagQueueHandle.queue->requests.empty()) {
			auto& delayedReq = tagQueueHandle.queue->requests.front();
			auto count = delayedReq.req.tags.begin()->second;
			ASSERT_EQ(tagQueueHandle.nextSeqNo, delayedReq.sequenceNumber);
			if (tagQueueHandle.queue->rateInfo.present() &&
			    !tagQueueHandle.queue->rateInfo.get().canStart(*(tagQueueHandle.numReleased), count)) {
				// Cannot release any more transaction from this tag (don't push the tag queue handle back into
				// pqOfQueues)
				CODE_PROBE(true, "GrvProxyTagThrottler throttling transaction");
				if (tagQueueHandle.queue->isMaxThrottled(maxThrottleDuration)) {
					// Requests in this queue have been throttled too long and errors
					// should be sent to clients.
					result.rejectedRequests += tagQueueHandle.queue->requests.size();
					tagQueueHandle.queue->rejectRequests(latencyBandsMap);
				}
				break;
			} else {
				if (tagQueueHandle.nextSeqNo < nextQueueSeqNo) {
					// Releasing transaction
					*(tagQueueHandle.numReleased) += count;
					delayedReq.updateProxyTagThrottledDuration(latencyBandsMap);
					if (delayedReq.req.priority == TransactionPriority::BATCH) {
						result.batchPriorityTransactionsReleased += delayedReq.req.transactionCount;
						++result.batchPriorityRequestsReleased;
						outBatchPriority.push_back(delayedReq.req);
					} else if (delayedReq.req.priority == TransactionPriority::DEFAULT) {
						result.defaultPriorityTransactionsReleased += delayedReq.req.transactionCount;
						++result.defaultPriorityRequestsReleased;
						outDefaultPriority.push_back(delayedReq.req);
					} else {
						// Immediate priority transactions should bypass the GrvProxyTagThrottler
						ASSERT(false);
					}
					tagQueueHandle.queue->requests.pop_front();
					if (!tagQueueHandle.queue->requests.empty()) {
						tagQueueHandle.nextSeqNo = tagQueueHandle.queue->requests.front().sequenceNumber;
					}
				} else {
					CODE_PROBE(true, "GrvProxyTagThrottler switching tags to preserve FIFO");
					pqOfQueues.push(tagQueueHandle);
					break;
				}
			}
		}
	}

	// End release windows for all tag queues
	{
		TransactionTagMap<uint32_t> transactionsReleasedMap;
		for (const auto& [tag, count] : transactionsReleased) {
			transactionsReleasedMap[tag] = count;
		}
		for (auto& [tag, queue] : queues) {
			queue.endReleaseWindow(transactionsReleasedMap[tag], elapsed);
		}
	}
	// If the capacity is increased, that means the vector has been illegally resized, potentially
	// corrupting memory
	ASSERT_EQ(transactionsReleased.capacity(), transactionsReleasedInitialCapacity);

	return result;
}

void GrvProxyTagThrottler::addLatencyBandThreshold(double value) {
	CODE_PROBE(size() > 0, "GrvProxyTagThrottler adding latency bands while actively throttling");
	latencyBandsMap.addThreshold(value);
}

uint32_t GrvProxyTagThrottler::size() const {
	return queues.size();
}

															#line 250 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via mockClient()
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
template <class MockClientActor>
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class MockClientActorState {
															#line 257 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	MockClientActorState(GrvProxyTagThrottler* const& throttler,TransactionPriority const& priority,TagSet const& tagSet,int const& batchSize,double const& desiredRate,TransactionTagMap<uint32_t>* const& counters) 
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		 : throttler(throttler),
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   priority(priority),
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   tagSet(tagSet),
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   batchSize(batchSize),
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   desiredRate(desiredRate),
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   counters(counters),
															#line 254 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   timer(),
															#line 255 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   tags()
															#line 278 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("mockClient", reinterpret_cast<unsigned long>(this));

	}
	~MockClientActorState() 
	{
		fdb_probe_actor_destroy("mockClient", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 256 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			for( const auto& tag : tagSet ) {
															#line 257 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				tags[tag] = batchSize;
															#line 295 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 259 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			;
															#line 299 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MockClientActorState();
		static_cast<MockClientActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 260 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		timer = delayJittered(static_cast<double>(batchSize) / desiredRate);
															#line 261 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		GetReadVersionRequest req;
															#line 262 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		req.tags = tags;
															#line 263 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		req.priority = priority;
															#line 264 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		throttler->addRequest(req);
															#line 265 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_0 = success(req.reply.getFuture()) && timer;
															#line 265 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (static_cast<MockClientActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 341 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MockClientActor*>(this)->actor_wait_state = 1;
															#line 265 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MockClientActor, 0, Void >*>(static_cast<MockClientActor*>(this)));
															#line 346 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 266 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		for( auto& [tag, _] : tags ) {
															#line 267 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			(*counters)[tag] += batchSize;
															#line 357 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 266 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		for( auto& [tag, _] : tags ) {
															#line 267 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			(*counters)[tag] += batchSize;
															#line 369 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MockClientActor*>(this)->actor_wait_state > 0) static_cast<MockClientActor*>(this)->actor_wait_state = 0;
		static_cast<MockClientActor*>(this)->ActorCallback< MockClientActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MockClientActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("mockClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MockClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MockClientActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("mockClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MockClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MockClientActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("mockClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MockClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockClient", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	GrvProxyTagThrottler* throttler;
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TransactionPriority priority;
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TagSet tagSet;
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	int batchSize;
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	double desiredRate;
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TransactionTagMap<uint32_t>* counters;
															#line 254 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> timer;
															#line 255 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TransactionTagMap<uint32_t> tags;
															#line 466 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via mockClient()
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class MockClientActor final : public Actor<Void>, public ActorCallback< MockClientActor, 0, Void >, public FastAllocated<MockClientActor>, public MockClientActorState<MockClientActor> {
															#line 471 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
	using FastAllocated<MockClientActor>::operator new;
	using FastAllocated<MockClientActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17335761199826300160UL, 7162893792595630848UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MockClientActor, 0, Void >;
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	MockClientActor(GrvProxyTagThrottler* const& throttler,TransactionPriority const& priority,TagSet const& tagSet,int const& batchSize,double const& desiredRate,TransactionTagMap<uint32_t>* const& counters) 
															#line 488 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   MockClientActorState<MockClientActor>(throttler, priority, tagSet, batchSize, desiredRate, counters),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("mockClient", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7285660657920109568UL, 11812854688363359744UL);
		ActorExecutionContextHelper __helper(static_cast<MockClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("mockClient");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("mockClient", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MockClientActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
[[nodiscard]] static Future<Void> mockClient( GrvProxyTagThrottler* const& throttler, TransactionPriority const& priority, TagSet const& tagSet, int const& batchSize, double const& desiredRate, TransactionTagMap<uint32_t>* const& counters ) {
															#line 248 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	return Future<Void>(new MockClientActor(throttler, priority, tagSet, batchSize, desiredRate, counters));
															#line 521 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
}

#line 271 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"

															#line 526 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via mockFifoClient()
															#line 272 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
template <class MockFifoClientActor>
															#line 272 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class MockFifoClientActorState {
															#line 533 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
															#line 272 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	MockFifoClientActorState(GrvProxyTagThrottler* const& throttler) 
															#line 272 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
															#line 272 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		 : throttler(throttler),
															#line 273 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   tagSet1(),
															#line 274 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   tagSet2(),
															#line 275 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   reqs(),
															#line 276 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   i(0),
															#line 278 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   replyIndices()
															#line 550 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("mockFifoClient", reinterpret_cast<unsigned long>(this));

	}
	~MockFifoClientActorState() 
	{
		fdb_probe_actor_destroy("mockFifoClient", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 281 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			tagSet1["sampleTag1"_sr] = 1;
															#line 282 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			tagSet2["sampleTag2"_sr] = 1;
															#line 283 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			for(i = 0;i < 2000;++i) {
															#line 284 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				auto& req = reqs.emplace_back();
															#line 285 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				req.priority = TransactionPriority::DEFAULT;
															#line 286 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				if (i < 1000)
															#line 575 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
				{
															#line 287 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
					req.tags = tagSet1;
															#line 579 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
				}
				else
				{
															#line 289 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
					req.tags = tagSet2;
															#line 585 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
				}
			}
															#line 292 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			deterministicRandom()->randomShuffle(reqs);
															#line 295 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			for( const auto& req : reqs ) {
															#line 296 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				throttler->addRequest(req);
															#line 594 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 298 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			futures = std::vector<Future<Void>>();
															#line 299 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			for(int j = 0;j < 2000;++j) {
															#line 301 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				auto* _replyIndices = &replyIndices;
															#line 302 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				futures.push_back(map(reqs[j].reply.getFuture(), [_replyIndices, j](auto const&) { (*_replyIndices).push_back(j); return Void(); }));
															#line 604 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 307 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = waitForAll(futures);
															#line 307 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			if (static_cast<MockFifoClientActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 610 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<MockFifoClientActor*>(this)->actor_wait_state = 1;
															#line 307 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MockFifoClientActor, 0, Void >*>(static_cast<MockFifoClientActor*>(this)));
															#line 615 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MockFifoClientActorState();
		static_cast<MockFifoClientActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 308 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		for(i = 0;i < 2000;++i) {
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			ASSERT_EQ(replyIndices[i], i);
															#line 640 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		}
															#line 311 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<MockFifoClientActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MockFifoClientActorState(); static_cast<MockFifoClientActor*>(this)->destroy(); return 0; }
															#line 644 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<MockFifoClientActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MockFifoClientActorState();
		static_cast<MockFifoClientActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 308 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		for(i = 0;i < 2000;++i) {
															#line 309 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			ASSERT_EQ(replyIndices[i], i);
															#line 658 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		}
															#line 311 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<MockFifoClientActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~MockFifoClientActorState(); static_cast<MockFifoClientActor*>(this)->destroy(); return 0; }
															#line 662 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<MockFifoClientActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~MockFifoClientActorState();
		static_cast<MockFifoClientActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MockFifoClientActor*>(this)->actor_wait_state > 0) static_cast<MockFifoClientActor*>(this)->actor_wait_state = 0;
		static_cast<MockFifoClientActor*>(this)->ActorCallback< MockFifoClientActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MockFifoClientActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("mockFifoClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MockFifoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockFifoClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MockFifoClientActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("mockFifoClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MockFifoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockFifoClient", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MockFifoClientActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("mockFifoClient", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MockFifoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockFifoClient", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 272 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	GrvProxyTagThrottler* throttler;
															#line 273 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TransactionTagMap<uint32_t> tagSet1;
															#line 274 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TransactionTagMap<uint32_t> tagSet2;
															#line 275 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	std::vector<GetReadVersionRequest> reqs;
															#line 276 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	int i;
															#line 278 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	std::vector<int> replyIndices;
															#line 298 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	std::vector<Future<Void>> futures;
															#line 759 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via mockFifoClient()
															#line 272 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class MockFifoClientActor final : public Actor<Void>, public ActorCallback< MockFifoClientActor, 0, Void >, public FastAllocated<MockFifoClientActor>, public MockFifoClientActorState<MockFifoClientActor> {
															#line 764 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
	using FastAllocated<MockFifoClientActor>::operator new;
	using FastAllocated<MockFifoClientActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6320899870532315648UL, 6607129348202809856UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MockFifoClientActor, 0, Void >;
															#line 272 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	MockFifoClientActor(GrvProxyTagThrottler* const& throttler) 
															#line 781 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   MockFifoClientActorState<MockFifoClientActor>(throttler),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("mockFifoClient", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5935461698524279296UL, 11989561225166146048UL);
		ActorExecutionContextHelper __helper(static_cast<MockFifoClientActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("mockFifoClient");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("mockFifoClient", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MockFifoClientActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 272 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
[[nodiscard]] static Future<Void> mockFifoClient( GrvProxyTagThrottler* const& throttler ) {
															#line 272 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	return Future<Void>(new MockFifoClientActor(throttler));
															#line 814 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
}

#line 313 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"

															#line 819 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via mockServer()
															#line 314 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
template <class MockServerActor>
															#line 314 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class MockServerActorState {
															#line 826 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
															#line 314 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	MockServerActorState(GrvProxyTagThrottler* const& throttler) 
															#line 314 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
															#line 314 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		 : throttler(throttler),
															#line 315 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   outBatchPriority(),
															#line 316 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   outDefaultPriority()
															#line 837 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("mockServer", reinterpret_cast<unsigned long>(this));

	}
	~MockServerActorState() 
	{
		fdb_probe_actor_destroy("mockServer", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 317 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			;
															#line 852 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MockServerActorState();
		static_cast<MockServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 318 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		elapsed = (0.009 + 0.002 * deterministicRandom()->random01());
															#line 319 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(elapsed);
															#line 319 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (static_cast<MockServerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 886 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<MockServerActor*>(this)->actor_wait_state = 1;
															#line 319 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MockServerActor, 0, Void >*>(static_cast<MockServerActor*>(this)));
															#line 891 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 320 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		throttler->releaseTransactions(elapsed, outBatchPriority, outDefaultPriority);
															#line 321 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		for(;!outBatchPriority.empty();) {
															#line 322 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			outBatchPriority.front().reply.send(GetReadVersionReply{});
															#line 323 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			outBatchPriority.pop_front();
															#line 906 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		}
															#line 325 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		for(;!outDefaultPriority.empty();) {
															#line 326 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			outDefaultPriority.front().reply.send(GetReadVersionReply{});
															#line 327 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			outDefaultPriority.pop_front();
															#line 914 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 320 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		throttler->releaseTransactions(elapsed, outBatchPriority, outDefaultPriority);
															#line 321 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		for(;!outBatchPriority.empty();) {
															#line 322 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			outBatchPriority.front().reply.send(GetReadVersionReply{});
															#line 323 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			outBatchPriority.pop_front();
															#line 930 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		}
															#line 325 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		for(;!outDefaultPriority.empty();) {
															#line 326 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			outDefaultPriority.front().reply.send(GetReadVersionReply{});
															#line 327 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			outDefaultPriority.pop_front();
															#line 938 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MockServerActor*>(this)->actor_wait_state > 0) static_cast<MockServerActor*>(this)->actor_wait_state = 0;
		static_cast<MockServerActor*>(this)->ActorCallback< MockServerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MockServerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("mockServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MockServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MockServerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("mockServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MockServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MockServerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("mockServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MockServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("mockServer", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 314 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	GrvProxyTagThrottler* throttler;
															#line 315 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Deque<GetReadVersionRequest> outBatchPriority;
															#line 316 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Deque<GetReadVersionRequest> outDefaultPriority;
															#line 318 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	double elapsed;
															#line 1027 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via mockServer()
															#line 314 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class MockServerActor final : public Actor<Void>, public ActorCallback< MockServerActor, 0, Void >, public FastAllocated<MockServerActor>, public MockServerActorState<MockServerActor> {
															#line 1032 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
	using FastAllocated<MockServerActor>::operator new;
	using FastAllocated<MockServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(109107351690568192UL, 14026672888469399552UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MockServerActor, 0, Void >;
															#line 314 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	MockServerActor(GrvProxyTagThrottler* const& throttler) 
															#line 1049 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   MockServerActorState<MockServerActor>(throttler),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("mockServer", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1192864816535556352UL, 7390335565917548800UL);
		ActorExecutionContextHelper __helper(static_cast<MockServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("mockServer");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("mockServer", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MockServerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 314 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
[[nodiscard]] static Future<Void> mockServer( GrvProxyTagThrottler* const& throttler ) {
															#line 314 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	return Future<Void>(new MockServerActor(throttler));
															#line 1082 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
}

#line 331 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"

static TransactionTag getRandomTag() {
	TransactionTag result;
	auto arr = new (result.arena()) uint8_t[32];
	for (int i = 0; i < 32; ++i) {
		arr[i] = (uint8_t)deterministicRandom()->randomInt(0, 256);
	}
	result.contents() = TransactionTagRef(arr, 32);
	return result;
}

static bool isNear(double desired, int64_t actual) {
	return std::abs(desired - actual) * 10 < desired;
}

// Rate limit set at 10, but client attempts 20 transactions per second.
// Client should be throttled to only 10 transactions per second.
															#line 1103 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase348()
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
template <class FlowTestCase348Actor>
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase348ActorState {
															#line 1110 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase348ActorState(UnitTestParameters const& params) 
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		 : params(params),
															#line 349 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   throttler(5.0),
															#line 350 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   tagSet(),
															#line 351 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   counters()
															#line 1123 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase348", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase348ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase348", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			{
															#line 353 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				TransactionTagMap<double> rates;
															#line 354 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				rates["sampleTag"_sr] = 10.0;
															#line 355 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				throttler.updateRates(rates);
															#line 1143 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 357 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			tagSet.addTag("sampleTag"_sr);
															#line 359 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			client = mockClient(&throttler, TransactionPriority::DEFAULT, tagSet, 1, 20.0, &counters);
															#line 360 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			server = mockServer(&throttler);
															#line 361 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeout(client && server, 60.0, Void());
															#line 361 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase348Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1155 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase348Actor*>(this)->actor_wait_state = 1;
															#line 361 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase348Actor, 0, Void >*>(static_cast<FlowTestCase348Actor*>(this)));
															#line 1160 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase348ActorState();
		static_cast<FlowTestCase348Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 362 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		TraceEvent("TagQuotaTest_Simple").detail("Counter", counters["sampleTag"_sr]);
															#line 363 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag"_sr], 60.0 * 10.0));
															#line 364 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase348Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase348ActorState(); static_cast<FlowTestCase348Actor*>(this)->destroy(); return 0; }
															#line 1187 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase348Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase348ActorState();
		static_cast<FlowTestCase348Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 362 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		TraceEvent("TagQuotaTest_Simple").detail("Counter", counters["sampleTag"_sr]);
															#line 363 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag"_sr], 60.0 * 10.0));
															#line 364 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase348Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase348ActorState(); static_cast<FlowTestCase348Actor*>(this)->destroy(); return 0; }
															#line 1203 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase348Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase348ActorState();
		static_cast<FlowTestCase348Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase348Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase348Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase348Actor*>(this)->ActorCallback< FlowTestCase348Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase348Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase348", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase348Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase348", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase348Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase348", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase348Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase348", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase348Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase348", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase348Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase348", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 349 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	GrvProxyTagThrottler throttler;
															#line 350 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TagSet tagSet;
															#line 351 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TransactionTagMap<uint32_t> counters;
															#line 359 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> client;
															#line 360 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> server;
															#line 1298 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase348()
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase348Actor final : public Actor<Void>, public ActorCallback< FlowTestCase348Actor, 0, Void >, public FastAllocated<FlowTestCase348Actor>, public FlowTestCase348ActorState<FlowTestCase348Actor> {
															#line 1303 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase348Actor>::operator new;
	using FastAllocated<FlowTestCase348Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14291431913078134784UL, 14088370400618966784UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase348Actor, 0, Void >;
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase348Actor(UnitTestParameters const& params) 
															#line 1320 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase348ActorState<FlowTestCase348Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase348", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11723605348918280192UL, 7053697639081114112UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase348Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase348");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase348", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase348Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
static Future<Void> flowTestCase348( UnitTestParameters const& params ) {
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase348Actor(params));
															#line 1353 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase348, "/GrvProxyTagThrottler/Simple")

#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"

// Clients share the available 30 transaction/second budget
															#line 1360 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase368()
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
template <class FlowTestCase368Actor>
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase368ActorState {
															#line 1367 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase368ActorState(UnitTestParameters const& params) 
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		 : params(params),
															#line 369 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   throttler(5.0),
															#line 370 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   tagSet(),
															#line 371 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   counters()
															#line 1380 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase368", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase368ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase368", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			{
															#line 373 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				TransactionTagMap<double> rates;
															#line 374 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				rates["sampleTag"_sr] = 30.0;
															#line 375 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				throttler.updateRates(rates);
															#line 1400 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 377 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			tagSet.addTag("sampleTag"_sr);
															#line 379 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			clients = std::vector<Future<Void>>();
															#line 380 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			clients.reserve(10);
															#line 381 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			for(int i = 0;i < 10;++i) {
															#line 382 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				clients.push_back(mockClient(&throttler, TransactionPriority::DEFAULT, tagSet, 1, 10.0, &counters));
															#line 1412 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 385 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			server = mockServer(&throttler);
															#line 386 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeout(waitForAll(clients) && server, 60.0, Void());
															#line 386 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase368Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1420 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase368Actor*>(this)->actor_wait_state = 1;
															#line 386 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase368Actor, 0, Void >*>(static_cast<FlowTestCase368Actor*>(this)));
															#line 1425 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase368ActorState();
		static_cast<FlowTestCase368Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 387 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		TraceEvent("TagQuotaTest_MultiClient").detail("Counter", counters["sampleTag"_sr]);
															#line 388 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag"_sr], 60.0 * 30.0));
															#line 389 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase368Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase368ActorState(); static_cast<FlowTestCase368Actor*>(this)->destroy(); return 0; }
															#line 1452 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase368Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase368ActorState();
		static_cast<FlowTestCase368Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 387 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		TraceEvent("TagQuotaTest_MultiClient").detail("Counter", counters["sampleTag"_sr]);
															#line 388 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag"_sr], 60.0 * 30.0));
															#line 389 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase368Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase368ActorState(); static_cast<FlowTestCase368Actor*>(this)->destroy(); return 0; }
															#line 1468 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase368Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase368ActorState();
		static_cast<FlowTestCase368Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase368Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase368Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase368Actor*>(this)->ActorCallback< FlowTestCase368Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase368Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase368", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase368Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase368", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase368Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase368", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase368Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase368", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase368Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase368", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase368Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase368", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 369 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	GrvProxyTagThrottler throttler;
															#line 370 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TagSet tagSet;
															#line 371 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TransactionTagMap<uint32_t> counters;
															#line 379 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	std::vector<Future<Void>> clients;
															#line 385 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> server;
															#line 1563 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase368()
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase368Actor final : public Actor<Void>, public ActorCallback< FlowTestCase368Actor, 0, Void >, public FastAllocated<FlowTestCase368Actor>, public FlowTestCase368ActorState<FlowTestCase368Actor> {
															#line 1568 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase368Actor>::operator new;
	using FastAllocated<FlowTestCase368Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2827248898029391360UL, 18314075824528499456UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase368Actor, 0, Void >;
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase368Actor(UnitTestParameters const& params) 
															#line 1585 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase368ActorState<FlowTestCase368Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase368", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11456213915922423040UL, 17085289642107127040UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase368Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase368");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase368", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase368Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
static Future<Void> flowTestCase368( UnitTestParameters const& params ) {
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase368Actor(params));
															#line 1618 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase368, "/GrvProxyTagThrottler/MultiClient")

#line 391 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"

// Each tag receives 10 transaction/second budget
															#line 1625 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase393()
															#line 393 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
template <class FlowTestCase393Actor>
															#line 393 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase393ActorState {
															#line 1632 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
															#line 393 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase393ActorState(UnitTestParameters const& params) 
															#line 393 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
															#line 393 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		 : params(params),
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   throttler(5.0),
															#line 395 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   tagSet1(),
															#line 396 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   tagSet2(),
															#line 397 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   counters()
															#line 1647 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase393", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase393ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase393", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			{
															#line 399 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				TransactionTagMap<double> rates;
															#line 400 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				rates["sampleTag1"_sr] = 10.0;
															#line 401 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				rates["sampleTag2"_sr] = 10.0;
															#line 402 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				throttler.updateRates(rates);
															#line 1669 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 404 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			tagSet1.addTag("sampleTag1"_sr);
															#line 405 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			tagSet2.addTag("sampleTag2"_sr);
															#line 406 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			client1 = mockClient(&throttler, TransactionPriority::DEFAULT, tagSet1, 5, 20.0, &counters);
															#line 407 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			client2 = mockClient(&throttler, TransactionPriority::DEFAULT, tagSet2, 5, 20.0, &counters);
															#line 408 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			server = mockServer(&throttler);
															#line 409 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeout(client1 && client2 && server, 60.0, Void());
															#line 409 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase393Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1685 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase393Actor*>(this)->actor_wait_state = 1;
															#line 409 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase393Actor, 0, Void >*>(static_cast<FlowTestCase393Actor*>(this)));
															#line 1690 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase393ActorState();
		static_cast<FlowTestCase393Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 410 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		TraceEvent("TagQuotaTest_MultiTag") .detail("Counter1", counters["sampleTag1"_sr]) .detail("Counter2", counters["sampleTag2"_sr]);
															#line 413 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag1"_sr], 60 * 10.0));
															#line 414 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag2"_sr], 60 * 10.0));
															#line 415 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase393Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase393ActorState(); static_cast<FlowTestCase393Actor*>(this)->destroy(); return 0; }
															#line 1719 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase393Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase393ActorState();
		static_cast<FlowTestCase393Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 410 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		TraceEvent("TagQuotaTest_MultiTag") .detail("Counter1", counters["sampleTag1"_sr]) .detail("Counter2", counters["sampleTag2"_sr]);
															#line 413 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag1"_sr], 60 * 10.0));
															#line 414 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag2"_sr], 60 * 10.0));
															#line 415 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase393Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase393ActorState(); static_cast<FlowTestCase393Actor*>(this)->destroy(); return 0; }
															#line 1737 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase393Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase393ActorState();
		static_cast<FlowTestCase393Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase393Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase393Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase393Actor*>(this)->ActorCallback< FlowTestCase393Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase393Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase393", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase393Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase393", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase393Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase393", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase393Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase393", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase393Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase393", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase393Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase393", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 393 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 394 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	GrvProxyTagThrottler throttler;
															#line 395 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TagSet tagSet1;
															#line 396 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TagSet tagSet2;
															#line 397 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TransactionTagMap<uint32_t> counters;
															#line 406 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> client1;
															#line 407 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> client2;
															#line 408 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> server;
															#line 1836 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase393()
															#line 393 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase393Actor final : public Actor<Void>, public ActorCallback< FlowTestCase393Actor, 0, Void >, public FastAllocated<FlowTestCase393Actor>, public FlowTestCase393ActorState<FlowTestCase393Actor> {
															#line 1841 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase393Actor>::operator new;
	using FastAllocated<FlowTestCase393Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3535008399269950976UL, 3076158892572429824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase393Actor, 0, Void >;
															#line 393 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase393Actor(UnitTestParameters const& params) 
															#line 1858 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase393ActorState<FlowTestCase393Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase393", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9849616840405276416UL, 13405162071094611456UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase393Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase393");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase393", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase393Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 393 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
static Future<Void> flowTestCase393( UnitTestParameters const& params ) {
															#line 393 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase393Actor(params));
															#line 1891 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase393, "/GrvProxyTagThrottler/MultiTag")

#line 417 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"

// Test processing GetReadVersionRequests that batch several transactions
															#line 1898 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase419()
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
template <class FlowTestCase419Actor>
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase419ActorState {
															#line 1905 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase419ActorState(UnitTestParameters const& params) 
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		 : params(params),
															#line 420 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   throttler(5.0),
															#line 421 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   tagSet(),
															#line 422 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   counters()
															#line 1918 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase419", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase419ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase419", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			{
															#line 424 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				TransactionTagMap<double> rates;
															#line 425 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				rates["sampleTag"_sr] = 10.0;
															#line 426 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				throttler.updateRates(rates);
															#line 1938 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 428 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			tagSet.addTag("sampleTag"_sr);
															#line 430 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			client = mockClient(&throttler, TransactionPriority::DEFAULT, tagSet, 5, 20.0, &counters);
															#line 431 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			server = mockServer(&throttler);
															#line 432 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = timeout(client && server, 60.0, Void());
															#line 432 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase419Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1950 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase419Actor*>(this)->actor_wait_state = 1;
															#line 432 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase419Actor, 0, Void >*>(static_cast<FlowTestCase419Actor*>(this)));
															#line 1955 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase419ActorState();
		static_cast<FlowTestCase419Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 434 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		TraceEvent("TagQuotaTest_Batch").detail("Counter", counters["sampleTag"_sr]);
															#line 435 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag"_sr], 60.0 * 10.0));
															#line 436 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase419Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase419ActorState(); static_cast<FlowTestCase419Actor*>(this)->destroy(); return 0; }
															#line 1982 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase419Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase419ActorState();
		static_cast<FlowTestCase419Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 434 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		TraceEvent("TagQuotaTest_Batch").detail("Counter", counters["sampleTag"_sr]);
															#line 435 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag"_sr], 60.0 * 10.0));
															#line 436 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase419Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase419ActorState(); static_cast<FlowTestCase419Actor*>(this)->destroy(); return 0; }
															#line 1998 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase419Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase419ActorState();
		static_cast<FlowTestCase419Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase419Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase419Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase419Actor*>(this)->ActorCallback< FlowTestCase419Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase419Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase419", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase419Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase419", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase419Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase419", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase419Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase419", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase419Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase419", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase419Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase419", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 420 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	GrvProxyTagThrottler throttler;
															#line 421 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TagSet tagSet;
															#line 422 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TransactionTagMap<uint32_t> counters;
															#line 430 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> client;
															#line 431 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> server;
															#line 2093 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase419()
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase419Actor final : public Actor<Void>, public ActorCallback< FlowTestCase419Actor, 0, Void >, public FastAllocated<FlowTestCase419Actor>, public FlowTestCase419ActorState<FlowTestCase419Actor> {
															#line 2098 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase419Actor>::operator new;
	using FastAllocated<FlowTestCase419Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18292139613704359680UL, 17288435554756573184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase419Actor, 0, Void >;
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase419Actor(UnitTestParameters const& params) 
															#line 2115 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase419ActorState<FlowTestCase419Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase419", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15593287614925827840UL, 14717929419283747584UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase419Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase419");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase419", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase419Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
static Future<Void> flowTestCase419( UnitTestParameters const& params ) {
															#line 419 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase419Actor(params));
															#line 2148 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase419, "/GrvProxyTagThrottler/Batch")

#line 438 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"

// Tests cleanup of tags that are no longer throttled.
															#line 2155 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase440()
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
template <class FlowTestCase440Actor>
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase440ActorState {
															#line 2162 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase440ActorState(UnitTestParameters const& params) 
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		 : params(params)
															#line 2169 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase440", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase440ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase440", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 441 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			GrvProxyTagThrottler throttler(5.0);
															#line 442 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			for(int i = 0;i < 1000;++i) {
															#line 443 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				auto const tag = getRandomTag();
															#line 444 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				TransactionTagMap<double> rates;
															#line 445 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				rates[tag] = 10.0;
															#line 446 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				throttler.updateRates(rates);
															#line 447 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				ASSERT_EQ(throttler.size(), 1);
															#line 2196 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 449 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			if (!static_cast<FlowTestCase440Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase440ActorState(); static_cast<FlowTestCase440Actor*>(this)->destroy(); return 0; }
															#line 2200 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			new (&static_cast<FlowTestCase440Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase440ActorState();
			static_cast<FlowTestCase440Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase440ActorState();
		static_cast<FlowTestCase440Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 2224 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase440()
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase440Actor final : public Actor<Void>, public FastAllocated<FlowTestCase440Actor>, public FlowTestCase440ActorState<FlowTestCase440Actor> {
															#line 2229 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase440Actor>::operator new;
	using FastAllocated<FlowTestCase440Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6117883827678506752UL, 18187308698860867328UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase440Actor(UnitTestParameters const& params) 
															#line 2245 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase440ActorState<FlowTestCase440Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase440", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1762963680670260224UL, 1016828828881312256UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase440Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase440");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase440", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
static Future<Void> flowTestCase440( UnitTestParameters const& params ) {
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase440Actor(params));
															#line 2277 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase440, "/GrvProxyTagThrottler/Cleanup1")

#line 451 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"

// Tests cleanup of tags once queues have been emptied
															#line 2284 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase453()
															#line 453 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
template <class FlowTestCase453Actor>
															#line 453 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase453ActorState {
															#line 2291 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
															#line 453 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase453ActorState(UnitTestParameters const& params) 
															#line 453 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
															#line 453 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		 : params(params)
															#line 2298 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase453", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase453ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase453", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 454 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			GrvProxyTagThrottler throttler(5.0);
															#line 2313 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			{
															#line 456 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				GetReadVersionRequest req;
															#line 457 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				req.tags["sampleTag"_sr] = 1;
															#line 458 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				req.priority = TransactionPriority::DEFAULT;
															#line 459 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				throttler.addRequest(req);
															#line 2323 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 461 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			ASSERT_EQ(throttler.size(), 1);
															#line 462 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			throttler.updateRates(TransactionTagMap<double>{});
															#line 463 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			ASSERT_EQ(throttler.size(), 1);
															#line 2331 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			{
															#line 465 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				Deque<GetReadVersionRequest> outBatchPriority;
															#line 466 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				Deque<GetReadVersionRequest> outDefaultPriority;
															#line 467 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				throttler.releaseTransactions(0.1, outBatchPriority, outDefaultPriority);
															#line 2339 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 470 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			throttler.updateRates(TransactionTagMap<double>{});
															#line 471 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			ASSERT_EQ(throttler.size(), 0);
															#line 472 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			if (!static_cast<FlowTestCase453Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase453ActorState(); static_cast<FlowTestCase453Actor*>(this)->destroy(); return 0; }
															#line 2347 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			new (&static_cast<FlowTestCase453Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase453ActorState();
			static_cast<FlowTestCase453Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase453ActorState();
		static_cast<FlowTestCase453Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 453 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 2371 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase453()
															#line 453 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase453Actor final : public Actor<Void>, public FastAllocated<FlowTestCase453Actor>, public FlowTestCase453ActorState<FlowTestCase453Actor> {
															#line 2376 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase453Actor>::operator new;
	using FastAllocated<FlowTestCase453Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16026792584562062080UL, 4754782423741345792UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 453 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase453Actor(UnitTestParameters const& params) 
															#line 2392 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase453ActorState<FlowTestCase453Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase453", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3780861377891986944UL, 16838074452003899392UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase453Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase453");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase453", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 453 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
static Future<Void> flowTestCase453( UnitTestParameters const& params ) {
															#line 453 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase453Actor(params));
															#line 2424 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase453, "/GrvProxyTagThrottler/Cleanup2")

#line 474 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"

// Tests that unthrottled transactions are released in FIFO order, even when they
// have different tags
															#line 2432 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase477()
															#line 477 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
template <class FlowTestCase477Actor>
															#line 477 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase477ActorState {
															#line 2439 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
															#line 477 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase477ActorState(UnitTestParameters const& params) 
															#line 477 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
															#line 477 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		 : params(params),
															#line 478 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   throttler(5.0),
															#line 479 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   server(mockServer(&throttler))
															#line 2450 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase477", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase477ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase477", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 480 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = mockFifoClient(&throttler);
															#line 480 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase477Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2467 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase477Actor*>(this)->actor_wait_state = 1;
															#line 480 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase477Actor, 0, Void >*>(static_cast<FlowTestCase477Actor*>(this)));
															#line 2472 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase477ActorState();
		static_cast<FlowTestCase477Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 481 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase477Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase477ActorState(); static_cast<FlowTestCase477Actor*>(this)->destroy(); return 0; }
															#line 2495 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase477Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase477ActorState();
		static_cast<FlowTestCase477Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 481 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase477Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase477ActorState(); static_cast<FlowTestCase477Actor*>(this)->destroy(); return 0; }
															#line 2507 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase477Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase477ActorState();
		static_cast<FlowTestCase477Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase477Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase477Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase477Actor*>(this)->ActorCallback< FlowTestCase477Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase477Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase477", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase477Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase477", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase477Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase477", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase477Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase477", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase477Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase477", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase477Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase477", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 477 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 478 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	GrvProxyTagThrottler throttler;
															#line 479 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> server;
															#line 2596 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase477()
															#line 477 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase477Actor final : public Actor<Void>, public ActorCallback< FlowTestCase477Actor, 0, Void >, public FastAllocated<FlowTestCase477Actor>, public FlowTestCase477ActorState<FlowTestCase477Actor> {
															#line 2601 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase477Actor>::operator new;
	using FastAllocated<FlowTestCase477Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12464845504010339072UL, 17120688708721514496UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase477Actor, 0, Void >;
															#line 477 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase477Actor(UnitTestParameters const& params) 
															#line 2618 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase477ActorState<FlowTestCase477Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase477", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11245903412891513088UL, 17135232202511675904UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase477Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase477");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase477", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase477Actor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 477 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
static Future<Void> flowTestCase477( UnitTestParameters const& params ) {
															#line 477 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase477Actor(params));
															#line 2651 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase477, "/GrvProxyTagThrottler/Fifo")

#line 483 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"

// Tests that while throughput is low, the tag throttler
// does not accumulate too much budget.
//
// A server is setup to server 100 transactions per second,
// then runs idly for 60 seconds. Then a client starts
// and attempts 200 transactions per second for 60 seconds.
// The server throttles the client to only achieve
// 10 transactions per second during this 60 second window.
// If the throttler is allowed to accumulate budget indefinitely
// during the idle 60 seconds, this test will fail.
															#line 2667 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase494()
															#line 494 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
template <class FlowTestCase494Actor>
															#line 494 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase494ActorState {
															#line 2674 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
															#line 494 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase494ActorState(UnitTestParameters const& params) 
															#line 494 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
															#line 494 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		 : params(params),
															#line 495 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   throttler(5.0),
															#line 496 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   tagSet(),
															#line 497 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		   counters()
															#line 2687 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase494", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase494ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase494", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			{
															#line 499 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				TransactionTagMap<double> rates;
															#line 500 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				rates["sampleTag"_sr] = 100.0;
															#line 501 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
				throttler.updateRates(rates);
															#line 2707 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			}
															#line 503 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			tagSet.addTag("sampleTag"_sr);
															#line 505 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			server = mockServer(&throttler);
															#line 506 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(60.0);
															#line 506 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			if (static_cast<FlowTestCase494Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2717 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase494Actor*>(this)->actor_wait_state = 1;
															#line 506 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase494Actor, 0, Void >*>(static_cast<FlowTestCase494Actor*>(this)));
															#line 2722 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase494ActorState();
		static_cast<FlowTestCase494Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 507 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		client = mockClient(&throttler, TransactionPriority::DEFAULT, tagSet, 1, 200.0, &counters);
															#line 508 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_1 = timeout(client && server, 60.0, Void());
															#line 508 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (static_cast<FlowTestCase494Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2749 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase494Actor*>(this)->actor_wait_state = 2;
															#line 508 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase494Actor, 1, Void >*>(static_cast<FlowTestCase494Actor*>(this)));
															#line 2754 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 507 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		client = mockClient(&throttler, TransactionPriority::DEFAULT, tagSet, 1, 200.0, &counters);
															#line 508 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		StrictFuture<Void> __when_expr_1 = timeout(client && server, 60.0, Void());
															#line 508 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (static_cast<FlowTestCase494Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2767 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase494Actor*>(this)->actor_wait_state = 2;
															#line 508 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase494Actor, 1, Void >*>(static_cast<FlowTestCase494Actor*>(this)));
															#line 2772 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase494Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase494Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase494Actor*>(this)->ActorCallback< FlowTestCase494Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase494Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase494", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase494Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase494", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase494Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase494", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase494Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase494", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase494Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase494", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase494Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase494", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 509 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		TraceEvent("TagQuotaTest_LimitedIdleBudget").detail("Counter", counters["sampleTag"_sr]);
															#line 510 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag"_sr], 60.0 * 100.0));
															#line 511 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase494Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase494ActorState(); static_cast<FlowTestCase494Actor*>(this)->destroy(); return 0; }
															#line 2860 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase494Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase494ActorState();
		static_cast<FlowTestCase494Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 509 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		TraceEvent("TagQuotaTest_LimitedIdleBudget").detail("Counter", counters["sampleTag"_sr]);
															#line 510 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		ASSERT(isNear(counters["sampleTag"_sr], 60.0 * 100.0));
															#line 511 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
		if (!static_cast<FlowTestCase494Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase494ActorState(); static_cast<FlowTestCase494Actor*>(this)->destroy(); return 0; }
															#line 2876 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		new (&static_cast<FlowTestCase494Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase494ActorState();
		static_cast<FlowTestCase494Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase494Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase494Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase494Actor*>(this)->ActorCallback< FlowTestCase494Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase494Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase494", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase494Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase494", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase494Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase494", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase494Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase494", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase494Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase494", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase494Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase494", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 494 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	UnitTestParameters params;
															#line 495 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	GrvProxyTagThrottler throttler;
															#line 496 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TagSet tagSet;
															#line 497 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	TransactionTagMap<uint32_t> counters;
															#line 505 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> server;
															#line 507 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	Future<Void> client;
															#line 2971 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase494()
															#line 494 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
class FlowTestCase494Actor final : public Actor<Void>, public ActorCallback< FlowTestCase494Actor, 0, Void >, public ActorCallback< FlowTestCase494Actor, 1, Void >, public FastAllocated<FlowTestCase494Actor>, public FlowTestCase494ActorState<FlowTestCase494Actor> {
															#line 2976 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase494Actor>::operator new;
	using FastAllocated<FlowTestCase494Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12940705323584145664UL, 7297971143746227968UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase494Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase494Actor, 1, Void >;
															#line 494 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	FlowTestCase494Actor(UnitTestParameters const& params) 
															#line 2994 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase494ActorState<FlowTestCase494Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase494", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10341393952872072192UL, 18086017399662167552UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase494Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase494");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase494", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase494Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase494Actor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 494 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
static Future<Void> flowTestCase494( UnitTestParameters const& params ) {
															#line 494 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
	return Future<Void>(new FlowTestCase494Actor(params));
															#line 3028 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbserver/GrvProxyTagThrottler.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase494, "/GrvProxyTagThrottler/LimitedIdleBudget")

#line 513 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbserver/GrvProxyTagThrottler.actor.cpp"
