#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
/*
 * S3Client_cli.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/ClientKnobCollection.h"
#include <algorithm>
#include <cctype>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <limits>
#include <memory>
#include <string>
#include <vector>
#include <fcntl.h>
#include <filesystem>

#ifdef _WIN32
#include <io.h>
#endif

#include <boost/algorithm/hex.hpp>
#include "fdbclient/BuildFlags.h"
#include "fdbclient/BackupContainerFileSystem.h"
#import "fdbclient/BackupTLSConfig.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/IKnobCollection.h"
#include "fdbclient/Knobs.h"
#include "fdbclient/versions.h"
#include "fdbclient/S3Client.actor.h"
#include "fdbclient/BackupAgent.actor.h"
#include "flow/Platform.h"
#include "flow/ArgParseUtil.h"
#include "flow/FastRef.h"
#include "flow/Trace.h"
#include "flow/flow.h"
#include "flow/TLSConfig.actor.h"
#include "SimpleOpt/SimpleOpt.h"

#include "flow/actorcompiler.h" // has to be last include

// CLI for S3Client.

extern const char* getSourceVersion();

namespace s3client_cli {

enum {
	OPT_BLOB_CREDENTIALS,
	OPT_TRACE,
	OPT_TRACE_DIR,
	OPT_TRACE_FORMAT,
	OPT_TRACE_LOG_GROUP,
	OPT_BUILD_FLAGS,
	OPT_KNOB,
	OPT_PROXY,
	OPT_HELP,
	OPT_LS_RECURSIVE
};

CSimpleOpt::SOption Options[] = { { OPT_TRACE, "--log", SO_NONE },
	                              { OPT_TRACE, "--logs", SO_NONE },
	                              { OPT_TRACE, "-l", SO_NONE },
	                              { OPT_TRACE_DIR, "--logdir", SO_REQ_SEP },
	                              { OPT_TRACE_FORMAT, "--trace-format", SO_REQ_SEP },
	                              { OPT_TRACE_LOG_GROUP, "--loggroup", SO_REQ_SEP },
	                              { OPT_BLOB_CREDENTIALS, "--blob-credentials", SO_REQ_SEP },
	                              TLS_OPTION_FLAGS,
	                              { OPT_BUILD_FLAGS, "--build-flags", SO_NONE },
	                              { OPT_KNOB, "--knob-", SO_REQ_SEP },
	                              { OPT_PROXY, "--proxy", SO_REQ_SEP },
	                              { OPT_HELP, "-h", SO_NONE },
	                              { OPT_HELP, "--help", SO_NONE },
	                              { OPT_LS_RECURSIVE, "--recursive", SO_NONE },
	                              SO_END_OF_OPTIONS };

static void printUsage(std::string const& programName) {
	std::cout << "Usage: " << programName
	          << " [OPTIONS] COMMAND SOURCE [TARGET]\n"
	             "Run basic s3 operations from the command-line (using the S3BlobStore engine).\n"
	             "Use https://docs.aws.amazon.com/cli/latest/userguide/getting-started-install.html\n"
	             "if you need finesse -- listing resources, etc.\n"
	             "COMMAND:\n"
	             "  cp             Copy SOURCE to TARGET. SOURCE is file, directory, or s3/blobstore\n"
	             "                 'Backup URL' to copy from. If SOURCE is a Backup URL,\n"
	             "                 TARGET must be a local directory and vice versa. See 'Backup URLs'\n"
	             "                 in https://apple.github.io/foundationdb/backups.html for\n"
	             "                 more on the fdb s3 'blobstore://' URL format.\n"
	             "  ls             List contents of SOURCE. Must be a s3/blobstore 'Backup URL'.\n"
	             "  rm             Delete SOURCE. Must be a s3/blobstore 'Backup URL'.\n"
	             "OPTIONS:\n"
	             "  --recursive    Recursively list contents of SOURCE. Only valid with 'ls' command.\n"
	             "  --log          Enables trace file logging for the CLI session.\n"
	             "  --logdir PATH  Specifies the output directory for trace files. If\n"
	             "                 unspecified, defaults to the current directory. Has\n"
	             "                 no effect unless --log is specified.\n"
	             "  --loggroup     LOG_GROUP\n"
	             "                 Sets the LogGroup field with the specified value for all\n"
	             "                 events in the trace output (defaults to `default').\n"
	             "  --trace-format FORMAT\n"
	             "                 Select the format of the trace files, xml (the default) or json.\n"
	             "                 Has no effect unless --log is specified.\n"
	             "  --blob-credentials FILE\n"
	             "                 File containing blob credentials in JSON format.\n"
	             "                 The same credential format/file fdbbackup uses.\n"
	             "                 See 'Blob Credential Files' in https://apple.github.io/foundationdb/backups.html.\n"
	             "  --proxy HOST:PORT\n"
	             "                 Connect to S3 through proxy at given host:port.\n"
	             "  --build-flags  Print build information and exit.\n"
	             "  --knob-KNOBNAME KNOBVALUE\n"
	             "                 Changes a knob value. KNOBNAME should be lowercase.\n"
	             "EXAMPLES:\n"
	             " "
	          << programName
	          << " --knob_http_verbose_level=10 --tls-ca-file /etc/ssl/cert.pem \\\n"
	             " --blob-credentials /tmp/s3.6GWo/blob_credentials.json --log --logdir /tmp/s3.6GWo/logs cp \\\n"
	             " 'blobstore://@backup-us-west-2.s3.amazonaws.com/dir/x.txt?bucket=backup-us-west-2&region=us-west-2' "
	             "/tmp/x.txt\n";
	return;
}

static void printBuildInformation() {
	printf("%s", jsonBuildInformation().c_str());
}

struct Params : public ReferenceCounted<Params> {
	Optional<std::string> proxy;
	bool log_enabled = false;
	std::string log_dir, trace_format, trace_log_group;
	BackupTLSConfig tlsConfig;
	std::vector<std::pair<std::string, std::string>> knobs;
	std::string src;
	std::string tgt;
	std::string command;
	int whichIsBlobstoreURL = -1;
	const std::string blobstore_enable_object_integrity_check = "blobstore_enable_object_integrity_check";
	bool ls_recursive = false;

	std::string toString() {
		std::string s;
		if (proxy.present()) {
			s.append(", Proxy: ");
			s.append(proxy.get());
		}
		if (log_enabled) {
			if (!log_dir.empty()) {
				s.append(" LogDir:").append(log_dir);
			}
			if (!trace_format.empty()) {
				s.append(" Format:").append(trace_format);
			}
			if (!trace_log_group.empty()) {
				s.append(" LogGroup:").append(trace_log_group);
			}
		}
		for (const auto& [knob, value] : knobs) {
			s.append(", KNOB-").append(knob).append(" = ").append(value);
		}
		s.append(", Source: ").append(src);
		s.append(", Target: ").append(tgt);
		return s;
	}

	void updateKnobs() {
		// Set default to 'true' for blobstore_enable_object_integrity_check if not explicitly set
		bool blobstore_enable_object_integrity_check_set = false;
		for (const std::pair p : knobs) {
			if (p.first == blobstore_enable_object_integrity_check) {
				blobstore_enable_object_integrity_check_set = true;
				break;
			}
		}
		if (!blobstore_enable_object_integrity_check_set) {
			knobs.push_back(std::pair(blobstore_enable_object_integrity_check, "true"));
		}
		IKnobCollection::setupKnobs(knobs);

		// Reinitialize knobs in order to update knobs that are dependent on explicitly set knobs
		IKnobCollection::getMutableGlobalKnobCollection().initialize(Randomize::False, IsSimulated::False);
	}
};

static int isBlobStoreURL(const std::string& url) {
	return url.starts_with(BLOBSTORE_PREFIX);
}

static int parseCommandLine(Reference<Params> param, CSimpleOpt* args) {
	while (args->Next()) {
		auto lastError = args->LastError();
		switch (lastError) {
		case SO_SUCCESS:
			break;

		default:
			return FDB_EXIT_ERROR;
			break;
		}
		int optId = args->OptionId();
		switch (optId) {
		case OPT_HELP:
			return FDB_EXIT_ERROR;
		case OPT_TRACE:
			param->log_enabled = true;
			break;
		case OPT_TRACE_DIR:
			param->log_dir = args->OptionArg();
			break;
		case OPT_TRACE_FORMAT:
			if (!selectTraceFormatter(args->OptionArg())) {
				std::cerr << "ERROR: Unrecognized trace format " << args->OptionArg() << "\n";
				return FDB_EXIT_ERROR;
			}
			param->trace_format = args->OptionArg();
			break;
		case OPT_TRACE_LOG_GROUP:
			param->trace_log_group = args->OptionArg();
			break;
		case OPT_BLOB_CREDENTIALS:
			param->tlsConfig.blobCredentials.push_back(args->OptionArg());
			break;
		case OPT_KNOB: {
			Optional<std::string> knobName = extractPrefixedArgument("--knob", args->OptionSyntax());
			if (!knobName.present()) {
				std::cerr << "ERROR: unable to parse knob option '" << args->OptionSyntax() << "'\n";
				return FDB_EXIT_ERROR;
			}
			param->knobs.emplace_back(knobName.get(), args->OptionArg());
			break;
		}
		case OPT_LS_RECURSIVE:
			param->ls_recursive = true;
			break;
		case TLSConfig::OPT_TLS_PLUGIN:
			args->OptionArg();
			break;
		case TLSConfig::OPT_TLS_CERTIFICATES:
			param->tlsConfig.tlsCertPath = args->OptionArg();
			break;
		case TLSConfig::OPT_TLS_PASSWORD:
			param->tlsConfig.tlsPassword = args->OptionArg();
			break;
		case TLSConfig::OPT_TLS_CA_FILE:
			param->tlsConfig.tlsCAPath = args->OptionArg();
			break;
		case TLSConfig::OPT_TLS_KEY:
			param->tlsConfig.tlsKeyPath = args->OptionArg();
			break;
		case TLSConfig::OPT_TLS_VERIFY_PEERS:
			param->tlsConfig.tlsVerifyPeers = args->OptionArg();
			break;
		case OPT_BUILD_FLAGS:
			printBuildInformation();
			return FDB_EXIT_ERROR;
			break;
		case OPT_PROXY:
			param->proxy = args->OptionArg();
			break;
		}
	}
	if (args->FileCount() < 1) {
		std::cerr << "ERROR: Not enough arguments; need a COMMAND" << std::endl;
		return FDB_EXIT_ERROR;
	}
	std::string command = args->Files()[0];
	// Command are modelled on 'https://docs.aws.amazon.com/cli/latest/reference/s3/'.
	param->command = command;
	if (command == "cp") {
		if (args->FileCount() != 3) {
			std::cerr << "ERROR: cp command requires a SOURCE and a TARGET" << std::endl;
			return FDB_EXIT_ERROR;
		}
		param->src = args->Files()[1];
		param->tgt = args->Files()[2];
		param->whichIsBlobstoreURL = isBlobStoreURL(param->src) ? 0 : isBlobStoreURL(param->tgt) ? 1 : -1;
		if (param->whichIsBlobstoreURL < 0) {
			std::cerr << "ERROR: Either SOURCE or TARGET needs to be a blobstore URL "
			          << "(e.g. blobstore://myKey:mySecret@something.domain.com:80/dec_1_2017_0400?bucket=backups)"
			          << std::endl;
			return FDB_EXIT_ERROR;
		}
	} else if (command == "rm") {
		if (args->FileCount() != 2) {
			std::cerr << "ERROR: rm command requires a SOURCE" << std::endl;
			return FDB_EXIT_ERROR;
		}
		param->src = args->Files()[1];
		if (!isBlobStoreURL(param->src)) {
			std::cerr << "ERROR: SOURCE must be a blobstore URL for rm command" << std::endl;
			return FDB_EXIT_ERROR;
		}
		param->whichIsBlobstoreURL = 0;
		param->tgt = "";
	} else if (command == "ls") {
		if (args->FileCount() != 2) {
			std::cerr << "ERROR: ls command requires a SOURCE" << std::endl;
			return FDB_EXIT_ERROR;
		}
		param->src = args->Files()[1];
		if (!isBlobStoreURL(param->src)) {
			std::cerr << "ERROR: SOURCE must be a blobstore URL for ls command" << std::endl;
			return FDB_EXIT_ERROR;
		}
		param->whichIsBlobstoreURL = 0;
		param->tgt = "";
	} else {
		std::cerr << "ERROR: Invalid command: " << command << std::endl;
		return FDB_EXIT_ERROR;
	}
	return FDB_EXIT_SUCCESS;
}

// Method called by main. Figures which of copy_up or copy_down to call.
															#line 331 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
// This generated class is to be used only via run()
															#line 329 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
template <class RunActor>
															#line 329 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
class RunActorState {
															#line 337 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
public:
															#line 329 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	RunActorState(Reference<Params> const& params) 
															#line 329 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
															#line 329 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
		 : params(params)
															#line 344 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
	{
		fdb_probe_actor_create("run", reinterpret_cast<unsigned long>(this));

	}
	~RunActorState() 
	{
		fdb_probe_actor_destroy("run", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 330 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
			if (params->command == "cp")
															#line 359 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
			{
															#line 331 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
				if (params->whichIsBlobstoreURL == 1)
															#line 363 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
				{
															#line 332 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
					if (std::filesystem::is_directory(params->src))
															#line 367 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
					{
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						StrictFuture<Void> __when_expr_0 = copyUpDirectory(params->src, params->tgt);
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						if (static_cast<RunActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 373 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
						if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
						static_cast<RunActor*>(this)->actor_wait_state = 1;
															#line 333 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RunActor, 0, Void >*>(static_cast<RunActor*>(this)));
															#line 378 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
															#line 335 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						StrictFuture<Void> __when_expr_1 = copyUpFile(params->src, params->tgt);
															#line 335 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						if (static_cast<RunActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 387 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
						if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
						static_cast<RunActor*>(this)->actor_wait_state = 2;
															#line 335 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RunActor, 1, Void >*>(static_cast<RunActor*>(this)));
															#line 392 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
						loopDepth = 0;
					}
				}
				else
				{
															#line 338 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
					if (std::filesystem::is_directory(params->tgt))
															#line 400 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
					{
															#line 339 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						StrictFuture<Void> __when_expr_2 = copyDownDirectory(params->src, params->tgt);
															#line 339 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						if (static_cast<RunActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 406 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
						if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1when3(__when_expr_2.get(), loopDepth); };
						static_cast<RunActor*>(this)->actor_wait_state = 3;
															#line 339 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RunActor, 2, Void >*>(static_cast<RunActor*>(this)));
															#line 411 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
						loopDepth = 0;
					}
					else
					{
															#line 341 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						StrictFuture<Void> __when_expr_3 = copyDownFile(params->src, params->tgt);
															#line 341 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						if (static_cast<RunActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 420 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
						if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1when4(__when_expr_3.get(), loopDepth); };
						static_cast<RunActor*>(this)->actor_wait_state = 4;
															#line 341 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< RunActor, 3, Void >*>(static_cast<RunActor*>(this)));
															#line 425 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
						loopDepth = 0;
					}
				}
			}
			else
			{
															#line 344 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
				if (params->command == "rm")
															#line 434 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
				{
															#line 345 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
					StrictFuture<Void> __when_expr_4 = deleteResource(params->src);
															#line 345 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
					if (static_cast<RunActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 440 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
					if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1when5(__when_expr_4.get(), loopDepth); };
					static_cast<RunActor*>(this)->actor_wait_state = 5;
															#line 345 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
					__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< RunActor, 4, Void >*>(static_cast<RunActor*>(this)));
															#line 445 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
															#line 346 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
					if (params->command == "ls")
															#line 452 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
					{
															#line 347 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
						if (params->ls_recursive)
															#line 456 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
						{
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
							StrictFuture<Void> __when_expr_5 = listFiles(params->src, std::numeric_limits<int>::max());
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
							if (static_cast<RunActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 462 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
							if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1when6(__when_expr_5.get(), loopDepth); };
							static_cast<RunActor*>(this)->actor_wait_state = 6;
															#line 348 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
							__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< RunActor, 5, Void >*>(static_cast<RunActor*>(this)));
															#line 467 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
							loopDepth = 0;
						}
						else
						{
															#line 350 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
							StrictFuture<Void> __when_expr_6 = listFiles(params->src, 1);
															#line 350 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
							if (static_cast<RunActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 476 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
							if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1when7(__when_expr_6.get(), loopDepth); };
							static_cast<RunActor*>(this)->actor_wait_state = 7;
															#line 350 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
							__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< RunActor, 6, Void >*>(static_cast<RunActor*>(this)));
															#line 481 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
							loopDepth = 0;
						}
					}
					else
					{
						loopDepth = a_body1cont11(loopDepth);
					}
				}
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunActorState();
		static_cast<RunActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 353 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
		if (!static_cast<RunActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RunActorState(); static_cast<RunActor*>(this)->destroy(); return 0; }
															#line 512 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
		new (&static_cast<RunActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RunActorState();
		static_cast<RunActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunActor*>(this)->actor_wait_state > 0) static_cast<RunActor*>(this)->actor_wait_state = 0;
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RunActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RunActor*>(this)->actor_wait_state > 0) static_cast<RunActor*>(this)->actor_wait_state = 0;
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RunActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RunActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RunActor*>(this)->actor_wait_state > 0) static_cast<RunActor*>(this)->actor_wait_state = 0;
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RunActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RunActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1when4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1when4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<RunActor*>(this)->actor_wait_state > 0) static_cast<RunActor*>(this)->actor_wait_state = 0;
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< RunActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< RunActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1when5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1when5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<RunActor*>(this)->actor_wait_state > 0) static_cast<RunActor*>(this)->actor_wait_state = 0;
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< RunActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< RunActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont11(int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont12(int loopDepth) 
	{
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1when6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13(_, loopDepth);

		return loopDepth;
	}
	int a_body1when6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<RunActor*>(this)->actor_wait_state > 0) static_cast<RunActor*>(this)->actor_wait_state = 0;
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1when6(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< RunActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< RunActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont14(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont14(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1when7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14(_, loopDepth);

		return loopDepth;
	}
	int a_body1when7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<RunActor*>(this)->actor_wait_state > 0) static_cast<RunActor*>(this)->actor_wait_state = 0;
		static_cast<RunActor*>(this)->ActorCallback< RunActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1when7(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< RunActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< RunActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 329 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	Reference<Params> params;
															#line 1167 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
};
// This generated class is to be used only via run()
															#line 329 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
class RunActor final : public Actor<Void>, public ActorCallback< RunActor, 0, Void >, public ActorCallback< RunActor, 1, Void >, public ActorCallback< RunActor, 2, Void >, public ActorCallback< RunActor, 3, Void >, public ActorCallback< RunActor, 4, Void >, public ActorCallback< RunActor, 5, Void >, public ActorCallback< RunActor, 6, Void >, public FastAllocated<RunActor>, public RunActorState<RunActor> {
															#line 1172 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
public:
	using FastAllocated<RunActor>::operator new;
	using FastAllocated<RunActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5674512423698670336UL, 336909793350222848UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RunActor, 0, Void >;
friend struct ActorCallback< RunActor, 1, Void >;
friend struct ActorCallback< RunActor, 2, Void >;
friend struct ActorCallback< RunActor, 3, Void >;
friend struct ActorCallback< RunActor, 4, Void >;
friend struct ActorCallback< RunActor, 5, Void >;
friend struct ActorCallback< RunActor, 6, Void >;
															#line 329 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	RunActor(Reference<Params> const& params) 
															#line 1195 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
		 : Actor<Void>(),
		   RunActorState<RunActor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("run", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(633236942609902848UL, 16810589115111353856UL);
		ActorExecutionContextHelper __helper(static_cast<RunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("run");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("run", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RunActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RunActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RunActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< RunActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< RunActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< RunActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< RunActor, 6, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 329 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
[[nodiscard]] Future<Void> run( Reference<Params> const& params ) {
															#line 329 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	return Future<Void>(new RunActor(params));
															#line 1233 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
}

#line 355 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"

															#line 1238 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
// This generated class is to be used only via deleteResource()
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
template <class DeleteResourceActor>
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
class DeleteResourceActorState {
															#line 1244 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
public:
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	DeleteResourceActorState(std::string const& src) 
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
		 : src(src)
															#line 1251 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
	{
		fdb_probe_actor_create("deleteResource", reinterpret_cast<unsigned long>(this));

	}
	~DeleteResourceActorState() 
	{
		fdb_probe_actor_destroy("deleteResource", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 358 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
				StrictFuture<Void> __when_expr_0 = ::deleteResource(src);
															#line 358 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
				if (static_cast<DeleteResourceActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1269 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeleteResourceActor*>(this)->actor_wait_state = 1;
															#line 358 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeleteResourceActor, 0, Void >*>(static_cast<DeleteResourceActor*>(this)));
															#line 1274 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeleteResourceActorState();
		static_cast<DeleteResourceActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 363 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
		if (!static_cast<DeleteResourceActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeleteResourceActorState(); static_cast<DeleteResourceActor*>(this)->destroy(); return 0; }
															#line 1303 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
		new (&static_cast<DeleteResourceActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeleteResourceActorState();
		static_cast<DeleteResourceActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 361 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 1316 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeleteResourceActor*>(this)->actor_wait_state > 0) static_cast<DeleteResourceActor*>(this)->actor_wait_state = 0;
		static_cast<DeleteResourceActor*>(this)->ActorCallback< DeleteResourceActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeleteResourceActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deleteResource", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteResourceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteResource", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeleteResourceActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deleteResource", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteResourceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteResource", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeleteResourceActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deleteResource", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteResourceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deleteResource", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	std::string src;
															#line 1428 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
};
// This generated class is to be used only via deleteResource()
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
class DeleteResourceActor final : public Actor<Void>, public ActorCallback< DeleteResourceActor, 0, Void >, public FastAllocated<DeleteResourceActor>, public DeleteResourceActorState<DeleteResourceActor> {
															#line 1433 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
public:
	using FastAllocated<DeleteResourceActor>::operator new;
	using FastAllocated<DeleteResourceActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4815620487712202240UL, 10035436564947367168UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeleteResourceActor, 0, Void >;
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	DeleteResourceActor(std::string const& src) 
															#line 1450 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
		 : Actor<Void>(),
		   DeleteResourceActorState<DeleteResourceActor>(src),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deleteResource", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14827851668417892864UL, 17501710632754782464UL);
		ActorExecutionContextHelper __helper(static_cast<DeleteResourceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deleteResource");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deleteResource", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeleteResourceActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
[[nodiscard]] Future<Void> deleteResource( std::string const& src ) {
															#line 356 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	return Future<Void>(new DeleteResourceActor(src));
															#line 1482 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
}

#line 365 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"

															#line 1487 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
// This generated class is to be used only via listFiles()
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
template <class ListFilesActor>
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
class ListFilesActorState {
															#line 1493 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
public:
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	ListFilesActorState(std::string const& src,int const& maxDepth) 
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
		 : src(src),
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
		   maxDepth(maxDepth)
															#line 1502 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
	{
		fdb_probe_actor_create("listFiles", reinterpret_cast<unsigned long>(this));

	}
	~ListFilesActorState() 
	{
		fdb_probe_actor_destroy("listFiles", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
				StrictFuture<Void> __when_expr_0 = ::listFiles(src, maxDepth);
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
				if (static_cast<ListFilesActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1520 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ListFilesActor*>(this)->actor_wait_state = 1;
															#line 368 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ListFilesActor, 0, Void >*>(static_cast<ListFilesActor*>(this)));
															#line 1525 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ListFilesActorState();
		static_cast<ListFilesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 373 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
		if (!static_cast<ListFilesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ListFilesActorState(); static_cast<ListFilesActor*>(this)->destroy(); return 0; }
															#line 1554 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
		new (&static_cast<ListFilesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ListFilesActorState();
		static_cast<ListFilesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 371 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 1567 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ListFilesActor*>(this)->actor_wait_state > 0) static_cast<ListFilesActor*>(this)->actor_wait_state = 0;
		static_cast<ListFilesActor*>(this)->ActorCallback< ListFilesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ListFilesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("listFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ListFilesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("listFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ListFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ListFilesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("listFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ListFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("listFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	std::string src;
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	int maxDepth;
															#line 1681 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
};
// This generated class is to be used only via listFiles()
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
class ListFilesActor final : public Actor<Void>, public ActorCallback< ListFilesActor, 0, Void >, public FastAllocated<ListFilesActor>, public ListFilesActorState<ListFilesActor> {
															#line 1686 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
public:
	using FastAllocated<ListFilesActor>::operator new;
	using FastAllocated<ListFilesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1206425423432889856UL, 9098110653735248128UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ListFilesActor, 0, Void >;
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	ListFilesActor(std::string const& src,int const& maxDepth) 
															#line 1703 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
		 : Actor<Void>(),
		   ListFilesActorState<ListFilesActor>(src, maxDepth),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("listFiles", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7548829570599198464UL, 17955669203805414144UL);
		ActorExecutionContextHelper __helper(static_cast<ListFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("listFiles");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("listFiles", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ListFilesActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
[[nodiscard]] Future<Void> listFiles( std::string const& src, int const& maxDepth ) {
															#line 366 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
	return Future<Void>(new ListFilesActor(src, maxDepth));
															#line 1735 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/build_output/fdbclient/S3Client_cli.actor.g.cpp"
}

#line 375 "/codebuild/output/src2620178984/src/github.com/apple/foundationdb/fdbclient/S3Client_cli.actor.cpp"
} // namespace s3client_cli

int main(int argc, char** argv) {
	std::string commandLine;
	for (int a = 0; a < argc; a++) {
		if (a) {
			commandLine += ' ';
		}
		commandLine += argv[a];
	}
	int status;
	try {
		// This csimpleopt parser is not the smartest. If you pass --logs instead of --log, it will
		// treat the --logs as an argument though it is a misnamed option. Doesn't seem to be a way
		// around it.
		std::unique_ptr<CSimpleOpt> args(
		    new CSimpleOpt(argc, argv, s3client_cli::Options, SO_O_EXACT | SO_O_HYPHEN_TO_UNDERSCORE | SO_O_NOERR));
		auto param = makeReference<s3client_cli::Params>();
		status = s3client_cli::parseCommandLine(param, args.get());
		TraceEvent("S3ClientCommandLine")
		    .detail("CommandLine", commandLine)
		    .detail("Params", param->toString())
		    .detail("Status", status);
		if (status != FDB_EXIT_SUCCESS) {
			s3client_cli::printUsage(argv[0]);
			return status;
		}
		if (param->log_enabled) {
			if (param->log_dir.empty()) {
				setNetworkOption(FDBNetworkOptions::TRACE_ENABLE);
			} else {
				setNetworkOption(FDBNetworkOptions::TRACE_ENABLE, StringRef(param->log_dir));
			}
			if (!param->trace_format.empty()) {
				setNetworkOption(FDBNetworkOptions::TRACE_FORMAT, StringRef(param->trace_format));
			} else {
				setNetworkOption(FDBNetworkOptions::TRACE_FORMAT, "json"_sr);
			}
			if (!param->trace_log_group.empty()) {
				setNetworkOption(FDBNetworkOptions::TRACE_LOG_GROUP, StringRef(param->trace_log_group));
			}
		}
		if (!param->tlsConfig.setupTLS()) {
			TraceEvent(SevError, "TLSError").log();
			throw tls_error();
		}

		platformInit();
		Error::init();

		setupNetwork(0, UseMetrics::True);

		// Must be called after setupNetwork() to be effective
		param->updateKnobs();

		// Check for proxy from environment variable if not set via command line
		if (!param->proxy.present()) {
			const char* proxyEnv = getenv("FDB_PROXY");
			if (proxyEnv != nullptr) {
				param->proxy = std::string(proxyEnv);
			}
		}

		// Set the proxy in g_network if it's present
		if (param->proxy.present()) {
			if (!Hostname::isHostname(param->proxy.get()) &&
			    !NetworkAddress::parseOptional(param->proxy.get()).present()) {
				fprintf(stderr, "ERROR: proxy format should be either IP:port or host:port\n");
				flushAndExit(FDB_EXIT_ERROR);
			}
			Optional<std::string>* pProxy = (Optional<std::string>*)g_network->global(INetwork::enProxy);
			*pProxy = param->proxy.get();
		}

		TraceEvent("ProgramStart")
		    .setMaxEventLength(12000)
		    .detail("SourceVersion", getSourceVersion())
		    .detail("Version", FDB_VT_VERSION)
		    .detail("PackageName", FDB_VT_PACKAGE_NAME)
		    .detailf("ActualTime", "%lld", DEBUG_DETERMINISM ? 0 : time(NULL))
		    .setMaxFieldLength(10000)
		    .detail("CommandLine", commandLine)
		    .setMaxFieldLength(0)
		    .trackLatest("ProgramStart");

		TraceEvent::setNetworkThread();
		std::string path(argv[0]);
		openTraceFile(
		    {}, 10 << 20, 500 << 20, param->log_dir, path.substr(path.find_last_of("/\\") + 1), param->trace_log_group);
		param->tlsConfig.setupBlobCredentials();
		auto f = stopAfter(run(param));
		runNetwork();
		if (f.isValid() && f.isReady() && !f.isError() && !f.get().present()) {
			status = FDB_EXIT_ERROR;
		}
	} catch (Error& e) {
		std::cerr << "ERROR: " << e.what() << "\n";
		return FDB_EXIT_ERROR;
	} catch (std::exception& e) {
		TraceEvent(SevError, "MainError").error(unknown_error()).detail("RootException", e.what());
		return FDB_EXIT_MAIN_EXCEPTION;
	}
	flushAndExit(status);
}
