#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
/*
 * MockS3Server.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2025 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbserver/MockS3Server.h"

#include "fdbrpc/HTTP.h"
#include "fdbrpc/simulator.h"
#include "flow/Trace.h"
#include "flow/IRandom.h"
#include "flow/serialize.h"

#include <string>
#include <map>
#include <sstream>
#include <regex>
#include <utility>
#include <iostream>
#include <algorithm>

#include "flow/IAsyncFile.h" // For IAsyncFileSystem
#include "flow/Platform.h" // For platform::createDirectory

#include "rapidjson/document.h"
#include "rapidjson/writer.h"
#include "rapidjson/stringbuffer.h"
#include "rapidjson/prettywriter.h"

#include "flow/actorcompiler.h" // This must be the last #include.

/*
 * ACTOR STATE VARIABLE INITIALIZATION REQUIREMENT
 *
 * ACTORs with early returns (before any wait()) crash with canBeSet() assertion if no state
 * variable is declared before the return. The actor compiler generates a member initialization
 * list (": member(value)") in the state class constructor only when it sees state variables.
 * This initialization list ensures the Actor<T> base class and its internal Promise are fully
 * initialized before any code runs. Without it, early returns try to use an uninitialized Promise.
 *
 * FIX: Declare at least one state variable BEFORE any early return. Declaration alone is enough.
 *
 * CORRECT:
 *   ACTOR Future<Void> someActor(...) {
 *       state std::string data;                // Triggers member init list (requires default ctor)
 *       if (earlyExitCondition) return Void(); // Safe - Promise is initialized
 *       data = computeValue();                 // Can initialize later
 *   }
 *
 *   // Or if no default constructor: state MyType x(params); and initialize at declaration
 *
 * WRONG (canBeSet() crash):
 *   ACTOR Future<Void> someActor(...) {
 *       if (earlyExitCondition) return Void(); // CRASH - no member init list generated yet
 *       state std::string data;                // Too late - compiler didn't see it early enough
 *   }
 */

// MockS3 persistence file extensions and constants
namespace {
constexpr const char* DEFAULT_MOCKS3_PERSISTENCE_DIR = "simfdb/mocks3";
constexpr const char* OBJECT_DATA_SUFFIX = ".data";
constexpr const char* OBJECT_META_SUFFIX = ".meta.json";
constexpr const char* MULTIPART_STATE_SUFFIX = ".state.json";
constexpr size_t OBJECT_META_SUFFIX_LEN = 10; // strlen(".meta.json")
constexpr size_t MULTIPART_STATE_SUFFIX_LEN = 11; // strlen(".state.json")
} // namespace

// Global storage for MockS3 (shared across all simulated processes)
struct MockS3GlobalStorage {
	struct ObjectData {
		std::string content;
		HTTP::Headers headers;
		std::map<std::string, std::string> tags;
		std::string etag;
		double lastModified;

		ObjectData() : lastModified(now()) {}
		ObjectData(const std::string& data) : content(data), lastModified(now()) { etag = generateETag(data); }

		static std::string generateETag(const std::string& content) {
			return "\"" + HTTP::computeMD5Sum(content) + "\"";
		}
	};

	struct MultipartUpload {
		std::string uploadId;
		std::string bucket;
		std::string object;
		std::map<int, std::pair<std::string, std::string>> parts; // partNum -> {etag, content}
		HTTP::Headers metadata;
		double initiated;

		MultipartUpload() = default;
		MultipartUpload(const std::string& b, const std::string& o) : bucket(b), object(o), initiated(now()) {
			uploadId = deterministicRandom()->randomUniqueID().toString();
		}
	};

	std::map<std::string, std::map<std::string, ObjectData>> buckets;
	std::map<std::string, MultipartUpload> multipartUploads;

	// Persistence configuration
	std::string persistenceDir;
	bool persistenceEnabled = false;
	bool persistenceLoaded = false;

	// Note: In FDB simulation, function-local statics are SHARED across all simulated processes
	// because they all run on the same OS thread. This is exactly what we want for MockS3 storage.
	MockS3GlobalStorage() { TraceEvent("MockS3GlobalStorageCreated").detail("Address", format("%p", this)); }

	// Clear all stored data - called at the start of each simulation test to prevent
	// data accumulation across multiple tests
	void clearStorage() {
		size_t totalObjects = 0;
		size_t totalBuckets = buckets.size();
		for (const auto& [bucketName, objects] : buckets) {
			totalObjects += objects.size();
		}
		size_t totalMultipart = multipartUploads.size();

		TraceEvent("MockS3GlobalStorageClearingDebug")
		    .detail("Address", format("%p", this))
		    .detail("BucketsBeforeClear", totalBuckets)
		    .detail("ObjectsBeforeClear", totalObjects)
		    .detail("MultipartUploadsBeforeClear", totalMultipart);

		buckets.clear();
		multipartUploads.clear();

		TraceEvent("MockS3GlobalStorageCleared")
		    .detail("Address", format("%p", this))
		    .detail("ClearedBuckets", totalBuckets)
		    .detail("ClearedObjects", totalObjects)
		    .detail("ClearedMultipart", totalMultipart);
	}

	// Enable persistence to specified directory
	// Note: When using simulation filesystem, directories are created automatically by file open()
	// and each simulated machine has its own isolated directory structure
	void enablePersistence(const std::string& dir) {
		persistenceDir = dir;
		persistenceEnabled = true;
		persistenceLoaded = false;

		TraceEvent("MockS3PersistenceEnabled")
		    .detail("Directory", dir)
		    .detail("Address", format("%p", this))
		    .detail("UsingSimulationFS", g_network->isSimulated());
	}

	// Get paths for persistence files
	// Note: Object names with slashes map directly to filesystem directory structure
	std::string getObjectDataPath(const std::string& bucket, const std::string& object) const {
		return persistenceDir + "/objects/" + bucket + "/" + object + OBJECT_DATA_SUFFIX;
	}

	std::string getObjectMetaPath(const std::string& bucket, const std::string& object) const {
		return persistenceDir + "/objects/" + bucket + "/" + object + OBJECT_META_SUFFIX;
	}

	std::string getMultipartStatePath(const std::string& uploadId) const {
		return persistenceDir + "/multipart/" + uploadId + MULTIPART_STATE_SUFFIX;
	}

	std::string getMultipartPartPath(const std::string& uploadId, int partNum) const {
		return persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
	}

	std::string getMultipartPartMetaPath(const std::string& uploadId, int partNum) const {
		return persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum) + OBJECT_META_SUFFIX;
	}
};

// Accessor function - uses function-local static for lazy initialization
// In simulation, this static is shared across all simulated processes (same OS thread)
static MockS3GlobalStorage& getGlobalStorage() {
	static MockS3GlobalStorage storage;
	return storage;
}

// Helper: Create all parent directories for a file path
// Uses platform::createDirectory which handles recursive creation and EEXIST errors
static void createParentDirectories(const std::string& filePath) {
	size_t lastSlash = filePath.find_last_of('/');
	if (lastSlash != std::string::npos && lastSlash > 0) {
		std::string parentDir = filePath.substr(0, lastSlash);
		platform::createDirectory(parentDir); // Handles recursive creation and EEXIST
	}
}

// ACTOR: Atomic file write using simulation filesystem without chaos injection
// Chaos-free because AsyncFileChaos only affects files with "storage-" in the name
// (see AsyncFileChaos.h:40). OPEN_NO_AIO controls AsyncFileNonDurable behavior.
															#line 212 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via atomicWriteFile()
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class AtomicWriteFileActor>
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class AtomicWriteFileActorState {
															#line 219 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	AtomicWriteFileActorState(std::string const& path,std::string const& content) 
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path),
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   content(content),
															#line 211 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   initialized(true)
															#line 230 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("atomicWriteFile", reinterpret_cast<unsigned long>(this));

	}
	~AtomicWriteFileActorState() 
	{
		fdb_probe_actor_destroy("atomicWriteFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 215 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				createParentDirectories(path);
															#line 219 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( path, IAsyncFile::OPEN_ATOMIC_WRITE_AND_CREATE | IAsyncFile::OPEN_CREATE | IAsyncFile::OPEN_READWRITE | IAsyncFile::OPEN_UNCACHED | IAsyncFile::OPEN_NO_AIO, 0644);
															#line 219 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 250 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 1;
															#line 219 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 255 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AtomicWriteFileActorState();
		static_cast<AtomicWriteFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 233 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<AtomicWriteFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AtomicWriteFileActorState(); static_cast<AtomicWriteFileActor*>(this)->destroy(); return 0; }
															#line 284 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<AtomicWriteFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AtomicWriteFileActorState();
		static_cast<AtomicWriteFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 231 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceWriteException").error(e).detail("Path", path);
															#line 297 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 225 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = file->write(content.data(), content.size(), 0);
															#line 225 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 314 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 2;
															#line 225 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 1, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 319 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& __file,int loopDepth) 
	{
															#line 219 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = __file;
															#line 328 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && __file,int loopDepth) 
	{
		file = std::move(__file);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 226 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = file->sync();
															#line 226 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 409 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 3;
															#line 226 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 2, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 414 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 226 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = file->sync();
															#line 226 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 425 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 3;
															#line 226 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< AtomicWriteFileActor, 2, Void >*>(static_cast<AtomicWriteFileActor*>(this)));
															#line 430 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 227 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 229 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceWriteSuccess").detail("Path", path).detail("Size", content.size());
															#line 516 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 227 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 229 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceWriteSuccess").detail("Path", path).detail("Size", content.size());
															#line 527 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<AtomicWriteFileActor*>(this)->actor_wait_state > 0) static_cast<AtomicWriteFileActor*>(this)->actor_wait_state = 0;
		static_cast<AtomicWriteFileActor*>(this)->ActorCallback< AtomicWriteFileActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< AtomicWriteFileActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< AtomicWriteFileActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 211 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	bool initialized;
															#line 219 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<IAsyncFile> file;
															#line 628 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via atomicWriteFile()
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class AtomicWriteFileActor final : public Actor<Void>, public ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >, public ActorCallback< AtomicWriteFileActor, 1, Void >, public ActorCallback< AtomicWriteFileActor, 2, Void >, public FastAllocated<AtomicWriteFileActor>, public AtomicWriteFileActorState<AtomicWriteFileActor> {
															#line 633 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<AtomicWriteFileActor>::operator new;
	using FastAllocated<AtomicWriteFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11878423700189261824UL, 17289563703425160704UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< AtomicWriteFileActor, 1, Void >;
friend struct ActorCallback< AtomicWriteFileActor, 2, Void >;
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	AtomicWriteFileActor(std::string const& path,std::string const& content) 
															#line 652 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   AtomicWriteFileActorState<AtomicWriteFileActor>(path, content),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("atomicWriteFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1551474338982279168UL, 5682969234323159552UL);
		ActorExecutionContextHelper __helper(static_cast<AtomicWriteFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("atomicWriteFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("atomicWriteFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< AtomicWriteFileActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> atomicWriteFile( std::string const& path, std::string const& content ) {
															#line 210 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new AtomicWriteFileActor(path, content));
															#line 687 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 235 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Read file content using simulation filesystem without chaos
// Chaos-free because AsyncFileChaos only affects files with "storage-" in the name
															#line 694 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via readFileContent()
															#line 238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class ReadFileContentActor>
															#line 238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class ReadFileContentActorState {
															#line 701 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	ReadFileContentActorState(std::string const& path) 
															#line 238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path),
															#line 239 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   exists(fileExists(path))
															#line 710 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("readFileContent", reinterpret_cast<unsigned long>(this));

	}
	~ReadFileContentActorState() 
	{
		fdb_probe_actor_destroy("readFileContent", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 242 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!exists)
															#line 726 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 243 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(std::string()); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 730 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::string());
					this->~ReadFileContentActorState();
					static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 246 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( path, IAsyncFile::OPEN_READONLY | IAsyncFile::OPEN_UNCACHED | IAsyncFile::OPEN_NO_AIO, 0644);
															#line 246 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 740 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ReadFileContentActor*>(this)->actor_wait_state = 1;
															#line 246 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*>(static_cast<ReadFileContentActor*>(this)));
															#line 745 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 264 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadException").error(e).detail("Path", path);
															#line 265 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(std::string()); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 777 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::string());
			this->~ReadFileContentActorState();
			static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 248 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<int64_t> __when_expr_1 = file->size();
															#line 248 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 797 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadFileContentActor*>(this)->actor_wait_state = 2;
															#line 248 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 1, int64_t >*>(static_cast<ReadFileContentActor*>(this)));
															#line 802 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& __file,int loopDepth) 
	{
															#line 246 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = __file;
															#line 811 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && __file,int loopDepth) 
	{
		file = std::move(__file);
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 250 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content = std::string();
															#line 251 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content.resize(fileSize);
															#line 252 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<int> __when_expr_2 = file->read((uint8_t*)content.data(), fileSize, 0);
															#line 252 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 896 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReadFileContentActor*>(this)->actor_wait_state = 3;
															#line 252 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadFileContentActor, 2, int >*>(static_cast<ReadFileContentActor*>(this)));
															#line 901 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(int64_t const& __fileSize,int loopDepth) 
	{
															#line 248 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileSize = __fileSize;
															#line 910 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(int64_t && __fileSize,int loopDepth) 
	{
		fileSize = std::move(__fileSize);
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 1, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 1, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 1, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 1, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int const& bytesRead,int loopDepth) 
	{
															#line 253 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 255 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bytesRead != fileSize)
															#line 991 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 256 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadShort") .detail("Path", path) .detail("Expected", fileSize) .detail("Actual", bytesRead);
															#line 995 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
															#line 262 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(content); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 999 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::move(content)); // state_var_RVO
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5(int && bytesRead,int loopDepth) 
	{
															#line 253 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		file = Reference<IAsyncFile>();
															#line 255 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bytesRead != fileSize)
															#line 1013 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 256 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceReadShort") .detail("Path", path) .detail("Expected", fileSize) .detail("Actual", bytesRead);
															#line 1017 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
															#line 262 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<ReadFileContentActor*>(this)->SAV<std::string>::futures) { (void)(content); this->~ReadFileContentActorState(); static_cast<ReadFileContentActor*>(this)->destroy(); return 0; }
															#line 1021 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<ReadFileContentActor*>(this)->SAV< std::string >::value()) std::string(std::move(content)); // state_var_RVO
		this->~ReadFileContentActorState();
		static_cast<ReadFileContentActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4when1(int const& bytesRead,int loopDepth) 
	{
		loopDepth = a_body1cont5(bytesRead, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(int && bytesRead,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(bytesRead), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadFileContentActor*>(this)->actor_wait_state > 0) static_cast<ReadFileContentActor*>(this)->actor_wait_state = 0;
		static_cast<ReadFileContentActor*>(this)->ActorCallback< ReadFileContentActor, 2, int >::remove();

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 2, int >*,int const& value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadFileContentActor, 2, int >*,int && value) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadFileContentActor, 2, int >*,Error err) 
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 239 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	bool exists;
															#line 246 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<IAsyncFile> file;
															#line 248 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int64_t fileSize;
															#line 250 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 1114 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via readFileContent()
															#line 238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class ReadFileContentActor final : public Actor<std::string>, public ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >, public ActorCallback< ReadFileContentActor, 1, int64_t >, public ActorCallback< ReadFileContentActor, 2, int >, public FastAllocated<ReadFileContentActor>, public ReadFileContentActorState<ReadFileContentActor> {
															#line 1119 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<ReadFileContentActor>::operator new;
	using FastAllocated<ReadFileContentActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11808789249219257344UL, 1688774657357059328UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<std::string>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< ReadFileContentActor, 1, int64_t >;
friend struct ActorCallback< ReadFileContentActor, 2, int >;
															#line 238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	ReadFileContentActor(std::string const& path) 
															#line 1138 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<std::string>(),
		   ReadFileContentActorState<ReadFileContentActor>(path),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readFileContent", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7623102229524567296UL, 7126613595367428096UL);
		ActorExecutionContextHelper __helper(static_cast<ReadFileContentActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readFileContent");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readFileContent", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadFileContentActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadFileContentActor, 1, int64_t >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadFileContentActor, 2, int >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<std::string> readFileContent( std::string const& path ) {
															#line 238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<std::string>(new ReadFileContentActor(path));
															#line 1173 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 268 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete file using simulation filesystem
// Wraps deleteFile with trace events and error handling for MockS3 persistence cleanup
															#line 1180 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via deletePersistedFile()
															#line 271 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedFileActor>
															#line 271 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedFileActorState {
															#line 1187 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 271 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedFileActorState(std::string const& path) 
															#line 271 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 271 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : path(path)
															#line 1194 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("deletePersistedFile", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedFileActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 273 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = IAsyncFileSystem::filesystem()->deleteFile(path, true);
															#line 273 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1212 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedFileActor*>(this)->actor_wait_state = 1;
															#line 273 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedFileActor, 0, Void >*>(static_cast<DeletePersistedFileActor*>(this)));
															#line 1217 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedFileActorState();
		static_cast<DeletePersistedFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 278 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedFileActorState(); static_cast<DeletePersistedFileActor*>(this)->destroy(); return 0; }
															#line 1246 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<DeletePersistedFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedFileActorState();
		static_cast<DeletePersistedFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 276 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistenceDeleteException").error(e).detail("Path", path);
															#line 1259 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 274 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceDelete").detail("Path", path);
															#line 1274 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 274 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PersistenceDelete").detail("Path", path);
															#line 1283 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedFileActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedFileActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedFileActor*>(this)->ActorCallback< DeletePersistedFileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedFileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedFileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedFileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 271 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string path;
															#line 1378 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via deletePersistedFile()
															#line 271 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedFileActor final : public Actor<Void>, public ActorCallback< DeletePersistedFileActor, 0, Void >, public FastAllocated<DeletePersistedFileActor>, public DeletePersistedFileActorState<DeletePersistedFileActor> {
															#line 1383 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<DeletePersistedFileActor>::operator new;
	using FastAllocated<DeletePersistedFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6923559244319430400UL, 10569543311612271616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedFileActor, 0, Void >;
															#line 271 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedFileActor(std::string const& path) 
															#line 1400 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   DeletePersistedFileActorState<DeletePersistedFileActor>(path),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8878611606913210624UL, 16427746229923233536UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedFileActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 271 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedFile( std::string const& path ) {
															#line 271 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedFileActor(path));
															#line 1433 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 280 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// JSON Serialization using rapidjson
static std::string serializeObjectMeta(const MockS3GlobalStorage::ObjectData& obj) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();

	doc.AddMember("etag", Value(obj.etag.c_str(), allocator), allocator);
	doc.AddMember("lastModified", obj.lastModified, allocator);

	Value tagsObj(kObjectType);
	for (const auto& tag : obj.tags) {
		tagsObj.AddMember(Value(tag.first.c_str(), allocator), Value(tag.second.c_str(), allocator), allocator);
	}
	doc.AddMember("tags", tagsObj, allocator);

	StringBuffer buffer;
	PrettyWriter<StringBuffer> writer(buffer);
	doc.Accept(writer);
	return buffer.GetString();
}

// JSON Deserialization using rapidjson
static void deserializeObjectMeta(const std::string& jsonStr, MockS3GlobalStorage::ObjectData& obj) {
	using namespace rapidjson;
	Document doc;
	doc.Parse(jsonStr.c_str());

	if (doc.HasMember("etag") && doc["etag"].IsString())
		obj.etag = doc["etag"].GetString();
	if (doc.HasMember("lastModified") && doc["lastModified"].IsNumber())
		obj.lastModified = doc["lastModified"].GetDouble();
	if (doc.HasMember("tags") && doc["tags"].IsObject()) {
		for (auto& m : doc["tags"].GetObject()) {
			if (m.value.IsString())
				obj.tags[m.name.GetString()] = m.value.GetString();
		}
	}
}

static std::string serializeMultipartState(const MockS3GlobalStorage::MultipartUpload& upload) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();

	doc.AddMember("uploadId", Value(upload.uploadId.c_str(), allocator), allocator);
	doc.AddMember("bucket", Value(upload.bucket.c_str(), allocator), allocator);
	doc.AddMember("object", Value(upload.object.c_str(), allocator), allocator);
	doc.AddMember("initiated", upload.initiated, allocator);

	Value partsArray(kArrayType);
	for (const auto& part : upload.parts) {
		Value partObj(kObjectType);
		partObj.AddMember("partNum", part.first, allocator);
		partObj.AddMember("etag", Value(part.second.first.c_str(), allocator), allocator);
		partsArray.PushBack(partObj, allocator);
	}
	doc.AddMember("parts", partsArray, allocator);

	StringBuffer buffer;
	PrettyWriter<StringBuffer> writer(buffer);
	doc.Accept(writer);
	return buffer.GetString();
}

static void deserializeMultipartState(const std::string& jsonStr, MockS3GlobalStorage::MultipartUpload& upload) {
	using namespace rapidjson;
	Document doc;
	doc.Parse(jsonStr.c_str());

	if (doc.HasMember("uploadId") && doc["uploadId"].IsString())
		upload.uploadId = doc["uploadId"].GetString();
	if (doc.HasMember("bucket") && doc["bucket"].IsString())
		upload.bucket = doc["bucket"].GetString();
	if (doc.HasMember("object") && doc["object"].IsString())
		upload.object = doc["object"].GetString();
	if (doc.HasMember("initiated") && doc["initiated"].IsNumber())
		upload.initiated = doc["initiated"].GetDouble();
	if (doc.HasMember("parts") && doc["parts"].IsArray()) {
		for (auto& partVal : doc["parts"].GetArray()) {
			if (partVal.HasMember("partNum") && partVal["partNum"].IsInt() && partVal.HasMember("etag") &&
			    partVal["etag"].IsString()) {
				int partNum = partVal["partNum"].GetInt();
				std::string etag = partVal["etag"].GetString();
				upload.parts[partNum] = { etag, "" }; // Content loaded separately from .part.N files
			}
		}
	}
}

// Forward declarations for state loading functions
															#line 1530 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
[[nodiscard]] static Future<Void> loadPersistedObjects( std::string const& persistenceDir );

#line 374 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1534 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
[[nodiscard]] static Future<Void> loadPersistedMultipartUploads( std::string const& persistenceDir );

#line 375 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1538 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
[[nodiscard]] static Future<Void> loadMockS3PersistedStateImpl(  );

#line 376 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
Future<Void> loadMockS3PersistedStateFuture();

static std::string serializePartMeta(const std::string& etag) {
	using namespace rapidjson;
	Document doc;
	doc.SetObject();
	auto& allocator = doc.GetAllocator();
	doc.AddMember("etag", Value(etag.c_str(), allocator), allocator);

	StringBuffer buffer;
	Writer<StringBuffer> writer(buffer); // Use Writer instead of PrettyWriter for compact output
	doc.Accept(writer);
	return buffer.GetString();
}

// ACTOR: Persist object data and metadata
															#line 1558 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via persistObject()
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class PersistObjectActor>
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistObjectActorState {
															#line 1565 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistObjectActorState(std::string const& bucket,std::string const& object) 
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : bucket(bucket),
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 1574 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("persistObject", reinterpret_cast<unsigned long>(this));

	}
	~PersistObjectActorState() 
	{
		fdb_probe_actor_destroy("persistObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 393 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 396 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!storage.persistenceEnabled && g_network->isSimulated())
															#line 1591 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 397 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				storage.enablePersistence(DEFAULT_MOCKS3_PERSISTENCE_DIR);
															#line 398 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3PersistenceAutoEnabled").detail("Bucket", bucket).detail("Object", object);
															#line 1597 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			}
															#line 401 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!storage.persistenceEnabled)
															#line 1601 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 402 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1605 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistObjectActorState();
				static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 405 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto bucketIter = storage.buckets.find(bucket);
															#line 406 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (bucketIter == storage.buckets.end())
															#line 1615 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 407 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1619 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistObjectActorState();
				static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 410 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto objectIter = bucketIter->second.find(object);
															#line 411 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (objectIter == bucketIter->second.end())
															#line 1629 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 412 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1633 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistObjectActorState();
				static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 416 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			content = objectIter->second.content;
															#line 417 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaJson = serializeObjectMeta(objectIter->second);
															#line 1643 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 422 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				dataPath = storage.getObjectDataPath(bucket, object);
															#line 423 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				metaPath = storage.getObjectMetaPath(bucket, object);
															#line 425 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3PersistingObject") .detail("Bucket", bucket) .detail("Object", object) .detail("DataPath", dataPath) .detail("Size", content.size());
															#line 432 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = atomicWriteFile(dataPath, content);
															#line 432 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1655 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PersistObjectActor*>(this)->actor_wait_state = 1;
															#line 432 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 0, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1660 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistObjectActorState();
		static_cast<PersistObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 445 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<PersistObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistObjectActorState(); static_cast<PersistObjectActor*>(this)->destroy(); return 0; }
															#line 1689 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<PersistObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistObjectActorState();
		static_cast<PersistObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 442 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3PersistObjectFailed").error(e).detail("Bucket", bucket).detail("Object", object);
															#line 1702 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 435 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(metaPath, metaJson);
															#line 435 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1719 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont6when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistObjectActor*>(this)->actor_wait_state = 2;
															#line 435 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 1, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1724 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 435 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(metaPath, metaJson);
															#line 435 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1735 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont6when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistObjectActor*>(this)->actor_wait_state = 2;
															#line 435 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistObjectActor, 1, Void >*>(static_cast<PersistObjectActor*>(this)));
															#line 1740 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state > 0) static_cast<PersistObjectActor*>(this)->actor_wait_state = 0;
		static_cast<PersistObjectActor*>(this)->ActorCallback< PersistObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 437 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectPersisted") .detail("Bucket", bucket) .detail("Object", object) .detail("Size", content.size());
															#line 1824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 437 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectPersisted") .detail("Bucket", bucket) .detail("Object", object) .detail("Size", content.size());
															#line 1833 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistObjectActor*>(this)->actor_wait_state > 0) static_cast<PersistObjectActor*>(this)->actor_wait_state = 0;
		static_cast<PersistObjectActor*>(this)->ActorCallback< PersistObjectActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistObjectActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistObjectActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont9(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 416 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 417 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaJson;
															#line 422 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 423 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 1938 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via persistObject()
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistObjectActor final : public Actor<Void>, public ActorCallback< PersistObjectActor, 0, Void >, public ActorCallback< PersistObjectActor, 1, Void >, public FastAllocated<PersistObjectActor>, public PersistObjectActorState<PersistObjectActor> {
															#line 1943 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<PersistObjectActor>::operator new;
	using FastAllocated<PersistObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5967799815222256896UL, 5839522459781140224UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistObjectActor, 0, Void >;
friend struct ActorCallback< PersistObjectActor, 1, Void >;
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistObjectActor(std::string const& bucket,std::string const& object) 
															#line 1961 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   PersistObjectActorState<PersistObjectActor>(bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1247695763033469440UL, 1096140585632004864UL);
		ActorExecutionContextHelper __helper(static_cast<PersistObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistObjectActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistObjectActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> persistObject( std::string const& bucket, std::string const& object ) {
															#line 392 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new PersistObjectActor(bucket, object));
															#line 1995 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 447 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Persist multipart upload state
															#line 2001 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via persistMultipartState()
															#line 449 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class PersistMultipartStateActor>
															#line 449 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistMultipartStateActorState {
															#line 2008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 449 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistMultipartStateActorState(std::string const& uploadId) 
															#line 449 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 449 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : uploadId(uploadId),
															#line 450 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(),
															#line 451 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   parts()
															#line 2019 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("persistMultipartState", reinterpret_cast<unsigned long>(this));

	}
	~PersistMultipartStateActorState() 
	{
		fdb_probe_actor_destroy("persistMultipartState", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 453 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 456 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!storage.persistenceEnabled && g_network->isSimulated())
															#line 2036 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 457 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				storage.enablePersistence(DEFAULT_MOCKS3_PERSISTENCE_DIR);
															#line 458 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3PersistenceAutoEnabled").detail("UploadId", uploadId);
															#line 2042 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			}
															#line 461 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!storage.persistenceEnabled)
															#line 2046 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 462 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistMultipartStateActorState();
				static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 465 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = storage.multipartUploads.find(uploadId);
															#line 466 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == storage.multipartUploads.end())
															#line 2060 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 467 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2064 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~PersistMultipartStateActorState();
				static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 470 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			const auto& upload = uploadIter->second;
															#line 471 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			persistenceDir = storage.persistenceDir;
															#line 472 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			parts = upload.parts;
															#line 2076 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 477 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string statePath = persistenceDir + "/multipart/" + uploadId + ".state.json";
															#line 478 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string stateJson = serializeMultipartState(upload);
															#line 479 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = atomicWriteFile(statePath, stateJson);
															#line 479 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2086 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 1;
															#line 479 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 0, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2091 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~PersistMultipartStateActorState();
		static_cast<PersistMultipartStateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 503 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<PersistMultipartStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~PersistMultipartStateActorState(); static_cast<PersistMultipartStateActor*>(this)->destroy(); return 0; }
															#line 2120 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<PersistMultipartStateActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~PersistMultipartStateActorState();
		static_cast<PersistMultipartStateActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 500 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3PersistMultipartFailed").error(e).detail("UploadId", uploadId);
															#line 2133 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 482 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter = parts.begin();
															#line 483 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 2150 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 482 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter = parts.begin();
															#line 483 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 2161 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 498 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartPersisted").detail("UploadId", uploadId).detail("PartsCount", parts.size());
															#line 2245 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont5loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1(int loopDepth) 
	{
															#line 483 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partIter != parts.end()))
															#line 2261 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1cont5break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 484 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = partIter->first;
															#line 485 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		etag = partIter->second.first;
															#line 486 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partData = partIter->second.second;
															#line 488 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
															#line 489 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = atomicWriteFile(partPath, partData);
															#line 489 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2277 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont5loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 2;
															#line 489 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 1, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2282 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5break1(int loopDepth) 
	{
		try {
			return a_body1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 491 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 492 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = serializePartMeta(etag);
															#line 493 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = atomicWriteFile(partMetaPath, partMetaJson);
															#line 493 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2310 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont5loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 3;
															#line 493 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 2, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2315 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 491 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 492 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = serializePartMeta(etag);
															#line 493 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = atomicWriteFile(partMetaPath, partMetaJson);
															#line 493 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2330 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont5loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 3;
															#line 493 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< PersistMultipartStateActor, 2, Void >*>(static_cast<PersistMultipartStateActor*>(this)));
															#line 2335 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont5loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont5loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 495 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter++;
															#line 2419 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont5loopHead1(0);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 495 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partIter++;
															#line 2428 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont5loopHead1(0);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<PersistMultipartStateActor*>(this)->actor_wait_state > 0) static_cast<PersistMultipartStateActor*>(this)->actor_wait_state = 0;
		static_cast<PersistMultipartStateActor*>(this)->ActorCallback< PersistMultipartStateActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< PersistMultipartStateActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< PersistMultipartStateActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 449 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 450 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 451 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<int, std::pair<std::string, std::string>> parts;
															#line 482 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<int, std::pair<std::string, std::string>>::iterator partIter;
															#line 484 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 485 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 486 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partData;
															#line 488 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 491 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 492 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaJson;
															#line 2541 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via persistMultipartState()
															#line 449 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class PersistMultipartStateActor final : public Actor<Void>, public ActorCallback< PersistMultipartStateActor, 0, Void >, public ActorCallback< PersistMultipartStateActor, 1, Void >, public ActorCallback< PersistMultipartStateActor, 2, Void >, public FastAllocated<PersistMultipartStateActor>, public PersistMultipartStateActorState<PersistMultipartStateActor> {
															#line 2546 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<PersistMultipartStateActor>::operator new;
	using FastAllocated<PersistMultipartStateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17064957336697683712UL, 6644531351620709632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< PersistMultipartStateActor, 0, Void >;
friend struct ActorCallback< PersistMultipartStateActor, 1, Void >;
friend struct ActorCallback< PersistMultipartStateActor, 2, Void >;
															#line 449 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	PersistMultipartStateActor(std::string const& uploadId) 
															#line 2565 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   PersistMultipartStateActorState<PersistMultipartStateActor>(uploadId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("persistMultipartState", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9066204211401481728UL, 7216261254142226944UL);
		ActorExecutionContextHelper __helper(static_cast<PersistMultipartStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("persistMultipartState");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("persistMultipartState", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< PersistMultipartStateActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 449 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> persistMultipartState( std::string const& uploadId ) {
															#line 449 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new PersistMultipartStateActor(uploadId));
															#line 2600 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 505 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete persisted object
															#line 2606 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via deletePersistedObject()
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedObjectActor>
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedObjectActorState {
															#line 2613 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedObjectActorState(std::string const& bucket,std::string const& object) 
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : bucket(bucket),
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 508 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   dataPath(),
															#line 509 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   metaPath()
															#line 2626 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("deletePersistedObject", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedObjectActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 511 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 512 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 514 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			dataPath = storage.getObjectDataPath(bucket, object);
															#line 515 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaPath = storage.getObjectMetaPath(bucket, object);
															#line 2647 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 518 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedFile(dataPath);
															#line 518 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2653 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 1;
															#line 518 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 0, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2658 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedObjectActorState();
		static_cast<DeletePersistedObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 529 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedObjectActorState(); static_cast<DeletePersistedObjectActor*>(this)->destroy(); return 0; }
															#line 2687 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<DeletePersistedObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedObjectActorState();
		static_cast<DeletePersistedObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 523 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3DeletePersistedObjectFailed") .error(e) .detail("Bucket", bucket) .detail("Object", object);
															#line 2700 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 519 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(metaPath);
															#line 519 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2717 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 2;
															#line 519 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 1, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2722 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 519 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(metaPath);
															#line 519 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2733 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 2;
															#line 519 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedObjectActor, 1, Void >*>(static_cast<DeletePersistedObjectActor*>(this)));
															#line 2738 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedObjectActor*>(this)->ActorCallback< DeletePersistedObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 521 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 2822 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 521 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 2831 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedObjectActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedObjectActor*>(this)->ActorCallback< DeletePersistedObjectActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeletePersistedObjectActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 508 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 509 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 2932 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via deletePersistedObject()
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedObjectActor final : public Actor<Void>, public ActorCallback< DeletePersistedObjectActor, 0, Void >, public ActorCallback< DeletePersistedObjectActor, 1, Void >, public FastAllocated<DeletePersistedObjectActor>, public DeletePersistedObjectActorState<DeletePersistedObjectActor> {
															#line 2937 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<DeletePersistedObjectActor>::operator new;
	using FastAllocated<DeletePersistedObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(347857931132335360UL, 4980797872990648064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedObjectActor, 0, Void >;
friend struct ActorCallback< DeletePersistedObjectActor, 1, Void >;
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedObjectActor(std::string const& bucket,std::string const& object) 
															#line 2955 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   DeletePersistedObjectActorState<DeletePersistedObjectActor>(bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5402173951988437248UL, 4607022702540543488UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedObjectActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeletePersistedObjectActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedObject( std::string const& bucket, std::string const& object ) {
															#line 507 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedObjectActor(bucket, object));
															#line 2989 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 531 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Delete persisted multipart upload
															#line 2995 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via deletePersistedMultipart()
															#line 533 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class DeletePersistedMultipartActor>
															#line 533 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedMultipartActorState {
															#line 3002 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 533 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedMultipartActorState(std::string const& uploadId) 
															#line 533 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 533 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : uploadId(uploadId),
															#line 534 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   maxPart(),
															#line 535 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(),
															#line 536 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partNum(),
															#line 537 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partPath(),
															#line 538 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partMetaPath()
															#line 3019 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("deletePersistedMultipart", reinterpret_cast<unsigned long>(this));

	}
	~DeletePersistedMultipartActorState() 
	{
		fdb_probe_actor_destroy("deletePersistedMultipart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 540 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto& storage = getGlobalStorage();
															#line 541 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(storage.persistenceEnabled);
															#line 3036 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 545 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				auto uploadIter = storage.multipartUploads.find(uploadId);
															#line 546 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				maxPart = 100;
															#line 547 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (uploadIter != storage.multipartUploads.end())
															#line 3044 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 548 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					for( const auto& part : uploadIter->second.parts ) {
															#line 549 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						maxPart = std::max(maxPart, part.first);
															#line 3050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					}
				}
															#line 554 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				persistenceDir = storage.persistenceDir;
															#line 557 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string statePath = persistenceDir + "/multipart/" + uploadId + ".state.json";
															#line 558 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = deletePersistedFile(statePath);
															#line 558 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3061 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 1;
															#line 558 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 0, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3066 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DeletePersistedMultipartActorState();
		static_cast<DeletePersistedMultipartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 582 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<DeletePersistedMultipartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DeletePersistedMultipartActorState(); static_cast<DeletePersistedMultipartActor*>(this)->destroy(); return 0; }
															#line 3095 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<DeletePersistedMultipartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DeletePersistedMultipartActorState();
		static_cast<DeletePersistedMultipartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 579 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3DeletePersistedMultipartFailed").error(e).detail("UploadId", uploadId);
															#line 3108 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 562 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 3125 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 562 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		;
															#line 3136 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 577 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartDeleted").detail("UploadId", uploadId);
															#line 3220 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partNum <= maxPart + 10))
															#line 3236 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 564 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = persistenceDir + "/multipart/" + uploadId + ".part." + std::to_string(partNum);
															#line 565 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 566 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedFile(partPath);
															#line 566 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3248 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 2;
															#line 566 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 1, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3253 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 567 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = deletePersistedFile(partMetaPath);
															#line 567 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3277 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 3;
															#line 567 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 2, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3282 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 567 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_2 = deletePersistedFile(partMetaPath);
															#line 567 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3293 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 3;
															#line 567 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 2, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3298 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 568 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 572 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (partNum % 2 == 0)
															#line 3384 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 573 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0.02);
															#line 573 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3390 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 4;
															#line 573 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 3, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3395 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 568 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 572 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (partNum % 2 == 0)
															#line 3411 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 573 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0.02);
															#line 573 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3417 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 4;
															#line 573 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DeletePersistedMultipartActor, 3, Void >*>(static_cast<DeletePersistedMultipartActor*>(this)));
															#line 3422 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2loopBody1cont4(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state > 0) static_cast<DeletePersistedMultipartActor*>(this)->actor_wait_state = 0;
		static_cast<DeletePersistedMultipartActor*>(this)->ActorCallback< DeletePersistedMultipartActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DeletePersistedMultipartActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 533 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 534 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int maxPart;
															#line 535 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 536 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 537 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 538 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 3625 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via deletePersistedMultipart()
															#line 533 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class DeletePersistedMultipartActor final : public Actor<Void>, public ActorCallback< DeletePersistedMultipartActor, 0, Void >, public ActorCallback< DeletePersistedMultipartActor, 1, Void >, public ActorCallback< DeletePersistedMultipartActor, 2, Void >, public ActorCallback< DeletePersistedMultipartActor, 3, Void >, public FastAllocated<DeletePersistedMultipartActor>, public DeletePersistedMultipartActorState<DeletePersistedMultipartActor> {
															#line 3630 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<DeletePersistedMultipartActor>::operator new;
	using FastAllocated<DeletePersistedMultipartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15818059773574295040UL, 10582062469822832896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DeletePersistedMultipartActor, 0, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 1, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 2, Void >;
friend struct ActorCallback< DeletePersistedMultipartActor, 3, Void >;
															#line 533 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	DeletePersistedMultipartActor(std::string const& uploadId) 
															#line 3650 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   DeletePersistedMultipartActorState<DeletePersistedMultipartActor>(uploadId),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15695025802244908800UL, 1209722574274508288UL);
		ActorExecutionContextHelper __helper(static_cast<DeletePersistedMultipartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("deletePersistedMultipart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("deletePersistedMultipart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DeletePersistedMultipartActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 533 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> deletePersistedMultipart( std::string const& uploadId ) {
															#line 533 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new DeletePersistedMultipartActor(uploadId));
															#line 3686 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 584 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Mock S3 Server Implementation for deterministic testing
class MockS3ServerImpl {
public:
	using ObjectData = MockS3GlobalStorage::ObjectData;
	using MultipartUpload = MockS3GlobalStorage::MultipartUpload;

	MockS3ServerImpl() { TraceEvent("MockS3ServerImpl_Constructor").detail("Address", format("%p", this)); }

	~MockS3ServerImpl() { TraceEvent("MockS3ServerImpl_Destructor").detail("Address", format("%p", this)); }

	// S3 Operation Handlers
																#line 3702 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleRequest()
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleRequestActor>
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleRequestActorState {
															#line 3708 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleRequestActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response) 
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response)
															#line 3719 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleRequest", reinterpret_cast<unsigned long>(this));

	}
	~HandleRequestActorState() 
	{
		fdb_probe_actor_destroy("handleRequest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 600 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3Request") .detail("Method", req->verb) .detail("Resource", req->resource) .detail("ContentLength", req->data.contentLen) .detail("Headers", req->data.headers.size()) .detail("UserAgent", req->data.headers.find("User-Agent") != req->data.headers.end() ? req->data.headers.at("User-Agent") : "N/A") .detail("Host", req->data.headers.find("Host") != req->data.headers.end() ? req->data.headers.at("Host") : "N/A");
															#line 3734 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 613 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::string bucket, object;
															#line 614 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::map<std::string, std::string> queryParams;
															#line 615 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->parseS3Request(req->resource, bucket, object, queryParams);
															#line 617 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ParsedRequest") .detail("Bucket", bucket) .detail("Object", object) .detail("QueryParamCount", queryParams.size());
															#line 623 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (queryParams.count("uploads"))
															#line 3746 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 624 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					StrictFuture<Void> __when_expr_0 = self->handleMultipartStart(self, req, response, bucket, object);
															#line 624 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3752 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<HandleRequestActor*>(this)->actor_wait_state = 1;
															#line 624 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 0, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3757 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
															#line 625 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					if (queryParams.count("uploadId"))
															#line 3764 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					{
															#line 626 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						if (queryParams.count("partNumber"))
															#line 3768 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
						{
															#line 627 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							StrictFuture<Void> __when_expr_1 = self->handleUploadPart(self, req, response, bucket, object, queryParams);
															#line 627 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3774 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
							static_cast<HandleRequestActor*>(this)->actor_wait_state = 2;
															#line 627 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 1, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3779 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							loopDepth = 0;
						}
						else
						{
															#line 628 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (req->verb == "POST")
															#line 3786 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							{
															#line 629 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_2 = self->handleMultipartComplete(self, req, response, bucket, object, queryParams);
															#line 629 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3792 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), loopDepth); else return a_body1when3(__when_expr_2.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 3;
															#line 629 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 2, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3797 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								loopDepth = 0;
							}
							else
							{
															#line 630 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (req->verb == "DELETE")
															#line 3804 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								{
															#line 631 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									StrictFuture<Void> __when_expr_3 = self->handleMultipartAbort(self, req, response, bucket, object, queryParams);
															#line 631 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3810 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), loopDepth); else return a_body1when4(__when_expr_3.get(), loopDepth); };
									static_cast<HandleRequestActor*>(this)->actor_wait_state = 4;
															#line 631 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 3, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3815 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = 0;
								}
								else
								{
															#line 633 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									self->sendError( response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "InvalidRequest", "Unknown multipart operation");
															#line 3822 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = a_body1cont9(loopDepth);
								}
							}
						}
					}
					else
					{
															#line 636 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
						if (queryParams.count("tagging"))
															#line 3832 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
						{
															#line 637 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (req->verb == "PUT")
															#line 3836 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							{
															#line 638 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_4 = self->handlePutObjectTags(self, req, response, bucket, object);
															#line 638 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3842 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), loopDepth); else return a_body1when5(__when_expr_4.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 5;
															#line 638 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 4, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3847 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								loopDepth = 0;
							}
							else
							{
															#line 639 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (req->verb == "GET")
															#line 3854 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								{
															#line 640 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									StrictFuture<Void> __when_expr_5 = self->handleGetObjectTags(self, req, response, bucket, object);
															#line 640 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch2(__when_expr_5.getError(), loopDepth); else return a_body1when6(__when_expr_5.get(), loopDepth); };
									static_cast<HandleRequestActor*>(this)->actor_wait_state = 6;
															#line 640 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 5, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3865 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = 0;
								}
								else
								{
															#line 642 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									self->sendError(response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Method not allowed for tagging");
															#line 3872 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									loopDepth = a_body1cont15(loopDepth);
								}
							}
						}
						else
						{
															#line 647 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
							if (queryParams.count("list-type") || (req->verb == "GET" && object.empty()))
															#line 3881 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
							{
															#line 649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								StrictFuture<Void> __when_expr_6 = self->handleListObjects(self, req, response, bucket, queryParams);
															#line 649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3887 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch2(__when_expr_6.getError(), loopDepth); else return a_body1when7(__when_expr_6.get(), loopDepth); };
								static_cast<HandleRequestActor*>(this)->actor_wait_state = 7;
															#line 649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 6, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3892 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								loopDepth = 0;
							}
							else
							{
															#line 650 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
								if (object.empty())
															#line 3899 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
								{
															#line 652 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (req->verb == "HEAD")
															#line 3903 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									{
															#line 653 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										StrictFuture<Void> __when_expr_7 = self->handleHeadBucket(self, req, response, bucket);
															#line 653 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3909 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch2(__when_expr_7.getError(), loopDepth); else return a_body1when8(__when_expr_7.get(), loopDepth); };
										static_cast<HandleRequestActor*>(this)->actor_wait_state = 8;
															#line 653 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 7, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3914 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										loopDepth = 0;
									}
									else
									{
															#line 654 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (req->verb == "PUT")
															#line 3921 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										{
															#line 655 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											StrictFuture<Void> __when_expr_8 = self->handlePutBucket(self, req, response, bucket);
															#line 655 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3927 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch2(__when_expr_8.getError(), loopDepth); else return a_body1when9(__when_expr_8.get(), loopDepth); };
											static_cast<HandleRequestActor*>(this)->actor_wait_state = 9;
															#line 655 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 8, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3932 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											loopDepth = 0;
										}
										else
										{
															#line 657 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											self->sendError(response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Bucket operation not supported");
															#line 3939 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											loopDepth = a_body1cont23(loopDepth);
										}
									}
								}
								else
								{
															#line 664 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
									if (req->verb == "PUT")
															#line 3948 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
									{
															#line 665 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										StrictFuture<Void> __when_expr_9 = self->handlePutObject(self, req, response, bucket, object);
															#line 665 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3954 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch2(__when_expr_9.getError(), loopDepth); else return a_body1when10(__when_expr_9.get(), loopDepth); };
										static_cast<HandleRequestActor*>(this)->actor_wait_state = 10;
															#line 665 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 9, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3959 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										loopDepth = 0;
									}
									else
									{
															#line 666 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
										if (req->verb == "GET")
															#line 3966 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
										{
															#line 667 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											StrictFuture<Void> __when_expr_10 = self->handleGetObject(self, req, response, bucket, object);
															#line 667 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3972 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch2(__when_expr_10.getError(), loopDepth); else return a_body1when11(__when_expr_10.get(), loopDepth); };
											static_cast<HandleRequestActor*>(this)->actor_wait_state = 11;
															#line 667 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 10, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3977 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											loopDepth = 0;
										}
										else
										{
															#line 668 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
											if (req->verb == "DELETE")
															#line 3984 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
											{
															#line 669 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												StrictFuture<Void> __when_expr_11 = self->handleDeleteObject(self, req, response, bucket, object);
															#line 669 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3990 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
												if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1Catch2(__when_expr_11.getError(), loopDepth); else return a_body1when12(__when_expr_11.get(), loopDepth); };
												static_cast<HandleRequestActor*>(this)->actor_wait_state = 12;
															#line 669 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 11, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 3995 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
												loopDepth = 0;
											}
											else
											{
															#line 670 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
												if (req->verb == "HEAD")
															#line 4002 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
												{
															#line 671 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													StrictFuture<Void> __when_expr_12 = self->handleHeadObject(self, req, response, bucket, object);
															#line 671 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													if (static_cast<HandleRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
													if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1Catch2(__when_expr_12.getError(), loopDepth); else return a_body1when13(__when_expr_12.get(), loopDepth); };
													static_cast<HandleRequestActor*>(this)->actor_wait_state = 13;
															#line 671 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< HandleRequestActor, 12, Void >*>(static_cast<HandleRequestActor*>(this)));
															#line 4013 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
													loopDepth = 0;
												}
												else
												{
															#line 673 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
													self->sendError( response, HTTP::HTTP_STATUS_CODE_BAD_GATEWAY, "MethodNotAllowed", "Method not supported");
															#line 4020 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
													loopDepth = a_body1cont32(loopDepth);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleRequestActorState();
		static_cast<HandleRequestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 683 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleRequestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleRequestActorState(); static_cast<HandleRequestActor*>(this)->destroy(); return 0; }
															#line 4058 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleRequestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleRequestActorState();
		static_cast<HandleRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 679 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3RequestError").error(e).detail("Resource", req->resource);
															#line 680 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			self->sendError(response, 500, "InternalError", "Internal server error");
															#line 4073 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont35(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1when3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont9(int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1when4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1when4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont12(int loopDepth) 
	{
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont14(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont14(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1when5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont14(_, loopDepth);

		return loopDepth;
	}
	int a_body1when5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont14(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont15(int loopDepth) 
	{
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont16(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1when6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont16(_, loopDepth);

		return loopDepth;
	}
	int a_body1when6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont16(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1when6(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont18(int loopDepth) 
	{
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont19(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont19(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1when7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont19(_, loopDepth);

		return loopDepth;
	}
	int a_body1when7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont19(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1when7(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont20(int loopDepth) 
	{
		loopDepth = a_body1cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont21(int loopDepth) 
	{
		loopDepth = a_body1cont20(loopDepth);

		return loopDepth;
	}
	int a_body1cont22(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1cont22(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1when8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont22(_, loopDepth);

		return loopDepth;
	}
	int a_body1when8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont22(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1when8(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1when8(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont23(int loopDepth) 
	{
		loopDepth = a_body1cont21(loopDepth);

		return loopDepth;
	}
	int a_body1cont24(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont23(loopDepth);

		return loopDepth;
	}
	int a_body1cont24(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont23(loopDepth);

		return loopDepth;
	}
	int a_body1when9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont24(_, loopDepth);

		return loopDepth;
	}
	int a_body1when9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont24(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1when9(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1when9(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont26(int loopDepth) 
	{
		loopDepth = a_body1cont20(loopDepth);

		return loopDepth;
	}
	int a_body1cont27(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1cont27(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1when10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont27(_, loopDepth);

		return loopDepth;
	}
	int a_body1when10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont27(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1when10(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1when10(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont28(int loopDepth) 
	{
		loopDepth = a_body1cont26(loopDepth);

		return loopDepth;
	}
	int a_body1cont29(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1cont29(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1when11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont29(_, loopDepth);

		return loopDepth;
	}
	int a_body1when11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont29(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1when11(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1when11(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont30(int loopDepth) 
	{
		loopDepth = a_body1cont28(loopDepth);

		return loopDepth;
	}
	int a_body1cont31(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont31(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1when12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont31(_, loopDepth);

		return loopDepth;
	}
	int a_body1when12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont31(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1when12(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1when12(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont32(int loopDepth) 
	{
		loopDepth = a_body1cont30(loopDepth);

		return loopDepth;
	}
	int a_body1cont33(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont32(loopDepth);

		return loopDepth;
	}
	int a_body1cont33(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont32(loopDepth);

		return loopDepth;
	}
	int a_body1when13(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont33(_, loopDepth);

		return loopDepth;
	}
	int a_body1when13(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont33(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<HandleRequestActor*>(this)->actor_wait_state > 0) static_cast<HandleRequestActor*>(this)->actor_wait_state = 0;
		static_cast<HandleRequestActor*>(this)->ActorCallback< HandleRequestActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1when13(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< HandleRequestActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1when13(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< HandleRequestActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), 12);

	}
	int a_body1cont35(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 5330 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleRequest()
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleRequestActor final : public Actor<Void>, public ActorCallback< HandleRequestActor, 0, Void >, public ActorCallback< HandleRequestActor, 1, Void >, public ActorCallback< HandleRequestActor, 2, Void >, public ActorCallback< HandleRequestActor, 3, Void >, public ActorCallback< HandleRequestActor, 4, Void >, public ActorCallback< HandleRequestActor, 5, Void >, public ActorCallback< HandleRequestActor, 6, Void >, public ActorCallback< HandleRequestActor, 7, Void >, public ActorCallback< HandleRequestActor, 8, Void >, public ActorCallback< HandleRequestActor, 9, Void >, public ActorCallback< HandleRequestActor, 10, Void >, public ActorCallback< HandleRequestActor, 11, Void >, public ActorCallback< HandleRequestActor, 12, Void >, public FastAllocated<HandleRequestActor>, public HandleRequestActorState<HandleRequestActor> {
															#line 5335 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleRequestActor>::operator new;
	using FastAllocated<HandleRequestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8038049552992900352UL, 9238080013022759424UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleRequestActor, 0, Void >;
friend struct ActorCallback< HandleRequestActor, 1, Void >;
friend struct ActorCallback< HandleRequestActor, 2, Void >;
friend struct ActorCallback< HandleRequestActor, 3, Void >;
friend struct ActorCallback< HandleRequestActor, 4, Void >;
friend struct ActorCallback< HandleRequestActor, 5, Void >;
friend struct ActorCallback< HandleRequestActor, 6, Void >;
friend struct ActorCallback< HandleRequestActor, 7, Void >;
friend struct ActorCallback< HandleRequestActor, 8, Void >;
friend struct ActorCallback< HandleRequestActor, 9, Void >;
friend struct ActorCallback< HandleRequestActor, 10, Void >;
friend struct ActorCallback< HandleRequestActor, 11, Void >;
friend struct ActorCallback< HandleRequestActor, 12, Void >;
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleRequestActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response) 
															#line 5364 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleRequestActorState<HandleRequestActor>(self, req, response),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleRequest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3085149730912622080UL, 127431907239638784UL);
		ActorExecutionContextHelper __helper(static_cast<HandleRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleRequest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleRequest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleRequestActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< HandleRequestActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< HandleRequestActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< HandleRequestActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< HandleRequestActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< HandleRequestActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< HandleRequestActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< HandleRequestActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< HandleRequestActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< HandleRequestActor, 9, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< HandleRequestActor, 10, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< HandleRequestActor, 11, Void >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< HandleRequestActor, 12, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleRequest( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response ) {
															#line 596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleRequestActor(self, req, response));
															#line 5408 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 685 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	void parseS3Request(const std::string& resource,
	                    std::string& bucket,
	                    std::string& object,
	                    std::map<std::string, std::string>& queryParams) {

		// Split resource into path and query string
		size_t queryPos = resource.find('?');
		std::string path = (queryPos != std::string::npos) ? resource.substr(0, queryPos) : resource;
		std::string query = (queryPos != std::string::npos) ? resource.substr(queryPos + 1) : "";

		// Parse path: /bucket/object (like real S3)
		if (path.size() > 1) {
			path = path.substr(1); // Remove leading /
			size_t slashPos = path.find('/');
			if (slashPos != std::string::npos) {
				bucket = path.substr(0, slashPos);
				object = path.substr(slashPos + 1);
			} else {
				bucket = path;
				object = "";
			}
		}

		// Parse query parameters
		if (!query.empty()) {
			std::regex paramRegex("([^&=]+)=?([^&]*)");
			std::sregex_iterator iter(query.begin(), query.end(), paramRegex);
			std::sregex_iterator end;

			for (; iter != end; ++iter) {
				std::string key = iter->str(1);
				std::string value = iter->str(2);
				// URL decode the parameter value
				queryParams[key] = HTTP::urlDecode(value);
			}
		}

		// MockS3Server handles S3 HTTP requests where bucket is always the first path component
		// For bucket operations: HEAD /bucket_name
		// For object operations: HEAD /bucket_name/object_path
		if (bucket.empty()) {
			TraceEvent(SevWarn, "MockS3MissingBucketInPath").detail("Resource", resource).detail("QueryString", query);
			throw backup_invalid_url();
		}

		TraceEvent("MockS3ParsedPath")
		    .detail("OriginalResource", resource)
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("QueryString", query);
	}

	// Parse HTTP Range header: "bytes=start-end"
	// Returns true if parsing succeeded, false otherwise
	// Sets rangeStart and rangeEnd to the parsed values
	static bool parseRangeHeader(const std::string& rangeHeader, int64_t& rangeStart, int64_t& rangeEnd) {
		if (rangeHeader.empty()) {
			return false;
		}

		// Check for "bytes=" prefix
		if (rangeHeader.substr(0, 6) != "bytes=") {
			return false;
		}

		std::string range = rangeHeader.substr(6);
		size_t dashPos = range.find('-');
		if (dashPos == std::string::npos) {
			return false;
		}

		try {
			rangeStart = std::stoll(range.substr(0, dashPos));
			std::string endStr = range.substr(dashPos + 1);
			if (endStr.empty()) {
				// Open-ended range (e.g., "bytes=100-")
				rangeEnd = -1; // Indicates open-ended
			} else {
				rangeEnd = std::stoll(endStr);
			}
			return true;
		} catch (...) {
			return false;
		}
	}

	// Multipart Upload Operations
																#line 5500 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleMultipartStart()
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartStartActor>
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartStartActorState {
															#line 5506 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartStartActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 5521 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleMultipartStart", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartStartActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartStart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 779 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartStart").detail("Bucket", bucket).detail("Object", object);
															#line 784 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string existingUploadId;
															#line 785 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			for( const auto& pair : getGlobalStorage().multipartUploads ) {
															#line 786 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (pair.second.bucket == bucket && pair.second.object == object)
															#line 5542 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 787 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					existingUploadId = pair.first;
															#line 788 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					TraceEvent("MockS3MultipartStartIdempotent") .detail("Bucket", bucket) .detail("Object", object) .detail("ExistingUploadId", existingUploadId);
															#line 5548 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
					break;
				}
			}
															#line 796 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadId = std::string();
															#line 797 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!existingUploadId.empty())
															#line 5556 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 798 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				uploadId = existingUploadId;
															#line 5560 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			}
			else
			{
															#line 800 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				MultipartUpload upload(bucket, object);
															#line 801 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				uploadId = upload.uploadId;
															#line 802 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				getGlobalStorage().multipartUploads[uploadId] = std::move(upload);
															#line 803 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3MultipartStarted").detail("UploadId", uploadId);
															#line 5572 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			}
															#line 806 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = persistMultipartState(uploadId);
															#line 806 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleMultipartStartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5578 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandleMultipartStartActor*>(this)->actor_wait_state = 1;
															#line 806 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartStartActor, 0, Void >*>(static_cast<HandleMultipartStartActor*>(this)));
															#line 5583 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartStartActorState();
		static_cast<HandleMultipartStartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 809 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<InitiateMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <UploadId>%s</UploadId>\n" "</InitiateMultipartUploadResult>", bucket.c_str(), object.c_str(), uploadId.c_str());
															#line 819 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 821 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartStartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartStartActorState(); static_cast<HandleMultipartStartActor*>(this)->destroy(); return 0; }
															#line 5610 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartStartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartStartActorState();
		static_cast<HandleMultipartStartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 809 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<InitiateMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <UploadId>%s</UploadId>\n" "</InitiateMultipartUploadResult>", bucket.c_str(), object.c_str(), uploadId.c_str());
															#line 819 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 821 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartStartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartStartActorState(); static_cast<HandleMultipartStartActor*>(this)->destroy(); return 0; }
															#line 5626 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartStartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartStartActorState();
		static_cast<HandleMultipartStartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartStartActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartStartActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartStartActor*>(this)->ActorCallback< HandleMultipartStartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartStartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartStartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartStartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 796 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 5721 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleMultipartStart()
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartStartActor final : public Actor<Void>, public ActorCallback< HandleMultipartStartActor, 0, Void >, public FastAllocated<HandleMultipartStartActor>, public HandleMultipartStartActorState<HandleMultipartStartActor> {
															#line 5726 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleMultipartStartActor>::operator new;
	using FastAllocated<HandleMultipartStartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15384466730469719040UL, 1162329390522324736UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartStartActor, 0, Void >;
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartStartActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 5743 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleMultipartStartActorState<HandleMultipartStartActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartStart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10763588338010117376UL, 16830117502434504448UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartStartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartStart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartStart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartStartActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartStart( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 773 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartStartActor(self, req, response, bucket, object));
															#line 5775 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 823 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 5780 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleUploadPart()
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleUploadPartActor>
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleUploadPartActorState {
															#line 5786 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleUploadPartActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 831 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId")),
															#line 832 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   partNumber(std::stoi(queryParams.at("partNumber")))
															#line 5807 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleUploadPart", reinterpret_cast<unsigned long>(this));

	}
	~HandleUploadPartActorState() 
	{
		fdb_probe_actor_destroy("handleUploadPart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 834 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3UploadPart") .detail("UploadId", uploadId) .detail("PartNumber", partNumber) .detail("ContentLength", req->data.contentLen) .detail("ActualContentSize", req->data.content.size()) .detail("ContentPreview", req->data.content.size() > 0 ? req->data.content.substr(0, std::min((size_t)20, req->data.content.size())) : "EMPTY");
															#line 844 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 845 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 5826 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 847 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleUploadPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleUploadPartActorState(); static_cast<HandleUploadPartActor*>(this)->destroy(); return 0; }
															#line 5832 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<HandleUploadPartActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleUploadPartActorState();
				static_cast<HandleUploadPartActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 851 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			etag = ObjectData::generateETag(req->data.content);
															#line 852 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadIter->second.parts[partNumber] = { etag, req->data.content };
															#line 855 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = persistMultipartState(uploadId);
															#line 855 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleUploadPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandleUploadPartActor*>(this)->actor_wait_state = 1;
															#line 855 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleUploadPartActor, 0, Void >*>(static_cast<HandleUploadPartActor*>(this)));
															#line 5851 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleUploadPartActorState();
		static_cast<HandleUploadPartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 858 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 859 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 861 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 863 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PartUploaded") .detail("UploadId", uploadId) .detail("PartNumber", partNumber) .detail("ETag", etag);
															#line 868 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleUploadPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleUploadPartActorState(); static_cast<HandleUploadPartActor*>(this)->destroy(); return 0; }
															#line 5884 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleUploadPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleUploadPartActorState();
		static_cast<HandleUploadPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 858 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 859 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 861 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 863 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PartUploaded") .detail("UploadId", uploadId) .detail("PartNumber", partNumber) .detail("ETag", etag);
															#line 868 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleUploadPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleUploadPartActorState(); static_cast<HandleUploadPartActor*>(this)->destroy(); return 0; }
															#line 5906 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleUploadPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleUploadPartActorState();
		static_cast<HandleUploadPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleUploadPartActor*>(this)->actor_wait_state > 0) static_cast<HandleUploadPartActor*>(this)->actor_wait_state = 0;
		static_cast<HandleUploadPartActor*>(this)->ActorCallback< HandleUploadPartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleUploadPartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleUploadPartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleUploadPartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 831 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 832 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNumber;
															#line 851 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 6007 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleUploadPart()
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleUploadPartActor final : public Actor<Void>, public ActorCallback< HandleUploadPartActor, 0, Void >, public FastAllocated<HandleUploadPartActor>, public HandleUploadPartActorState<HandleUploadPartActor> {
															#line 6012 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleUploadPartActor>::operator new;
	using FastAllocated<HandleUploadPartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(185189453097117440UL, 2914830828606868736UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleUploadPartActor, 0, Void >;
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleUploadPartActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 6029 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleUploadPartActorState<HandleUploadPartActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleUploadPart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2867735766649073408UL, 15512396953593662720UL);
		ActorExecutionContextHelper __helper(static_cast<HandleUploadPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleUploadPart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleUploadPart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleUploadPartActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleUploadPart( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleUploadPartActor(self, req, response, bucket, object, queryParams));
															#line 6061 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 870 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 6066 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleMultipartComplete()
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartCompleteActor>
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartCompleteActorState {
															#line 6072 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartCompleteActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 878 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId"))
															#line 6091 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleMultipartComplete", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartCompleteActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartComplete", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 880 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartComplete").detail("UploadId", uploadId);
															#line 882 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 883 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 6110 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 884 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 885 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleMultipartCompleteActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartCompleteActorState(); static_cast<HandleMultipartCompleteActor*>(this)->destroy(); return 0; }
															#line 6116 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<HandleMultipartCompleteActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleMultipartCompleteActorState();
				static_cast<HandleMultipartCompleteActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 889 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			combinedContent = std::string();
															#line 890 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			for( auto& part : uploadIter->second.parts ) {
															#line 891 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				combinedContent += part.second.second;
															#line 6128 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			}
															#line 894 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartDebug") .detail("UploadId", uploadId) .detail("PartsCount", uploadIter->second.parts.size()) .detail("CombinedSize", combinedContent.size()) .detail("CombinedPreview", combinedContent.size() > 0 ? combinedContent.substr(0, std::min((size_t)20, combinedContent.size())) : "EMPTY");
															#line 903 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ObjectData obj(combinedContent);
															#line 904 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().buckets[bucket][object] = std::move(obj);
															#line 906 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartFinalObject") .detail("UploadId", uploadId) .detail("StoredSize", getGlobalStorage().buckets[bucket][object].content.size()) .detail("StoredPreview", getGlobalStorage().buckets[bucket][object].content.size() > 0 ? getGlobalStorage().buckets[bucket][object].content.substr( 0, std::min((size_t)20, getGlobalStorage().buckets[bucket][object].content.size())) : "EMPTY");
															#line 916 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = persistObject(bucket, object);
															#line 916 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6142 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 1;
															#line 916 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartCompleteActor, 0, Void >*>(static_cast<HandleMultipartCompleteActor*>(this)));
															#line 6147 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartCompleteActorState();
		static_cast<HandleMultipartCompleteActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 919 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().multipartUploads.erase(uploadId);
															#line 920 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedMultipart(uploadId);
															#line 920 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6174 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 2;
															#line 920 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartCompleteActor, 1, Void >*>(static_cast<HandleMultipartCompleteActor*>(this)));
															#line 6179 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 919 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().multipartUploads.erase(uploadId);
															#line 920 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = deletePersistedMultipart(uploadId);
															#line 920 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6192 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 2;
															#line 920 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartCompleteActor, 1, Void >*>(static_cast<HandleMultipartCompleteActor*>(this)));
															#line 6197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartCompleteActor*>(this)->ActorCallback< HandleMultipartCompleteActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartCompleteActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 923 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<CompleteMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <ETag>%s</ETag>\n" "</CompleteMultipartUploadResult>", bucket.c_str(), object.c_str(), getGlobalStorage().buckets[bucket][object].etag.c_str());
															#line 933 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 935 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartCompleted").detail("UploadId", uploadId).detail("FinalSize", combinedContent.size());
															#line 937 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartCompleteActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartCompleteActorState(); static_cast<HandleMultipartCompleteActor*>(this)->destroy(); return 0; }
															#line 6287 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartCompleteActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartCompleteActorState();
		static_cast<HandleMultipartCompleteActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 923 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" "<CompleteMultipartUploadResult>\n" "  <Bucket>%s</Bucket>\n" "  <Key>%s</Key>\n" "  <ETag>%s</ETag>\n" "</CompleteMultipartUploadResult>", bucket.c_str(), object.c_str(), getGlobalStorage().buckets[bucket][object].etag.c_str());
															#line 933 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		self->sendXMLResponse(response, 200, xml);
															#line 935 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartCompleted").detail("UploadId", uploadId).detail("FinalSize", combinedContent.size());
															#line 937 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartCompleteActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartCompleteActorState(); static_cast<HandleMultipartCompleteActor*>(this)->destroy(); return 0; }
															#line 6305 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartCompleteActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartCompleteActorState();
		static_cast<HandleMultipartCompleteActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartCompleteActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartCompleteActor*>(this)->ActorCallback< HandleMultipartCompleteActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< HandleMultipartCompleteActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 878 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 889 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string combinedContent;
															#line 6404 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleMultipartComplete()
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartCompleteActor final : public Actor<Void>, public ActorCallback< HandleMultipartCompleteActor, 0, Void >, public ActorCallback< HandleMultipartCompleteActor, 1, Void >, public FastAllocated<HandleMultipartCompleteActor>, public HandleMultipartCompleteActorState<HandleMultipartCompleteActor> {
															#line 6409 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleMultipartCompleteActor>::operator new;
	using FastAllocated<HandleMultipartCompleteActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8632363371020265728UL, 10206005356870244864UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartCompleteActor, 0, Void >;
friend struct ActorCallback< HandleMultipartCompleteActor, 1, Void >;
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartCompleteActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 6427 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleMultipartCompleteActorState<HandleMultipartCompleteActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartComplete", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7470605948036193280UL, 16274542347142593024UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartCompleteActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartComplete");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartComplete", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartCompleteActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< HandleMultipartCompleteActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartComplete( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartCompleteActor(self, req, response, bucket, object, queryParams));
															#line 6460 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 939 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

																#line 6465 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleMultipartAbort()
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleMultipartAbortActor>
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartAbortActorState {
															#line 6471 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartAbortActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object),
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   queryParams(queryParams),
															#line 947 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   uploadId(queryParams.at("uploadId"))
															#line 6490 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleMultipartAbort", reinterpret_cast<unsigned long>(this));

	}
	~HandleMultipartAbortActorState() 
	{
		fdb_probe_actor_destroy("handleMultipartAbort", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 949 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3MultipartAbort").detail("UploadId", uploadId);
															#line 951 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto uploadIter = getGlobalStorage().multipartUploads.find(uploadId);
															#line 952 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (uploadIter == getGlobalStorage().multipartUploads.end())
															#line 6509 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 953 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchUpload", "Upload not found");
															#line 954 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<HandleMultipartAbortActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartAbortActorState(); static_cast<HandleMultipartAbortActor*>(this)->destroy(); return 0; }
															#line 6515 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<HandleMultipartAbortActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~HandleMultipartAbortActorState();
				static_cast<HandleMultipartAbortActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().multipartUploads.erase(uploadId);
															#line 959 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = deletePersistedMultipart(uploadId);
															#line 959 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6527 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state = 1;
															#line 959 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleMultipartAbortActor, 0, Void >*>(static_cast<HandleMultipartAbortActor*>(this)));
															#line 6532 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleMultipartAbortActorState();
		static_cast<HandleMultipartAbortActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 961 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 962 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 963 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 965 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartAborted").detail("UploadId", uploadId);
															#line 967 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartAbortActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartAbortActorState(); static_cast<HandleMultipartAbortActor*>(this)->destroy(); return 0; }
															#line 6563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartAbortActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartAbortActorState();
		static_cast<HandleMultipartAbortActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 961 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 962 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 963 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 965 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartAborted").detail("UploadId", uploadId);
															#line 967 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleMultipartAbortActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleMultipartAbortActorState(); static_cast<HandleMultipartAbortActor*>(this)->destroy(); return 0; }
															#line 6583 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleMultipartAbortActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleMultipartAbortActorState();
		static_cast<HandleMultipartAbortActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state > 0) static_cast<HandleMultipartAbortActor*>(this)->actor_wait_state = 0;
		static_cast<HandleMultipartAbortActor*>(this)->ActorCallback< HandleMultipartAbortActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleMultipartAbortActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::map<std::string, std::string> queryParams;
															#line 947 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 6680 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleMultipartAbort()
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleMultipartAbortActor final : public Actor<Void>, public ActorCallback< HandleMultipartAbortActor, 0, Void >, public FastAllocated<HandleMultipartAbortActor>, public HandleMultipartAbortActorState<HandleMultipartAbortActor> {
															#line 6685 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleMultipartAbortActor>::operator new;
	using FastAllocated<HandleMultipartAbortActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17129663919060321024UL, 15335462459598554624UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleMultipartAbortActor, 0, Void >;
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleMultipartAbortActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object,std::map<std::string, std::string> const& queryParams) 
															#line 6702 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleMultipartAbortActorState<HandleMultipartAbortActor>(self, req, response, bucket, object, queryParams),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleMultipartAbort", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3855017746569260032UL, 6136799802664629248UL);
		ActorExecutionContextHelper __helper(static_cast<HandleMultipartAbortActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleMultipartAbort");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleMultipartAbort", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleMultipartAbortActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleMultipartAbort( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object, std::map<std::string, std::string> const& queryParams ) {
															#line 940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleMultipartAbortActor(self, req, response, bucket, object, queryParams));
															#line 6734 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 969 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	// Object Tagging Operations
	static Future<Void> handlePutObjectTags(MockS3ServerImpl* self,
	                                        Reference<HTTP::IncomingRequest> req,
	                                        Reference<HTTP::OutgoingResponse> response,
	                                        std::string bucket,
	                                        std::string object) {

		TraceEvent("MockS3PutObjectTags")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagsXML", req->data.content);

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		// Parse tags XML (simplified parser)
		std::map<std::string, std::string> tags = self->parseTagsXML(req->data.content);
		objectIter->second.tags = tags;

		response->code = 200;
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3ObjectTagsSet")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagCount", tags.size());

		return Void();
	}

	static Future<Void> handleGetObjectTags(MockS3ServerImpl* self,
	                                        Reference<HTTP::IncomingRequest> req,
	                                        Reference<HTTP::OutgoingResponse> response,
	                                        std::string bucket,
	                                        std::string object) {

		TraceEvent("MockS3GetObjectTags").detail("Bucket", bucket).detail("Object", object);

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		// Generate tags XML response
		std::string xml = self->generateTagsXML(objectIter->second.tags);
		self->sendXMLResponse(response, 200, xml);

		TraceEvent("MockS3ObjectTagsRetrieved")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("TagCount", objectIter->second.tags.size());

		return Void();
	}

	// Basic Object Operations
																#line 6812 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handlePutObject()
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandlePutObjectActor>
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandlePutObjectActorState {
															#line 6818 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandlePutObjectActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 6833 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handlePutObject", reinterpret_cast<unsigned long>(this));

	}
	~HandlePutObjectActorState() 
	{
		fdb_probe_actor_destroy("handlePutObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1049 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3PutObject_Debug") .detail("Bucket", bucket) .detail("Object", object) .detail("ContentLength", req->data.contentLen) .detail("ContentSize", req->data.content.size()) .detail("ContentPreview", req->data.content.substr(0, std::min(100, (int)req->data.content.size())));
															#line 1056 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ObjectData obj(req->data.content);
															#line 1057 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			etag = obj.etag;
															#line 1058 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			getGlobalStorage().buckets[bucket][object] = std::move(obj);
															#line 1060 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3PutObject_Stored") .detail("Bucket", bucket) .detail("Object", object) .detail("ETag", etag) .detail("StoredSize", getGlobalStorage().buckets[bucket][object].content.size());
															#line 1067 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = persistObject(bucket, object);
															#line 1067 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandlePutObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandlePutObjectActor*>(this)->actor_wait_state = 1;
															#line 1067 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandlePutObjectActor, 0, Void >*>(static_cast<HandlePutObjectActor*>(this)));
															#line 6865 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandlePutObjectActorState();
		static_cast<HandlePutObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1069 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 1070 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 1071 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PutObject_Response") .detail("Bucket", bucket) .detail("Object", object) .detail("ResponseCode", response->code) .detail("ContentLen", response->data.contentLen) .detail("HasContent", response->data.content != nullptr);
															#line 1081 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandlePutObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandlePutObjectActorState(); static_cast<HandlePutObjectActor*>(this)->destroy(); return 0; }
															#line 6896 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandlePutObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandlePutObjectActorState();
		static_cast<HandlePutObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1069 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 200;
															#line 1070 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.headers["ETag"] = etag;
															#line 1071 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3PutObject_Response") .detail("Bucket", bucket) .detail("Object", object) .detail("ResponseCode", response->code) .detail("ContentLen", response->data.contentLen) .detail("HasContent", response->data.content != nullptr);
															#line 1081 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandlePutObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandlePutObjectActorState(); static_cast<HandlePutObjectActor*>(this)->destroy(); return 0; }
															#line 6916 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandlePutObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandlePutObjectActorState();
		static_cast<HandlePutObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandlePutObjectActor*>(this)->actor_wait_state > 0) static_cast<HandlePutObjectActor*>(this)->actor_wait_state = 0;
		static_cast<HandlePutObjectActor*>(this)->ActorCallback< HandlePutObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandlePutObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandlePutObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandlePutObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 1057 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string etag;
															#line 7011 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handlePutObject()
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandlePutObjectActor final : public Actor<Void>, public ActorCallback< HandlePutObjectActor, 0, Void >, public FastAllocated<HandlePutObjectActor>, public HandlePutObjectActorState<HandlePutObjectActor> {
															#line 7016 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandlePutObjectActor>::operator new;
	using FastAllocated<HandlePutObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7965279563671826176UL, 1864694557032560896UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandlePutObjectActor, 0, Void >;
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandlePutObjectActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 7033 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandlePutObjectActorState<HandlePutObjectActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handlePutObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4560658774791891968UL, 15471708591759102464UL);
		ActorExecutionContextHelper __helper(static_cast<HandlePutObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handlePutObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handlePutObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandlePutObjectActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handlePutObject( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 1043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandlePutObjectActor(self, req, response, bucket, object));
															#line 7065 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1083 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	static Future<Void> handleGetObject(MockS3ServerImpl* self,
	                                    Reference<HTTP::IncomingRequest> req,
	                                    Reference<HTTP::OutgoingResponse> response,
	                                    std::string bucket,
	                                    std::string object) {

		TraceEvent("MockS3GetObjectStart").detail("Bucket", bucket).detail("Object", object);

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			TraceEvent("MockS3GetObjectBucketNotFound").detail("Bucket", bucket).detail("Object", object);
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			TraceEvent("MockS3GetObjectNotFound").detail("Bucket", bucket).detail("Object", object);
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		std::string content = objectIter->second.content;
		std::string etag = objectIter->second.etag;
		std::string contentMD5 = HTTP::computeMD5Sum(content);

		TraceEvent("MockS3GetObjectFound")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("ContentSize", content.size())
		    .detail("ETag", etag);

		// Handle HTTP Range header for partial content requests
		// This is essential for AsyncFileEncrypted to read encrypted blocks correctly
		int64_t rangeStart = 0;
		int64_t rangeEnd = static_cast<int64_t>(content.size() - 1);
		bool isRangeRequest = false;

		auto rangeHeader = req->data.headers.find("Range");
		if (rangeHeader != req->data.headers.end()) {
			int64_t parsedStart, parsedEnd;
			if (parseRangeHeader(rangeHeader->second, parsedStart, parsedEnd)) {
				rangeStart = parsedStart;
				if (parsedEnd == -1) {
					// Open-ended range (e.g., "bytes=100-")
					rangeEnd = static_cast<int64_t>(content.size() - 1);
				} else {
					rangeEnd = parsedEnd;
				}
				// Clamp range to actual content size
				int64_t contentSize = static_cast<int64_t>(content.size() - 1);
				rangeEnd = std::min(rangeEnd, contentSize);
				rangeStart = std::min(rangeStart, contentSize);
				if (rangeStart <= rangeEnd) {
					isRangeRequest = true;
				}
			}
		}

		// Extract the requested range
		std::string responseContent;
		if (isRangeRequest && rangeStart <= rangeEnd) {
			responseContent =
			    content.substr(static_cast<size_t>(rangeStart), static_cast<size_t>(rangeEnd - rangeStart + 1));
			response->code = 206; // Partial Content
			response->data.headers["Content-Range"] =
			    format("bytes %lld-%lld/%zu", rangeStart, rangeEnd, content.size());
			// For range requests, calculate MD5 of the partial content, not full content
			contentMD5 = HTTP::computeMD5Sum(responseContent);

			TraceEvent("MockS3GetObjectRangeRequest")
			    .detail("Bucket", bucket)
			    .detail("Object", object)
			    .detail("RangeStart", rangeStart)
			    .detail("RangeEnd", rangeEnd)
			    .detail("TotalSize", content.size())
			    .detail("ResponseSize", responseContent.size());
		} else {
			responseContent = content;
			response->code = 200;

			TraceEvent("MockS3GetObjectFullRequest")
			    .detail("Bucket", bucket)
			    .detail("Object", object)
			    .detail("ResponseSize", responseContent.size());
		}

		response->data.headers["ETag"] = etag;
		response->data.headers["Content-Type"] = "binary/octet-stream";
		response->data.headers["Content-MD5"] = contentMD5;

		// Write content to response
		response->data.contentLen = responseContent.size();
		response->data.headers["Content-Length"] = std::to_string(responseContent.size());
		response->data.content->discardAll(); // Clear existing content

		if (!responseContent.empty()) {
			// Use the correct approach: getWriteBuffer from the UnsentPacketQueue
			PacketBuffer* buffer = response->data.content->getWriteBuffer(responseContent.size());
			PacketWriter pw(buffer, nullptr, Unversioned());
			pw.serializeBytes(responseContent);
			pw.finish();
		}

		TraceEvent("MockS3GetObjectComplete")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("ResponseCode", response->code)
		    .detail("ResponseSize", responseContent.size());

		return Void();
	}

																#line 7183 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
// This generated class is to be used only via handleDeleteObject()
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class HandleDeleteObjectActor>
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleDeleteObjectActorState {
															#line 7189 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleDeleteObjectActorState(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : self(self),
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   req(req),
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   response(response),
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   bucket(bucket),
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   object(object)
															#line 7204 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("handleDeleteObject", reinterpret_cast<unsigned long>(this));

	}
	~HandleDeleteObjectActorState() 
	{
		fdb_probe_actor_destroy("handleDeleteObject", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1203 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3DeleteObject").detail("Bucket", bucket).detail("Object", object);
															#line 1205 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			auto bucketIter = getGlobalStorage().buckets.find(bucket);
															#line 1206 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (bucketIter != getGlobalStorage().buckets.end())
															#line 7223 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1207 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIter->second.erase(object);
															#line 7227 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			}
															#line 1211 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<Void> __when_expr_0 = deletePersistedObject(bucket, object);
															#line 1211 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7233 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state = 1;
															#line 1211 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleDeleteObjectActor, 0, Void >*>(static_cast<HandleDeleteObjectActor*>(this)));
															#line 7238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleDeleteObjectActorState();
		static_cast<HandleDeleteObjectActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1213 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 1214 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1215 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 1217 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 1219 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleDeleteObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleDeleteObjectActorState(); static_cast<HandleDeleteObjectActor*>(this)->destroy(); return 0; }
															#line 7269 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleDeleteObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleDeleteObjectActorState();
		static_cast<HandleDeleteObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1213 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->code = 204;
															#line 1214 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.contentLen = 0;
															#line 1215 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		response->data.content->discardAll();
															#line 1217 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectDeleted").detail("Bucket", bucket).detail("Object", object);
															#line 1219 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<HandleDeleteObjectActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~HandleDeleteObjectActorState(); static_cast<HandleDeleteObjectActor*>(this)->destroy(); return 0; }
															#line 7289 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<HandleDeleteObjectActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~HandleDeleteObjectActorState();
		static_cast<HandleDeleteObjectActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state > 0) static_cast<HandleDeleteObjectActor*>(this)->actor_wait_state = 0;
		static_cast<HandleDeleteObjectActor*>(this)->ActorCallback< HandleDeleteObjectActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleDeleteObjectActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3ServerImpl* self;
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::IncomingRequest> req;
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::OutgoingResponse> response;
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string object;
															#line 7382 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via handleDeleteObject()
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class HandleDeleteObjectActor final : public Actor<Void>, public ActorCallback< HandleDeleteObjectActor, 0, Void >, public FastAllocated<HandleDeleteObjectActor>, public HandleDeleteObjectActorState<HandleDeleteObjectActor> {
															#line 7387 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<HandleDeleteObjectActor>::operator new;
	using FastAllocated<HandleDeleteObjectActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(442867593906376960UL, 9926672506457285120UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleDeleteObjectActor, 0, Void >;
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	HandleDeleteObjectActor(MockS3ServerImpl* const& self,Reference<HTTP::IncomingRequest> const& req,Reference<HTTP::OutgoingResponse> const& response,std::string const& bucket,std::string const& object) 
															#line 7404 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   HandleDeleteObjectActorState<HandleDeleteObjectActor>(self, req, response, bucket, object),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleDeleteObject", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7788094060450111232UL, 5901935141690097664UL);
		ActorExecutionContextHelper __helper(static_cast<HandleDeleteObjectActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleDeleteObject");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleDeleteObject", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleDeleteObjectActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> handleDeleteObject( MockS3ServerImpl* const& self, Reference<HTTP::IncomingRequest> const& req, Reference<HTTP::OutgoingResponse> const& response, std::string const& bucket, std::string const& object ) {
															#line 1197 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new HandleDeleteObjectActor(self, req, response, bucket, object));
															#line 7436 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1221 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

	static Future<Void> handleHeadObject(MockS3ServerImpl* self,
	                                     Reference<HTTP::IncomingRequest> req,
	                                     Reference<HTTP::OutgoingResponse> response,
	                                     std::string bucket,
	                                     std::string object) {

		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			TraceEvent("MockS3HeadObjectNoBucket")
			    .detail("Bucket", bucket)
			    .detail("Object", object)
			    .detail("AvailableBuckets", getGlobalStorage().buckets.size());
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		auto objectIter = bucketIter->second.find(object);
		if (objectIter == bucketIter->second.end()) {
			TraceEvent("MockS3HeadObjectNoObject")
			    .detail("Bucket", bucket)
			    .detail("Object", object)
			    .detail("ObjectsInBucket", bucketIter->second.size());
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchKey", "Object not found");
			return Void();
		}

		const ObjectData& obj = objectIter->second;
		std::string etag = obj.etag;
		size_t contentSize = obj.content.size();
		std::string preview = contentSize > 0 ? obj.content.substr(0, std::min((size_t)20, contentSize)) : "EMPTY";

		TraceEvent("MockS3HeadObjectFound")
		    .detail("Bucket", bucket)
		    .detail("Object", object)
		    .detail("Size", contentSize)
		    .detail("Preview", preview);

		response->code = 200;
		response->data.headers["ETag"] = etag;
		response->data.headers["Content-Length"] = std::to_string(contentSize);
		response->data.headers["Content-Type"] = "binary/octet-stream";
		// HEAD requests need contentLen set to actual size for headers
		response->data.contentLen = contentSize; // This controls ResponseContentSize in HTTP logs

		return Void();
	}

	// S3 ListObjects Operation
	static Future<Void> handleListObjects(MockS3ServerImpl* self,
	                                      Reference<HTTP::IncomingRequest> req,
	                                      Reference<HTTP::OutgoingResponse> response,
	                                      std::string bucket,
	                                      std::map<std::string, std::string> queryParams) {

		TraceEvent("MockS3ListObjects").detail("Bucket", bucket).detail("QueryParamCount", queryParams.size());

		// Get query parameters for listing
		std::string prefix = queryParams.count("prefix") ? queryParams.at("prefix") : "";
		std::string delimiter = queryParams.count("delimiter") ? queryParams.at("delimiter") : "";
		std::string marker = queryParams.count("marker") ? queryParams.at("marker") : "";
		std::string continuationToken =
		    queryParams.count("continuation-token") ? queryParams.at("continuation-token") : "";
		int maxKeys = queryParams.count("max-keys") ? std::stoi(queryParams.at("max-keys")) : 1000;

		TraceEvent("MockS3ListObjectsDebug")
		    .detail("Bucket", bucket)
		    .detail("Prefix", prefix)
		    .detail("Delimiter", delimiter)
		    .detail("Marker", marker)
		    .detail("ContinuationToken", continuationToken)
		    .detail("MaxKeys", maxKeys);

		// Find bucket
		auto bucketIter = getGlobalStorage().buckets.find(bucket);
		if (bucketIter == getGlobalStorage().buckets.end()) {
			self->sendError(response, HTTP::HTTP_STATUS_CODE_NOT_FOUND, "NoSuchBucket", "Bucket not found");
			return Void();
		}

		// Collect all matching objects first
		std::vector<std::pair<std::string, const ObjectData*>> matchingObjects;
		for (const auto& objectPair : bucketIter->second) {
			const std::string& objectName = objectPair.first;
			const ObjectData& objectData = objectPair.second;

			// Apply prefix filter
			if (!prefix.empty() && objectName.find(prefix) != 0) {
				continue;
			}

			matchingObjects.push_back({ objectName, &objectData });
		}

		// Sort objects by name for consistent pagination
		std::sort(matchingObjects.begin(), matchingObjects.end());

		// Find starting point for pagination
		size_t startIndex = 0;
		if (!marker.empty()) {
			for (size_t i = 0; i < matchingObjects.size(); i++) {
				if (matchingObjects[i].first > marker) {
					startIndex = i;
					break;
				}
			}
		} else if (!continuationToken.empty()) {
			// Simple continuation token implementation (just use the last object name)
			for (size_t i = 0; i < matchingObjects.size(); i++) {
				if (matchingObjects[i].first > continuationToken) {
					startIndex = i;
					break;
				}
			}
		}

		// Build list of objects for this page
		std::string xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<ListBucketResult>\n";
		xml += "<Name>" + bucket + "</Name>\n";
		xml += "<Prefix>" + prefix + "</Prefix>\n";
		xml += "<MaxKeys>" + std::to_string(maxKeys) + "</MaxKeys>\n";

		if (!marker.empty()) {
			xml += "<Marker>" + marker + "</Marker>\n";
		}

		int count = 0;
		std::string lastKey;
		size_t totalMatching = matchingObjects.size();

		for (size_t i = startIndex; i < matchingObjects.size() && count < maxKeys; i++) {
			const std::string& objectName = matchingObjects[i].first;
			const ObjectData* objectData = matchingObjects[i].second;

			xml += "<Contents>\n";
			xml += "<Key>" + objectName + "</Key>\n";
			xml += "<LastModified>" + std::to_string((int64_t)objectData->lastModified) + "</LastModified>\n";
			xml += "<ETag>" + objectData->etag + "</ETag>\n";
			xml += "<Size>" + std::to_string(objectData->content.size()) + "</Size>\n";
			xml += "<StorageClass>STANDARD</StorageClass>\n";
			xml += "</Contents>\n";

			lastKey = objectName;
			count++;
		}

		// Determine if there are more results
		bool isTruncated = (startIndex + count) < totalMatching;
		xml += "<IsTruncated>" + std::string(isTruncated ? "true" : "false") + "</IsTruncated>\n";

		if (isTruncated && !lastKey.empty()) {
			xml += "<NextMarker>" + lastKey + "</NextMarker>\n";
		}

		xml += "</ListBucketResult>";

		self->sendXMLResponse(response, 200, xml);

		TraceEvent("MockS3ListObjectsCompleted")
		    .detail("Bucket", bucket)
		    .detail("Prefix", prefix)
		    .detail("ObjectCount", count)
		    .detail("StartIndex", startIndex)
		    .detail("TotalMatching", totalMatching)
		    .detail("IsTruncated", isTruncated)
		    .detail("NextMarker", isTruncated ? lastKey : "");

		return Void();
	}

	// S3 Bucket Operations
	static Future<Void> handleHeadBucket(MockS3ServerImpl* self,
	                                     Reference<HTTP::IncomingRequest> req,
	                                     Reference<HTTP::OutgoingResponse> response,
	                                     std::string bucket) {

		TraceEvent("MockS3HeadBucket").detail("Bucket", bucket);

		// Ensure bucket exists in our storage (implicit creation like real S3)
		if (getGlobalStorage().buckets.find(bucket) == getGlobalStorage().buckets.end()) {
			getGlobalStorage().buckets[bucket] = std::map<std::string, ObjectData>();
		}

		response->code = 200;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3BucketHead").detail("Bucket", bucket);

		return Void();
	}

	static Future<Void> handlePutBucket(MockS3ServerImpl* self,
	                                    Reference<HTTP::IncomingRequest> req,
	                                    Reference<HTTP::OutgoingResponse> response,
	                                    std::string bucket) {

		TraceEvent("MockS3PutBucket").detail("Bucket", bucket);

		// Ensure bucket exists in our storage (implicit creation)
		if (getGlobalStorage().buckets.find(bucket) == getGlobalStorage().buckets.end()) {
			getGlobalStorage().buckets[bucket] = std::map<std::string, ObjectData>();
		}

		response->code = 200;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.contentLen = 0;
		response->data.content->discardAll(); // Clear existing content

		TraceEvent("MockS3BucketCreated").detail("Bucket", bucket);

		return Void();
	}

	// Utility Methods

	void sendError(Reference<HTTP::OutgoingResponse> response,
	               int code,
	               const std::string& errorCode,
	               const std::string& message) {

		TraceEvent("MockS3Error").detail("Code", code).detail("ErrorCode", errorCode).detail("Message", message);

		std::string xml = format("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
		                         "<Error>\n"
		                         "  <Code>%s</Code>\n"
		                         "  <Message>%s</Message>\n"
		                         "</Error>",
		                         errorCode.c_str(),
		                         message.c_str());

		sendXMLResponse(response, code, xml);
	}

	void sendXMLResponse(Reference<HTTP::OutgoingResponse> response, int code, const std::string& xml) {
		TraceEvent("MockS3SendXMLResponse_Start")
		    .detail("Code", code)
		    .detail("XMLSize", xml.size())
		    .detail("XMLPreview", xml.size() > 0 ? xml.substr(0, std::min((size_t)50, xml.size())) : "EMPTY");

		response->code = code;
		response->data.headers["Content-Type"] = "application/xml";
		response->data.headers["Content-Length"] = std::to_string(xml.size());
		response->data.headers["Content-MD5"] = HTTP::computeMD5Sum(xml);

		// Actually put the XML content into the response
		if (xml.empty()) {
			response->data.contentLen = 0;
			TraceEvent("MockS3SendXMLResponse_Empty").detail("ResponseCode", response->code);
		} else {
			// Use the existing content queue instead of creating a new one
			// This prevents memory management issues and potential canBeSet() failures
			size_t contentSize = xml.size();
			response->data.contentLen = contentSize;

			// Clear any existing content and write the XML
			response->data.content->discardAll();
			PacketBuffer* buffer = response->data.content->getWriteBuffer(contentSize);
			PacketWriter pw(buffer, nullptr, Unversioned());
			pw.serializeBytes(xml);
			pw.finish();
		}

		TraceEvent("MockS3SendXMLResponse_Complete")
		    .detail("FinalCode", response->code)
		    .detail("FinalContentLen", response->data.contentLen)
		    .detail("XMLSize", xml.size());
	}

	std::map<std::string, std::string> parseTagsXML(const std::string& xml) {
		std::map<std::string, std::string> tags;

		// Simplified XML parsing for tags - this would need a proper XML parser in production
		std::regex tagRegex("<Tag><Key>([^<]+)</Key><Value>([^<]*)</Value></Tag>");
		std::sregex_iterator iter(xml.begin(), xml.end(), tagRegex);
		std::sregex_iterator end;

		for (; iter != end; ++iter) {
			std::string key = iter->str(1);
			std::string value = iter->str(2);
			tags[key] = value;

			TraceEvent("MockS3ParsedTag").detail("Key", key).detail("Value", value);
		}

		return tags;
	}

	std::string generateTagsXML(const std::map<std::string, std::string>& tags) {
		std::string xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<Tagging><TagSet>";

		for (const auto& tag : tags) {
			xml += "<Tag><Key>" + tag.first + "</Key><Value>" + tag.second + "</Value></Tag>";
		}

		xml += "</TagSet></Tagging>";
		return xml;
	}
};

// Global registry to track registered servers and avoid conflicts
static std::map<std::string, bool> registeredServers;

// Clear global storage state for clean test runs
static void clearSingletonState() {
	getGlobalStorage().buckets.clear();
	getGlobalStorage().multipartUploads.clear();
	TraceEvent("MockS3ServerImpl_StateCleared");
}

// Process a Mock S3 request directly (for wrapping/chaos injection)
Future<Void> processMockS3Request(Reference<HTTP::IncomingRequest> req, Reference<HTTP::OutgoingResponse> response) {
	static MockS3ServerImpl serverInstance;
	return MockS3ServerImpl::handleRequest(&serverInstance, req, response);
}

// Request Handler Implementation - Each handler instance works with global storage
Future<Void> MockS3RequestHandler::handleRequest(Reference<HTTP::IncomingRequest> req,
                                                 Reference<HTTP::OutgoingResponse> response) {
	// Guard against calling virtual functions during destruction
	if (destructing.load()) {
		TraceEvent(SevWarn, "MockS3RequestHandler_DestructingGuard")
		    .detail("Method", "handleRequest")
		    .detail("Resource", req->resource);
		return Void();
	}

	return processMockS3Request(req, response);
}

Reference<HTTP::IRequestHandler> MockS3RequestHandler::clone() {
	// Guard against calling virtual functions during destruction
	if (destructing.load()) {
		TraceEvent(SevWarn, "MockS3RequestHandler_DestructingGuard").detail("Method", "clone");
		// Return nullptr - caller must handle this gracefully
		return Reference<HTTP::IRequestHandler>();
	}
	return makeReference<MockS3RequestHandler>();
}

// Safe server registration that prevents conflicts (internal implementation)
															#line 7782 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via registerMockS3Server_impl()
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class RegisterMockS3Server_implActor>
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class RegisterMockS3Server_implActorState {
															#line 7789 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	RegisterMockS3Server_implActorState(std::string const& ip,std::string const& port) 
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : ip(ip),
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   port(port),
															#line 1564 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   serverKey(ip + ":" + port)
															#line 7800 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this));

	}
	~RegisterMockS3Server_implActorState() 
	{
		fdb_probe_actor_destroy("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1567 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Registration Start") .detail("IP", ip) .detail("Port", port) .detail("ServerKey", serverKey) .detail("IsSimulated", g_network->isSimulated()) .detail("AlreadyRegistered", registeredServers.count(serverKey) > 0);
															#line 1576 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (registeredServers.count(serverKey))
															#line 7817 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1577 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent(SevWarn, "MockS3ServerAlreadyRegistered").detail("Address", serverKey);
															#line 1578 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<RegisterMockS3Server_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterMockS3Server_implActorState(); static_cast<RegisterMockS3Server_implActor*>(this)->destroy(); return 0; }
															#line 7823 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<RegisterMockS3Server_implActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~RegisterMockS3Server_implActorState();
				static_cast<RegisterMockS3Server_implActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1582 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Calling registerSimHTTPServer") .detail("Address", serverKey);
															#line 1586 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = g_simulator->registerSimHTTPServer(ip, port, makeReference<MockS3RequestHandler>());
															#line 1586 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7836 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 1;
															#line 1586 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RegisterMockS3Server_implActor, 0, Void >*>(static_cast<RegisterMockS3Server_implActor*>(this)));
															#line 7841 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RegisterMockS3Server_implActorState();
		static_cast<RegisterMockS3Server_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1607 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<RegisterMockS3Server_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RegisterMockS3Server_implActorState(); static_cast<RegisterMockS3Server_implActor*>(this)->destroy(); return 0; }
															#line 7870 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<RegisterMockS3Server_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RegisterMockS3Server_implActorState();
		static_cast<RegisterMockS3Server_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1599 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3ServerRegistrationFailed") .error(e) .detail("Address", serverKey) .detail("ErrorCode", e.code()) .detail("ErrorName", e.name());
															#line 1604 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 7885 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1587 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		registeredServers[serverKey] = true;
															#line 1590 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = initializeMockS3Persistence(serverKey);
															#line 1590 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7903 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 2;
															#line 1590 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RegisterMockS3Server_implActor, 1, Void >*>(static_cast<RegisterMockS3Server_implActor*>(this)));
															#line 7908 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1587 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		registeredServers[serverKey] = true;
															#line 1590 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = initializeMockS3Persistence(serverKey);
															#line 1590 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 7921 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 2;
															#line 1590 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RegisterMockS3Server_implActor, 1, Void >*>(static_cast<RegisterMockS3Server_implActor*>(this)));
															#line 7926 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state > 0) static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterMockS3Server_implActor*>(this)->ActorCallback< RegisterMockS3Server_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RegisterMockS3Server_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1592 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerRegistered").detail("Address", serverKey).detail("Success", true);
															#line 1594 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Registration Complete") .detail("Address", serverKey) .detail("TotalRegistered", registeredServers.size());
															#line 8012 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1592 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerRegistered").detail("Address", serverKey).detail("Success", true);
															#line 1594 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerDiagnostic") .detail("Phase", "Registration Complete") .detail("Address", serverKey) .detail("TotalRegistered", registeredServers.size());
															#line 8023 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state > 0) static_cast<RegisterMockS3Server_implActor*>(this)->actor_wait_state = 0;
		static_cast<RegisterMockS3Server_implActor*>(this)->ActorCallback< RegisterMockS3Server_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RegisterMockS3Server_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string ip;
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string port;
															#line 1564 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string serverKey;
															#line 8122 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via registerMockS3Server_impl()
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class RegisterMockS3Server_implActor final : public Actor<Void>, public ActorCallback< RegisterMockS3Server_implActor, 0, Void >, public ActorCallback< RegisterMockS3Server_implActor, 1, Void >, public FastAllocated<RegisterMockS3Server_implActor>, public RegisterMockS3Server_implActorState<RegisterMockS3Server_implActor> {
															#line 8127 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<RegisterMockS3Server_implActor>::operator new;
	using FastAllocated<RegisterMockS3Server_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13980894035001152000UL, 43096422522094592UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RegisterMockS3Server_implActor, 0, Void >;
friend struct ActorCallback< RegisterMockS3Server_implActor, 1, Void >;
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	RegisterMockS3Server_implActor(std::string const& ip,std::string const& port) 
															#line 8145 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   RegisterMockS3Server_implActorState<RegisterMockS3Server_implActor>(ip, port),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13948469852865528320UL, 9795983047084281344UL);
		ActorExecutionContextHelper __helper(static_cast<RegisterMockS3Server_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("registerMockS3Server_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("registerMockS3Server_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RegisterMockS3Server_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RegisterMockS3Server_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> registerMockS3Server_impl( std::string const& ip, std::string const& port ) {
															#line 1563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new RegisterMockS3Server_implActor(ip, port));
															#line 8179 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1609 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Public Interface Implementation
															#line 8185 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via startMockS3Server()
															#line 1611 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class StartMockS3ServerActor>
															#line 1611 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerActorState {
															#line 8192 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1611 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerActorState(NetworkAddress const& listenAddress) 
															#line 1611 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1611 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : listenAddress(listenAddress)
															#line 8199 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("startMockS3Server", reinterpret_cast<unsigned long>(this));

	}
	~StartMockS3ServerActorState() 
	{
		fdb_probe_actor_destroy("startMockS3Server", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1612 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerStarting").detail("ListenAddress", listenAddress.toString());
															#line 8214 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1615 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerRegistering") .detail("IP", listenAddress.ip.toString()) .detail("Port", std::to_string(listenAddress.port)) .detail("IsSimulated", g_network->isSimulated());
															#line 1621 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = registerMockS3Server_impl(listenAddress.ip.toString(), std::to_string(listenAddress.port));
															#line 1621 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<StartMockS3ServerActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 8222 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<StartMockS3ServerActor*>(this)->actor_wait_state = 1;
															#line 1621 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerActor, 0, Void >*>(static_cast<StartMockS3ServerActor*>(this)));
															#line 8227 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StartMockS3ServerActorState();
		static_cast<StartMockS3ServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1632 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerActorState(); static_cast<StartMockS3ServerActor*>(this)->destroy(); return 0; }
															#line 8256 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<StartMockS3ServerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerActorState();
		static_cast<StartMockS3ServerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1628 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3ServerStartError").error(e).detail("ListenAddress", listenAddress.toString());
															#line 1629 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 8271 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1623 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerStarted") .detail("ListenAddress", listenAddress.toString()) .detail("HandlerCreated", true);
															#line 8285 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1623 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerStarted") .detail("ListenAddress", listenAddress.toString()) .detail("HandlerCreated", true);
															#line 8294 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StartMockS3ServerActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerActor*>(this)->ActorCallback< StartMockS3ServerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1611 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	NetworkAddress listenAddress;
															#line 8389 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via startMockS3Server()
															#line 1611 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerActor final : public Actor<Void>, public ActorCallback< StartMockS3ServerActor, 0, Void >, public FastAllocated<StartMockS3ServerActor>, public StartMockS3ServerActorState<StartMockS3ServerActor> {
															#line 8394 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<StartMockS3ServerActor>::operator new;
	using FastAllocated<StartMockS3ServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15237250613113131264UL, 1691277130271915776UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StartMockS3ServerActor, 0, Void >;
															#line 1611 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerActor(NetworkAddress const& listenAddress) 
															#line 8411 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   StartMockS3ServerActorState<StartMockS3ServerActor>(listenAddress),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("startMockS3Server", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16442790091880884736UL, 12597676879247326208UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("startMockS3Server");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("startMockS3Server", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StartMockS3ServerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1611 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> startMockS3Server( NetworkAddress const& listenAddress ) {
															#line 1611 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new StartMockS3ServerActor(listenAddress));
															#line 8444 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1634 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Clear all MockS3 global storage - called at the start of each simulation test
void clearMockS3Storage() {
	getGlobalStorage().clearStorage();
	// Note: Do NOT clear chaos server registry here - it must persist across tests
	// like the simulator's httpHandlers map, to prevent duplicate registration attempts
}

// Enable persistence for MockS3 storage
void enableMockS3Persistence(const std::string& persistenceDir) {
	getGlobalStorage().enablePersistence(persistenceDir);
	TraceEvent("MockS3PersistenceConfigured").detail("Directory", persistenceDir);
}

// ACTOR: Load persisted objects from disk
															#line 8463 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via loadPersistedObjects()
															#line 1649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadPersistedObjectsActor>
															#line 1649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedObjectsActorState {
															#line 8470 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedObjectsActorState(std::string const& persistenceDir) 
															#line 1649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir(persistenceDir),
															#line 1650 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   objectsDir(persistenceDir + "/objects")
															#line 8479 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("loadPersistedObjects", reinterpret_cast<unsigned long>(this));

	}
	~LoadPersistedObjectsActorState() 
	{
		fdb_probe_actor_destroy("loadPersistedObjects", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1652 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(objectsDir))
															#line 8494 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1653 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3LoadObjects").detail("Status", "NoObjectsDir");
															#line 1654 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadPersistedObjectsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedObjectsActorState(); static_cast<LoadPersistedObjectsActor*>(this)->destroy(); return 0; }
															#line 8500 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<LoadPersistedObjectsActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadPersistedObjectsActorState();
				static_cast<LoadPersistedObjectsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1659 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				buckets = platform::listFiles(objectsDir, "");
															#line 1661 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::sort(buckets.begin(), buckets.end());
															#line 1662 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIdx = 0;
															#line 1664 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				bucketIdx = 0;
															#line 8515 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadPersistedObjectsActorState();
		static_cast<LoadPersistedObjectsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1719 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadPersistedObjectsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedObjectsActorState(); static_cast<LoadPersistedObjectsActor*>(this)->destroy(); return 0; }
															#line 8544 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<LoadPersistedObjectsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadPersistedObjectsActorState();
		static_cast<LoadPersistedObjectsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1716 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadObjectsFailed").error(e);
															#line 8557 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1714 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectsLoaded").detail("BucketsCount", getGlobalStorage().buckets.size());
															#line 8572 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1664 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(bucketIdx < buckets.size()))
															#line 8588 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1665 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucket = buckets[bucketIdx];
															#line 1666 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (bucket == "." || bucket == "..")
															#line 8596 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1continue1(loopDepth); // continue
		}
															#line 1669 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketDir = objectsDir + "/" + bucket;
															#line 1670 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!directoryExists(bucketDir))
															#line 8604 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1continue1(loopDepth); // continue
		}
															#line 1674 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		files = platform::listFiles(bucketDir, "");
															#line 1675 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::sort(files.begin(), files.end());
															#line 1676 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx = 0;
															#line 1678 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx = 0;
															#line 8616 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1continue1(int loopDepth) 
	{
															#line 1664 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketIdx++;
															#line 8638 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1664 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		bucketIdx++;
															#line 8647 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 1678 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(fileIdx < files.size()))
															#line 8663 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1679 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileName = files[fileIdx];
															#line 1682 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (fileName.size() > OBJECT_META_SUFFIX_LEN && fileName.substr(fileName.size() - OBJECT_META_SUFFIX_LEN) == OBJECT_META_SUFFIX)
															#line 8671 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 1685 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			objectName = fileName.substr(0, fileName.size() - OBJECT_META_SUFFIX_LEN);
															#line 1686 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			dataPath = bucketDir + "/" + objectName + OBJECT_DATA_SUFFIX;
															#line 1687 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			metaPath = bucketDir + "/" + fileName;
															#line 1689 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(dataPath))
															#line 8681 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1690 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent(SevWarn, "MockS3LoadObjectSkipped") .detail("Bucket", bucket) .detail("Object", objectName) .detail("Reason", "NoDataFile");
															#line 8685 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				return a_body1loopBody1continue1(loopDepth); // continue
			}
															#line 1698 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<std::string> __when_expr_0 = readFileContent(dataPath);
															#line 1698 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8692 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 1;
															#line 1698 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedObjectsActor, 0, std::string >*>(static_cast<LoadPersistedObjectsActor*>(this)));
															#line 8697 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1continue1(int loopDepth) 
	{
															#line 1678 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 8724 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 1678 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 8733 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(int loopDepth) 
	{
															#line 1699 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_1 = readFileContent(metaPath);
															#line 1699 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 8744 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 2;
															#line 1699 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedObjectsActor, 1, std::string >*>(static_cast<LoadPersistedObjectsActor*>(this)));
															#line 8749 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(std::string const& __content,int loopDepth) 
	{
															#line 1698 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		content = __content;
															#line 8758 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(std::string && __content,int loopDepth) 
	{
		content = std::move(__content);
		loopDepth = a_body1loopBody1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedObjectsActor*>(this)->ActorCallback< LoadPersistedObjectsActor, 0, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadPersistedObjectsActor, 0, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont5(int loopDepth) 
	{
															#line 1702 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		MockS3GlobalStorage::ObjectData obj(content);
															#line 1703 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		deserializeObjectMeta(metaJson, obj);
															#line 1704 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().buckets[bucket][objectName] = std::move(obj);
															#line 1706 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ObjectRestored") .detail("Bucket", bucket) .detail("Object", objectName) .detail("Size", content.size());
															#line 8843 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(std::string const& __metaJson,int loopDepth) 
	{
															#line 1699 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		metaJson = __metaJson;
															#line 8852 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3when1(std::string && __metaJson,int loopDepth) 
	{
		metaJson = std::move(__metaJson);
		loopDepth = a_body1loopBody1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedObjectsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedObjectsActor*>(this)->ActorCallback< LoadPersistedObjectsActor, 1, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadPersistedObjectsActor, 1, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 1650 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string objectsDir;
															#line 1659 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> buckets;
															#line 1662 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int bucketIdx;
															#line 1665 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucket;
															#line 1669 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string bucketDir;
															#line 1674 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> files;
															#line 1676 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int fileIdx;
															#line 1679 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string fileName;
															#line 1685 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string objectName;
															#line 1686 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string dataPath;
															#line 1687 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaPath;
															#line 1698 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string content;
															#line 1699 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string metaJson;
															#line 8968 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via loadPersistedObjects()
															#line 1649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedObjectsActor final : public Actor<Void>, public ActorCallback< LoadPersistedObjectsActor, 0, std::string >, public ActorCallback< LoadPersistedObjectsActor, 1, std::string >, public FastAllocated<LoadPersistedObjectsActor>, public LoadPersistedObjectsActorState<LoadPersistedObjectsActor> {
															#line 8973 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<LoadPersistedObjectsActor>::operator new;
	using FastAllocated<LoadPersistedObjectsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5143274530972582144UL, 16573649396292898048UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadPersistedObjectsActor, 0, std::string >;
friend struct ActorCallback< LoadPersistedObjectsActor, 1, std::string >;
															#line 1649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedObjectsActor(std::string const& persistenceDir) 
															#line 8991 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   LoadPersistedObjectsActorState<LoadPersistedObjectsActor>(persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadPersistedObjects", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15447654064422368256UL, 7597335790883620352UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedObjectsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadPersistedObjects");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadPersistedObjects", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadPersistedObjectsActor, 0, std::string >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadPersistedObjectsActor, 1, std::string >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadPersistedObjects( std::string const& persistenceDir ) {
															#line 1649 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadPersistedObjectsActor(persistenceDir));
															#line 9025 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1721 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Load persisted multipart uploads from disk
															#line 9031 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via loadPersistedMultipartUploads()
															#line 1723 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadPersistedMultipartUploadsActor>
															#line 1723 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedMultipartUploadsActorState {
															#line 9038 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1723 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedMultipartUploadsActorState(std::string const& persistenceDir) 
															#line 1723 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1723 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir(persistenceDir),
															#line 1724 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   multipartDir(persistenceDir + "/multipart")
															#line 9047 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this));

	}
	~LoadPersistedMultipartUploadsActorState() 
	{
		fdb_probe_actor_destroy("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1726 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!fileExists(multipartDir))
															#line 9062 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1727 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3LoadMultipart").detail("Status", "NoMultipartDir");
															#line 1728 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedMultipartUploadsActorState(); static_cast<LoadPersistedMultipartUploadsActor*>(this)->destroy(); return 0; }
															#line 9068 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadPersistedMultipartUploadsActorState();
				static_cast<LoadPersistedMultipartUploadsActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
			try {
															#line 1733 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				files = platform::listFiles(multipartDir, "");
															#line 1734 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				std::sort(files.begin(), files.end());
															#line 1735 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				fileIdx = 0;
															#line 1737 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				fileIdx = 0;
															#line 9083 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadPersistedMultipartUploadsActorState();
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1803 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadPersistedMultipartUploadsActorState(); static_cast<LoadPersistedMultipartUploadsActor*>(this)->destroy(); return 0; }
															#line 9112 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<LoadPersistedMultipartUploadsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadPersistedMultipartUploadsActorState();
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1800 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadMultipartFailed").error(e);
															#line 9125 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 1798 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartUploadsLoaded").detail("UploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9140 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1737 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(fileIdx < files.size()))
															#line 9156 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1738 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileName = files[fileIdx];
															#line 1741 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (fileName.size() > MULTIPART_STATE_SUFFIX_LEN && fileName.substr(fileName.size() - MULTIPART_STATE_SUFFIX_LEN) == MULTIPART_STATE_SUFFIX)
															#line 9164 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 1743 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			uploadId = fileName.substr(0, fileName.size() - MULTIPART_STATE_SUFFIX_LEN);
															#line 1744 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			statePath = multipartDir + "/" + fileName;
															#line 1747 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			StrictFuture<std::string> __when_expr_0 = readFileContent(statePath);
															#line 1747 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9174 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 1;
															#line 1747 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9179 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1continue1(int loopDepth) 
	{
															#line 1737 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 9206 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1737 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		fileIdx++;
															#line 9215 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1748 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (stateJson.empty())
															#line 9224 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
															#line 1749 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevWarn, "MockS3LoadMultipartSkipped") .detail("UploadId", uploadId) .detail("Reason", "EmptyStateFile");
															#line 9228 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			return a_body1continue1(loopDepth); // continue
		}
															#line 1756 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload = MockS3GlobalStorage::MultipartUpload("", "");
															#line 1757 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload.uploadId = uploadId;
															#line 1758 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		deserializeMultipartState(stateJson, upload);
															#line 1761 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 1762 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		maxAttempts = 10000;
															#line 1763 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum = 1;
															#line 9243 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::string const& __stateJson,int loopDepth) 
	{
															#line 1747 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		stateJson = __stateJson;
															#line 9252 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::string && __stateJson,int loopDepth) 
	{
		stateJson = std::move(__stateJson);
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1788 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartUploadRestored") .detail("UploadId", uploadId) .detail("Bucket", upload.bucket) .detail("Object", upload.object) .detail("PartsCount", upload.parts.size());
															#line 1794 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().multipartUploads[uploadId] = std::move(upload);
															#line 9333 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont3loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1(int loopDepth) 
	{
															#line 1763 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!(partNum <= maxAttempts))
															#line 9349 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1764 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partPath = multipartDir + "/" + uploadId + ".part." + std::to_string(partNum);
															#line 1765 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaPath = partPath + ".meta.json";
															#line 1767 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!fileExists(partPath) || !fileExists(partMetaPath))
															#line 9359 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		{
			return a_body1loopBody1cont3break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1771 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_1 = readFileContent(partPath);
															#line 1771 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 9367 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 2;
															#line 1771 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9372 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1(int loopDepth) 
	{
															#line 1772 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<std::string> __when_expr_2 = readFileContent(partMetaPath);
															#line 1772 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 9396 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont3loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 3;
															#line 1772 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*>(static_cast<LoadPersistedMultipartUploadsActor*>(this)));
															#line 9401 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(std::string const& __partData,int loopDepth) 
	{
															#line 1771 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partData = __partData;
															#line 9410 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1when1(std::string && __partData,int loopDepth) 
	{
		partData = std::move(__partData);
		loopDepth = a_body1loopBody1cont3loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3loopBody1cont4(int loopDepth) 
	{
															#line 1775 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		using namespace rapidjson;
															#line 1776 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		Document doc;
															#line 1777 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		doc.Parse(partMetaJson.c_str());
															#line 1778 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		std::string etag = doc.HasMember("etag") && doc["etag"].IsString() ? doc["etag"].GetString() : "";
															#line 1779 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		upload.parts[partNum] = { etag, partData };
															#line 1781 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3MultipartPartRestored") .detail("UploadId", uploadId) .detail("PartNumber", partNum) .detail("Size", partData.size());
															#line 1763 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partNum++;
															#line 9501 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopBody1cont3loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(std::string const& __partMetaJson,int loopDepth) 
	{
															#line 1772 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		partMetaJson = __partMetaJson;
															#line 9510 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1loopBody1cont3loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3loopBody1cont1when1(std::string && __partMetaJson,int loopDepth) 
	{
		partMetaJson = std::move(__partMetaJson);
		loopDepth = a_body1loopBody1cont3loopBody1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state > 0) static_cast<LoadPersistedMultipartUploadsActor*>(this)->actor_wait_state = 0;
		static_cast<LoadPersistedMultipartUploadsActor*>(this)->ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >::remove();

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,std::string const& value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,std::string && value) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*,Error err) 
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1723 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 1724 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string multipartDir;
															#line 1733 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::vector<std::string> files;
															#line 1735 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int fileIdx;
															#line 1738 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string fileName;
															#line 1743 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string uploadId;
															#line 1744 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string statePath;
															#line 1747 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string stateJson;
															#line 1756 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	MockS3GlobalStorage::MultipartUpload upload;
															#line 1761 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int partNum;
															#line 1762 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	int maxAttempts;
															#line 1764 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partPath;
															#line 1765 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaPath;
															#line 1771 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partData;
															#line 1772 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string partMetaJson;
															#line 9628 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via loadPersistedMultipartUploads()
															#line 1723 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadPersistedMultipartUploadsActor final : public Actor<Void>, public ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >, public ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >, public ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >, public FastAllocated<LoadPersistedMultipartUploadsActor>, public LoadPersistedMultipartUploadsActorState<LoadPersistedMultipartUploadsActor> {
															#line 9633 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<LoadPersistedMultipartUploadsActor>::operator new;
	using FastAllocated<LoadPersistedMultipartUploadsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2798706548104289024UL, 6497294233603759616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >;
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >;
friend struct ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >;
															#line 1723 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadPersistedMultipartUploadsActor(std::string const& persistenceDir) 
															#line 9652 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   LoadPersistedMultipartUploadsActorState<LoadPersistedMultipartUploadsActor>(persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17523876871774881024UL, 15657059887496463360UL);
		ActorExecutionContextHelper __helper(static_cast<LoadPersistedMultipartUploadsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadPersistedMultipartUploads");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadPersistedMultipartUploads", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 0, std::string >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 1, std::string >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< LoadPersistedMultipartUploadsActor, 2, std::string >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1723 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadPersistedMultipartUploads( std::string const& persistenceDir ) {
															#line 1723 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadPersistedMultipartUploadsActor(persistenceDir));
															#line 9687 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1805 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// ACTOR: Load all persisted state from disk
															#line 9693 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via loadMockS3PersistedStateImpl()
															#line 1807 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class LoadMockS3PersistedStateImplActor>
															#line 1807 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadMockS3PersistedStateImplActorState {
															#line 9700 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1807 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadMockS3PersistedStateImplActorState() 
															#line 1807 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1808 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : persistenceDir()
															#line 9707 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this));

	}
	~LoadMockS3PersistedStateImplActorState() 
	{
		fdb_probe_actor_destroy("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1810 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!getGlobalStorage().persistenceEnabled || getGlobalStorage().persistenceLoaded)
															#line 9722 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1811 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (!static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadMockS3PersistedStateImplActorState(); static_cast<LoadMockS3PersistedStateImplActor*>(this)->destroy(); return 0; }
															#line 9726 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				new (&static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~LoadMockS3PersistedStateImplActorState();
				static_cast<LoadMockS3PersistedStateImplActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 1814 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			persistenceDir = getGlobalStorage().persistenceDir;
															#line 1815 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3LoadPersistedStateStart").detail("PersistenceDir", persistenceDir);
															#line 9736 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1819 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = loadPersistedObjects(persistenceDir);
															#line 1819 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9742 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 1;
															#line 1819 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9747 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~LoadMockS3PersistedStateImplActorState();
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1834 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~LoadMockS3PersistedStateImplActorState(); static_cast<LoadMockS3PersistedStateImplActor*>(this)->destroy(); return 0; }
															#line 9776 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<LoadMockS3PersistedStateImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~LoadMockS3PersistedStateImplActorState();
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1830 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent(SevError, "MockS3LoadPersistedStateFailed").error(e);
															#line 1831 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 9791 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1822 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = loadPersistedMultipartUploads(persistenceDir);
															#line 1822 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9807 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 2;
															#line 1822 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9812 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1822 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = loadPersistedMultipartUploads(persistenceDir);
															#line 1822 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 9823 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), loopDepth); else return a_body1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 2;
															#line 1822 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*>(static_cast<LoadMockS3PersistedStateImplActor*>(this)));
															#line 9828 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state > 0) static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 0;
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().persistenceLoaded = true;
															#line 1826 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3LoadPersistedStateComplete") .detail("ObjectsCount", getGlobalStorage().buckets.size()) .detail("MultipartUploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9914 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1824 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		getGlobalStorage().persistenceLoaded = true;
															#line 1826 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3LoadPersistedStateComplete") .detail("ObjectsCount", getGlobalStorage().buckets.size()) .detail("MultipartUploadsCount", getGlobalStorage().multipartUploads.size());
															#line 9925 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state > 0) static_cast<LoadMockS3PersistedStateImplActor*>(this)->actor_wait_state = 0;
		static_cast<LoadMockS3PersistedStateImplActor*>(this)->ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1808 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 10020 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via loadMockS3PersistedStateImpl()
															#line 1807 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class LoadMockS3PersistedStateImplActor final : public Actor<Void>, public ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >, public ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >, public FastAllocated<LoadMockS3PersistedStateImplActor>, public LoadMockS3PersistedStateImplActorState<LoadMockS3PersistedStateImplActor> {
															#line 10025 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<LoadMockS3PersistedStateImplActor>::operator new;
	using FastAllocated<LoadMockS3PersistedStateImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15263961221113072640UL, 10303393743573532672UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >;
friend struct ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >;
															#line 1807 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	LoadMockS3PersistedStateImplActor() 
															#line 10043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   LoadMockS3PersistedStateImplActorState<LoadMockS3PersistedStateImplActor>(),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10516119037031120128UL, 15944437332465256704UL);
		ActorExecutionContextHelper __helper(static_cast<LoadMockS3PersistedStateImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("loadMockS3PersistedStateImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("loadMockS3PersistedStateImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< LoadMockS3PersistedStateImplActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< LoadMockS3PersistedStateImplActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1807 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] static Future<Void> loadMockS3PersistedStateImpl(  ) {
															#line 1807 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new LoadMockS3PersistedStateImplActor());
															#line 10077 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1836 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Load persisted state from disk (called at server startup) - returns Future for use in ACTOR context
Future<Void> loadMockS3PersistedStateFuture() {
	if (getGlobalStorage().persistenceEnabled && !getGlobalStorage().persistenceLoaded) {
		return loadMockS3PersistedStateImpl();
	}
	return Void();
}

// Initialize MockS3 persistence for simulation tests (exported for MockS3ServerChaos)
															#line 10091 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via initializeMockS3Persistence()
															#line 1846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class InitializeMockS3PersistenceActor>
															#line 1846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class InitializeMockS3PersistenceActorState {
															#line 10098 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	InitializeMockS3PersistenceActorState(std::string const& serverKey) 
															#line 1846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : serverKey(serverKey)
															#line 10105 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this));

	}
	~InitializeMockS3PersistenceActorState() 
	{
		fdb_probe_actor_destroy("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1847 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!getGlobalStorage().persistenceEnabled)
															#line 10120 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 1848 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				enableMockS3Persistence(DEFAULT_MOCKS3_PERSISTENCE_DIR);
															#line 1849 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerPersistenceEnabled") .detail("Address", serverKey) .detail("PersistenceDir", DEFAULT_MOCKS3_PERSISTENCE_DIR);
															#line 1854 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = loadMockS3PersistedStateFuture();
															#line 1854 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<InitializeMockS3PersistenceActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10130 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<InitializeMockS3PersistenceActor*>(this)->actor_wait_state = 1;
															#line 1854 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< InitializeMockS3PersistenceActor, 0, Void >*>(static_cast<InitializeMockS3PersistenceActor*>(this)));
															#line 10135 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~InitializeMockS3PersistenceActorState();
		static_cast<InitializeMockS3PersistenceActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1856 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<InitializeMockS3PersistenceActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~InitializeMockS3PersistenceActorState(); static_cast<InitializeMockS3PersistenceActor*>(this)->destroy(); return 0; }
															#line 10163 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<InitializeMockS3PersistenceActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~InitializeMockS3PersistenceActorState();
		static_cast<InitializeMockS3PersistenceActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<InitializeMockS3PersistenceActor*>(this)->actor_wait_state > 0) static_cast<InitializeMockS3PersistenceActor*>(this)->actor_wait_state = 0;
		static_cast<InitializeMockS3PersistenceActor*>(this)->ActorCallback< InitializeMockS3PersistenceActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< InitializeMockS3PersistenceActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeMockS3PersistenceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< InitializeMockS3PersistenceActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeMockS3PersistenceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< InitializeMockS3PersistenceActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeMockS3PersistenceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string serverKey;
															#line 10260 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via initializeMockS3Persistence()
															#line 1846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class InitializeMockS3PersistenceActor final : public Actor<Void>, public ActorCallback< InitializeMockS3PersistenceActor, 0, Void >, public FastAllocated<InitializeMockS3PersistenceActor>, public InitializeMockS3PersistenceActorState<InitializeMockS3PersistenceActor> {
															#line 10265 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<InitializeMockS3PersistenceActor>::operator new;
	using FastAllocated<InitializeMockS3PersistenceActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1128923379304032256UL, 18337894282155197184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< InitializeMockS3PersistenceActor, 0, Void >;
															#line 1846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	InitializeMockS3PersistenceActor(std::string const& serverKey) 
															#line 10282 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   InitializeMockS3PersistenceActorState<InitializeMockS3PersistenceActor>(serverKey),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15097143290471136000UL, 6435086413190992384UL);
		ActorExecutionContextHelper __helper(static_cast<InitializeMockS3PersistenceActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("initializeMockS3Persistence");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("initializeMockS3Persistence", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< InitializeMockS3PersistenceActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> initializeMockS3Persistence( std::string const& serverKey ) {
															#line 1846 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new InitializeMockS3PersistenceActor(serverKey));
															#line 10315 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 1858 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Unit Tests for MockS3Server
															#line 10321 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1860()
															#line 1860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1860Actor>
															#line 1860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1860ActorState {
															#line 10328 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1860ActorState(UnitTestParameters const& params) 
															#line 1860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10335 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1860", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1860ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1860", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1862 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1863 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1864 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1865 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1867 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1869 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1870 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1871 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1873 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1860Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1860ActorState(); static_cast<FlowTestCase1860Actor*>(this)->destroy(); return 0; }
															#line 10366 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1860Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1860ActorState();
			static_cast<FlowTestCase1860Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1860ActorState();
		static_cast<FlowTestCase1860Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10390 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1860()
															#line 1860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1860Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1860Actor>, public FlowTestCase1860ActorState<FlowTestCase1860Actor> {
															#line 10395 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1860Actor>::operator new;
	using FastAllocated<FlowTestCase1860Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1702850884528193280UL, 16582459250366124800UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1860Actor(UnitTestParameters const& params) 
															#line 10411 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1860ActorState<FlowTestCase1860Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1860", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9490600770812761088UL, 2532780761947624704UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1860Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1860");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1860", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1860( UnitTestParameters const& params ) {
															#line 1860 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1860Actor(params));
															#line 10443 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1860, "/MockS3Server/parseS3Request/ValidBucketParameter")

#line 1875 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10449 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1876()
															#line 1876 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1876Actor>
															#line 1876 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1876ActorState {
															#line 10456 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1876 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1876ActorState(UnitTestParameters const& params) 
															#line 1876 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1876 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10463 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1876", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1876ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1876", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1878 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1879 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/?region=us-east-1";
															#line 1880 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1881 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 10484 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1884 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				server.parseS3Request(resource, bucket, object, queryParams);
															#line 1885 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				ASSERT(false);
															#line 10490 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1876ActorState();
		static_cast<FlowTestCase1876Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1890 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<FlowTestCase1876Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1876ActorState(); static_cast<FlowTestCase1876Actor*>(this)->destroy(); return 0; }
															#line 10519 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<FlowTestCase1876Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1876ActorState();
		static_cast<FlowTestCase1876Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1887 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(e.code() == error_code_backup_invalid_url);
															#line 10532 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1876 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10558 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1876()
															#line 1876 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1876Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1876Actor>, public FlowTestCase1876ActorState<FlowTestCase1876Actor> {
															#line 10563 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1876Actor>::operator new;
	using FastAllocated<FlowTestCase1876Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13598968414092452864UL, 17392797876562318848UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1876 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1876Actor(UnitTestParameters const& params) 
															#line 10579 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1876ActorState<FlowTestCase1876Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1876", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2774087583918859520UL, 72397983422908416UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1876Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1876");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1876", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1876 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1876( UnitTestParameters const& params ) {
															#line 1876 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1876Actor(params));
															#line 10611 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1876, "/MockS3Server/parseS3Request/MissingBucketParameter")

#line 1892 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10617 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1893()
															#line 1893 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1893Actor>
															#line 1893 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1893ActorState {
															#line 10624 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1893 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1893ActorState(UnitTestParameters const& params) 
															#line 1893 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1893 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10631 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1893", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1893ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1893", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1895 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1896 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/";
															#line 1897 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1898 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 10652 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			try {
															#line 1901 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				server.parseS3Request(resource, bucket, object, queryParams);
															#line 1902 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				ASSERT(false);
															#line 10658 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1893ActorState();
		static_cast<FlowTestCase1893Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1907 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<FlowTestCase1893Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1893ActorState(); static_cast<FlowTestCase1893Actor*>(this)->destroy(); return 0; }
															#line 10687 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<FlowTestCase1893Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1893ActorState();
		static_cast<FlowTestCase1893Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1904 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(e.code() == error_code_backup_invalid_url);
															#line 10700 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1893 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10726 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1893()
															#line 1893 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1893Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1893Actor>, public FlowTestCase1893ActorState<FlowTestCase1893Actor> {
															#line 10731 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1893Actor>::operator new;
	using FastAllocated<FlowTestCase1893Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6544323081761924864UL, 9305186431970266368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1893 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1893Actor(UnitTestParameters const& params) 
															#line 10747 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1893ActorState<FlowTestCase1893Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1893", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3218957982273941248UL, 11196005967260126208UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1893Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1893");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1893", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1893 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1893( UnitTestParameters const& params ) {
															#line 1893 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1893Actor(params));
															#line 10779 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1893, "/MockS3Server/parseS3Request/EmptyQueryString")

#line 1909 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10785 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1910()
															#line 1910 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1910Actor>
															#line 1910 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1910ActorState {
															#line 10792 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1910 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1910ActorState(UnitTestParameters const& params) 
															#line 1910 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1910 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10799 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1910", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1910ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1910", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1912 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1913 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket/testobject?region=us-east-1";
															#line 1914 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1915 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1917 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1919 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1920 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "testobject");
															#line 1921 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1923 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1910Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1910ActorState(); static_cast<FlowTestCase1910Actor*>(this)->destroy(); return 0; }
															#line 10830 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1910Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1910ActorState();
			static_cast<FlowTestCase1910Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1910ActorState();
		static_cast<FlowTestCase1910Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1910 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10854 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1910()
															#line 1910 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1910Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1910Actor>, public FlowTestCase1910ActorState<FlowTestCase1910Actor> {
															#line 10859 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1910Actor>::operator new;
	using FastAllocated<FlowTestCase1910Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5631708236082012160UL, 7011539159414697984UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1910 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1910Actor(UnitTestParameters const& params) 
															#line 10875 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1910ActorState<FlowTestCase1910Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1910", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12102103180346197504UL, 18238244337250394368UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1910Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1910");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1910", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1910 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1910( UnitTestParameters const& params ) {
															#line 1910 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1910Actor(params));
															#line 10907 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1910, "/MockS3Server/parseS3Request/BucketParameterOverride")

#line 1925 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 10913 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1926()
															#line 1926 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1926Actor>
															#line 1926 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1926ActorState {
															#line 10920 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1926 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1926ActorState(UnitTestParameters const& params) 
															#line 1926 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1926 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 10927 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1926", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1926ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1926", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1928 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1929 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket/folder/subfolder/file.txt?region=us-east-1";
															#line 1930 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1931 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1933 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1935 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1936 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "folder/subfolder/file.txt");
															#line 1937 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1939 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1926Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1926ActorState(); static_cast<FlowTestCase1926Actor*>(this)->destroy(); return 0; }
															#line 10958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1926Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1926ActorState();
			static_cast<FlowTestCase1926Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1926ActorState();
		static_cast<FlowTestCase1926Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1926 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 10982 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1926()
															#line 1926 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1926Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1926Actor>, public FlowTestCase1926ActorState<FlowTestCase1926Actor> {
															#line 10987 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1926Actor>::operator new;
	using FastAllocated<FlowTestCase1926Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9057230597868997376UL, 14962226789509362688UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1926 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1926Actor(UnitTestParameters const& params) 
															#line 11003 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1926ActorState<FlowTestCase1926Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1926", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8971900688956400640UL, 857432341951948288UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1926Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1926");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1926", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1926 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1926( UnitTestParameters const& params ) {
															#line 1926 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1926Actor(params));
															#line 11035 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1926, "/MockS3Server/parseS3Request/ComplexPath")

#line 1941 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11041 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1942()
															#line 1942 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1942Actor>
															#line 1942 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1942ActorState {
															#line 11048 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1942 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1942ActorState(UnitTestParameters const& params) 
															#line 1942 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1942 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11055 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1942", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1942ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1942", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1944 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1945 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1&param=value%3Dtest";
															#line 1946 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1947 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1949 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1951 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1952 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1953 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["param"] == "value=test");
															#line 1955 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1942Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1942ActorState(); static_cast<FlowTestCase1942Actor*>(this)->destroy(); return 0; }
															#line 11086 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1942Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1942ActorState();
			static_cast<FlowTestCase1942Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1942ActorState();
		static_cast<FlowTestCase1942Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1942 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11110 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1942()
															#line 1942 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1942Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1942Actor>, public FlowTestCase1942ActorState<FlowTestCase1942Actor> {
															#line 11115 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1942Actor>::operator new;
	using FastAllocated<FlowTestCase1942Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(404872233400689152UL, 2902916527943256320UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1942 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1942Actor(UnitTestParameters const& params) 
															#line 11131 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1942ActorState<FlowTestCase1942Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1942", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7009561507078476288UL, 4663137970561448960UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1942Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1942");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1942", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1942 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1942( UnitTestParameters const& params ) {
															#line 1942 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1942Actor(params));
															#line 11163 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1942, "/MockS3Server/parseS3Request/URLEncodedParameters")

#line 1957 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11169 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1958()
															#line 1958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1958Actor>
															#line 1958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1958ActorState {
															#line 11176 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1958ActorState(UnitTestParameters const& params) 
															#line 1958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11183 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1958", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1958ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1958", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1960 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1961 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1962 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1963 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1965 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1967 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1968 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1969 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1971 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1958Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1958ActorState(); static_cast<FlowTestCase1958Actor*>(this)->destroy(); return 0; }
															#line 11214 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1958Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1958ActorState();
			static_cast<FlowTestCase1958Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1958ActorState();
		static_cast<FlowTestCase1958Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1958()
															#line 1958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1958Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1958Actor>, public FlowTestCase1958ActorState<FlowTestCase1958Actor> {
															#line 11243 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1958Actor>::operator new;
	using FastAllocated<FlowTestCase1958Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4556156447124014848UL, 13223328964417143296UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1958Actor(UnitTestParameters const& params) 
															#line 11259 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1958ActorState<FlowTestCase1958Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1958", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9921252078891117312UL, 5208520797973763072UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1958Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1958");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1958", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1958( UnitTestParameters const& params ) {
															#line 1958 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1958Actor(params));
															#line 11291 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1958, "/MockS3Server/parseS3Request/EmptyPath")

#line 1973 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11297 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1974()
															#line 1974 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1974Actor>
															#line 1974 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1974ActorState {
															#line 11304 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1974 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1974ActorState(UnitTestParameters const& params) 
															#line 1974 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1974 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11311 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1974", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1974ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1974", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1976 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1977 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1";
															#line 1978 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1979 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1981 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1983 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 1984 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(object == "");
															#line 1985 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 1987 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1974Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1974ActorState(); static_cast<FlowTestCase1974Actor*>(this)->destroy(); return 0; }
															#line 11342 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1974Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1974ActorState();
			static_cast<FlowTestCase1974Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1974ActorState();
		static_cast<FlowTestCase1974Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1974 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11366 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1974()
															#line 1974 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1974Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1974Actor>, public FlowTestCase1974ActorState<FlowTestCase1974Actor> {
															#line 11371 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1974Actor>::operator new;
	using FastAllocated<FlowTestCase1974Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18225748338666894848UL, 900496347895019008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1974 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1974Actor(UnitTestParameters const& params) 
															#line 11387 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1974ActorState<FlowTestCase1974Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18100637140583322368UL, 7823974797634655744UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1974");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1974 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1974( UnitTestParameters const& params ) {
															#line 1974 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1974Actor(params));
															#line 11419 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1974, "/MockS3Server/parseS3Request/OnlyBucketInPath")

#line 1989 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11425 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase1990()
															#line 1990 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase1990Actor>
															#line 1990 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1990ActorState {
															#line 11432 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 1990 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1990ActorState(UnitTestParameters const& params) 
															#line 1990 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 1990 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11439 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase1990", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1990ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1990", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1992 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 1993 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?region=us-east-1&version=1&encoding=utf8";
															#line 1994 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 1995 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 1997 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 1999 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 2000 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 2001 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["version"] == "1");
															#line 2002 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["encoding"] == "utf8");
															#line 2003 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams.size() == 3);
															#line 2005 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase1990Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1990ActorState(); static_cast<FlowTestCase1990Actor*>(this)->destroy(); return 0; }
															#line 11474 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase1990Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1990ActorState();
			static_cast<FlowTestCase1990Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1990ActorState();
		static_cast<FlowTestCase1990Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1990 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11498 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase1990()
															#line 1990 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase1990Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1990Actor>, public FlowTestCase1990ActorState<FlowTestCase1990Actor> {
															#line 11503 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase1990Actor>::operator new;
	using FastAllocated<FlowTestCase1990Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10023941595247892992UL, 11166159021161574912UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1990 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase1990Actor(UnitTestParameters const& params) 
															#line 11519 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase1990ActorState<FlowTestCase1990Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1093206899862294016UL, 17026273717551034624UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1990");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1990 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase1990( UnitTestParameters const& params ) {
															#line 1990 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase1990Actor(params));
															#line 11551 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase1990, "/MockS3Server/parseS3Request/MultipleParameters")

#line 2007 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11557 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2008()
															#line 2008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2008Actor>
															#line 2008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2008ActorState {
															#line 11564 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2008ActorState(UnitTestParameters const& params) 
															#line 2008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11571 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2008", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2008ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2008", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2010 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			MockS3ServerImpl server;
															#line 2011 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string resource = "/testbucket?flag&region=us-east-1";
															#line 2012 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string bucket, object;
															#line 2013 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::map<std::string, std::string> queryParams;
															#line 2015 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			server.parseS3Request(resource, bucket, object, queryParams);
															#line 2017 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(bucket == "testbucket");
															#line 2018 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["flag"] == "");
															#line 2019 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(queryParams["region"] == "us-east-1");
															#line 2021 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2008Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2008ActorState(); static_cast<FlowTestCase2008Actor*>(this)->destroy(); return 0; }
															#line 11602 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2008Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2008ActorState();
			static_cast<FlowTestCase2008Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2008ActorState();
		static_cast<FlowTestCase2008Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11626 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2008()
															#line 2008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2008Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2008Actor>, public FlowTestCase2008ActorState<FlowTestCase2008Actor> {
															#line 11631 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2008Actor>::operator new;
	using FastAllocated<FlowTestCase2008Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2859806410120608512UL, 15801855462660089856UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2008Actor(UnitTestParameters const& params) 
															#line 11647 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2008ActorState<FlowTestCase2008Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2008", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9806764820203201280UL, 9475331387441784320UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2008Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2008");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2008", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2008( UnitTestParameters const& params ) {
															#line 2008 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2008Actor(params));
															#line 11679 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2008, "/MockS3Server/parseS3Request/ParametersWithoutValues")

#line 2023 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11685 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2024()
															#line 2024 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2024Actor>
															#line 2024 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2024ActorState {
															#line 11692 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2024 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2024ActorState(UnitTestParameters const& params) 
															#line 2024 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2024 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11699 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2024", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2024ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2024", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2025 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=0-99";
															#line 2026 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2028 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2030 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2031 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 0);
															#line 2032 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 99);
															#line 2034 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2024Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2024ActorState(); static_cast<FlowTestCase2024Actor*>(this)->destroy(); return 0; }
															#line 11726 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2024Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2024ActorState();
			static_cast<FlowTestCase2024Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2024ActorState();
		static_cast<FlowTestCase2024Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2024 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11750 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2024()
															#line 2024 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2024Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2024Actor>, public FlowTestCase2024ActorState<FlowTestCase2024Actor> {
															#line 11755 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2024Actor>::operator new;
	using FastAllocated<FlowTestCase2024Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4346896401178174464UL, 7985771279949171456UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2024 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2024Actor(UnitTestParameters const& params) 
															#line 11771 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2024ActorState<FlowTestCase2024Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2024", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3638059160900261376UL, 8092817940385817856UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2024Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2024");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2024", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2024 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2024( UnitTestParameters const& params ) {
															#line 2024 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2024Actor(params));
															#line 11803 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2024, "/MockS3Server/RangeHeader/SimpleByteRange")

#line 2036 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11809 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2037()
															#line 2037 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2037Actor>
															#line 2037 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2037ActorState {
															#line 11816 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2037 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2037ActorState(UnitTestParameters const& params) 
															#line 2037 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2037 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11823 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2037", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2037ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2037", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2038 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=100-199";
															#line 2039 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2041 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2043 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2044 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 100);
															#line 2045 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 199);
															#line 2047 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2037Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2037ActorState(); static_cast<FlowTestCase2037Actor*>(this)->destroy(); return 0; }
															#line 11850 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2037Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2037ActorState();
			static_cast<FlowTestCase2037Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2037ActorState();
		static_cast<FlowTestCase2037Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2037 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11874 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2037()
															#line 2037 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2037Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2037Actor>, public FlowTestCase2037ActorState<FlowTestCase2037Actor> {
															#line 11879 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2037Actor>::operator new;
	using FastAllocated<FlowTestCase2037Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2172023257105099008UL, 12827859229143351296UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2037 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2037Actor(UnitTestParameters const& params) 
															#line 11895 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2037ActorState<FlowTestCase2037Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2037", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11516127994804049408UL, 8958415098019529216UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2037Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2037");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2037", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2037 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2037( UnitTestParameters const& params ) {
															#line 2037 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2037Actor(params));
															#line 11927 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2037, "/MockS3Server/RangeHeader/MiddleRange")

#line 2049 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 11933 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2050()
															#line 2050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2050Actor>
															#line 2050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2050ActorState {
															#line 11940 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2050ActorState(UnitTestParameters const& params) 
															#line 2050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 11947 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2050", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2050ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2050", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2051 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=1000000-1999999";
															#line 2052 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2054 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2056 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2057 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 1000000);
															#line 2058 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 1999999);
															#line 2060 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2050Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2050ActorState(); static_cast<FlowTestCase2050Actor*>(this)->destroy(); return 0; }
															#line 11974 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2050Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2050ActorState();
			static_cast<FlowTestCase2050Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2050ActorState();
		static_cast<FlowTestCase2050Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 11998 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2050()
															#line 2050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2050Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2050Actor>, public FlowTestCase2050ActorState<FlowTestCase2050Actor> {
															#line 12003 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2050Actor>::operator new;
	using FastAllocated<FlowTestCase2050Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5860623832190487808UL, 470502039002974208UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2050Actor(UnitTestParameters const& params) 
															#line 12019 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2050ActorState<FlowTestCase2050Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2050", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12474116207401914368UL, 14512261519994337280UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2050Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2050");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2050", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2050( UnitTestParameters const& params ) {
															#line 2050 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2050Actor(params));
															#line 12051 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2050, "/MockS3Server/RangeHeader/LargeOffsets")

#line 2062 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12057 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2063()
															#line 2063 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2063Actor>
															#line 2063 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2063ActorState {
															#line 12064 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2063 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2063ActorState(UnitTestParameters const& params) 
															#line 2063 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2063 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12071 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2063", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2063ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2063", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2064 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "invalid-range";
															#line 2065 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2067 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2069 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2071 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2063Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2063ActorState(); static_cast<FlowTestCase2063Actor*>(this)->destroy(); return 0; }
															#line 12094 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2063Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2063ActorState();
			static_cast<FlowTestCase2063Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2063ActorState();
		static_cast<FlowTestCase2063Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2063 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12118 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2063()
															#line 2063 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2063Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2063Actor>, public FlowTestCase2063ActorState<FlowTestCase2063Actor> {
															#line 12123 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2063Actor>::operator new;
	using FastAllocated<FlowTestCase2063Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10607086228549566208UL, 13695599391995632128UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2063 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2063Actor(UnitTestParameters const& params) 
															#line 12139 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2063ActorState<FlowTestCase2063Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2063", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10024756261257064704UL, 6150053523691575552UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2063Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2063");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2063", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2063 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2063( UnitTestParameters const& params ) {
															#line 2063 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2063Actor(params));
															#line 12171 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2063, "/MockS3Server/RangeHeader/InvalidFormat")

#line 2073 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12177 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2074()
															#line 2074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2074Actor>
															#line 2074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2074ActorState {
															#line 12184 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2074ActorState(UnitTestParameters const& params) 
															#line 2074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12191 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2074", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2074ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2074", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2075 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "0-99";
															#line 2076 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2078 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2080 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2082 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2074Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2074ActorState(); static_cast<FlowTestCase2074Actor*>(this)->destroy(); return 0; }
															#line 12214 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2074Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2074ActorState();
			static_cast<FlowTestCase2074Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2074ActorState();
		static_cast<FlowTestCase2074Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12238 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2074()
															#line 2074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2074Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2074Actor>, public FlowTestCase2074ActorState<FlowTestCase2074Actor> {
															#line 12243 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2074Actor>::operator new;
	using FastAllocated<FlowTestCase2074Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13555699762729974528UL, 10816719844243896064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2074Actor(UnitTestParameters const& params) 
															#line 12259 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2074ActorState<FlowTestCase2074Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2074", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5313871499119712256UL, 4540502925954524160UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2074Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2074");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2074", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2074( UnitTestParameters const& params ) {
															#line 2074 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2074Actor(params));
															#line 12291 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2074, "/MockS3Server/RangeHeader/MissingBytesPrefix")

#line 2084 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12297 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2085()
															#line 2085 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2085Actor>
															#line 2085 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2085ActorState {
															#line 12304 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2085 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2085ActorState(UnitTestParameters const& params) 
															#line 2085 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2085 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12311 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2085", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2085ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2085", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2086 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=0";
															#line 2087 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2089 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2091 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2093 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2085Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2085ActorState(); static_cast<FlowTestCase2085Actor*>(this)->destroy(); return 0; }
															#line 12334 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2085Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2085ActorState();
			static_cast<FlowTestCase2085Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2085ActorState();
		static_cast<FlowTestCase2085Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2085 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12358 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2085()
															#line 2085 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2085Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2085Actor>, public FlowTestCase2085ActorState<FlowTestCase2085Actor> {
															#line 12363 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2085Actor>::operator new;
	using FastAllocated<FlowTestCase2085Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8867482133373710336UL, 8869176559948488960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2085 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2085Actor(UnitTestParameters const& params) 
															#line 12379 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2085ActorState<FlowTestCase2085Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2085", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10285220636179945472UL, 10151859032696499968UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2085Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2085");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2085", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2085 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2085( UnitTestParameters const& params ) {
															#line 2085 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2085Actor(params));
															#line 12411 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2085, "/MockS3Server/RangeHeader/MissingDash")

#line 2095 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12417 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2096()
															#line 2096 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2096Actor>
															#line 2096 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2096ActorState {
															#line 12424 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2096 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2096ActorState(UnitTestParameters const& params) 
															#line 2096 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2096 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12431 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2096", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2096ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2096", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2097 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "";
															#line 2098 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2100 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2102 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2104 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2096Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2096ActorState(); static_cast<FlowTestCase2096Actor*>(this)->destroy(); return 0; }
															#line 12454 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2096Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2096ActorState();
			static_cast<FlowTestCase2096Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2096ActorState();
		static_cast<FlowTestCase2096Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2096 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12478 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2096()
															#line 2096 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2096Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2096Actor>, public FlowTestCase2096ActorState<FlowTestCase2096Actor> {
															#line 12483 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2096Actor>::operator new;
	using FastAllocated<FlowTestCase2096Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12506919502754729984UL, 9915644519830581760UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2096 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2096Actor(UnitTestParameters const& params) 
															#line 12499 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2096ActorState<FlowTestCase2096Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2096", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1910473369915824896UL, 14710271759254564608UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2096Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2096");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2096", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2096 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2096( UnitTestParameters const& params ) {
															#line 2096 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2096Actor(params));
															#line 12531 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2096, "/MockS3Server/RangeHeader/EmptyString")

#line 2106 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12537 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2107()
															#line 2107 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2107Actor>
															#line 2107 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2107ActorState {
															#line 12544 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2107 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2107ActorState(UnitTestParameters const& params) 
															#line 2107 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2107 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12551 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2107", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2107ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2107", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2108 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=-100-200";
															#line 2109 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2111 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2114 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == false);
															#line 2116 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2107Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2107ActorState(); static_cast<FlowTestCase2107Actor*>(this)->destroy(); return 0; }
															#line 12574 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2107Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2107ActorState();
			static_cast<FlowTestCase2107Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2107ActorState();
		static_cast<FlowTestCase2107Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2107 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12598 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2107()
															#line 2107 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2107Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2107Actor>, public FlowTestCase2107ActorState<FlowTestCase2107Actor> {
															#line 12603 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2107Actor>::operator new;
	using FastAllocated<FlowTestCase2107Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7465948124891687424UL, 16479040305773594368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2107 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2107Actor(UnitTestParameters const& params) 
															#line 12619 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2107ActorState<FlowTestCase2107Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2107", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11978555983944944640UL, 13724063305597775360UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2107Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2107");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2107", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2107 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2107( UnitTestParameters const& params ) {
															#line 2107 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2107Actor(params));
															#line 12651 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2107, "/MockS3Server/RangeHeader/NegativeStart")

#line 2118 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

															#line 12657 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase2119()
															#line 2119 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class FlowTestCase2119Actor>
															#line 2119 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2119ActorState {
															#line 12664 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2119 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2119ActorState(UnitTestParameters const& params) 
															#line 2119 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2119 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : params(params)
															#line 12671 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase2119", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2119ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2119", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2120 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			std::string rangeHeader = "bytes=200-100";
															#line 2121 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			int64_t rangeStart, rangeEnd;
															#line 2123 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			bool result = MockS3ServerImpl::parseRangeHeader(rangeHeader, rangeStart, rangeEnd);
															#line 2126 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(result == true);
															#line 2127 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeStart == 200);
															#line 2128 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			ASSERT(rangeEnd == 100);
															#line 2130 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!static_cast<FlowTestCase2119Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2119ActorState(); static_cast<FlowTestCase2119Actor*>(this)->destroy(); return 0; }
															#line 12698 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			new (&static_cast<FlowTestCase2119Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2119ActorState();
			static_cast<FlowTestCase2119Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2119ActorState();
		static_cast<FlowTestCase2119Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2119 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	UnitTestParameters params;
															#line 12722 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase2119()
															#line 2119 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class FlowTestCase2119Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2119Actor>, public FlowTestCase2119ActorState<FlowTestCase2119Actor> {
															#line 12727 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase2119Actor>::operator new;
	using FastAllocated<FlowTestCase2119Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6620044969149990912UL, 8444859466071352064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2119 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	FlowTestCase2119Actor(UnitTestParameters const& params) 
															#line 12743 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase2119ActorState<FlowTestCase2119Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2119", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6666171426196121344UL, 18129796740537758720UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2119Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2119");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2119", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2119 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
static Future<Void> flowTestCase2119( UnitTestParameters const& params ) {
															#line 2119 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new FlowTestCase2119Actor(params));
															#line 12775 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase2119, "/MockS3Server/RangeHeader/StartGreaterThanEnd")

#line 2132 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

// Real HTTP Server Implementation for ctests
															#line 12782 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
namespace {
// This generated class is to be used only via startMockS3ServerReal_impl()
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
template <class StartMockS3ServerReal_implActor>
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerReal_implActorState {
															#line 12789 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerReal_implActorState(NetworkAddress const& listenAddress,std::string const& persistenceDir) 
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		 : listenAddress(listenAddress),
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		   persistenceDir(persistenceDir)
															#line 12798 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
	{
		fdb_probe_actor_create("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this));

	}
	~StartMockS3ServerReal_implActorState() 
	{
		fdb_probe_actor_destroy("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2135 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			TraceEvent("MockS3ServerRealStarting").detail("ListenAddress", listenAddress.toString());
															#line 2138 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
			if (!getGlobalStorage().persistenceEnabled)
															#line 12815 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
			{
															#line 2140 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (persistenceDir.empty())
															#line 12819 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				{
															#line 2141 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
					persistenceDir = DEFAULT_MOCKS3_PERSISTENCE_DIR;
															#line 12823 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				}
															#line 2143 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				enableMockS3Persistence(persistenceDir);
															#line 2144 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				TraceEvent("MockS3ServerRealPersistenceEnabled") .detail("ListenAddress", listenAddress.toString()) .detail("PersistenceDir", persistenceDir);
															#line 2149 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				StrictFuture<Void> __when_expr_0 = loadMockS3PersistedStateFuture();
															#line 2149 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12833 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 1;
															#line 2149 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*>(static_cast<StartMockS3ServerReal_implActor*>(this)));
															#line 12838 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2152 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		server = makeReference<HTTP::SimServerContext>();
															#line 2153 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		server->registerNewServer(listenAddress, makeReference<MockS3RequestHandler>());
															#line 2155 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		TraceEvent("MockS3ServerRealStarted") .detail("ListenAddress", listenAddress.toString()) .detail("ServerPtr", format("%p", server.getPtr()));
															#line 2160 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		StrictFuture<Void> __when_expr_1 = Never();
															#line 2160 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12874 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 2;
															#line 2160 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*>(static_cast<StartMockS3ServerReal_implActor*>(this)));
															#line 12879 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerReal_implActor*>(this)->ActorCallback< StartMockS3ServerReal_implActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 2161 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerReal_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerReal_implActorState(); static_cast<StartMockS3ServerReal_implActor*>(this)->destroy(); return 0; }
															#line 12975 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<StartMockS3ServerReal_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 2161 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
		if (!static_cast<StartMockS3ServerReal_implActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~StartMockS3ServerReal_implActorState(); static_cast<StartMockS3ServerReal_implActor*>(this)->destroy(); return 0; }
															#line 12987 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		new (&static_cast<StartMockS3ServerReal_implActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~StartMockS3ServerReal_implActorState();
		static_cast<StartMockS3ServerReal_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state > 0) static_cast<StartMockS3ServerReal_implActor*>(this)->actor_wait_state = 0;
		static_cast<StartMockS3ServerReal_implActor*>(this)->ActorCallback< StartMockS3ServerReal_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	NetworkAddress listenAddress;
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	std::string persistenceDir;
															#line 2152 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	Reference<HTTP::SimServerContext> server;
															#line 13076 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
};
// This generated class is to be used only via startMockS3ServerReal_impl()
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
class StartMockS3ServerReal_implActor final : public Actor<Void>, public ActorCallback< StartMockS3ServerReal_implActor, 0, Void >, public ActorCallback< StartMockS3ServerReal_implActor, 1, Void >, public FastAllocated<StartMockS3ServerReal_implActor>, public StartMockS3ServerReal_implActorState<StartMockS3ServerReal_implActor> {
															#line 13081 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
public:
	using FastAllocated<StartMockS3ServerReal_implActor>::operator new;
	using FastAllocated<StartMockS3ServerReal_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10613562249404862464UL, 5561216478090931200UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StartMockS3ServerReal_implActor, 0, Void >;
friend struct ActorCallback< StartMockS3ServerReal_implActor, 1, Void >;
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	StartMockS3ServerReal_implActor(NetworkAddress const& listenAddress,std::string const& persistenceDir) 
															#line 13099 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
		 : Actor<Void>(),
		   StartMockS3ServerReal_implActorState<StartMockS3ServerReal_implActor>(listenAddress, persistenceDir),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15792334911365886208UL, 3052106341508935936UL);
		ActorExecutionContextHelper __helper(static_cast<StartMockS3ServerReal_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("startMockS3ServerReal_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("startMockS3ServerReal_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StartMockS3ServerReal_implActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< StartMockS3ServerReal_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
[[nodiscard]] Future<Void> startMockS3ServerReal_impl( NetworkAddress const& listenAddress, std::string const& persistenceDir ) {
															#line 2134 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"
	return Future<Void>(new StartMockS3ServerReal_implActor(listenAddress, persistenceDir));
															#line 13133 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/build_output/fdbserver/MockS3Server.actor.g.cpp"
}

#line 2163 "/codebuild/output/src2085254572/src/github.com/apple/foundationdb/fdbserver/MockS3Server.actor.cpp"

Future<Void> startMockS3ServerReal(const NetworkAddress& listenAddress, const std::string& persistenceDir) {
	return startMockS3ServerReal_impl(listenAddress, persistenceDir);
}

// Wrapper for registerMockS3Server (calls the ACTOR implementation)
Future<Void> registerMockS3Server(std::string ip, std::string port) {
	return registerMockS3Server_impl(ip, port);
}
