#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
/*
 * StorageMetrics.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "flow/UnitTest.h"
#include "fdbserver/StorageMetrics.actor.h"
#include "flow/actorcompiler.h" // This must be the last #include.

CommonStorageCounters::CommonStorageCounters(const std::string& name,
                                             const std::string& id,
                                             const StorageServerMetrics* metrics)
  : cc(name, id), finishedQueries("FinishedQueries", cc), bytesQueried("BytesQueried", cc),
    bytesFetched("BytesFetched", cc), bytesInput("BytesInput", cc), mutationBytes("MutationBytes", cc),
    kvFetched("KVFetched", cc), mutations("Mutations", cc), setMutations("SetMutations", cc),
    clearRangeMutations("ClearRangeMutations", cc), kvFetchServed("KVFetchServed", cc),
    kvFetchBytesServed("KVFetchBytesServed", cc), fetchKeyErrors("FetchKeyErrors", cc) {
	if (metrics) {
		specialCounter(cc, "BytesStored", [metrics]() { return metrics->byteSample.getEstimate(allKeys); });
		specialCounter(cc, "BytesReadSampleCount", [metrics]() { return metrics->bytesReadSample.queue.size(); });
		specialCounter(cc, "OpsReadSampleCount", [metrics]() { return metrics->opsReadSample.queue.size(); });
		specialCounter(cc, "BytesWriteSampleCount", [metrics]() { return metrics->bytesWriteSample.queue.size(); });
		specialCounter(cc, "IopsReadSampleCount", [metrics]() { return metrics->iopsSample.queue.size(); });
	}
}

// TODO: update the cost as bytesReadPerKSecond + opsReadPerKSecond * SERVER_KNOBS->EMPTY_READ_PENALTY. The source of
// this model is Redwood will have a constant cost of seek of each read ops then read the actual data.
// As by 71.2.8, bytesReadPerKSecond should be larger than opsReadPerKSecond * SERVER_KNOBS->EMPTY_READ_PENALTY because
// the bytes always round to EMPTY_READ_PENALTY when the returned result size is less than EMPTY_READ_PENALTY. This cost
// is different from what tag throttling use to produce throttling decision.
int64_t StorageMetrics::readLoadKSecond() const {
	return std::max(bytesReadPerKSecond, opsReadPerKSecond * SERVER_KNOBS->EMPTY_READ_PENALTY);
}

int64_t StorageMetricSample::getEstimate(KeyRangeRef keys) const {
	return sample.sumRange(keys.begin, keys.end);
}

KeyRef StorageMetricSample::splitEstimate(KeyRangeRef range, int64_t offset, bool front) const {
	auto fwd_split = sample.index(front ? sample.sumTo(sample.lower_bound(range.begin)) + offset
	                                    : sample.sumTo(sample.lower_bound(range.end)) - offset);

	if (fwd_split == sample.end() || *fwd_split >= range.end)
		return range.end;

	if (!front && *fwd_split <= range.begin)
		return range.begin;

	auto bck_split = fwd_split;

	// Butterfly search - start at midpoint then go in both directions.
	while ((fwd_split != sample.end() && *fwd_split < range.end) ||
	       (bck_split != sample.begin() && *bck_split > range.begin)) {
		if (bck_split != sample.begin() && *bck_split > range.begin) {
			auto it = bck_split;
			bck_split.decrementNonEnd();

			KeyRef split = keyBetween(KeyRangeRef(
			    bck_split != sample.begin() ? std::max<KeyRef>(*bck_split, range.begin) : range.begin, *it));
			if (!front || (getEstimate(KeyRangeRef(range.begin, split)) > 0 &&
			               split.size() <= CLIENT_KNOBS->SPLIT_KEY_SIZE_LIMIT))
				return split;
		}

		if (fwd_split != sample.end() && *fwd_split < range.end) {
			auto it = fwd_split;
			++it;

			KeyRef split =
			    keyBetween(KeyRangeRef(*fwd_split, it != sample.end() ? std::min<KeyRef>(*it, range.end) : range.end));
			if (front ||
			    (getEstimate(KeyRangeRef(split, range.end)) > 0 && split.size() <= CLIENT_KNOBS->SPLIT_KEY_SIZE_LIMIT))
				return split;

			fwd_split = it;
		}
	}

	// If we didn't return above, we didn't find anything.
	TraceEvent(SevWarn, "CannotSplitLastSampleKey").detail("Range", range).detail("Offset", offset);
	return front ? range.end : range.begin;
}

// Get the current estimated metrics for the given keys
StorageMetrics StorageServerMetrics::getMetrics(KeyRangeRef const& keys) const {
	StorageMetrics result;
	result.bytes = byteSample.getEstimate(keys);
	result.bytesWrittenPerKSecond =
	    bytesWriteSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	result.iosPerKSecond = iopsSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;

	result.bytesReadPerKSecond =
	    bytesReadSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	result.opsReadPerKSecond =
	    opsReadSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	return result;
}

// Called when metrics should change (IO for a given key)
// Notifies waiting WaitMetricsRequests through waitMetricsMap, and updates metricsAverageQueue and metricsSampleMap
void StorageServerMetrics::notify(const Key& key, StorageMetrics& metrics) {
	ASSERT(metrics.bytes == 0); // ShardNotifyMetrics
	if (g_network->isSimulated()) {
		CODE_PROBE(metrics.bytesWrittenPerKSecond != 0, "ShardNotifyMetrics bytes");
		CODE_PROBE(metrics.iosPerKSecond != 0, "ShardNotifyMetrics ios");
		CODE_PROBE(metrics.bytesReadPerKSecond != 0, "ShardNotifyMetrics bytesRead");
		CODE_PROBE(metrics.opsReadPerKSecond != 0, "ShardNotifyMetrics opsRead");
	}

	double expire = now() + SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL;

	StorageMetrics notifyMetrics;

	if (metrics.bytesWrittenPerKSecond)
		notifyMetrics.bytesWrittenPerKSecond =
		    bytesWriteSample.addAndExpire(key, metrics.bytesWrittenPerKSecond, expire) *
		    SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	if (metrics.iosPerKSecond)
		notifyMetrics.iosPerKSecond = iopsSample.addAndExpire(key, metrics.iosPerKSecond, expire) *
		                              SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	if (metrics.bytesReadPerKSecond)
		notifyMetrics.bytesReadPerKSecond = bytesReadSample.addAndExpire(key, metrics.bytesReadPerKSecond, expire) *
		                                    SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	if (metrics.opsReadPerKSecond) {
		notifyMetrics.opsReadPerKSecond = opsReadSample.addAndExpire(key, metrics.opsReadPerKSecond, expire) *
		                                  SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	}

	if (!notifyMetrics.allZero()) {
		auto& v = waitMetricsMap[key];
		for (int i = 0; i < v.size(); i++) {
			if (g_network->isSimulated()) {
				CODE_PROBE(true, "shard notify metrics");
			}
			// ShardNotifyMetrics
			v[i].send(notifyMetrics);
		}
	}
}

// Due to the fact that read sampling will be called on all reads, use this specialized function to avoid overhead
// around branch misses and unnecessary stack allocation which eventually adds up under heavy load.
void StorageServerMetrics::notifyBytesReadPerKSecond(const Key& key, int64_t in) {
	double expire = now() + SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL;
	int64_t bytesReadPerKSecond =
	    bytesReadSample.addAndExpire(key, in, expire) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	int64_t opsReadPerKSecond =
	    opsReadSample.addAndExpire(key, 1, expire) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;

	if (bytesReadPerKSecond > 0 || opsReadPerKSecond > 0) {
		StorageMetrics notifyMetrics;
		notifyMetrics.bytesReadPerKSecond = bytesReadPerKSecond;
		notifyMetrics.opsReadPerKSecond = opsReadPerKSecond;
		auto& v = waitMetricsMap[key];
		for (int i = 0; i < v.size(); i++) {
			CODE_PROBE(bytesReadPerKSecond > 0, "ShardNotifyMetrics bytesRead");
			CODE_PROBE(opsReadPerKSecond > 0, "ShardNotifyMetrics opsRead");
			v[i].send(notifyMetrics);
		}
	}
}

// Called by StorageServerDisk when the size of a key in byteSample changes, to notify WaitMetricsRequest
// Should not be called for keys past allKeys.end
void StorageServerMetrics::notifyBytes(
    RangeMap<Key, std::vector<PromiseStream<StorageMetrics>>, KeyRangeRef>::iterator shard,
    int64_t bytes) {
	ASSERT(shard.end() <= allKeys.end);

	StorageMetrics notifyMetrics;
	notifyMetrics.bytes = bytes;
	auto size = shard->cvalue().size();
	for (int i = 0; i < size; i++) {
		// fmt::print("NotifyBytes {} {}\n", shard->value().size(), shard->range().toString());
		CODE_PROBE(true, "notifyBytes");
		shard.value()[i].send(notifyMetrics);
	}
}

// Called by StorageServerDisk when the size of a key in byteSample changes, to notify WaitMetricsRequest
void StorageServerMetrics::notifyBytes(const KeyRef& key, int64_t bytes) {
	if (key >= allKeys.end) // Do not notify on changes to internal storage server state
		return;

	notifyBytes(waitMetricsMap.rangeContaining(key), bytes);
}

// Called when a range of keys becomes unassigned (and therefore not readable), to notify waiting
// WaitMetricsRequests (also other types of wait
//   requests in the future?)
void StorageServerMetrics::notifyNotReadable(KeyRangeRef keys) {
	auto rs = waitMetricsMap.intersectingRanges(keys);
	for (auto r = rs.begin(); r != rs.end(); ++r) {
		auto& v = r->value();
		CODE_PROBE(v.size(), "notifyNotReadable() sending errors to intersecting ranges");
		for (int n = 0; n < v.size(); n++)
			v[n].sendError(wrong_shard_server());
	}
}

// Called periodically (~1 sec intervals) to remove older IOs from the averages
// Removes old entries from metricsAverageQueue, updates metricsSampleMap accordingly, and notifies
//   WaitMetricsRequests through waitMetricsMap.
void StorageServerMetrics::poll() {
	{
		StorageMetrics m;
		m.bytesWrittenPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		bytesWriteSample.poll(waitMetricsMap, m);
	}
	{
		StorageMetrics m;
		m.iosPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		iopsSample.poll(waitMetricsMap, m);
	}
	{
		StorageMetrics m;
		m.bytesReadPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		bytesReadSample.poll(waitMetricsMap, m);
	}
	{
		StorageMetrics m;
		m.opsReadPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		opsReadSample.poll(waitMetricsMap, m);
	}
	// bytesSample doesn't need polling because we never call addExpire() on it
}

// This function can run on untrusted user data.  We must validate all divisions carefully.
KeyRef StorageServerMetrics::getSplitKey(int64_t remaining,
                                         int64_t estimated,
                                         int64_t limits,
                                         int64_t used,
                                         int64_t infinity,
                                         bool isLastShard,
                                         const StorageMetricSample& sample,
                                         double divisor,
                                         KeyRef const& lastKey,
                                         KeyRef const& key,
                                         bool hasUsed) const {
	ASSERT(remaining >= 0);
	ASSERT(limits > 0);
	ASSERT(divisor > 0);

	if (limits < infinity / 2) {
		int64_t expectedSize;
		if (isLastShard || remaining > estimated) {
			double remaining_divisor = (double(remaining) / limits) + 0.5;
			expectedSize = remaining / remaining_divisor;
		} else {
			// If we are here, then estimated >= remaining >= 0
			double estimated_divisor = (double(estimated) / limits) + 0.5;
			expectedSize = remaining / estimated_divisor;
		}

		if (remaining > expectedSize) {
			// This does the conversion from native units to bytes using the divisor.
			double offset = (expectedSize - used) / divisor;
			if (offset <= 0)
				return hasUsed ? lastKey : key;
			return sample.splitEstimate(
			    KeyRangeRef(lastKey, key),
			    offset * ((1.0 - SERVER_KNOBS->SPLIT_JITTER_AMOUNT) +
			              2 * deterministicRandom()->random01() * SERVER_KNOBS->SPLIT_JITTER_AMOUNT));
		}
	}

	return key;
}

void StorageServerMetrics::splitMetrics(SplitMetricsRequest req) const {
	int minSplitBytes = req.minSplitBytes.present() ? req.minSplitBytes.get() : SERVER_KNOBS->MIN_SHARD_BYTES;
	int minSplitWriteTraffic = SERVER_KNOBS->SHARD_SPLIT_BYTES_PER_KSEC;
	try {
		SplitMetricsReply reply;
		KeyRef lastKey = req.keys.begin;
		StorageMetrics used = req.used;
		StorageMetrics estimated = req.estimated;
		StorageMetrics remaining = getMetrics(req.keys) + used;

		//TraceEvent("SplitMetrics").detail("Begin", req.keys.begin).detail("End", req.keys.end).detail("Remaining", remaining.bytes).detail("Used", used.bytes).detail("MinSplitBytes", minSplitBytes);

		while (true) {
			if (remaining.bytes < 2 * minSplitBytes && (!SERVER_KNOBS->ENABLE_WRITE_BASED_SHARD_SPLIT ||
			                                            remaining.bytesWrittenPerKSecond < minSplitWriteTraffic))
				break;
			KeyRef key = req.keys.end;
			bool hasUsed = used.bytes != 0 || used.bytesWrittenPerKSecond != 0 || used.iosPerKSecond != 0;
			key = getSplitKey(remaining.bytes,
			                  estimated.bytes,
			                  req.limits.bytes,
			                  used.bytes,
			                  req.limits.infinity,
			                  req.isLastShard,
			                  byteSample,
			                  1,
			                  lastKey,
			                  key,
			                  hasUsed);
			if (used.bytes < minSplitBytes)
				key = std::max(
				    key, byteSample.splitEstimate(KeyRangeRef(lastKey, req.keys.end), minSplitBytes - used.bytes));
			key = getSplitKey(remaining.iosPerKSecond,
			                  estimated.iosPerKSecond,
			                  req.limits.iosPerKSecond,
			                  used.iosPerKSecond,
			                  req.limits.infinity,
			                  req.isLastShard,
			                  iopsSample,
			                  SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS,
			                  lastKey,
			                  key,
			                  hasUsed);
			key = getSplitKey(remaining.bytesWrittenPerKSecond,
			                  estimated.bytesWrittenPerKSecond,
			                  req.limits.bytesWrittenPerKSecond,
			                  used.bytesWrittenPerKSecond,
			                  req.limits.infinity,
			                  req.isLastShard,
			                  bytesWriteSample,
			                  SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS,
			                  lastKey,
			                  key,
			                  hasUsed);
			ASSERT(key != lastKey || hasUsed);
			if (key == req.keys.end)
				break;
			reply.splits.push_back_deep(reply.splits.arena(), key);
			if (reply.splits.size() > SERVER_KNOBS->SPLIT_METRICS_MAX_ROWS) {
				reply.more = true;
				break;
			}

			StorageMetrics diff = (getMetrics(KeyRangeRef(lastKey, key)) + used);
			remaining -= diff;
			estimated -= diff;

			used = StorageMetrics();
			lastKey = key;
		}

		reply.used = reply.more ? StorageMetrics() : getMetrics(KeyRangeRef(lastKey, req.keys.end)) + used;
		req.reply.send(reply);
	} catch (Error& e) {
		req.reply.sendError(e);
	}
}

void StorageServerMetrics::getStorageMetrics(GetStorageMetricsRequest req,
                                             StorageBytes sb,
                                             double bytesInputRate,
                                             int64_t versionLag,
                                             double lastUpdate,
                                             int64_t bytesDurable,
                                             int64_t bytesInput,
                                             int ongoingBulkLoadTaskCount) const {
	GetStorageMetricsReply rep;

	// SOMEDAY: make bytes dynamic with hard disk space
	rep.load = getMetrics(allKeys);

	if (sb.free < 1e9) {
		TraceEvent(SevWarn, "PhysicalDiskMetrics")
		    .suppressFor(60.0)
		    .detail("Free", sb.free)
		    .detail("Total", sb.total)
		    .detail("Available", sb.available)
		    .detail("Load", rep.load.bytes);
	}

	rep.available.bytes = sb.available;
	rep.available.iosPerKSecond = 10e6;
	rep.available.bytesWrittenPerKSecond = 100e9;
	rep.available.bytesReadPerKSecond = 100e9;

	rep.capacity.bytes = sb.total;
	rep.capacity.iosPerKSecond = 10e6;
	rep.capacity.bytesWrittenPerKSecond = 100e9;
	rep.capacity.bytesReadPerKSecond = 100e9;

	rep.bytesInputRate = bytesInputRate;

	rep.versionLag = versionLag;
	rep.lastUpdate = lastUpdate;

	rep.bytesDurable = bytesDurable;
	rep.bytesInput = bytesInput;

	rep.ongoingBulkLoadTaskCount = ongoingBulkLoadTaskCount;

	req.reply.send(rep);
}

// Equally split the metrics (specified by splitType) of parentRange into chunkCount and return all the sampled metrics
// (bytes, readBytes and readOps) of each chunk
// NOTE: update unit test "equalDivide" after change
std::vector<ReadHotRangeWithMetrics> StorageServerMetrics::getReadHotRanges(KeyRangeRef parentRange,
                                                                            int chunkCount,
                                                                            uint8_t splitType) const {
	const StorageMetricSample* sampler = nullptr;
	switch (splitType) {
	case ReadHotSubRangeRequest::SplitType::BYTES:
		sampler = &byteSample;
		break;
	case ReadHotSubRangeRequest::SplitType::READ_BYTES:
		sampler = &bytesReadSample;
		break;
	case ReadHotSubRangeRequest::SplitType::READ_OPS:
		sampler = &opsReadSample;
		break;
	default:
		ASSERT(false);
	}

	std::vector<ReadHotRangeWithMetrics> toReturn;
	if (sampler->sample.empty()) {
		return toReturn;
	}

	auto total = sampler->getEstimate(parentRange);
	double splitChunk = std::max(1.0, (double)total / chunkCount);

	KeyRef beginKey = parentRange.begin;
	while (parentRange.contains(beginKey)) {
		auto beginIt = sampler->sample.lower_bound(beginKey);
		if (beginIt == sampler->sample.end()) {
			break;
		}
		auto endIt = sampler->sample.index(sampler->sample.sumTo(beginIt) + splitChunk - 1);
		// because index return x where sumTo(x+1) (that including sample at x) > metrics, we have to forward endIt here
		if (endIt != sampler->sample.end())
			++endIt;

		if (endIt == sampler->sample.end()) {
			KeyRangeRef lastRange(beginKey, parentRange.end);
			toReturn.emplace_back(
			    lastRange,
			    byteSample.getEstimate(lastRange),
			    (double)bytesReadSample.getEstimate(lastRange) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL,
			    (double)opsReadSample.getEstimate(lastRange) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
			break;
		}

		ASSERT_LT(beginKey, *endIt);
		KeyRangeRef range(beginKey, *endIt);
		toReturn.emplace_back(
		    range,
		    byteSample.getEstimate(range),
		    (double)bytesReadSample.getEstimate(range) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL,
		    (double)opsReadSample.getEstimate(range) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);

		beginKey = *endIt;
	}
	return toReturn;
}

// Given a read hot shard, this function will divide the shard into chunks and find those chunks whose
// readBytes/sizeBytes exceeds the `readDensityRatio`. Please make sure to run unit tests
// `StorageMetricsSampleTests.txt` after change made.
std::vector<ReadHotRangeWithMetrics> StorageServerMetrics::_getReadHotRanges(
    KeyRangeRef shard,
    double readDensityRatio,
    int64_t baseChunkSize,
    int64_t minShardReadBandwidthPerKSeconds) const {
	std::vector<ReadHotRangeWithMetrics> toReturn;

	double shardSize = (double)byteSample.getEstimate(shard);
	int64_t shardReadBandwidth = bytesReadSample.getEstimate(shard);
	if (shardReadBandwidth * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS <=
	    minShardReadBandwidthPerKSeconds) {
		return toReturn;
	}
	if (shardSize <= baseChunkSize) {
		// Shard is small, use it as is
		if (bytesReadSample.getEstimate(shard) > (readDensityRatio * shardSize)) {
			toReturn.emplace_back(shard,
			                      bytesReadSample.getEstimate(shard) / shardSize,
			                      bytesReadSample.getEstimate(shard) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
		}
		return toReturn;
	}
	KeyRef beginKey = shard.begin;
	auto endKey =
	    byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + baseChunkSize);
	while (endKey != byteSample.sample.end()) {
		if (*endKey > shard.end) {
			endKey = byteSample.sample.lower_bound(shard.end);
			if (*endKey == beginKey) {
				// No need to increment endKey since otherwise it would stuck here forever.
				break;
			}
		}
		if (*endKey == beginKey) {
			++endKey;
			continue;
		}
		if (bytesReadSample.getEstimate(KeyRangeRef(beginKey, *endKey)) >
		    (readDensityRatio * std::max(baseChunkSize, byteSample.getEstimate(KeyRangeRef(beginKey, *endKey))))) {
			auto range = KeyRangeRef(beginKey, *endKey);
			if (!toReturn.empty() && toReturn.back().keys.end == range.begin) {
				// in case two consecutive chunks both are over the ratio, merge them.
				range = KeyRangeRef(toReturn.back().keys.begin, *endKey);
				toReturn.pop_back();
			}
			toReturn.emplace_back(range,
			                      (double)bytesReadSample.getEstimate(range) /
			                          std::max(baseChunkSize, byteSample.getEstimate(range)),
			                      bytesReadSample.getEstimate(range) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
		}
		beginKey = *endKey;
		endKey =
		    byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + baseChunkSize);
	}
	return toReturn;
}

int64_t StorageServerMetrics::getHotShards(const KeyRange& range) const {
	auto bytesWrittenPerKSecond =
	    bytesWriteSample.getEstimate(range) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	return bytesWrittenPerKSecond;
}

void StorageServerMetrics::getReadHotRanges(ReadHotSubRangeRequest req) const {
	ReadHotSubRangeReply reply;
	auto _ranges = getReadHotRanges(req.keys, req.chunkCount, req.type);
	reply.readHotRanges = VectorRef(_ranges.data(), _ranges.size());
	req.reply.send(reply);
}

void StorageServerMetrics::getSplitPoints(SplitRangeRequest req, Optional<KeyRef> prefix) const {
	SplitRangeReply reply;
	KeyRangeRef range = req.keys;
	if (prefix.present()) {
		range = range.withPrefix(prefix.get(), req.arena);
	}

	std::vector<KeyRef> points = getSplitPoints(range, req.chunkSize, prefix);

	reply.splitPoints.append_deep(reply.splitPoints.arena(), points.data(), points.size());
	req.reply.send(reply);
}

std::vector<KeyRef> StorageServerMetrics::getSplitPoints(KeyRangeRef range,
                                                         int64_t chunkSize,
                                                         Optional<KeyRef> prefixToRemove) const {
	std::vector<KeyRef> toReturn;
	KeyRef beginKey = range.begin;
	IndexedSet<Key, int64_t>::const_iterator endKey =
	    byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + chunkSize);
	while (endKey != byteSample.sample.end()) {
		if (*endKey > range.end) {
			break;
		}
		if (*endKey == beginKey) {
			++endKey;
			continue;
		}
		KeyRef splitPoint = *endKey;
		if (prefixToRemove.present()) {
			splitPoint = splitPoint.removePrefix(prefixToRemove.get());
		}
		toReturn.push_back(splitPoint);
		beginKey = *endKey;
		endKey = byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + chunkSize);
	}
	return toReturn;
}

void StorageServerMetrics::collapse(KeyRangeMap<int>& map, KeyRef const& key) {
	auto range = map.rangeContaining(key);
	if (range == map.ranges().begin() || range == map.ranges().end())
		return;
	int value = range->value();
	auto prev = range;
	--prev;
	if (prev->value() != value)
		return;
	KeyRange keys = KeyRangeRef(prev->begin(), range->end());
	map.insert(keys, value);
}

void StorageServerMetrics::add(KeyRangeMap<int>& map, KeyRangeRef const& keys, int delta) {
	auto rs = map.modify(keys);
	for (auto r = rs.begin(); r != rs.end(); ++r)
		r->value() += delta;
	collapse(map, keys.begin);
	collapse(map, keys.end);
}

// Returns the sampled metric value (possibly 0, possibly increased by the sampling factor)
int64_t TransientStorageMetricSample::addAndExpire(const Key& key, int64_t metric, double expiration) {
	auto x = add(key, metric);
	if (x)
		queue.emplace_back(expiration, std::make_pair(key, -x));
	return x;
}

// FIXME: both versions of erase are broken, because they do not remove items in the queue with will subtract a
// metric from the value sometime in the future
int64_t TransientStorageMetricSample::erase(KeyRef key) {
	auto it = sample.find(key);
	if (it == sample.end())
		return 0;
	int64_t x = sample.getMetric(it);
	sample.erase(it);
	return x;
}

void TransientStorageMetricSample::erase(KeyRangeRef keys) {
	sample.erase(keys.begin, keys.end);
}

bool TransientStorageMetricSample::roll(int64_t metric) const {
	return deterministicRandom()->random01() < (double)metric / metricUnitsPerSample; //< SOMEDAY: Better randomInt64?
}

void TransientStorageMetricSample::poll(KeyRangeMap<std::vector<PromiseStream<StorageMetrics>>>& waitMap,
                                        StorageMetrics metrics) {
	double now = ::now();
	while (queue.size() && queue.front().first <= now) {
		KeyRef key = queue.front().second.first;
		int64_t delta = queue.front().second.second;
		ASSERT(delta != 0);

		auto [m, it] = sample.addMetric(key, delta);
		if (m == 0)
			sample.erase(it);

		StorageMetrics deltaM = metrics * delta;
		auto v = waitMap[key];
		for (int i = 0; i < v.size(); i++) {
			CODE_PROBE(true, "TransientStorageMetricSample poll update");
			v[i].send(deltaM);
		}

		queue.pop_front();
	}
}

void TransientStorageMetricSample::poll() {
	double now = ::now();
	while (queue.size() && queue.front().first <= now) {
		KeyRef key = queue.front().second.first;
		int64_t delta = queue.front().second.second;
		ASSERT(delta != 0);

		auto [m, it] = sample.addMetric(key, delta);
		if (m == 0)
			sample.erase(it);

		queue.pop_front();
	}
}

int64_t TransientStorageMetricSample::add(const Key& key, int64_t metric) {
	if (!metric)
		return 0;
	int64_t mag = metric < 0 ? -metric : metric;

	if (mag < metricUnitsPerSample) {
		if (!roll(mag))
			return 0;
		metric = metric < 0 ? -metricUnitsPerSample : metricUnitsPerSample;
	}

	auto [m, it] = sample.addMetric(key, metric);
	if (m == 0)
		sample.erase(it);

	return metric;
}

															#line 690 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase688()
															#line 688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase688Actor>
															#line 688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase688ActorState {
															#line 697 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
															#line 688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase688ActorState(UnitTestParameters const& params) 
															#line 688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 704 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase688", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase688ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase688", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 689 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageMetricSample s(1000);
															#line 690 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Apple"_sr, 1000);
															#line 691 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Banana"_sr, 2000);
															#line 692 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Cat"_sr, 1000);
															#line 693 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Cathode"_sr, 1000);
															#line 694 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Dog"_sr, 1000);
															#line 696 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(s.getEstimate(KeyRangeRef("A"_sr, "D"_sr)) == 5000);
															#line 697 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(s.getEstimate(KeyRangeRef("A"_sr, "E"_sr)) == 6000);
															#line 698 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(s.getEstimate(KeyRangeRef("B"_sr, "C"_sr)) == 2000);
															#line 703 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase688Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase688ActorState(); static_cast<FlowTestCase688Actor*>(this)->destroy(); return 0; }
															#line 737 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase688Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase688ActorState();
			static_cast<FlowTestCase688Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase688ActorState();
		static_cast<FlowTestCase688Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 761 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase688()
															#line 688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase688Actor final : public Actor<Void>, public FastAllocated<FlowTestCase688Actor>, public FlowTestCase688ActorState<FlowTestCase688Actor> {
															#line 766 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase688Actor>::operator new;
	using FastAllocated<FlowTestCase688Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2960322457769833216UL, 10448111468965073152UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase688Actor(UnitTestParameters const& params) 
															#line 782 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase688ActorState<FlowTestCase688Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase688", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4809578758301895168UL, 15566773527449175552UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase688Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase688");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase688", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase688( UnitTestParameters const& params ) {
															#line 688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase688Actor(params));
															#line 814 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase688, "/fdbserver/StorageMetricSample/simple")

#line 705 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 820 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase706()
															#line 706 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase706Actor>
															#line 706 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase706ActorState {
															#line 827 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
															#line 706 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase706ActorState(UnitTestParameters const& params) 
															#line 706 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 706 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 834 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase706", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase706ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase706", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 708 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 709 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 711 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 200 * sampleUnit);
															#line 712 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 800 * sampleUnit);
															#line 713 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 714 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 716 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 717 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 718 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 720 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 2000 * sampleUnit, {});
															#line 722 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 1 && t[0] == "Bah"_sr);
															#line 724 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase706Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase706ActorState(); static_cast<FlowTestCase706Actor*>(this)->destroy(); return 0; }
															#line 873 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase706Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase706ActorState();
			static_cast<FlowTestCase706Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase706ActorState();
		static_cast<FlowTestCase706Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 706 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 897 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase706()
															#line 706 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase706Actor final : public Actor<Void>, public FastAllocated<FlowTestCase706Actor>, public FlowTestCase706ActorState<FlowTestCase706Actor> {
															#line 902 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase706Actor>::operator new;
	using FastAllocated<FlowTestCase706Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10715208995651661568UL, 17968286665067096576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 706 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase706Actor(UnitTestParameters const& params) 
															#line 918 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase706ActorState<FlowTestCase706Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase706", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12388691682232539136UL, 11847394637888134144UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase706Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase706");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase706", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 706 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase706( UnitTestParameters const& params ) {
															#line 706 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase706Actor(params));
															#line 950 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase706, "/fdbserver/StorageMetricSample/rangeSplitPoints/simple")

#line 726 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 956 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase727()
															#line 727 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase727Actor>
															#line 727 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase727ActorState {
															#line 963 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
															#line 727 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase727ActorState(UnitTestParameters const& params) 
															#line 727 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 727 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 970 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase727", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase727ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase727", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 729 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 730 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 732 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 200 * sampleUnit);
															#line 733 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 800 * sampleUnit);
															#line 734 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 735 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 736 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 737 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 738 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 739 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 741 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 600 * sampleUnit, {});
															#line 743 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 3 && t[0] == "Absolute"_sr && t[1] == "Apple"_sr && t[2] == "Bah"_sr);
															#line 745 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase727Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase727ActorState(); static_cast<FlowTestCase727Actor*>(this)->destroy(); return 0; }
															#line 1009 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase727Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase727ActorState();
			static_cast<FlowTestCase727Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase727ActorState();
		static_cast<FlowTestCase727Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 727 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1033 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase727()
															#line 727 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase727Actor final : public Actor<Void>, public FastAllocated<FlowTestCase727Actor>, public FlowTestCase727ActorState<FlowTestCase727Actor> {
															#line 1038 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase727Actor>::operator new;
	using FastAllocated<FlowTestCase727Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1269515343530246912UL, 14514544799021298176UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 727 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase727Actor(UnitTestParameters const& params) 
															#line 1054 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase727ActorState<FlowTestCase727Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase727", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4409634954555124992UL, 16751488767419794688UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase727Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase727");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase727", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 727 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase727( UnitTestParameters const& params ) {
															#line 727 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase727Actor(params));
															#line 1086 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase727, "/fdbserver/StorageMetricSample/rangeSplitPoints/multipleReturnedPoints")

#line 747 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1092 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase748()
															#line 748 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase748Actor>
															#line 748 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase748ActorState {
															#line 1099 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
															#line 748 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase748ActorState(UnitTestParameters const& params) 
															#line 748 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 748 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1106 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase748", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase748ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase748", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 750 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 751 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 753 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 200 * sampleUnit);
															#line 754 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 800 * sampleUnit);
															#line 755 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 756 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 757 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 758 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 759 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 760 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 762 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 10000 * sampleUnit, {});
															#line 764 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 0);
															#line 766 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase748Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase748ActorState(); static_cast<FlowTestCase748Actor*>(this)->destroy(); return 0; }
															#line 1145 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase748Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase748ActorState();
			static_cast<FlowTestCase748Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase748ActorState();
		static_cast<FlowTestCase748Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 748 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1169 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase748()
															#line 748 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase748Actor final : public Actor<Void>, public FastAllocated<FlowTestCase748Actor>, public FlowTestCase748ActorState<FlowTestCase748Actor> {
															#line 1174 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase748Actor>::operator new;
	using FastAllocated<FlowTestCase748Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(597254268661998592UL, 5431304298669431808UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 748 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase748Actor(UnitTestParameters const& params) 
															#line 1190 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase748ActorState<FlowTestCase748Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase748", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18431792263612847104UL, 13893518547466676480UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase748Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase748");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase748", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 748 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase748( UnitTestParameters const& params ) {
															#line 748 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase748Actor(params));
															#line 1222 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase748, "/fdbserver/StorageMetricSample/rangeSplitPoints/noneSplitable")

#line 768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1228 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase769()
															#line 769 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase769Actor>
															#line 769 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase769ActorState {
															#line 1235 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
															#line 769 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase769ActorState(UnitTestParameters const& params) 
															#line 769 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 769 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1242 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase769", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase769ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase769", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 771 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 772 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 774 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 775 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 776 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 10 * sampleUnit);
															#line 777 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 778 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 779 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 20 * sampleUnit);
															#line 780 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 10 * sampleUnit);
															#line 781 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 30 * sampleUnit);
															#line 783 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 1000 * sampleUnit, {});
															#line 785 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 0);
															#line 787 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase769Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase769ActorState(); static_cast<FlowTestCase769Actor*>(this)->destroy(); return 0; }
															#line 1281 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase769Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase769ActorState();
			static_cast<FlowTestCase769Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase769ActorState();
		static_cast<FlowTestCase769Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 769 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1305 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase769()
															#line 769 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase769Actor final : public Actor<Void>, public FastAllocated<FlowTestCase769Actor>, public FlowTestCase769ActorState<FlowTestCase769Actor> {
															#line 1310 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase769Actor>::operator new;
	using FastAllocated<FlowTestCase769Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11776977928882867968UL, 4917475852475666176UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 769 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase769Actor(UnitTestParameters const& params) 
															#line 1326 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase769ActorState<FlowTestCase769Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase769", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10515145315543764480UL, 4019529408049027584UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase769Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase769");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase769", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 769 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase769( UnitTestParameters const& params ) {
															#line 769 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase769Actor(params));
															#line 1358 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase769, "/fdbserver/StorageMetricSample/rangeSplitPoints/chunkTooLarge")

#line 789 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1364 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase790()
															#line 790 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase790Actor>
															#line 790 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase790ActorState {
															#line 1371 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
															#line 790 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase790ActorState(UnitTestParameters const& params) 
															#line 790 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 790 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1378 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase790", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase790ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase790", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 792 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 793 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 795 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 796 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 797 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 798 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 799 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 1000 * sampleUnit);
															#line 801 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 802 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 803 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 804 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 805 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 806 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 807 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 808 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 810 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm._getReadHotRanges(KeyRangeRef("A"_sr, "C"_sr), 2.0, 200 * sampleUnit, 0);
															#line 813 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 1 && (*t.begin()).keys.begin == "Bah"_sr && (*t.begin()).keys.end == "Bob"_sr);
															#line 815 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase790Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase790ActorState(); static_cast<FlowTestCase790Actor*>(this)->destroy(); return 0; }
															#line 1427 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase790Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase790ActorState();
			static_cast<FlowTestCase790Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase790ActorState();
		static_cast<FlowTestCase790Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 790 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1451 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase790()
															#line 790 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase790Actor final : public Actor<Void>, public FastAllocated<FlowTestCase790Actor>, public FlowTestCase790ActorState<FlowTestCase790Actor> {
															#line 1456 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase790Actor>::operator new;
	using FastAllocated<FlowTestCase790Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12816386691689697280UL, 7560832940618717440UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 790 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase790Actor(UnitTestParameters const& params) 
															#line 1472 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase790ActorState<FlowTestCase790Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase790", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14360389705571275264UL, 11998061871995402496UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase790Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase790");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase790", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 790 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase790( UnitTestParameters const& params ) {
															#line 790 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase790Actor(params));
															#line 1504 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase790, "/fdbserver/StorageMetricSample/readHotDetect/simple")

#line 817 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1510 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase818()
															#line 818 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase818Actor>
															#line 818 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase818ActorState {
															#line 1517 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
															#line 818 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase818ActorState(UnitTestParameters const& params) 
															#line 818 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 818 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1524 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase818", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase818ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase818", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 820 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 821 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 823 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 824 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 825 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 826 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 827 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 1000 * sampleUnit);
															#line 828 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Final"_sr, 2000 * sampleUnit);
															#line 830 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 831 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 832 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 833 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 834 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 835 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 836 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 837 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 838 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Dah"_sr, 300 * sampleUnit);
															#line 840 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm._getReadHotRanges(KeyRangeRef("A"_sr, "D"_sr), 2.0, 200 * sampleUnit, 0);
															#line 843 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 2 && (*t.begin()).keys.begin == "Bah"_sr && (*t.begin()).keys.end == "Bob"_sr);
															#line 844 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.at(1).keys.begin == "Cat"_sr && t.at(1).keys.end == "Dah"_sr);
															#line 846 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase818Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase818ActorState(); static_cast<FlowTestCase818Actor*>(this)->destroy(); return 0; }
															#line 1579 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase818Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase818ActorState();
			static_cast<FlowTestCase818Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase818ActorState();
		static_cast<FlowTestCase818Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 818 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1603 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase818()
															#line 818 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase818Actor final : public Actor<Void>, public FastAllocated<FlowTestCase818Actor>, public FlowTestCase818ActorState<FlowTestCase818Actor> {
															#line 1608 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase818Actor>::operator new;
	using FastAllocated<FlowTestCase818Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14676651678586358784UL, 13648208438415546368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 818 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase818Actor(UnitTestParameters const& params) 
															#line 1624 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase818ActorState<FlowTestCase818Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase818", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7549350623616053504UL, 5702060599045202944UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase818Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase818");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase818", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 818 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase818( UnitTestParameters const& params ) {
															#line 818 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase818Actor(params));
															#line 1656 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase818, "/fdbserver/StorageMetricSample/readHotDetect/moreThanOneRange")

#line 848 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1662 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase849()
															#line 849 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase849Actor>
															#line 849 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase849ActorState {
															#line 1669 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
															#line 849 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase849ActorState(UnitTestParameters const& params) 
															#line 849 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 849 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1676 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase849", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase849ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase849", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 851 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 852 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 854 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 855 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 856 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Bucket"_sr, 2000 * sampleUnit);
															#line 857 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 858 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 859 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 5000 * sampleUnit);
															#line 860 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Final"_sr, 2000 * sampleUnit);
															#line 862 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 863 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 864 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 865 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 866 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 867 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 868 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 869 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 870 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Dah"_sr, 300 * sampleUnit);
															#line 872 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm._getReadHotRanges(KeyRangeRef("A"_sr, "D"_sr), 2.0, 200 * sampleUnit, 0);
															#line 875 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 2 && (*t.begin()).keys.begin == "Bah"_sr && (*t.begin()).keys.end == "But"_sr);
															#line 876 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.at(1).keys.begin == "Cat"_sr && t.at(1).keys.end == "Dah"_sr);
															#line 878 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase849Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase849ActorState(); static_cast<FlowTestCase849Actor*>(this)->destroy(); return 0; }
															#line 1733 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase849Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase849ActorState();
			static_cast<FlowTestCase849Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase849ActorState();
		static_cast<FlowTestCase849Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 849 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1757 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase849()
															#line 849 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase849Actor final : public Actor<Void>, public FastAllocated<FlowTestCase849Actor>, public FlowTestCase849ActorState<FlowTestCase849Actor> {
															#line 1762 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase849Actor>::operator new;
	using FastAllocated<FlowTestCase849Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1343683221695224320UL, 9045704832350027008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 849 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase849Actor(UnitTestParameters const& params) 
															#line 1778 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase849ActorState<FlowTestCase849Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase849", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(149511579398379520UL, 588255415613623040UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase849Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase849");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase849", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 849 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase849( UnitTestParameters const& params ) {
															#line 849 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase849Actor(params));
															#line 1810 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase849, "/fdbserver/StorageMetricSample/readHotDetect/consecutiveRanges")

#line 880 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1816 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase881()
															#line 881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase881Actor>
															#line 881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase881ActorState {
															#line 1823 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
															#line 881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase881ActorState(UnitTestParameters const& params) 
															#line 881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1830 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase881", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase881ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase881", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 883 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 884 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 888 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 889 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 891 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Bucket"_sr, 2000 * sampleUnit);
															#line 893 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 894 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 896 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 5000 * sampleUnit);
															#line 898 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Final"_sr, 2000 * sampleUnit);
															#line 901 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20);
															#line 902 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80);
															#line 903 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000);
															#line 904 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20);
															#line 905 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80);
															#line 906 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200);
															#line 908 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100);
															#line 910 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300);
															#line 911 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Dah"_sr, 300);
															#line 914 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm.getReadHotRanges(KeyRangeRef("Y"_sr, "Z"_sr), 7, ReadHotSubRangeRequest::SplitType::READ_BYTES);
															#line 916 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.size(), 0);
															#line 919 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			t = ssm.getReadHotRanges(KeyRangeRef(""_sr, "\xff"_sr), 7, ReadHotSubRangeRequest::SplitType::READ_BYTES);
															#line 920 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.size(), 5);
															#line 924 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ((*t.begin()).keys.begin, ""_sr);
															#line 926 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ((*t.begin()).keys.end, "Bucket"_sr);
															#line 927 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t[0].bytes, 1400);
															#line 929 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(1).keys.begin, "Bucket"_sr);
															#line 930 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(1).keys.end, "Cat"_sr);
															#line 932 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(2).bytes, 600);
															#line 933 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(3).readBandwidthSec, 5000 * sampleUnit / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
															#line 934 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(3).bytes, 0);
															#line 935 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase881Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase881ActorState(); static_cast<FlowTestCase881Actor*>(this)->destroy(); return 0; }
															#line 1905 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase881Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase881ActorState();
			static_cast<FlowTestCase881Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase881ActorState();
		static_cast<FlowTestCase881Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1929 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase881()
															#line 881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase881Actor final : public Actor<Void>, public FastAllocated<FlowTestCase881Actor>, public FlowTestCase881ActorState<FlowTestCase881Actor> {
															#line 1934 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase881Actor>::operator new;
	using FastAllocated<FlowTestCase881Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15501192906881411072UL, 5838455467279576832UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase881Actor(UnitTestParameters const& params) 
															#line 1950 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase881ActorState<FlowTestCase881Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase881", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14122163549893164288UL, 17159159028816462080UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase881Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase881");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase881", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase881( UnitTestParameters const& params ) {
															#line 881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase881Actor(params));
															#line 1982 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase881, "/fdbserver/StorageMetricSample/readHotDetect/equalDivide")

#line 937 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
