#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
/*
 * SimulatedCluster.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <cstdint>
#include <fstream>
#include <ostream>
#include <set>
#include <sstream>
#include <string_view>
#include <unordered_map>

#include <toml.hpp>

#include "fdbclient/DatabaseConfiguration.h"
#include "fdbclient/FDBTypes.h"
#include "fdbrpc/Locality.h"
#include "fdbrpc/simulator.h"
#include "fdbrpc/IPAllowList.h"
#include "fdbrpc/SimulatorProcessInfo.h"
#include "fdbclient/ClusterConnectionFile.h"
#include "fdbclient/ClusterConnectionMemoryRecord.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbserver/TesterInterface.actor.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "fdbclient/ClusterInterface.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/CoordinationInterface.h"
#include "fdbclient/SimpleIni.h"
#include "fdbrpc/AsyncFileNonDurable.actor.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/BackupAgent.actor.h"
#include "fdbclient/versions.h"
#include "flow/IRandom.h"
#include "flow/MkCert.h"
#include "flow/ProcessEvents.h"
#include "fdbrpc/WellKnownEndpoints.h"
#include "flow/ProtocolVersion.h"
#include "flow/flow.h"
#include "flow/network.h"
#include "flow/TypeTraits.h"
#include "flow/FaultInjection.h"
#include "flow/CodeProbeUtils.h"
#include "fdbserver/SimulatedCluster.h"
#include "flow/IConnection.h"
#include "fdbserver/MockGlobalState.h"
#include "flow/actorcompiler.h" // This must be the last #include.

#undef max
#undef min

extern "C" int g_expect_full_pointermap;
extern const char* getSourceVersion();

using namespace std::literals;

bool isSimulatorProcessUnreliable() {
	return g_network->isSimulated() && !g_simulator->getCurrentProcess()->isReliable();
}

namespace {

constexpr bool hasRocksDB =
#ifdef WITH_ROCKSDB
    true
#else
    false
#endif
    ;

} // anonymous namespace

namespace probe {

namespace assert {

struct HasRocksDB {
	constexpr static AnnotationType type = AnnotationType::Assertion;
	constexpr bool operator()(ICodeProbe const* self) const { return ::hasRocksDB; }
};

constexpr HasRocksDB hasRocksDB;

} // namespace assert

} // namespace probe

// TODO: Defining these here is just asking for ODR violations.
template <>
std::string describe(bool const& val) {
	return val ? "true" : "false";
}

template <>
std::string describe(int const& val) {
	return format("%d", val);
}

template <>
std::string describe(const SimulationStorageEngine& val) {
	return std::to_string(static_cast<uint32_t>(val));
}

namespace {

bool isValidSimulationStorageEngineValue(const std::underlying_type_t<SimulationStorageEngine>& value) {
	return value < static_cast<std::underlying_type_t<SimulationStorageEngine>>(
	                   SimulationStorageEngine::SIMULATION_STORAGE_ENGINE_INVALID_VALUE);
}

constexpr int MACHINE_REBOOT_TIME = 10;

bool destructed = false;

// Configuration details specified in workload test files that change the simulation
// environment details
class TestConfig : public BasicTestConfig {
	class ConfigBuilder {
		using value_type = toml::value;
		using base_variant = std::variant<int,
		                                  float,
		                                  double,
		                                  bool,
		                                  std::string,
		                                  std::vector<int>,
		                                  std::vector<std::string>,
		                                  ConfigDBType,
		                                  SimulationStorageEngine,
		                                  std::set<SimulationStorageEngine>>;
		using types =
		    variant_map<variant_concat<base_variant, variant_map<base_variant, Optional>>, std::add_pointer_t>;
		std::unordered_map<std::string_view, types> confMap;

		struct visitor {
			const value_type& value;
			visitor(const value_type& v) : value(v) {}
			void operator()(int* val) const { *val = value.as_integer(); }
			void operator()(Optional<int>* val) const { *val = value.as_integer(); }
			void operator()(float* val) const { *val = value.as_floating(); }
			void operator()(Optional<float>* val) const { *val = value.as_floating(); }
			void operator()(double* val) const { *val = value.as_floating(); }
			void operator()(Optional<double>* val) const { *val = value.as_floating(); }
			void operator()(bool* val) const { *val = value.as_boolean(); }
			void operator()(Optional<bool>* val) const { *val = value.as_boolean(); }
			void operator()(std::string* val) const { *val = value.as_string(); }
			void operator()(Optional<std::string>* val) const { *val = value.as_string(); }
			void operator()(std::vector<int>* val) const {
				auto arr = value.as_array();
				for (const auto& i : arr) {
					val->emplace_back(i.as_integer());
				}
			}
			void operator()(Optional<std::vector<int>>* val) const {
				std::vector<int> res;
				(*this)(&res);
				*val = std::move(res);
			}
			void operator()(ConfigDBType* val) const {
				if (value.as_string() == "random") {
					*val = deterministicRandom()->coinflip() ? ConfigDBType::SIMPLE : ConfigDBType::PAXOS;
				} else {
					*val = configDBTypeFromString(value.as_string());
				}
			}
			void operator()(Optional<ConfigDBType>* val) const {
				ConfigDBType res;
				(*this)(&res);
				*val = std::move(res);
			}
			void operator()(std::vector<std::string>* val) const {
				auto arr = value.as_array();
				for (const auto& i : arr) {
					val->emplace_back(i.as_string());
				}
			}
			void operator()(Optional<std::vector<std::string>>* val) const {
				std::vector<std::string> res;
				(*this)(&res);
				*val = std::move(res);
			}
			void operator()(std::set<SimulationStorageEngine>* val) const {
				auto arr = value.as_array();
				for (const auto& i : arr) {
					const auto intVal = static_cast<uint8_t>(i.as_integer());
					ASSERT(isValidSimulationStorageEngineValue(intVal));
					val->insert(static_cast<SimulationStorageEngine>(intVal));
				}
			}
			void operator()(Optional<std::set<SimulationStorageEngine>>* val) const {
				std::set<SimulationStorageEngine> res;
				(*this)(&res);
				*val = std::move(res);
			}
			void operator()(SimulationStorageEngine* val) const {
				uint8_t intVal = static_cast<uint8_t>(value.as_integer());
				ASSERT(isValidSimulationStorageEngineValue(intVal));
				*val = static_cast<SimulationStorageEngine>(intVal);
			}
			void operator()(Optional<SimulationStorageEngine>* val) const {
				SimulationStorageEngine v;
				(*this)(&v);
				*val = v;
			}
		};

		struct trace_visitor {
			std::string key;
			TraceEvent& evt;
			trace_visitor(std::string const& key, TraceEvent& e) : key("Key" + key), evt(e) {}
			template <class T>
			void operator()(T const* val) const {
				evt.detail(key.c_str(), *val);
			}
			void operator()(std::vector<int> const* val) const {
				if (val->empty()) {
					evt.detail(key.c_str(), "[]");
					return;
				}
				std::stringstream value;
				value << "[" << val->at(0);
				for (int i = 1; i < val->size(); ++i) {
					value << "," << val->at(i);
				}
				value << "]";
				evt.detail(key.c_str(), value.str());
			}
			void operator()(Optional<std::vector<int>> const* val) const {
				if (!val->present()) {
					evt.detail(key.c_str(), *val);
				} else {
					(*this)(&(val->get()));
				}
			}
			void operator()(std::vector<std::string> const* val) const {
				if (val->empty()) {
					evt.detail(key.c_str(), "[]");
					return;
				}
				std::stringstream value;
				value << "[" << val->at(0);
				for (int i = 1; i < val->size(); ++i) {
					value << "," << val->at(i);
				}
				value << "]";
				evt.detail(key.c_str(), value.str());
			}
			void operator()(Optional<std::vector<std::string>> const* val) const {
				if (!val->present()) {
					evt.detail(key.c_str(), *val);
				} else {
					(*this)(&(val->get()));
				}
			}
			void operator()(ConfigDBType const* val) const { evt.detail(key.c_str(), *val); }
			void operator()(Optional<ConfigDBType> const* val) const {
				Optional<std::string> optStr;
				if (val->present()) {
					optStr = configDBTypeToString(val->get());
				}
				evt.detail(key.c_str(), optStr);
			}
		};

	public:
		~ConfigBuilder() {
			TraceEvent evt("SimulatorConfigFromToml");
			for (const auto& p : confMap) {
				std::visit(trace_visitor(std::string(p.first), evt), p.second);
			}
		}

		template <class V>
		ConfigBuilder& add(std::string_view key, V value) {
			confMap.emplace(key, value);
			return *this;
		}

		void set(std::string_view key, const value_type& value) {
			auto iter = confMap.find(key);
			if (iter == confMap.end()) {
				std::cerr << "Unknown configuration attribute " << key << std::endl;
				TraceEvent("UnknownConfigurationAttribute").detail("Name", std::string(key));
				throw unknown_error();
			}
			std::visit(visitor(value), iter->second);
		}
	};

	bool isIniFile(const char* fileName) {
		std::string name = fileName;
		auto pos = name.find_last_of('.');
		ASSERT(pos != std::string::npos && pos + 1 < name.size());
		auto extension = name.substr(pos + 1);
		return extension == "txt"sv;
	}

	void loadIniFile(const char* testFile) {
		std::ifstream ifs;
		ifs.open(testFile, std::ifstream::in);
		if (!ifs.good())
			return;

		std::string cline;

		while (ifs.good()) {
			getline(ifs, cline);
			std::string line = removeWhitespace(std::string(cline));
			if (!line.size() || line.find(';') == 0)
				continue;

			size_t found = line.find('=');
			if (found == std::string::npos)
				// hmmm, not good
				continue;
			std::string attrib = removeWhitespace(line.substr(0, found));
			std::string value = removeWhitespace(line.substr(found + 1));

			if (attrib == "extraDatabaseMode") {
				extraDatabaseMode = ISimulator::stringToExtraDatabaseMode(value);
			}

			if (attrib == "extraDatabaseCount") {
				sscanf(value.c_str(), "%d", &extraDatabaseCount);
			}

			if (attrib == "minimumReplication") {
				sscanf(value.c_str(), "%d", &minimumReplication);
			}

			if (attrib == "minimumRegions") {
				sscanf(value.c_str(), "%d", &minimumRegions);
			}

			if (attrib == "configureLocked") {
				int configureLockedInt;
				sscanf(value.c_str(), "%d", &configureLockedInt);
				configureLocked = (configureLockedInt != 0);
			}

			if (attrib == "startIncompatibleProcess") {
				startIncompatibleProcess = strcmp(value.c_str(), "true") == 0;
			}

			if (attrib == "logAntiQuorum") {
				sscanf(value.c_str(), "%d", &logAntiQuorum);
			}

			if (attrib == "storageEngineExcludeTypes") {
				std::stringstream ss(value);
				for (int i; ss >> i;) {
					ASSERT(isValidSimulationStorageEngineValue(i));
					storageEngineExcludeTypes.insert(static_cast<SimulationStorageEngine>(i));
					if (ss.peek() == ',') {
						ss.ignore();
					}
				}
			}
			if (attrib == "maxTLogVersion") {
				sscanf(value.c_str(), "%d", &maxTLogVersion);
			}
			if (attrib == "disableTss") {
				disableTss = strcmp(value.c_str(), "true") == 0;
			}
			if (attrib == "disableHostname") {
				disableHostname = strcmp(value.c_str(), "true") == 0;
			}
			if (attrib == "disableRemoteKVS") {
				disableRemoteKVS = strcmp(value.c_str(), "true") == 0;
			}
			if (attrib == "encryptModes") {
				std::stringstream ss(value);
				std::string token;
				while (std::getline(ss, token, ',')) {
					encryptModes.push_back(token);
				}
			}
			if (attrib == "restartInfoLocation") {
				isFirstTestInRestart = true;
			}
			if (attrib == "configDBType") {
				if (value == "random") {
					configDBType = deterministicRandom()->random01() < 0.1 ? ConfigDBType::SIMPLE : ConfigDBType::PAXOS;
				} else {
					configDBType = configDBTypeFromString(value);
				}
			}
			if (attrib == "randomlyRenameZoneId") {
				randomlyRenameZoneId = strcmp(value.c_str(), "true") == 0;
			}
			if (attrib == "simHTTPServerEnabled") {
				simHTTPServerEnabled = strcmp(value.c_str(), "true") == 0;
			}
			if (attrib == "injectSSTargetedRestart") {
				injectTargetedSSRestart = strcmp(value.c_str(), "true") == 0;
			}
			if (attrib == "longRunningTest") {
				longRunningTest = strcmp(value.c_str(), "true") == 0;
			}
			if (attrib == "simulationNormalRunTestsTimeoutSeconds") {
				sscanf(value.c_str(), "%d", &simulationNormalRunTestsTimeoutSeconds);
			}
			if (attrib == "simulationBuggifyRunTestsTimeoutSeconds") {
				sscanf(value.c_str(), "%d", &simulationBuggifyRunTestsTimeoutSeconds);
			}
		}

		ifs.close();
	}

	ConfigDBType configDBType{ ConfigDBType::DISABLED };

public:
	int extraDB = 0;
	ISimulator::ExtraDatabaseMode extraDatabaseMode = ISimulator::ExtraDatabaseMode::Disabled;
	// The number of extra database used if the database mode is MULTIPLE
	int extraDatabaseCount = 1;
	bool extraDatabaseBackupAgents = false;
	int minimumReplication = 0;
	int minimumRegions = 0;
	bool configureLocked = false;
	bool startIncompatibleProcess = false;
	bool isFirstTestInRestart = false;
	// 7.0 cannot be downgraded to 6.3 after enabling TSS, so disable TSS for 6.3 downgrade tests
	bool disableTss = false;
	// 7.1 cannot be downgraded to 7.0 and below after enabling hostname, so disable hostname for 7.0 downgrade tests
	bool disableHostname = false;
	// remote key value store is a child process spawned by the SS process to run the storage engine
	bool disableRemoteKVS = false;
	// TODO(gglass): see about removing `encryptModes` (it relates to legacy deleted tenant support).
	// If provided, set using EncryptionAtRestMode::fromString
	std::vector<std::string> encryptModes;
	// Storage Engine Types: Verify match with SimulationConfig::generateNormalConfig
	//	0 = "ssd"
	//	1 = "memory"
	//	2 = "memory-radixtree"
	//	3 = "ssd-redwood-1"
	//	4 = "ssd-rocksdb-v1"
	//	5 = "ssd-sharded-rocksdb"
	// Requires a comma-separated list of numbers WITHOUT whitespaces
	// See SimulationStorageEngine for more details
	std::set<SimulationStorageEngine> storageEngineExcludeTypes;
	Optional<int> datacenters, stderrSeverity, processesPerMachine;
	// Set the maximum TLog version that can be selected for a test
	// Refer to FDBTypes.h::TLogVersion. Defaults to the maximum supported version.
	int maxTLogVersion = TLogVersion::MAX_SUPPORTED;
	int extraMachineCountDC = 0;
	int extraStorageMachineCountPerDC = 0;

	Optional<bool> generateFearless, buggify, faultInjection;
	Optional<std::string> config;
	Optional<std::string> remoteConfig;
	bool randomlyRenameZoneId = false;
	bool simHTTPServerEnabled = true;

	bool injectTargetedSSRestart = false;
	bool injectSSDelay = false;
	std::string testClass; // unused -- used in TestHarness
	float testPriority; // unused -- used in TestHarness

	bool longRunningTest = false;
	int simulationNormalRunTestsTimeoutSeconds = 5400;
	int simulationBuggifyRunTestsTimeoutSeconds = 36000;

	// Number of tlogs in the remote region
	Optional<int> remoteDesiredTLogCount;
	// Number of process classes explictly set as Stateless in all DCs
	Optional<int> statelessProcessClassesPerDC;

	ConfigDBType getConfigDBType() const { return configDBType; }

	bool tomlKeyPresent(const toml::value& data, std::string key) {
		if (data.is_table()) {
			for (const auto& [k, v] : data.as_table()) {
				if (k == key || tomlKeyPresent(v, key)) {
					return true;
				}
			}
		} else if (data.is_array()) {
			for (const auto& v : data.as_array()) {
				if (tomlKeyPresent(v, key)) {
					return true;
				}
			}
		}
		return false;
	}

	void readFromConfig(const char* testFile) {
		if (isIniFile(testFile)) {
			loadIniFile(testFile);
			return;
		}
		std::string extraDatabaseModeStr;
		ConfigBuilder builder;
		builder.add("testClass", &testClass)
		    .add("testPriority", &testPriority)
		    .add("extraDatabaseMode", &extraDatabaseModeStr)
		    .add("extraDatabaseCount", &extraDatabaseCount)
		    .add("extraDatabaseBackupAgents", &extraDatabaseBackupAgents)
		    .add("minimumReplication", &minimumReplication)
		    .add("minimumRegions", &minimumRegions)
		    .add("configureLocked", &configureLocked)
		    .add("startIncompatibleProcess", &startIncompatibleProcess)
		    .add("logAntiQuorum", &logAntiQuorum)
		    .add("storageEngineExcludeTypes", &storageEngineExcludeTypes)
		    .add("maxTLogVersion", &maxTLogVersion)
		    .add("disableTss", &disableTss)
		    .add("disableHostname", &disableHostname)
		    .add("disableRemoteKVS", &disableRemoteKVS)
		    .add("encryptModes", &encryptModes)
		    .add("simpleConfig", &simpleConfig)
		    .add("generateFearless", &generateFearless)
		    .add("datacenters", &datacenters)
		    .add("desiredTLogCount", &desiredTLogCount)
		    .add("remoteDesiredTLogCount", &remoteDesiredTLogCount)
		    .add("commitProxyCount", &commitProxyCount)
		    .add("grvProxyCount", &grvProxyCount)
		    .add("resolverCount", &resolverCount)
		    .add("storageEngineType", &storageEngineType)
		    .add("config", &config)
		    .add("remoteConfig", &remoteConfig)
		    .add("buggify", &buggify)
		    .add("faultInjection", &faultInjection)
		    .add("StderrSeverity", &stderrSeverity)
		    .add("machineCount", &machineCount)
		    .add("asanMachineCount", &asanMachineCount)
		    .add("processesPerMachine", &processesPerMachine)
		    .add("coordinators", &coordinators)
		    .add("configDB", &configDBType)
		    .add("extraMachineCountDC", &extraMachineCountDC)
		    .add("extraStorageMachineCountPerDC", &extraStorageMachineCountPerDC)
		    .add("simHTTPServerEnabled", &simHTTPServerEnabled)
		    .add("randomlyRenameZoneId", &randomlyRenameZoneId)
		    .add("injectTargetedSSRestart", &injectTargetedSSRestart)
		    .add("injectSSDelay", &injectSSDelay)
		    .add("longRunningTest", &longRunningTest)
		    .add("simulationNormalRunTestsTimeoutSeconds", &simulationNormalRunTestsTimeoutSeconds)
		    .add("simulationBuggifyRunTestsTimeoutSeconds", &simulationBuggifyRunTestsTimeoutSeconds)
		    .add("statelessProcessClassesPerDC", &statelessProcessClassesPerDC);
		try {
			auto file = toml::parse(testFile);
			if (file.contains("configuration") && toml::find(file, "configuration").is_table()) {
				auto conf = toml::find(file, "configuration").as_table();
				for (const auto& [key, value] : conf) {
					if (key == "ClientInfoLogging") {
						setNetworkOption(FDBNetworkOptions::DISABLE_CLIENT_STATISTICS_LOGGING);
					} else if (key == "restartInfoLocation") {
						isFirstTestInRestart = true;
					} else {
						builder.set(key, value);
					}
				}
				if (stderrSeverity.present()) {
					TraceEvent("StderrSeverity").detail("NewSeverity", stderrSeverity.get());
				}
			}
			// look for restartInfoLocation to mark isFirstTestInRestart
			if (!isFirstTestInRestart) {
				isFirstTestInRestart = tomlKeyPresent(file, "restartInfoLocation");
			}
			if (!extraDatabaseModeStr.empty()) {
				extraDatabaseMode = ISimulator::stringToExtraDatabaseMode(extraDatabaseModeStr);
			}
		} catch (std::exception& e) {
			std::cerr << e.what() << std::endl;
			TraceEvent("TOMLParseError").detail("Error", printable(e.what()));
			throw unknown_error();
		}
		// Verify that we can use the passed config
		if (simpleConfig) {
			if (minimumRegions > 1) {
				TraceEvent("ElapsedTime").detail("SimTime", now()).detail("RealTime", 0).detail("RandomUnseed", 0);
				flushAndExit(0);
			}
		}
	}

	bool excludedStorageEngineType(SimulationStorageEngine storageEngineType) const {
		return storageEngineExcludeTypes.contains(storageEngineType);
	}

	TestConfig() = default;
	TestConfig(const BasicTestConfig& config) : BasicTestConfig(config) {}
};

template <class T>
T simulate(const T& in) {
	BinaryWriter writer(AssumeVersion(g_network->protocolVersion()));
	writer << in;
	BinaryReader reader(writer.getData(), writer.getLength(), AssumeVersion(g_network->protocolVersion()));
	T out;
	reader >> out;
	return out;
}

															#line 615 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
// This generated class is to be used only via runBackup()
															#line 613 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
template <class RunBackupActor>
															#line 613 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class RunBackupActorState {
															#line 621 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
															#line 613 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	RunBackupActorState(Reference<IClusterConnectionRecord> const& connRecord) 
															#line 613 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
															#line 613 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		 : connRecord(connRecord),
															#line 614 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   agentFuture(),
															#line 615 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   fileAgent()
															#line 632 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("runBackup", reinterpret_cast<unsigned long>(this));

	}
	~RunBackupActorState() 
	{
		fdb_probe_actor_destroy("runBackup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 617 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			;
															#line 647 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunBackupActorState();
		static_cast<RunBackupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 621 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (g_simulator->backupAgents == ISimulator::BackupAgentType::BackupToFile)
															#line 670 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 622 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			Database cx = Database::createDatabase(connRecord, ApiVersion::LATEST_VERSION);
															#line 624 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("SimBackupAgentsStarting").log();
															#line 625 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			agentFuture = fileAgent.run(cx, 1.0 / CLIENT_KNOBS->BACKUP_AGGREGATE_POLL_RATE, CLIENT_KNOBS->SIM_BACKUP_TASKS_PER_AGENT);
															#line 628 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			;
															#line 680 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 617 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (!(g_simulator->backupAgents == ISimulator::BackupAgentType::WaitForType))
															#line 701 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 618 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(1.0);
															#line 618 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<RunBackupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 709 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RunBackupActor*>(this)->actor_wait_state = 1;
															#line 618 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RunBackupActor, 0, Void >*>(static_cast<RunBackupActor*>(this)));
															#line 714 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunBackupActor*>(this)->actor_wait_state > 0) static_cast<RunBackupActor*>(this)->actor_wait_state = 0;
		static_cast<RunBackupActor*>(this)->ActorCallback< RunBackupActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunBackupActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunBackupActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RunBackupActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 636 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_2 = Future<Void>(Never());
															#line 636 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<RunBackupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 825 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<RunBackupActor*>(this)->actor_wait_state = 3;
															#line 636 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RunBackupActor, 2, Void >*>(static_cast<RunBackupActor*>(this)));
															#line 830 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 632 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimBackupAgentsStopping").log();
															#line 633 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		agentFuture.cancel();
															#line 841 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 628 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (!(g_simulator->backupAgents == ISimulator::BackupAgentType::BackupToFile))
															#line 857 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 629 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(1.0);
															#line 629 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<RunBackupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 865 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RunBackupActor*>(this)->actor_wait_state = 2;
															#line 629 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RunBackupActor, 1, Void >*>(static_cast<RunBackupActor*>(this)));
															#line 870 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RunBackupActor*>(this)->actor_wait_state > 0) static_cast<RunBackupActor*>(this)->actor_wait_state = 0;
		static_cast<RunBackupActor*>(this)->ActorCallback< RunBackupActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunBackupActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RunBackupActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RunBackupActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 637 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		return a_body1Catch1(internal_error(), loopDepth);
															#line 979 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 637 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		return a_body1Catch1(internal_error(), loopDepth);
															#line 987 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RunBackupActor*>(this)->actor_wait_state > 0) static_cast<RunBackupActor*>(this)->actor_wait_state = 0;
		static_cast<RunBackupActor*>(this)->ActorCallback< RunBackupActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunBackupActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runBackup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RunBackupActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runBackup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RunBackupActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runBackup", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 613 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Reference<IClusterConnectionRecord> connRecord;
															#line 614 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Future<Void> agentFuture;
															#line 615 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	FileBackupAgent fileAgent;
															#line 1072 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
};
// This generated class is to be used only via runBackup()
															#line 613 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class RunBackupActor final : public Actor<Void>, public ActorCallback< RunBackupActor, 0, Void >, public ActorCallback< RunBackupActor, 1, Void >, public ActorCallback< RunBackupActor, 2, Void >, public FastAllocated<RunBackupActor>, public RunBackupActorState<RunBackupActor> {
															#line 1077 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
	using FastAllocated<RunBackupActor>::operator new;
	using FastAllocated<RunBackupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14207943757877479168UL, 10026410792937765120UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RunBackupActor, 0, Void >;
friend struct ActorCallback< RunBackupActor, 1, Void >;
friend struct ActorCallback< RunBackupActor, 2, Void >;
															#line 613 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	RunBackupActor(Reference<IClusterConnectionRecord> const& connRecord) 
															#line 1096 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RunBackupActorState<RunBackupActor>(connRecord),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("runBackup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8128639066585784576UL, 222009843484957696UL);
		ActorExecutionContextHelper __helper(static_cast<RunBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("runBackup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("runBackup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RunBackupActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RunBackupActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RunBackupActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 613 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
[[nodiscard]] Future<Void> runBackup( Reference<IClusterConnectionRecord> const& connRecord ) {
															#line 613 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	return Future<Void>(new RunBackupActor(connRecord));
															#line 1130 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
}

#line 639 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"

															#line 1135 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
// This generated class is to be used only via runDr()
															#line 640 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
template <class RunDrActor>
															#line 640 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class RunDrActorState {
															#line 1141 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
															#line 640 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	RunDrActorState(Reference<IClusterConnectionRecord> const& connRecord) 
															#line 640 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
															#line 640 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		 : connRecord(connRecord),
															#line 641 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   agentFutures()
															#line 1150 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("runDr", reinterpret_cast<unsigned long>(this));

	}
	~RunDrActorState() 
	{
		fdb_probe_actor_destroy("runDr", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 643 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			;
															#line 1165 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunDrActorState();
		static_cast<RunDrActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 647 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (g_simulator->drAgents == ISimulator::BackupAgentType::BackupToDB)
															#line 1188 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 648 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			ASSERT(g_simulator->extraDatabases.size() == 1);
															#line 649 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			Database cx = Database::createDatabase(connRecord, ApiVersion::LATEST_VERSION);
															#line 651 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			drDatabase = Database::createSimulatedExtraDatabase(g_simulator->extraDatabases[0]);
															#line 653 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("StartingDrAgents") .detail("ConnectionString", connRecord->getConnectionString().toString()) .detail("ExtraString", g_simulator->extraDatabases[0]);
															#line 657 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			dbAgent = DatabaseBackupAgent(cx);
															#line 658 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			extraAgent = DatabaseBackupAgent(drDatabase);
															#line 660 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			auto drPollDelay = 1.0 / CLIENT_KNOBS->BACKUP_AGGREGATE_POLL_RATE;
															#line 662 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			agentFutures.push_back(extraAgent.run(cx, drPollDelay, CLIENT_KNOBS->SIM_BACKUP_TASKS_PER_AGENT));
															#line 663 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			agentFutures.push_back(dbAgent.run(drDatabase, drPollDelay, CLIENT_KNOBS->SIM_BACKUP_TASKS_PER_AGENT));
															#line 665 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			;
															#line 1210 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 643 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (!(g_simulator->drAgents == ISimulator::BackupAgentType::WaitForType))
															#line 1231 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 644 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(1.0);
															#line 644 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<RunDrActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1239 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RunDrActor*>(this)->actor_wait_state = 1;
															#line 644 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RunDrActor, 0, Void >*>(static_cast<RunDrActor*>(this)));
															#line 1244 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunDrActor*>(this)->actor_wait_state > 0) static_cast<RunDrActor*>(this)->actor_wait_state = 0;
		static_cast<RunDrActor*>(this)->ActorCallback< RunDrActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunDrActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runDr", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunDrActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runDr", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunDrActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runDr", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunDrActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runDr", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RunDrActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runDr", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunDrActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runDr", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 676 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_2 = Future<Void>(Never());
															#line 676 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<RunDrActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1355 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<RunDrActor*>(this)->actor_wait_state = 3;
															#line 676 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RunDrActor, 2, Void >*>(static_cast<RunDrActor*>(this)));
															#line 1360 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
															#line 669 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("StoppingDrAgents").log();
															#line 671 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		for( auto it : agentFutures ) {
															#line 672 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			it.cancel();
															#line 1373 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 665 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (!(g_simulator->drAgents == ISimulator::BackupAgentType::BackupToDB))
															#line 1390 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 666 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(1.0);
															#line 666 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<RunDrActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1398 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RunDrActor*>(this)->actor_wait_state = 2;
															#line 666 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RunDrActor, 1, Void >*>(static_cast<RunDrActor*>(this)));
															#line 1403 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RunDrActor*>(this)->actor_wait_state > 0) static_cast<RunDrActor*>(this)->actor_wait_state = 0;
		static_cast<RunDrActor*>(this)->ActorCallback< RunDrActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunDrActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runDr", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunDrActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runDr", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RunDrActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runDr", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunDrActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runDr", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RunDrActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runDr", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunDrActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runDr", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 677 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		return a_body1Catch1(internal_error(), loopDepth);
															#line 1512 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 677 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		return a_body1Catch1(internal_error(), loopDepth);
															#line 1520 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RunDrActor*>(this)->actor_wait_state > 0) static_cast<RunDrActor*>(this)->actor_wait_state = 0;
		static_cast<RunDrActor*>(this)->ActorCallback< RunDrActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunDrActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runDr", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunDrActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runDr", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RunDrActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runDr", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunDrActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runDr", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RunDrActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runDr", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunDrActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runDr", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 640 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Reference<IClusterConnectionRecord> connRecord;
															#line 641 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::vector<Future<Void>> agentFutures;
															#line 651 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Database drDatabase;
															#line 657 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	DatabaseBackupAgent dbAgent;
															#line 658 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	DatabaseBackupAgent extraAgent;
															#line 1609 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
};
// This generated class is to be used only via runDr()
															#line 640 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class RunDrActor final : public Actor<Void>, public ActorCallback< RunDrActor, 0, Void >, public ActorCallback< RunDrActor, 1, Void >, public ActorCallback< RunDrActor, 2, Void >, public FastAllocated<RunDrActor>, public RunDrActorState<RunDrActor> {
															#line 1614 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
	using FastAllocated<RunDrActor>::operator new;
	using FastAllocated<RunDrActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5499127927548142336UL, 9027688815475777536UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RunDrActor, 0, Void >;
friend struct ActorCallback< RunDrActor, 1, Void >;
friend struct ActorCallback< RunDrActor, 2, Void >;
															#line 640 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	RunDrActor(Reference<IClusterConnectionRecord> const& connRecord) 
															#line 1633 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RunDrActorState<RunDrActor>(connRecord),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("runDr", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8938195149834069504UL, 4332675399481671168UL);
		ActorExecutionContextHelper __helper(static_cast<RunDrActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("runDr");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("runDr", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RunDrActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RunDrActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RunDrActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 640 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
[[nodiscard]] Future<Void> runDr( Reference<IClusterConnectionRecord> const& connRecord ) {
															#line 640 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	return Future<Void>(new RunDrActor(connRecord));
															#line 1667 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
}

#line 679 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"

															#line 1672 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
// This generated class is to be used only via runSimHTTPServer()
															#line 680 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
template <class RunSimHTTPServerActor>
															#line 680 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class RunSimHTTPServerActorState {
															#line 1678 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
															#line 680 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	RunSimHTTPServerActorState() 
															#line 680 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	{
															#line 1684 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		fdb_probe_actor_create("runSimHTTPServer", reinterpret_cast<unsigned long>(this));

	}
	~RunSimHTTPServerActorState() 
	{
		fdb_probe_actor_destroy("runSimHTTPServer", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 681 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("SimHTTPServerStarting");
															#line 682 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			context = makeReference<HTTP::SimServerContext>();
															#line 683 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			g_simulator->addSimHTTPProcess(context);
															#line 1702 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			try {
															#line 686 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				StrictFuture<Void> __when_expr_0 = context->actors.getResult();
															#line 686 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (static_cast<RunSimHTTPServerActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1708 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<RunSimHTTPServerActor*>(this)->actor_wait_state = 1;
															#line 686 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RunSimHTTPServerActor, 0, Void >*>(static_cast<RunSimHTTPServerActor*>(this)));
															#line 1713 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunSimHTTPServerActorState();
		static_cast<RunSimHTTPServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 693 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		return a_body1Catch1(internal_error(), loopDepth);
															#line 1742 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("SimHTTPServerDied").errorUnsuppressed(e);
															#line 689 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			context->stop();
															#line 690 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			g_simulator->removeSimHTTPProcess();
															#line 691 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 1757 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunSimHTTPServerActor*>(this)->actor_wait_state > 0) static_cast<RunSimHTTPServerActor*>(this)->actor_wait_state = 0;
		static_cast<RunSimHTTPServerActor*>(this)->ActorCallback< RunSimHTTPServerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RunSimHTTPServerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("runSimHTTPServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunSimHTTPServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runSimHTTPServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunSimHTTPServerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("runSimHTTPServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunSimHTTPServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runSimHTTPServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RunSimHTTPServerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("runSimHTTPServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunSimHTTPServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runSimHTTPServer", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 682 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Reference<HTTP::SimServerContext> context;
															#line 1869 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
};
// This generated class is to be used only via runSimHTTPServer()
															#line 680 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class RunSimHTTPServerActor final : public Actor<Void>, public ActorCallback< RunSimHTTPServerActor, 0, Void >, public FastAllocated<RunSimHTTPServerActor>, public RunSimHTTPServerActorState<RunSimHTTPServerActor> {
															#line 1874 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
	using FastAllocated<RunSimHTTPServerActor>::operator new;
	using FastAllocated<RunSimHTTPServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4910172442551733248UL, 15337717994535959296UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RunSimHTTPServerActor, 0, Void >;
															#line 680 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	RunSimHTTPServerActor() 
															#line 1891 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RunSimHTTPServerActorState<RunSimHTTPServerActor>(),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("runSimHTTPServer", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2544057753507577600UL, 13835641485291669248UL);
		ActorExecutionContextHelper __helper(static_cast<RunSimHTTPServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("runSimHTTPServer");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("runSimHTTPServer", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RunSimHTTPServerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 680 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
[[nodiscard]] Future<Void> runSimHTTPServer(  ) {
															#line 680 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	return Future<Void>(new RunSimHTTPServerActor());
															#line 1923 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
}

#line 695 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"

// enum AgentMode { AgentNone = 0, AgentOnly = 1, AgentAddition = 2 };
// FIXME: could do this as bit flags of (fdbd) (backup agent) (http) etc... if the space gets more complicated
enum ProcessMode { FDBDOnly = 0, BackupAgentOnly = 1, FDBDAndBackupAgent = 2, SimHTTPServer = 3 };

bool processRunBackupAgent(ProcessMode mode) {
	return mode == BackupAgentOnly || mode == FDBDAndBackupAgent;
}

bool processRunFDBD(ProcessMode mode) {
	return mode == FDBDOnly || mode == FDBDAndBackupAgent;
}

bool processRunHTTPServer(ProcessMode mode) {
	return mode == SimHTTPServer;
}

// SOMEDAY: when a process can be rebooted in isolation from the other on that machine,
//  a loop{} will be needed around the waiting on simulatedFDBD(). For now this simply
//  takes care of house-keeping such as context switching and file closing.
															#line 1947 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
// This generated class is to be used only via simulatedFDBDRebooter()
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
template <class SimulatedFDBDRebooterActor>
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class SimulatedFDBDRebooterActorState {
															#line 1953 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	SimulatedFDBDRebooterActorState(Reference<IClusterConnectionRecord> const& connRecord,IPAddress const& ip,bool const& sslEnabled,uint16_t const& port,uint16_t const& listenPerProcess,LocalityData const& localities,ProcessClass const& processClass,std::string* const& dataFolder,std::string* const& coordFolder,std::string const& baseFolder,ClusterConnectionString const& connStr,ClusterConnectionString const& otherConnStr,bool const& useSeedFile,ProcessMode const& processMode,std::string const& whitelistBinPaths,ProtocolVersion const& protocolVersion,ConfigDBType const& configDBType,bool const& isDr) 
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		 : connRecord(connRecord),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   ip(ip),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   sslEnabled(sslEnabled),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   port(port),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   listenPerProcess(listenPerProcess),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   localities(localities),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   processClass(processClass),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   dataFolder(dataFolder),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   coordFolder(coordFolder),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   baseFolder(baseFolder),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   connStr(connStr),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   otherConnStr(otherConnStr),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   useSeedFile(useSeedFile),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   processMode(processMode),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   whitelistBinPaths(whitelistBinPaths),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   protocolVersion(protocolVersion),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   configDBType(configDBType),
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   isDr(isDr),
															#line 733 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   simProcess(g_simulator->getCurrentProcess()),
															#line 734 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   randomId(nondeterministicRandom()->randomUniqueID()),
															#line 735 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   cycles(0),
															#line 736 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   allowList()
															#line 2002 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this));

	}
	~SimulatedFDBDRebooterActorState() 
	{
		fdb_probe_actor_destroy("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 738 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			allowList.addTrustedSubnet("0.0.0.0/2"sv);
															#line 739 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			allowList.addTrustedSubnet("abcd::/16"sv);
															#line 741 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			;
															#line 2021 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SimulatedFDBDRebooterActorState();
		static_cast<SimulatedFDBDRebooterActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 742 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		auto waitTime = SERVER_KNOBS->MIN_REBOOT_TIME + (SERVER_KNOBS->MAX_REBOOT_TIME - SERVER_KNOBS->MIN_REBOOT_TIME) * deterministicRandom()->random01();
															#line 745 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		cycles++;
															#line 746 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedFDBDPreWait") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", NetworkAddress(ip, port, true, false)) .detail("ZoneId", localities.zoneId()) .detail("WaitTime", waitTime) .detail("Port", port) .detail("IsDr", isDr);
															#line 755 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(waitTime);
															#line 755 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2059 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 1;
															#line 755 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SimulatedFDBDRebooterActor, 0, Void >*>(static_cast<SimulatedFDBDRebooterActor*>(this)));
															#line 2064 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 757 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		process = g_simulator->newProcess("Server", ip, port, sslEnabled, listenPerProcess, localities, processClass, dataFolder->c_str(), coordFolder->c_str(), protocolVersion, isDr);
															#line 768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_1 = g_simulator->onProcess( process, TaskPriority::DefaultYield);
															#line 768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2077 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 2;
															#line 768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SimulatedFDBDRebooterActor, 1, Void >*>(static_cast<SimulatedFDBDRebooterActor*>(this)));
															#line 2082 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 757 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		process = g_simulator->newProcess("Server", ip, port, sslEnabled, listenPerProcess, localities, processClass, dataFolder->c_str(), coordFolder->c_str(), protocolVersion, isDr);
															#line 768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_1 = g_simulator->onProcess( process, TaskPriority::DefaultYield);
															#line 768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2095 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 2;
															#line 768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SimulatedFDBDRebooterActor, 1, Void >*>(static_cast<SimulatedFDBDRebooterActor*>(this)));
															#line 2100 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state > 0) static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 0;
		static_cast<SimulatedFDBDRebooterActor*>(this)->ActorCallback< SimulatedFDBDRebooterActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulatedFDBDRebooterActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SimulatedFDBDRebooterActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SimulatedFDBDRebooterActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 771 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		onShutdown = process->onShutdown();
															#line 2184 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		try {
															#line 774 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("SimulatedRebooterStarting") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("ZoneId", localities.zoneId()) .detail("DataHall", localities.dataHallId()) .detail("Address", process->address.toString()) .detail("Excluded", process->excluded) .detail("UsingSSL", sslEnabled) .detail("ProcessMode", processMode);
															#line 783 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("ProgramStart") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("SourceVersion", getSourceVersion()) .detail("Version", FDB_VT_VERSION) .detail("PackageName", FDB_VT_PACKAGE_NAME) .detail("DataFolder", *dataFolder) .detail("ConnectionString", connRecord ? connRecord->getConnectionString().toString() : "") .detailf("ActualTime", "%lld", DEBUG_DETERMINISM ? 0 : time(nullptr)) .detail("CommandLine", "fdbserver -r simulation") .detail("BuggifyEnabled", isGeneralBuggifyEnabled()) .detail("Simulated", true) .trackLatest("ProgramStart");
															#line 2190 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			try {
															#line 800 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				bool client = processClass == ProcessClass::TesterClass || processMode == BackupAgentOnly || processMode == SimHTTPServer;
															#line 802 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				FlowTransport::createInstance(client, 1, WLTOKEN_RESERVED_COUNT, &allowList);
															#line 803 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				for( const auto& p : g_simulator->authKeys ) {
															#line 804 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					FlowTransport::transport().addPublicKey(p.first, p.second.toPublic());
															#line 2200 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 806 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				Sim2FileSystem::newFileSystem();
															#line 808 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				std::vector<Future<Void>> futures;
															#line 809 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				for(int listenPort = port;listenPort < port + listenPerProcess;++listenPort) {
															#line 810 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					NetworkAddress n(ip, listenPort, true, sslEnabled && listenPort == port);
															#line 811 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					futures.push_back(FlowTransport::transport().bind(n, n));
															#line 2212 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 813 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (processRunFDBD(processMode))
															#line 2216 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 814 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					futures.push_back(fdbd(connRecord, localities, processClass, *dataFolder, *coordFolder, 500e6, "", "", -1, whitelistBinPaths, "", {}, configDBType, false));
															#line 2220 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 829 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (processRunBackupAgent(processMode))
															#line 2224 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 830 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					futures.push_back(runBackup(connRecord));
															#line 831 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					futures.push_back(runDr(connRecord));
															#line 2230 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 833 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (processRunHTTPServer(processMode))
															#line 2234 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 834 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					fmt::print("Process {0} run http server\n", ip.toString());
															#line 835 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					futures.push_back(runSimHTTPServer());
															#line 2240 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 838 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				futures.push_back(success(onShutdown));
															#line 839 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (!g_simulator->globalHasSwitchedCluster() && g_simulator->hasSwitchedCluster(process->address))
															#line 2246 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 845 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					TraceEvent("RebootProcessAndSwitchLateReboot").detail("Address", process->address);
															#line 846 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					g_simulator->switchCluster(process->address);
															#line 847 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					process->shutdownSignal.send(ISimulator::KillType::RebootProcessAndSwitch);
															#line 851 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					process->rebooting = true;
															#line 2256 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 853 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				StrictFuture<Void> __when_expr_2 = waitForAny(futures);
															#line 853 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2Catch2(actor_cancelled(), loopDepth);
															#line 2262 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont2Catch2(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
				static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 3;
															#line 853 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SimulatedFDBDRebooterActor, 2, Void >*>(static_cast<SimulatedFDBDRebooterActor*>(this)));
															#line 2267 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1loopBody1cont2Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1loopBody1cont2Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 771 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		onShutdown = process->onShutdown();
															#line 2288 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		try {
															#line 774 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("SimulatedRebooterStarting") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("ZoneId", localities.zoneId()) .detail("DataHall", localities.dataHallId()) .detail("Address", process->address.toString()) .detail("Excluded", process->excluded) .detail("UsingSSL", sslEnabled) .detail("ProcessMode", processMode);
															#line 783 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("ProgramStart") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("SourceVersion", getSourceVersion()) .detail("Version", FDB_VT_VERSION) .detail("PackageName", FDB_VT_PACKAGE_NAME) .detail("DataFolder", *dataFolder) .detail("ConnectionString", connRecord ? connRecord->getConnectionString().toString() : "") .detailf("ActualTime", "%lld", DEBUG_DETERMINISM ? 0 : time(nullptr)) .detail("CommandLine", "fdbserver -r simulation") .detail("BuggifyEnabled", isGeneralBuggifyEnabled()) .detail("Simulated", true) .trackLatest("ProgramStart");
															#line 2294 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			try {
															#line 800 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				bool client = processClass == ProcessClass::TesterClass || processMode == BackupAgentOnly || processMode == SimHTTPServer;
															#line 802 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				FlowTransport::createInstance(client, 1, WLTOKEN_RESERVED_COUNT, &allowList);
															#line 803 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				for( const auto& p : g_simulator->authKeys ) {
															#line 804 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					FlowTransport::transport().addPublicKey(p.first, p.second.toPublic());
															#line 2304 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 806 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				Sim2FileSystem::newFileSystem();
															#line 808 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				std::vector<Future<Void>> futures;
															#line 809 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				for(int listenPort = port;listenPort < port + listenPerProcess;++listenPort) {
															#line 810 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					NetworkAddress n(ip, listenPort, true, sslEnabled && listenPort == port);
															#line 811 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					futures.push_back(FlowTransport::transport().bind(n, n));
															#line 2316 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 813 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (processRunFDBD(processMode))
															#line 2320 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 814 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					futures.push_back(fdbd(connRecord, localities, processClass, *dataFolder, *coordFolder, 500e6, "", "", -1, whitelistBinPaths, "", {}, configDBType, false));
															#line 2324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 829 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (processRunBackupAgent(processMode))
															#line 2328 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 830 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					futures.push_back(runBackup(connRecord));
															#line 831 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					futures.push_back(runDr(connRecord));
															#line 2334 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 833 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (processRunHTTPServer(processMode))
															#line 2338 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 834 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					fmt::print("Process {0} run http server\n", ip.toString());
															#line 835 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					futures.push_back(runSimHTTPServer());
															#line 2344 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 838 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				futures.push_back(success(onShutdown));
															#line 839 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (!g_simulator->globalHasSwitchedCluster() && g_simulator->hasSwitchedCluster(process->address))
															#line 2350 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 845 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					TraceEvent("RebootProcessAndSwitchLateReboot").detail("Address", process->address);
															#line 846 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					g_simulator->switchCluster(process->address);
															#line 847 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					process->shutdownSignal.send(ISimulator::KillType::RebootProcessAndSwitch);
															#line 851 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					process->rebooting = true;
															#line 2360 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 853 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				StrictFuture<Void> __when_expr_2 = waitForAny(futures);
															#line 853 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2Catch2(actor_cancelled(), loopDepth);
															#line 2366 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont2Catch2(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
				static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 3;
															#line 853 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SimulatedFDBDRebooterActor, 2, Void >*>(static_cast<SimulatedFDBDRebooterActor*>(this)));
															#line 2371 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1loopBody1cont2Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1loopBody1cont2Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state > 0) static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 0;
		static_cast<SimulatedFDBDRebooterActor*>(this)->ActorCallback< SimulatedFDBDRebooterActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulatedFDBDRebooterActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SimulatedFDBDRebooterActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SimulatedFDBDRebooterActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 902 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		ASSERT(destructed || g_simulator->getCurrentProcess() == process);
															#line 904 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (!process->shutdownSignal.isSet() && !destructed)
															#line 2469 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 905 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			process->rebooting = true;
															#line 906 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			process->shutdownSignal.send(ISimulator::KillType::None);
															#line 2475 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
															#line 908 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedFDBDWait") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("Excluded", process->excluded) .detail("Rebooting", process->rebooting) .detail("ZoneId", localities.zoneId());
															#line 915 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_3 = g_simulator->onProcess(simProcess);
															#line 915 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2483 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 4;
															#line 915 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< SimulatedFDBDRebooterActor, 3, Void >*>(static_cast<SimulatedFDBDRebooterActor*>(this)));
															#line 2488 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 895 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent(destructed ? SevInfo : SevError, "SimulatedFDBDRebooterError") .errorUnsuppressed(e) .detail("ZoneId", localities.zoneId()) .detail("RandomId", randomId);
															#line 899 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			onShutdown = e;
															#line 2500 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 884 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedFDBDDone") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("Excluded", process->excluded) .detail("ZoneId", localities.zoneId()) .detail("KillType", onShutdown.isReady() ? onShutdown.get() : ISimulator::KillType::None);
															#line 892 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (!onShutdown.isReady())
															#line 2517 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 893 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			onShutdown = ISimulator::KillType::InjectFaults;
															#line 2521 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1loopBody1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 857 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (g_network->isSimulated() && e.code() != error_code_io_timeout && (bool)g_network->global(INetwork::enASIOTimedOut))
															#line 2532 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 859 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				TraceEvent(SevError, "IOTimeoutErrorSuppressed") .detail("ErrorCode", e.code()) .detail("RandomId", randomId) .backtrace();
															#line 2536 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 864 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (e.code() == error_code_io_timeout && !onShutdown.isReady())
															#line 2540 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 865 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				onShutdown = ISimulator::KillType::RebootProcess;
															#line 2544 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 868 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (onShutdown.isReady() && onShutdown.isError())
															#line 2548 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 869 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				return a_body1loopBody1cont2Catch1(onShutdown.getError(), loopDepth);
															#line 2552 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 870 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (e.code() != error_code_actor_cancelled)
															#line 2556 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 871 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				printf("SimulatedFDBDTerminated: %s\n", e.what());
															#line 2560 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 872 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			ASSERT(destructed || g_simulator->getCurrentProcess() == process);
															#line 874 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent(e.code() == error_code_actor_cancelled || e.code() == error_code_file_not_found || e.code() == error_code_incompatible_software_version || e.code() == error_code_broken_promise || destructed ? SevInfo : SevError, "SimulatedFDBDTerminated") .errorUnsuppressed(e) .detail("ZoneId", localities.zoneId());
															#line 2566 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state > 0) static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 0;
		static_cast<SimulatedFDBDRebooterActor*>(this)->ActorCallback< SimulatedFDBDRebooterActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulatedFDBDRebooterActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch2(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< SimulatedFDBDRebooterActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch2(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< SimulatedFDBDRebooterActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch2(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont15(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont16(Void const& _,int loopDepth) 
	{
															#line 917 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_4 = delay(0.00001 + FLOW_KNOBS->MAX_BUGGIFIED_DELAY);
															#line 917 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2696 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont16when1(__when_expr_4.get(), loopDepth); };
		static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 5;
															#line 917 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< SimulatedFDBDRebooterActor, 4, Void >*>(static_cast<SimulatedFDBDRebooterActor*>(this)));
															#line 2701 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont16(Void && _,int loopDepth) 
	{
															#line 917 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_4 = delay(0.00001 + FLOW_KNOBS->MAX_BUGGIFIED_DELAY);
															#line 917 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2712 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont16when1(__when_expr_4.get(), loopDepth); };
		static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 5;
															#line 917 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< SimulatedFDBDRebooterActor, 4, Void >*>(static_cast<SimulatedFDBDRebooterActor*>(this)));
															#line 2717 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont16(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont16(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state > 0) static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 0;
		static_cast<SimulatedFDBDRebooterActor*>(this)->ActorCallback< SimulatedFDBDRebooterActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulatedFDBDRebooterActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< SimulatedFDBDRebooterActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< SimulatedFDBDRebooterActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont16cont1(Void const& _,int loopDepth) 
	{
															#line 919 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		g_simulator->destroyProcess( process);
															#line 922 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		auto shutdownResult = onShutdown.get();
															#line 923 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedFDBDShutdown") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("Excluded", process->excluded) .detail("ZoneId", localities.zoneId()) .detail("KillType", shutdownResult);
															#line 931 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (shutdownResult < ISimulator::KillType::RebootProcessAndDelete)
															#line 2807 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 932 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("SimulatedFDBDLowerReboot") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("Excluded", process->excluded) .detail("ZoneId", localities.zoneId()) .detail("KillType", shutdownResult);
															#line 939 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (!static_cast<SimulatedFDBDRebooterActor*>(this)->SAV<ISimulator::KillType>::futures) { (void)(onShutdown.get()); this->~SimulatedFDBDRebooterActorState(); static_cast<SimulatedFDBDRebooterActor*>(this)->destroy(); return 0; }
															#line 2813 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			new (&static_cast<SimulatedFDBDRebooterActor*>(this)->SAV< ISimulator::KillType >::value()) ISimulator::KillType(onShutdown.get());
			this->~SimulatedFDBDRebooterActorState();
			static_cast<SimulatedFDBDRebooterActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 942 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (onShutdown.get() == ISimulator::KillType::RebootProcessAndDelete)
															#line 2821 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 943 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("SimulatedFDBDRebootAndDelete") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("ZoneId", localities.zoneId()) .detail("KillType", shutdownResult);
															#line 949 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			*coordFolder = joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString());
															#line 950 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			*dataFolder = joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString());
															#line 951 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			platform::createDirectory(*dataFolder);
															#line 953 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (!useSeedFile)
															#line 2833 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 954 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				writeFile(joinPath(*dataFolder, "fdb.cluster"), connStr.toString());
															#line 955 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				connRecord = makeReference<ClusterConnectionFile>(joinPath(*dataFolder, "fdb.cluster"));
															#line 2839 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
			else
			{
															#line 957 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				connRecord = makeReference<ClusterConnectionFile>(joinPath(*dataFolder, "fdb.cluster"), connStr.toString());
															#line 2845 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
		}
		else
		{
															#line 960 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (onShutdown.get() == ISimulator::KillType::RebootProcessAndSwitch)
															#line 2852 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 961 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				TraceEvent("SimulatedFDBDRebootAndSwitch") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("ZoneId", localities.zoneId()) .detail("KillType", shutdownResult) .detail("ConnectionString", connStr.toString()) .detail("OtherConnectionString", otherConnStr.toString()) .detail("SwitchingTo", g_simulator->hasSwitchedCluster(process->address)) .detail("MachineId", process->machine->machineId);
															#line 973 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (otherConnStr.toString().size() > 1)
															#line 2858 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 974 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					std::string newConnStr = g_simulator->hasSwitchedCluster(process->address) ? otherConnStr.toString() : connStr.toString();
															#line 976 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					connRecord = makeReference<ClusterConnectionFile>(joinPath(*dataFolder, "fdb.cluster"), newConnStr);
															#line 2864 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
			}
			else
			{
															#line 979 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				TraceEvent("SimulatedFDBDJustRepeat") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("ZoneId", localities.zoneId()) .detail("KillType", shutdownResult);
															#line 2871 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont16cont1(Void && _,int loopDepth) 
	{
															#line 919 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		g_simulator->destroyProcess( process);
															#line 922 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		auto shutdownResult = onShutdown.get();
															#line 923 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedFDBDShutdown") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("Excluded", process->excluded) .detail("ZoneId", localities.zoneId()) .detail("KillType", shutdownResult);
															#line 931 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (shutdownResult < ISimulator::KillType::RebootProcessAndDelete)
															#line 2888 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 932 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("SimulatedFDBDLowerReboot") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("Excluded", process->excluded) .detail("ZoneId", localities.zoneId()) .detail("KillType", shutdownResult);
															#line 939 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (!static_cast<SimulatedFDBDRebooterActor*>(this)->SAV<ISimulator::KillType>::futures) { (void)(onShutdown.get()); this->~SimulatedFDBDRebooterActorState(); static_cast<SimulatedFDBDRebooterActor*>(this)->destroy(); return 0; }
															#line 2894 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			new (&static_cast<SimulatedFDBDRebooterActor*>(this)->SAV< ISimulator::KillType >::value()) ISimulator::KillType(onShutdown.get());
			this->~SimulatedFDBDRebooterActorState();
			static_cast<SimulatedFDBDRebooterActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 942 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (onShutdown.get() == ISimulator::KillType::RebootProcessAndDelete)
															#line 2902 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 943 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("SimulatedFDBDRebootAndDelete") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("ZoneId", localities.zoneId()) .detail("KillType", shutdownResult);
															#line 949 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			*coordFolder = joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString());
															#line 950 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			*dataFolder = joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString());
															#line 951 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			platform::createDirectory(*dataFolder);
															#line 953 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (!useSeedFile)
															#line 2914 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 954 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				writeFile(joinPath(*dataFolder, "fdb.cluster"), connStr.toString());
															#line 955 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				connRecord = makeReference<ClusterConnectionFile>(joinPath(*dataFolder, "fdb.cluster"));
															#line 2920 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
			else
			{
															#line 957 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				connRecord = makeReference<ClusterConnectionFile>(joinPath(*dataFolder, "fdb.cluster"), connStr.toString());
															#line 2926 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
		}
		else
		{
															#line 960 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (onShutdown.get() == ISimulator::KillType::RebootProcessAndSwitch)
															#line 2933 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 961 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				TraceEvent("SimulatedFDBDRebootAndSwitch") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("ZoneId", localities.zoneId()) .detail("KillType", shutdownResult) .detail("ConnectionString", connStr.toString()) .detail("OtherConnectionString", otherConnStr.toString()) .detail("SwitchingTo", g_simulator->hasSwitchedCluster(process->address)) .detail("MachineId", process->machine->machineId);
															#line 973 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (otherConnStr.toString().size() > 1)
															#line 2939 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 974 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					std::string newConnStr = g_simulator->hasSwitchedCluster(process->address) ? otherConnStr.toString() : connStr.toString();
															#line 976 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					connRecord = makeReference<ClusterConnectionFile>(joinPath(*dataFolder, "fdb.cluster"), newConnStr);
															#line 2945 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
			}
			else
			{
															#line 979 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				TraceEvent("SimulatedFDBDJustRepeat") .detail("Cycles", cycles) .detail("RandomId", randomId) .detail("Address", process->address) .detail("ZoneId", localities.zoneId()) .detail("KillType", shutdownResult);
															#line 2952 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont16when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont16cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont16when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont16cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state > 0) static_cast<SimulatedFDBDRebooterActor*>(this)->actor_wait_state = 0;
		static_cast<SimulatedFDBDRebooterActor*>(this)->ActorCallback< SimulatedFDBDRebooterActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulatedFDBDRebooterActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont16when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< SimulatedFDBDRebooterActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont16when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< SimulatedFDBDRebooterActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Reference<IClusterConnectionRecord> connRecord;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	IPAddress ip;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	bool sslEnabled;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	uint16_t port;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	uint16_t listenPerProcess;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	LocalityData localities;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ProcessClass processClass;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::string* dataFolder;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::string* coordFolder;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::string baseFolder;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ClusterConnectionString connStr;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ClusterConnectionString otherConnStr;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	bool useSeedFile;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ProcessMode processMode;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::string whitelistBinPaths;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ProtocolVersion protocolVersion;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ConfigDBType configDBType;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	bool isDr;
															#line 733 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ISimulator::ProcessInfo* simProcess;
															#line 734 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	UID randomId;
															#line 735 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	int cycles;
															#line 736 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	IPAllowList allowList;
															#line 757 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ISimulator::ProcessInfo* process;
															#line 771 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Future<ISimulator::KillType> onShutdown;
															#line 3082 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
};
// This generated class is to be used only via simulatedFDBDRebooter()
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class SimulatedFDBDRebooterActor final : public Actor<ISimulator::KillType>, public ActorCallback< SimulatedFDBDRebooterActor, 0, Void >, public ActorCallback< SimulatedFDBDRebooterActor, 1, Void >, public ActorCallback< SimulatedFDBDRebooterActor, 2, Void >, public ActorCallback< SimulatedFDBDRebooterActor, 3, Void >, public ActorCallback< SimulatedFDBDRebooterActor, 4, Void >, public FastAllocated<SimulatedFDBDRebooterActor>, public SimulatedFDBDRebooterActorState<SimulatedFDBDRebooterActor> {
															#line 3087 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
	using FastAllocated<SimulatedFDBDRebooterActor>::operator new;
	using FastAllocated<SimulatedFDBDRebooterActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17707983326667880704UL, 14875275807852351744UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<ISimulator::KillType>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SimulatedFDBDRebooterActor, 0, Void >;
friend struct ActorCallback< SimulatedFDBDRebooterActor, 1, Void >;
friend struct ActorCallback< SimulatedFDBDRebooterActor, 2, Void >;
friend struct ActorCallback< SimulatedFDBDRebooterActor, 3, Void >;
friend struct ActorCallback< SimulatedFDBDRebooterActor, 4, Void >;
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	SimulatedFDBDRebooterActor(Reference<IClusterConnectionRecord> const& connRecord,IPAddress const& ip,bool const& sslEnabled,uint16_t const& port,uint16_t const& listenPerProcess,LocalityData const& localities,ProcessClass const& processClass,std::string* const& dataFolder,std::string* const& coordFolder,std::string const& baseFolder,ClusterConnectionString const& connStr,ClusterConnectionString const& otherConnStr,bool const& useSeedFile,ProcessMode const& processMode,std::string const& whitelistBinPaths,ProtocolVersion const& protocolVersion,ConfigDBType const& configDBType,bool const& isDr) 
															#line 3108 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		 : Actor<ISimulator::KillType>(),
		   SimulatedFDBDRebooterActorState<SimulatedFDBDRebooterActor>(connRecord, ip, sslEnabled, port, listenPerProcess, localities, processClass, dataFolder, coordFolder, baseFolder, connStr, otherConnStr, useSeedFile, processMode, whitelistBinPaths, protocolVersion, configDBType, isDr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15551843614329839360UL, 15909686783315870720UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedFDBDRebooterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("simulatedFDBDRebooter");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("simulatedFDBDRebooter", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SimulatedFDBDRebooterActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SimulatedFDBDRebooterActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< SimulatedFDBDRebooterActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< SimulatedFDBDRebooterActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< SimulatedFDBDRebooterActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
[[nodiscard]] Future<ISimulator::KillType> simulatedFDBDRebooter( Reference<IClusterConnectionRecord> const& connRecord, IPAddress const& ip, bool const& sslEnabled, uint16_t const& port, uint16_t const& listenPerProcess, LocalityData const& localities, ProcessClass const& processClass, std::string* const& dataFolder, std::string* const& coordFolder, std::string const& baseFolder, ClusterConnectionString const& connStr, ClusterConnectionString const& otherConnStr, bool const& useSeedFile, ProcessMode const& processMode, std::string const& whitelistBinPaths, ProtocolVersion const& protocolVersion, ConfigDBType const& configDBType, bool const& isDr ) {
															#line 715 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	return Future<ISimulator::KillType>(new SimulatedFDBDRebooterActor(connRecord, ip, sslEnabled, port, listenPerProcess, localities, processClass, dataFolder, coordFolder, baseFolder, connStr, otherConnStr, useSeedFile, processMode, whitelistBinPaths, protocolVersion, configDBType, isDr));
															#line 3144 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
}

#line 988 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"

// Since a datacenter kill is considered to be the same as killing a machine, files cannot be swapped across datacenters
std::map<Optional<Standalone<StringRef>>, std::vector<std::vector<std::string>>> availableFolders;
// process count is no longer needed because it is now the length of the vector of ip's, because it was one ip per
// process
															#line 3153 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
// This generated class is to be used only via simulatedMachine()
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
template <class SimulatedMachineActor>
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class SimulatedMachineActorState {
															#line 3159 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	SimulatedMachineActorState(ClusterConnectionString const& connStr,ClusterConnectionString const& otherConnStr,std::vector<IPAddress> const& ips,bool const& sslEnabled,LocalityData const& localities,ProcessClass const& processClass,std::string const& baseFolder,bool const& restarting,bool const& useSeedFile,ProcessMode const& processMode,bool const& sslOnly,std::string const& whitelistBinPaths,ProtocolVersion const& protocolVersion,ConfigDBType const& configDBType,bool const& isDr) 
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		 : connStr(connStr),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   otherConnStr(otherConnStr),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   ips(ips),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   sslEnabled(sslEnabled),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   localities(localities),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   processClass(processClass),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   baseFolder(baseFolder),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   restarting(restarting),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   useSeedFile(useSeedFile),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   processMode(processMode),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   sslOnly(sslOnly),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   whitelistBinPaths(whitelistBinPaths),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   protocolVersion(protocolVersion),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   configDBType(configDBType),
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   isDr(isDr),
															#line 1008 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   bootCount(0),
															#line 1009 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   myFolders(),
															#line 1010 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   coordFolders(),
															#line 1011 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   randomId(nondeterministicRandom()->randomUniqueID()),
															#line 1012 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   listenPerProcess((sslEnabled && !sslOnly) ? 2 : 1)
															#line 3204 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("simulatedMachine", reinterpret_cast<unsigned long>(this));

	}
	~SimulatedMachineActorState() 
	{
		fdb_probe_actor_destroy("simulatedMachine", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1015 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				CSimpleIni ini;
															#line 1016 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				ini.SetUnicode();
															#line 1017 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				ini.LoadFile(joinPath(baseFolder, "restartInfo.ini").c_str());
															#line 1019 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				for(int i = 0;i < ips.size();i++) {
															#line 1020 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					if (restarting)
															#line 3228 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					{
															#line 1021 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						myFolders.push_back( ini.GetValue(printable(localities.machineId()).c_str(), format("%d", i * listenPerProcess).c_str(), joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString()).c_str()));
															#line 1026 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						if (i == 0)
															#line 3234 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
						{
															#line 1027 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
							std::string coordinationFolder = ini.GetValue(printable(localities.machineId()).c_str(), "coordinationFolder", "");
															#line 1029 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
							if (!coordinationFolder.size())
															#line 3240 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
							{
															#line 1030 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
								coordinationFolder = ini.GetValue( printable(localities.machineId()).c_str(), format("c%d", i * listenPerProcess).c_str(), joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString()).c_str());
															#line 3244 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
							}
															#line 1034 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
							coordFolders.push_back(coordinationFolder);
															#line 3248 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
						}
						else
						{
															#line 1036 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
							coordFolders.push_back( ini.GetValue(printable(localities.machineId()).c_str(), format("c%d", i * listenPerProcess).c_str(), joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString()).c_str()));
															#line 3254 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
						}
					}
					else
					{
															#line 1042 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						coordFolders.push_back(joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString()));
															#line 1043 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						std::string thisFolder = deterministicRandom()->randomUniqueID().toString();
															#line 1044 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						myFolders.push_back(joinPath(baseFolder, thisFolder));
															#line 1045 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						platform::createDirectory(myFolders[i]);
															#line 1047 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						if (!useSeedFile)
															#line 3269 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
						{
															#line 1048 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
							writeFile(joinPath(myFolders[i], "fdb.cluster"), connStr.toString());
															#line 3273 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
						}
					}
				}
															#line 1053 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				;
															#line 3279 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SimulatedMachineActorState();
		static_cast<SimulatedMachineActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1300 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			g_simulator->getMachineById(localities.machineId())->openFiles.clear();
															#line 1301 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 3311 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1054 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		processes = std::vector<Future<ISimulator::KillType>>();
															#line 1055 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		for(int i = 0;i < ips.size();i++) {
															#line 1056 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			std::string path = joinPath(myFolders[i], "fdb.cluster");
															#line 1057 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			ProcessMode ipProcessMode = processMode == BackupAgentOnly ? (i == ips.size() - 1 ? BackupAgentOnly : FDBDOnly) : processMode;
															#line 1062 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			Reference<IClusterConnectionRecord> clusterFile;
															#line 1063 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (ipProcessMode != SimHTTPServer)
															#line 3342 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 1066 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				clusterFile = Reference<IClusterConnectionRecord>( useSeedFile || !fileExists(path) ? new ClusterConnectionFile(path, connStr.toString()) : new ClusterConnectionFile(path));
															#line 3346 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 1070 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			const int listenPort = i * listenPerProcess + 1;
															#line 1072 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (g_simulator->hasDiffProtocolProcess && !g_simulator->setDiffProtocol && ipProcessMode == FDBDOnly)
															#line 3352 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 1073 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				processes.push_back(simulatedFDBDRebooter(clusterFile, ips[i], sslEnabled, listenPort, listenPerProcess, localities, processClass, &myFolders[i], &coordFolders[i], baseFolder, connStr, otherConnStr, useSeedFile, ipProcessMode, whitelistBinPaths, protocolVersion, configDBType, isDr));
															#line 1091 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_simulator->setDiffProtocol = true;
															#line 3358 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1093 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				processes.push_back(simulatedFDBDRebooter(clusterFile, ips[i], sslEnabled, listenPort, listenPerProcess, localities, processClass, &myFolders[i], &coordFolders[i], baseFolder, connStr, otherConnStr, useSeedFile, ipProcessMode, whitelistBinPaths, g_network->protocolVersion(), configDBType, isDr));
															#line 3364 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 1112 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("SimulatedMachineProcess", randomId) .detail("Address", NetworkAddress(ips[i], listenPort, true, false)) .detail("ZoneId", localities.zoneId()) .detail("DataHall", localities.dataHallId()) .detail("Folder", myFolders[i]);
															#line 3368 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
															#line 1119 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		CODE_PROBE(bootCount >= 1, "Simulated machine rebooted");
															#line 1120 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		CODE_PROBE(bootCount >= 2, "Simulated machine rebooted twice");
															#line 1121 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		CODE_PROBE(bootCount >= 3, "Simulated machine rebooted three times", probe::decoration::rare);
															#line 1122 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		++bootCount;
															#line 1124 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedMachineStart", randomId) .detail("Folder0", myFolders[0]) .detail("CFolder0", coordFolders[0]) .detail("MachineIPs", toIPVectorString(ips)) .detail("SSL", sslEnabled) .detail("Processes", processes.size()) .detail("BootCount", bootCount) .detail("ProcessClass", processClass.toString()) .detail("Restarting", restarting) .detail("UseSeedFile", useSeedFile) .detail("ZoneId", localities.zoneId()) .detail("DataHall", localities.dataHallId()) .detail("Locality", localities.toString());
															#line 1138 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_0 = waitForAll(processes);
															#line 1138 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<SimulatedMachineActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3384 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<SimulatedMachineActor*>(this)->actor_wait_state = 1;
															#line 1138 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SimulatedMachineActor, 0, Void >*>(static_cast<SimulatedMachineActor*>(this)));
															#line 3389 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1140 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedMachineRebootStart", randomId) .detail("Folder0", myFolders[0]) .detail("CFolder0", coordFolders[0]) .detail("MachineIPs", toIPVectorString(ips)) .detail("ZoneId", localities.zoneId()) .detail("DataHall", localities.dataHallId());
															#line 3398 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 1149 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			auto& machineCache = g_simulator->getMachineById(localities.machineId())->openFiles;
															#line 1152 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			std::vector<AsyncFileNonDurable*> files;
															#line 1153 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			for(auto fileItr = machineCache.begin();fileItr != machineCache.end();++fileItr) {
															#line 1154 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				ASSERT(fileItr->second.get().isReady());
															#line 1155 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				files.push_back((AsyncFileNonDurable*)fileItr->second.get().get().getPtr());
															#line 3410 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 1158 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			std::vector<Future<Void>> killFutures;
															#line 1159 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			for(auto fileItr = files.begin();fileItr != files.end();++fileItr) {
															#line 1160 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				killFutures.push_back((*fileItr)->kill());
															#line 3418 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 1162 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			StrictFuture<Void> __when_expr_1 = waitForAll(killFutures);
															#line 1162 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (static_cast<SimulatedMachineActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3424 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<SimulatedMachineActor*>(this)->actor_wait_state = 2;
															#line 1162 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SimulatedMachineActor, 1, Void >*>(static_cast<SimulatedMachineActor*>(this)));
															#line 3429 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1140 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedMachineRebootStart", randomId) .detail("Folder0", myFolders[0]) .detail("CFolder0", coordFolders[0]) .detail("MachineIPs", toIPVectorString(ips)) .detail("ZoneId", localities.zoneId()) .detail("DataHall", localities.dataHallId());
															#line 3439 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 1149 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			auto& machineCache = g_simulator->getMachineById(localities.machineId())->openFiles;
															#line 1152 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			std::vector<AsyncFileNonDurable*> files;
															#line 1153 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			for(auto fileItr = machineCache.begin();fileItr != machineCache.end();++fileItr) {
															#line 1154 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				ASSERT(fileItr->second.get().isReady());
															#line 1155 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				files.push_back((AsyncFileNonDurable*)fileItr->second.get().get().getPtr());
															#line 3451 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 1158 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			std::vector<Future<Void>> killFutures;
															#line 1159 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			for(auto fileItr = files.begin();fileItr != files.end();++fileItr) {
															#line 1160 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				killFutures.push_back((*fileItr)->kill());
															#line 3459 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 1162 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			StrictFuture<Void> __when_expr_1 = waitForAll(killFutures);
															#line 1162 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (static_cast<SimulatedMachineActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3465 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<SimulatedMachineActor*>(this)->actor_wait_state = 2;
															#line 1162 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SimulatedMachineActor, 1, Void >*>(static_cast<SimulatedMachineActor*>(this)));
															#line 3470 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SimulatedMachineActor*>(this)->actor_wait_state > 0) static_cast<SimulatedMachineActor*>(this)->actor_wait_state = 0;
		static_cast<SimulatedMachineActor*>(this)->ActorCallback< SimulatedMachineActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulatedMachineActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SimulatedMachineActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SimulatedMachineActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
															#line 1165 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		filenames = std::set<std::string>();
															#line 1166 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		closingStr = std::string();
															#line 1167 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		auto& machineCache = g_simulator->getMachineById(localities.machineId())->openFiles;
															#line 1168 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		for( auto it : machineCache ) {
															#line 1169 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			filenames.insert(it.first);
															#line 1170 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			closingStr += it.first + ", ";
															#line 1171 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			ASSERT(it.second.get().canGet());
															#line 3567 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
															#line 1174 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		for( auto it : g_simulator->getMachineById(localities.machineId())->deletingOrClosingFiles ) {
															#line 1175 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			filenames.insert(it);
															#line 1176 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			closingStr += it + ", ";
															#line 3575 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
															#line 1179 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedMachineRebootAfterKills", randomId) .detail("Folder0", myFolders[0]) .detail("CFolder0", coordFolders[0]) .detail("MachineIPs", toIPVectorString(ips)) .detail("Closing", closingStr) .detail("ZoneId", localities.zoneId()) .detail("DataHall", localities.dataHallId());
															#line 1187 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		ISimulator::MachineInfo* machine = g_simulator->getMachineById(localities.machineId());
															#line 1188 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		machine->closingFiles = filenames;
															#line 1189 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		g_simulator->getMachineById(localities.machineId())->openFiles.clear();
															#line 1196 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		shutdownDelayCount = 0;
															#line 1197 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		backoff = 0;
															#line 1198 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		;
															#line 3591 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont6loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SimulatedMachineActor*>(this)->actor_wait_state > 0) static_cast<SimulatedMachineActor*>(this)->actor_wait_state = 0;
		static_cast<SimulatedMachineActor*>(this)->ActorCallback< SimulatedMachineActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulatedMachineActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SimulatedMachineActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SimulatedMachineActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont10(int loopDepth) 
	{
															#line 1226 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedFDBDFilesClosed", randomId) .detail("Address", toIPVectorString(ips)) .detail("ZoneId", localities.zoneId()) .detail("DataHall", localities.dataHallId());
															#line 1231 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		g_simulator->destroyMachine(localities.machineId());
															#line 1237 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		killType = processes[0].get();
															#line 1238 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		for(int i = 1;i < ips.size();i++) {
															#line 1239 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			killType = std::max(processes[i].get(), killType);
															#line 3695 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
															#line 1241 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		CODE_PROBE(true, "Simulated machine has been rebooted");
															#line 1243 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		swap = killType == ISimulator::KillType::Reboot && BUGGIFY_WITH_PROB(0.75) && g_simulator->canSwapToMachine(localities.zoneId());
															#line 1245 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (swap)
															#line 3703 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 1246 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			availableFolders[localities.dcId()].push_back(myFolders);
															#line 3707 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
															#line 1248 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		auto rebootTime = deterministicRandom()->random01() * MACHINE_REBOOT_TIME;
															#line 1250 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedMachineShutdown", randomId) .detail("Swap", swap) .detail("KillType", killType) .detail("RebootTime", rebootTime) .detail("ZoneId", localities.zoneId()) .detail("DataHall", localities.dataHallId()) .detail("MachineIPs", toIPVectorString(ips));
															#line 1258 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_3 = delay(rebootTime);
															#line 1258 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<SimulatedMachineActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3717 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont10when1(__when_expr_3.get(), loopDepth); };
		static_cast<SimulatedMachineActor*>(this)->actor_wait_state = 4;
															#line 1258 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< SimulatedMachineActor, 3, Void >*>(static_cast<SimulatedMachineActor*>(this)));
															#line 3722 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont6loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6loopBody1(int loopDepth) 
	{
															#line 1199 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		auto& machineCache = g_simulator->getMachineById(localities.machineId())->closingFiles;
															#line 1201 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (!machineCache.empty())
															#line 3740 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 1202 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			std::string openFiles;
															#line 1203 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			int i = 0;
															#line 1204 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			for(auto it = machineCache.begin();it != machineCache.end() && i < 5;++it) {
															#line 1205 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				openFiles += *it + ", ";
															#line 1206 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				i++;
															#line 3752 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 1208 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("MachineFilesOpen", randomId) .detail("PAddr", toIPVectorString(ips)) .detail("OpenFiles", openFiles);
															#line 3756 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
		else
		{
			return a_body1loopBody1cont6break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1214 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (shutdownDelayCount++ >= 50)
															#line 3764 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 1215 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent(SevError, "SimulatedFDBDFilesCheck", randomId) .detail("PAddrs", toIPVectorString(ips)) .detail("ZoneId", localities.zoneId()) .detail("DataHall", localities.dataHallId());
															#line 1219 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			ASSERT(false);
															#line 3770 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
															#line 1222 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(backoff);
															#line 1222 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<SimulatedMachineActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 3776 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont6loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<SimulatedMachineActor*>(this)->actor_wait_state = 3;
															#line 1222 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SimulatedMachineActor, 2, Void >*>(static_cast<SimulatedMachineActor*>(this)));
															#line 3781 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont6loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1223 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		backoff = std::min(backoff + 1.0, 6.0);
															#line 3803 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont6loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1223 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		backoff = std::min(backoff + 1.0, 6.0);
															#line 3812 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1cont6loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont6loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<SimulatedMachineActor*>(this)->actor_wait_state > 0) static_cast<SimulatedMachineActor*>(this)->actor_wait_state = 0;
		static_cast<SimulatedMachineActor*>(this)->ActorCallback< SimulatedMachineActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulatedMachineActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont6loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< SimulatedMachineActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont6loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< SimulatedMachineActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont10cont1(Void const& _,int loopDepth) 
	{
															#line 1260 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (swap)
															#line 3896 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 1261 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			auto& avail = availableFolders[localities.dcId()];
															#line 1262 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			int i = deterministicRandom()->randomInt(0, avail.size());
															#line 1263 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (i != avail.size() - 1)
															#line 3904 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 1264 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				std::swap(avail[i], avail.back());
															#line 3908 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 1265 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			auto toRebootFrom = avail.back();
															#line 1266 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			avail.pop_back();
															#line 1268 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (myFolders != toRebootFrom)
															#line 3916 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 1269 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				CODE_PROBE(true, "Simulated machine swapped data folders", probe::decoration::rare);
															#line 1270 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				TraceEvent("SimulatedMachineFolderSwap", randomId) .detail("OldFolder0", myFolders[0]) .detail("NewFolder0", toRebootFrom[0]) .detail("MachineIPs", toIPVectorString(ips));
															#line 3922 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 1275 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			myFolders = toRebootFrom;
															#line 1276 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (!useSeedFile)
															#line 3928 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 1277 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				for( auto f : toRebootFrom ) {
															#line 1278 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					if (!fileExists(joinPath(f, "fdb.cluster")))
															#line 3934 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					{
															#line 1279 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						writeFile(joinPath(f, "fdb.cluster"), connStr.toString());
															#line 3938 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					}
				}
			}
		}
		else
		{
															#line 1283 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (killType == ISimulator::KillType::RebootAndDelete)
															#line 3947 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 1284 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				for(int i = 0;i < ips.size();i++) {
															#line 1285 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					coordFolders[i] = joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString());
															#line 1286 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					myFolders[i] = joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString());
															#line 1287 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					platform::createDirectory(myFolders[i]);
															#line 1289 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					if (!useSeedFile)
															#line 3959 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					{
															#line 1290 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						writeFile(joinPath(myFolders[i], "fdb.cluster"), connStr.toString());
															#line 3963 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					}
				}
															#line 1294 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				CODE_PROBE(true, "Simulated machine rebooted with data loss");
															#line 3968 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont10cont1(Void && _,int loopDepth) 
	{
															#line 1260 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (swap)
															#line 3979 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 1261 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			auto& avail = availableFolders[localities.dcId()];
															#line 1262 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			int i = deterministicRandom()->randomInt(0, avail.size());
															#line 1263 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (i != avail.size() - 1)
															#line 3987 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 1264 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				std::swap(avail[i], avail.back());
															#line 3991 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 1265 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			auto toRebootFrom = avail.back();
															#line 1266 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			avail.pop_back();
															#line 1268 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (myFolders != toRebootFrom)
															#line 3999 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 1269 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				CODE_PROBE(true, "Simulated machine swapped data folders", probe::decoration::rare);
															#line 1270 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				TraceEvent("SimulatedMachineFolderSwap", randomId) .detail("OldFolder0", myFolders[0]) .detail("NewFolder0", toRebootFrom[0]) .detail("MachineIPs", toIPVectorString(ips));
															#line 4005 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 1275 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			myFolders = toRebootFrom;
															#line 1276 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (!useSeedFile)
															#line 4011 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 1277 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				for( auto f : toRebootFrom ) {
															#line 1278 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					if (!fileExists(joinPath(f, "fdb.cluster")))
															#line 4017 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					{
															#line 1279 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						writeFile(joinPath(f, "fdb.cluster"), connStr.toString());
															#line 4021 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					}
				}
			}
		}
		else
		{
															#line 1283 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (killType == ISimulator::KillType::RebootAndDelete)
															#line 4030 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 1284 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				for(int i = 0;i < ips.size();i++) {
															#line 1285 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					coordFolders[i] = joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString());
															#line 1286 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					myFolders[i] = joinPath(baseFolder, deterministicRandom()->randomUniqueID().toString());
															#line 1287 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					platform::createDirectory(myFolders[i]);
															#line 1289 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					if (!useSeedFile)
															#line 4042 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					{
															#line 1290 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						writeFile(joinPath(myFolders[i], "fdb.cluster"), connStr.toString());
															#line 4046 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					}
				}
															#line 1294 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				CODE_PROBE(true, "Simulated machine rebooted with data loss");
															#line 4051 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<SimulatedMachineActor*>(this)->actor_wait_state > 0) static_cast<SimulatedMachineActor*>(this)->actor_wait_state = 0;
		static_cast<SimulatedMachineActor*>(this)->ActorCallback< SimulatedMachineActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulatedMachineActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< SimulatedMachineActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< SimulatedMachineActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ClusterConnectionString connStr;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ClusterConnectionString otherConnStr;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::vector<IPAddress> ips;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	bool sslEnabled;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	LocalityData localities;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ProcessClass processClass;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::string baseFolder;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	bool restarting;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	bool useSeedFile;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ProcessMode processMode;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	bool sslOnly;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::string whitelistBinPaths;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ProtocolVersion protocolVersion;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ConfigDBType configDBType;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	bool isDr;
															#line 1008 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	int bootCount;
															#line 1009 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::vector<std::string> myFolders;
															#line 1010 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::vector<std::string> coordFolders;
															#line 1011 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	UID randomId;
															#line 1012 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	int listenPerProcess;
															#line 1054 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::vector<Future<ISimulator::KillType>> processes;
															#line 1165 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::set<std::string> filenames;
															#line 1166 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::string closingStr;
															#line 1196 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	int shutdownDelayCount;
															#line 1197 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	double backoff;
															#line 1237 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ISimulator::KillType killType;
															#line 1243 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	bool swap;
															#line 4187 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
};
// This generated class is to be used only via simulatedMachine()
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class SimulatedMachineActor final : public Actor<Void>, public ActorCallback< SimulatedMachineActor, 0, Void >, public ActorCallback< SimulatedMachineActor, 1, Void >, public ActorCallback< SimulatedMachineActor, 2, Void >, public ActorCallback< SimulatedMachineActor, 3, Void >, public FastAllocated<SimulatedMachineActor>, public SimulatedMachineActorState<SimulatedMachineActor> {
															#line 4192 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
	using FastAllocated<SimulatedMachineActor>::operator new;
	using FastAllocated<SimulatedMachineActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16378651824743064832UL, 4886195206910609152UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SimulatedMachineActor, 0, Void >;
friend struct ActorCallback< SimulatedMachineActor, 1, Void >;
friend struct ActorCallback< SimulatedMachineActor, 2, Void >;
friend struct ActorCallback< SimulatedMachineActor, 3, Void >;
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	SimulatedMachineActor(ClusterConnectionString const& connStr,ClusterConnectionString const& otherConnStr,std::vector<IPAddress> const& ips,bool const& sslEnabled,LocalityData const& localities,ProcessClass const& processClass,std::string const& baseFolder,bool const& restarting,bool const& useSeedFile,ProcessMode const& processMode,bool const& sslOnly,std::string const& whitelistBinPaths,ProtocolVersion const& protocolVersion,ConfigDBType const& configDBType,bool const& isDr) 
															#line 4212 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   SimulatedMachineActorState<SimulatedMachineActor>(connStr, otherConnStr, ips, sslEnabled, localities, processClass, baseFolder, restarting, useSeedFile, processMode, sslOnly, whitelistBinPaths, protocolVersion, configDBType, isDr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("simulatedMachine", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9396936184670874624UL, 10527453599203259392UL);
		ActorExecutionContextHelper __helper(static_cast<SimulatedMachineActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("simulatedMachine");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("simulatedMachine", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SimulatedMachineActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SimulatedMachineActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< SimulatedMachineActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< SimulatedMachineActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
[[nodiscard]] Future<Void> simulatedMachine( ClusterConnectionString const& connStr, ClusterConnectionString const& otherConnStr, std::vector<IPAddress> const& ips, bool const& sslEnabled, LocalityData const& localities, ProcessClass const& processClass, std::string const& baseFolder, bool const& restarting, bool const& useSeedFile, ProcessMode const& processMode, bool const& sslOnly, std::string const& whitelistBinPaths, ProtocolVersion const& protocolVersion, ConfigDBType const& configDBType, bool const& isDr ) {
															#line 993 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	return Future<Void>(new SimulatedMachineActor(connStr, otherConnStr, ips, sslEnabled, localities, processClass, baseFolder, restarting, useSeedFile, processMode, sslOnly, whitelistBinPaths, protocolVersion, configDBType, isDr));
															#line 4247 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
}

#line 1304 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"

IPAddress makeIPAddressForSim(bool isIPv6, std::array<int, 4> parts) {
	if (isIPv6) {
		IPAddress::IPAddressStore addrStore{ 0xAB, 0xCD };
		uint16_t* ptr = (uint16_t*)addrStore.data();
		ptr[4] = (uint16_t)(parts[0] << 8);
		ptr[5] = (uint16_t)(parts[1] << 8);
		ptr[6] = (uint16_t)(parts[2] << 8);
		ptr[7] = (uint16_t)(parts[3] << 8);
		return IPAddress(addrStore);
	} else {
		return IPAddress(parts[0] << 24 | parts[1] << 16 | parts[2] << 8 | parts[3]);
	}
}

#include "fdbclient/MonitorLeader.h"

// Configures the system according to the given specifications in order to run
// simulation, but with the additional consideration that it is meant to act
// like a "rebooted" machine, mostly used for restarting tests.
															#line 4271 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
// This generated class is to be used only via restartSimulatedSystem()
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
template <class RestartSimulatedSystemActor>
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class RestartSimulatedSystemActorState {
															#line 4277 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	RestartSimulatedSystemActorState(std::vector<Future<Void>>* const& systemActors,std::string const& baseFolder,int* const& pTesterCount,Optional<ClusterConnectionString>* const& pConnString,Standalone<StringRef>* const& pStartingConfiguration,TestConfig* const& testConfig,std::string const& whitelistBinPaths,ProtocolVersion const& protocolVersion) 
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		 : systemActors(systemActors),
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   baseFolder(baseFolder),
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   pTesterCount(pTesterCount),
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   pConnString(pConnString),
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   pStartingConfiguration(pStartingConfiguration),
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   testConfig(testConfig),
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   whitelistBinPaths(whitelistBinPaths),
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   protocolVersion(protocolVersion)
															#line 4298 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("restartSimulatedSystem", reinterpret_cast<unsigned long>(this));

	}
	~RestartSimulatedSystemActorState() 
	{
		fdb_probe_actor_destroy("restartSimulatedSystem", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1332 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			CSimpleIni ini;
															#line 1333 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			ini.SetUnicode();
															#line 1334 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			ini.LoadFile(joinPath(baseFolder, "restartInfo.ini").c_str());
															#line 1336 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			auto configDBType = testConfig->getConfigDBType();
															#line 1340 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			bool renameZoneIds = testConfig->randomlyRenameZoneId ? deterministicRandom()->random01() < 0.1 : false;
															#line 1341 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			CODE_PROBE(renameZoneIds, "Zone ID names altered in restart test");
															#line 1344 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			ini.SetMultiKey();
															#line 4325 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			try {
															#line 1347 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				int machineCount = atoi(ini.GetValue("META", "machineCount"));
															#line 1348 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				int processesPerMachine = atoi(ini.GetValue("META", "processesPerMachine"));
															#line 1349 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				int listenersPerProcess = 1;
															#line 1350 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				auto listenersPerProcessStr = ini.GetValue("META", "listenersPerProcess");
															#line 1351 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (listenersPerProcessStr != nullptr)
															#line 4337 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 1352 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					listenersPerProcess = atoi(listenersPerProcessStr);
															#line 4341 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 1354 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				int desiredCoordinators = atoi(ini.GetValue("META", "desiredCoordinators"));
															#line 1355 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				int testerCount = atoi(ini.GetValue("META", "testerCount"));
															#line 1356 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				auto tssModeStr = ini.GetValue("META", "tssMode");
															#line 1357 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (tssModeStr != nullptr)
															#line 4351 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 1358 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					g_simulator->tssMode = (ISimulator::TSSMode)atoi(tssModeStr);
															#line 4355 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 1360 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				ClusterConnectionString conn(ini.GetValue("META", "connectionString"));
															#line 1361 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (testConfig->extraDatabaseMode == ISimulator::ExtraDatabaseMode::Local)
															#line 4361 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 1362 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					g_simulator->extraDatabases.clear();
															#line 1363 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					g_simulator->extraDatabases.push_back(conn.toString());
															#line 4367 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 1365 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (!testConfig->disableHostname)
															#line 4371 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 1366 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					auto mockDNSStr = ini.GetValue("META", "mockDNS");
															#line 1367 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					if (mockDNSStr != nullptr)
															#line 4377 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					{
															#line 1368 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						INetworkConnections::net()->parseMockDNSFromString(mockDNSStr);
															#line 4381 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					}
				}
															#line 1371 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				auto& g_knobs = IKnobCollection::getMutableGlobalKnobCollection();
															#line 1372 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (testConfig->disableRemoteKVS)
															#line 4388 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 1373 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					g_knobs.setKnob("remote_kv_store", KnobValueRef::create(bool{ false }));
															#line 1374 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					TraceEvent(SevDebug, "DisableRemoteKVS");
															#line 4394 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 1376 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				*pConnString = conn;
															#line 1377 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				*pTesterCount = testerCount;
															#line 1378 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				bool usingSSL = conn.toString().find(":tls") != std::string::npos || listenersPerProcess > 1;
															#line 1379 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				int useSeedForMachine = deterministicRandom()->randomInt(0, machineCount);
															#line 1380 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				std::vector<std::string> dcIds;
															#line 1381 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				for(int i = 0;i < machineCount;i++) {
															#line 1382 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					Optional<Standalone<StringRef>> dcUID;
															#line 1383 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					Optional<Standalone<StringRef>> zoneId;
															#line 1384 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					std::string machineIdString = ini.GetValue("META", format("%d", i).c_str());
															#line 1385 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					Standalone<StringRef> machineId = StringRef(machineIdString);
															#line 1387 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					std::string dcUIDini = ini.GetValue(machineIdString.c_str(), "dcUID");
															#line 1388 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					if (!dcUIDini.empty())
															#line 4420 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					{
															#line 1389 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						dcUID = StringRef(dcUIDini);
															#line 4424 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					}
															#line 1392 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					auto zoneIDini = ini.GetValue(machineIdString.c_str(), "zoneId");
															#line 1393 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					if (zoneIDini == nullptr)
															#line 4430 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					{
															#line 1394 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						zoneId = machineId;
															#line 4434 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					}
					else
					{
															#line 1396 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						auto zoneIdStr = std::string(zoneIDini);
															#line 1397 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						if (renameZoneIds)
															#line 4442 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
						{
															#line 1398 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
							zoneIdStr = "modified/" + zoneIdStr;
															#line 4446 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
						}
															#line 1400 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						zoneId = Standalone<StringRef>(zoneIdStr);
															#line 4450 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					}
															#line 1403 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					ProcessClass::ClassType cType = (ProcessClass::ClassType)(atoi(ini.GetValue(machineIdString.c_str(), "mClass")));
															#line 1406 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					if (cType == ProcessClass::MasterClass || cType == ProcessClass::ResolutionClass)
															#line 4456 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					{
															#line 1407 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						cType = ProcessClass::StatelessClass;
															#line 4460 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					}
															#line 1409 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					ProcessClass processClass = ProcessClass(cType, ProcessClass::CommandLineSource);
															#line 1411 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					if (processClass != ProcessClass::TesterClass)
															#line 4466 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					{
															#line 1412 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						dcIds.push_back(dcUIDini);
															#line 4470 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					}
															#line 1415 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					std::vector<IPAddress> ipAddrs;
															#line 1416 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					int processes = atoi(ini.GetValue(machineIdString.c_str(), "processes"));
															#line 1418 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					auto ip = ini.GetValue(machineIdString.c_str(), "ipAddr");
															#line 1423 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					auto parseIp = [](const char* ipStr) -> IPAddress { Optional<IPAddress> parsedIp = IPAddress::parse(ipStr); if (parsedIp.present()) { return parsedIp.get(); } else { return IPAddress(strtoul(ipStr, nullptr, 10)); } };
															#line 1432 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					if (ip == nullptr)
															#line 4482 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
					{
															#line 1433 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						for(int i = 0;i < processes;i++) {
															#line 1434 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
							const char* val = ini.GetValue(machineIdString.c_str(), format("ipAddr%d", i * listenersPerProcess).c_str());
															#line 1436 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
							ipAddrs.push_back(parseIp(val));
															#line 4490 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
						}
					}
					else
					{
															#line 1440 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						ipAddrs.push_back(parseIp(ip));
															#line 1442 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
						for(int i = 1;i < processes;i++) {
															#line 1443 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
							if (ipAddrs.back().isV6())
															#line 4501 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
							{
															#line 1444 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
								IPAddress::IPAddressStore store = ipAddrs.back().toV6();
															#line 1445 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
								uint16_t* ptr = (uint16_t*)store.data();
															#line 1446 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
								ptr[7] += 1;
															#line 1447 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
								ipAddrs.push_back(IPAddress(store));
															#line 4511 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
							}
							else
							{
															#line 1449 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
								ipAddrs.push_back(IPAddress(ipAddrs.back().toV4() + 1));
															#line 4517 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
							}
						}
					}
															#line 1454 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					LocalityData localities(Optional<Standalone<StringRef>>(), zoneId, machineId, dcUID);
															#line 1455 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					localities.set("data_hall"_sr, dcUID);
															#line 1458 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					systemActors->push_back(reportErrors( simulatedMachine(conn, ClusterConnectionString(), ipAddrs, usingSSL, localities, processClass, baseFolder, true, i == useSeedForMachine, processClass == ProcessClass::SimHTTPServerClass ? SimHTTPServer : FDBDAndBackupAgent, usingSSL && (listenersPerProcess == 1 || processClass == ProcessClass::TesterClass), whitelistBinPaths, protocolVersion, configDBType, false), processClass == ProcessClass::TesterClass ? "SimulatedTesterMachine" : "SimulatedMachine"));
															#line 4527 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 1477 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_simulator->desiredCoordinators = desiredCoordinators;
															#line 1478 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_simulator->processesPerMachine = processesPerMachine;
															#line 1480 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				uniquify(dcIds);
															#line 1481 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (!BUGGIFY && dcIds.size() == 2 && dcIds[0] != "" && dcIds[1] != "")
															#line 4537 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				{
															#line 1482 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					StatusObject primaryObj;
															#line 1483 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					StatusObject primaryDcObj;
															#line 1484 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					primaryDcObj["id"] = dcIds[0];
															#line 1485 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					primaryDcObj["priority"] = 2;
															#line 1486 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					StatusArray primaryDcArr;
															#line 1487 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					primaryDcArr.push_back(primaryDcObj);
															#line 1489 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					StatusObject remoteObj;
															#line 1490 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					StatusObject remoteDcObj;
															#line 1491 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					remoteDcObj["id"] = dcIds[1];
															#line 1492 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					remoteDcObj["priority"] = 1;
															#line 1493 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					StatusArray remoteDcArr;
															#line 1494 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					remoteDcArr.push_back(remoteDcObj);
															#line 1496 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					primaryObj["datacenters"] = primaryDcArr;
															#line 1497 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					remoteObj["datacenters"] = remoteDcArr;
															#line 1499 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					StatusArray regionArr;
															#line 1500 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					regionArr.push_back(primaryObj);
															#line 1501 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					regionArr.push_back(remoteObj);
															#line 1503 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
					*pStartingConfiguration = "single usable_regions=2 regions=" + json_spirit::write_string(json_spirit::mValue(regionArr), json_spirit::Output_options::none);
															#line 4575 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				}
															#line 1508 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_simulator->restarted = true;
															#line 1510 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				TraceEvent("RestartSimulatorSettings") .detail("DesiredCoordinators", g_simulator->desiredCoordinators) .detail("ProcessesPerMachine", g_simulator->processesPerMachine) .detail("ListenersPerProcess", listenersPerProcess);
															#line 4581 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				loopDepth = a_body1cont23(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RestartSimulatedSystemActorState();
		static_cast<RestartSimulatedSystemActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1518 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(1.0);
															#line 1518 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (static_cast<RestartSimulatedSystemActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4612 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1cont1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RestartSimulatedSystemActor*>(this)->actor_wait_state = 1;
															#line 1518 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RestartSimulatedSystemActor, 0, Void >*>(static_cast<RestartSimulatedSystemActor*>(this)));
															#line 4617 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1515 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent(SevError, "RestartSimulationError").error(e);
															#line 4627 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont23(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont24(Void const& _,int loopDepth) 
	{
															#line 1520 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (!static_cast<RestartSimulatedSystemActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RestartSimulatedSystemActorState(); static_cast<RestartSimulatedSystemActor*>(this)->destroy(); return 0; }
															#line 4655 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		new (&static_cast<RestartSimulatedSystemActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RestartSimulatedSystemActorState();
		static_cast<RestartSimulatedSystemActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont24(Void && _,int loopDepth) 
	{
															#line 1520 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (!static_cast<RestartSimulatedSystemActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RestartSimulatedSystemActorState(); static_cast<RestartSimulatedSystemActor*>(this)->destroy(); return 0; }
															#line 4667 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		new (&static_cast<RestartSimulatedSystemActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RestartSimulatedSystemActorState();
		static_cast<RestartSimulatedSystemActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont24(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont24(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RestartSimulatedSystemActor*>(this)->actor_wait_state > 0) static_cast<RestartSimulatedSystemActor*>(this)->actor_wait_state = 0;
		static_cast<RestartSimulatedSystemActor*>(this)->ActorCallback< RestartSimulatedSystemActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RestartSimulatedSystemActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("restartSimulatedSystem", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestartSimulatedSystemActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restartSimulatedSystem", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RestartSimulatedSystemActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("restartSimulatedSystem", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestartSimulatedSystemActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restartSimulatedSystem", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RestartSimulatedSystemActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("restartSimulatedSystem", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RestartSimulatedSystemActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restartSimulatedSystem", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::vector<Future<Void>>* systemActors;
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::string baseFolder;
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	int* pTesterCount;
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Optional<ClusterConnectionString>* pConnString;
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Standalone<StringRef>* pStartingConfiguration;
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	TestConfig* testConfig;
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::string whitelistBinPaths;
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ProtocolVersion protocolVersion;
															#line 4766 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
};
// This generated class is to be used only via restartSimulatedSystem()
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class RestartSimulatedSystemActor final : public Actor<Void>, public ActorCallback< RestartSimulatedSystemActor, 0, Void >, public FastAllocated<RestartSimulatedSystemActor>, public RestartSimulatedSystemActorState<RestartSimulatedSystemActor> {
															#line 4771 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
	using FastAllocated<RestartSimulatedSystemActor>::operator new;
	using FastAllocated<RestartSimulatedSystemActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11568508454949301248UL, 3134874495607635712UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RestartSimulatedSystemActor, 0, Void >;
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	RestartSimulatedSystemActor(std::vector<Future<Void>>* const& systemActors,std::string const& baseFolder,int* const& pTesterCount,Optional<ClusterConnectionString>* const& pConnString,Standalone<StringRef>* const& pStartingConfiguration,TestConfig* const& testConfig,std::string const& whitelistBinPaths,ProtocolVersion const& protocolVersion) 
															#line 4788 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RestartSimulatedSystemActorState<RestartSimulatedSystemActor>(systemActors, baseFolder, pTesterCount, pConnString, pStartingConfiguration, testConfig, whitelistBinPaths, protocolVersion),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("restartSimulatedSystem", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11816297150536197376UL, 12373310510592618752UL);
		ActorExecutionContextHelper __helper(static_cast<RestartSimulatedSystemActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("restartSimulatedSystem");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("restartSimulatedSystem", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RestartSimulatedSystemActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
[[nodiscard]] Future<Void> restartSimulatedSystem( std::vector<Future<Void>>* const& systemActors, std::string const& baseFolder, int* const& pTesterCount, Optional<ClusterConnectionString>* const& pConnString, Standalone<StringRef>* const& pStartingConfiguration, TestConfig* const& testConfig, std::string const& whitelistBinPaths, ProtocolVersion const& protocolVersion ) {
															#line 1324 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	return Future<Void>(new RestartSimulatedSystemActor(systemActors, baseFolder, pTesterCount, pConnString, pStartingConfiguration, testConfig, whitelistBinPaths, protocolVersion));
															#line 4820 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
}

#line 1522 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"

// Configuration details compiled in a structure used when setting up a simulated cluster
struct SimulationConfig : public BasicSimulationConfig {
	explicit SimulationConfig(const TestConfig& testConfig);
	ISimulator::ExtraDatabaseMode extraDatabaseMode;
	int extraDatabaseCount;
	bool generateFearless;

	void set_config(std::string config);

	int coordinators;

private:
	void setRandomConfig();
	void setSimpleConfig();
	void setSpecificConfig(const TestConfig& testConfig);
	void setDatacenters(const TestConfig& testConfig);
	void setEncryptionAtRestMode(const TestConfig& testConfig);
	void setStorageEngine(const TestConfig& testConfig);
	void setRegions(const TestConfig& testConfig);
	void setReplicationType(const TestConfig& testConfig);
	void setMachineCount(const TestConfig& testConfig);
	void setCoordinators(const TestConfig& testConfig);
	void setProcessesPerMachine(const TestConfig& testConfig);
	void setTss(const TestConfig& testConfig);
	void generateNormalConfig(const TestConfig& testConfig);
};

SimulationConfig::SimulationConfig(const TestConfig& testConfig)
  : extraDatabaseMode(testConfig.extraDatabaseMode), extraDatabaseCount(testConfig.extraDatabaseCount) {
	generateNormalConfig(testConfig);
}

void SimulationConfig::set_config(std::string config) {
	// The only mechanism we have for turning "single" into what single means
	// is buildConfiguration()... :/
	std::map<std::string, std::string> hack_map;
	const auto buildResult = buildConfiguration(config, hack_map);
	ASSERT(buildResult != ConfigurationResult::NO_OPTIONS_PROVIDED);
	for (auto kv : hack_map)
		db.set(kv.first, kv.second);
}

[[maybe_unused]] StringRef StringRefOf(const char* s) {
	return StringRef((uint8_t*)s, strlen(s));
}

// Set the randomly generated options of the config. Compiled here to easily observe and trace random options
void SimulationConfig::setRandomConfig() {
	if (deterministicRandom()->random01() < 0.25) {
		db.desiredTLogCount = deterministicRandom()->randomInt(1, 7);
	}
	if (deterministicRandom()->random01() < 0.25) {
		db.commitProxyCount = deterministicRandom()->randomInt(1, 7);
	}
	if (deterministicRandom()->random01() < 0.25) {
		db.grvProxyCount = deterministicRandom()->randomInt(1, 4);
	}
	if (deterministicRandom()->random01() < 0.25) {
		db.resolverCount = deterministicRandom()->randomInt(1, 7);
	}
	// TraceEvent("SimulatedConfigRandom")
	// 	.detail("DesiredTLogCount", db.desiredTLogCount)
	// 	.detail("CommitProxyCount", db.commitProxyCount)
	// 	.detail("GRVProxyCount", db.grvProxyCount)
	// 	.detail("ResolverCount", db.resolverCount);

	if (deterministicRandom()->random01() < 0.5) {
		// TraceEvent("SimulatedConfigRandom").detail("PerpetualWiggle", 0);
		set_config("perpetual_storage_wiggle=0");
	} else {
		// TraceEvent("SimulatedConfigRandom").detail("PerpetualWiggle", 1);
		set_config("perpetual_storage_wiggle=1");
	}

	if (deterministicRandom()->random01() < 0.5) {
		set_config("backup_worker_enabled:=1");
	}
}

// Overwrite DB with simple options, used when simpleConfig is true in the TestConfig
void SimulationConfig::setSimpleConfig() {
	db.desiredTLogCount = 1;
	db.commitProxyCount = 1;
	db.grvProxyCount = 1;
	db.resolverCount = 1;
}

// Overwrite previous options with ones specified by TestConfig
void SimulationConfig::setSpecificConfig(const TestConfig& testConfig) {
	if (testConfig.desiredTLogCount.present()) {
		db.desiredTLogCount = testConfig.desiredTLogCount.get();
	}
	if (testConfig.remoteDesiredTLogCount.present()) {
		db.remoteDesiredTLogCount = testConfig.remoteDesiredTLogCount.get();
	}
	if (testConfig.commitProxyCount.present()) {
		db.commitProxyCount = testConfig.commitProxyCount.get();
	}
	if (testConfig.grvProxyCount.present()) {
		db.grvProxyCount = testConfig.grvProxyCount.get();
	}
	if (testConfig.resolverCount.present()) {
		db.resolverCount = testConfig.resolverCount.get();
	}
}

// Sets generateFearless and number of dataCenters based on testConfig details
// The number of datacenters may be overwritten in setRegions
void SimulationConfig::setDatacenters(const TestConfig& testConfig) {

#ifdef NO_MULTIREGION_TEST
	if (testConfig.minimumRegions > 1 || (testConfig.generateFearless.present() && testConfig.generateFearless.get())) {
		throw internal_error_msg("Test requires multi-region while the flag is turned off in the build process");
	}
	generateFearless = false;
#else
	generateFearless =
	    testConfig.generateFearless.present()
	        ? testConfig.generateFearless.get()
	        : (!testConfig.simpleConfig && (testConfig.minimumRegions > 1 || deterministicRandom()->random01() < 0.5));
#endif
	datacenters =
	    testConfig.simpleConfig
	        ? 1
	        : (generateFearless ? (testConfig.minimumReplication > 0 || deterministicRandom()->random01() < 0.5 ? 4 : 6)
	                            : deterministicRandom()->randomInt(1, 4));

	// Overwrite with specific option if present
	if (testConfig.datacenters.present()) {
		datacenters = testConfig.datacenters.get();
	}
}

// TODO(gglass): consider removing this.
void SimulationConfig::setEncryptionAtRestMode(const TestConfig& testConfig) {
	// Non-DISABLED encryption at rest values are experimental and are being removed.
	EncryptionAtRestMode encryptionMode = EncryptionAtRestMode::DISABLED;
	TraceEvent("SimulatedClusterEncryptionMode").detail("Mode", encryptionMode.toString());
	CODE_PROBE(true, "Enforce to disable encryption in simulation", probe::decoration::rare);
	set_config("encryption_at_rest_mode=" + encryptionMode.toString());
}

namespace {

using StorageEngineConfigFunc = void (*)(SimulationConfig*);

void ssdStorageEngineConfig(SimulationConfig* simCfg) {
	CODE_PROBE(true, "Simulated cluster using ssd storage engine");
	simCfg->set_config("ssd");
}
void memoryStorageEngineConfig(SimulationConfig* simCfg) {
	CODE_PROBE(true, "Simulated cluster using default memory storage engine");
	simCfg->set_config("memory");
}

void radixTreeStorageEngineConfig(SimulationConfig* simCfg) {
	CODE_PROBE(true, "Simulated cluster using radix-tree storage engine");
	simCfg->set_config("memory-radixtree");
}

void redwoodStorageEngineConfig(SimulationConfig* simCfg) {
	CODE_PROBE(true, "Simulated cluster using redwood storage engine");
	// The experimental suffix is still supported so test it randomly
	simCfg->set_config("ssd-redwood-1");
}

void rocksdbStorageEngineConfig(SimulationConfig* simCfg) {
	CODE_PROBE(true, "Simulated cluster using RocksDB storage engine", probe::assert::hasRocksDB);
	simCfg->set_config("ssd-rocksdb-v1");
}

void shardedRocksDBStorageEngineConfig(SimulationConfig* simCfg) {
	CODE_PROBE(true, "Simulated cluster using Sharded RocksDB storage engine", probe::assert::hasRocksDB);
	simCfg->set_config("encryption_at_rest_mode=disabled");
	simCfg->set_config("ssd-sharded-rocksdb");
}

const std::unordered_map<SimulationStorageEngine, StorageEngineConfigFunc> STORAGE_ENGINE_CONFIG_MAPPER = {
	{ SimulationStorageEngine::SSD, ssdStorageEngineConfig },
	{ SimulationStorageEngine::MEMORY, memoryStorageEngineConfig },
	{ SimulationStorageEngine::RADIX_TREE, radixTreeStorageEngineConfig },
	{ SimulationStorageEngine::REDWOOD, redwoodStorageEngineConfig },
	{ SimulationStorageEngine::ROCKSDB, rocksdbStorageEngineConfig },
	{ SimulationStorageEngine::SHARDED_ROCKSDB, shardedRocksDBStorageEngineConfig }
};

// TODO: Figure out what is broken with the RocksDB engine in simulation.
const std::vector<SimulationStorageEngine> SIMULATION_STORAGE_ENGINE = {
	SimulationStorageEngine::SSD,        SimulationStorageEngine::MEMORY,
	SimulationStorageEngine::RADIX_TREE, SimulationStorageEngine::REDWOOD,
#ifdef WITH_ROCKSDB
	SimulationStorageEngine::ROCKSDB,    SimulationStorageEngine::SHARDED_ROCKSDB,
#endif
};

std::string getExcludedStorageEngineTypesInString(const std::set<SimulationStorageEngine>& excluded) {
	std::string str;
	for (const auto& e : excluded) {
		str += std::to_string(static_cast<uint32_t>(e));
		str += ',';
	}
	if (!excluded.empty())
		str.pop_back();
	return str;
}

SimulationStorageEngine chooseSimulationStorageEngine(const TestConfig& testConfig, const bool isEncryptionEnabled) {
	StringRef reason;
	SimulationStorageEngine result = SimulationStorageEngine::SIMULATION_STORAGE_ENGINE_INVALID_VALUE;

	if (isEncryptionEnabled) {
		// Only storage engine supporting encryption is Redwood.
		reason = "EncryptionEnabled"_sr;
		result = SimulationStorageEngine::REDWOOD;

	} else if (testConfig.storageEngineType.present()) {
		reason = "ConfigureSpecified"_sr;
		result = testConfig.storageEngineType.get();
		if (testConfig.excludedStorageEngineType(result) ||
		    std::find(std::begin(SIMULATION_STORAGE_ENGINE), std::end(SIMULATION_STORAGE_ENGINE), result) ==
		        std::end(SIMULATION_STORAGE_ENGINE)) {

			TraceEvent(SevError, "StorageEngineNotSupported").detail("StorageEngineType", result);
			ASSERT(false);
		}

	} else if (SERVER_KNOBS->ENFORCE_SHARDED_ROCKSDB_SIM_IF_AVALIABLE &&
	           testConfig.storageEngineExcludeTypes.find(SimulationStorageEngine::SHARDED_ROCKSDB) ==
	               testConfig.storageEngineExcludeTypes.end()) {
		reason = "ENFORCE_SHARDED_ROCKSDB_SIM_IF_AVALIABLE is enabled"_sr;
		result = SimulationStorageEngine::SHARDED_ROCKSDB;

	} else {
		std::unordered_set<SimulationStorageEngine> storageEngineAvailable;
		for (const auto& storageEngine : SIMULATION_STORAGE_ENGINE) {
			storageEngineAvailable.insert(storageEngine);
		}
		for (const auto& storageEngineExcluded : testConfig.storageEngineExcludeTypes) {
			storageEngineAvailable.erase(storageEngineExcluded);
		}
		ASSERT(storageEngineAvailable.size() > 0);
		std::vector<SimulationStorageEngine> storageEngineCandidates;
		for (const auto& storageEngine : storageEngineAvailable) {
			if (storageEngine == SimulationStorageEngine::MEMORY) {
				// Adjust the chance that Memory is selected
				storageEngineCandidates.insert(
				    storageEngineCandidates.end(), SERVER_KNOBS->PROBABILITY_FACTOR_MEMORY_SELECTED_SIM, storageEngine);
			} else if (storageEngine == SimulationStorageEngine::SHARDED_ROCKSDB) {
				// Adjust the chance that ShardedRocksDB is selected
				storageEngineCandidates.insert(storageEngineCandidates.end(),
				                               SERVER_KNOBS->PROBABILITY_FACTOR_SHARDED_ROCKSDB_ENGINE_SELECTED_SIM,
				                               storageEngine);
			} else if (storageEngine == SimulationStorageEngine::SSD) {
				// Adjust the chance that SQLite is selected
				storageEngineCandidates.insert(storageEngineCandidates.end(),
				                               SERVER_KNOBS->PROBABILITY_FACTOR_SQLITE_ENGINE_SELECTED_SIM,
				                               storageEngine);
			} else if (storageEngine == SimulationStorageEngine::ROCKSDB) {
				// Adjust the chance that RocksDB is selected
				storageEngineCandidates.insert(storageEngineCandidates.end(),
				                               SERVER_KNOBS->PROBABILITY_FACTOR_ROCKSDB_ENGINE_SELECTED_SIM,
				                               storageEngine);
			} else {
				storageEngineCandidates.push_back(storageEngine);
			}
		}
		reason = "RandomlyChosen"_sr;
		result = deterministicRandom()->randomChoice(storageEngineCandidates);
		if (result == SimulationStorageEngine::SIMULATION_STORAGE_ENGINE_INVALID_VALUE) {
			UNREACHABLE();
		}
	}

	TraceEvent(SevInfo, "SimulationStorageEngine")
	    .detail("StorageEngine", static_cast<uint8_t>(result))
	    .detail("Reason", reason)
	    .detail("Excluded", getExcludedStorageEngineTypesInString(testConfig.storageEngineExcludeTypes))
	    .detail("RocksDBEngineChoosable", hasRocksDB);

	return result;
}

} // anonymous namespace

// Sets storage engine based on testConfig details
void SimulationConfig::setStorageEngine(const TestConfig& testConfig) {
	auto storageEngineType = chooseSimulationStorageEngine(testConfig, db.encryptionAtRestMode.isEncryptionEnabled());
	STORAGE_ENGINE_CONFIG_MAPPER.at(storageEngineType)(this);
}

// Sets replication type and TLogSpillType and Version
void SimulationConfig::setReplicationType(const TestConfig& testConfig) {
	replication_type = testConfig.simpleConfig
	                       ? 1
	                       : (std::max(testConfig.minimumReplication,
	                                   datacenters > 4 ? deterministicRandom()->randomInt(1, 3)
	                                                   : std::min(deterministicRandom()->randomInt(0, 6), 3)));
	if (testConfig.config.present()) {
		set_config(testConfig.config.get());
	} else {
		switch (replication_type) {
		case 0: {
			CODE_PROBE(true, "Simulated cluster using custom redundancy mode");
			int storage_servers = deterministicRandom()->randomInt(1, generateFearless ? 4 : 5);
			// FIXME: log replicas must be more than storage replicas because otherwise better master exists will not
			// recognize it needs to change dcs
			int replication_factor = deterministicRandom()->randomInt(storage_servers, generateFearless ? 4 : 5);
			// Version vector is an experimental feature that does not support logWriteAntiQuorum
			// feature. Disable logWriteAntiQuorum (when version vector is enabled) in simulation
			// tests for now.
			// @todo extend version vector to support logWriteAntiQuorum feature.
			int anti_quorum =
			    SERVER_KNOBS->ENABLE_VERSION_VECTOR
			        ? 0
			        : (deterministicRandom()->randomInt(
			              0,
			              (replication_factor / 2) +
			                  1)); // The anti quorum cannot be more than half of the replication factor, or the
			                       // log system will continue to accept commits when a recovery is impossible
			// Go through buildConfiguration, as it sets tLogPolicy/storagePolicy.
			set_config(format("storage_replicas:=%d log_replicas:=%d log_anti_quorum:=%d "
			                  "replica_datacenters:=1 min_replica_datacenters:=1",
			                  storage_servers,
			                  replication_factor,
			                  anti_quorum));
			break;
		}
		case 1: {
			CODE_PROBE(true, "Simulated cluster running in single redundancy mode");
			set_config("single");
			break;
		}
		case 2: {
			CODE_PROBE(true, "Simulated cluster running in double redundancy mode");
			set_config("double");
			break;
		}
		case 3: {
			if (datacenters <= 2 || generateFearless) {
				CODE_PROBE(true, "Simulated cluster running in triple redundancy mode");
				set_config("triple");
			} else if (datacenters == 3) {
				CODE_PROBE(true, "Simulated cluster running in 3 data-hall mode");
				set_config("three_data_hall");
			} else {
				ASSERT(false);
			}
			break;
		}
		default:
			ASSERT(false); // Programmer forgot to adjust cases.
		}
		if (deterministicRandom()->random01() < 0.5) {
			int logSpill = deterministicRandom()->randomInt(TLogSpillType::VALUE, TLogSpillType::END);
			set_config(format("log_spill:=%d", logSpill));
			int logVersion =
			    deterministicRandom()->randomInt(TLogVersion::MIN_RECRUITABLE, testConfig.maxTLogVersion + 1);
			set_config(format("log_version:=%d", logVersion));
		} else {
			if (deterministicRandom()->random01() < 0.7)
				set_config(format("log_version:=%d", testConfig.maxTLogVersion));
			if (deterministicRandom()->random01() < 0.5)
				set_config(format("log_spill:=%d", TLogSpillType::DEFAULT));
		}
	}
}

// Set the regions of the config, including the primary and remote options
// This will also determine the replication types used for satellite and remote.
void SimulationConfig::setRegions(const TestConfig& testConfig) {
	// The kill region workload relies on the fact that all "0", "2", and "4" are all of the possible primary dcids.
	StatusObject primaryObj;
	StatusObject primaryDcObj;
	primaryDcObj["id"] = "0";
	primaryDcObj["priority"] = 2;
	StatusArray primaryDcArr;
	primaryDcArr.push_back(primaryDcObj);

	StatusObject remoteObj;
	StatusObject remoteDcObj;
	remoteDcObj["id"] = "1";
	remoteDcObj["priority"] = 1;
	StatusArray remoteDcArr;
	remoteDcArr.push_back(remoteDcObj);

	bool needsRemote = generateFearless;
	if (generateFearless) {
		if (datacenters > 4) {
			// FIXME: we cannot use one satellite replication with more than one satellite per region because
			// canKillProcesses does not respect usable_dcs
			int satellite_replication_type = deterministicRandom()->randomInt(0, 3);
			if (SERVER_KNOBS->ENABLE_VERSION_VECTOR_TLOG_UNICAST && satellite_replication_type == 1) {
				// two_satellite_fast sets antiQuorum to 2, because only one out of two satellites need reply before
				// commit. disabling for version vector in simulation until quorum feature works with it.
				satellite_replication_type = 2;
			}
			switch (satellite_replication_type) {
			case 0: {
				CODE_PROBE(true, "Simulated cluster using no satellite redundancy mode (>4 datacenters)");
				break;
			}
			case 1: {
				CODE_PROBE(true, "Simulated cluster using two satellite fast redundancy mode");
				primaryObj["satellite_redundancy_mode"] = "two_satellite_fast";
				remoteObj["satellite_redundancy_mode"] = "two_satellite_fast";
				break;
			}
			case 2: {
				CODE_PROBE(true, "Simulated cluster using two satellite safe redundancy mode");
				primaryObj["satellite_redundancy_mode"] = "two_satellite_safe";
				remoteObj["satellite_redundancy_mode"] = "two_satellite_safe";
				break;
			}
			default:
				ASSERT(false); // Programmer forgot to adjust cases.
			}
		} else {
			int satellite_replication_type = deterministicRandom()->randomInt(0, 5);
			switch (satellite_replication_type) {
			case 0: {
				// FIXME: implement
				CODE_PROBE(true, "Simulated cluster using custom satellite redundancy mode");
				break;
			}
			case 1: {
				CODE_PROBE(true, "Simulated cluster using no satellite redundancy mode (<4 datacenters)");
				break;
			}
			case 2: {
				CODE_PROBE(true, "Simulated cluster using single satellite redundancy mode");
				primaryObj["satellite_redundancy_mode"] = "one_satellite_single";
				remoteObj["satellite_redundancy_mode"] = "one_satellite_single";
				break;
			}
			case 3: {
				CODE_PROBE(true, "Simulated cluster using double satellite redundancy mode");
				primaryObj["satellite_redundancy_mode"] = "one_satellite_double";
				remoteObj["satellite_redundancy_mode"] = "one_satellite_double";
				break;
			}
			case 4: {
				CODE_PROBE(true, "Simulated cluster using triple satellite redundancy mode");
				primaryObj["satellite_redundancy_mode"] = "one_satellite_triple";
				remoteObj["satellite_redundancy_mode"] = "one_satellite_triple";
				break;
			}
			default:
				ASSERT(false); // Programmer forgot to adjust cases.
			}
		}

		// Calculate the maximum satellite_logs we can support based on available machines
		bool useNormalDCsAsSatellites =
		    datacenters > 4 && testConfig.minimumRegions < 2 && deterministicRandom()->random01() < 0.3;
		int maxSatelliteLogs = getMaxSatelliteLogs();

		if (deterministicRandom()->random01() < 0.25)
			primaryObj["satellite_logs"] = deterministicRandom()->randomInt(1, maxSatelliteLogs + 1);
		if (deterministicRandom()->random01() < 0.25)
			remoteObj["satellite_logs"] = deterministicRandom()->randomInt(1, maxSatelliteLogs + 1);

		// We cannot run with a remote DC when MAX_READ_TRANSACTION_LIFE_VERSIONS is too small, because the log
		// routers will not be able to keep up.
		if (testConfig.minimumRegions <= 1 &&
		    (deterministicRandom()->random01() < 0.25 ||
		     SERVER_KNOBS->MAX_READ_TRANSACTION_LIFE_VERSIONS < SERVER_KNOBS->VERSIONS_PER_SECOND)) {
			CODE_PROBE(true, "Simulated cluster using one region");
			needsRemote = false;
		} else {
			CODE_PROBE(true, "Simulated cluster using two regions");
			db.usableRegions = 2;
		}

		if (testConfig.remoteConfig.present()) {
			set_config(testConfig.remoteConfig.get());
		} else {
			int remote_replication_type = deterministicRandom()->randomInt(0, datacenters > 4 ? 4 : 5);
			switch (remote_replication_type) {
			case 0: {
				// FIXME: implement
				CODE_PROBE(true, "Simulated cluster using custom remote redundancy mode");
				break;
			}
			case 1: {
				CODE_PROBE(true, "Simulated cluster using default remote redundancy mode");
				break;
			}
			case 2: {
				CODE_PROBE(true, "Simulated cluster using single remote redundancy mode");
				set_config("remote_single");
				break;
			}
			case 3: {
				CODE_PROBE(true, "Simulated cluster using double remote redundancy mode");
				set_config("remote_double");
				break;
			}
			case 4: {
				CODE_PROBE(true, "Simulated cluster using triple remote redundancy mode");
				set_config("remote_triple");
				break;
			}
			default:
				ASSERT(false); // Programmer forgot to adjust cases.
			}
		}

		if (deterministicRandom()->random01() < 0.25)
			db.desiredLogRouterCount = deterministicRandom()->randomInt(1, 7);

		if (testConfig.remoteDesiredTLogCount.present()) {
			db.remoteDesiredTLogCount = testConfig.remoteDesiredTLogCount.get();
		} else if (deterministicRandom()->random01() < 0.25) {
			db.remoteDesiredTLogCount = deterministicRandom()->randomInt(1, 7);
		}

		StatusObject primarySatelliteObj;
		primarySatelliteObj["id"] = useNormalDCsAsSatellites ? "1" : "2";
		primarySatelliteObj["priority"] = 1;
		primarySatelliteObj["satellite"] = 1;
		if (deterministicRandom()->random01() < 0.25)
			primarySatelliteObj["satellite_logs"] = deterministicRandom()->randomInt(1, maxSatelliteLogs + 1);
		primaryDcArr.push_back(primarySatelliteObj);

		StatusObject remoteSatelliteObj;
		remoteSatelliteObj["id"] = useNormalDCsAsSatellites ? "0" : "3";
		remoteSatelliteObj["priority"] = 1;
		remoteSatelliteObj["satellite"] = 1;
		if (deterministicRandom()->random01() < 0.25)
			remoteSatelliteObj["satellite_logs"] = deterministicRandom()->randomInt(1, maxSatelliteLogs + 1);
		remoteDcArr.push_back(remoteSatelliteObj);

		if (datacenters > 4) {
			StatusObject primarySatelliteObjB;
			primarySatelliteObjB["id"] = useNormalDCsAsSatellites ? "2" : "4";
			primarySatelliteObjB["priority"] = 1;
			primarySatelliteObjB["satellite"] = 1;
			if (deterministicRandom()->random01() < 0.25)
				primarySatelliteObjB["satellite_logs"] = deterministicRandom()->randomInt(1, maxSatelliteLogs + 1);
			primaryDcArr.push_back(primarySatelliteObjB);

			StatusObject remoteSatelliteObjB;
			remoteSatelliteObjB["id"] = useNormalDCsAsSatellites ? "2" : "5";
			remoteSatelliteObjB["priority"] = 1;
			remoteSatelliteObjB["satellite"] = 1;
			if (deterministicRandom()->random01() < 0.25)
				remoteSatelliteObjB["satellite_logs"] = deterministicRandom()->randomInt(1, maxSatelliteLogs + 1);
			remoteDcArr.push_back(remoteSatelliteObjB);
		}
		if (useNormalDCsAsSatellites) {
			datacenters = 3;
		}
	}

	primaryObj["datacenters"] = primaryDcArr;
	remoteObj["datacenters"] = remoteDcArr;

	StatusArray regionArr;
	regionArr.push_back(primaryObj);
	if (needsRemote || deterministicRandom()->random01() < 0.5) {
		regionArr.push_back(remoteObj);
	}

	if (needsRemote) {
		g_simulator->originalRegions =
		    "regions=" + json_spirit::write_string(json_spirit::mValue(regionArr), json_spirit::Output_options::none);

		StatusArray disablePrimary = regionArr;
		disablePrimary[0].get_obj()["datacenters"].get_array()[0].get_obj()["priority"] = -1;
		g_simulator->disablePrimary = "regions=" + json_spirit::write_string(json_spirit::mValue(disablePrimary),
		                                                                     json_spirit::Output_options::none);

		StatusArray disableRemote = regionArr;
		disableRemote[1].get_obj()["datacenters"].get_array()[0].get_obj()["priority"] = -1;
		g_simulator->disableRemote = "regions=" + json_spirit::write_string(json_spirit::mValue(disableRemote),
		                                                                    json_spirit::Output_options::none);
	} else {
		// In order to generate a starting configuration with the remote disabled, do not apply the region
		// configuration to the DatabaseConfiguration until after creating the starting conf string.
		set_config("regions=" +
		           json_spirit::write_string(json_spirit::mValue(regionArr), json_spirit::Output_options::none));
	}
}

// Sets the machine count based on the testConfig. May be overwritten later
// if the end result is not a viable config.
void SimulationConfig::setMachineCount(const TestConfig& testConfig) {
	if (testConfig.machineCount.present()) {
		machine_count = testConfig.machineCount.get();
#ifdef ADDRESS_SANITIZER
		if (testConfig.asanMachineCount.present()) {
			machine_count = testConfig.asanMachineCount.get();
		}
#endif
	} else if (generateFearless && testConfig.minimumReplication > 1) {
		// low latency tests in fearless configurations need 4 machines per datacenter (3 for triple replication, 1 that
		// is down during failures).
		machine_count = 16;
	} else if (generateFearless) {
		machine_count = 12;
	} else if (db.tLogPolicy && db.tLogPolicy->info() == "data_hall^2 x zoneid^2 x 1") {
		machine_count = 9;
	} else {
		// datacenters+2 so that the configure database workload can configure into three_data_hall
		machine_count = std::max(datacenters + 2,
		                         ((db.minDatacentersRequired() > 0) ? datacenters : 1) *
		                             std::max(3, db.minZonesRequiredPerDatacenter()));
		machine_count = deterministicRandom()->randomInt(
		    machine_count,
		    std::max(machine_count + 1, extraDatabaseMode == ISimulator::ExtraDatabaseMode::Disabled ? 10 : 6));
		// generateMachineTeamTestConfig set up the number of servers per machine and the number of machines such that
		// if we do not remove the surplus server and machine teams, the simulation test will report error.
		// This is needed to make sure the number of server (and machine) teams is no larger than the desired number.
		bool generateMachineTeamTestConfig = BUGGIFY_WITH_PROB(0.1) ? true : false;
		if (generateMachineTeamTestConfig) {
			// When DESIRED_TEAMS_PER_SERVER is set to 1, the desired machine team number is 5
			// while the max possible machine team number is 10.
			// If machine_count > 5, we can still test the effectivenss of machine teams
			// Note: machine_count may be much larger than 5 because we may have a big replication factor
			machine_count = std::max(machine_count,
			                         deterministicRandom()->randomInt(
			                             5, extraDatabaseMode == ISimulator::ExtraDatabaseMode::Disabled ? 10 : 6));
		}
	}
	machine_count += datacenters * (testConfig.extraMachineCountDC + testConfig.extraStorageMachineCountPerDC);
}

// Sets the coordinator count based on the testConfig. May be overwritten later
// if the end result is not a viable config.
void SimulationConfig::setCoordinators(const TestConfig& testConfig) {
	if (testConfig.coordinators.present()) {
		coordinators = testConfig.coordinators.get();
	} else {
		// because we protect a majority of coordinators from being killed, it is better to run with low numbers of
		// coordinators to prevent too many processes from being protected
		coordinators = (testConfig.minimumRegions <= 1 && BUGGIFY)
		                   ? deterministicRandom()->randomInt(1, std::max(machine_count, 2))
		                   : 1;
	}
}

// Sets the processes per machine based on the testConfig.
void SimulationConfig::setProcessesPerMachine(const TestConfig& testConfig) {
	if (testConfig.processesPerMachine.present()) {
		processes_per_machine = testConfig.processesPerMachine.get();
	} else if (generateFearless) {
		processes_per_machine = 1;
	} else {
		processes_per_machine = deterministicRandom()->randomInt(
		    1, (extraDatabaseMode == ISimulator::ExtraDatabaseMode::Disabled ? 28 : 14) / machine_count + 2);
	}
}

// Sets the TSS configuration based on the testConfig.
// Also configures the cluster behaviour through setting some flags on the simulator.
void SimulationConfig::setTss(const TestConfig& testConfig) {
	int tssCount = 0;
	// TODO: Support TSS in SHARD_ENCODE_LOCATION_METADATA mode.
	if (!testConfig.simpleConfig && !testConfig.disableTss && deterministicRandom()->random01() < 0.25) {
		// 1 or 2 tss
		tssCount = deterministicRandom()->randomInt(1, 3);
	}

	// reduce tss to half of extra non-seed servers that can be recruited in usable regions.
	tssCount =
	    std::max(0, std::min(tssCount, db.usableRegions * ((machine_count / datacenters) - db.storageTeamSize) / 2));

	if (!testConfig.config.present() && tssCount > 0) {
		std::string confStr = format("tss_count:=%d tss_storage_engine:=%d", tssCount, db.storageServerStoreType);
		set_config(confStr);
		double tssRandom = deterministicRandom()->random01();
		if (tssRandom > 0.5 || !faultInjectionActivated) {
			// normal tss mode
			g_simulator->tssMode = ISimulator::TSSMode::EnabledNormal;
		} else if (tssRandom < 0.25 && !testConfig.isFirstTestInRestart) {
			// fault injection - don't enable in first test in restart because second test won't know it intentionally
			// lost data
			g_simulator->tssMode = ISimulator::TSSMode::EnabledDropMutations;
		} else {
			// delay injection
			g_simulator->tssMode = ISimulator::TSSMode::EnabledAddDelay;
		}
		printf("enabling tss for simulation in mode %d: %s\n", g_simulator->tssMode, confStr.c_str());
	}
}

void setConfigDB(TestConfig const& testConfig) {
	if (g_simulator) {
		g_simulator->configDBType = testConfig.getConfigDBType();
	}
}

// Generates and sets an appropriate configuration for the database according to
// the provided testConfig. Some attributes are randomly generated for more coverage
// of different combinations
void SimulationConfig::generateNormalConfig(const TestConfig& testConfig) {
	set_config("new");
	// Some of these options will overwrite one another so the ordering is important.
	// This is a bit inefficient but separates the different types of option setting paths for better readability.
	setDatacenters(testConfig);

	// These 3 sets will only change the settings with trivial logic and low coupling with
	// other portions of the configuration. The parameters that are more involved and use
	// complex logic will be found in their respective "set----" methods following after.
	setRandomConfig();
	if (testConfig.simpleConfig) {
		setSimpleConfig();
	}
	setSpecificConfig(testConfig);
	setEncryptionAtRestMode(testConfig);
	setStorageEngine(testConfig);
	setReplicationType(testConfig);
#if (!NO_MULTIREGION_TEST)
	if (!testConfig.singleRegion &&
	    (generateFearless || (datacenters == 2 && deterministicRandom()->random01() < 0.5))) {
		setRegions(testConfig);
	}
#endif
	setMachineCount(testConfig);
	setCoordinators(testConfig);

	if (testConfig.minimumReplication > 1 && datacenters == 3) {
		// low latency tests in 3 data hall mode need 2 other data centers with 2 machines each to avoid waiting for
		// logs to recover.
		machine_count = std::max(machine_count, 6);
		coordinators = 3;
	}

	setProcessesPerMachine(testConfig);
	setTss(testConfig);
	setConfigDB(testConfig);
}

// Configures the system according to the given specifications in order to run
// simulation under the correct conditions
void setupSimulatedSystem(std::vector<Future<Void>>* systemActors,
                          std::string baseFolder,
                          int* pTesterCount,
                          Optional<ClusterConnectionString>* pConnString,
                          Standalone<StringRef>* pStartingConfiguration,
                          std::string whitelistBinPaths,
                          TestConfig testConfig,
                          ProtocolVersion protocolVersion) {
	auto& g_knobs = IKnobCollection::getMutableGlobalKnobCollection();
	// SOMEDAY: this does not test multi-interface configurations
	SimulationConfig simconfig(testConfig);

	if (testConfig.testClass == MOCK_DD_TEST_CLASS) {
		MockGlobalState::g_mockState()->initializeClusterLayout(simconfig);
	}

	if (testConfig.logAntiQuorum != -1) {
		simconfig.db.tLogWriteAntiQuorum = testConfig.logAntiQuorum;
	}

	StatusObject startingConfigJSON = simconfig.db.toJSON(true);
	std::string startingConfigString = "new";
	if (testConfig.configureLocked) {
		startingConfigString += " locked";
	}
	if (testConfig.disableRemoteKVS) {
		g_knobs.setKnob("remote_kv_store", KnobValueRef::create(bool{ false }));
		TraceEvent(SevDebug, "DisableRemoteKVS");
	}
	auto configDBType = testConfig.getConfigDBType();
	startingConfigString += DatabaseConfiguration::configureStringFromJSON(startingConfigJSON);

	// Set a random locality for the perpetual wiggle if any locality is set in the config
	// This behavior is preserved in a refactor but it unclear why the source configuration JSON
	// is translated to a random locality instead of something matching the source.
	auto ipwLocality = startingConfigJSON.find("perpetual_storage_wiggle_locality");
	if (ipwLocality != startingConfigJSON.end()) {
		if (deterministicRandom()->random01() < 0.25) {
			int dcId = deterministicRandom()->randomInt(0, simconfig.datacenters);
			startingConfigString += " " + ipwLocality->first + "=" + "data_hall:" + std::to_string(dcId);
		}
	}

	// handle tss_storage_engine separately because the passthrough needs the enum ordinal, but it's serialized to json
	// as the string name
	if (simconfig.db.desiredTSSCount > 0) {
		startingConfigString += format(" tss_storage_engine:=%d", simconfig.db.testingStorageServerStoreType);
	}

	if (g_simulator->originalRegions != "") {
		simconfig.set_config(g_simulator->originalRegions);
		g_simulator->startingDisabledConfiguration = startingConfigString + " " + g_simulator->disableRemote;
		startingConfigString += " " + g_simulator->originalRegions;
	}

	TraceEvent("SimulatorConfig").setMaxFieldLength(10000).detail("ConfigString", StringRef(startingConfigString));

	const int dataCenters = simconfig.datacenters;
	const int machineCount = simconfig.machine_count;
	const int coordinatorCount = simconfig.coordinators;
	const int processesPerMachine = simconfig.processes_per_machine;

	// half the time, when we have more than 4 machines that are not the first in their dataCenter, assign classes
	bool assignClasses = machineCount - dataCenters > 4 && deterministicRandom()->random01() < 0.5;

	// Use SSL 5% of the time
	bool sslEnabled = deterministicRandom()->random01() < 0.10;
	bool sslOnly = sslEnabled && deterministicRandom()->coinflip();
	bool isTLS = sslEnabled && sslOnly;
	g_simulator->listenersPerProcess = sslEnabled && !sslOnly ? 2 : 1;
	CODE_PROBE(sslEnabled, "SSL enabled");
	CODE_PROBE(!sslEnabled, "SSL disabled");

	// Use IPv6 25% of the time
	bool useIPv6 = deterministicRandom()->random01() < 0.25;
	CODE_PROBE(useIPv6, "Use IPv6");
	CODE_PROBE(!useIPv6, "Use IPv4");

	// Use hostname 25% of the time, unless it is disabled
	bool useHostname = !testConfig.disableHostname && deterministicRandom()->random01() < 0.25;
	CODE_PROBE(useHostname, "Use hostname");
	CODE_PROBE(!useHostname, "Use IP address");
	NetworkAddressFromHostname fromHostname =
	    useHostname ? NetworkAddressFromHostname::True : NetworkAddressFromHostname::False;

	int extraDatabaseCount = 0;
	bool useLocalDatabase = (testConfig.extraDatabaseMode == ISimulator::ExtraDatabaseMode::LocalOrSingle && BUGGIFY) ||
	                        testConfig.extraDatabaseMode == ISimulator::ExtraDatabaseMode::Local;
	if (!useLocalDatabase && testConfig.extraDatabaseMode != ISimulator::ExtraDatabaseMode::Disabled) {
		extraDatabaseCount =
		    testConfig.extraDatabaseMode == ISimulator::ExtraDatabaseMode::Multiple && testConfig.extraDatabaseCount > 0
		        ? testConfig.extraDatabaseCount
		        : 1;
	}

	std::vector<NetworkAddress> coordinatorAddresses;
	std::vector<Hostname> coordinatorHostnames;

	// A list of coordinators for each extra database being created. The Nth vector in the outer vector
	// contains the coordinators for the Nth extra database.
	std::vector<std::vector<NetworkAddress>> extraCoordinatorAddresses(extraDatabaseCount);
	std::vector<std::vector<Hostname>> extraCoordinatorHostnames(extraDatabaseCount);

	if (testConfig.minimumRegions > 1) {
		// do not put coordinators in the primary region so that we can kill that region safely
		int nonPrimaryDcs = dataCenters / 2;
		for (int dc = 1; dc < dataCenters; dc += 2) {
			int dcCoordinators = coordinatorCount / nonPrimaryDcs + ((dc - 1) / 2 < coordinatorCount % nonPrimaryDcs);
			for (int m = 0; m < dcCoordinators; m++) {
				auto ip = makeIPAddressForSim(useIPv6, { 2, dc, 1, m });
				uint16_t port = sslEnabled && !sslOnly ? 2 : 1;
				NetworkAddress coordinator(ip, port, true, isTLS, fromHostname);
				coordinatorAddresses.push_back(coordinator);

				if (useHostname) {
					std::string hostname = "fakeCoordinatorDC" + std::to_string(dc) + "M" + std::to_string(m);
					Hostname coordinatorHostname(hostname, std::to_string(port), isTLS);
					coordinatorHostnames.push_back(coordinatorHostname);
					INetworkConnections::net()->addMockTCPEndpoint(hostname, std::to_string(port), { coordinator });
				}

				for (int edb = 0; edb < extraDatabaseCount; ++edb) {
					auto extraIp = makeIPAddressForSim(useIPv6, { 4 + edb, dc, 1, m });
					NetworkAddress extraCoordinator(extraIp, port, true, isTLS, fromHostname);
					extraCoordinatorAddresses[edb].push_back(extraCoordinator);

					if (useHostname) {
						std::string hostname = "fakeExtraCoordinatorDC" + std::to_string(dc) + "M" + std::to_string(m) +
						                       "C" + std::to_string(edb);
						Hostname extraCoordinatorHostname(hostname, std::to_string(port), isTLS);
						extraCoordinatorHostnames[edb].push_back(extraCoordinatorHostname);
						INetworkConnections::net()->addMockTCPEndpoint(
						    hostname, std::to_string(port), { extraCoordinator });
					}
				}

				TraceEvent("SelectedCoordinator")
				    .detail("Hostname", useHostname ? coordinatorHostnames.back().toString().c_str() : "N/A")
				    .detail("Address", coordinatorAddresses.back());
			}
		}
	} else {
		int assignedMachines = 0;
		int coordCount = coordinatorCount;
		if (coordinatorCount > 4) {
			++coordCount;
		}
		for (int dc = 0; dc < dataCenters; dc++) {
			int dcCoordinators = coordCount / dataCenters + (dc < coordCount % dataCenters);
			int machines = machineCount / dataCenters + (dc < machineCount % dataCenters);
			for (int m = 0; m < dcCoordinators; m++) {
				if (coordinatorCount > 4 &&
				    (assignedMachines == 4 || (m + 1 == dcCoordinators && assignedMachines < 4 &&
				                               assignedMachines + machines - dcCoordinators >= 4))) {
					auto ip = makeIPAddressForSim(useIPv6, { 2, dc, 1, m });
					TraceEvent("SkippedCoordinator")
					    .detail("Address", ip.toString())
					    .detail("M", m)
					    .detail("Machines", machines)
					    .detail("Assigned", assignedMachines)
					    .detail("DcCoord", dcCoordinators)
					    .detail("CoordinatorCount", coordinatorCount);
				} else {
					auto ip = makeIPAddressForSim(useIPv6, { 2, dc, 1, m });
					uint16_t port = sslEnabled && !sslOnly ? 2 : 1;
					NetworkAddress coordinator(ip, port, true, isTLS, fromHostname);
					coordinatorAddresses.push_back(coordinator);

					if (useHostname) {
						std::string hostname = "fakeCoordinatorDC" + std::to_string(dc) + "M" + std::to_string(m);
						Hostname coordinatorHostname(hostname, std::to_string(port), isTLS);
						coordinatorHostnames.push_back(coordinatorHostname);
						INetworkConnections::net()->addMockTCPEndpoint(hostname, std::to_string(port), { coordinator });
					}

					for (int edb = 0; edb < extraDatabaseCount; ++edb) {
						auto extraIp = makeIPAddressForSim(useIPv6, { 4 + edb, dc, 1, m });
						NetworkAddress extraCoordinator(extraIp, port, true, isTLS, fromHostname);
						extraCoordinatorAddresses[edb].push_back(extraCoordinator);
						std::string hostname = "fakeExtraCoordinatorDC" + std::to_string(dc) + "M" + std::to_string(m) +
						                       "C" + std::to_string(edb);
						Hostname extraCoordinatorHostname(hostname, std::to_string(port), isTLS);
						extraCoordinatorHostnames[edb].push_back(extraCoordinatorHostname);
						INetworkConnections::net()->addMockTCPEndpoint(
						    hostname, std::to_string(port), { extraCoordinator });
					}

					TraceEvent("SelectedCoordinator")
					    .detail("Hostname", useHostname ? coordinatorHostnames.back().toString().c_str() : "N/A")
					    .detail("Address", coordinatorAddresses.back())
					    .detail("M", m)
					    .detail("Machines", machines)
					    .detail("Assigned", assignedMachines)
					    .detail("DcCoord", dcCoordinators)
					    .detail("P1", (m + 1 == dcCoordinators))
					    .detail("P2", (assignedMachines < 4))
					    .detail("P3", (assignedMachines + machines - dcCoordinators >= 4))
					    .detail("CoordinatorCount", coordinatorCount);
				}
				assignedMachines++;
			}
			assignedMachines += machines - dcCoordinators;
		}
	}

	ASSERT(coordinatorAddresses.size() > 0);

	// Mark a random majority of the coordinators as protected, so
	// we won't accidentally kill off a quorum and render the
	// cluster unrecoverable.
	deterministicRandom()->randomShuffle(coordinatorAddresses);
	for (int i = 0; i < (coordinatorAddresses.size() / 2) + 1; i++) {
		TraceEvent("ProtectCoordinator")
		    .detail("Address", coordinatorAddresses[i])
		    .detail("Coordinators", describe(coordinatorAddresses));
		g_simulator->protectedAddresses.insert(NetworkAddress(
		    coordinatorAddresses[i].ip, coordinatorAddresses[i].port, true, coordinatorAddresses[i].isTLS()));
		if (coordinatorAddresses[i].port == 2) {
			g_simulator->protectedAddresses.insert(NetworkAddress(coordinatorAddresses[i].ip, 1, true, true));
		}
	}
	deterministicRandom()->randomShuffle(coordinatorAddresses);

	for (const auto& coordinators : extraCoordinatorAddresses) {
		for (int i = 0; i < (coordinators.size() / 2) + 1; i++) {
			TraceEvent("ProtectCoordinator")
			    .detail("Address", coordinators[i])
			    .detail("Coordinators", describe(coordinators));
			g_simulator->protectedAddresses.insert(
			    NetworkAddress(coordinators[i].ip, coordinators[i].port, true, coordinators[i].isTLS()));
			if (coordinators[i].port == 2) {
				g_simulator->protectedAddresses.insert(NetworkAddress(coordinators[i].ip, 1, true, true));
			}
		}
	}

	ASSERT_EQ(coordinatorAddresses.size(), coordinatorCount);
	ClusterConnectionString conn(coordinatorAddresses, "TestCluster:0"_sr);
	if (useHostname) {
		conn = ClusterConnectionString(coordinatorHostnames, "TestCluster:0"_sr);
	}

	if (useLocalDatabase) {
		g_simulator->extraDatabases.push_back(
		    useHostname ? ClusterConnectionString(coordinatorHostnames, "TestCluster:0"_sr).toString()
		                : ClusterConnectionString(coordinatorAddresses, "TestCluster:0"_sr).toString());
	} else if (testConfig.extraDatabaseMode != ISimulator::ExtraDatabaseMode::Disabled) {
		for (int i = 0; i < extraDatabaseCount; ++i) {
			g_simulator->extraDatabases.push_back(
			    useHostname
			        ? ClusterConnectionString(extraCoordinatorHostnames[i], StringRef(format("ExtraCluster%04d:0", i)))
			              .toString()
			        : ClusterConnectionString(extraCoordinatorAddresses[i], StringRef(format("ExtraCluster%04d:0", i)))
			              .toString());
		}
	}

	*pConnString = conn;

	TraceEvent("SimulatedConnectionString")
	    .detail("String", conn.toString())
	    .detail("ConfigString", startingConfigString);

	bool requiresExtraDBMachines = !g_simulator->extraDatabases.empty() && !useLocalDatabase;
	int assignedMachines = 0;
	bool gradualMigrationPossible = true;
	std::vector<ProcessClass::ClassType> processClassesSubSet = { ProcessClass::UnsetClass,
		                                                          ProcessClass::StatelessClass };
	for (int dc = 0; dc < dataCenters; dc++) {
		// FIXME: test unset dcID
		Optional<Standalone<StringRef>> dcUID = StringRef(format("%d", dc));
		std::vector<UID> machineIdentities;
		int machines = machineCount / dataCenters +
		               (dc < machineCount % dataCenters); // add remainder of machines to first datacenter
		int possible_ss = 0;
		int dcCoordinators = coordinatorCount / dataCenters + (dc < coordinatorCount % dataCenters);

		int simHTTPMachines = 0;
		if (testConfig.simHTTPServerEnabled) {
			simHTTPMachines = deterministicRandom()->randomInt(1, 4);
			fmt::print("sim http machines = {0}\n", simHTTPMachines);
		}
		int extraStorageMachineCount = testConfig.extraStorageMachineCountPerDC;

		int totalMachines = machines + simHTTPMachines + extraStorageMachineCount;

		printf("Datacenter %d: %d/%d machines, %d/%d coordinators, %d other machines\n",
		       dc,
		       machines,
		       machineCount,
		       dcCoordinators,
		       coordinatorCount,
		       totalMachines - machines);
		ASSERT_LE(dcCoordinators, machines);

		int useSeedForMachine = deterministicRandom()->randomInt(0, totalMachines);
		Standalone<StringRef> zoneId;
		Standalone<StringRef> newZoneId;

		Optional<int> desiredStatelessClasses;
		int actualStatelessClasses = 0;
		if (testConfig.statelessProcessClassesPerDC.present()) {
			desiredStatelessClasses = testConfig.statelessProcessClassesPerDC.get();
		}

		for (int machine = 0; machine < totalMachines; machine++) {
			Standalone<StringRef> machineId(deterministicRandom()->randomUniqueID().toString());
			if (machine == 0 || machineCount - dataCenters <= 4 || assignedMachines != 4 ||
			    simconfig.db.regions.size() || deterministicRandom()->random01() < 0.5) {
				zoneId = deterministicRandom()->randomUniqueID().toString();
				newZoneId = deterministicRandom()->randomUniqueID().toString();
			}

			// Choose a machine class
			ProcessClass processClass = ProcessClass(ProcessClass::UnsetClass, ProcessClass::CommandLineSource);
			if (assignClasses) {
				if (assignedMachines < 4)
					processClass = ProcessClass((ProcessClass::ClassType)deterministicRandom()->randomInt(0, 2),
					                            ProcessClass::CommandLineSource); // Unset or Storage
				else if (assignedMachines == 4 && !simconfig.db.regions.size())
					processClass = ProcessClass(
					    processClassesSubSet[deterministicRandom()->randomInt(0, processClassesSubSet.size())],
					    ProcessClass::CommandLineSource); // Unset or Stateless
				else
					processClass = ProcessClass((ProcessClass::ClassType)deterministicRandom()->randomInt(0, 3),
					                            ProcessClass::CommandLineSource); // Unset, Storage, or Transaction

				if (processClass == ProcessClass::UnsetClass || processClass == ProcessClass::StorageClass) {
					possible_ss++;
				}
			}

			if (desiredStatelessClasses.present() && actualStatelessClasses < desiredStatelessClasses.get()) {
				processClass = ProcessClass(ProcessClass::StatelessClass, ProcessClass::CommandLineSource);
				actualStatelessClasses++;
			}

			// FIXME: hack to add machines specifically to (some removed process types and) http server.
			// `machines` here is the normal (non-temporary) machines that totalMachines comprises of
			int processCount = processesPerMachine;
			ProcessMode processMode = requiresExtraDBMachines ? BackupAgentOnly : FDBDAndBackupAgent;
			if (machine >= machines) {
				if (extraStorageMachineCount > 0) {
					processClass = ProcessClass(ProcessClass::ClassType::StorageClass,
					                            ProcessClass::CommandLineSource); // Storage
					extraStorageMachineCount--;
					possible_ss++;
				} else if (simHTTPMachines > 0) {
					processClass = ProcessClass(ProcessClass::SimHTTPServerClass, ProcessClass::CommandLineSource);
					processCount = 1;
					processMode = SimHTTPServer;
					simHTTPMachines--;
				}
			}

			std::vector<IPAddress> ips;
			ips.reserve(processesPerMachine);
			for (int i = 0; i < processCount; i++) {
				ips.push_back(
				    makeIPAddressForSim(useIPv6, { 2, dc, deterministicRandom()->randomInt(1, i + 2), machine }));
			}
			if (requiresExtraDBMachines) {
				ips.push_back(makeIPAddressForSim(useIPv6, { 2, dc, 1, machine }));
			}

			// check the sslEnablementMap using only one ip
			LocalityData localities(Optional<Standalone<StringRef>>(), zoneId, machineId, dcUID);
			localities.set("data_hall"_sr, dcUID);
			systemActors->push_back(reportErrors(
			    simulatedMachine(conn,
			                     requiresExtraDBMachines ? ClusterConnectionString(g_simulator->extraDatabases.at(0))
			                                             : ClusterConnectionString(),
			                     ips,
			                     sslEnabled,
			                     localities,
			                     processClass,
			                     baseFolder,
			                     false,
			                     machine == useSeedForMachine,
			                     processMode,
			                     sslOnly,
			                     whitelistBinPaths,
			                     protocolVersion,
			                     configDBType,
			                     false),
			    "SimulatedMachine"));

			if (requiresExtraDBMachines) {
				int cluster = 4;
				for (auto extraDatabase : g_simulator->extraDatabases) {
					std::vector<IPAddress> extraIps;
					extraIps.reserve(processesPerMachine);
					for (int i = 0; i < processesPerMachine; i++) {
						extraIps.push_back(makeIPAddressForSim(
						    useIPv6, { cluster, dc, deterministicRandom()->randomInt(1, i + 2), machine }));
					}

					Standalone<StringRef> newMachineId(deterministicRandom()->randomUniqueID().toString());

					LocalityData localities(Optional<Standalone<StringRef>>(), newZoneId, newMachineId, dcUID);
					localities.set("data_hall"_sr, dcUID);
					systemActors->push_back(reportErrors(
					    simulatedMachine(ClusterConnectionString(extraDatabase),
					                     conn,
					                     extraIps,
					                     sslEnabled,
					                     localities,
					                     processClass,
					                     baseFolder,
					                     false,
					                     machine == useSeedForMachine,
					                     testConfig.extraDatabaseBackupAgents ? FDBDAndBackupAgent : FDBDOnly,
					                     sslOnly,
					                     whitelistBinPaths,
					                     protocolVersion,
					                     configDBType,
					                     true),
					    "SimulatedMachine"));
					++cluster;
				}
			}

			assignedMachines++;
		}

		if (desiredStatelessClasses.present()) {
			// If this assertion fails, that measn that there were not enough machines in the DC (primary or remote)
			// to match desired stateless classes
			ASSERT(actualStatelessClasses == desiredStatelessClasses.get());
		}

		if (possible_ss - simconfig.db.desiredTSSCount / simconfig.db.usableRegions <= simconfig.db.storageTeamSize) {
			gradualMigrationPossible = false;
		}

		TraceEvent("SimulatedClusterAssignMachineToDC")
		    .detail("DC", dc)
		    .detail("PossibleSS", possible_ss)
		    .detail("Machines", machines)
		    .detail("DcCoordinators", dcCoordinators);
	}

	g_simulator->desiredCoordinators = coordinatorCount;
	g_simulator->physicalDatacenters = dataCenters;
	g_simulator->processesPerMachine = processesPerMachine;

	TraceEvent("SetupSimulatorSettings")
	    .detail("DesiredCoordinators", g_simulator->desiredCoordinators)
	    .detail("PhysicalDatacenters", g_simulator->physicalDatacenters)
	    .detail("ProcessesPerMachine", g_simulator->processesPerMachine);

	// SOMEDAY: add locality for testers to simulate network topology
	// FIXME: Start workers with tester class instead, at least sometimes run tests with the testers-only flag
	int testerCount = *pTesterCount = deterministicRandom()->randomInt(4, 9);
	int useSeedForMachine = deterministicRandom()->randomInt(0, testerCount);
	for (int i = 0; i < testerCount; i++) {
		std::vector<IPAddress> ips;
		ips.push_back(makeIPAddressForSim(useIPv6, { 3, 4, 3, i + 1 }));
		Standalone<StringRef> newZoneId = Standalone<StringRef>(deterministicRandom()->randomUniqueID().toString());
		LocalityData localities(
		    Optional<Standalone<StringRef>>(), newZoneId, newZoneId, Optional<Standalone<StringRef>>());
		systemActors->push_back(
		    reportErrors(simulatedMachine(conn,
		                                  ClusterConnectionString(),
		                                  ips,
		                                  sslEnabled,
		                                  localities,
		                                  ProcessClass(ProcessClass::TesterClass, ProcessClass::CommandLineSource),
		                                  baseFolder,
		                                  false,
		                                  i == useSeedForMachine,
		                                  FDBDOnly,
		                                  sslOnly,
		                                  whitelistBinPaths,
		                                  protocolVersion,
		                                  configDBType,
		                                  false),
		                 "SimulatedTesterMachine"));
	}

	if (g_simulator->setDiffProtocol) {
		--(*pTesterCount);
	}

	*pStartingConfiguration = startingConfigString;

	// save some state that we only need when restarting the simulator.
	g_simulator->connectionString = conn.toString();
	g_simulator->testerCount = testerCount;
	g_simulator->allowStorageMigrationTypeChange = gradualMigrationPossible;
	g_simulator->willRestart = testConfig.isFirstTestInRestart;

	TraceEvent("SimulatedClusterStarted")
	    .detail("DataCenters", dataCenters)
	    .detail("ServerMachineCount", machineCount)
	    .detail("ProcessesPerServer", processesPerMachine)
	    .detail("SSLEnabled", sslEnabled)
	    .detail("SSLOnly", sslOnly)
	    .detail("ClassesAssigned", assignClasses)
	    .detail("GradualMigrationPossible", gradualMigrationPossible)
	    .detail("StartingConfiguration", pStartingConfiguration->toString());
}

using namespace std::literals;

// Populates the TestConfig fields according to what is found in the test file.
[[maybe_unused]] void checkTestConf(const char* testFile, TestConfig* testConfig) {}

} // namespace

															#line 6070 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via simulationSetupAndRun()
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
template <class SimulationSetupAndRunActor>
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class SimulationSetupAndRunActorState {
															#line 6077 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	SimulationSetupAndRunActorState(std::string const& dataFolder,const char* const& testFile,bool const& rebooting,bool const& restoring,std::string const& whitelistBinPaths) 
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		 : dataFolder(dataFolder),
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   testFile(testFile),
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   rebooting(rebooting),
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   restoring(restoring),
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   whitelistBinPaths(whitelistBinPaths),
															#line 2773 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   systemActors(),
															#line 2774 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   connectionString(),
															#line 2775 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   startingConfiguration(),
															#line 2776 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   testerCount(1),
															#line 2777 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   testConfig(),
															#line 2778 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		   allowList()
															#line 6104 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("simulationSetupAndRun", reinterpret_cast<unsigned long>(this));

	}
	~SimulationSetupAndRunActorState() 
	{
		fdb_probe_actor_destroy("simulationSetupAndRun", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2779 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			testConfig.readFromConfig(testFile);
															#line 2780 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			g_simulator->hasDiffProtocolProcess = testConfig.startIncompatibleProcess;
															#line 2781 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			g_simulator->setDiffProtocol = false;
															#line 2782 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (testConfig.injectTargetedSSRestart && deterministicRandom()->random01() < 0.25)
															#line 6125 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 2783 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_simulator->injectTargetedSSRestartTime = 60.0 + 340.0 * deterministicRandom()->random01();
															#line 6129 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 2786 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (testConfig.injectSSDelay && deterministicRandom()->random01() < 0.25)
															#line 6133 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 2787 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_simulator->injectSSDelayTime = 60.0 + 240.0 * deterministicRandom()->random01();
															#line 6137 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 2790 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (deterministicRandom()->random01() < 0.25)
															#line 6141 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 2791 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_simulator->injectTargetedBMRestartTime = 60.0 + 340.0 * deterministicRandom()->random01();
															#line 6145 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 2794 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (deterministicRandom()->random01() < 0.25)
															#line 6149 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 2795 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_simulator->injectTargetedBWRestartTime = 60.0 + 340.0 * deterministicRandom()->random01();
															#line 6153 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 2799 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			allowList.addTrustedSubnet("0.0.0.0/2"sv);
															#line 2800 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			allowList.addTrustedSubnet("abcd::/16"sv);
															#line 2802 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (!SERVER_KNOBS->SHARD_ENCODE_LOCATION_METADATA && (std::string_view(testFile).find("PhysicalShardMove") == std::string_view::npos && std::string_view(testFile).find("BulkLoading") == std::string_view::npos && std::string_view(testFile).find("BulkDumping") == std::string_view::npos && std::string_view(testFile).find("ShardedRocksNondeterministicTest") == std::string_view::npos))
															#line 6161 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 2810 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				testConfig.storageEngineExcludeTypes.insert(SimulationStorageEngine::SHARDED_ROCKSDB);
															#line 6165 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 2813 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			protocolVersion = currentProtocolVersion();
															#line 2814 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (testConfig.startIncompatibleProcess)
															#line 6171 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 2816 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				uint64_t minAddToMakeIncompatible = ProtocolVersion::compatibleProtocolVersionMask & ~(ProtocolVersion::compatibleProtocolVersionMask - 1);
															#line 2818 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				protocolVersion = ProtocolVersion(currentProtocolVersion().version() + minAddToMakeIncompatible);
															#line 6177 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
															#line 2822 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			auto testSystem = g_simulator->newProcess("TestSystem", IPAddress(0x01010101), 1, false, 1, LocalityData(Optional<Standalone<StringRef>>(), Standalone<StringRef>(deterministicRandom()->randomUniqueID().toString()), Standalone<StringRef>(deterministicRandom()->randomUniqueID().toString()), Optional<Standalone<StringRef>>()), ProcessClass(ProcessClass::TesterClass, ProcessClass::CommandLineSource), "", "", currentProtocolVersion(), false);
															#line 2837 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			testSystem->excludeFromRestarts = true;
															#line 2838 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			StrictFuture<Void> __when_expr_0 = g_simulator->onProcess(testSystem, TaskPriority::DefaultYield);
															#line 2838 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 6187 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 1;
															#line 2838 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SimulationSetupAndRunActor, 0, Void >*>(static_cast<SimulationSetupAndRunActor*>(this)));
															#line 6191 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		delete static_cast<SimulationSetupAndRunActor*>(this);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2839 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		Sim2FileSystem::newFileSystem();
															#line 2840 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		FlowTransport::createInstance(true, 1, WLTOKEN_RESERVED_COUNT, &allowList);
															#line 2841 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		CODE_PROBE(true, "Simulation start");
															#line 6217 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		try {
															#line 2845 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (rebooting)
															#line 6221 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 2846 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				StrictFuture<Void> __when_expr_1 = timeoutError(restartSimulatedSystem(&systemActors, dataFolder, &testerCount, &connectionString, &startingConfiguration, &testConfig, whitelistBinPaths, protocolVersion), 100.0);
															#line 2846 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
															#line 6227 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 2;
															#line 2846 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SimulationSetupAndRunActor, 1, Void >*>(static_cast<SimulationSetupAndRunActor*>(this)));
															#line 6231 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 2860 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_expect_full_pointermap = 1;
															#line 2861 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				setupSimulatedSystem(&systemActors, dataFolder, &testerCount, &connectionString, &startingConfiguration, whitelistBinPaths, testConfig, protocolVersion);
															#line 2869 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				StrictFuture<Void> __when_expr_2 = delay(1.0);
															#line 2869 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when2(__when_expr_2.get(), loopDepth); };
															#line 6244 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 3;
															#line 2869 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SimulationSetupAndRunActor, 2, Void >*>(static_cast<SimulationSetupAndRunActor*>(this)));
															#line 6248 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2839 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		Sim2FileSystem::newFileSystem();
															#line 2840 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		FlowTransport::createInstance(true, 1, WLTOKEN_RESERVED_COUNT, &allowList);
															#line 2841 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		CODE_PROBE(true, "Simulation start");
															#line 6268 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		try {
															#line 2845 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (rebooting)
															#line 6272 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 2846 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				StrictFuture<Void> __when_expr_1 = timeoutError(restartSimulatedSystem(&systemActors, dataFolder, &testerCount, &connectionString, &startingConfiguration, &testConfig, whitelistBinPaths, protocolVersion), 100.0);
															#line 2846 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1cont1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
															#line 6278 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 2;
															#line 2846 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SimulationSetupAndRunActor, 1, Void >*>(static_cast<SimulationSetupAndRunActor*>(this)));
															#line 6282 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 2860 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_expect_full_pointermap = 1;
															#line 2861 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				setupSimulatedSystem(&systemActors, dataFolder, &testerCount, &connectionString, &startingConfiguration, whitelistBinPaths, testConfig, protocolVersion);
															#line 2869 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				StrictFuture<Void> __when_expr_2 = delay(1.0);
															#line 2869 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1cont1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when2(__when_expr_2.get(), loopDepth); };
															#line 6295 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 3;
															#line 2869 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< SimulationSetupAndRunActor, 2, Void >*>(static_cast<SimulationSetupAndRunActor*>(this)));
															#line 6299 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state > 0) static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 0;
		static_cast<SimulationSetupAndRunActor*>(this)->ActorCallback< SimulationSetupAndRunActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SimulationSetupAndRunActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont8(int loopDepth) 
	{
															#line 2919 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("TracingMissingCodeProbes").log();
															#line 2920 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		probe::traceMissedProbes(probe::ExecutionContext::Simulation);
															#line 2921 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		TraceEvent("SimulatedSystemDestruct").log();
															#line 2922 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		g_simulator->stop();
															#line 2923 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		destructed = true;
															#line 2924 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_6 = Never();
															#line 2924 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont8when1(__when_expr_6.get(), loopDepth); };
															#line 6402 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 7;
															#line 2924 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< SimulationSetupAndRunActor, 6, Void >*>(static_cast<SimulationSetupAndRunActor*>(this)));
															#line 6406 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2912 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			auto timeoutVal = isGeneralBuggifyEnabled() ? testConfig.simulationBuggifyRunTestsTimeoutSeconds : testConfig.simulationNormalRunTestsTimeoutSeconds;
															#line 2914 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			auto msg = fmt::format("Timeout after {} simulated seconds", timeoutVal);
															#line 2915 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			ProcessEvents::trigger("Timeout"_sr, StringRef(msg), e);
															#line 2916 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent(SevError, "SetupAndRunError").error(e);
															#line 6422 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = a_body1cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9(int loopDepth) 
	{
															#line 2874 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_3 = HTTP::registerAlwaysFailHTTPHandler();
															#line 2874 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1cont1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont9when1(__when_expr_3.get(), loopDepth); };
															#line 6439 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 4;
															#line 2874 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< SimulationSetupAndRunActor, 3, Void >*>(static_cast<SimulationSetupAndRunActor*>(this)));
															#line 6443 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
															#line 2856 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (restoring)
															#line 6452 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 2857 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			startingConfiguration = "usable_regions=1"_sr;
															#line 6456 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
															#line 2856 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (restoring)
															#line 6466 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 2857 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			startingConfiguration = "usable_regions=1"_sr;
															#line 6470 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state > 0) static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 0;
		static_cast<SimulationSetupAndRunActor*>(this)->ActorCallback< SimulationSetupAndRunActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SimulationSetupAndRunActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state > 0) static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 0;
		static_cast<SimulationSetupAndRunActor*>(this)->ActorCallback< SimulationSetupAndRunActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< SimulationSetupAndRunActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont12(Void const& _,int loopDepth) 
	{
															#line 2876 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		std::string clusterFileDir = joinPath(dataFolder, deterministicRandom()->randomUniqueID().toString());
															#line 2877 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		platform::createDirectory(clusterFileDir);
															#line 2878 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		writeFile(joinPath(clusterFileDir, "fdb.cluster"), connectionString.get().toString());
															#line 2879 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		connFile = makeReference<ClusterConnectionFile>(joinPath(clusterFileDir, "fdb.cluster"));
															#line 2881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (rebooting)
															#line 6650 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 2883 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			StrictFuture<std::vector<NetworkAddress>> __when_expr_4 = connFile->getConnectionString().tryResolveHostnames();
															#line 2883 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont12when1(__when_expr_4.get(), loopDepth); };
															#line 6656 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 5;
															#line 2883 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< SimulationSetupAndRunActor, 4, std::vector<NetworkAddress> >*>(static_cast<SimulationSetupAndRunActor*>(this)));
															#line 6660 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont12cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont12(Void && _,int loopDepth) 
	{
															#line 2876 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		std::string clusterFileDir = joinPath(dataFolder, deterministicRandom()->randomUniqueID().toString());
															#line 2877 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		platform::createDirectory(clusterFileDir);
															#line 2878 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		writeFile(joinPath(clusterFileDir, "fdb.cluster"), connectionString.get().toString());
															#line 2879 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		connFile = makeReference<ClusterConnectionFile>(joinPath(clusterFileDir, "fdb.cluster"));
															#line 2881 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (rebooting)
															#line 6682 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		{
															#line 2883 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			StrictFuture<std::vector<NetworkAddress>> __when_expr_4 = connFile->getConnectionString().tryResolveHostnames();
															#line 2883 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont12when1(__when_expr_4.get(), loopDepth); };
															#line 6688 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 5;
															#line 2883 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< SimulationSetupAndRunActor, 4, std::vector<NetworkAddress> >*>(static_cast<SimulationSetupAndRunActor*>(this)));
															#line 6692 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont12cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state > 0) static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 0;
		static_cast<SimulationSetupAndRunActor*>(this)->ActorCallback< SimulationSetupAndRunActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< SimulationSetupAndRunActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont12cont1(int loopDepth) 
	{
															#line 2897 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		Future<Void> runTestsF = runTests(connFile, TEST_TYPE_FROM_FILE, TEST_ON_TESTERS, testerCount, testFile, startingConfiguration, LocalityData(), UnitTestParameters(), rebooting);
															#line 2906 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		StrictFuture<Void> __when_expr_5 = testConfig.longRunningTest ? runTestsF : timeoutError(runTestsF, isGeneralBuggifyEnabled() ? testConfig.simulationBuggifyRunTestsTimeoutSeconds : testConfig.simulationNormalRunTestsTimeoutSeconds);
															#line 2906 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont12cont1when1(__when_expr_5.get(), loopDepth); };
															#line 6785 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 6;
															#line 2906 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< SimulationSetupAndRunActor, 5, Void >*>(static_cast<SimulationSetupAndRunActor*>(this)));
															#line 6789 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont12cont2(std::vector<NetworkAddress> const& coordinatorAddresses,int loopDepth) 
	{
															#line 2885 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		ASSERT(coordinatorAddresses.size() > 0);
															#line 2886 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		for(int i = 0;i < (coordinatorAddresses.size() / 2) + 1;i++) {
															#line 2887 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("ProtectCoordinator") .detail("Address", coordinatorAddresses[i]) .detail("Coordinators", describe(coordinatorAddresses));
															#line 2890 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			g_simulator->protectedAddresses.insert(NetworkAddress( coordinatorAddresses[i].ip, coordinatorAddresses[i].port, true, coordinatorAddresses[i].isTLS()));
															#line 2892 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (coordinatorAddresses[i].port == 2)
															#line 6806 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 2893 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_simulator->protectedAddresses.insert(NetworkAddress(coordinatorAddresses[i].ip, 1, true, true));
															#line 6810 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1cont12cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont12cont2(std::vector<NetworkAddress> && coordinatorAddresses,int loopDepth) 
	{
															#line 2885 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		ASSERT(coordinatorAddresses.size() > 0);
															#line 2886 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		for(int i = 0;i < (coordinatorAddresses.size() / 2) + 1;i++) {
															#line 2887 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			TraceEvent("ProtectCoordinator") .detail("Address", coordinatorAddresses[i]) .detail("Coordinators", describe(coordinatorAddresses));
															#line 2890 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			g_simulator->protectedAddresses.insert(NetworkAddress( coordinatorAddresses[i].ip, coordinatorAddresses[i].port, true, coordinatorAddresses[i].isTLS()));
															#line 2892 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
			if (coordinatorAddresses[i].port == 2)
															#line 6829 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			{
															#line 2893 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
				g_simulator->protectedAddresses.insert(NetworkAddress(coordinatorAddresses[i].ip, 1, true, true));
															#line 6833 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1cont12cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont12when1(std::vector<NetworkAddress> const& coordinatorAddresses,int loopDepth) 
	{
		loopDepth = a_body1cont12cont2(coordinatorAddresses, loopDepth);

		return loopDepth;
	}
	int a_body1cont12when1(std::vector<NetworkAddress> && coordinatorAddresses,int loopDepth) 
	{
		loopDepth = a_body1cont12cont2(std::move(coordinatorAddresses), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state > 0) static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 0;
		static_cast<SimulationSetupAndRunActor*>(this)->ActorCallback< SimulationSetupAndRunActor, 4, std::vector<NetworkAddress> >::remove();

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 4, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> const& value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont12when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 4, std::vector<NetworkAddress> >*,std::vector<NetworkAddress> && value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont12when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< SimulationSetupAndRunActor, 4, std::vector<NetworkAddress> >*,Error err) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont12cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont12cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont12cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont12cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont12cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state > 0) static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 0;
		static_cast<SimulationSetupAndRunActor*>(this)->ActorCallback< SimulationSetupAndRunActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont12cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont12cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< SimulationSetupAndRunActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont12cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont8(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont13(Void const& _,int loopDepth) 
	{
															#line 2925 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		ASSERT(false);
															#line 7019 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = a_body1cont13cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont13(Void && _,int loopDepth) 
	{
															#line 2925 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		ASSERT(false);
															#line 7028 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		loopDepth = a_body1cont13cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont13(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont13(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state > 0) static_cast<SimulationSetupAndRunActor*>(this)->actor_wait_state = 0;
		static_cast<SimulationSetupAndRunActor*>(this)->ActorCallback< SimulationSetupAndRunActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< SimulationSetupAndRunActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< SimulationSetupAndRunActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont13cont2(int loopDepth) 
	{
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
		delete static_cast<SimulationSetupAndRunActor*>(this);
															#line 7112 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		return 0;

		return loopDepth;
	}
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::string dataFolder;
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	const char* testFile;
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	bool rebooting;
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	bool restoring;
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::string whitelistBinPaths;
															#line 2773 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	std::vector<Future<Void>> systemActors;
															#line 2774 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Optional<ClusterConnectionString> connectionString;
															#line 2775 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Standalone<StringRef> startingConfiguration;
															#line 2776 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	int testerCount;
															#line 2777 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	TestConfig testConfig;
															#line 2778 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	IPAllowList allowList;
															#line 2813 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	ProtocolVersion protocolVersion;
															#line 2879 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	Reference<ClusterConnectionFile> connFile;
															#line 7143 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
};
// This generated class is to be used only via simulationSetupAndRun()
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
class SimulationSetupAndRunActor final : public Actor<void>, public ActorCallback< SimulationSetupAndRunActor, 0, Void >, public ActorCallback< SimulationSetupAndRunActor, 1, Void >, public ActorCallback< SimulationSetupAndRunActor, 2, Void >, public ActorCallback< SimulationSetupAndRunActor, 3, Void >, public ActorCallback< SimulationSetupAndRunActor, 4, std::vector<NetworkAddress> >, public ActorCallback< SimulationSetupAndRunActor, 5, Void >, public ActorCallback< SimulationSetupAndRunActor, 6, Void >, public FastAllocated<SimulationSetupAndRunActor>, public SimulationSetupAndRunActorState<SimulationSetupAndRunActor> {
															#line 7148 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
public:
	using FastAllocated<SimulationSetupAndRunActor>::operator new;
	using FastAllocated<SimulationSetupAndRunActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14102792058997301760UL, 6164658073855621632UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() {{
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<void>*>(this)->~Actor();
        operator delete(this);
    }}
#pragma clang diagnostic pop
friend struct ActorCallback< SimulationSetupAndRunActor, 0, Void >;
friend struct ActorCallback< SimulationSetupAndRunActor, 1, Void >;
friend struct ActorCallback< SimulationSetupAndRunActor, 2, Void >;
friend struct ActorCallback< SimulationSetupAndRunActor, 3, Void >;
friend struct ActorCallback< SimulationSetupAndRunActor, 4, std::vector<NetworkAddress> >;
friend struct ActorCallback< SimulationSetupAndRunActor, 5, Void >;
friend struct ActorCallback< SimulationSetupAndRunActor, 6, Void >;
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	SimulationSetupAndRunActor(std::string const& dataFolder,const char* const& testFile,bool const& rebooting,bool const& restoring,std::string const& whitelistBinPaths) 
															#line 7171 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
		 : Actor<void>(),
		   SimulationSetupAndRunActorState<SimulationSetupAndRunActor>(dataFolder, testFile, rebooting, restoring, whitelistBinPaths),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10317229901811515136UL, 7715916063272469504UL);
		ActorExecutionContextHelper __helper(static_cast<SimulationSetupAndRunActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("simulationSetupAndRun");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("simulationSetupAndRun", reinterpret_cast<unsigned long>(this), -1);

	}
};
} // namespace
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
void simulationSetupAndRun( std::string const& dataFolder, const char* const& testFile, bool const& rebooting, bool const& restoring, std::string const& whitelistBinPaths ) {
															#line 2768 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"
	new SimulationSetupAndRunActor(dataFolder, testFile, rebooting, restoring, whitelistBinPaths);
															#line 7195 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/build_output/fdbserver/SimulatedCluster.actor.g.cpp.py_gen"
}

#line 2927 "/codebuild/output/src2236818296/src/github.com/apple/foundationdb/fdbserver/SimulatedCluster.actor.cpp"

// Helper function to calculate the maximum satellite_logs based on available machines per datacenter
// We count the minimum number of machines in any satellite datacenter to ensure we don't over-provision
int getMaxSatelliteLogs() {
	if (!g_network->isSimulated()) {
		return 6; // Conservative default for non-simulated environments
	}

	// Count machines per datacenter
	std::map<Optional<Standalone<StringRef>>, int> machinesPerDC;
	for (auto& process : g_simulator->getAllProcesses()) {
		if (process->locality.dcId().present()) {
			machinesPerDC[process->locality.dcId()]++;
		}
	}

	// Find the minimum machines in satellite DCs (0, 1, 2, 3, 4, 5).
	// Note normal DCs can be selected as satellites, see usage of useNormalDCsAsSatellites.
	int minSatelliteMachines = 6; // Start with max possible
	for (int dcId = 0; dcId <= 5; dcId++) {
		auto dcIdStr = Standalone<StringRef>(std::to_string(dcId));
		int count = machinesPerDC[dcIdStr];
		if (count > 0) {
			minSatelliteMachines = std::min(minSatelliteMachines, count);
		}
	}

	// Cap at 6 (the original max) and ensure at least 1
	return std::max(1, std::min(6, minSatelliteMachines));
}

BasicSimulationConfig generateBasicSimulationConfig(const BasicTestConfig& testConfig) {
	TestConfig config(testConfig);
	return SimulationConfig(config);
}
