#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
/*
 * KeyValueStoreRocksDB.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/FDBTypes.h"
#include "flow/IRandom.h"
#ifdef WITH_ROCKSDB

#include <rocksdb/c.h>
#include <rocksdb/cache.h>
#include <rocksdb/db.h>
#include <rocksdb/env.h>
#include <rocksdb/filter_policy.h>
#include <rocksdb/listener.h>
#include <rocksdb/metadata.h>
#include <rocksdb/options.h>
#include <rocksdb/perf_context.h>
#include <rocksdb/rate_limiter.h>
#include <rocksdb/slice.h>
#include <rocksdb/slice_transform.h>
#include <rocksdb/sst_file_reader.h>
#include <rocksdb/sst_file_writer.h>
#include <rocksdb/statistics.h>
#include <rocksdb/table.h>
#include <rocksdb/types.h>
#include <rocksdb/utilities/checkpoint.h>
#include <rocksdb/utilities/table_properties_collectors.h>
#include <rocksdb/version.h>

#if defined __has_include
#if __has_include(<liburing.h>)
#include <liburing.h>
#endif
#endif
#include "fdbclient/SystemData.h"
#include "fdbserver/CoroFlow.h"
#include "fdbserver/FDBRocksDBVersion.h"
#include "fdbserver/RocksDBLogForwarder.h"
#include "flow/ActorCollection.h"
#include "flow/flow.h"
#include "flow/IThreadPool.h"
#include "flow/ThreadHelper.actor.h"
#include "flow/Histogram.h"

#include <memory>
#include <tuple>
#include <vector>
#include <fstream>

#endif // WITH_ROCKSDB

#include "fdbserver/Knobs.h"
#include "fdbserver/IKeyValueStore.h"
#include "fdbserver/RocksDBCheckpointUtils.actor.h"

#include "flow/actorcompiler.h" // has to be last include

#ifdef WITH_ROCKSDB

// Enforcing rocksdb version.
static_assert(ROCKSDB_MAJOR == FDB_ROCKSDB_MAJOR, "Unsupported RocksDB major version");
static_assert(ROCKSDB_MINOR == FDB_ROCKSDB_MINOR, "Unsupported RocksDB minor version");
static_assert(ROCKSDB_PATCH == FDB_ROCKSDB_PATCH, "Unsupported RocksDB patch version");

namespace {
using rocksdb::BackgroundErrorReason;

rocksdb::WALRecoveryMode getWalRecoveryMode() {
	switch (SERVER_KNOBS->ROCKSDB_WAL_RECOVERY_MODE) {
	case 0:
		return rocksdb::WALRecoveryMode::kTolerateCorruptedTailRecords;
	case 1:
		return rocksdb::WALRecoveryMode::kAbsoluteConsistency;
	case 2:
		return rocksdb::WALRecoveryMode::kPointInTimeRecovery;
	case 3:
		return rocksdb::WALRecoveryMode::kSkipAnyCorruptedRecords;
	default:
		TraceEvent(SevWarn, "InvalidWalRecoveryMode").detail("KnobValue", SERVER_KNOBS->ROCKSDB_WAL_RECOVERY_MODE);
		return rocksdb::WALRecoveryMode::kPointInTimeRecovery;
	}
}

rocksdb::CompactionPri getCompactionPriority() {
	switch (SERVER_KNOBS->ROCKSDB_COMPACTION_PRI) {
	case 0:
		return rocksdb::CompactionPri::kByCompensatedSize;
	case 1:
		return rocksdb::CompactionPri::kOldestLargestSeqFirst;
	case 2:
		return rocksdb::CompactionPri::kOldestSmallestSeqFirst;
	case 3:
		return rocksdb::CompactionPri::kMinOverlappingRatio;
	case 4:
		return rocksdb::CompactionPri::kRoundRobin;
	default:
		TraceEvent(SevWarn, "InvalidCompactionPriority").detail("KnobValue", SERVER_KNOBS->ROCKSDB_COMPACTION_PRI);
		return rocksdb::CompactionPri::kMinOverlappingRatio;
	}
}

class SharedRocksDBState {
public:
	SharedRocksDBState(UID id);

	void setClosing() { this->closing = true; }
	bool isClosing() const { return this->closing; }

	rocksdb::DBOptions getDbOptions() const { return this->dbOptions; }
	rocksdb::ColumnFamilyOptions getCfOptions() const { return this->cfOptions; }
	rocksdb::Options getOptions() const { return rocksdb::Options(this->dbOptions, this->cfOptions); }
	rocksdb::ReadOptions getReadOptions() { return this->readOptions; }
	rocksdb::FlushOptions getFlushOptions() { return this->flushOptions; }
	double getLastFlushTime() const { return this->lastFlushTime_; }
	void setLastFlushTime(double lastFlushTime) { this->lastFlushTime_ = lastFlushTime; }

private:
	const UID id;
	rocksdb::ColumnFamilyOptions initialCfOptions();
	rocksdb::DBOptions initialDbOptions();
	rocksdb::ReadOptions initialReadOptions();
	rocksdb::FlushOptions initialFlushOptions();

	bool closing;
	rocksdb::DBOptions dbOptions;
	rocksdb::ColumnFamilyOptions cfOptions;
	rocksdb::ReadOptions readOptions;
	rocksdb::FlushOptions flushOptions;
	std::atomic<double> lastFlushTime_;
};

SharedRocksDBState::SharedRocksDBState(UID id)
  : id(id), closing(false), dbOptions(initialDbOptions()), cfOptions(initialCfOptions()),
    readOptions(initialReadOptions()), flushOptions(initialFlushOptions()) {}

rocksdb::FlushOptions SharedRocksDBState::initialFlushOptions() {
	rocksdb::FlushOptions fOptions;
	fOptions.wait = SERVER_KNOBS->ROCKSDB_WAIT_ON_CF_FLUSH;
	fOptions.allow_write_stall = SERVER_KNOBS->ROCKSDB_ALLOW_WRITE_STALL_ON_FLUSH;

	return fOptions;
}

rocksdb::ColumnFamilyOptions SharedRocksDBState::initialCfOptions() {
	rocksdb::ColumnFamilyOptions options;
	options.level_compaction_dynamic_level_bytes = SERVER_KNOBS->ROCKSDB_LEVEL_COMPACTION_DYNAMIC_LEVEL_BYTES;
	options.OptimizeLevelStyleCompaction(SERVER_KNOBS->ROCKSDB_MEMTABLE_BYTES);

	if (SERVER_KNOBS->ROCKSDB_DISABLE_AUTO_COMPACTIONS) {
		options.disable_auto_compactions = SERVER_KNOBS->ROCKSDB_DISABLE_AUTO_COMPACTIONS;
	}
	if (SERVER_KNOBS->ROCKSDB_PERIODIC_COMPACTION_SECONDS > 0) {
		// Adding two days range of jitter.
		int64_t jitter = 2 * 24 * 60 * 60 * deterministicRandom()->random01();
		options.periodic_compaction_seconds = SERVER_KNOBS->ROCKSDB_PERIODIC_COMPACTION_SECONDS + jitter;
	}
	if (SERVER_KNOBS->ROCKSDB_TTL_COMPACTION_SECONDS > 0) {
		// Adding two days range of jitter.
		int64_t jitter = 2 * 24 * 60 * 60 * deterministicRandom()->random01();
		options.ttl = SERVER_KNOBS->ROCKSDB_TTL_COMPACTION_SECONDS + jitter;
	}
	if (SERVER_KNOBS->ROCKSDB_MAX_COMPACTION_BYTES > 0) {
		options.max_compaction_bytes = SERVER_KNOBS->ROCKSDB_MAX_COMPACTION_BYTES;
	}
	if (SERVER_KNOBS->ROCKSDB_SOFT_PENDING_COMPACT_BYTES_LIMIT > 0) {
		options.soft_pending_compaction_bytes_limit = SERVER_KNOBS->ROCKSDB_SOFT_PENDING_COMPACT_BYTES_LIMIT;
	}
	if (SERVER_KNOBS->ROCKSDB_HARD_PENDING_COMPACT_BYTES_LIMIT > 0) {
		options.hard_pending_compaction_bytes_limit = SERVER_KNOBS->ROCKSDB_HARD_PENDING_COMPACT_BYTES_LIMIT;
	}
	options.memtable_protection_bytes_per_key = SERVER_KNOBS->ROCKSDB_MEMTABLE_PROTECTION_BYTES_PER_KEY;
	options.block_protection_bytes_per_key = SERVER_KNOBS->ROCKSDB_BLOCK_PROTECTION_BYTES_PER_KEY;
	options.paranoid_file_checks = SERVER_KNOBS->ROCKSDB_PARANOID_FILE_CHECKS;
	options.memtable_max_range_deletions = SERVER_KNOBS->ROCKSDB_MEMTABLE_MAX_RANGE_DELETIONS;
	if (SERVER_KNOBS->ROCKSDB_TARGET_FILE_SIZE_BASE > 0) {
		options.target_file_size_base = SERVER_KNOBS->ROCKSDB_TARGET_FILE_SIZE_BASE;
	}
	options.target_file_size_multiplier = SERVER_KNOBS->ROCKSDB_TARGET_FILE_SIZE_MULTIPLIER;
	options.write_buffer_size = SERVER_KNOBS->ROCKSDB_WRITE_BUFFER_SIZE;
	options.max_write_buffer_number = SERVER_KNOBS->ROCKSDB_MAX_WRITE_BUFFER_NUMBER;
	options.min_write_buffer_number_to_merge = SERVER_KNOBS->ROCKSDB_MIN_WRITE_BUFFER_NUMBER_TO_MERGE;
	options.level0_file_num_compaction_trigger = SERVER_KNOBS->ROCKSDB_LEVEL0_FILENUM_COMPACTION_TRIGGER;
	options.level0_slowdown_writes_trigger = SERVER_KNOBS->ROCKSDB_LEVEL0_SLOWDOWN_WRITES_TRIGGER;
	options.level0_stop_writes_trigger = SERVER_KNOBS->ROCKSDB_LEVEL0_STOP_WRITES_TRIGGER;

	// Compact sstables when there's too much deleted stuff.
	if (SERVER_KNOBS->ROCKSDB_ENABLE_COMPACT_ON_DELETION) {
		// Creates a factory of a table property collector that marks a SST
		// file as need-compaction when it observe at least "D" deletion
		// entries in any "N" consecutive entries, or the ratio of tombstone
		// entries >= deletion_ratio.

		// @param sliding_window_size "N". Note that this number will be
		//     round up to the smallest multiple of 128 that is no less
		//     than the specified size.
		// @param deletion_trigger "D".  Note that even when "N" is changed,
		//     the specified number for "D" will not be changed.
		// @param deletion_ratio, if <= 0 or > 1, disable triggering compaction
		//     based on deletion ratio. Disabled by default.
		options.table_properties_collector_factories = { rocksdb::NewCompactOnDeletionCollectorFactory(
			SERVER_KNOBS->ROCKSDB_CDCF_SLIDING_WINDOW_SIZE,
			SERVER_KNOBS->ROCKSDB_CDCF_DELETION_TRIGGER,
			SERVER_KNOBS->ROCKSDB_CDCF_DELETION_RATIO) };
	}

	rocksdb::BlockBasedTableOptions bbOpts;
	if (SERVER_KNOBS->ROCKSDB_PREFIX_LEN > 0) {
		// Prefix blooms are used during Seek.
		options.prefix_extractor.reset(rocksdb::NewFixedPrefixTransform(SERVER_KNOBS->ROCKSDB_PREFIX_LEN));

		// Also turn on bloom filters in the memtable.
		options.memtable_prefix_bloom_size_ratio = SERVER_KNOBS->ROCKSDB_MEMTABLE_PREFIX_BLOOM_SIZE_RATIO;

		// 5 -- Can be read by RocksDB's versions since 6.6.0. Full and partitioned
		// filters use a generally faster and more accurate Bloom filter
		// implementation, with a different schema.
		// https://github.com/facebook/rocksdb/blob/b77569f18bfc77fb1d8a0b3218f6ecf571bc4988/include/rocksdb/table.h#L391
		bbOpts.format_version = 5;

		// Create and apply a bloom filter using the 10 bits
		// which should yield a ~1% false positive rate:
		// https://github.com/facebook/rocksdb/wiki/RocksDB-Bloom-Filter#full-filters-new-format
		bbOpts.filter_policy.reset(rocksdb::NewBloomFilterPolicy(SERVER_KNOBS->ROCKSDB_BLOOM_BITS_PER_KEY));

		// The whole key blooms are only used for point lookups.
		// https://github.com/facebook/rocksdb/wiki/RocksDB-Bloom-Filter#prefix-vs-whole-key
		bbOpts.whole_key_filtering = SERVER_KNOBS->ROCKSDB_BLOOM_WHOLE_KEY_FILTERING;
	}

	if (SERVER_KNOBS->ROCKSDB_BLOCK_CACHE_SIZE > 0) {
		bbOpts.block_cache =
		    rocksdb::NewLRUCache(SERVER_KNOBS->ROCKSDB_BLOCK_CACHE_SIZE,
		                         -1, /* num_shard_bits, default value:-1*/
		                         false, /* strict_capacity_limit, default value:false */
		                         SERVER_KNOBS->ROCKSDB_CACHE_HIGH_PRI_POOL_RATIO /* high_pri_pool_ratio */);
		bbOpts.cache_index_and_filter_blocks = SERVER_KNOBS->ROCKSDB_CACHE_INDEX_AND_FILTER_BLOCKS;
		bbOpts.pin_l0_filter_and_index_blocks_in_cache = SERVER_KNOBS->ROCKSDB_CACHE_INDEX_AND_FILTER_BLOCKS;
		bbOpts.cache_index_and_filter_blocks_with_high_priority = SERVER_KNOBS->ROCKSDB_CACHE_INDEX_AND_FILTER_BLOCKS;
	}

	if (SERVER_KNOBS->ROCKSDB_BLOCK_SIZE > 0) {
		bbOpts.block_size = SERVER_KNOBS->ROCKSDB_BLOCK_SIZE;
	}

	// The readahead size starts with 8KB and is exponentially increased on each additional sequential IO,
	// up to a max of BlockBasedTableOptions.max_auto_readahead_size (default 256 KB)
	if (SERVER_KNOBS->ROCKSDB_MAX_AUTO_READAHEAD_SIZE > 0) {
		bbOpts.max_auto_readahead_size = SERVER_KNOBS->ROCKSDB_MAX_AUTO_READAHEAD_SIZE;
	}

	options.table_factory.reset(rocksdb::NewBlockBasedTableFactory(bbOpts));

	options.compaction_pri = getCompactionPriority();

	return options;
}

rocksdb::DBOptions SharedRocksDBState::initialDbOptions() {
	rocksdb::DBOptions options;
	options.use_direct_reads = SERVER_KNOBS->ROCKSDB_USE_DIRECT_READS;
	options.use_direct_io_for_flush_and_compaction = SERVER_KNOBS->ROCKSDB_USE_DIRECT_IO_FLUSH_COMPACTION;
	options.avoid_unnecessary_blocking_io = true;
	options.max_open_files = SERVER_KNOBS->ROCKSDB_MAX_OPEN_FILES;
	options.create_if_missing = true;
	if (SERVER_KNOBS->ROCKSDB_BACKGROUND_PARALLELISM > 0) {
		options.IncreaseParallelism(SERVER_KNOBS->ROCKSDB_BACKGROUND_PARALLELISM);
	}
	if (SERVER_KNOBS->ROCKSDB_MAX_SUBCOMPACTIONS > 0) {
		options.max_subcompactions = SERVER_KNOBS->ROCKSDB_MAX_SUBCOMPACTIONS;
	}
	if (SERVER_KNOBS->ROCKSDB_COMPACTION_READAHEAD_SIZE > 0) {
		options.compaction_readahead_size = SERVER_KNOBS->ROCKSDB_COMPACTION_READAHEAD_SIZE;
	}
	options.wal_recovery_mode = getWalRecoveryMode();
	// The following two fields affect how archived logs will be deleted.
	// 1. If both set to 0, logs will be deleted asap and will not get into
	//    the archive.
	// 2. If WAL_ttl_seconds is 0 and WAL_size_limit_MB is not 0,
	//    WAL files will be checked every 10 min and if total size is greater
	//    then WAL_size_limit_MB, they will be deleted starting with the
	//    earliest until size_limit is met. All empty files will be deleted.
	// 3. If WAL_ttl_seconds is not 0 and WAL_size_limit_MB is 0, then
	//    WAL files will be checked every WAL_ttl_seconds / 2 and those that
	//    are older than WAL_ttl_seconds will be deleted.
	// 4. If both are not 0, WAL files will be checked every 10 min and both
	//    checks will be performed with ttl being first.
	options.WAL_ttl_seconds = SERVER_KNOBS->ROCKSDB_WAL_TTL_SECONDS;
	options.WAL_size_limit_MB = SERVER_KNOBS->ROCKSDB_WAL_SIZE_LIMIT_MB;

	options.statistics = rocksdb::CreateDBStatistics();
	options.statistics->set_stats_level(rocksdb::StatsLevel(SERVER_KNOBS->ROCKSDB_STATS_LEVEL));

	options.db_log_dir = g_network->isSimulated() ? "" : SERVER_KNOBS->LOG_DIRECTORY;
	if (SERVER_KNOBS->ROCKSDB_LOG_LEVEL_DEBUG) {
		options.info_log_level = rocksdb::InfoLogLevel::DEBUG_LEVEL;
	}

	options.max_log_file_size = SERVER_KNOBS->ROCKSDB_MAX_LOG_FILE_SIZE;
	options.keep_log_file_num = SERVER_KNOBS->ROCKSDB_KEEP_LOG_FILE_NUM;

	if (!SERVER_KNOBS->ROCKSDB_MUTE_LOGS) {
		options.info_log = std::make_shared<RocksDBLogForwarder>(id, options.info_log_level);
	}

	if (SERVER_KNOBS->ROCKSDB_FULLFILE_CHECKSUM) {
		// We want this sst level checksum for many scenarios, such as compaction, backup, and physicalshardmove
		// https://github.com/facebook/rocksdb/wiki/Full-File-Checksum-and-Checksum-Handoff
		options.file_checksum_gen_factory = rocksdb::GetFileChecksumGenCrc32cFactory();
	}
	return options;
}

rocksdb::ReadOptions SharedRocksDBState::initialReadOptions() {
	rocksdb::ReadOptions options;
	options.background_purge_on_iterator_cleanup = true;
	options.auto_prefix_mode = (SERVER_KNOBS->ROCKSDB_PREFIX_LEN > 0);
	return options;
}

// Returns string representation of RocksDB background error reason.
// Error reason code:
// https://github.com/facebook/rocksdb/blob/12d798ac06bcce36be703b057d5f5f4dab3b270c/include/rocksdb/listener.h#L125
// This function needs to be updated when error code changes.
std::string getErrorReason(BackgroundErrorReason reason) {
	switch (reason) {
	case BackgroundErrorReason::kFlush:
		return format("%d Flush", reason);
	case BackgroundErrorReason::kCompaction:
		return format("%d Compaction", reason);
	case BackgroundErrorReason::kWriteCallback:
		return format("%d WriteCallback", reason);
	case BackgroundErrorReason::kMemTable:
		return format("%d MemTable", reason);
	case BackgroundErrorReason::kManifestWrite:
		return format("%d ManifestWrite", reason);
	case BackgroundErrorReason::kFlushNoWAL:
		return format("%d FlushNoWAL", reason);
	case BackgroundErrorReason::kManifestWriteNoWAL:
		return format("%d ManifestWriteNoWAL", reason);
	default:
		return format("%d Unknown", reason);
	}
}
// Background error handling is tested with Chaos test.
// TODO: Test background error in simulation. RocksDB doesn't use flow IO in simulation, which limits our ability to
// inject IO errors. We could implement rocksdb::FileSystem using flow IO to unblock simulation. Also, trace event is
// not available on background threads because trace event requires setting up special thread locals. Using trace event
// could potentially cause segmentation fault.
class RocksDBErrorListener : public rocksdb::EventListener {
public:
	RocksDBErrorListener(UID id) : id(id) {};
	void OnBackgroundError(rocksdb::BackgroundErrorReason reason, rocksdb::Status* bg_error) override {
		TraceEvent(SevError, "RocksDBBGError", id)
		    .detail("Reason", getErrorReason(reason))
		    .detail("RocksDBSeverity", bg_error->severity())
		    .detail("Status", bg_error->ToString());
		std::unique_lock<std::mutex> lock(mutex);
		if (!errorPromise.isValid())
			return;
		// RocksDB generates two types of background errors, IO Error and Corruption
		// Error type and severity map could be found at
		// https://github.com/facebook/rocksdb/blob/2e09a54c4fb82e88bcaa3e7cfa8ccbbbbf3635d5/db/error_handler.cc#L138.
		// All background errors will be treated as storage engine failure. Send the error to storage server.
		if (bg_error->IsIOError()) {
			errorPromise.sendError(io_error());
		} else if (bg_error->IsCorruption()) {
			errorPromise.sendError(file_corrupt());
		} else {
			errorPromise.sendError(unknown_error());
		}
	}
	Future<Void> getFuture() {
		std::unique_lock<std::mutex> lock(mutex);
		return errorPromise.getFuture();
	}
	~RocksDBErrorListener() {
		std::unique_lock<std::mutex> lock(mutex);
		if (!errorPromise.isValid())
			return;
		errorPromise.send(Never());
	}

private:
	ThreadReturnPromise<Void> errorPromise;
	std::mutex mutex;
	UID id;
};

class RocksDBEventListener : public rocksdb::EventListener {
public:
	RocksDBEventListener(std::shared_ptr<SharedRocksDBState> sharedState) : sharedState(sharedState) {};

	void OnFlushCompleted(rocksdb::DB* db, const rocksdb::FlushJobInfo& info) override {
		sharedState->setLastFlushTime(now());
	}

private:
	std::shared_ptr<SharedRocksDBState> sharedState;
};

using DB = rocksdb::DB*;
using CF = rocksdb::ColumnFamilyHandle*;

#define PERSIST_PREFIX "\xff\xff"
const KeyRef persistVersion = PERSIST_PREFIX "Version"_sr;
const StringRef ROCKSDBSTORAGE_HISTOGRAM_GROUP = "RocksDBStorage"_sr;
const StringRef ROCKSDB_COMMIT_LATENCY_HISTOGRAM = "RocksDBCommitLatency"_sr;
const StringRef ROCKSDB_COMMIT_ACTION_HISTOGRAM = "RocksDBCommitAction"_sr;
const StringRef ROCKSDB_COMMIT_QUEUEWAIT_HISTOGRAM = "RocksDBCommitQueueWait"_sr;
const StringRef ROCKSDB_WRITE_HISTOGRAM = "RocksDBWrite"_sr;
const StringRef ROCKSDB_DELETE_COMPACTRANGE_HISTOGRAM = "RocksDBDeleteCompactRange"_sr;
const StringRef ROCKSDB_READRANGE_LATENCY_HISTOGRAM = "RocksDBReadRangeLatency"_sr;
const StringRef ROCKSDB_READVALUE_LATENCY_HISTOGRAM = "RocksDBReadValueLatency"_sr;
const StringRef ROCKSDB_READPREFIX_LATENCY_HISTOGRAM = "RocksDBReadPrefixLatency"_sr;
const StringRef ROCKSDB_READRANGE_ACTION_HISTOGRAM = "RocksDBReadRangeAction"_sr;
const StringRef ROCKSDB_READVALUE_ACTION_HISTOGRAM = "RocksDBReadValueAction"_sr;
const StringRef ROCKSDB_READPREFIX_ACTION_HISTOGRAM = "RocksDBReadPrefixAction"_sr;
const StringRef ROCKSDB_READRANGE_QUEUEWAIT_HISTOGRAM = "RocksDBReadRangeQueueWait"_sr;
const StringRef ROCKSDB_READVALUE_QUEUEWAIT_HISTOGRAM = "RocksDBReadValueQueueWait"_sr;
const StringRef ROCKSDB_READPREFIX_QUEUEWAIT_HISTOGRAM = "RocksDBReadPrefixQueueWait"_sr;
const StringRef ROCKSDB_READRANGE_NEWITERATOR_HISTOGRAM = "RocksDBReadRangeNewIterator"_sr;
const StringRef ROCKSDB_READVALUE_GET_HISTOGRAM = "RocksDBReadValueGet"_sr;
const StringRef ROCKSDB_READPREFIX_GET_HISTOGRAM = "RocksDBReadPrefixGet"_sr;
const StringRef ROCKSDB_READ_RANGE_BYTES_RETURNED_HISTOGRAM = "RocksDBReadRangeBytesReturned"_sr;
const StringRef ROCKSDB_READ_RANGE_KV_PAIRS_RETURNED_HISTOGRAM = "RocksDBReadRangeKVPairsReturned"_sr;
const StringRef ROCKSDB_DELETES_PER_COMMIT_HISTOGRAM = "RocksDBDeletesPerCommit"_sr;
const StringRef ROCKSDB_DELETE_RANGES_PER_COMMIT_HISTOGRAM = "RocksDBDeleteRangesPerCommit"_sr;

rocksdb::ExportImportFilesMetaData getMetaData(const CheckpointMetaData& checkpoint) {
	rocksdb::ExportImportFilesMetaData metaData;
	if (checkpoint.getFormat() != DataMoveRocksCF) {
		return metaData;
	}

	RocksDBColumnFamilyCheckpoint rocksCF = getRocksCF(checkpoint);
	metaData.db_comparator_name = rocksCF.dbComparatorName;

	for (const LiveFileMetaData& fileMetaData : rocksCF.sstFiles) {
		rocksdb::LiveFileMetaData liveFileMetaData;
		liveFileMetaData.relative_filename = fileMetaData.relative_filename;
		liveFileMetaData.directory = fileMetaData.directory;
		liveFileMetaData.file_number = fileMetaData.file_number;
		liveFileMetaData.file_type = static_cast<rocksdb::FileType>(fileMetaData.file_type);
		liveFileMetaData.size = fileMetaData.size;
		liveFileMetaData.temperature = static_cast<rocksdb::Temperature>(fileMetaData.temperature);
		liveFileMetaData.file_checksum = fileMetaData.file_checksum;
		liveFileMetaData.file_checksum_func_name = fileMetaData.file_checksum_func_name;
		liveFileMetaData.smallest_seqno = fileMetaData.smallest_seqno;
		liveFileMetaData.largest_seqno = fileMetaData.largest_seqno;
		liveFileMetaData.smallestkey = fileMetaData.smallestkey;
		liveFileMetaData.largestkey = fileMetaData.largestkey;
		liveFileMetaData.num_reads_sampled = fileMetaData.num_reads_sampled;
		liveFileMetaData.being_compacted = fileMetaData.being_compacted;
		liveFileMetaData.num_entries = fileMetaData.num_entries;
		liveFileMetaData.num_deletions = fileMetaData.num_deletions;
		liveFileMetaData.oldest_blob_file_number = fileMetaData.oldest_blob_file_number;
		liveFileMetaData.oldest_ancester_time = fileMetaData.oldest_ancester_time;
		liveFileMetaData.file_creation_time = fileMetaData.file_creation_time;
		liveFileMetaData.epoch_number = fileMetaData.epoch_number;
		liveFileMetaData.name = fileMetaData.name;
		liveFileMetaData.db_path = fileMetaData.db_path;
		liveFileMetaData.column_family_name = fileMetaData.column_family_name;
		liveFileMetaData.level = fileMetaData.level;
		metaData.files.push_back(liveFileMetaData);
	}

	return metaData;
}

void populateMetaData(CheckpointMetaData* checkpoint, const rocksdb::ExportImportFilesMetaData& metaData) {
	RocksDBColumnFamilyCheckpoint rocksCF;
	rocksCF.dbComparatorName = metaData.db_comparator_name;
	for (const rocksdb::LiveFileMetaData& fileMetaData : metaData.files) {
		LiveFileMetaData liveFileMetaData;
		liveFileMetaData.relative_filename = fileMetaData.relative_filename;
		liveFileMetaData.directory = fileMetaData.directory;
		liveFileMetaData.file_number = fileMetaData.file_number;
		liveFileMetaData.file_type = static_cast<int>(fileMetaData.file_type);
		liveFileMetaData.size = fileMetaData.size;
		liveFileMetaData.temperature = static_cast<uint8_t>(fileMetaData.temperature);
		liveFileMetaData.file_checksum = fileMetaData.file_checksum;
		liveFileMetaData.file_checksum_func_name = fileMetaData.file_checksum_func_name;
		liveFileMetaData.smallest_seqno = fileMetaData.smallest_seqno;
		liveFileMetaData.largest_seqno = fileMetaData.largest_seqno;
		liveFileMetaData.smallestkey = fileMetaData.smallestkey;
		liveFileMetaData.largestkey = fileMetaData.largestkey;
		liveFileMetaData.num_reads_sampled = fileMetaData.num_reads_sampled;
		liveFileMetaData.being_compacted = fileMetaData.being_compacted;
		liveFileMetaData.num_entries = fileMetaData.num_entries;
		liveFileMetaData.num_deletions = fileMetaData.num_deletions;
		liveFileMetaData.oldest_blob_file_number = fileMetaData.oldest_blob_file_number;
		liveFileMetaData.oldest_ancester_time = fileMetaData.oldest_ancester_time;
		liveFileMetaData.file_creation_time = fileMetaData.file_creation_time;
		liveFileMetaData.epoch_number = fileMetaData.epoch_number;
		liveFileMetaData.name = fileMetaData.name;
		liveFileMetaData.db_path = fileMetaData.db_path;
		liveFileMetaData.column_family_name = fileMetaData.column_family_name;
		liveFileMetaData.level = fileMetaData.level;
		rocksCF.sstFiles.push_back(liveFileMetaData);
	}
	checkpoint->setFormat(DataMoveRocksCF);
	checkpoint->setSerializedCheckpoint(ObjectWriter::toValue(rocksCF, IncludeVersion()));
}

rocksdb::Slice toSlice(StringRef s) {
	return rocksdb::Slice(reinterpret_cast<const char*>(s.begin()), s.size());
}

StringRef toStringRef(rocksdb::Slice s) {
	return StringRef(reinterpret_cast<const uint8_t*>(s.data()), s.size());
}

struct Counters {
	CounterCollection cc;
	Counter immediateThrottle;
	Counter failedToAcquire;
	Counter deleteKeyReqs;
	Counter deleteRangeReqs;
	Counter convertedDeleteKeyReqs;
	Counter convertedDeleteRangeReqs;
	Counter rocksdbReadRangeQueries;
	Counter commitDelayed;

	Counters()
	  : cc("RocksDBThrottle"), immediateThrottle("ImmediateThrottle", cc), failedToAcquire("FailedToAcquire", cc),
	    deleteKeyReqs("DeleteKeyRequests", cc), deleteRangeReqs("DeleteRangeRequests", cc),
	    convertedDeleteKeyReqs("ConvertedDeleteKeyRequests", cc),
	    convertedDeleteRangeReqs("ConvertedDeleteRangeRequests", cc),
	    rocksdbReadRangeQueries("RocksdbReadRangeQueries", cc), commitDelayed("CommitDelayed", cc) {}
};

struct ReadIterator {
	uint64_t index; // incrementing counter to uniquely identify read iterator.
	bool inUse;
	std::shared_ptr<rocksdb::Iterator> iter;
	double creationTime;
	KeyRange keyRange;
	std::shared_ptr<rocksdb::Slice> beginSlice, endSlice;
	ReadIterator(CF& cf, uint64_t index, DB& db, std::shared_ptr<SharedRocksDBState> sharedState)
	  : index(index), inUse(true), creationTime(now()), iter(db->NewIterator(sharedState->getReadOptions(), cf)) {}
	ReadIterator(CF& cf, uint64_t index, DB& db, std::shared_ptr<SharedRocksDBState> sharedState, KeyRange keyRange)
	  : index(index), inUse(true), creationTime(now()), keyRange(keyRange) {
		rocksdb::ReadOptions readOptions = sharedState->getReadOptions();
		beginSlice = std::shared_ptr<rocksdb::Slice>(new rocksdb::Slice(toSlice(keyRange.begin)));
		readOptions.iterate_lower_bound = beginSlice.get();
		endSlice = std::shared_ptr<rocksdb::Slice>(new rocksdb::Slice(toSlice(keyRange.end)));
		readOptions.iterate_upper_bound = endSlice.get();

		iter = std::shared_ptr<rocksdb::Iterator>(db->NewIterator(readOptions, cf));
	}
};

/*
ReadIteratorPool: Collection of iterators. Reuses iterators on non-concurrent multiple read operations,
instead of creating and deleting for every read.

Read: IteratorPool provides an unused iterator if exists or creates and gives a new iterator.
Returns back the iterator after the read is done.

Write: Iterators in the pool are deleted, forcing new iterator creation on next reads. The iterators
which are currently used by the reads can continue using the iterator as it is a shared_ptr. Once
the read is processed, shared_ptr goes out of scope and gets deleted. Eventually the iterator object
gets deleted as the ref count becomes 0.
*/
class ReadIteratorPool {
public:
	ReadIteratorPool(UID id, DB& db, CF& cf, std::shared_ptr<SharedRocksDBState> sharedState)
	  : db(db), cf(cf), index(0), deletedUptoIndex(0), iteratorsReuseCount(0), sharedState(sharedState) {
		TraceEvent("ReadIteratorPool", id)
		    .detail("KnobRocksDBReadRangeReuseIterators", SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_ITERATORS)
		    .detail("KnobRocksDBReadRangeReuseBoundedIterators",
		            SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_BOUNDED_ITERATORS)
		    .detail("KnobRocksDBReadRangeBoundedIteratorsMaxLimit",
		            SERVER_KNOBS->ROCKSDB_READ_RANGE_BOUNDED_ITERATORS_MAX_LIMIT)
		    .detail("KnobRocksDBPrefixLen", SERVER_KNOBS->ROCKSDB_PREFIX_LEN);
		if (SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_ITERATORS &&
		    SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_BOUNDED_ITERATORS) {
			TraceEvent(SevWarn, "ReadIteratorKnobsMismatch");
		}
	}

	// Called on every db commit.
	void update() {
		if (SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_ITERATORS ||
		    SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_BOUNDED_ITERATORS) {
			mutex.lock();
			// The latest index might contain the current iterator which is getting created.
			// But, that should be ok to avoid adding more code complexity.
			deletedUptoIndex = index;
			mutex.unlock();
			deleteIteratorsPromise.send(Void());
		}
	}

	// Called on every read operation.
	ReadIterator getIterator(KeyRange keyRange) {
		// Reusing iterator in simulation can cause slow down
		// We avoid to always reuse iterator in simulation to speed up the simulation
		if (g_network->isSimulated() &&
		    deterministicRandom()->random01() > SERVER_KNOBS->ROCKSDB_PROBABILITY_REUSE_ITERATOR_SIM) {
			index++;
			ReadIterator iter(cf, index, db, sharedState, keyRange);
			return iter;
		}

		if (SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_ITERATORS) {
			mutex.lock();
			for (it = iteratorsMap.begin(); it != iteratorsMap.end(); it++) {
				if (!it->second.inUse && it->second.index > deletedUptoIndex) {
					it->second.inUse = true;
					iteratorsReuseCount++;
					ReadIterator iter = it->second;
					mutex.unlock();
					return iter;
				}
			}
			index++;
			uint64_t readIteratorIndex = index;
			mutex.unlock();

			ReadIterator iter(cf, readIteratorIndex, db, sharedState);
			mutex.lock();
			iteratorsMap.insert({ readIteratorIndex, iter });
			mutex.unlock();
			return iter;
		} else if (SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_BOUNDED_ITERATORS) {
			// TODO: Based on the datasize in the keyrange, decide whether to store the iterator for reuse.
			mutex.lock();
			for (it = iteratorsMap.begin(); it != iteratorsMap.end(); it++) {
				if (!it->second.inUse && it->second.index > deletedUptoIndex &&
				    it->second.keyRange.contains(keyRange)) {
					it->second.inUse = true;
					iteratorsReuseCount++;
					ReadIterator iter = it->second;
					mutex.unlock();
					return iter;
				}
			}
			index++;
			uint64_t readIteratorIndex = index;
			mutex.unlock();

			ReadIterator iter(cf, readIteratorIndex, db, sharedState, keyRange);
			if (iteratorsMap.size() < SERVER_KNOBS->ROCKSDB_READ_RANGE_BOUNDED_ITERATORS_MAX_LIMIT) {
				// Not storing more than ROCKSDB_READ_RANGE_BOUNDED_ITERATORS_MAX_LIMIT of iterators
				// to avoid 'out of memory' issues.
				mutex.lock();
				iteratorsMap.insert({ readIteratorIndex, iter });
				mutex.unlock();
			}
			return iter;
		} else {
			index++;
			ReadIterator iter(cf, index, db, sharedState, keyRange);
			return iter;
		}
	}

	// Called on every read operation, after the keys are collected.
	void returnIterator(ReadIterator& iter) {
		if (SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_ITERATORS ||
		    SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_BOUNDED_ITERATORS) {
			std::lock_guard<std::mutex> lock(mutex);
			it = iteratorsMap.find(iter.index);
			// iterator found: put the iterator back to the pool(inUse=false).
			// iterator not found: update would have removed the iterator from pool, so nothing to do.
			if (it != iteratorsMap.end()) {
				ASSERT(it->second.inUse);
				it->second.inUse = false;
			}
		}
	}

	// Called for every ROCKSDB_READ_RANGE_ITERATOR_REFRESH_TIME seconds in a loop.
	void refreshIterators() {
		std::lock_guard<std::mutex> lock(mutex);
		it = iteratorsMap.begin();
		auto currTime = now();
		while (it != iteratorsMap.end()) {
			if ((it->second.index <= deletedUptoIndex) ||
			    ((currTime - it->second.creationTime) > SERVER_KNOBS->ROCKSDB_READ_RANGE_ITERATOR_REFRESH_TIME)) {
				it = iteratorsMap.erase(it);
			} else {
				it++;
			}
		}
	}

	uint64_t numReadIteratorsCreated() { return index; }

	uint64_t numTimesReadIteratorsReused() { return iteratorsReuseCount; }

	ThreadFutureStream<Void> getDeleteIteratorsFutureStream() { return deleteIteratorsPromise.getFuture(); }

private:
	std::unordered_map<int, ReadIterator> iteratorsMap;
	std::unordered_map<int, ReadIterator>::iterator it;
	DB& db;
	CF& cf;
	std::shared_ptr<SharedRocksDBState> sharedState;
	std::mutex mutex;
	// incrementing counter for every new iterator creation, to uniquely identify the iterator in returnIterator().
	uint64_t index;
	uint64_t deletedUptoIndex;
	uint64_t iteratorsReuseCount;
	ThreadReturnPromiseStream<Void> deleteIteratorsPromise;
};

class PerfContextMetrics {
public:
	PerfContextMetrics();
	void reset();
	void set(int index);
	void log(bool ignoreZeroMetric);

private:
	std::vector<std::tuple<const char*, int, std::vector<uint64_t>>> metrics;
	uint64_t getRocksdbPerfcontextMetric(int metric);
};

PerfContextMetrics::PerfContextMetrics() {
	metrics = {
		{ "UserKeyComparisonCount", rocksdb_user_key_comparison_count, {} },
		{ "BlockCacheHitCount", rocksdb_block_cache_hit_count, {} },
		{ "BlockReadCount", rocksdb_block_read_count, {} },
		{ "BlockReadByte", rocksdb_block_read_byte, {} },
		{ "BlockReadTime", rocksdb_block_read_time, {} },
		{ "BlockChecksumTime", rocksdb_block_checksum_time, {} },
		{ "BlockDecompressTime", rocksdb_block_decompress_time, {} },
		{ "GetReadBytes", rocksdb_get_read_bytes, {} },
		{ "MultigetReadBytes", rocksdb_multiget_read_bytes, {} },
		{ "IterReadBytes", rocksdb_iter_read_bytes, {} },
		{ "InternalKeySkippedCount", rocksdb_internal_key_skipped_count, {} },
		{ "InternalDeleteSkippedCount", rocksdb_internal_delete_skipped_count, {} },
		{ "InternalRecentSkippedCount", rocksdb_internal_recent_skipped_count, {} },
		{ "InternalMergeCount", rocksdb_internal_merge_count, {} },
		{ "GetSnapshotTime", rocksdb_get_snapshot_time, {} },
		{ "GetFromMemtableTime", rocksdb_get_from_memtable_time, {} },
		{ "GetFromMemtableCount", rocksdb_get_from_memtable_count, {} },
		{ "GetPostProcessTime", rocksdb_get_post_process_time, {} },
		{ "GetFromOutputFilesTime", rocksdb_get_from_output_files_time, {} },
		{ "SeekOnMemtableTime", rocksdb_seek_on_memtable_time, {} },
		{ "SeekOnMemtableCount", rocksdb_seek_on_memtable_count, {} },
		{ "NextOnMemtableCount", rocksdb_next_on_memtable_count, {} },
		{ "PrevOnMemtableCount", rocksdb_prev_on_memtable_count, {} },
		{ "SeekChildSeekTime", rocksdb_seek_child_seek_time, {} },
		{ "SeekChildSeekCount", rocksdb_seek_child_seek_count, {} },
		{ "SeekMinHeapTime", rocksdb_seek_min_heap_time, {} },
		{ "SeekMaxHeapTime", rocksdb_seek_max_heap_time, {} },
		{ "SeekInternalSeekTime", rocksdb_seek_internal_seek_time, {} },
		{ "FindNextUserEntryTime", rocksdb_find_next_user_entry_time, {} },
		{ "WriteWalTime", rocksdb_write_wal_time, {} },
		{ "WriteMemtableTime", rocksdb_write_memtable_time, {} },
		{ "WriteDelayTime", rocksdb_write_delay_time, {} },
		{ "WritePreAndPostProcessTime", rocksdb_write_pre_and_post_process_time, {} },
		{ "DbMutexLockNanos", rocksdb_db_mutex_lock_nanos, {} },
		{ "DbConditionWaitNanos", rocksdb_db_condition_wait_nanos, {} },
		{ "MergeOperatorTimeNanos", rocksdb_merge_operator_time_nanos, {} },
		{ "ReadIndexBlockNanos", rocksdb_read_index_block_nanos, {} },
		{ "ReadFilterBlockNanos", rocksdb_read_filter_block_nanos, {} },
		{ "NewTableBlockIterNanos", rocksdb_new_table_block_iter_nanos, {} },
		{ "NewTableIteratorNanos", rocksdb_new_table_iterator_nanos, {} },
		{ "BlockSeekNanos", rocksdb_block_seek_nanos, {} },
		{ "FindTableNanos", rocksdb_find_table_nanos, {} },
		{ "BloomMemtableHitCount", rocksdb_bloom_memtable_hit_count, {} },
		{ "BloomMemtableMissCount", rocksdb_bloom_memtable_miss_count, {} },
		{ "BloomSstHitCount", rocksdb_bloom_sst_hit_count, {} },
		{ "BloomSstMissCount", rocksdb_bloom_sst_miss_count, {} },
		{ "KeyLockWaitTime", rocksdb_key_lock_wait_time, {} },
		{ "KeyLockWaitCount", rocksdb_key_lock_wait_count, {} },
		{ "EnvNewSequentialFileNanos", rocksdb_env_new_sequential_file_nanos, {} },
		{ "EnvNewRandomAccessFileNanos", rocksdb_env_new_random_access_file_nanos, {} },
		{ "EnvNewWritableFileNanos", rocksdb_env_new_writable_file_nanos, {} },
		{ "EnvReuseWritableFileNanos", rocksdb_env_reuse_writable_file_nanos, {} },
		{ "EnvNewRandomRwFileNanos", rocksdb_env_new_random_rw_file_nanos, {} },
		{ "EnvNewDirectoryNanos", rocksdb_env_new_directory_nanos, {} },
		{ "EnvFileExistsNanos", rocksdb_env_file_exists_nanos, {} },
		{ "EnvGetChildrenNanos", rocksdb_env_get_children_nanos, {} },
		{ "EnvGetChildrenFileAttributesNanos", rocksdb_env_get_children_file_attributes_nanos, {} },
		{ "EnvDeleteFileNanos", rocksdb_env_delete_file_nanos, {} },
		{ "EnvCreateDirNanos", rocksdb_env_create_dir_nanos, {} },
		{ "EnvCreateDirIfMissingNanos", rocksdb_env_create_dir_if_missing_nanos, {} },
		{ "EnvDeleteDirNanos", rocksdb_env_delete_dir_nanos, {} },
		{ "EnvGetFileSizeNanos", rocksdb_env_get_file_size_nanos, {} },
		{ "EnvGetFileModificationTimeNanos", rocksdb_env_get_file_modification_time_nanos, {} },
		{ "EnvRenameFileNanos", rocksdb_env_rename_file_nanos, {} },
		{ "EnvLinkFileNanos", rocksdb_env_link_file_nanos, {} },
		{ "EnvLockFileNanos", rocksdb_env_lock_file_nanos, {} },
		{ "EnvUnlockFileNanos", rocksdb_env_unlock_file_nanos, {} },
		{ "EnvNewLoggerNanos", rocksdb_env_new_logger_nanos, {} },
	};
	for (auto& [name, metric, vals] : metrics) { // readers, then writer
		for (int i = 0; i < SERVER_KNOBS->ROCKSDB_READ_PARALLELISM; i++) {
			vals.push_back(0); // add reader
		}
		vals.push_back(0); // add writer
	}
}

void PerfContextMetrics::reset() {
	rocksdb::get_perf_context()->Reset();
}

void PerfContextMetrics::set(int index) {
	for (auto& [name, metric, vals] : metrics) {
		vals[index] = getRocksdbPerfcontextMetric(metric);
	}
}

void PerfContextMetrics::log(bool ignoreZeroMetric) {
	TraceEvent e("RocksDBPerfContextMetrics");
	e.setMaxEventLength(20000);
	for (auto& [name, metric, vals] : metrics) {
		uint64_t s = 0;
		for (auto& v : vals) {
			s = s + v;
		}
		if (ignoreZeroMetric && s == 0)
			continue;
		e.detail("Sum" + (std::string)name, s);
		for (int i = 0; i < SERVER_KNOBS->ROCKSDB_READ_PARALLELISM; i++) {
			if (vals[i] != 0)
				e.detail("RD" + std::to_string(i) + name, vals[i]);
		}
		if (vals[SERVER_KNOBS->ROCKSDB_READ_PARALLELISM] != 0)
			e.detail("WR" + (std::string)name, vals[SERVER_KNOBS->ROCKSDB_READ_PARALLELISM]);
	}
}

uint64_t PerfContextMetrics::getRocksdbPerfcontextMetric(int metric) {
	switch (metric) {
	case rocksdb_user_key_comparison_count:
		return rocksdb::get_perf_context()->user_key_comparison_count;
	case rocksdb_block_cache_hit_count:
		return rocksdb::get_perf_context()->block_cache_hit_count;
	case rocksdb_block_read_count:
		return rocksdb::get_perf_context()->block_read_count;
	case rocksdb_block_read_byte:
		return rocksdb::get_perf_context()->block_read_byte;
	case rocksdb_block_read_time:
		return rocksdb::get_perf_context()->block_read_time;
	case rocksdb_block_checksum_time:
		return rocksdb::get_perf_context()->block_checksum_time;
	case rocksdb_block_decompress_time:
		return rocksdb::get_perf_context()->block_decompress_time;
	case rocksdb_get_read_bytes:
		return rocksdb::get_perf_context()->get_read_bytes;
	case rocksdb_multiget_read_bytes:
		return rocksdb::get_perf_context()->multiget_read_bytes;
	case rocksdb_iter_read_bytes:
		return rocksdb::get_perf_context()->iter_read_bytes;
	case rocksdb_internal_key_skipped_count:
		return rocksdb::get_perf_context()->internal_key_skipped_count;
	case rocksdb_internal_delete_skipped_count:
		return rocksdb::get_perf_context()->internal_delete_skipped_count;
	case rocksdb_internal_recent_skipped_count:
		return rocksdb::get_perf_context()->internal_recent_skipped_count;
	case rocksdb_internal_merge_count:
		return rocksdb::get_perf_context()->internal_merge_count;
	case rocksdb_get_snapshot_time:
		return rocksdb::get_perf_context()->get_snapshot_time;
	case rocksdb_get_from_memtable_time:
		return rocksdb::get_perf_context()->get_from_memtable_time;
	case rocksdb_get_from_memtable_count:
		return rocksdb::get_perf_context()->get_from_memtable_count;
	case rocksdb_get_post_process_time:
		return rocksdb::get_perf_context()->get_post_process_time;
	case rocksdb_get_from_output_files_time:
		return rocksdb::get_perf_context()->get_from_output_files_time;
	case rocksdb_seek_on_memtable_time:
		return rocksdb::get_perf_context()->seek_on_memtable_time;
	case rocksdb_seek_on_memtable_count:
		return rocksdb::get_perf_context()->seek_on_memtable_count;
	case rocksdb_next_on_memtable_count:
		return rocksdb::get_perf_context()->next_on_memtable_count;
	case rocksdb_prev_on_memtable_count:
		return rocksdb::get_perf_context()->prev_on_memtable_count;
	case rocksdb_seek_child_seek_time:
		return rocksdb::get_perf_context()->seek_child_seek_time;
	case rocksdb_seek_child_seek_count:
		return rocksdb::get_perf_context()->seek_child_seek_count;
	case rocksdb_seek_min_heap_time:
		return rocksdb::get_perf_context()->seek_min_heap_time;
	case rocksdb_seek_max_heap_time:
		return rocksdb::get_perf_context()->seek_max_heap_time;
	case rocksdb_seek_internal_seek_time:
		return rocksdb::get_perf_context()->seek_internal_seek_time;
	case rocksdb_find_next_user_entry_time:
		return rocksdb::get_perf_context()->find_next_user_entry_time;
	case rocksdb_write_wal_time:
		return rocksdb::get_perf_context()->write_wal_time;
	case rocksdb_write_memtable_time:
		return rocksdb::get_perf_context()->write_memtable_time;
	case rocksdb_write_delay_time:
		return rocksdb::get_perf_context()->write_delay_time;
	case rocksdb_write_pre_and_post_process_time:
		return rocksdb::get_perf_context()->write_pre_and_post_process_time;
	case rocksdb_db_mutex_lock_nanos:
		return rocksdb::get_perf_context()->db_mutex_lock_nanos;
	case rocksdb_db_condition_wait_nanos:
		return rocksdb::get_perf_context()->db_condition_wait_nanos;
	case rocksdb_merge_operator_time_nanos:
		return rocksdb::get_perf_context()->merge_operator_time_nanos;
	case rocksdb_read_index_block_nanos:
		return rocksdb::get_perf_context()->read_index_block_nanos;
	case rocksdb_read_filter_block_nanos:
		return rocksdb::get_perf_context()->read_filter_block_nanos;
	case rocksdb_new_table_block_iter_nanos:
		return rocksdb::get_perf_context()->new_table_block_iter_nanos;
	case rocksdb_new_table_iterator_nanos:
		return rocksdb::get_perf_context()->new_table_iterator_nanos;
	case rocksdb_block_seek_nanos:
		return rocksdb::get_perf_context()->block_seek_nanos;
	case rocksdb_find_table_nanos:
		return rocksdb::get_perf_context()->find_table_nanos;
	case rocksdb_bloom_memtable_hit_count:
		return rocksdb::get_perf_context()->bloom_memtable_hit_count;
	case rocksdb_bloom_memtable_miss_count:
		return rocksdb::get_perf_context()->bloom_memtable_miss_count;
	case rocksdb_bloom_sst_hit_count:
		return rocksdb::get_perf_context()->bloom_sst_hit_count;
	case rocksdb_bloom_sst_miss_count:
		return rocksdb::get_perf_context()->bloom_sst_miss_count;
	case rocksdb_key_lock_wait_time:
		return rocksdb::get_perf_context()->key_lock_wait_time;
	case rocksdb_key_lock_wait_count:
		return rocksdb::get_perf_context()->key_lock_wait_count;
	case rocksdb_env_new_sequential_file_nanos:
		return rocksdb::get_perf_context()->env_new_sequential_file_nanos;
	case rocksdb_env_new_random_access_file_nanos:
		return rocksdb::get_perf_context()->env_new_random_access_file_nanos;
	case rocksdb_env_new_writable_file_nanos:
		return rocksdb::get_perf_context()->env_new_writable_file_nanos;
	case rocksdb_env_reuse_writable_file_nanos:
		return rocksdb::get_perf_context()->env_reuse_writable_file_nanos;
	case rocksdb_env_new_random_rw_file_nanos:
		return rocksdb::get_perf_context()->env_new_random_rw_file_nanos;
	case rocksdb_env_new_directory_nanos:
		return rocksdb::get_perf_context()->env_new_directory_nanos;
	case rocksdb_env_file_exists_nanos:
		return rocksdb::get_perf_context()->env_file_exists_nanos;
	case rocksdb_env_get_children_nanos:
		return rocksdb::get_perf_context()->env_get_children_nanos;
	case rocksdb_env_get_children_file_attributes_nanos:
		return rocksdb::get_perf_context()->env_get_children_file_attributes_nanos;
	case rocksdb_env_delete_file_nanos:
		return rocksdb::get_perf_context()->env_delete_file_nanos;
	case rocksdb_env_create_dir_nanos:
		return rocksdb::get_perf_context()->env_create_dir_nanos;
	case rocksdb_env_create_dir_if_missing_nanos:
		return rocksdb::get_perf_context()->env_create_dir_if_missing_nanos;
	case rocksdb_env_delete_dir_nanos:
		return rocksdb::get_perf_context()->env_delete_dir_nanos;
	case rocksdb_env_get_file_size_nanos:
		return rocksdb::get_perf_context()->env_get_file_size_nanos;
	case rocksdb_env_get_file_modification_time_nanos:
		return rocksdb::get_perf_context()->env_get_file_modification_time_nanos;
	case rocksdb_env_rename_file_nanos:
		return rocksdb::get_perf_context()->env_rename_file_nanos;
	case rocksdb_env_link_file_nanos:
		return rocksdb::get_perf_context()->env_link_file_nanos;
	case rocksdb_env_lock_file_nanos:
		return rocksdb::get_perf_context()->env_lock_file_nanos;
	case rocksdb_env_unlock_file_nanos:
		return rocksdb::get_perf_context()->env_unlock_file_nanos;
	case rocksdb_env_new_logger_nanos:
		return rocksdb::get_perf_context()->env_new_logger_nanos;
	default:
		break;
	}
	return 0;
}

															#line 992 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via refreshReadIteratorPool()
															#line 990 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class RefreshReadIteratorPoolActor>
															#line 990 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class RefreshReadIteratorPoolActorState {
															#line 998 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 990 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	RefreshReadIteratorPoolActorState(std::shared_ptr<ReadIteratorPool> const& readIterPool) 
															#line 990 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 990 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : readIterPool(readIterPool)
															#line 1005 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this));

	}
	~RefreshReadIteratorPoolActorState() 
	{
		fdb_probe_actor_destroy("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 991 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_ITERATORS || SERVER_KNOBS->ROCKSDB_READ_RANGE_REUSE_BOUNDED_ITERATORS)
															#line 1020 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			{
															#line 992 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				deleteIteratorsFutureStream = readIterPool->getDeleteIteratorsFutureStream();
															#line 993 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				;
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RefreshReadIteratorPoolActorState();
		static_cast<RefreshReadIteratorPoolActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1007 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<RefreshReadIteratorPoolActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RefreshReadIteratorPoolActorState(); static_cast<RefreshReadIteratorPoolActor*>(this)->destroy(); return 0; }
															#line 1054 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<RefreshReadIteratorPoolActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RefreshReadIteratorPoolActorState();
		static_cast<RefreshReadIteratorPoolActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 995 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->ROCKSDB_READ_RANGE_ITERATOR_REFRESH_TIME);
															#line 994 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<RefreshReadIteratorPoolActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1075 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 998 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		auto __when_expr_1 = deleteIteratorsFutureStream;
															#line 1079 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<Void>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<Void>>::value, "invalid type");
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.pop(), loopDepth); };
		static_cast<RefreshReadIteratorPoolActor*>(this)->actor_wait_state = 1;
															#line 995 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RefreshReadIteratorPoolActor, 0, Void >*>(static_cast<RefreshReadIteratorPoolActor*>(this)));
															#line 998 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< RefreshReadIteratorPoolActor, 1, Void >*>(static_cast<RefreshReadIteratorPoolActor*>(this)));
															#line 1087 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 996 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		readIterPool->refreshIterators();
															#line 1102 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 996 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		readIterPool->refreshIterators();
															#line 1111 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 1001 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(0.0);
															#line 1001 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<RefreshReadIteratorPoolActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1122 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2when1(__when_expr_2.get(), loopDepth); };
		static_cast<RefreshReadIteratorPoolActor*>(this)->actor_wait_state = 2;
															#line 1001 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RefreshReadIteratorPoolActor, 2, Void >*>(static_cast<RefreshReadIteratorPoolActor*>(this)));
															#line 1127 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 1001 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delay(0.0);
															#line 1001 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<RefreshReadIteratorPoolActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1138 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2when1(__when_expr_2.get(), loopDepth); };
		static_cast<RefreshReadIteratorPoolActor*>(this)->actor_wait_state = 2;
															#line 1001 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RefreshReadIteratorPoolActor, 2, Void >*>(static_cast<RefreshReadIteratorPoolActor*>(this)));
															#line 1143 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RefreshReadIteratorPoolActor*>(this)->actor_wait_state > 0) static_cast<RefreshReadIteratorPoolActor*>(this)->actor_wait_state = 0;
		static_cast<RefreshReadIteratorPoolActor*>(this)->ActorCallback< RefreshReadIteratorPoolActor, 0, Void >::remove();
		static_cast<RefreshReadIteratorPoolActor*>(this)->ActorSingleCallback< RefreshReadIteratorPoolActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RefreshReadIteratorPoolActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshReadIteratorPoolActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RefreshReadIteratorPoolActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshReadIteratorPoolActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RefreshReadIteratorPoolActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshReadIteratorPoolActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1when2cont1(Void const& _,int loopDepth) 
	{
															#line 1002 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		readIterPool->refreshIterators();
															#line 1216 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2cont1(Void && _,int loopDepth) 
	{
															#line 1002 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		readIterPool->refreshIterators();
															#line 1225 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1when2cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1when2cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RefreshReadIteratorPoolActor*>(this)->actor_wait_state > 0) static_cast<RefreshReadIteratorPoolActor*>(this)->actor_wait_state = 0;
		static_cast<RefreshReadIteratorPoolActor*>(this)->ActorCallback< RefreshReadIteratorPoolActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RefreshReadIteratorPoolActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshReadIteratorPoolActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1when2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RefreshReadIteratorPoolActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshReadIteratorPoolActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1when2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RefreshReadIteratorPoolActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshReadIteratorPoolActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< RefreshReadIteratorPoolActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshReadIteratorPoolActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< RefreshReadIteratorPoolActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshReadIteratorPoolActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< RefreshReadIteratorPoolActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshReadIteratorPoolActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 990 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::shared_ptr<ReadIteratorPool> readIterPool;
															#line 992 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	ThreadFutureStream<Void> deleteIteratorsFutureStream;
															#line 1366 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via refreshReadIteratorPool()
															#line 990 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class RefreshReadIteratorPoolActor final : public Actor<Void>, public ActorCallback< RefreshReadIteratorPoolActor, 0, Void >, public ActorSingleCallback< RefreshReadIteratorPoolActor, 1, Void >, public ActorCallback< RefreshReadIteratorPoolActor, 2, Void >, public FastAllocated<RefreshReadIteratorPoolActor>, public RefreshReadIteratorPoolActorState<RefreshReadIteratorPoolActor> {
															#line 1371 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<RefreshReadIteratorPoolActor>::operator new;
	using FastAllocated<RefreshReadIteratorPoolActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14886284252552229888UL, 14799471197600244480UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RefreshReadIteratorPoolActor, 0, Void >;
friend struct ActorSingleCallback< RefreshReadIteratorPoolActor, 1, Void >;
friend struct ActorCallback< RefreshReadIteratorPoolActor, 2, Void >;
															#line 990 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	RefreshReadIteratorPoolActor(std::shared_ptr<ReadIteratorPool> const& readIterPool) 
															#line 1390 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RefreshReadIteratorPoolActorState<RefreshReadIteratorPoolActor>(readIterPool),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7507560516737225728UL, 12616974376578173952UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshReadIteratorPoolActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("refreshReadIteratorPool");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("refreshReadIteratorPool", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RefreshReadIteratorPoolActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RefreshReadIteratorPoolActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 990 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
[[nodiscard]] Future<Void> refreshReadIteratorPool( std::shared_ptr<ReadIteratorPool> const& readIterPool ) {
															#line 990 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new RefreshReadIteratorPoolActor(readIterPool));
															#line 1423 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}

#line 1009 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

															#line 1428 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via flowLockLogger()
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class FlowLockLoggerActor>
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowLockLoggerActorState {
															#line 1434 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowLockLoggerActorState(UID const& id,const FlowLock* const& readLock,const FlowLock* const& fetchLock) 
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : id(id),
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readLock(readLock),
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   fetchLock(fetchLock)
															#line 1445 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowLockLogger", reinterpret_cast<unsigned long>(this));

	}
	~FlowLockLoggerActorState() 
	{
		fdb_probe_actor_destroy("flowLockLogger", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1011 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			;
															#line 1460 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowLockLoggerActorState();
		static_cast<FlowLockLoggerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1012 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->ROCKSDB_METRICS_DELAY);
															#line 1012 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowLockLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1492 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<FlowLockLoggerActor*>(this)->actor_wait_state = 1;
															#line 1012 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowLockLoggerActor, 0, Void >*>(static_cast<FlowLockLoggerActor*>(this)));
															#line 1497 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1013 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		TraceEvent e("RocksDBFlowLock", id);
															#line 1014 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("ReadAvailable", readLock->available());
															#line 1015 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("ReadActivePermits", readLock->activePermits());
															#line 1016 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("ReadWaiters", readLock->waiters());
															#line 1017 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("FetchAvailable", fetchLock->available());
															#line 1018 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("FetchActivePermits", fetchLock->activePermits());
															#line 1019 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("FetchWaiters", fetchLock->waiters());
															#line 1518 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1013 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		TraceEvent e("RocksDBFlowLock", id);
															#line 1014 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("ReadAvailable", readLock->available());
															#line 1015 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("ReadActivePermits", readLock->activePermits());
															#line 1016 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("ReadWaiters", readLock->waiters());
															#line 1017 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("FetchAvailable", fetchLock->available());
															#line 1018 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("FetchActivePermits", fetchLock->activePermits());
															#line 1019 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("FetchWaiters", fetchLock->waiters());
															#line 1539 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowLockLoggerActor*>(this)->actor_wait_state > 0) static_cast<FlowLockLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<FlowLockLoggerActor*>(this)->ActorCallback< FlowLockLoggerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowLockLoggerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowLockLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowLockLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowLockLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowLockLoggerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowLockLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowLockLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowLockLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowLockLoggerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowLockLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowLockLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowLockLogger", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UID id;
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	const FlowLock* readLock;
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	const FlowLock* fetchLock;
															#line 1625 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowLockLogger()
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowLockLoggerActor final : public Actor<Void>, public ActorCallback< FlowLockLoggerActor, 0, Void >, public FastAllocated<FlowLockLoggerActor>, public FlowLockLoggerActorState<FlowLockLoggerActor> {
															#line 1630 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowLockLoggerActor>::operator new;
	using FastAllocated<FlowLockLoggerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6242678241900532992UL, 13289868256408068096UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowLockLoggerActor, 0, Void >;
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowLockLoggerActor(UID const& id,const FlowLock* const& readLock,const FlowLock* const& fetchLock) 
															#line 1647 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowLockLoggerActorState<FlowLockLoggerActor>(id, readLock, fetchLock),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowLockLogger", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12599808689398097408UL, 15182406346863084288UL);
		ActorExecutionContextHelper __helper(static_cast<FlowLockLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowLockLogger");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowLockLogger", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowLockLoggerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
[[nodiscard]] Future<Void> flowLockLogger( UID const& id, const FlowLock* const& readLock, const FlowLock* const& fetchLock ) {
															#line 1010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new FlowLockLoggerActor(id, readLock, fetchLock));
															#line 1679 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}

#line 1022 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

															#line 1684 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via manualFlush()
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class ManualFlushActor>
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class ManualFlushActorState {
															#line 1690 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	ManualFlushActorState(UID const& id,rocksdb::DB* const& db,std::shared_ptr<SharedRocksDBState> const& sharedState,CF const& cf) 
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : id(id),
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   db(db),
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   sharedState(sharedState),
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   cf(cf)
															#line 1703 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("manualFlush", reinterpret_cast<unsigned long>(this));

	}
	~ManualFlushActorState() 
	{
		fdb_probe_actor_destroy("manualFlush", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1024 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (SERVER_KNOBS->ROCKSDB_MANUAL_FLUSH_TIME_INTERVAL)
															#line 1718 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			{
															#line 1025 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				fOptions = sharedState->getFlushOptions();
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				waitTime = SERVER_KNOBS->ROCKSDB_MANUAL_FLUSH_TIME_INTERVAL;
															#line 1027 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				currTime = 0;
															#line 1028 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				timeElapsedAfterLastFlush = 0;
															#line 1029 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				;
															#line 1730 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ManualFlushActorState();
		static_cast<ManualFlushActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1050 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<ManualFlushActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ManualFlushActorState(); static_cast<ManualFlushActor*>(this)->destroy(); return 0; }
															#line 1758 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<ManualFlushActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ManualFlushActorState();
		static_cast<ManualFlushActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1030 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(waitTime);
															#line 1030 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<ManualFlushActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1779 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ManualFlushActor*>(this)->actor_wait_state = 1;
															#line 1030 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ManualFlushActor, 0, Void >*>(static_cast<ManualFlushActor*>(this)));
															#line 1784 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1032 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		currTime = now();
															#line 1033 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		timeElapsedAfterLastFlush = currTime - sharedState->getLastFlushTime();
															#line 1034 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (timeElapsedAfterLastFlush >= SERVER_KNOBS->ROCKSDB_MANUAL_FLUSH_TIME_INTERVAL)
															#line 1797 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 1035 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			db->Flush(fOptions, cf);
															#line 1036 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			waitTime = SERVER_KNOBS->ROCKSDB_MANUAL_FLUSH_TIME_INTERVAL;
															#line 1037 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			TraceEvent("RocksDBManualFlush", id).detail("TimeElapsedAfterLastFlush", timeElapsedAfterLastFlush);
															#line 1805 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
		else
		{
															#line 1039 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			waitTime = SERVER_KNOBS->ROCKSDB_MANUAL_FLUSH_TIME_INTERVAL - timeElapsedAfterLastFlush;
															#line 1811 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 1045 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (g_network->isSimulated() && !SERVER_KNOBS->ROCKSDB_ENABLE_NONDETERMINISM)
															#line 1815 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 1046 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			waitTime = SERVER_KNOBS->ROCKSDB_MANUAL_FLUSH_TIME_INTERVAL;
															#line 1819 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1032 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		currTime = now();
															#line 1033 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		timeElapsedAfterLastFlush = currTime - sharedState->getLastFlushTime();
															#line 1034 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (timeElapsedAfterLastFlush >= SERVER_KNOBS->ROCKSDB_MANUAL_FLUSH_TIME_INTERVAL)
															#line 1833 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 1035 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			db->Flush(fOptions, cf);
															#line 1036 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			waitTime = SERVER_KNOBS->ROCKSDB_MANUAL_FLUSH_TIME_INTERVAL;
															#line 1037 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			TraceEvent("RocksDBManualFlush", id).detail("TimeElapsedAfterLastFlush", timeElapsedAfterLastFlush);
															#line 1841 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
		else
		{
															#line 1039 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			waitTime = SERVER_KNOBS->ROCKSDB_MANUAL_FLUSH_TIME_INTERVAL - timeElapsedAfterLastFlush;
															#line 1847 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 1045 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (g_network->isSimulated() && !SERVER_KNOBS->ROCKSDB_ENABLE_NONDETERMINISM)
															#line 1851 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 1046 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			waitTime = SERVER_KNOBS->ROCKSDB_MANUAL_FLUSH_TIME_INTERVAL;
															#line 1855 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ManualFlushActor*>(this)->actor_wait_state > 0) static_cast<ManualFlushActor*>(this)->actor_wait_state = 0;
		static_cast<ManualFlushActor*>(this)->ActorCallback< ManualFlushActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ManualFlushActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("manualFlush", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ManualFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("manualFlush", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ManualFlushActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("manualFlush", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ManualFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("manualFlush", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ManualFlushActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("manualFlush", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ManualFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("manualFlush", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UID id;
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	rocksdb::DB* db;
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::shared_ptr<SharedRocksDBState> sharedState;
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	CF cf;
															#line 1025 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	rocksdb::FlushOptions fOptions;
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	double waitTime;
															#line 1027 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	double currTime;
															#line 1028 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	int timeElapsedAfterLastFlush;
															#line 1952 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via manualFlush()
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class ManualFlushActor final : public Actor<Void>, public ActorCallback< ManualFlushActor, 0, Void >, public FastAllocated<ManualFlushActor>, public ManualFlushActorState<ManualFlushActor> {
															#line 1957 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<ManualFlushActor>::operator new;
	using FastAllocated<ManualFlushActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16229419294587073792UL, 3048422244122182400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ManualFlushActor, 0, Void >;
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	ManualFlushActor(UID const& id,rocksdb::DB* const& db,std::shared_ptr<SharedRocksDBState> const& sharedState,CF const& cf) 
															#line 1974 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ManualFlushActorState<ManualFlushActor>(id, db, sharedState, cf),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("manualFlush", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3091281260415529472UL, 6672087945880837120UL);
		ActorExecutionContextHelper __helper(static_cast<ManualFlushActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("manualFlush");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("manualFlush", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ManualFlushActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
[[nodiscard]] Future<Void> manualFlush( UID const& id, rocksdb::DB* const& db, std::shared_ptr<SharedRocksDBState> const& sharedState, CF const& cf ) {
															#line 1023 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new ManualFlushActor(id, db, sharedState, cf));
															#line 2006 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}

#line 1052 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

															#line 2011 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via rocksDBMetricLogger()
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class RocksDBMetricLoggerActor>
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class RocksDBMetricLoggerActorState {
															#line 2017 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	RocksDBMetricLoggerActorState(UID const& id,std::shared_ptr<SharedRocksDBState> const& sharedState,std::shared_ptr<rocksdb::Statistics> const& statistics,std::shared_ptr<PerfContextMetrics> const& perfContextMetrics,rocksdb::DB* const& db,std::shared_ptr<ReadIteratorPool> const& readIterPool,Counters* const& counters,CF const& cf) 
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : id(id),
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   sharedState(sharedState),
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   statistics(statistics),
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   perfContextMetrics(perfContextMetrics),
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   db(db),
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readIterPool(readIterPool),
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   counters(counters),
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   cf(cf),
															#line 1061 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   tickerStats({ { "StallMicros", rocksdb::STALL_MICROS, 0 }, { "BytesRead", rocksdb::BYTES_READ, 0 }, { "IterBytesRead", rocksdb::ITER_BYTES_READ, 0 }, { "BytesWritten", rocksdb::BYTES_WRITTEN, 0 }, { "BlockCacheMisses", rocksdb::BLOCK_CACHE_MISS, 0 }, { "BlockCacheHits", rocksdb::BLOCK_CACHE_HIT, 0 }, { "BlockCacheIndexMisses", rocksdb::BLOCK_CACHE_INDEX_MISS, 0 }, { "BlockCacheIndexHits", rocksdb::BLOCK_CACHE_INDEX_HIT, 0 }, { "BlockCacheFilterMisses", rocksdb::BLOCK_CACHE_FILTER_MISS, 0 }, { "BlockCacheFilterHits", rocksdb::BLOCK_CACHE_FILTER_HIT, 0 }, { "BlockCacheDataMisses", rocksdb::BLOCK_CACHE_DATA_MISS, 0 }, { "BlockCacheDataHits", rocksdb::BLOCK_CACHE_DATA_HIT, 0 }, { "BlockCacheBytesRead", rocksdb::BLOCK_CACHE_BYTES_READ, 0 }, { "BlockCacheBytesWrite", rocksdb::BLOCK_CACHE_BYTES_WRITE, 0 }, { "BloomFilterUseful", rocksdb::BLOOM_FILTER_USEFUL, 0 }, { "BloomFilterFullPositive", rocksdb::BLOOM_FILTER_FULL_POSITIVE, 0 }, { "BloomFilterTruePositive", rocksdb::BLOOM_FILTER_FULL_TRUE_POSITIVE, 0 }, { "MemtableHit", rocksdb::MEMTABLE_HIT, 0 }, { "MemtableMiss", rocksdb::MEMTABLE_MISS, 0 }, { "GetHitL0", rocksdb::GET_HIT_L0, 0 }, { "GetHitL1", rocksdb::GET_HIT_L1, 0 }, { "GetHitL2AndUp", rocksdb::GET_HIT_L2_AND_UP, 0 }, { "CountKeysWritten", rocksdb::NUMBER_KEYS_WRITTEN, 0 }, { "CountKeysRead", rocksdb::NUMBER_KEYS_READ, 0 }, { "CountDBSeek", rocksdb::NUMBER_DB_SEEK, 0 }, { "CountDBNext", rocksdb::NUMBER_DB_NEXT, 0 }, { "CountDBPrev", rocksdb::NUMBER_DB_PREV, 0 }, { "BloomFilterPrefixChecked", rocksdb::BLOOM_FILTER_PREFIX_CHECKED, 0 }, { "BloomFilterPrefixUseful", rocksdb::BLOOM_FILTER_PREFIX_USEFUL, 0 }, { "CountWalFileSyncs", rocksdb::WAL_FILE_SYNCED, 0 }, { "CountWalFileBytes", rocksdb::WAL_FILE_BYTES, 0 }, { "CompactReadBytes", rocksdb::COMPACT_READ_BYTES, 0 }, { "CompactReadBytesMarked", rocksdb::COMPACT_READ_BYTES_MARKED, 0 }, { "CompactReadBytesPeriodic", rocksdb::COMPACT_READ_BYTES_PERIODIC, 0 }, { "CompactReadBytesTtl", rocksdb::COMPACT_READ_BYTES_TTL, 0 }, { "CompactWriteBytes", rocksdb::COMPACT_WRITE_BYTES, 0 }, { "CompactWriteBytesMarked", rocksdb::COMPACT_WRITE_BYTES_MARKED, 0 }, { "CompactWriteBytesPeriodic", rocksdb::COMPACT_WRITE_BYTES_PERIODIC, 0 }, { "CompactWriteBytesTtl", rocksdb::COMPACT_WRITE_BYTES_TTL, 0 }, { "FlushWriteBytes", rocksdb::FLUSH_WRITE_BYTES, 0 }, { "CountBlocksCompressed", rocksdb::NUMBER_BLOCK_COMPRESSED, 0 }, { "CountBlocksDecompressed", rocksdb::NUMBER_BLOCK_DECOMPRESSED, 0 }, { "RowCacheHit", rocksdb::ROW_CACHE_HIT, 0 }, { "RowCacheMiss", rocksdb::ROW_CACHE_MISS, 0 }, { "CountIterSkippedKeys", rocksdb::NUMBER_ITER_SKIP, 0 }, { "NoIteratorCreated", rocksdb::NO_ITERATOR_CREATED, 0 }, { "NoIteratorDeleted", rocksdb::NO_ITERATOR_DELETED, 0 }, }),
															#line 1118 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   histogramStats({ { "CompactionTime", rocksdb::COMPACTION_TIME }, { "CompactionCPUTime", rocksdb::COMPACTION_CPU_TIME }, { "CompressionTimeNanos", rocksdb::COMPRESSION_TIMES_NANOS }, { "DecompressionTimeNanos", rocksdb::DECOMPRESSION_TIMES_NANOS }, { "WriteStall", rocksdb::WRITE_STALL }, }),
															#line 1133 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   intPropertyStats({ { "NumImmutableMemtables", rocksdb::DB::Properties::kNumImmutableMemTable }, { "NumImmutableMemtablesFlushed", rocksdb::DB::Properties::kNumImmutableMemTableFlushed }, { "IsMemtableFlushPending", rocksdb::DB::Properties::kMemTableFlushPending }, { "NumRunningFlushes", rocksdb::DB::Properties::kNumRunningFlushes }, { "IsCompactionPending", rocksdb::DB::Properties::kCompactionPending }, { "NumRunningCompactions", rocksdb::DB::Properties::kNumRunningCompactions }, { "CumulativeBackgroundErrors", rocksdb::DB::Properties::kBackgroundErrors }, { "CurrentSizeActiveMemtable", rocksdb::DB::Properties::kCurSizeActiveMemTable }, { "AllMemtablesBytes", rocksdb::DB::Properties::kCurSizeAllMemTables }, { "ActiveMemtableBytes", rocksdb::DB::Properties::kSizeAllMemTables }, { "CountEntriesActiveMemtable", rocksdb::DB::Properties::kNumEntriesActiveMemTable }, { "CountEntriesImmutMemtables", rocksdb::DB::Properties::kNumEntriesImmMemTables }, { "CountDeletesActiveMemtable", rocksdb::DB::Properties::kNumDeletesActiveMemTable }, { "CountDeletesImmutMemtables", rocksdb::DB::Properties::kNumDeletesImmMemTables }, { "EstimatedCountKeys", rocksdb::DB::Properties::kEstimateNumKeys }, { "EstimateSstReaderBytes", rocksdb::DB::Properties::kEstimateTableReadersMem }, { "CountActiveSnapshots", rocksdb::DB::Properties::kNumSnapshots }, { "OldestSnapshotTime", rocksdb::DB::Properties::kOldestSnapshotTime }, { "CountLiveVersions", rocksdb::DB::Properties::kNumLiveVersions }, { "EstimateLiveDataSize", rocksdb::DB::Properties::kEstimateLiveDataSize }, { "BaseLevel", rocksdb::DB::Properties::kBaseLevel }, { "EstPendCompactBytes", rocksdb::DB::Properties::kEstimatePendingCompactionBytes }, { "BlockCacheUsage", rocksdb::DB::Properties::kBlockCacheUsage }, { "BlockCachePinnedUsage", rocksdb::DB::Properties::kBlockCachePinnedUsage }, { "LiveSstFilesSize", rocksdb::DB::Properties::kLiveSstFilesSize }, { "ObsoleteSstFilesSize", rocksdb::DB::Properties::kObsoleteSstFilesSize }, }),
															#line 1162 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   strPropertyStats({ { "LevelStats", rocksdb::DB::Properties::kLevelStats }, { "BlockCacheEntryStats", rocksdb::DB::Properties::kBlockCacheEntryStats }, }),
															#line 1167 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   levelStrPropertyStats({ { "CompressionRatioAtLevel", rocksdb::DB::Properties::kCompressionRatioAtLevelPrefix }, }),
															#line 1171 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readIteratorPoolStats({ { "NumReadIteratorsCreated", 0 }, { "NumTimesReadIteratorsReused", 0 }, }),
															#line 1176 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   rocksdbMetricsTrackingKey(id.toString() + "/RocksDBMetrics")
															#line 2052 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("rocksDBMetricLogger", reinterpret_cast<unsigned long>(this));

	}
	~RocksDBMetricLoggerActorState() 
	{
		fdb_probe_actor_destroy("rocksDBMetricLogger", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1177 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			;
															#line 2067 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RocksDBMetricLoggerActorState();
		static_cast<RocksDBMetricLoggerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1248 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<RocksDBMetricLoggerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RocksDBMetricLoggerActorState(); static_cast<RocksDBMetricLoggerActor*>(this)->destroy(); return 0; }
															#line 2090 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<RocksDBMetricLoggerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RocksDBMetricLoggerActorState();
		static_cast<RocksDBMetricLoggerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1178 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->ROCKSDB_METRICS_DELAY);
															#line 1178 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<RocksDBMetricLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2111 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RocksDBMetricLoggerActor*>(this)->actor_wait_state = 1;
															#line 1178 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RocksDBMetricLoggerActor, 0, Void >*>(static_cast<RocksDBMetricLoggerActor*>(this)));
															#line 2116 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 1179 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (sharedState->isClosing())
															#line 2138 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1182 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		TraceEvent e("RocksDBMetrics", id);
															#line 1183 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.trackLatest(rocksdbMetricsTrackingKey);
															#line 1185 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		uint64_t stat;
															#line 1186 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		for( auto& [name, ticker, cum] : tickerStats ) {
															#line 1187 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			stat = statistics->getTickerCount(ticker);
															#line 1188 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			e.detail(name, stat - cum);
															#line 1189 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			cum = stat;
															#line 2156 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 1194 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (SERVER_KNOBS->ROCKSDB_STATS_LEVEL > rocksdb::kExceptHistogramOrTimers)
															#line 2160 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 1195 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			for( auto& [name, histogram] : histogramStats ) {
															#line 1196 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				rocksdb::HistogramData histogram_data;
															#line 1197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				statistics->histogramData(histogram, &histogram_data);
															#line 1198 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				e.detail(format("%s%s", name, "P95"), histogram_data.percentile95);
															#line 1199 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				e.detail(format("%s%s", name, "P99"), histogram_data.percentile99);
															#line 2172 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			}
		}
															#line 1203 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		for( const auto& [name, property] : intPropertyStats ) {
															#line 1204 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			stat = 0;
															#line 1206 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			ASSERT(db->GetAggregatedIntProperty(property, &stat));
															#line 1207 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			e.detail(name, stat);
															#line 2183 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 1210 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::string propValue;
															#line 1211 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		for( const auto& [name, property] : strPropertyStats ) {
															#line 1212 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			propValue = "";
															#line 1213 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			ASSERT(db->GetProperty(cf, property, &propValue));
															#line 1214 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			e.detail(name, propValue);
															#line 2195 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 1217 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		rocksdb::ColumnFamilyMetaData cf_meta_data;
															#line 1218 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		db->GetColumnFamilyMetaData(cf, &cf_meta_data);
															#line 1219 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		int numLevels = static_cast<int>(cf_meta_data.levels.size());
															#line 1220 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::string levelProp;
															#line 1221 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		for( const auto& [name, property] : levelStrPropertyStats ) {
															#line 1222 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			levelProp = "";
															#line 1223 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			for(int level = 0;level < numLevels;level++) {
															#line 1224 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				propValue = "";
															#line 1225 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				ASSERT(db->GetProperty(cf, property + std::to_string(level), &propValue));
															#line 1226 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				levelProp += std::to_string(level) + ":" + propValue + (level == numLevels - 1 ? "" : ",");
															#line 2217 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			}
															#line 1228 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			e.detail(name, levelProp);
															#line 2221 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 1231 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		stat = readIterPool->numReadIteratorsCreated();
															#line 1232 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("NumReadIteratorsCreated", stat - readIteratorPoolStats["NumReadIteratorsCreated"]);
															#line 1233 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		readIteratorPoolStats["NumReadIteratorsCreated"] = stat;
															#line 1235 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		stat = readIterPool->numTimesReadIteratorsReused();
															#line 1236 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("NumTimesReadIteratorsReused", stat - readIteratorPoolStats["NumTimesReadIteratorsReused"]);
															#line 1237 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		readIteratorPoolStats["NumTimesReadIteratorsReused"] = stat;
															#line 1239 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("BlockCacheSize", SERVER_KNOBS->ROCKSDB_BLOCK_CACHE_SIZE);
															#line 1241 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		counters->cc.logToTraceEvent(e);
															#line 1243 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (SERVER_KNOBS->ROCKSDB_PERFCONTEXT_ENABLE)
															#line 2241 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 1244 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			perfContextMetrics->log(true);
															#line 2245 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 1179 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (sharedState->isClosing())
															#line 2255 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1182 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		TraceEvent e("RocksDBMetrics", id);
															#line 1183 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.trackLatest(rocksdbMetricsTrackingKey);
															#line 1185 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		uint64_t stat;
															#line 1186 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		for( auto& [name, ticker, cum] : tickerStats ) {
															#line 1187 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			stat = statistics->getTickerCount(ticker);
															#line 1188 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			e.detail(name, stat - cum);
															#line 1189 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			cum = stat;
															#line 2273 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 1194 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (SERVER_KNOBS->ROCKSDB_STATS_LEVEL > rocksdb::kExceptHistogramOrTimers)
															#line 2277 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 1195 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			for( auto& [name, histogram] : histogramStats ) {
															#line 1196 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				rocksdb::HistogramData histogram_data;
															#line 1197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				statistics->histogramData(histogram, &histogram_data);
															#line 1198 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				e.detail(format("%s%s", name, "P95"), histogram_data.percentile95);
															#line 1199 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				e.detail(format("%s%s", name, "P99"), histogram_data.percentile99);
															#line 2289 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			}
		}
															#line 1203 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		for( const auto& [name, property] : intPropertyStats ) {
															#line 1204 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			stat = 0;
															#line 1206 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			ASSERT(db->GetAggregatedIntProperty(property, &stat));
															#line 1207 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			e.detail(name, stat);
															#line 2300 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 1210 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::string propValue;
															#line 1211 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		for( const auto& [name, property] : strPropertyStats ) {
															#line 1212 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			propValue = "";
															#line 1213 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			ASSERT(db->GetProperty(cf, property, &propValue));
															#line 1214 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			e.detail(name, propValue);
															#line 2312 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 1217 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		rocksdb::ColumnFamilyMetaData cf_meta_data;
															#line 1218 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		db->GetColumnFamilyMetaData(cf, &cf_meta_data);
															#line 1219 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		int numLevels = static_cast<int>(cf_meta_data.levels.size());
															#line 1220 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::string levelProp;
															#line 1221 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		for( const auto& [name, property] : levelStrPropertyStats ) {
															#line 1222 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			levelProp = "";
															#line 1223 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			for(int level = 0;level < numLevels;level++) {
															#line 1224 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				propValue = "";
															#line 1225 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				ASSERT(db->GetProperty(cf, property + std::to_string(level), &propValue));
															#line 1226 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				levelProp += std::to_string(level) + ":" + propValue + (level == numLevels - 1 ? "" : ",");
															#line 2334 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			}
															#line 1228 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			e.detail(name, levelProp);
															#line 2338 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 1231 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		stat = readIterPool->numReadIteratorsCreated();
															#line 1232 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("NumReadIteratorsCreated", stat - readIteratorPoolStats["NumReadIteratorsCreated"]);
															#line 1233 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		readIteratorPoolStats["NumReadIteratorsCreated"] = stat;
															#line 1235 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		stat = readIterPool->numTimesReadIteratorsReused();
															#line 1236 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("NumTimesReadIteratorsReused", stat - readIteratorPoolStats["NumTimesReadIteratorsReused"]);
															#line 1237 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		readIteratorPoolStats["NumTimesReadIteratorsReused"] = stat;
															#line 1239 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		e.detail("BlockCacheSize", SERVER_KNOBS->ROCKSDB_BLOCK_CACHE_SIZE);
															#line 1241 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		counters->cc.logToTraceEvent(e);
															#line 1243 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (SERVER_KNOBS->ROCKSDB_PERFCONTEXT_ENABLE)
															#line 2358 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 1244 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			perfContextMetrics->log(true);
															#line 2362 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RocksDBMetricLoggerActor*>(this)->actor_wait_state > 0) static_cast<RocksDBMetricLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<RocksDBMetricLoggerActor*>(this)->ActorCallback< RocksDBMetricLoggerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RocksDBMetricLoggerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("rocksDBMetricLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RocksDBMetricLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rocksDBMetricLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RocksDBMetricLoggerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("rocksDBMetricLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RocksDBMetricLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rocksDBMetricLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RocksDBMetricLoggerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("rocksDBMetricLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RocksDBMetricLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("rocksDBMetricLogger", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UID id;
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::shared_ptr<SharedRocksDBState> sharedState;
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::shared_ptr<rocksdb::Statistics> statistics;
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::shared_ptr<PerfContextMetrics> perfContextMetrics;
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	rocksdb::DB* db;
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::shared_ptr<ReadIteratorPool> readIterPool;
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Counters* counters;
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	CF cf;
															#line 1061 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::vector<std::tuple<const char*, uint32_t, uint64_t>> tickerStats;
															#line 1118 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::vector<std::pair<const char*, uint32_t>> histogramStats;
															#line 1133 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::vector<std::pair<const char*, std::string>> intPropertyStats;
															#line 1162 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::vector<std::pair<const char*, std::string>> strPropertyStats;
															#line 1167 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::vector<std::pair<const char*, std::string>> levelStrPropertyStats;
															#line 1171 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::unordered_map<std::string, uint64_t> readIteratorPoolStats;
															#line 1176 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string rocksdbMetricsTrackingKey;
															#line 2473 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via rocksDBMetricLogger()
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class RocksDBMetricLoggerActor final : public Actor<Void>, public ActorCallback< RocksDBMetricLoggerActor, 0, Void >, public FastAllocated<RocksDBMetricLoggerActor>, public RocksDBMetricLoggerActorState<RocksDBMetricLoggerActor> {
															#line 2478 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<RocksDBMetricLoggerActor>::operator new;
	using FastAllocated<RocksDBMetricLoggerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5639661702087893760UL, 1158519769056029184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RocksDBMetricLoggerActor, 0, Void >;
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	RocksDBMetricLoggerActor(UID const& id,std::shared_ptr<SharedRocksDBState> const& sharedState,std::shared_ptr<rocksdb::Statistics> const& statistics,std::shared_ptr<PerfContextMetrics> const& perfContextMetrics,rocksdb::DB* const& db,std::shared_ptr<ReadIteratorPool> const& readIterPool,Counters* const& counters,CF const& cf) 
															#line 2495 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RocksDBMetricLoggerActorState<RocksDBMetricLoggerActor>(id, sharedState, statistics, perfContextMetrics, db, readIterPool, counters, cf),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("rocksDBMetricLogger", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14984653816242247680UL, 2752637896947694080UL);
		ActorExecutionContextHelper __helper(static_cast<RocksDBMetricLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("rocksDBMetricLogger");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("rocksDBMetricLogger", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RocksDBMetricLoggerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
[[nodiscard]] Future<Void> rocksDBMetricLogger( UID const& id, std::shared_ptr<SharedRocksDBState> const& sharedState, std::shared_ptr<rocksdb::Statistics> const& statistics, std::shared_ptr<PerfContextMetrics> const& perfContextMetrics, rocksdb::DB* const& db, std::shared_ptr<ReadIteratorPool> const& readIterPool, Counters* const& counters, CF const& cf ) {
															#line 1053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new RocksDBMetricLoggerActor(id, sharedState, statistics, perfContextMetrics, db, readIterPool, counters, cf));
															#line 2527 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}

#line 1250 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

void logRocksDBError(UID id,
                     const rocksdb::Status& status,
                     const std::string& method,
                     Optional<Severity> sev = Optional<Severity>()) {
	Severity level = sev.present() ? sev.get() : (status.IsTimedOut() ? SevWarn : SevError);
	TraceEvent e(level, "RocksDBError", id);
	e.setMaxFieldLength(10000)
	    .detail("Error", status.ToString())
	    .detail("Method", method)
	    .detail("RocksDBSeverity", status.severity());
	if (status.IsIOError()) {
		e.detail("SubCode", status.subcode());
	}
}

Error statusToError(const rocksdb::Status& s) {
	if (s.IsIOError()) {
		return io_error();
	} else if (s.IsTimedOut()) {
		return transaction_too_old();
	} else {
		return unknown_error();
	}
}

struct RocksDBKeyValueStore : IKeyValueStore {
	struct Writer : IThreadPoolReceiver {
		struct CheckpointAction : TypedAction<Writer, CheckpointAction> {
			CheckpointAction(const CheckpointRequest& request) : request(request) {}

			double getTimeEstimate() const override { return SERVER_KNOBS->COMMIT_TIME_ESTIMATE; }

			const CheckpointRequest request;
			ThreadReturnPromise<CheckpointMetaData> reply;
		};

		struct RestoreAction : TypedAction<Writer, RestoreAction> {
			RestoreAction(const std::string& path, const std::vector<CheckpointMetaData>& checkpoints)
			  : path(path), checkpoints(checkpoints) {}

			double getTimeEstimate() const override { return SERVER_KNOBS->COMMIT_TIME_ESTIMATE; }

			const std::string path;
			const std::vector<CheckpointMetaData> checkpoints;
			ThreadReturnPromise<Void> done;
		};

		explicit Writer(DB& db,
		                CF& cf,
		                UID id,
		                std::shared_ptr<SharedRocksDBState> sharedState,
		                std::shared_ptr<ReadIteratorPool> readIterPool,
		                std::shared_ptr<PerfContextMetrics> perfContextMetrics,
		                int threadIndex,
		                ThreadReturnPromiseStream<std::pair<std::string, double>>* metricPromiseStream)
		  : db(db), cf(cf), id(id), sharedState(sharedState), readIterPool(readIterPool),
		    perfContextMetrics(perfContextMetrics), threadIndex(threadIndex), metricPromiseStream(metricPromiseStream),
		    rateLimiter(SERVER_KNOBS->ROCKSDB_WRITE_RATE_LIMITER_BYTES_PER_SEC > 0
		                    ? rocksdb::NewGenericRateLimiter(
		                          SERVER_KNOBS->ROCKSDB_WRITE_RATE_LIMITER_BYTES_PER_SEC, // rate_bytes_per_sec
		                          100 * 1000, // refill_period_us
		                          SERVER_KNOBS->ROCKSDB_WRITE_RATE_LIMITER_FAIRNESS, // fairness
		                          rocksdb::RateLimiter::Mode::kAllIo,
		                          SERVER_KNOBS->ROCKSDB_WRITE_RATE_LIMITER_AUTO_TUNE)
		                    : nullptr) {
			if (SERVER_KNOBS->ROCKSDB_PERFCONTEXT_ENABLE) {
				// Enable perf context on the same thread with the db thread
				rocksdb::SetPerfLevel(rocksdb::PerfLevel::kEnableTimeExceptForMutex);
				perfContextMetrics->reset();
			}
		}

		void init() override {}

		struct IngestSSTFilesAction : TypedAction<Writer, IngestSSTFilesAction> {
			IngestSSTFilesAction(std::shared_ptr<BulkLoadFileSetKeyMap> localFileSets) : localFileSets(localFileSets) {}

			double getTimeEstimate() const override { return SERVER_KNOBS->COMMIT_TIME_ESTIMATE; }

			std::shared_ptr<BulkLoadFileSetKeyMap> localFileSets;
			ThreadReturnPromise<Void> done;
		};

		void action(IngestSSTFilesAction& a) {
			// Create a list of SST files to ingest
			std::vector<std::string> sstFiles;
			for (const auto& [range, fileSet] : *a.localFileSets) {
				if (fileSet.hasDataFile()) {
					sstFiles.push_back(fileSet.getDataFileFullPath());
				}
			}

			if (sstFiles.empty()) {
				TraceEvent(SevInfo, "RocksDBIngestSSTFilesNoFiles", id);
				a.done.send(Void()); // Nothing to ingest
				return;
			}

			// Configure ingestion options
			rocksdb::IngestExternalFileOptions options;
			options.move_files = true;
			options.verify_checksums_before_ingest = true;
			options.allow_blocking_flush = true;
			// write_global_seqno is default true which means on ingest the SST file is rewritten w/ seqno injected for
			// each KV.

			// Ingest the SST files
			// The default column family parameter is necessary here; w/o it the ingested keyvalues are unreadable
			rocksdb::Status status = db->IngestExternalFile(cf, sstFiles, options);

			if (!status.ok()) {
				logRocksDBError(id, status, "IngestSSTFiles");
				a.done.sendError(statusToError(status));
				return;
			}

			a.done.send(Void());
		}

		struct CompactRangeAction : TypedAction<Writer, CompactRangeAction> {
			CompactRangeAction(KeyRangeRef range) : range(range) {}

			double getTimeEstimate() const override { return SERVER_KNOBS->COMMIT_TIME_ESTIMATE; }

			const KeyRange range;
			ThreadReturnPromise<Void> done;
		};

		void action(CompactRangeAction& a) {
			// Configure compaction options
			rocksdb::CompactRangeOptions options;
			// Force RocksDB to rewrite file to last level
			options.bottommost_level_compaction = rocksdb::BottommostLevelCompaction::kForceOptimized;

			// Convert key range to slices
			auto begin = toSlice(a.range.begin);
			auto end = toSlice(a.range.end);

			// Perform the compaction
			rocksdb::Status status = db->CompactRange(options, cf, &begin, &end);

			if (!status.ok()) {
				logRocksDBError(id, status, "CompactRange");
				a.done.sendError(statusToError(status));
				return;
			}

			a.done.send(Void());
		}

		struct OpenAction : TypedAction<Writer, OpenAction> {
			std::string path;
			ThreadReturnPromise<Void> done;
			Optional<Future<Void>>& metrics;
			const FlowLock* readLock;
			const FlowLock* fetchLock;
			std::shared_ptr<RocksDBErrorListener> errorListener;
			std::shared_ptr<RocksDBEventListener> eventListener;
			std::shared_ptr<double> lastFlushTime;
			Counters& counters;
			OpenAction(std::string path,
			           Optional<Future<Void>>& metrics,
			           const FlowLock* readLock,
			           const FlowLock* fetchLock,
			           std::shared_ptr<RocksDBErrorListener> errorListener,
			           std::shared_ptr<RocksDBEventListener> eventListener,
			           Counters& counters)
			  : path(std::move(path)), metrics(metrics), readLock(readLock), fetchLock(fetchLock),
			    errorListener(errorListener), eventListener(eventListener), counters(counters) {}

			double getTimeEstimate() const override { return SERVER_KNOBS->COMMIT_TIME_ESTIMATE; }
		};
		void action(OpenAction& a) {
			ASSERT(cf == nullptr);

			std::vector<std::string> columnFamilies;
			rocksdb::DBOptions options = sharedState->getDbOptions();
			rocksdb::Status status = rocksdb::DB::ListColumnFamilies(options, a.path, &columnFamilies);
			if (std::find(columnFamilies.begin(), columnFamilies.end(), "default") == columnFamilies.end()) {
				columnFamilies.push_back("default");
			}

			rocksdb::ColumnFamilyOptions cfOptions = sharedState->getCfOptions();
			std::vector<rocksdb::ColumnFamilyDescriptor> descriptors;
			for (const std::string& name : columnFamilies) {
				descriptors.push_back(rocksdb::ColumnFamilyDescriptor{ name, cfOptions });
			}

			options.listeners.push_back(a.errorListener);
			if (SERVER_KNOBS->ROCKSDB_MANUAL_FLUSH_TIME_INTERVAL > 0) {
				options.listeners.push_back(a.eventListener);
			}
			if (SERVER_KNOBS->ROCKSDB_WRITE_RATE_LIMITER_BYTES_PER_SEC > 0) {
				options.rate_limiter = rateLimiter;
			}

			std::vector<rocksdb::ColumnFamilyHandle*> handles;
			status = rocksdb::DB::Open(options, a.path, descriptors, &handles, &db);
			cfHandles.insert(handles.begin(), handles.end());

			if (!status.ok()) {
				logRocksDBError(id, status, "Open");
				a.done.sendError(statusToError(status));
				return;
			}

			for (rocksdb::ColumnFamilyHandle* handle : handles) {
				if (handle->GetName() == SERVER_KNOBS->DEFAULT_FDB_ROCKSDB_COLUMN_FAMILY) {
					cf = handle;
					break;
				}
			}

			if (cf == nullptr) {
				status = db->CreateColumnFamily(cfOptions, SERVER_KNOBS->DEFAULT_FDB_ROCKSDB_COLUMN_FAMILY, &cf);
				cfHandles.insert(cf);
				if (!status.ok()) {
					logRocksDBError(id, status, "Open");
					a.done.sendError(statusToError(status));
				}
			}

			TraceEvent(SevInfo, "RocksDB", id)
			    .detail("Path", a.path)
			    .detail("Method", "Open")
			    .detail("KnobRocksDBWriteRateLimiterBytesPerSec",
			            SERVER_KNOBS->ROCKSDB_WRITE_RATE_LIMITER_BYTES_PER_SEC)
			    .detail("KnobRocksDBWriteRateLimiterAutoTune", SERVER_KNOBS->ROCKSDB_WRITE_RATE_LIMITER_AUTO_TUNE)
			    .detail("ColumnFamily", cf->GetName());
			if (g_network->isSimulated()) {
				// The current thread and main thread are same when the code runs in simulation.
				// blockUntilReady() is getting the thread into deadlock state, so directly calling
				// the metricsLogger.
				if (SERVER_KNOBS->ROCKSDB_ENABLE_NONDETERMINISM) {
					a.metrics = rocksDBMetricLogger(id,
					                                sharedState,
					                                options.statistics,
					                                perfContextMetrics,
					                                db,
					                                readIterPool,
					                                &a.counters,
					                                cf) &&
					            flowLockLogger(id, a.readLock, a.fetchLock) && refreshReadIteratorPool(readIterPool) &&
					            manualFlush(id, db, sharedState, cf);
				} else {
					a.metrics = flowLockLogger(id, a.readLock, a.fetchLock) && refreshReadIteratorPool(readIterPool) &&
					            manualFlush(id, db, sharedState, cf);
				}
			} else {
				onMainThread([&] {
					a.metrics = rocksDBMetricLogger(id,
					                                sharedState,
					                                options.statistics,
					                                perfContextMetrics,
					                                db,
					                                readIterPool,
					                                &a.counters,
					                                cf) &&
					            flowLockLogger(id, a.readLock, a.fetchLock) && refreshReadIteratorPool(readIterPool) &&
					            manualFlush(id, db, sharedState, cf);
					return Future<bool>(true);
				}).blockUntilReady();
			}
			a.done.send(Void());
		}

		struct DeleteVisitor : public rocksdb::WriteBatch::Handler {
			VectorRef<KeyRangeRef>& deletes;
			Arena& arena;

			DeleteVisitor(VectorRef<KeyRangeRef>& deletes, Arena& arena) : deletes(deletes), arena(arena) {}

			rocksdb::Status DeleteRangeCF(uint32_t /*column_family_id*/,
			                              const rocksdb::Slice& begin,
			                              const rocksdb::Slice& end) override {
				KeyRangeRef kr(toStringRef(begin), toStringRef(end));
				deletes.push_back_deep(arena, kr);
				return rocksdb::Status::OK();
			}

			rocksdb::Status PutCF(uint32_t column_family_id,
			                      const rocksdb::Slice& key,
			                      const rocksdb::Slice& value) override {
				return rocksdb::Status::OK();
			}

			rocksdb::Status DeleteCF(uint32_t column_family_id, const rocksdb::Slice& key) override {
				return rocksdb::Status::OK();
			}

			rocksdb::Status SingleDeleteCF(uint32_t column_family_id, const rocksdb::Slice& key) override {
				return rocksdb::Status::OK();
			}

			rocksdb::Status MergeCF(uint32_t column_family_id,
			                        const rocksdb::Slice& key,
			                        const rocksdb::Slice& value) override {
				return rocksdb::Status::OK();
			}
		};

		struct CommitAction : TypedAction<Writer, CommitAction> {
			std::unique_ptr<rocksdb::WriteBatch> batchToCommit;
			ThreadReturnPromise<Void> done;
			double startTime;
			bool getHistograms;
			double getTimeEstimate() const override { return SERVER_KNOBS->COMMIT_TIME_ESTIMATE; }
			CommitAction()
			  : startTime(timer_monotonic()),
			    getHistograms(deterministicRandom()->random01() < SERVER_KNOBS->ROCKSDB_HISTOGRAMS_SAMPLE_RATE) {}
		};
		void action(CommitAction& a) {
			bool doPerfContextMetrics =
			    SERVER_KNOBS->ROCKSDB_PERFCONTEXT_ENABLE &&
			    (deterministicRandom()->random01() < SERVER_KNOBS->ROCKSDB_PERFCONTEXT_SAMPLE_RATE);
			if (doPerfContextMetrics) {
				perfContextMetrics->reset();
			}
			double commitBeginTime = timer_monotonic();
			if (a.getHistograms) {
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_COMMIT_QUEUEWAIT_HISTOGRAM.toString(), commitBeginTime - a.startTime));
			}
			Standalone<VectorRef<KeyRangeRef>> deletes;
			if (SERVER_KNOBS->ROCKSDB_SUGGEST_COMPACT_CLEAR_RANGE) {
				DeleteVisitor dv(deletes, deletes.arena());
				rocksdb::Status s = a.batchToCommit->Iterate(&dv);
				if (!s.ok()) {
					logRocksDBError(id, s, "CommitDeleteVisitor");
					a.done.sendError(statusToError(s));
					return;
				}
				// If there are any range deletes, we should have added them to be deleted.
				ASSERT(!deletes.empty() || !a.batchToCommit->HasDeleteRange());
			}

			rocksdb::WriteOptions options;
			options.sync = !SERVER_KNOBS->ROCKSDB_UNSAFE_AUTO_FSYNC;
			rocksdb::Status s = db->Write(options, a.batchToCommit.get());
			readIterPool->update();

			if (!s.ok()) {
				logRocksDBError(id, s, "Commit");
				a.done.sendError(statusToError(s));
			} else {
				a.done.send(Void());

				if (SERVER_KNOBS->ROCKSDB_SUGGEST_COMPACT_CLEAR_RANGE) {
					double compactRangeBeginTime = a.getHistograms ? timer_monotonic() : 0;
					for (const auto& keyRange : deletes) {
						auto begin = toSlice(keyRange.begin);
						auto end = toSlice(keyRange.end);

						ASSERT(db->SuggestCompactRange(cf, &begin, &end).ok());
					}
					if (a.getHistograms) {
						metricPromiseStream->send(std::make_pair(ROCKSDB_DELETE_COMPACTRANGE_HISTOGRAM.toString(),
						                                         timer_monotonic() - compactRangeBeginTime));
					}
				}
			}
			double currTime = timer_monotonic();
			if (a.getHistograms) {
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_COMMIT_ACTION_HISTOGRAM.toString(), currTime - commitBeginTime));
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_COMMIT_LATENCY_HISTOGRAM.toString(), currTime - a.startTime));
			}
			if (doPerfContextMetrics) {
				perfContextMetrics->set(threadIndex);
			}
		}

		struct CloseAction : TypedAction<Writer, CloseAction> {
			ThreadReturnPromise<Void> done;
			std::string path;
			bool deleteOnClose;
			CloseAction(std::string path, bool deleteOnClose) : path(path), deleteOnClose(deleteOnClose) {}
			double getTimeEstimate() const override { return SERVER_KNOBS->COMMIT_TIME_ESTIMATE; }
		};
		void action(CloseAction& a) {
			readIterPool.reset();
			if (db == nullptr) {
				a.done.send(Void());
				return;
			}
			for (rocksdb::ColumnFamilyHandle* handle : cfHandles) {
				if (handle != nullptr) {
					db->DestroyColumnFamilyHandle(handle);
				}
			}
			cfHandles.clear();
			auto s = db->Close();
			if (!s.ok()) {
				logRocksDBError(id, s, "Close");
			}
			if (a.deleteOnClose) {
				std::set<std::string> columnFamilies{ "default" };
				columnFamilies.insert(SERVER_KNOBS->DEFAULT_FDB_ROCKSDB_COLUMN_FAMILY);
				std::vector<rocksdb::ColumnFamilyDescriptor> descriptors;
				for (const std::string name : columnFamilies) {
					descriptors.push_back(rocksdb::ColumnFamilyDescriptor{ name, sharedState->getCfOptions() });
				}
				s = rocksdb::DestroyDB(a.path, sharedState->getOptions(), descriptors);
				if (!s.ok()) {
					logRocksDBError(id, s, "Destroy");
				} else {
					TraceEvent("RocksDB", id).detail("Path", a.path).detail("Method", "Destroy");
				}
			}
			TraceEvent("RocksDB", id).detail("Path", a.path).detail("Method", "Close");
			a.done.send(Void());
		}

		void action(CheckpointAction& a);

		void action(RestoreAction& a);

		std::shared_ptr<SharedRocksDBState> sharedState;
		DB& db;
		CF& cf;
		std::unordered_set<rocksdb::ColumnFamilyHandle*> cfHandles;
		UID id;
		std::shared_ptr<rocksdb::RateLimiter> rateLimiter;
		std::shared_ptr<ReadIteratorPool> readIterPool;
		std::shared_ptr<PerfContextMetrics> perfContextMetrics;
		int threadIndex;

		// ThreadReturnPromiseStream pair.first stores the histogram name and
		// pair.second stores the corresponding measured latency (seconds)
		ThreadReturnPromiseStream<std::pair<std::string, double>>* metricPromiseStream;
	};

	struct Reader : IThreadPoolReceiver {
		UID id;
		DB& db;
		CF& cf;
		std::shared_ptr<SharedRocksDBState> sharedState;
		double readValueTimeout;
		double readValuePrefixTimeout;
		double readRangeTimeout;
		std::shared_ptr<ReadIteratorPool> readIterPool;
		std::shared_ptr<PerfContextMetrics> perfContextMetrics;
		int threadIndex;
		ThreadReturnPromiseStream<std::pair<std::string, double>>* metricPromiseStream;
		// ThreadReturnPromiseStream pair.first stores the histogram name and
		// pair.second stores the corresponding measured latency (seconds)

		explicit Reader(UID id,
		                DB& db,
		                CF& cf,
		                std::shared_ptr<SharedRocksDBState> sharedState,
		                std::shared_ptr<ReadIteratorPool> readIterPool,
		                std::shared_ptr<PerfContextMetrics> perfContextMetrics,
		                int threadIndex,
		                ThreadReturnPromiseStream<std::pair<std::string, double>>* metricPromiseStream)
		  : id(id), db(db), cf(cf), sharedState(sharedState), readIterPool(readIterPool),
		    perfContextMetrics(perfContextMetrics), metricPromiseStream(metricPromiseStream), threadIndex(threadIndex) {

			readValueTimeout = SERVER_KNOBS->ROCKSDB_READ_VALUE_TIMEOUT;
			readValuePrefixTimeout = SERVER_KNOBS->ROCKSDB_READ_VALUE_PREFIX_TIMEOUT;
			readRangeTimeout = SERVER_KNOBS->ROCKSDB_READ_RANGE_TIMEOUT;

			if (SERVER_KNOBS->ROCKSDB_PERFCONTEXT_ENABLE) {
				// Enable perf context on the same thread with the db thread
				rocksdb::SetPerfLevel(rocksdb::PerfLevel::kEnableTimeExceptForMutex);
				perfContextMetrics->reset();
			}
		}

		void init() override {}

		struct ReadValueAction : TypedAction<Reader, ReadValueAction> {
			Key key;
			ReadType type;
			Optional<UID> debugID;
			double startTime;
			bool getHistograms;
			ThreadReturnPromise<Optional<Value>> result;
			ReadValueAction(KeyRef key, ReadType type, Optional<UID> debugID)
			  : key(key), type(type), debugID(debugID), startTime(timer_monotonic()),
			    getHistograms(deterministicRandom()->random01() < SERVER_KNOBS->ROCKSDB_HISTOGRAMS_SAMPLE_RATE) {}
			double getTimeEstimate() const override { return SERVER_KNOBS->READ_VALUE_TIME_ESTIMATE; }
		};
		void action(ReadValueAction& a) {
			ASSERT(cf != nullptr);
			bool doPerfContextMetrics =
			    SERVER_KNOBS->ROCKSDB_PERFCONTEXT_ENABLE &&
			    (deterministicRandom()->random01() < SERVER_KNOBS->ROCKSDB_PERFCONTEXT_SAMPLE_RATE);
			if (doPerfContextMetrics) {
				perfContextMetrics->reset();
			}
			const double readBeginTime = timer_monotonic();
			if (a.getHistograms) {
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_READVALUE_QUEUEWAIT_HISTOGRAM.toString(), readBeginTime - a.startTime));
			}
			Optional<TraceBatch> traceBatch;
			if (a.debugID.present()) {
				traceBatch = { TraceBatch{} };
				traceBatch.get().addEvent("GetValueDebug", a.debugID.get().first(), "Reader.Before");
			}
			if (shouldThrottle(a.type, a.key) && SERVER_KNOBS->ROCKSDB_SET_READ_TIMEOUT &&
			    readBeginTime - a.startTime > readValueTimeout) {
				TraceEvent(SevWarn, "KVSTimeout", id)
				    .detail("Error", "Read value request timedout")
				    .detail("Method", "ReadValueAction")
				    .detail("TimeoutValue", readValueTimeout);
				a.result.sendError(transaction_too_old());
				return;
			}

			rocksdb::PinnableSlice value;
			rocksdb::ReadOptions readOptions = sharedState->getReadOptions();
			if (shouldThrottle(a.type, a.key) && SERVER_KNOBS->ROCKSDB_SET_READ_TIMEOUT) {
				uint64_t deadlineMircos =
				    db->GetEnv()->NowMicros() + (readValueTimeout - (readBeginTime - a.startTime)) * 1000000;
				std::chrono::seconds deadlineSeconds(deadlineMircos / 1000000);
				readOptions.deadline = std::chrono::duration_cast<std::chrono::microseconds>(deadlineSeconds);
			}

			auto s = db->Get(readOptions, cf, toSlice(a.key), &value);
			if (!s.ok() && !s.IsNotFound()) {
				logRocksDBError(id, s, "ReadValue");
				a.result.sendError(statusToError(s));
				return;
			}

			if (a.debugID.present()) {
				traceBatch.get().addEvent("GetValueDebug", a.debugID.get().first(), "Reader.After");
				traceBatch.get().dump();
			}
			if (s.ok()) {
				a.result.send(Value(toStringRef(value)));
			} else if (s.IsNotFound()) {
				a.result.send(Optional<Value>());
			} else {
				logRocksDBError(id, s, "ReadValue");
				a.result.sendError(statusToError(s));
			}

			const double endTime = timer_monotonic();
			if (a.getHistograms) {
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_READVALUE_ACTION_HISTOGRAM.toString(), endTime - readBeginTime));
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_READVALUE_LATENCY_HISTOGRAM.toString(), endTime - a.startTime));
			}
			if (doPerfContextMetrics) {
				perfContextMetrics->set(threadIndex);
			}
		}

		struct ReadValuePrefixAction : TypedAction<Reader, ReadValuePrefixAction> {
			Key key;
			int maxLength;
			ReadType type;
			Optional<UID> debugID;
			double startTime;
			bool getHistograms;
			ThreadReturnPromise<Optional<Value>> result;
			ReadValuePrefixAction(Key key, int maxLength, ReadType type, Optional<UID> debugID)
			  : key(key), maxLength(maxLength), type(type), debugID(debugID), startTime(timer_monotonic()),
			    getHistograms(deterministicRandom()->random01() < SERVER_KNOBS->ROCKSDB_HISTOGRAMS_SAMPLE_RATE) {}
			double getTimeEstimate() const override { return SERVER_KNOBS->READ_VALUE_TIME_ESTIMATE; }
		};
		void action(ReadValuePrefixAction& a) {
			bool doPerfContextMetrics =
			    SERVER_KNOBS->ROCKSDB_PERFCONTEXT_ENABLE &&
			    (deterministicRandom()->random01() < SERVER_KNOBS->ROCKSDB_PERFCONTEXT_SAMPLE_RATE);
			if (doPerfContextMetrics) {
				perfContextMetrics->reset();
			}
			const double readBeginTime = timer_monotonic();
			if (a.getHistograms) {
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_READPREFIX_QUEUEWAIT_HISTOGRAM.toString(), readBeginTime - a.startTime));
			}
			Optional<TraceBatch> traceBatch;
			if (a.debugID.present()) {
				traceBatch = { TraceBatch{} };
				traceBatch.get().addEvent("GetValuePrefixDebug",
				                          a.debugID.get().first(),
				                          "Reader.Before"); //.detail("TaskID", g_network->getCurrentTask());
			}
			if (shouldThrottle(a.type, a.key) && SERVER_KNOBS->ROCKSDB_SET_READ_TIMEOUT &&
			    readBeginTime - a.startTime > readValuePrefixTimeout) {
				TraceEvent(SevWarn, "KVSTimeout", id)
				    .detail("Error", "Read value prefix request timedout")
				    .detail("Method", "ReadValuePrefixAction")
				    .detail("TimeoutValue", readValuePrefixTimeout);
				a.result.sendError(transaction_too_old());
				return;
			}

			rocksdb::PinnableSlice value;
			rocksdb::ReadOptions readOptions = sharedState->getReadOptions();
			if (shouldThrottle(a.type, a.key) && SERVER_KNOBS->ROCKSDB_SET_READ_TIMEOUT) {
				uint64_t deadlineMircos =
				    db->GetEnv()->NowMicros() + (readValuePrefixTimeout - (readBeginTime - a.startTime)) * 1000000;
				std::chrono::seconds deadlineSeconds(deadlineMircos / 1000000);
				readOptions.deadline = std::chrono::duration_cast<std::chrono::microseconds>(deadlineSeconds);
			}

			auto s = db->Get(readOptions, cf, toSlice(a.key), &value);

			if (a.debugID.present()) {
				traceBatch.get().addEvent("GetValuePrefixDebug",
				                          a.debugID.get().first(),
				                          "Reader.After"); //.detail("TaskID", g_network->getCurrentTask());
				traceBatch.get().dump();
			}
			if (s.ok()) {
				a.result.send(Value(StringRef(reinterpret_cast<const uint8_t*>(value.data()),
				                              std::min(value.size(), size_t(a.maxLength)))));
			} else if (s.IsNotFound()) {
				a.result.send(Optional<Value>());
			} else {
				logRocksDBError(id, s, "ReadValuePrefix");
				a.result.sendError(statusToError(s));
			}
			const double endTime = timer_monotonic();
			if (a.getHistograms) {
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_READPREFIX_ACTION_HISTOGRAM.toString(), endTime - readBeginTime));
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_READPREFIX_LATENCY_HISTOGRAM.toString(), endTime - a.startTime));
			}
			if (doPerfContextMetrics) {
				perfContextMetrics->set(threadIndex);
			}
		}

		struct ReadRangeAction : TypedAction<Reader, ReadRangeAction>, FastAllocated<ReadRangeAction> {
			KeyRange keys;
			int rowLimit, byteLimit;
			ReadType type;
			double startTime;
			bool getHistograms;
			ThreadReturnPromise<RangeResult> result;
			ReadRangeAction(KeyRange keys, int rowLimit, int byteLimit, ReadType type)
			  : keys(keys), rowLimit(rowLimit), byteLimit(byteLimit), type(type), startTime(timer_monotonic()),
			    getHistograms(deterministicRandom()->random01() < SERVER_KNOBS->ROCKSDB_HISTOGRAMS_SAMPLE_RATE) {}
			double getTimeEstimate() const override { return SERVER_KNOBS->READ_RANGE_TIME_ESTIMATE; }
		};
		void action(ReadRangeAction& a) {
			bool doPerfContextMetrics =
			    SERVER_KNOBS->ROCKSDB_PERFCONTEXT_ENABLE &&
			    (deterministicRandom()->random01() < SERVER_KNOBS->ROCKSDB_PERFCONTEXT_SAMPLE_RATE);
			if (doPerfContextMetrics) {
				perfContextMetrics->reset();
			}
			const double readBeginTime = timer_monotonic();
			if (a.getHistograms) {
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_READRANGE_QUEUEWAIT_HISTOGRAM.toString(), readBeginTime - a.startTime));
			}
			if (shouldThrottle(a.type, a.keys.begin) && SERVER_KNOBS->ROCKSDB_SET_READ_TIMEOUT &&
			    readBeginTime - a.startTime > readRangeTimeout) {
				TraceEvent(SevWarn, "KVSTimeout", id)
				    .detail("Error", "Read range request timedout")
				    .detail("Method", "ReadRangeAction")
				    .detail("TimeoutValue", readRangeTimeout);
				a.result.sendError(transaction_too_old());
				return;
			}

			RangeResult result;
			if (a.rowLimit == 0 || a.byteLimit == 0) {
				result.more = false;
				a.result.send(result);
				return;
			}
			int accumulatedBytes = 0;
			rocksdb::Status s;
			if (a.rowLimit >= 0) {
				double iterCreationBeginTime = a.getHistograms ? timer_monotonic() : 0;
				ReadIterator readIter = readIterPool->getIterator(a.keys);
				if (a.getHistograms) {
					metricPromiseStream->send(std::make_pair(ROCKSDB_READRANGE_NEWITERATOR_HISTOGRAM.toString(),
					                                         timer_monotonic() - iterCreationBeginTime));
				}
				auto cursor = readIter.iter;
				cursor->Seek(toSlice(a.keys.begin));
				while (cursor->Valid() && toStringRef(cursor->key()) < a.keys.end) {
					KeyValueRef kv(toStringRef(cursor->key()), toStringRef(cursor->value()));
					accumulatedBytes += sizeof(KeyValueRef) + kv.expectedSize();
					result.push_back_deep(result.arena(), kv);
					// Calling `cursor->Next()` is potentially expensive, so short-circut here just in case.
					if (result.size() >= a.rowLimit || accumulatedBytes >= a.byteLimit) {
						break;
					}
					if (shouldThrottle(a.type, a.keys.begin) && SERVER_KNOBS->ROCKSDB_SET_READ_TIMEOUT &&
					    timer_monotonic() - a.startTime > readRangeTimeout) {
						TraceEvent(SevWarn, "KVSTimeout", id)
						    .detail("Error", "Read range request timedout")
						    .detail("Method", "ReadRangeAction")
						    .detail("TimeoutValue", readRangeTimeout);
						a.result.sendError(transaction_too_old());
						return;
					}
					cursor->Next();
				}
				s = cursor->status();
				readIterPool->returnIterator(readIter);
			} else {
				double iterCreationBeginTime = a.getHistograms ? timer_monotonic() : 0;
				ReadIterator readIter = readIterPool->getIterator(a.keys);
				if (a.getHistograms) {
					metricPromiseStream->send(std::make_pair(ROCKSDB_READRANGE_NEWITERATOR_HISTOGRAM.toString(),
					                                         timer_monotonic() - iterCreationBeginTime));
				}
				auto cursor = readIter.iter;
				cursor->SeekForPrev(toSlice(a.keys.end));
				if (cursor->Valid() && toStringRef(cursor->key()) == a.keys.end) {
					cursor->Prev();
				}
				while (cursor->Valid() && toStringRef(cursor->key()) >= a.keys.begin) {
					KeyValueRef kv(toStringRef(cursor->key()), toStringRef(cursor->value()));
					accumulatedBytes += sizeof(KeyValueRef) + kv.expectedSize();
					result.push_back_deep(result.arena(), kv);
					// Calling `cursor->Prev()` is potentially expensive, so short-circut here just in case.
					if (result.size() >= -a.rowLimit || accumulatedBytes >= a.byteLimit) {
						break;
					}
					if (shouldThrottle(a.type, a.keys.begin) && SERVER_KNOBS->ROCKSDB_SET_READ_TIMEOUT &&
					    timer_monotonic() - a.startTime > readRangeTimeout) {
						TraceEvent(SevWarn, "KVSTimeout", id)
						    .detail("Error", "Read range request timedout")
						    .detail("Method", "ReadRangeAction")
						    .detail("TimeoutValue", readRangeTimeout);
						a.result.sendError(transaction_too_old());
						return;
					}
					cursor->Prev();
				}
				s = cursor->status();
				readIterPool->returnIterator(readIter);
			}

			if (!s.ok()) {
				logRocksDBError(id, s, "ReadRange");
				a.result.sendError(statusToError(s));
				return;
			}
			result.more =
			    (result.size() == a.rowLimit) || (result.size() == -a.rowLimit) || (accumulatedBytes >= a.byteLimit);
			a.result.send(result);

			const double endTime = timer_monotonic();
			if (a.getHistograms) {
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_READ_RANGE_BYTES_RETURNED_HISTOGRAM.toString(), result.logicalSize()));
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_READ_RANGE_KV_PAIRS_RETURNED_HISTOGRAM.toString(), result.size()));
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_READRANGE_ACTION_HISTOGRAM.toString(), endTime - readBeginTime));
				metricPromiseStream->send(
				    std::make_pair(ROCKSDB_READRANGE_LATENCY_HISTOGRAM.toString(), endTime - a.startTime));
			}
			if (doPerfContextMetrics) {
				perfContextMetrics->set(threadIndex);
			}
		}
	};

	explicit RocksDBKeyValueStore(const std::string& path, UID id)
	  : id(id), sharedState(std::make_shared<SharedRocksDBState>(id)), path(path),
	    perfContextMetrics(new PerfContextMetrics()),
	    readIterPool(new ReadIteratorPool(id, db, defaultFdbCF, sharedState)),
	    readSemaphore(SERVER_KNOBS->ROCKSDB_READ_QUEUE_SOFT_MAX),
	    fetchSemaphore(SERVER_KNOBS->ROCKSDB_FETCH_QUEUE_SOFT_MAX),
	    numReadWaiters(SERVER_KNOBS->ROCKSDB_READ_QUEUE_HARD_MAX - SERVER_KNOBS->ROCKSDB_READ_QUEUE_SOFT_MAX),
	    numFetchWaiters(SERVER_KNOBS->ROCKSDB_FETCH_QUEUE_HARD_MAX - SERVER_KNOBS->ROCKSDB_FETCH_QUEUE_SOFT_MAX),
	    errorListener(std::make_shared<RocksDBErrorListener>(id)), errorFuture(errorListener->getFuture()),
	    deletesPerCommitHistogram(Histogram::getHistogram(ROCKSDBSTORAGE_HISTOGRAM_GROUP,
	                                                      ROCKSDB_DELETES_PER_COMMIT_HISTOGRAM,
	                                                      Histogram::Unit::countLinear,
	                                                      0,
	                                                      10000)),
	    deleteRangesPerCommitHistogram(Histogram::getHistogram(ROCKSDBSTORAGE_HISTOGRAM_GROUP,
	                                                           ROCKSDB_DELETE_RANGES_PER_COMMIT_HISTOGRAM,
	                                                           Histogram::Unit::countLinear,
	                                                           0,
	                                                           10000)) {
		eventListener = std::make_shared<RocksDBEventListener>(sharedState);
		// In simluation, run the reader/writer threads as Coro threads (i.e. in the network thread. The storage engine
		// is still multi-threaded as background compaction threads are still present. Reads/writes to disk will also
		// block the network thread in a way that would be unacceptable in production but is a necessary evil here. When
		// performing the reads in background threads in simulation, the event loop thinks there is no work to do and
		// advances time faster than 1 sec/sec. By the time the blocking read actually finishes, simulation has advanced
		// time by more than 5 seconds, so every read fails with a transaction_too_old error. Doing blocking IO on the
		// main thread solves this issue. There are almost certainly better fixes, but my goal was to get a less
		// invasive change merged first and work on a more realistic version if/when we think that would provide
		// substantially more confidence in the correctness.
		// TODO: Adapt the simulation framework to not advance time quickly when background reads/writes are occurring.
		if (g_network->isSimulated()) {
			writeThread = CoroThreadPool::createThreadPool();
			readThreads = CoroThreadPool::createThreadPool();
		} else {
			writeThread = createGenericThreadPool(/*stackSize=*/0, SERVER_KNOBS->ROCKSDB_WRITER_THREAD_PRIORITY);
			readThreads = createGenericThreadPool(/*stackSize=*/0, SERVER_KNOBS->ROCKSDB_READER_THREAD_PRIORITY);
		}
		if (SERVER_KNOBS->ROCKSDB_HISTOGRAMS_SAMPLE_RATE > 0) {
			collection = actorCollection(addActor.getFuture());
			for (int i = 0; i < SERVER_KNOBS->ROCKSDB_READ_PARALLELISM + 1; i++) {
				// ROCKSDB_READ_PARALLELISM readers and 1 writer
				metricPromiseStreams.emplace_back(
				    std::make_unique<ThreadReturnPromiseStream<std::pair<std::string, double>>>());
				addActor.send(updateHistogram(metricPromiseStreams[i]->getFuture()));
			}
		}

		// the writer uses SERVER_KNOBS->ROCKSDB_READ_PARALLELISM as its threadIndex
		// threadIndex is used for metricPromiseStreams and perfContextMetrics
		writeThread->addThread(new Writer(db,
		                                  defaultFdbCF,
		                                  id,
		                                  this->sharedState,
		                                  readIterPool,
		                                  perfContextMetrics,
		                                  SERVER_KNOBS->ROCKSDB_READ_PARALLELISM,
		                                  SERVER_KNOBS->ROCKSDB_HISTOGRAMS_SAMPLE_RATE > 0
		                                      ? metricPromiseStreams[SERVER_KNOBS->ROCKSDB_READ_PARALLELISM].get()
		                                      : nullptr),
		                       "fdb-rocksdb-wr");
		TraceEvent("RocksDBReadThreads", id)
		    .detail("KnobRocksDBReadParallelism", SERVER_KNOBS->ROCKSDB_READ_PARALLELISM);
		for (unsigned i = 0; i < SERVER_KNOBS->ROCKSDB_READ_PARALLELISM; ++i) {
			readThreads->addThread(
			    new Reader(id,
			               db,
			               defaultFdbCF,
			               this->sharedState,
			               readIterPool,
			               perfContextMetrics,
			               i,
			               SERVER_KNOBS->ROCKSDB_HISTOGRAMS_SAMPLE_RATE > 0 ? metricPromiseStreams[i].get() : nullptr),
			    "fdb-rocksdb-re");
		}
	}

																#line 3374 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via errorListenActor()
															#line 2093 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class ErrorListenActorActor>
															#line 2093 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class ErrorListenActorActorState {
															#line 3380 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 2093 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	ErrorListenActorActorState(Future<Void> const& collection) 
															#line 2093 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 2093 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : collection(collection)
															#line 3387 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("errorListenActor", reinterpret_cast<unsigned long>(this));

	}
	~ErrorListenActorActorState() 
	{
		fdb_probe_actor_destroy("errorListenActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 2095 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				StrictFuture<Void> __when_expr_0 = collection;
															#line 2095 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				if (static_cast<ErrorListenActorActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 3405 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ErrorListenActorActor*>(this)->actor_wait_state = 1;
															#line 2095 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ErrorListenActorActor, 0, Void >*>(static_cast<ErrorListenActorActor*>(this)));
															#line 3410 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ErrorListenActorActorState();
		static_cast<ErrorListenActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2099 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 3440 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2096 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(false);
															#line 2097 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		return a_body1Catch2(internal_error(), loopDepth);
															#line 3456 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2096 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(false);
															#line 2097 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		return a_body1Catch2(internal_error(), loopDepth);
															#line 3466 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ErrorListenActorActor*>(this)->actor_wait_state > 0) static_cast<ErrorListenActorActor*>(this)->actor_wait_state = 0;
		static_cast<ErrorListenActorActor*>(this)->ActorCallback< ErrorListenActorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ErrorListenActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("errorListenActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ErrorListenActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("errorListenActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ErrorListenActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("errorListenActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ErrorListenActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("errorListenActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ErrorListenActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("errorListenActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ErrorListenActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("errorListenActor", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2093 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Future<Void> collection;
															#line 3547 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via errorListenActor()
															#line 2093 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class ErrorListenActorActor final : public Actor<Void>, public ActorCallback< ErrorListenActorActor, 0, Void >, public FastAllocated<ErrorListenActorActor>, public ErrorListenActorActorState<ErrorListenActorActor> {
															#line 3552 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<ErrorListenActorActor>::operator new;
	using FastAllocated<ErrorListenActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17652004770603520UL, 12054094274963045376UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ErrorListenActorActor, 0, Void >;
															#line 2093 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	ErrorListenActorActor(Future<Void> const& collection) 
															#line 3569 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ErrorListenActorActorState<ErrorListenActorActor>(collection),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("errorListenActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15076706140325191936UL, 17767563907757718528UL);
		ActorExecutionContextHelper __helper(static_cast<ErrorListenActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("errorListenActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("errorListenActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ErrorListenActorActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2093 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
[[nodiscard]] Future<Void> errorListenActor( Future<Void> const& collection ) {
															#line 2093 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new ErrorListenActorActor(collection));
															#line 3601 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}

#line 2102 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

																#line 3606 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via updateHistogram()
															#line 2103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class UpdateHistogramActor>
															#line 2103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class UpdateHistogramActorState {
															#line 3612 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 2103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UpdateHistogramActorState(ThreadFutureStream<std::pair<std::string, double>> const& metricFutureStream) 
															#line 2103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 2103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : metricFutureStream(metricFutureStream),
															#line 2104 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   commitLatencyHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_COMMIT_LATENCY_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2106 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   commitActionHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_COMMIT_ACTION_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2108 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   commitQueueWaitHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_COMMIT_QUEUEWAIT_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2110 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   writeHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_WRITE_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2112 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   deleteCompactRangeHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_DELETE_COMPACTRANGE_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2114 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readRangeLatencyHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READRANGE_LATENCY_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2116 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readValueLatencyHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READVALUE_LATENCY_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2118 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readPrefixLatencyHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READPREFIX_LATENCY_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2120 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readRangeActionHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READRANGE_ACTION_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2122 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readValueActionHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READVALUE_ACTION_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2124 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readPrefixActionHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READPREFIX_ACTION_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2126 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readRangeQueueWaitHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READRANGE_QUEUEWAIT_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2128 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readValueQueueWaitHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READVALUE_QUEUEWAIT_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2130 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readPrefixQueueWaitHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READPREFIX_QUEUEWAIT_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2132 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readRangeNewIteratorHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READRANGE_NEWITERATOR_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2134 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readValueGetHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READVALUE_GET_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2136 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   readPrefixGetHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READPREFIX_GET_HISTOGRAM, Histogram::Unit::milliseconds)),
															#line 2138 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   rocksdbReadRangeBytesReturnedHistogram(Histogram::getHistogram( ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READ_RANGE_BYTES_RETURNED_HISTOGRAM, Histogram::Unit::bytes)),
															#line 2140 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   rocksdbReadRangeKVPairsReturnedHistogram(Histogram::getHistogram(ROCKSDBSTORAGE_HISTOGRAM_GROUP, ROCKSDB_READ_RANGE_KV_PAIRS_RETURNED_HISTOGRAM, Histogram::Unit::countLinear))
															#line 3657 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("updateHistogram", reinterpret_cast<unsigned long>(this));

	}
	~UpdateHistogramActorState() 
	{
		fdb_probe_actor_destroy("updateHistogram", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2144 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			;
															#line 3672 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateHistogramActorState();
		static_cast<UpdateHistogramActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2146 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		auto __when_expr_0 = metricFutureStream;
															#line 3702 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_0), FutureStream<std::pair<std::string, double>>>::value || std::is_same<decltype(__when_expr_0), ThreadFutureStream<std::pair<std::string, double>>>::value, "invalid type");
															#line 2145 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<UpdateHistogramActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3706 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.pop(), loopDepth); };
		static_cast<UpdateHistogramActor*>(this)->actor_wait_state = 1;
															#line 2146 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorSingleCallback< UpdateHistogramActor, 0, std::pair<std::string, double> >*>(static_cast<UpdateHistogramActor*>(this)));
															#line 3711 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::pair<std::string, double> const& measure,int loopDepth) 
	{
															#line 2147 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::string metricName = measure.first;
															#line 2148 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		double metricValue = measure.second;
															#line 2149 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (metricName == ROCKSDB_COMMIT_LATENCY_HISTOGRAM.toString())
															#line 3730 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 2150 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			commitLatencyHistogram->sampleSeconds(metricValue);
															#line 3734 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2151 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (metricName == ROCKSDB_COMMIT_ACTION_HISTOGRAM.toString())
															#line 3740 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			{
															#line 2152 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				commitActionHistogram->sampleSeconds(metricValue);
															#line 3744 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			}
			else
			{
															#line 2153 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				if (metricName == ROCKSDB_COMMIT_QUEUEWAIT_HISTOGRAM.toString())
															#line 3750 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
				{
															#line 2154 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
					commitQueueWaitHistogram->sampleSeconds(metricValue);
															#line 3754 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
				}
				else
				{
															#line 2155 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
					if (metricName == ROCKSDB_WRITE_HISTOGRAM.toString())
															#line 3760 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
					{
															#line 2156 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
						writeHistogram->sampleSeconds(metricValue);
															#line 3764 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
					}
					else
					{
															#line 2157 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
						if (metricName == ROCKSDB_DELETE_COMPACTRANGE_HISTOGRAM.toString())
															#line 3770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
						{
															#line 2158 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
							deleteCompactRangeHistogram->sampleSeconds(metricValue);
															#line 3774 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
						}
						else
						{
															#line 2159 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
							if (metricName == ROCKSDB_READRANGE_LATENCY_HISTOGRAM.toString())
															#line 3780 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
							{
															#line 2160 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
								readRangeLatencyHistogram->sampleSeconds(metricValue);
															#line 3784 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
							}
							else
							{
															#line 2161 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
								if (metricName == ROCKSDB_READVALUE_LATENCY_HISTOGRAM.toString())
															#line 3790 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
								{
															#line 2162 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
									readValueLatencyHistogram->sampleSeconds(metricValue);
															#line 3794 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
								}
								else
								{
															#line 2163 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
									if (metricName == ROCKSDB_READPREFIX_LATENCY_HISTOGRAM.toString())
															#line 3800 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
									{
															#line 2164 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
										readPrefixLatencyHistogram->sampleSeconds(metricValue);
															#line 3804 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
									}
									else
									{
															#line 2165 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
										if (metricName == ROCKSDB_READRANGE_ACTION_HISTOGRAM.toString())
															#line 3810 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
										{
															#line 2166 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
											readRangeActionHistogram->sampleSeconds(metricValue);
															#line 3814 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
										}
										else
										{
															#line 2167 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
											if (metricName == ROCKSDB_READVALUE_ACTION_HISTOGRAM.toString())
															#line 3820 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
											{
															#line 2168 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
												readValueActionHistogram->sampleSeconds(metricValue);
															#line 3824 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
											}
											else
											{
															#line 2169 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
												if (metricName == ROCKSDB_READPREFIX_ACTION_HISTOGRAM.toString())
															#line 3830 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
												{
															#line 2170 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
													readPrefixActionHistogram->sampleSeconds(metricValue);
															#line 3834 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
												}
												else
												{
															#line 2171 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
													if (metricName == ROCKSDB_READRANGE_QUEUEWAIT_HISTOGRAM.toString())
															#line 3840 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
													{
															#line 2172 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
														readRangeQueueWaitHistogram->sampleSeconds(metricValue);
															#line 3844 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
													}
													else
													{
															#line 2173 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
														if (metricName == ROCKSDB_READVALUE_QUEUEWAIT_HISTOGRAM.toString())
															#line 3850 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
														{
															#line 2174 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															readValueQueueWaitHistogram->sampleSeconds(metricValue);
															#line 3854 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
														}
														else
														{
															#line 2175 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															if (metricName == ROCKSDB_READPREFIX_QUEUEWAIT_HISTOGRAM.toString())
															#line 3860 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
															{
															#line 2176 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																readPrefixQueueWaitHistogram->sampleSeconds(metricValue);
															#line 3864 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
															}
															else
															{
															#line 2177 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																if (metricName == ROCKSDB_READRANGE_NEWITERATOR_HISTOGRAM.toString())
															#line 3870 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																{
															#line 2178 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																	readRangeNewIteratorHistogram->sampleSeconds(metricValue);
															#line 3874 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																}
																else
																{
															#line 2179 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																	if (metricName == ROCKSDB_READVALUE_GET_HISTOGRAM.toString())
															#line 3880 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																	{
															#line 2180 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																		readValueGetHistogram->sampleSeconds(metricValue);
															#line 3884 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																	}
																	else
																	{
															#line 2181 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																		if (metricName == ROCKSDB_READPREFIX_GET_HISTOGRAM.toString())
															#line 3890 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																		{
															#line 2182 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																			readPrefixGetHistogram->sampleSeconds(metricValue);
															#line 3894 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																		}
																		else
																		{
															#line 2183 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																			if (metricName == ROCKSDB_READ_RANGE_BYTES_RETURNED_HISTOGRAM.toString())
															#line 3900 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																			{
															#line 2184 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																				rocksdbReadRangeBytesReturnedHistogram->sample(metricValue);
															#line 3904 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																			}
																			else
																			{
															#line 2185 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																				if (metricName == ROCKSDB_READ_RANGE_KV_PAIRS_RETURNED_HISTOGRAM.toString())
															#line 3910 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																				{
															#line 2186 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																					rocksdbReadRangeKVPairsReturnedHistogram->sample(metricValue);
															#line 3914 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																				}
																				else
																				{
															#line 2188 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																					UNREACHABLE();
															#line 3920 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(std::pair<std::string, double> && measure,int loopDepth) 
	{
															#line 2147 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::string metricName = measure.first;
															#line 2148 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		double metricValue = measure.second;
															#line 2149 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (metricName == ROCKSDB_COMMIT_LATENCY_HISTOGRAM.toString())
															#line 3952 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 2150 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			commitLatencyHistogram->sampleSeconds(metricValue);
															#line 3956 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
		else
		{
															#line 2151 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (metricName == ROCKSDB_COMMIT_ACTION_HISTOGRAM.toString())
															#line 3962 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			{
															#line 2152 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				commitActionHistogram->sampleSeconds(metricValue);
															#line 3966 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			}
			else
			{
															#line 2153 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				if (metricName == ROCKSDB_COMMIT_QUEUEWAIT_HISTOGRAM.toString())
															#line 3972 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
				{
															#line 2154 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
					commitQueueWaitHistogram->sampleSeconds(metricValue);
															#line 3976 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
				}
				else
				{
															#line 2155 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
					if (metricName == ROCKSDB_WRITE_HISTOGRAM.toString())
															#line 3982 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
					{
															#line 2156 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
						writeHistogram->sampleSeconds(metricValue);
															#line 3986 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
					}
					else
					{
															#line 2157 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
						if (metricName == ROCKSDB_DELETE_COMPACTRANGE_HISTOGRAM.toString())
															#line 3992 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
						{
															#line 2158 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
							deleteCompactRangeHistogram->sampleSeconds(metricValue);
															#line 3996 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
						}
						else
						{
															#line 2159 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
							if (metricName == ROCKSDB_READRANGE_LATENCY_HISTOGRAM.toString())
															#line 4002 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
							{
															#line 2160 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
								readRangeLatencyHistogram->sampleSeconds(metricValue);
															#line 4006 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
							}
							else
							{
															#line 2161 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
								if (metricName == ROCKSDB_READVALUE_LATENCY_HISTOGRAM.toString())
															#line 4012 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
								{
															#line 2162 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
									readValueLatencyHistogram->sampleSeconds(metricValue);
															#line 4016 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
								}
								else
								{
															#line 2163 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
									if (metricName == ROCKSDB_READPREFIX_LATENCY_HISTOGRAM.toString())
															#line 4022 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
									{
															#line 2164 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
										readPrefixLatencyHistogram->sampleSeconds(metricValue);
															#line 4026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
									}
									else
									{
															#line 2165 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
										if (metricName == ROCKSDB_READRANGE_ACTION_HISTOGRAM.toString())
															#line 4032 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
										{
															#line 2166 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
											readRangeActionHistogram->sampleSeconds(metricValue);
															#line 4036 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
										}
										else
										{
															#line 2167 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
											if (metricName == ROCKSDB_READVALUE_ACTION_HISTOGRAM.toString())
															#line 4042 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
											{
															#line 2168 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
												readValueActionHistogram->sampleSeconds(metricValue);
															#line 4046 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
											}
											else
											{
															#line 2169 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
												if (metricName == ROCKSDB_READPREFIX_ACTION_HISTOGRAM.toString())
															#line 4052 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
												{
															#line 2170 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
													readPrefixActionHistogram->sampleSeconds(metricValue);
															#line 4056 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
												}
												else
												{
															#line 2171 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
													if (metricName == ROCKSDB_READRANGE_QUEUEWAIT_HISTOGRAM.toString())
															#line 4062 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
													{
															#line 2172 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
														readRangeQueueWaitHistogram->sampleSeconds(metricValue);
															#line 4066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
													}
													else
													{
															#line 2173 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
														if (metricName == ROCKSDB_READVALUE_QUEUEWAIT_HISTOGRAM.toString())
															#line 4072 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
														{
															#line 2174 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															readValueQueueWaitHistogram->sampleSeconds(metricValue);
															#line 4076 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
														}
														else
														{
															#line 2175 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															if (metricName == ROCKSDB_READPREFIX_QUEUEWAIT_HISTOGRAM.toString())
															#line 4082 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
															{
															#line 2176 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																readPrefixQueueWaitHistogram->sampleSeconds(metricValue);
															#line 4086 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
															}
															else
															{
															#line 2177 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																if (metricName == ROCKSDB_READRANGE_NEWITERATOR_HISTOGRAM.toString())
															#line 4092 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																{
															#line 2178 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																	readRangeNewIteratorHistogram->sampleSeconds(metricValue);
															#line 4096 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																}
																else
																{
															#line 2179 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																	if (metricName == ROCKSDB_READVALUE_GET_HISTOGRAM.toString())
															#line 4102 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																	{
															#line 2180 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																		readValueGetHistogram->sampleSeconds(metricValue);
															#line 4106 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																	}
																	else
																	{
															#line 2181 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																		if (metricName == ROCKSDB_READPREFIX_GET_HISTOGRAM.toString())
															#line 4112 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																		{
															#line 2182 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																			readPrefixGetHistogram->sampleSeconds(metricValue);
															#line 4116 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																		}
																		else
																		{
															#line 2183 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																			if (metricName == ROCKSDB_READ_RANGE_BYTES_RETURNED_HISTOGRAM.toString())
															#line 4122 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																			{
															#line 2184 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																				rocksdbReadRangeBytesReturnedHistogram->sample(metricValue);
															#line 4126 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																			}
																			else
																			{
															#line 2185 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																				if (metricName == ROCKSDB_READ_RANGE_KV_PAIRS_RETURNED_HISTOGRAM.toString())
															#line 4132 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																				{
															#line 2186 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																					rocksdbReadRangeKVPairsReturnedHistogram->sample(metricValue);
															#line 4136 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																				}
																				else
																				{
															#line 2188 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
																					UNREACHABLE();
															#line 4142 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateHistogramActor*>(this)->actor_wait_state > 0) static_cast<UpdateHistogramActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateHistogramActor*>(this)->ActorSingleCallback< UpdateHistogramActor, 0, std::pair<std::string, double> >::remove();

	}
	void a_callback_fire(ActorSingleCallback< UpdateHistogramActor, 0, std::pair<std::string, double> >*,std::pair<std::string, double> const& value) 
	{
		fdb_probe_actor_enter("updateHistogram", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateHistogramActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateHistogram", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< UpdateHistogramActor, 0, std::pair<std::string, double> >*,std::pair<std::string, double> && value) 
	{
		fdb_probe_actor_enter("updateHistogram", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateHistogramActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateHistogram", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorSingleCallback< UpdateHistogramActor, 0, std::pair<std::string, double> >*,Error err) 
	{
		fdb_probe_actor_enter("updateHistogram", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateHistogramActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateHistogram", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	ThreadFutureStream<std::pair<std::string, double>> metricFutureStream;
															#line 2104 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> commitLatencyHistogram;
															#line 2106 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> commitActionHistogram;
															#line 2108 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> commitQueueWaitHistogram;
															#line 2110 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> writeHistogram;
															#line 2112 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> deleteCompactRangeHistogram;
															#line 2114 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readRangeLatencyHistogram;
															#line 2116 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readValueLatencyHistogram;
															#line 2118 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readPrefixLatencyHistogram;
															#line 2120 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readRangeActionHistogram;
															#line 2122 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readValueActionHistogram;
															#line 2124 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readPrefixActionHistogram;
															#line 2126 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readRangeQueueWaitHistogram;
															#line 2128 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readValueQueueWaitHistogram;
															#line 2130 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readPrefixQueueWaitHistogram;
															#line 2132 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readRangeNewIteratorHistogram;
															#line 2134 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readValueGetHistogram;
															#line 2136 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> readPrefixGetHistogram;
															#line 2138 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> rocksdbReadRangeBytesReturnedHistogram;
															#line 2140 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reference<Histogram> rocksdbReadRangeKVPairsReturnedHistogram;
															#line 4269 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via updateHistogram()
															#line 2103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class UpdateHistogramActor final : public Actor<Void>, public ActorSingleCallback< UpdateHistogramActor, 0, std::pair<std::string, double> >, public FastAllocated<UpdateHistogramActor>, public UpdateHistogramActorState<UpdateHistogramActor> {
															#line 4274 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<UpdateHistogramActor>::operator new;
	using FastAllocated<UpdateHistogramActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18354889516697115904UL, 11256061613259080960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorSingleCallback< UpdateHistogramActor, 0, std::pair<std::string, double> >;
															#line 2103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UpdateHistogramActor(ThreadFutureStream<std::pair<std::string, double>> const& metricFutureStream) 
															#line 4291 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   UpdateHistogramActorState<UpdateHistogramActor>(metricFutureStream),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateHistogram", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11448717511008380160UL, 11220236045797536000UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateHistogramActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateHistogram");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateHistogram", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorSingleCallback< UpdateHistogramActor, 0, std::pair<std::string, double> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
[[nodiscard]] Future<Void> updateHistogram( ThreadFutureStream<std::pair<std::string, double>> const& metricFutureStream ) {
															#line 2103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new UpdateHistogramActor(metricFutureStream));
															#line 4323 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}

#line 2194 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

	Future<Void> getError() const override { return errorFuture; }

																#line 4330 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via doClose()
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class DoCloseActor>
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class DoCloseActorState {
															#line 4336 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	DoCloseActorState(RocksDBKeyValueStore* const& self,bool const& deleteOnClose) 
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : self(self),
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   deleteOnClose(deleteOnClose)
															#line 4345 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("doClose", reinterpret_cast<unsigned long>(this));

	}
	~DoCloseActorState() 
	{
		fdb_probe_actor_destroy("doClose", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2198 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->sharedState->setClosing();
															#line 2201 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->metrics.reset();
															#line 2203 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Void> __when_expr_0 = self->readThreads->stop();
															#line 2203 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 4366 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			static_cast<DoCloseActor*>(this)->actor_wait_state = 1;
															#line 2203 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DoCloseActor, 0, Void >*>(static_cast<DoCloseActor*>(this)));
															#line 4370 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		delete static_cast<DoCloseActor*>(this);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2204 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		self->readIterPool.reset();
															#line 2205 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		auto a = new Writer::CloseAction(self->path, deleteOnClose);
															#line 2206 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		auto f = a->done.getFuture();
															#line 2207 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		self->writeThread->post(a);
															#line 2208 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = f;
															#line 2208 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
															#line 4402 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		static_cast<DoCloseActor*>(this)->actor_wait_state = 2;
															#line 2208 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoCloseActor, 1, Void >*>(static_cast<DoCloseActor*>(this)));
															#line 4406 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2204 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		self->readIterPool.reset();
															#line 2205 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		auto a = new Writer::CloseAction(self->path, deleteOnClose);
															#line 2206 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		auto f = a->done.getFuture();
															#line 2207 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		self->writeThread->post(a);
															#line 2208 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = f;
															#line 2208 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
															#line 4425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		static_cast<DoCloseActor*>(this)->actor_wait_state = 2;
															#line 2208 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DoCloseActor, 1, Void >*>(static_cast<DoCloseActor*>(this)));
															#line 4429 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DoCloseActor*>(this)->actor_wait_state > 0) static_cast<DoCloseActor*>(this)->actor_wait_state = 0;
		static_cast<DoCloseActor*>(this)->ActorCallback< DoCloseActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoCloseActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doClose", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoCloseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doClose", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DoCloseActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doClose", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoCloseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doClose", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DoCloseActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doClose", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoCloseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doClose", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2209 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_2 = self->writeThread->stop();
															#line 2209 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
															#line 4515 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		static_cast<DoCloseActor*>(this)->actor_wait_state = 3;
															#line 2209 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoCloseActor, 2, Void >*>(static_cast<DoCloseActor*>(this)));
															#line 4519 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2209 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_2 = self->writeThread->stop();
															#line 2209 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
															#line 4530 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		static_cast<DoCloseActor*>(this)->actor_wait_state = 3;
															#line 2209 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DoCloseActor, 2, Void >*>(static_cast<DoCloseActor*>(this)));
															#line 4534 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DoCloseActor*>(this)->actor_wait_state > 0) static_cast<DoCloseActor*>(this)->actor_wait_state = 0;
		static_cast<DoCloseActor*>(this)->ActorCallback< DoCloseActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoCloseActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doClose", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoCloseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doClose", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DoCloseActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doClose", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoCloseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doClose", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DoCloseActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doClose", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoCloseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doClose", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 2210 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (self->closePromise.canBeSet())
															#line 4618 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 2211 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->closePromise.send(Void());
															#line 4622 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 2213 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (self->db != nullptr)
															#line 4626 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 2214 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			delete self->db;
															#line 4630 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 2216 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		delete self;
															#line 4634 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 2210 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (self->closePromise.canBeSet())
															#line 4643 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 2211 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->closePromise.send(Void());
															#line 4647 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 2213 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (self->db != nullptr)
															#line 4651 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 2214 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			delete self->db;
															#line 4655 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 2216 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		delete self;
															#line 4659 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DoCloseActor*>(this)->actor_wait_state > 0) static_cast<DoCloseActor*>(this)->actor_wait_state = 0;
		static_cast<DoCloseActor*>(this)->ActorCallback< DoCloseActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DoCloseActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("doClose", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoCloseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doClose", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DoCloseActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("doClose", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DoCloseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doClose", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DoCloseActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("doClose", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DoCloseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("doClose", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		delete static_cast<DoCloseActor*>(this);
															#line 4743 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		return 0;

		return loopDepth;
	}
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	RocksDBKeyValueStore* self;
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	bool deleteOnClose;
															#line 4752 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via doClose()
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class DoCloseActor final : public Actor<void>, public ActorCallback< DoCloseActor, 0, Void >, public ActorCallback< DoCloseActor, 1, Void >, public ActorCallback< DoCloseActor, 2, Void >, public FastAllocated<DoCloseActor>, public DoCloseActorState<DoCloseActor> {
															#line 4757 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<DoCloseActor>::operator new;
	using FastAllocated<DoCloseActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8833529633422490112UL, 11909700856164934656UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() {{
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<void>*>(this)->~Actor();
        operator delete(this);
    }}
#pragma clang diagnostic pop
friend struct ActorCallback< DoCloseActor, 0, Void >;
friend struct ActorCallback< DoCloseActor, 1, Void >;
friend struct ActorCallback< DoCloseActor, 2, Void >;
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	DoCloseActor(RocksDBKeyValueStore* const& self,bool const& deleteOnClose) 
															#line 4776 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<void>(),
		   DoCloseActorState<DoCloseActor>(self, deleteOnClose),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("doClose", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8249376994020974080UL, 9463511211515355904UL);
		ActorExecutionContextHelper __helper(static_cast<DoCloseActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("doClose");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("doClose", reinterpret_cast<unsigned long>(this), -1);

	}
};
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
static void doClose( RocksDBKeyValueStore* const& self, bool const& deleteOnClose ) {
															#line 2197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	new DoCloseActor(self, deleteOnClose);
															#line 4799 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}

#line 2218 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

	Future<Void> onClosed() const override { return closePromise.getFuture(); }

	void dispose() override { doClose(this, true); }

	void close() override { doClose(this, false); }

	KeyValueStoreType getType() const override { return KeyValueStoreType(KeyValueStoreType::SSD_ROCKSDB_V1); }
	bool supportsSstIngestion() const override { return true; }

	Future<Void> init() override {
		if (openFuture.isValid()) {
			return openFuture;
		}
		auto a = std::make_unique<Writer::OpenAction>(
		    path, metrics, &readSemaphore, &fetchSemaphore, errorListener, eventListener, counters);
		openFuture = a->done.getFuture();
		writeThread->post(a.release());
		return openFuture;
	}

	void set(KeyValueRef kv, const Arena*) override {
		if (writeBatch == nullptr) {
			writeBatch.reset(new rocksdb::WriteBatch(
			    0, // reserved_bytes default:0
			    0, // max_bytes default:0
			    SERVER_KNOBS->ROCKSDB_WRITEBATCH_PROTECTION_BYTES_PER_KEY, // protection_bytes_per_key
			    0 /* default_cf_ts_sz default:0 */));
			keysSet.clear();
			maxDeletes = SERVER_KNOBS->ROCKSDB_SINGLEKEY_DELETES_MAX;
			deletesPerCommit = 0;
			deleteRangesPerCommit = 0;
		}
		ASSERT(defaultFdbCF != nullptr);
		writeBatch->Put(defaultFdbCF, toSlice(kv.key), toSlice(kv.value));
		if (SERVER_KNOBS->ROCKSDB_SINGLEKEY_DELETES_ON_CLEARRANGE) {
			keysSet.insert(kv.key);
		}
	}

	void clear(KeyRangeRef keyRange, const Arena*) override {
		if (writeBatch == nullptr) {
			writeBatch.reset(new rocksdb::WriteBatch(
			    0, // reserved_bytes default:0
			    0, // max_bytes default:0
			    SERVER_KNOBS->ROCKSDB_WRITEBATCH_PROTECTION_BYTES_PER_KEY, // protection_bytes_per_key
			    0 /* default_cf_ts_sz default:0 */));
			keysSet.clear();
			maxDeletes = SERVER_KNOBS->ROCKSDB_SINGLEKEY_DELETES_MAX;
			deletesPerCommit = 0;
			deleteRangesPerCommit = 0;
		}

		ASSERT(defaultFdbCF != nullptr);
		// Number of deletes to rocksdb = counters.deleteKeyReqs + convertedDeleteKeyReqs;
		// Number of deleteRanges to rocksdb = counters.deleteRangeReqs - counters.convertedDeleteRangeReqs;
		if (keyRange.singleKeyRange() && !SERVER_KNOBS->ROCKSDB_FORCE_DELETERANGE_FOR_CLEARRANGE) {
			writeBatch->Delete(defaultFdbCF, toSlice(keyRange.begin));
			++counters.deleteKeyReqs;
			--maxDeletes;
			++deletesPerCommit;
		} else {
			++counters.deleteRangeReqs;
			if (SERVER_KNOBS->ROCKSDB_SINGLEKEY_DELETES_ON_CLEARRANGE &&
			    !SERVER_KNOBS->ROCKSDB_FORCE_DELETERANGE_FOR_CLEARRANGE && maxDeletes > 0) {
				++counters.convertedDeleteRangeReqs;
				rocksdb::ReadOptions readOptions = sharedState->getReadOptions();
				auto beginSlice = toSlice(keyRange.begin);
				auto endSlice = toSlice(keyRange.end);
				readOptions.iterate_lower_bound = &beginSlice;
				readOptions.iterate_upper_bound = &endSlice;
				auto cursor = std::unique_ptr<rocksdb::Iterator>(db->NewIterator(readOptions, defaultFdbCF));
				cursor->Seek(toSlice(keyRange.begin));
				while (cursor->Valid() && toStringRef(cursor->key()) < keyRange.end && maxDeletes > 0) {
					writeBatch->Delete(defaultFdbCF, cursor->key());
					++counters.convertedDeleteKeyReqs;
					--maxDeletes;
					++deletesPerCommit;
					cursor->Next();
				}
				if (!cursor->status().ok() || maxDeletes <= 0) {
					// if readrange iteration fails, then do a deleteRange.
					writeBatch->DeleteRange(defaultFdbCF, toSlice(keyRange.begin), toSlice(keyRange.end));
					++deleteRangesPerCommit;
				} else {
					auto it = keysSet.lower_bound(keyRange.begin);
					while (it != keysSet.end() && *it < keyRange.end) {
						writeBatch->Delete(defaultFdbCF, toSlice(*it));
						++counters.convertedDeleteKeyReqs;
						--maxDeletes;
						it++;
						++deletesPerCommit;
					}
					it = previousCommitKeysSet.lower_bound(keyRange.begin);
					while (it != previousCommitKeysSet.end() && *it < keyRange.end) {
						writeBatch->Delete(defaultFdbCF, toSlice(*it));
						++counters.convertedDeleteKeyReqs;
						--maxDeletes;
						it++;
						++deletesPerCommit;
					}
				}
			} else {
				writeBatch->DeleteRange(defaultFdbCF, toSlice(keyRange.begin), toSlice(keyRange.end));
				++deleteRangesPerCommit;
			}
		}
	}

	static bool overloaded(const uint64_t estPendCompactBytes, const uint64_t numImmutableMemtables) {
		// Rocksdb metadata estPendCompactBytes and numImmutableMemtables is not deterministic so we don't
		// use it in simulation. We still want to exercise the overload functionality for test coverage, so we return
		// overloaded = true 5% of the time.
		if (g_network->isSimulated()) {
			return deterministicRandom()->randomInt(0, 100) < 5;
		}
		return (estPendCompactBytes > SERVER_KNOBS->ROCKSDB_CAN_COMMIT_COMPACT_BYTES_LIMIT ||
		        numImmutableMemtables >= SERVER_KNOBS->ROCKSDB_CAN_COMMIT_IMMUTABLE_MEMTABLES_LIMIT);
	}

	// Checks and waits for few seconds if rocskdb is overloaded.
																#line 4924 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via checkRocksdbState()
															#line 2339 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class CheckRocksdbStateActor>
															#line 2339 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class CheckRocksdbStateActorState {
															#line 4930 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 2339 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	CheckRocksdbStateActorState(RocksDBKeyValueStore* const& self) 
															#line 2339 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 2339 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : self(self),
															#line 2340 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   estPendCompactBytes(),
															#line 2341 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   numImmutableMemtables(),
															#line 2342 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   count(SERVER_KNOBS->ROCKSDB_CAN_COMMIT_DELAY_TIMES_ON_OVERLOAD)
															#line 4943 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("checkRocksdbState", reinterpret_cast<unsigned long>(this));

	}
	~CheckRocksdbStateActorState() 
	{
		fdb_probe_actor_destroy("checkRocksdbState", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2343 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->db->GetAggregatedIntProperty(rocksdb::DB::Properties::kEstimatePendingCompactionBytes, &estPendCompactBytes);
															#line 2345 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->db->GetAggregatedIntProperty(rocksdb::DB::Properties::kNumImmutableMemTable, &numImmutableMemtables);
															#line 2346 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			;
															#line 4962 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckRocksdbStateActorState();
		static_cast<CheckRocksdbStateActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2359 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<CheckRocksdbStateActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckRocksdbStateActorState(); static_cast<CheckRocksdbStateActor*>(this)->destroy(); return 0; }
															#line 4985 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<CheckRocksdbStateActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckRocksdbStateActorState();
		static_cast<CheckRocksdbStateActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 2346 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!(count && overloaded(estPendCompactBytes, numImmutableMemtables)))
															#line 5004 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2347 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(SERVER_KNOBS->ROCKSDB_CAN_COMMIT_DELAY_ON_OVERLOAD);
															#line 2347 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<CheckRocksdbStateActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5012 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<CheckRocksdbStateActor*>(this)->actor_wait_state = 1;
															#line 2347 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckRocksdbStateActor, 0, Void >*>(static_cast<CheckRocksdbStateActor*>(this)));
															#line 5017 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 2348 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		++self->counters.commitDelayed;
															#line 2349 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		count--;
															#line 2350 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 5043 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 2351 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			TraceEvent(SevWarn, "RocksDBCommitsDelayed1000x", self->id) .detail("EstPendCompactBytes", estPendCompactBytes) .detail("NumImmutableMemtables", numImmutableMemtables);
															#line 5047 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 2354 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		self->db->GetAggregatedIntProperty(rocksdb::DB::Properties::kEstimatePendingCompactionBytes, &estPendCompactBytes);
															#line 2356 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		self->db->GetAggregatedIntProperty(rocksdb::DB::Properties::kNumImmutableMemTable, &numImmutableMemtables);
															#line 5053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 2348 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		++self->counters.commitDelayed;
															#line 2349 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		count--;
															#line 2350 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (deterministicRandom()->random01() < 0.001)
															#line 5066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 2351 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			TraceEvent(SevWarn, "RocksDBCommitsDelayed1000x", self->id) .detail("EstPendCompactBytes", estPendCompactBytes) .detail("NumImmutableMemtables", numImmutableMemtables);
															#line 5070 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 2354 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		self->db->GetAggregatedIntProperty(rocksdb::DB::Properties::kEstimatePendingCompactionBytes, &estPendCompactBytes);
															#line 2356 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		self->db->GetAggregatedIntProperty(rocksdb::DB::Properties::kNumImmutableMemTable, &numImmutableMemtables);
															#line 5076 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckRocksdbStateActor*>(this)->actor_wait_state > 0) static_cast<CheckRocksdbStateActor*>(this)->actor_wait_state = 0;
		static_cast<CheckRocksdbStateActor*>(this)->ActorCallback< CheckRocksdbStateActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckRocksdbStateActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkRocksdbState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRocksdbStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRocksdbState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckRocksdbStateActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkRocksdbState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRocksdbStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRocksdbState", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckRocksdbStateActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkRocksdbState", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRocksdbStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRocksdbState", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2339 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	RocksDBKeyValueStore* self;
															#line 2340 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	uint64_t estPendCompactBytes;
															#line 2341 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	uint64_t numImmutableMemtables;
															#line 2342 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	int count;
															#line 5164 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via checkRocksdbState()
															#line 2339 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class CheckRocksdbStateActor final : public Actor<Void>, public ActorCallback< CheckRocksdbStateActor, 0, Void >, public FastAllocated<CheckRocksdbStateActor>, public CheckRocksdbStateActorState<CheckRocksdbStateActor> {
															#line 5169 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<CheckRocksdbStateActor>::operator new;
	using FastAllocated<CheckRocksdbStateActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16635236449861892352UL, 4280603997516396288UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckRocksdbStateActor, 0, Void >;
															#line 2339 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	CheckRocksdbStateActor(RocksDBKeyValueStore* const& self) 
															#line 5186 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   CheckRocksdbStateActorState<CheckRocksdbStateActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkRocksdbState", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12911928222912983808UL, 6268774229734680576UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRocksdbStateActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkRocksdbState");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkRocksdbState", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckRocksdbStateActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2339 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
[[nodiscard]] Future<Void> checkRocksdbState( RocksDBKeyValueStore* const& self ) {
															#line 2339 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new CheckRocksdbStateActor(self));
															#line 5218 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}

#line 2361 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

	Future<Void> canCommit() override { return checkRocksdbState(this); }

																#line 5225 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via commitInRocksDB()
															#line 2364 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class CommitInRocksDBActor>
															#line 2364 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class CommitInRocksDBActorState {
															#line 5231 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 2364 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	CommitInRocksDBActorState(RocksDBKeyValueStore* const& self) 
															#line 2364 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 2364 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : self(self)
															#line 5238 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("commitInRocksDB", reinterpret_cast<unsigned long>(this));

	}
	~CommitInRocksDBActorState() 
	{
		fdb_probe_actor_destroy("commitInRocksDB", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2366 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (self->writeBatch == nullptr)
															#line 5253 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			{
															#line 2367 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				if (!static_cast<CommitInRocksDBActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CommitInRocksDBActorState(); static_cast<CommitInRocksDBActor*>(this)->destroy(); return 0; }
															#line 5257 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
				new (&static_cast<CommitInRocksDBActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~CommitInRocksDBActorState();
				static_cast<CommitInRocksDBActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 2369 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			auto a = new Writer::CommitAction();
															#line 2370 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			a->batchToCommit = std::move(self->writeBatch);
															#line 2371 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->previousCommitKeysSet = std::move(self->keysSet);
															#line 2372 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->maxDeletes = SERVER_KNOBS->ROCKSDB_SINGLEKEY_DELETES_MAX;
															#line 2373 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->deletesPerCommitHistogram->sampleRecordCounter(self->deletesPerCommit);
															#line 2374 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->deleteRangesPerCommitHistogram->sampleRecordCounter(self->deleteRangesPerCommit);
															#line 2375 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (self->deletesPerCommit > 8000 || self->deleteRangesPerCommit > 1000)
															#line 5277 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			{
															#line 2376 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				TraceEvent("RocksDBDeletesCount", self->id) .detail("DeletesPerCommit", self->deletesPerCommit) .detail("DeleteRangesPerCommit", self->deleteRangesPerCommit);
															#line 5281 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			}
															#line 2379 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->deletesPerCommit = 0;
															#line 2380 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->deleteRangesPerCommit = 0;
															#line 2381 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			fut = a->done.getFuture();
															#line 2382 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			self->writeThread->post(a);
															#line 2383 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Void> __when_expr_0 = fut;
															#line 2383 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<CommitInRocksDBActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5295 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CommitInRocksDBActor*>(this)->actor_wait_state = 1;
															#line 2383 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CommitInRocksDBActor, 0, Void >*>(static_cast<CommitInRocksDBActor*>(this)));
															#line 5300 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CommitInRocksDBActorState();
		static_cast<CommitInRocksDBActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2384 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		self->previousCommitKeysSet.clear();
															#line 2385 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<CommitInRocksDBActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CommitInRocksDBActorState(); static_cast<CommitInRocksDBActor*>(this)->destroy(); return 0; }
															#line 5325 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<CommitInRocksDBActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CommitInRocksDBActorState();
		static_cast<CommitInRocksDBActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2384 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		self->previousCommitKeysSet.clear();
															#line 2385 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<CommitInRocksDBActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CommitInRocksDBActorState(); static_cast<CommitInRocksDBActor*>(this)->destroy(); return 0; }
															#line 5339 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<CommitInRocksDBActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CommitInRocksDBActorState();
		static_cast<CommitInRocksDBActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CommitInRocksDBActor*>(this)->actor_wait_state > 0) static_cast<CommitInRocksDBActor*>(this)->actor_wait_state = 0;
		static_cast<CommitInRocksDBActor*>(this)->ActorCallback< CommitInRocksDBActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CommitInRocksDBActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("commitInRocksDB", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CommitInRocksDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitInRocksDB", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CommitInRocksDBActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("commitInRocksDB", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CommitInRocksDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitInRocksDB", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CommitInRocksDBActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("commitInRocksDB", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CommitInRocksDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitInRocksDB", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2364 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	RocksDBKeyValueStore* self;
															#line 2381 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Future<Void> fut;
															#line 5426 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via commitInRocksDB()
															#line 2364 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class CommitInRocksDBActor final : public Actor<Void>, public ActorCallback< CommitInRocksDBActor, 0, Void >, public FastAllocated<CommitInRocksDBActor>, public CommitInRocksDBActorState<CommitInRocksDBActor> {
															#line 5431 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<CommitInRocksDBActor>::operator new;
	using FastAllocated<CommitInRocksDBActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11053986159654627072UL, 12433071870492610048UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CommitInRocksDBActor, 0, Void >;
															#line 2364 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	CommitInRocksDBActor(RocksDBKeyValueStore* const& self) 
															#line 5448 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   CommitInRocksDBActorState<CommitInRocksDBActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("commitInRocksDB", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11869706016978475264UL, 17336036884070694656UL);
		ActorExecutionContextHelper __helper(static_cast<CommitInRocksDBActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("commitInRocksDB");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("commitInRocksDB", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CommitInRocksDBActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2364 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
[[nodiscard]] Future<Void> commitInRocksDB( RocksDBKeyValueStore* const& self ) {
															#line 2364 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new CommitInRocksDBActor(self));
															#line 5480 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}

#line 2387 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

	Future<Void> commit(bool) override { return commitInRocksDB(this); }

	void checkWaiters(const FlowLock& semaphore, int maxWaiters) {
		if (semaphore.waiters() > maxWaiters) {
			++counters.immediateThrottle;
			throw server_overloaded();
		}
	}

	// We don't throttle eager reads and reads to the FF keyspace because FDB struggles when those reads fail.
	// Thus far, they have been low enough volume to not cause an issue.
	static bool shouldThrottle(ReadType type, KeyRef key) {
		return type != ReadType::EAGER && !(key.startsWith(systemKeys.begin));
	}

																#line 5500 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via read()
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class Action, class ReadActor>
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class ReadActorState {
															#line 5506 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	ReadActorState(Action* const& action,FlowLock* const& semaphore,IThreadPool* const& pool,Counter* const& counter) 
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : action(action),
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   semaphore(semaphore),
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   pool(pool),
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   counter(counter),
															#line 2405 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   a(action)
															#line 5521 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("read", reinterpret_cast<unsigned long>(this));

	}
	~ReadActorState() 
	{
		fdb_probe_actor_destroy("read", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2406 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Void>> __when_expr_0 = timeout(semaphore->take(), SERVER_KNOBS->ROCKSDB_READ_QUEUE_WAIT);
															#line 2406 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<ReadActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5538 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadActor*>(this)->actor_wait_state = 1;
															#line 2406 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadActor, 0, Optional<Void> >*>(static_cast<ReadActor*>(this)));
															#line 5543 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadActorState();
		static_cast<ReadActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2407 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!slot.present())
															#line 5566 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 2408 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			++(*counter);
															#line 2409 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			return a_body1Catch1(server_overloaded(), loopDepth);
															#line 5572 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 2412 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		release = FlowLock::Releaser(*semaphore);
															#line 2414 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		auto fut = a->result.getFuture();
															#line 2415 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		pool->post(a.release());
															#line 2416 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_1 = fut;
															#line 2416 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<ReadActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5584 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadActor*>(this)->actor_wait_state = 2;
															#line 2416 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadActor, 1, Optional<Value> >*>(static_cast<ReadActor*>(this)));
															#line 5589 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Optional<Void> const& __slot,int loopDepth) 
	{
															#line 2406 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		slot = __slot;
															#line 5598 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Void> && __slot,int loopDepth) 
	{
		slot = std::move(__slot);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadActor*>(this)->actor_wait_state > 0) static_cast<ReadActor*>(this)->actor_wait_state = 0;
		static_cast<ReadActor*>(this)->ActorCallback< ReadActor, 0, Optional<Void> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadActor, 0, Optional<Void> >*,Optional<Void> const& value) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadActor, 0, Optional<Void> >*,Optional<Void> && value) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadActor, 0, Optional<Void> >*,Error err) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Optional<Value> const& result,int loopDepth) 
	{
															#line 2418 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<ReadActor*>(this)->SAV<Optional<Value>>::futures) { (void)(result); this->~ReadActorState(); static_cast<ReadActor*>(this)->destroy(); return 0; }
															#line 5677 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<ReadActor*>(this)->SAV< Optional<Value> >::value()) Optional<Value>(result);
		this->~ReadActorState();
		static_cast<ReadActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Optional<Value> && result,int loopDepth) 
	{
															#line 2418 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<ReadActor*>(this)->SAV<Optional<Value>>::futures) { (void)(result); this->~ReadActorState(); static_cast<ReadActor*>(this)->destroy(); return 0; }
															#line 5689 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<ReadActor*>(this)->SAV< Optional<Value> >::value()) Optional<Value>(result);
		this->~ReadActorState();
		static_cast<ReadActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Value> const& result,int loopDepth) 
	{
		loopDepth = a_body1cont2(result, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Optional<Value> && result,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadActor*>(this)->actor_wait_state > 0) static_cast<ReadActor*>(this)->actor_wait_state = 0;
		static_cast<ReadActor*>(this)->ActorCallback< ReadActor, 1, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadActor, 1, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadActor, 1, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadActor, 1, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Action* action;
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowLock* semaphore;
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	IThreadPool* pool;
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Counter* counter;
															#line 2405 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::unique_ptr<Action> a;
															#line 2406 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Optional<Void> slot;
															#line 2412 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowLock::Releaser release;
															#line 5786 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via read()
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class Action>
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class ReadActor final : public Actor<Optional<Value>>, public ActorCallback< ReadActor<Action>, 0, Optional<Void> >, public ActorCallback< ReadActor<Action>, 1, Optional<Value> >, public FastAllocated<ReadActor<Action>>, public ReadActorState<Action, ReadActor<Action>> {
															#line 5793 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<ReadActor<Action>>::operator new;
	using FastAllocated<ReadActor<Action>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5410252502904791040UL, 10686028720999509504UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Optional<Value>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadActor<Action>, 0, Optional<Void> >;
friend struct ActorCallback< ReadActor<Action>, 1, Optional<Value> >;
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	ReadActor(Action* const& action,FlowLock* const& semaphore,IThreadPool* const& pool,Counter* const& counter) 
															#line 5811 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Optional<Value>>(),
		   ReadActorState<Action, ReadActor<Action>>(action, semaphore, pool, counter),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3377648328873480704UL, 6210503387287458048UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("read");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadActor<Action>, 0, Optional<Void> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadActor<Action>, 1, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class Action>
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
[[nodiscard]] static Future<Optional<Value>> read( Action* const& action, FlowLock* const& semaphore, IThreadPool* const& pool, Counter* const& counter ) {
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Optional<Value>>(new ReadActor<Action>(action, semaphore, pool, counter));
															#line 5846 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}

#line 2420 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

	Future<Optional<Value>> readValue(KeyRef key, Optional<ReadOptions> options) override {
		ReadType type = ReadType::NORMAL;
		Optional<UID> debugID;

		if (options.present()) {
			type = options.get().type;
			debugID = options.get().debugID;
		}

		if (!shouldThrottle(type, key)) {
			auto a = new Reader::ReadValueAction(key, type, debugID);
			auto res = a->result.getFuture();
			readThreads->post(a);
			return res;
		}

		auto& semaphore = (type == ReadType::FETCH) ? fetchSemaphore : readSemaphore;
		int maxWaiters = (type == ReadType::FETCH) ? numFetchWaiters : numReadWaiters;

		checkWaiters(semaphore, maxWaiters);
		auto a = std::make_unique<Reader::ReadValueAction>(key, type, debugID);
		return read(a.release(), &semaphore, readThreads.getPtr(), &counters.failedToAcquire);
	}

	Future<Optional<Value>> readValuePrefix(KeyRef key, int maxLength, Optional<ReadOptions> options) override {
		ReadType type = ReadType::NORMAL;
		Optional<UID> debugID;

		if (options.present()) {
			type = options.get().type;
			debugID = options.get().debugID;
		}

		if (!shouldThrottle(type, key)) {
			auto a = new Reader::ReadValuePrefixAction(key, maxLength, type, debugID);
			auto res = a->result.getFuture();
			readThreads->post(a);
			return res;
		}

		auto& semaphore = (type == ReadType::FETCH) ? fetchSemaphore : readSemaphore;
		int maxWaiters = (type == ReadType::FETCH) ? numFetchWaiters : numReadWaiters;

		checkWaiters(semaphore, maxWaiters);
		auto a = std::make_unique<Reader::ReadValuePrefixAction>(key, maxLength, type, debugID);
		return read(a.release(), &semaphore, readThreads.getPtr(), &counters.failedToAcquire);
	}

																#line 5899 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via read()
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class ReadActor1>
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class ReadActor1State {
															#line 5905 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	ReadActor1State(Reader::ReadRangeAction* const& action,FlowLock* const& semaphore,IThreadPool* const& pool,Counter* const& counter) 
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : action(action),
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   semaphore(semaphore),
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   pool(pool),
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   counter(counter),
															#line 2473 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   a(action)
															#line 5920 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("read", reinterpret_cast<unsigned long>(this));

	}
	~ReadActor1State() 
	{
		fdb_probe_actor_destroy("read", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2474 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Void>> __when_expr_0 = timeout(semaphore->take(), SERVER_KNOBS->ROCKSDB_READ_QUEUE_WAIT);
															#line 2474 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<ReadActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5937 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadActor1*>(this)->actor_wait_state = 1;
															#line 2474 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadActor1, 0, Optional<Void> >*>(static_cast<ReadActor1*>(this)));
															#line 5942 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadActor1State();
		static_cast<ReadActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2475 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!slot.present())
															#line 5965 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
															#line 2476 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			++(*counter);
															#line 2477 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			return a_body1Catch1(server_overloaded(), loopDepth);
															#line 5971 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		}
															#line 2480 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		release = FlowLock::Releaser(*semaphore);
															#line 2482 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		auto fut = a->result.getFuture();
															#line 2483 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		pool->post(a.release());
															#line 2484 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Standalone<RangeResultRef>> __when_expr_1 = fut;
															#line 2484 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<ReadActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5983 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadActor1*>(this)->actor_wait_state = 2;
															#line 2484 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadActor1, 1, Standalone<RangeResultRef> >*>(static_cast<ReadActor1*>(this)));
															#line 5988 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Optional<Void> const& __slot,int loopDepth) 
	{
															#line 2474 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		slot = __slot;
															#line 5997 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Void> && __slot,int loopDepth) 
	{
		slot = std::move(__slot);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadActor1*>(this)->actor_wait_state > 0) static_cast<ReadActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadActor1*>(this)->ActorCallback< ReadActor1, 0, Optional<Void> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadActor1, 0, Optional<Void> >*,Optional<Void> const& value) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadActor1, 0, Optional<Void> >*,Optional<Void> && value) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadActor1, 0, Optional<Void> >*,Error err) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Standalone<RangeResultRef> const& result,int loopDepth) 
	{
															#line 2486 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<ReadActor1*>(this)->SAV<Standalone<RangeResultRef>>::futures) { (void)(result); this->~ReadActor1State(); static_cast<ReadActor1*>(this)->destroy(); return 0; }
															#line 6076 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<ReadActor1*>(this)->SAV< Standalone<RangeResultRef> >::value()) Standalone<RangeResultRef>(result);
		this->~ReadActor1State();
		static_cast<ReadActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Standalone<RangeResultRef> && result,int loopDepth) 
	{
															#line 2486 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<ReadActor1*>(this)->SAV<Standalone<RangeResultRef>>::futures) { (void)(result); this->~ReadActor1State(); static_cast<ReadActor1*>(this)->destroy(); return 0; }
															#line 6088 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<ReadActor1*>(this)->SAV< Standalone<RangeResultRef> >::value()) Standalone<RangeResultRef>(result);
		this->~ReadActor1State();
		static_cast<ReadActor1*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Standalone<RangeResultRef> const& result,int loopDepth) 
	{
		loopDepth = a_body1cont2(result, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Standalone<RangeResultRef> && result,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadActor1*>(this)->actor_wait_state > 0) static_cast<ReadActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadActor1*>(this)->ActorCallback< ReadActor1, 1, Standalone<RangeResultRef> >::remove();

	}
	void a_callback_fire(ActorCallback< ReadActor1, 1, Standalone<RangeResultRef> >*,Standalone<RangeResultRef> const& value) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadActor1, 1, Standalone<RangeResultRef> >*,Standalone<RangeResultRef> && value) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadActor1, 1, Standalone<RangeResultRef> >*,Error err) 
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Reader::ReadRangeAction* action;
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowLock* semaphore;
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	IThreadPool* pool;
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Counter* counter;
															#line 2473 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::unique_ptr<Reader::ReadRangeAction> a;
															#line 2474 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Optional<Void> slot;
															#line 2480 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowLock::Releaser release;
															#line 6185 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via read()
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class ReadActor1 final : public Actor<Standalone<RangeResultRef>>, public ActorCallback< ReadActor1, 0, Optional<Void> >, public ActorCallback< ReadActor1, 1, Standalone<RangeResultRef> >, public FastAllocated<ReadActor1>, public ReadActor1State<ReadActor1> {
															#line 6190 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<ReadActor1>::operator new;
	using FastAllocated<ReadActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5410252502904791040UL, 10686028720999509504UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Standalone<RangeResultRef>>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadActor1, 0, Optional<Void> >;
friend struct ActorCallback< ReadActor1, 1, Standalone<RangeResultRef> >;
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	ReadActor1(Reader::ReadRangeAction* const& action,FlowLock* const& semaphore,IThreadPool* const& pool,Counter* const& counter) 
															#line 6208 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Standalone<RangeResultRef>>(),
		   ReadActor1State<ReadActor1>(action, semaphore, pool, counter),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("read", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13045688915661361920UL, 7552795543224692992UL);
		ActorExecutionContextHelper __helper(static_cast<ReadActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("read");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("read", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadActor1, 0, Optional<Void> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadActor1, 1, Standalone<RangeResultRef> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
[[nodiscard]] static Future<Standalone<RangeResultRef>> read( Reader::ReadRangeAction* const& action, FlowLock* const& semaphore, IThreadPool* const& pool, Counter* const& counter ) {
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Standalone<RangeResultRef>>(new ReadActor1(action, semaphore, pool, counter));
															#line 6241 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}

#line 2488 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

	Future<RangeResult> readRange(KeyRangeRef keys,
	                              int rowLimit,
	                              int byteLimit,
	                              Optional<ReadOptions> options) override {
		ReadType type = ReadType::NORMAL;

		if (options.present()) {
			type = options.get().type;
		}

		if (!shouldThrottle(type, keys.begin)) {
			++counters.rocksdbReadRangeQueries;
			auto a = new Reader::ReadRangeAction(keys, rowLimit, byteLimit, type);
			auto res = a->result.getFuture();
			readThreads->post(a);
			return res;
		}

		auto& semaphore = (type == ReadType::FETCH) ? fetchSemaphore : readSemaphore;
		int maxWaiters = (type == ReadType::FETCH) ? numFetchWaiters : numReadWaiters;

		checkWaiters(semaphore, maxWaiters);
		++counters.rocksdbReadRangeQueries;
		auto a = std::make_unique<Reader::ReadRangeAction>(keys, rowLimit, byteLimit, type);
		return read(a.release(), &semaphore, readThreads.getPtr(), &counters.failedToAcquire);
	}

	StorageBytes getStorageBytes() const override {
		uint64_t liveFilesSize = 0;
		uint64_t obsoleteFilesSize = 0;
		ASSERT(db->GetAggregatedIntProperty(rocksdb::DB::Properties::kLiveSstFilesSize, &liveFilesSize));
		ASSERT(db->GetAggregatedIntProperty(rocksdb::DB::Properties::kObsoleteSstFilesSize, &obsoleteFilesSize));

		int64_t free;
		int64_t total;
		g_network->getDiskBytes(path, free, total);

		// Rocksdb metadata kLiveSstFilesSize and kObsoleteSstFilesSize are not deterministic so don't rely on it for
		// simulation. Instead, we pick a sane value that is deterministically random.
		if (g_network->isSimulated()) {
			liveFilesSize = (total - free) * deterministicRandom()->random01();
			obsoleteFilesSize = 0;
		}

		return StorageBytes(free, total, liveFilesSize + obsoleteFilesSize, free);
	}

	Future<CheckpointMetaData> checkpoint(const CheckpointRequest& request) override {
		auto a = new Writer::CheckpointAction(request);

		auto res = a->reply.getFuture();
		writeThread->post(a);
		return res;
	}

	Future<Void> restore(const std::vector<CheckpointMetaData>& checkpoints) override {
		auto a = new Writer::RestoreAction(path, checkpoints);
		auto res = a->done.getFuture();
		writeThread->post(a);
		return res;
	}

	// Delete a checkpoint.
	Future<Void> deleteCheckpoint(const CheckpointMetaData& checkpoint) override {
		if (checkpoint.format == DataMoveRocksCF) {
			RocksDBColumnFamilyCheckpoint rocksCF;
			ObjectReader reader(checkpoint.getSerializedCheckpoint().begin(), IncludeVersion());
			reader.deserialize(rocksCF);

			std::unordered_set<std::string> dirs;
			for (const LiveFileMetaData& file : rocksCF.sstFiles) {
				dirs.insert(file.db_path);
			}
			for (const std::string dir : dirs) {
				platform::eraseDirectoryRecursive(dir);
				TraceEvent("DeleteCheckpointRemovedDir", id)
				    .detail("CheckpointID", checkpoint.checkpointID)
				    .detail("Dir", dir);
			}
		} else if (checkpoint.format == RocksDB) {
			throw not_implemented();
		} else {
			throw internal_error();
		}
		return Void();
	}

	Future<EncryptionAtRestMode> encryptionMode() override {
		return EncryptionAtRestMode(EncryptionAtRestMode::DISABLED);
	}

	Future<Void> ingestSSTFiles(std::shared_ptr<BulkLoadFileSetKeyMap> localFileSets) override {
		auto a = new Writer::IngestSSTFilesAction(localFileSets);
		auto res = a->done.getFuture();
		writeThread->post(a);
		return res;
	}

	Future<Void> compactRange(KeyRangeRef range) override {
		auto a = new Writer::CompactRangeAction(range);
		auto res = a->done.getFuture();
		writeThread->post(a);
		return res;
	}

	DB db = nullptr;
	std::shared_ptr<SharedRocksDBState> sharedState;
	std::shared_ptr<PerfContextMetrics> perfContextMetrics;
	std::string path;
	rocksdb::ColumnFamilyHandle* defaultFdbCF = nullptr;
	UID id;
	Reference<IThreadPool> writeThread;
	Reference<IThreadPool> readThreads;
	std::shared_ptr<RocksDBErrorListener> errorListener;
	std::shared_ptr<RocksDBEventListener> eventListener;
	Future<Void> errorFuture;
	Promise<Void> closePromise;
	Future<Void> openFuture;
	std::unique_ptr<rocksdb::WriteBatch> writeBatch;
	// keysSet will store the written keys in the current transaction.
	// previousCommitKeysSet will store the written keys that are currently in the rocksdb commit path.
	// When one commit is in the rocksdb commit path, the other processing commit in the kvsstorerocksdb
	// read iterators will not see the the writes set in previousCommitKeysSet. To avoid that, we will
	// maintain the previousCommitKeysSet until the rocksdb commit is processed and returned.
	std::set<Key> keysSet;
	std::set<Key> previousCommitKeysSet;
	// maximum number of single key deletes in a commit, if ROCKSDB_SINGLEKEY_DELETES_ON_CLEARRANGE is enabled.
	int maxDeletes;
	int deletesPerCommit;
	int deleteRangesPerCommit;
	Reference<Histogram> deletesPerCommitHistogram;
	Reference<Histogram> deleteRangesPerCommitHistogram;
	Optional<Future<Void>> metrics;
	FlowLock readSemaphore;
	int numReadWaiters;
	FlowLock fetchSemaphore;
	int numFetchWaiters;
	std::shared_ptr<ReadIteratorPool> readIterPool;
	std::vector<std::unique_ptr<ThreadReturnPromiseStream<std::pair<std::string, double>>>> metricPromiseStreams;
	// ThreadReturnPromiseStream pair.first stores the histogram name and
	// pair.second stores the corresponding measured latency (seconds)
	Future<Void> collection;
	PromiseStream<Future<Void>> addActor;
	Counters counters;
};

void RocksDBKeyValueStore::Writer::action(CheckpointAction& a) {
	TraceEvent("RocksDBServeCheckpointBegin", id)
	    .detail("MinVersion", a.request.version)
	    .detail("Ranges", describe(a.request.ranges))
	    .detail("Format", static_cast<int>(a.request.format))
	    .detail("CheckpointDir", a.request.checkpointDir);

	rocksdb::Checkpoint* checkpoint = nullptr;
	rocksdb::Status s = rocksdb::Checkpoint::Create(db, &checkpoint);
	if (!s.ok()) {
		logRocksDBError(id, s, "Checkpoint");
		a.reply.sendError(statusToError(s));
		return;
	}

	rocksdb::PinnableSlice value;
	rocksdb::ReadOptions readOptions = sharedState->getReadOptions();
	s = db->Get(readOptions, cf, toSlice(persistVersion), &value);

	if (!s.ok() && !s.IsNotFound()) {
		logRocksDBError(id, s, "Checkpoint");
		a.reply.sendError(statusToError(s));
		return;
	}

	const Version version =
	    s.IsNotFound() ? latestVersion : BinaryReader::fromStringRef<Version>(toStringRef(value), Unversioned());

	ASSERT(a.request.version == version || a.request.version == latestVersion);
	TraceEvent(SevDebug, "RocksDBServeCheckpointVersion", id)
	    .detail("CheckpointVersion", a.request.version)
	    .detail("PersistVersion", version);

	// TODO: set the range as the actual shard range.
	CheckpointMetaData res(version, a.request.format, a.request.checkpointID);
	res.ranges = a.request.ranges;
	const std::string& checkpointDir = abspath(a.request.checkpointDir);

	if (a.request.format == DataMoveRocksCF) {
		rocksdb::ExportImportFilesMetaData* pMetadata{ nullptr };
		platform::eraseDirectoryRecursive(checkpointDir);
		s = checkpoint->ExportColumnFamily(cf, checkpointDir, &pMetadata);
		if (!s.ok()) {
			logRocksDBError(id, s, "ExportColumnFamily");
			a.reply.sendError(statusToError(s));
			return;
		}

		populateMetaData(&res, *pMetadata);
		delete pMetadata;
		TraceEvent("RocksDBServeCheckpointSuccess", id)
		    .detail("CheckpointMetaData", res.toString())
		    .detail("RocksDBCF", getRocksCF(res).toString());
	} else if (a.request.format == RocksDB) {
		platform::eraseDirectoryRecursive(checkpointDir);
		uint64_t debugCheckpointSeq = -1;
		s = checkpoint->CreateCheckpoint(checkpointDir, /*log_size_for_flush=*/0, &debugCheckpointSeq);
		if (!s.ok()) {
			logRocksDBError(id, s, "Checkpoint");
			a.reply.sendError(statusToError(s));
			return;
		}

		RocksDBCheckpoint rcp;
		rcp.checkpointDir = checkpointDir;
		rcp.sstFiles = platform::listFiles(checkpointDir, ".sst");
		res.setSerializedCheckpoint(ObjectWriter::toValue(rcp, IncludeVersion()));
		TraceEvent("RocksDBCheckpointCreated", id)
		    .detail("CheckpointVersion", a.request.version)
		    .detail("RocksSequenceNumber", debugCheckpointSeq)
		    .detail("CheckpointDir", checkpointDir);
	} else {
		if (checkpoint != nullptr) {
			delete checkpoint;
		}
		throw not_implemented();
	}

	if (checkpoint != nullptr) {
		delete checkpoint;
	}
	res.setState(CheckpointMetaData::Complete);
	a.reply.send(res);
}

void RocksDBKeyValueStore::Writer::action(RestoreAction& a) {
	TraceEvent("RocksDBRestoreBegin", id).detail("Path", a.path).detail("Checkpoints", describe(a.checkpoints));

	ASSERT(db != nullptr);
	ASSERT(!a.checkpoints.empty());

	const CheckpointFormat format = a.checkpoints[0].getFormat();
	for (int i = 1; i < a.checkpoints.size(); ++i) {
		if (a.checkpoints[i].getFormat() != format) {
			throw invalid_checkpoint_format();
		}
	}

	rocksdb::Status status;
	if (format == DataMoveRocksCF) {
		ASSERT_EQ(a.checkpoints.size(), 1);
		TraceEvent("RocksDBServeRestoreCF", id)
		    .detail("Path", a.path)
		    .detail("Checkpoint", a.checkpoints[0].toString())
		    .detail("RocksDBCF", getRocksCF(a.checkpoints[0]).toString());

		if (cf != nullptr) {
			ASSERT(db->DropColumnFamily(cf).ok());
			db->DestroyColumnFamilyHandle(cf);
			cfHandles.erase(cf);
			cf = nullptr;
		}

		rocksdb::ExportImportFilesMetaData metaData = getMetaData(a.checkpoints[0]);
		rocksdb::ImportColumnFamilyOptions importOptions;
		importOptions.move_files = true;
		status = db->CreateColumnFamilyWithImport(
		    sharedState->getCfOptions(), SERVER_KNOBS->DEFAULT_FDB_ROCKSDB_COLUMN_FAMILY, importOptions, metaData, &cf);
		cfHandles.insert(cf);

		if (!status.ok()) {
			logRocksDBError(id, status, "Restore");
			a.done.sendError(statusToError(status));
		} else {
			TraceEvent(SevInfo, "RocksDBRestoreCFSuccess", id)
			    .detail("Path", a.path)
			    .detail("Checkpoint", a.checkpoints[0].toString());
			a.done.send(Void());
		}
	} else if (format == RocksDB) {
		if (cf == nullptr) {
			status = db->CreateColumnFamily(
			    sharedState->getCfOptions(), SERVER_KNOBS->DEFAULT_FDB_ROCKSDB_COLUMN_FAMILY, &cf);
			cfHandles.insert(cf);
			TraceEvent("RocksDBServeRestoreRange", id)
			    .detail("Path", a.path)
			    .detail("Checkpoint", describe(a.checkpoints));
			if (!status.ok()) {
				logRocksDBError(id, status, "CreateColumnFamily");
				a.done.sendError(statusToError(status));
				return;
			}
		}

		std::vector<std::string> sstFiles;
		for (const auto& checkpoint : a.checkpoints) {
			const RocksDBCheckpoint rocksCheckpoint = getRocksCheckpoint(checkpoint);
			for (const auto& file : rocksCheckpoint.fetchedFiles) {
				TraceEvent("RocksDBRestoreFile", id)
				    .detail("Checkpoint", rocksCheckpoint.toString())
				    .detail("File", file.toString());
				sstFiles.push_back(file.path);
			}
		}

		if (!sstFiles.empty()) {
			rocksdb::IngestExternalFileOptions ingestOptions;
			ingestOptions.move_files = true;
			ingestOptions.write_global_seqno = false;
			ingestOptions.verify_checksums_before_ingest = true;
			status = db->IngestExternalFile(cf, sstFiles, ingestOptions);
			if (!status.ok()) {
				logRocksDBError(id, status, "IngestExternalFile", SevWarnAlways);
				a.done.sendError(statusToError(status));
				return;
			}
		} else {
			TraceEvent(SevDebug, "RocksDBServeRestoreEmptyRange", id)
			    .detail("Path", a.path)
			    .detail("Checkpoint", describe(a.checkpoints));
		}
		TraceEvent("RocksDBServeRestoreEnd", id).detail("Path", a.path).detail("Checkpoint", describe(a.checkpoints));
		a.done.send(Void());
	} else {
		throw not_implemented();
	}
}

} // namespace

#endif // WITH_ROCKSDB

IKeyValueStore* keyValueStoreRocksDB(std::string const& path,
                                     UID logID,
                                     KeyValueStoreType storeType,
                                     bool checkChecksums,
                                     bool checkIntegrity) {
#ifdef WITH_ROCKSDB
	return new RocksDBKeyValueStore(path, logID);
#else
	TraceEvent(SevError, "RocksDBEngineInitFailure", logID).detail("Reason", "Built without RocksDB");
	ASSERT(false);
	return nullptr;
#endif // WITH_ROCKSDB
}

#ifdef WITH_ROCKSDB
#include "flow/UnitTest.h"

namespace {

															#line 6593 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via flowTestCase2836()
															#line 2836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class FlowTestCase2836Actor>
															#line 2836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase2836ActorState {
															#line 6599 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 2836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase2836ActorState(UnitTestParameters const& params) 
															#line 2836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 2836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : params(params),
															#line 2837 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   rocksDBTestDir("rocksdb-kvstore-basic-test-db")
															#line 6608 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2836", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2836ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2836", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2838 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 2840 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			kvStore = new RocksDBKeyValueStore(rocksDBTestDir, deterministicRandom()->randomUniqueID());
															#line 2841 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Void> __when_expr_0 = kvStore->init();
															#line 2841 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6629 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 1;
															#line 2841 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 0, Void >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 6634 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2836ActorState();
		static_cast<FlowTestCase2836Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2843 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		foo = "foo"_sr;
															#line 2844 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		bar = "ibar"_sr;
															#line 2845 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ foo, foo });
															#line 2846 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ keyAfter(foo), keyAfter(foo) });
															#line 2847 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ bar, bar });
															#line 2848 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ keyAfter(bar), keyAfter(bar) });
															#line 2849 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = kvStore->commit(false);
															#line 2849 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6671 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 2;
															#line 2849 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 1, Void >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 6676 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2843 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		foo = "foo"_sr;
															#line 2844 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		bar = "ibar"_sr;
															#line 2845 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ foo, foo });
															#line 2846 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ keyAfter(foo), keyAfter(foo) });
															#line 2847 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ bar, bar });
															#line 2848 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ keyAfter(bar), keyAfter(bar) });
															#line 2849 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = kvStore->commit(false);
															#line 2849 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6699 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 2;
															#line 2849 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 1, Void >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 6704 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2836Actor*>(this)->ActorCallback< FlowTestCase2836Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase2836Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		{
															#line 2852 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_2 = kvStore->readValue(foo);
															#line 2852 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6791 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 3;
															#line 2852 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 2, Optional<Value> >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 6796 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		{
															#line 2852 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_2 = kvStore->readValue(foo);
															#line 2852 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6809 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
			static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 3;
															#line 2852 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 2, Optional<Value> >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 6814 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2836Actor*>(this)->ActorCallback< FlowTestCase2836Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase2836Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 2857 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->clear(singleKeyRange(foo));
															#line 2858 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_3 = kvStore->commit(false);
															#line 2858 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6903 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 4;
															#line 2858 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 3, Void >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 6908 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Optional<Value> const& val,int loopDepth) 
	{
															#line 2853 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(foo == val.get());
															#line 6917 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Optional<Value> && val,int loopDepth) 
	{
															#line 2853 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(foo == val.get());
															#line 6926 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont4(val, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2836Actor*>(this)->ActorCallback< FlowTestCase2836Actor, 2, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 2, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 2, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FlowTestCase2836Actor, 2, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		{
															#line 2861 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_4 = kvStore->readValue(foo);
															#line 2861 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7013 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 5;
															#line 2861 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 4, Optional<Value> >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 7018 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		{
															#line 2861 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_4 = kvStore->readValue(foo);
															#line 2861 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7031 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 5;
															#line 2861 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 4, Optional<Value> >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 7036 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2836Actor*>(this)->ActorCallback< FlowTestCase2836Actor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FlowTestCase2836Actor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont7(int loopDepth) 
	{
		{
															#line 2866 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_5 = kvStore->readValue(keyAfter(foo));
															#line 2866 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7124 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont7when1(__when_expr_5.get(), loopDepth); };
			static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 6;
															#line 2866 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 5, Optional<Value> >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 7129 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> const& val,int loopDepth) 
	{
															#line 2862 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(!val.present());
															#line 7139 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> && val,int loopDepth) 
	{
															#line 2862 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(!val.present());
															#line 7148 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont8(val, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2836Actor*>(this)->ActorCallback< FlowTestCase2836Actor, 4, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 4, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 4, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FlowTestCase2836Actor, 4, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont10(int loopDepth) 
	{
															#line 2871 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->clear(KeyRangeRef(keyAfter(foo), keyAfter(bar)));
															#line 2872 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_6 = kvStore->commit(false);
															#line 2872 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7236 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont10when1(__when_expr_6.get(), loopDepth); };
		static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 7;
															#line 2872 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 6, Void >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 7241 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont11(Optional<Value> const& val,int loopDepth) 
	{
															#line 2867 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(keyAfter(foo) == val.get());
															#line 7250 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont11(Optional<Value> && val,int loopDepth) 
	{
															#line 2867 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(keyAfter(foo) == val.get());
															#line 7259 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont11(val, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont11(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2836Actor*>(this)->ActorCallback< FlowTestCase2836Actor, 5, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 5, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 5, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< FlowTestCase2836Actor, 5, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont10cont1(Void const& _,int loopDepth) 
	{
		{
															#line 2875 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_7 = kvStore->readValue(bar);
															#line 2875 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7346 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont10cont1when1(__when_expr_7.get(), loopDepth); };
			static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 8;
															#line 2875 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 7, Optional<Value> >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 7351 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont10cont1(Void && _,int loopDepth) 
	{
		{
															#line 2875 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_7 = kvStore->readValue(bar);
															#line 2875 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7364 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont10cont1when1(__when_expr_7.get(), loopDepth); };
			static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 8;
															#line 2875 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 7, Optional<Value> >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 7369 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2836Actor*>(this)->ActorCallback< FlowTestCase2836Actor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< FlowTestCase2836Actor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont10cont2(int loopDepth) 
	{
		{
															#line 2880 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_8 = kvStore->readValue(keyAfter(bar));
															#line 2880 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7457 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont10cont2when1(__when_expr_8.get(), loopDepth); };
			static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 9;
															#line 2880 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 8, Optional<Value> >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 7462 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont10cont3(Optional<Value> const& val,int loopDepth) 
	{
															#line 2876 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(!val.present());
															#line 7472 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont10cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont3(Optional<Value> && val,int loopDepth) 
	{
															#line 2876 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(!val.present());
															#line 7481 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont10cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont10cont3(val, loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont10cont3(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2836Actor*>(this)->ActorCallback< FlowTestCase2836Actor, 7, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 7, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont10cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 7, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont10cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< FlowTestCase2836Actor, 7, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont10cont5(int loopDepth) 
	{
															#line 2884 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		Future<Void> closed = kvStore->onClosed();
															#line 2885 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->dispose();
															#line 2886 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_9 = closed;
															#line 2886 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7571 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont10cont5when1(__when_expr_9.get(), loopDepth); };
		static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 10;
															#line 2886 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2836Actor, 9, Void >*>(static_cast<FlowTestCase2836Actor*>(this)));
															#line 7576 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont10cont6(Optional<Value> const& val,int loopDepth) 
	{
															#line 2881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(keyAfter(bar) == val.get());
															#line 7585 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont10cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont6(Optional<Value> && val,int loopDepth) 
	{
															#line 2881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(keyAfter(bar) == val.get());
															#line 7594 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont10cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont2when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont10cont6(val, loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont2when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont10cont6(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2836Actor*>(this)->ActorCallback< FlowTestCase2836Actor, 8, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 8, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont10cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 8, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont10cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< FlowTestCase2836Actor, 8, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont10cont8(Void const& _,int loopDepth) 
	{
															#line 2888 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 2889 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<FlowTestCase2836Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2836ActorState(); static_cast<FlowTestCase2836Actor*>(this)->destroy(); return 0; }
															#line 7680 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase2836Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2836ActorState();
		static_cast<FlowTestCase2836Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10cont8(Void && _,int loopDepth) 
	{
															#line 2888 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 2889 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<FlowTestCase2836Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2836ActorState(); static_cast<FlowTestCase2836Actor*>(this)->destroy(); return 0; }
															#line 7694 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase2836Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2836ActorState();
		static_cast<FlowTestCase2836Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2836Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2836Actor*>(this)->ActorCallback< FlowTestCase2836Actor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont10cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2836Actor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont10cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< FlowTestCase2836Actor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), 9);

	}
															#line 2836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UnitTestParameters params;
															#line 2837 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	const std::string rocksDBTestDir;
															#line 2840 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	IKeyValueStore* kvStore;
															#line 2843 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	StringRef foo;
															#line 2844 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	StringRef bar;
															#line 7787 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2836()
															#line 2836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase2836Actor final : public Actor<Void>, public ActorCallback< FlowTestCase2836Actor, 0, Void >, public ActorCallback< FlowTestCase2836Actor, 1, Void >, public ActorCallback< FlowTestCase2836Actor, 2, Optional<Value> >, public ActorCallback< FlowTestCase2836Actor, 3, Void >, public ActorCallback< FlowTestCase2836Actor, 4, Optional<Value> >, public ActorCallback< FlowTestCase2836Actor, 5, Optional<Value> >, public ActorCallback< FlowTestCase2836Actor, 6, Void >, public ActorCallback< FlowTestCase2836Actor, 7, Optional<Value> >, public ActorCallback< FlowTestCase2836Actor, 8, Optional<Value> >, public ActorCallback< FlowTestCase2836Actor, 9, Void >, public FastAllocated<FlowTestCase2836Actor>, public FlowTestCase2836ActorState<FlowTestCase2836Actor> {
															#line 7792 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2836Actor>::operator new;
	using FastAllocated<FlowTestCase2836Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9010982958365724672UL, 11558817570802493696UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase2836Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase2836Actor, 1, Void >;
friend struct ActorCallback< FlowTestCase2836Actor, 2, Optional<Value> >;
friend struct ActorCallback< FlowTestCase2836Actor, 3, Void >;
friend struct ActorCallback< FlowTestCase2836Actor, 4, Optional<Value> >;
friend struct ActorCallback< FlowTestCase2836Actor, 5, Optional<Value> >;
friend struct ActorCallback< FlowTestCase2836Actor, 6, Void >;
friend struct ActorCallback< FlowTestCase2836Actor, 7, Optional<Value> >;
friend struct ActorCallback< FlowTestCase2836Actor, 8, Optional<Value> >;
friend struct ActorCallback< FlowTestCase2836Actor, 9, Void >;
															#line 2836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase2836Actor(UnitTestParameters const& params) 
															#line 7818 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2836ActorState<FlowTestCase2836Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2836", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17644504824809698304UL, 2913119859251144192UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2836Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2836");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2836", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase2836Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase2836Actor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FlowTestCase2836Actor, 2, Optional<Value> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FlowTestCase2836Actor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FlowTestCase2836Actor, 4, Optional<Value> >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< FlowTestCase2836Actor, 5, Optional<Value> >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< FlowTestCase2836Actor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< FlowTestCase2836Actor, 7, Optional<Value> >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< FlowTestCase2836Actor, 8, Optional<Value> >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< FlowTestCase2836Actor, 9, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
static Future<Void> flowTestCase2836( UnitTestParameters const& params ) {
															#line 2836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new FlowTestCase2836Actor(params));
															#line 7859 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2836, "noSim/fdbserver/KeyValueStoreRocksDB/RocksDBBasic")

#line 2891 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

															#line 7865 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via flowTestCase2892()
															#line 2892 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class FlowTestCase2892Actor>
															#line 2892 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase2892ActorState {
															#line 7871 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 2892 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase2892ActorState(UnitTestParameters const& params) 
															#line 2892 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 2892 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : params(params),
															#line 2893 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   rocksDBTestDir("rocksdb-kvstore-reopen-test-db")
															#line 7880 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2892", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2892ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2892", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2894 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 2896 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			kvStore = new RocksDBKeyValueStore(rocksDBTestDir, deterministicRandom()->randomUniqueID());
															#line 2897 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Void> __when_expr_0 = kvStore->init();
															#line 2897 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7901 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 1;
															#line 2897 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 0, Void >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 7906 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2892ActorState();
		static_cast<FlowTestCase2892Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2899 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ "foo"_sr, "bar"_sr });
															#line 2900 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = kvStore->commit(false);
															#line 2900 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7933 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 2;
															#line 2900 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 1, Void >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 7938 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2899 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ "foo"_sr, "bar"_sr });
															#line 2900 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = kvStore->commit(false);
															#line 2900 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7951 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 2;
															#line 2900 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 1, Void >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 7956 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2892Actor*>(this)->ActorCallback< FlowTestCase2892Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase2892Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2902 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_2 = kvStore->readValue("foo"_sr);
															#line 2902 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8042 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 3;
															#line 2902 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 2, Optional<Value> >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 8047 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2902 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_2 = kvStore->readValue("foo"_sr);
															#line 2902 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8058 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 3;
															#line 2902 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 2, Optional<Value> >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 8063 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2892Actor*>(this)->ActorCallback< FlowTestCase2892Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase2892Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Optional<Value> const& val,int loopDepth) 
	{
															#line 2903 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(Optional<Value>("bar"_sr) == val);
															#line 2905 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		Future<Void> closed = kvStore->onClosed();
															#line 2906 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->close();
															#line 2907 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_3 = closed;
															#line 2907 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8155 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 4;
															#line 2907 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 3, Void >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 8160 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<Value> && val,int loopDepth) 
	{
															#line 2903 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(Optional<Value>("bar"_sr) == val);
															#line 2905 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		Future<Void> closed = kvStore->onClosed();
															#line 2906 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->close();
															#line 2907 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_3 = closed;
															#line 2907 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8177 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 4;
															#line 2907 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 3, Void >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 8182 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont3(val, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2892Actor*>(this)->ActorCallback< FlowTestCase2892Actor, 2, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 2, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 2, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FlowTestCase2892Actor, 2, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 2909 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore = new RocksDBKeyValueStore(rocksDBTestDir, deterministicRandom()->randomUniqueID());
															#line 2910 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_4 = kvStore->init();
															#line 2910 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8270 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 5;
															#line 2910 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 4, Void >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 8275 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 2909 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore = new RocksDBKeyValueStore(rocksDBTestDir, deterministicRandom()->randomUniqueID());
															#line 2910 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_4 = kvStore->init();
															#line 2910 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8288 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 5;
															#line 2910 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 4, Void >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 8293 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2892Actor*>(this)->ActorCallback< FlowTestCase2892Actor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FlowTestCase2892Actor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 2912 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_5 = kvStore->init();
															#line 2912 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8379 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont5when1(__when_expr_5.get(), loopDepth); };
		static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 6;
															#line 2912 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 5, Void >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 8384 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 2912 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_5 = kvStore->init();
															#line 2912 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8395 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont5when1(__when_expr_5.get(), loopDepth); };
		static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 6;
															#line 2912 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 5, Void >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 8400 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2892Actor*>(this)->ActorCallback< FlowTestCase2892Actor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FlowTestCase2892Actor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		{
															#line 2915 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_6 = kvStore->readValue("foo"_sr);
															#line 2915 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8487 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont6when1(__when_expr_6.get(), loopDepth); };
			static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 7;
															#line 2915 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 6, Optional<Value> >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 8492 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		{
															#line 2915 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_6 = kvStore->readValue("foo"_sr);
															#line 2915 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8505 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont6when1(__when_expr_6.get(), loopDepth); };
			static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 7;
															#line 2915 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 6, Optional<Value> >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 8510 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2892Actor*>(this)->ActorCallback< FlowTestCase2892Actor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< FlowTestCase2892Actor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont7(int loopDepth) 
	{
		{
															#line 2920 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			Future<Void> closed = kvStore->onClosed();
															#line 2921 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			kvStore->dispose();
															#line 2922 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Void> __when_expr_7 = closed;
															#line 2922 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8602 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont7when1(__when_expr_7.get(), loopDepth); };
			static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 8;
															#line 2922 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2892Actor, 7, Void >*>(static_cast<FlowTestCase2892Actor*>(this)));
															#line 8607 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> const& val,int loopDepth) 
	{
															#line 2916 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(Optional<Value>("bar"_sr) == val);
															#line 8617 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> && val,int loopDepth) 
	{
															#line 2916 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(Optional<Value>("bar"_sr) == val);
															#line 8626 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont8(val, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2892Actor*>(this)->ActorCallback< FlowTestCase2892Actor, 6, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 6, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 6, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< FlowTestCase2892Actor, 6, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont10(int loopDepth) 
	{
															#line 2925 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 2926 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<FlowTestCase2892Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2892ActorState(); static_cast<FlowTestCase2892Actor*>(this)->destroy(); return 0; }
															#line 8712 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase2892Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2892ActorState();
		static_cast<FlowTestCase2892Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2892Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2892Actor*>(this)->ActorCallback< FlowTestCase2892Actor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2892Actor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< FlowTestCase2892Actor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), 7);

	}
															#line 2892 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UnitTestParameters params;
															#line 2893 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	const std::string rocksDBTestDir;
															#line 2896 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	IKeyValueStore* kvStore;
															#line 8813 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2892()
															#line 2892 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase2892Actor final : public Actor<Void>, public ActorCallback< FlowTestCase2892Actor, 0, Void >, public ActorCallback< FlowTestCase2892Actor, 1, Void >, public ActorCallback< FlowTestCase2892Actor, 2, Optional<Value> >, public ActorCallback< FlowTestCase2892Actor, 3, Void >, public ActorCallback< FlowTestCase2892Actor, 4, Void >, public ActorCallback< FlowTestCase2892Actor, 5, Void >, public ActorCallback< FlowTestCase2892Actor, 6, Optional<Value> >, public ActorCallback< FlowTestCase2892Actor, 7, Void >, public FastAllocated<FlowTestCase2892Actor>, public FlowTestCase2892ActorState<FlowTestCase2892Actor> {
															#line 8818 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2892Actor>::operator new;
	using FastAllocated<FlowTestCase2892Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1493999452268137216UL, 9546354774490052096UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase2892Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase2892Actor, 1, Void >;
friend struct ActorCallback< FlowTestCase2892Actor, 2, Optional<Value> >;
friend struct ActorCallback< FlowTestCase2892Actor, 3, Void >;
friend struct ActorCallback< FlowTestCase2892Actor, 4, Void >;
friend struct ActorCallback< FlowTestCase2892Actor, 5, Void >;
friend struct ActorCallback< FlowTestCase2892Actor, 6, Optional<Value> >;
friend struct ActorCallback< FlowTestCase2892Actor, 7, Void >;
															#line 2892 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase2892Actor(UnitTestParameters const& params) 
															#line 8842 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2892ActorState<FlowTestCase2892Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2892", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7131020592421591808UL, 16477339081648044544UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2892Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2892");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2892", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase2892Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase2892Actor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FlowTestCase2892Actor, 2, Optional<Value> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FlowTestCase2892Actor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FlowTestCase2892Actor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< FlowTestCase2892Actor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< FlowTestCase2892Actor, 6, Optional<Value> >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< FlowTestCase2892Actor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2892 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
static Future<Void> flowTestCase2892( UnitTestParameters const& params ) {
															#line 2892 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new FlowTestCase2892Actor(params));
															#line 8881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2892, "noSim/fdbserver/KeyValueStoreRocksDB/RocksDBReopen")

#line 2928 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

															#line 8887 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via flowTestCase2929()
															#line 2929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class FlowTestCase2929Actor>
															#line 2929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase2929ActorState {
															#line 8893 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 2929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase2929ActorState(UnitTestParameters const& params) 
															#line 2929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 2929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : params(params),
															#line 2930 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   cwd(platform::getWorkingDirectory() + "/"),
															#line 2931 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   rocksDBTestDir("rocksdb-kvstore-br-test-db")
															#line 8904 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2929", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2929ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2929", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2932 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 2934 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			kvStore = new RocksDBKeyValueStore(rocksDBTestDir, deterministicRandom()->randomUniqueID());
															#line 2935 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Void> __when_expr_0 = kvStore->init();
															#line 2935 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8925 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 1;
															#line 2935 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 0, Void >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 8930 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2929ActorState();
		static_cast<FlowTestCase2929Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2937 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ "foo"_sr, "bar"_sr });
															#line 2938 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = kvStore->commit(false);
															#line 2938 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8957 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 2;
															#line 2938 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 1, Void >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 8962 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2937 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ "foo"_sr, "bar"_sr });
															#line 2938 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = kvStore->commit(false);
															#line 2938 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8975 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 2;
															#line 2938 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 1, Void >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 8980 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2929Actor*>(this)->ActorCallback< FlowTestCase2929Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase2929Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2940 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_2 = kvStore->readValue("foo"_sr);
															#line 2940 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 3;
															#line 2940 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 2, Optional<Value> >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 9071 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2940 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_2 = kvStore->readValue("foo"_sr);
															#line 2940 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9082 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 3;
															#line 2940 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 2, Optional<Value> >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 9087 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2929Actor*>(this)->ActorCallback< FlowTestCase2929Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase2929Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Optional<Value> const& val,int loopDepth) 
	{
															#line 2941 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(Optional<Value>("bar"_sr) == val);
															#line 2943 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		rocksDBRestoreDir = "rocksdb-kvstore-br-restore-db";
															#line 2944 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(rocksDBRestoreDir);
															#line 2946 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStoreCopy = new RocksDBKeyValueStore(rocksDBRestoreDir, deterministicRandom()->randomUniqueID());
															#line 2948 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_3 = kvStoreCopy->init();
															#line 2948 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9181 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 4;
															#line 2948 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 3, Void >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 9186 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<Value> && val,int loopDepth) 
	{
															#line 2941 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(Optional<Value>("bar"_sr) == val);
															#line 2943 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		rocksDBRestoreDir = "rocksdb-kvstore-br-restore-db";
															#line 2944 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(rocksDBRestoreDir);
															#line 2946 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStoreCopy = new RocksDBKeyValueStore(rocksDBRestoreDir, deterministicRandom()->randomUniqueID());
															#line 2948 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_3 = kvStoreCopy->init();
															#line 2948 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9205 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 4;
															#line 2948 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 3, Void >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 9210 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont3(val, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2929Actor*>(this)->ActorCallback< FlowTestCase2929Actor, 2, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 2, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 2, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FlowTestCase2929Actor, 2, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 2950 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive("checkpoint");
															#line 2951 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		checkpointDir = cwd + "checkpoint";
															#line 2953 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		CheckpointRequest request( latestVersion, { allKeys }, DataMoveRocksCF, deterministicRandom()->randomUniqueID(), checkpointDir);
															#line 2955 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<CheckpointMetaData> __when_expr_4 = kvStore->checkpoint(request);
															#line 2955 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9302 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 5;
															#line 2955 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 4, CheckpointMetaData >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 9307 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 2950 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive("checkpoint");
															#line 2951 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		checkpointDir = cwd + "checkpoint";
															#line 2953 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		CheckpointRequest request( latestVersion, { allKeys }, DataMoveRocksCF, deterministicRandom()->randomUniqueID(), checkpointDir);
															#line 2955 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<CheckpointMetaData> __when_expr_4 = kvStore->checkpoint(request);
															#line 2955 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9324 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 5;
															#line 2955 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 4, CheckpointMetaData >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 9329 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2929Actor*>(this)->ActorCallback< FlowTestCase2929Actor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FlowTestCase2929Actor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont5(CheckpointMetaData const& metaData,int loopDepth) 
	{
															#line 2957 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::vector<CheckpointMetaData> checkpoints;
															#line 2958 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		checkpoints.push_back(metaData);
															#line 2959 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_5 = kvStoreCopy->restore(checkpoints);
															#line 2959 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9419 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont5when1(__when_expr_5.get(), loopDepth); };
		static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 6;
															#line 2959 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 5, Void >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 9424 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5(CheckpointMetaData && metaData,int loopDepth) 
	{
															#line 2957 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::vector<CheckpointMetaData> checkpoints;
															#line 2958 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		checkpoints.push_back(metaData);
															#line 2959 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_5 = kvStoreCopy->restore(checkpoints);
															#line 2959 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9439 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont5when1(__when_expr_5.get(), loopDepth); };
		static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 6;
															#line 2959 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 5, Void >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 9444 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4when1(CheckpointMetaData const& metaData,int loopDepth) 
	{
		loopDepth = a_body1cont5(metaData, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(CheckpointMetaData && metaData,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(metaData), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2929Actor*>(this)->ActorCallback< FlowTestCase2929Actor, 4, CheckpointMetaData >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 4, CheckpointMetaData >*,CheckpointMetaData const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 4, CheckpointMetaData >*,CheckpointMetaData && value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FlowTestCase2929Actor, 4, CheckpointMetaData >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
		{
															#line 2962 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_6 = kvStoreCopy->readValue("foo"_sr);
															#line 2962 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9531 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont6when1(__when_expr_6.get(), loopDepth); };
			static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 7;
															#line 2962 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 6, Optional<Value> >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 9536 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
		{
															#line 2962 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_6 = kvStoreCopy->readValue("foo"_sr);
															#line 2962 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9549 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont6when1(__when_expr_6.get(), loopDepth); };
			static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 7;
															#line 2962 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 6, Optional<Value> >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 9554 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2929Actor*>(this)->ActorCallback< FlowTestCase2929Actor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< FlowTestCase2929Actor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 2966 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::vector<Future<Void>> closes;
															#line 2967 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		closes.push_back(kvStore->onClosed());
															#line 2968 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		closes.push_back(kvStoreCopy->onClosed());
															#line 2969 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->dispose();
															#line 2970 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStoreCopy->dispose();
															#line 2971 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_7 = waitForAll(closes);
															#line 2971 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9651 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont7when1(__when_expr_7.get(), loopDepth); };
		static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 8;
															#line 2971 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2929Actor, 7, Void >*>(static_cast<FlowTestCase2929Actor*>(this)));
															#line 9656 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> const& val,int loopDepth) 
	{
															#line 2963 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(Optional<Value>("bar"_sr) == val);
															#line 9665 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Optional<Value> && val,int loopDepth) 
	{
															#line 2963 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(Optional<Value>("bar"_sr) == val);
															#line 9674 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont8(val, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2929Actor*>(this)->ActorCallback< FlowTestCase2929Actor, 6, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 6, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 6, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< FlowTestCase2929Actor, 6, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
															#line 2973 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 2974 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(rocksDBRestoreDir);
															#line 2976 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<FlowTestCase2929Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2929ActorState(); static_cast<FlowTestCase2929Actor*>(this)->destroy(); return 0; }
															#line 9762 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase2929Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2929ActorState();
		static_cast<FlowTestCase2929Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
															#line 2973 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 2974 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(rocksDBRestoreDir);
															#line 2976 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<FlowTestCase2929Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2929ActorState(); static_cast<FlowTestCase2929Actor*>(this)->destroy(); return 0; }
															#line 9778 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase2929Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2929ActorState();
		static_cast<FlowTestCase2929Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2929Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2929Actor*>(this)->ActorCallback< FlowTestCase2929Actor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2929Actor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< FlowTestCase2929Actor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), 7);

	}
															#line 2929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UnitTestParameters params;
															#line 2930 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string cwd;
															#line 2931 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string rocksDBTestDir;
															#line 2934 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	IKeyValueStore* kvStore;
															#line 2943 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string rocksDBRestoreDir;
															#line 2946 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	IKeyValueStore* kvStoreCopy;
															#line 2951 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string checkpointDir;
															#line 9875 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2929()
															#line 2929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase2929Actor final : public Actor<Void>, public ActorCallback< FlowTestCase2929Actor, 0, Void >, public ActorCallback< FlowTestCase2929Actor, 1, Void >, public ActorCallback< FlowTestCase2929Actor, 2, Optional<Value> >, public ActorCallback< FlowTestCase2929Actor, 3, Void >, public ActorCallback< FlowTestCase2929Actor, 4, CheckpointMetaData >, public ActorCallback< FlowTestCase2929Actor, 5, Void >, public ActorCallback< FlowTestCase2929Actor, 6, Optional<Value> >, public ActorCallback< FlowTestCase2929Actor, 7, Void >, public FastAllocated<FlowTestCase2929Actor>, public FlowTestCase2929ActorState<FlowTestCase2929Actor> {
															#line 9880 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2929Actor>::operator new;
	using FastAllocated<FlowTestCase2929Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2464335301301580544UL, 4468228854193023744UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase2929Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase2929Actor, 1, Void >;
friend struct ActorCallback< FlowTestCase2929Actor, 2, Optional<Value> >;
friend struct ActorCallback< FlowTestCase2929Actor, 3, Void >;
friend struct ActorCallback< FlowTestCase2929Actor, 4, CheckpointMetaData >;
friend struct ActorCallback< FlowTestCase2929Actor, 5, Void >;
friend struct ActorCallback< FlowTestCase2929Actor, 6, Optional<Value> >;
friend struct ActorCallback< FlowTestCase2929Actor, 7, Void >;
															#line 2929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase2929Actor(UnitTestParameters const& params) 
															#line 9904 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2929ActorState<FlowTestCase2929Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2929", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5783480388495546368UL, 12386124090955985408UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2929Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2929");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2929", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase2929Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase2929Actor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FlowTestCase2929Actor, 2, Optional<Value> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FlowTestCase2929Actor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FlowTestCase2929Actor, 4, CheckpointMetaData >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< FlowTestCase2929Actor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< FlowTestCase2929Actor, 6, Optional<Value> >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< FlowTestCase2929Actor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
static Future<Void> flowTestCase2929( UnitTestParameters const& params ) {
															#line 2929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new FlowTestCase2929Actor(params));
															#line 9943 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2929, "noSim/fdbserver/KeyValueStoreRocksDB/CheckpointRestoreColumnFamily")

#line 2978 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

															#line 9949 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via flowTestCase2979()
															#line 2979 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class FlowTestCase2979Actor>
															#line 2979 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase2979ActorState {
															#line 9955 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 2979 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase2979ActorState(UnitTestParameters const& params) 
															#line 2979 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 2979 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : params(params),
															#line 2980 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   cwd(platform::getWorkingDirectory() + "/"),
															#line 2981 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   rocksDBTestDir("rocksdb-kvstore-brsst-test-db")
															#line 9966 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2979", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2979ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2979", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2982 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 2983 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			kvStore = new RocksDBKeyValueStore(rocksDBTestDir, deterministicRandom()->randomUniqueID());
															#line 2984 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Void> __when_expr_0 = kvStore->init();
															#line 2984 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9987 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 1;
															#line 2984 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 0, Void >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 9992 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2979ActorState();
		static_cast<FlowTestCase2979Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2986 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ "foo"_sr, "bar"_sr });
															#line 2987 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = kvStore->commit(false);
															#line 2987 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10019 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 2;
															#line 2987 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 1, Void >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 10024 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2986 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ "foo"_sr, "bar"_sr });
															#line 2987 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = kvStore->commit(false);
															#line 2987 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10037 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 2;
															#line 2987 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 1, Void >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 10042 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2979Actor*>(this)->ActorCallback< FlowTestCase2979Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase2979Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2988 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_2 = kvStore->readValue("foo"_sr);
															#line 2988 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10128 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 3;
															#line 2988 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 2, Optional<Value> >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 10133 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2988 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_2 = kvStore->readValue("foo"_sr);
															#line 2988 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10144 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 3;
															#line 2988 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 2, Optional<Value> >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 10149 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2979Actor*>(this)->ActorCallback< FlowTestCase2979Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase2979Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Optional<Value> const& val,int loopDepth) 
	{
															#line 2989 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(Optional<Value>("bar"_sr) == val);
															#line 2991 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive("checkpoint");
															#line 2992 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::string checkpointDir = cwd + "checkpoint";
															#line 2994 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		CheckpointRequest request( latestVersion, { allKeys }, DataMoveRocksCF, deterministicRandom()->randomUniqueID(), checkpointDir);
															#line 2996 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<CheckpointMetaData> __when_expr_3 = kvStore->checkpoint(request);
															#line 2996 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10243 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 4;
															#line 2996 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 3, CheckpointMetaData >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 10248 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<Value> && val,int loopDepth) 
	{
															#line 2989 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(Optional<Value>("bar"_sr) == val);
															#line 2991 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive("checkpoint");
															#line 2992 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::string checkpointDir = cwd + "checkpoint";
															#line 2994 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		CheckpointRequest request( latestVersion, { allKeys }, DataMoveRocksCF, deterministicRandom()->randomUniqueID(), checkpointDir);
															#line 2996 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<CheckpointMetaData> __when_expr_3 = kvStore->checkpoint(request);
															#line 2996 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10267 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 4;
															#line 2996 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 3, CheckpointMetaData >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 10272 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont3(val, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2979Actor*>(this)->ActorCallback< FlowTestCase2979Actor, 2, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 2, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 2, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FlowTestCase2979Actor, 2, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(CheckpointMetaData const& metaData,int loopDepth) 
	{
															#line 2998 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		TraceEvent(SevDebug, "RocksDBCreatedCheckpoint");
															#line 2999 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		testRange = KeyRangeRef("foo"_sr, "foobar"_sr);
															#line 3000 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		token = BinaryWriter::toValue(testRange, IncludeVersion());
															#line 3001 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		cpReader = newCheckpointReader(metaData, CheckpointAsKeyValues::True, deterministicRandom()->randomUniqueID());
															#line 3003 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		TraceEvent(SevDebug, "RocksDBCheckpointReaderCreated");
															#line 3004 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(cpReader != nullptr);
															#line 3005 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_4 = cpReader->init(token);
															#line 3005 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10370 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 5;
															#line 3005 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 4, Void >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 10375 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(CheckpointMetaData && metaData,int loopDepth) 
	{
															#line 2998 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		TraceEvent(SevDebug, "RocksDBCreatedCheckpoint");
															#line 2999 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		testRange = KeyRangeRef("foo"_sr, "foobar"_sr);
															#line 3000 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		token = BinaryWriter::toValue(testRange, IncludeVersion());
															#line 3001 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		cpReader = newCheckpointReader(metaData, CheckpointAsKeyValues::True, deterministicRandom()->randomUniqueID());
															#line 3003 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		TraceEvent(SevDebug, "RocksDBCheckpointReaderCreated");
															#line 3004 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(cpReader != nullptr);
															#line 3005 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_4 = cpReader->init(token);
															#line 3005 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10398 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 5;
															#line 3005 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 4, Void >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 10403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3when1(CheckpointMetaData const& metaData,int loopDepth) 
	{
		loopDepth = a_body1cont4(metaData, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(CheckpointMetaData && metaData,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(metaData), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2979Actor*>(this)->ActorCallback< FlowTestCase2979Actor, 3, CheckpointMetaData >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 3, CheckpointMetaData >*,CheckpointMetaData const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 3, CheckpointMetaData >*,CheckpointMetaData && value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FlowTestCase2979Actor, 3, CheckpointMetaData >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 3006 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		TraceEvent(SevDebug, "RocksDBCheckpointReaderInited");
															#line 3007 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		iter = cpReader->getIterator(testRange);
															#line 3008 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		;
															#line 10491 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 3006 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		TraceEvent(SevDebug, "RocksDBCheckpointReaderInited");
															#line 3007 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		iter = cpReader->getIterator(testRange);
															#line 3008 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		;
															#line 10504 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2979Actor*>(this)->ActorCallback< FlowTestCase2979Actor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FlowTestCase2979Actor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 3026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		iter.reset();
															#line 3027 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		std::vector<Future<Void>> closes;
															#line 3028 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		closes.push_back(cpReader->close());
															#line 3029 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		closes.push_back(kvStore->onClosed());
															#line 3030 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->dispose();
															#line 3031 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_7 = waitForAll(closes);
															#line 3031 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 10600 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont6when1(__when_expr_7.get(), loopDepth); };
		static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 8;
															#line 3031 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 7, Void >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 10605 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont5loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1(int loopDepth) 
	{
		try {
															#line 3010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<RangeResult> __when_expr_5 = iter->nextBatch(CLIENT_KNOBS->REPLY_BYTE_LIMIT, CLIENT_KNOBS->REPLY_BYTE_LIMIT);
															#line 3010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1cont5loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 10624 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont5loopBody1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont5loopBody1when1(__when_expr_5.get(), loopDepth); };
			static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 6;
															#line 3010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 5, RangeResult >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 10629 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont5loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont5loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5break1(int loopDepth) 
	{
		try {
			return a_body1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont5loopHead1(0);

		return loopDepth;
	}
	int a_body1cont5loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 3018 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (e.code() == error_code_end_of_stream)
															#line 10664 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			{
				return a_body1cont5break1(loopDepth==0?0:loopDepth-1); // break
			}
			else
			{
															#line 3021 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
				TraceEvent(SevError, "TestFailed").error(e);
															#line 10672 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			}
			loopDepth = a_body1cont5loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2(int loopDepth) 
	{
															#line 3012 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		i = 0;
															#line 3013 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		;
															#line 10690 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(RangeResult const& __res,int loopDepth) 
	{
															#line 3010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		res = __res;
															#line 10699 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1when1(RangeResult && __res,int loopDepth) 
	{
		res = std::move(__res);
		loopDepth = a_body1cont5loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2979Actor*>(this)->ActorCallback< FlowTestCase2979Actor, 5, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 5, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 5, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< FlowTestCase2979Actor, 5, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont5loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont5loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 3013 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!(i < res.size()))
															#line 10791 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
			return a_body1cont5loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3014 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_6 = kvStore->readValue(res[i].key);
															#line 3014 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state < 0) return a_body1cont5loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10799 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont5loopBody1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont5loopBody1cont2loopBody1when1(__when_expr_6.get(), loopDepth); };
		static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 7;
															#line 3014 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2979Actor, 6, Optional<Value> >*>(static_cast<FlowTestCase2979Actor*>(this)));
															#line 10804 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1cont5loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont5loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont5loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2loopBody1cont1(Optional<Value> const& val,int loopDepth) 
	{
															#line 3015 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(val.present() && val.get() == res[i].value);
															#line 3013 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		++i;
															#line 10828 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont5loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2loopBody1cont1(Optional<Value> && val,int loopDepth) 
	{
															#line 3015 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(val.present() && val.get() == res[i].value);
															#line 3013 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		++i;
															#line 10839 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont5loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2loopBody1when1(Optional<Value> const& val,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont2loopBody1cont1(val, loopDepth);

		return loopDepth;
	}
	int a_body1cont5loopBody1cont2loopBody1when1(Optional<Value> && val,int loopDepth) 
	{
		loopDepth = a_body1cont5loopBody1cont2loopBody1cont1(std::move(val), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2979Actor*>(this)->ActorCallback< FlowTestCase2979Actor, 6, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 6, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont5loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 6, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont5loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< FlowTestCase2979Actor, 6, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont5loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont5loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont5loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont5loopBody1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont5loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 3033 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 3035 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<FlowTestCase2979Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2979ActorState(); static_cast<FlowTestCase2979Actor*>(this)->destroy(); return 0; }
															#line 10938 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase2979Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2979ActorState();
		static_cast<FlowTestCase2979Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 3033 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 3035 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<FlowTestCase2979Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2979ActorState(); static_cast<FlowTestCase2979Actor*>(this)->destroy(); return 0; }
															#line 10952 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase2979Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2979ActorState();
		static_cast<FlowTestCase2979Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2979Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2979Actor*>(this)->ActorCallback< FlowTestCase2979Actor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2979Actor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< FlowTestCase2979Actor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), 7);

	}
															#line 2979 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UnitTestParameters params;
															#line 2980 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string cwd;
															#line 2981 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string rocksDBTestDir;
															#line 2983 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	IKeyValueStore* kvStore;
															#line 2999 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	KeyRange testRange;
															#line 3000 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	Standalone<StringRef> token;
															#line 3001 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	ICheckpointReader* cpReader;
															#line 3007 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::unique_ptr<ICheckpointIterator> iter;
															#line 3010 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	RangeResult res;
															#line 3012 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	int i;
															#line 11055 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2979()
															#line 2979 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase2979Actor final : public Actor<Void>, public ActorCallback< FlowTestCase2979Actor, 0, Void >, public ActorCallback< FlowTestCase2979Actor, 1, Void >, public ActorCallback< FlowTestCase2979Actor, 2, Optional<Value> >, public ActorCallback< FlowTestCase2979Actor, 3, CheckpointMetaData >, public ActorCallback< FlowTestCase2979Actor, 4, Void >, public ActorCallback< FlowTestCase2979Actor, 5, RangeResult >, public ActorCallback< FlowTestCase2979Actor, 6, Optional<Value> >, public ActorCallback< FlowTestCase2979Actor, 7, Void >, public FastAllocated<FlowTestCase2979Actor>, public FlowTestCase2979ActorState<FlowTestCase2979Actor> {
															#line 11060 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2979Actor>::operator new;
	using FastAllocated<FlowTestCase2979Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8277586893373192448UL, 2898824125789333248UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase2979Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase2979Actor, 1, Void >;
friend struct ActorCallback< FlowTestCase2979Actor, 2, Optional<Value> >;
friend struct ActorCallback< FlowTestCase2979Actor, 3, CheckpointMetaData >;
friend struct ActorCallback< FlowTestCase2979Actor, 4, Void >;
friend struct ActorCallback< FlowTestCase2979Actor, 5, RangeResult >;
friend struct ActorCallback< FlowTestCase2979Actor, 6, Optional<Value> >;
friend struct ActorCallback< FlowTestCase2979Actor, 7, Void >;
															#line 2979 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase2979Actor(UnitTestParameters const& params) 
															#line 11084 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2979ActorState<FlowTestCase2979Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2979", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3492693846684944128UL, 16467299578121083136UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2979Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2979");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2979", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase2979Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase2979Actor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FlowTestCase2979Actor, 2, Optional<Value> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FlowTestCase2979Actor, 3, CheckpointMetaData >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FlowTestCase2979Actor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< FlowTestCase2979Actor, 5, RangeResult >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< FlowTestCase2979Actor, 6, Optional<Value> >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< FlowTestCase2979Actor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 2979 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
static Future<Void> flowTestCase2979( UnitTestParameters const& params ) {
															#line 2979 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new FlowTestCase2979Actor(params));
															#line 11123 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2979, "noSim/fdbserver/KeyValueStoreRocksDB/CheckpointRestoreKeyValues")

#line 3037 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

															#line 11129 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via flowTestCase3038()
															#line 3038 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class FlowTestCase3038Actor>
															#line 3038 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase3038ActorState {
															#line 11135 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 3038 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase3038ActorState(UnitTestParameters const& params) 
															#line 3038 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 3038 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : params(params),
															#line 3039 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   rocksDBTestDir("rocksdb-perf-db")
															#line 11144 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase3038", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase3038ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase3038", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3040 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			platform::eraseDirectoryRecursive(rocksDBTestDir);
															#line 3042 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			kvStore = new RocksDBKeyValueStore(rocksDBTestDir, deterministicRandom()->randomUniqueID());
															#line 3043 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Void> __when_expr_0 = kvStore->init();
															#line 3043 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11165 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state = 1;
															#line 3043 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3038Actor, 0, Void >*>(static_cast<FlowTestCase3038Actor*>(this)));
															#line 11170 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase3038ActorState();
		static_cast<FlowTestCase3038Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3045 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		shardPrefix = "\xffprefix/"_sr;
															#line 3047 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		i = 0;
															#line 3048 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		;
															#line 11197 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3045 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		shardPrefix = "\xffprefix/"_sr;
															#line 3047 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		i = 0;
															#line 3048 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		;
															#line 11210 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase3038Actor*>(this)->ActorCallback< FlowTestCase3038Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase3038Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3038Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase3038Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 3060 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		Future<Void> closed = kvStore->onClosed();
															#line 3061 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->dispose();
															#line 3062 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_3 = closed;
															#line 3062 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11300 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont2when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state = 4;
															#line 3062 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3038Actor, 3, Void >*>(static_cast<FlowTestCase3038Actor*>(this)));
															#line 11305 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 3048 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!(i < 50000))
															#line 11321 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 3049 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		key1 = format("\xffprefix/%d", i);
															#line 3050 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		key2 = format("\xffprefix/%d", i + 1);
															#line 3052 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->set({ key2, std::to_string(i) });
															#line 3053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = kvStore->readRange(KeyRangeRef(shardPrefix, key1), 10000, 10000);
															#line 3053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11335 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state = 2;
															#line 3053 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3038Actor, 1, RangeResult >*>(static_cast<FlowTestCase3038Actor*>(this)));
															#line 11340 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(RangeResult const& result,int loopDepth) 
	{
															#line 3054 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->clear({ KeyRangeRef(shardPrefix, key1) });
															#line 3055 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_2 = kvStore->commit(false);
															#line 3055 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11366 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state = 3;
															#line 3055 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3038Actor, 2, Void >*>(static_cast<FlowTestCase3038Actor*>(this)));
															#line 11371 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(RangeResult && result,int loopDepth) 
	{
															#line 3054 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->clear({ KeyRangeRef(shardPrefix, key1) });
															#line 3055 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_2 = kvStore->commit(false);
															#line 3055 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11384 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state = 3;
															#line 3055 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3038Actor, 2, Void >*>(static_cast<FlowTestCase3038Actor*>(this)));
															#line 11389 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(RangeResult const& result,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(result, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(RangeResult && result,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase3038Actor*>(this)->ActorCallback< FlowTestCase3038Actor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase3038Actor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3038Actor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase3038Actor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 3048 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		++i;
															#line 11473 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 3048 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		++i;
															#line 11482 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase3038Actor*>(this)->ActorCallback< FlowTestCase3038Actor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase3038Actor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3038Actor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FlowTestCase3038Actor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 3063 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<FlowTestCase3038Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase3038ActorState(); static_cast<FlowTestCase3038Actor*>(this)->destroy(); return 0; }
															#line 11566 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase3038Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase3038ActorState();
		static_cast<FlowTestCase3038Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 3063 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<FlowTestCase3038Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase3038ActorState(); static_cast<FlowTestCase3038Actor*>(this)->destroy(); return 0; }
															#line 11578 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase3038Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase3038ActorState();
		static_cast<FlowTestCase3038Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase3038Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase3038Actor*>(this)->ActorCallback< FlowTestCase3038Actor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase3038Actor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3038Actor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FlowTestCase3038Actor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 3038 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UnitTestParameters params;
															#line 3039 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	const std::string rocksDBTestDir;
															#line 3042 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	IKeyValueStore* kvStore;
															#line 3045 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	KeyRef shardPrefix;
															#line 3047 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	int i;
															#line 3049 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string key1;
															#line 3050 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string key2;
															#line 11675 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase3038()
															#line 3038 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase3038Actor final : public Actor<Void>, public ActorCallback< FlowTestCase3038Actor, 0, Void >, public ActorCallback< FlowTestCase3038Actor, 1, RangeResult >, public ActorCallback< FlowTestCase3038Actor, 2, Void >, public ActorCallback< FlowTestCase3038Actor, 3, Void >, public FastAllocated<FlowTestCase3038Actor>, public FlowTestCase3038ActorState<FlowTestCase3038Actor> {
															#line 11680 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase3038Actor>::operator new;
	using FastAllocated<FlowTestCase3038Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11390279148344738816UL, 10179670460740335616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase3038Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase3038Actor, 1, RangeResult >;
friend struct ActorCallback< FlowTestCase3038Actor, 2, Void >;
friend struct ActorCallback< FlowTestCase3038Actor, 3, Void >;
															#line 3038 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase3038Actor(UnitTestParameters const& params) 
															#line 11700 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase3038ActorState<FlowTestCase3038Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase3038", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(74412399730402560UL, 16764109078723212800UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3038Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase3038");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase3038", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase3038Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase3038Actor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FlowTestCase3038Actor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FlowTestCase3038Actor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 3038 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
static Future<Void> flowTestCase3038( UnitTestParameters const& params ) {
															#line 3038 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new FlowTestCase3038Actor(params));
															#line 11735 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase3038, "noSim/RocksDB/RangeClear")

#line 3065 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

															#line 11741 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
// This generated class is to be used only via flowTestCase3066()
															#line 3066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
template <class FlowTestCase3066Actor>
															#line 3066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase3066ActorState {
															#line 11747 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
															#line 3066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase3066ActorState(UnitTestParameters const& params) 
															#line 3066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
															#line 3066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		 : params(params),
															#line 3067 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   testDir("test_ingest_sst_visibility"),
															#line 3068 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   testStoreID(deterministicRandom()->randomUniqueID()),
															#line 3069 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		   kvStore(new RocksDBKeyValueStore(testDir, testStoreID))
															#line 11760 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase3066", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase3066ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase3066", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 3072 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			StrictFuture<Void> __when_expr_0 = kvStore->init();
															#line 3072 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			if (static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11777 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state = 1;
															#line 3072 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3066Actor, 0, Void >*>(static_cast<FlowTestCase3066Actor*>(this)));
															#line 11782 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase3066ActorState();
		static_cast<FlowTestCase3066Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 3075 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		sstFilename = "test.sst";
															#line 3076 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		sstFileFullPath = joinPath(testDir, sstFilename);
															#line 3077 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		rocksdb::SstFileWriter sstWriter(rocksdb::EnvOptions(), kvStore->sharedState->getOptions());
															#line 3078 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(sstWriter.Open(sstFileFullPath).ok());
															#line 3079 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(sstWriter.Put("test_key", "test_value").ok());
															#line 3080 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(sstWriter.Finish().ok());
															#line 3083 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		fileSetMap = std::make_shared<BulkLoadFileSetKeyMap>();
															#line 3084 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		dummyManifestFile = "dummy_manifest.txt";
															#line 3088 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		BulkLoadFileSet fileSet(testDir, "", dummyManifestFile, sstFilename, "", BulkLoadChecksum());
															#line 3095 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		fileSetMap->emplace_back(allKeys, fileSet);
															#line 3098 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = kvStore->ingestSSTFiles(fileSetMap);
															#line 3098 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11827 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state = 2;
															#line 3098 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3066Actor, 1, Void >*>(static_cast<FlowTestCase3066Actor*>(this)));
															#line 11832 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 3075 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		sstFilename = "test.sst";
															#line 3076 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		sstFileFullPath = joinPath(testDir, sstFilename);
															#line 3077 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		rocksdb::SstFileWriter sstWriter(rocksdb::EnvOptions(), kvStore->sharedState->getOptions());
															#line 3078 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(sstWriter.Open(sstFileFullPath).ok());
															#line 3079 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(sstWriter.Put("test_key", "test_value").ok());
															#line 3080 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(sstWriter.Finish().ok());
															#line 3083 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		fileSetMap = std::make_shared<BulkLoadFileSetKeyMap>();
															#line 3084 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		dummyManifestFile = "dummy_manifest.txt";
															#line 3088 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		BulkLoadFileSet fileSet(testDir, "", dummyManifestFile, sstFilename, "", BulkLoadChecksum());
															#line 3095 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		fileSetMap->emplace_back(allKeys, fileSet);
															#line 3098 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_1 = kvStore->ingestSSTFiles(fileSetMap);
															#line 3098 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11863 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state = 2;
															#line 3098 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3066Actor, 1, Void >*>(static_cast<FlowTestCase3066Actor*>(this)));
															#line 11868 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase3066Actor*>(this)->ActorCallback< FlowTestCase3066Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase3066Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3066Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase3066Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 3101 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_2 = kvStore->readValue("test_key"_sr, Optional<ReadOptions>());
															#line 3101 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11954 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state = 3;
															#line 3101 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3066Actor, 2, Optional<Value> >*>(static_cast<FlowTestCase3066Actor*>(this)));
															#line 11959 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 3101 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Optional<Value>> __when_expr_2 = kvStore->readValue("test_key"_sr, Optional<ReadOptions>());
															#line 3101 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 11970 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state = 3;
															#line 3101 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3066Actor, 2, Optional<Value> >*>(static_cast<FlowTestCase3066Actor*>(this)));
															#line 11975 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase3066Actor*>(this)->ActorCallback< FlowTestCase3066Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase3066Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3066Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase3066Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Optional<Value> const& value,int loopDepth) 
	{
															#line 3102 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(value.present());
															#line 3103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(value.get() == "test_value"_sr);
															#line 3106 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		Future<Void> closed = kvStore->onClosed();
															#line 3107 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->dispose();
															#line 3108 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_3 = closed;
															#line 3108 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12069 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state = 4;
															#line 3108 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3066Actor, 3, Void >*>(static_cast<FlowTestCase3066Actor*>(this)));
															#line 12074 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<Value> && value,int loopDepth) 
	{
															#line 3102 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(value.present());
															#line 3103 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		ASSERT(value.get() == "test_value"_sr);
															#line 3106 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		Future<Void> closed = kvStore->onClosed();
															#line 3107 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		kvStore->dispose();
															#line 3108 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		StrictFuture<Void> __when_expr_3 = closed;
															#line 3108 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 12093 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state = 4;
															#line 3108 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase3066Actor, 3, Void >*>(static_cast<FlowTestCase3066Actor*>(this)));
															#line 12098 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> const& value,int loopDepth) 
	{
		loopDepth = a_body1cont3(value, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Value> && value,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(value), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase3066Actor*>(this)->ActorCallback< FlowTestCase3066Actor, 2, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase3066Actor, 2, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3066Actor, 2, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FlowTestCase3066Actor, 2, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 3109 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(testDir);
															#line 3111 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<FlowTestCase3066Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase3066ActorState(); static_cast<FlowTestCase3066Actor*>(this)->destroy(); return 0; }
															#line 12184 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase3066Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase3066ActorState();
		static_cast<FlowTestCase3066Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 3109 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		platform::eraseDirectoryRecursive(testDir);
															#line 3111 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
		if (!static_cast<FlowTestCase3066Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase3066ActorState(); static_cast<FlowTestCase3066Actor*>(this)->destroy(); return 0; }
															#line 12198 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase3066Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase3066ActorState();
		static_cast<FlowTestCase3066Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase3066Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase3066Actor*>(this)->ActorCallback< FlowTestCase3066Actor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase3066Actor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FlowTestCase3066Actor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FlowTestCase3066Actor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 3066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UnitTestParameters params;
															#line 3067 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string testDir;
															#line 3068 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	UID testStoreID;
															#line 3069 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	RocksDBKeyValueStore* kvStore;
															#line 3075 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string sstFilename;
															#line 3076 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string sstFileFullPath;
															#line 3083 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::shared_ptr<BulkLoadFileSetKeyMap> fileSetMap;
															#line 3084 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	std::string dummyManifestFile;
															#line 12297 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase3066()
															#line 3066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
class FlowTestCase3066Actor final : public Actor<Void>, public ActorCallback< FlowTestCase3066Actor, 0, Void >, public ActorCallback< FlowTestCase3066Actor, 1, Void >, public ActorCallback< FlowTestCase3066Actor, 2, Optional<Value> >, public ActorCallback< FlowTestCase3066Actor, 3, Void >, public FastAllocated<FlowTestCase3066Actor>, public FlowTestCase3066ActorState<FlowTestCase3066Actor> {
															#line 12302 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase3066Actor>::operator new;
	using FastAllocated<FlowTestCase3066Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1605405039980655104UL, 16570637218318715904UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase3066Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase3066Actor, 1, Void >;
friend struct ActorCallback< FlowTestCase3066Actor, 2, Optional<Value> >;
friend struct ActorCallback< FlowTestCase3066Actor, 3, Void >;
															#line 3066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	FlowTestCase3066Actor(UnitTestParameters const& params) 
															#line 12322 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase3066ActorState<FlowTestCase3066Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase3066", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2145378300697336576UL, 3654257394827336448UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase3066Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase3066");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase3066", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase3066Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase3066Actor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FlowTestCase3066Actor, 2, Optional<Value> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FlowTestCase3066Actor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 3066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
static Future<Void> flowTestCase3066( UnitTestParameters const& params ) {
															#line 3066 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"
	return Future<Void>(new FlowTestCase3066Actor(params));
															#line 12357 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/KeyValueStoreRocksDB.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase3066, "noSim/fdbserver/KeyValueStoreRocksDB/IngestSSTFileVisibility")

#line 3113 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/KeyValueStoreRocksDB.actor.cpp"

} // namespace
#endif // WITH_ROCKSDB
