#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
/*
 * EncryptKeyProxy.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/BlobMetadataUtils.h"
#include "fdbclient/EncryptKeyProxyInterface.h"

#include "fdbrpc/Locality.h"
#include "fdbserver/KmsConnector.h"
#include "fdbserver/KmsConnectorInterface.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/RESTKmsConnector.h"
#include "fdbserver/ServerDBInfo.actor.h"
#include "fdbserver/SimKmsConnector.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "fdbserver/ServerDBInfo.h"
#include "flow/Arena.h"
#include "flow/CodeProbe.h"
#include "flow/EncryptUtils.h"
#include "flow/Error.h"
#include "flow/EventTypes.actor.h"
#include "flow/FastRef.h"
#include "flow/IRandom.h"
#include "flow/Knobs.h"
#include "flow/Trace.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"
#include "flow/network.h"

#include <boost/functional/hash.hpp>
#include <boost/mpl/not.hpp>
#include <limits>
#include <string>
#include <utility>
#include <memory>

#include "flow/actorcompiler.h" // This must be the last #include.

namespace {

const std::string REST_KMS_CONNECTOR_TYPE_STR = "RESTKmsConnector";
const std::string FDB_PREF_KMS_CONNECTOR_TYPE_STR = "FDBPerfKmsConnector";
const std::string FDB_SIM_KMS_CONNECTOR_TYPE_STR = "SimKmsConnector";

struct CipherKeyValidityTS {
	int64_t refreshAtTS;
	int64_t expAtTS;
};

bool canReplyWith(Error e) {
	switch (e.code()) {
	case error_code_encrypt_key_not_found:
	case error_code_encrypt_keys_fetch_failed:
	// FDB <-> KMS connection may be observing transient issues
	// Caller processes should consider reusing 'non-revocable' CipherKeys iff ONLY below error codes lead to CipherKey
	// refresh failure
	case error_code_timed_out:
	case error_code_connection_failed:
		return true;
	default:
		return false;
	}
}

bool isKmsConnectionError(Error e) {
	switch (e.code()) {
	case error_code_timed_out:
	case error_code_connection_failed:
		return true;
	default:
		return false;
	}
}

int64_t computeCipherRefreshTS(Optional<int64_t> refreshInterval, int64_t currTS) {
	int64_t refreshAtTS = -1;
	const int64_t defaultTTL = FLOW_KNOBS->ENCRYPT_CIPHER_KEY_CACHE_TTL;

	if (refreshInterval.present()) {
		if (refreshInterval.get() < 0) {
			// Never refresh the CipherKey
			refreshAtTS = std::numeric_limits<int64_t>::max();
		} else if (refreshInterval.get() > 0) {
			refreshAtTS = currTS + refreshInterval.get();
		} else {
			ASSERT(refreshInterval.get() == 0);
			// Fallback to default refreshInterval if not specified
			refreshAtTS = currTS + defaultTTL;
		}
	} else {
		// Fallback to default refreshInterval if not specified
		refreshAtTS = currTS + defaultTTL;
	}

	ASSERT(refreshAtTS > 0);

	return refreshAtTS;
}

int64_t computeCipherExpireTS(Optional<int64_t> expiryInterval, int64_t currTS, int64_t refreshAtTS) {
	int64_t expireAtTS = -1;

	ASSERT(refreshAtTS > 0);

	if (expiryInterval.present()) {
		if (expiryInterval.get() < 0) {
			// Non-revocable CipherKey, never expire
			expireAtTS = std::numeric_limits<int64_t>::max();
		} else if (expiryInterval.get() > 0) {
			expireAtTS = currTS + expiryInterval.get();
		} else {
			ASSERT(expiryInterval.get() == 0);
			// None supplied, match expiry to refresh timestamp
			expireAtTS = refreshAtTS;
		}
	} else {
		// None supplied, match expiry to refresh timestamp
		expireAtTS = refreshAtTS;
	}

	ASSERT(expireAtTS > 0);

	return expireAtTS;
}

CipherKeyValidityTS getCipherKeyValidityTS(Optional<int64_t> refreshInterval, Optional<int64_t> expiryInterval) {
	int64_t currTS = (int64_t)now();

	CipherKeyValidityTS validityTS;
	validityTS.refreshAtTS = computeCipherRefreshTS(refreshInterval, currTS);
	validityTS.expAtTS = computeCipherExpireTS(expiryInterval, currTS, validityTS.refreshAtTS);

	return validityTS;
}

} // namespace

struct EncryptBaseCipherKey {
	EncryptCipherDomainId domainId;
	EncryptCipherBaseKeyId baseCipherId;
	Standalone<StringRef> baseCipherKey;
	// Key check value for the 'baseCipher'
	EncryptCipherKeyCheckValue baseCipherKCV;
	// Timestamp after which the cached CipherKey is eligible for KMS refresh
	int64_t refreshAt;
	// Timestamp after which the cached CipherKey 'should' be considered as 'expired'
	// KMS can define two type of keys:
	// 1. Revocable CipherKeys    : CipherKeys that has a finite expiry interval.
	// 2. Non-revocable CipherKeys: CipherKeys which 'do not' expire, however, are still eligible for KMS refreshes to
	// support KMS CipherKey rotation.
	//
	// If/when CipherKey refresh fails due to transient outage in FDB <-> KMS connectivity, a caller is allowed to
	// leverage already cached CipherKey iff it is 'Non-revocable CipherKey'. PerpetualWiggle would update old/retired
	// CipherKeys with the latest CipherKeys sometime soon in the future.
	int64_t expireAt;

	EncryptBaseCipherKey()
	  : domainId(0), baseCipherId(0), baseCipherKey(StringRef()), baseCipherKCV(0), refreshAt(0), expireAt(0) {}
	explicit EncryptBaseCipherKey(EncryptCipherDomainId dId,
	                              EncryptCipherBaseKeyId cipherId,
	                              Standalone<StringRef> cipherKey,
	                              EncryptCipherKeyCheckValue cipherKCV,
	                              int64_t refAtTS,
	                              int64_t expAtTS)
	  : domainId(dId), baseCipherId(cipherId), baseCipherKey(cipherKey), baseCipherKCV(cipherKCV), refreshAt(refAtTS),
	    expireAt(expAtTS) {}

	bool needsRefresh() const {
		bool shouldRefresh = now() > refreshAt;
		CODE_PROBE(shouldRefresh, "EKP: Key needs refresh");
		return shouldRefresh;
	}

	bool isExpired() const {
		bool expired = now() > expireAt;
		CODE_PROBE(expired, "EKP: Key is expired");
		return expired;
	}
};

// TODO: could refactor both into CacheEntry<T> with T data, creationTimeSec, and noExpiry
struct BlobMetadataCacheEntry {
	Standalone<BlobMetadataDetailsRef> metadataDetails;
	uint64_t creationTimeSec;

	BlobMetadataCacheEntry() : creationTimeSec(0) {}
	explicit BlobMetadataCacheEntry(Standalone<BlobMetadataDetailsRef> metadataDetails)
	  : metadataDetails(metadataDetails), creationTimeSec(now()) {}

	bool isValid() const { return (now() - creationTimeSec) < SERVER_KNOBS->BLOB_METADATA_CACHE_TTL; }
};

// TODO: Bound the size of the cache (implement LRU/LFU...)
using EncryptBaseDomainIdCache = std::unordered_map<EncryptCipherDomainId, EncryptBaseCipherKey>;

using EncryptBaseCipherDomainIdKeyIdCacheKey = std::pair<EncryptCipherDomainId, EncryptCipherBaseKeyId>;
using EncryptBaseCipherDomainIdKeyIdCacheKeyHash = boost::hash<EncryptBaseCipherDomainIdKeyIdCacheKey>;
using EncryptBaseCipherDomainIdKeyIdCache = std::unordered_map<EncryptBaseCipherDomainIdKeyIdCacheKey,
                                                               EncryptBaseCipherKey,
                                                               EncryptBaseCipherDomainIdKeyIdCacheKeyHash>;
using BlobMetadataDomainIdCache = std::unordered_map<BlobMetadataDomainId, BlobMetadataCacheEntry>;

struct EncryptKeyProxyData : NonCopyable, ReferenceCounted<EncryptKeyProxyData> {
public:
	UID myId;
	PromiseStream<Future<Void>> addActor;
	Future<Void> encryptionKeyRefresher;
	Future<Void> blobMetadataRefresher;
	Future<Void> healthChecker;
	Future<Void> logger;

	EncryptBaseDomainIdCache baseCipherDomainIdCache;
	EncryptBaseCipherDomainIdKeyIdCache baseCipherDomainIdKeyIdCache;
	BlobMetadataDomainIdCache blobMetadataDomainIdCache;

	std::unique_ptr<KmsConnector> kmsConnector;

	bool canConnectToKms = true;
	double canConnectToKmsLastUpdatedTS = 0;

	CounterCollection ekpCacheMetrics;

	Counter baseCipherKeyIdCacheMisses;
	Counter baseCipherKeyIdCacheHits;
	Counter baseCipherDomainIdCacheMisses;
	Counter baseCipherDomainIdCacheHits;
	Counter baseCipherKeysRefreshed;
	Counter numResponseWithErrors;
	Counter numEncryptionKeyRefreshErrors;
	Counter blobMetadataCacheHits;
	Counter blobMetadataCacheMisses;
	Counter blobMetadataRefreshed;
	Counter numBlobMetadataRefreshErrors;
	Counter numHealthCheckErrors;
	Counter numHealthCheckRequests;

	LatencySample kmsLookupByIdsReqLatency;
	LatencySample kmsLookupByDomainIdsReqLatency;
	LatencySample kmsBlobMetadataReqLatency;

	explicit EncryptKeyProxyData(UID id)
	  : myId(id), ekpCacheMetrics("EKPMetrics", myId.toString()),
	    baseCipherKeyIdCacheMisses("EKPCipherIdCacheMisses", ekpCacheMetrics),
	    baseCipherKeyIdCacheHits("EKPCipherIdCacheHits", ekpCacheMetrics),
	    baseCipherDomainIdCacheMisses("EKPCipherDomainIdCacheMisses", ekpCacheMetrics),
	    baseCipherDomainIdCacheHits("EKPCipherDomainIdCacheHits", ekpCacheMetrics),
	    baseCipherKeysRefreshed("EKPCipherKeysRefreshed", ekpCacheMetrics),
	    numResponseWithErrors("EKPNumResponseWithErrors", ekpCacheMetrics),
	    numEncryptionKeyRefreshErrors("EKPNumEncryptionKeyRefreshErrors", ekpCacheMetrics),
	    blobMetadataCacheHits("EKPBlobMetadataCacheHits", ekpCacheMetrics),
	    blobMetadataCacheMisses("EKPBlobMetadataCacheMisses", ekpCacheMetrics),
	    blobMetadataRefreshed("EKPBlobMetadataRefreshed", ekpCacheMetrics),
	    numBlobMetadataRefreshErrors("EKPBlobMetadataRefreshErrors", ekpCacheMetrics),
	    numHealthCheckErrors("KMSHealthCheckErrors", ekpCacheMetrics),
	    numHealthCheckRequests("KMSHealthCheckRequests", ekpCacheMetrics),
	    kmsLookupByIdsReqLatency("EKPKmsLookupByIdsReqLatency",
	                             id,
	                             SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
	                             SERVER_KNOBS->LATENCY_SKETCH_ACCURACY),
	    kmsLookupByDomainIdsReqLatency("EKPKmsLookupByDomainIdsReqLatency",
	                                   id,
	                                   SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
	                                   SERVER_KNOBS->LATENCY_SKETCH_ACCURACY),
	    kmsBlobMetadataReqLatency("EKPKmsBlobMetadataReqLatency",
	                              id,
	                              SERVER_KNOBS->LATENCY_METRICS_LOGGING_INTERVAL,
	                              SERVER_KNOBS->LATENCY_SKETCH_ACCURACY) {
		logger = ekpCacheMetrics.traceCounters(
		    "EncryptKeyProxyMetrics", id, SERVER_KNOBS->ENCRYPTION_LOGGING_INTERVAL, "EncryptKeyProxyMetrics");
	}

	void setKMSHealthiness(bool canConnect) {
		canConnectToKms = canConnect;
		canConnectToKmsLastUpdatedTS = now();
	}

	static EncryptBaseCipherDomainIdKeyIdCacheKey getBaseCipherDomainIdKeyIdCacheKey(
	    const EncryptCipherDomainId domainId,
	    const EncryptCipherBaseKeyId baseCipherId) {
		return std::make_pair(domainId, baseCipherId);
	}

	void insertIntoBaseDomainIdCache(const EncryptCipherDomainId domainId,
	                                 const EncryptCipherBaseKeyId baseCipherId,
	                                 Standalone<StringRef> baseCipherKey,
	                                 const EncryptCipherKeyCheckValue baseCipherKCV,
	                                 int64_t refreshAtTS,
	                                 int64_t expireAtTS) {
		// Entries in domainId cache are eligible for periodic refreshes to support 'limiting lifetime of encryption
		// key' support if enabled on external KMS solutions.

		baseCipherDomainIdCache[domainId] =
		    EncryptBaseCipherKey(domainId, baseCipherId, baseCipherKey, baseCipherKCV, refreshAtTS, expireAtTS);

		// Update cached the information indexed using baseCipherId
		// Cache indexed by 'baseCipherId' need not refresh cipher, however, it still needs to abide by KMS governed
		// CipherKey lifetime rules
		insertIntoBaseCipherIdCache(
		    domainId, baseCipherId, baseCipherKey, baseCipherKCV, std::numeric_limits<int64_t>::max(), expireAtTS);
	}

	void insertIntoBaseCipherIdCache(const EncryptCipherDomainId domainId,
	                                 const EncryptCipherBaseKeyId baseCipherId,
	                                 const Standalone<StringRef> baseCipherKey,
	                                 const EncryptCipherKeyCheckValue baseCipherKCV,
	                                 int64_t refreshAtTS,
	                                 int64_t expireAtTS) {
		// Given an cipherKey is immutable, it is OK to NOT expire cached information.
		// TODO: Update cache to support LRU eviction policy to limit the total cache size.
		const EncryptCipherKeyCheckValue computedKCV =
		    Sha256KCV().computeKCV(baseCipherKey.begin(), baseCipherKey.size());
		if (computedKCV != baseCipherKCV) {
			TraceEvent(SevWarnAlways, "BlobCipherKeyInitBaseCipherKCVMismatch")
			    .detail("DomId", domainId)
			    .detail("BaseCipherId", baseCipherId)
			    .detail("Computed", computedKCV)
			    .detail("BaseCipherKCV", baseCipherKCV);
			throw encrypt_key_check_value_mismatch();
		}
		EncryptBaseCipherDomainIdKeyIdCacheKey cacheKey = getBaseCipherDomainIdKeyIdCacheKey(domainId, baseCipherId);
		baseCipherDomainIdKeyIdCache[cacheKey] =
		    EncryptBaseCipherKey(domainId, baseCipherId, baseCipherKey, baseCipherKCV, refreshAtTS, expireAtTS);
		TraceEvent("InsertIntoBaseCipherIdCache")
		    .detail("DomId", domainId)
		    .detail("BaseCipherId", baseCipherId)
		    .detail("BaseCipherLen", baseCipherKey.size())
		    .detail("BaseCipherKCV", baseCipherKCV)
		    .detail("RefreshAt", refreshAtTS)
		    .detail("ExpireAt", expireAtTS);
	}

	void insertIntoBlobMetadataCache(const BlobMetadataDomainId domainId,
	                                 const Standalone<BlobMetadataDetailsRef>& entry) {
		blobMetadataDomainIdCache[domainId] = BlobMetadataCacheEntry(entry);
	}

	template <class Reply>
	using isEKPGetLatestBaseCipherKeysReply = std::is_base_of<EKPGetLatestBaseCipherKeysReply, Reply>;
	template <class Reply>
	using isEKPGetBaseCipherKeysByIdsReply = std::is_base_of<EKPGetBaseCipherKeysByIdsReply, Reply>;

	// For errors occurring due to invalid input parameters such as: invalid encryptionDomainId or
	// invalid baseCipherId, piggyback error with response to the client; approach allows clients
	// to take necessary corrective actions such as: clearing up cache with invalid ids, log relevant
	// details for further investigation etc.

	template <class Reply>
	typename std::enable_if<isEKPGetBaseCipherKeysByIdsReply<Reply>::value ||
	                            isEKPGetLatestBaseCipherKeysReply<Reply>::value,
	                        void>::type
	sendErrorResponse(const ReplyPromise<Reply>& promise, const Error& e) {
		Reply reply;
		++numResponseWithErrors;
		reply.error = e;
		promise.send(reply);
	}
};

std::unordered_map<std::pair<EncryptCipherDomainId, EncryptCipherBaseKeyId>,
                   EKPGetBaseCipherKeysRequestInfo,
                   boost::hash<std::pair<EncryptCipherDomainId, EncryptCipherBaseKeyId>>>
getLookupDetails(
    Reference<EncryptKeyProxyData> ekpProxyData,
    Optional<TraceEvent>& dbgTrace,
    EKPGetBaseCipherKeysByIdsReply& keyIdsReply,
    int& numHits,
    std::unordered_set<EKPGetBaseCipherKeysRequestInfo, EKPGetBaseCipherKeysRequestInfo_Hash> dedupedCipherInfos) {
	std::unordered_map<std::pair<EncryptCipherDomainId, EncryptCipherBaseKeyId>,
	                   EKPGetBaseCipherKeysRequestInfo,
	                   boost::hash<std::pair<EncryptCipherDomainId, EncryptCipherBaseKeyId>>>
	    lookupCipherInfoMap;
	if (dbgTrace.present()) {
		dbgTrace.get().detail("NKeys", dedupedCipherInfos.size());
		for (const auto& item : dedupedCipherInfos) {
			// Record {encryptDomainId, baseCipherId} queried
			dbgTrace.get().detail(
			    getEncryptDbgTraceKey(ENCRYPT_DBG_TRACE_QUERY_PREFIX, item.domainId, item.baseCipherId), "");
		}
	}

	for (const auto& item : dedupedCipherInfos) {
		const EncryptBaseCipherDomainIdKeyIdCacheKey cacheKey =
		    ekpProxyData->getBaseCipherDomainIdKeyIdCacheKey(item.domainId, item.baseCipherId);
		const auto itr = ekpProxyData->baseCipherDomainIdKeyIdCache.find(cacheKey);
		if (itr != ekpProxyData->baseCipherDomainIdKeyIdCache.end() && !itr->second.isExpired()) {
			keyIdsReply.baseCipherDetails.emplace_back(
			    itr->second.domainId, itr->second.baseCipherId, itr->second.baseCipherKey, itr->second.baseCipherKCV);
			numHits++;

			if (dbgTrace.present()) {
				// {encryptId, baseCipherId} forms a unique tuple across encryption domains
				dbgTrace.get().detail(getEncryptDbgTraceKey(ENCRYPT_DBG_TRACE_CACHED_PREFIX,
				                                            itr->second.domainId,
				                                            itr->second.baseCipherId),
				                      "");
			}
		} else {
			lookupCipherInfoMap.emplace(std::make_pair(item.domainId, item.baseCipherId), item);
		}
	}

	ASSERT_EQ(keyIdsReply.baseCipherDetails.size(), numHits);

	ekpProxyData->baseCipherKeyIdCacheHits += numHits;
	ekpProxyData->baseCipherKeyIdCacheMisses += lookupCipherInfoMap.size();
	return lookupCipherInfoMap;
}

															#line 427 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getCipherKeysByBaseCipherKeyIds()
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
template <class GetCipherKeysByBaseCipherKeyIdsActor>
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class GetCipherKeysByBaseCipherKeyIdsActorState {
															#line 434 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	GetCipherKeysByBaseCipherKeyIdsActorState(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf,EKPGetBaseCipherKeysByIdsRequest const& req) 
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		 : ekpProxyData(ekpProxyData),
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   kmsConnectorInf(kmsConnectorInf),
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   req(req),
															#line 430 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   numHits(0),
															#line 431 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   keysByIds(req),
															#line 432 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   keyIdsReply(),
															#line 433 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   dbgTrace(keysByIds.debugId.present() ? TraceEvent("GetByKeyIds", ekpProxyData->myId) : Optional<TraceEvent>())
															#line 453 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getCipherKeysByBaseCipherKeyIds", reinterpret_cast<unsigned long>(this));

	}
	~GetCipherKeysByBaseCipherKeyIdsActorState() 
	{
		fdb_probe_actor_destroy("getCipherKeysByBaseCipherKeyIds", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 436 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (dbgTrace.present())
															#line 468 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 437 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				dbgTrace.get().setMaxEventLength(SERVER_KNOBS->ENCRYPT_PROXY_MAX_DBG_TRACE_LENGTH);
															#line 438 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				dbgTrace.get().detail("DbgId", keysByIds.debugId.get());
															#line 474 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 443 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			std::unordered_set<EKPGetBaseCipherKeysRequestInfo, EKPGetBaseCipherKeysRequestInfo_Hash> dedupedCipherInfos;
															#line 444 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			for( const auto& item : req.baseCipherInfos ) {
															#line 445 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				dedupedCipherInfos.emplace(item);
															#line 482 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 448 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			lookupCipherInfoMap = getLookupDetails(ekpProxyData, dbgTrace, keyIdsReply, numHits, dedupedCipherInfos);
															#line 452 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (!lookupCipherInfoMap.empty())
															#line 488 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
				try {
															#line 454 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					KmsConnLookupEKsByKeyIdsReq keysByIdsReq;
															#line 455 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					for( const auto& item : lookupCipherInfoMap ) {
															#line 456 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
						keysByIdsReq.encryptKeyInfos.emplace_back(item.second.domainId, item.second.baseCipherId);
															#line 497 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					}
															#line 458 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					keysByIdsReq.debugId = keysByIds.debugId;
															#line 459 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					startTime = now();
															#line 460 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					StrictFuture<KmsConnLookupEKsByKeyIdsRep> __when_expr_0 = kmsConnectorInf.ekLookupByIds.getReply(keysByIdsReq);
															#line 460 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					if (static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 507 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->actor_wait_state = 1;
															#line 460 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetCipherKeysByBaseCipherKeyIdsActor, 0, KmsConnLookupEKsByKeyIdsRep >*>(static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)));
															#line 512 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					loopDepth = 0;
				}
				catch (Error& error) {
					loopDepth = a_body1Catch2(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1Catch2(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetCipherKeysByBaseCipherKeyIdsActorState();
		static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 519 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		keyIdsReply.numHits = numHits;
															#line 520 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		keysByIds.reply.send(keyIdsReply);
															#line 522 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		CODE_PROBE(!lookupCipherInfoMap.empty(), "EKP fetch cipherKeys by KeyId from KMS");
															#line 524 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (!static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetCipherKeysByBaseCipherKeyIdsActorState(); static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->destroy(); return 0; }
															#line 552 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		new (&static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetCipherKeysByBaseCipherKeyIdsActorState();
		static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 505 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (isKmsConnectionError(e))
															#line 571 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 506 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				ekpProxyData->setKMSHealthiness(false);
															#line 575 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 509 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (!canReplyWith(e))
															#line 579 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 510 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				TraceEvent("GetCipherKeysByKeyIds", ekpProxyData->myId).error(e);
															#line 511 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 585 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 513 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent("GetCipherKeysByKeyIds", ekpProxyData->myId).detail("ErrorCode", e.code());
															#line 514 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->sendErrorResponse(keysByIds.reply, e);
															#line 515 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (!static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetCipherKeysByBaseCipherKeyIdsActorState(); static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->destroy(); return 0; }
															#line 593 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			new (&static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~GetCipherKeysByBaseCipherKeyIdsActorState();
			static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(KmsConnLookupEKsByKeyIdsRep const& keysByIdsRep,int loopDepth) 
	{
															#line 461 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->kmsLookupByIdsReqLatency.addMeasurement(now() - startTime);
															#line 463 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		for( const auto& item : keysByIdsRep.cipherKeyDetails ) {
															#line 464 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			keyIdsReply.baseCipherDetails.emplace_back( item.encryptDomainId, item.encryptKeyId, item.encryptKey, item.encryptKCV);
															#line 615 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 471 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		for( auto& item : keysByIdsRep.cipherKeyDetails ) {
															#line 476 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			CipherKeyValidityTS validityTS = getCipherKeyValidityTS(Optional<int64_t>(-1), item.expireAfterSec);
															#line 478 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			const auto itr = lookupCipherInfoMap.find(std::make_pair(item.encryptDomainId, item.encryptKeyId));
															#line 479 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (itr == lookupCipherInfoMap.end())
															#line 625 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 480 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				TraceEvent(SevError, "GetCipherKeysByKeyIdsMappingNotFound", ekpProxyData->myId) .detail("DomainId", item.encryptDomainId);
															#line 482 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				return a_body1Catch2(encrypt_keys_fetch_failed(), loopDepth);
															#line 631 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 484 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->insertIntoBaseCipherIdCache(item.encryptDomainId, item.encryptKeyId, item.encryptKey, item.encryptKCV, validityTS.refreshAtTS, validityTS.expAtTS);
															#line 491 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (dbgTrace.present())
															#line 637 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 493 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				dbgTrace.get().detail(getEncryptDbgTraceKeyWithTS(ENCRYPT_DBG_TRACE_INSERT_PREFIX, item.encryptDomainId, item.encryptKeyId, validityTS.refreshAtTS, validityTS.expAtTS), "");
															#line 641 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
		}
															#line 501 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (keysByIdsRep.cipherKeyDetails.size() > 0)
															#line 646 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 502 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->setKMSHealthiness(true);
															#line 650 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(KmsConnLookupEKsByKeyIdsRep && keysByIdsRep,int loopDepth) 
	{
															#line 461 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->kmsLookupByIdsReqLatency.addMeasurement(now() - startTime);
															#line 463 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		for( const auto& item : keysByIdsRep.cipherKeyDetails ) {
															#line 464 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			keyIdsReply.baseCipherDetails.emplace_back( item.encryptDomainId, item.encryptKeyId, item.encryptKey, item.encryptKCV);
															#line 664 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 471 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		for( auto& item : keysByIdsRep.cipherKeyDetails ) {
															#line 476 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			CipherKeyValidityTS validityTS = getCipherKeyValidityTS(Optional<int64_t>(-1), item.expireAfterSec);
															#line 478 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			const auto itr = lookupCipherInfoMap.find(std::make_pair(item.encryptDomainId, item.encryptKeyId));
															#line 479 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (itr == lookupCipherInfoMap.end())
															#line 674 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 480 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				TraceEvent(SevError, "GetCipherKeysByKeyIdsMappingNotFound", ekpProxyData->myId) .detail("DomainId", item.encryptDomainId);
															#line 482 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				return a_body1Catch2(encrypt_keys_fetch_failed(), loopDepth);
															#line 680 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 484 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->insertIntoBaseCipherIdCache(item.encryptDomainId, item.encryptKeyId, item.encryptKey, item.encryptKCV, validityTS.refreshAtTS, validityTS.expAtTS);
															#line 491 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (dbgTrace.present())
															#line 686 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 493 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				dbgTrace.get().detail(getEncryptDbgTraceKeyWithTS(ENCRYPT_DBG_TRACE_INSERT_PREFIX, item.encryptDomainId, item.encryptKeyId, validityTS.refreshAtTS, validityTS.expAtTS), "");
															#line 690 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
		}
															#line 501 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (keysByIdsRep.cipherKeyDetails.size() > 0)
															#line 695 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 502 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->setKMSHealthiness(true);
															#line 699 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont13(loopDepth);

		return loopDepth;
	}
	int a_body1when1(KmsConnLookupEKsByKeyIdsRep const& keysByIdsRep,int loopDepth) 
	{
		loopDepth = a_body1cont5(keysByIdsRep, loopDepth);

		return loopDepth;
	}
	int a_body1when1(KmsConnLookupEKsByKeyIdsRep && keysByIdsRep,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(keysByIdsRep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->actor_wait_state > 0) static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->actor_wait_state = 0;
		static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->ActorCallback< GetCipherKeysByBaseCipherKeyIdsActor, 0, KmsConnLookupEKsByKeyIdsRep >::remove();

	}
	void a_callback_fire(ActorCallback< GetCipherKeysByBaseCipherKeyIdsActor, 0, KmsConnLookupEKsByKeyIdsRep >*,KmsConnLookupEKsByKeyIdsRep const& value) 
	{
		fdb_probe_actor_enter("getCipherKeysByBaseCipherKeyIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCipherKeysByBaseCipherKeyIds", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetCipherKeysByBaseCipherKeyIdsActor, 0, KmsConnLookupEKsByKeyIdsRep >*,KmsConnLookupEKsByKeyIdsRep && value) 
	{
		fdb_probe_actor_enter("getCipherKeysByBaseCipherKeyIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCipherKeysByBaseCipherKeyIds", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetCipherKeysByBaseCipherKeyIdsActor, 0, KmsConnLookupEKsByKeyIdsRep >*,Error err) 
	{
		fdb_probe_actor_enter("getCipherKeysByBaseCipherKeyIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getCipherKeysByBaseCipherKeyIds", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont13(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	Reference<EncryptKeyProxyData> ekpProxyData;
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	KmsConnectorInterface kmsConnectorInf;
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	EKPGetBaseCipherKeysByIdsRequest req;
															#line 430 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	int numHits;
															#line 431 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	EKPGetBaseCipherKeysByIdsRequest keysByIds;
															#line 432 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	EKPGetBaseCipherKeysByIdsReply keyIdsReply;
															#line 433 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	Optional<TraceEvent> dbgTrace;
															#line 448 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	std::unordered_map<std::pair<EncryptCipherDomainId, EncryptCipherBaseKeyId>, EKPGetBaseCipherKeysRequestInfo, boost::hash<std::pair<EncryptCipherDomainId, EncryptCipherBaseKeyId>>> lookupCipherInfoMap;
															#line 459 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	double startTime;
															#line 811 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getCipherKeysByBaseCipherKeyIds()
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class GetCipherKeysByBaseCipherKeyIdsActor final : public Actor<Void>, public ActorCallback< GetCipherKeysByBaseCipherKeyIdsActor, 0, KmsConnLookupEKsByKeyIdsRep >, public FastAllocated<GetCipherKeysByBaseCipherKeyIdsActor>, public GetCipherKeysByBaseCipherKeyIdsActorState<GetCipherKeysByBaseCipherKeyIdsActor> {
															#line 816 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetCipherKeysByBaseCipherKeyIdsActor>::operator new;
	using FastAllocated<GetCipherKeysByBaseCipherKeyIdsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15211099771576990464UL, 9623359321597673728UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetCipherKeysByBaseCipherKeyIdsActor, 0, KmsConnLookupEKsByKeyIdsRep >;
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	GetCipherKeysByBaseCipherKeyIdsActor(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf,EKPGetBaseCipherKeysByIdsRequest const& req) 
															#line 833 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   GetCipherKeysByBaseCipherKeyIdsActorState<GetCipherKeysByBaseCipherKeyIdsActor>(ekpProxyData, kmsConnectorInf, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getCipherKeysByBaseCipherKeyIds", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1114291997493772544UL, 15237477432170543616UL);
		ActorExecutionContextHelper __helper(static_cast<GetCipherKeysByBaseCipherKeyIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getCipherKeysByBaseCipherKeyIds");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getCipherKeysByBaseCipherKeyIds", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetCipherKeysByBaseCipherKeyIdsActor, 0, KmsConnLookupEKsByKeyIdsRep >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
[[nodiscard]] Future<Void> getCipherKeysByBaseCipherKeyIds( Reference<EncryptKeyProxyData> const& ekpProxyData, KmsConnectorInterface const& kmsConnectorInf, EKPGetBaseCipherKeysByIdsRequest const& req ) {
															#line 425 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	return Future<Void>(new GetCipherKeysByBaseCipherKeyIdsActor(ekpProxyData, kmsConnectorInf, req));
															#line 866 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
}

#line 526 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"

std::unordered_set<EncryptCipherDomainId> getLookupDetailsLatest(
    Reference<EncryptKeyProxyData> ekpProxyData,
    Optional<TraceEvent>& dbgTrace,
    EKPGetLatestBaseCipherKeysReply& latestCipherReply,
    int& numHits,
    std::unordered_set<EncryptCipherDomainId> dedupedDomainIds) {
	std::unordered_set<EncryptCipherDomainId> lookupCipherDomainIds;
	for (const auto domainId : dedupedDomainIds) {
		const auto itr = ekpProxyData->baseCipherDomainIdCache.find(domainId);
		if (itr != ekpProxyData->baseCipherDomainIdCache.end() && !itr->second.needsRefresh() &&
		    !itr->second.isExpired()) {
			latestCipherReply.baseCipherDetails.emplace_back(domainId,
			                                                 itr->second.baseCipherId,
			                                                 itr->second.baseCipherKey,
			                                                 itr->second.baseCipherKCV,
			                                                 itr->second.refreshAt,
			                                                 itr->second.expireAt);
			numHits++;

			if (dbgTrace.present()) {
				// {encryptDomainId, baseCipherId} forms a unique tuple across encryption domains
				dbgTrace.get().detail(getEncryptDbgTraceKeyWithTS(ENCRYPT_DBG_TRACE_CACHED_PREFIX,
				                                                  domainId,
				                                                  itr->second.baseCipherId,
				                                                  itr->second.refreshAt,
				                                                  itr->second.expireAt),
				                      "");
			}
		} else {
			lookupCipherDomainIds.emplace(domainId);
		}
	}

	ASSERT_EQ(numHits, latestCipherReply.baseCipherDetails.size());

	ekpProxyData->baseCipherDomainIdCacheHits += numHits;
	ekpProxyData->baseCipherDomainIdCacheMisses += lookupCipherDomainIds.size();
	return lookupCipherDomainIds;
}

															#line 911 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getLatestCipherKeys()
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
template <class GetLatestCipherKeysActor>
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class GetLatestCipherKeysActorState {
															#line 918 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	GetLatestCipherKeysActorState(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf,EKPGetLatestBaseCipherKeysRequest const& req) 
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		 : ekpProxyData(ekpProxyData),
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   kmsConnectorInf(kmsConnectorInf),
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   req(req),
															#line 572 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   numHits(0),
															#line 573 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   latestKeysReq(req),
															#line 574 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   latestCipherReply(),
															#line 575 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   dbgTrace(latestKeysReq.debugId.present() ? TraceEvent("GetByDomIds", ekpProxyData->myId) : Optional<TraceEvent>())
															#line 937 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getLatestCipherKeys", reinterpret_cast<unsigned long>(this));

	}
	~GetLatestCipherKeysActorState() 
	{
		fdb_probe_actor_destroy("getLatestCipherKeys", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 578 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (dbgTrace.present())
															#line 952 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 579 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				dbgTrace.get().setMaxEventLength(SERVER_KNOBS->ENCRYPT_PROXY_MAX_DBG_TRACE_LENGTH);
															#line 580 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				dbgTrace.get().detail("DbgId", latestKeysReq.debugId.get());
															#line 958 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 585 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			std::unordered_set<EncryptCipherDomainId> dedupedDomainIds;
															#line 586 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			for( const auto domainId : req.encryptDomainIds ) {
															#line 587 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				dedupedDomainIds.emplace(domainId);
															#line 966 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 590 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (dbgTrace.present())
															#line 970 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 591 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				dbgTrace.get().detail("NKeys", dedupedDomainIds.size());
															#line 592 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				for( const auto domainId : dedupedDomainIds ) {
															#line 594 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					dbgTrace.get().detail(getEncryptDbgTraceKey(ENCRYPT_DBG_TRACE_QUERY_PREFIX, domainId), "");
															#line 978 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				}
			}
															#line 600 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			lookupCipherDomainIds = getLookupDetailsLatest(ekpProxyData, dbgTrace, latestCipherReply, numHits, dedupedDomainIds);
															#line 602 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (!lookupCipherDomainIds.empty())
															#line 985 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
				try {
															#line 604 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					KmsConnLookupEKsByDomainIdsReq keysByDomainIdReq;
															#line 605 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					for( const auto domainId : lookupCipherDomainIds ) {
															#line 606 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
						keysByDomainIdReq.encryptDomainIds.emplace_back(domainId);
															#line 994 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					}
															#line 608 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					keysByDomainIdReq.debugId = latestKeysReq.debugId;
															#line 610 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					startTime = now();
															#line 611 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					StrictFuture<KmsConnLookupEKsByDomainIdsRep> __when_expr_0 = kmsConnectorInf.ekLookupByDomainIds.getReply(keysByDomainIdReq);
															#line 611 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					if (static_cast<GetLatestCipherKeysActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1004 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
					static_cast<GetLatestCipherKeysActor*>(this)->actor_wait_state = 1;
															#line 611 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetLatestCipherKeysActor, 0, KmsConnLookupEKsByDomainIdsRep >*>(static_cast<GetLatestCipherKeysActor*>(this)));
															#line 1009 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					loopDepth = 0;
				}
				catch (Error& error) {
					loopDepth = a_body1Catch2(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1Catch2(unknown_error(), loopDepth);
				}
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetLatestCipherKeysActorState();
		static_cast<GetLatestCipherKeysActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 666 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		latestCipherReply.numHits = numHits;
															#line 667 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		latestKeysReq.reply.send(latestCipherReply);
															#line 669 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		CODE_PROBE(!lookupCipherDomainIds.empty(), "EKP fetch latest cipherKeys from KMS");
															#line 671 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (!static_cast<GetLatestCipherKeysActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetLatestCipherKeysActorState(); static_cast<GetLatestCipherKeysActor*>(this)->destroy(); return 0; }
															#line 1049 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		new (&static_cast<GetLatestCipherKeysActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetLatestCipherKeysActorState();
		static_cast<GetLatestCipherKeysActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 653 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (isKmsConnectionError(e))
															#line 1068 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 654 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				ekpProxyData->setKMSHealthiness(false);
															#line 1072 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 656 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (!canReplyWith(e))
															#line 1076 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 657 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				TraceEvent("GetLatestCipherKeys", ekpProxyData->myId).error(e);
															#line 658 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 1082 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 660 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent("GetLatestCipherKeys", ekpProxyData->myId).detail("ErrorCode", e.code());
															#line 661 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->sendErrorResponse(latestKeysReq.reply, e);
															#line 662 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (!static_cast<GetLatestCipherKeysActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetLatestCipherKeysActorState(); static_cast<GetLatestCipherKeysActor*>(this)->destroy(); return 0; }
															#line 1090 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			new (&static_cast<GetLatestCipherKeysActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~GetLatestCipherKeysActorState();
			static_cast<GetLatestCipherKeysActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7(KmsConnLookupEKsByDomainIdsRep const& keysByDomainIdRep,int loopDepth) 
	{
															#line 613 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->kmsLookupByDomainIdsReqLatency.addMeasurement(now() - startTime);
															#line 615 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		for( auto& item : keysByDomainIdRep.cipherKeyDetails ) {
															#line 616 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			CipherKeyValidityTS validityTS = getCipherKeyValidityTS(item.refreshAfterSec, item.expireAfterSec);
															#line 618 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			latestCipherReply.baseCipherDetails.emplace_back(item.encryptDomainId, item.encryptKeyId, item.encryptKey, item.encryptKCV, validityTS.refreshAtTS, validityTS.expAtTS);
															#line 626 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			const auto itr = lookupCipherDomainIds.find(item.encryptDomainId);
															#line 627 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (itr == lookupCipherDomainIds.end())
															#line 1118 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 628 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				TraceEvent(SevError, "GetLatestCipherKeysDomainIdNotFound", ekpProxyData->myId) .detail("DomainId", item.encryptDomainId);
															#line 630 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				return a_body1Catch2(encrypt_keys_fetch_failed(), loopDepth);
															#line 1124 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 632 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->insertIntoBaseDomainIdCache(item.encryptDomainId, item.encryptKeyId, item.encryptKey, item.encryptKCV, validityTS.refreshAtTS, validityTS.expAtTS);
															#line 639 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (dbgTrace.present())
															#line 1130 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 641 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				dbgTrace.get().detail(getEncryptDbgTraceKeyWithTS(ENCRYPT_DBG_TRACE_INSERT_PREFIX, item.encryptDomainId, item.encryptKeyId, validityTS.refreshAtTS, validityTS.expAtTS), "");
															#line 1134 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
		}
															#line 649 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (keysByDomainIdRep.cipherKeyDetails.size() > 0)
															#line 1139 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 650 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->setKMSHealthiness(true);
															#line 1143 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(KmsConnLookupEKsByDomainIdsRep && keysByDomainIdRep,int loopDepth) 
	{
															#line 613 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->kmsLookupByDomainIdsReqLatency.addMeasurement(now() - startTime);
															#line 615 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		for( auto& item : keysByDomainIdRep.cipherKeyDetails ) {
															#line 616 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			CipherKeyValidityTS validityTS = getCipherKeyValidityTS(item.refreshAfterSec, item.expireAfterSec);
															#line 618 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			latestCipherReply.baseCipherDetails.emplace_back(item.encryptDomainId, item.encryptKeyId, item.encryptKey, item.encryptKCV, validityTS.refreshAtTS, validityTS.expAtTS);
															#line 626 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			const auto itr = lookupCipherDomainIds.find(item.encryptDomainId);
															#line 627 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (itr == lookupCipherDomainIds.end())
															#line 1163 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 628 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				TraceEvent(SevError, "GetLatestCipherKeysDomainIdNotFound", ekpProxyData->myId) .detail("DomainId", item.encryptDomainId);
															#line 630 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				return a_body1Catch2(encrypt_keys_fetch_failed(), loopDepth);
															#line 1169 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 632 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->insertIntoBaseDomainIdCache(item.encryptDomainId, item.encryptKeyId, item.encryptKey, item.encryptKCV, validityTS.refreshAtTS, validityTS.expAtTS);
															#line 639 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (dbgTrace.present())
															#line 1175 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 641 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				dbgTrace.get().detail(getEncryptDbgTraceKeyWithTS(ENCRYPT_DBG_TRACE_INSERT_PREFIX, item.encryptDomainId, item.encryptKeyId, validityTS.refreshAtTS, validityTS.expAtTS), "");
															#line 1179 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
		}
															#line 649 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (keysByDomainIdRep.cipherKeyDetails.size() > 0)
															#line 1184 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 650 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->setKMSHealthiness(true);
															#line 1188 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont14(loopDepth);

		return loopDepth;
	}
	int a_body1when1(KmsConnLookupEKsByDomainIdsRep const& keysByDomainIdRep,int loopDepth) 
	{
		loopDepth = a_body1cont7(keysByDomainIdRep, loopDepth);

		return loopDepth;
	}
	int a_body1when1(KmsConnLookupEKsByDomainIdsRep && keysByDomainIdRep,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(keysByDomainIdRep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetLatestCipherKeysActor*>(this)->actor_wait_state > 0) static_cast<GetLatestCipherKeysActor*>(this)->actor_wait_state = 0;
		static_cast<GetLatestCipherKeysActor*>(this)->ActorCallback< GetLatestCipherKeysActor, 0, KmsConnLookupEKsByDomainIdsRep >::remove();

	}
	void a_callback_fire(ActorCallback< GetLatestCipherKeysActor, 0, KmsConnLookupEKsByDomainIdsRep >*,KmsConnLookupEKsByDomainIdsRep const& value) 
	{
		fdb_probe_actor_enter("getLatestCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetLatestCipherKeysActor, 0, KmsConnLookupEKsByDomainIdsRep >*,KmsConnLookupEKsByDomainIdsRep && value) 
	{
		fdb_probe_actor_enter("getLatestCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetLatestCipherKeysActor, 0, KmsConnLookupEKsByDomainIdsRep >*,Error err) 
	{
		fdb_probe_actor_enter("getLatestCipherKeys", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getLatestCipherKeys", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont14(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	Reference<EncryptKeyProxyData> ekpProxyData;
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	KmsConnectorInterface kmsConnectorInf;
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	EKPGetLatestBaseCipherKeysRequest req;
															#line 572 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	int numHits;
															#line 573 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	EKPGetLatestBaseCipherKeysRequest latestKeysReq;
															#line 574 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	EKPGetLatestBaseCipherKeysReply latestCipherReply;
															#line 575 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	Optional<TraceEvent> dbgTrace;
															#line 600 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	std::unordered_set<EncryptCipherDomainId> lookupCipherDomainIds;
															#line 610 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	double startTime;
															#line 1300 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getLatestCipherKeys()
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class GetLatestCipherKeysActor final : public Actor<Void>, public ActorCallback< GetLatestCipherKeysActor, 0, KmsConnLookupEKsByDomainIdsRep >, public FastAllocated<GetLatestCipherKeysActor>, public GetLatestCipherKeysActorState<GetLatestCipherKeysActor> {
															#line 1305 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetLatestCipherKeysActor>::operator new;
	using FastAllocated<GetLatestCipherKeysActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4037216001881048064UL, 4231786441938043648UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetLatestCipherKeysActor, 0, KmsConnLookupEKsByDomainIdsRep >;
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	GetLatestCipherKeysActor(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf,EKPGetLatestBaseCipherKeysRequest const& req) 
															#line 1322 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   GetLatestCipherKeysActorState<GetLatestCipherKeysActor>(ekpProxyData, kmsConnectorInf, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getLatestCipherKeys", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8353423732201806848UL, 13421465162602851584UL);
		ActorExecutionContextHelper __helper(static_cast<GetLatestCipherKeysActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getLatestCipherKeys");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getLatestCipherKeys", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetLatestCipherKeysActor, 0, KmsConnLookupEKsByDomainIdsRep >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
[[nodiscard]] Future<Void> getLatestCipherKeys( Reference<EncryptKeyProxyData> const& ekpProxyData, KmsConnectorInterface const& kmsConnectorInf, EKPGetLatestBaseCipherKeysRequest const& req ) {
															#line 567 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	return Future<Void>(new GetLatestCipherKeysActor(ekpProxyData, kmsConnectorInf, req));
															#line 1355 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
}

#line 673 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"

bool isCipherKeyEligibleForRefresh(const EncryptBaseCipherKey& cipherKey, int64_t currTS) {
	// Candidate eligible for refresh iff either is true:
	// 1. CipherKey cell is either expired/needs-refresh right now.
	// 2. CipherKey cell 'will' be expired/needs-refresh before next refresh cycle interval (proactive refresh)
	if (BUGGIFY_WITH_PROB(0.01)) {
		return true;
	}
	int64_t nextRefreshCycleTS = currTS + FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL;
	return nextRefreshCycleTS > cipherKey.expireAt || nextRefreshCycleTS > cipherKey.refreshAt;
}

															#line 1371 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getHealthStatusImpl()
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
template <class GetHealthStatusImplActor>
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class GetHealthStatusImplActorState {
															#line 1378 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	GetHealthStatusImplActorState(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf) 
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		 : ekpProxyData(ekpProxyData),
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   kmsConnectorInf(kmsConnectorInf),
															#line 687 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   debugId(deterministicRandom()->randomUniqueID())
															#line 1389 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getHealthStatusImpl", reinterpret_cast<unsigned long>(this));

	}
	~GetHealthStatusImplActorState() 
	{
		fdb_probe_actor_destroy("getHealthStatusImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 688 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (DEBUG_ENCRYPT_KEY_PROXY)
															#line 1404 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 689 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				TraceEvent(SevDebug, "KMSHealthCheckStart", ekpProxyData->myId);
															#line 1408 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
			try {
															#line 694 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				KmsConnLookupEKsByDomainIdsReq req;
															#line 695 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				req.debugId = debugId;
															#line 696 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				req.encryptDomainIds.push_back(SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID);
															#line 697 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				++ekpProxyData->numHealthCheckRequests;
															#line 698 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				StrictFuture<KmsConnLookupEKsByDomainIdsRep> __when_expr_0 = timeoutError(kmsConnectorInf.ekLookupByDomainIds.getReply(req), FLOW_KNOBS->EKP_HEALTH_CHECK_REQUEST_TIMEOUT);
															#line 698 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				if (static_cast<GetHealthStatusImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 1423 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetHealthStatusImplActor*>(this)->actor_wait_state = 1;
															#line 698 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetHealthStatusImplActor, 0, KmsConnLookupEKsByDomainIdsRep >*>(static_cast<GetHealthStatusImplActor*>(this)));
															#line 1428 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetHealthStatusImplActorState();
		static_cast<GetHealthStatusImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 725 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (!static_cast<GetHealthStatusImplActor*>(this)->SAV<bool>::futures) { (void)(false); this->~GetHealthStatusImplActorState(); static_cast<GetHealthStatusImplActor*>(this)->destroy(); return 0; }
															#line 1457 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		new (&static_cast<GetHealthStatusImplActor*>(this)->SAV< bool >::value()) bool(false);
		this->~GetHealthStatusImplActorState();
		static_cast<GetHealthStatusImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 719 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent(SevWarn, "KMSHealthCheckError").error(e);
															#line 720 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (!canReplyWith(e))
															#line 1472 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 721 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 1476 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 723 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			++ekpProxyData->numHealthCheckErrors;
															#line 1480 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(KmsConnLookupEKsByDomainIdsRep const& rep,int loopDepth) 
	{
															#line 700 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (rep.cipherKeyDetails.size() < 1)
															#line 1495 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 701 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent(SevWarn, "KMSHealthCheckResponseEmpty");
															#line 702 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			return a_body1Catch2(encrypt_key_not_found(), loopDepth);
															#line 1501 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 704 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		EncryptCipherKeyDetailsRef cipherDetails = rep.cipherKeyDetails[0];
															#line 705 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (cipherDetails.encryptDomainId != SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID)
															#line 1507 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 706 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent(SevWarn, "KMSHealthCheckNoSystemKeyFound");
															#line 707 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			return a_body1Catch2(encrypt_key_not_found(), loopDepth);
															#line 1513 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 709 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		CipherKeyValidityTS validityTS = getCipherKeyValidityTS(cipherDetails.refreshAfterSec, cipherDetails.expireAfterSec);
															#line 711 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->insertIntoBaseDomainIdCache(cipherDetails.encryptDomainId, cipherDetails.encryptKeyId, cipherDetails.encryptKey, cipherDetails.encryptKCV, validityTS.refreshAtTS, validityTS.expAtTS);
															#line 717 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (!static_cast<GetHealthStatusImplActor*>(this)->SAV<bool>::futures) { (void)(true); this->~GetHealthStatusImplActorState(); static_cast<GetHealthStatusImplActor*>(this)->destroy(); return 0; }
															#line 1521 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		new (&static_cast<GetHealthStatusImplActor*>(this)->SAV< bool >::value()) bool(true);
		this->~GetHealthStatusImplActorState();
		static_cast<GetHealthStatusImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(KmsConnLookupEKsByDomainIdsRep && rep,int loopDepth) 
	{
															#line 700 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (rep.cipherKeyDetails.size() < 1)
															#line 1533 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 701 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent(SevWarn, "KMSHealthCheckResponseEmpty");
															#line 702 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			return a_body1Catch2(encrypt_key_not_found(), loopDepth);
															#line 1539 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 704 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		EncryptCipherKeyDetailsRef cipherDetails = rep.cipherKeyDetails[0];
															#line 705 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (cipherDetails.encryptDomainId != SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID)
															#line 1545 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 706 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent(SevWarn, "KMSHealthCheckNoSystemKeyFound");
															#line 707 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			return a_body1Catch2(encrypt_key_not_found(), loopDepth);
															#line 1551 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 709 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		CipherKeyValidityTS validityTS = getCipherKeyValidityTS(cipherDetails.refreshAfterSec, cipherDetails.expireAfterSec);
															#line 711 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->insertIntoBaseDomainIdCache(cipherDetails.encryptDomainId, cipherDetails.encryptKeyId, cipherDetails.encryptKey, cipherDetails.encryptKCV, validityTS.refreshAtTS, validityTS.expAtTS);
															#line 717 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (!static_cast<GetHealthStatusImplActor*>(this)->SAV<bool>::futures) { (void)(true); this->~GetHealthStatusImplActorState(); static_cast<GetHealthStatusImplActor*>(this)->destroy(); return 0; }
															#line 1559 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		new (&static_cast<GetHealthStatusImplActor*>(this)->SAV< bool >::value()) bool(true);
		this->~GetHealthStatusImplActorState();
		static_cast<GetHealthStatusImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(KmsConnLookupEKsByDomainIdsRep const& rep,int loopDepth) 
	{
		loopDepth = a_body1cont3(rep, loopDepth);

		return loopDepth;
	}
	int a_body1when1(KmsConnLookupEKsByDomainIdsRep && rep,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(rep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetHealthStatusImplActor*>(this)->actor_wait_state > 0) static_cast<GetHealthStatusImplActor*>(this)->actor_wait_state = 0;
		static_cast<GetHealthStatusImplActor*>(this)->ActorCallback< GetHealthStatusImplActor, 0, KmsConnLookupEKsByDomainIdsRep >::remove();

	}
	void a_callback_fire(ActorCallback< GetHealthStatusImplActor, 0, KmsConnLookupEKsByDomainIdsRep >*,KmsConnLookupEKsByDomainIdsRep const& value) 
	{
		fdb_probe_actor_enter("getHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthStatusImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetHealthStatusImplActor, 0, KmsConnLookupEKsByDomainIdsRep >*,KmsConnLookupEKsByDomainIdsRep && value) 
	{
		fdb_probe_actor_enter("getHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthStatusImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetHealthStatusImplActor, 0, KmsConnLookupEKsByDomainIdsRep >*,Error err) 
	{
		fdb_probe_actor_enter("getHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthStatusImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	Reference<EncryptKeyProxyData> ekpProxyData;
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	KmsConnectorInterface kmsConnectorInf;
															#line 687 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	UID debugId;
															#line 1648 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getHealthStatusImpl()
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class GetHealthStatusImplActor final : public Actor<bool>, public ActorCallback< GetHealthStatusImplActor, 0, KmsConnLookupEKsByDomainIdsRep >, public FastAllocated<GetHealthStatusImplActor>, public GetHealthStatusImplActorState<GetHealthStatusImplActor> {
															#line 1653 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetHealthStatusImplActor>::operator new;
	using FastAllocated<GetHealthStatusImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2016170637812403968UL, 12307475568392766976UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetHealthStatusImplActor, 0, KmsConnLookupEKsByDomainIdsRep >;
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	GetHealthStatusImplActor(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf) 
															#line 1670 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		 : Actor<bool>(),
		   GetHealthStatusImplActorState<GetHealthStatusImplActor>(ekpProxyData, kmsConnectorInf),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getHealthStatusImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1169554197488953088UL, 12981757224618950144UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthStatusImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getHealthStatusImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getHealthStatusImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetHealthStatusImplActor, 0, KmsConnLookupEKsByDomainIdsRep >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
[[nodiscard]] Future<bool> getHealthStatusImpl( Reference<EncryptKeyProxyData> const& ekpProxyData, KmsConnectorInterface const& kmsConnectorInf ) {
															#line 685 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	return Future<bool>(new GetHealthStatusImplActor(ekpProxyData, kmsConnectorInf));
															#line 1703 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
}

#line 727 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"

															#line 1708 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via updateHealthStatusImpl()
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
template <class UpdateHealthStatusImplActor>
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class UpdateHealthStatusImplActorState {
															#line 1715 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	UpdateHealthStatusImplActorState(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf) 
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		 : ekpProxyData(ekpProxyData),
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   kmsConnectorInf(kmsConnectorInf)
															#line 1724 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("updateHealthStatusImpl", reinterpret_cast<unsigned long>(this));

	}
	~UpdateHealthStatusImplActorState() 
	{
		fdb_probe_actor_destroy("updateHealthStatusImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 731 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (now() - ekpProxyData->canConnectToKmsLastUpdatedTS < FLOW_KNOBS->ENCRYPT_KEY_HEALTH_CHECK_INTERVAL)
															#line 1739 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 732 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				if (!static_cast<UpdateHealthStatusImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateHealthStatusImplActorState(); static_cast<UpdateHealthStatusImplActor*>(this)->destroy(); return 0; }
															#line 1743 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				new (&static_cast<UpdateHealthStatusImplActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~UpdateHealthStatusImplActorState();
				static_cast<UpdateHealthStatusImplActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 735 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			StrictFuture<bool> __when_expr_0 = getHealthStatusImpl(ekpProxyData, kmsConnectorInf);
															#line 735 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (static_cast<UpdateHealthStatusImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1753 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UpdateHealthStatusImplActor*>(this)->actor_wait_state = 1;
															#line 735 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateHealthStatusImplActor, 0, bool >*>(static_cast<UpdateHealthStatusImplActor*>(this)));
															#line 1758 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateHealthStatusImplActorState();
		static_cast<UpdateHealthStatusImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(bool const& canConnectToKms,int loopDepth) 
	{
															#line 736 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (canConnectToKms != ekpProxyData->canConnectToKms)
															#line 1781 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 737 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent("KmsConnectorHealthStatusChange") .detail("OldStatus", ekpProxyData->canConnectToKms) .detail("NewStatus", canConnectToKms);
															#line 1785 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 741 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->setKMSHealthiness(canConnectToKms);
															#line 742 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (!static_cast<UpdateHealthStatusImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateHealthStatusImplActorState(); static_cast<UpdateHealthStatusImplActor*>(this)->destroy(); return 0; }
															#line 1791 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		new (&static_cast<UpdateHealthStatusImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateHealthStatusImplActorState();
		static_cast<UpdateHealthStatusImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(bool && canConnectToKms,int loopDepth) 
	{
															#line 736 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (canConnectToKms != ekpProxyData->canConnectToKms)
															#line 1803 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 737 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent("KmsConnectorHealthStatusChange") .detail("OldStatus", ekpProxyData->canConnectToKms) .detail("NewStatus", canConnectToKms);
															#line 1807 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 741 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->setKMSHealthiness(canConnectToKms);
															#line 742 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (!static_cast<UpdateHealthStatusImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateHealthStatusImplActorState(); static_cast<UpdateHealthStatusImplActor*>(this)->destroy(); return 0; }
															#line 1813 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		new (&static_cast<UpdateHealthStatusImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateHealthStatusImplActorState();
		static_cast<UpdateHealthStatusImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(bool const& canConnectToKms,int loopDepth) 
	{
		loopDepth = a_body1cont1(canConnectToKms, loopDepth);

		return loopDepth;
	}
	int a_body1when1(bool && canConnectToKms,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(canConnectToKms), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateHealthStatusImplActor*>(this)->actor_wait_state > 0) static_cast<UpdateHealthStatusImplActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateHealthStatusImplActor*>(this)->ActorCallback< UpdateHealthStatusImplActor, 0, bool >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateHealthStatusImplActor, 0, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("updateHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateHealthStatusImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateHealthStatusImplActor, 0, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("updateHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateHealthStatusImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateHealthStatusImplActor, 0, bool >*,Error err) 
	{
		fdb_probe_actor_enter("updateHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateHealthStatusImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateHealthStatusImpl", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	Reference<EncryptKeyProxyData> ekpProxyData;
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	KmsConnectorInterface kmsConnectorInf;
															#line 1900 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
};
// This generated class is to be used only via updateHealthStatusImpl()
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class UpdateHealthStatusImplActor final : public Actor<Void>, public ActorCallback< UpdateHealthStatusImplActor, 0, bool >, public FastAllocated<UpdateHealthStatusImplActor>, public UpdateHealthStatusImplActorState<UpdateHealthStatusImplActor> {
															#line 1905 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
	using FastAllocated<UpdateHealthStatusImplActor>::operator new;
	using FastAllocated<UpdateHealthStatusImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17347706046144488448UL, 4601442196306469376UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateHealthStatusImplActor, 0, bool >;
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	UpdateHealthStatusImplActor(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf) 
															#line 1922 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   UpdateHealthStatusImplActorState<UpdateHealthStatusImplActor>(ekpProxyData, kmsConnectorInf),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateHealthStatusImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6628932730401293568UL, 11401167851095044352UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateHealthStatusImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateHealthStatusImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateHealthStatusImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateHealthStatusImplActor, 0, bool >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
[[nodiscard]] Future<Void> updateHealthStatusImpl( Reference<EncryptKeyProxyData> const& ekpProxyData, KmsConnectorInterface const& kmsConnectorInf ) {
															#line 728 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	return Future<Void>(new UpdateHealthStatusImplActor(ekpProxyData, kmsConnectorInf));
															#line 1955 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
}

#line 744 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"

															#line 1960 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getEKPStatus()
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
template <class GetEKPStatusActor>
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class GetEKPStatusActorState {
															#line 1967 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	GetEKPStatusActorState(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf,EncryptKeyProxyHealthStatusRequest const& req) 
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		 : ekpProxyData(ekpProxyData),
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   kmsConnectorInf(kmsConnectorInf),
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   req(req),
															#line 748 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   status()
															#line 1980 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getEKPStatus", reinterpret_cast<unsigned long>(this));

	}
	~GetEKPStatusActorState() 
	{
		fdb_probe_actor_destroy("getEKPStatus", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 749 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			status.canConnectToEKP = true;
															#line 750 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			status.canConnectToKms = ekpProxyData->canConnectToKms;
															#line 751 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			status.lastUpdatedTS = ekpProxyData->canConnectToKmsLastUpdatedTS;
															#line 752 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			status.kmsConnectorType = ekpProxyData->kmsConnector->getConnectorStr();
															#line 754 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			KmsConnGetKMSStateReq getKMSStateReq;
															#line 2003 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			try {
															#line 756 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				StrictFuture<KmsConnGetKMSStateRep> __when_expr_0 = kmsConnectorInf.getKMSStateReq.getReply(getKMSStateReq);
															#line 756 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				if (static_cast<GetEKPStatusActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2009 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<GetEKPStatusActor*>(this)->actor_wait_state = 1;
															#line 756 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetEKPStatusActor, 0, KmsConnGetKMSStateRep >*>(static_cast<GetEKPStatusActor*>(this)));
															#line 2014 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetEKPStatusActorState();
		static_cast<GetEKPStatusActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 767 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (!static_cast<GetEKPStatusActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~GetEKPStatusActorState(); static_cast<GetEKPStatusActor*>(this)->destroy(); return 0; }
															#line 2043 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		new (&static_cast<GetEKPStatusActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~GetEKPStatusActorState();
		static_cast<GetEKPStatusActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 763 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent("EKPGetKMSStateFailed", ekpProxyData->myId).error(e);
															#line 764 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 2058 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(KmsConnGetKMSStateRep const& getKMSStateRep,int loopDepth) 
	{
															#line 757 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		for( const auto& url : getKMSStateRep.restKMSUrls ) {
															#line 758 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			status.restKMSUrls.push_back(url.toString());
															#line 2074 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 760 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		status.kmsStable = getKMSStateRep.kmsStable;
															#line 761 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		req.reply.send(status);
															#line 2080 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(KmsConnGetKMSStateRep && getKMSStateRep,int loopDepth) 
	{
															#line 757 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		for( const auto& url : getKMSStateRep.restKMSUrls ) {
															#line 758 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			status.restKMSUrls.push_back(url.toString());
															#line 2091 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 760 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		status.kmsStable = getKMSStateRep.kmsStable;
															#line 761 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		req.reply.send(status);
															#line 2097 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1when1(KmsConnGetKMSStateRep const& getKMSStateRep,int loopDepth) 
	{
		loopDepth = a_body1cont2(getKMSStateRep, loopDepth);

		return loopDepth;
	}
	int a_body1when1(KmsConnGetKMSStateRep && getKMSStateRep,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(getKMSStateRep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetEKPStatusActor*>(this)->actor_wait_state > 0) static_cast<GetEKPStatusActor*>(this)->actor_wait_state = 0;
		static_cast<GetEKPStatusActor*>(this)->ActorCallback< GetEKPStatusActor, 0, KmsConnGetKMSStateRep >::remove();

	}
	void a_callback_fire(ActorCallback< GetEKPStatusActor, 0, KmsConnGetKMSStateRep >*,KmsConnGetKMSStateRep const& value) 
	{
		fdb_probe_actor_enter("getEKPStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetEKPStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getEKPStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetEKPStatusActor, 0, KmsConnGetKMSStateRep >*,KmsConnGetKMSStateRep && value) 
	{
		fdb_probe_actor_enter("getEKPStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetEKPStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getEKPStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetEKPStatusActor, 0, KmsConnGetKMSStateRep >*,Error err) 
	{
		fdb_probe_actor_enter("getEKPStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetEKPStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getEKPStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	Reference<EncryptKeyProxyData> ekpProxyData;
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	KmsConnectorInterface kmsConnectorInf;
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	EncryptKeyProxyHealthStatusRequest req;
															#line 748 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	KMSHealthStatus status;
															#line 2198 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getEKPStatus()
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class GetEKPStatusActor final : public Actor<Void>, public ActorCallback< GetEKPStatusActor, 0, KmsConnGetKMSStateRep >, public FastAllocated<GetEKPStatusActor>, public GetEKPStatusActorState<GetEKPStatusActor> {
															#line 2203 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetEKPStatusActor>::operator new;
	using FastAllocated<GetEKPStatusActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17733359217841201920UL, 14409138580005596160UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetEKPStatusActor, 0, KmsConnGetKMSStateRep >;
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	GetEKPStatusActor(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf,EncryptKeyProxyHealthStatusRequest const& req) 
															#line 2220 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   GetEKPStatusActorState<GetEKPStatusActor>(ekpProxyData, kmsConnectorInf, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getEKPStatus", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14902081074932302592UL, 16713244416702240512UL);
		ActorExecutionContextHelper __helper(static_cast<GetEKPStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getEKPStatus");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getEKPStatus", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetEKPStatusActor, 0, KmsConnGetKMSStateRep >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
[[nodiscard]] Future<Void> getEKPStatus( Reference<EncryptKeyProxyData> const& ekpProxyData, KmsConnectorInterface const& kmsConnectorInf, EncryptKeyProxyHealthStatusRequest const& req ) {
															#line 745 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	return Future<Void>(new GetEKPStatusActor(ekpProxyData, kmsConnectorInf, req));
															#line 2253 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
}

#line 769 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"

															#line 2258 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via refreshEncryptionKeysImpl()
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
template <class RefreshEncryptionKeysImplActor>
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class RefreshEncryptionKeysImplActorState {
															#line 2265 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	RefreshEncryptionKeysImplActorState(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf) 
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		 : ekpProxyData(ekpProxyData),
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   kmsConnectorInf(kmsConnectorInf),
															#line 772 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   debugId(deterministicRandom()->randomUniqueID()),
															#line 774 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   t("RefreshEKsStart", ekpProxyData->myId)
															#line 2278 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("refreshEncryptionKeysImpl", reinterpret_cast<unsigned long>(this));

	}
	~RefreshEncryptionKeysImplActorState() 
	{
		fdb_probe_actor_destroy("refreshEncryptionKeysImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 775 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			t.setMaxEventLength(SERVER_KNOBS->ENCRYPT_PROXY_MAX_DBG_TRACE_LENGTH);
															#line 776 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			t.detail("KmsConnInf", kmsConnectorInf.id());
															#line 777 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			t.detail("DebugId", debugId);
															#line 2297 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			try {
															#line 780 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				KmsConnLookupEKsByDomainIdsReq req;
															#line 781 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				req.debugId = debugId;
															#line 784 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				int64_t currTS = (int64_t)now();
															#line 785 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				for(auto itr = ekpProxyData->baseCipherDomainIdCache.begin();itr != ekpProxyData->baseCipherDomainIdCache.end();) {
															#line 787 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					if (isCipherKeyEligibleForRefresh(itr->second, currTS))
															#line 2309 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					{
															#line 788 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
						TraceEvent("RefreshEKs").detail("Id", itr->first);
															#line 789 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
						req.encryptDomainIds.push_back(itr->first);
															#line 2315 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					}
															#line 793 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					if (itr->second.isExpired())
															#line 2319 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					{
															#line 794 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
						itr = ekpProxyData->baseCipherDomainIdCache.erase(itr);
															#line 2323 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					}
					else
					{
															#line 796 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
						itr++;
															#line 2329 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					}
				}
															#line 800 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				if (req.encryptDomainIds.empty())
															#line 2334 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				{
															#line 802 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					TraceEvent(SevDebug, "RefreshEKsEmptyRefresh");
															#line 803 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
					if (!static_cast<RefreshEncryptionKeysImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RefreshEncryptionKeysImplActorState(); static_cast<RefreshEncryptionKeysImplActor*>(this)->destroy(); return 0; }
															#line 2340 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
					new (&static_cast<RefreshEncryptionKeysImplActor*>(this)->SAV< Void >::value()) Void(Void());
					this->~RefreshEncryptionKeysImplActorState();
					static_cast<RefreshEncryptionKeysImplActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 806 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				startTime = now();
															#line 807 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				StrictFuture<KmsConnLookupEKsByDomainIdsRep> __when_expr_0 = kmsConnectorInf.ekLookupByDomainIds.getReply(req);
															#line 807 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				if (static_cast<RefreshEncryptionKeysImplActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2352 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<RefreshEncryptionKeysImplActor*>(this)->actor_wait_state = 1;
															#line 807 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RefreshEncryptionKeysImplActor, 0, KmsConnLookupEKsByDomainIdsRep >*>(static_cast<RefreshEncryptionKeysImplActor*>(this)));
															#line 2357 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RefreshEncryptionKeysImplActorState();
		static_cast<RefreshEncryptionKeysImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 852 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (!static_cast<RefreshEncryptionKeysImplActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RefreshEncryptionKeysImplActorState(); static_cast<RefreshEncryptionKeysImplActor*>(this)->destroy(); return 0; }
															#line 2386 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		new (&static_cast<RefreshEncryptionKeysImplActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RefreshEncryptionKeysImplActorState();
		static_cast<RefreshEncryptionKeysImplActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 841 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (isKmsConnectionError(e))
															#line 2399 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 842 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				ekpProxyData->setKMSHealthiness(false);
															#line 2403 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 844 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (!canReplyWith(e))
															#line 2407 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 845 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				TraceEvent(SevWarn, "RefreshEKsError").error(e);
															#line 846 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 2413 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			}
															#line 848 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent("RefreshEKs").detail("ErrorCode", e.code());
															#line 849 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			++ekpProxyData->numEncryptionKeyRefreshErrors;
															#line 2419 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(KmsConnLookupEKsByDomainIdsRep const& rep,int loopDepth) 
	{
															#line 808 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->kmsLookupByDomainIdsReqLatency.addMeasurement(now() - startTime);
															#line 809 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		for( const auto& item : rep.cipherKeyDetails ) {
															#line 810 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			const auto itr = ekpProxyData->baseCipherDomainIdCache.find(item.encryptDomainId);
															#line 811 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (itr == ekpProxyData->baseCipherDomainIdCache.end())
															#line 2440 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 812 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				TraceEvent(SevInfo, "RefreshEKsDomainIdNotFound", ekpProxyData->myId) .detail("DomainId", item.encryptDomainId);
															#line 2444 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				continue;
			}
															#line 818 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			CipherKeyValidityTS validityTS = getCipherKeyValidityTS(item.refreshAfterSec, item.expireAfterSec);
															#line 819 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->insertIntoBaseDomainIdCache(item.encryptDomainId, item.encryptKeyId, item.encryptKey, item.encryptKCV, validityTS.refreshAtTS, validityTS.expAtTS);
															#line 826 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			t.detail(getEncryptDbgTraceKeyWithTS(ENCRYPT_DBG_TRACE_INSERT_PREFIX, item.encryptDomainId, item.encryptKeyId, validityTS.refreshAtTS, validityTS.expAtTS), "");
															#line 2453 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 834 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->baseCipherKeysRefreshed += rep.cipherKeyDetails.size();
															#line 835 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (rep.cipherKeyDetails.size() > 0)
															#line 2459 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->setKMSHealthiness(true);
															#line 2463 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 838 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		t.detail("NumKeys", rep.cipherKeyDetails.size());
															#line 839 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		CODE_PROBE(!rep.cipherKeyDetails.empty(), "EKP refresh cipherKeys");
															#line 2469 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(KmsConnLookupEKsByDomainIdsRep && rep,int loopDepth) 
	{
															#line 808 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->kmsLookupByDomainIdsReqLatency.addMeasurement(now() - startTime);
															#line 809 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		for( const auto& item : rep.cipherKeyDetails ) {
															#line 810 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			const auto itr = ekpProxyData->baseCipherDomainIdCache.find(item.encryptDomainId);
															#line 811 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (itr == ekpProxyData->baseCipherDomainIdCache.end())
															#line 2484 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			{
															#line 812 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				TraceEvent(SevInfo, "RefreshEKsDomainIdNotFound", ekpProxyData->myId) .detail("DomainId", item.encryptDomainId);
															#line 2488 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				continue;
			}
															#line 818 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			CipherKeyValidityTS validityTS = getCipherKeyValidityTS(item.refreshAfterSec, item.expireAfterSec);
															#line 819 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->insertIntoBaseDomainIdCache(item.encryptDomainId, item.encryptKeyId, item.encryptKey, item.encryptKCV, validityTS.refreshAtTS, validityTS.expAtTS);
															#line 826 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			t.detail(getEncryptDbgTraceKeyWithTS(ENCRYPT_DBG_TRACE_INSERT_PREFIX, item.encryptDomainId, item.encryptKeyId, validityTS.refreshAtTS, validityTS.expAtTS), "");
															#line 2497 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 834 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ekpProxyData->baseCipherKeysRefreshed += rep.cipherKeyDetails.size();
															#line 835 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (rep.cipherKeyDetails.size() > 0)
															#line 2503 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		{
															#line 836 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			ekpProxyData->setKMSHealthiness(true);
															#line 2507 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		}
															#line 838 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		t.detail("NumKeys", rep.cipherKeyDetails.size());
															#line 839 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		CODE_PROBE(!rep.cipherKeyDetails.empty(), "EKP refresh cipherKeys");
															#line 2513 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		loopDepth = a_body1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1when1(KmsConnLookupEKsByDomainIdsRep const& rep,int loopDepth) 
	{
		loopDepth = a_body1cont2(rep, loopDepth);

		return loopDepth;
	}
	int a_body1when1(KmsConnLookupEKsByDomainIdsRep && rep,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(rep), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RefreshEncryptionKeysImplActor*>(this)->actor_wait_state > 0) static_cast<RefreshEncryptionKeysImplActor*>(this)->actor_wait_state = 0;
		static_cast<RefreshEncryptionKeysImplActor*>(this)->ActorCallback< RefreshEncryptionKeysImplActor, 0, KmsConnLookupEKsByDomainIdsRep >::remove();

	}
	void a_callback_fire(ActorCallback< RefreshEncryptionKeysImplActor, 0, KmsConnLookupEKsByDomainIdsRep >*,KmsConnLookupEKsByDomainIdsRep const& value) 
	{
		fdb_probe_actor_enter("refreshEncryptionKeysImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshEncryptionKeysImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshEncryptionKeysImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RefreshEncryptionKeysImplActor, 0, KmsConnLookupEKsByDomainIdsRep >*,KmsConnLookupEKsByDomainIdsRep && value) 
	{
		fdb_probe_actor_enter("refreshEncryptionKeysImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshEncryptionKeysImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshEncryptionKeysImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RefreshEncryptionKeysImplActor, 0, KmsConnLookupEKsByDomainIdsRep >*,Error err) 
	{
		fdb_probe_actor_enter("refreshEncryptionKeysImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshEncryptionKeysImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("refreshEncryptionKeysImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	Reference<EncryptKeyProxyData> ekpProxyData;
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	KmsConnectorInterface kmsConnectorInf;
															#line 772 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	UID debugId;
															#line 774 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	TraceEvent t;
															#line 806 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	double startTime;
															#line 2616 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
};
// This generated class is to be used only via refreshEncryptionKeysImpl()
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class RefreshEncryptionKeysImplActor final : public Actor<Void>, public ActorCallback< RefreshEncryptionKeysImplActor, 0, KmsConnLookupEKsByDomainIdsRep >, public FastAllocated<RefreshEncryptionKeysImplActor>, public RefreshEncryptionKeysImplActorState<RefreshEncryptionKeysImplActor> {
															#line 2621 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
	using FastAllocated<RefreshEncryptionKeysImplActor>::operator new;
	using FastAllocated<RefreshEncryptionKeysImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14023365159269471488UL, 7492289875283356416UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RefreshEncryptionKeysImplActor, 0, KmsConnLookupEKsByDomainIdsRep >;
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	RefreshEncryptionKeysImplActor(Reference<EncryptKeyProxyData> const& ekpProxyData,KmsConnectorInterface const& kmsConnectorInf) 
															#line 2638 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RefreshEncryptionKeysImplActorState<RefreshEncryptionKeysImplActor>(ekpProxyData, kmsConnectorInf),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("refreshEncryptionKeysImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10582508617539912448UL, 988660230950166784UL);
		ActorExecutionContextHelper __helper(static_cast<RefreshEncryptionKeysImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("refreshEncryptionKeysImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("refreshEncryptionKeysImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RefreshEncryptionKeysImplActor, 0, KmsConnLookupEKsByDomainIdsRep >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
[[nodiscard]] Future<Void> refreshEncryptionKeysImpl( Reference<EncryptKeyProxyData> const& ekpProxyData, KmsConnectorInterface const& kmsConnectorInf ) {
															#line 770 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	return Future<Void>(new RefreshEncryptionKeysImplActor(ekpProxyData, kmsConnectorInf));
															#line 2671 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
}

#line 854 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"

Future<Void> refreshEncryptionKeys(Reference<EncryptKeyProxyData> ekpProxyData, KmsConnectorInterface kmsConnectorInf) {
	return refreshEncryptionKeysImpl(ekpProxyData, kmsConnectorInf);
}

Future<Void> updateHealthStatus(Reference<EncryptKeyProxyData> ekpProxyData, KmsConnectorInterface kmsConnectorInf) {
	return updateHealthStatusImpl(ekpProxyData, kmsConnectorInf);
}

void activateKmsConnector(Reference<EncryptKeyProxyData> ekpProxyData, KmsConnectorInterface kmsConnectorInf) {
	if (g_network->isSimulated()) {
		ekpProxyData->kmsConnector = std::make_unique<SimKmsConnector>(FDB_SIM_KMS_CONNECTOR_TYPE_STR);
	} else if (SERVER_KNOBS->KMS_CONNECTOR_TYPE.compare(FDB_PREF_KMS_CONNECTOR_TYPE_STR) == 0) {
		ekpProxyData->kmsConnector = std::make_unique<SimKmsConnector>(FDB_PREF_KMS_CONNECTOR_TYPE_STR);
	} else if (SERVER_KNOBS->KMS_CONNECTOR_TYPE.compare(REST_KMS_CONNECTOR_TYPE_STR) == 0) {
		ekpProxyData->kmsConnector = std::make_unique<RESTKmsConnector>(REST_KMS_CONNECTOR_TYPE_STR);
	} else {
		throw not_implemented();
	}

	TraceEvent("EKPActiveKmsConnector", ekpProxyData->myId)
	    .detail("ConnectorType", ekpProxyData->kmsConnector->getConnectorStr())
	    .detail("InfId", kmsConnectorInf.id());

	ekpProxyData->addActor.send(ekpProxyData->kmsConnector->connectorCore(kmsConnectorInf));
}

															#line 2702 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via encryptKeyProxyServer()
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
template <class EncryptKeyProxyServerActor>
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class EncryptKeyProxyServerActorState {
															#line 2709 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	EncryptKeyProxyServerActorState(EncryptKeyProxyInterface const& ekpInterface,Reference<AsyncVar<ServerDBInfo>> const& db,EncryptionAtRestMode const& encryptMode) 
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		 : ekpInterface(ekpInterface),
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   db(db),
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   encryptMode(encryptMode),
															#line 884 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   self(makeReference<EncryptKeyProxyData>(ekpInterface.id())),
															#line 885 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		   collection(actorCollection(self->addActor.getFuture()))
															#line 2724 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this));

	}
	~EncryptKeyProxyServerActorState() 
	{
		fdb_probe_actor_destroy("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 886 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			self->addActor.send(traceRole(Role::ENCRYPT_KEY_PROXY, ekpInterface.id()));
															#line 888 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			kmsConnectorInf = KmsConnectorInterface();
															#line 889 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			kmsConnectorInf.initEndpoints();
															#line 891 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent("EKPStart", self->myId).detail("KmsConnectorInf", kmsConnectorInf.id());
															#line 893 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			activateKmsConnector(self, kmsConnectorInf);
															#line 902 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			self->encryptionKeyRefresher = recurringAsync([&]() { return refreshEncryptionKeys(self, kmsConnectorInf); }, FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL, true, FLOW_KNOBS->ENCRYPT_KEY_REFRESH_INTERVAL, TaskPriority::Worker);
															#line 908 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			self->healthChecker = recurringAsync([&]() { return updateHealthStatus(self, kmsConnectorInf); }, FLOW_KNOBS->ENCRYPT_KEY_HEALTH_CHECK_INTERVAL, true, FLOW_KNOBS->ENCRYPT_KEY_HEALTH_CHECK_INTERVAL, TaskPriority::Worker, true);
															#line 915 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			CODE_PROBE(!encryptMode.isEncryptionEnabled() && SERVER_KNOBS->ENABLE_REST_KMS_COMMUNICATION, "Encryption disabled and EKP Recruited");
															#line 2753 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			try {
															#line 918 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
				;
															#line 2757 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~EncryptKeyProxyServerActorState();
		static_cast<EncryptKeyProxyServerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 946 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (!static_cast<EncryptKeyProxyServerActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~EncryptKeyProxyServerActorState(); static_cast<EncryptKeyProxyServerActor*>(this)->destroy(); return 0; }
															#line 2786 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		new (&static_cast<EncryptKeyProxyServerActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~EncryptKeyProxyServerActorState();
		static_cast<EncryptKeyProxyServerActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 943 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			TraceEvent("EKPTerminated", self->myId).errorUnsuppressed(e);
															#line 2799 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 919 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		auto __when_expr_0 = ekpInterface.getBaseCipherKeysByIds.getFuture();
															#line 2827 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_0), FutureStream<EKPGetBaseCipherKeysByIdsRequest>>::value || std::is_same<decltype(__when_expr_0), ThreadFutureStream<EKPGetBaseCipherKeysByIdsRequest>>::value, "invalid type");
															#line 918 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		if (static_cast<EncryptKeyProxyServerActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2831 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.pop(), loopDepth); };
															#line 923 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		auto __when_expr_1 = ekpInterface.getLatestBaseCipherKeys.getFuture();
															#line 2835 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<EKPGetLatestBaseCipherKeysRequest>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<EKPGetLatestBaseCipherKeysRequest>>::value, "invalid type");
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.pop(), loopDepth); };
															#line 927 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		auto __when_expr_2 = ekpInterface.haltEncryptKeyProxy.getFuture();
															#line 2840 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_2), FutureStream<HaltEncryptKeyProxyRequest>>::value || std::is_same<decltype(__when_expr_2), ThreadFutureStream<HaltEncryptKeyProxyRequest>>::value, "invalid type");
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when3(__when_expr_2.pop(), loopDepth); };
															#line 933 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		auto __when_expr_3 = ekpInterface.getHealthStatus.getFuture();
															#line 2845 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_3), FutureStream<EncryptKeyProxyHealthStatusRequest>>::value || std::is_same<decltype(__when_expr_3), ThreadFutureStream<EncryptKeyProxyHealthStatusRequest>>::value, "invalid type");
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when4(__when_expr_3.pop(), loopDepth); };
															#line 937 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		StrictFuture<Void> __when_expr_4 = collection;
															#line 2850 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when5(__when_expr_4.get(), loopDepth); };
		static_cast<EncryptKeyProxyServerActor*>(this)->actor_wait_state = 1;
															#line 919 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorSingleCallback< EncryptKeyProxyServerActor, 0, EKPGetBaseCipherKeysByIdsRequest >*>(static_cast<EncryptKeyProxyServerActor*>(this)));
															#line 923 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< EncryptKeyProxyServerActor, 1, EKPGetLatestBaseCipherKeysRequest >*>(static_cast<EncryptKeyProxyServerActor*>(this)));
															#line 927 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorSingleCallback< EncryptKeyProxyServerActor, 2, HaltEncryptKeyProxyRequest >*>(static_cast<EncryptKeyProxyServerActor*>(this)));
															#line 933 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorSingleCallback< EncryptKeyProxyServerActor, 3, EncryptKeyProxyHealthStatusRequest >*>(static_cast<EncryptKeyProxyServerActor*>(this)));
															#line 937 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< EncryptKeyProxyServerActor, 4, Void >*>(static_cast<EncryptKeyProxyServerActor*>(this)));
															#line 2863 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(EKPGetBaseCipherKeysByIdsRequest const& req,int loopDepth) 
	{
															#line 920 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ASSERT(encryptMode.isEncryptionEnabled());
															#line 921 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		self->addActor.send(getCipherKeysByBaseCipherKeyIds(self, kmsConnectorInf, req));
															#line 2893 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(EKPGetBaseCipherKeysByIdsRequest && req,int loopDepth) 
	{
															#line 920 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ASSERT(encryptMode.isEncryptionEnabled());
															#line 921 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		self->addActor.send(getCipherKeysByBaseCipherKeyIds(self, kmsConnectorInf, req));
															#line 2904 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(EKPGetLatestBaseCipherKeysRequest const& req,int loopDepth) 
	{
															#line 924 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ASSERT(encryptMode.isEncryptionEnabled());
															#line 925 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		self->addActor.send(getLatestCipherKeys(self, kmsConnectorInf, req));
															#line 2915 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(EKPGetLatestBaseCipherKeysRequest && req,int loopDepth) 
	{
															#line 924 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ASSERT(encryptMode.isEncryptionEnabled());
															#line 925 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		self->addActor.send(getLatestCipherKeys(self, kmsConnectorInf, req));
															#line 2926 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when3(HaltEncryptKeyProxyRequest const& req,int loopDepth) 
	{
															#line 928 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ASSERT(encryptMode.isEncryptionEnabled() || SERVER_KNOBS->ENABLE_REST_KMS_COMMUNICATION);
															#line 929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		TraceEvent("EKPHalted", self->myId).detail("ReqID", req.requesterID);
															#line 930 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		req.reply.send(Void());
															#line 2939 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when3(HaltEncryptKeyProxyRequest && req,int loopDepth) 
	{
															#line 928 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ASSERT(encryptMode.isEncryptionEnabled() || SERVER_KNOBS->ENABLE_REST_KMS_COMMUNICATION);
															#line 929 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		TraceEvent("EKPHalted", self->myId).detail("ReqID", req.requesterID);
															#line 930 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		req.reply.send(Void());
															#line 2952 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		return a_body1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1when4(EncryptKeyProxyHealthStatusRequest const& req,int loopDepth) 
	{
															#line 934 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ASSERT(encryptMode.isEncryptionEnabled() || SERVER_KNOBS->ENABLE_REST_KMS_COMMUNICATION);
															#line 935 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		self->addActor.send(getEKPStatus(self, kmsConnectorInf, req));
															#line 2963 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when4(EncryptKeyProxyHealthStatusRequest && req,int loopDepth) 
	{
															#line 934 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ASSERT(encryptMode.isEncryptionEnabled() || SERVER_KNOBS->ENABLE_REST_KMS_COMMUNICATION);
															#line 935 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		self->addActor.send(getEKPStatus(self, kmsConnectorInf, req));
															#line 2974 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when5(Void const& _,int loopDepth) 
	{
															#line 938 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ASSERT(false);
															#line 939 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		return a_body1Catch2(internal_error(), std::max(0, loopDepth - 1));
															#line 2985 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"

		return loopDepth;
	}
	int a_body1loopBody1when5(Void && _,int loopDepth) 
	{
															#line 938 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		ASSERT(false);
															#line 939 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		return a_body1Catch2(internal_error(), std::max(0, loopDepth - 1));
															#line 2995 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<EncryptKeyProxyServerActor*>(this)->actor_wait_state > 0) static_cast<EncryptKeyProxyServerActor*>(this)->actor_wait_state = 0;
		static_cast<EncryptKeyProxyServerActor*>(this)->ActorSingleCallback< EncryptKeyProxyServerActor, 0, EKPGetBaseCipherKeysByIdsRequest >::remove();
		static_cast<EncryptKeyProxyServerActor*>(this)->ActorSingleCallback< EncryptKeyProxyServerActor, 1, EKPGetLatestBaseCipherKeysRequest >::remove();
		static_cast<EncryptKeyProxyServerActor*>(this)->ActorSingleCallback< EncryptKeyProxyServerActor, 2, HaltEncryptKeyProxyRequest >::remove();
		static_cast<EncryptKeyProxyServerActor*>(this)->ActorSingleCallback< EncryptKeyProxyServerActor, 3, EncryptKeyProxyHealthStatusRequest >::remove();
		static_cast<EncryptKeyProxyServerActor*>(this)->ActorCallback< EncryptKeyProxyServerActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorSingleCallback< EncryptKeyProxyServerActor, 0, EKPGetBaseCipherKeysByIdsRequest >*,EKPGetBaseCipherKeysByIdsRequest const& value) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< EncryptKeyProxyServerActor, 0, EKPGetBaseCipherKeysByIdsRequest >*,EKPGetBaseCipherKeysByIdsRequest && value) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorSingleCallback< EncryptKeyProxyServerActor, 0, EKPGetBaseCipherKeysByIdsRequest >*,Error err) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< EncryptKeyProxyServerActor, 1, EKPGetLatestBaseCipherKeysRequest >*,EKPGetLatestBaseCipherKeysRequest const& value) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< EncryptKeyProxyServerActor, 1, EKPGetLatestBaseCipherKeysRequest >*,EKPGetLatestBaseCipherKeysRequest && value) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< EncryptKeyProxyServerActor, 1, EKPGetLatestBaseCipherKeysRequest >*,Error err) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< EncryptKeyProxyServerActor, 2, HaltEncryptKeyProxyRequest >*,HaltEncryptKeyProxyRequest const& value) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< EncryptKeyProxyServerActor, 2, HaltEncryptKeyProxyRequest >*,HaltEncryptKeyProxyRequest && value) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorSingleCallback< EncryptKeyProxyServerActor, 2, HaltEncryptKeyProxyRequest >*,Error err) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< EncryptKeyProxyServerActor, 3, EncryptKeyProxyHealthStatusRequest >*,EncryptKeyProxyHealthStatusRequest const& value) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< EncryptKeyProxyServerActor, 3, EncryptKeyProxyHealthStatusRequest >*,EncryptKeyProxyHealthStatusRequest && value) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorSingleCallback< EncryptKeyProxyServerActor, 3, EncryptKeyProxyHealthStatusRequest >*,Error err) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< EncryptKeyProxyServerActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< EncryptKeyProxyServerActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< EncryptKeyProxyServerActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	EncryptKeyProxyInterface ekpInterface;
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	Reference<AsyncVar<ServerDBInfo>> db;
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	EncryptionAtRestMode encryptMode;
															#line 884 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	Reference<EncryptKeyProxyData> self;
															#line 885 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	Future<Void> collection;
															#line 888 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	KmsConnectorInterface kmsConnectorInf;
															#line 3319 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
};
// This generated class is to be used only via encryptKeyProxyServer()
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class EncryptKeyProxyServerActor final : public Actor<Void>, public ActorSingleCallback< EncryptKeyProxyServerActor, 0, EKPGetBaseCipherKeysByIdsRequest >, public ActorSingleCallback< EncryptKeyProxyServerActor, 1, EKPGetLatestBaseCipherKeysRequest >, public ActorSingleCallback< EncryptKeyProxyServerActor, 2, HaltEncryptKeyProxyRequest >, public ActorSingleCallback< EncryptKeyProxyServerActor, 3, EncryptKeyProxyHealthStatusRequest >, public ActorCallback< EncryptKeyProxyServerActor, 4, Void >, public FastAllocated<EncryptKeyProxyServerActor>, public EncryptKeyProxyServerActorState<EncryptKeyProxyServerActor> {
															#line 3324 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
	using FastAllocated<EncryptKeyProxyServerActor>::operator new;
	using FastAllocated<EncryptKeyProxyServerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14044775150309453568UL, 854187855441196288UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorSingleCallback< EncryptKeyProxyServerActor, 0, EKPGetBaseCipherKeysByIdsRequest >;
friend struct ActorSingleCallback< EncryptKeyProxyServerActor, 1, EKPGetLatestBaseCipherKeysRequest >;
friend struct ActorSingleCallback< EncryptKeyProxyServerActor, 2, HaltEncryptKeyProxyRequest >;
friend struct ActorSingleCallback< EncryptKeyProxyServerActor, 3, EncryptKeyProxyHealthStatusRequest >;
friend struct ActorCallback< EncryptKeyProxyServerActor, 4, Void >;
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	EncryptKeyProxyServerActor(EncryptKeyProxyInterface const& ekpInterface,Reference<AsyncVar<ServerDBInfo>> const& db,EncryptionAtRestMode const& encryptMode) 
															#line 3345 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   EncryptKeyProxyServerActorState<EncryptKeyProxyServerActor>(ekpInterface, db, encryptMode),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(753260908361884672UL, 16592891810734151680UL);
		ActorExecutionContextHelper __helper(static_cast<EncryptKeyProxyServerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("encryptKeyProxyServer");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("encryptKeyProxyServer", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorSingleCallback< EncryptKeyProxyServerActor, 0, EKPGetBaseCipherKeysByIdsRequest >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
[[nodiscard]] Future<Void> encryptKeyProxyServer( EncryptKeyProxyInterface const& ekpInterface, Reference<AsyncVar<ServerDBInfo>> const& db, EncryptionAtRestMode const& encryptMode ) {
															#line 881 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	return Future<Void>(new EncryptKeyProxyServerActor(ekpInterface, db, encryptMode));
															#line 3378 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
}

#line 948 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"

void testLookupLatestCipherDetails() {
	Reference<EncryptKeyProxyData> ekpProxyData =
	    makeReference<EncryptKeyProxyData>(deterministicRandom()->randomUniqueID());
	Optional<TraceEvent> dbgTrace = TraceEvent("GetLatestCipherTest", ekpProxyData->myId);
	int numHits = 0;
	EKPGetLatestBaseCipherKeysReply latestCipherReply;
	std::unordered_set<EncryptCipherDomainId> dedupedDomainIds = { 1, 2, 3, 4 };
	double startTime = now();
	ekpProxyData->baseCipherDomainIdCache[1] =
	    EncryptBaseCipherKey(1, 1, "dom1"_sr, 0, startTime + 300, startTime + 300);
	// key needs refresh
	ekpProxyData->baseCipherDomainIdCache[2] =
	    EncryptBaseCipherKey(2, 2, "dom2"_sr, 0, startTime - 10, startTime + 300);
	// key is expired
	ekpProxyData->baseCipherDomainIdCache[3] =
	    EncryptBaseCipherKey(3, 3, "dom3"_sr, 0, startTime + 300, startTime - 10);

	std::unordered_set<EncryptCipherDomainId> lookupCipherDomainIds =
	    getLookupDetailsLatest(ekpProxyData, dbgTrace, latestCipherReply, numHits, dedupedDomainIds);
	std::unordered_set<EncryptCipherDomainId> expectedLookupCipherDomainIds = { 2, 3, 4 };

	ASSERT_EQ(numHits, 1);
	if (lookupCipherDomainIds != expectedLookupCipherDomainIds) {
		ASSERT(false);
	}
	EKPBaseCipherDetails expectedCipherDetails =
	    EKPBaseCipherDetails(1, 1, "dom1"_sr, 0, startTime + 300, startTime + 300);
	ASSERT_EQ(latestCipherReply.baseCipherDetails.size(), 1);
	ASSERT(latestCipherReply.baseCipherDetails[0] == expectedCipherDetails);
	ASSERT_EQ(ekpProxyData->baseCipherDomainIdCacheHits.getValue(), 1);
	ASSERT_EQ(ekpProxyData->baseCipherDomainIdCacheMisses.getValue(), 3);
}

void testLookupCipherDetails() {
	Reference<EncryptKeyProxyData> ekpProxyData =
	    makeReference<EncryptKeyProxyData>(deterministicRandom()->randomUniqueID());
	Optional<TraceEvent> dbgTrace = TraceEvent("GetCipherTest", ekpProxyData->myId);
	int numHits = 0;
	EKPGetBaseCipherKeysByIdsReply keyIdsReply;
	std::unordered_set<EKPGetBaseCipherKeysRequestInfo, EKPGetBaseCipherKeysRequestInfo_Hash> dedupedCipherInfos = {
		{ 1, 1 }, { 2, 2 }, { 3, 3 }, { 4, 4 }
	};
	double startTime = now();
	ekpProxyData->baseCipherDomainIdKeyIdCache[EncryptKeyProxyData::getBaseCipherDomainIdKeyIdCacheKey(1, 1)] =
	    EncryptBaseCipherKey(1, 1, "dom1"_sr, 0, startTime + 300, startTime + 300);
	// key needs refresh
	ekpProxyData->baseCipherDomainIdKeyIdCache[EncryptKeyProxyData::getBaseCipherDomainIdKeyIdCacheKey(2, 2)] =
	    EncryptBaseCipherKey(2, 2, "dom2"_sr, 0, startTime - 10, startTime + 300);
	// key is expired
	ekpProxyData->baseCipherDomainIdKeyIdCache[EncryptKeyProxyData::getBaseCipherDomainIdKeyIdCacheKey(3, 3)] =
	    EncryptBaseCipherKey(3, 3, "dom3"_sr, 0, startTime + 300, startTime - 10);

	std::unordered_map<std::pair<EncryptCipherDomainId, EncryptCipherBaseKeyId>,
	                   EKPGetBaseCipherKeysRequestInfo,
	                   boost::hash<std::pair<EncryptCipherDomainId, EncryptCipherBaseKeyId>>>
	    lookupCipherInfoMap = getLookupDetails(ekpProxyData, dbgTrace, keyIdsReply, numHits, dedupedCipherInfos);
	ASSERT_EQ(numHits, 2);
	ASSERT(lookupCipherInfoMap.find({ 3, 3 }) != lookupCipherInfoMap.end());
	ASSERT(lookupCipherInfoMap.find({ 4, 4 }) != lookupCipherInfoMap.end());
	ASSERT_EQ(keyIdsReply.baseCipherDetails.size(), 2);
	EKPBaseCipherDetails expectedCipherDetails1 = EKPBaseCipherDetails(
	    1, 1, "dom1"_sr, 0, std::numeric_limits<int64_t>::max(), std::numeric_limits<int64_t>::max());
	EKPBaseCipherDetails expectedCipherDetails2 = EKPBaseCipherDetails(
	    2, 2, "dom2"_sr, 0, std::numeric_limits<int64_t>::max(), std::numeric_limits<int64_t>::max());
	for (EKPBaseCipherDetails details : keyIdsReply.baseCipherDetails) {
		if (details.encryptDomainId == 1) {
			ASSERT(details == expectedCipherDetails1);
		} else if (details.encryptDomainId == 2) {
			ASSERT(details == expectedCipherDetails2);
		} else {
			ASSERT(false);
		}
	}
	ASSERT_EQ(ekpProxyData->baseCipherKeyIdCacheHits.getValue(), 2);
	ASSERT_EQ(ekpProxyData->baseCipherKeyIdCacheMisses.getValue(), 2);
}

															#line 3460 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1026()
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
template <class FlowTestCase1026Actor>
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class FlowTestCase1026ActorState {
															#line 3467 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	FlowTestCase1026ActorState(UnitTestParameters const& params) 
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
		 : params(params)
															#line 3474 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1026", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1026ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1026", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1027 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			testLookupLatestCipherDetails();
															#line 1028 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			testLookupCipherDetails();
															#line 1029 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
			if (!static_cast<FlowTestCase1026Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1026ActorState(); static_cast<FlowTestCase1026Actor*>(this)->destroy(); return 0; }
															#line 3493 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase1026Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase1026ActorState();
			static_cast<FlowTestCase1026Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1026ActorState();
		static_cast<FlowTestCase1026Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	UnitTestParameters params;
															#line 3517 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1026()
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
class FlowTestCase1026Actor final : public Actor<Void>, public FastAllocated<FlowTestCase1026Actor>, public FlowTestCase1026ActorState<FlowTestCase1026Actor> {
															#line 3522 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1026Actor>::operator new;
	using FastAllocated<FlowTestCase1026Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17546640266136549376UL, 14549627785233718016UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	FlowTestCase1026Actor(UnitTestParameters const& params) 
															#line 3538 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1026ActorState<FlowTestCase1026Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1026", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18210846001031630080UL, 13747550748181303552UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1026Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1026");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1026", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
static Future<Void> flowTestCase1026( UnitTestParameters const& params ) {
															#line 1026 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
	return Future<Void>(new FlowTestCase1026Actor(params));
															#line 3570 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/build_output/fdbserver/EncryptKeyProxy.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1026, "/EncryptKeyProxy")

#line 1031 "/codebuild/output/src2000075677/src/github.com/apple/foundationdb/fdbserver/EncryptKeyProxy.actor.cpp"
