#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
/*
 * RESTKmsConnector.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/RESTUtils.h"
#include "fdbserver/RESTKmsConnector.h"

#include "fdbclient/BlobCipher.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/RESTClient.h"

#include "fdbrpc/HTTP.h"

#include "fdbserver/KmsConnectorInterface.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/RESTKmsConnectorUtils.h"

#include "flow/Arena.h"
#include "flow/ActorCollection.h"
#include "flow/BooleanParam.h"
#include "flow/EncryptUtils.h"
#include "flow/Error.h"
#include "flow/FastRef.h"
#include "flow/IAsyncFile.h"
#include "flow/IConnection.h"
#include "flow/IRandom.h"
#include "flow/Knobs.h"
#include "flow/Platform.h"
#include "flow/Trace.h"
#include "flow/UnitTest.h"

#include <algorithm>
#include <limits>
#include <boost/algorithm/string.hpp>
#include <cstring>
#include <stack>
#include <memory>
#include <queue>
#include <sstream>
#include <unordered_map>
#include <utility>

#include "flow/actorcompiler.h" // This must be the last #include

using namespace RESTKmsConnectorUtils;

namespace {
bool canReplyWith(Error e) {
	switch (e.code()) {
	case error_code_encrypt_invalid_kms_config:
	case error_code_encrypt_keys_fetch_failed:
	case error_code_file_not_found:
	case error_code_file_too_large:
	case error_code_http_request_failed:
	case error_code_io_error:
	case error_code_operation_failed:
	case error_code_value_too_large:
	case error_code_timed_out:
	case error_code_connection_failed:
	case error_code_rest_malformed_response:
		return true;
	default:
		return false;
	}
}

bool isKmsNotReachable(const int errCode) {
	return errCode == error_code_timed_out || errCode == error_code_connection_failed;
}

void removeTrailingChar(std::string& str, char c) {
	while (!str.empty() && str[str.length() - 1] == c) {
		str.erase(str.length() - 1);
	}
}

} // namespace

template <class Params>
struct KmsUrlCtx {
	enum class PenaltyType { TIMEOUT = 1, MALFORMED_RESPONSE = 2 };

	std::string url;
	uint64_t nRequests;
	uint64_t nFailedResponses;
	uint64_t nResponseParseFailures;
	double unresponsivenessPenalty;
	double unresponsivenessPenaltyTS;

	KmsUrlCtx()
	  : url(""), nRequests(0), nFailedResponses(0), nResponseParseFailures(0), unresponsivenessPenalty(0.0),
	    unresponsivenessPenaltyTS(0) {}
	explicit KmsUrlCtx(const std::string& u)
	  : url(u), nRequests(0), nFailedResponses(0), nResponseParseFailures(0), unresponsivenessPenalty(0.0),
	    unresponsivenessPenaltyTS(0) {}

	bool operator==(const KmsUrlCtx& toCompare) const { return url.compare(toCompare.url) == 0; }

	void refreshUnresponsivenessPenalty() {
		if (unresponsivenessPenaltyTS == 0) {
			return;
		}
		int64_t timeSinceLastPenalty = now() - unresponsivenessPenaltyTS;
		unresponsivenessPenalty = Params::penalty(timeSinceLastPenalty);
	}

	void penalize(const PenaltyType type) {
		if (type == PenaltyType::TIMEOUT) {
			nFailedResponses++;
			unresponsivenessPenaltyTS = now();
		} else {
			ASSERT_EQ(type, PenaltyType::MALFORMED_RESPONSE);
			nResponseParseFailures++;
		}
	}

	std::string toString() const {
		return fmt::format(
		    "{} {} {} {} {}", url, nRequests, nFailedResponses, nResponseParseFailures, unresponsivenessPenalty);
	}
};

// Current implementation is designed to favor the most-preferable KMS for all outbound requests allowing leveraging KMS
// implemented caching if supported
//
// TODO: Implement load-balancing requests to available KMS servers maintaining prioritized KMS server list based on
// observed errors/connection failures/timeouts etc.

template <class Params>
struct KmsUrlStore {
	void sort() {
		std::sort(kmsUrls.begin(), kmsUrls.end(), [](const KmsUrlCtx<Params>& l, const KmsUrlCtx<Params>& r) {
			// Sort the available URLs based on following rules:
			// 1. URL with higher unresponsiveness-penalty are least preferred
			// 2. Among URLs with same unresponsiveness-penalty weight, URLs with more number of failed-respones are
			// less preferred
			// 3. Lastly, URLs with more malformed response messages are less preferred

			if (l.unresponsivenessPenalty != r.unresponsivenessPenalty) {
				return l.unresponsivenessPenalty < r.unresponsivenessPenalty;
			}
			if (l.nFailedResponses != r.nFailedResponses) {
				return l.nFailedResponses < r.nFailedResponses;
			}
			return l.nResponseParseFailures < r.nResponseParseFailures;
		});
	}

	void penalize(const KmsUrlCtx<Params>& toPenalize, const typename KmsUrlCtx<Params>::PenaltyType type) {
		bool found = false;
		for (KmsUrlCtx<Params>& urlCtx : kmsUrls) {
			if (urlCtx == toPenalize) {
				urlCtx.penalize(type);
				found = true;
				break;
			}
		}
		ASSERT(found);

		// update the penalties
		for (auto& url : kmsUrls) {
			url.refreshUnresponsivenessPenalty();
		}

		if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::DEBUG) {
			std::string details;
			for (const auto& url : kmsUrls) {
				details.append(fmt::format("[ {} ], ", url.toString()));
			}
			TraceEvent("RESTKmsUrlStoreBeforeSort")
			    .detail("Details", details)
			    .detail("Penalize", toPenalize.toString());
		}

		// Reshuffle the URLs
		sort();

		if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::DEBUG) {
			std::string details;
			for (const auto& url : kmsUrls) {
				details.append(fmt::format("[ {} ], ", url.toString()));
			}
			TraceEvent("RESTKmsUrlStoreAfterSort").detail("Details", details);
		}
	}

	std::vector<KmsUrlCtx<Params>> kmsUrls;
};

FDB_BOOLEAN_PARAM(RefreshPersistedUrls);
FDB_BOOLEAN_PARAM(IsCipherType);

// Routine to determine penalty for cached KMSUrl based on unresponsive KMS behavior observed in recent past. The
// routine is designed to assign a maximum penalty if KMS responses are unacceptable in very recent past, with time the
// the penalty weight deteriorates (matches real world outage OR server overload scenario)

struct KmsUrlPenaltyParams {
	static double penalty(int64_t timeSinceLastPenalty) { return continuousTimeDecay(1.0, 0.1, timeSinceLastPenalty); }
};

struct RESTKmsConnectorCtx : public ReferenceCounted<RESTKmsConnectorCtx> {
	UID uid;
	KmsUrlStore<KmsUrlPenaltyParams> kmsUrlStore;
	double lastKmsUrlsRefreshTs;
	double lastKmsUrlDiscoverTS;
	RESTClient restClient;
	ValidationTokenMap validationTokenMap;
	PromiseStream<Future<Void>> addActor;
	bool kmsStable;
	Future<Void> kmsStabilityChecker;

	RESTKmsConnectorCtx()
	  : uid(deterministicRandom()->randomUniqueID()), lastKmsUrlsRefreshTs(0), lastKmsUrlDiscoverTS(0.0),
	    kmsStable(true) {}
	explicit RESTKmsConnectorCtx(const UID& id)
	  : uid(id), lastKmsUrlsRefreshTs(0), lastKmsUrlDiscoverTS(0.0), kmsStable(true) {}
};

std::string getFullRequestUrl(Reference<RESTKmsConnectorCtx> ctx, const std::string& url, const std::string& suffix) {
	if (suffix.empty()) {
		TraceEvent(SevWarn, "RESTGetFullUrlEmptyEndpoint", ctx->uid);
		throw encrypt_invalid_kms_config();
	}
	std::string fullUrl(url);
	return (suffix[0] == '/') ? fullUrl.append(suffix) : fullUrl.append("/").append(suffix);
}

void dropCachedKmsUrls(Reference<RESTKmsConnectorCtx> ctx,
                       std::unordered_map<std::string, KmsUrlCtx<KmsUrlPenaltyParams>>* urlMap) {
	for (const auto& url : ctx->kmsUrlStore.kmsUrls) {
		if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::VERBOSE) {
			TraceEvent("RESTDropCachedKmsUrls", ctx->uid)
			    .detail("Url", url.url)
			    .detail("NumRequests", url.nRequests)
			    .detail("NumFailedResponses", url.nFailedResponses)
			    .detail("NumRespParseFailures", url.nResponseParseFailures);
		}
		urlMap->insert(std::make_pair(url.url, url));
	}
	ctx->kmsUrlStore.kmsUrls.clear();
}

bool shouldRefreshKmsUrls(Reference<RESTKmsConnectorCtx> ctx) {
	if (!SERVER_KNOBS->REST_KMS_CONNECTOR_REFRESH_KMS_URLS) {
		return false;
	}

	return (now() - ctx->lastKmsUrlsRefreshTs) > SERVER_KNOBS->REST_KMS_CONNECTOR_REFRESH_KMS_URLS_INTERVAL_SEC;
}

void extractKmsUrls(Reference<RESTKmsConnectorCtx> ctx,
                    const rapidjson::Document& doc,
                    Reference<HTTP::IncomingResponse> httpResp) {
	// Refresh KmsUrls cache
	std::unordered_map<std::string, KmsUrlCtx<KmsUrlPenaltyParams>> urlMap;
	dropCachedKmsUrls(ctx, &urlMap);
	ASSERT_EQ(ctx->kmsUrlStore.kmsUrls.size(), 0);

	for (const auto& url : doc[KMS_URLS_TAG].GetArray()) {
		if (!url.IsString()) {
			// TODO: We need to log only the kms section of the document
			TraceEvent(SevWarnAlways, "RESTDiscoverKmsUrlsMalformedResp", ctx->uid).detail("UrlType", url.GetType());
			throw operation_failed();
		}

		std::string urlStr;
		urlStr.resize(url.GetStringLength());
		memcpy(urlStr.data(), url.GetString(), url.GetStringLength());

		// preserve the KmsUrl stats while (re)discovering KMS URLs, preferable to select the servers with lesser count
		// of unexpected events in the past

		auto itr = urlMap.find(urlStr);
		if (itr != urlMap.end()) {
			if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::INFO) {
				TraceEvent("RESTDiscoverExistingKmsUrl", ctx->uid).detail("UrlCtx", itr->second.toString());
			}
			ctx->kmsUrlStore.kmsUrls.emplace_back(itr->second);
		} else {
			auto urlCtx = KmsUrlCtx<KmsUrlPenaltyParams>(urlStr);
			if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::INFO) {
				TraceEvent("RESTDiscoverNewKmsUrl", ctx->uid).detail("UrlCtx", urlCtx.toString());
			}
			ctx->kmsUrlStore.kmsUrls.emplace_back(urlCtx);
		}
	}

	// Reshuffle URLs to re-arrange them in appropriate priority
	ctx->kmsUrlStore.sort();

	// Update Kms URLs refresh timestamp
	ctx->lastKmsUrlsRefreshTs = now();
}

															#line 313 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via parseDiscoverKmsUrlFile()
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class ParseDiscoverKmsUrlFileActor>
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class ParseDiscoverKmsUrlFileActorState {
															#line 320 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	ParseDiscoverKmsUrlFileActorState(Reference<RESTKmsConnectorCtx> const& ctx,std::string const& filename) 
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx),
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   filename(filename)
															#line 329 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this));

	}
	~ParseDiscoverKmsUrlFileActorState() 
	{
		fdb_probe_actor_destroy("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 312 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (filename.empty() || !fileExists(filename))
															#line 344 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 313 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				TraceEvent(SevWarnAlways, "RESTDiscoverKmsUrlsFileNotFound", ctx->uid).log();
															#line 314 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				return a_body1Catch1(encrypt_invalid_kms_config(), loopDepth);
															#line 350 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 317 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( filename, IAsyncFile::OPEN_NO_AIO | IAsyncFile::OPEN_READONLY | IAsyncFile::OPEN_UNCACHED, 0644);
															#line 317 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 356 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state = 1;
															#line 317 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ParseDiscoverKmsUrlFileActor, 0, Reference<IAsyncFile> >*>(static_cast<ParseDiscoverKmsUrlFileActor*>(this)));
															#line 361 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ParseDiscoverKmsUrlFileActorState();
		static_cast<ParseDiscoverKmsUrlFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 319 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<int64_t> __when_expr_1 = dFile->size();
															#line 319 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 386 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state = 2;
															#line 319 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ParseDiscoverKmsUrlFileActor, 1, int64_t >*>(static_cast<ParseDiscoverKmsUrlFileActor*>(this)));
															#line 391 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> const& __dFile,int loopDepth) 
	{
															#line 317 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		dFile = __dFile;
															#line 400 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IAsyncFile> && __dFile,int loopDepth) 
	{
		dFile = std::move(__dFile);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state > 0) static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state = 0;
		static_cast<ParseDiscoverKmsUrlFileActor*>(this)->ActorCallback< ParseDiscoverKmsUrlFileActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< ParseDiscoverKmsUrlFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ParseDiscoverKmsUrlFileActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ParseDiscoverKmsUrlFileActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont3(int loopDepth) 
	{
															#line 320 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		buff = makeString(fSize);
															#line 321 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<int> __when_expr_2 = dFile->read(mutateString(buff), fSize, 0);
															#line 321 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 483 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont3when1(__when_expr_2.get(), loopDepth); };
		static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state = 3;
															#line 321 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ParseDiscoverKmsUrlFileActor, 2, int >*>(static_cast<ParseDiscoverKmsUrlFileActor*>(this)));
															#line 488 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(int64_t const& __fSize,int loopDepth) 
	{
															#line 319 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		fSize = __fSize;
															#line 497 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(int64_t && __fSize,int loopDepth) 
	{
		fSize = std::move(__fSize);
		loopDepth = a_body1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state > 0) static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state = 0;
		static_cast<ParseDiscoverKmsUrlFileActor*>(this)->ActorCallback< ParseDiscoverKmsUrlFileActor, 1, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< ParseDiscoverKmsUrlFileActor, 1, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ParseDiscoverKmsUrlFileActor, 1, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ParseDiscoverKmsUrlFileActor, 1, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont4(int const& bytesRead,int loopDepth) 
	{
															#line 322 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (bytesRead != fSize)
															#line 576 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
															#line 323 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent(SevWarnAlways, "RESTDiscoveryKmsUrlFileReadShort", ctx->uid) .detail("Filename", filename) .detail("Expected", fSize) .detail("Actual", bytesRead);
															#line 327 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			return a_body1Catch1(io_error(), loopDepth);
															#line 582 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 334 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		std::unordered_map<std::string, KmsUrlCtx<KmsUrlPenaltyParams>> urlMap;
															#line 335 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		dropCachedKmsUrls(ctx, &urlMap);
															#line 336 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT_EQ(ctx->kmsUrlStore.kmsUrls.size(), 0);
															#line 338 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		std::stringstream ss(buff.toString());
															#line 339 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		std::string url;
															#line 340 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		for(;std::getline(ss, url, DISCOVER_URL_FILE_URL_SEP);) {
															#line 341 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string trimedUrl = boost::trim_copy(url);
															#line 343 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			for(;!trimedUrl.empty() && trimedUrl.ends_with('/');) {
															#line 344 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				trimedUrl.pop_back();
															#line 602 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 346 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (trimedUrl.empty())
															#line 606 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
				continue;
			}
															#line 350 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			auto itr = urlMap.find(trimedUrl);
															#line 351 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (itr != urlMap.end())
															#line 614 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 352 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::INFO)
															#line 618 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
															#line 353 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					TraceEvent("RESTParseDiscoverKmsUrlsExistingUrl", ctx->uid).detail("UrlCtx", itr->second.toString());
															#line 622 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				}
															#line 355 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ctx->kmsUrlStore.kmsUrls.emplace_back(itr->second);
															#line 626 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
			else
			{
															#line 357 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				auto urlCtx = KmsUrlCtx<KmsUrlPenaltyParams>(trimedUrl);
															#line 358 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::INFO)
															#line 634 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
															#line 359 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					TraceEvent("RESTParseDiscoverKmsUrlsAddUrl", ctx->uid).detail("UrlCtx", urlCtx.toString());
															#line 638 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				}
															#line 361 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ctx->kmsUrlStore.kmsUrls.emplace_back(urlCtx);
															#line 642 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
		}
															#line 366 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ctx->kmsUrlStore.sort();
															#line 368 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<ParseDiscoverKmsUrlFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ParseDiscoverKmsUrlFileActorState(); static_cast<ParseDiscoverKmsUrlFileActor*>(this)->destroy(); return 0; }
															#line 649 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<ParseDiscoverKmsUrlFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ParseDiscoverKmsUrlFileActorState();
		static_cast<ParseDiscoverKmsUrlFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4(int && bytesRead,int loopDepth) 
	{
															#line 322 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (bytesRead != fSize)
															#line 661 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
															#line 323 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent(SevWarnAlways, "RESTDiscoveryKmsUrlFileReadShort", ctx->uid) .detail("Filename", filename) .detail("Expected", fSize) .detail("Actual", bytesRead);
															#line 327 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			return a_body1Catch1(io_error(), loopDepth);
															#line 667 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 334 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		std::unordered_map<std::string, KmsUrlCtx<KmsUrlPenaltyParams>> urlMap;
															#line 335 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		dropCachedKmsUrls(ctx, &urlMap);
															#line 336 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT_EQ(ctx->kmsUrlStore.kmsUrls.size(), 0);
															#line 338 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		std::stringstream ss(buff.toString());
															#line 339 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		std::string url;
															#line 340 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		for(;std::getline(ss, url, DISCOVER_URL_FILE_URL_SEP);) {
															#line 341 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string trimedUrl = boost::trim_copy(url);
															#line 343 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			for(;!trimedUrl.empty() && trimedUrl.ends_with('/');) {
															#line 344 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				trimedUrl.pop_back();
															#line 687 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 346 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (trimedUrl.empty())
															#line 691 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
				continue;
			}
															#line 350 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			auto itr = urlMap.find(trimedUrl);
															#line 351 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (itr != urlMap.end())
															#line 699 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 352 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::INFO)
															#line 703 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
															#line 353 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					TraceEvent("RESTParseDiscoverKmsUrlsExistingUrl", ctx->uid).detail("UrlCtx", itr->second.toString());
															#line 707 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				}
															#line 355 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ctx->kmsUrlStore.kmsUrls.emplace_back(itr->second);
															#line 711 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
			else
			{
															#line 357 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				auto urlCtx = KmsUrlCtx<KmsUrlPenaltyParams>(trimedUrl);
															#line 358 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::INFO)
															#line 719 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
															#line 359 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					TraceEvent("RESTParseDiscoverKmsUrlsAddUrl", ctx->uid).detail("UrlCtx", urlCtx.toString());
															#line 723 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				}
															#line 361 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ctx->kmsUrlStore.kmsUrls.emplace_back(urlCtx);
															#line 727 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
		}
															#line 366 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ctx->kmsUrlStore.sort();
															#line 368 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<ParseDiscoverKmsUrlFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ParseDiscoverKmsUrlFileActorState(); static_cast<ParseDiscoverKmsUrlFileActor*>(this)->destroy(); return 0; }
															#line 734 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<ParseDiscoverKmsUrlFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ParseDiscoverKmsUrlFileActorState();
		static_cast<ParseDiscoverKmsUrlFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3when1(int const& bytesRead,int loopDepth) 
	{
		loopDepth = a_body1cont4(bytesRead, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(int && bytesRead,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(bytesRead), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state > 0) static_cast<ParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state = 0;
		static_cast<ParseDiscoverKmsUrlFileActor*>(this)->ActorCallback< ParseDiscoverKmsUrlFileActor, 2, int >::remove();

	}
	void a_callback_fire(ActorCallback< ParseDiscoverKmsUrlFileActor, 2, int >*,int const& value) 
	{
		fdb_probe_actor_enter("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ParseDiscoverKmsUrlFileActor, 2, int >*,int && value) 
	{
		fdb_probe_actor_enter("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ParseDiscoverKmsUrlFileActor, 2, int >*,Error err) 
	{
		fdb_probe_actor_enter("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::string filename;
															#line 317 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<IAsyncFile> dFile;
															#line 319 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	int64_t fSize;
															#line 320 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Standalone<StringRef> buff;
															#line 827 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via parseDiscoverKmsUrlFile()
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class ParseDiscoverKmsUrlFileActor final : public Actor<Void>, public ActorCallback< ParseDiscoverKmsUrlFileActor, 0, Reference<IAsyncFile> >, public ActorCallback< ParseDiscoverKmsUrlFileActor, 1, int64_t >, public ActorCallback< ParseDiscoverKmsUrlFileActor, 2, int >, public FastAllocated<ParseDiscoverKmsUrlFileActor>, public ParseDiscoverKmsUrlFileActorState<ParseDiscoverKmsUrlFileActor> {
															#line 832 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<ParseDiscoverKmsUrlFileActor>::operator new;
	using FastAllocated<ParseDiscoverKmsUrlFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11897804603327839232UL, 4786559008297226240UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ParseDiscoverKmsUrlFileActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< ParseDiscoverKmsUrlFileActor, 1, int64_t >;
friend struct ActorCallback< ParseDiscoverKmsUrlFileActor, 2, int >;
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	ParseDiscoverKmsUrlFileActor(Reference<RESTKmsConnectorCtx> const& ctx,std::string const& filename) 
															#line 851 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ParseDiscoverKmsUrlFileActorState<ParseDiscoverKmsUrlFileActor>(ctx, filename),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8349089322283310080UL, 13870883122760371712UL);
		ActorExecutionContextHelper __helper(static_cast<ParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("parseDiscoverKmsUrlFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("parseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ParseDiscoverKmsUrlFileActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ParseDiscoverKmsUrlFileActor, 1, int64_t >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ParseDiscoverKmsUrlFileActor, 2, int >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> parseDiscoverKmsUrlFile( Reference<RESTKmsConnectorCtx> const& ctx, std::string const& filename ) {
															#line 311 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new ParseDiscoverKmsUrlFileActor(ctx, filename));
															#line 886 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 370 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 891 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via discoverKmsUrls()
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class DiscoverKmsUrlsActor>
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class DiscoverKmsUrlsActorState {
															#line 898 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	DiscoverKmsUrlsActorState(Reference<RESTKmsConnectorCtx> const& ctx,RefreshPersistedUrls const& refreshPersistedUrls) 
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx),
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   refreshPersistedUrls(refreshPersistedUrls)
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("discoverKmsUrls", reinterpret_cast<unsigned long>(this));

	}
	~DiscoverKmsUrlsActorState() 
	{
		fdb_probe_actor_destroy("discoverKmsUrls", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 384 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (!refreshPersistedUrls)
															#line 922 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
			}
															#line 389 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string_view mode{ SERVER_KNOBS->REST_KMS_CONNECTOR_VALIDATION_TOKEN_MODE };
															#line 391 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (mode.compare("file") == 0)
															#line 929 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 392 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StrictFuture<Void> __when_expr_0 = parseDiscoverKmsUrlFile(ctx, SERVER_KNOBS->REST_KMS_CONNECTOR_DISCOVER_KMS_URL_FILE);
															#line 392 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (static_cast<DiscoverKmsUrlsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 935 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DiscoverKmsUrlsActor*>(this)->actor_wait_state = 1;
															#line 392 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DiscoverKmsUrlsActor, 0, Void >*>(static_cast<DiscoverKmsUrlsActor*>(this)));
															#line 940 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 394 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				return a_body1Catch1(not_implemented(), loopDepth);
															#line 947 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DiscoverKmsUrlsActorState();
		static_cast<DiscoverKmsUrlsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 397 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ctx->lastKmsUrlDiscoverTS = now();
															#line 399 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<DiscoverKmsUrlsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DiscoverKmsUrlsActorState(); static_cast<DiscoverKmsUrlsActor*>(this)->destroy(); return 0; }
															#line 972 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<DiscoverKmsUrlsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DiscoverKmsUrlsActorState();
		static_cast<DiscoverKmsUrlsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DiscoverKmsUrlsActor*>(this)->actor_wait_state > 0) static_cast<DiscoverKmsUrlsActor*>(this)->actor_wait_state = 0;
		static_cast<DiscoverKmsUrlsActor*>(this)->ActorCallback< DiscoverKmsUrlsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DiscoverKmsUrlsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("discoverKmsUrls", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DiscoverKmsUrlsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("discoverKmsUrls", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DiscoverKmsUrlsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("discoverKmsUrls", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DiscoverKmsUrlsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("discoverKmsUrls", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DiscoverKmsUrlsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("discoverKmsUrls", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DiscoverKmsUrlsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("discoverKmsUrls", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	RefreshPersistedUrls refreshPersistedUrls;
															#line 1071 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via discoverKmsUrls()
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class DiscoverKmsUrlsActor final : public Actor<Void>, public ActorCallback< DiscoverKmsUrlsActor, 0, Void >, public FastAllocated<DiscoverKmsUrlsActor>, public DiscoverKmsUrlsActorState<DiscoverKmsUrlsActor> {
															#line 1076 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<DiscoverKmsUrlsActor>::operator new;
	using FastAllocated<DiscoverKmsUrlsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10571773211760736256UL, 687258442027214592UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DiscoverKmsUrlsActor, 0, Void >;
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	DiscoverKmsUrlsActor(Reference<RESTKmsConnectorCtx> const& ctx,RefreshPersistedUrls const& refreshPersistedUrls) 
															#line 1093 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   DiscoverKmsUrlsActorState<DiscoverKmsUrlsActor>(ctx, refreshPersistedUrls),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("discoverKmsUrls", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5701154263732512000UL, 355808445171514112UL);
		ActorExecutionContextHelper __helper(static_cast<DiscoverKmsUrlsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("discoverKmsUrls");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("discoverKmsUrls", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DiscoverKmsUrlsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> discoverKmsUrls( Reference<RESTKmsConnectorCtx> const& ctx, RefreshPersistedUrls const& refreshPersistedUrls ) {
															#line 371 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new DiscoverKmsUrlsActor(ctx, refreshPersistedUrls));
															#line 1126 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 401 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

void checkResponseForError(Reference<RESTKmsConnectorCtx> ctx,
                           const rapidjson::Document& doc,
                           IsCipherType isCipherType) {
	// check version tag sanity
	if (!doc.HasMember(REQUEST_VERSION_TAG) || !doc[REQUEST_VERSION_TAG].IsInt()) {
		TraceEvent(SevWarnAlways, "RESTKMSResponseMissingVersion", ctx->uid).log();
		CODE_PROBE(true, "KMS response missing version");
		throw rest_malformed_response();
	}

	const int version = doc[REQUEST_VERSION_TAG].GetInt();
	const int maxSupportedVersion = isCipherType ? SERVER_KNOBS->REST_KMS_MAX_CIPHER_REQUEST_VERSION
	                                             : SERVER_KNOBS->REST_KMS_MAX_BLOB_METADATA_REQUEST_VERSION;
	if (version == INVALID_REQUEST_VERSION || version > maxSupportedVersion) {
		TraceEvent(SevWarnAlways, "RESTKMSResponseInvalidVersion", ctx->uid)
		    .detail("Version", version)
		    .detail("MaxSupportedVersion", maxSupportedVersion);
		CODE_PROBE(true, "KMS response invalid version");
		throw rest_malformed_response();
	}

	// Check if response has error
	Optional<ErrorDetail> errorDetails = RESTKmsConnectorUtils::getError(doc);
	if (errorDetails.present()) {
		TraceEvent("RESTKMSErrorResponse", ctx->uid)
		    .detail("ErrorMsg", errorDetails->errorMsg)
		    .detail("ErrorCode", errorDetails->errorCode);
		throw encrypt_keys_fetch_failed();
	}
}

void checkDocForNewKmsUrls(Reference<RESTKmsConnectorCtx> ctx,
                           Reference<HTTP::IncomingResponse> resp,
                           const rapidjson::Document& doc) {
	if (doc.HasMember(KMS_URLS_TAG) && !doc[KMS_URLS_TAG].IsNull()) {
		try {
			extractKmsUrls(ctx, doc, resp);
		} catch (Error& e) {
			TraceEvent("RESTRefreshKmsUrlsFailed", ctx->uid).error(e);
			// Given cipherKeyDetails extraction was done successfully, ignore KmsUrls parsing error
		}
	}
}

Standalone<VectorRef<EncryptCipherKeyDetailsRef>> parseEncryptCipherResponse(Reference<RESTKmsConnectorCtx> ctx,
                                                                             Reference<HTTP::IncomingResponse> resp) {
	// Acceptable response payload json format:
	//
	// response_json_payload {
	//   "version" = <version>
	//   "cipher_key_details" : [
	//     {
	//        "base_cipher_id"    : <cipherKeyId>,
	//        "encrypt_domain_id" : <domainId>,
	//        "base_cipher"       : <baseCipher>,
	//        "refresh_after_sec"   : <refreshTimeInterval>, (Optional)
	//        "expire_after_sec"    : <expireTimeInterval>  (Optional)
	//     },
	//     {
	//         ....
	//	   }
	//   ],
	//   "kms_urls" : [
	//         "url1", "url2", ...
	//   ],
	//	 "error" : {					// Optional, populated by the KMS, if present, rest of payload is ignored.
	//		"errMsg" : <message>,
	//		"errCode": <code>
	// 	  }
	// }

	if (!resp.isValid() || resp->code != HTTP::HTTP_STATUS_CODE_OK) {
		// STATUS_OK is gating factor for REST request success
		throw http_request_failed();
	}

	if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::VERBOSE) {
		TraceEvent("RESTParseEncryptCipherResponseStart", ctx->uid);
	}

	rapidjson::Document doc;
	doc.Parse(resp->data.content.data());

	checkResponseForError(ctx, doc, IsCipherType::True);

	Standalone<VectorRef<EncryptCipherKeyDetailsRef>> result;

	// Extract CipherKeyDetails
	if (!doc.HasMember(CIPHER_KEY_DETAILS_TAG) || !doc[CIPHER_KEY_DETAILS_TAG].IsArray()) {
		TraceEvent(SevWarn, "RESTParseEncryptCipherResponseFailed", ctx->uid)
		    .detail("Reason", "MissingCipherKeyDetails");
		CODE_PROBE(true, "REST CipherKeyDetails not array");
		throw rest_malformed_response();
	}

	for (const auto& cipherDetail : doc[CIPHER_KEY_DETAILS_TAG].GetArray()) {
		if (!cipherDetail.IsObject()) {
			TraceEvent(SevWarn, "RESTParseEncryptCipherResponseFailed", ctx->uid)
			    .detail("CipherDetailType", cipherDetail.GetType())
			    .detail("Reason", "EncryptKeyDetailsNotObject");
			CODE_PROBE(true, "REST CipherKeyDetail not object");
			throw rest_malformed_response();
		}

		const bool isBaseCipherIdPresent = cipherDetail.HasMember(BASE_CIPHER_ID_TAG);
		const bool isBaseCipherPresent = cipherDetail.HasMember(BASE_CIPHER_TAG);
		const bool isEncryptDomainIdPresent = cipherDetail.HasMember(ENCRYPT_DOMAIN_ID_TAG);
		if (!isBaseCipherIdPresent || !isBaseCipherPresent || !isEncryptDomainIdPresent) {
			TraceEvent(SevWarn, "RESTParseEncryptCipherResponseFailed", ctx->uid)
			    .detail("Reason", "MalformedKeyDetail")
			    .detail("BaseCipherIdPresent", isBaseCipherIdPresent)
			    .detail("BaseCipherPresent", isBaseCipherPresent)
			    .detail("EncryptDomainIdPresent", isEncryptDomainIdPresent);
			CODE_PROBE(true, "REST CipherKeyDetail malformed");
			throw rest_malformed_response();
		}

		const int cipherKeyLen = cipherDetail[BASE_CIPHER_TAG].GetStringLength();
		std::unique_ptr<uint8_t[]> cipherKey = std::make_unique<uint8_t[]>(cipherKeyLen);
		memcpy(cipherKey.get(), cipherDetail[BASE_CIPHER_TAG].GetString(), cipherKeyLen);

		// Extract cipher refresh and/or expiry interval if supplied
		Optional<int64_t> refreshAfterSec =
		    cipherDetail.HasMember(REFRESH_AFTER_SEC) && cipherDetail[REFRESH_AFTER_SEC].GetInt64() > 0
		        ? cipherDetail[REFRESH_AFTER_SEC].GetInt64()
		        : Optional<int64_t>();
		Optional<int64_t> expireAfterSec =
		    cipherDetail.HasMember(EXPIRE_AFTER_SEC) ? cipherDetail[EXPIRE_AFTER_SEC].GetInt64() : Optional<int64_t>();

		EncryptCipherDomainId domainId = cipherDetail[ENCRYPT_DOMAIN_ID_TAG].GetInt64();
		EncryptCipherBaseKeyId baseCipherId = cipherDetail[BASE_CIPHER_ID_TAG].GetUint64();
		StringRef cipher = StringRef(cipherKey.get(), cipherKeyLen);

		// https://en.wikipedia.org/wiki/Key_checksum_value
		// Key Check Value (KCV) is the checksum of a cryptographic key, it is used to validate integrity of the
		// 'baseCipher' key supplied by the external KMS. The checksum computed is eventually persisted as part of
		// EncryptionHeader and assists in following scenarios: a) 'baseCipher' corruption happen external to FDB b)
		// 'baseCipher' corruption within FDB processes
		//
		// Approach compute KCV after reading it from the network buffer, HTTP checksum protects against potential
		// on-wire corruption
		if (cipher.size() > MAX_BASE_CIPHER_LEN) {
			// HMAC_SHA digest generation accepts upto MAX_BASE_CIPHER_LEN key-buffer, longer keys are truncated and
			// weakens the security guarantees.
			TraceEvent(SevWarnAlways, "RESTKmsConnectorMaxBaseCipherKeyLimit")
			    .detail("MaxAllowed", MAX_BASE_CIPHER_LEN)
			    .detail("BaseCipherLen", cipher.size());
			throw rest_max_base_cipher_len();
		}

		EncryptCipherKeyCheckValue cipherKCV = Sha256KCV().computeKCV(cipher.begin(), cipher.size());

		if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::DEBUG) {
			TraceEvent event("RESTParseEncryptCipherResponse", ctx->uid);
			event.detail("DomainId", domainId);
			event.detail("BaseCipherId", baseCipherId);
			event.detail("BaseCipherLen", cipher.size());
			event.detail("BaseCipherKCV", cipherKCV);
			if (refreshAfterSec.present()) {
				event.detail("RefreshAt", refreshAfterSec.get());
			}
			if (expireAfterSec.present()) {
				event.detail("ExpireAt", expireAfterSec.get());
			}
		}

		result.emplace_back_deep(
		    result.arena(), domainId, baseCipherId, cipher, cipherKCV, refreshAfterSec, expireAfterSec);
	}

	checkDocForNewKmsUrls(ctx, resp, doc);

	return result;
}

Standalone<VectorRef<BlobMetadataDetailsRef>> parseBlobMetadataResponse(Reference<RESTKmsConnectorCtx> ctx,
                                                                        Reference<HTTP::IncomingResponse> resp) {
	// Acceptable response payload json format:
	// (baseLocation and partitions follow the same properties as described in BlobMetadataUtils.h)
	//
	// response_json_payload {
	//   "version" = <version>
	//   "blob_metadata_details" : [
	//     {
	//        "domain_id" : <domainId>,
	//        "locations" : [
	//			  { id: 1, path: "fdbblob://partition1"} , {id: 2, path: "fdbblob://partition2"}, ...
	//		  ],
	//        "refresh_after_sec"   : <refreshTimeInterval>, (Optional)
	//        "expire_after_sec"    : <expireTimeInterval>  (Optional)
	//     },
	//     {
	//         ....
	//	   }
	//   ],
	//   "kms_urls" : [
	//         "url1", "url2", ...
	//   ],
	//	 "error" : {					// Optional, populated by the KMS, if present, rest of payload is ignored.
	//		"errMsg" : <message>,
	//		"errCode": <code>
	// 	  }
	// }

	if (resp->code != HTTP::HTTP_STATUS_CODE_OK) {
		// STATUS_OK is gating factor for REST request success
		throw http_request_failed();
	}

	rapidjson::Document doc;
	doc.Parse(resp->data.content.data());

	checkResponseForError(ctx, doc, IsCipherType::False);

	Standalone<VectorRef<BlobMetadataDetailsRef>> result;

	// Extract CipherKeyDetails
	if (!doc.HasMember(BLOB_METADATA_DETAILS_TAG) || !doc[BLOB_METADATA_DETAILS_TAG].IsArray()) {
		TraceEvent(SevWarn, "ParseBlobMetadataResponseFailureMissingDetails", ctx->uid).log();
		CODE_PROBE(true, "REST BlobMetadata details missing or not-array");
		throw rest_malformed_response();
	}

	for (const auto& detail : doc[BLOB_METADATA_DETAILS_TAG].GetArray()) {
		if (!detail.IsObject()) {
			TraceEvent(SevWarn, "ParseBlobMetadataResponseFailureDetailsNotObject", ctx->uid)
			    .detail("CipherDetailType", detail.GetType());
			CODE_PROBE(true, "REST BlobMetadata detail not-object");
			throw rest_malformed_response();
		}

		const bool isDomainIdPresent = detail.HasMember(BLOB_METADATA_DOMAIN_ID_TAG);
		const bool isLocationsPresent =
		    detail.HasMember(BLOB_METADATA_LOCATIONS_TAG) && detail[BLOB_METADATA_LOCATIONS_TAG].IsArray();
		if (!isDomainIdPresent || !isLocationsPresent) {
			TraceEvent(SevWarn, "ParseBlobMetadataResponseMalformedDetail", ctx->uid)
			    .detail("DomainIdPresent", isDomainIdPresent)
			    .detail("LocationsPresent", isLocationsPresent);
			CODE_PROBE(true, "REST BlobMetadata detail malformed");
			throw rest_malformed_response();
		}

		BlobMetadataDomainId domainId = detail[BLOB_METADATA_DOMAIN_ID_TAG].GetInt64();

		// just do extra memory copy for simplicity here
		Standalone<VectorRef<BlobMetadataLocationRef>> locations;
		for (const auto& location : detail[BLOB_METADATA_LOCATIONS_TAG].GetArray()) {
			if (!location.IsObject()) {
				TraceEvent("ParseBlobMetadataResponseFailureLocationNotObject", ctx->uid)
				    .detail("LocationType", location.GetType());
				throw rest_malformed_response();
			}
			const bool isLocationIdPresent = location.HasMember(BLOB_METADATA_LOCATION_ID_TAG);
			const bool isPathPresent = location.HasMember(BLOB_METADATA_LOCATION_PATH_TAG);
			if (!isLocationIdPresent || !isPathPresent) {
				TraceEvent(SevWarn, "ParseBlobMetadataResponseMalformedLocation", ctx->uid)
				    .detail("LocationIdPresent", isLocationIdPresent)
				    .detail("PathPresent", isPathPresent);
				CODE_PROBE(true, "REST BlobMetadata location malformed");
				throw rest_malformed_response();
			}

			BlobMetadataLocationId locationId = location[BLOB_METADATA_LOCATION_ID_TAG].GetInt64();

			const int pathLen = location[BLOB_METADATA_LOCATION_PATH_TAG].GetStringLength();
			std::unique_ptr<uint8_t[]> pathStr = std::make_unique<uint8_t[]>(pathLen);
			memcpy(pathStr.get(), location[BLOB_METADATA_LOCATION_PATH_TAG].GetString(), pathLen);
			locations.emplace_back_deep(locations.arena(), locationId, StringRef(pathStr.get(), pathLen));
		}

		// Extract refresh and/or expiry interval if supplied
		double refreshAt = detail.HasMember(REFRESH_AFTER_SEC) && detail[REFRESH_AFTER_SEC].GetInt64() > 0
		                       ? now() + detail[REFRESH_AFTER_SEC].GetInt64()
		                       : std::numeric_limits<double>::max();
		double expireAt = detail.HasMember(EXPIRE_AFTER_SEC) ? now() + detail[EXPIRE_AFTER_SEC].GetInt64()
		                                                     : std::numeric_limits<double>::max();
		result.emplace_back_deep(result.arena(), domainId, locations, refreshAt, expireAt);
	}

	checkDocForNewKmsUrls(ctx, resp, doc);

	return result;
}

StringRef getEncryptKeysByKeyIdsRequestBody(Reference<RESTKmsConnectorCtx> ctx,
                                            const KmsConnLookupEKsByKeyIdsReq& req,
                                            const bool refreshKmsUrls,
                                            Arena& arena) {
	// Acceptable request payload json format:
	//
	// request_json_payload {
	//   "version" = <version>
	//   "cipher_key_details" = [
	//     {
	//        "base_cipher_id"      : <cipherKeyId>
	//        "encrypt_domain_id"   : <domainId>		// Optional
	//     },
	//     {
	//         ....
	//	   }
	//   ],
	//   "validation_tokens" = [
	//     {
	//        "token_name" : <name>,
	//        "token_value": <value>
	//     },
	//     {
	//         ....
	//     }
	//   ]
	//   "refresh_kms_urls" = 1/0
	//   "debug_uid" = <uid-string>   // Optional debug info to trace requests across FDB <--> KMS
	// }

	rapidjson::Document doc;
	doc.SetObject();

	// Append 'request version'
	addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_CURRENT_BLOB_METADATA_REQUEST_VERSION);

	// Append 'cipher_key_details' as json array
	rapidjson::Value keyIdDetails(rapidjson::kArrayType);
	for (const auto& detail : req.encryptKeyInfos) {
		addBaseCipherIdDomIdToDoc(doc, keyIdDetails, detail.baseCipherId, detail.domainId);
	}
	rapidjson::Value memberKey(CIPHER_KEY_DETAILS_TAG, doc.GetAllocator());
	doc.AddMember(memberKey, keyIdDetails, doc.GetAllocator());

	// Append 'validation_tokens' as json array
	addValidationTokensSectionToJsonDoc(doc, ctx->validationTokenMap);

	// Append 'refresh_kms_urls'
	addRefreshKmsUrlsSectionToJsonDoc(doc, refreshKmsUrls);

	// Append 'debug_uid' section if needed
	addDebugUidSectionToJsonDoc(doc, req.debugId);

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	StringRef ref = makeString(sb.GetSize(), arena);
	memcpy(mutateString(ref), sb.GetString(), sb.GetSize());
	return ref;
}

															#line 1478 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via kmsRequestImpl()
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class T, class KmsRequestImplActor>
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class KmsRequestImplActorState {
															#line 1485 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	KmsRequestImplActorState(Reference<RESTKmsConnectorCtx> const& ctx,std::string const& urlSuffix,StringRef const& requestBodyRef,std::function<T(Reference<RESTKmsConnectorCtx>, Reference<HTTP::IncomingResponse>)> const& parseFunc) 
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx),
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   urlSuffix(urlSuffix),
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   requestBodyRef(requestBodyRef),
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   parseFunc(parseFunc),
															#line 755 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   requestID(deterministicRandom()->randomUniqueID()),
															#line 762 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   pass(0),
															#line 763 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   urlCtx()
															#line 1504 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("kmsRequestImpl", reinterpret_cast<unsigned long>(this));

	}
	~KmsRequestImplActorState() 
	{
		fdb_probe_actor_destroy("kmsRequestImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 764 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			;
															#line 1519 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~KmsRequestImplActorState();
		static_cast<KmsRequestImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 765 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		idx = 0;
															#line 766 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		start = now();
															#line 768 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		pass++;
															#line 769 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		;
															#line 1555 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 822 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_1 = discoverKmsUrls(ctx, RefreshPersistedUrls::True);
															#line 822 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<KmsRequestImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1566 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<KmsRequestImplActor*>(this)->actor_wait_state = 2;
															#line 822 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< KmsRequestImplActor, 1, Void >*>(static_cast<KmsRequestImplActor*>(this)));
															#line 1571 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
															#line 769 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!(idx < ctx->kmsUrlStore.kmsUrls.size()))
															#line 1587 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 770 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		urlCtx = &ctx->kmsUrlStore.kmsUrls[idx++];
															#line 1593 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		try {
															#line 772 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string kmsEncryptionFullUrl = getFullRequestUrl(ctx, urlCtx->url, urlSuffix);
															#line 774 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::DEBUG)
															#line 1599 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 775 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				TraceEvent("RESTKmsRequestImpl", ctx->uid) .detail("Pass", pass) .detail("RequestID", requestID) .detail("FullUrl", kmsEncryptionFullUrl) .detail("StartIdx", start) .detail("CurIdx", idx) .detail("LastKmsUrlDiscoverTS", ctx->lastKmsUrlDiscoverTS);
															#line 1603 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 784 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			StrictFuture<Reference<HTTP::IncomingResponse>> __when_expr_0 = ctx->restClient.doPost( kmsEncryptionFullUrl, requestBodyRef.toString(), RESTKmsConnectorUtils::getHTTPHeaders());
															#line 784 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (static_cast<KmsRequestImplActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1609 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<KmsRequestImplActor*>(this)->actor_wait_state = 1;
															#line 784 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< KmsRequestImplActor, 0, Reference<HTTP::IncomingResponse> >*>(static_cast<KmsRequestImplActor*>(this)));
															#line 1614 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 817 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (start < ctx->lastKmsUrlDiscoverTS)
															#line 1642 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
															#line 818 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			idx = 0;
															#line 1646 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 797 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ctx->kmsUrlStore.penalize(*urlCtx, KmsUrlCtx<KmsUrlPenaltyParams>::PenaltyType::TIMEOUT);
															#line 799 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (!isKmsNotReachable(e.code()))
															#line 1659 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 800 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::DEBUG)
															#line 1663 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
															#line 801 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					TraceEvent("KmsRequestFailedUnreachable", ctx->uid).error(e).detail("RequestID", requestID);
															#line 1667 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				}
															#line 803 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 2));
															#line 1671 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 805 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent(SevDebug, "KmsRequestError", ctx->uid).error(e).detail("RequestID", requestID);
															#line 1675 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(Reference<HTTP::IncomingResponse> const& resp,int loopDepth) 
	{
															#line 786 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		urlCtx->nRequests++;
															#line 1690 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		try {
															#line 789 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			T parsedResp = parseFunc(ctx, resp);
															#line 790 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (!static_cast<KmsRequestImplActor*>(this)->SAV<T>::futures) { (void)(parsedResp); this->~KmsRequestImplActorState(); static_cast<KmsRequestImplActor*>(this)->destroy(); return 0; }
															#line 1696 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			new (&static_cast<KmsRequestImplActor*>(this)->SAV< T >::value()) T(parsedResp);
			this->~KmsRequestImplActorState();
			static_cast<KmsRequestImplActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(Reference<HTTP::IncomingResponse> && resp,int loopDepth) 
	{
															#line 786 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		urlCtx->nRequests++;
															#line 1714 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		try {
															#line 789 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			T parsedResp = parseFunc(ctx, resp);
															#line 790 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (!static_cast<KmsRequestImplActor*>(this)->SAV<T>::futures) { (void)(parsedResp); this->~KmsRequestImplActorState(); static_cast<KmsRequestImplActor*>(this)->destroy(); return 0; }
															#line 1720 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			new (&static_cast<KmsRequestImplActor*>(this)->SAV< T >::value()) T(parsedResp);
			this->~KmsRequestImplActorState();
			static_cast<KmsRequestImplActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1cont3Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1cont3Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Reference<HTTP::IncomingResponse> const& resp,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(resp, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(Reference<HTTP::IncomingResponse> && resp,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(std::move(resp), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<KmsRequestImplActor*>(this)->actor_wait_state > 0) static_cast<KmsRequestImplActor*>(this)->actor_wait_state = 0;
		static_cast<KmsRequestImplActor*>(this)->ActorCallback< KmsRequestImplActor, 0, Reference<HTTP::IncomingResponse> >::remove();

	}
	void a_callback_fire(ActorCallback< KmsRequestImplActor, 0, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> const& value) 
	{
		fdb_probe_actor_enter("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KmsRequestImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< KmsRequestImplActor, 0, Reference<HTTP::IncomingResponse> >*,Reference<HTTP::IncomingResponse> && value) 
	{
		fdb_probe_actor_enter("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KmsRequestImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< KmsRequestImplActor, 0, Reference<HTTP::IncomingResponse> >*,Error err) 
	{
		fdb_probe_actor_enter("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<KmsRequestImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 792 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent(SevWarn, "KmsRequestRespParseFailure").error(e).detail("RequestID", requestID);
															#line 793 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ctx->kmsUrlStore.penalize(*urlCtx, KmsUrlCtx<KmsUrlPenaltyParams>::PenaltyType::MALFORMED_RESPONSE);
															#line 1822 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<KmsRequestImplActor*>(this)->actor_wait_state > 0) static_cast<KmsRequestImplActor*>(this)->actor_wait_state = 0;
		static_cast<KmsRequestImplActor*>(this)->ActorCallback< KmsRequestImplActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< KmsRequestImplActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KmsRequestImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< KmsRequestImplActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KmsRequestImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< KmsRequestImplActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<KmsRequestImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kmsRequestImpl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::string urlSuffix;
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	StringRef requestBodyRef;
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::function<T(Reference<RESTKmsConnectorCtx>, Reference<HTTP::IncomingResponse>)> parseFunc;
															#line 755 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	UID requestID;
															#line 762 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	int pass;
															#line 763 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	KmsUrlCtx<KmsUrlPenaltyParams>* urlCtx;
															#line 765 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	int idx;
															#line 766 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	double start;
															#line 1951 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via kmsRequestImpl()
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class T>
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class KmsRequestImplActor final : public Actor<T>, public ActorCallback< KmsRequestImplActor<T>, 0, Reference<HTTP::IncomingResponse> >, public ActorCallback< KmsRequestImplActor<T>, 1, Void >, public FastAllocated<KmsRequestImplActor<T>>, public KmsRequestImplActorState<T, KmsRequestImplActor<T>> {
															#line 1958 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<KmsRequestImplActor<T>>::operator new;
	using FastAllocated<KmsRequestImplActor<T>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5635664223725345536UL, 14815515009387323648UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<T>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< KmsRequestImplActor<T>, 0, Reference<HTTP::IncomingResponse> >;
friend struct ActorCallback< KmsRequestImplActor<T>, 1, Void >;
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	KmsRequestImplActor(Reference<RESTKmsConnectorCtx> const& ctx,std::string const& urlSuffix,StringRef const& requestBodyRef,std::function<T(Reference<RESTKmsConnectorCtx>, Reference<HTTP::IncomingResponse>)> const& parseFunc) 
															#line 1976 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<T>(),
		   KmsRequestImplActorState<T, KmsRequestImplActor<T>>(ctx, urlSuffix, requestBodyRef, parseFunc),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("kmsRequestImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(348943008905229568UL, 16609751683208066816UL);
		ActorExecutionContextHelper __helper(static_cast<KmsRequestImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("kmsRequestImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("kmsRequestImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< KmsRequestImplActor<T>, 0, Reference<HTTP::IncomingResponse> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< KmsRequestImplActor<T>, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class T>
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<T> kmsRequestImpl( Reference<RESTKmsConnectorCtx> const& ctx, std::string const& urlSuffix, StringRef const& requestBodyRef, std::function<T(Reference<RESTKmsConnectorCtx>, Reference<HTTP::IncomingResponse>)> const& parseFunc ) {
															#line 749 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<T>(new KmsRequestImplActor<T>(ctx, urlSuffix, requestBodyRef, parseFunc));
															#line 2012 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 825 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 2017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via fetchEncryptionKeysByKeyIds()
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class FetchEncryptionKeysByKeyIdsActor>
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FetchEncryptionKeysByKeyIdsActorState {
															#line 2024 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FetchEncryptionKeysByKeyIdsActorState(Reference<RESTKmsConnectorCtx> const& ctx,KmsConnLookupEKsByKeyIdsReq const& req) 
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx),
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   req(req),
															#line 827 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   reply()
															#line 2035 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("fetchEncryptionKeysByKeyIds", reinterpret_cast<unsigned long>(this));

	}
	~FetchEncryptionKeysByKeyIdsActorState() 
	{
		fdb_probe_actor_destroy("fetchEncryptionKeysByKeyIds", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 830 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				bool refreshKmsUrls = shouldRefreshKmsUrls(ctx);
															#line 831 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StringRef requestBodyRef = getEncryptKeysByKeyIdsRequestBody(ctx, req, refreshKmsUrls, req.arena);
															#line 832 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				std::function<Standalone<VectorRef<EncryptCipherKeyDetailsRef>>(Reference<RESTKmsConnectorCtx>, Reference<HTTP::IncomingResponse>)> f = &parseEncryptCipherResponse;
															#line 835 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store( reply.cipherKeyDetails, kmsRequestImpl( ctx, SERVER_KNOBS->REST_KMS_CONNECTOR_GET_ENCRYPTION_KEYS_ENDPOINT, requestBodyRef, std::move(f)));
															#line 835 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2059 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->actor_wait_state = 1;
															#line 835 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchEncryptionKeysByKeyIdsActor, 0, Void >*>(static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)));
															#line 2064 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchEncryptionKeysByKeyIdsActorState();
		static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 847 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchEncryptionKeysByKeyIdsActorState(); static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->destroy(); return 0; }
															#line 2093 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchEncryptionKeysByKeyIdsActorState();
		static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 841 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent("RESTLookupEKsByKeyIdsFailed", ctx->uid).error(e);
															#line 842 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (!canReplyWith(e))
															#line 2108 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 843 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 2112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 845 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			req.reply.sendError(e);
															#line 2116 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 839 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		req.reply.send(reply);
															#line 2131 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 839 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		req.reply.send(reply);
															#line 2140 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->actor_wait_state > 0) static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->actor_wait_state = 0;
		static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->ActorCallback< FetchEncryptionKeysByKeyIdsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchEncryptionKeysByKeyIdsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchEncryptionKeysByKeyIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchEncryptionKeysByKeyIds", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchEncryptionKeysByKeyIdsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchEncryptionKeysByKeyIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchEncryptionKeysByKeyIds", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchEncryptionKeysByKeyIdsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchEncryptionKeysByKeyIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchEncryptionKeysByKeyIds", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	KmsConnLookupEKsByKeyIdsReq req;
															#line 827 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	KmsConnLookupEKsByKeyIdsRep reply;
															#line 2239 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via fetchEncryptionKeysByKeyIds()
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FetchEncryptionKeysByKeyIdsActor final : public Actor<Void>, public ActorCallback< FetchEncryptionKeysByKeyIdsActor, 0, Void >, public FastAllocated<FetchEncryptionKeysByKeyIdsActor>, public FetchEncryptionKeysByKeyIdsActorState<FetchEncryptionKeysByKeyIdsActor> {
															#line 2244 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<FetchEncryptionKeysByKeyIdsActor>::operator new;
	using FastAllocated<FetchEncryptionKeysByKeyIdsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7180954954193494016UL, 6414005636416131840UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchEncryptionKeysByKeyIdsActor, 0, Void >;
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FetchEncryptionKeysByKeyIdsActor(Reference<RESTKmsConnectorCtx> const& ctx,KmsConnLookupEKsByKeyIdsReq const& req) 
															#line 2261 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FetchEncryptionKeysByKeyIdsActorState<FetchEncryptionKeysByKeyIdsActor>(ctx, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchEncryptionKeysByKeyIds", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4167165272695444992UL, 15050557084529312512UL);
		ActorExecutionContextHelper __helper(static_cast<FetchEncryptionKeysByKeyIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchEncryptionKeysByKeyIds");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchEncryptionKeysByKeyIds", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchEncryptionKeysByKeyIdsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> fetchEncryptionKeysByKeyIds( Reference<RESTKmsConnectorCtx> const& ctx, KmsConnLookupEKsByKeyIdsReq const& req ) {
															#line 826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new FetchEncryptionKeysByKeyIdsActor(ctx, req));
															#line 2294 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 849 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

StringRef getEncryptKeysByDomainIdsRequestBody(Reference<RESTKmsConnectorCtx> ctx,
                                               const KmsConnLookupEKsByDomainIdsReq& req,
                                               const bool refreshKmsUrls,
                                               Arena& arena) {
	// Acceptable request payload json format:
	//
	// request_json_payload {
	//   "version" = <version>
	//   "cipher_key_details" = [
	//     {
	//        "encrypt_domain_id"   : <domainId>
	//     },
	//     {
	//         ....
	//	   }
	//   ],
	//   "validation_tokens" = [
	//     {
	//        "token_name" : <name>,
	//        "token_value": <value>
	//     },
	//     {
	//         ....
	//     }
	//   ]
	//   "refresh_kms_urls" = 1/0
	//   "debug_uid" = <uid-string>     // Optional debug info to trace requests across FDB <--> KMS
	// }

	rapidjson::Document doc;
	doc.SetObject();

	// Append 'request version'
	addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_CURRENT_CIPHER_REQUEST_VERSION);

	// Append 'cipher_key_details' as json array
	addLatestDomainDetailsToDoc(doc, CIPHER_KEY_DETAILS_TAG, ENCRYPT_DOMAIN_ID_TAG, req.encryptDomainIds);

	// Append 'validation_tokens' as json array
	addValidationTokensSectionToJsonDoc(doc, ctx->validationTokenMap);

	// Append 'refresh_kms_urls'
	addRefreshKmsUrlsSectionToJsonDoc(doc, refreshKmsUrls);

	// Append 'debug_uid' section if needed
	addDebugUidSectionToJsonDoc(doc, req.debugId);

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	StringRef ref = makeString(sb.GetSize(), arena);
	memcpy(mutateString(ref), sb.GetString(), sb.GetSize());
	return ref;
}

															#line 2356 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via fetchEncryptionKeysByDomainIds()
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class FetchEncryptionKeysByDomainIdsActor>
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FetchEncryptionKeysByDomainIdsActorState {
															#line 2363 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FetchEncryptionKeysByDomainIdsActorState(Reference<RESTKmsConnectorCtx> const& ctx,KmsConnLookupEKsByDomainIdsReq const& req) 
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx),
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   req(req),
															#line 909 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   reply()
															#line 2374 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("fetchEncryptionKeysByDomainIds", reinterpret_cast<unsigned long>(this));

	}
	~FetchEncryptionKeysByDomainIdsActorState() 
	{
		fdb_probe_actor_destroy("fetchEncryptionKeysByDomainIds", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 911 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				bool refreshKmsUrls = shouldRefreshKmsUrls(ctx);
															#line 912 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StringRef requestBodyRef = getEncryptKeysByDomainIdsRequestBody(ctx, req, refreshKmsUrls, req.arena);
															#line 914 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				std::function<Standalone<VectorRef<EncryptCipherKeyDetailsRef>>(Reference<RESTKmsConnectorCtx>, Reference<HTTP::IncomingResponse>)> f = &parseEncryptCipherResponse;
															#line 918 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(reply.cipherKeyDetails, kmsRequestImpl(ctx, SERVER_KNOBS->REST_KMS_CONNECTOR_GET_LATEST_ENCRYPTION_KEYS_ENDPOINT, requestBodyRef, std::move(f)));
															#line 918 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2398 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->actor_wait_state = 1;
															#line 918 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchEncryptionKeysByDomainIdsActor, 0, Void >*>(static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)));
															#line 2403 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchEncryptionKeysByDomainIdsActorState();
		static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 931 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchEncryptionKeysByDomainIdsActorState(); static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->destroy(); return 0; }
															#line 2432 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchEncryptionKeysByDomainIdsActorState();
		static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 925 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent("RESTLookupEKsByDomainIdsFailed", ctx->uid).error(e);
															#line 926 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (!canReplyWith(e))
															#line 2447 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 927 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 2451 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 929 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			req.reply.sendError(e);
															#line 2455 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 923 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		req.reply.send(reply);
															#line 2470 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 923 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		req.reply.send(reply);
															#line 2479 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->actor_wait_state > 0) static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->actor_wait_state = 0;
		static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->ActorCallback< FetchEncryptionKeysByDomainIdsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchEncryptionKeysByDomainIdsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchEncryptionKeysByDomainIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchEncryptionKeysByDomainIds", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchEncryptionKeysByDomainIdsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchEncryptionKeysByDomainIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchEncryptionKeysByDomainIds", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchEncryptionKeysByDomainIdsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchEncryptionKeysByDomainIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchEncryptionKeysByDomainIds", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	KmsConnLookupEKsByDomainIdsReq req;
															#line 909 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	KmsConnLookupEKsByDomainIdsRep reply;
															#line 2578 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via fetchEncryptionKeysByDomainIds()
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FetchEncryptionKeysByDomainIdsActor final : public Actor<Void>, public ActorCallback< FetchEncryptionKeysByDomainIdsActor, 0, Void >, public FastAllocated<FetchEncryptionKeysByDomainIdsActor>, public FetchEncryptionKeysByDomainIdsActorState<FetchEncryptionKeysByDomainIdsActor> {
															#line 2583 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<FetchEncryptionKeysByDomainIdsActor>::operator new;
	using FastAllocated<FetchEncryptionKeysByDomainIdsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11505054244051774976UL, 15436391048890584320UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchEncryptionKeysByDomainIdsActor, 0, Void >;
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FetchEncryptionKeysByDomainIdsActor(Reference<RESTKmsConnectorCtx> const& ctx,KmsConnLookupEKsByDomainIdsReq const& req) 
															#line 2600 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FetchEncryptionKeysByDomainIdsActorState<FetchEncryptionKeysByDomainIdsActor>(ctx, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchEncryptionKeysByDomainIds", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12011643318214593536UL, 14765249188923834112UL);
		ActorExecutionContextHelper __helper(static_cast<FetchEncryptionKeysByDomainIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchEncryptionKeysByDomainIds");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchEncryptionKeysByDomainIds", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchEncryptionKeysByDomainIdsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> fetchEncryptionKeysByDomainIds( Reference<RESTKmsConnectorCtx> const& ctx, KmsConnLookupEKsByDomainIdsReq const& req ) {
															#line 907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new FetchEncryptionKeysByDomainIdsActor(ctx, req));
															#line 2633 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 933 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

StringRef getBlobMetadataRequestBody(Reference<RESTKmsConnectorCtx> ctx,
                                     KmsConnBlobMetadataReq& req,
                                     const bool refreshKmsUrls) {
	// Acceptable request payload json format:
	//
	// request_json_payload {
	//   "version" = <version>
	//   "blob_metadata_details" = [
	//     {
	//        "domain_id"   : <domainId>
	//     },
	//     {
	//         ....
	//	   }
	//   ],
	//   "validation_tokens" = [
	//     {
	//        "token_name" : <name>,
	//        "token_value": <value>
	//     },
	//     {
	//         ....
	//     }
	//   ]
	//   "refresh_kms_urls" = 1/0
	//   "debug_uid" = <uid-string>     // Optional debug info to trace requests across FDB <--> KMS
	// }

	rapidjson::Document doc;
	doc.SetObject();

	// Append 'request version'
	addVersionToDoc(doc, SERVER_KNOBS->REST_KMS_CURRENT_BLOB_METADATA_REQUEST_VERSION);

	// Append 'blob_metadata_details' as json array
	addLatestDomainDetailsToDoc(doc, BLOB_METADATA_DETAILS_TAG, BLOB_METADATA_DOMAIN_ID_TAG, req.domainIds);

	// Append 'validation_tokens' as json array
	addValidationTokensSectionToJsonDoc(doc, ctx->validationTokenMap);

	// Append 'refresh_kms_urls'
	addRefreshKmsUrlsSectionToJsonDoc(doc, refreshKmsUrls);

	// Append 'debug_uid' section if needed
	addDebugUidSectionToJsonDoc(doc, req.debugId);

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);

	StringRef ref = makeString(sb.GetSize(), req.arena);
	memcpy(mutateString(ref), sb.GetString(), sb.GetSize());
	return ref;
}

// FIXME: add lookup error stats and suppress error trace events on interval
															#line 2695 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via fetchBlobMetadata()
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class FetchBlobMetadataActor>
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FetchBlobMetadataActorState {
															#line 2702 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FetchBlobMetadataActorState(Reference<RESTKmsConnectorCtx> const& ctx,KmsConnBlobMetadataReq const& req) 
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx),
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   req(req),
															#line 992 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   reply()
															#line 2713 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("fetchBlobMetadata", reinterpret_cast<unsigned long>(this));

	}
	~FetchBlobMetadataActorState() 
	{
		fdb_probe_actor_destroy("fetchBlobMetadata", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 994 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				bool refreshKmsUrls = shouldRefreshKmsUrls(ctx);
															#line 995 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StringRef requestBodyRef = getBlobMetadataRequestBody(ctx, req, refreshKmsUrls);
															#line 999 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				std::function<Standalone<VectorRef<BlobMetadataDetailsRef>>(Reference<RESTKmsConnectorCtx>, Reference<HTTP::IncomingResponse>)> f = &parseBlobMetadataResponse;
															#line 1002 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(reply.metadataDetails, kmsRequestImpl( ctx, SERVER_KNOBS->REST_KMS_CONNECTOR_GET_BLOB_METADATA_ENDPOINT, requestBodyRef, std::move(f)));
															#line 1002 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (static_cast<FetchBlobMetadataActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2737 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<FetchBlobMetadataActor*>(this)->actor_wait_state = 1;
															#line 1002 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FetchBlobMetadataActor, 0, Void >*>(static_cast<FetchBlobMetadataActor*>(this)));
															#line 2742 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FetchBlobMetadataActorState();
		static_cast<FetchBlobMetadataActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1014 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<FetchBlobMetadataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FetchBlobMetadataActorState(); static_cast<FetchBlobMetadataActor*>(this)->destroy(); return 0; }
															#line 2771 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<FetchBlobMetadataActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FetchBlobMetadataActorState();
		static_cast<FetchBlobMetadataActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1008 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent("RESTLookupBlobMetadataFailed", ctx->uid).error(e);
															#line 1009 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (!canReplyWith(e))
															#line 2786 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 1010 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 2790 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 1012 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			req.reply.sendError(e);
															#line 2794 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1006 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		req.reply.send(reply);
															#line 2809 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1006 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		req.reply.send(reply);
															#line 2818 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FetchBlobMetadataActor*>(this)->actor_wait_state > 0) static_cast<FetchBlobMetadataActor*>(this)->actor_wait_state = 0;
		static_cast<FetchBlobMetadataActor*>(this)->ActorCallback< FetchBlobMetadataActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FetchBlobMetadataActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("fetchBlobMetadata", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBlobMetadataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBlobMetadata", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FetchBlobMetadataActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("fetchBlobMetadata", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBlobMetadataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBlobMetadata", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FetchBlobMetadataActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("fetchBlobMetadata", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBlobMetadataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("fetchBlobMetadata", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	KmsConnBlobMetadataReq req;
															#line 992 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	KmsConnBlobMetadataRep reply;
															#line 2917 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via fetchBlobMetadata()
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FetchBlobMetadataActor final : public Actor<Void>, public ActorCallback< FetchBlobMetadataActor, 0, Void >, public FastAllocated<FetchBlobMetadataActor>, public FetchBlobMetadataActorState<FetchBlobMetadataActor> {
															#line 2922 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<FetchBlobMetadataActor>::operator new;
	using FastAllocated<FetchBlobMetadataActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13900643176445464832UL, 15374785656787749376UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FetchBlobMetadataActor, 0, Void >;
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FetchBlobMetadataActor(Reference<RESTKmsConnectorCtx> const& ctx,KmsConnBlobMetadataReq const& req) 
															#line 2939 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FetchBlobMetadataActorState<FetchBlobMetadataActor>(ctx, req),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("fetchBlobMetadata", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13510688306193718528UL, 7769973268822050048UL);
		ActorExecutionContextHelper __helper(static_cast<FetchBlobMetadataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("fetchBlobMetadata");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("fetchBlobMetadata", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FetchBlobMetadataActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> fetchBlobMetadata( Reference<RESTKmsConnectorCtx> const& ctx, KmsConnBlobMetadataReq const& req ) {
															#line 991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new FetchBlobMetadataActor(ctx, req));
															#line 2972 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1016 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 2977 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via procureValidationTokensFromFiles()
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class ProcureValidationTokensFromFilesActor>
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class ProcureValidationTokensFromFilesActorState {
															#line 2984 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	ProcureValidationTokensFromFilesActorState(Reference<RESTKmsConnectorCtx> const& ctx,std::string const& details) 
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx),
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   details(details)
															#line 2993 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this));

	}
	~ProcureValidationTokensFromFilesActorState() 
	{
		fdb_probe_actor_destroy("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1018 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			Standalone<StringRef> detailsRef(details);
															#line 1019 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (details.empty())
															#line 3010 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 1020 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				TraceEvent("RESTValidationTokenEmptyFileDetails", ctx->uid).log();
															#line 1021 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				return a_body1Catch1(encrypt_invalid_kms_config(), loopDepth);
															#line 3016 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 1024 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent("RESTValidationToken", ctx->uid).detail("DetailsStr", details);
															#line 1026 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			tokenFilePathMap = std::unordered_map<std::string, std::string>();
															#line 1027 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			for(;;) {
															#line 1028 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StringRef name = detailsRef.eat(TOKEN_NAME_FILE_SEP);
															#line 1029 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (name.empty())
															#line 3028 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
					break;
				}
															#line 1032 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StringRef path = detailsRef.eat(TOKEN_TUPLE_SEP);
															#line 1033 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (path.empty())
															#line 3036 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
															#line 1034 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					TraceEvent("RESTValidationTokenFileDetailsMalformed", ctx->uid).detail("FileDetails", details);
															#line 1035 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					return a_body1Catch1(operation_failed(), loopDepth);
															#line 3042 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				}
															#line 1038 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				std::string tokenName = boost::trim_copy(name.toString());
															#line 1039 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				std::string tokenFile = boost::trim_copy(path.toString());
															#line 1040 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (!fileExists(tokenFile))
															#line 3050 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
															#line 1041 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					TraceEvent("RESTValidationTokenFileNotFound", ctx->uid) .detail("TokenName", tokenName) .detail("Filename", tokenFile);
															#line 1044 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					return a_body1Catch1(encrypt_invalid_kms_config(), loopDepth);
															#line 3056 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				}
															#line 1047 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				tokenFilePathMap.emplace(tokenName, tokenFile);
															#line 1048 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				TraceEvent("RESTValidationToken", ctx->uid).detail("FName", tokenName).detail("Filename", tokenFile);
															#line 3062 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 1052 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ctx->validationTokenMap.clear();
															#line 1055 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			tokensPayloadSize = 0;
															#line 1056 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			RangeForbody1Iterator0 = std::begin(tokenFilePathMap);
															#line 3070 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ProcureValidationTokensFromFilesActorState();
		static_cast<ProcureValidationTokensFromFilesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1109 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<ProcureValidationTokensFromFilesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcureValidationTokensFromFilesActorState(); static_cast<ProcureValidationTokensFromFilesActor*>(this)->destroy(); return 0; }
															#line 3093 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<ProcureValidationTokensFromFilesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcureValidationTokensFromFilesActorState();
		static_cast<ProcureValidationTokensFromFilesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1056 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!(RangeForbody1Iterator0 != std::end(tokenFilePathMap)))
															#line 3112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1056 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		const auto& item = *RangeForbody1Iterator0;
															#line 3118 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
															#line 1057 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			tokenName = item.first;
															#line 1058 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			tokenFile = item.second;
															#line 1059 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			StrictFuture<Reference<IAsyncFile>> __when_expr_0 = IAsyncFileSystem::filesystem()->open( tokenFile, IAsyncFile::OPEN_NO_AIO | IAsyncFile::OPEN_READONLY | IAsyncFile::OPEN_UNCACHED, 0644);
															#line 1059 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3128 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state = 1;
															#line 1059 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ProcureValidationTokensFromFilesActor, 0, Reference<IAsyncFile> >*>(static_cast<ProcureValidationTokensFromFilesActor*>(this)));
															#line 3133 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 1056 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		++RangeForbody1Iterator0;
															#line 3156 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1062 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<int64_t> __when_expr_1 = tFile->size();
															#line 1062 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3167 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_1.get(), loopDepth); };
		static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state = 2;
															#line 1062 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ProcureValidationTokensFromFilesActor, 1, int64_t >*>(static_cast<ProcureValidationTokensFromFilesActor*>(this)));
															#line 3172 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Reference<IAsyncFile> const& __tFile,int loopDepth) 
	{
															#line 1059 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		tFile = __tFile;
															#line 3181 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Reference<IAsyncFile> && __tFile,int loopDepth) 
	{
		tFile = std::move(__tFile);
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state > 0) static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state = 0;
		static_cast<ProcureValidationTokensFromFilesActor*>(this)->ActorCallback< ProcureValidationTokensFromFilesActor, 0, Reference<IAsyncFile> >::remove();

	}
	void a_callback_fire(ActorCallback< ProcureValidationTokensFromFilesActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> const& value) 
	{
		fdb_probe_actor_enter("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensFromFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ProcureValidationTokensFromFilesActor, 0, Reference<IAsyncFile> >*,Reference<IAsyncFile> && value) 
	{
		fdb_probe_actor_enter("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensFromFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ProcureValidationTokensFromFilesActor, 0, Reference<IAsyncFile> >*,Error err) 
	{
		fdb_probe_actor_enter("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensFromFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1063 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (fSize > SERVER_KNOBS->REST_KMS_CONNECTOR_VALIDATION_TOKEN_MAX_SIZE)
															#line 3260 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
															#line 1064 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent(SevWarnAlways, "RESTValidationTokenFileTooLarge", ctx->uid) .detail("FileName", tokenFile) .detail("Size", fSize) .detail("MaxAllowedSize", SERVER_KNOBS->REST_KMS_CONNECTOR_VALIDATION_TOKEN_MAX_SIZE);
															#line 1068 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			return a_body1Catch1(file_too_large(), std::max(0, loopDepth - 1));
															#line 3266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 1071 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		tokensPayloadSize += fSize;
															#line 1072 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (tokensPayloadSize > SERVER_KNOBS->REST_KMS_CONNECTOR_VALIDATION_TOKENS_MAX_PAYLOAD_SIZE)
															#line 3272 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
															#line 1073 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent(SevWarnAlways, "RESTValidationTokenPayloadTooLarge", ctx->uid) .detail("MaxAllowedSize", SERVER_KNOBS->REST_KMS_CONNECTOR_VALIDATION_TOKENS_MAX_PAYLOAD_SIZE);
															#line 1075 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			return a_body1Catch1(value_too_large(), std::max(0, loopDepth - 1));
															#line 3278 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 1078 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		buff = makeString(fSize);
															#line 1079 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<int> __when_expr_2 = tFile->read(mutateString(buff), fSize, 0);
															#line 1079 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3286 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state = 3;
															#line 1079 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ProcureValidationTokensFromFilesActor, 2, int >*>(static_cast<ProcureValidationTokensFromFilesActor*>(this)));
															#line 3291 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(int64_t const& __fSize,int loopDepth) 
	{
															#line 1062 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		fSize = __fSize;
															#line 3300 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(int64_t && __fSize,int loopDepth) 
	{
		fSize = std::move(__fSize);
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state > 0) static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state = 0;
		static_cast<ProcureValidationTokensFromFilesActor*>(this)->ActorCallback< ProcureValidationTokensFromFilesActor, 1, int64_t >::remove();

	}
	void a_callback_fire(ActorCallback< ProcureValidationTokensFromFilesActor, 1, int64_t >*,int64_t const& value) 
	{
		fdb_probe_actor_enter("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensFromFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ProcureValidationTokensFromFilesActor, 1, int64_t >*,int64_t && value) 
	{
		fdb_probe_actor_enter("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensFromFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ProcureValidationTokensFromFilesActor, 1, int64_t >*,Error err) 
	{
		fdb_probe_actor_enter("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensFromFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont5(int const& bytesRead,int loopDepth) 
	{
															#line 1080 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (bytesRead != fSize)
															#line 3379 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
															#line 1081 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent(SevError, "RESTDiscoveryKmsUrlFileReadShort", ctx->uid) .detail("Filename", tokenFile) .detail("Expected", fSize) .detail("Actual", bytesRead);
															#line 1085 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			return a_body1Catch1(io_error(), std::max(0, loopDepth - 1));
															#line 3385 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 1089 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ValidationTokenCtx tokenCtx = ValidationTokenCtx(tokenName, ValidationTokenSource::VALIDATION_TOKEN_SOURCE_FILE);
															#line 1091 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		tokenCtx.value.resize(fSize);
															#line 1092 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		memcpy(tokenCtx.value.data(), buff.begin(), fSize);
															#line 1093 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		tokenCtx.filePath = tokenFile;
															#line 1095 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (SERVER_KNOBS->REST_KMS_CONNECTOR_REMOVE_TRAILING_NEWLINE)
															#line 3397 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
															#line 1096 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			removeTrailingChar(tokenCtx.value, '\n');
															#line 3401 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 1100 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		TraceEvent("RESTValidationTokenReadFile", ctx->uid) .detail("TokenName", tokenCtx.name) .detail("TokenSize", tokenCtx.value.size()) .detail("TokenFilePath", tokenCtx.filePath.get()) .detail("TotalPayloadSize", tokensPayloadSize);
															#line 1106 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ctx->validationTokenMap.emplace(tokenName, std::move(tokenCtx));
															#line 3407 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5(int && bytesRead,int loopDepth) 
	{
															#line 1080 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (bytesRead != fSize)
															#line 3416 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
															#line 1081 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent(SevError, "RESTDiscoveryKmsUrlFileReadShort", ctx->uid) .detail("Filename", tokenFile) .detail("Expected", fSize) .detail("Actual", bytesRead);
															#line 1085 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			return a_body1Catch1(io_error(), std::max(0, loopDepth - 1));
															#line 3422 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 1089 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ValidationTokenCtx tokenCtx = ValidationTokenCtx(tokenName, ValidationTokenSource::VALIDATION_TOKEN_SOURCE_FILE);
															#line 1091 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		tokenCtx.value.resize(fSize);
															#line 1092 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		memcpy(tokenCtx.value.data(), buff.begin(), fSize);
															#line 1093 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		tokenCtx.filePath = tokenFile;
															#line 1095 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (SERVER_KNOBS->REST_KMS_CONNECTOR_REMOVE_TRAILING_NEWLINE)
															#line 3434 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		{
															#line 1096 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			removeTrailingChar(tokenCtx.value, '\n');
															#line 3438 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 1100 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		TraceEvent("RESTValidationTokenReadFile", ctx->uid) .detail("TokenName", tokenCtx.name) .detail("TokenSize", tokenCtx.value.size()) .detail("TokenFilePath", tokenCtx.filePath.get()) .detail("TotalPayloadSize", tokensPayloadSize);
															#line 1106 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ctx->validationTokenMap.emplace(tokenName, std::move(tokenCtx));
															#line 3444 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(int const& bytesRead,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(bytesRead, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(int && bytesRead,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(bytesRead), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state > 0) static_cast<ProcureValidationTokensFromFilesActor*>(this)->actor_wait_state = 0;
		static_cast<ProcureValidationTokensFromFilesActor*>(this)->ActorCallback< ProcureValidationTokensFromFilesActor, 2, int >::remove();

	}
	void a_callback_fire(ActorCallback< ProcureValidationTokensFromFilesActor, 2, int >*,int const& value) 
	{
		fdb_probe_actor_enter("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensFromFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ProcureValidationTokensFromFilesActor, 2, int >*,int && value) 
	{
		fdb_probe_actor_enter("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensFromFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ProcureValidationTokensFromFilesActor, 2, int >*,Error err) 
	{
		fdb_probe_actor_enter("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensFromFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::string details;
															#line 1026 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::unordered_map<std::string, std::string> tokenFilePathMap;
															#line 1055 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	uint64_t tokensPayloadSize;
															#line 1056 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	decltype(std::begin(std::declval<std::unordered_map<std::string, std::string>>())) RangeForbody1Iterator0;
															#line 1057 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::string tokenName;
															#line 1058 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::string tokenFile;
															#line 1059 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<IAsyncFile> tFile;
															#line 1062 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	int64_t fSize;
															#line 1078 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Standalone<StringRef> buff;
															#line 3544 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via procureValidationTokensFromFiles()
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class ProcureValidationTokensFromFilesActor final : public Actor<Void>, public ActorCallback< ProcureValidationTokensFromFilesActor, 0, Reference<IAsyncFile> >, public ActorCallback< ProcureValidationTokensFromFilesActor, 1, int64_t >, public ActorCallback< ProcureValidationTokensFromFilesActor, 2, int >, public FastAllocated<ProcureValidationTokensFromFilesActor>, public ProcureValidationTokensFromFilesActorState<ProcureValidationTokensFromFilesActor> {
															#line 3549 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<ProcureValidationTokensFromFilesActor>::operator new;
	using FastAllocated<ProcureValidationTokensFromFilesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11525703641395275776UL, 5995667139568689408UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ProcureValidationTokensFromFilesActor, 0, Reference<IAsyncFile> >;
friend struct ActorCallback< ProcureValidationTokensFromFilesActor, 1, int64_t >;
friend struct ActorCallback< ProcureValidationTokensFromFilesActor, 2, int >;
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	ProcureValidationTokensFromFilesActor(Reference<RESTKmsConnectorCtx> const& ctx,std::string const& details) 
															#line 3568 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ProcureValidationTokensFromFilesActorState<ProcureValidationTokensFromFilesActor>(ctx, details),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6628245782674118912UL, 8963236149006903040UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensFromFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("procureValidationTokensFromFiles");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("procureValidationTokensFromFiles", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ProcureValidationTokensFromFilesActor, 0, Reference<IAsyncFile> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ProcureValidationTokensFromFilesActor, 1, int64_t >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ProcureValidationTokensFromFilesActor, 2, int >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> procureValidationTokensFromFiles( Reference<RESTKmsConnectorCtx> const& ctx, std::string const& details ) {
															#line 1017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new ProcureValidationTokensFromFilesActor(ctx, details));
															#line 3603 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1111 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 3608 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via procureValidationTokens()
															#line 1112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class ProcureValidationTokensActor>
															#line 1112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class ProcureValidationTokensActorState {
															#line 3615 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	ProcureValidationTokensActorState(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 1112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx)
															#line 3622 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("procureValidationTokens", reinterpret_cast<unsigned long>(this));

	}
	~ProcureValidationTokensActorState() 
	{
		fdb_probe_actor_destroy("procureValidationTokens", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1113 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string_view mode{ SERVER_KNOBS->REST_KMS_CONNECTOR_VALIDATION_TOKEN_MODE };
															#line 1115 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (mode.compare("file") == 0)
															#line 3639 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 1116 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StrictFuture<Void> __when_expr_0 = procureValidationTokensFromFiles(ctx, SERVER_KNOBS->REST_KMS_CONNECTOR_VALIDATION_TOKEN_DETAILS);
															#line 1116 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (static_cast<ProcureValidationTokensActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3645 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ProcureValidationTokensActor*>(this)->actor_wait_state = 1;
															#line 1116 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ProcureValidationTokensActor, 0, Void >*>(static_cast<ProcureValidationTokensActor*>(this)));
															#line 3650 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 1118 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				return a_body1Catch1(not_implemented(), loopDepth);
															#line 3657 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ProcureValidationTokensActorState();
		static_cast<ProcureValidationTokensActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1121 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<ProcureValidationTokensActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcureValidationTokensActorState(); static_cast<ProcureValidationTokensActor*>(this)->destroy(); return 0; }
															#line 3680 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<ProcureValidationTokensActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcureValidationTokensActorState();
		static_cast<ProcureValidationTokensActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ProcureValidationTokensActor*>(this)->actor_wait_state > 0) static_cast<ProcureValidationTokensActor*>(this)->actor_wait_state = 0;
		static_cast<ProcureValidationTokensActor*>(this)->ActorCallback< ProcureValidationTokensActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ProcureValidationTokensActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("procureValidationTokens", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokens", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ProcureValidationTokensActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("procureValidationTokens", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokens", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ProcureValidationTokensActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("procureValidationTokens", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("procureValidationTokens", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 3777 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via procureValidationTokens()
															#line 1112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class ProcureValidationTokensActor final : public Actor<Void>, public ActorCallback< ProcureValidationTokensActor, 0, Void >, public FastAllocated<ProcureValidationTokensActor>, public ProcureValidationTokensActorState<ProcureValidationTokensActor> {
															#line 3782 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<ProcureValidationTokensActor>::operator new;
	using FastAllocated<ProcureValidationTokensActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6887986609449435136UL, 7063693550560656128UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ProcureValidationTokensActor, 0, Void >;
															#line 1112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	ProcureValidationTokensActor(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 3799 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ProcureValidationTokensActorState<ProcureValidationTokensActor>(ctx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("procureValidationTokens", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10439913350516423680UL, 14049100379557911040UL);
		ActorExecutionContextHelper __helper(static_cast<ProcureValidationTokensActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("procureValidationTokens");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("procureValidationTokens", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ProcureValidationTokensActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> procureValidationTokens( Reference<RESTKmsConnectorCtx> const& ctx ) {
															#line 1112 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new ProcureValidationTokensActor(ctx));
															#line 3832 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1123 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

// Check if KMS is table by checking request failure count from RESTClient metrics.
// Will clear RESTClient metrics afterward, assuming it is the only user of the metrics.
//
// TODO(yiwu): make RESTClient periodically report and clear the stats.
void updateKMSStability(Reference<RESTKmsConnectorCtx> self) {
	bool stable = true;
	for (auto& s : self->restClient.statsMap) {
		if (s.second->requests_failed > 0) {
			stable = false;
		}
		s.second->clear();
	}
	self->kmsStable = stable;
}

Future<Void> getKMSState(Reference<RESTKmsConnectorCtx> self, KmsConnGetKMSStateReq req) {
	KmsConnGetKMSStateRep reply;
	reply.kmsStable = self->kmsStable;

	try {
		reply.restKMSUrls.reserve(reply.arena, self->kmsUrlStore.kmsUrls.size());
		for (const auto& url : self->kmsUrlStore.kmsUrls) {
			reply.restKMSUrls.emplace_back(reply.arena, url.toString());
		}
		req.reply.send(reply);
	} catch (Error& e) {
		TraceEvent(SevWarn, "RestKMSGetKMSStateFailed", self->uid).error(e);
		throw e;
	}

	return Void();
}

															#line 3870 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via restConnectorCoreImpl()
															#line 1157 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class RestConnectorCoreImplActor>
															#line 1157 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class RestConnectorCoreImplActorState {
															#line 3877 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1157 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	RestConnectorCoreImplActorState(KmsConnectorInterface const& interf) 
															#line 1157 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1157 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : interf(interf),
															#line 1158 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   self(makeReference<RESTKmsConnectorCtx>(interf.id())),
															#line 1159 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   collection(actorCollection(self->addActor.getFuture()))
															#line 3888 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this));

	}
	~RestConnectorCoreImplActorState() 
	{
		fdb_probe_actor_destroy("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1161 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			TraceEvent("RESTKmsConnectorInit", self->uid).log();
															#line 1163 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			self->kmsStabilityChecker = recurring([self = self]() { updateKMSStability(self); }, SERVER_KNOBS->REST_KMS_STABILITY_CHECK_INTERVAL);
															#line 1165 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			StrictFuture<Void> __when_expr_0 = discoverKmsUrls(self, RefreshPersistedUrls::False);
															#line 1165 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3909 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state = 1;
															#line 1165 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RestConnectorCoreImplActor, 0, Void >*>(static_cast<RestConnectorCoreImplActor*>(this)));
															#line 3914 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RestConnectorCoreImplActorState();
		static_cast<RestConnectorCoreImplActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1166 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_1 = procureValidationTokens(self);
															#line 1166 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3939 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state = 2;
															#line 1166 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RestConnectorCoreImplActor, 1, Void >*>(static_cast<RestConnectorCoreImplActor*>(this)));
															#line 3944 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1166 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_1 = procureValidationTokens(self);
															#line 1166 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3955 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state = 2;
															#line 1166 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RestConnectorCoreImplActor, 1, Void >*>(static_cast<RestConnectorCoreImplActor*>(this)));
															#line 3960 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state > 0) static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state = 0;
		static_cast<RestConnectorCoreImplActor*>(this)->ActorCallback< RestConnectorCoreImplActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RestConnectorCoreImplActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RestConnectorCoreImplActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RestConnectorCoreImplActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1168 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		;
															#line 4044 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1168 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		;
															#line 4053 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state > 0) static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state = 0;
		static_cast<RestConnectorCoreImplActor*>(this)->ActorCallback< RestConnectorCoreImplActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RestConnectorCoreImplActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RestConnectorCoreImplActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RestConnectorCoreImplActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 1170 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		auto __when_expr_2 = interf.ekLookupByIds.getFuture();
															#line 4144 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_2), FutureStream<KmsConnLookupEKsByKeyIdsReq>>::value || std::is_same<decltype(__when_expr_2), ThreadFutureStream<KmsConnLookupEKsByKeyIdsReq>>::value, "invalid type");
															#line 1169 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4148 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_2.pop(), loopDepth); };
															#line 1173 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		auto __when_expr_3 = interf.ekLookupByDomainIds.getFuture();
															#line 4152 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_3), FutureStream<KmsConnLookupEKsByDomainIdsReq>>::value || std::is_same<decltype(__when_expr_3), ThreadFutureStream<KmsConnLookupEKsByDomainIdsReq>>::value, "invalid type");
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when2(__when_expr_3.pop(), loopDepth); };
															#line 1176 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		auto __when_expr_4 = interf.blobMetadataReq.getFuture();
															#line 4157 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_4), FutureStream<KmsConnBlobMetadataReq>>::value || std::is_same<decltype(__when_expr_4), ThreadFutureStream<KmsConnBlobMetadataReq>>::value, "invalid type");
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when3(__when_expr_4.pop(), loopDepth); };
															#line 1179 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		auto __when_expr_5 = interf.getKMSStateReq.getFuture();
															#line 4162 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_5), FutureStream<KmsConnGetKMSStateReq>>::value || std::is_same<decltype(__when_expr_5), ThreadFutureStream<KmsConnGetKMSStateReq>>::value, "invalid type");
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when4(__when_expr_5.pop(), loopDepth); };
															#line 1182 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_6 = collection;
															#line 4167 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when5(__when_expr_6.get(), loopDepth); };
		static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state = 3;
															#line 1170 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorSingleCallback< RestConnectorCoreImplActor, 2, KmsConnLookupEKsByKeyIdsReq >*>(static_cast<RestConnectorCoreImplActor*>(this)));
															#line 1173 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorSingleCallback< RestConnectorCoreImplActor, 3, KmsConnLookupEKsByDomainIdsReq >*>(static_cast<RestConnectorCoreImplActor*>(this)));
															#line 1176 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorSingleCallback< RestConnectorCoreImplActor, 4, KmsConnBlobMetadataReq >*>(static_cast<RestConnectorCoreImplActor*>(this)));
															#line 1179 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorSingleCallback< RestConnectorCoreImplActor, 5, KmsConnGetKMSStateReq >*>(static_cast<RestConnectorCoreImplActor*>(this)));
															#line 1182 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< RestConnectorCoreImplActor, 6, Void >*>(static_cast<RestConnectorCoreImplActor*>(this)));
															#line 4180 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(KmsConnLookupEKsByKeyIdsReq const& req,int loopDepth) 
	{
															#line 1171 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		self->addActor.send(fetchEncryptionKeysByKeyIds(self, req));
															#line 4195 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(KmsConnLookupEKsByKeyIdsReq && req,int loopDepth) 
	{
															#line 1171 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		self->addActor.send(fetchEncryptionKeysByKeyIds(self, req));
															#line 4204 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when2(KmsConnLookupEKsByDomainIdsReq const& req,int loopDepth) 
	{
															#line 1174 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		self->addActor.send(fetchEncryptionKeysByDomainIds(self, req));
															#line 4213 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when2(KmsConnLookupEKsByDomainIdsReq && req,int loopDepth) 
	{
															#line 1174 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		self->addActor.send(fetchEncryptionKeysByDomainIds(self, req));
															#line 4222 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when3(KmsConnBlobMetadataReq const& req,int loopDepth) 
	{
															#line 1177 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		self->addActor.send(fetchBlobMetadata(self, req));
															#line 4231 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when3(KmsConnBlobMetadataReq && req,int loopDepth) 
	{
															#line 1177 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		self->addActor.send(fetchBlobMetadata(self, req));
															#line 4240 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when4(KmsConnGetKMSStateReq const& req,int loopDepth) 
	{
															#line 1180 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		self->addActor.send(getKMSState(self, req));
															#line 4249 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when4(KmsConnGetKMSStateReq && req,int loopDepth) 
	{
															#line 1180 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		self->addActor.send(getKMSState(self, req));
															#line 4258 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when5(Void const& _,int loopDepth) 
	{
															#line 1184 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 4267 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when5(Void && _,int loopDepth) 
	{
															#line 1184 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 4276 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state > 0) static_cast<RestConnectorCoreImplActor*>(this)->actor_wait_state = 0;
		static_cast<RestConnectorCoreImplActor*>(this)->ActorSingleCallback< RestConnectorCoreImplActor, 2, KmsConnLookupEKsByKeyIdsReq >::remove();
		static_cast<RestConnectorCoreImplActor*>(this)->ActorSingleCallback< RestConnectorCoreImplActor, 3, KmsConnLookupEKsByDomainIdsReq >::remove();
		static_cast<RestConnectorCoreImplActor*>(this)->ActorSingleCallback< RestConnectorCoreImplActor, 4, KmsConnBlobMetadataReq >::remove();
		static_cast<RestConnectorCoreImplActor*>(this)->ActorSingleCallback< RestConnectorCoreImplActor, 5, KmsConnGetKMSStateReq >::remove();
		static_cast<RestConnectorCoreImplActor*>(this)->ActorCallback< RestConnectorCoreImplActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorSingleCallback< RestConnectorCoreImplActor, 2, KmsConnLookupEKsByKeyIdsReq >*,KmsConnLookupEKsByKeyIdsReq const& value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< RestConnectorCoreImplActor, 2, KmsConnLookupEKsByKeyIdsReq >*,KmsConnLookupEKsByKeyIdsReq && value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorSingleCallback< RestConnectorCoreImplActor, 2, KmsConnLookupEKsByKeyIdsReq >*,Error err) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< RestConnectorCoreImplActor, 3, KmsConnLookupEKsByDomainIdsReq >*,KmsConnLookupEKsByDomainIdsReq const& value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< RestConnectorCoreImplActor, 3, KmsConnLookupEKsByDomainIdsReq >*,KmsConnLookupEKsByDomainIdsReq && value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorSingleCallback< RestConnectorCoreImplActor, 3, KmsConnLookupEKsByDomainIdsReq >*,Error err) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< RestConnectorCoreImplActor, 4, KmsConnBlobMetadataReq >*,KmsConnBlobMetadataReq const& value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorSingleCallback< RestConnectorCoreImplActor, 4, KmsConnBlobMetadataReq >*,KmsConnBlobMetadataReq && value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorSingleCallback< RestConnectorCoreImplActor, 4, KmsConnBlobMetadataReq >*,Error err) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorSingleCallback< RestConnectorCoreImplActor, 5, KmsConnGetKMSStateReq >*,KmsConnGetKMSStateReq const& value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorSingleCallback< RestConnectorCoreImplActor, 5, KmsConnGetKMSStateReq >*,KmsConnGetKMSStateReq && value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorSingleCallback< RestConnectorCoreImplActor, 5, KmsConnGetKMSStateReq >*,Error err) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< RestConnectorCoreImplActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< RestConnectorCoreImplActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< RestConnectorCoreImplActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), 6);

	}
															#line 1157 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	KmsConnectorInterface interf;
															#line 1158 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> self;
															#line 1159 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Future<Void> collection;
															#line 4582 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via restConnectorCoreImpl()
															#line 1157 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class RestConnectorCoreImplActor final : public Actor<Void>, public ActorCallback< RestConnectorCoreImplActor, 0, Void >, public ActorCallback< RestConnectorCoreImplActor, 1, Void >, public ActorSingleCallback< RestConnectorCoreImplActor, 2, KmsConnLookupEKsByKeyIdsReq >, public ActorSingleCallback< RestConnectorCoreImplActor, 3, KmsConnLookupEKsByDomainIdsReq >, public ActorSingleCallback< RestConnectorCoreImplActor, 4, KmsConnBlobMetadataReq >, public ActorSingleCallback< RestConnectorCoreImplActor, 5, KmsConnGetKMSStateReq >, public ActorCallback< RestConnectorCoreImplActor, 6, Void >, public FastAllocated<RestConnectorCoreImplActor>, public RestConnectorCoreImplActorState<RestConnectorCoreImplActor> {
															#line 4587 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<RestConnectorCoreImplActor>::operator new;
	using FastAllocated<RestConnectorCoreImplActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(153479892277957888UL, 5178394617738548736UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RestConnectorCoreImplActor, 0, Void >;
friend struct ActorCallback< RestConnectorCoreImplActor, 1, Void >;
friend struct ActorSingleCallback< RestConnectorCoreImplActor, 2, KmsConnLookupEKsByKeyIdsReq >;
friend struct ActorSingleCallback< RestConnectorCoreImplActor, 3, KmsConnLookupEKsByDomainIdsReq >;
friend struct ActorSingleCallback< RestConnectorCoreImplActor, 4, KmsConnBlobMetadataReq >;
friend struct ActorSingleCallback< RestConnectorCoreImplActor, 5, KmsConnGetKMSStateReq >;
friend struct ActorCallback< RestConnectorCoreImplActor, 6, Void >;
															#line 1157 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	RestConnectorCoreImplActor(KmsConnectorInterface const& interf) 
															#line 4610 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   RestConnectorCoreImplActorState<RestConnectorCoreImplActor>(interf),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17703871393385212672UL, 12195251967357033216UL);
		ActorExecutionContextHelper __helper(static_cast<RestConnectorCoreImplActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("restConnectorCoreImpl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("restConnectorCoreImpl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RestConnectorCoreImplActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RestConnectorCoreImplActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorSingleCallback< RestConnectorCoreImplActor, 2, KmsConnLookupEKsByKeyIdsReq >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1157 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> restConnectorCoreImpl( KmsConnectorInterface const& interf ) {
															#line 1157 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new RestConnectorCoreImplActor(interf));
															#line 4645 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1189 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

Future<Void> RESTKmsConnector::connectorCore(KmsConnectorInterface interf) {
	return restConnectorCoreImpl(interf);
}

// Only used to link unit tests
void forceLinkRESTKmsConnectorTest() {}

namespace {
std::string_view KMS_URL_NAME_TEST = "http://foo/bar";
std::string_view BLOB_METADATA_BASE_LOCATION_TEST = "file://local";
uint8_t BASE_CIPHER_KEY_TEST[32];

std::shared_ptr<platform::TmpFile> prepareTokenFile(const uint8_t* buff, const int len) {
	std::shared_ptr<platform::TmpFile> tmpFile = std::make_shared<platform::TmpFile>("/tmp");
	ASSERT(fileExists(tmpFile->getFileName()));
	tmpFile->write(buff, len);
	return tmpFile;
}

std::shared_ptr<platform::TmpFile> prepareTokenFile(const int tokenLen) {
	Standalone<StringRef> buff = makeString(tokenLen);
	deterministicRandom()->randomBytes(mutateString(buff), tokenLen);

	return prepareTokenFile(buff.begin(), tokenLen);
}

															#line 4676 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
// This generated class is to be used only via testEmptyValidationFileDetails()
															#line 1216 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class TestEmptyValidationFileDetailsActor>
															#line 1216 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestEmptyValidationFileDetailsActorState {
															#line 4682 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1216 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestEmptyValidationFileDetailsActorState(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 1216 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1216 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx)
															#line 4689 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("testEmptyValidationFileDetails", reinterpret_cast<unsigned long>(this));

	}
	~TestEmptyValidationFileDetailsActorState() 
	{
		fdb_probe_actor_destroy("testEmptyValidationFileDetails", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1218 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StrictFuture<Void> __when_expr_0 = procureValidationTokensFromFiles(ctx, "");
															#line 1218 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (static_cast<TestEmptyValidationFileDetailsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4707 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<TestEmptyValidationFileDetailsActor*>(this)->actor_wait_state = 1;
															#line 1218 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestEmptyValidationFileDetailsActor, 0, Void >*>(static_cast<TestEmptyValidationFileDetailsActor*>(this)));
															#line 4712 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestEmptyValidationFileDetailsActorState();
		static_cast<TestEmptyValidationFileDetailsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1223 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestEmptyValidationFileDetailsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestEmptyValidationFileDetailsActorState(); static_cast<TestEmptyValidationFileDetailsActor*>(this)->destroy(); return 0; }
															#line 4741 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestEmptyValidationFileDetailsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestEmptyValidationFileDetailsActorState();
		static_cast<TestEmptyValidationFileDetailsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1221 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(e.code(), error_code_encrypt_invalid_kms_config);
															#line 4754 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1219 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 4769 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1219 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 4778 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestEmptyValidationFileDetailsActor*>(this)->actor_wait_state > 0) static_cast<TestEmptyValidationFileDetailsActor*>(this)->actor_wait_state = 0;
		static_cast<TestEmptyValidationFileDetailsActor*>(this)->ActorCallback< TestEmptyValidationFileDetailsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestEmptyValidationFileDetailsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testEmptyValidationFileDetails", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestEmptyValidationFileDetailsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testEmptyValidationFileDetails", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestEmptyValidationFileDetailsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testEmptyValidationFileDetails", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestEmptyValidationFileDetailsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testEmptyValidationFileDetails", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestEmptyValidationFileDetailsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testEmptyValidationFileDetails", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestEmptyValidationFileDetailsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testEmptyValidationFileDetails", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1216 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 4873 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via testEmptyValidationFileDetails()
															#line 1216 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestEmptyValidationFileDetailsActor final : public Actor<Void>, public ActorCallback< TestEmptyValidationFileDetailsActor, 0, Void >, public FastAllocated<TestEmptyValidationFileDetailsActor>, public TestEmptyValidationFileDetailsActorState<TestEmptyValidationFileDetailsActor> {
															#line 4878 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<TestEmptyValidationFileDetailsActor>::operator new;
	using FastAllocated<TestEmptyValidationFileDetailsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14768139143134551296UL, 1353758061337604608UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestEmptyValidationFileDetailsActor, 0, Void >;
															#line 1216 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestEmptyValidationFileDetailsActor(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 4895 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   TestEmptyValidationFileDetailsActorState<TestEmptyValidationFileDetailsActor>(ctx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testEmptyValidationFileDetails", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8412797799100940032UL, 4130831369046660096UL);
		ActorExecutionContextHelper __helper(static_cast<TestEmptyValidationFileDetailsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testEmptyValidationFileDetails");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testEmptyValidationFileDetails", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestEmptyValidationFileDetailsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1216 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> testEmptyValidationFileDetails( Reference<RESTKmsConnectorCtx> const& ctx ) {
															#line 1216 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new TestEmptyValidationFileDetailsActor(ctx));
															#line 4927 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1225 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 4932 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
// This generated class is to be used only via testMalformedFileValidationTokenDetails()
															#line 1226 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class TestMalformedFileValidationTokenDetailsActor>
															#line 1226 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestMalformedFileValidationTokenDetailsActorState {
															#line 4938 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1226 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestMalformedFileValidationTokenDetailsActorState(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 1226 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1226 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx)
															#line 4945 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("testMalformedFileValidationTokenDetails", reinterpret_cast<unsigned long>(this));

	}
	~TestMalformedFileValidationTokenDetailsActorState() 
	{
		fdb_probe_actor_destroy("testMalformedFileValidationTokenDetails", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1228 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StrictFuture<Void> __when_expr_0 = procureValidationTokensFromFiles(ctx, "abdc/tmp/foo");
															#line 1228 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4963 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->actor_wait_state = 1;
															#line 1228 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestMalformedFileValidationTokenDetailsActor, 0, Void >*>(static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)));
															#line 4968 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestMalformedFileValidationTokenDetailsActorState();
		static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1234 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestMalformedFileValidationTokenDetailsActorState(); static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->destroy(); return 0; }
															#line 4997 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestMalformedFileValidationTokenDetailsActorState();
		static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1231 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(e.code(), error_code_operation_failed);
															#line 5010 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1229 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 5025 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1229 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 5034 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->actor_wait_state > 0) static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->actor_wait_state = 0;
		static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->ActorCallback< TestMalformedFileValidationTokenDetailsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestMalformedFileValidationTokenDetailsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testMalformedFileValidationTokenDetails", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testMalformedFileValidationTokenDetails", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestMalformedFileValidationTokenDetailsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testMalformedFileValidationTokenDetails", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testMalformedFileValidationTokenDetails", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestMalformedFileValidationTokenDetailsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testMalformedFileValidationTokenDetails", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testMalformedFileValidationTokenDetails", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1226 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 5129 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via testMalformedFileValidationTokenDetails()
															#line 1226 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestMalformedFileValidationTokenDetailsActor final : public Actor<Void>, public ActorCallback< TestMalformedFileValidationTokenDetailsActor, 0, Void >, public FastAllocated<TestMalformedFileValidationTokenDetailsActor>, public TestMalformedFileValidationTokenDetailsActorState<TestMalformedFileValidationTokenDetailsActor> {
															#line 5134 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<TestMalformedFileValidationTokenDetailsActor>::operator new;
	using FastAllocated<TestMalformedFileValidationTokenDetailsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4389483522365920256UL, 15098735887874684416UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestMalformedFileValidationTokenDetailsActor, 0, Void >;
															#line 1226 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestMalformedFileValidationTokenDetailsActor(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 5151 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   TestMalformedFileValidationTokenDetailsActorState<TestMalformedFileValidationTokenDetailsActor>(ctx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testMalformedFileValidationTokenDetails", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5545442567747869184UL, 11877485476858114816UL);
		ActorExecutionContextHelper __helper(static_cast<TestMalformedFileValidationTokenDetailsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testMalformedFileValidationTokenDetails");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testMalformedFileValidationTokenDetails", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestMalformedFileValidationTokenDetailsActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1226 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> testMalformedFileValidationTokenDetails( Reference<RESTKmsConnectorCtx> const& ctx ) {
															#line 1226 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new TestMalformedFileValidationTokenDetailsActor(ctx));
															#line 5183 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1236 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 5188 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
// This generated class is to be used only via testValidationTokenFileNotFound()
															#line 1237 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class TestValidationTokenFileNotFoundActor>
															#line 1237 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestValidationTokenFileNotFoundActorState {
															#line 5194 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1237 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestValidationTokenFileNotFoundActorState(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 1237 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1237 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx)
															#line 5201 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("testValidationTokenFileNotFound", reinterpret_cast<unsigned long>(this));

	}
	~TestValidationTokenFileNotFoundActorState() 
	{
		fdb_probe_actor_destroy("testValidationTokenFileNotFound", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1239 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StrictFuture<Void> __when_expr_0 = procureValidationTokensFromFiles(ctx, "foo$/imaginary-dir/dream/phantom-file");
															#line 1239 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (static_cast<TestValidationTokenFileNotFoundActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5219 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<TestValidationTokenFileNotFoundActor*>(this)->actor_wait_state = 1;
															#line 1239 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestValidationTokenFileNotFoundActor, 0, Void >*>(static_cast<TestValidationTokenFileNotFoundActor*>(this)));
															#line 5224 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestValidationTokenFileNotFoundActorState();
		static_cast<TestValidationTokenFileNotFoundActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1244 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestValidationTokenFileNotFoundActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestValidationTokenFileNotFoundActorState(); static_cast<TestValidationTokenFileNotFoundActor*>(this)->destroy(); return 0; }
															#line 5253 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestValidationTokenFileNotFoundActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestValidationTokenFileNotFoundActorState();
		static_cast<TestValidationTokenFileNotFoundActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1242 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(e.code(), error_code_encrypt_invalid_kms_config);
															#line 5266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1240 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 5281 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1240 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 5290 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestValidationTokenFileNotFoundActor*>(this)->actor_wait_state > 0) static_cast<TestValidationTokenFileNotFoundActor*>(this)->actor_wait_state = 0;
		static_cast<TestValidationTokenFileNotFoundActor*>(this)->ActorCallback< TestValidationTokenFileNotFoundActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestValidationTokenFileNotFoundActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testValidationTokenFileNotFound", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestValidationTokenFileNotFoundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testValidationTokenFileNotFound", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestValidationTokenFileNotFoundActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testValidationTokenFileNotFound", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestValidationTokenFileNotFoundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testValidationTokenFileNotFound", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestValidationTokenFileNotFoundActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testValidationTokenFileNotFound", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestValidationTokenFileNotFoundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testValidationTokenFileNotFound", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1237 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 5385 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via testValidationTokenFileNotFound()
															#line 1237 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestValidationTokenFileNotFoundActor final : public Actor<Void>, public ActorCallback< TestValidationTokenFileNotFoundActor, 0, Void >, public FastAllocated<TestValidationTokenFileNotFoundActor>, public TestValidationTokenFileNotFoundActorState<TestValidationTokenFileNotFoundActor> {
															#line 5390 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<TestValidationTokenFileNotFoundActor>::operator new;
	using FastAllocated<TestValidationTokenFileNotFoundActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3404527798032802816UL, 17048508712388149248UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestValidationTokenFileNotFoundActor, 0, Void >;
															#line 1237 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestValidationTokenFileNotFoundActor(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 5407 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   TestValidationTokenFileNotFoundActorState<TestValidationTokenFileNotFoundActor>(ctx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testValidationTokenFileNotFound", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10275298182887830528UL, 15376322865295475456UL);
		ActorExecutionContextHelper __helper(static_cast<TestValidationTokenFileNotFoundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testValidationTokenFileNotFound");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testValidationTokenFileNotFound", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestValidationTokenFileNotFoundActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1237 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> testValidationTokenFileNotFound( Reference<RESTKmsConnectorCtx> const& ctx ) {
															#line 1237 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new TestValidationTokenFileNotFoundActor(ctx));
															#line 5439 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1246 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 5444 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
// This generated class is to be used only via testTooLargeValidationTokenFile()
															#line 1247 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class TestTooLargeValidationTokenFileActor>
															#line 1247 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestTooLargeValidationTokenFileActorState {
															#line 5450 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1247 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestTooLargeValidationTokenFileActorState(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 1247 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1247 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx)
															#line 5457 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("testTooLargeValidationTokenFile", reinterpret_cast<unsigned long>(this));

	}
	~TestTooLargeValidationTokenFileActorState() 
	{
		fdb_probe_actor_destroy("testTooLargeValidationTokenFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1248 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string name("foo");
															#line 1249 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			const int tokenLen = SERVER_KNOBS->REST_KMS_CONNECTOR_VALIDATION_TOKEN_MAX_SIZE + 1;
															#line 1251 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			tmpFile = prepareTokenFile(tokenLen);
															#line 1253 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string details;
															#line 1254 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			details.append(name).append(TOKEN_NAME_FILE_SEP).append(tmpFile->getFileName());
															#line 5480 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			try {
															#line 1257 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StrictFuture<Void> __when_expr_0 = procureValidationTokensFromFiles(ctx, details);
															#line 1257 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (static_cast<TestTooLargeValidationTokenFileActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5486 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<TestTooLargeValidationTokenFileActor*>(this)->actor_wait_state = 1;
															#line 1257 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestTooLargeValidationTokenFileActor, 0, Void >*>(static_cast<TestTooLargeValidationTokenFileActor*>(this)));
															#line 5491 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestTooLargeValidationTokenFileActorState();
		static_cast<TestTooLargeValidationTokenFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1263 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestTooLargeValidationTokenFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestTooLargeValidationTokenFileActorState(); static_cast<TestTooLargeValidationTokenFileActor*>(this)->destroy(); return 0; }
															#line 5520 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestTooLargeValidationTokenFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestTooLargeValidationTokenFileActorState();
		static_cast<TestTooLargeValidationTokenFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1260 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(e.code(), error_code_file_too_large);
															#line 5533 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1258 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 5548 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1258 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 5557 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestTooLargeValidationTokenFileActor*>(this)->actor_wait_state > 0) static_cast<TestTooLargeValidationTokenFileActor*>(this)->actor_wait_state = 0;
		static_cast<TestTooLargeValidationTokenFileActor*>(this)->ActorCallback< TestTooLargeValidationTokenFileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestTooLargeValidationTokenFileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testTooLargeValidationTokenFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestTooLargeValidationTokenFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTooLargeValidationTokenFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestTooLargeValidationTokenFileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testTooLargeValidationTokenFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestTooLargeValidationTokenFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTooLargeValidationTokenFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestTooLargeValidationTokenFileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testTooLargeValidationTokenFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestTooLargeValidationTokenFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testTooLargeValidationTokenFile", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1247 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 1251 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::shared_ptr<platform::TmpFile> tmpFile;
															#line 5654 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via testTooLargeValidationTokenFile()
															#line 1247 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestTooLargeValidationTokenFileActor final : public Actor<Void>, public ActorCallback< TestTooLargeValidationTokenFileActor, 0, Void >, public FastAllocated<TestTooLargeValidationTokenFileActor>, public TestTooLargeValidationTokenFileActorState<TestTooLargeValidationTokenFileActor> {
															#line 5659 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<TestTooLargeValidationTokenFileActor>::operator new;
	using FastAllocated<TestTooLargeValidationTokenFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11873802240775632640UL, 12326995590886932992UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestTooLargeValidationTokenFileActor, 0, Void >;
															#line 1247 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestTooLargeValidationTokenFileActor(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 5676 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   TestTooLargeValidationTokenFileActorState<TestTooLargeValidationTokenFileActor>(ctx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testTooLargeValidationTokenFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(117888384434193664UL, 13954494296919063296UL);
		ActorExecutionContextHelper __helper(static_cast<TestTooLargeValidationTokenFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testTooLargeValidationTokenFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testTooLargeValidationTokenFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestTooLargeValidationTokenFileActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1247 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> testTooLargeValidationTokenFile( Reference<RESTKmsConnectorCtx> const& ctx ) {
															#line 1247 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new TestTooLargeValidationTokenFileActor(ctx));
															#line 5708 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1265 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 5713 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
// This generated class is to be used only via testValidationFileTokenPayloadTooLarge()
															#line 1266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class TestValidationFileTokenPayloadTooLargeActor>
															#line 1266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestValidationFileTokenPayloadTooLargeActorState {
															#line 5719 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestValidationFileTokenPayloadTooLargeActorState(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 1266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx)
															#line 5726 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("testValidationFileTokenPayloadTooLarge", reinterpret_cast<unsigned long>(this));

	}
	~TestValidationFileTokenPayloadTooLargeActorState() 
	{
		fdb_probe_actor_destroy("testValidationFileTokenPayloadTooLarge", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1267 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			const int tokenLen = SERVER_KNOBS->REST_KMS_CONNECTOR_VALIDATION_TOKEN_MAX_SIZE;
															#line 1268 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			const int nTokens = SERVER_KNOBS->REST_KMS_CONNECTOR_VALIDATION_TOKENS_MAX_PAYLOAD_SIZE / SERVER_KNOBS->REST_KMS_CONNECTOR_VALIDATION_TOKEN_MAX_SIZE + 2;
															#line 1271 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			Standalone<StringRef> buff = makeString(tokenLen);
															#line 1272 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			deterministicRandom()->randomBytes(mutateString(buff), tokenLen);
															#line 1274 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string details;
															#line 1275 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			tokenfiles = std::vector<std::shared_ptr<platform::TmpFile>>();
															#line 1276 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			for(int i = 0;i < nTokens;i++) {
															#line 1277 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				std::shared_ptr<platform::TmpFile> tokenfile = prepareTokenFile(buff.begin(), tokenLen);
															#line 1279 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				details.append(std::to_string(i)).append(TOKEN_NAME_FILE_SEP).append(tokenfile->getFileName());
															#line 1280 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (i < nTokens)
															#line 5759 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
															#line 1281 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					details.append(TOKEN_TUPLE_SEP);
															#line 5763 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				}
															#line 1282 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				tokenfiles.emplace_back(tokenfile);
															#line 5767 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
			try {
															#line 1286 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StrictFuture<Void> __when_expr_0 = procureValidationTokensFromFiles(ctx, details);
															#line 1286 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 5774 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->actor_wait_state = 1;
															#line 1286 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestValidationFileTokenPayloadTooLargeActor, 0, Void >*>(static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)));
															#line 5779 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestValidationFileTokenPayloadTooLargeActorState();
		static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1292 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestValidationFileTokenPayloadTooLargeActorState(); static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->destroy(); return 0; }
															#line 5808 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestValidationFileTokenPayloadTooLargeActorState();
		static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1289 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(e.code(), error_code_value_too_large);
															#line 5821 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 1287 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 5836 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 1287 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 5845 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->actor_wait_state > 0) static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->actor_wait_state = 0;
		static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->ActorCallback< TestValidationFileTokenPayloadTooLargeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestValidationFileTokenPayloadTooLargeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testValidationFileTokenPayloadTooLarge", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testValidationFileTokenPayloadTooLarge", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestValidationFileTokenPayloadTooLargeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testValidationFileTokenPayloadTooLarge", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testValidationFileTokenPayloadTooLarge", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestValidationFileTokenPayloadTooLargeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testValidationFileTokenPayloadTooLarge", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testValidationFileTokenPayloadTooLarge", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 1275 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::vector<std::shared_ptr<platform::TmpFile>> tokenfiles;
															#line 5942 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via testValidationFileTokenPayloadTooLarge()
															#line 1266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestValidationFileTokenPayloadTooLargeActor final : public Actor<Void>, public ActorCallback< TestValidationFileTokenPayloadTooLargeActor, 0, Void >, public FastAllocated<TestValidationFileTokenPayloadTooLargeActor>, public TestValidationFileTokenPayloadTooLargeActorState<TestValidationFileTokenPayloadTooLargeActor> {
															#line 5947 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<TestValidationFileTokenPayloadTooLargeActor>::operator new;
	using FastAllocated<TestValidationFileTokenPayloadTooLargeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13974956970755965696UL, 16175310828935339008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestValidationFileTokenPayloadTooLargeActor, 0, Void >;
															#line 1266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestValidationFileTokenPayloadTooLargeActor(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 5964 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   TestValidationFileTokenPayloadTooLargeActorState<TestValidationFileTokenPayloadTooLargeActor>(ctx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testValidationFileTokenPayloadTooLarge", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9599145780086675200UL, 605867831632008704UL);
		ActorExecutionContextHelper __helper(static_cast<TestValidationFileTokenPayloadTooLargeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testValidationFileTokenPayloadTooLarge");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testValidationFileTokenPayloadTooLarge", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestValidationFileTokenPayloadTooLargeActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> testValidationFileTokenPayloadTooLarge( Reference<RESTKmsConnectorCtx> const& ctx ) {
															#line 1266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new TestValidationFileTokenPayloadTooLargeActor(ctx));
															#line 5996 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1294 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 6001 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
// This generated class is to be used only via testMultiValidationFileTokenFiles()
															#line 1295 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class TestMultiValidationFileTokenFilesActor>
															#line 1295 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestMultiValidationFileTokenFilesActorState {
															#line 6007 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1295 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestMultiValidationFileTokenFilesActorState(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 1295 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1295 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx),
															#line 1296 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   numFiles(deterministicRandom()->randomInt(2, 5)),
															#line 1297 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   tokenLen(deterministicRandom()->randomInt(26, 75)),
															#line 1298 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   buff(makeString(tokenLen)),
															#line 1299 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   tokenFiles(),
															#line 1300 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   tokenNameValueMap(),
															#line 1301 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   tokenDetailsStr(),
															#line 1302 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   newLineAppended(BUGGIFY ? true : false)
															#line 6028 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("testMultiValidationFileTokenFiles", reinterpret_cast<unsigned long>(this));

	}
	~TestMultiValidationFileTokenFilesActorState() 
	{
		fdb_probe_actor_destroy("testMultiValidationFileTokenFiles", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1304 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string token;
															#line 1306 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			for(;;) {
															#line 1307 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				deterministicRandom()->randomBytes(mutateString(buff), tokenLen);
															#line 1308 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				token = std::string((char*)buff.begin(), tokenLen);
															#line 1309 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				removeTrailingChar(token, '\n');
															#line 1310 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (token.size() > 0)
															#line 6053 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
					break;
				}
			}
															#line 1314 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			tokenLen = token.size();
															#line 1315 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string tokenWithNewLine(token);
															#line 1316 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			tokenWithNewLine.push_back('\n');
															#line 1318 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			for(int i = 1;i <= numFiles;i++) {
															#line 1319 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				std::string tokenName = std::to_string(i);
															#line 1320 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				std::shared_ptr<platform::TmpFile> tokenfile = newLineAppended ? prepareTokenFile(reinterpret_cast<uint8_t*>(tokenWithNewLine.data()), tokenLen + 1) : prepareTokenFile(reinterpret_cast<uint8_t*>(token.data()), tokenLen);
															#line 1324 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				tokenFiles.emplace(tokenName, tokenfile);
															#line 1325 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				tokenDetailsStr.append(tokenName).append(TOKEN_NAME_FILE_SEP).append(tokenfile->getFileName());
															#line 1326 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (i < numFiles)
															#line 6076 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
															#line 1327 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					tokenDetailsStr.append(TOKEN_TUPLE_SEP);
															#line 6080 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				}
															#line 1329 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				tokenNameValueMap.emplace(std::to_string(i), token);
															#line 6084 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 1332 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			StrictFuture<Void> __when_expr_0 = procureValidationTokensFromFiles(ctx, tokenDetailsStr);
															#line 1332 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (static_cast<TestMultiValidationFileTokenFilesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 6090 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestMultiValidationFileTokenFilesActor*>(this)->actor_wait_state = 1;
															#line 1332 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestMultiValidationFileTokenFilesActor, 0, Void >*>(static_cast<TestMultiValidationFileTokenFilesActor*>(this)));
															#line 6095 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestMultiValidationFileTokenFilesActorState();
		static_cast<TestMultiValidationFileTokenFilesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1334 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT_EQ(ctx->validationTokenMap.size(), tokenNameValueMap.size());
															#line 1336 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		for( const auto& token : ctx->validationTokenMap ) {
															#line 1337 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			const auto& itr = tokenNameValueMap.find(token.first);
															#line 1338 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			const ValidationTokenCtx& tokenCtx = token.second;
															#line 1340 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT(itr != tokenNameValueMap.end());
															#line 1341 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(token.first.compare(itr->first), 0);
															#line 1342 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(tokenCtx.value.compare(itr->second), 0);
															#line 1343 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(tokenCtx.source, ValidationTokenSource::VALIDATION_TOKEN_SOURCE_FILE);
															#line 1344 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT(tokenCtx.filePath.present());
															#line 1345 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(tokenCtx.filePath.compare(tokenFiles[tokenCtx.name]->getFileName()), 0);
															#line 1346 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_NE(tokenCtx.getReadTS(), 0);
															#line 6138 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 1349 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		CODE_PROBE(newLineAppended, "RESTKmsConnector remove trailing newline");
															#line 1351 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestMultiValidationFileTokenFilesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestMultiValidationFileTokenFilesActorState(); static_cast<TestMultiValidationFileTokenFilesActor*>(this)->destroy(); return 0; }
															#line 6144 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestMultiValidationFileTokenFilesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestMultiValidationFileTokenFilesActorState();
		static_cast<TestMultiValidationFileTokenFilesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1334 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT_EQ(ctx->validationTokenMap.size(), tokenNameValueMap.size());
															#line 1336 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		for( const auto& token : ctx->validationTokenMap ) {
															#line 1337 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			const auto& itr = tokenNameValueMap.find(token.first);
															#line 1338 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			const ValidationTokenCtx& tokenCtx = token.second;
															#line 1340 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT(itr != tokenNameValueMap.end());
															#line 1341 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(token.first.compare(itr->first), 0);
															#line 1342 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(tokenCtx.value.compare(itr->second), 0);
															#line 1343 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(tokenCtx.source, ValidationTokenSource::VALIDATION_TOKEN_SOURCE_FILE);
															#line 1344 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT(tokenCtx.filePath.present());
															#line 1345 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(tokenCtx.filePath.compare(tokenFiles[tokenCtx.name]->getFileName()), 0);
															#line 1346 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_NE(tokenCtx.getReadTS(), 0);
															#line 6176 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 1349 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		CODE_PROBE(newLineAppended, "RESTKmsConnector remove trailing newline");
															#line 1351 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestMultiValidationFileTokenFilesActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestMultiValidationFileTokenFilesActorState(); static_cast<TestMultiValidationFileTokenFilesActor*>(this)->destroy(); return 0; }
															#line 6182 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestMultiValidationFileTokenFilesActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestMultiValidationFileTokenFilesActorState();
		static_cast<TestMultiValidationFileTokenFilesActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestMultiValidationFileTokenFilesActor*>(this)->actor_wait_state > 0) static_cast<TestMultiValidationFileTokenFilesActor*>(this)->actor_wait_state = 0;
		static_cast<TestMultiValidationFileTokenFilesActor*>(this)->ActorCallback< TestMultiValidationFileTokenFilesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestMultiValidationFileTokenFilesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testMultiValidationFileTokenFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestMultiValidationFileTokenFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testMultiValidationFileTokenFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestMultiValidationFileTokenFilesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testMultiValidationFileTokenFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestMultiValidationFileTokenFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testMultiValidationFileTokenFiles", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestMultiValidationFileTokenFilesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testMultiValidationFileTokenFiles", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestMultiValidationFileTokenFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testMultiValidationFileTokenFiles", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1295 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 1296 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	int numFiles;
															#line 1297 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	int tokenLen;
															#line 1298 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Standalone<StringRef> buff;
															#line 1299 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::unordered_map<std::string, std::shared_ptr<platform::TmpFile>> tokenFiles;
															#line 1300 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::unordered_map<std::string, std::string> tokenNameValueMap;
															#line 1301 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::string tokenDetailsStr;
															#line 1302 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	bool newLineAppended;
															#line 6281 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via testMultiValidationFileTokenFiles()
															#line 1295 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestMultiValidationFileTokenFilesActor final : public Actor<Void>, public ActorCallback< TestMultiValidationFileTokenFilesActor, 0, Void >, public FastAllocated<TestMultiValidationFileTokenFilesActor>, public TestMultiValidationFileTokenFilesActorState<TestMultiValidationFileTokenFilesActor> {
															#line 6286 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<TestMultiValidationFileTokenFilesActor>::operator new;
	using FastAllocated<TestMultiValidationFileTokenFilesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(18163210517267035648UL, 13608531710812694528UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestMultiValidationFileTokenFilesActor, 0, Void >;
															#line 1295 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestMultiValidationFileTokenFilesActor(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 6303 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   TestMultiValidationFileTokenFilesActorState<TestMultiValidationFileTokenFilesActor>(ctx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testMultiValidationFileTokenFiles", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8787259583700750336UL, 18323398735666912768UL);
		ActorExecutionContextHelper __helper(static_cast<TestMultiValidationFileTokenFilesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testMultiValidationFileTokenFiles");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testMultiValidationFileTokenFiles", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestMultiValidationFileTokenFilesActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1295 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> testMultiValidationFileTokenFiles( Reference<RESTKmsConnectorCtx> const& ctx ) {
															#line 1295 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new TestMultiValidationFileTokenFilesActor(ctx));
															#line 6335 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1353 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

EncryptCipherDomainId getRandomDomainId() {
	const int lottery = deterministicRandom()->randomInt(0, 100);
	if (lottery < 10) {
		return SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID;
	} else if (lottery >= 10 && lottery < 25) {
		return ENCRYPT_HEADER_DOMAIN_ID;
	} else {
		return lottery;
	}
}

void addFakeRefreshExpire(rapidjson::Document& resDoc, rapidjson::Value& detail, rapidjson::Value& key) {
	if (deterministicRandom()->coinflip()) {
		key.SetString(REFRESH_AFTER_SEC, resDoc.GetAllocator());
		rapidjson::Value refreshInterval;
		refreshInterval.SetInt64(10);
		detail.AddMember(key, refreshInterval, resDoc.GetAllocator());
	}
	if (deterministicRandom()->coinflip()) {
		key.SetString(EXPIRE_AFTER_SEC, resDoc.GetAllocator());
		rapidjson::Value expireInterval;
		deterministicRandom()->coinflip() ? expireInterval.SetInt64(10) : expireInterval.SetInt64(-1);
		detail.AddMember(key, expireInterval, resDoc.GetAllocator());
	}
}

void addFakeKmsUrls(const rapidjson::Document& reqDoc, rapidjson::Document& resDoc) {
	ASSERT(reqDoc.HasMember(REFRESH_KMS_URLS_TAG));
	if (reqDoc[REFRESH_KMS_URLS_TAG].GetBool()) {
		rapidjson::Value kmsUrls(rapidjson::kArrayType);
		for (int i = 0; i < 3; i++) {
			rapidjson::Value url;
			url.SetString(KMS_URL_NAME_TEST.data(), resDoc.GetAllocator());
			kmsUrls.PushBack(url, resDoc.GetAllocator());
		}
		rapidjson::Value memberKey(KMS_URLS_TAG, resDoc.GetAllocator());
		resDoc.AddMember(memberKey, kmsUrls, resDoc.GetAllocator());
	}
}

void getFakeEncryptCipherResponse(StringRef jsonReqRef,
                                  const bool baseCipherIdPresent,
                                  Reference<HTTP::IncomingResponse> httpResponse) {
	rapidjson::Document reqDoc;
	reqDoc.Parse(jsonReqRef.toString().data());

	rapidjson::Document resDoc;
	resDoc.SetObject();

	ASSERT(reqDoc.HasMember(REQUEST_VERSION_TAG) && reqDoc[REQUEST_VERSION_TAG].IsInt());
	ASSERT(reqDoc.HasMember(CIPHER_KEY_DETAILS_TAG) && reqDoc[CIPHER_KEY_DETAILS_TAG].IsArray());

	addVersionToDoc(resDoc, reqDoc[REQUEST_VERSION_TAG].GetInt());

	rapidjson::Value cipherKeyDetails(rapidjson::kArrayType);
	for (const auto& detail : reqDoc[CIPHER_KEY_DETAILS_TAG].GetArray()) {
		// Fix: Initialize JSON value as object type to prevent undefined behavior
		rapidjson::Value keyDetail(rapidjson::kObjectType);

		ASSERT(detail.HasMember(ENCRYPT_DOMAIN_ID_TAG));

		rapidjson::Value key(ENCRYPT_DOMAIN_ID_TAG, resDoc.GetAllocator());
		rapidjson::Value domainId;
		domainId.SetInt64(detail[ENCRYPT_DOMAIN_ID_TAG].GetInt64());
		keyDetail.AddMember(key, domainId, resDoc.GetAllocator());

		key.SetString(BASE_CIPHER_ID_TAG, resDoc.GetAllocator());
		rapidjson::Value baseCipherId;
		if (detail.HasMember(BASE_CIPHER_ID_TAG)) {
			domainId.SetUint64(detail[BASE_CIPHER_ID_TAG].GetUint64());
		} else {
			ASSERT(!baseCipherIdPresent);
			domainId.SetUint(1234);
		}
		keyDetail.AddMember(key, domainId, resDoc.GetAllocator());

		key.SetString(BASE_CIPHER_TAG, resDoc.GetAllocator());
		rapidjson::Value baseCipher;
		baseCipher.SetString((char*)&BASE_CIPHER_KEY_TEST[0], sizeof(BASE_CIPHER_KEY_TEST), resDoc.GetAllocator());
		keyDetail.AddMember(key, baseCipher, resDoc.GetAllocator());

		addFakeRefreshExpire(resDoc, keyDetail, key);

		cipherKeyDetails.PushBack(keyDetail, resDoc.GetAllocator());
	}
	rapidjson::Value memberKey(CIPHER_KEY_DETAILS_TAG, resDoc.GetAllocator());
	resDoc.AddMember(memberKey, cipherKeyDetails, resDoc.GetAllocator());

	addFakeKmsUrls(reqDoc, resDoc);

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	resDoc.Accept(writer);
	httpResponse->data.content.resize(sb.GetSize(), '\0');
	memcpy(httpResponse->data.content.data(), sb.GetString(), sb.GetSize());
	httpResponse->data.contentLen = sb.GetSize();
}

void getFakeBlobMetadataResponse(StringRef jsonReqRef,
                                 const bool baseCipherIdPresent,
                                 Reference<HTTP::IncomingResponse> httpResponse) {
	rapidjson::Document reqDoc;
	reqDoc.Parse(jsonReqRef.toString().data());

	rapidjson::Document resDoc;
	resDoc.SetObject();

	ASSERT(reqDoc.HasMember(REQUEST_VERSION_TAG) && reqDoc[REQUEST_VERSION_TAG].IsInt());
	ASSERT(reqDoc.HasMember(BLOB_METADATA_DETAILS_TAG) && reqDoc[BLOB_METADATA_DETAILS_TAG].IsArray());

	addVersionToDoc(resDoc, reqDoc[REQUEST_VERSION_TAG].GetInt());

	rapidjson::Value blobMetadataDetails(rapidjson::kArrayType);
	for (const auto& detail : reqDoc[BLOB_METADATA_DETAILS_TAG].GetArray()) {
		rapidjson::Value keyDetail(rapidjson::kObjectType);

		ASSERT(detail.HasMember(BLOB_METADATA_DOMAIN_ID_TAG));

		rapidjson::Value key(BLOB_METADATA_DOMAIN_ID_TAG, resDoc.GetAllocator());
		rapidjson::Value domainId;
		domainId.SetInt64(detail[BLOB_METADATA_DOMAIN_ID_TAG].GetInt64());
		keyDetail.AddMember(key, domainId, resDoc.GetAllocator());

		int locationCount = deterministicRandom()->randomInt(1, 6);
		rapidjson::Value locations(rapidjson::kArrayType);
		for (int i = 0; i < locationCount; i++) {
			rapidjson::Value location(rapidjson::kObjectType);

			rapidjson::Value locId;
			key.SetString(BLOB_METADATA_LOCATION_ID_TAG, resDoc.GetAllocator());
			locId.SetInt64(i);
			location.AddMember(key, locId, resDoc.GetAllocator());

			rapidjson::Value path;
			key.SetString(BLOB_METADATA_LOCATION_PATH_TAG, resDoc.GetAllocator());
			path.SetString(BLOB_METADATA_BASE_LOCATION_TEST.data(), resDoc.GetAllocator());
			location.AddMember(key, path, resDoc.GetAllocator());

			locations.PushBack(location, resDoc.GetAllocator());
		}

		key.SetString(BLOB_METADATA_LOCATIONS_TAG, resDoc.GetAllocator());
		keyDetail.AddMember(key, locations, resDoc.GetAllocator());

		addFakeRefreshExpire(resDoc, keyDetail, key);

		blobMetadataDetails.PushBack(keyDetail, resDoc.GetAllocator());
	}
	rapidjson::Value memberKey(BLOB_METADATA_DETAILS_TAG, resDoc.GetAllocator());
	resDoc.AddMember(memberKey, blobMetadataDetails, resDoc.GetAllocator());

	addFakeKmsUrls(reqDoc, resDoc);

	// Serialize json to string
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	resDoc.Accept(writer);
	httpResponse->data.content.resize(sb.GetSize(), '\0');
	memcpy(httpResponse->data.content.data(), sb.GetString(), sb.GetSize());
}

void validateKmsUrls(Reference<RESTKmsConnectorCtx> ctx) {
	ASSERT_EQ(ctx->kmsUrlStore.kmsUrls.size(), 3);
	ASSERT_EQ(ctx->kmsUrlStore.kmsUrls[0].url.compare(KMS_URL_NAME_TEST), 0);
}

void testGetEncryptKeysByKeyIdsRequestBody(Reference<RESTKmsConnectorCtx> ctx, Arena& arena) {
	KmsConnLookupEKsByKeyIdsReq req;
	std::unordered_map<EncryptCipherBaseKeyId, EncryptCipherDomainId> keyMap;
	const int nKeys = deterministicRandom()->randomInt(7, 8);
	for (int i = 1; i < nKeys; i++) {
		EncryptCipherDomainId domainId = getRandomDomainId();
		req.encryptKeyInfos.emplace_back(domainId, i);
		keyMap[i] = domainId;
	}

	bool refreshKmsUrls = deterministicRandom()->coinflip();
	if (deterministicRandom()->coinflip()) {
		req.debugId = deterministicRandom()->randomUniqueID();
	}

	StringRef requestBodyRef = getEncryptKeysByKeyIdsRequestBody(ctx, req, refreshKmsUrls, arena);
	TraceEvent("FetchKeysByKeyIds", ctx->uid).setMaxFieldLength(100000).detail("JsonReqStr", requestBodyRef.toString());
	Reference<HTTP::IncomingResponse> httpResp = makeReference<HTTP::IncomingResponse>();
	httpResp->code = HTTP::HTTP_STATUS_CODE_OK;
	getFakeEncryptCipherResponse(requestBodyRef, true, httpResp);
	TraceEvent("FetchKeysByKeyIds", ctx->uid).setMaxFieldLength(100000).detail("HttpRespStr", httpResp->data.content);

	Standalone<VectorRef<EncryptCipherKeyDetailsRef>> cipherDetails = parseEncryptCipherResponse(ctx, httpResp);
	ASSERT_EQ(cipherDetails.size(), keyMap.size());
	for (const auto& detail : cipherDetails) {
		ASSERT(keyMap.find(detail.encryptKeyId) != keyMap.end());
		ASSERT_EQ(keyMap[detail.encryptKeyId], detail.encryptDomainId);
		ASSERT_EQ(detail.encryptKey.size(), sizeof(BASE_CIPHER_KEY_TEST));
		ASSERT_EQ(memcmp(detail.encryptKey.begin(), &BASE_CIPHER_KEY_TEST[0], sizeof(BASE_CIPHER_KEY_TEST)), 0);
	}
	if (refreshKmsUrls) {
		validateKmsUrls(ctx);
	}
}

void testGetEncryptKeysByDomainIdsRequestBody(Reference<RESTKmsConnectorCtx> ctx, Arena& arena) {
	KmsConnLookupEKsByDomainIdsReq req;
	std::unordered_set<EncryptCipherDomainId> domainIds;
	const int nKeys = deterministicRandom()->randomInt(7, 25);
	for (int i = 1; i < nKeys; i++) {
		EncryptCipherDomainId domainId = getRandomDomainId();
		if (domainIds.insert(domainId).second) {
			req.encryptDomainIds.push_back(domainId);
		}
	}

	bool refreshKmsUrls = deterministicRandom()->coinflip();

	StringRef jsonReqRef = getEncryptKeysByDomainIdsRequestBody(ctx, req, refreshKmsUrls, arena);
	TraceEvent("FetchKeysByDomainIds", ctx->uid).detail("JsonReqStr", jsonReqRef.toString());
	Reference<HTTP::IncomingResponse> httpResp = makeReference<HTTP::IncomingResponse>();
	httpResp->code = HTTP::HTTP_STATUS_CODE_OK;
	getFakeEncryptCipherResponse(jsonReqRef, false, httpResp);
	TraceEvent("FetchKeysByDomainIds", ctx->uid).detail("HttpRespStr", httpResp->data.content);

	Standalone<VectorRef<EncryptCipherKeyDetailsRef>> cipherDetails = parseEncryptCipherResponse(ctx, httpResp);
	ASSERT_EQ(domainIds.size(), cipherDetails.size());
	for (const auto& detail : cipherDetails) {
		ASSERT(domainIds.find(detail.encryptDomainId) != domainIds.end());
		ASSERT_EQ(detail.encryptKey.size(), sizeof(BASE_CIPHER_KEY_TEST));
		ASSERT_EQ(memcmp(detail.encryptKey.begin(), &BASE_CIPHER_KEY_TEST[0], sizeof(BASE_CIPHER_KEY_TEST)), 0);
	}
	if (refreshKmsUrls) {
		validateKmsUrls(ctx);
	}
}

void testGetBlobMetadataRequestBody(Reference<RESTKmsConnectorCtx> ctx) {
	KmsConnBlobMetadataReq req;
	std::unordered_set<BlobMetadataDomainId> domainIds;
	const int nKeys = deterministicRandom()->randomInt(7, 25);
	for (int i = 1; i < nKeys; i++) {
		EncryptCipherDomainId domainId = deterministicRandom()->randomInt(0, 1000);
		if (domainIds.insert(domainId).second) {
			req.domainIds.push_back(domainId);
		}
	}

	bool refreshKmsUrls = deterministicRandom()->coinflip();

	TraceEvent("FetchBlobMetadataStart", ctx->uid);
	StringRef jsonReqRef = getBlobMetadataRequestBody(ctx, req, refreshKmsUrls);
	TraceEvent("FetchBlobMetadataReq", ctx->uid).detail("JsonReqStr", jsonReqRef.toString());
	Reference<HTTP::IncomingResponse> httpResp = makeReference<HTTP::IncomingResponse>();
	httpResp->code = HTTP::HTTP_STATUS_CODE_OK;
	getFakeBlobMetadataResponse(jsonReqRef, false, httpResp);
	TraceEvent("FetchBlobMetadataResp", ctx->uid).detail("HttpRespStr", httpResp->data.content);

	Standalone<VectorRef<BlobMetadataDetailsRef>> details = parseBlobMetadataResponse(ctx, httpResp);

	ASSERT_EQ(domainIds.size(), details.size());
	for (const auto& detail : details) {
		auto it = domainIds.find(detail.domainId);
		ASSERT(it != domainIds.end());
		ASSERT(!detail.locations.empty());
	}
	if (refreshKmsUrls) {
		validateKmsUrls(ctx);
	}
}

void testMissingOrInvalidVersion(Reference<RESTKmsConnectorCtx> ctx, bool isCipher) {
	rapidjson::Document doc;
	doc.SetObject();

	rapidjson::Value cDetails(rapidjson::kArrayType);
	rapidjson::Value detail(rapidjson::kObjectType);
	rapidjson::Value key(isCipher ? BASE_CIPHER_ID_TAG : BLOB_METADATA_DOMAIN_ID_TAG, doc.GetAllocator());
	rapidjson::Value id;
	id.SetUint(12345);
	detail.AddMember(key, id, doc.GetAllocator());
	cDetails.PushBack(detail, doc.GetAllocator());
	key.SetString(isCipher ? CIPHER_KEY_DETAILS_TAG : BLOB_METADATA_DETAILS_TAG, doc.GetAllocator());
	doc.AddMember(key, cDetails, doc.GetAllocator());

	rapidjson::Value versionKey(REQUEST_VERSION_TAG, doc.GetAllocator());
	rapidjson::Value versionValue;
	int version = INVALID_REQUEST_VERSION;
	if (deterministicRandom()->coinflip()) {
		if (deterministicRandom()->coinflip()) {
			version = -7;
		} else {
			version = (isCipher ? SERVER_KNOBS->REST_KMS_CURRENT_CIPHER_REQUEST_VERSION
			                    : SERVER_KNOBS->REST_KMS_CURRENT_BLOB_METADATA_REQUEST_VERSION) +
			          10;
		}
	} else {
		// set to invalid_version
	}
	versionValue.SetInt(version);
	doc.AddMember(versionKey, versionValue, doc.GetAllocator());

	Reference<HTTP::IncomingResponse> httpResp = makeReference<HTTP::IncomingResponse>();
	httpResp->code = HTTP::HTTP_STATUS_CODE_OK;
	httpResp->data.contentLen = 0;
	httpResp->data.content = "";
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);
	httpResp->data.content.resize(sb.GetSize(), '\0');
	memcpy(httpResp->data.content.data(), sb.GetString(), sb.GetSize());

	try {
		if (isCipher) {
			parseEncryptCipherResponse(ctx, httpResp);
		} else {
			parseBlobMetadataResponse(ctx, httpResp);
		}
	} catch (Error& e) {
		ASSERT_EQ(e.code(), error_code_rest_malformed_response);
	}
}

void testMissingDetailsTag(Reference<RESTKmsConnectorCtx> ctx, bool isCipher) {
	rapidjson::Document doc;
	doc.SetObject();

	rapidjson::Value key(KMS_URLS_TAG, doc.GetAllocator());
	rapidjson::Value refreshUrl;
	refreshUrl.SetBool(true);
	doc.AddMember(key, refreshUrl, doc.GetAllocator());

	Reference<HTTP::IncomingResponse> httpResp = makeReference<HTTP::IncomingResponse>();
	httpResp->code = HTTP::HTTP_STATUS_CODE_OK;
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);
	httpResp->data.content.resize(sb.GetSize(), '\0');
	memcpy(httpResp->data.content.data(), sb.GetString(), sb.GetSize());
	httpResp->data.contentLen = sb.GetSize();

	try {
		if (isCipher) {
			parseEncryptCipherResponse(ctx, httpResp);
		} else {
			parseBlobMetadataResponse(ctx, httpResp);
		}
		ASSERT(false); // error expected
	} catch (Error& e) {
		ASSERT_EQ(e.code(), error_code_rest_malformed_response);
	}
}

void testMalformedDetails(Reference<RESTKmsConnectorCtx> ctx, bool isCipher) {
	TraceEvent("TestMalformedDetailsStart");
	rapidjson::Document doc;
	doc.SetObject();

	rapidjson::Value key(isCipher ? CIPHER_KEY_DETAILS_TAG : BLOB_METADATA_DETAILS_TAG, doc.GetAllocator());
	rapidjson::Value details;
	details.SetBool(true);
	doc.AddMember(key, details, doc.GetAllocator());

	addVersionToDoc(doc, 1);

	Reference<HTTP::IncomingResponse> httpResp = makeReference<HTTP::IncomingResponse>();
	httpResp->code = HTTP::HTTP_STATUS_CODE_OK;
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);
	httpResp->data.content.resize(sb.GetSize(), '\0');
	memcpy(httpResp->data.content.data(), sb.GetString(), sb.GetSize());
	httpResp->data.contentLen = sb.GetSize();

	try {
		if (isCipher) {
			parseEncryptCipherResponse(ctx, httpResp);
		} else {
			parseBlobMetadataResponse(ctx, httpResp);
		}
		ASSERT(false); // error expected
	} catch (Error& e) {
		ASSERT_EQ(e.code(), error_code_rest_malformed_response);
	}
	TraceEvent("TestMalformedDetailsEnd");
}

void testMalformedDetailNotObj(Reference<RESTKmsConnectorCtx> ctx, bool isCipher) {
	TraceEvent("TestMalformedDetailNotObjStart");
	rapidjson::Document doc;
	doc.SetObject();

	rapidjson::Value cDetails(rapidjson::kArrayType);
	rapidjson::Value detail(rapidjson::kObjectType);
	rapidjson::Value key(isCipher ? BASE_CIPHER_ID_TAG : BLOB_METADATA_DOMAIN_ID_TAG, doc.GetAllocator());
	rapidjson::Value id;
	id.SetUint(12345);
	detail.AddMember(key, id, doc.GetAllocator());
	cDetails.PushBack(detail, doc.GetAllocator());
	key.SetString(isCipher ? CIPHER_KEY_DETAILS_TAG : BLOB_METADATA_DETAILS_TAG, doc.GetAllocator());
	doc.AddMember(key, cDetails, doc.GetAllocator());

	addVersionToDoc(doc, 1);

	Reference<HTTP::IncomingResponse> httpResp = makeReference<HTTP::IncomingResponse>();
	httpResp->code = HTTP::HTTP_STATUS_CODE_OK;
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);
	httpResp->data.content.resize(sb.GetSize(), '\0');
	memcpy(httpResp->data.content.data(), sb.GetString(), sb.GetSize());
	httpResp->data.contentLen = sb.GetSize();

	try {
		if (isCipher) {
			parseEncryptCipherResponse(ctx, httpResp);
		} else {
			parseBlobMetadataResponse(ctx, httpResp);
		}
		ASSERT(false); // error expected
	} catch (Error& e) {
		ASSERT_EQ(e.code(), error_code_rest_malformed_response);
	}
	TraceEvent("TestMalformedDetailNotObjEnd");
}

void testMalformedDetailObj(Reference<RESTKmsConnectorCtx> ctx, bool isCipher) {
	TraceEvent("TestMalformedDetailObjStart");
	rapidjson::Document doc;
	doc.SetObject();

	rapidjson::Value cDetails(rapidjson::kArrayType);
	rapidjson::Value detail(rapidjson::kObjectType);
	rapidjson::Value key(isCipher ? BASE_CIPHER_ID_TAG : BLOB_METADATA_DOMAIN_ID_TAG, doc.GetAllocator());
	rapidjson::Value id;
	id.SetUint(12345);
	detail.AddMember(key, id, doc.GetAllocator());
	cDetails.PushBack(detail, doc.GetAllocator());
	key.SetString(isCipher ? CIPHER_KEY_DETAILS_TAG : BLOB_METADATA_DETAILS_TAG, doc.GetAllocator());
	doc.AddMember(key, cDetails, doc.GetAllocator());

	addVersionToDoc(doc, 1);

	Reference<HTTP::IncomingResponse> httpResp = makeReference<HTTP::IncomingResponse>();
	httpResp->code = HTTP::HTTP_STATUS_CODE_OK;
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);
	httpResp->data.content.resize(sb.GetSize(), '\0');
	memcpy(httpResp->data.content.data(), sb.GetString(), sb.GetSize());
	httpResp->data.contentLen = sb.GetSize();

	try {
		if (isCipher) {
			parseEncryptCipherResponse(ctx, httpResp);
		} else {
			parseBlobMetadataResponse(ctx, httpResp);
		}
		ASSERT(false); // error expected
	} catch (Error& e) {
		ASSERT_EQ(e.code(), error_code_rest_malformed_response);
	}
	TraceEvent("TestMalformedDetailObjEnd");
}

void testKMSErrorResponse(Reference<RESTKmsConnectorCtx> ctx, bool isCipher) {
	rapidjson::Document doc;
	doc.SetObject();

	addVersionToDoc(doc, 1);

	// Construct fake response, it should get ignored anyways
	rapidjson::Value cDetails(rapidjson::kArrayType);
	rapidjson::Value detail(rapidjson::kObjectType);
	rapidjson::Value key(BASE_CIPHER_ID_TAG, doc.GetAllocator());
	rapidjson::Value id;
	id.SetUint(12345);
	detail.AddMember(key, id, doc.GetAllocator());
	cDetails.PushBack(detail, doc.GetAllocator());
	key.SetString(isCipher ? CIPHER_KEY_DETAILS_TAG : BLOB_METADATA_DETAILS_TAG, doc.GetAllocator());
	doc.AddMember(key, cDetails, doc.GetAllocator());

	// Add error tag
	rapidjson::Value errorTag(rapidjson::kObjectType);

	// Add 'error_detail'
	rapidjson::Value eKey(ERROR_MSG_TAG, doc.GetAllocator());
	rapidjson::Value detailInfo;
	detailInfo.SetString("Foo is always bad", doc.GetAllocator());
	errorTag.AddMember(eKey, detailInfo, doc.GetAllocator());

	key.SetString(ERROR_TAG, doc.GetAllocator());
	doc.AddMember(key, errorTag, doc.GetAllocator());

	Reference<HTTP::IncomingResponse> httpResp = makeReference<HTTP::IncomingResponse>();
	httpResp->code = HTTP::HTTP_STATUS_CODE_OK;
	rapidjson::StringBuffer sb;
	rapidjson::Writer<rapidjson::StringBuffer> writer(sb);
	doc.Accept(writer);
	httpResp->data.content.resize(sb.GetSize(), '\0');
	memcpy(httpResp->data.content.data(), sb.GetString(), sb.GetSize());
	httpResp->data.contentLen = sb.GetSize();

	try {
		if (isCipher) {
			parseEncryptCipherResponse(ctx, httpResp);
		} else {
			parseBlobMetadataResponse(ctx, httpResp);
		}
		ASSERT(false); // error expected
	} catch (Error& e) {
		ASSERT_EQ(e.code(), error_code_encrypt_keys_fetch_failed);
	}
}

															#line 6852 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
// This generated class is to be used only via testParseDiscoverKmsUrlFileNotFound()
															#line 1866 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class TestParseDiscoverKmsUrlFileNotFoundActor>
															#line 1866 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestParseDiscoverKmsUrlFileNotFoundActorState {
															#line 6858 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1866 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestParseDiscoverKmsUrlFileNotFoundActorState(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 1866 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1866 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx)
															#line 6865 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("testParseDiscoverKmsUrlFileNotFound", reinterpret_cast<unsigned long>(this));

	}
	~TestParseDiscoverKmsUrlFileNotFoundActorState() 
	{
		fdb_probe_actor_destroy("testParseDiscoverKmsUrlFileNotFound", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1868 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				StrictFuture<Void> __when_expr_0 = parseDiscoverKmsUrlFile(ctx, "/imaginary-dir/dream/phantom-file");
															#line 1868 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 6883 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->actor_wait_state = 1;
															#line 1868 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestParseDiscoverKmsUrlFileNotFoundActor, 0, Void >*>(static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)));
															#line 6888 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestParseDiscoverKmsUrlFileNotFoundActorState();
		static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1873 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestParseDiscoverKmsUrlFileNotFoundActorState(); static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->destroy(); return 0; }
															#line 6917 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestParseDiscoverKmsUrlFileNotFoundActorState();
		static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1871 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(e.code(), error_code_encrypt_invalid_kms_config);
															#line 6930 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1869 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 6945 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1869 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT(false);
															#line 6954 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->actor_wait_state > 0) static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->actor_wait_state = 0;
		static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->ActorCallback< TestParseDiscoverKmsUrlFileNotFoundActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestParseDiscoverKmsUrlFileNotFoundActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFileNotFound", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFileNotFound", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestParseDiscoverKmsUrlFileNotFoundActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFileNotFound", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFileNotFound", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestParseDiscoverKmsUrlFileNotFoundActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFileNotFound", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFileNotFound", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 1866 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 7049 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via testParseDiscoverKmsUrlFileNotFound()
															#line 1866 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestParseDiscoverKmsUrlFileNotFoundActor final : public Actor<Void>, public ActorCallback< TestParseDiscoverKmsUrlFileNotFoundActor, 0, Void >, public FastAllocated<TestParseDiscoverKmsUrlFileNotFoundActor>, public TestParseDiscoverKmsUrlFileNotFoundActorState<TestParseDiscoverKmsUrlFileNotFoundActor> {
															#line 7054 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<TestParseDiscoverKmsUrlFileNotFoundActor>::operator new;
	using FastAllocated<TestParseDiscoverKmsUrlFileNotFoundActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11928811551589333248UL, 13982366030813242368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestParseDiscoverKmsUrlFileNotFoundActor, 0, Void >;
															#line 1866 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestParseDiscoverKmsUrlFileNotFoundActor(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 7071 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   TestParseDiscoverKmsUrlFileNotFoundActorState<TestParseDiscoverKmsUrlFileNotFoundActor>(ctx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFileNotFound", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14525047893300351488UL, 2345080055439509760UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileNotFoundActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testParseDiscoverKmsUrlFileNotFound");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFileNotFound", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestParseDiscoverKmsUrlFileNotFoundActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1866 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> testParseDiscoverKmsUrlFileNotFound( Reference<RESTKmsConnectorCtx> const& ctx ) {
															#line 1866 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new TestParseDiscoverKmsUrlFileNotFoundActor(ctx));
															#line 7103 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1875 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 7108 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
// This generated class is to be used only via testParseDiscoverKmsUrlFile()
															#line 1876 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class TestParseDiscoverKmsUrlFileActor>
															#line 1876 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestParseDiscoverKmsUrlFileActorState {
															#line 7114 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1876 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestParseDiscoverKmsUrlFileActorState(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 1876 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1876 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx),
															#line 1877 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   tmpFile(std::make_shared<platform::TmpFile>("/tmp"))
															#line 7123 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("testParseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this));

	}
	~TestParseDiscoverKmsUrlFileActorState() 
	{
		fdb_probe_actor_destroy("testParseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1878 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT(fileExists(tmpFile->getFileName()));
															#line 1880 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urls = std::unordered_set<std::string>();
															#line 1881 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urls.emplace("https://127.0.0.1/foo  ");
															#line 1882 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urls.emplace("  https://127.0.0.1/foo1");
															#line 1883 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urls.emplace("  https://127.0.0.1/foo2  ");
															#line 1884 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urls.emplace("https://127.0.0.1/foo3/");
															#line 1885 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urls.emplace("https://127.0.0.1/foo4///");
															#line 1887 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			compareUrls = std::unordered_set<std::string>();
															#line 1888 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			compareUrls.emplace("https://127.0.0.1/foo");
															#line 1889 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			compareUrls.emplace("https://127.0.0.1/foo1");
															#line 1890 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			compareUrls.emplace("https://127.0.0.1/foo2");
															#line 1891 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			compareUrls.emplace("https://127.0.0.1/foo3");
															#line 1892 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			compareUrls.emplace("https://127.0.0.1/foo4");
															#line 1894 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string content;
															#line 1895 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			for( auto& url : urls ) {
															#line 1896 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				content.append(url);
															#line 1897 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				content.push_back(DISCOVER_URL_FILE_URL_SEP);
															#line 7170 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 1899 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			tmpFile->write((const uint8_t*)content.data(), content.size());
															#line 1900 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			StrictFuture<Void> __when_expr_0 = parseDiscoverKmsUrlFile(ctx, tmpFile->getFileName());
															#line 1900 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7178 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state = 1;
															#line 1900 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestParseDiscoverKmsUrlFileActor, 0, Void >*>(static_cast<TestParseDiscoverKmsUrlFileActor*>(this)));
															#line 7183 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestParseDiscoverKmsUrlFileActorState();
		static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1902 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT_EQ(ctx->kmsUrlStore.kmsUrls.size(), urls.size());
															#line 1903 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		for( const auto& url : ctx->kmsUrlStore.kmsUrls ) {
															#line 1904 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT(compareUrls.find(url.url) != compareUrls.end());
															#line 1905 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(url.nFailedResponses, 0);
															#line 1906 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(url.nRequests, 0);
															#line 1907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(url.nResponseParseFailures, 0);
															#line 7216 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 1910 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestParseDiscoverKmsUrlFileActorState(); static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->destroy(); return 0; }
															#line 7220 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestParseDiscoverKmsUrlFileActorState();
		static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1902 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT_EQ(ctx->kmsUrlStore.kmsUrls.size(), urls.size());
															#line 1903 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		for( const auto& url : ctx->kmsUrlStore.kmsUrls ) {
															#line 1904 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT(compareUrls.find(url.url) != compareUrls.end());
															#line 1905 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(url.nFailedResponses, 0);
															#line 1906 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(url.nRequests, 0);
															#line 1907 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(url.nResponseParseFailures, 0);
															#line 7242 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 1910 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestParseDiscoverKmsUrlFileActorState(); static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->destroy(); return 0; }
															#line 7246 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestParseDiscoverKmsUrlFileActorState();
		static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state > 0) static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->actor_wait_state = 0;
		static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->ActorCallback< TestParseDiscoverKmsUrlFileActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestParseDiscoverKmsUrlFileActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestParseDiscoverKmsUrlFileActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestParseDiscoverKmsUrlFileActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1876 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 1877 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::shared_ptr<platform::TmpFile> tmpFile;
															#line 1880 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::unordered_set<std::string> urls;
															#line 1887 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::unordered_set<std::string> compareUrls;
															#line 7337 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via testParseDiscoverKmsUrlFile()
															#line 1876 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestParseDiscoverKmsUrlFileActor final : public Actor<Void>, public ActorCallback< TestParseDiscoverKmsUrlFileActor, 0, Void >, public FastAllocated<TestParseDiscoverKmsUrlFileActor>, public TestParseDiscoverKmsUrlFileActorState<TestParseDiscoverKmsUrlFileActor> {
															#line 7342 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<TestParseDiscoverKmsUrlFileActor>::operator new;
	using FastAllocated<TestParseDiscoverKmsUrlFileActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15161085407137806336UL, 9866340029046616064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestParseDiscoverKmsUrlFileActor, 0, Void >;
															#line 1876 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestParseDiscoverKmsUrlFileActor(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 7359 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   TestParseDiscoverKmsUrlFileActorState<TestParseDiscoverKmsUrlFileActor>(ctx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16187203404140478976UL, 4421232396576303360UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testParseDiscoverKmsUrlFile");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFile", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestParseDiscoverKmsUrlFileActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1876 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> testParseDiscoverKmsUrlFile( Reference<RESTKmsConnectorCtx> const& ctx ) {
															#line 1876 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new TestParseDiscoverKmsUrlFileActor(ctx));
															#line 7391 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1912 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 7396 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
// This generated class is to be used only via testParseDiscoverKmsUrlFileAlreadyExisting()
															#line 1913 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class TestParseDiscoverKmsUrlFileAlreadyExistingActor>
															#line 1913 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestParseDiscoverKmsUrlFileAlreadyExistingActorState {
															#line 7402 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1913 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestParseDiscoverKmsUrlFileAlreadyExistingActorState(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 1913 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1913 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : ctx(ctx)
															#line 7409 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("testParseDiscoverKmsUrlFileAlreadyExisting", reinterpret_cast<unsigned long>(this));

	}
	~TestParseDiscoverKmsUrlFileAlreadyExistingActorState() 
	{
		fdb_probe_actor_destroy("testParseDiscoverKmsUrlFileAlreadyExisting", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1914 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::unordered_map<std::string, KmsUrlCtx<KmsUrlPenaltyParams>> urlMap;
															#line 1915 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			dropCachedKmsUrls(ctx, &urlMap);
															#line 1916 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(ctx->kmsUrlStore.kmsUrls.size(), 0);
															#line 1918 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			auto urlCtx = KmsUrlCtx<KmsUrlPenaltyParams>("https://127.0.0.1/foo2");
															#line 1919 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urlCtx.nFailedResponses = 1;
															#line 1920 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urlCtx.nRequests = 2;
															#line 1921 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urlCtx.nResponseParseFailures = 3;
															#line 1922 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ctx->kmsUrlStore.kmsUrls.push_back(KmsUrlCtx<KmsUrlPenaltyParams>("https://127.0.0.1/foo4"));
															#line 1923 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ctx->kmsUrlStore.kmsUrls.push_back(KmsUrlCtx<KmsUrlPenaltyParams>("https://127.0.0.1/foo5"));
															#line 1924 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ctx->kmsUrlStore.kmsUrls.push_back(KmsUrlCtx<KmsUrlPenaltyParams>(urlCtx));
															#line 1926 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			tmpFile = std::make_shared<platform::TmpFile>("/tmp");
															#line 1927 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT(fileExists(tmpFile->getFileName()));
															#line 1929 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urls = std::unordered_set<std::string>();
															#line 1930 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urls.emplace("https://127.0.0.1/foo  ");
															#line 1931 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urls.emplace("  https://127.0.0.1/foo1");
															#line 1932 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			urls.emplace("  https://127.0.0.1/foo2  ");
															#line 1934 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			compareUrls = std::unordered_set<std::string>();
															#line 1935 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			compareUrls.emplace("https://127.0.0.1/foo");
															#line 1936 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			compareUrls.emplace("https://127.0.0.1/foo1");
															#line 1937 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			compareUrls.emplace("https://127.0.0.1/foo2");
															#line 1939 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			std::string content;
															#line 1940 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			for( auto& url : urls ) {
															#line 1941 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				content.append(url);
															#line 1942 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				content.push_back(DISCOVER_URL_FILE_URL_SEP);
															#line 7470 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 1944 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			tmpFile->write((const uint8_t*)content.data(), content.size());
															#line 1945 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			StrictFuture<Void> __when_expr_0 = parseDiscoverKmsUrlFile(ctx, tmpFile->getFileName());
															#line 1945 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7478 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->actor_wait_state = 1;
															#line 1945 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestParseDiscoverKmsUrlFileAlreadyExistingActor, 0, Void >*>(static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)));
															#line 7483 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestParseDiscoverKmsUrlFileAlreadyExistingActorState();
		static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1947 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT_EQ(ctx->kmsUrlStore.kmsUrls.size(), urls.size());
															#line 1948 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		for( const auto& url : ctx->kmsUrlStore.kmsUrls ) {
															#line 1949 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT(compareUrls.find(url.url) != compareUrls.end());
															#line 1950 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (url.url == "https://127.0.0.1/foo2")
															#line 7512 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 1951 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nFailedResponses, 1);
															#line 1952 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nRequests, 2);
															#line 1953 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nResponseParseFailures, 3);
															#line 7520 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1955 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nFailedResponses, 0);
															#line 1956 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nRequests, 0);
															#line 1957 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nResponseParseFailures, 0);
															#line 7530 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
		}
															#line 1961 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestParseDiscoverKmsUrlFileAlreadyExistingActorState(); static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->destroy(); return 0; }
															#line 7535 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestParseDiscoverKmsUrlFileAlreadyExistingActorState();
		static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1947 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		ASSERT_EQ(ctx->kmsUrlStore.kmsUrls.size(), urls.size());
															#line 1948 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		for( const auto& url : ctx->kmsUrlStore.kmsUrls ) {
															#line 1949 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT(compareUrls.find(url.url) != compareUrls.end());
															#line 1950 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (url.url == "https://127.0.0.1/foo2")
															#line 7553 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			{
															#line 1951 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nFailedResponses, 1);
															#line 1952 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nRequests, 2);
															#line 1953 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nResponseParseFailures, 3);
															#line 7561 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1955 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nFailedResponses, 0);
															#line 1956 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nRequests, 0);
															#line 1957 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nResponseParseFailures, 0);
															#line 7571 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
		}
															#line 1961 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestParseDiscoverKmsUrlFileAlreadyExistingActorState(); static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->destroy(); return 0; }
															#line 7576 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestParseDiscoverKmsUrlFileAlreadyExistingActorState();
		static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->actor_wait_state > 0) static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->actor_wait_state = 0;
		static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->ActorCallback< TestParseDiscoverKmsUrlFileAlreadyExistingActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestParseDiscoverKmsUrlFileAlreadyExistingActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFileAlreadyExisting", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFileAlreadyExisting", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestParseDiscoverKmsUrlFileAlreadyExistingActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFileAlreadyExisting", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFileAlreadyExisting", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestParseDiscoverKmsUrlFileAlreadyExistingActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFileAlreadyExisting", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFileAlreadyExisting", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1913 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 1926 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::shared_ptr<platform::TmpFile> tmpFile;
															#line 1929 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::unordered_set<std::string> urls;
															#line 1934 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	std::unordered_set<std::string> compareUrls;
															#line 7667 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via testParseDiscoverKmsUrlFileAlreadyExisting()
															#line 1913 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class TestParseDiscoverKmsUrlFileAlreadyExistingActor final : public Actor<Void>, public ActorCallback< TestParseDiscoverKmsUrlFileAlreadyExistingActor, 0, Void >, public FastAllocated<TestParseDiscoverKmsUrlFileAlreadyExistingActor>, public TestParseDiscoverKmsUrlFileAlreadyExistingActorState<TestParseDiscoverKmsUrlFileAlreadyExistingActor> {
															#line 7672 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<TestParseDiscoverKmsUrlFileAlreadyExistingActor>::operator new;
	using FastAllocated<TestParseDiscoverKmsUrlFileAlreadyExistingActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17261536704492050688UL, 13739697331917070080UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestParseDiscoverKmsUrlFileAlreadyExistingActor, 0, Void >;
															#line 1913 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	TestParseDiscoverKmsUrlFileAlreadyExistingActor(Reference<RESTKmsConnectorCtx> const& ctx) 
															#line 7689 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   TestParseDiscoverKmsUrlFileAlreadyExistingActorState<TestParseDiscoverKmsUrlFileAlreadyExistingActor>(ctx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testParseDiscoverKmsUrlFileAlreadyExisting", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12860797886940755456UL, 9069843595874204672UL);
		ActorExecutionContextHelper __helper(static_cast<TestParseDiscoverKmsUrlFileAlreadyExistingActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testParseDiscoverKmsUrlFileAlreadyExisting");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testParseDiscoverKmsUrlFileAlreadyExisting", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestParseDiscoverKmsUrlFileAlreadyExistingActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 1913 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
[[nodiscard]] Future<Void> testParseDiscoverKmsUrlFileAlreadyExisting( Reference<RESTKmsConnectorCtx> const& ctx ) {
															#line 1913 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new TestParseDiscoverKmsUrlFileAlreadyExistingActor(ctx));
															#line 7721 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}

#line 1963 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

void setKnobs() {
	auto& g_knobs = IKnobCollection::getMutableGlobalKnobCollection();
	g_knobs.setKnob("rest_kms_current_cipher_request_version", KnobValueRef::create(int{ 1 }));
	g_knobs.setKnob("rest_kms_current_blob_metadata_request_version", KnobValueRef::create(int{ 1 }));
	g_knobs.setKnob("rest_log_level", KnobValueRef::create(int{ 3 }));
	g_knobs.setKnob("rest_kms_connector_remove_trailing_newline", KnobValueRef::create(bool{ true }));
}

} // namespace

															#line 7736 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1974()
															#line 1974 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class FlowTestCase1974Actor>
															#line 1974 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase1974ActorState {
															#line 7743 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1974 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase1974ActorState(UnitTestParameters const& params) 
															#line 1974 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1974 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : params(params),
															#line 1975 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   ctx(makeReference<RESTKmsConnectorCtx>()),
															#line 1976 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   arena()
															#line 7754 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1974", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1974ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1974", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1978 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			setKnobs();
															#line 1981 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			deterministicRandom()->randomBytes(&BASE_CIPHER_KEY_TEST[0], 32);
															#line 1983 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			StrictFuture<Void> __when_expr_0 = testParseDiscoverKmsUrlFileNotFound(ctx);
															#line 1983 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7775 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state = 1;
															#line 1983 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1974Actor, 0, Void >*>(static_cast<FlowTestCase1974Actor*>(this)));
															#line 7780 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1974ActorState();
		static_cast<FlowTestCase1974Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 1984 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_1 = testParseDiscoverKmsUrlFile(ctx);
															#line 1984 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7805 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state = 2;
															#line 1984 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1974Actor, 1, Void >*>(static_cast<FlowTestCase1974Actor*>(this)));
															#line 7810 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 1984 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_1 = testParseDiscoverKmsUrlFile(ctx);
															#line 1984 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7821 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state = 2;
															#line 1984 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1974Actor, 1, Void >*>(static_cast<FlowTestCase1974Actor*>(this)));
															#line 7826 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1974Actor*>(this)->ActorCallback< FlowTestCase1974Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1974Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1974Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase1974Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 1985 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_2 = testParseDiscoverKmsUrlFileAlreadyExisting(ctx);
															#line 1985 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7912 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state = 3;
															#line 1985 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1974Actor, 2, Void >*>(static_cast<FlowTestCase1974Actor*>(this)));
															#line 7917 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 1985 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_2 = testParseDiscoverKmsUrlFileAlreadyExisting(ctx);
															#line 1985 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 7928 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state = 3;
															#line 1985 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1974Actor, 2, Void >*>(static_cast<FlowTestCase1974Actor*>(this)));
															#line 7933 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1974Actor*>(this)->ActorCallback< FlowTestCase1974Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1974Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1974Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase1974Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 1987 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<FlowTestCase1974Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1974ActorState(); static_cast<FlowTestCase1974Actor*>(this)->destroy(); return 0; }
															#line 8017 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase1974Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1974ActorState();
		static_cast<FlowTestCase1974Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 1987 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<FlowTestCase1974Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1974ActorState(); static_cast<FlowTestCase1974Actor*>(this)->destroy(); return 0; }
															#line 8029 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase1974Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1974ActorState();
		static_cast<FlowTestCase1974Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1974Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1974Actor*>(this)->ActorCallback< FlowTestCase1974Actor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1974Actor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1974Actor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FlowTestCase1974Actor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1974 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	UnitTestParameters params;
															#line 1975 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 1976 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Arena arena;
															#line 8118 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1974()
															#line 1974 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase1974Actor final : public Actor<Void>, public ActorCallback< FlowTestCase1974Actor, 0, Void >, public ActorCallback< FlowTestCase1974Actor, 1, Void >, public ActorCallback< FlowTestCase1974Actor, 2, Void >, public FastAllocated<FlowTestCase1974Actor>, public FlowTestCase1974ActorState<FlowTestCase1974Actor> {
															#line 8123 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1974Actor>::operator new;
	using FastAllocated<FlowTestCase1974Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2687408128802504704UL, 3065297564273814016UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase1974Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase1974Actor, 1, Void >;
friend struct ActorCallback< FlowTestCase1974Actor, 2, Void >;
															#line 1974 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase1974Actor(UnitTestParameters const& params) 
															#line 8142 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1974ActorState<FlowTestCase1974Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1974", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18100637140583322368UL, 7823974797634655744UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1974Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1974");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1974", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase1974Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase1974Actor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FlowTestCase1974Actor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1974 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
static Future<Void> flowTestCase1974( UnitTestParameters const& params ) {
															#line 1974 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new FlowTestCase1974Actor(params));
															#line 8177 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1974, "/KmsConnector/REST/ParseKmsDiscoveryUrls")

#line 1989 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 8183 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase1990()
															#line 1990 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class FlowTestCase1990Actor>
															#line 1990 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase1990ActorState {
															#line 8190 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 1990 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase1990ActorState(UnitTestParameters const& params) 
															#line 1990 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 1990 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : params(params),
															#line 1991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   ctx(makeReference<RESTKmsConnectorCtx>()),
															#line 1992 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   arena()
															#line 8201 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase1990", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase1990ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase1990", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1994 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			setKnobs();
															#line 1997 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			deterministicRandom()->randomBytes(&BASE_CIPHER_KEY_TEST[0], 32);
															#line 1999 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			StrictFuture<Void> __when_expr_0 = testEmptyValidationFileDetails(ctx);
															#line 1999 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8222 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 1;
															#line 1999 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1990Actor, 0, Void >*>(static_cast<FlowTestCase1990Actor*>(this)));
															#line 8227 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase1990ActorState();
		static_cast<FlowTestCase1990Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2000 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_1 = testMalformedFileValidationTokenDetails(ctx);
															#line 2000 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8252 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 2;
															#line 2000 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1990Actor, 1, Void >*>(static_cast<FlowTestCase1990Actor*>(this)));
															#line 8257 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2000 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_1 = testMalformedFileValidationTokenDetails(ctx);
															#line 2000 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8268 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 2;
															#line 2000 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1990Actor, 1, Void >*>(static_cast<FlowTestCase1990Actor*>(this)));
															#line 8273 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1990Actor*>(this)->ActorCallback< FlowTestCase1990Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase1990Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2001 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_2 = testValidationTokenFileNotFound(ctx);
															#line 2001 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8359 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 3;
															#line 2001 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1990Actor, 2, Void >*>(static_cast<FlowTestCase1990Actor*>(this)));
															#line 8364 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2001 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_2 = testValidationTokenFileNotFound(ctx);
															#line 2001 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8375 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 3;
															#line 2001 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1990Actor, 2, Void >*>(static_cast<FlowTestCase1990Actor*>(this)));
															#line 8380 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1990Actor*>(this)->ActorCallback< FlowTestCase1990Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase1990Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 2002 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_3 = testTooLargeValidationTokenFile(ctx);
															#line 2002 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8466 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 4;
															#line 2002 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1990Actor, 3, Void >*>(static_cast<FlowTestCase1990Actor*>(this)));
															#line 8471 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 2002 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_3 = testTooLargeValidationTokenFile(ctx);
															#line 2002 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8482 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 4;
															#line 2002 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1990Actor, 3, Void >*>(static_cast<FlowTestCase1990Actor*>(this)));
															#line 8487 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1990Actor*>(this)->ActorCallback< FlowTestCase1990Actor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< FlowTestCase1990Actor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 2003 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_4 = testValidationFileTokenPayloadTooLarge(ctx);
															#line 2003 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8573 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 5;
															#line 2003 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1990Actor, 4, Void >*>(static_cast<FlowTestCase1990Actor*>(this)));
															#line 8578 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 2003 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_4 = testValidationFileTokenPayloadTooLarge(ctx);
															#line 2003 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8589 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 5;
															#line 2003 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1990Actor, 4, Void >*>(static_cast<FlowTestCase1990Actor*>(this)));
															#line 8594 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1990Actor*>(this)->ActorCallback< FlowTestCase1990Actor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< FlowTestCase1990Actor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 2004 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_5 = testMultiValidationFileTokenFiles(ctx);
															#line 2004 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8680 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont5when1(__when_expr_5.get(), loopDepth); };
		static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 6;
															#line 2004 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1990Actor, 5, Void >*>(static_cast<FlowTestCase1990Actor*>(this)));
															#line 8685 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 2004 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_5 = testMultiValidationFileTokenFiles(ctx);
															#line 2004 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 8696 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont5when1(__when_expr_5.get(), loopDepth); };
		static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 6;
															#line 2004 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase1990Actor, 5, Void >*>(static_cast<FlowTestCase1990Actor*>(this)));
															#line 8701 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1990Actor*>(this)->ActorCallback< FlowTestCase1990Actor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< FlowTestCase1990Actor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 2006 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<FlowTestCase1990Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1990ActorState(); static_cast<FlowTestCase1990Actor*>(this)->destroy(); return 0; }
															#line 8785 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase1990Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1990ActorState();
		static_cast<FlowTestCase1990Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 2006 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<FlowTestCase1990Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase1990ActorState(); static_cast<FlowTestCase1990Actor*>(this)->destroy(); return 0; }
															#line 8797 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase1990Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase1990ActorState();
		static_cast<FlowTestCase1990Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase1990Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase1990Actor*>(this)->ActorCallback< FlowTestCase1990Actor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< FlowTestCase1990Actor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< FlowTestCase1990Actor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), 5);

	}
															#line 1990 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	UnitTestParameters params;
															#line 1991 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 1992 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Arena arena;
															#line 8886 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase1990()
															#line 1990 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase1990Actor final : public Actor<Void>, public ActorCallback< FlowTestCase1990Actor, 0, Void >, public ActorCallback< FlowTestCase1990Actor, 1, Void >, public ActorCallback< FlowTestCase1990Actor, 2, Void >, public ActorCallback< FlowTestCase1990Actor, 3, Void >, public ActorCallback< FlowTestCase1990Actor, 4, Void >, public ActorCallback< FlowTestCase1990Actor, 5, Void >, public FastAllocated<FlowTestCase1990Actor>, public FlowTestCase1990ActorState<FlowTestCase1990Actor> {
															#line 8891 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase1990Actor>::operator new;
	using FastAllocated<FlowTestCase1990Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9218962868562251008UL, 1987028781058460672UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase1990Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase1990Actor, 1, Void >;
friend struct ActorCallback< FlowTestCase1990Actor, 2, Void >;
friend struct ActorCallback< FlowTestCase1990Actor, 3, Void >;
friend struct ActorCallback< FlowTestCase1990Actor, 4, Void >;
friend struct ActorCallback< FlowTestCase1990Actor, 5, Void >;
															#line 1990 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase1990Actor(UnitTestParameters const& params) 
															#line 8913 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase1990ActorState<FlowTestCase1990Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase1990", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1093206899862294016UL, 17026273717551034624UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase1990Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase1990");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase1990", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase1990Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase1990Actor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< FlowTestCase1990Actor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< FlowTestCase1990Actor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< FlowTestCase1990Actor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< FlowTestCase1990Actor, 5, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1990 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
static Future<Void> flowTestCase1990( UnitTestParameters const& params ) {
															#line 1990 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new FlowTestCase1990Actor(params));
															#line 8951 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase1990, "/KmsConnector/REST/ParseValidationTokenFile")

#line 2008 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 8957 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2009()
															#line 2009 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class FlowTestCase2009Actor>
															#line 2009 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase2009ActorState {
															#line 8964 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 2009 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase2009ActorState(UnitTestParameters const& params) 
															#line 2009 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 2009 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : params(params),
															#line 2010 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   ctx(makeReference<RESTKmsConnectorCtx>()),
															#line 2011 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   arena()
															#line 8975 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2009", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2009ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2009", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2013 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			setKnobs();
															#line 2016 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			deterministicRandom()->randomBytes(&BASE_CIPHER_KEY_TEST[0], 32);
															#line 2018 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testMissingOrInvalidVersion(ctx, true);
															#line 2019 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testMissingDetailsTag(ctx, true);
															#line 2020 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testMalformedDetails(ctx, true);
															#line 2021 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testMalformedDetailNotObj(ctx, true);
															#line 2022 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testMalformedDetailObj(ctx, true);
															#line 2023 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testKMSErrorResponse(ctx, true);
															#line 2024 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (!static_cast<FlowTestCase2009Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2009ActorState(); static_cast<FlowTestCase2009Actor*>(this)->destroy(); return 0; }
															#line 9006 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2009Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2009ActorState();
			static_cast<FlowTestCase2009Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2009ActorState();
		static_cast<FlowTestCase2009Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2009 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	UnitTestParameters params;
															#line 2010 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 2011 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Arena arena;
															#line 9034 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2009()
															#line 2009 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase2009Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2009Actor>, public FlowTestCase2009ActorState<FlowTestCase2009Actor> {
															#line 9039 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2009Actor>::operator new;
	using FastAllocated<FlowTestCase2009Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(974786836740415744UL, 1930969934555866112UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2009 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase2009Actor(UnitTestParameters const& params) 
															#line 9055 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2009ActorState<FlowTestCase2009Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2009", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13576115846491637248UL, 1211027230964504576UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2009Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2009");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2009", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2009 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
static Future<Void> flowTestCase2009( UnitTestParameters const& params ) {
															#line 2009 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new FlowTestCase2009Actor(params));
															#line 9087 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2009, "/KmsConnector/REST/ParseEncryptCipherResponse")

#line 2026 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 9093 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2027()
															#line 2027 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class FlowTestCase2027Actor>
															#line 2027 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase2027ActorState {
															#line 9100 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 2027 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase2027ActorState(UnitTestParameters const& params) 
															#line 2027 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 2027 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : params(params),
															#line 2028 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   ctx(makeReference<RESTKmsConnectorCtx>()),
															#line 2029 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   arena()
															#line 9111 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2027", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2027ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2027", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2031 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			setKnobs();
															#line 2033 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testMissingOrInvalidVersion(ctx, true);
															#line 2034 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testMissingDetailsTag(ctx, false);
															#line 2035 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testMalformedDetails(ctx, false);
															#line 2036 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testMalformedDetailNotObj(ctx, false);
															#line 2037 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testMalformedDetailObj(ctx, true);
															#line 2038 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testKMSErrorResponse(ctx, false);
															#line 2039 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (!static_cast<FlowTestCase2027Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2027ActorState(); static_cast<FlowTestCase2027Actor*>(this)->destroy(); return 0; }
															#line 9140 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2027Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2027ActorState();
			static_cast<FlowTestCase2027Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2027ActorState();
		static_cast<FlowTestCase2027Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2027 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	UnitTestParameters params;
															#line 2028 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 2029 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Arena arena;
															#line 9168 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2027()
															#line 2027 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase2027Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2027Actor>, public FlowTestCase2027ActorState<FlowTestCase2027Actor> {
															#line 9173 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2027Actor>::operator new;
	using FastAllocated<FlowTestCase2027Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3400673279482960896UL, 12713467616432071424UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2027 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase2027Actor(UnitTestParameters const& params) 
															#line 9189 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2027ActorState<FlowTestCase2027Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2027", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16236332943724244224UL, 12145729724406796544UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2027Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2027");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2027", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2027 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
static Future<Void> flowTestCase2027( UnitTestParameters const& params ) {
															#line 2027 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new FlowTestCase2027Actor(params));
															#line 9221 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2027, "/KmsConnector/REST/ParseBlobMetadataResponse")

#line 2041 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

															#line 9227 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2042()
															#line 2042 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class FlowTestCase2042Actor>
															#line 2042 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase2042ActorState {
															#line 9234 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 2042 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase2042ActorState(UnitTestParameters const& params) 
															#line 2042 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 2042 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : params(params),
															#line 2043 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   ctx(makeReference<RESTKmsConnectorCtx>()),
															#line 2044 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		   arena()
															#line 9245 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2042", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2042ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2042", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2046 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			setKnobs();
															#line 2049 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			deterministicRandom()->randomBytes(&BASE_CIPHER_KEY_TEST[0], 32);
															#line 2052 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			StrictFuture<Void> __when_expr_0 = testParseDiscoverKmsUrlFile(ctx);
															#line 2052 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (static_cast<FlowTestCase2042Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9266 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<FlowTestCase2042Actor*>(this)->actor_wait_state = 1;
															#line 2052 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2042Actor, 0, Void >*>(static_cast<FlowTestCase2042Actor*>(this)));
															#line 9271 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2042ActorState();
		static_cast<FlowTestCase2042Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2053 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_1 = testMultiValidationFileTokenFiles(ctx);
															#line 2053 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase2042Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9296 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2042Actor*>(this)->actor_wait_state = 2;
															#line 2053 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2042Actor, 1, Void >*>(static_cast<FlowTestCase2042Actor*>(this)));
															#line 9301 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2053 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		StrictFuture<Void> __when_expr_1 = testMultiValidationFileTokenFiles(ctx);
															#line 2053 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (static_cast<FlowTestCase2042Actor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 9312 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<FlowTestCase2042Actor*>(this)->actor_wait_state = 2;
															#line 2053 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< FlowTestCase2042Actor, 1, Void >*>(static_cast<FlowTestCase2042Actor*>(this)));
															#line 9317 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<FlowTestCase2042Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2042Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2042Actor*>(this)->ActorCallback< FlowTestCase2042Actor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2042Actor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2042", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2042Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2042", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2042Actor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2042", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2042Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2042", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< FlowTestCase2042Actor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2042", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2042Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2042", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 2055 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		const int numIterations = deterministicRandom()->randomInt(512, 786);
															#line 2056 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		for(int i = 0;i < numIterations;i++) {
															#line 2057 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testGetEncryptKeysByKeyIdsRequestBody(ctx, arena);
															#line 2058 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testGetEncryptKeysByDomainIdsRequestBody(ctx, arena);
															#line 2059 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testGetBlobMetadataRequestBody(ctx);
															#line 9409 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 2061 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<FlowTestCase2042Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2042ActorState(); static_cast<FlowTestCase2042Actor*>(this)->destroy(); return 0; }
															#line 9413 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase2042Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2042ActorState();
		static_cast<FlowTestCase2042Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 2055 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		const int numIterations = deterministicRandom()->randomInt(512, 786);
															#line 2056 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		for(int i = 0;i < numIterations;i++) {
															#line 2057 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testGetEncryptKeysByKeyIdsRequestBody(ctx, arena);
															#line 2058 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testGetEncryptKeysByDomainIdsRequestBody(ctx, arena);
															#line 2059 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			testGetBlobMetadataRequestBody(ctx);
															#line 9433 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		}
															#line 2061 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		if (!static_cast<FlowTestCase2042Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2042ActorState(); static_cast<FlowTestCase2042Actor*>(this)->destroy(); return 0; }
															#line 9437 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		new (&static_cast<FlowTestCase2042Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase2042ActorState();
		static_cast<FlowTestCase2042Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<FlowTestCase2042Actor*>(this)->actor_wait_state > 0) static_cast<FlowTestCase2042Actor*>(this)->actor_wait_state = 0;
		static_cast<FlowTestCase2042Actor*>(this)->ActorCallback< FlowTestCase2042Actor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< FlowTestCase2042Actor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("flowTestCase2042", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2042Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2042", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< FlowTestCase2042Actor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("flowTestCase2042", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2042Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2042", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< FlowTestCase2042Actor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("flowTestCase2042", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2042Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("flowTestCase2042", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 2042 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	UnitTestParameters params;
															#line 2043 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Reference<RESTKmsConnectorCtx> ctx;
															#line 2044 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	Arena arena;
															#line 9526 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2042()
															#line 2042 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase2042Actor final : public Actor<Void>, public ActorCallback< FlowTestCase2042Actor, 0, Void >, public ActorCallback< FlowTestCase2042Actor, 1, Void >, public FastAllocated<FlowTestCase2042Actor>, public FlowTestCase2042ActorState<FlowTestCase2042Actor> {
															#line 9531 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2042Actor>::operator new;
	using FastAllocated<FlowTestCase2042Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13723090953135151104UL, 10941781933823181824UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< FlowTestCase2042Actor, 0, Void >;
friend struct ActorCallback< FlowTestCase2042Actor, 1, Void >;
															#line 2042 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase2042Actor(UnitTestParameters const& params) 
															#line 9549 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2042ActorState<FlowTestCase2042Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2042", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14744226308146629120UL, 16106846869226614528UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2042Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2042");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2042", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< FlowTestCase2042Actor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< FlowTestCase2042Actor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2042 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
static Future<Void> flowTestCase2042( UnitTestParameters const& params ) {
															#line 2042 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new FlowTestCase2042Actor(params));
															#line 9583 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2042, "/KmsConnector/REST/GetEncryptionKeyOps")

#line 2063 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"

namespace {
struct TestUrlPenaltyParam {
	static double penalty(int64_t ignored) {
		int elapsed = deterministicRandom()->randomInt(1, 120);
		return KmsUrlPenaltyParams::penalty(elapsed);
	}
};
} // namespace

															#line 9598 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via flowTestCase2073()
															#line 2073 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
template <class FlowTestCase2073Actor>
															#line 2073 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase2073ActorState {
															#line 9605 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
															#line 2073 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase2073ActorState(UnitTestParameters const& params) 
															#line 2073 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
															#line 2073 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
		 : params(params)
															#line 9612 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase2073", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase2073ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase2073", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2074 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			KmsUrlStore<TestUrlPenaltyParam> store;
															#line 2075 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			const int nUrls = deterministicRandom()->randomInt(2, 10);
															#line 2076 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			for(int i = 0;i < nUrls;i++) {
															#line 2077 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				store.kmsUrls.emplace_back("foo" + std::to_string(i));
															#line 9633 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 2079 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			ASSERT_EQ(store.kmsUrls.size(), nUrls);
															#line 2080 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			for( const auto& url : store.kmsUrls ) {
															#line 2081 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.unresponsivenessPenalty, 0.0);
															#line 2082 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.unresponsivenessPenaltyTS, 0);
															#line 2083 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nFailedResponses, 0);
															#line 2084 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nResponseParseFailures, 0);
															#line 2085 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				ASSERT_EQ(url.nRequests, 0);
															#line 9649 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			}
															#line 2088 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			const int nIterations = deterministicRandom()->randomInt(100, 500);
															#line 2089 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			for(int i = 0;i < nIterations;i++) {
															#line 2090 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				const int idx = deterministicRandom()->randomInt(0, nUrls);
															#line 2092 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				if (deterministicRandom()->coinflip())
															#line 9659 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
				{
															#line 2093 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					if (deterministicRandom()->coinflip())
															#line 9663 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
					{
															#line 2094 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
						store.penalize(store.kmsUrls[idx], KmsUrlCtx<TestUrlPenaltyParam>::PenaltyType::TIMEOUT);
															#line 9667 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
					}
					else
					{
															#line 2096 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
						store.penalize(store.kmsUrls[idx], KmsUrlCtx<TestUrlPenaltyParam>::PenaltyType::MALFORMED_RESPONSE);
															#line 9673 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
					}
				}
				else
				{
				}
															#line 2102 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
				for(int j = 0;j < store.kmsUrls.size() - 1;j++) {
															#line 2103 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
					if (store.kmsUrls[j].unresponsivenessPenalty != store.kmsUrls[j + 1].unresponsivenessPenalty)
															#line 9683 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
					{
															#line 2104 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
						ASSERT_LE(store.kmsUrls[j].unresponsivenessPenalty, store.kmsUrls[j + 1].unresponsivenessPenalty);
															#line 9687 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
					}
					else
					{
															#line 2106 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
						if (store.kmsUrls[j].nFailedResponses != store.kmsUrls[j + 1].nFailedResponses)
															#line 9693 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
						{
															#line 2107 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
							ASSERT_LE(store.kmsUrls[j].nFailedResponses, store.kmsUrls[j + 1].nFailedResponses);
															#line 9697 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
						}
						else
						{
															#line 2109 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
							ASSERT_LE(store.kmsUrls[j].nResponseParseFailures, store.kmsUrls[j + 1].nResponseParseFailures);
															#line 9703 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
						}
					}
				}
			}
															#line 2114 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
			if (!static_cast<FlowTestCase2073Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase2073ActorState(); static_cast<FlowTestCase2073Actor*>(this)->destroy(); return 0; }
															#line 9710 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase2073Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase2073ActorState();
			static_cast<FlowTestCase2073Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase2073ActorState();
		static_cast<FlowTestCase2073Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 2073 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	UnitTestParameters params;
															#line 9734 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase2073()
															#line 2073 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
class FlowTestCase2073Actor final : public Actor<Void>, public FastAllocated<FlowTestCase2073Actor>, public FlowTestCase2073ActorState<FlowTestCase2073Actor> {
															#line 9739 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase2073Actor>::operator new;
	using FastAllocated<FlowTestCase2073Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17090252073743535872UL, 12613567134740979456UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 2073 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	FlowTestCase2073Actor(UnitTestParameters const& params) 
															#line 9755 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase2073ActorState<FlowTestCase2073Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase2073", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7913572192206814720UL, 18156498231524054272UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase2073Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase2073");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase2073", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 2073 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
static Future<Void> flowTestCase2073( UnitTestParameters const& params ) {
															#line 2073 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/fdbserver/RESTKmsConnector.actor.cpp"
	return Future<Void>(new FlowTestCase2073Actor(params));
															#line 9787 "/codebuild/output/src1317373735/src/github.com/apple/foundationdb/build_output/fdbserver/RESTKmsConnector.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase2073, "/KmsConnector/KmsUrlStore")

