#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
/*
 * EncryptionOps.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/BlobCipher.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/Knobs.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "flow/EncryptUtils.h"
#include "flow/Error.h"
#include "flow/flow.h"
#include "flow/IRandom.h"
#include "flow/ITrace.h"
#include "flow/serialize.h"
#include "flow/Trace.h"

#include <chrono>
#include <cstring>
#include <limits>
#include <memory>
#include <random>
#include <variant>

#include "flow/actorcompiler.h" // This must be the last #include.

#define MEGA_BYTES (1024 * 1024)
#define NANO_SECOND (1000 * 1000 * 1000)

struct WorkloadMetrics {
	double totalEncryptTimeNS;
	double totalDecryptTimeNS;
	double totalKeyDerivationTimeNS;
	int64_t totalBytes;

	void reset() {
		totalEncryptTimeNS = 0;
		totalDecryptTimeNS = 0;
		totalKeyDerivationTimeNS = 0;
		totalBytes = 0;
	}

	WorkloadMetrics() { reset(); }

	double computeEncryptThroughputMBPS() {
		// convert bytes -> MBs & nano-seonds -> seconds
		return (totalBytes * NANO_SECOND) / (totalEncryptTimeNS * MEGA_BYTES);
	}

	double computeDecryptThroughputMBPS() {
		// convert bytes -> MBs & nano-seonds -> seconds
		return (totalBytes * NANO_SECOND) / (totalDecryptTimeNS * MEGA_BYTES);
	}

	void updateKeyDerivationTime(double val) { totalKeyDerivationTimeNS += val; }
	void updateEncryptionTime(double val) { totalEncryptTimeNS += val; }
	void updateDecryptionTime(double val) { totalDecryptTimeNS += val; }
	void updateBytes(int64_t val) { totalBytes += val; }

	void recordMetrics(const std::string& mode, const int numIterations) {
		TraceEvent("EncryptionOpsWorkload")
		    .detail("Mode", mode)
		    .detail("EncryptTimeMS", totalEncryptTimeNS / 1000)
		    .detail("DecryptTimeMS", totalDecryptTimeNS / 1000)
		    .detail("EncryptMBPS", computeEncryptThroughputMBPS())
		    .detail("DecryptMBPS", computeDecryptThroughputMBPS())
		    .detail("KeyDerivationTimeMS", totalKeyDerivationTimeNS / 1000)
		    .detail("TotalBytes", totalBytes)
		    .detail("AvgCommitSize", totalBytes / numIterations);
	}
};

// Workload generator for encryption/decryption operations.
// 1. For every client run, it generate unique random encryptionDomainId range and simulate encryption of
//    either fixed size or variable size payload.
// 2. For each encryption run, it would interact with BlobCipherKeyCache to fetch the desired encryption key,
//    which then is used for encrypting the plaintext payload.
// 3. Encryption operation generates 'encryption header', it is leveraged to decrypt the ciphertext obtained from
//    step#2 (simulate real-world scenario)
//
// Correctness validations:
// -----------------------
// Correctness invariants are validated at various steps:
// 1. Encryption key correctness: as part of performing decryption, BlobCipherKeyCache lookup is done to procure
//    desired encryption key based on: {encryptionDomainId, baseCipherId}; the obtained key is validated against
//    the encryption key used for encrypting the data.
// 2. After encryption, generated 'encryption header' fields are validated, encrypted buffer size and contents are
//    validated.
// 3. After decryption, the obtained deciphertext is validated against the original plaintext payload.
//
// Performance metrics:
// -------------------
// The workload generator profiles below operations across the iterations and logs the details at the end, they are:
// 1. Time spent in encryption key fetch (and derivation) operations.
// 2. Time spent encrypting the buffer (doesn't include key lookup time); also records the throughput in MB/sec.
// 3. Time spent decrypting the buffer (doesn't include key lookup time); also records the throughput in MB/sec.

struct EncryptionOpsWorkload : TestWorkload {
	static constexpr auto NAME = "EncryptionOps";
	int mode;
	int64_t numIterations;
	int pageSize;
	int maxBufSize;
	std::unique_ptr<uint8_t[]> buff;
	int enableTTLTest;
	int minBaseCipherLen;
	int maxBaseCipherLen;

	std::unique_ptr<WorkloadMetrics> metrics;

	EncryptCipherDomainId minDomainId;
	EncryptCipherDomainId maxDomainId;
	EncryptCipherBaseKeyId minBaseCipherId;
	EncryptCipherBaseKeyId headerBaseCipherId;
	EncryptCipherRandomSalt headerRandomSalt;

	EncryptionOpsWorkload(WorkloadContext const& wcx) : TestWorkload(wcx), enableTTLTest(false) {
		mode = getOption(options, "fixedSize"_sr, 1);
		numIterations = getOption(options, "numIterations"_sr, 10);
		pageSize = getOption(options, "pageSize"_sr, 4096);
		maxBufSize = getOption(options, "maxBufSize"_sr, 512 * 1024);
		buff = std::make_unique<uint8_t[]>(maxBufSize);

		// assign unique encryptionDomainId range per workload clients
		minDomainId = wcx.clientId * 100 + mode * 30 + 1;
		maxDomainId = deterministicRandom()->randomInt(minDomainId, minDomainId + 10) + 5;
		minBaseCipherId = 100;
		headerBaseCipherId = wcx.clientId * 100 + 1;

		metrics = std::make_unique<WorkloadMetrics>();

		minBaseCipherLen = deterministicRandom()->randomInt(4, 11);
		maxBaseCipherLen = deterministicRandom()->randomInt(AES_256_KEY_LENGTH, MAX_BASE_CIPHER_LEN + 1);

		ASSERT_LT(minBaseCipherLen, maxBaseCipherLen);

		if (wcx.clientId == 0 && mode == 1) {
			enableTTLTest = true;
		}

		TraceEvent("EncryptionOpsWorkload")
		    .detail("Mode", getModeStr())
		    .detail("MinDomainId", minDomainId)
		    .detail("MaxDomainId", maxDomainId)
		    .detail("EnableTTL", enableTTLTest);
	}

	~EncryptionOpsWorkload() { TraceEvent("EncryptionOpsWorkloadDone").log(); }

	bool isFixedSizePayload() const { return mode == 1; }

	std::string getModeStr() const {
		if (mode == 1) {
			return "FixedSize";
		} else if (mode == 0) {
			return "VariableSize";
		}
		// no other mode supported
		throw internal_error();
	}

	static void generateRandomBaseCipher(const int minLen, const int maxLen, uint8_t* buff, int* retLen) {
		memset(buff, 0, maxLen);
		*retLen = deterministicRandom()->randomInt(minLen, maxLen);
		deterministicRandom()->randomBytes(buff, *retLen);
	}

	void setupCipherEssentials() {
		Reference<BlobCipherKeyCache> cipherKeyCache = BlobCipherKeyCache::getInstance();

		TraceEvent("SetupCipherEssentialsStart").detail("MinDomainId", minDomainId).detail("MaxDomainId", maxDomainId);

		uint8_t buff[maxBaseCipherLen];
		std::vector<Reference<BlobCipherKey>> cipherKeys;
		int cipherLen = 0;
		for (EncryptCipherDomainId id = minDomainId; id <= maxDomainId; id++) {
			generateRandomBaseCipher(minBaseCipherLen, maxBaseCipherLen, &buff[0], &cipherLen);
			const EncryptCipherKeyCheckValue cipherKCV = Sha256KCV().computeKCV(&buff[0], cipherLen);

			cipherKeyCache->insertCipherKey(id,
			                                minBaseCipherId,
			                                buff,
			                                cipherLen,
			                                cipherKCV,
			                                std::numeric_limits<int64_t>::max(),
			                                std::numeric_limits<int64_t>::max());

			ASSERT(cipherLen > 0 && cipherLen <= maxBaseCipherLen);

			cipherKeys = cipherKeyCache->getAllCiphers(id);
			ASSERT_EQ(cipherKeys.size(), 1);
		}

		// insert the Encrypt Header cipherKey; record cipherDetails as getLatestCipher() may not work with multiple
		// test clients
		generateRandomBaseCipher(minBaseCipherLen, maxBaseCipherLen, &buff[0], &cipherLen);
		const EncryptCipherKeyCheckValue cipherKCV = Sha256KCV().computeKCV(&buff[0], cipherLen);
		cipherKeyCache->insertCipherKey(ENCRYPT_HEADER_DOMAIN_ID,
		                                headerBaseCipherId,
		                                buff,
		                                cipherLen,
		                                cipherKCV,
		                                std::numeric_limits<int64_t>::max(),
		                                std::numeric_limits<int64_t>::max());
		Reference<BlobCipherKey> latestCipher = cipherKeyCache->getLatestCipherKey(ENCRYPT_HEADER_DOMAIN_ID);
		ASSERT_EQ(latestCipher->getBaseCipherId(), headerBaseCipherId);
		ASSERT_EQ(memcmp(latestCipher->rawBaseCipher(), buff, cipherLen), 0);
		headerRandomSalt = latestCipher->getSalt();

		TraceEvent("SetupCipherEssentialsDone")
		    .detail("MinDomainId", minDomainId)
		    .detail("MaxDomainId", maxDomainId)
		    .detail("HeaderBaseCipherId", headerBaseCipherId)
		    .detail("HeaderRandomSalt", headerRandomSalt);
	}

	void resetCipherEssentials() {
		TraceEvent("ResetCipherEssentialsStart").detail("Min", minDomainId).detail("Max", maxDomainId);

		Reference<BlobCipherKeyCache> cipherKeyCache = BlobCipherKeyCache::getInstance();
		for (EncryptCipherDomainId id = minDomainId; id <= maxDomainId; id++) {
			cipherKeyCache->resetEncryptDomainId(id);
		}

		cipherKeyCache->resetEncryptDomainId(FDB_DEFAULT_ENCRYPT_DOMAIN_ID);
		cipherKeyCache->resetEncryptDomainId(SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID);
		cipherKeyCache->resetEncryptDomainId(ENCRYPT_HEADER_DOMAIN_ID);

		TraceEvent("ResetCipherEssentialsDone");
	}

	void updateLatestBaseCipher(const EncryptCipherDomainId encryptDomainId,
	                            uint8_t* baseCipher,
	                            int* baseCipherLen,
	                            EncryptCipherBaseKeyId* nextBaseCipherId) {
		Reference<BlobCipherKeyCache> cipherKeyCache = BlobCipherKeyCache::getInstance();
		Reference<BlobCipherKey> cipherKey = cipherKeyCache->getLatestCipherKey(encryptDomainId);
		*nextBaseCipherId = cipherKey->getBaseCipherId() + 1;

		generateRandomBaseCipher(minBaseCipherLen, maxBaseCipherLen, baseCipher, baseCipherLen);

		ASSERT(*baseCipherLen > 0 && *baseCipherLen <= maxBaseCipherLen);
		TraceEvent("UpdateBaseCipher")
		    .detail("DomainId", encryptDomainId)
		    .detail("BaseCipherId", *nextBaseCipherId)
		    .detail("BaseCipherLen", *baseCipherLen)
		    .detail("ExistingBaseCipherId", cipherKey->getBaseCipherId())
		    .detail("ExistingBaseCipherLen", cipherKey->getBaseCipherLen());
	}

	Reference<BlobCipherKey> getEncryptionKey(const EncryptCipherDomainId& domainId,
	                                          const EncryptCipherBaseKeyId& baseCipherId,
	                                          const EncryptCipherRandomSalt& salt) {
		const bool simCacheMiss = deterministicRandom()->randomInt(1, 100) < 15;

		Reference<BlobCipherKeyCache> cipherKeyCache = BlobCipherKeyCache::getInstance();
		Reference<BlobCipherKey> cipherKey = cipherKeyCache->getCipherKey(domainId, baseCipherId, salt);

		if (simCacheMiss) {
			TraceEvent("SimKeyCacheMiss").detail("EncryptDomainId", domainId).detail("BaseCipherId", baseCipherId);
			// simulate KeyCache miss that may happen during decryption; insert a CipherKey with known 'salt'
			cipherKeyCache->insertCipherKey(domainId,
			                                baseCipherId,
			                                cipherKey->rawBaseCipher(),
			                                cipherKey->getBaseCipherLen(),
			                                cipherKey->getSalt(),
			                                std::numeric_limits<int64_t>::max(),
			                                std::numeric_limits<int64_t>::max());
			// Ensure the update was a NOP
			Reference<BlobCipherKey> cKey = cipherKeyCache->getCipherKey(domainId, baseCipherId, salt);
			ASSERT(cKey->isEqual(cipherKey));
		}
		return cipherKey;
	}

	Reference<EncryptBuf> doEncryption(Reference<BlobCipherKey> textCipherKey,
	                                   Reference<BlobCipherKey> headerCipherKey,
	                                   uint8_t* payload,
	                                   int len,
	                                   const EncryptAuthTokenMode authMode,
	                                   const EncryptAuthTokenAlgo authAlgo,
	                                   BlobCipherEncryptHeader* header,
	                                   Arena& arena) {
		uint8_t iv[AES_256_IV_LENGTH];
		deterministicRandom()->randomBytes(&iv[0], AES_256_IV_LENGTH);
		EncryptBlobCipherAes265Ctr encryptor(
		    textCipherKey, headerCipherKey, &iv[0], AES_256_IV_LENGTH, authMode, authAlgo, BlobCipherMetrics::TEST);

		auto start = std::chrono::high_resolution_clock::now();
		Reference<EncryptBuf> encrypted = encryptor.encrypt(payload, len, header, arena);
		auto end = std::chrono::high_resolution_clock::now();

		// validate encrypted buffer size and contents (not matching with plaintext)
		ASSERT_EQ(encrypted->getLogicalSize(), len);
		if (g_network->isSimulated() && ENABLE_MUTATION_TRACKING_WITH_BLOB_CIPHER)
			ASSERT_EQ(memcmp(encrypted->begin(), payload, len), 0);
		else
			ASSERT_NE(memcmp(encrypted->begin(), payload, len), 0);
		ASSERT_EQ(header->flags.headerVersion, EncryptBlobCipherAes265Ctr::ENCRYPT_HEADER_VERSION);

		metrics->updateEncryptionTime(std::chrono::duration<double, std::nano>(end - start).count());
		return encrypted;
	}

	StringRef doEncryption(Reference<BlobCipherKey> textCipherKey,
	                       Reference<BlobCipherKey> headerCipherKey,
	                       uint8_t* payload,
	                       int len,
	                       const EncryptAuthTokenMode authMode,
	                       const EncryptAuthTokenAlgo authAlgo,
	                       BlobCipherEncryptHeaderRef* headerRef,
	                       Arena& arena) {
		uint8_t iv[AES_256_IV_LENGTH];
		deterministicRandom()->randomBytes(&iv[0], AES_256_IV_LENGTH);
		EncryptBlobCipherAes265Ctr encryptor(
		    textCipherKey, headerCipherKey, &iv[0], AES_256_IV_LENGTH, authMode, authAlgo, BlobCipherMetrics::TEST);

		auto start = std::chrono::high_resolution_clock::now();
		StringRef encrypted = encryptor.encrypt(payload, len, headerRef, arena);
		auto end = std::chrono::high_resolution_clock::now();

		// validate encrypted buffer size and contents (not matching with plaintext)
		const uint8_t* headerIV = headerRef->getIV();
		ASSERT_EQ(memcmp(&headerIV[0], &iv[0], AES_256_IV_LENGTH), 0);

		EncryptHeaderCipherDetails validateDetails = headerRef->getCipherDetails();
		ASSERT(validateDetails.textCipherDetails.isValid() &&
		       validateDetails.textCipherDetails == BlobCipherDetails(textCipherKey->getDomainId(),
		                                                              textCipherKey->getBaseCipherId(),
		                                                              textCipherKey->getSalt()));
		if (authMode == ENCRYPT_HEADER_AUTH_TOKEN_MODE_NONE) {
			ASSERT(!validateDetails.headerCipherDetails.present());
		} else {
			ASSERT(validateDetails.headerCipherDetails.present() &&
			       validateDetails.headerCipherDetails.get().isValid() &&
			       validateDetails.headerCipherDetails.get() == BlobCipherDetails(headerCipherKey->getDomainId(),
			                                                                      headerCipherKey->getBaseCipherId(),
			                                                                      headerCipherKey->getSalt()));
		}

		ASSERT_EQ(encrypted.size(), len);
		ASSERT_EQ(headerRef->flagsVersion(), CLIENT_KNOBS->ENCRYPT_HEADER_FLAGS_VERSION);
		if (g_network->isSimulated() && ENABLE_MUTATION_TRACKING_WITH_BLOB_CIPHER)
			ASSERT_EQ(memcmp(encrypted.begin(), payload, len), 0);
		else
			ASSERT_NE(memcmp(encrypted.begin(), payload, len), 0);

		metrics->updateEncryptionTime(std::chrono::duration<double, std::nano>(end - start).count());
		return encrypted;
	}

	void doDecryption(Reference<EncryptBuf> encrypted,
	                  int len,
	                  const BlobCipherEncryptHeader& header,
	                  uint8_t* originalPayload,
	                  Reference<BlobCipherKey> orgCipherKey,
	                  Arena& arena) {
		ASSERT_EQ(header.flags.headerVersion, EncryptBlobCipherAes265Ctr::ENCRYPT_HEADER_VERSION);
		ASSERT_EQ(header.flags.encryptMode, ENCRYPT_CIPHER_MODE_AES_256_CTR);

		Reference<BlobCipherKey> cipherKey = getEncryptionKey(header.cipherTextDetails.encryptDomainId,
		                                                      header.cipherTextDetails.baseCipherId,
		                                                      header.cipherTextDetails.salt);
		Reference<BlobCipherKey> headerCipherKey;
		if (header.flags.authTokenMode != EncryptAuthTokenMode::ENCRYPT_HEADER_AUTH_TOKEN_MODE_NONE) {
			headerCipherKey = getEncryptionKey(header.cipherHeaderDetails.encryptDomainId,
			                                   header.cipherHeaderDetails.baseCipherId,
			                                   header.cipherHeaderDetails.salt);
		}
		ASSERT(cipherKey.isValid());
		ASSERT(cipherKey->isEqual(orgCipherKey));
		ASSERT(headerCipherKey.isValid() ||
		       header.flags.authTokenMode == EncryptAuthTokenMode::ENCRYPT_HEADER_AUTH_TOKEN_MODE_NONE);

		DecryptBlobCipherAes256Ctr decryptor(cipherKey, headerCipherKey, header.iv, BlobCipherMetrics::TEST);

		auto start = std::chrono::high_resolution_clock::now();
		Reference<EncryptBuf> decrypted = decryptor.decrypt(encrypted->begin(), len, header, arena);
		auto end = std::chrono::high_resolution_clock::now();
		metrics->updateDecryptionTime(std::chrono::duration<double, std::nano>(end - start).count());

		// validate decrypted buffer size and contents (matching with original plaintext)
		ASSERT_EQ(decrypted->getLogicalSize(), len);
		ASSERT_EQ(memcmp(decrypted->begin(), originalPayload, len), 0);
	}

	void doDecryption(StringRef encrypted,
	                  int len,
	                  const Standalone<StringRef>& headerStr,
	                  uint8_t* originalPayload,
	                  Reference<BlobCipherKey> orgCipherKey,
	                  Arena& arena) {
		BlobCipherEncryptHeaderRef headerRef = BlobCipherEncryptHeaderRef::fromStringRef(headerStr);

		ASSERT_EQ(headerRef.flagsVersion(), CLIENT_KNOBS->ENCRYPT_HEADER_FLAGS_VERSION);

		// validate flags
		BlobCipherDetails textCipherDetails;
		BlobCipherDetails headerCipherDetails;
		uint8_t iv[AES_256_IV_LENGTH];
		if (std::holds_alternative<AesCtrNoAuth>(headerRef.algoHeader)) {
			AesCtrNoAuth noAuth = std::get<AesCtrNoAuth>(headerRef.algoHeader);
			memcpy(&iv[0], &noAuth.v1.iv[0], AES_256_IV_LENGTH);
			textCipherDetails = noAuth.v1.cipherTextDetails;
			headerCipherDetails = BlobCipherDetails();
		} else if (std::holds_alternative<AesCtrWithHmac>(headerRef.algoHeader)) {
			AesCtrWithHmac hmacSha = std::get<AesCtrWithHmac>(headerRef.algoHeader);
			memcpy(&iv[0], &hmacSha.v1.iv[0], AES_256_IV_LENGTH);
			textCipherDetails = hmacSha.v1.cipherTextDetails;
			headerCipherDetails = hmacSha.v1.cipherHeaderDetails;
		} else {
			ASSERT(std::holds_alternative<AesCtrWithCmac>(headerRef.algoHeader));
			AesCtrWithCmac aesCmac = std::get<AesCtrWithCmac>(headerRef.algoHeader);
			memcpy(&iv[0], &aesCmac.v1.iv[0], AES_256_IV_LENGTH);
			textCipherDetails = aesCmac.v1.cipherTextDetails;
			headerCipherDetails = aesCmac.v1.cipherHeaderDetails;
		}
		Reference<BlobCipherKey> cipherKey =
		    getEncryptionKey(textCipherDetails.encryptDomainId, textCipherDetails.baseCipherId, textCipherDetails.salt);
		Reference<BlobCipherKey> headerCipherKey =
		    !headerCipherDetails.isValid()
		        ? Reference<BlobCipherKey>() // no authentication mode cipher header-key is not needed
		        : getEncryptionKey(
		              headerCipherDetails.encryptDomainId, headerCipherDetails.baseCipherId, headerCipherDetails.salt);
		ASSERT(cipherKey.isValid());
		ASSERT(cipherKey->isEqual(orgCipherKey));

		DecryptBlobCipherAes256Ctr decryptor(cipherKey, headerCipherKey, &iv[0], BlobCipherMetrics::TEST);

		auto start = std::chrono::high_resolution_clock::now();
		StringRef decrypted = decryptor.decrypt(encrypted.begin(), len, headerRef, arena);
		auto end = std::chrono::high_resolution_clock::now();
		metrics->updateDecryptionTime(std::chrono::duration<double, std::nano>(end - start).count());

		// validate decrypted buffer size and contents (matching with original plaintext)
		ASSERT_EQ(decrypted.size(), len);
		ASSERT_EQ(memcmp(decrypted.begin(), originalPayload, len), 0);
	}

	void testBlobCipherKeyCacheOps() {
		uint8_t baseCipher[maxBaseCipherLen];
		int baseCipherLen = 0;
		EncryptCipherBaseKeyId nextBaseCipherId;

		// Setup encryptDomainIds and corresponding baseCipher details
		setupCipherEssentials();

		for (int i = 0; i < numIterations; i++) {
			Arena tmpArena;
			bool updateBaseCipher = deterministicRandom()->randomInt(1, 100) < 5;

			// Step-1: Encryption key derivation, caching the cipher for later use
			Reference<BlobCipherKeyCache> cipherKeyCache = BlobCipherKeyCache::getInstance();

			// randomly select a domainId
			const EncryptCipherDomainId encryptDomainId = deterministicRandom()->randomInt(minDomainId, maxDomainId);
			ASSERT(encryptDomainId >= minDomainId && encryptDomainId <= maxDomainId);

			if (updateBaseCipher) {
				// simulate baseCipherId getting refreshed/updated
				updateLatestBaseCipher(encryptDomainId, &baseCipher[0], &baseCipherLen, &nextBaseCipherId);
				const EncryptCipherKeyCheckValue baseCipherKCV = Sha256KCV().computeKCV(&baseCipher[0], baseCipherLen);
				cipherKeyCache->insertCipherKey(encryptDomainId,
				                                nextBaseCipherId,
				                                &baseCipher[0],
				                                baseCipherLen,
				                                baseCipherKCV,
				                                std::numeric_limits<int64_t>::max(),
				                                std::numeric_limits<int64_t>::max());
			}

			auto start = std::chrono::high_resolution_clock::now();
			Reference<BlobCipherKey> cipherKey = cipherKeyCache->getLatestCipherKey(encryptDomainId);
			// Each client working with their own version of encryptHeaderCipherKey, avoid using getLatest()
			Reference<BlobCipherKey> headerCipherKey =
			    cipherKeyCache->getCipherKey(ENCRYPT_HEADER_DOMAIN_ID, headerBaseCipherId, headerRandomSalt);
			auto end = std::chrono::high_resolution_clock::now();
			metrics->updateKeyDerivationTime(std::chrono::duration<double, std::nano>(end - start).count());

			// Validate sanity of "getLatestCipher", especially when baseCipher gets updated
			if (updateBaseCipher) {
				ASSERT_EQ(cipherKey->getBaseCipherId(), nextBaseCipherId);
				ASSERT_EQ(cipherKey->getBaseCipherLen(), baseCipherLen);
				ASSERT_EQ(memcmp(cipherKey->rawBaseCipher(), baseCipher, baseCipherLen), 0);
			}

			int dataLen = isFixedSizePayload() ? pageSize : deterministicRandom()->randomInt(100, maxBufSize);
			deterministicRandom()->randomBytes(buff.get(), dataLen);

			// Encrypt the payload - generates BlobCipherEncryptHeader to assist decryption later
			const EncryptAuthTokenMode authMode = getRandomAuthTokenMode();
			const EncryptAuthTokenAlgo authAlgo = authMode == EncryptAuthTokenMode::ENCRYPT_HEADER_AUTH_TOKEN_MODE_NONE
			                                          ? EncryptAuthTokenAlgo::ENCRYPT_HEADER_AUTH_TOKEN_ALGO_NONE
			                                          : getRandomAuthTokenAlgo();

			try {
				BlobCipherEncryptHeader header;
				Reference<EncryptBuf> encrypted = doEncryption(
				    cipherKey, headerCipherKey, buff.get(), dataLen, authMode, authAlgo, &header, tmpArena);

				// Decrypt the payload - parses the BlobCipherEncryptHeader, fetch corresponding cipherKey and
				// decrypt
				doDecryption(encrypted, dataLen, header, buff.get(), cipherKey, tmpArena);

				{
					BlobCipherEncryptHeaderRef headerRef;
					StringRef encrypted = doEncryption(
					    cipherKey, headerCipherKey, buff.get(), dataLen, authMode, authAlgo, &headerRef, tmpArena);
					// simulate 'header' on-disk read, serialize buffer and deserialize on decryption
					Standalone<StringRef> serHeader = BlobCipherEncryptHeaderRef::toStringRef(headerRef);

					// Decrypt the payload - parses the BlobCipherEncryptHeader, fetch corresponding cipherKey and
					// decrypt
					doDecryption(encrypted, dataLen, serHeader, buff.get(), cipherKey, tmpArena);
				}
			} catch (Error& e) {
				TraceEvent("Failed")
				    .detail("DomainId", encryptDomainId)
				    .detail("BaseCipherId", cipherKey->getBaseCipherId())
				    .detail("AuthTokenMode", authMode)
				    .detail("AuthTokenAlgo", authAlgo);
				throw;
			}

			metrics->updateBytes(dataLen);
		}

		// Cleanup cipherKeys
		resetCipherEssentials();
	}

	static void compareCipherDetails(Reference<BlobCipherKey> cipherKey,
	                                 const EncryptCipherDomainId domId,
	                                 const EncryptCipherBaseKeyId baseCipherId,
	                                 const uint8_t* baseCipher,
	                                 const int baseCipherLen,
	                                 const int64_t refreshAt,
	                                 const int64_t expAt) {
		ASSERT(cipherKey.isValid());
		ASSERT_EQ(cipherKey->getDomainId(), domId);
		ASSERT_EQ(cipherKey->getBaseCipherId(), baseCipherId);
		ASSERT_EQ(memcmp(cipherKey->rawBaseCipher(), baseCipher, baseCipherLen), 0);
		ASSERT_EQ(cipherKey->getRefreshAtTS(), refreshAt);
		ASSERT_EQ(cipherKey->getExpireAtTS(), expAt);
	}

																#line 565 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
// This generated class is to be used only via testBlobCipherKeyCacheTTL()
															#line 563 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
template <class TestBlobCipherKeyCacheTTLActor>
															#line 563 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
class TestBlobCipherKeyCacheTTLActorState {
															#line 571 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
public:
															#line 563 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	TestBlobCipherKeyCacheTTLActorState(EncryptionOpsWorkload* const& self) 
															#line 563 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
															#line 563 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		 : self(self),
															#line 564 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		   cipherKeyCache(BlobCipherKeyCache::getInstance()),
															#line 566 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		   domId(deterministicRandom()->randomInt(120000, 150000)),
															#line 567 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		   baseCipherId(deterministicRandom()->randomInt(786, 1024)),
															#line 568 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		   baseCipherLen(deterministicRandom()->randomInt(4, MAX_BASE_CIPHER_LEN + 1)),
															#line 569 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		   baseCipher(std::make_unique<uint8_t[]>(baseCipherLen)),
															#line 570 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		   baseCipherKCV(),
															#line 571 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		   cipherKey(),
															#line 572 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		   salt(),
															#line 573 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		   refreshAt(),
															#line 574 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		   expAt()
															#line 598 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
	{
		fdb_probe_actor_create("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this));

	}
	~TestBlobCipherKeyCacheTTLActorState() 
	{
		fdb_probe_actor_destroy("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 576 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			TraceEvent("TestBlobCipherCacheTTLStart").detail("DomId", domId);
															#line 578 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			deterministicRandom()->randomBytes(baseCipher.get(), baseCipherLen);
															#line 579 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			baseCipherKCV = Sha256KCV().computeKCV(baseCipher.get(), baseCipherLen);
															#line 582 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			refreshAt = std::numeric_limits<int64_t>::max();
															#line 583 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			expAt = std::numeric_limits<int64_t>::max();
															#line 584 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			cipherKeyCache->insertCipherKey( domId, baseCipherId, baseCipher.get(), baseCipherLen, baseCipherKCV, refreshAt, expAt);
															#line 586 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			cipherKey = cipherKeyCache->getLatestCipherKey(domId);
															#line 587 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			compareCipherDetails(cipherKey, domId, baseCipherId, baseCipher.get(), baseCipherLen, refreshAt, expAt);
															#line 589 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			TraceEvent("TestBlobCipherCacheTTLNonRevocableNoExpiry").detail("DomId", domId);
															#line 592 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			baseCipherId_1 = baseCipherId + 1;
															#line 593 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			refreshAt = now() + 5;
															#line 594 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			cipherKeyCache->insertCipherKey( domId, baseCipherId_1, baseCipher.get(), baseCipherLen, baseCipherKCV, refreshAt, expAt);
															#line 596 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			cipherKey = cipherKeyCache->getLatestCipherKey(domId);
															#line 597 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			ASSERT(cipherKey.isValid());
															#line 598 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			compareCipherDetails(cipherKey, domId, baseCipherId_1, baseCipher.get(), baseCipherLen, refreshAt, expAt);
															#line 599 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			salt = cipherKey->getSalt();
															#line 600 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delayUntil(refreshAt);
															#line 600 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			if (static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 647 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state = 1;
															#line 600 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TestBlobCipherKeyCacheTTLActor, 0, Void >*>(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)));
															#line 652 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TestBlobCipherKeyCacheTTLActorState();
		static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 602 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getLatestCipherKey(domId);
															#line 603 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(!cipherKey.isValid());
															#line 604 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getCipherKey(domId, baseCipherId_1, salt);
															#line 605 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(cipherKey.isValid());
															#line 606 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		compareCipherDetails(cipherKey, domId, baseCipherId_1, baseCipher.get(), baseCipherLen, refreshAt, expAt);
															#line 608 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		TraceEvent("TestBlobCipherCacheTTLNonRevocableWithExpiry").detail("DomId", domId);
															#line 611 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		baseCipherId_2 = baseCipherId + 2;
															#line 612 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		refreshAt = now() + 5;
															#line 613 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		expAt = refreshAt + 5;
															#line 614 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKeyCache->insertCipherKey( domId, baseCipherId_2, baseCipher.get(), baseCipherLen, baseCipherKCV, refreshAt, expAt);
															#line 616 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getLatestCipherKey(domId);
															#line 617 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(cipherKey.isValid());
															#line 618 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		compareCipherDetails(cipherKey, domId, baseCipherId_2, baseCipher.get(), baseCipherLen, refreshAt, expAt);
															#line 619 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		salt = cipherKey->getSalt();
															#line 620 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delayUntil(refreshAt);
															#line 620 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		if (static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 705 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state = 2;
															#line 620 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestBlobCipherKeyCacheTTLActor, 1, Void >*>(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)));
															#line 710 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 602 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getLatestCipherKey(domId);
															#line 603 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(!cipherKey.isValid());
															#line 604 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getCipherKey(domId, baseCipherId_1, salt);
															#line 605 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(cipherKey.isValid());
															#line 606 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		compareCipherDetails(cipherKey, domId, baseCipherId_1, baseCipher.get(), baseCipherLen, refreshAt, expAt);
															#line 608 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		TraceEvent("TestBlobCipherCacheTTLNonRevocableWithExpiry").detail("DomId", domId);
															#line 611 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		baseCipherId_2 = baseCipherId + 2;
															#line 612 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		refreshAt = now() + 5;
															#line 613 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		expAt = refreshAt + 5;
															#line 614 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKeyCache->insertCipherKey( domId, baseCipherId_2, baseCipher.get(), baseCipherLen, baseCipherKCV, refreshAt, expAt);
															#line 616 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getLatestCipherKey(domId);
															#line 617 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(cipherKey.isValid());
															#line 618 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		compareCipherDetails(cipherKey, domId, baseCipherId_2, baseCipher.get(), baseCipherLen, refreshAt, expAt);
															#line 619 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		salt = cipherKey->getSalt();
															#line 620 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delayUntil(refreshAt);
															#line 620 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		if (static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 749 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state = 2;
															#line 620 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TestBlobCipherKeyCacheTTLActor, 1, Void >*>(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)));
															#line 754 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state > 0) static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state = 0;
		static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->ActorCallback< TestBlobCipherKeyCacheTTLActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestBlobCipherKeyCacheTTLActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TestBlobCipherKeyCacheTTLActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TestBlobCipherKeyCacheTTLActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 622 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getLatestCipherKey(domId);
															#line 623 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(!cipherKey.isValid());
															#line 624 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getCipherKey(domId, baseCipherId_2, salt);
															#line 625 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(cipherKey.isValid());
															#line 626 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		compareCipherDetails(cipherKey, domId, baseCipherId_2, baseCipher.get(), baseCipherLen, refreshAt, expAt);
															#line 627 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delayUntil(expAt);
															#line 627 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		if (static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 850 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state = 3;
															#line 627 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TestBlobCipherKeyCacheTTLActor, 2, Void >*>(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)));
															#line 855 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 622 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getLatestCipherKey(domId);
															#line 623 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(!cipherKey.isValid());
															#line 624 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getCipherKey(domId, baseCipherId_2, salt);
															#line 625 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(cipherKey.isValid());
															#line 626 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		compareCipherDetails(cipherKey, domId, baseCipherId_2, baseCipher.get(), baseCipherLen, refreshAt, expAt);
															#line 627 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		StrictFuture<Void> __when_expr_2 = delayUntil(expAt);
															#line 627 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		if (static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 876 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state = 3;
															#line 627 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< TestBlobCipherKeyCacheTTLActor, 2, Void >*>(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)));
															#line 881 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state > 0) static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state = 0;
		static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->ActorCallback< TestBlobCipherKeyCacheTTLActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestBlobCipherKeyCacheTTLActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TestBlobCipherKeyCacheTTLActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TestBlobCipherKeyCacheTTLActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 629 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getLatestCipherKey(domId);
															#line 630 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(!cipherKey.isValid());
															#line 631 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getCipherKey(domId, baseCipherId_2, salt);
															#line 632 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(!cipherKey.isValid());
															#line 634 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		TraceEvent("TestBlobCipherCacheTTLEnd").detail("DomId", domId);
															#line 635 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		if (!static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestBlobCipherKeyCacheTTLActorState(); static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->destroy(); return 0; }
															#line 975 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		new (&static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestBlobCipherKeyCacheTTLActorState();
		static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 629 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getLatestCipherKey(domId);
															#line 630 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(!cipherKey.isValid());
															#line 631 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		cipherKey = cipherKeyCache->getCipherKey(domId, baseCipherId_2, salt);
															#line 632 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		ASSERT(!cipherKey.isValid());
															#line 634 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		TraceEvent("TestBlobCipherCacheTTLEnd").detail("DomId", domId);
															#line 635 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		if (!static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TestBlobCipherKeyCacheTTLActorState(); static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->destroy(); return 0; }
															#line 997 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		new (&static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TestBlobCipherKeyCacheTTLActorState();
		static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state > 0) static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->actor_wait_state = 0;
		static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->ActorCallback< TestBlobCipherKeyCacheTTLActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TestBlobCipherKeyCacheTTLActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< TestBlobCipherKeyCacheTTLActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< TestBlobCipherKeyCacheTTLActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 563 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	EncryptionOpsWorkload* self;
															#line 564 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	Reference<BlobCipherKeyCache> cipherKeyCache;
															#line 566 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	EncryptCipherDomainId domId;
															#line 567 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	EncryptCipherBaseKeyId baseCipherId;
															#line 568 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	int baseCipherLen;
															#line 569 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	std::unique_ptr<uint8_t[]> baseCipher;
															#line 570 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	EncryptCipherKeyCheckValue baseCipherKCV;
															#line 571 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	Reference<BlobCipherKey> cipherKey;
															#line 572 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	EncryptCipherRandomSalt salt;
															#line 573 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	int64_t refreshAt;
															#line 574 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	int64_t expAt;
															#line 592 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	EncryptCipherBaseKeyId baseCipherId_1;
															#line 611 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	EncryptCipherBaseKeyId baseCipherId_2;
															#line 1106 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
};
// This generated class is to be used only via testBlobCipherKeyCacheTTL()
															#line 563 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
class TestBlobCipherKeyCacheTTLActor final : public Actor<Void>, public ActorCallback< TestBlobCipherKeyCacheTTLActor, 0, Void >, public ActorCallback< TestBlobCipherKeyCacheTTLActor, 1, Void >, public ActorCallback< TestBlobCipherKeyCacheTTLActor, 2, Void >, public FastAllocated<TestBlobCipherKeyCacheTTLActor>, public TestBlobCipherKeyCacheTTLActorState<TestBlobCipherKeyCacheTTLActor> {
															#line 1111 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
public:
	using FastAllocated<TestBlobCipherKeyCacheTTLActor>::operator new;
	using FastAllocated<TestBlobCipherKeyCacheTTLActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9082895264481608448UL, 7560573698036158976UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TestBlobCipherKeyCacheTTLActor, 0, Void >;
friend struct ActorCallback< TestBlobCipherKeyCacheTTLActor, 1, Void >;
friend struct ActorCallback< TestBlobCipherKeyCacheTTLActor, 2, Void >;
															#line 563 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	TestBlobCipherKeyCacheTTLActor(EncryptionOpsWorkload* const& self) 
															#line 1130 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		 : Actor<Void>(),
		   TestBlobCipherKeyCacheTTLActorState<TestBlobCipherKeyCacheTTLActor>(self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4284139672280799232UL, 4639925176254949120UL);
		ActorExecutionContextHelper __helper(static_cast<TestBlobCipherKeyCacheTTLActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("testBlobCipherKeyCacheTTL");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("testBlobCipherKeyCacheTTL", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TestBlobCipherKeyCacheTTLActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TestBlobCipherKeyCacheTTLActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< TestBlobCipherKeyCacheTTLActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 563 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
[[nodiscard]] Future<Void> testBlobCipherKeyCacheTTL( EncryptionOpsWorkload* const& self ) {
															#line 563 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	return Future<Void>(new TestBlobCipherKeyCacheTTLActor(self));
															#line 1164 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
}

#line 637 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"

	Future<Void> setup(Database const& ctx) override { return Void(); }

	Future<Void> start(Database const& cx) override { return _start(cx, this); }

																#line 1173 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
// This generated class is to be used only via _start()
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
template <class _startActor>
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
class _startActorState {
															#line 1179 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
public:
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	_startActorState(Database const& cx,EncryptionOpsWorkload* const& self) 
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		 : cx(cx),
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		   self(self)
															#line 1188 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
	{
		fdb_probe_actor_create("_start", reinterpret_cast<unsigned long>(this));

	}
	~_startActorState() 
	{
		fdb_probe_actor_destroy("_start", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 643 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			self->testBlobCipherKeyCacheOps();
															#line 644 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
			if (self->enableTTLTest)
															#line 1205 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
			{
															#line 645 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
				StrictFuture<Void> __when_expr_0 = self->testBlobCipherKeyCacheTTL(self);
															#line 645 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
				if (static_cast<_startActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1211 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<_startActor*>(this)->actor_wait_state = 1;
															#line 645 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _startActor, 0, Void >*>(static_cast<_startActor*>(this)));
															#line 1216 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_startActorState();
		static_cast<_startActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 647 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
		if (!static_cast<_startActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_startActorState(); static_cast<_startActor*>(this)->destroy(); return 0; }
															#line 1244 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		new (&static_cast<_startActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_startActorState();
		static_cast<_startActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_startActor*>(this)->actor_wait_state > 0) static_cast<_startActor*>(this)->actor_wait_state = 0;
		static_cast<_startActor*>(this)->ActorCallback< _startActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< _startActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _startActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _startActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	Database cx;
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	EncryptionOpsWorkload* self;
															#line 1343 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
};
// This generated class is to be used only via _start()
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
class _startActor final : public Actor<Void>, public ActorCallback< _startActor, 0, Void >, public FastAllocated<_startActor>, public _startActorState<_startActor> {
															#line 1348 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
public:
	using FastAllocated<_startActor>::operator new;
	using FastAllocated<_startActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16853390886812121088UL, 434362193176809472UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _startActor, 0, Void >;
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	_startActor(Database const& cx,EncryptionOpsWorkload* const& self) 
															#line 1365 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
		 : Actor<Void>(),
		   _startActorState<_startActor>(cx, self),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_start", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16516870674810565632UL, 12235738791954228224UL);
		ActorExecutionContextHelper __helper(static_cast<_startActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_start");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_start", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _startActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
[[nodiscard]] Future<Void> _start( Database const& cx, EncryptionOpsWorkload* const& self ) {
															#line 642 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"
	return Future<Void>(new _startActor(cx, self));
															#line 1397 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/EncryptionOps.actor.g.cpp"
}

#line 649 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbserver/workloads/EncryptionOps.actor.cpp"

	Future<bool> check(Database const& cx) override { return true; }

	void getMetrics(std::vector<PerfMetric>& m) override { metrics->recordMetrics(getModeStr(), numIterations); }
};

WorkloadFactory<EncryptionOpsWorkload> EncryptionOpsWorkloadFactory;
