#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
/*
 * RESTUtils.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/RESTUtils.h"
#include "fdbclient/Knobs.h"

#include "flow/flat_buffers.h"
#include "flow/UnitTest.h"
#include "flow/IConnection.h"

#include <boost/algorithm/string.hpp>
#include <queue>

#include "flow/actorcompiler.h" // always the last include

// RESTConnectionPool destructor implementation
RESTConnectionPool::~RESTConnectionPool() {
	// In simulation, explicitly close all pooled connections before destruction.
	// This satisfies Sim2Conn's assertion: !opened || closedByCaller
	// Without this, connections would be destroyed without being closed, causing assertion failures.
	if (g_network && g_network->isSimulated()) {
		for (auto& kv : connectionPoolMap) {
			while (!kv.second.empty()) {
				ReusableConnection& rconn = kv.second.front();
				if (rconn.conn.isValid()) {
					rconn.conn->close();
				}
				kv.second.pop();
			}
		}
	}
}

const std::unordered_map<std::string, RESTConnectionType> RESTConnectionType::supportedConnTypes = {
	{ "http", RESTConnectionType("http", RESTConnectionType::NOT_SECURE_CONNECTION) },
	{ "https", RESTConnectionType("https", RESTConnectionType::SECURE_CONNECTION) }
};

RESTConnectionType RESTConnectionType::getConnectionType(const std::string& protocol) {
	auto itr = RESTConnectionType::supportedConnTypes.find(protocol);
	if (itr == RESTConnectionType::supportedConnTypes.end()) {
		TraceEvent("RESTConnectionTypeUnsupportedPrototocol").detail("Protocol", protocol);
		CODE_PROBE(true, "REST URI unsupported protocol");
		throw rest_unsupported_protocol();
	}
	return itr->second;
}

bool RESTConnectionType::isProtocolSupported(const std::string& protocol) {
	auto itr = RESTConnectionType::supportedConnTypes.find(protocol);
	return itr != RESTConnectionType::supportedConnTypes.end();
}

bool RESTConnectionType::isSecure(const std::string& protocol) {
	auto itr = RESTConnectionType::supportedConnTypes.find(protocol);
	if (itr == RESTConnectionType::supportedConnTypes.end()) {
		TraceEvent("RESTConnectionTypeUnsupportedPrototocol").detail("Protocol", protocol);
		throw rest_unsupported_protocol();
	}
	return itr->second.secure == RESTConnectionType::SECURE_CONNECTION;
}

RESTClientKnobs::RESTClientKnobs() {
	connection_pool_size = FLOW_KNOBS->RESTCLIENT_MAX_CONNECTIONPOOL_SIZE;
	connect_tries = FLOW_KNOBS->RESTCLIENT_CONNECT_TRIES;
	connect_timeout = FLOW_KNOBS->RESTCLIENT_CONNECT_TIMEOUT;
	max_connection_life = FLOW_KNOBS->RESTCLIENT_MAX_CONNECTION_LIFE;
	request_tries = FLOW_KNOBS->RESTCLIENT_REQUEST_TRIES;
	request_timeout_secs = FLOW_KNOBS->RESTCLIENT_REQUEST_TIMEOUT_SEC;

	knobMap["connection_pool_size"] = std::addressof(connection_pool_size);
	knobMap["pz"] = std::addressof(connection_pool_size);
	knobMap["connect_tries"] = std::addressof(connect_tries);
	knobMap["ct"] = std::addressof(connect_tries);
	knobMap["connect_timeout"] = std::addressof(connect_timeout);
	knobMap["cto"] = std::addressof(connect_timeout);
	knobMap["max_connection_life"] = std::addressof(max_connection_life);
	knobMap["mcl"] = std::addressof(max_connection_life);
	knobMap["request_tries"] = std::addressof(request_tries);
	knobMap["rt"] = std::addressof(request_tries);
	knobMap["request_timeout_secs"] = std::addressof(request_timeout_secs);
	knobMap["rtom"] = std::addressof(request_timeout_secs);
}

void RESTClientKnobs::set(const std::unordered_map<std::string, int>& knobSettings) {

	for (const auto& itr : knobSettings) {
		const auto& kItr = RESTClientKnobs::knobMap.find(itr.first);
		if (kItr == RESTClientKnobs::knobMap.end()) {
			TraceEvent("RESTClientInvalidKnobName").detail("KnobName", itr.first);
			throw rest_invalid_rest_client_knob();
		}
		ASSERT_EQ(itr.first.compare(kItr->first), 0);
		*(kItr->second) = itr.second;
	}
}

std::unordered_map<std::string, int> RESTClientKnobs::get() const {
	std::unordered_map<std::string, int> details = {
		{ "connection_pool_size", connection_pool_size },
		{ "connect_tries", connect_tries },
		{ "connect_timeout", connect_timeout },
		{ "max_connection_life", max_connection_life },
		{ "request_tries", request_tries },
		{ "request_timeout_secs", request_timeout_secs },
	};

	return details;
}

															#line 130 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
namespace {
// This generated class is to be used only via connect_impl()
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
template <class Connect_implActor>
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class Connect_implActorState {
															#line 137 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	Connect_implActorState(Reference<RESTConnectionPool> const& connectionPool,RESTConnectionPoolKey const& connectKey,bool const& isSecure,int const& maxConnLife) 
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		 : connectionPool(connectionPool),
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		   connectKey(connectKey),
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		   isSecure(isSecure),
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		   maxConnLife(maxConnLife)
															#line 150 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
	{
		fdb_probe_actor_create("connect_impl", reinterpret_cast<unsigned long>(this));

	}
	~Connect_implActorState() 
	{
		fdb_probe_actor_destroy("connect_impl", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 133 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::VERBOSE)
															#line 165 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			{
															#line 134 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				TraceEvent("RESTUtilConnectStart") .detail("Host", connectKey.first) .detail("Service", connectKey.second) .detail("IsSecure", isSecure) .detail("ConnectPoolNumKeys", connectionPool->connectionPoolMap.size());
															#line 169 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			}
															#line 141 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			auto poolItr = connectionPool->connectionPoolMap.find(connectKey);
															#line 142 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			for(;poolItr != connectionPool->connectionPoolMap.end() && !poolItr->second.empty();) {
															#line 143 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				RESTConnectionPool::ReusableConnection rconn = poolItr->second.front();
															#line 144 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				poolItr->second.pop();
															#line 146 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				if (rconn.expirationTime > now())
															#line 181 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
				{
															#line 147 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
					if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::DEBUG)
															#line 185 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
					{
															#line 148 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
						TraceEvent("RESTUtilReuseConn") .detail("Host", connectKey.first) .detail("Service", connectKey.second) .detail("RemoteEndpoint", rconn.conn->getPeerAddress()) .detail("ExpireIn", rconn.expirationTime - now()) .detail("NumConnsInPool", poolItr->second.size());
															#line 189 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
					}
															#line 155 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
					if (!static_cast<Connect_implActor*>(this)->SAV<RESTConnectionPool::ReusableConnection>::futures) { (void)(rconn); this->~Connect_implActorState(); static_cast<Connect_implActor*>(this)->destroy(); return 0; }
															#line 193 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
					new (&static_cast<Connect_implActor*>(this)->SAV< RESTConnectionPool::ReusableConnection >::value()) RESTConnectionPool::ReusableConnection(rconn);
					this->~Connect_implActorState();
					static_cast<Connect_implActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
			}
															#line 159 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT(poolItr == connectionPool->connectionPoolMap.end() || poolItr->second.empty());
															#line 162 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			StrictFuture<Reference<IConnection>> __when_expr_0 = INetworkConnections::net()->connect(connectKey.first, connectKey.second, isSecure);
															#line 162 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 206 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<Connect_implActor*>(this)->actor_wait_state = 1;
															#line 162 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 0, Reference<IConnection> >*>(static_cast<Connect_implActor*>(this)));
															#line 211 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~Connect_implActorState();
		static_cast<Connect_implActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 164 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		StrictFuture<Void> __when_expr_1 = conn->connectHandshake();
															#line 164 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		if (static_cast<Connect_implActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 236 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<Connect_implActor*>(this)->actor_wait_state = 2;
															#line 164 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< Connect_implActor, 1, Void >*>(static_cast<Connect_implActor*>(this)));
															#line 241 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Reference<IConnection> const& __conn,int loopDepth) 
	{
															#line 162 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		conn = __conn;
															#line 250 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Reference<IConnection> && __conn,int loopDepth) 
	{
		conn = std::move(__conn);
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 0, Reference<IConnection> >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 0, Reference<IConnection> >*,Reference<IConnection> const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 0, Reference<IConnection> >*,Reference<IConnection> && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 0, Reference<IConnection> >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 166 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		TraceEvent("RESTTUilCreateNewConn") .suppressFor(60) .detail("Host", connectKey.first) .detail("Service", connectKey.second) .detail("RemoteEndpoint", conn->getPeerAddress()) .detail("ConnPoolSize", connectionPool->connectionPoolMap.size());
															#line 173 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		if (!static_cast<Connect_implActor*>(this)->SAV<RESTConnectionPool::ReusableConnection>::futures) { (void)(RESTConnectionPool::ReusableConnection({ conn, now() + maxConnLife })); this->~Connect_implActorState(); static_cast<Connect_implActor*>(this)->destroy(); return 0; }
															#line 331 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		new (&static_cast<Connect_implActor*>(this)->SAV< RESTConnectionPool::ReusableConnection >::value()) RESTConnectionPool::ReusableConnection(RESTConnectionPool::ReusableConnection({ conn, now() + maxConnLife }));
		this->~Connect_implActorState();
		static_cast<Connect_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 166 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		TraceEvent("RESTTUilCreateNewConn") .suppressFor(60) .detail("Host", connectKey.first) .detail("Service", connectKey.second) .detail("RemoteEndpoint", conn->getPeerAddress()) .detail("ConnPoolSize", connectionPool->connectionPoolMap.size());
															#line 173 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		if (!static_cast<Connect_implActor*>(this)->SAV<RESTConnectionPool::ReusableConnection>::futures) { (void)(RESTConnectionPool::ReusableConnection({ conn, now() + maxConnLife })); this->~Connect_implActorState(); static_cast<Connect_implActor*>(this)->destroy(); return 0; }
															#line 345 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		new (&static_cast<Connect_implActor*>(this)->SAV< RESTConnectionPool::ReusableConnection >::value()) RESTConnectionPool::ReusableConnection(RESTConnectionPool::ReusableConnection({ conn, now() + maxConnLife }));
		this->~Connect_implActorState();
		static_cast<Connect_implActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<Connect_implActor*>(this)->actor_wait_state > 0) static_cast<Connect_implActor*>(this)->actor_wait_state = 0;
		static_cast<Connect_implActor*>(this)->ActorCallback< Connect_implActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< Connect_implActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< Connect_implActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	Reference<RESTConnectionPool> connectionPool;
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	RESTConnectionPoolKey connectKey;
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	bool isSecure;
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	int maxConnLife;
															#line 162 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	Reference<IConnection> conn;
															#line 438 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
};
// This generated class is to be used only via connect_impl()
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class Connect_implActor final : public Actor<RESTConnectionPool::ReusableConnection>, public ActorCallback< Connect_implActor, 0, Reference<IConnection> >, public ActorCallback< Connect_implActor, 1, Void >, public FastAllocated<Connect_implActor>, public Connect_implActorState<Connect_implActor> {
															#line 443 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
	using FastAllocated<Connect_implActor>::operator new;
	using FastAllocated<Connect_implActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(511436827003500288UL, 10775589030820715520UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<RESTConnectionPool::ReusableConnection>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< Connect_implActor, 0, Reference<IConnection> >;
friend struct ActorCallback< Connect_implActor, 1, Void >;
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	Connect_implActor(Reference<RESTConnectionPool> const& connectionPool,RESTConnectionPoolKey const& connectKey,bool const& isSecure,int const& maxConnLife) 
															#line 461 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		 : Actor<RESTConnectionPool::ReusableConnection>(),
		   Connect_implActorState<Connect_implActor>(connectionPool, connectKey, isSecure, maxConnLife),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("connect_impl", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7092978256320716288UL, 13901349198287939840UL);
		ActorExecutionContextHelper __helper(static_cast<Connect_implActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("connect_impl");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("connect_impl", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< Connect_implActor, 0, Reference<IConnection> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< Connect_implActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
[[nodiscard]] Future<RESTConnectionPool::ReusableConnection> connect_impl( Reference<RESTConnectionPool> const& connectionPool, RESTConnectionPoolKey const& connectKey, bool const& isSecure, int const& maxConnLife ) {
															#line 128 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	return Future<RESTConnectionPool::ReusableConnection>(new Connect_implActor(connectionPool, connectKey, isSecure, maxConnLife));
															#line 495 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
}

#line 175 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"

Future<RESTConnectionPool::ReusableConnection> RESTConnectionPool::connect(RESTConnectionPoolKey connectKey,
                                                                           const bool isSecure,
                                                                           const int maxConnLife) {
	return connect_impl(Reference<RESTConnectionPool>::addRef(this), connectKey, isSecure, maxConnLife);
}

void RESTConnectionPool::returnConnection(RESTConnectionPoolKey connectKey,
                                          ReusableConnection& rconn,
                                          const int maxConnections) {
	if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::VERBOSE) {
		TraceEvent("RESTUtilReturnConnStart")
		    .detail("Host", connectKey.first)
		    .detail("Service", connectKey.second)
		    .detail("ConnectPoolNumKeys", connectionPoolMap.size());
	}

	auto poolItr = connectionPoolMap.find(connectKey);
	// If it expires in the future then add it to the pool in the front iff connection pool size is not maxed
	if (rconn.expirationTime > now()) {
		bool returned = true;
		if (poolItr == connectionPoolMap.end()) {
			connectionPoolMap.insert({ connectKey, std::queue<RESTConnectionPool::ReusableConnection>({ rconn }) });
		} else if (poolItr->second.size() < maxConnections) {
			poolItr->second.push(rconn);
		} else {
			// Connection pool at its capacity; do nothing
			returned = false;
		}

		if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::DEBUG && returned) {
			poolItr = connectionPoolMap.find(connectKey);
			TraceEvent("RESTUtilReturnConnToPool")
			    .detail("Host", connectKey.first)
			    .detail("Service", connectKey.second)
			    .detail("ConnPoolSize", connectionPoolMap.size())
			    .detail("CachedConns", poolItr->second.size())
			    .detail("TimeToExpire", rconn.expirationTime - now());
		}
	}
	rconn.conn = Reference<IConnection>();
}

RESTUrl::RESTUrl(const std::string& fUrl) {
	parseUrl(fUrl);
}

RESTUrl::RESTUrl(const std::string& fullUrl, const std::string& b) : body(b) {
	parseUrl(fullUrl);
}

void RESTUrl::parseUrl(const std::string& fullUrl) {
	// Sample valid URIs
	// 1. With 'host' & 'resource' := '<protocol>://<host>/<resource>'
	// 2. With 'host', 'service' & 'resource' := '<protocol>://<host>:port/<resource>'
	// 3. With 'host', 'service', 'resource' & 'reqParameters' := '<protocol>://<host>:port/<resource>?<parameter-list>'

	if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::VERBOSE) {
		TraceEvent("RESTParseURI").detail("URI", fullUrl);
	}

	try {
		StringRef t(fullUrl);
		StringRef p = t.eat("://");
		std::string protocol = p.toString();
		boost::algorithm::to_lower(protocol);
		this->connType = RESTConnectionType::getConnectionType(protocol);
		if (!this->connType.secure && !CLIENT_KNOBS->REST_KMS_ALLOW_NOT_SECURE_CONNECTION) {
			TraceEvent(SevWarnAlways, "RESTUtilsUnSupportedNotSecureConn").detail("Protocol", protocol);
			CODE_PROBE(true, "REST URI not-secure connection not supported");
			throw rest_unsupported_protocol();
		}

		// extract 'resource' and optional 'parameter list' if supplied in the URL
		uint8_t foundSeparator = 0;
		StringRef hostPort = t.eatAny("/?", &foundSeparator);
		this->resource = "/";
		if (foundSeparator == '/') {
			this->resource += t.eat("?").toString();
			this->reqParameters = t.eat().toString();
		}

		// hostPort is at least a host or IP address, optionally followed by :portNumber or :serviceName
		StringRef hRef(hostPort);
		StringRef h = hRef.eat(":");
		if (h.size() == 0) {
			CODE_PROBE(true, "REST URI empty host");
			throw std::string("host cannot be empty");
		}
		this->host = h.toString();
		this->service = hRef.eat().toString();

		if (FLOW_KNOBS->REST_LOG_LEVEL >= RESTLogSeverity::DEBUG) {
			TraceEvent("RESTUtilParseURI")
			    .detail("URI", fullUrl)
			    .detail("Host", this->host)
			    .detail("Service", this->service)
			    .detail("Resource", this->resource)
			    .detail("ReqParameters", this->reqParameters)
			    .detail("ConnectionType", this->connType.toString());
		}
	} catch (std::string& err) {
		TraceEvent(SevWarnAlways, "RESTUtilParseError").detail("URI", fullUrl).detail("Error", err);
		throw rest_invalid_uri();
	}
}

double continuousTimeDecay(double initialValue, double decayRate, double time) {
	return initialValue * exp(-decayRate * time);
}

// Only used to link unit tests
void forceLinkRESTUtilsTests() {}

															#line 613 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase289()
															#line 289 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
template <class FlowTestCase289Actor>
															#line 289 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase289ActorState {
															#line 620 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
															#line 289 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase289ActorState(UnitTestParameters const& params) 
															#line 289 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
															#line 289 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		 : params(params)
															#line 627 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase289", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase289ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase289", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 291 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				std::string uri("httpx://foo/bar");
															#line 292 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				RESTUrl r(uri);
															#line 293 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				ASSERT(false);
															#line 647 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase289ActorState();
		static_cast<FlowTestCase289Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 299 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		if (!static_cast<FlowTestCase289Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase289ActorState(); static_cast<FlowTestCase289Actor*>(this)->destroy(); return 0; }
															#line 676 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		new (&static_cast<FlowTestCase289Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase289ActorState();
		static_cast<FlowTestCase289Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 295 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			if (e.code() != error_code_rest_unsupported_protocol)
															#line 689 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			{
															#line 296 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 693 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 289 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	UnitTestParameters params;
															#line 720 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase289()
															#line 289 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase289Actor final : public Actor<Void>, public FastAllocated<FlowTestCase289Actor>, public FlowTestCase289ActorState<FlowTestCase289Actor> {
															#line 725 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase289Actor>::operator new;
	using FastAllocated<FlowTestCase289Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14748224526745796352UL, 11772074166811812864UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 289 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase289Actor(UnitTestParameters const& params) 
															#line 741 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase289ActorState<FlowTestCase289Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase289", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8394282903702220032UL, 17062758169699755264UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase289Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase289");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase289", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 289 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
static Future<Void> flowTestCase289( UnitTestParameters const& params ) {
															#line 289 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	return Future<Void>(new FlowTestCase289Actor(params));
															#line 773 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase289, "/RESTUtils/InvalidProtocol")

#line 301 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"

															#line 779 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase302()
															#line 302 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
template <class FlowTestCase302Actor>
															#line 302 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase302ActorState {
															#line 786 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
															#line 302 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase302ActorState(UnitTestParameters const& params) 
															#line 302 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
															#line 302 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		 : params(params)
															#line 793 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase302", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase302ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase302", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 304 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				std::string uri("https://:/bar");
															#line 305 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				RESTUrl r(uri);
															#line 306 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				ASSERT(false);
															#line 813 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase302ActorState();
		static_cast<FlowTestCase302Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 312 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		if (!static_cast<FlowTestCase302Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase302ActorState(); static_cast<FlowTestCase302Actor*>(this)->destroy(); return 0; }
															#line 842 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		new (&static_cast<FlowTestCase302Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase302ActorState();
		static_cast<FlowTestCase302Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 308 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			if (e.code() != error_code_rest_invalid_uri)
															#line 855 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			{
															#line 309 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 859 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 302 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	UnitTestParameters params;
															#line 886 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase302()
															#line 302 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase302Actor final : public Actor<Void>, public FastAllocated<FlowTestCase302Actor>, public FlowTestCase302ActorState<FlowTestCase302Actor> {
															#line 891 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase302Actor>::operator new;
	using FastAllocated<FlowTestCase302Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2019447972578702592UL, 9826019984510647040UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 302 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase302Actor(UnitTestParameters const& params) 
															#line 907 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase302ActorState<FlowTestCase302Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase302", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9982220374302255104UL, 11867521624599648512UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase302Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase302");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase302", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 302 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
static Future<Void> flowTestCase302( UnitTestParameters const& params ) {
															#line 302 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	return Future<Void>(new FlowTestCase302Actor(params));
															#line 939 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase302, "/RESTUtils/MissingHost")

#line 314 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"

															#line 945 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase315()
															#line 315 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
template <class FlowTestCase315Actor>
															#line 315 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase315ActorState {
															#line 952 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
															#line 315 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase315ActorState(UnitTestParameters const& params) 
															#line 315 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
															#line 315 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		 : params(params)
															#line 959 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase315", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase315ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase315", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 316 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			std::string uri("https://host:80/foo/bar");
															#line 317 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			RESTUrl r(uri);
															#line 318 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.connType.secure, RESTConnectionType::SECURE_CONNECTION);
															#line 319 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.host.compare("host"), 0);
															#line 320 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.service.compare("80"), 0);
															#line 321 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.resource.compare("/foo/bar"), 0);
															#line 322 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			if (!static_cast<FlowTestCase315Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase315ActorState(); static_cast<FlowTestCase315Actor*>(this)->destroy(); return 0; }
															#line 986 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			new (&static_cast<FlowTestCase315Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase315ActorState();
			static_cast<FlowTestCase315Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase315ActorState();
		static_cast<FlowTestCase315Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 315 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	UnitTestParameters params;
															#line 1010 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase315()
															#line 315 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase315Actor final : public Actor<Void>, public FastAllocated<FlowTestCase315Actor>, public FlowTestCase315ActorState<FlowTestCase315Actor> {
															#line 1015 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase315Actor>::operator new;
	using FastAllocated<FlowTestCase315Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1572832459082671616UL, 16807147959487225856UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 315 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase315Actor(UnitTestParameters const& params) 
															#line 1031 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase315ActorState<FlowTestCase315Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase315", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8691705209224595968UL, 12806436817554523648UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase315Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase315");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase315", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 315 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
static Future<Void> flowTestCase315( UnitTestParameters const& params ) {
															#line 315 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	return Future<Void>(new FlowTestCase315Actor(params));
															#line 1063 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase315, "/RESTUtils/ValidURIWithService")

#line 324 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"

															#line 1069 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase325()
															#line 325 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
template <class FlowTestCase325Actor>
															#line 325 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase325ActorState {
															#line 1076 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
															#line 325 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase325ActorState(UnitTestParameters const& params) 
															#line 325 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
															#line 325 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		 : params(params)
															#line 1083 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase325", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase325ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase325", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 326 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			std::string uri("https://host/foo/bar");
															#line 327 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			RESTUrl r(uri);
															#line 328 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.connType.secure, RESTConnectionType::SECURE_CONNECTION);
															#line 329 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.host.compare("host"), 0);
															#line 330 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT(r.service.empty());
															#line 331 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.resource.compare("/foo/bar"), 0);
															#line 332 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			if (!static_cast<FlowTestCase325Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase325ActorState(); static_cast<FlowTestCase325Actor*>(this)->destroy(); return 0; }
															#line 1110 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			new (&static_cast<FlowTestCase325Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase325ActorState();
			static_cast<FlowTestCase325Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase325ActorState();
		static_cast<FlowTestCase325Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 325 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	UnitTestParameters params;
															#line 1134 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase325()
															#line 325 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase325Actor final : public Actor<Void>, public FastAllocated<FlowTestCase325Actor>, public FlowTestCase325ActorState<FlowTestCase325Actor> {
															#line 1139 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase325Actor>::operator new;
	using FastAllocated<FlowTestCase325Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2017263800681799936UL, 9864393446397842944UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 325 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase325Actor(UnitTestParameters const& params) 
															#line 1155 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase325ActorState<FlowTestCase325Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase325", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16815383832306191616UL, 12066985519106116608UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase325Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase325");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase325", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 325 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
static Future<Void> flowTestCase325( UnitTestParameters const& params ) {
															#line 325 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	return Future<Void>(new FlowTestCase325Actor(params));
															#line 1187 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase325, "/RESTUtils/ValidURIWithoutService")

#line 334 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"

															#line 1193 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase335()
															#line 335 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
template <class FlowTestCase335Actor>
															#line 335 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase335ActorState {
															#line 1200 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
															#line 335 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase335ActorState(UnitTestParameters const& params) 
															#line 335 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
															#line 335 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		 : params(params)
															#line 1207 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase335", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase335ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase335", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 336 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			std::string uri("https://host//foo/bar");
															#line 337 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			RESTUrl r(uri);
															#line 338 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.connType.secure, RESTConnectionType::SECURE_CONNECTION);
															#line 339 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.host.compare("host"), 0);
															#line 340 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT(r.service.empty());
															#line 341 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.resource.compare("//foo/bar"), 0);
															#line 342 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			if (!static_cast<FlowTestCase335Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase335ActorState(); static_cast<FlowTestCase335Actor*>(this)->destroy(); return 0; }
															#line 1234 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			new (&static_cast<FlowTestCase335Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase335ActorState();
			static_cast<FlowTestCase335Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase335ActorState();
		static_cast<FlowTestCase335Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 335 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	UnitTestParameters params;
															#line 1258 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase335()
															#line 335 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase335Actor final : public Actor<Void>, public FastAllocated<FlowTestCase335Actor>, public FlowTestCase335ActorState<FlowTestCase335Actor> {
															#line 1263 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase335Actor>::operator new;
	using FastAllocated<FlowTestCase335Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(342905409384655104UL, 16737337197897412864UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 335 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase335Actor(UnitTestParameters const& params) 
															#line 1279 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase335ActorState<FlowTestCase335Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase335", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17608831897142540800UL, 15211983560567129344UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase335Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase335");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase335", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 335 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
static Future<Void> flowTestCase335( UnitTestParameters const& params ) {
															#line 335 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	return Future<Void>(new FlowTestCase335Actor(params));
															#line 1311 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase335, "/RESTUtils/ValidURIWithExtraForwardSlash")

#line 344 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"

															#line 1317 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase345()
															#line 345 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
template <class FlowTestCase345Actor>
															#line 345 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase345ActorState {
															#line 1324 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
															#line 345 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase345ActorState(UnitTestParameters const& params) 
															#line 345 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
															#line 345 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		 : params(params)
															#line 1331 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase345", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase345ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase345", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 346 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			std::string uri("https://host/foo/bar?param1,param2");
															#line 347 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			RESTUrl r(uri);
															#line 348 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.connType.secure, RESTConnectionType::SECURE_CONNECTION);
															#line 349 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.host.compare("host"), 0);
															#line 350 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT(r.service.empty());
															#line 351 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.resource.compare("/foo/bar"), 0);
															#line 352 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.reqParameters.compare("param1,param2"), 0);
															#line 353 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			if (!static_cast<FlowTestCase345Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase345ActorState(); static_cast<FlowTestCase345Actor*>(this)->destroy(); return 0; }
															#line 1360 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			new (&static_cast<FlowTestCase345Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase345ActorState();
			static_cast<FlowTestCase345Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase345ActorState();
		static_cast<FlowTestCase345Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 345 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	UnitTestParameters params;
															#line 1384 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase345()
															#line 345 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase345Actor final : public Actor<Void>, public FastAllocated<FlowTestCase345Actor>, public FlowTestCase345ActorState<FlowTestCase345Actor> {
															#line 1389 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase345Actor>::operator new;
	using FastAllocated<FlowTestCase345Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14397821818783233024UL, 9842675718230787840UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 345 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase345Actor(UnitTestParameters const& params) 
															#line 1405 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase345ActorState<FlowTestCase345Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase345", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1784490442332942336UL, 6815571583949155840UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase345Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase345");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase345", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 345 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
static Future<Void> flowTestCase345( UnitTestParameters const& params ) {
															#line 345 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	return Future<Void>(new FlowTestCase345Actor(params));
															#line 1437 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase345, "/RESTUtils/ValidURIWithParamsSecure")

#line 355 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"

															#line 1443 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase356()
															#line 356 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
template <class FlowTestCase356Actor>
															#line 356 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase356ActorState {
															#line 1450 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
															#line 356 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase356ActorState(UnitTestParameters const& params) 
															#line 356 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
															#line 356 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		 : params(params)
															#line 1457 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase356", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase356ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase356", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 357 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			auto& g_knobs = IKnobCollection::getMutableGlobalKnobCollection();
															#line 358 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			g_knobs.setKnob("rest_kms_allow_not_secure_connection", KnobValueRef::create(bool{ false }));
															#line 359 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			std::string uri("http://host/foo/bar?param1,param2");
															#line 1476 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			try {
															#line 361 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				RESTUrl r(uri);
															#line 362 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
				ASSERT(false);
															#line 1482 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase356ActorState();
		static_cast<FlowTestCase356Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 366 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		if (!static_cast<FlowTestCase356Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase356ActorState(); static_cast<FlowTestCase356Actor*>(this)->destroy(); return 0; }
															#line 1511 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		new (&static_cast<FlowTestCase356Actor*>(this)->SAV< Void >::value()) Void(Void());
		this->~FlowTestCase356ActorState();
		static_cast<FlowTestCase356Actor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 364 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(e.code(), error_code_rest_unsupported_protocol);
															#line 1524 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 356 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	UnitTestParameters params;
															#line 1550 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase356()
															#line 356 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase356Actor final : public Actor<Void>, public FastAllocated<FlowTestCase356Actor>, public FlowTestCase356ActorState<FlowTestCase356Actor> {
															#line 1555 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase356Actor>::operator new;
	using FastAllocated<FlowTestCase356Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(1203406023604718848UL, 1102548050137648128UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 356 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase356Actor(UnitTestParameters const& params) 
															#line 1571 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase356ActorState<FlowTestCase356Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase356", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11587220580997284608UL, 4302225637328386560UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase356Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase356");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase356", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 356 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
static Future<Void> flowTestCase356( UnitTestParameters const& params ) {
															#line 356 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	return Future<Void>(new FlowTestCase356Actor(params));
															#line 1603 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase356, "/RESTUtils/ValidURIWithParamsKnobNotEnabled")

#line 368 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"

															#line 1609 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase369()
															#line 369 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
template <class FlowTestCase369Actor>
															#line 369 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase369ActorState {
															#line 1616 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
															#line 369 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase369ActorState(UnitTestParameters const& params) 
															#line 369 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
															#line 369 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
		 : params(params)
															#line 1623 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase369", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase369ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase369", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 370 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			auto& g_knobs = IKnobCollection::getMutableGlobalKnobCollection();
															#line 371 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			g_knobs.setKnob("rest_kms_allow_not_secure_connection", KnobValueRef::create(bool{ true }));
															#line 372 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			std::string uri("http://host/foo/bar?param1,param2");
															#line 373 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			RESTUrl r(uri);
															#line 374 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.connType.secure, RESTConnectionType::NOT_SECURE_CONNECTION);
															#line 375 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.host.compare("host"), 0);
															#line 376 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT(r.service.empty());
															#line 377 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.resource.compare("/foo/bar"), 0);
															#line 378 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			ASSERT_EQ(r.reqParameters.compare("param1,param2"), 0);
															#line 379 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
			if (!static_cast<FlowTestCase369Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase369ActorState(); static_cast<FlowTestCase369Actor*>(this)->destroy(); return 0; }
															#line 1656 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
			new (&static_cast<FlowTestCase369Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase369ActorState();
			static_cast<FlowTestCase369Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase369ActorState();
		static_cast<FlowTestCase369Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 369 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	UnitTestParameters params;
															#line 1680 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase369()
															#line 369 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
class FlowTestCase369Actor final : public Actor<Void>, public FastAllocated<FlowTestCase369Actor>, public FlowTestCase369ActorState<FlowTestCase369Actor> {
															#line 1685 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase369Actor>::operator new;
	using FastAllocated<FlowTestCase369Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3460702615764599808UL, 10834717947965784576UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 369 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	FlowTestCase369Actor(UnitTestParameters const& params) 
															#line 1701 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase369ActorState<FlowTestCase369Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase369", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9919857435579211520UL, 9239333301527876352UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase369Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase369");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase369", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 369 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
static Future<Void> flowTestCase369( UnitTestParameters const& params ) {
															#line 369 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/fdbclient/RESTUtils.actor.cpp"
	return Future<Void>(new FlowTestCase369Actor(params));
															#line 1733 "/codebuild/output/src35625984/src/github.com/apple/foundationdb/build_output/fdbclient/RESTUtils.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase369, "/RESTUtils/ValidURIWithParams")

