#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
/*
 * Resolver.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <algorithm>
#include <memory>

#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/Notified.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbclient/SystemData.h"
#include "fdbrpc/Stats.h"
#include "fdbserver/ApplyMetadataMutation.h"
#include "fdbserver/ConflictSet.h"
#include "fdbserver/IKeyValueStore.h"
#include "fdbserver/Knobs.h"
#include "fdbserver/LogSystem.h"
#include "fdbserver/LogSystemDiskQueueAdapter.h"
#include "fdbserver/MasterInterface.h"
#include "fdbserver/ResolverInterface.h"
#include "fdbserver/RestoreUtil.h"
#include "fdbserver/ServerDBInfo.h"
#include "fdbserver/StorageMetrics.actor.h"
#include "fdbserver/WaitFailure.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "flow/ActorCollection.h"
#include "flow/Error.h"
#include "flow/Histogram.h"

#include "flow/actorcompiler.h" // This must be the last #include.

namespace {
struct ProxyRequestsInfo {
	std::map<Version, ResolveTransactionBatchReply> outstandingBatches;
	Version lastVersion;

	ProxyRequestsInfo() : lastVersion(-1) {}
};
} // namespace

namespace {

class RecentStateTransactionsInfo {
public:
	RecentStateTransactionsInfo() = default;

	// Erases state transactions up to the given version (inclusive) and returns
	// the number of bytes for the erased mutations.
	int64_t eraseUpTo(Version oldestVersion) {
		recentStateTransactions.erase(recentStateTransactions.begin(),
		                              recentStateTransactions.upper_bound(oldestVersion));

		int64_t stateBytes = 0;
		while (recentStateTransactionSizes.size() && recentStateTransactionSizes.front().first <= oldestVersion) {
			stateBytes += recentStateTransactionSizes.front().second;
			recentStateTransactionSizes.pop_front();
		}
		return stateBytes;
	}

	// Adds state transactions between two versions to the reply message.
	// "initialShardChanged" indicates if commitVersion has shard changes.
	// Returns if shardChanged or a state transaction has ever happened for these versions.
	[[nodiscard]] bool applyStateTxnsToBatchReply(ResolveTransactionBatchReply* reply,
	                                              Version firstUnseenVersion,
	                                              Version commitVersion,
	                                              bool initialShardChanged) {
		bool shardChanged = initialShardChanged;
		auto stateTransactionItr = recentStateTransactions.lower_bound(firstUnseenVersion);
		auto endItr = recentStateTransactions.lower_bound(commitVersion);
		// Resolver only sends back prior state txns back, because the proxy
		// sends this request has them and will apply them via applyMetadataToCommittedTransactions();
		// and other proxies will get this version's state txns as a prior version.
		for (; stateTransactionItr != endItr; ++stateTransactionItr) {
			shardChanged = shardChanged || stateTransactionItr->value.first;
			reply->stateMutations.push_back(reply->arena, stateTransactionItr->value.second);
			reply->arena.dependsOn(stateTransactionItr->value.second.arena());
		}
		return shardChanged;
	}

	bool empty() const { return recentStateTransactionSizes.empty(); }
	// Returns the number of versions with non-empty state transactions.
	uint32_t size() const { return recentStateTransactionSizes.size(); }

	// Returns the first/smallest version of the state transactions.
	// This can only be called when empty() returns false or size() > 0.
	Version firstVersion() const { return recentStateTransactionSizes.front().first; }

	// Records non-zero stateBytes for a version.
	void addVersionBytes(Version commitVersion, int64_t stateBytes) {
		if (stateBytes > 0) {
			recentStateTransactionSizes.emplace_back(commitVersion, stateBytes);
		}
	}

	// Returns the reference to the pair of (shardChanged, stateMutations) for the given version
	std::pair<bool, Standalone<VectorRef<StateTransactionRef>>>& getStateTransactionsRef(Version commitVersion) {
		return recentStateTransactions[commitVersion];
	}

private:
	// Commit version to a pair of (shardChanged, stateMutations).
	Map<Version, std::pair<bool, Standalone<VectorRef<StateTransactionRef>>>> recentStateTransactions;

	// Only keep versions with non-zero size state transactions.
	Deque<std::pair<Version, int64_t>> recentStateTransactionSizes;
};

struct Resolver : ReferenceCounted<Resolver> {
	const UID dbgid;
	const int commitProxyCount;
	const int resolverCount;
	NotifiedVersion version;
	AsyncVar<Version> neededVersion;

	RecentStateTransactionsInfo recentStateTransactionsInfo;
	AsyncVar<int64_t> totalStateBytes;
	AsyncTrigger checkNeededVersion;
	std::map<NetworkAddress, ProxyRequestsInfo> proxyInfoMap;
	ConflictSet* conflictSet;
	TransientStorageMetricSample iopsSample;

	// Use LogSystem as backend for txnStateStore. However, the real commit
	// happens at commit proxies and we never "write" to the LogSystem at
	// Resolvers.
	LogSystemDiskQueueAdapter* logAdapter = nullptr;
	Reference<ILogSystem> logSystem;
	IKeyValueStore* txnStateStore = nullptr;
	int localTLogCount = -1;

	std::map<UID, Reference<StorageInfo>> storageCache;
	KeyRangeMap<ServerCacheInfo> keyInfo; // keyrange -> all storage servers in all DCs for the keyrange
	std::unordered_map<UID, StorageServerInterface> tssMapping;
	bool forceRecovery = false;

	Version debugMinRecentStateVersion = 0;

	// The previous commit versions per tlog
	std::vector<Version> tpcvVector;

	CounterCollection cc;
	Counter resolveBatchIn;
	Counter resolveBatchStart;
	Counter resolvedTransactions;
	Counter resolvedBytes;
	Counter resolvedReadConflictRanges;
	Counter resolvedWriteConflictRanges;
	Counter transactionsAccepted;
	Counter transactionsTooOld;
	Counter transactionsConflicted;
	Counter resolvedStateTransactions;
	Counter resolvedStateMutations;
	Counter resolvedStateBytes;
	Counter resolveBatchOut;
	Counter metricsRequests;
	Counter splitRequests;
	int numLogs;

	// End-to-end server latency of resolver requests.
	Reference<Histogram> resolverLatencyDist;

	// Queue wait times, per request.
	Reference<Histogram> queueWaitLatencyDist;

	// Actual work, per req request.
	Reference<Histogram> computeTimeDist;

	// Distribution of waiters in queue.
	// 0 or 1 will be most common, but higher values are interesting.
	Reference<Histogram> queueDepthDist;

	Future<Void> logger;

	EncryptionAtRestMode encryptMode;

	Version lastShardMove;

	Resolver(UID dbgid, int commitProxyCount, int resolverCount, EncryptionAtRestMode encryptMode)
	  : dbgid(dbgid), commitProxyCount(commitProxyCount), resolverCount(resolverCount), encryptMode(encryptMode),
	    version(-1), conflictSet(newConflictSet()), iopsSample(SERVER_KNOBS->KEY_BYTES_PER_SAMPLE),
	    cc("Resolver", dbgid.toString()), resolveBatchIn("ResolveBatchIn", cc),
	    resolveBatchStart("ResolveBatchStart", cc), resolvedTransactions("ResolvedTransactions", cc),
	    resolvedBytes("ResolvedBytes", cc), resolvedReadConflictRanges("ResolvedReadConflictRanges", cc),
	    resolvedWriteConflictRanges("ResolvedWriteConflictRanges", cc),
	    transactionsAccepted("TransactionsAccepted", cc), transactionsTooOld("TransactionsTooOld", cc),
	    transactionsConflicted("TransactionsConflicted", cc),
	    resolvedStateTransactions("ResolvedStateTransactions", cc),
	    resolvedStateMutations("ResolvedStateMutations", cc), resolvedStateBytes("ResolvedStateBytes", cc),
	    resolveBatchOut("ResolveBatchOut", cc), metricsRequests("MetricsRequests", cc),
	    splitRequests("SplitRequests", cc),
	    resolverLatencyDist(Histogram::getHistogram("Resolver"_sr, "Latency"_sr, Histogram::Unit::milliseconds)),
	    queueWaitLatencyDist(Histogram::getHistogram("Resolver"_sr, "QueueWait"_sr, Histogram::Unit::milliseconds)),
	    computeTimeDist(Histogram::getHistogram("Resolver"_sr, "ComputeTime"_sr, Histogram::Unit::milliseconds)),
	    // Distribution of queue depths, with knowledge that Histogram has 32 buckets, and each bucket will have size 1.
	    queueDepthDist(Histogram::getHistogram("Resolver"_sr, "QueueDepth"_sr, Histogram::Unit::countLinear, 0, 31)),
	    lastShardMove(invalidVersion) {
		specialCounter(cc, "Version", [this]() { return this->version.get(); });
		specialCounter(cc, "NeededVersion", [this]() { return this->neededVersion.get(); });
		specialCounter(cc, "TotalStateBytes", [this]() { return this->totalStateBytes.get(); });

		logger = cc.traceCounters("ResolverMetrics", dbgid, SERVER_KNOBS->WORKER_LOGGING_INTERVAL, "ResolverMetrics");
	}
	~Resolver() { destroyConflictSet(conflictSet); }
};
} // namespace

															#line 226 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via versionReady()
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
template <class VersionReadyActor>
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class VersionReadyActorState {
															#line 233 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	VersionReadyActorState(Resolver* const& self,ProxyRequestsInfo* const& proxyInfo,Version const& prevVersion) 
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		 : self(self),
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   proxyInfo(proxyInfo),
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   prevVersion(prevVersion)
															#line 244 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("versionReady", reinterpret_cast<unsigned long>(this));

	}
	~VersionReadyActorState() 
	{
		fdb_probe_actor_destroy("versionReady", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 225 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			;
															#line 259 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~VersionReadyActorState();
		static_cast<VersionReadyActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 226 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (self->recentStateTransactionsInfo.size() && proxyInfo->lastVersion <= self->recentStateTransactionsInfo.firstVersion())
															#line 289 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 228 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->neededVersion.set(std::max(self->neededVersion.get(), prevVersion));
															#line 293 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
															#line 232 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		int waiters = self->version.numWaiting();
															#line 233 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (self->version.get() < prevVersion)
															#line 299 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 234 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			waiters++;
															#line 303 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
															#line 236 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		self->queueDepthDist->sampleRecordCounter(waiters);
															#line 239 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = self->version.whenAtLeast(prevVersion);
															#line 238 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<VersionReadyActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 311 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 244 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->checkNeededVersion.onTrigger();
															#line 315 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<VersionReadyActor*>(this)->actor_wait_state = 1;
															#line 239 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< VersionReadyActor, 0, Void >*>(static_cast<VersionReadyActor*>(this)));
															#line 244 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< VersionReadyActor, 1, Void >*>(static_cast<VersionReadyActor*>(this)));
															#line 322 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 241 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		self->queueDepthDist->sampleRecordCounter(self->version.numWaiting());
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<VersionReadyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~VersionReadyActorState(); static_cast<VersionReadyActor*>(this)->destroy(); return 0; }
															#line 339 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		new (&static_cast<VersionReadyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~VersionReadyActorState();
		static_cast<VersionReadyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 241 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		self->queueDepthDist->sampleRecordCounter(self->version.numWaiting());
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<VersionReadyActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~VersionReadyActorState(); static_cast<VersionReadyActor*>(this)->destroy(); return 0; }
															#line 353 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		new (&static_cast<VersionReadyActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~VersionReadyActorState();
		static_cast<VersionReadyActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<VersionReadyActor*>(this)->actor_wait_state > 0) static_cast<VersionReadyActor*>(this)->actor_wait_state = 0;
		static_cast<VersionReadyActor*>(this)->ActorCallback< VersionReadyActor, 0, Void >::remove();
		static_cast<VersionReadyActor*>(this)->ActorCallback< VersionReadyActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< VersionReadyActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< VersionReadyActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< VersionReadyActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< VersionReadyActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< VersionReadyActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< VersionReadyActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Resolver* self;
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ProxyRequestsInfo* proxyInfo;
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Version prevVersion;
															#line 500 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via versionReady()
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class VersionReadyActor final : public Actor<Void>, public ActorCallback< VersionReadyActor, 0, Void >, public ActorCallback< VersionReadyActor, 1, Void >, public FastAllocated<VersionReadyActor>, public VersionReadyActorState<VersionReadyActor> {
															#line 505 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
	using FastAllocated<VersionReadyActor>::operator new;
	using FastAllocated<VersionReadyActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13473167698623859200UL, 8019788682649168128UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< VersionReadyActor, 0, Void >;
friend struct ActorCallback< VersionReadyActor, 1, Void >;
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	VersionReadyActor(Resolver* const& self,ProxyRequestsInfo* const& proxyInfo,Version const& prevVersion) 
															#line 523 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   VersionReadyActorState<VersionReadyActor>(self, proxyInfo, prevVersion),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("versionReady", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(14159113740431511296UL, 12582743155816252928UL);
		ActorExecutionContextHelper __helper(static_cast<VersionReadyActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("versionReady");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("versionReady", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< VersionReadyActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> versionReady( Resolver* const& self, ProxyRequestsInfo* const& proxyInfo, Version const& prevVersion ) {
															#line 224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new VersionReadyActor(self, proxyInfo, prevVersion));
															#line 556 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
}

#line 248 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"

// Check if the given set of tags contain any tags other than the log router tags.
// Used to check if a given "ResolveTransactionBatchRequest" corresponds to an
// empty commit message or not.
static bool hasNonLogRouterTags(const std::set<Tag>& allTags) {
	for (auto& versionTag : allTags) {
		if (versionTag.locality != tagLocalityLogRouter) {
			return true;
		}
	}
	return false;
}

															#line 573 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via resolveBatch()
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
template <class ResolveBatchActor>
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolveBatchActorState {
															#line 580 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolveBatchActorState(Reference<Resolver> const& self,ResolveTransactionBatchRequest const& req,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		 : self(self),
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   req(req),
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   db(db),
															#line 264 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   debugID(),
															#line 265 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   span("R:resolveBatch"_loc, req.spanContext),
															#line 268 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   proxyAddress(req.prevVersion >= 0 ? req.reply.getEndpoint().getPrimaryAddress() : NetworkAddress()),
															#line 270 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   proxyInfo(self->proxyInfoMap[proxyAddress]),
															#line 272 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   cipherKeys()
															#line 601 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("resolveBatch", reinterpret_cast<unsigned long>(this));

	}
	~ResolveBatchActorState() 
	{
		fdb_probe_actor_destroy("resolveBatch", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 273 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (self->encryptMode.isEncryptionEnabled())
															#line 616 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 274 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				static const std::unordered_set<EncryptCipherDomainId> metadataDomainIds = { SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID, ENCRYPT_HEADER_DOMAIN_ID };
															#line 276 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				StrictFuture<std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>> __when_expr_0 = GetEncryptCipherKeys<ServerDBInfo>::getLatestEncryptCipherKeys( db, metadataDomainIds, BlobCipherMetrics::TLOG);
															#line 276 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				if (static_cast<ResolveBatchActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 624 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ResolveBatchActor*>(this)->actor_wait_state = 1;
															#line 276 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*>(static_cast<ResolveBatchActor*>(this)));
															#line 629 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResolveBatchActorState();
		static_cast<ResolveBatchActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 282 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->resolveBatchIn;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (req.debugID.present())
															#line 659 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 285 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			debugID = nondeterministicRandom()->randomUniqueID();
															#line 286 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			g_traceBatch.addAttach("CommitAttachID", req.debugID.get().first(), debugID.get().first());
															#line 287 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			g_traceBatch.addEvent("CommitDebug", debugID.get().first(), "Resolver.resolveBatch.Before");
															#line 667 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
															#line 296 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		;
															#line 671 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& cks,int loopDepth) 
	{
															#line 279 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		cipherKeys = cks;
															#line 680 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && cks,int loopDepth) 
	{
															#line 279 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		cipherKeys = cks;
															#line 689 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& cks,int loopDepth) 
	{
		loopDepth = a_body1cont2(cks, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && cks,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(cks), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state > 0) static_cast<ResolveBatchActor*>(this)->actor_wait_state = 0;
		static_cast<ResolveBatchActor*>(this)->ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >::remove();

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,Error err) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 308 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (debugID.present())
															#line 773 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 309 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			g_traceBatch.addEvent("CommitDebug", debugID.get().first(), "Resolver.resolveBatch.AfterQueueSizeCheck");
															#line 777 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
															#line 312 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = versionReady(self.getPtr(), &proxyInfo, req.prevVersion);
															#line 312 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 783 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont4when1(__when_expr_2.get(), loopDepth); };
		static_cast<ResolveBatchActor*>(this)->actor_wait_state = 3;
															#line 312 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ResolveBatchActor, 2, Void >*>(static_cast<ResolveBatchActor*>(this)));
															#line 788 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
															#line 296 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!(self->totalStateBytes.get() > SERVER_KNOBS->RESOLVER_STATE_MEMORY_LIMIT && self->recentStateTransactionsInfo.size() && proxyInfo.lastVersion > self->recentStateTransactionsInfo.firstVersion() && req.version > self->neededVersion.get()))
															#line 804 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
			return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 305 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->totalStateBytes.onChange() || self->neededVersion.onChange();
															#line 305 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 812 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ResolveBatchActor*>(this)->actor_wait_state = 2;
															#line 305 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResolveBatchActor, 1, Void >*>(static_cast<ResolveBatchActor*>(this)));
															#line 817 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state > 0) static_cast<ResolveBatchActor*>(this)->actor_wait_state = 0;
		static_cast<ResolveBatchActor*>(this)->ActorCallback< ResolveBatchActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResolveBatchActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 314 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (check_yield(TaskPriority::DefaultEndpoint))
															#line 926 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 315 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0, TaskPriority::Low) || delay(SERVER_KNOBS->COMMIT_SLEEP_TIME);
															#line 315 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (static_cast<ResolveBatchActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 932 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont6when1(__when_expr_3.get(), loopDepth); };
			static_cast<ResolveBatchActor*>(this)->actor_wait_state = 4;
															#line 315 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ResolveBatchActor, 3, Void >*>(static_cast<ResolveBatchActor*>(this)));
															#line 937 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 314 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (check_yield(TaskPriority::DefaultEndpoint))
															#line 951 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 315 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			StrictFuture<Void> __when_expr_3 = delay(0, TaskPriority::Low) || delay(SERVER_KNOBS->COMMIT_SLEEP_TIME);
															#line 315 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (static_cast<ResolveBatchActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 957 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont6when1(__when_expr_3.get(), loopDepth); };
			static_cast<ResolveBatchActor*>(this)->actor_wait_state = 4;
															#line 315 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ResolveBatchActor, 3, Void >*>(static_cast<ResolveBatchActor*>(this)));
															#line 962 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state > 0) static_cast<ResolveBatchActor*>(this)->actor_wait_state = 0;
		static_cast<ResolveBatchActor*>(this)->ActorCallback< ResolveBatchActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ResolveBatchActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont8(int loopDepth) 
	{
															#line 320 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		double queueWaitEndTime = g_network->timer();
															#line 321 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		self->queueWaitLatencyDist->sampleSeconds(queueWaitEndTime - req.requestTime());
															#line 323 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (self->version.get() == req.prevVersion)
															#line 1055 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 327 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			const double beginComputeTime = g_network->timer();
															#line 329 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			++self->resolveBatchStart;
															#line 330 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->resolvedTransactions += req.transactions.size();
															#line 331 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->resolvedBytes += req.transactions.expectedSize();
															#line 333 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (proxyInfo.lastVersion > 0)
															#line 1067 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 334 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				proxyInfo.outstandingBatches.erase(proxyInfo.outstandingBatches.begin(), proxyInfo.outstandingBatches.upper_bound(req.lastReceivedVersion));
															#line 1071 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
															#line 338 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			Version firstUnseenVersion = proxyInfo.lastVersion + 1;
															#line 339 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			proxyInfo.lastVersion = req.version;
															#line 341 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (req.debugID.present())
															#line 1079 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 342 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				g_traceBatch.addEvent("CommitDebug", debugID.get().first(), "Resolver.resolveBatch.AfterOrderer");
															#line 1083 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
															#line 344 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ResolveTransactionBatchReply& reply = proxyInfo.outstandingBatches[req.version];
															#line 345 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			reply.writtenTags = req.writtenTags;
															#line 347 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			std::vector<int> commitList;
															#line 348 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			std::vector<int> tooOldList;
															#line 351 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			double expire = now() + SERVER_KNOBS->SAMPLE_EXPIRATION_TIME;
															#line 352 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ConflictBatch conflictBatch(self->conflictSet, &reply.conflictingKeyRangeMap, &reply.arena);
															#line 353 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			const Version newOldestVersion = req.version - SERVER_KNOBS->MAX_WRITE_TRANSACTION_LIFE_VERSIONS;
															#line 354 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			for(int t = 0;t < req.transactions.size();t++) {
															#line 355 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				conflictBatch.addTransaction(req.transactions[t], newOldestVersion);
															#line 356 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				self->resolvedReadConflictRanges += req.transactions[t].read_conflict_ranges.size();
															#line 357 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				self->resolvedWriteConflictRanges += req.transactions[t].write_conflict_ranges.size();
															#line 359 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				if (self->resolverCount > 1)
															#line 1109 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				{
															#line 360 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					for( auto it : req.transactions[t].write_conflict_ranges ) {
															#line 361 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
						self->iopsSample.addAndExpire( it.begin, SERVER_KNOBS->SAMPLE_OFFSET_PER_KEY + it.begin.size(), expire);
															#line 1115 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
					}
															#line 363 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					for( auto it : req.transactions[t].read_conflict_ranges ) {
															#line 364 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
						self->iopsSample.addAndExpire( it.begin, SERVER_KNOBS->SAMPLE_OFFSET_PER_KEY + it.begin.size(), expire);
															#line 1121 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
					}
				}
			}
															#line 368 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			conflictBatch.detectConflicts(req.version, newOldestVersion, commitList, &tooOldList);
															#line 370 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			reply.debugID = req.debugID;
															#line 371 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			reply.committed.resize(reply.arena, req.transactions.size());
															#line 372 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			for(int c = 0;c < commitList.size();c++) {
															#line 373 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				reply.committed[commitList[c]] = ConflictBatch::TransactionCommitted;
															#line 1135 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
															#line 375 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			for(int c = 0;c < tooOldList.size();c++) {
															#line 376 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				ASSERT(reply.committed[tooOldList[c]] == ConflictBatch::TransactionConflict);
															#line 377 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				reply.committed[tooOldList[c]] = ConflictBatch::TransactionTooOld;
															#line 1143 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
															#line 380 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->transactionsAccepted += commitList.size();
															#line 381 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->transactionsTooOld += tooOldList.size();
															#line 382 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->transactionsConflicted += req.transactions.size() - commitList.size() - tooOldList.size();
															#line 384 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(req.prevVersion >= 0 || req.txnStateTransactions.size() == 0);
															#line 387 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			auto& stateTransactionsPair = self->recentStateTransactionsInfo.getStateTransactionsRef(req.version);
															#line 388 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			auto& stateTransactions = stateTransactionsPair.second;
															#line 389 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			int64_t stateMutations = 0;
															#line 390 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			int64_t stateBytes = 0;
															#line 391 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			const bool shouldApplyResolverPrivateMutations = SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS && !req.txnStateTransactions.empty();
															#line 394 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			std::unique_ptr<LogPushData> toCommit;
															#line 395 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			std::unique_ptr<ResolverData> resolverData;
															#line 396 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			bool isLocked = false;
															#line 397 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS)
															#line 1171 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 398 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				toCommit.reset(new LogPushData(self->logSystem, self->localTLogCount));
															#line 399 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				if (shouldApplyResolverPrivateMutations)
															#line 1177 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				{
															#line 400 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					auto lockedKey = self->txnStateStore->readValue(databaseLockedKey).get();
															#line 401 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					isLocked = lockedKey.present() && lockedKey.get().size();
															#line 402 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					resolverData.reset(new ResolverData(self->dbgid, self->logSystem, self->txnStateStore, &self->keyInfo, toCommit.get(), self->forceRecovery, req.version + 1, &self->storageCache, &self->tssMapping));
															#line 1185 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				}
			}
															#line 413 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			for( int t : req.txnStateTransactions ) {
															#line 414 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				stateMutations += req.transactions[t].mutations.size();
															#line 415 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				stateBytes += req.transactions[t].mutations.expectedSize();
															#line 416 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				stateTransactions.push_back_deep( stateTransactions.arena(), StateTransactionRef(reply.committed[t] == ConflictBatch::TransactionCommitted, req.transactions[t].mutations));
															#line 426 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				if (reply.committed[t] == ConflictBatch::TransactionCommitted && !self->forceRecovery && shouldApplyResolverPrivateMutations && (!isLocked || req.transactions[t].lock_aware))
															#line 1198 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				{
															#line 428 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					SpanContext spanContext = req.transactions[t].spanContext.present() ? req.transactions[t].spanContext.get() : SpanContext();
															#line 431 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					applyMetadataMutations(spanContext, *resolverData, req.transactions[t].mutations, self->encryptMode.isEncryptionEnabled() ? &cipherKeys : nullptr, self->encryptMode);
															#line 1204 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				}
															#line 437 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				CODE_PROBE(self->forceRecovery, "Resolver detects forced recovery");
															#line 1208 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
															#line 440 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->resolvedStateTransactions += req.txnStateTransactions.size();
															#line 441 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->resolvedStateMutations += stateMutations;
															#line 442 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->resolvedStateBytes += stateBytes;
															#line 444 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->recentStateTransactionsInfo.addVersionBytes(req.version, stateBytes);
															#line 446 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(req.version >= firstUnseenVersion);
															#line 447 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(firstUnseenVersion >= self->debugMinRecentStateVersion);
															#line 449 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			CODE_PROBE(firstUnseenVersion == req.version, "Resolver first unseen version is current version");
															#line 453 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			bool logsChanged = shouldApplyResolverPrivateMutations && toCommit->haveLogsChanged();
															#line 454 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			stateTransactionsPair.first = logsChanged;
															#line 455 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			bool shardChanged = self->recentStateTransactionsInfo.applyStateTxnsToBatchReply( &reply, firstUnseenVersion, req.version, logsChanged);
															#line 459 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS)
															#line 1232 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 460 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				auto privateMutations = toCommit->getAllMessages();
															#line 461 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				for( const auto& mutations : privateMutations ) {
															#line 462 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					reply.privateMutations.push_back(reply.arena, mutations);
															#line 463 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					reply.arena.dependsOn(mutations.arena());
															#line 1242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				}
															#line 466 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				toCommit->saveTags(reply.writtenTags);
															#line 467 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				reply.privateMutationCount = shouldApplyResolverPrivateMutations ? toCommit->getMutationCount() : 0;
															#line 1248 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
															#line 472 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(!proxyInfo.outstandingBatches.empty());
															#line 473 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(self->proxyInfoMap.size() <= self->commitProxyCount + 1);
															#line 476 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			Version oldestProxyVersion = req.version;
															#line 477 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			for(auto itr = self->proxyInfoMap.begin();itr != self->proxyInfoMap.end();++itr) {
															#line 479 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				if (itr->first.isValid())
															#line 1260 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				{
															#line 480 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					oldestProxyVersion = std::min(itr->second.lastVersion, oldestProxyVersion);
															#line 1264 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				}
				else
				{
															#line 483 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					ASSERT(self->debugMinRecentStateVersion == 0 || self->debugMinRecentStateVersion > itr->second.lastVersion);
															#line 1270 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				}
			}
															#line 488 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			CODE_PROBE(oldestProxyVersion == req.version, "The proxy that sent this request has the oldest current version");
															#line 490 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			CODE_PROBE(oldestProxyVersion != req.version, "The proxy that sent this request does not have the oldest current version");
															#line 493 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			bool anyPopped = false;
															#line 494 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (firstUnseenVersion <= oldestProxyVersion && self->proxyInfoMap.size() == self->commitProxyCount + 1)
															#line 1281 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 495 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				CODE_PROBE(true, "Deleting old state transactions");
															#line 496 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				int64_t erasedBytes = self->recentStateTransactionsInfo.eraseUpTo(oldestProxyVersion);
															#line 497 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				self->debugMinRecentStateVersion = oldestProxyVersion + 1;
															#line 498 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				anyPopped = erasedBytes > 0;
															#line 499 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				stateBytes -= erasedBytes;
															#line 1293 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
															#line 502 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (SERVER_KNOBS->ENABLE_VERSION_VECTOR_TLOG_UNICAST)
															#line 1297 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 503 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				if (!self->numLogs)
															#line 1301 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				{
															#line 504 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					reply.tpcvMap.clear();
															#line 1305 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				}
				else
				{
															#line 508 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					ASSERT(SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS);
															#line 509 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					std::set<uint16_t> writtenTLogs;
															#line 513 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					bool isEmpty = !hasNonLogRouterTags(req.writtenTags);
															#line 514 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					if (req.lastShardMove < self->lastShardMove || shardChanged || req.txnStateTransactions.size() || isEmpty)
															#line 1317 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
					{
															#line 516 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
						for(int i = 0;i < self->numLogs;i++) {
															#line 517 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
							writtenTLogs.insert(i);
															#line 1323 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
						}
															#line 519 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
						if (shardChanged)
															#line 1327 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
						{
															#line 520 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
							self->lastShardMove = req.version;
															#line 1331 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
						}
					}
					else
					{
															#line 523 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
						toCommit->getLocations(reply.writtenTags, writtenTLogs);
															#line 1338 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
					}
															#line 525 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					if (self->tpcvVector[0] == invalidVersion)
															#line 1342 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
					{
															#line 526 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
						std::fill(self->tpcvVector.begin(), self->tpcvVector.end(), req.prevVersion);
															#line 1346 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
					}
															#line 528 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
					for( uint16_t tLog : writtenTLogs ) {
															#line 529 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
						reply.tpcvMap[tLog] = self->tpcvVector[tLog];
															#line 530 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
						self->tpcvVector[tLog] = req.version;
															#line 1354 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
					}
				}
															#line 533 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				reply.lastShardMove = self->lastShardMove;
															#line 1359 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
															#line 535 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->version.set(req.version);
															#line 536 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			bool breachedLimit = self->totalStateBytes.get() <= SERVER_KNOBS->RESOLVER_STATE_MEMORY_LIMIT && self->totalStateBytes.get() + stateBytes > SERVER_KNOBS->RESOLVER_STATE_MEMORY_LIMIT;
															#line 538 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->totalStateBytes.setUnconditional(self->totalStateBytes.get() + stateBytes);
															#line 539 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (anyPopped || breachedLimit)
															#line 1369 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 540 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				self->checkNeededVersion.trigger();
															#line 1373 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
															#line 544 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			const double endComputeTime = g_network->timer();
															#line 545 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->computeTimeDist->sampleSeconds(endComputeTime - beginComputeTime);
															#line 547 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (req.debugID.present())
															#line 1381 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 548 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				g_traceBatch.addEvent("CommitDebug", debugID.get().first(), "Resolver.resolveBatch.After");
															#line 1385 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
		}
		else
		{
															#line 550 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			CODE_PROBE(true, "Duplicate resolve batch request");
															#line 1392 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
															#line 554 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		auto proxyInfoItr = self->proxyInfoMap.find(proxyAddress);
															#line 556 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (proxyInfoItr != self->proxyInfoMap.end())
															#line 1398 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 557 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			auto batchItr = proxyInfoItr->second.outstandingBatches.find(req.version);
															#line 558 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (batchItr != proxyInfoItr->second.outstandingBatches.end())
															#line 1404 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 559 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				req.reply.send(batchItr->second);
															#line 1408 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
			else
			{
															#line 561 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				CODE_PROBE(true, "No outstanding batches for version on proxy", probe::decoration::rare);
															#line 562 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				req.reply.send(Never());
															#line 1416 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
		}
		else
		{
															#line 565 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT_WE_THINK(false);
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			req.reply.send(Never());
															#line 1425 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
															#line 573 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		const double endTime = g_network->timer();
															#line 574 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		self->resolverLatencyDist->sampleSeconds(endTime - req.requestTime());
															#line 576 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->resolveBatchOut;
															#line 578 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ResolveBatchActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResolveBatchActorState(); static_cast<ResolveBatchActor*>(this)->destroy(); return 0; }
															#line 1435 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		new (&static_cast<ResolveBatchActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResolveBatchActorState();
		static_cast<ResolveBatchActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
															#line 316 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		g_network->setCurrentTask(TaskPriority::DefaultEndpoint);
															#line 1447 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
															#line 316 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		g_network->setCurrentTask(TaskPriority::DefaultEndpoint);
															#line 1456 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ResolveBatchActor*>(this)->actor_wait_state > 0) static_cast<ResolveBatchActor*>(this)->actor_wait_state = 0;
		static_cast<ResolveBatchActor*>(this)->ActorCallback< ResolveBatchActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ResolveBatchActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ResolveBatchActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<Resolver> self;
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolveTransactionBatchRequest req;
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 264 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Optional<UID> debugID;
															#line 265 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Span span;
															#line 268 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	NetworkAddress proxyAddress;
															#line 270 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ProxyRequestsInfo& proxyInfo;
															#line 272 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> cipherKeys;
															#line 1552 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via resolveBatch()
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolveBatchActor final : public Actor<Void>, public ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >, public ActorCallback< ResolveBatchActor, 1, Void >, public ActorCallback< ResolveBatchActor, 2, Void >, public ActorCallback< ResolveBatchActor, 3, Void >, public FastAllocated<ResolveBatchActor>, public ResolveBatchActorState<ResolveBatchActor> {
															#line 1557 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
	using FastAllocated<ResolveBatchActor>::operator new;
	using FastAllocated<ResolveBatchActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(16496548866001722624UL, 261704128452110080UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >;
friend struct ActorCallback< ResolveBatchActor, 1, Void >;
friend struct ActorCallback< ResolveBatchActor, 2, Void >;
friend struct ActorCallback< ResolveBatchActor, 3, Void >;
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolveBatchActor(Reference<Resolver> const& self,ResolveTransactionBatchRequest const& req,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 1577 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ResolveBatchActorState<ResolveBatchActor>(self, req, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resolveBatch", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4614177339645472000UL, 11787679010325726976UL);
		ActorExecutionContextHelper __helper(static_cast<ResolveBatchActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resolveBatch");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resolveBatch", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResolveBatchActor, 0, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ResolveBatchActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ResolveBatchActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ResolveBatchActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> resolveBatch( Reference<Resolver> const& self, ResolveTransactionBatchRequest const& req, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new ResolveBatchActor(self, req, db));
															#line 1613 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
}

#line 580 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"

namespace {

// TODO: refactor with the one in CommitProxyServer.actor.cpp
struct TransactionStateResolveContext {
	// Maximum sequence for txnStateRequest, this is defined when the request last flag is set.
	Sequence maxSequence = std::numeric_limits<Sequence>::max();

	// Flags marks received transaction state requests, we only process the transaction request when *all* requests are
	// received.
	std::unordered_set<Sequence> receivedSequences;

	Reference<Resolver> pResolverData;

	// Pointer to transaction state store, shortcut for commitData.txnStateStore
	IKeyValueStore* pTxnStateStore = nullptr;

	// Actor streams
	PromiseStream<Future<Void>>* pActors = nullptr;

	// Flag reports if the transaction state request is complete. This request should only happen during recover, i.e.
	// once per Resolver.
	bool processed = false;

	TransactionStateResolveContext() = default;

	TransactionStateResolveContext(Reference<Resolver> pResolverData_, PromiseStream<Future<Void>>* pActors_)
	  : pResolverData(pResolverData_), pTxnStateStore(pResolverData_->txnStateStore), pActors(pActors_) {
		ASSERT(pTxnStateStore != nullptr || !SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS);
	}
};

															#line 1649 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
// This generated class is to be used only via processCompleteTransactionStateRequest()
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
template <class ProcessCompleteTransactionStateRequestActor>
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class ProcessCompleteTransactionStateRequestActorState {
															#line 1655 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ProcessCompleteTransactionStateRequestActorState(Reference<Resolver> const& self,TransactionStateResolveContext* const& pContext,Reference<AsyncVar<ServerDBInfo> const> const& db,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>* const& cipherKeys) 
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		 : self(self),
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   pContext(pContext),
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   db(db),
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   cipherKeys(cipherKeys),
															#line 617 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   txnKeys(allKeys),
															#line 618 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   tag_uid()
															#line 1672 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this));

	}
	~ProcessCompleteTransactionStateRequestActorState() 
	{
		fdb_probe_actor_destroy("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 620 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			RangeResult UIDtoTagMap = pContext->pTxnStateStore->readRange(serverTagKeys).get();
															#line 621 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			for( const KeyValueRef& kv : UIDtoTagMap ) {
															#line 622 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				tag_uid[decodeServerTagValue(kv.value)] = decodeServerTagKey(kv.key);
															#line 1691 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			}
															#line 625 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			;
															#line 1695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ProcessCompleteTransactionStateRequestActorState();
		static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 686 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		auto lockedKey = pContext->pTxnStateStore->readValue(databaseLockedKey).get();
															#line 690 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->pTxnStateStore->enableSnapshot();
															#line 692 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcessCompleteTransactionStateRequestActorState(); static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->destroy(); return 0; }
															#line 1722 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		new (&static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcessCompleteTransactionStateRequestActorState();
		static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 626 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = yield();
															#line 626 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1743 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->actor_wait_state = 1;
															#line 626 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >*>(static_cast<ProcessCompleteTransactionStateRequestActor*>(this)));
															#line 1748 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 628 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		RangeResult data = pContext->pTxnStateStore ->readRange(txnKeys, SERVER_KNOBS->BUGGIFIED_ROW_LIMIT, SERVER_KNOBS->APPLY_MUTATION_BYTES) .get();
															#line 632 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!data.size())
															#line 1772 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 635 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		((KeyRangeRef&)txnKeys) = KeyRangeRef(keyAfter(data.back().key, txnKeys.arena()), txnKeys.end);
															#line 637 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		MutationsVec mutations;
															#line 638 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		std::vector<std::pair<MapPair<Key, ServerCacheInfo>, int>> keyInfoData;
															#line 639 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		std::vector<UID> src, dest;
															#line 640 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		ServerCacheInfo info;
															#line 642 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		auto updateTagInfo = [pContext = pContext](const std::vector<UID>& uids, std::vector<Tag>& tags, std::vector<Reference<StorageInfo>>& storageInfoItems) { for (const auto& id : uids) { auto storageInfo = getStorageInfo(id, &pContext->pResolverData->storageCache, pContext->pTxnStateStore); ASSERT(storageInfo->tag != invalidTag); tags.push_back(storageInfo->tag); storageInfoItems.push_back(storageInfo); } };
															#line 652 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		for( auto& kv : data ) {
															#line 653 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (!kv.key.startsWith(keyServersPrefix))
															#line 1792 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 654 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				mutations.emplace_back(mutations.arena(), MutationRef::SetValue, kv.key, kv.value);
															#line 1796 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				continue;
			}
															#line 658 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			KeyRef k = kv.key.removePrefix(keyServersPrefix);
															#line 659 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (k == allKeys.end)
															#line 1803 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
				continue;
			}
															#line 662 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			decodeKeyServersValue(tag_uid, kv.value, src, dest);
															#line 664 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			info.tags.clear();
															#line 666 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			info.src_info.clear();
															#line 667 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			updateTagInfo(src, info.tags, info.src_info);
															#line 669 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			info.dest_info.clear();
															#line 670 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			updateTagInfo(dest, info.tags, info.dest_info);
															#line 672 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			uniquify(info.tags);
															#line 673 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			keyInfoData.emplace_back(MapPair<Key, ServerCacheInfo>(k, info), 1);
															#line 1823 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
															#line 678 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->pResolverData->keyInfo.rawInsert(keyInfoData);
															#line 680 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		bool confChanges;
															#line 681 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		ResolverData resolverData( pContext->pResolverData->dbgid, pContext->pTxnStateStore, &pContext->pResolverData->keyInfo, confChanges);
															#line 683 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		applyMetadataMutations(SpanContext(), resolverData, mutations, cipherKeys, self->encryptMode);
															#line 1833 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 628 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		RangeResult data = pContext->pTxnStateStore ->readRange(txnKeys, SERVER_KNOBS->BUGGIFIED_ROW_LIMIT, SERVER_KNOBS->APPLY_MUTATION_BYTES) .get();
															#line 632 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!data.size())
															#line 1844 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 635 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		((KeyRangeRef&)txnKeys) = KeyRangeRef(keyAfter(data.back().key, txnKeys.arena()), txnKeys.end);
															#line 637 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		MutationsVec mutations;
															#line 638 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		std::vector<std::pair<MapPair<Key, ServerCacheInfo>, int>> keyInfoData;
															#line 639 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		std::vector<UID> src, dest;
															#line 640 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		ServerCacheInfo info;
															#line 642 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		auto updateTagInfo = [pContext = pContext](const std::vector<UID>& uids, std::vector<Tag>& tags, std::vector<Reference<StorageInfo>>& storageInfoItems) { for (const auto& id : uids) { auto storageInfo = getStorageInfo(id, &pContext->pResolverData->storageCache, pContext->pTxnStateStore); ASSERT(storageInfo->tag != invalidTag); tags.push_back(storageInfo->tag); storageInfoItems.push_back(storageInfo); } };
															#line 652 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		for( auto& kv : data ) {
															#line 653 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (!kv.key.startsWith(keyServersPrefix))
															#line 1864 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 654 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				mutations.emplace_back(mutations.arena(), MutationRef::SetValue, kv.key, kv.value);
															#line 1868 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				continue;
			}
															#line 658 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			KeyRef k = kv.key.removePrefix(keyServersPrefix);
															#line 659 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (k == allKeys.end)
															#line 1875 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
				continue;
			}
															#line 662 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			decodeKeyServersValue(tag_uid, kv.value, src, dest);
															#line 664 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			info.tags.clear();
															#line 666 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			info.src_info.clear();
															#line 667 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			updateTagInfo(src, info.tags, info.src_info);
															#line 669 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			info.dest_info.clear();
															#line 670 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			updateTagInfo(dest, info.tags, info.dest_info);
															#line 672 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			uniquify(info.tags);
															#line 673 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			keyInfoData.emplace_back(MapPair<Key, ServerCacheInfo>(k, info), 1);
															#line 1895 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
															#line 678 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->pResolverData->keyInfo.rawInsert(keyInfoData);
															#line 680 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		bool confChanges;
															#line 681 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		ResolverData resolverData( pContext->pResolverData->dbgid, pContext->pTxnStateStore, &pContext->pResolverData->keyInfo, confChanges);
															#line 683 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		applyMetadataMutations(SpanContext(), resolverData, mutations, cipherKeys, self->encryptMode);
															#line 1905 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->actor_wait_state > 0) static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->actor_wait_state = 0;
		static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<Resolver> self;
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	TransactionStateResolveContext* pContext;
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>* cipherKeys;
															#line 617 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	KeyRange txnKeys;
															#line 618 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	std::map<Tag, UID> tag_uid;
															#line 1997 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via processCompleteTransactionStateRequest()
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class ProcessCompleteTransactionStateRequestActor final : public Actor<Void>, public ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >, public FastAllocated<ProcessCompleteTransactionStateRequestActor>, public ProcessCompleteTransactionStateRequestActorState<ProcessCompleteTransactionStateRequestActor> {
															#line 2002 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
	using FastAllocated<ProcessCompleteTransactionStateRequestActor>::operator new;
	using FastAllocated<ProcessCompleteTransactionStateRequestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3253127076705136128UL, 15092041995318755328UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >;
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ProcessCompleteTransactionStateRequestActor(Reference<Resolver> const& self,TransactionStateResolveContext* const& pContext,Reference<AsyncVar<ServerDBInfo> const> const& db,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>* const& cipherKeys) 
															#line 2019 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ProcessCompleteTransactionStateRequestActorState<ProcessCompleteTransactionStateRequestActor>(self, pContext, db, cipherKeys),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8499504063907237888UL, 15656031568419423488UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessCompleteTransactionStateRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("processCompleteTransactionStateRequest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("processCompleteTransactionStateRequest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ProcessCompleteTransactionStateRequestActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> processCompleteTransactionStateRequest( Reference<Resolver> const& self, TransactionStateResolveContext* const& pContext, Reference<AsyncVar<ServerDBInfo> const> const& db, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>* const& cipherKeys ) {
															#line 612 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new ProcessCompleteTransactionStateRequestActor(self, pContext, db, cipherKeys));
															#line 2051 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
}

#line 694 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"

															#line 2056 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
// This generated class is to be used only via processTransactionStateRequestPart()
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
template <class ProcessTransactionStateRequestPartActor>
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class ProcessTransactionStateRequestPartActorState {
															#line 2062 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ProcessTransactionStateRequestPartActorState(Reference<Resolver> const& self,TransactionStateResolveContext* const& pContext,TxnStateRequest const& request,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		 : self(self),
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   pContext(pContext),
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   request(request),
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   db(db)
															#line 2075 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this));

	}
	~ProcessTransactionStateRequestPartActorState() 
	{
		fdb_probe_actor_destroy("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 699 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(pContext->pResolverData.getPtr() != nullptr);
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(pContext->pActors != nullptr);
															#line 702 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (pContext->receivedSequences.contains(request.sequence))
															#line 2094 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 704 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				pContext->pActors->send(broadcastTxnRequest(request, SERVER_KNOBS->TXN_STATE_SEND_AMOUNT, true));
															#line 705 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				StrictFuture<Void> __when_expr_0 = yield();
															#line 705 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2102 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 1;
															#line 705 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >*>(static_cast<ProcessTransactionStateRequestPartActor*>(this)));
															#line 2107 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ProcessTransactionStateRequestPartActorState();
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 709 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (request.last)
															#line 2135 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 711 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			pContext->maxSequence = request.sequence + 1;
															#line 2139 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
															#line 713 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->receivedSequences.insert(request.sequence);
															#line 717 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		for( auto& kv : request.data ) {
															#line 718 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			pContext->pTxnStateStore->set(kv, &request.arena);
															#line 2147 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
															#line 720 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->pTxnStateStore->commit(true);
															#line 722 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (pContext->receivedSequences.size() == pContext->maxSequence)
															#line 2153 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 724 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(!pContext->processed);
															#line 725 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			cipherKeys = std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>();
															#line 726 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (self->encryptMode.isEncryptionEnabled())
															#line 2161 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 727 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				static const std::unordered_set<EncryptCipherDomainId> metadataDomainIds = { SYSTEM_KEYSPACE_ENCRYPT_DOMAIN_ID, ENCRYPT_HEADER_DOMAIN_ID };
															#line 730 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				StrictFuture<std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>>> __when_expr_1 = GetEncryptCipherKeys<ServerDBInfo>::getLatestEncryptCipherKeys( db, metadataDomainIds, BlobCipherMetrics::TLOG);
															#line 730 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2169 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
				static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 2;
															#line 730 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*>(static_cast<ProcessTransactionStateRequestPartActor*>(this)));
															#line 2174 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont7(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 706 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcessTransactionStateRequestPartActorState(); static_cast<ProcessTransactionStateRequestPartActor*>(this)->destroy(); return 0; }
															#line 2193 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		new (&static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcessTransactionStateRequestPartActorState();
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 706 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcessTransactionStateRequestPartActorState(); static_cast<ProcessTransactionStateRequestPartActor*>(this)->destroy(); return 0; }
															#line 2205 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		new (&static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcessTransactionStateRequestPartActorState();
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state > 0) static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 0;
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 740 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->pActors->send(broadcastTxnRequest(request, SERVER_KNOBS->TXN_STATE_SEND_AMOUNT, true));
															#line 741 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_3 = yield();
															#line 741 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2296 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont4when1(__when_expr_3.get(), loopDepth); };
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 4;
															#line 741 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >*>(static_cast<ProcessTransactionStateRequestPartActor*>(this)));
															#line 2301 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7(int loopDepth) 
	{
															#line 735 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_2 = processCompleteTransactionStateRequest( self, pContext, db, self->encryptMode.isEncryptionEnabled() ? &cipherKeys : nullptr);
															#line 735 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2312 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont7when1(__when_expr_2.get(), loopDepth); };
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 3;
															#line 735 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >*>(static_cast<ProcessTransactionStateRequestPartActor*>(this)));
															#line 2317 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont8(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& cks,int loopDepth) 
	{
															#line 733 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		cipherKeys = cks;
															#line 2326 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && cks,int loopDepth) 
	{
															#line 733 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		cipherKeys = cks;
															#line 2335 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& cks,int loopDepth) 
	{
		loopDepth = a_body1cont8(cks, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && cks,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(cks), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state > 0) static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 0;
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >::remove();

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> const& value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> && value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*,Error err) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
															#line 737 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->processed = true;
															#line 2419 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
															#line 737 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		pContext->processed = true;
															#line 2428 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state > 0) static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 0;
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont11(Void const& _,int loopDepth) 
	{
															#line 742 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcessTransactionStateRequestPartActorState(); static_cast<ProcessTransactionStateRequestPartActor*>(this)->destroy(); return 0; }
															#line 2512 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		new (&static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcessTransactionStateRequestPartActorState();
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11(Void && _,int loopDepth) 
	{
															#line 742 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ProcessTransactionStateRequestPartActorState(); static_cast<ProcessTransactionStateRequestPartActor*>(this)->destroy(); return 0; }
															#line 2524 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		new (&static_cast<ProcessTransactionStateRequestPartActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ProcessTransactionStateRequestPartActorState();
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state > 0) static_cast<ProcessTransactionStateRequestPartActor*>(this)->actor_wait_state = 0;
		static_cast<ProcessTransactionStateRequestPartActor*>(this)->ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<Resolver> self;
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	TransactionStateResolveContext* pContext;
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	TxnStateRequest request;
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 725 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> cipherKeys;
															#line 2617 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via processTransactionStateRequestPart()
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class ProcessTransactionStateRequestPartActor final : public Actor<Void>, public ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >, public ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >, public ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >, public ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >, public FastAllocated<ProcessTransactionStateRequestPartActor>, public ProcessTransactionStateRequestPartActorState<ProcessTransactionStateRequestPartActor> {
															#line 2622 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
	using FastAllocated<ProcessTransactionStateRequestPartActor>::operator new;
	using FastAllocated<ProcessTransactionStateRequestPartActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12792114269153031936UL, 6998009768567518208UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >;
friend struct ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >;
friend struct ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >;
friend struct ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >;
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ProcessTransactionStateRequestPartActor(Reference<Resolver> const& self,TransactionStateResolveContext* const& pContext,TxnStateRequest const& request,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 2642 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ProcessTransactionStateRequestPartActorState<ProcessTransactionStateRequestPartActor>(self, pContext, request, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6249502656935116032UL, 7110093004433443328UL);
		ActorExecutionContextHelper __helper(static_cast<ProcessTransactionStateRequestPartActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("processTransactionStateRequestPart");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("processTransactionStateRequestPart", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ProcessTransactionStateRequestPartActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ProcessTransactionStateRequestPartActor, 1, std::unordered_map<EncryptCipherDomainId, Reference<BlobCipherKey>> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ProcessTransactionStateRequestPartActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ProcessTransactionStateRequestPartActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> processTransactionStateRequestPart( Reference<Resolver> const& self, TransactionStateResolveContext* const& pContext, TxnStateRequest const& request, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 695 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new ProcessTransactionStateRequestPartActor(self, pContext, request, db));
															#line 2677 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
}

#line 744 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"

} // anonymous namespace

															#line 2684 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via resolverCore()
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
template <class ResolverCoreActor>
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolverCoreActorState {
															#line 2691 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverCoreActorState(ResolverInterface const& resolver,InitializeResolverRequest const& initReq,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		 : resolver(resolver),
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   initReq(initReq),
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   db(db),
															#line 750 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   self(new Resolver(resolver.id(), initReq.commitProxyCount, initReq.resolverCount, initReq.encryptMode)),
															#line 752 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   actors(false),
															#line 753 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   doPollMetrics(self->resolverCount > 1 ? Void() : Future<Void>(Never())),
															#line 754 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   addActor()
															#line 2710 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("resolverCore", reinterpret_cast<unsigned long>(this));

	}
	~ResolverCoreActorState() 
	{
		fdb_probe_actor_destroy("resolverCore", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 755 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			actors.add(waitFailureServer(resolver.waitFailure.getFuture()));
															#line 756 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			actors.add(traceRole(Role::RESOLVER, resolver.id()));
															#line 758 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			TraceEvent("ResolverInit", resolver.id()) .detail("RecoveryCount", initReq.recoveryCount) .detail("EncryptMode", initReq.encryptMode.toString());
															#line 763 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			;
															#line 2731 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResolverCoreActorState();
		static_cast<ResolverCoreActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 770 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		self->logSystem = ILogSystem::fromServerDBInfo(resolver.id(), db->get(), false, addActor);
															#line 771 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		self->localTLogCount = db->get().logSystemConfig.numLogs();
															#line 772 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		onError = transformError(actorCollection(addActor.getFuture()), broken_promise(), resolver_failed());
															#line 774 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		transactionStateResolveContext = TransactionStateResolveContext();
															#line 775 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS)
															#line 2762 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 776 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->logAdapter = new LogSystemDiskQueueAdapter(self->logSystem, Reference<AsyncVar<PeekTxsInfo>>(), 1, false);
															#line 777 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->txnStateStore = keyValueStoreLogSystem( self->logAdapter, db, resolver.id(), 2e9, true, true, true, self->encryptMode.isEncryptionEnabled());
															#line 781 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			StrictFuture<Void> __when_expr_1 = success(self->txnStateStore->readValue(StringRef()));
															#line 781 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (static_cast<ResolverCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2772 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ResolverCoreActor*>(this)->actor_wait_state = 2;
															#line 781 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResolverCoreActor, 1, Void >*>(static_cast<ResolverCoreActor*>(this)));
															#line 2777 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 763 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!(!(initReq.masterLifetime.isEqual(db->get().masterLifetime) && db->get().recoveryState >= RecoveryState::RECOVERY_TRANSACTION)))
															#line 2798 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 766 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 766 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ResolverCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2806 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ResolverCoreActor*>(this)->actor_wait_state = 1;
															#line 766 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResolverCoreActor, 0, Void >*>(static_cast<ResolverCoreActor*>(this)));
															#line 2811 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResolverCoreActor*>(this)->actor_wait_state > 0) static_cast<ResolverCoreActor*>(this)->actor_wait_state = 0;
		static_cast<ResolverCoreActor*>(this)->ActorCallback< ResolverCoreActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResolverCoreActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 793 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		;
															#line 2920 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 784 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		transactionStateResolveContext = TransactionStateResolveContext(self, &addActor);
															#line 786 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (SERVER_KNOBS->ENABLE_VERSION_VECTOR_TLOG_UNICAST)
															#line 2931 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 787 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->numLogs = db->get().logSystemConfig.numLogs();
															#line 788 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->tpcvVector.resize(1 + self->numLogs, 0);
															#line 789 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			std::fill(self->tpcvVector.begin(), self->tpcvVector.end(), invalidVersion);
															#line 2939 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 784 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		transactionStateResolveContext = TransactionStateResolveContext(self, &addActor);
															#line 786 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (SERVER_KNOBS->ENABLE_VERSION_VECTOR_TLOG_UNICAST)
															#line 2951 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 787 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->numLogs = db->get().logSystemConfig.numLogs();
															#line 788 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			self->tpcvVector.resize(1 + self->numLogs, 0);
															#line 789 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			std::fill(self->tpcvVector.begin(), self->tpcvVector.end(), invalidVersion);
															#line 2959 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ResolverCoreActor*>(this)->actor_wait_state > 0) static_cast<ResolverCoreActor*>(this)->actor_wait_state = 0;
		static_cast<ResolverCoreActor*>(this)->ActorCallback< ResolverCoreActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResolverCoreActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 794 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		auto __when_expr_2 = resolver.resolve.getFuture();
															#line 3051 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_2), FutureStream<ResolveTransactionBatchRequest>>::value || std::is_same<decltype(__when_expr_2), ThreadFutureStream<ResolveTransactionBatchRequest>>::value, "invalid type");
															#line 793 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ResolverCoreActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3055 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_2.pop(), loopDepth); };
															#line 797 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		auto __when_expr_3 = resolver.metrics.getFuture();
															#line 3059 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_3), FutureStream<ResolutionMetricsRequest>>::value || std::is_same<decltype(__when_expr_3), ThreadFutureStream<ResolutionMetricsRequest>>::value, "invalid type");
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when2(__when_expr_3.pop(), loopDepth); };
															#line 802 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		auto __when_expr_4 = resolver.split.getFuture();
															#line 3064 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_4), FutureStream<ResolutionSplitRequest>>::value || std::is_same<decltype(__when_expr_4), ThreadFutureStream<ResolutionSplitRequest>>::value, "invalid type");
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when3(__when_expr_4.pop(), loopDepth); };
															#line 810 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_5 = actors.getResult();
															#line 3069 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when4(__when_expr_5.get(), loopDepth); };
															#line 811 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_6 = onError;
															#line 3073 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when5(__when_expr_6.get(), loopDepth); };
															#line 812 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_7 = doPollMetrics;
															#line 3077 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when6(__when_expr_7.get(), loopDepth); };
															#line 816 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		auto __when_expr_8 = resolver.txnState.getFuture();
															#line 3081 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_8), FutureStream<TxnStateRequest>>::value || std::is_same<decltype(__when_expr_8), ThreadFutureStream<TxnStateRequest>>::value, "invalid type");
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when7(__when_expr_8.pop(), loopDepth); };
		static_cast<ResolverCoreActor*>(this)->actor_wait_state = 3;
															#line 794 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >*>(static_cast<ResolverCoreActor*>(this)));
															#line 797 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >*>(static_cast<ResolverCoreActor*>(this)));
															#line 802 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >*>(static_cast<ResolverCoreActor*>(this)));
															#line 810 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ResolverCoreActor, 5, Void >*>(static_cast<ResolverCoreActor*>(this)));
															#line 811 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ResolverCoreActor, 6, Void >*>(static_cast<ResolverCoreActor*>(this)));
															#line 812 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< ResolverCoreActor, 7, Void >*>(static_cast<ResolverCoreActor*>(this)));
															#line 816 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >*>(static_cast<ResolverCoreActor*>(this)));
															#line 3099 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(ResolveTransactionBatchRequest const& batch,int loopDepth) 
	{
															#line 795 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		actors.add(resolveBatch(self, batch, db));
															#line 3114 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(ResolveTransactionBatchRequest && batch,int loopDepth) 
	{
															#line 795 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		actors.add(resolveBatch(self, batch, db));
															#line 3123 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when2(ResolutionMetricsRequest const& req,int loopDepth) 
	{
															#line 798 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->metricsRequests;
															#line 799 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		req.reply.send(self->iopsSample.getEstimate(SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS ? normalKeys : allKeys));
															#line 3134 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when2(ResolutionMetricsRequest && req,int loopDepth) 
	{
															#line 798 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->metricsRequests;
															#line 799 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		req.reply.send(self->iopsSample.getEstimate(SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS ? normalKeys : allKeys));
															#line 3145 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when3(ResolutionSplitRequest const& req,int loopDepth) 
	{
															#line 803 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->splitRequests;
															#line 804 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		ResolutionSplitReply rep;
															#line 805 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		rep.key = self->iopsSample.splitEstimate(req.range, req.offset, req.front);
															#line 806 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		rep.used = self->iopsSample.getEstimate(req.front ? KeyRangeRef(req.range.begin, rep.key) : KeyRangeRef(rep.key, req.range.end));
															#line 808 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		req.reply.send(rep);
															#line 3162 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when3(ResolutionSplitRequest && req,int loopDepth) 
	{
															#line 803 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		++self->splitRequests;
															#line 804 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		ResolutionSplitReply rep;
															#line 805 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		rep.key = self->iopsSample.splitEstimate(req.range, req.offset, req.front);
															#line 806 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		rep.used = self->iopsSample.getEstimate(req.front ? KeyRangeRef(req.range.begin, rep.key) : KeyRangeRef(rep.key, req.range.end));
															#line 808 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		req.reply.send(rep);
															#line 3179 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when4(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when4(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when6(Void const& _,int loopDepth) 
	{
															#line 813 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		self->iopsSample.poll();
															#line 814 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		doPollMetrics = delay(SERVER_KNOBS->SAMPLE_POLL_TIME);
															#line 3214 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when6(Void && _,int loopDepth) 
	{
															#line 813 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		self->iopsSample.poll();
															#line 814 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		doPollMetrics = delay(SERVER_KNOBS->SAMPLE_POLL_TIME);
															#line 3225 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when7(TxnStateRequest const& request,int loopDepth) 
	{
															#line 817 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS)
															#line 3234 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 818 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			addActor.send(processTransactionStateRequestPart(self, &transactionStateResolveContext, request, db));
															#line 3238 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 820 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(false);
															#line 3244 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when7(TxnStateRequest && request,int loopDepth) 
	{
															#line 817 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (SERVER_KNOBS->PROXY_USE_RESOLVER_PRIVATE_MUTATIONS)
															#line 3254 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 818 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			addActor.send(processTransactionStateRequestPart(self, &transactionStateResolveContext, request, db));
															#line 3258 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
		else
		{
															#line 820 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			ASSERT(false);
															#line 3264 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ResolverCoreActor*>(this)->actor_wait_state > 0) static_cast<ResolverCoreActor*>(this)->actor_wait_state = 0;
		static_cast<ResolverCoreActor*>(this)->ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorCallback< ResolverCoreActor, 5, Void >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorCallback< ResolverCoreActor, 6, Void >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorCallback< ResolverCoreActor, 7, Void >::remove();
		static_cast<ResolverCoreActor*>(this)->ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >::remove();

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >*,ResolveTransactionBatchRequest const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >*,ResolveTransactionBatchRequest && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >*,ResolutionMetricsRequest const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >*,ResolutionMetricsRequest && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >*,ResolutionSplitRequest const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when3(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >*,ResolutionSplitRequest && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when3(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when4(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when4(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ResolverCoreActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when5(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when5(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ResolverCoreActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when6(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< ResolverCoreActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when6(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< ResolverCoreActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >*,TxnStateRequest const& value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when7(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >*,TxnStateRequest && value) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1when7(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >*,Error err) 
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), 8);

	}
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverInterface resolver;
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	InitializeResolverRequest initReq;
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 750 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<Resolver> self;
															#line 752 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ActorCollection actors;
															#line 753 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Future<Void> doPollMetrics;
															#line 754 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	PromiseStream<Future<Void>> addActor;
															#line 772 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Future<Void> onError;
															#line 774 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	TransactionStateResolveContext transactionStateResolveContext;
															#line 3699 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via resolverCore()
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolverCoreActor final : public Actor<Void>, public ActorCallback< ResolverCoreActor, 0, Void >, public ActorCallback< ResolverCoreActor, 1, Void >, public ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >, public ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >, public ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >, public ActorCallback< ResolverCoreActor, 5, Void >, public ActorCallback< ResolverCoreActor, 6, Void >, public ActorCallback< ResolverCoreActor, 7, Void >, public ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >, public FastAllocated<ResolverCoreActor>, public ResolverCoreActorState<ResolverCoreActor> {
															#line 3704 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
	using FastAllocated<ResolverCoreActor>::operator new;
	using FastAllocated<ResolverCoreActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(3371894482745382400UL, 511544889623850240UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResolverCoreActor, 0, Void >;
friend struct ActorCallback< ResolverCoreActor, 1, Void >;
friend struct ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >;
friend struct ActorSingleCallback< ResolverCoreActor, 3, ResolutionMetricsRequest >;
friend struct ActorSingleCallback< ResolverCoreActor, 4, ResolutionSplitRequest >;
friend struct ActorCallback< ResolverCoreActor, 5, Void >;
friend struct ActorCallback< ResolverCoreActor, 6, Void >;
friend struct ActorCallback< ResolverCoreActor, 7, Void >;
friend struct ActorSingleCallback< ResolverCoreActor, 8, TxnStateRequest >;
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverCoreActor(ResolverInterface const& resolver,InitializeResolverRequest const& initReq,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 3729 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ResolverCoreActorState<ResolverCoreActor>(resolver, initReq, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resolverCore", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(574549493760680960UL, 16065327264668821760UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverCoreActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resolverCore");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resolverCore", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResolverCoreActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ResolverCoreActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorSingleCallback< ResolverCoreActor, 2, ResolveTransactionBatchRequest >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> resolverCore( ResolverInterface const& resolver, InitializeResolverRequest const& initReq, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 747 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new ResolverCoreActor(resolver, initReq, db));
															#line 3764 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
}

#line 825 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"

															#line 3769 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via checkRemoved()
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
template <class CheckRemovedActor>
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class CheckRemovedActorState {
															#line 3776 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	CheckRemovedActorState(Reference<AsyncVar<ServerDBInfo> const> const& db,uint64_t const& recoveryCount,ResolverInterface const& myInterface) 
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		 : db(db),
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   recoveryCount(recoveryCount),
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   myInterface(myInterface)
															#line 3787 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("checkRemoved", reinterpret_cast<unsigned long>(this));

	}
	~CheckRemovedActorState() 
	{
		fdb_probe_actor_destroy("checkRemoved", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 829 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			;
															#line 3802 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckRemovedActorState();
		static_cast<CheckRemovedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 830 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (db->get().recoveryCount >= recoveryCount && std::find(db->get().resolvers.begin(), db->get().resolvers.end(), myInterface) == db->get().resolvers.end())
															#line 3832 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		{
															#line 833 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			return a_body1Catch1(worker_removed(), std::max(0, loopDepth - 1));
															#line 3836 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
															#line 835 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->onChange();
															#line 835 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<CheckRemovedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3842 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<CheckRemovedActor*>(this)->actor_wait_state = 1;
															#line 835 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckRemovedActor, 0, Void >*>(static_cast<CheckRemovedActor*>(this)));
															#line 3847 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckRemovedActor*>(this)->actor_wait_state > 0) static_cast<CheckRemovedActor*>(this)->actor_wait_state = 0;
		static_cast<CheckRemovedActor*>(this)->ActorCallback< CheckRemovedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckRemovedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckRemovedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckRemovedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	uint64_t recoveryCount;
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverInterface myInterface;
															#line 3945 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via checkRemoved()
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class CheckRemovedActor final : public Actor<Void>, public ActorCallback< CheckRemovedActor, 0, Void >, public FastAllocated<CheckRemovedActor>, public CheckRemovedActorState<CheckRemovedActor> {
															#line 3950 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
	using FastAllocated<CheckRemovedActor>::operator new;
	using FastAllocated<CheckRemovedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10293314742454122240UL, 18136178359364109568UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckRemovedActor, 0, Void >;
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	CheckRemovedActor(Reference<AsyncVar<ServerDBInfo> const> const& db,uint64_t const& recoveryCount,ResolverInterface const& myInterface) 
															#line 3967 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   CheckRemovedActorState<CheckRemovedActor>(db, recoveryCount, myInterface),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkRemoved", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(15518436168711072512UL, 6662910151609573120UL);
		ActorExecutionContextHelper __helper(static_cast<CheckRemovedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkRemoved");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkRemoved", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckRemovedActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> checkRemoved( Reference<AsyncVar<ServerDBInfo> const> const& db, uint64_t const& recoveryCount, ResolverInterface const& myInterface ) {
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new CheckRemovedActor(db, recoveryCount, myInterface));
															#line 4000 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
}

#line 838 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"

															#line 4005 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via resolver()
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
template <class ResolverActor>
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolverActorState {
															#line 4012 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverActorState(ResolverInterface const& resolver,InitializeResolverRequest const& initReq,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		 : resolver(resolver),
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   initReq(initReq),
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		   db(db)
															#line 4023 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("resolver", reinterpret_cast<unsigned long>(this));

	}
	~ResolverActorState() 
	{
		fdb_probe_actor_destroy("resolver", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 843 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				core = resolverCore(resolver, initReq, db);
															#line 844 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				;
															#line 4041 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ResolverActorState();
		static_cast<ResolverActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 851 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			if (e.code() == error_code_actor_cancelled || e.code() == error_code_worker_removed)
															#line 4071 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
			{
															#line 852 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				TraceEvent("ResolverTerminated", resolver.id()).errorUnsuppressed(e);
															#line 853 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
				if (!static_cast<ResolverActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResolverActorState(); static_cast<ResolverActor*>(this)->destroy(); return 0; }
															#line 4077 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
				new (&static_cast<ResolverActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~ResolverActorState();
				static_cast<ResolverActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 855 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 4085 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 845 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_0 = core;
															#line 844 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (static_cast<ResolverActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4108 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 848 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		StrictFuture<Void> __when_expr_1 = checkRemoved(db, initReq.recoveryCount, resolver);
															#line 4112 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<ResolverActor*>(this)->actor_wait_state = 1;
															#line 845 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ResolverActor, 0, Void >*>(static_cast<ResolverActor*>(this)));
															#line 848 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ResolverActor, 1, Void >*>(static_cast<ResolverActor*>(this)));
															#line 4119 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 846 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ResolverActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResolverActorState(); static_cast<ResolverActor*>(this)->destroy(); return 0; }
															#line 4134 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		new (&static_cast<ResolverActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResolverActorState();
		static_cast<ResolverActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 846 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
		if (!static_cast<ResolverActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ResolverActorState(); static_cast<ResolverActor*>(this)->destroy(); return 0; }
															#line 4146 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		new (&static_cast<ResolverActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~ResolverActorState();
		static_cast<ResolverActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ResolverActor*>(this)->actor_wait_state > 0) static_cast<ResolverActor*>(this)->actor_wait_state = 0;
		static_cast<ResolverActor*>(this)->ActorCallback< ResolverActor, 0, Void >::remove();
		static_cast<ResolverActor*>(this)->ActorCallback< ResolverActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ResolverActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResolverActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ResolverActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ResolverActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ResolverActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ResolverActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverInterface resolver;
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	InitializeResolverRequest initReq;
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Reference<AsyncVar<ServerDBInfo> const> db;
															#line 843 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	Future<Void> core;
															#line 4295 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
};
// This generated class is to be used only via resolver()
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
class ResolverActor final : public Actor<Void>, public ActorCallback< ResolverActor, 0, Void >, public ActorCallback< ResolverActor, 1, Void >, public FastAllocated<ResolverActor>, public ResolverActorState<ResolverActor> {
															#line 4300 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
public:
	using FastAllocated<ResolverActor>::operator new;
	using FastAllocated<ResolverActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15501308702183256320UL, 9871983606068145408UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ResolverActor, 0, Void >;
friend struct ActorCallback< ResolverActor, 1, Void >;
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	ResolverActor(ResolverInterface const& resolver,InitializeResolverRequest const& initReq,Reference<AsyncVar<ServerDBInfo> const> const& db) 
															#line 4318 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ResolverActorState<ResolverActor>(resolver, initReq, db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("resolver", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18233311343312639744UL, 9120145405007874816UL);
		ActorExecutionContextHelper __helper(static_cast<ResolverActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("resolver");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("resolver", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ResolverActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
[[nodiscard]] Future<Void> resolver( ResolverInterface const& resolver, InitializeResolverRequest const& initReq, Reference<AsyncVar<ServerDBInfo> const> const& db ) {
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
	return Future<Void>(new ResolverActor(resolver, initReq, db));
															#line 4351 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbserver/Resolver.actor.g.cpp.py_gen"
}

#line 858 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbserver/Resolver.actor.cpp"
