#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
/*
 * BackupAgentBase.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <iomanip>
#include <time.h>

#include "fdbclient/BackupAgent.actor.h"
#include "fdbclient/BlobCipher.h"
#include "fdbclient/CommitProxyInterface.h"
#include "fdbclient/CommitTransaction.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/GetEncryptCipherKeys.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/SystemData.h"
#include "fdbrpc/simulator.h"
#include "flow/ActorCollection.h"
#include "flow/DeterministicRandom.h"
#include "flow/network.h"

#include "flow/actorcompiler.h" // has to be last include

std::string BackupAgentBase::formatTime(int64_t epochs) {
	time_t curTime = (time_t)epochs;
	char buffer[30];
	struct tm timeinfo;
	getLocalTime(&curTime, &timeinfo);
	strftime(buffer, 30, "%Y/%m/%d.%H:%M:%S%z", &timeinfo);
	return buffer;
}

int64_t BackupAgentBase::parseTime(std::string timestamp) {
	struct tm out;
	out.tm_isdst = -1; // This field is not set by strptime. -1 tells mktime to determine whether DST is in effect

	std::string timeOnly = timestamp.substr(0, 19);

	// TODO:  Use std::get_time implementation for all platforms once supported
	// It would be nice to read the timezone using %z, but it seems not all get_time()
	// or strptime() implementations handle it correctly in all environments so we
	// will read the date and time independent of timezone at first and then adjust it.
#ifdef _WIN32
	std::istringstream s(timeOnly);
	s.imbue(std::locale(setlocale(LC_TIME, nullptr)));
	s >> std::get_time(&out, "%Y/%m/%d.%H:%M:%S");
	if (s.fail()) {
		return -1;
	}
#else
	if (strptime(timeOnly.c_str(), "%Y/%m/%d.%H:%M:%S", &out) == nullptr) {
		return -1;
	}
#endif

	// Read timezone offset in +/-HHMM format then convert to seconds
	int tzHH;
	int tzMM;
	if (sscanf(timestamp.substr(19, 5).c_str(), "%3d%2d", &tzHH, &tzMM) != 2) {
		return -1;
	}
	if (tzHH < 0) {
		tzMM = -tzMM;
	}
	// tzOffset is the number of seconds EAST of GMT
	int tzOffset = tzHH * 60 * 60 + tzMM * 60;

	// The goal is to convert the timestamp string to epoch seconds assuming the date/time was expressed in the timezone
	// at the end of the string. However, mktime() will ONLY return epoch seconds assuming the date/time is expressed in
	// local time (based on locale / environment) mktime() will set out.tm_gmtoff when available
	int64_t ts = mktime(&out);

	// localTZOffset is the number of seconds EAST of GMT
	long localTZOffset;
#ifdef _WIN32
	// _get_timezone() returns the number of seconds WEST of GMT
	if (_get_timezone(&localTZOffset) != 0) {
		return -1;
	}
	// Negate offset to match the orientation of tzOffset
	localTZOffset = -localTZOffset;
#else
	// tm.tm_gmtoff is the number of seconds EAST of GMT
	localTZOffset = out.tm_gmtoff;
#endif

	// Add back the difference between the local timezone assumed by mktime() and the intended timezone from the input
	// string
	ts += (localTZOffset - tzOffset);
	return ts;
}

const Key BackupAgentBase::keyFolderId = "config_folderid"_sr;
const Key BackupAgentBase::keyBeginVersion = "beginVersion"_sr;
const Key BackupAgentBase::keyEndVersion = "endVersion"_sr;
const Key BackupAgentBase::keyPrevBeginVersion = "prevBeginVersion"_sr;
const Key BackupAgentBase::keyConfigBackupTag = "config_backup_tag"_sr;
const Key BackupAgentBase::keyConfigLogUid = "config_log_uid"_sr;
const Key BackupAgentBase::keyConfigBackupRanges = "config_backup_ranges"_sr;
const Key BackupAgentBase::keyConfigStopWhenDoneKey = "config_stop_when_done"_sr;
const Key BackupAgentBase::keyStateStop = "state_stop"_sr;
const Key BackupAgentBase::keyStateStatus = "state_status"_sr;
const Key BackupAgentBase::keyStateLogBeginVersion = "last_begin_version"_sr;
const Key BackupAgentBase::keyLastUid = "last_uid"_sr;
const Key BackupAgentBase::keyBeginKey = "beginKey"_sr;
const Key BackupAgentBase::keyEndKey = "endKey"_sr;
const Key BackupAgentBase::keyDrVersion = "drVersion"_sr;
const Key BackupAgentBase::destUid = "destUid"_sr;
const Key BackupAgentBase::backupStartVersion = "backupStartVersion"_sr;

const Key BackupAgentBase::keyTagName = "tagname"_sr;
const Key BackupAgentBase::keyStates = "state"_sr;
const Key BackupAgentBase::keyConfig = "config"_sr;
const Key BackupAgentBase::keyErrors = "errors"_sr;
const Key BackupAgentBase::keyRanges = "ranges"_sr;
const Key BackupAgentBase::keyTasks = "tasks"_sr;
const Key BackupAgentBase::keyFutures = "futures"_sr;
const Key BackupAgentBase::keySourceStates = "source_states"_sr;
const Key BackupAgentBase::keySourceTagName = "source_tagname"_sr;

bool copyParameter(Reference<Task> source, Reference<Task> dest, Key key) {
	if (source) {
		dest->params[key] = source->params[key];
		return true;
	}

	return false;
}

Version getVersionFromString(std::string const& value) {
	Version version = invalidVersion;
	int n = 0;
	if (sscanf(value.c_str(), "%lld%n", (long long*)&version, &n) != 1 || n != value.size()) {
		TraceEvent(SevWarnAlways, "GetVersionFromString").detail("InvalidVersion", value);
		throw restore_invalid_version();
	}
	return version;
}

// Transaction log data is stored by the FoundationDB core in the
// "backupLogKeys" (i.e., \xff\x02/blog/) keyspace in a funny order for
// performance reasons.
// Returns the ranges of keys that contain the data for the given range
// of versions.
// assert CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE % blocksize = 0. Otherwise calculation of hash will be incorrect
Standalone<VectorRef<KeyRangeRef>> getLogRanges(Version beginVersion,
                                                Version endVersion,
                                                Key destUidValue,
                                                int blockSize) {
	Standalone<VectorRef<KeyRangeRef>> ret;

	Key baLogRangePrefix = destUidValue.withPrefix(backupLogKeys.begin);

	//TraceEvent("GetLogRanges").detail("DestUidValue", destUidValue).detail("Prefix", baLogRangePrefix);

	for (int64_t vblock = beginVersion / blockSize; vblock < (endVersion + blockSize - 1) / blockSize; ++vblock) {
		int64_t tb = vblock * blockSize / CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE;
		uint64_t bv = bigEndian64(std::max(beginVersion, vblock * blockSize));
		uint64_t ev = bigEndian64(std::min(endVersion, (vblock + 1) * blockSize));
		uint32_t data = tb & 0xffffffff;
		uint8_t hash = (uint8_t)hashlittle(&data, sizeof(uint32_t), 0);

		Key vblockPrefix = StringRef(&hash, sizeof(uint8_t)).withPrefix(baLogRangePrefix);

		ret.push_back_deep(ret.arena(),
		                   KeyRangeRef(StringRef((uint8_t*)&bv, sizeof(uint64_t)).withPrefix(vblockPrefix),
		                               StringRef((uint8_t*)&ev, sizeof(uint64_t)).withPrefix(vblockPrefix)));
	}

	return ret;
}

// Given a begin and end version, get the prefix in the database for this range
// i.e. applyLogKeys.begin/backupUid/hash(uint8)/version(64bites)/part
// returns multiple key ranges, each of length APPLY_BLOCK_SIZE
// e.g. (64, 200) -> [(64, 128), (128, 192), (192, 200)]
Standalone<VectorRef<KeyRangeRef>> getApplyRanges(Version beginVersion, Version endVersion, Key backupUid) {
	Standalone<VectorRef<KeyRangeRef>> ret;
	Key baLogRangePrefix = backupUid.withPrefix(applyLogKeys.begin);

	//TraceEvent("GetLogRanges").detail("BackupUid", backupUid).detail("Prefix", baLogRangePrefix);

	for (int64_t vblock = beginVersion / CLIENT_KNOBS->APPLY_BLOCK_SIZE;
	     vblock < (endVersion + CLIENT_KNOBS->APPLY_BLOCK_SIZE - 1) / CLIENT_KNOBS->APPLY_BLOCK_SIZE;
	     ++vblock) {
		int64_t tb = vblock * CLIENT_KNOBS->APPLY_BLOCK_SIZE / CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE;
		uint64_t bv = bigEndian64(std::max(beginVersion, vblock * CLIENT_KNOBS->APPLY_BLOCK_SIZE));
		uint64_t ev = bigEndian64(std::min(endVersion, (vblock + 1) * CLIENT_KNOBS->APPLY_BLOCK_SIZE));
		uint32_t data = tb & 0xffffffff;
		uint8_t hash = (uint8_t)hashlittle(&data, sizeof(uint32_t), 0);

		Key vblockPrefix = StringRef(&hash, sizeof(uint8_t)).withPrefix(baLogRangePrefix);

		ret.push_back_deep(ret.arena(),
		                   KeyRangeRef(StringRef((uint8_t*)&bv, sizeof(uint64_t)).withPrefix(vblockPrefix),
		                               StringRef((uint8_t*)&ev, sizeof(uint64_t)).withPrefix(vblockPrefix)));
	}

	return ret;
}

Key getApplyKey(Version version, Key backupUid) {
	int64_t vblock = (version - 1) / CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE;
	uint64_t v = bigEndian64(version);
	uint32_t data = vblock & 0xffffffff;
	uint8_t hash = (uint8_t)hashlittle(&data, sizeof(uint32_t), 0);
	Key k1 = StringRef((uint8_t*)&v, sizeof(uint64_t)).withPrefix(StringRef(&hash, sizeof(uint8_t)));
	Key k2 = k1.withPrefix(backupUid);
	return k2.withPrefix(applyLogKeys.begin);
}

// It's important to keep the hash value consistent with the one used in getLogRanges.
// Otherwise, the same version will result in different keys.
Key getLogKey(Version version, Key backupUid, int blockSize) {
	int64_t vblock = version / blockSize;
	vblock = vblock * blockSize / CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE;
	uint64_t v = bigEndian64(version);
	uint32_t data = vblock & 0xffffffff;
	uint8_t hash = (uint8_t)hashlittle(&data, sizeof(uint32_t), 0);
	Key k1 = StringRef((uint8_t*)&v, sizeof(uint64_t)).withPrefix(StringRef(&hash, sizeof(uint8_t)));
	Key k2 = k1.withPrefix(backupUid);
	return k2.withPrefix(backupLogKeys.begin);
}

namespace {
															#line 244 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
// This generated class is to be used only via flowTestCase242()
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class FlowTestCase242Actor>
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class FlowTestCase242ActorState {
															#line 250 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowTestCase242ActorState(UnitTestParameters const& params) 
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : params(params)
															#line 257 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("flowTestCase242", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase242ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase242", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 243 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			std::vector<Version> versions = { 100, 841000000 };
															#line 244 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for(int i = 0;i < 10;i++) {
															#line 245 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				versions.push_back(deterministicRandom()->randomInt64(0, std::numeric_limits<int32_t>::max()));
															#line 276 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 247 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key backupUid = "backupUid0"_sr;
															#line 248 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			int blockSize = deterministicRandom()->coinflip() ? CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE : 100'000;
															#line 249 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for( const auto v : versions ) {
															#line 250 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Key k = getLogKey(v, backupUid, blockSize);
															#line 251 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Standalone<VectorRef<KeyRangeRef>> ranges = getLogRanges(v, v + 1, backupUid, blockSize);
															#line 252 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				ASSERT(ranges[0].contains(k));
															#line 290 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 254 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<FlowTestCase242Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase242ActorState(); static_cast<FlowTestCase242Actor*>(this)->destroy(); return 0; }
															#line 294 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<FlowTestCase242Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase242ActorState();
			static_cast<FlowTestCase242Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase242ActorState();
		static_cast<FlowTestCase242Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	UnitTestParameters params;
															#line 318 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via flowTestCase242()
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class FlowTestCase242Actor final : public Actor<Void>, public FastAllocated<FlowTestCase242Actor>, public FlowTestCase242ActorState<FlowTestCase242Actor> {
															#line 323 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<FlowTestCase242Actor>::operator new;
	using FastAllocated<FlowTestCase242Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6382914553360497152UL, 13239372739917059328UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowTestCase242Actor(UnitTestParameters const& params) 
															#line 339 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   FlowTestCase242ActorState<FlowTestCase242Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase242", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2822182176824976640UL, 17533110519103100160UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase242Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase242");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase242", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
static Future<Void> flowTestCase242( UnitTestParameters const& params ) {
															#line 242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new FlowTestCase242Actor(params));
															#line 370 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}
ACTOR_TEST_CASE(flowTestCase242, "/backup/logversion")

#line 256 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
} // namespace

Version getLogKeyVersion(Key key) {
	return bigEndian64(*(int64_t*)(key.begin() + backupLogPrefixBytes + sizeof(UID) + sizeof(uint8_t)));
}

// Given a key from one of the ranges returned by get_log_ranges,
// returns(version, part) where version is the database version number of
// the transaction log data in the value, and part is 0 for the first such
// data for a given version, 1 for the second block of data, etc.
std::pair<Version, uint32_t> decodeBKMutationLogKey(Key key) {
	return std::make_pair(
	    getLogKeyVersion(key),
	    bigEndian32(*(int32_t*)(key.begin() + backupLogPrefixBytes + sizeof(UID) + sizeof(uint8_t) + sizeof(int64_t))));
}

void _addResult(VectorRef<MutationRef>* result, int* mutationSize, Arena* arena, MutationRef logValue) {
	result->push_back_deep(*arena, logValue);
	*mutationSize += logValue.expectedSize();
}

/*
 This actor is responsible for taking an original transaction which was added to the backup mutation log (represented
 by "value" parameter), breaking it up into the individual MutationRefs (that constitute the transaction), decrypting
 each mutation (if needed) and adding/removing prefixes from the mutations. The final mutations are then added to the
 "result" vector alongside their encrypted counterparts (which is added to the "encryptedResult" vector)
 Each `value` is a param2
*/
															#line 403 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via decodeBackupLogValue()
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class DecodeBackupLogValueActor>
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class DecodeBackupLogValueActorState {
															#line 410 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	DecodeBackupLogValueActorState(Arena* const& arena,VectorRef<MutationRef>* const& result,VectorRef<Optional<MutationRef>>* const& encryptedResult,int* const& mutationSize,Standalone<StringRef> const& value,Key const& addPrefix,Key const& removePrefix,Version const& version,Reference<KeyRangeMap<Version>> const& key_version,Database const& cx,bool const& provisionalProxy,std::shared_ptr<DatabaseConfiguration> const& dbConfig) 
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : arena(arena),
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   result(result),
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   encryptedResult(encryptedResult),
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   mutationSize(mutationSize),
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   value(value),
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   addPrefix(addPrefix),
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   removePrefix(removePrefix),
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   version(version),
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   key_version(key_version),
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   cx(cx),
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   provisionalProxy(provisionalProxy),
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   dbConfig(dbConfig)
															#line 439 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("decodeBackupLogValue", reinterpret_cast<unsigned long>(this));

	}
	~DecodeBackupLogValueActorState() 
	{
		fdb_probe_actor_destroy("decodeBackupLogValue", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 297 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				offset = uint64_t(0);
															#line 298 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				uint64_t protocolVersion = 0;
															#line 299 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				memcpy(&protocolVersion, value.begin(), sizeof(uint64_t));
															#line 300 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				offset += sizeof(uint64_t);
															#line 301 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (protocolVersion <= 0x0FDB00A200090001)
															#line 463 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 302 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					TraceEvent(SevError, "DecodeBackupLogValue") .detail("IncompatibleProtocolVersion", protocolVersion) .detail("ValueSize", value.size()) .detail("Value", value);
															#line 306 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					return a_body1Catch2(incompatible_protocol_version(), loopDepth);
															#line 469 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
															#line 309 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				totalBytes = 0;
															#line 310 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				memcpy(&totalBytes, value.begin() + offset, sizeof(uint32_t));
															#line 311 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				offset += sizeof(uint32_t);
															#line 312 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				consumed = 0;
															#line 314 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (totalBytes + offset > value.size())
															#line 481 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 315 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					TraceEvent(SevError, "OffsetOutOfBoundary") .detail("TotalBytes", totalBytes) .detail("Offset", offset) .detail("Version", version) .detail("ValueSize", value.size());
															#line 320 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					return a_body1Catch2(restore_missing_data(), loopDepth);
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
															#line 323 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				originalOffset = offset;
															#line 325 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				;
															#line 493 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = a_body1loopHead1(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DecodeBackupLogValueActorState();
		static_cast<DecodeBackupLogValueActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 461 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<DecodeBackupLogValueActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DecodeBackupLogValueActorState(); static_cast<DecodeBackupLogValueActor*>(this)->destroy(); return 0; }
															#line 522 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<DecodeBackupLogValueActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~DecodeBackupLogValueActorState();
		static_cast<DecodeBackupLogValueActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 454 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			TraceEvent(e.code() == error_code_restore_missing_data ? SevWarn : SevError, "BA_DecodeBackupLogValue") .error(e) .GetLastError() .detail("ValueSize", value.size()) .detail("Value", value);
															#line 459 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 537 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(int loopDepth) 
	{
															#line 442 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		ASSERT(consumed == totalBytes);
															#line 443 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (value.size() != offset)
															#line 553 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 444 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			TraceEvent(SevError, "BA_DecodeBackupLogValue") .detail("UnexpectedExtraDataSize", value.size()) .detail("Offset", offset) .detail("GroupKey", version) .detail("TotalBytes", totalBytes) .detail("Consumed", consumed) .detail("OriginalOffset", originalOffset);
															#line 451 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			return a_body1Catch2(restore_corrupted_data(), loopDepth);
															#line 559 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 325 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!(consumed < totalBytes))
															#line 576 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 326 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		uint32_t type = 0;
															#line 328 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		memcpy(&type, value.begin() + offset, sizeof(uint32_t));
															#line 329 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		offset += sizeof(uint32_t);
															#line 330 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		len1 = 0;
															#line 331 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		memcpy(&len1, value.begin() + offset, sizeof(uint32_t));
															#line 332 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		offset += sizeof(uint32_t);
															#line 333 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		len2 = 0;
															#line 334 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		memcpy(&len2, value.begin() + offset, sizeof(uint32_t));
															#line 335 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		offset += sizeof(uint32_t);
															#line 337 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		ASSERT(offset + len1 + len2 <= value.size() && isValidMutationType(type));
															#line 340 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		logValue = MutationRef();
															#line 341 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		tempArena = Arena();
															#line 342 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		logValue.type = type;
															#line 343 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		logValue.param1 = value.substr(offset, len1);
															#line 344 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		offset += len1;
															#line 345 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		logValue.param2 = value.substr(offset, len2);
															#line 346 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		offset += len2;
															#line 347 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		encryptedLogValue = Optional<MutationRef>();
															#line 348 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		ASSERT(!dbConfig->encryptionAtRestMode.isEncryptionEnabled() || logValue.isEncrypted());
															#line 352 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (logValue.isEncrypted())
															#line 620 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 353 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			encryptedLogValue = logValue;
															#line 354 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			domainId = logValue.encryptDomainId();
															#line 355 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Reference<AsyncVar<ClientDBInfo> const> dbInfo = cx->clientInfo;
															#line 628 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			try {
															#line 357 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<TextAndHeaderCipherKeys> __when_expr_0 = GetEncryptCipherKeys<ClientDBInfo>::getEncryptCipherKeys( dbInfo, logValue.configurableEncryptionHeader(), BlobCipherMetrics::RESTORE);
															#line 357 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<DecodeBackupLogValueActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 634 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
				static_cast<DecodeBackupLogValueActor*>(this)->actor_wait_state = 1;
															#line 357 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DecodeBackupLogValueActor, 0, TextAndHeaderCipherKeys >*>(static_cast<DecodeBackupLogValueActor*>(this)));
															#line 639 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1loopBody1Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 375 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		ASSERT(!logValue.isEncrypted());
															#line 377 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		MutationRef originalLogValue = logValue;
															#line 379 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (logValue.type == MutationRef::ClearRange)
															#line 676 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 380 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			KeyRangeRef range(logValue.param1, logValue.param2);
															#line 381 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			auto ranges = key_version->intersectingRanges(range);
															#line 382 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for( auto r : ranges ) {
															#line 383 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (version > r.value() && r.value() != invalidVersion)
															#line 686 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 384 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					KeyRef minKey = std::min(r.range().end, range.end);
															#line 385 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (minKey == (removePrefix == StringRef() ? allKeys.end : strinc(removePrefix)))
															#line 692 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					{
															#line 386 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						logValue.param1 = std::max(r.range().begin, range.begin);
															#line 387 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						if (removePrefix.size())
															#line 698 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						{
															#line 388 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
							logValue.param1 = logValue.param1.removePrefix(removePrefix);
															#line 702 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						}
															#line 390 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						if (addPrefix.size())
															#line 706 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						{
															#line 391 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
							logValue.param1 = logValue.param1.withPrefix(addPrefix, tempArena);
															#line 710 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						}
															#line 393 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						logValue.param2 = addPrefix == StringRef() ? allKeys.end : strinc(addPrefix, tempArena);
															#line 394 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						_addResult(result, mutationSize, arena, logValue);
															#line 716 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
					else
					{
															#line 396 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						logValue.param1 = std::max(r.range().begin, range.begin);
															#line 397 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						logValue.param2 = minKey;
															#line 398 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						if (removePrefix.size())
															#line 726 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						{
															#line 399 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
							logValue.param1 = logValue.param1.removePrefix(removePrefix);
															#line 400 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
							logValue.param2 = logValue.param2.removePrefix(removePrefix);
															#line 732 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						}
															#line 402 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						if (addPrefix.size())
															#line 736 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						{
															#line 403 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
							logValue.param1 = logValue.param1.withPrefix(addPrefix, tempArena);
															#line 404 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
							logValue.param2 = logValue.param2.withPrefix(addPrefix, tempArena);
															#line 742 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						}
															#line 406 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						_addResult(result, mutationSize, arena, logValue);
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
															#line 408 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (originalLogValue.param1 == logValue.param1 && originalLogValue.param2 == logValue.param2)
															#line 750 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					{
															#line 409 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						encryptedResult->push_back_deep(*arena, encryptedLogValue);
															#line 754 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
					else
					{
															#line 411 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						encryptedResult->push_back_deep(*arena, Optional<MutationRef>());
															#line 760 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
				}
			}
		}
		else
		{
															#line 416 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Version ver = key_version->rangeContaining(logValue.param1).value();
															#line 421 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (version > ver && ver != invalidVersion)
															#line 771 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 422 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (removePrefix.size())
															#line 775 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 423 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					logValue.param1 = logValue.param1.removePrefix(removePrefix);
															#line 779 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
															#line 425 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (addPrefix.size())
															#line 783 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 426 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					logValue.param1 = logValue.param1.withPrefix(addPrefix, tempArena);
															#line 787 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
															#line 428 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				_addResult(result, mutationSize, arena, logValue);
															#line 431 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (originalLogValue.param1 == logValue.param1 && originalLogValue.param2 == logValue.param2)
															#line 793 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 432 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					encryptedResult->push_back_deep(*arena, encryptedLogValue);
															#line 797 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 434 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					encryptedResult->push_back_deep(*arena, Optional<MutationRef>());
															#line 803 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
			}
		}
															#line 439 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		consumed += BackupAgentBase::logHeaderSize + len1 + len2;
															#line 809 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 362 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			TraceEvent(SevWarnAlways, "MutationLogRestoreEncryptKeyFetchFailed") .detail("Version", version) .detail("TenantId", domainId);
															#line 365 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (e.code() == error_code_encrypt_keys_fetch_failed || e.code() == error_code_encrypt_key_not_found)
															#line 827 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 367 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				CODE_PROBE(true, "mutation log restore encrypt keys not found", probe::decoration::rare);
															#line 368 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				consumed += BackupAgentBase::logHeaderSize + len1 + len2;
															#line 833 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				return a_body1loopHead1(loopDepth); // continue
			}
			else
			{
															#line 371 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				return a_body1Catch2(e, std::max(0, loopDepth - 1));
															#line 840 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4(TextAndHeaderCipherKeys const& cipherKeys,int loopDepth) 
	{
															#line 359 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		logValue = logValue.decrypt(cipherKeys, tempArena, BlobCipherMetrics::RESTORE);
															#line 855 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4(TextAndHeaderCipherKeys && cipherKeys,int loopDepth) 
	{
															#line 359 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		logValue = logValue.decrypt(cipherKeys, tempArena, BlobCipherMetrics::RESTORE);
															#line 864 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(TextAndHeaderCipherKeys const& cipherKeys,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(cipherKeys, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(TextAndHeaderCipherKeys && cipherKeys,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(std::move(cipherKeys), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DecodeBackupLogValueActor*>(this)->actor_wait_state > 0) static_cast<DecodeBackupLogValueActor*>(this)->actor_wait_state = 0;
		static_cast<DecodeBackupLogValueActor*>(this)->ActorCallback< DecodeBackupLogValueActor, 0, TextAndHeaderCipherKeys >::remove();

	}
	void a_callback_fire(ActorCallback< DecodeBackupLogValueActor, 0, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys const& value) 
	{
		fdb_probe_actor_enter("decodeBackupLogValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DecodeBackupLogValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decodeBackupLogValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DecodeBackupLogValueActor, 0, TextAndHeaderCipherKeys >*,TextAndHeaderCipherKeys && value) 
	{
		fdb_probe_actor_enter("decodeBackupLogValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DecodeBackupLogValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decodeBackupLogValue", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DecodeBackupLogValueActor, 0, TextAndHeaderCipherKeys >*,Error err) 
	{
		fdb_probe_actor_enter("decodeBackupLogValue", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DecodeBackupLogValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("decodeBackupLogValue", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Arena* arena;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	VectorRef<MutationRef>* result;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	VectorRef<Optional<MutationRef>>* encryptedResult;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int* mutationSize;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Standalone<StringRef> value;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key addPrefix;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key removePrefix;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version version;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<KeyRangeMap<Version>> key_version;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	bool provisionalProxy;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::shared_ptr<DatabaseConfiguration> dbConfig;
															#line 297 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint64_t offset;
															#line 309 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint32_t totalBytes;
															#line 312 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint32_t consumed;
															#line 323 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int originalOffset;
															#line 330 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint32_t len1;
															#line 333 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint32_t len2;
															#line 340 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	MutationRef logValue;
															#line 341 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Arena tempArena;
															#line 347 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Optional<MutationRef> encryptedLogValue;
															#line 354 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	EncryptCipherDomainId domainId;
															#line 1014 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via decodeBackupLogValue()
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class DecodeBackupLogValueActor final : public Actor<Void>, public ActorCallback< DecodeBackupLogValueActor, 0, TextAndHeaderCipherKeys >, public FastAllocated<DecodeBackupLogValueActor>, public DecodeBackupLogValueActorState<DecodeBackupLogValueActor> {
															#line 1019 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<DecodeBackupLogValueActor>::operator new;
	using FastAllocated<DecodeBackupLogValueActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17768509748045335808UL, 6651391252387942400UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DecodeBackupLogValueActor, 0, TextAndHeaderCipherKeys >;
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	DecodeBackupLogValueActor(Arena* const& arena,VectorRef<MutationRef>* const& result,VectorRef<Optional<MutationRef>>* const& encryptedResult,int* const& mutationSize,Standalone<StringRef> const& value,Key const& addPrefix,Key const& removePrefix,Version const& version,Reference<KeyRangeMap<Version>> const& key_version,Database const& cx,bool const& provisionalProxy,std::shared_ptr<DatabaseConfiguration> const& dbConfig) 
															#line 1036 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   DecodeBackupLogValueActorState<DecodeBackupLogValueActor>(arena, result, encryptedResult, mutationSize, value, addPrefix, removePrefix, version, key_version, cx, provisionalProxy, dbConfig),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("decodeBackupLogValue", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3565411968530254592UL, 18378739926066530816UL);
		ActorExecutionContextHelper __helper(static_cast<DecodeBackupLogValueActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("decodeBackupLogValue");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("decodeBackupLogValue", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DecodeBackupLogValueActor, 0, TextAndHeaderCipherKeys >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] static Future<Void> decodeBackupLogValue( Arena* const& arena, VectorRef<MutationRef>* const& result, VectorRef<Optional<MutationRef>>* const& encryptedResult, int* const& mutationSize, Standalone<StringRef> const& value, Key const& addPrefix, Key const& removePrefix, Version const& version, Reference<KeyRangeMap<Version>> const& key_version, Database const& cx, bool const& provisionalProxy, std::shared_ptr<DatabaseConfiguration> const& dbConfig ) {
															#line 284 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new DecodeBackupLogValueActor(arena, result, encryptedResult, mutationSize, value, addPrefix, removePrefix, version, key_version, cx, provisionalProxy, dbConfig));
															#line 1069 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 463 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

static double lastErrorTime = 0;

void logErrorWorker(Reference<ReadYourWritesTransaction> tr, Key keyErrors, std::string message) {
	tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
	tr->setOption(FDBTransactionOptions::LOCK_AWARE);
	if (now() - lastErrorTime > CLIENT_KNOBS->BACKUP_ERROR_DELAY) {
		TraceEvent("BA_LogError").detail("Key", keyErrors).detail("Message", message);
		lastErrorTime = now();
	}
	tr->set(keyErrors, message);
}

Future<Void> logError(Database cx, Key keyErrors, const std::string& message) {
	return runRYWTransaction(cx, [=](Reference<ReadYourWritesTransaction> tr) {
		logErrorWorker(tr, keyErrors, message);
		return Future<Void>(Void());
	});
}

Future<Void> logError(Reference<ReadYourWritesTransaction> tr, Key keyErrors, const std::string& message) {
	return logError(tr->getDatabase(), keyErrors, message);
}

															#line 1097 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via readCommitted()
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class ReadCommittedActor>
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ReadCommittedActorState {
															#line 1104 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ReadCommittedActorState(Database const& cx,PromiseStream<RangeResultWithVersion> const& results,Reference<FlowLock> const& lock,KeyRangeRef const& range,Terminator const& terminator,AccessSystemKeys const& systemAccess,LockAware const& lockAware) 
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   results(results),
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lock(lock),
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   range(range),
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   terminator(terminator),
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   systemAccess(systemAccess),
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lockAware(lockAware),
															#line 494 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   begin(firstGreaterOrEqual(range.begin)),
															#line 495 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   end(firstGreaterOrEqual(range.end)),
															#line 496 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   tr(cx),
															#line 497 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   releaser()
															#line 1131 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("readCommitted", reinterpret_cast<unsigned long>(this));

	}
	~ReadCommittedActorState() 
	{
		fdb_probe_actor_destroy("readCommitted", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 499 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			;
															#line 1146 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadCommittedActorState();
		static_cast<ReadCommittedActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 501 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			limits = GetRangeLimits(GetRangeLimits::ROW_LIMIT_UNLIMITED, (g_network->isSimulated() && !g_simulator->speedUpSimulation) ? CLIENT_KNOBS->BACKUP_SIMULATED_LIMIT_BYTES : CLIENT_KNOBS->BACKUP_GET_RANGE_LIMIT_BYTES);
															#line 506 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (systemAccess)
															#line 1179 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 507 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1183 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 508 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (lockAware)
															#line 1187 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 509 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1191 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 510 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (CLIENT_KNOBS->ENABLE_REPLICA_CONSISTENCY_CHECK_ON_BACKUP_READS)
															#line 1195 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 511 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::ENABLE_REPLICA_CONSISTENCY_CHECK);
															#line 512 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				int64_t requiredReplicas = CLIENT_KNOBS->BACKUP_CONSISTENCY_CHECK_REQUIRED_REPLICAS;
															#line 513 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::CONSISTENCY_CHECK_REQUIRED_REPLICAS, StringRef((uint8_t*)&requiredReplicas, sizeof(int64_t)));
															#line 1203 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 518 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			releaser.release();
															#line 519 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_0 = lock->take(TaskPriority::DefaultYield, limits.bytes + CLIENT_KNOBS->VALUE_SIZE_LIMIT + CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 519 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<ReadCommittedActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1211 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadCommittedActor*>(this)->actor_wait_state = 1;
															#line 519 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor, 0, Void >*>(static_cast<ReadCommittedActor*>(this)));
															#line 1216 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 556 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (e.code() == error_code_transaction_too_old)
															#line 1238 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 559 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.fullReset();
															#line 1242 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = a_body1loopBody1Catch1cont1(loopDepth);
			}
			else
			{
															#line 561 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_3 = tr.onError(e);
															#line 561 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<ReadCommittedActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1251 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
				static_cast<ReadCommittedActor*>(this)->actor_wait_state = 4;
															#line 561 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor, 3, Void >*>(static_cast<ReadCommittedActor*>(this)));
															#line 1256 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 521 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser = FlowLock::Releaser( *lock, limits.bytes + CLIENT_KNOBS->VALUE_SIZE_LIMIT + CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 524 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = tr.getRange(begin, end, limits);
															#line 524 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1276 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadCommittedActor*>(this)->actor_wait_state = 2;
															#line 524 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor, 1, RangeResult >*>(static_cast<ReadCommittedActor*>(this)));
															#line 1281 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 521 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser = FlowLock::Releaser( *lock, limits.bytes + CLIENT_KNOBS->VALUE_SIZE_LIMIT + CLIENT_KNOBS->SYSTEM_KEY_SIZE_LIMIT);
															#line 524 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = tr.getRange(begin, end, limits);
															#line 524 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1294 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<ReadCommittedActor*>(this)->actor_wait_state = 2;
															#line 524 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor, 1, RangeResult >*>(static_cast<ReadCommittedActor*>(this)));
															#line 1299 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor*>(this)->ActorCallback< ReadCommittedActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
															#line 528 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (values.size() > 1 && BUGGIFY)
															#line 1383 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 529 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			RangeResult copy;
															#line 531 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for(int i = 0;i < values.size() / 2;i++) {
															#line 532 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				copy.push_back_deep(copy.arena(), values[i]);
															#line 1391 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 534 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			values = copy;
															#line 535 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			values.more = true;
															#line 537 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (deterministicRandom()->random01() < 0.5)
															#line 1399 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 538 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_2 = delay(6.0);
															#line 538 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<ReadCommittedActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1405 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_2.get(), loopDepth); };
				static_cast<ReadCommittedActor*>(this)->actor_wait_state = 3;
															#line 538 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor, 2, Void >*>(static_cast<ReadCommittedActor*>(this)));
															#line 1410 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont8(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult const& __values,int loopDepth) 
	{
															#line 524 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		values = __values;
															#line 1429 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult && __values,int loopDepth) 
	{
		values = std::move(__values);
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor*>(this)->ActorCallback< ReadCommittedActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont7(int loopDepth) 
	{
															#line 541 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser.remaining -= values.expectedSize();
															#line 543 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		ASSERT(releaser.remaining >= 0);
															#line 545 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		results.send(RangeResultWithVersion(values, tr.getReadVersion().get()));
															#line 547 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (values.size() > 0)
															#line 1514 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 548 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			begin = firstGreaterThan(values.end()[-1].key);
															#line 1518 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 550 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!values.more && !limits.isReached())
															#line 1522 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 551 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (terminator)
															#line 1526 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 552 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				results.sendError(end_of_stream());
															#line 1530 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 553 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<ReadCommittedActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadCommittedActorState(); static_cast<ReadCommittedActor*>(this)->destroy(); return 0; }
															#line 1534 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<ReadCommittedActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReadCommittedActorState();
			static_cast<ReadCommittedActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont15(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor*>(this)->ActorCallback< ReadCommittedActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont15(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ReadCommittedActor*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor*>(this)->ActorCallback< ReadCommittedActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PromiseStream<RangeResultWithVersion> results;
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<FlowLock> lock;
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeyRangeRef range;
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Terminator terminator;
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	AccessSystemKeys systemAccess;
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	LockAware lockAware;
															#line 494 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeySelector begin;
															#line 495 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeySelector end;
															#line 496 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Transaction tr;
															#line 497 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock::Releaser releaser;
															#line 501 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	GetRangeLimits limits;
															#line 524 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RangeResult values;
															#line 1769 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via readCommitted()
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ReadCommittedActor final : public Actor<Void>, public ActorCallback< ReadCommittedActor, 0, Void >, public ActorCallback< ReadCommittedActor, 1, RangeResult >, public ActorCallback< ReadCommittedActor, 2, Void >, public ActorCallback< ReadCommittedActor, 3, Void >, public FastAllocated<ReadCommittedActor>, public ReadCommittedActorState<ReadCommittedActor> {
															#line 1774 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<ReadCommittedActor>::operator new;
	using FastAllocated<ReadCommittedActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12490419512601771264UL, 13621378939295054336UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadCommittedActor, 0, Void >;
friend struct ActorCallback< ReadCommittedActor, 1, RangeResult >;
friend struct ActorCallback< ReadCommittedActor, 2, Void >;
friend struct ActorCallback< ReadCommittedActor, 3, Void >;
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ReadCommittedActor(Database const& cx,PromiseStream<RangeResultWithVersion> const& results,Reference<FlowLock> const& lock,KeyRangeRef const& range,Terminator const& terminator,AccessSystemKeys const& systemAccess,LockAware const& lockAware) 
															#line 1794 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ReadCommittedActorState<ReadCommittedActor>(cx, results, lock, range, terminator, systemAccess, lockAware),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(614696038618302720UL, 17105894356619358976UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readCommitted");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadCommittedActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadCommittedActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadCommittedActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ReadCommittedActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> readCommitted( Database const& cx, PromiseStream<RangeResultWithVersion> const& results, Reference<FlowLock> const& lock, KeyRangeRef const& range, Terminator const& terminator, AccessSystemKeys const& systemAccess, LockAware const& lockAware ) {
															#line 487 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new ReadCommittedActor(cx, results, lock, range, terminator, systemAccess, lockAware));
															#line 1830 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 566 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

// read each version, potentially multiple part within the same version
															#line 1836 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via readCommitted()
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class ReadCommittedActor1>
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ReadCommittedActor1State {
															#line 1843 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ReadCommittedActor1State(Database const& cx,PromiseStream<RCGroup> const& results,Future<Void> const& active,Reference<FlowLock> const& lock,KeyRangeRef const& range,std::function<std::pair<uint64_t, uint32_t>(Key key)> const& groupBy,Terminator const& terminator,AccessSystemKeys const& systemAccess,LockAware const& lockAware) 
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   results(results),
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   active(active),
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lock(lock),
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   range(range),
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   groupBy(groupBy),
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   terminator(terminator),
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   systemAccess(systemAccess),
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lockAware(lockAware),
															#line 577 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   nextKey(firstGreaterOrEqual(range.begin)),
															#line 578 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   end(firstGreaterOrEqual(range.end)),
															#line 580 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   rcGroup(RCGroup()),
															#line 581 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   skipGroup(ULLONG_MAX),
															#line 582 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   tr(cx),
															#line 583 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   releaser()
															#line 1878 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("readCommitted", reinterpret_cast<unsigned long>(this));

	}
	~ReadCommittedActor1State() 
	{
		fdb_probe_actor_destroy("readCommitted", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 585 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			;
															#line 1893 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ReadCommittedActor1State();
		static_cast<ReadCommittedActor1*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 587 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			limits = GetRangeLimits(GetRangeLimits::ROW_LIMIT_UNLIMITED, (g_network->isSimulated() && !g_simulator->speedUpSimulation) ? CLIENT_KNOBS->BACKUP_SIMULATED_LIMIT_BYTES : CLIENT_KNOBS->BACKUP_GET_RANGE_LIMIT_BYTES);
															#line 592 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (systemAccess)
															#line 1926 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 593 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1930 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 594 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (lockAware)
															#line 1934 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 595 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1938 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 597 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr.getRange(nextKey, end, limits);
															#line 597 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1944 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 1;
															#line 597 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 0, RangeResult >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 1949 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 678 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (e.code() == error_code_transaction_too_old)
															#line 1971 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 681 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr.fullReset();
															#line 1975 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = a_body1loopBody1Catch1cont1(loopDepth);
			}
			else
			{
															#line 683 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_4 = tr.onError(e);
															#line 683 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1984 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
				static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 5;
															#line 683 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 4, Void >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 1989 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 601 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (rangevalue.size() > 1 && BUGGIFY)
															#line 2005 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 602 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			RangeResult copy;
															#line 604 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for(int i = 0;i < rangevalue.size() / 2;i++) {
															#line 605 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				copy.push_back_deep(copy.arena(), rangevalue[i]);
															#line 2013 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 607 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			rangevalue = copy;
															#line 608 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			rangevalue.more = true;
															#line 610 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (deterministicRandom()->random01() < 0.01)
															#line 2021 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 611 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_1 = delay(6.0);
															#line 611 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2027 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
				static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 2;
															#line 611 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 1, Void >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 2032 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1loopBody1cont6(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1loopBody1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __rangevalue,int loopDepth) 
	{
															#line 597 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		rangevalue = __rangevalue;
															#line 2051 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __rangevalue,int loopDepth) 
	{
		rangevalue = std::move(__rangevalue);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor1*>(this)->ActorCallback< ReadCommittedActor1, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor1, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
															#line 615 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_2 = active;
															#line 615 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2132 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont5when1(__when_expr_2.get(), loopDepth); };
		static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 3;
															#line 615 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 2, Void >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 2137 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor1*>(this)->ActorCallback< ReadCommittedActor1, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor1, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
															#line 616 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser.release();
															#line 617 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_3 = lock->take(TaskPriority::DefaultYield, rangevalue.expectedSize() + rcGroup.items.expectedSize());
															#line 617 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2243 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_3.get(), loopDepth); };
		static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 4;
															#line 617 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 3, Void >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 2248 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
															#line 616 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser.release();
															#line 617 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_3 = lock->take(TaskPriority::DefaultYield, rangevalue.expectedSize() + rcGroup.items.expectedSize());
															#line 617 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_3.get(), loopDepth); };
		static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 4;
															#line 617 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ReadCommittedActor1, 3, Void >*>(static_cast<ReadCommittedActor1*>(this)));
															#line 2266 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor1*>(this)->ActorCallback< ReadCommittedActor1, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor1, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont10(Void const& _,int loopDepth) 
	{
															#line 618 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser = FlowLock::Releaser(*lock, rangevalue.expectedSize() + rcGroup.items.expectedSize());
															#line 621 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for( auto& s : rangevalue ) {
															#line 622 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Version groupKey = groupBy(s.key).first;
															#line 629 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (groupKey != skipGroup)
															#line 2356 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 630 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (rcGroup.version == -1)
															#line 2360 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 631 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					rcGroup.version = tr.getReadVersion().get();
															#line 632 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					rcGroup.groupKey = groupKey;
															#line 2366 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 633 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (rcGroup.groupKey != groupKey)
															#line 2372 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					{
															#line 644 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						releaser.remaining -= rcGroup.items .expectedSize();
															#line 647 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						ASSERT(releaser.remaining >= 0);
															#line 648 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						results.send(rcGroup);
															#line 649 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						nextKey = firstGreaterThan(rcGroup.items.end()[-1].key);
															#line 650 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						skipGroup = rcGroup.groupKey;
															#line 652 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup = RCGroup();
															#line 653 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup.version = tr.getReadVersion().get();
															#line 654 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup.groupKey = groupKey;
															#line 2390 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
				}
															#line 657 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				rcGroup.items.push_back_deep(rcGroup.items.arena(), s);
															#line 2395 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
		}
															#line 661 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!rangevalue.more)
															#line 2400 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 662 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (rcGroup.version != -1)
															#line 2404 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 663 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				releaser.remaining -= rcGroup.items .expectedSize();
															#line 666 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				ASSERT(releaser.remaining >= 0);
															#line 668 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				results.send(rcGroup);
															#line 2412 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 671 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (terminator)
															#line 2416 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 672 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				results.sendError(end_of_stream());
															#line 2420 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 673 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<ReadCommittedActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadCommittedActor1State(); static_cast<ReadCommittedActor1*>(this)->destroy(); return 0; }
															#line 2424 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<ReadCommittedActor1*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReadCommittedActor1State();
			static_cast<ReadCommittedActor1*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 676 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		nextKey = firstGreaterThan(rangevalue.end()[-1].key);
															#line 2432 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont10cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10(Void && _,int loopDepth) 
	{
															#line 618 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		releaser = FlowLock::Releaser(*lock, rangevalue.expectedSize() + rcGroup.items.expectedSize());
															#line 621 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for( auto& s : rangevalue ) {
															#line 622 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Version groupKey = groupBy(s.key).first;
															#line 629 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (groupKey != skipGroup)
															#line 2447 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 630 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (rcGroup.version == -1)
															#line 2451 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 631 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					rcGroup.version = tr.getReadVersion().get();
															#line 632 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					rcGroup.groupKey = groupKey;
															#line 2457 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 633 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (rcGroup.groupKey != groupKey)
															#line 2463 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					{
															#line 644 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						releaser.remaining -= rcGroup.items .expectedSize();
															#line 647 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						ASSERT(releaser.remaining >= 0);
															#line 648 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						results.send(rcGroup);
															#line 649 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						nextKey = firstGreaterThan(rcGroup.items.end()[-1].key);
															#line 650 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						skipGroup = rcGroup.groupKey;
															#line 652 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup = RCGroup();
															#line 653 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup.version = tr.getReadVersion().get();
															#line 654 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						rcGroup.groupKey = groupKey;
															#line 2481 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
				}
															#line 657 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				rcGroup.items.push_back_deep(rcGroup.items.arena(), s);
															#line 2486 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
		}
															#line 661 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!rangevalue.more)
															#line 2491 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 662 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (rcGroup.version != -1)
															#line 2495 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 663 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				releaser.remaining -= rcGroup.items .expectedSize();
															#line 666 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				ASSERT(releaser.remaining >= 0);
															#line 668 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				results.send(rcGroup);
															#line 2503 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 671 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (terminator)
															#line 2507 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 672 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				results.sendError(end_of_stream());
															#line 2511 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 673 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<ReadCommittedActor1*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ReadCommittedActor1State(); static_cast<ReadCommittedActor1*>(this)->destroy(); return 0; }
															#line 2515 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<ReadCommittedActor1*>(this)->SAV< Void >::value()) Void(Void());
			this->~ReadCommittedActor1State();
			static_cast<ReadCommittedActor1*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 676 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		nextKey = firstGreaterThan(rangevalue.end()[-1].key);
															#line 2523 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont10cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor1*>(this)->ActorCallback< ReadCommittedActor1, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor1, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont10cont10(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ReadCommittedActor1*>(this)->actor_wait_state > 0) static_cast<ReadCommittedActor1*>(this)->actor_wait_state = 0;
		static_cast<ReadCommittedActor1*>(this)->ActorCallback< ReadCommittedActor1, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ReadCommittedActor1, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ReadCommittedActor1, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PromiseStream<RCGroup> results;
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Future<Void> active;
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<FlowLock> lock;
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeyRangeRef range;
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::function<std::pair<uint64_t, uint32_t>(Key key)> groupBy;
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Terminator terminator;
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	AccessSystemKeys systemAccess;
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	LockAware lockAware;
															#line 577 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeySelector nextKey;
															#line 578 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KeySelector end;
															#line 580 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RCGroup rcGroup;
															#line 581 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	uint64_t skipGroup;
															#line 582 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Transaction tr;
															#line 583 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock::Releaser releaser;
															#line 587 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	GetRangeLimits limits;
															#line 597 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RangeResult rangevalue;
															#line 2743 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via readCommitted()
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ReadCommittedActor1 final : public Actor<Void>, public ActorCallback< ReadCommittedActor1, 0, RangeResult >, public ActorCallback< ReadCommittedActor1, 1, Void >, public ActorCallback< ReadCommittedActor1, 2, Void >, public ActorCallback< ReadCommittedActor1, 3, Void >, public ActorCallback< ReadCommittedActor1, 4, Void >, public FastAllocated<ReadCommittedActor1>, public ReadCommittedActor1State<ReadCommittedActor1> {
															#line 2748 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<ReadCommittedActor1>::operator new;
	using FastAllocated<ReadCommittedActor1>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12490419512601771264UL, 13621378939295054336UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ReadCommittedActor1, 0, RangeResult >;
friend struct ActorCallback< ReadCommittedActor1, 1, Void >;
friend struct ActorCallback< ReadCommittedActor1, 2, Void >;
friend struct ActorCallback< ReadCommittedActor1, 3, Void >;
friend struct ActorCallback< ReadCommittedActor1, 4, Void >;
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ReadCommittedActor1(Database const& cx,PromiseStream<RCGroup> const& results,Future<Void> const& active,Reference<FlowLock> const& lock,KeyRangeRef const& range,std::function<std::pair<uint64_t, uint32_t>(Key key)> const& groupBy,Terminator const& terminator,AccessSystemKeys const& systemAccess,LockAware const& lockAware) 
															#line 2769 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ReadCommittedActor1State<ReadCommittedActor1>(cx, results, active, lock, range, groupBy, terminator, systemAccess, lockAware),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("readCommitted", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11877085914525018880UL, 18281691565276970752UL);
		ActorExecutionContextHelper __helper(static_cast<ReadCommittedActor1*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("readCommitted");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("readCommitted", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ReadCommittedActor1, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ReadCommittedActor1, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ReadCommittedActor1, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ReadCommittedActor1, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ReadCommittedActor1, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> readCommitted( Database const& cx, PromiseStream<RCGroup> const& results, Future<Void> const& active, Reference<FlowLock> const& lock, KeyRangeRef const& range, std::function<std::pair<uint64_t, uint32_t>(Key key)> const& groupBy, Terminator const& terminator, AccessSystemKeys const& systemAccess, LockAware const& lockAware ) {
															#line 568 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new ReadCommittedActor1(cx, results, active, lock, range, groupBy, terminator, systemAccess, lockAware));
															#line 2806 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 688 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

Future<Void> readCommitted(Database cx,
                           PromiseStream<RCGroup> results,
                           Reference<FlowLock> lock,
                           KeyRangeRef range,
                           std::function<std::pair<uint64_t, uint32_t>(Key key)> groupBy) {
	return readCommitted(
	    cx, results, Void(), lock, range, groupBy, Terminator::True, AccessSystemKeys::True, LockAware::True);
}

// restore transaction has to be first in the batch, or it is the only txn in batch to make sure it never conflicts with
// others.
															#line 2822 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via sendCommitTransactionRequest()
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class SendCommitTransactionRequestActor>
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class SendCommitTransactionRequestActorState {
															#line 2829 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	SendCommitTransactionRequestActorState(CommitTransactionRequest const& req,Key const& uid,Version const& newBeginVersion,Key const& rangeBegin,NotifiedVersion* const& committedVersion,int* const& totalBytes,int* const& mutationSize,FlowLock* const& commitLock,PublicRequestStream<CommitTransactionRequest> const& commit) 
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : req(req),
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   uid(uid),
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   newBeginVersion(newBeginVersion),
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   rangeBegin(rangeBegin),
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   committedVersion(committedVersion),
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   totalBytes(totalBytes),
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   mutationSize(mutationSize),
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commitLock(commitLock),
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commit(commit)
															#line 2852 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this));

	}
	~SendCommitTransactionRequestActorState() 
	{
		fdb_probe_actor_destroy("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 709 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key applyBegin = uid.withPrefix(applyMutationsBeginRange.begin);
															#line 710 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key versionKey = BinaryWriter::toValue(newBeginVersion, Unversioned());
															#line 711 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key rangeEnd = getApplyKey(newBeginVersion, uid);
															#line 715 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.mutations.push_back_deep(req.arena, MutationRef(MutationRef::SetValue, applyBegin, versionKey));
															#line 716 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.encryptedMutations.push_back_deep(req.arena, Optional<MutationRef>());
															#line 717 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.write_conflict_ranges.push_back_deep(req.arena, singleKeyRange(applyBegin));
															#line 718 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.mutations.push_back_deep(req.arena, MutationRef(MutationRef::ClearRange, rangeBegin, rangeEnd));
															#line 719 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.encryptedMutations.push_back_deep(req.arena, Optional<MutationRef>());
															#line 720 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.write_conflict_ranges.push_back_deep(req.arena, singleKeyRange(rangeBegin));
															#line 725 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.read_snapshot = committedVersion->get();
															#line 726 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.flags = req.flags | CommitTransactionRequest::FLAG_IS_LOCK_AWARE;
															#line 728 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			*totalBytes += *mutationSize;
															#line 729 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_0 = commitLock->take(TaskPriority::DefaultYield, *mutationSize);
															#line 729 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2893 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state = 1;
															#line 729 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< SendCommitTransactionRequestActor, 0, Void >*>(static_cast<SendCommitTransactionRequestActor*>(this)));
															#line 2898 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~SendCommitTransactionRequestActorState();
		static_cast<SendCommitTransactionRequestActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 730 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		Future<Void> commitAndUnlock = commitLock->releaseWhen(success(commit.getReply(req)), *mutationSize);
															#line 739 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_1 = commitAndUnlock;
															#line 739 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2925 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state = 2;
															#line 739 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SendCommitTransactionRequestActor, 1, Void >*>(static_cast<SendCommitTransactionRequestActor*>(this)));
															#line 2930 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 730 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		Future<Void> commitAndUnlock = commitLock->releaseWhen(success(commit.getReply(req)), *mutationSize);
															#line 739 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_1 = commitAndUnlock;
															#line 739 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2943 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state = 2;
															#line 739 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< SendCommitTransactionRequestActor, 1, Void >*>(static_cast<SendCommitTransactionRequestActor*>(this)));
															#line 2948 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state > 0) static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state = 0;
		static_cast<SendCommitTransactionRequestActor*>(this)->ActorCallback< SendCommitTransactionRequestActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SendCommitTransactionRequestActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< SendCommitTransactionRequestActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< SendCommitTransactionRequestActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 740 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<SendCommitTransactionRequestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SendCommitTransactionRequestActorState(); static_cast<SendCommitTransactionRequestActor*>(this)->destroy(); return 0; }
															#line 3032 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<SendCommitTransactionRequestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SendCommitTransactionRequestActorState();
		static_cast<SendCommitTransactionRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 740 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<SendCommitTransactionRequestActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~SendCommitTransactionRequestActorState(); static_cast<SendCommitTransactionRequestActor*>(this)->destroy(); return 0; }
															#line 3044 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<SendCommitTransactionRequestActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~SendCommitTransactionRequestActorState();
		static_cast<SendCommitTransactionRequestActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state > 0) static_cast<SendCommitTransactionRequestActor*>(this)->actor_wait_state = 0;
		static_cast<SendCommitTransactionRequestActor*>(this)->ActorCallback< SendCommitTransactionRequestActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< SendCommitTransactionRequestActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< SendCommitTransactionRequestActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< SendCommitTransactionRequestActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CommitTransactionRequest req;
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key uid;
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version newBeginVersion;
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key rangeBegin;
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	NotifiedVersion* committedVersion;
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int* totalBytes;
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int* mutationSize;
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock* commitLock;
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PublicRequestStream<CommitTransactionRequest> commit;
															#line 3145 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via sendCommitTransactionRequest()
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class SendCommitTransactionRequestActor final : public Actor<Void>, public ActorCallback< SendCommitTransactionRequestActor, 0, Void >, public ActorCallback< SendCommitTransactionRequestActor, 1, Void >, public FastAllocated<SendCommitTransactionRequestActor>, public SendCommitTransactionRequestActorState<SendCommitTransactionRequestActor> {
															#line 3150 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<SendCommitTransactionRequestActor>::operator new;
	using FastAllocated<SendCommitTransactionRequestActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2422540639968613376UL, 13915507041448860672UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< SendCommitTransactionRequestActor, 0, Void >;
friend struct ActorCallback< SendCommitTransactionRequestActor, 1, Void >;
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	SendCommitTransactionRequestActor(CommitTransactionRequest const& req,Key const& uid,Version const& newBeginVersion,Key const& rangeBegin,NotifiedVersion* const& committedVersion,int* const& totalBytes,int* const& mutationSize,FlowLock* const& commitLock,PublicRequestStream<CommitTransactionRequest> const& commit) 
															#line 3168 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   SendCommitTransactionRequestActorState<SendCommitTransactionRequestActor>(req, uid, newBeginVersion, rangeBegin, committedVersion, totalBytes, mutationSize, commitLock, commit),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1297593344567152896UL, 10219936147972304640UL);
		ActorExecutionContextHelper __helper(static_cast<SendCommitTransactionRequestActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("sendCommitTransactionRequest");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("sendCommitTransactionRequest", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< SendCommitTransactionRequestActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< SendCommitTransactionRequestActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> sendCommitTransactionRequest( CommitTransactionRequest const& req, Key const& uid, Version const& newBeginVersion, Key const& rangeBegin, NotifiedVersion* const& committedVersion, int* const& totalBytes, int* const& mutationSize, FlowLock* const& commitLock, PublicRequestStream<CommitTransactionRequest> const& commit ) {
															#line 700 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new SendCommitTransactionRequestActor(req, uid, newBeginVersion, rangeBegin, committedVersion, totalBytes, mutationSize, commitLock, commit));
															#line 3202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 742 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

// Decodes the backup mutation log and send the mutations to the CommitProxy.
// The mutation logs are grouped by version and passed in as a stream of RCGroup from readCommitted().
// The mutations are then decoded and sent to the CommitProxy in a batch.
															#line 3210 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via kvMutationLogToTransactions()
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class KvMutationLogToTransactionsActor>
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class KvMutationLogToTransactionsActorState {
															#line 3217 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KvMutationLogToTransactionsActorState(Database const& cx,PromiseStream<RCGroup> const& results,Reference<FlowLock> const& lock,Key const& uid,Key const& addPrefix,Key const& removePrefix,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,Optional<Version> const& endVersion,Key const& rangeBegin,FlowLock* const& commitLock,Reference<KeyRangeMap<Version>> const& keyVersion,bool const& provisionalProxy,std::shared_ptr<DatabaseConfiguration> const& dbConfig) 
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   results(results),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lock(lock),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   uid(uid),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   addPrefix(addPrefix),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   removePrefix(removePrefix),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commit(commit),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   committedVersion(committedVersion),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   endVersion(endVersion),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   rangeBegin(rangeBegin),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commitLock(commitLock),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   keyVersion(keyVersion),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   provisionalProxy(provisionalProxy),
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   dbConfig(dbConfig),
															#line 760 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   lastVersion(invalidVersion),
															#line 761 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   endOfStream(false),
															#line 762 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   totalBytes(0)
															#line 3256 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this));

	}
	~KvMutationLogToTransactionsActorState() 
	{
		fdb_probe_actor_destroy("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 764 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			;
															#line 3271 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~KvMutationLogToTransactionsActorState();
		static_cast<KvMutationLogToTransactionsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 765 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		req = CommitTransactionRequest();
															#line 766 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		newBeginVersion = invalidVersion;
															#line 767 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		mutationSize = 0;
															#line 768 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		;
															#line 3307 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
															#line 818 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_2 = sendCommitTransactionRequest( req, uid, newBeginVersion, rangeBegin, committedVersion, &totalBytes, &mutationSize, commitLock, commit);
															#line 818 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3318 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_2.get(), loopDepth); };
		static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 3;
															#line 818 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< KvMutationLogToTransactionsActor, 2, Void >*>(static_cast<KvMutationLogToTransactionsActor*>(this)));
															#line 3323 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1(int loopDepth) 
	{
		try {
															#line 770 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			auto __when_expr_0 = results.getFuture();
															#line 3340 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			static_assert(std::is_same<decltype(__when_expr_0), FutureStream<RCGroup>>::value || std::is_same<decltype(__when_expr_0), ThreadFutureStream<RCGroup>>::value, "invalid type");
															#line 770 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3344 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1loopBody1when1(__when_expr_0.pop(), loopDepth); };
			static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 1;
															#line 770 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >*>(static_cast<KvMutationLogToTransactionsActor*>(this)));
															#line 3349 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 805 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (e.code() == error_code_end_of_stream)
															#line 3384 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 806 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (endVersion.present() && endVersion.get() > lastVersion && endVersion.get() > newBeginVersion)
															#line 3388 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 807 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					newBeginVersion = endVersion.get();
															#line 3392 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
															#line 809 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (newBeginVersion == invalidVersion)
															#line 3396 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 810 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (!static_cast<KvMutationLogToTransactionsActor*>(this)->SAV<int>::futures) { (void)(totalBytes); this->~KvMutationLogToTransactionsActorState(); static_cast<KvMutationLogToTransactionsActor*>(this)->destroy(); return 0; }
															#line 3400 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					new (&static_cast<KvMutationLogToTransactionsActor*>(this)->SAV< int >::value()) int(std::move(totalBytes)); // state_var_RVO
					this->~KvMutationLogToTransactionsActorState();
					static_cast<KvMutationLogToTransactionsActor*>(this)->finishSendAndDelPromiseRef();
					return 0;
				}
															#line 811 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				endOfStream = true;
															#line 3408 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				return a_body1loopBody1break1(loopDepth==0?0:loopDepth-1); // break
			}
															#line 814 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			return a_body1Catch1(e, std::max(0, loopDepth - 2));
															#line 3413 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2(int loopDepth) 
	{
															#line 771 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		curReq = CommitTransactionRequest();
															#line 772 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		lock->release(group.items.expectedSize());
															#line 773 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		curBatchMutationSize = 0;
															#line 775 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		BinaryWriter bw(Unversioned());
															#line 777 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for(int i = 0;i < group.items.size();++i) {
															#line 780 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			bw.serializeBytes(group.items[i].value);
															#line 3437 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 783 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		Standalone<StringRef> value = bw.toValue();
															#line 784 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_1 = decodeBackupLogValue(&curReq.arena, &curReq.transaction.mutations, &curReq.transaction.encryptedMutations, &curBatchMutationSize, value, addPrefix, removePrefix, group.groupKey, keyVersion, cx, provisionalProxy, dbConfig);
															#line 784 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3445 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 2;
															#line 784 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< KvMutationLogToTransactionsActor, 1, Void >*>(static_cast<KvMutationLogToTransactionsActor*>(this)));
															#line 3450 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(RCGroup const& __group,int loopDepth) 
	{
															#line 770 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		group = __group;
															#line 3459 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1when1(RCGroup && __group,int loopDepth) 
	{
		group = std::move(__group);
		loopDepth = a_body1loopBody1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state > 0) static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 0;
		static_cast<KvMutationLogToTransactionsActor*>(this)->ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >::remove();

	}
	void a_callback_fire(ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >*,RCGroup const& value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >*,RCGroup && value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >*,Error err) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 797 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for(int i = 0;i < curReq.transaction.mutations.size();i++) {
															#line 798 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.mutations.push_back_deep(req.arena, curReq.transaction.mutations[i]);
															#line 799 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.encryptedMutations.push_back_deep(req.arena, curReq.transaction.encryptedMutations[i]);
															#line 3542 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 802 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		mutationSize += curBatchMutationSize;
															#line 803 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		newBeginVersion = group.groupKey + 1;
															#line 3548 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 797 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for(int i = 0;i < curReq.transaction.mutations.size();i++) {
															#line 798 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.mutations.push_back_deep(req.arena, curReq.transaction.mutations[i]);
															#line 799 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req.transaction.encryptedMutations.push_back_deep(req.arena, curReq.transaction.encryptedMutations[i]);
															#line 3561 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 802 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		mutationSize += curBatchMutationSize;
															#line 803 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		newBeginVersion = group.groupKey + 1;
															#line 3567 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1loopBody1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state > 0) static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 0;
		static_cast<KvMutationLogToTransactionsActor*>(this)->ActorCallback< KvMutationLogToTransactionsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< KvMutationLogToTransactionsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< KvMutationLogToTransactionsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< KvMutationLogToTransactionsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1loopBody1cont7(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 820 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (endOfStream)
															#line 3664 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 821 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<KvMutationLogToTransactionsActor*>(this)->SAV<int>::futures) { (void)(totalBytes); this->~KvMutationLogToTransactionsActorState(); static_cast<KvMutationLogToTransactionsActor*>(this)->destroy(); return 0; }
															#line 3668 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<KvMutationLogToTransactionsActor*>(this)->SAV< int >::value()) int(std::move(totalBytes)); // state_var_RVO
			this->~KvMutationLogToTransactionsActorState();
			static_cast<KvMutationLogToTransactionsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 820 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (endOfStream)
															#line 3682 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 821 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<KvMutationLogToTransactionsActor*>(this)->SAV<int>::futures) { (void)(totalBytes); this->~KvMutationLogToTransactionsActorState(); static_cast<KvMutationLogToTransactionsActor*>(this)->destroy(); return 0; }
															#line 3686 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<KvMutationLogToTransactionsActor*>(this)->SAV< int >::value()) int(std::move(totalBytes)); // state_var_RVO
			this->~KvMutationLogToTransactionsActorState();
			static_cast<KvMutationLogToTransactionsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state > 0) static_cast<KvMutationLogToTransactionsActor*>(this)->actor_wait_state = 0;
		static_cast<KvMutationLogToTransactionsActor*>(this)->ActorCallback< KvMutationLogToTransactionsActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< KvMutationLogToTransactionsActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< KvMutationLogToTransactionsActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< KvMutationLogToTransactionsActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PromiseStream<RCGroup> results;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<FlowLock> lock;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key uid;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key addPrefix;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key removePrefix;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PublicRequestStream<CommitTransactionRequest> commit;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	NotifiedVersion* committedVersion;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Optional<Version> endVersion;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key rangeBegin;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock* commitLock;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<KeyRangeMap<Version>> keyVersion;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	bool provisionalProxy;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::shared_ptr<DatabaseConfiguration> dbConfig;
															#line 760 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version lastVersion;
															#line 761 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	bool endOfStream;
															#line 762 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int totalBytes;
															#line 765 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CommitTransactionRequest req;
															#line 766 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version newBeginVersion;
															#line 767 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int mutationSize;
															#line 770 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RCGroup group;
															#line 771 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CommitTransactionRequest curReq;
															#line 773 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int curBatchMutationSize;
															#line 3817 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via kvMutationLogToTransactions()
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class KvMutationLogToTransactionsActor final : public Actor<int>, public ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >, public ActorCallback< KvMutationLogToTransactionsActor, 1, Void >, public ActorCallback< KvMutationLogToTransactionsActor, 2, Void >, public FastAllocated<KvMutationLogToTransactionsActor>, public KvMutationLogToTransactionsActorState<KvMutationLogToTransactionsActor> {
															#line 3822 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<KvMutationLogToTransactionsActor>::operator new;
	using FastAllocated<KvMutationLogToTransactionsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6202529512380287744UL, 9482922324195998720UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >;
friend struct ActorCallback< KvMutationLogToTransactionsActor, 1, Void >;
friend struct ActorCallback< KvMutationLogToTransactionsActor, 2, Void >;
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	KvMutationLogToTransactionsActor(Database const& cx,PromiseStream<RCGroup> const& results,Reference<FlowLock> const& lock,Key const& uid,Key const& addPrefix,Key const& removePrefix,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,Optional<Version> const& endVersion,Key const& rangeBegin,FlowLock* const& commitLock,Reference<KeyRangeMap<Version>> const& keyVersion,bool const& provisionalProxy,std::shared_ptr<DatabaseConfiguration> const& dbConfig) 
															#line 3841 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<int>(),
		   KvMutationLogToTransactionsActorState<KvMutationLogToTransactionsActor>(cx, results, lock, uid, addPrefix, removePrefix, commit, committedVersion, endVersion, rangeBegin, commitLock, keyVersion, provisionalProxy, dbConfig),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(4574830504988014848UL, 15510128346966688768UL);
		ActorExecutionContextHelper __helper(static_cast<KvMutationLogToTransactionsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("kvMutationLogToTransactions");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("kvMutationLogToTransactions", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorSingleCallback< KvMutationLogToTransactionsActor, 0, RCGroup >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< KvMutationLogToTransactionsActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< KvMutationLogToTransactionsActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<int> kvMutationLogToTransactions( Database const& cx, PromiseStream<RCGroup> const& results, Reference<FlowLock> const& lock, Key const& uid, Key const& addPrefix, Key const& removePrefix, PublicRequestStream<CommitTransactionRequest> const& commit, NotifiedVersion* const& committedVersion, Optional<Version> const& endVersion, Key const& rangeBegin, FlowLock* const& commitLock, Reference<KeyRangeMap<Version>> const& keyVersion, bool const& provisionalProxy, std::shared_ptr<DatabaseConfiguration> const& dbConfig ) {
															#line 746 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<int>(new KvMutationLogToTransactionsActor(cx, results, lock, uid, addPrefix, removePrefix, commit, committedVersion, endVersion, rangeBegin, commitLock, keyVersion, provisionalProxy, dbConfig));
															#line 3876 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 825 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

															#line 3881 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via coalesceKeyVersionCache()
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class CoalesceKeyVersionCacheActor>
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CoalesceKeyVersionCacheActorState {
															#line 3888 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CoalesceKeyVersionCacheActorState(Key const& uid,Version const& endVersion,Reference<KeyRangeMap<Version>> const& keyVersion,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,PromiseStream<Future<Void>> const& addActor,FlowLock* const& commitLock) 
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : uid(uid),
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   endVersion(endVersion),
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   keyVersion(keyVersion),
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commit(commit),
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   committedVersion(committedVersion),
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   addActor(addActor),
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commitLock(commitLock)
															#line 3907 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this));

	}
	~CoalesceKeyVersionCacheActorState() 
	{
		fdb_probe_actor_destroy("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 833 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Version lastVersion = -1000;
															#line 834 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			int64_t removed = 0;
															#line 835 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			req = CommitTransactionRequest();
															#line 836 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			mutationSize = 0;
															#line 837 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key mapPrefix = uid.withPrefix(applyMutationsKeyVersionMapRange.begin);
															#line 839 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for( auto it : keyVersion->ranges() ) {
															#line 840 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (lastVersion == -1000)
															#line 3934 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 841 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					lastVersion = it.value();
															#line 3938 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 843 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					Version ver = it.value();
															#line 850 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (ver < endVersion && lastVersion < endVersion && ver != invalidVersion && lastVersion != invalidVersion)
															#line 3946 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					{
															#line 852 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						Key removeKey = it.range().begin.withPrefix(mapPrefix);
															#line 853 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						Key removeEnd = keyAfter(removeKey);
															#line 854 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						req.transaction.mutations.push_back_deep(req.arena, MutationRef(MutationRef::ClearRange, removeKey, removeEnd));
															#line 856 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						mutationSize += removeKey.size() + removeEnd.size();
															#line 857 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						removed--;
															#line 3958 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
					else
					{
															#line 859 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						lastVersion = ver;
															#line 3964 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
				}
			}
															#line 864 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (removed != 0)
															#line 3970 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 865 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Key countKey = uid.withPrefix(applyMutationsKeyVersionCountRange.begin);
															#line 866 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				req.transaction.write_conflict_ranges.push_back_deep(req.arena, singleKeyRange(countKey));
															#line 867 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				req.transaction.mutations.push_back_deep( req.arena, MutationRef(MutationRef::AddValue, countKey, StringRef((uint8_t*)&removed, 8)));
															#line 869 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				req.transaction.read_snapshot = committedVersion->get();
															#line 870 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				req.flags = req.flags | CommitTransactionRequest::FLAG_IS_LOCK_AWARE;
															#line 872 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_0 = commitLock->take(TaskPriority::DefaultYield, mutationSize);
															#line 872 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<CoalesceKeyVersionCacheActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3986 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<CoalesceKeyVersionCacheActor*>(this)->actor_wait_state = 1;
															#line 872 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >*>(static_cast<CoalesceKeyVersionCacheActor*>(this)));
															#line 3991 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CoalesceKeyVersionCacheActorState();
		static_cast<CoalesceKeyVersionCacheActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 876 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<CoalesceKeyVersionCacheActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CoalesceKeyVersionCacheActorState(); static_cast<CoalesceKeyVersionCacheActor*>(this)->destroy(); return 0; }
															#line 4019 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<CoalesceKeyVersionCacheActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CoalesceKeyVersionCacheActorState();
		static_cast<CoalesceKeyVersionCacheActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 873 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		addActor.send(commitLock->releaseWhen(success(commit.getReply(req)), mutationSize));
															#line 4031 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 873 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		addActor.send(commitLock->releaseWhen(success(commit.getReply(req)), mutationSize));
															#line 4040 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CoalesceKeyVersionCacheActor*>(this)->actor_wait_state > 0) static_cast<CoalesceKeyVersionCacheActor*>(this)->actor_wait_state = 0;
		static_cast<CoalesceKeyVersionCacheActor*>(this)->ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CoalesceKeyVersionCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CoalesceKeyVersionCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CoalesceKeyVersionCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key uid;
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version endVersion;
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<KeyRangeMap<Version>> keyVersion;
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PublicRequestStream<CommitTransactionRequest> commit;
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	NotifiedVersion* committedVersion;
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PromiseStream<Future<Void>> addActor;
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock* commitLock;
															#line 835 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CommitTransactionRequest req;
															#line 836 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int64_t mutationSize;
															#line 4138 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via coalesceKeyVersionCache()
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CoalesceKeyVersionCacheActor final : public Actor<Void>, public ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >, public FastAllocated<CoalesceKeyVersionCacheActor>, public CoalesceKeyVersionCacheActorState<CoalesceKeyVersionCacheActor> {
															#line 4143 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<CoalesceKeyVersionCacheActor>::operator new;
	using FastAllocated<CoalesceKeyVersionCacheActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(8633200019604793600UL, 6249096745648620800UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >;
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CoalesceKeyVersionCacheActor(Key const& uid,Version const& endVersion,Reference<KeyRangeMap<Version>> const& keyVersion,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,PromiseStream<Future<Void>> const& addActor,FlowLock* const& commitLock) 
															#line 4160 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   CoalesceKeyVersionCacheActorState<CoalesceKeyVersionCacheActor>(uid, endVersion, keyVersion, commit, committedVersion, addActor, commitLock),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18444184965993123072UL, 15771598966678214144UL);
		ActorExecutionContextHelper __helper(static_cast<CoalesceKeyVersionCacheActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("coalesceKeyVersionCache");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("coalesceKeyVersionCache", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CoalesceKeyVersionCacheActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> coalesceKeyVersionCache( Key const& uid, Version const& endVersion, Reference<KeyRangeMap<Version>> const& keyVersion, PublicRequestStream<CommitTransactionRequest> const& commit, NotifiedVersion* const& committedVersion, PromiseStream<Future<Void>> const& addActor, FlowLock* const& commitLock ) {
															#line 826 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new CoalesceKeyVersionCacheActor(uid, endVersion, keyVersion, commit, committedVersion, addActor, commitLock));
															#line 4193 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 878 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

															#line 4198 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via applyMutations()
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class ApplyMutationsActor>
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ApplyMutationsActorState {
															#line 4205 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ApplyMutationsActorState(Database const& cx,Key const& uid,Key const& addPrefix,Key const& removePrefix,Version const& beginVersion,Version* const& endVersion,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,Reference<KeyRangeMap<Version>> const& keyVersion,bool const& provisionalProxy) 
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   uid(uid),
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   addPrefix(addPrefix),
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   removePrefix(removePrefix),
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   beginVersion(beginVersion),
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   endVersion(endVersion),
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commit(commit),
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   committedVersion(committedVersion),
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   keyVersion(keyVersion),
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   provisionalProxy(provisionalProxy),
															#line 889 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   commitLock(CLIENT_KNOBS->BACKUP_LOCK_BYTES),
															#line 890 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   addActor(),
															#line 891 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   error(actorCollection(addActor.getFuture())),
															#line 892 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   maxBytes(CLIENT_KNOBS->APPLY_MIN_LOCK_BYTES),
															#line 893 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   dbConfig(std::make_shared<DatabaseConfiguration>())
															#line 4240 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("applyMutations", reinterpret_cast<unsigned long>(this));

	}
	~ApplyMutationsActorState() 
	{
		fdb_probe_actor_destroy("applyMutations", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 895 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			keyVersion->insert(metadataVersionKey, 0);
															#line 4255 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			try {
															#line 898 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_0 = store(*dbConfig, getDatabaseConfiguration(cx));
															#line 898 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4261 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 1;
															#line 898 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 0, Void >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 4266 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ApplyMutationsActorState();
		static_cast<ApplyMutationsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 964 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			ASSERT_WE_THINK(e.code() != error_code_transaction_rejected_range_locked);
															#line 965 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Severity sev = (e.code() == error_code_restore_missing_data || e.code() == error_code_transaction_throttled_hot_shard) ? SevWarnAlways : SevError;
															#line 969 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			TraceEvent(sev, "ApplyMutationsError").error(e);
															#line 970 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			return a_body1Catch1(e, loopDepth);
															#line 4302 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 900 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		;
															#line 4316 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 900 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		;
															#line 4325 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state > 0) static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<ApplyMutationsActor*>(this)->ActorCallback< ApplyMutationsActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ApplyMutationsActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1(int loopDepth) 
	{
															#line 901 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (beginVersion >= *endVersion)
															#line 4416 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 903 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_1 = commitLock.take(TaskPriority::DefaultYield, CLIENT_KNOBS->BACKUP_LOCK_BYTES);
															#line 903 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4422 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch2(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 2;
															#line 903 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 1, Void >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 4427 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1(int loopDepth) 
	{
															#line 910 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		int rangeCount = std::max(1, CLIENT_KNOBS->APPLY_MAX_LOCK_BYTES / maxBytes);
															#line 912 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		newEndVersion = std::min(*endVersion, ((beginVersion / CLIENT_KNOBS->APPLY_BLOCK_SIZE) + rangeCount) * CLIENT_KNOBS->APPLY_BLOCK_SIZE);
															#line 919 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		ranges = getApplyRanges(beginVersion, newEndVersion, uid);
															#line 920 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		idx = size_t();
															#line 921 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		results = std::vector<PromiseStream<RCGroup>>();
															#line 922 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		rc = std::vector<Future<Void>>();
															#line 923 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		locks = std::vector<Reference<FlowLock>>();
															#line 927 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for(int i = 0;i < ranges.size();++i) {
															#line 928 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			results.push_back(PromiseStream<RCGroup>());
															#line 929 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			locks.push_back(makeReference<FlowLock>( std::max(CLIENT_KNOBS->APPLY_MAX_LOCK_BYTES / ranges.size(), CLIENT_KNOBS->APPLY_MIN_LOCK_BYTES)));
															#line 931 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			rc.push_back(readCommitted(cx, results[i], locks[i], ranges[i], decodeBKMutationLogKey));
															#line 4461 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 934 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		maxBytes = std::max<int>(maxBytes * CLIENT_KNOBS->APPLY_MAX_DECAY_RATE, CLIENT_KNOBS->APPLY_MIN_LOCK_BYTES);
															#line 935 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		idx = 0;
															#line 4467 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont2loopBody1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 904 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		commitLock.release(CLIENT_KNOBS->BACKUP_LOCK_BYTES);
															#line 905 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (beginVersion >= *endVersion)
															#line 4478 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 906 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<ApplyMutationsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ApplyMutationsActorState(); static_cast<ApplyMutationsActor*>(this)->destroy(); return 0; }
															#line 4482 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<ApplyMutationsActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ApplyMutationsActorState();
			static_cast<ApplyMutationsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 904 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		commitLock.release(CLIENT_KNOBS->BACKUP_LOCK_BYTES);
															#line 905 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (beginVersion >= *endVersion)
															#line 4498 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 906 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<ApplyMutationsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~ApplyMutationsActorState(); static_cast<ApplyMutationsActor*>(this)->destroy(); return 0; }
															#line 4502 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<ApplyMutationsActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~ApplyMutationsActorState();
			static_cast<ApplyMutationsActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state > 0) static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<ApplyMutationsActor*>(this)->ActorCallback< ApplyMutationsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ApplyMutationsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont2loopBody1cont5(int loopDepth) 
	{
															#line 956 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_3 = coalesceKeyVersionCache( uid, newEndVersion, keyVersion, commit, committedVersion, addActor, &commitLock);
															#line 956 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4593 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch2(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont5when1(__when_expr_3.get(), loopDepth); };
		static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 4;
															#line 956 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 3, Void >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 4598 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont2loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 935 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!(idx < ranges.size()))
															#line 4614 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
			return a_body1cont2loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 936 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<int> __when_expr_2 = kvMutationLogToTransactions(cx, results[idx], locks[idx], uid, addPrefix, removePrefix, commit, committedVersion, idx == ranges.size() - 1 ? newEndVersion : Optional<Version>(), ranges[idx].begin, &commitLock, keyVersion, provisionalProxy, dbConfig);
															#line 936 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 4622 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch2(__when_expr_2.getError(), std::max(0, loopDepth - 2)); else return a_body1cont2loopBody1cont1loopBody1when1(__when_expr_2.get(), loopDepth); };
		static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 3;
															#line 936 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 2, int >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 4627 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont2loopBody1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch2(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch2(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopBody1cont1(int const& bytes,int loopDepth) 
	{
															#line 951 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		maxBytes = std::max<int>(CLIENT_KNOBS->APPLY_MAX_INCREASE_FACTOR * bytes, maxBytes);
															#line 952 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (error.isError())
															#line 4651 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 953 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			return a_body1Catch2(error.getError(), std::max(0, loopDepth - 2));
															#line 4655 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 935 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		++idx;
															#line 4659 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopBody1cont1(int && bytes,int loopDepth) 
	{
															#line 951 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		maxBytes = std::max<int>(CLIENT_KNOBS->APPLY_MAX_INCREASE_FACTOR * bytes, maxBytes);
															#line 952 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (error.isError())
															#line 4670 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 953 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			return a_body1Catch2(error.getError(), std::max(0, loopDepth - 2));
															#line 4674 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 935 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		++idx;
															#line 4678 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1cont2loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopBody1when1(int const& bytes,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1loopBody1cont1(bytes, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont1loopBody1when1(int && bytes,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont1loopBody1cont1(std::move(bytes), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state > 0) static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<ApplyMutationsActor*>(this)->ActorCallback< ApplyMutationsActor, 2, int >::remove();

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 2, int >*,int const& value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 2, int >*,int && value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ApplyMutationsActor, 2, int >*,Error err) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont2loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 958 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		beginVersion = newEndVersion;
															#line 959 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (BUGGIFY)
															#line 4764 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 960 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(2.0);
															#line 960 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4770 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
			static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 5;
															#line 960 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 4, Void >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 4775 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 958 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		beginVersion = newEndVersion;
															#line 959 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (BUGGIFY)
															#line 4791 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 960 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_4 = delay(2.0);
															#line 960 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4797 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch2(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont2loopBody1cont7when1(__when_expr_4.get(), loopDepth); };
			static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 5;
															#line 960 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ApplyMutationsActor, 4, Void >*>(static_cast<ApplyMutationsActor*>(this)));
															#line 4802 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont2loopBody1cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state > 0) static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<ApplyMutationsActor*>(this)->ActorCallback< ApplyMutationsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont2loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ApplyMutationsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont2loopBody1cont8(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2loopBody1cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ApplyMutationsActor*>(this)->actor_wait_state > 0) static_cast<ApplyMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<ApplyMutationsActor*>(this)->ActorCallback< ApplyMutationsActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2loopBody1cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ApplyMutationsActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont2loopBody1cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ApplyMutationsActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key uid;
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key addPrefix;
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key removePrefix;
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version beginVersion;
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version* endVersion;
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PublicRequestStream<CommitTransactionRequest> commit;
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	NotifiedVersion* committedVersion;
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<KeyRangeMap<Version>> keyVersion;
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	bool provisionalProxy;
															#line 889 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	FlowLock commitLock;
															#line 890 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	PromiseStream<Future<Void>> addActor;
															#line 891 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Future<Void> error;
															#line 892 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int maxBytes;
															#line 893 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::shared_ptr<DatabaseConfiguration> dbConfig;
															#line 912 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version newEndVersion;
															#line 919 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Standalone<VectorRef<KeyRangeRef>> ranges;
															#line 920 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	size_t idx;
															#line 921 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::vector<PromiseStream<RCGroup>> results;
															#line 922 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::vector<Future<Void>> rc;
															#line 923 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::vector<Reference<FlowLock>> locks;
															#line 5022 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via applyMutations()
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class ApplyMutationsActor final : public Actor<Void>, public ActorCallback< ApplyMutationsActor, 0, Void >, public ActorCallback< ApplyMutationsActor, 1, Void >, public ActorCallback< ApplyMutationsActor, 2, int >, public ActorCallback< ApplyMutationsActor, 3, Void >, public ActorCallback< ApplyMutationsActor, 4, Void >, public FastAllocated<ApplyMutationsActor>, public ApplyMutationsActorState<ApplyMutationsActor> {
															#line 5027 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<ApplyMutationsActor>::operator new;
	using FastAllocated<ApplyMutationsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(4650870759558101248UL, 2924517343442685184UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ApplyMutationsActor, 0, Void >;
friend struct ActorCallback< ApplyMutationsActor, 1, Void >;
friend struct ActorCallback< ApplyMutationsActor, 2, int >;
friend struct ActorCallback< ApplyMutationsActor, 3, Void >;
friend struct ActorCallback< ApplyMutationsActor, 4, Void >;
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	ApplyMutationsActor(Database const& cx,Key const& uid,Key const& addPrefix,Key const& removePrefix,Version const& beginVersion,Version* const& endVersion,PublicRequestStream<CommitTransactionRequest> const& commit,NotifiedVersion* const& committedVersion,Reference<KeyRangeMap<Version>> const& keyVersion,bool const& provisionalProxy) 
															#line 5048 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ApplyMutationsActorState<ApplyMutationsActor>(cx, uid, addPrefix, removePrefix, beginVersion, endVersion, commit, committedVersion, keyVersion, provisionalProxy),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("applyMutations", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17207310279585965312UL, 13534680049970660864UL);
		ActorExecutionContextHelper __helper(static_cast<ApplyMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("applyMutations");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("applyMutations", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ApplyMutationsActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ApplyMutationsActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ApplyMutationsActor, 2, int >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ApplyMutationsActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ApplyMutationsActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> applyMutations( Database const& cx, Key const& uid, Key const& addPrefix, Key const& removePrefix, Version const& beginVersion, Version* const& endVersion, PublicRequestStream<CommitTransactionRequest> const& commit, NotifiedVersion* const& committedVersion, Reference<KeyRangeMap<Version>> const& keyVersion, bool const& provisionalProxy ) {
															#line 879 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new ApplyMutationsActor(cx, uid, addPrefix, removePrefix, beginVersion, endVersion, commit, committedVersion, keyVersion, provisionalProxy));
															#line 5085 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 973 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

															#line 5090 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via _eraseLogData()
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class _eraseLogDataActor>
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class _eraseLogDataActorState {
															#line 5097 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	_eraseLogDataActorState(Reference<ReadYourWritesTransaction> const& tr,Key const& logUidValue,Key const& destUidValue,Optional<Version> const& endVersion,CheckBackupUID const& checkBackupUid,Version const& backupUid) 
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : tr(tr),
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   logUidValue(logUidValue),
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   destUidValue(destUidValue),
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   endVersion(endVersion),
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   checkBackupUid(checkBackupUid),
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   backupUid(backupUid),
															#line 980 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   backupLatestVersionsPath(destUidValue.withPrefix(backupLatestVersionsPrefix)),
															#line 981 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   backupLatestVersionsKey(logUidValue.withPrefix(backupLatestVersionsPath))
															#line 5118 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("_eraseLogData", reinterpret_cast<unsigned long>(this));

	}
	~_eraseLogDataActorState() 
	{
		fdb_probe_actor_destroy("_eraseLogData", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 983 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!destUidValue.size())
															#line 5133 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 984 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (!static_cast<_eraseLogDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_eraseLogDataActorState(); static_cast<_eraseLogDataActor*>(this)->destroy(); return 0; }
															#line 5137 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				new (&static_cast<_eraseLogDataActor*>(this)->SAV< Void >::value()) Void(Void());
				this->~_eraseLogDataActorState();
				static_cast<_eraseLogDataActor*>(this)->finishSendAndDelPromiseRef();
				return 0;
			}
															#line 987 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 988 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 990 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (checkBackupUid)
															#line 5149 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 991 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Subspace sourceStates = Subspace(databaseBackupPrefixRange.begin).get(BackupAgentBase::keySourceStates).get(logUidValue);
															#line 993 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Optional<Value>> __when_expr_0 = tr->get(sourceStates.pack(DatabaseBackupAgent::keyFolderId));
															#line 993 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5157 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 1;
															#line 993 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< _eraseLogDataActor, 0, Optional<Value> >*>(static_cast<_eraseLogDataActor*>(this)));
															#line 5162 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
				loopDepth = a_body1cont1(loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~_eraseLogDataActorState();
		static_cast<_eraseLogDataActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 998 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = tr->getRange(KeyRangeRef(backupLatestVersionsPath, strinc(backupLatestVersionsPath)), CLIENT_KNOBS->TOO_MANY);
															#line 998 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5192 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 2;
															#line 998 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< _eraseLogDataActor, 1, RangeResult >*>(static_cast<_eraseLogDataActor*>(this)));
															#line 5197 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3(Optional<Value> const& v,int loopDepth) 
	{
															#line 994 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (v.present() && BinaryReader::fromStringRef<Version>(v.get(), Unversioned()) > backupUid)
															#line 5206 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 995 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<_eraseLogDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_eraseLogDataActorState(); static_cast<_eraseLogDataActor*>(this)->destroy(); return 0; }
															#line 5210 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<_eraseLogDataActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~_eraseLogDataActorState();
			static_cast<_eraseLogDataActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(Optional<Value> && v,int loopDepth) 
	{
															#line 994 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (v.present() && BinaryReader::fromStringRef<Version>(v.get(), Unversioned()) > backupUid)
															#line 5224 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 995 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<_eraseLogDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_eraseLogDataActorState(); static_cast<_eraseLogDataActor*>(this)->destroy(); return 0; }
															#line 5228 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<_eraseLogDataActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~_eraseLogDataActorState();
			static_cast<_eraseLogDataActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> const& v,int loopDepth) 
	{
		loopDepth = a_body1cont3(v, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Optional<Value> && v,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(v), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state > 0) static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 0;
		static_cast<_eraseLogDataActor*>(this)->ActorCallback< _eraseLogDataActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< _eraseLogDataActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(int loopDepth) 
	{
															#line 1002 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		currBeginVersion = invalidVersion;
															#line 1003 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for( auto backupVersion : backupVersions ) {
															#line 1004 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			Key currLogUidValue = backupVersion.key.removePrefix(backupLatestVersionsPrefix).removePrefix(destUidValue);
															#line 1006 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (currLogUidValue == logUidValue)
															#line 5323 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1007 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				currBeginVersion = BinaryReader::fromStringRef<Version>(backupVersion.value, Unversioned());
															#line 5327 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				break;
			}
		}
															#line 1013 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (currBeginVersion == invalidVersion)
															#line 5333 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1014 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!static_cast<_eraseLogDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_eraseLogDataActorState(); static_cast<_eraseLogDataActor*>(this)->destroy(); return 0; }
															#line 5337 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			new (&static_cast<_eraseLogDataActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~_eraseLogDataActorState();
			static_cast<_eraseLogDataActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1017 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		currEndVersion = std::numeric_limits<Version>::max();
															#line 1018 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (endVersion.present())
															#line 5347 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1019 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			currEndVersion = std::min(currEndVersion, endVersion.get());
															#line 5351 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 1022 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		nextSmallestVersion = currEndVersion;
															#line 1023 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		bool clearLogRangesRequired = true;
															#line 1026 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (backupVersions.size() > 1)
															#line 5359 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1027 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			for( auto backupVersion : backupVersions ) {
															#line 1028 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Key currLogUidValue = backupVersion.key.removePrefix(backupLatestVersionsPrefix).removePrefix(destUidValue);
															#line 1029 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				Version currVersion = BinaryReader::fromStringRef<Version>(backupVersion.value, Unversioned());
															#line 1031 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (currLogUidValue == logUidValue)
															#line 5369 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
					continue;
				}
				else
				{
															#line 1033 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					if (currVersion > currBeginVersion)
															#line 5377 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					{
															#line 1034 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						nextSmallestVersion = std::min(currVersion, nextSmallestVersion);
															#line 5381 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
					else
					{
															#line 1037 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						clearLogRangesRequired = false;
															#line 5387 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
						break;
					}
				}
			}
		}
															#line 1043 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (endVersion.present() || backupVersions.size() != 1 || BUGGIFY)
															#line 5395 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1044 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (!endVersion.present())
															#line 5399 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1046 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr->clear(backupLatestVersionsKey);
															#line 1047 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (backupVersions.size() == 1)
															#line 5405 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 1048 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					tr->clear(prefixRange(destUidValue.withPrefix(logRangesRange.begin)));
															#line 5409 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
			}
			else
			{
															#line 1052 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr->set(backupLatestVersionsKey, BinaryWriter::toValue<Version>(currEndVersion, Unversioned()));
															#line 5416 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
															#line 1056 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (clearLogRangesRequired)
															#line 5420 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1057 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if ((nextSmallestVersion - currBeginVersion) / CLIENT_KNOBS->LOG_RANGE_BLOCK_SIZE >= std::numeric_limits<uint8_t>::max() || BUGGIFY)
															#line 5424 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 1060 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					Key baLogRangePrefix = destUidValue.withPrefix(backupLogKeys.begin);
															#line 1062 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					for(int h = 0;h <= std::numeric_limits<uint8_t>::max();h++) {
															#line 1063 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						uint64_t bv = bigEndian64(Version(0));
															#line 1064 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						uint64_t ev = bigEndian64(nextSmallestVersion);
															#line 1065 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						uint8_t h1 = h;
															#line 1066 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						Key vblockPrefix = StringRef(&h1, sizeof(uint8_t)).withPrefix(baLogRangePrefix);
															#line 1067 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						tr->clear(KeyRangeRef(StringRef((uint8_t*)&bv, sizeof(uint64_t)).withPrefix(vblockPrefix), StringRef((uint8_t*)&ev, sizeof(uint64_t)).withPrefix(vblockPrefix)));
															#line 5440 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
				}
				else
				{
															#line 1071 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					Standalone<VectorRef<KeyRangeRef>> ranges = getLogRanges(currBeginVersion, nextSmallestVersion, destUidValue);
															#line 1073 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					for( auto& range : ranges ) {
															#line 1074 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
						tr->clear(range);
															#line 5451 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
					}
				}
			}
		}
		else
		{
															#line 1080 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->clear(prefixRange(backupLatestVersionsPath));
															#line 1083 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->clear(prefixRange(destUidValue.withPrefix(backupLogKeys.begin)));
															#line 1086 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->clear(prefixRange(destUidValue.withPrefix(logRangesRange.begin)));
															#line 5464 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 1089 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!endVersion.present() && backupVersions.size() == 1)
															#line 5468 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1090 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<RangeResult> __when_expr_2 = tr->getRange(KeyRangeRef(destUidLookupPrefix, strinc(destUidLookupPrefix)), CLIENT_KNOBS->TOO_MANY);
															#line 1090 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5474 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont6when1(__when_expr_2.get(), loopDepth); };
			static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 3;
															#line 1090 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< _eraseLogDataActor, 2, RangeResult >*>(static_cast<_eraseLogDataActor*>(this)));
															#line 5479 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult const& __backupVersions,int loopDepth) 
	{
															#line 998 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		backupVersions = __backupVersions;
															#line 5493 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(RangeResult && __backupVersions,int loopDepth) 
	{
		backupVersions = std::move(__backupVersions);
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state > 0) static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 0;
		static_cast<_eraseLogDataActor*>(this)->ActorCallback< _eraseLogDataActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< _eraseLogDataActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 1099 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<_eraseLogDataActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~_eraseLogDataActorState(); static_cast<_eraseLogDataActor*>(this)->destroy(); return 0; }
															#line 5572 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<_eraseLogDataActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~_eraseLogDataActorState();
		static_cast<_eraseLogDataActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont28(RangeResult const& existingDestUidValues,int loopDepth) 
	{
															#line 1092 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for( auto it : existingDestUidValues ) {
															#line 1093 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (it.value == destUidValue)
															#line 5586 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1094 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr->clear(it.key);
															#line 5590 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont28(RangeResult && existingDestUidValues,int loopDepth) 
	{
															#line 1092 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		for( auto it : existingDestUidValues ) {
															#line 1093 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (it.value == destUidValue)
															#line 5603 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1094 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				tr->clear(it.key);
															#line 5607 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
		}
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(RangeResult const& existingDestUidValues,int loopDepth) 
	{
		loopDepth = a_body1cont28(existingDestUidValues, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(RangeResult && existingDestUidValues,int loopDepth) 
	{
		loopDepth = a_body1cont28(std::move(existingDestUidValues), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<_eraseLogDataActor*>(this)->actor_wait_state > 0) static_cast<_eraseLogDataActor*>(this)->actor_wait_state = 0;
		static_cast<_eraseLogDataActor*>(this)->ActorCallback< _eraseLogDataActor, 2, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 2, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< _eraseLogDataActor, 2, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< _eraseLogDataActor, 2, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key logUidValue;
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key destUidValue;
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Optional<Version> endVersion;
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CheckBackupUID checkBackupUid;
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version backupUid;
															#line 980 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key backupLatestVersionsPath;
															#line 981 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key backupLatestVersionsKey;
															#line 998 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RangeResult backupVersions;
															#line 1002 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version currBeginVersion;
															#line 1017 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version currEndVersion;
															#line 1022 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version nextSmallestVersion;
															#line 5713 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via _eraseLogData()
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class _eraseLogDataActor final : public Actor<Void>, public ActorCallback< _eraseLogDataActor, 0, Optional<Value> >, public ActorCallback< _eraseLogDataActor, 1, RangeResult >, public ActorCallback< _eraseLogDataActor, 2, RangeResult >, public FastAllocated<_eraseLogDataActor>, public _eraseLogDataActorState<_eraseLogDataActor> {
															#line 5718 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<_eraseLogDataActor>::operator new;
	using FastAllocated<_eraseLogDataActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6977497554565625600UL, 10694822610824132096UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< _eraseLogDataActor, 0, Optional<Value> >;
friend struct ActorCallback< _eraseLogDataActor, 1, RangeResult >;
friend struct ActorCallback< _eraseLogDataActor, 2, RangeResult >;
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	_eraseLogDataActor(Reference<ReadYourWritesTransaction> const& tr,Key const& logUidValue,Key const& destUidValue,Optional<Version> const& endVersion,CheckBackupUID const& checkBackupUid,Version const& backupUid) 
															#line 5737 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   _eraseLogDataActorState<_eraseLogDataActor>(tr, logUidValue, destUidValue, endVersion, checkBackupUid, backupUid),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("_eraseLogData", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(7090378008957281792UL, 14346794352726563584UL);
		ActorExecutionContextHelper __helper(static_cast<_eraseLogDataActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("_eraseLogData");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("_eraseLogData", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< _eraseLogDataActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< _eraseLogDataActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< _eraseLogDataActor, 2, RangeResult >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] static Future<Void> _eraseLogData( Reference<ReadYourWritesTransaction> const& tr, Key const& logUidValue, Key const& destUidValue, Optional<Version> const& endVersion, CheckBackupUID const& checkBackupUid, Version const& backupUid ) {
															#line 974 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new _eraseLogDataActor(tr, logUidValue, destUidValue, endVersion, checkBackupUid, backupUid));
															#line 5772 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 1101 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

Future<Void> eraseLogData(Reference<ReadYourWritesTransaction> tr,
                          Key logUidValue,
                          Key destUidValue,
                          Optional<Version> endVersion,
                          CheckBackupUID checkBackupUid,
                          Version backupUid) {
	return _eraseLogData(tr, logUidValue, destUidValue, endVersion, checkBackupUid, backupUid);
}

															#line 5786 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via cleanupLogMutations()
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class CleanupLogMutationsActor>
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CleanupLogMutationsActorState {
															#line 5793 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CleanupLogMutationsActorState(Database const& cx,Value const& destUidValue,bool const& deleteData) 
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   destUidValue(destUidValue),
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   deleteData(deleteData),
															#line 1112 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   backupLatestVersionsPath(destUidValue.withPrefix(backupLatestVersionsPrefix)),
															#line 1114 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   tr(new ReadYourWritesTransaction(cx)),
															#line 1115 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   removingLogUid(),
															#line 1116 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   loggedLogUids()
															#line 5812 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("cleanupLogMutations", reinterpret_cast<unsigned long>(this));

	}
	~CleanupLogMutationsActorState() 
	{
		fdb_probe_actor_destroy("cleanupLogMutations", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1118 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			;
															#line 5827 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CleanupLogMutationsActorState();
		static_cast<CleanupLogMutationsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1120 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1121 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1123 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange( KeyRangeRef(backupLatestVersionsPath, strinc(backupLatestVersionsPath)), CLIENT_KNOBS->TOO_MANY);
															#line 1123 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5864 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 1;
															#line 1123 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 0, RangeResult >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 5869 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1197 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_4 = tr->onError(e);
															#line 1197 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5893 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_4.get(), loopDepth); };
			static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 5;
															#line 1197 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 4, Void >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 5898 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 1125 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		readVer = tr->getReadVersion().get();
															#line 1127 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		minVersion = std::numeric_limits<Version>::max();
															#line 1128 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		minVersionLogUid = Key();
															#line 1130 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		backupIdx = 0;
															#line 1131 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		;
															#line 5921 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __backupVersions,int loopDepth) 
	{
															#line 1123 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		backupVersions = __backupVersions;
															#line 5930 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __backupVersions,int loopDepth) 
	{
		backupVersions = std::move(__backupVersions);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state > 0) static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupLogMutationsActor*>(this)->ActorCallback< CleanupLogMutationsActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CleanupLogMutationsActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1169 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (deleteData)
															#line 6009 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1170 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (readVer - minVersion > CLIENT_KNOBS->MIN_CLEANUP_SECONDS * CLIENT_KNOBS->CORE_VERSIONSPERSECOND && (!removingLogUid.present() || minVersionLogUid == removingLogUid.get()))
															#line 6013 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1172 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				removingLogUid = minVersionLogUid;
															#line 1173 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				StrictFuture<Void> __when_expr_2 = eraseLogData(tr, minVersionLogUid, destUidValue);
															#line 1173 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6021 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
				static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 3;
															#line 1173 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 2, Void >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 6026 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			else
			{
															#line 1177 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (removingLogUid.present() && minVersionLogUid != removingLogUid.get())
															#line 6033 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 1178 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("\nWARNING: The oldest tag was possibly removed, run again without `--delete-data' to " "check.\n\n");
															#line 6037 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1181 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("\nWARNING: Did not delete data because the tag is not at least %.4f hours behind. Change " "`--min-cleanup-seconds' to adjust this threshold.\n\n", CLIENT_KNOBS->MIN_CLEANUP_SECONDS / 3600.0);
															#line 6043 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				loopDepth = a_body1loopBody1cont5(loopDepth);
			}
		}
		else
		{
															#line 1185 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (readVer - minVersion > CLIENT_KNOBS->MIN_CLEANUP_SECONDS * CLIENT_KNOBS->CORE_VERSIONSPERSECOND)
															#line 6052 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1187 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				printf("\nPassing `--delete-data' would delete the tag that is %.4f hours behind.\n\n", (readVer - minVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6056 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1190 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				printf("\nPassing `--delete-data' would not delete the tag that is %.4f hours behind. Change " "`--min-cleanup-seconds' to adjust the cleanup threshold.\n\n", (readVer - minVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6062 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
			loopDepth = a_body1loopBody1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 1131 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!(backupIdx < backupVersions.size()))
															#line 6080 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1132 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		currVersion = BinaryReader::fromStringRef<Version>(backupVersions[backupIdx].value, Unversioned());
															#line 1134 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		currLogUid = backupVersions[backupIdx].key.removePrefix(backupLatestVersionsPrefix).removePrefix(destUidValue);
															#line 1136 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (currVersion < minVersion)
															#line 6090 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1137 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			minVersionLogUid = currLogUid;
															#line 1138 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			minVersion = currVersion;
															#line 6096 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
															#line 1141 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!loggedLogUids.count(currLogUid))
															#line 6100 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1142 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			foundDRKey = tr->get(Subspace(databaseBackupPrefixRange.begin) .get(BackupAgentBase::keySourceStates) .get(currLogUid) .pack(DatabaseBackupAgent::keyStateStatus));
															#line 1146 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			foundBackupKey = tr->get( Subspace(currLogUid.withPrefix("uid->config/"_sr).withPrefix(fileBackupPrefixRange.begin)) .pack("stateEnum"_sr));
															#line 1149 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_1 = success(foundDRKey) && success(foundBackupKey);
															#line 1149 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6110 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 2;
															#line 1149 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 1, Void >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 6115 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1131 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		backupIdx++;
															#line 6142 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 1151 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (foundDRKey.get().present() && foundBackupKey.get().present())
															#line 6151 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1152 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			printf("WARNING: Found a tag that looks like both a backup and a DR. This tag is %.4f hours " "behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6155 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
		else
		{
															#line 1155 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (foundDRKey.get().present() && !foundBackupKey.get().present())
															#line 6161 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1156 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				printf("Found a DR that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6165 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1158 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (!foundDRKey.get().present() && foundBackupKey.get().present())
															#line 6171 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 1159 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("Found a Backup that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6175 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1162 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("WARNING: Found an unknown tag that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6181 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
			}
		}
															#line 1165 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		loggedLogUids.insert(currLogUid);
															#line 6187 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 1151 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (foundDRKey.get().present() && foundBackupKey.get().present())
															#line 6196 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1152 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			printf("WARNING: Found a tag that looks like both a backup and a DR. This tag is %.4f hours " "behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6200 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		}
		else
		{
															#line 1155 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (foundDRKey.get().present() && !foundBackupKey.get().present())
															#line 6206 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			{
															#line 1156 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				printf("Found a DR that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6210 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1158 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
				if (!foundDRKey.get().present() && foundBackupKey.get().present())
															#line 6216 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				{
															#line 1159 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("Found a Backup that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6220 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
				else
				{
															#line 1162 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
					printf("WARNING: Found an unknown tag that is %.4f hours behind.\n", (readVer - currVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6226 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
				}
			}
		}
															#line 1165 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		loggedLogUids.insert(currLogUid);
															#line 6232 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state > 0) static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupLogMutationsActor*>(this)->ActorCallback< CleanupLogMutationsActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CleanupLogMutationsActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 1195 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<CleanupLogMutationsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CleanupLogMutationsActorState(); static_cast<CleanupLogMutationsActor*>(this)->destroy(); return 0; }
															#line 6316 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<CleanupLogMutationsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CleanupLogMutationsActorState();
		static_cast<CleanupLogMutationsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 1174 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr->commit();
															#line 1174 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6336 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_3.get(), loopDepth); };
		static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 4;
															#line 1174 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 3, Void >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 6341 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 1174 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		StrictFuture<Void> __when_expr_3 = tr->commit();
															#line 1174 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6352 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont6when1(__when_expr_3.get(), loopDepth); };
		static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 4;
															#line 1174 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CleanupLogMutationsActor, 3, Void >*>(static_cast<CleanupLogMutationsActor*>(this)));
															#line 6357 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state > 0) static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupLogMutationsActor*>(this)->ActorCallback< CleanupLogMutationsActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CleanupLogMutationsActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont7(Void const& _,int loopDepth) 
	{
															#line 1175 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		printf("\nSuccessfully removed the tag that was %.4f hours behind.\n\n", (readVer - minVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6441 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont7(Void && _,int loopDepth) 
	{
															#line 1175 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		printf("\nSuccessfully removed the tag that was %.4f hours behind.\n\n", (readVer - minVersion) / (3600.0 * CLIENT_KNOBS->CORE_VERSIONSPERSECOND));
															#line 6450 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state > 0) static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupLogMutationsActor*>(this)->ActorCallback< CleanupLogMutationsActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< CleanupLogMutationsActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state > 0) static_cast<CleanupLogMutationsActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupLogMutationsActor*>(this)->ActorCallback< CleanupLogMutationsActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< CleanupLogMutationsActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< CleanupLogMutationsActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Value destUidValue;
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	bool deleteData;
															#line 1112 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key backupLatestVersionsPath;
															#line 1114 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1115 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Optional<Key> removingLogUid;
															#line 1116 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	std::set<Key> loggedLogUids;
															#line 1123 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RangeResult backupVersions;
															#line 1125 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version readVer;
															#line 1127 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version minVersion;
															#line 1128 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key minVersionLogUid;
															#line 1130 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	int backupIdx;
															#line 1132 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Version currVersion;
															#line 1134 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Key currLogUid;
															#line 1142 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Future<Optional<Value>> foundDRKey;
															#line 1146 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Future<Optional<Value>> foundBackupKey;
															#line 6649 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via cleanupLogMutations()
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CleanupLogMutationsActor final : public Actor<Void>, public ActorCallback< CleanupLogMutationsActor, 0, RangeResult >, public ActorCallback< CleanupLogMutationsActor, 1, Void >, public ActorCallback< CleanupLogMutationsActor, 2, Void >, public ActorCallback< CleanupLogMutationsActor, 3, Void >, public ActorCallback< CleanupLogMutationsActor, 4, Void >, public FastAllocated<CleanupLogMutationsActor>, public CleanupLogMutationsActorState<CleanupLogMutationsActor> {
															#line 6654 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<CleanupLogMutationsActor>::operator new;
	using FastAllocated<CleanupLogMutationsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12101360650622931968UL, 15067830461370377472UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CleanupLogMutationsActor, 0, RangeResult >;
friend struct ActorCallback< CleanupLogMutationsActor, 1, Void >;
friend struct ActorCallback< CleanupLogMutationsActor, 2, Void >;
friend struct ActorCallback< CleanupLogMutationsActor, 3, Void >;
friend struct ActorCallback< CleanupLogMutationsActor, 4, Void >;
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CleanupLogMutationsActor(Database const& cx,Value const& destUidValue,bool const& deleteData) 
															#line 6675 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   CleanupLogMutationsActorState<CleanupLogMutationsActor>(cx, destUidValue, deleteData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("cleanupLogMutations", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5485089562015519744UL, 10719617229371527680UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupLogMutationsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("cleanupLogMutations");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("cleanupLogMutations", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CleanupLogMutationsActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CleanupLogMutationsActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CleanupLogMutationsActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< CleanupLogMutationsActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< CleanupLogMutationsActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> cleanupLogMutations( Database const& cx, Value const& destUidValue, bool const& deleteData ) {
															#line 1111 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new CleanupLogMutationsActor(cx, destUidValue, deleteData));
															#line 6712 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 1201 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

															#line 6717 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via cleanupBackup()
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
template <class CleanupBackupActor>
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CleanupBackupActorState {
															#line 6724 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CleanupBackupActorState(Database const& cx,DeleteData const& deleteData) 
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		 : cx(cx),
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   deleteData(deleteData),
															#line 1203 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		   tr(new ReadYourWritesTransaction(cx))
															#line 6735 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("cleanupBackup", reinterpret_cast<unsigned long>(this));

	}
	~CleanupBackupActorState() 
	{
		fdb_probe_actor_destroy("cleanupBackup", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1204 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			;
															#line 6750 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CleanupBackupActorState();
		static_cast<CleanupBackupActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 1206 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1207 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1209 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<RangeResult> __when_expr_0 = tr->getRange(KeyRangeRef(destUidLookupPrefix, strinc(destUidLookupPrefix)), CLIENT_KNOBS->TOO_MANY);
															#line 1209 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupBackupActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 6787 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CleanupBackupActor*>(this)->actor_wait_state = 1;
															#line 1209 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CleanupBackupActor, 0, RangeResult >*>(static_cast<CleanupBackupActor*>(this)));
															#line 6792 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1217 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr->onError(e);
															#line 1217 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupBackupActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6816 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_2.get(), loopDepth); };
			static_cast<CleanupBackupActor*>(this)->actor_wait_state = 3;
															#line 1217 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CleanupBackupActor, 2, Void >*>(static_cast<CleanupBackupActor*>(this)));
															#line 6821 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 1212 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		RangeForbody1loopBody1cont2Iterator0 = std::begin(destUids);
															#line 6836 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult const& __destUids,int loopDepth) 
	{
															#line 1209 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		destUids = __destUids;
															#line 6845 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(RangeResult && __destUids,int loopDepth) 
	{
		destUids = std::move(__destUids);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CleanupBackupActor*>(this)->actor_wait_state > 0) static_cast<CleanupBackupActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupBackupActor*>(this)->ActorCallback< CleanupBackupActor, 0, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 0, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 0, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CleanupBackupActor, 0, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
															#line 1215 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!static_cast<CleanupBackupActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CleanupBackupActorState(); static_cast<CleanupBackupActor*>(this)->destroy(); return 0; }
															#line 6924 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		new (&static_cast<CleanupBackupActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CleanupBackupActorState();
		static_cast<CleanupBackupActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 1212 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		if (!(RangeForbody1loopBody1cont2Iterator0 != std::end(destUids)))
															#line 6943 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1212 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		auto destUid = *RangeForbody1loopBody1cont2Iterator0;
															#line 6949 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		{
															#line 1213 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			StrictFuture<Void> __when_expr_1 = cleanupLogMutations(cx, destUid.value, deleteData);
															#line 1213 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			if (static_cast<CleanupBackupActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6955 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2loopBody1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CleanupBackupActor*>(this)->actor_wait_state = 2;
															#line 1213 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CleanupBackupActor, 1, Void >*>(static_cast<CleanupBackupActor*>(this)));
															#line 6960 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1212 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
		++RangeForbody1loopBody1cont2Iterator0;
															#line 6983 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CleanupBackupActor*>(this)->actor_wait_state > 0) static_cast<CleanupBackupActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupBackupActor*>(this)->ActorCallback< CleanupBackupActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CleanupBackupActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CleanupBackupActor*>(this)->actor_wait_state > 0) static_cast<CleanupBackupActor*>(this)->actor_wait_state = 0;
		static_cast<CleanupBackupActor*>(this)->ActorCallback< CleanupBackupActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CleanupBackupActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CleanupBackupActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Database cx;
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	DeleteData deleteData;
															#line 1203 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 1209 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	RangeResult destUids;
															#line 1212 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	decltype(std::begin(std::declval<RangeResult>())) RangeForbody1loopBody1cont2Iterator0;
															#line 7172 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
};
// This generated class is to be used only via cleanupBackup()
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
class CleanupBackupActor final : public Actor<Void>, public ActorCallback< CleanupBackupActor, 0, RangeResult >, public ActorCallback< CleanupBackupActor, 1, Void >, public ActorCallback< CleanupBackupActor, 2, Void >, public FastAllocated<CleanupBackupActor>, public CleanupBackupActorState<CleanupBackupActor> {
															#line 7177 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
public:
	using FastAllocated<CleanupBackupActor>::operator new;
	using FastAllocated<CleanupBackupActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14488605108394561280UL, 15194882660423087616UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CleanupBackupActor, 0, RangeResult >;
friend struct ActorCallback< CleanupBackupActor, 1, Void >;
friend struct ActorCallback< CleanupBackupActor, 2, Void >;
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	CleanupBackupActor(Database const& cx,DeleteData const& deleteData) 
															#line 7196 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   CleanupBackupActorState<CleanupBackupActor>(cx, deleteData),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("cleanupBackup", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8531268470707185664UL, 16321355424300612352UL);
		ActorExecutionContextHelper __helper(static_cast<CleanupBackupActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("cleanupBackup");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("cleanupBackup", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CleanupBackupActor, 0, RangeResult >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CleanupBackupActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CleanupBackupActor, 2, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
[[nodiscard]] Future<Void> cleanupBackup( Database const& cx, DeleteData const& deleteData ) {
															#line 1202 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"
	return Future<Void>(new CleanupBackupActor(cx, deleteData));
															#line 7231 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/build_output/fdbclient/BackupAgentBase.actor.g.cpp.py_gen"
}

#line 1221 "/codebuild/output/src3638569120/src/github.com/apple/foundationdb/fdbclient/BackupAgentBase.actor.cpp"

// Convert the status text to an enumerated value
BackupAgentBase::EnumState BackupAgentBase::getState(std::string const& stateText) {
	auto enState = EnumState::STATE_ERRORED;

	if (stateText.empty()) {
		enState = EnumState::STATE_NEVERRAN;
	}

	else if (!stateText.compare("has been submitted")) {
		enState = EnumState::STATE_SUBMITTED;
	}

	else if (!stateText.compare("has been started")) {
		enState = EnumState::STATE_RUNNING;
	}

	else if (!stateText.compare("is differential")) {
		enState = EnumState::STATE_RUNNING_DIFFERENTIAL;
	}

	else if (!stateText.compare("has been completed")) {
		enState = EnumState::STATE_COMPLETED;
	}

	else if (!stateText.compare("has been aborted")) {
		enState = EnumState::STATE_ABORTED;
	}

	else if (!stateText.compare("has been partially aborted")) {
		enState = EnumState::STATE_PARTIALLY_ABORTED;
	}

	return enState;
}

const char* BackupAgentBase::getStateText(EnumState enState) {
	const char* stateText;

	switch (enState) {
	case EnumState::STATE_ERRORED:
		stateText = "has errored";
		break;
	case EnumState::STATE_NEVERRAN:
		stateText = "has never been started";
		break;
	case EnumState::STATE_SUBMITTED:
		stateText = "has been submitted";
		break;
	case EnumState::STATE_RUNNING:
		stateText = "has been started";
		break;
	case EnumState::STATE_RUNNING_DIFFERENTIAL:
		stateText = "is differential";
		break;
	case EnumState::STATE_COMPLETED:
		stateText = "has been completed";
		break;
	case EnumState::STATE_ABORTED:
		stateText = "has been aborted";
		break;
	case EnumState::STATE_PARTIALLY_ABORTED:
		stateText = "has been partially aborted";
		break;
	default:
		stateText = "<undefined>";
		break;
	}

	return stateText;
}

const char* BackupAgentBase::getStateName(EnumState enState) {
	switch (enState) {
	case EnumState::STATE_ERRORED:
		return "Errored";
	case EnumState::STATE_NEVERRAN:
		return "NeverRan";
	case EnumState::STATE_SUBMITTED:
		return "Submitted";
		break;
	case EnumState::STATE_RUNNING:
		return "Running";
	case EnumState::STATE_RUNNING_DIFFERENTIAL:
		return "RunningDifferentially";
	case EnumState::STATE_COMPLETED:
		return "Completed";
	case EnumState::STATE_ABORTED:
		return "Aborted";
	case EnumState::STATE_PARTIALLY_ABORTED:
		return "Aborting";
	default:
		return "<undefined>";
	}
}

bool BackupAgentBase::isRunnable(EnumState enState) {
	switch (enState) {
	case EnumState::STATE_SUBMITTED:
	case EnumState::STATE_RUNNING:
	case EnumState::STATE_RUNNING_DIFFERENTIAL:
	case EnumState::STATE_PARTIALLY_ABORTED:
		return true;
	default:
		return false;
	}
}

Standalone<StringRef> BackupAgentBase::getCurrentTime() {
	double t = now();
	time_t curTime = t;
	char buffer[128];
	struct tm* timeinfo;
	timeinfo = localtime(&curTime);
	strftime(buffer, 128, "%Y-%m-%d-%H-%M-%S", timeinfo);

	std::string time(buffer);
	return StringRef(time + format(".%06d", (int)(1e6 * (t - curTime))));
}

std::string const BackupAgentBase::defaultTagName = "default";

void addDefaultBackupRanges(Standalone<VectorRef<KeyRangeRef>>& backupKeys) {
	backupKeys.push_back_deep(backupKeys.arena(), normalKeys);

	for (auto& r : getSystemBackupRanges()) {
		backupKeys.push_back_deep(backupKeys.arena(), r);
	}
}

VectorRef<KeyRangeRef> const& getSystemBackupRanges() {
	static Standalone<VectorRef<KeyRangeRef>> systemBackupRanges;
	return systemBackupRanges;
}

KeyRangeMap<bool> const& systemBackupMutationMask() {
	static KeyRangeMap<bool> mask;
	if (mask.size() == 1) {
		for (auto r : getSystemBackupRanges()) {
			mask.insert(r, true);
		}
	}

	return mask;
}

KeyRangeRef const& getDefaultBackupSharedRange() {
	static KeyRangeRef defaultSharedRange(""_sr, ""_sr);
	return defaultSharedRange;
}
