#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
/*
 * RemoveServersSafely.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "fdbclient/NativeAPI.actor.h"
#include "fdbserver/TesterInterface.actor.h"
#include "fdbserver/WorkerInterface.actor.h"
#include "fdbserver/workloads/workloads.actor.h"
#include "fdbrpc/simulator.h"
#include "fdbrpc/SimulatorProcessInfo.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "flow/actorcompiler.h" // This must be the last #include.

template <>
std::string describe(uint32_t const& item) {
	return format("%d", item);
}

struct RemoveServersSafelyWorkload : TestWorkload {
	static constexpr auto NAME = "RemoveServersSafely";

	bool enabled, killProcesses;
	int minMachinesToKill, maxMachinesToKill, maxSafetyCheckRetries;
	double minDelay, maxDelay;
	double kill1Timeout, kill2Timeout;

	std::set<AddressExclusion> toKill1, toKill2;
	std::map<AddressExclusion, Optional<Standalone<StringRef>>> machine_ids; // ip -> Locality Zone id
	std::map<AddressExclusion, std::set<AddressExclusion>> machineProcesses; // ip -> ip:port

	RemoveServersSafelyWorkload(WorkloadContext const& wcx) : TestWorkload(wcx) {
		enabled =
		    !clientId && g_network->isSimulated(); // only do this on the "first" client, and only when in simulation
		minMachinesToKill = getOption(options, "minMachinesToKill"_sr, 1);
		maxMachinesToKill = getOption(options, "maxMachinesToKill"_sr, 10);
		maxMachinesToKill = std::max(minMachinesToKill, maxMachinesToKill);
		maxSafetyCheckRetries = getOption(options, "maxSafetyCheckRetries"_sr, 50);
		minDelay = getOption(options, "minDelay"_sr, 0.0);
		maxDelay = getOption(options, "maxDelay"_sr, 60.0);
		kill1Timeout = getOption(options, "kill1Timeout"_sr, 60.0);
		kill2Timeout = getOption(options, "kill2Timeout"_sr, 6000.0);
		killProcesses = deterministicRandom()->random01() < 0.5;
		if (g_network->isSimulated()) {
			g_simulator->allowLogSetKills = false;
		}
	}

	Future<Void> setup(Database const& cx) override {
		if (!enabled)
			return Void();

		std::map<Optional<Standalone<StringRef>>, AddressExclusion> machinesMap; // Locality Zone Id -> ip address
		std::vector<AddressExclusion>
		    processAddrs; // IF (killProcesses) THEN ip:port ELSE ip addresses   unique list of the machines
		std::map<IPAddress, Optional<Standalone<StringRef>>> ip_dcid;
		auto processes = getServers();
		for (auto& it : processes) {
			AddressExclusion machineIp(it->address.ip);
			AddressExclusion pAddr(it->address.ip, it->address.port);

			TraceEvent("RemoveAndKill")
			    .detail("Step", "listAddresses")
			    .detail("Address", pAddr.toString())
			    .detail("Process", describe(*it))
			    .detail("Dcid", it->locality.dcId().get().toString())
			    .detail("Zoneid", it->locality.zoneId().get().toString())
			    .detail("MachineId", it->locality.machineId().get().toString());

			if (!g_simulator->protectedAddresses.contains(it->address))
				processAddrs.push_back(pAddr);
			machineProcesses[machineIp].insert(pAddr);

			// add only one entry for each machine
			if (!machinesMap.contains(it->locality.zoneId()))
				machinesMap[it->locality.zoneId()] = machineIp;

			machine_ids[machineIp] = it->locality.zoneId();
			ip_dcid[it->address.ip] = it->locality.dcId();
		}

		int processCount = processAddrs.size();
		int nToKill1 = deterministicRandom()->randomInt(std::min(processCount, minMachinesToKill),
		                                                std::min(processCount, maxMachinesToKill) + 1);
		int nToKill2 = deterministicRandom()->randomInt(std::min(processCount, minMachinesToKill),
		                                                std::min(processCount, maxMachinesToKill) + 1);
		toKill1 = random_subset(processAddrs, nToKill1);
		toKill2 = random_subset(processAddrs, nToKill2);

		if (!killProcesses) {
			std::set<AddressExclusion> processSet;

			for (auto k1 : toKill1) {
				AddressExclusion machineIp(k1.ip);
				ASSERT(machineProcesses.contains(machineIp));
				// kill all processes on this machine even if it has a different ip address
				std::copy(machineProcesses[machineIp].begin(),
				          machineProcesses[machineIp].end(),
				          std::inserter(processSet, processSet.end()));
			}
			toKill1.insert(processSet.begin(), processSet.end());

			processSet.clear();
			for (auto k2 : toKill2) {
				AddressExclusion machineIp(k2.ip);
				ASSERT(machineProcesses.contains(machineIp));
				std::copy(machineProcesses[machineIp].begin(),
				          machineProcesses[machineIp].end(),
				          std::inserter(processSet, processSet.end()));
			}
			toKill2.insert(processSet.begin(), processSet.end());
		}

		for (AddressExclusion ex : toKill1) {
			AddressExclusion machineIp(ex.ip);
			ASSERT(machine_ids.contains(machineIp));
			g_simulator->disableSwapToMachine(machine_ids[machineIp]);
		}

		for (AddressExclusion ex : toKill2) {
			AddressExclusion machineIp(ex.ip);
			ASSERT(machine_ids.contains(machineIp));
			g_simulator->disableSwapToMachine(machine_ids[machineIp]);
		}

		return Void();
	}

	Future<Void> start(Database const& cx) override {
		if (!enabled)
			return Void();
		double delay = deterministicRandom()->random01() * (maxDelay - minDelay) + minDelay;
		return workloadMain(this, cx, delay, toKill1, toKill2);
	}

	Future<bool> check(Database const& cx) override { return true; }

	void getMetrics(std::vector<PerfMetric>&) override {}

	std::set<AddressExclusion> getNetworks(std::vector<ISimulator::ProcessInfo*> const& processes) {
		std::set<AddressExclusion> processAddrs;

		for (auto& processInfo : processes) {
			processAddrs.insert(AddressExclusion(processInfo->address.ip, processInfo->address.port));
		}
		return processAddrs;
	}

	// Get the list of processes whose ip:port or ip matches netAddrs.
	// Note: item in netAddrs may be ip (representing a machine) or ip:port (representing a process)
	std::vector<ISimulator::ProcessInfo*> getProcesses(std::set<AddressExclusion> const& netAddrs) {
		std::vector<ISimulator::ProcessInfo*> processes;
		std::set<AddressExclusion> processAddrs;
		UID functionId = nondeterministicRandom()->randomUniqueID();

		// Get the list of process network addresses
		for (auto& netAddr : netAddrs) {
			auto machineIpPorts = machineProcesses.find(netAddr);
			if (machineIpPorts != machineProcesses.end()) {
				ASSERT(machineIpPorts->second.size());
				for (auto& processAdd : machineIpPorts->second)
					processAddrs.insert(processAdd);
			} else {
				processAddrs.insert(netAddr);
			}
		}
		// Get the list of processes matching network address
		for (auto processInfo : g_simulator->getAllProcesses()) {
			auto processNet = AddressExclusion(processInfo->address.ip, processInfo->address.port);
			if (processAddrs.find(processNet) != processAddrs.end()) {
				processes.push_back(processInfo);
				TraceEvent("RemoveAndKill", functionId)
				    .detail("Step", "ProcessToKill")
				    .detail("ProcessAddress", processInfo->address)
				    .detail("Process", describe(*processInfo))
				    .detail("Failed", processInfo->failed)
				    .detail("Excluded", processInfo->excluded)
				    .detail("Rebooting", processInfo->rebooting)
				    .detail("Protected", g_simulator->protectedAddresses.contains(processInfo->address));
			} else {
				TraceEvent("RemoveAndKill", functionId)
				    .detail("Step", "ProcessNotToKill")
				    .detail("ProcessAddress", processInfo->address)
				    .detail("Process", describe(*processInfo))
				    .detail("Failed", processInfo->failed)
				    .detail("Excluded", processInfo->excluded)
				    .detail("Rebooting", processInfo->rebooting)
				    .detail("Protected", g_simulator->protectedAddresses.contains(processInfo->address));
			}
		}
		TraceEvent("RemoveAndKill", functionId)
		    .detail("Step", "getProcesses")
		    .detail("NetAddrSize", netAddrs.size())
		    .detail("ProcessAddrSize", processAddrs.size())
		    .detail("NetAddrs", describe(netAddrs))
		    .detail("ProcessAddrs", describe(processAddrs))
		    .detail("Processes", processes.size())
		    .detail("MachineProcesses", machineProcesses.size());

		return processes;
	}

	std::vector<ISimulator::ProcessInfo*> excludeAddresses(std::set<AddressExclusion> const& procAddrs) {
		// Get the updated list of processes which may have changed due to reboots, deletes, etc
		std::vector<ISimulator::ProcessInfo*> procArray = getProcesses(procAddrs);

		// Include all of the excluded machines because the first command of the next section is includeall
		TraceEvent("RemoveAndKill")
		    .detail("Step", "exclude addresses")
		    .detail("AddrTotal", procAddrs.size())
		    .detail("ProcTotal", procArray.size())
		    .detail("Addresses", describe(procAddrs))
		    .detail("ClusterAvailable", g_simulator->isAvailable());
		for (auto& procAddr : procAddrs) {
			g_simulator->excludeAddress(NetworkAddress(procAddr.ip, procAddr.port, true, false));
		}
		for (auto& procRecord : procArray) {
			procRecord->excluded = true;
			TraceEvent("RemoveAndKill")
			    .detail("Step", "ExcludeAddress")
			    .detail("ProcessAddress", procRecord->address)
			    .detail("Process", describe(*procRecord))
			    .detail("Failed", procRecord->failed)
			    .detail("Rebooting", procRecord->rebooting)
			    .detail("ClusterAvailable", g_simulator->isAvailable());
		}
		return procArray;
	}

	std::vector<ISimulator::ProcessInfo*> includeAddresses(std::set<AddressExclusion> const& procAddrs) {
		// Get the updated list of processes which may have changed due to reboots, deletes, etc
		std::vector<ISimulator::ProcessInfo*> procArray = getProcesses(procAddrs);

		// Include all of the excluded machines because the first command of the next section is includeall
		TraceEvent("RemoveAndKill")
		    .detail("Step", "include addresses")
		    .detail("AddrTotal", procAddrs.size())
		    .detail("ProcTotal", procArray.size())
		    .detail("Addresses", describe(procAddrs))
		    .detail("ClusterAvailable", g_simulator->isAvailable());
		for (auto& procAddr : procAddrs) {
			g_simulator->includeAddress(NetworkAddress(procAddr.ip, procAddr.port, true, false));
		}
		for (auto& procRecord : procArray) {
			// Only change the exclusion member, if not failed since it will require a reboot to revive it
			if (!procRecord->failed)
				procRecord->excluded = false;
			TraceEvent("RemoveAndKill")
			    .detail("Step", "IncludeAddress")
			    .detail("ProcessAddress", procRecord->address)
			    .detail("Process", describe(*procRecord))
			    .detail("Failed", procRecord->failed)
			    .detail("Rebooting", procRecord->rebooting)
			    .detail("ClusterAvailable", g_simulator->isAvailable());
		}
		return procArray;
	}

	// Return processes that are intersection of killAddrs and allServers and that are safe to kill together;
	// killAddrs does not guarantee the addresses are safe to kill simultaneously.
	std::vector<ISimulator::ProcessInfo*> protectServers(std::set<AddressExclusion> const& killAddrs) {
		std::vector<ISimulator::ProcessInfo*> processes;
		std::set<AddressExclusion> processAddrs;
		std::vector<AddressExclusion> killableAddrs;
		std::vector<ISimulator::ProcessInfo*> killProcArray, killableProcesses, processesLeft, processesDead;

		// Get the list of processes matching network address
		for (auto processInfo : getServers()) {
			auto processNet = AddressExclusion(processInfo->address.ip, processInfo->address.port);
			// Mark all of the unavailable as dead
			if (!processInfo->isAvailable() || processInfo->isCleared())
				processesDead.push_back(processInfo);
			// Save all processes not specified within set
			else if (killAddrs.find(processNet) == killAddrs.end())
				processesLeft.push_back(processInfo);
			else
				killProcArray.push_back(processInfo);
		}

		// Identify the largest set of processes which can be killed
		int randomIndex;
		bool bCanKillProcess;
		ISimulator::ProcessInfo* randomProcess;

		for (int killsLeft = killProcArray.size(); killsLeft > 0; killsLeft--) {
			// Select a random kill process
			randomIndex = deterministicRandom()->randomInt(0, killsLeft);
			randomProcess = killProcArray[randomIndex];
			processesDead.push_back(randomProcess);
			killProcArray[randomIndex] = killProcArray.back();
			killProcArray.pop_back();
			// Add all of the remaining processes the leftover array
			processesLeft.insert(processesLeft.end(), killProcArray.begin(), killProcArray.end());

			// Check if we can kill the added process
			bCanKillProcess = g_simulator->canKillProcesses(
			    processesLeft, processesDead, ISimulator::KillType::KillInstantly, nullptr);

			// Remove the added processes
			processesLeft.resize(processesLeft.size() - killProcArray.size());

			if (bCanKillProcess) {
				killableProcesses.push_back(randomProcess);
				killableAddrs.push_back(AddressExclusion(randomProcess->address.ip, randomProcess->address.port));
				TraceEvent("RemoveAndKill")
				    .detail("Step", "IdentifyVictim")
				    .detail("VictimCount", killableAddrs.size())
				    .detail("Victim", randomProcess->toString())
				    .detail("Victims", describe(killableAddrs));
			}
			// Move the process to the keep array
			else {
				processesLeft.push_back(randomProcess);
				processesDead.pop_back();
			}
		}

		return killableProcesses;
	}

	// toKill1 and toKill2 are two random subsets of all processes. If simply kill all processes in toKill1 or toKill2,
	// we may kill too many processes to make the cluster unavailable and stuck.
																#line 340 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
// This generated class is to be used only via workloadMain()
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
template <class WorkloadMainActor>
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
class WorkloadMainActorState {
															#line 346 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
public:
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	WorkloadMainActorState(RemoveServersSafelyWorkload* const& self,Database const& cx,double const& waitSeconds,std::set<AddressExclusion> const& toKill1,std::set<AddressExclusion> const& toKill2) 
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		 : self(self),
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   cx(cx),
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   waitSeconds(waitSeconds),
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   toKill1(toKill1),
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   toKill2(toKill2)
															#line 361 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
	{
		fdb_probe_actor_create("workloadMain", reinterpret_cast<unsigned long>(this));

	}
	~WorkloadMainActorState() 
	{
		fdb_probe_actor_destroy("workloadMain", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 343 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			StrictFuture<Void> __when_expr_0 = updateProcessIds(cx);
															#line 343 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 378 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<WorkloadMainActor*>(this)->actor_wait_state = 1;
															#line 343 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 0, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 383 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~WorkloadMainActorState();
		static_cast<WorkloadMainActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 344 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(waitSeconds);
															#line 344 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 408 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 2;
															#line 344 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 1, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 413 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 344 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_1 = delay(waitSeconds);
															#line 344 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 424 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 2;
															#line 344 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 1, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 429 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state > 0) static_cast<WorkloadMainActor*>(this)->actor_wait_state = 0;
		static_cast<WorkloadMainActor*>(this)->ActorCallback< WorkloadMainActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< WorkloadMainActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 347 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		firstCoordinators = std::vector<NetworkAddress>();
															#line 348 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		killProcArray = std::vector<ISimulator::ProcessInfo*>();
															#line 349 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		bClearedFirst = bool();
															#line 351 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "exclude list first") .detail("ToKill", describe(toKill1)) .detail("KillTotal", toKill1.size()) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 359 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		killProcArray = self->protectServers(toKill1);
															#line 361 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		toKill1 = self->getNetworks(killProcArray);
															#line 362 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "exclude list first") .detail("ToKillModified", describe(toKill1)) .detail("KillTotalModified", toKill1.size()) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 368 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		self->excludeAddresses(toKill1);
															#line 370 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Optional<Void>> __when_expr_2 = timeout(removeAndKill(self, cx, toKill1, nullptr, false), self->kill1Timeout);
															#line 370 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 531 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 3;
															#line 370 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 2, Optional<Void> >*>(static_cast<WorkloadMainActor*>(this)));
															#line 536 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 347 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		firstCoordinators = std::vector<NetworkAddress>();
															#line 348 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		killProcArray = std::vector<ISimulator::ProcessInfo*>();
															#line 349 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		bClearedFirst = bool();
															#line 351 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "exclude list first") .detail("ToKill", describe(toKill1)) .detail("KillTotal", toKill1.size()) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 359 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		killProcArray = self->protectServers(toKill1);
															#line 361 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		toKill1 = self->getNetworks(killProcArray);
															#line 362 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "exclude list first") .detail("ToKillModified", describe(toKill1)) .detail("KillTotalModified", toKill1.size()) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 368 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		self->excludeAddresses(toKill1);
															#line 370 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Optional<Void>> __when_expr_2 = timeout(removeAndKill(self, cx, toKill1, nullptr, false), self->kill1Timeout);
															#line 370 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 563 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 3;
															#line 370 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 2, Optional<Void> >*>(static_cast<WorkloadMainActor*>(this)));
															#line 568 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state > 0) static_cast<WorkloadMainActor*>(this)->actor_wait_state = 0;
		static_cast<WorkloadMainActor*>(this)->ActorCallback< WorkloadMainActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< WorkloadMainActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Optional<Void> const& result,int loopDepth) 
	{
															#line 372 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		bClearedFirst = result.present();
															#line 373 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "excluded list first") .detail("ExcludeResult", bClearedFirst ? "succeeded" : "failed") .detail("KillTotal", toKill1.size()) .detail("Processes", killProcArray.size()) .detail("ToKill1", describe(toKill1)) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 383 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		failed = true;
															#line 384 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (!bClearedFirst || BUGGIFY)
															#line 658 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 386 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			TraceEvent("RemoveAndKill") .detail("Step", "include all first") .detail("KillTotal", toKill1.size()) .detail("ToKill", describe(toKill1)) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 391 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			StrictFuture<Void> __when_expr_3 = includeServers(cx, std::vector<AddressExclusion>(1));
															#line 391 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 666 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<WorkloadMainActor*>(this)->actor_wait_state = 4;
															#line 391 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 3, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 671 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Optional<Void> && result,int loopDepth) 
	{
															#line 372 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		bClearedFirst = result.present();
															#line 373 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "excluded list first") .detail("ExcludeResult", bClearedFirst ? "succeeded" : "failed") .detail("KillTotal", toKill1.size()) .detail("Processes", killProcArray.size()) .detail("ToKill1", describe(toKill1)) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 383 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		failed = true;
															#line 384 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (!bClearedFirst || BUGGIFY)
															#line 691 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 386 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			TraceEvent("RemoveAndKill") .detail("Step", "include all first") .detail("KillTotal", toKill1.size()) .detail("ToKill", describe(toKill1)) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 391 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			StrictFuture<Void> __when_expr_3 = includeServers(cx, std::vector<AddressExclusion>(1));
															#line 391 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 699 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<WorkloadMainActor*>(this)->actor_wait_state = 4;
															#line 391 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 3, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 704 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Void> const& result,int loopDepth) 
	{
		loopDepth = a_body1cont3(result, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Optional<Void> && result,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state > 0) static_cast<WorkloadMainActor*>(this)->actor_wait_state = 0;
		static_cast<WorkloadMainActor*>(this)->ActorCallback< WorkloadMainActor, 2, Optional<Void> >::remove();

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 2, Optional<Void> >*,Optional<Void> const& value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 2, Optional<Void> >*,Optional<Void> && value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< WorkloadMainActor, 2, Optional<Void> >*,Error err) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 399 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		killProcArray = self->protectServers(toKill2);
															#line 402 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		toKill2 = self->getNetworks(killProcArray);
															#line 404 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "exclude list second") .detail("KillTotal", toKill2.size()) .detail("ToKill", describe(toKill2)) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 409 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		self->excludeAddresses(toKill2);
															#line 413 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "exclude second list") .detail("ToKill2", describe(toKill2)) .detail("KillTotal", toKill2.size()) .detail("Processes", killProcArray.size()) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 419 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_6 = reportErrors(timeoutError(removeAndKill(self, cx, toKill2, bClearedFirst ? &toKill1 : nullptr, true), self->kill2Timeout), "RemoveServersSafelyError", UID());
															#line 419 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 805 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), loopDepth); else return a_body1cont4when1(__when_expr_6.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 7;
															#line 419 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 6, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 810 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5(Void const& _,int loopDepth) 
	{
															#line 392 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_4 = includeLocalities(cx, std::vector<std::string>(), failed, true);
															#line 392 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 821 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 5;
															#line 392 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 4, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 826 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5(Void && _,int loopDepth) 
	{
															#line 392 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_4 = includeLocalities(cx, std::vector<std::string>(), failed, true);
															#line 392 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 837 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 5;
															#line 392 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 4, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 842 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state > 0) static_cast<WorkloadMainActor*>(this)->actor_wait_state = 0;
		static_cast<WorkloadMainActor*>(this)->ActorCallback< WorkloadMainActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< WorkloadMainActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont6(Void const& _,int loopDepth) 
	{
															#line 393 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_5 = includeLocalities(cx, std::vector<std::string>(), !failed, true);
															#line 393 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 928 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont6when1(__when_expr_5.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 6;
															#line 393 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 5, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 933 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont6(Void && _,int loopDepth) 
	{
															#line 393 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_5 = includeLocalities(cx, std::vector<std::string>(), !failed, true);
															#line 393 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 944 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), loopDepth); else return a_body1cont6when1(__when_expr_5.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 6;
															#line 393 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 5, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 949 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state > 0) static_cast<WorkloadMainActor*>(this)->actor_wait_state = 0;
		static_cast<WorkloadMainActor*>(this)->ActorCallback< WorkloadMainActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< WorkloadMainActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont7(Void const& _,int loopDepth) 
	{
															#line 394 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		self->includeAddresses(toKill1);
															#line 1033 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont7(Void && _,int loopDepth) 
	{
															#line 394 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		self->includeAddresses(toKill1);
															#line 1042 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state > 0) static_cast<WorkloadMainActor*>(this)->actor_wait_state = 0;
		static_cast<WorkloadMainActor*>(this)->ActorCallback< WorkloadMainActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< WorkloadMainActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
															#line 424 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "excluded second list") .detail("KillTotal", toKill2.size()) .detail("ToKill", describe(toKill2)) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 431 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "include all second") .detail("KillTotal", toKill2.size()) .detail("ToKill", describe(toKill2)) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 436 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_7 = includeServers(cx, std::vector<AddressExclusion>(1));
															#line 436 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1132 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont9when1(__when_expr_7.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 8;
															#line 436 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 7, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 1137 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
															#line 424 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "excluded second list") .detail("KillTotal", toKill2.size()) .detail("ToKill", describe(toKill2)) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 431 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "include all second") .detail("KillTotal", toKill2.size()) .detail("ToKill", describe(toKill2)) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 436 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_7 = includeServers(cx, std::vector<AddressExclusion>(1));
															#line 436 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1152 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont9when1(__when_expr_7.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 8;
															#line 436 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 7, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 1157 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state > 0) static_cast<WorkloadMainActor*>(this)->actor_wait_state = 0;
		static_cast<WorkloadMainActor*>(this)->ActorCallback< WorkloadMainActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< WorkloadMainActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont10(Void const& _,int loopDepth) 
	{
															#line 437 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_8 = includeLocalities(cx, std::vector<std::string>(), failed, true);
															#line 437 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1243 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont10when1(__when_expr_8.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 9;
															#line 437 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 8, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 1248 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont10(Void && _,int loopDepth) 
	{
															#line 437 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_8 = includeLocalities(cx, std::vector<std::string>(), failed, true);
															#line 437 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1259 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont10when1(__when_expr_8.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 9;
															#line 437 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 8, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 1264 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state > 0) static_cast<WorkloadMainActor*>(this)->actor_wait_state = 0;
		static_cast<WorkloadMainActor*>(this)->ActorCallback< WorkloadMainActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< WorkloadMainActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont10cont1(Void const& _,int loopDepth) 
	{
															#line 438 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_9 = includeLocalities(cx, std::vector<std::string>(), !failed, true);
															#line 438 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1350 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont10cont1when1(__when_expr_9.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 10;
															#line 438 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 9, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 1355 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont10cont1(Void && _,int loopDepth) 
	{
															#line 438 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_9 = includeLocalities(cx, std::vector<std::string>(), !failed, true);
															#line 438 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1366 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont10cont1when1(__when_expr_9.get(), loopDepth); };
		static_cast<WorkloadMainActor*>(this)->actor_wait_state = 10;
															#line 438 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< WorkloadMainActor, 9, Void >*>(static_cast<WorkloadMainActor*>(this)));
															#line 1371 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state > 0) static_cast<WorkloadMainActor*>(this)->actor_wait_state = 0;
		static_cast<WorkloadMainActor*>(this)->ActorCallback< WorkloadMainActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< WorkloadMainActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont10cont2(Void const& _,int loopDepth) 
	{
															#line 439 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		self->includeAddresses(toKill2);
															#line 441 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (!static_cast<WorkloadMainActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WorkloadMainActorState(); static_cast<WorkloadMainActor*>(this)->destroy(); return 0; }
															#line 1457 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		new (&static_cast<WorkloadMainActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WorkloadMainActorState();
		static_cast<WorkloadMainActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10cont2(Void && _,int loopDepth) 
	{
															#line 439 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		self->includeAddresses(toKill2);
															#line 441 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (!static_cast<WorkloadMainActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~WorkloadMainActorState(); static_cast<WorkloadMainActor*>(this)->destroy(); return 0; }
															#line 1471 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		new (&static_cast<WorkloadMainActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~WorkloadMainActorState();
		static_cast<WorkloadMainActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont10cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont10cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont10cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<WorkloadMainActor*>(this)->actor_wait_state > 0) static_cast<WorkloadMainActor*>(this)->actor_wait_state = 0;
		static_cast<WorkloadMainActor*>(this)->ActorCallback< WorkloadMainActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont10cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< WorkloadMainActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont10cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< WorkloadMainActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), 9);

	}
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	RemoveServersSafelyWorkload* self;
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	Database cx;
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	double waitSeconds;
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::set<AddressExclusion> toKill1;
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::set<AddressExclusion> toKill2;
															#line 347 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::vector<NetworkAddress> firstCoordinators;
															#line 348 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::vector<ISimulator::ProcessInfo*> killProcArray;
															#line 349 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	bool bClearedFirst;
															#line 383 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	bool failed;
															#line 1572 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
};
// This generated class is to be used only via workloadMain()
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
class WorkloadMainActor final : public Actor<Void>, public ActorCallback< WorkloadMainActor, 0, Void >, public ActorCallback< WorkloadMainActor, 1, Void >, public ActorCallback< WorkloadMainActor, 2, Optional<Void> >, public ActorCallback< WorkloadMainActor, 3, Void >, public ActorCallback< WorkloadMainActor, 4, Void >, public ActorCallback< WorkloadMainActor, 5, Void >, public ActorCallback< WorkloadMainActor, 6, Void >, public ActorCallback< WorkloadMainActor, 7, Void >, public ActorCallback< WorkloadMainActor, 8, Void >, public ActorCallback< WorkloadMainActor, 9, Void >, public FastAllocated<WorkloadMainActor>, public WorkloadMainActorState<WorkloadMainActor> {
															#line 1577 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
public:
	using FastAllocated<WorkloadMainActor>::operator new;
	using FastAllocated<WorkloadMainActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(340812552181161472UL, 15586297179150221568UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< WorkloadMainActor, 0, Void >;
friend struct ActorCallback< WorkloadMainActor, 1, Void >;
friend struct ActorCallback< WorkloadMainActor, 2, Optional<Void> >;
friend struct ActorCallback< WorkloadMainActor, 3, Void >;
friend struct ActorCallback< WorkloadMainActor, 4, Void >;
friend struct ActorCallback< WorkloadMainActor, 5, Void >;
friend struct ActorCallback< WorkloadMainActor, 6, Void >;
friend struct ActorCallback< WorkloadMainActor, 7, Void >;
friend struct ActorCallback< WorkloadMainActor, 8, Void >;
friend struct ActorCallback< WorkloadMainActor, 9, Void >;
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	WorkloadMainActor(RemoveServersSafelyWorkload* const& self,Database const& cx,double const& waitSeconds,std::set<AddressExclusion> const& toKill1,std::set<AddressExclusion> const& toKill2) 
															#line 1603 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		 : Actor<Void>(),
		   WorkloadMainActorState<WorkloadMainActor>(self, cx, waitSeconds, toKill1, toKill2),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("workloadMain", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6473147158665632000UL, 16130932942585895168UL);
		ActorExecutionContextHelper __helper(static_cast<WorkloadMainActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("workloadMain");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("workloadMain", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< WorkloadMainActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< WorkloadMainActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< WorkloadMainActor, 2, Optional<Void> >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< WorkloadMainActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< WorkloadMainActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< WorkloadMainActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< WorkloadMainActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< WorkloadMainActor, 7, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< WorkloadMainActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< WorkloadMainActor, 9, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
[[nodiscard]] static Future<Void> workloadMain( RemoveServersSafelyWorkload* const& self, Database const& cx, double const& waitSeconds, std::set<AddressExclusion> const& toKill1, std::set<AddressExclusion> const& toKill2 ) {
															#line 338 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	return Future<Void>(new WorkloadMainActor(self, cx, waitSeconds, toKill1, toKill2));
															#line 1644 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
}

#line 443 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"

	std::vector<ISimulator::ProcessInfo*> killAddresses(std::set<AddressExclusion> const& killAddrs) {
		UID functionId = nondeterministicRandom()->randomUniqueID();
		bool removeViaClear = !BUGGIFY;
		std::vector<ISimulator::ProcessInfo*> killProcArray;
		std::vector<AddressExclusion> toKillArray;

		std::copy(killAddrs.begin(), killAddrs.end(), std::back_inserter(toKillArray));
		killProcArray = getProcesses(killAddrs);

		// Reboot and delete or kill the servers
		if (killProcesses) {
			TraceEvent("RemoveAndKill", functionId)
			    .detail("Step", removeViaClear ? "ClearProcesses" : "IgnoreProcesses")
			    .detail("Addresses", describe(killAddrs))
			    .detail("Processes", killProcArray.size())
			    .detail("ClusterAvailable", g_simulator->isAvailable())
			    .detail("RemoveViaClear", removeViaClear);
			for (auto& killProcess : killProcArray) {
				if (g_simulator->protectedAddresses.contains(killProcess->address))
					TraceEvent("RemoveAndKill", functionId)
					    .detail("Step", "NoKill Process")
					    .detail("Process", describe(*killProcess))
					    .detail("Failed", killProcess->failed)
					    .detail("Rebooting", killProcess->rebooting)
					    .detail("ClusterAvailable", g_simulator->isAvailable())
					    .detail("Protected", g_simulator->protectedAddresses.contains(killProcess->address));
				else if (removeViaClear) {
					g_simulator->rebootProcess(killProcess, ISimulator::KillType::RebootProcessAndDelete);
					TraceEvent("RemoveAndKill", functionId)
					    .detail("Step", "Clear Process")
					    .detail("Process", describe(*killProcess))
					    .detail("Failed", killProcess->failed)
					    .detail("Rebooting", killProcess->rebooting)
					    .detail("ClusterAvailable", g_simulator->isAvailable())
					    .detail("Protected", g_simulator->protectedAddresses.contains(killProcess->address));
				}
				/*
				                else {
				                    g_simulator->killProcess( killProcess, ISimulator::KillType::KillInstantly );
				                    TraceEvent("RemoveAndKill", functionId).detail("Step", "Kill Process").detail("Process", describe(*killProcess)).detail("Failed", killProcess->failed).detail("Rebooting", killProcess->rebooting).detail("ClusterAvailable", g_simulator->isAvailable()).detail("Protected", g_simulator->protectedAddresses.contains(killProcess->address));
				                }
				*/
			}
		} else {
			std::set<Optional<Standalone<StringRef>>> zoneIds;
			bool killedMachine;
			for (auto& killProcess : killProcArray) {
				zoneIds.insert(killProcess->locality.zoneId());
			}
			TraceEvent("RemoveAndKill", functionId)
			    .detail("Step", removeViaClear ? "ClearMachines" : "KillMachines")
			    .detail("Addresses", describe(killAddrs))
			    .detail("Processes", killProcArray.size())
			    .detail("Zones", zoneIds.size())
			    .detail("ClusterAvailable", g_simulator->isAvailable());
			for (auto& zoneId : zoneIds) {
				killedMachine = g_simulator->killZone(zoneId,
				                                      removeViaClear ? ISimulator::KillType::RebootAndDelete
				                                                     : ISimulator::KillType::KillInstantly);
				TraceEvent(killedMachine ? SevInfo : SevWarn, "RemoveAndKill")
				    .detail("Step", removeViaClear ? "Clear Machine" : "Kill Machine")
				    .detail("ZoneId", zoneId)
				    .detail(removeViaClear ? "Cleared" : "Killed", killedMachine)
				    .detail("ClusterAvailable", g_simulator->isAvailable());
			}
		}

		return killProcArray;
	}

	// If a process is rebooted, it's processid will change. So we need to monitor
	// such changes and re-issue the locality-based exclusion again.
																#line 1721 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
// This generated class is to be used only via checkLocalityChange()
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
template <class CheckLocalityChangeActor>
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
class CheckLocalityChangeActorState {
															#line 1727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
public:
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	CheckLocalityChangeActorState(RemoveServersSafelyWorkload* const& self,Database const& cx,std::vector<AddressExclusion> const& toKillArray,std::unordered_set<std::string> const& origKillLocalities,bool const& markExcludeAsFailed) 
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		 : self(self),
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   cx(cx),
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   toKillArray(toKillArray),
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   origKillLocalities(origKillLocalities),
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   markExcludeAsFailed(markExcludeAsFailed),
															#line 521 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   killLocalities(origKillLocalities)
															#line 1744 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
	{
		fdb_probe_actor_create("checkLocalityChange", reinterpret_cast<unsigned long>(this));

	}
	~CheckLocalityChangeActorState() 
	{
		fdb_probe_actor_destroy("checkLocalityChange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 523 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			;
															#line 1759 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckLocalityChangeActorState();
		static_cast<CheckLocalityChangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 524 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(10.0);
															#line 524 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1791 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 1;
															#line 524 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckLocalityChangeActor, 0, Void >*>(static_cast<CheckLocalityChangeActor*>(this)));
															#line 1796 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 525 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->updateProcessIds(cx);
															#line 525 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1807 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 2;
															#line 525 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckLocalityChangeActor, 1, Void >*>(static_cast<CheckLocalityChangeActor*>(this)));
															#line 1812 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 525 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_1 = self->updateProcessIds(cx);
															#line 525 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1823 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 2;
															#line 525 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckLocalityChangeActor, 1, Void >*>(static_cast<CheckLocalityChangeActor*>(this)));
															#line 1828 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state > 0) static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 0;
		static_cast<CheckLocalityChangeActor*>(this)->ActorCallback< CheckLocalityChangeActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckLocalityChangeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckLocalityChangeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckLocalityChangeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 526 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		std::unordered_set<std::string> toKillLocalities = self->getLocalitiesFromAddresses(toKillArray);
															#line 527 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (toKillLocalities == killLocalities)
															#line 1914 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
			return a_body1loopHead1(loopDepth); // continue
		}
															#line 532 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "localities changed") .detail("OrigKillLocalities", describe(origKillLocalities)) .detail("KillLocalities", describe(killLocalities)) .detail("ToKillLocalities", describe(toKillLocalities)) .detail("Failed", markExcludeAsFailed);
															#line 538 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		killLocalities = toKillLocalities;
															#line 541 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		failed = true;
															#line 542 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_2 = includeLocalities(cx, std::vector<std::string>(), failed, true);
															#line 542 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1928 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 3;
															#line 542 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckLocalityChangeActor, 2, Void >*>(static_cast<CheckLocalityChangeActor*>(this)));
															#line 1933 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 526 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		std::unordered_set<std::string> toKillLocalities = self->getLocalitiesFromAddresses(toKillArray);
															#line 527 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (toKillLocalities == killLocalities)
															#line 1944 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
			return a_body1loopHead1(loopDepth); // continue
		}
															#line 532 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "localities changed") .detail("OrigKillLocalities", describe(origKillLocalities)) .detail("KillLocalities", describe(killLocalities)) .detail("ToKillLocalities", describe(toKillLocalities)) .detail("Failed", markExcludeAsFailed);
															#line 538 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		killLocalities = toKillLocalities;
															#line 541 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		failed = true;
															#line 542 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_2 = includeLocalities(cx, std::vector<std::string>(), failed, true);
															#line 542 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1958 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 3;
															#line 542 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckLocalityChangeActor, 2, Void >*>(static_cast<CheckLocalityChangeActor*>(this)));
															#line 1963 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state > 0) static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 0;
		static_cast<CheckLocalityChangeActor*>(this)->ActorCallback< CheckLocalityChangeActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckLocalityChangeActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckLocalityChangeActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckLocalityChangeActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 543 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_3 = includeLocalities(cx, std::vector<std::string>(), !failed, true);
															#line 543 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2049 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 4;
															#line 543 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CheckLocalityChangeActor, 3, Void >*>(static_cast<CheckLocalityChangeActor*>(this)));
															#line 2054 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 543 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_3 = includeLocalities(cx, std::vector<std::string>(), !failed, true);
															#line 543 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2065 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont3when1(__when_expr_3.get(), loopDepth); };
		static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 4;
															#line 543 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CheckLocalityChangeActor, 3, Void >*>(static_cast<CheckLocalityChangeActor*>(this)));
															#line 2070 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state > 0) static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 0;
		static_cast<CheckLocalityChangeActor*>(this)->ActorCallback< CheckLocalityChangeActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckLocalityChangeActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CheckLocalityChangeActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CheckLocalityChangeActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont5(Void const& _,int loopDepth) 
	{
															#line 546 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_4 = excludeLocalities(cx, killLocalities, markExcludeAsFailed);
															#line 546 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2156 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 5;
															#line 546 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< CheckLocalityChangeActor, 4, Void >*>(static_cast<CheckLocalityChangeActor*>(this)));
															#line 2161 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont5(Void && _,int loopDepth) 
	{
															#line 546 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_4 = excludeLocalities(cx, killLocalities, markExcludeAsFailed);
															#line 546 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2172 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont5when1(__when_expr_4.get(), loopDepth); };
		static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 5;
															#line 546 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< CheckLocalityChangeActor, 4, Void >*>(static_cast<CheckLocalityChangeActor*>(this)));
															#line 2177 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state > 0) static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 0;
		static_cast<CheckLocalityChangeActor*>(this)->ActorCallback< CheckLocalityChangeActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckLocalityChangeActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< CheckLocalityChangeActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< CheckLocalityChangeActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont6(Void const& _,int loopDepth) 
	{
															#line 547 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "new localities excluded") .detail("Localities", describe(killLocalities));
															#line 2261 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont6(Void && _,int loopDepth) 
	{
															#line 547 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill") .detail("Step", "new localities excluded") .detail("Localities", describe(killLocalities));
															#line 2270 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state > 0) static_cast<CheckLocalityChangeActor*>(this)->actor_wait_state = 0;
		static_cast<CheckLocalityChangeActor*>(this)->ActorCallback< CheckLocalityChangeActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckLocalityChangeActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< CheckLocalityChangeActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< CheckLocalityChangeActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), 4);

	}
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	RemoveServersSafelyWorkload* self;
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	Database cx;
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::vector<AddressExclusion> toKillArray;
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::unordered_set<std::string> origKillLocalities;
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	bool markExcludeAsFailed;
															#line 521 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::unordered_set<std::string> killLocalities;
															#line 541 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	bool failed;
															#line 2364 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
};
// This generated class is to be used only via checkLocalityChange()
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
class CheckLocalityChangeActor final : public Actor<Void>, public ActorCallback< CheckLocalityChangeActor, 0, Void >, public ActorCallback< CheckLocalityChangeActor, 1, Void >, public ActorCallback< CheckLocalityChangeActor, 2, Void >, public ActorCallback< CheckLocalityChangeActor, 3, Void >, public ActorCallback< CheckLocalityChangeActor, 4, Void >, public FastAllocated<CheckLocalityChangeActor>, public CheckLocalityChangeActorState<CheckLocalityChangeActor> {
															#line 2369 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
public:
	using FastAllocated<CheckLocalityChangeActor>::operator new;
	using FastAllocated<CheckLocalityChangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(9797446648136962048UL, 7656668727502152960UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckLocalityChangeActor, 0, Void >;
friend struct ActorCallback< CheckLocalityChangeActor, 1, Void >;
friend struct ActorCallback< CheckLocalityChangeActor, 2, Void >;
friend struct ActorCallback< CheckLocalityChangeActor, 3, Void >;
friend struct ActorCallback< CheckLocalityChangeActor, 4, Void >;
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	CheckLocalityChangeActor(RemoveServersSafelyWorkload* const& self,Database const& cx,std::vector<AddressExclusion> const& toKillArray,std::unordered_set<std::string> const& origKillLocalities,bool const& markExcludeAsFailed) 
															#line 2390 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		 : Actor<Void>(),
		   CheckLocalityChangeActorState<CheckLocalityChangeActor>(self, cx, toKillArray, origKillLocalities, markExcludeAsFailed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkLocalityChange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3418294750804998656UL, 18180721268295372800UL);
		ActorExecutionContextHelper __helper(static_cast<CheckLocalityChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkLocalityChange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkLocalityChange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckLocalityChangeActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckLocalityChangeActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CheckLocalityChangeActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< CheckLocalityChangeActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< CheckLocalityChangeActor, 4, Void >*)0, actor_cancelled()); break;
		}

	}
};
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
[[nodiscard]] static Future<Void> checkLocalityChange( RemoveServersSafelyWorkload* const& self, Database const& cx, std::vector<AddressExclusion> const& toKillArray, std::unordered_set<std::string> const& origKillLocalities, bool const& markExcludeAsFailed ) {
															#line 516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	return Future<Void>(new CheckLocalityChangeActor(self, cx, toKillArray, origKillLocalities, markExcludeAsFailed));
															#line 2426 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
}

#line 552 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"

	// Attempts to exclude a set of processes, and once the exclusion is successful it kills them.
	// If markExcludeAsFailed is true, then it is an error if we cannot complete the exclusion.
																#line 2433 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
// This generated class is to be used only via removeAndKill()
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
template <class RemoveAndKillActor>
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
class RemoveAndKillActorState {
															#line 2439 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
public:
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	RemoveAndKillActorState(RemoveServersSafelyWorkload* const& self,Database const& cx,std::set<AddressExclusion> const& toKill,std::set<AddressExclusion>* const& pIncAddrs,bool const& markExcludeAsFailed) 
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		 : self(self),
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   cx(cx),
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   toKill(toKill),
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   pIncAddrs(pIncAddrs),
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   markExcludeAsFailed(markExcludeAsFailed),
															#line 560 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		   functionId(nondeterministicRandom()->randomUniqueID())
															#line 2456 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
	{
		fdb_probe_actor_create("removeAndKill", reinterpret_cast<unsigned long>(this));

	}
	~RemoveAndKillActorState() 
	{
		fdb_probe_actor_destroy("removeAndKill", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 563 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			TraceEvent("RemoveAndKill", functionId) .detail("Step", "Including all") .detail("ClusterAvailable", g_simulator->isAvailable()) .detail("MarkExcludeAsFailed", markExcludeAsFailed);
															#line 567 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			failed = true;
															#line 568 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			StrictFuture<Void> __when_expr_0 = includeServers(cx, std::vector<AddressExclusion>(1));
															#line 568 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2477 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 1;
															#line 568 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 0, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 2482 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RemoveAndKillActorState();
		static_cast<RemoveAndKillActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 569 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_1 = includeLocalities(cx, std::vector<std::string>(), failed, true);
															#line 569 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2507 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 2;
															#line 569 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 1, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 2512 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 569 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_1 = includeLocalities(cx, std::vector<std::string>(), failed, true);
															#line 569 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2523 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 2;
															#line 569 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 1, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 2528 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(Void const& _,int loopDepth) 
	{
															#line 570 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_2 = includeLocalities(cx, std::vector<std::string>(), !failed, true);
															#line 570 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2614 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 3;
															#line 570 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 2, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 2619 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont2(Void && _,int loopDepth) 
	{
															#line 570 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_2 = includeLocalities(cx, std::vector<std::string>(), !failed, true);
															#line 570 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2630 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont2when1(__when_expr_2.get(), loopDepth); };
		static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 3;
															#line 570 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 2, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 2635 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont3(Void const& _,int loopDepth) 
	{
															#line 571 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill", functionId) .detail("Step", "Included all") .detail("ClusterAvailable", g_simulator->isAvailable()) .detail("MarkExcludeAsFailed", markExcludeAsFailed);
															#line 576 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (pIncAddrs)
															#line 2721 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 577 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			self->includeAddresses(*pIncAddrs);
															#line 2725 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		}
															#line 580 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		killProcArray = std::vector<ISimulator::ProcessInfo*>();
															#line 581 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		toKillArray = std::vector<AddressExclusion>();
															#line 582 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		toKillMarkFailedArray = std::vector<AddressExclusion>();
															#line 583 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		coordExcl = AddressExclusion();
															#line 585 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (BUGGIFY && toKill.size())
															#line 2737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 586 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			StrictFuture<Optional<ClusterConnectionString>> __when_expr_3 = getConnectionString(cx);
															#line 586 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2743 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 4;
															#line 586 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 3, Optional<ClusterConnectionString> >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 2748 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(Void && _,int loopDepth) 
	{
															#line 571 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill", functionId) .detail("Step", "Included all") .detail("ClusterAvailable", g_simulator->isAvailable()) .detail("MarkExcludeAsFailed", markExcludeAsFailed);
															#line 576 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (pIncAddrs)
															#line 2764 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 577 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			self->includeAddresses(*pIncAddrs);
															#line 2768 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		}
															#line 580 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		killProcArray = std::vector<ISimulator::ProcessInfo*>();
															#line 581 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		toKillArray = std::vector<AddressExclusion>();
															#line 582 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		toKillMarkFailedArray = std::vector<AddressExclusion>();
															#line 583 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		coordExcl = AddressExclusion();
															#line 585 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (BUGGIFY && toKill.size())
															#line 2780 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 586 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			StrictFuture<Optional<ClusterConnectionString>> __when_expr_3 = getConnectionString(cx);
															#line 586 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2786 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1cont3when1(__when_expr_3.get(), loopDepth); };
			static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 4;
															#line 586 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 3, Optional<ClusterConnectionString> >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 2791 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont4(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont4(int loopDepth) 
	{
															#line 600 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		std::copy(toKill.begin(), toKill.end(), std::back_inserter(toKillArray));
															#line 601 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (markExcludeAsFailed)
															#line 2882 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 602 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			retries = 0;
															#line 603 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			;
															#line 2888 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = a_body1cont4loopHead1(loopDepth);
		}
		else
		{
			loopDepth = a_body1cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Optional<ClusterConnectionString> const& csOptional,int loopDepth) 
	{
															#line 587 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		coordinators = std::vector<NetworkAddress>();
															#line 588 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (csOptional.present())
															#line 2904 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 589 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			ClusterConnectionString cs = csOptional.get();
															#line 590 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			StrictFuture<Void> __when_expr_4 = store(coordinators, cs.tryResolveHostnames());
															#line 590 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2912 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 5;
															#line 590 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 4, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 2917 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont6(Optional<ClusterConnectionString> && csOptional,int loopDepth) 
	{
															#line 587 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		coordinators = std::vector<NetworkAddress>();
															#line 588 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (csOptional.present())
															#line 2933 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 589 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			ClusterConnectionString cs = csOptional.get();
															#line 590 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			StrictFuture<Void> __when_expr_4 = store(coordinators, cs.tryResolveHostnames());
															#line 590 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 2941 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1Catch1(__when_expr_4.getError(), loopDepth); else return a_body1cont6when1(__when_expr_4.get(), loopDepth); };
			static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 5;
															#line 590 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 4, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 2946 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3when1(Optional<ClusterConnectionString> const& csOptional,int loopDepth) 
	{
		loopDepth = a_body1cont6(csOptional, loopDepth);

		return loopDepth;
	}
	int a_body1cont3when1(Optional<ClusterConnectionString> && csOptional,int loopDepth) 
	{
		loopDepth = a_body1cont6(std::move(csOptional), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 3, Optional<ClusterConnectionString> >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 3, Optional<ClusterConnectionString> >*,Optional<ClusterConnectionString> const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 3, Optional<ClusterConnectionString> >*,Optional<ClusterConnectionString> && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 3, Optional<ClusterConnectionString> >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 592 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (coordinators.size() > 2)
															#line 3035 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 593 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			auto randomCoordinator = deterministicRandom()->randomChoice(coordinators);
															#line 594 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			coordExcl = AddressExclusion(randomCoordinator.ip, randomCoordinator.port);
															#line 595 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			TraceEvent("RemoveAndKill", functionId) .detail("Step", "ChooseCoordinator") .detail("Coordinator", describe(coordExcl));
															#line 3043 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		}
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont6when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont11(int loopDepth) 
	{
															#line 643 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (markExcludeAsFailed && coordExcl.isValid())
															#line 3140 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 646 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (toKillMarkFailedArray.size() > toKillArray.size())
															#line 3144 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			{
															#line 647 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				auto removeServer = toKillMarkFailedArray.begin();
															#line 648 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				TraceEvent("RemoveAndKill", functionId) .detail("Step", "ShrinkFailedKillSet") .detail("Removing", removeServer->toString());
															#line 651 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				toKillMarkFailedArray.erase(removeServer);
															#line 3152 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			}
															#line 653 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			ASSERT(toKillMarkFailedArray.size() <= toKillArray.size());
															#line 654 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			std::sort(toKillArray.begin(), toKillArray.end());
															#line 655 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			auto removeServer = toKill.begin();
															#line 656 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			TraceEvent("RemoveAndKill", functionId) .detail("Step", "ReplaceNonFailedKillSet") .detail("Removing", removeServer->toString()) .detail("Adding", coordExcl.toString());
															#line 660 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			toKillArray.erase(std::remove(toKillArray.begin(), toKillArray.end(), *removeServer), toKillArray.end());
															#line 661 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			toKillArray.push_back(coordExcl);
															#line 662 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			toKill.erase(removeServer);
															#line 663 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			toKill.insert(coordExcl);
															#line 3170 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		}
															#line 665 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		killProcArray = self->getProcesses(toKill);
															#line 666 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill", functionId) .detail("Step", "Activate Server Exclusion") .detail("KillAddrs", toKill.size()) .detail("KillProcs", killProcArray.size()) .detail("MissingProcs", toKill.size() != killProcArray.size()) .detail("ToKill", describe(toKill)) .detail("Addresses", describe(toKillArray)) .detail("FailedAddresses", describe(toKillMarkFailedArray)) .detail("ClusterAvailable", g_simulator->isAvailable()) .detail("MarkExcludeAsFailed", markExcludeAsFailed);
															#line 677 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		excludeLocalitiesInsteadOfServers = deterministicRandom()->coinflip();
															#line 678 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		toKillLocalitiesFailed = std::unordered_set<std::string>();
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (markExcludeAsFailed)
															#line 3182 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 680 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			toKillLocalitiesFailed = self->getLocalitiesFromAddresses(toKillMarkFailedArray);
															#line 681 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (excludeLocalitiesInsteadOfServers && toKillLocalitiesFailed.size() > 0)
															#line 3188 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			{
															#line 682 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				TraceEvent("RemoveAndKill", functionId) .detail("Step", "Excluding localities with failed option") .detail("FailedAddressesSize", toKillMarkFailedArray.size()) .detail("FailedAddresses", describe(toKillMarkFailedArray)) .detail("FailedLocaitiesSize", toKillLocalitiesFailed.size()) .detail("FailedLocaities", describe(toKillLocalitiesFailed));
															#line 689 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				StrictFuture<Void> __when_expr_7 = excludeLocalities(cx, toKillLocalitiesFailed, true);
															#line 689 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3196 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), loopDepth); else return a_body1cont11when1(__when_expr_7.get(), loopDepth); };
				static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 7;
															#line 689 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 7, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 3201 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 691 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				TraceEvent("RemoveAndKill", functionId) .detail("Step", "Excluding servers with failed option") .detail("FailedAddressesSize", toKillMarkFailedArray.size()) .detail("FailedAddresses", describe(toKillMarkFailedArray));
															#line 696 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				StrictFuture<Void> __when_expr_8 = excludeServers(cx, toKillMarkFailedArray, true);
															#line 696 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3212 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1Catch1(__when_expr_8.getError(), loopDepth); else return a_body1cont11when2(__when_expr_8.get(), loopDepth); };
				static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 8;
															#line 696 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 8, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 3217 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1cont11cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont12(int loopDepth) 
	{
		loopDepth = a_body1cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont4loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1(int loopDepth) 
	{
															#line 604 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		safe = false;
															#line 605 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		failSet = random_subset(toKillArray, deterministicRandom()->randomInt(0, toKillArray.size() + 1));
															#line 607 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		toKillMarkFailedArray.resize(failSet.size());
															#line 608 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		std::copy(failSet.begin(), failSet.end(), toKillMarkFailedArray.begin());
															#line 609 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		std::sort(toKillMarkFailedArray.begin(), toKillMarkFailedArray.end());
															#line 610 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (coordExcl.isValid())
															#line 3255 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 611 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			toKillMarkFailedArray.push_back(coordExcl);
															#line 3259 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		}
															#line 613 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill", functionId) .detail("Step", "SafetyCheck") .detail("Exclusions", describe(toKillMarkFailedArray));
															#line 617 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<bool> __when_expr_5 = checkSafeExclusions(cx, toKillMarkFailedArray);
															#line 616 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3267 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1when1(__when_expr_5.get(), loopDepth); };
															#line 622 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(5.0);
															#line 3271 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont4loopBody1when2(__when_expr_6.get(), loopDepth); };
		static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 6;
															#line 617 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 5, bool >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 622 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 6, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 3278 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont4break1(int loopDepth) 
	{
		try {
			return a_body1cont12(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4loopBody1cont1(int loopDepth) 
	{
															#line 628 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (retries == self->maxSafetyCheckRetries)
															#line 3300 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 630 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			TraceEvent("RemoveAndKill", functionId) .detail("Step", "SafetyCheckLimitReached") .detail("Retries", retries);
															#line 633 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			markExcludeAsFailed = false;
															#line 634 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			safe = true;
															#line 3308 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		}
															#line 636 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (safe)
															#line 3312 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
			return a_body1cont4break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 638 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		retries++;
															#line 3318 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont4loopHead1(0);

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(bool const& _safe,int loopDepth) 
	{
															#line 618 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		safe = _safe && self->protectServers(std::set<AddressExclusion>(toKillMarkFailedArray.begin(), toKillMarkFailedArray.end())) .size() == toKillMarkFailedArray.size();
															#line 3327 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1when1(bool && _safe,int loopDepth) 
	{
															#line 618 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		safe = _safe && self->protectServers(std::set<AddressExclusion>(toKillMarkFailedArray.begin(), toKillMarkFailedArray.end())) .size() == toKillMarkFailedArray.size();
															#line 3336 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 623 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill", functionId) .detail("Step", "SafetyCheckTimedOut") .detail("Exclusions", describe(toKillMarkFailedArray));
															#line 3345 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont4loopBody1when2(Void && _,int loopDepth) 
	{
															#line 623 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill", functionId) .detail("Step", "SafetyCheckTimedOut") .detail("Exclusions", describe(toKillMarkFailedArray));
															#line 3354 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = a_body1cont4loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 5, bool >::remove();
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 5, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont4loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 5, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont4loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 5, bool >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont4loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont4loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont11cont1(int loopDepth) 
	{
															#line 700 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		toKillLocalities = self->getLocalitiesFromAddresses(toKillArray);
															#line 701 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (excludeLocalitiesInsteadOfServers && toKillLocalities.size() > 0)
															#line 3486 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 702 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			TraceEvent("RemoveAndKill", functionId) .detail("Step", "Excluding localities without failed option") .detail("AddressesSize", toKillArray.size()) .detail("Addresses", describe(toKillArray)) .detail("LocaitiesSize", toKillLocalities.size()) .detail("Locaities", describe(toKillLocalities));
															#line 709 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			StrictFuture<Void> __when_expr_9 = excludeLocalities(cx, toKillLocalities, false);
															#line 709 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3494 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1Catch1(__when_expr_9.getError(), loopDepth); else return a_body1cont11cont1when1(__when_expr_9.get(), loopDepth); };
			static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 9;
															#line 709 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 9, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 3499 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 711 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			TraceEvent("RemoveAndKill", functionId) .detail("Step", "Excluding servers without failed option") .detail("AddressesSize", toKillArray.size()) .detail("Addresses", describe(toKillArray));
															#line 716 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			StrictFuture<Void> __when_expr_10 = excludeServers(cx, toKillArray);
															#line 716 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3510 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1Catch1(__when_expr_10.getError(), loopDepth); else return a_body1cont11cont1when2(__when_expr_10.get(), loopDepth); };
			static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 10;
															#line 716 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 10, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 3515 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont11cont4(int loopDepth) 
	{
		loopDepth = a_body1cont11cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont5(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont5(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont5(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont5(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont11cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont11when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont11when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont11when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont11cont7(int loopDepth) 
	{
															#line 721 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (toKill.size())
															#line 3705 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
															#line 723 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			TraceEvent("RemoveAndKill", functionId) .detail("Step", "Wait For Server Exclusion") .detail("Addresses", describe(toKill)) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (excludeLocalitiesInsteadOfServers)
															#line 3711 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			{
															#line 728 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				StrictFuture<Void> __when_expr_11 = success(checkForExcludingServers(cx, toKillArray, true )) || checkLocalityChange(self, cx, toKillArray, toKillLocalities, markExcludeAsFailed && toKillLocalitiesFailed.size() > 0);
															#line 728 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3717 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1Catch1(__when_expr_11.getError(), loopDepth); else return a_body1cont11cont7when1(__when_expr_11.get(), loopDepth); };
				static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 11;
															#line 728 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 11, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 3722 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 735 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				StrictFuture<Void> __when_expr_12 = success(checkForExcludingServers(cx, toKillArray, true ));
															#line 735 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3731 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1Catch1(__when_expr_12.getError(), loopDepth); else return a_body1cont11cont7when2(__when_expr_12.get(), loopDepth); };
				static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 12;
															#line 735 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 12, Void >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 3736 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
															#line 766 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			TraceEvent("RemoveAndKill", functionId) .detail("Step", "nothing to clear") .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 3744 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = a_body1cont11cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont8(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont8(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont8(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont8(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 9, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 9, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont11cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 9, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont11cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 9, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont11cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont1when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 10, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 10, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont11cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 10, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont11cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 10, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont11cont10(int loopDepth) 
	{
															#line 771 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill", functionId) .detail("Step", "done") .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 775 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (!static_cast<RemoveAndKillActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~RemoveAndKillActorState(); static_cast<RemoveAndKillActor*>(this)->destroy(); return 0; }
															#line 3930 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		new (&static_cast<RemoveAndKillActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~RemoveAndKillActorState();
		static_cast<RemoveAndKillActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont11cont11(int loopDepth) 
	{
															#line 738 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent("RemoveAndKill", functionId) .detail("Step", "coordinators auto") .detail("DesiredCoordinators", g_simulator->desiredCoordinators) .detail("ClusterAvailable", g_simulator->isAvailable());
															#line 745 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		cycle = 0;
															#line 746 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		nQuorum = int();
															#line 747 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		;
															#line 3948 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = a_body1cont11cont11loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont12(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont12(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont7when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont12(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont7when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont12(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 11, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 11, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont11cont7when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 11, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont11cont7when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 11, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont11cont13(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont13(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont7when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont13(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont7when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont11cont13(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 12, Void >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 12, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont11cont7when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 12, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont11cont7when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 12, Void >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 12);

	}
	int a_body1cont11cont11cont1(int loopDepth) 
	{
															#line 764 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		self->killAddresses(toKill);
															#line 4131 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = a_body1cont11cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont11loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont11cont11loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont11loopBody1(int loopDepth) 
	{
															#line 748 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		cycle++;
															#line 749 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		nQuorum = ((g_simulator->desiredCoordinators + 1) / 2) * 2 - 1;
															#line 750 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		StrictFuture<CoordinatorsResult> __when_expr_13 = changeQuorum(cx, autoQuorumChange(nQuorum));
															#line 750 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4153 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		if (__when_expr_13.isReady()) { if (__when_expr_13.isError()) return a_body1Catch1(__when_expr_13.getError(), std::max(0, loopDepth - 1)); else return a_body1cont11cont11loopBody1when1(__when_expr_13.get(), loopDepth); };
		static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 13;
															#line 750 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		__when_expr_13.addCallbackAndClear(static_cast<ActorCallback< RemoveAndKillActor, 13, CoordinatorsResult >*>(static_cast<RemoveAndKillActor*>(this)));
															#line 4158 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont11cont11break1(int loopDepth) 
	{
		try {
			return a_body1cont11cont11cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont11cont11loopBody1cont1(CoordinatorsResult const& result,int loopDepth) 
	{
															#line 751 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent(result == CoordinatorsResult::SUCCESS || result == CoordinatorsResult::SAME_NETWORK_ADDRESSES ? SevInfo : SevWarn, "RemoveAndKillQuorumChangeResult") .detail("Step", "coordinators auto") .detail("Result", (int)result) .detail("Attempt", cycle) .detail("Quorum", nQuorum) .detail("DesiredCoordinators", g_simulator->desiredCoordinators);
															#line 760 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (result == CoordinatorsResult::SUCCESS || result == CoordinatorsResult::SAME_NETWORK_ADDRESSES)
															#line 4182 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
			return a_body1cont11cont11break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1cont11cont11loopHead1(0);

		return loopDepth;
	}
	int a_body1cont11cont11loopBody1cont1(CoordinatorsResult && result,int loopDepth) 
	{
															#line 751 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		TraceEvent(result == CoordinatorsResult::SUCCESS || result == CoordinatorsResult::SAME_NETWORK_ADDRESSES ? SevInfo : SevWarn, "RemoveAndKillQuorumChangeResult") .detail("Step", "coordinators auto") .detail("Result", (int)result) .detail("Attempt", cycle) .detail("Quorum", nQuorum) .detail("DesiredCoordinators", g_simulator->desiredCoordinators);
															#line 760 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (result == CoordinatorsResult::SUCCESS || result == CoordinatorsResult::SAME_NETWORK_ADDRESSES)
															#line 4196 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		{
			return a_body1cont11cont11break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1cont11cont11loopHead1(0);

		return loopDepth;
	}
	int a_body1cont11cont11loopBody1when1(CoordinatorsResult const& result,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11loopBody1cont1(result, loopDepth);

		return loopDepth;
	}
	int a_body1cont11cont11loopBody1when1(CoordinatorsResult && result,int loopDepth) 
	{
		loopDepth = a_body1cont11cont11loopBody1cont1(std::move(result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<RemoveAndKillActor*>(this)->actor_wait_state > 0) static_cast<RemoveAndKillActor*>(this)->actor_wait_state = 0;
		static_cast<RemoveAndKillActor*>(this)->ActorCallback< RemoveAndKillActor, 13, CoordinatorsResult >::remove();

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 13, CoordinatorsResult >*,CoordinatorsResult const& value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont11cont11loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_fire(ActorCallback< RemoveAndKillActor, 13, CoordinatorsResult >*,CoordinatorsResult && value) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont11cont11loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_error(ActorCallback< RemoveAndKillActor, 13, CoordinatorsResult >*,Error err) 
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), 13);

	}
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	RemoveServersSafelyWorkload* self;
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	Database cx;
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::set<AddressExclusion> toKill;
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::set<AddressExclusion>* pIncAddrs;
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	bool markExcludeAsFailed;
															#line 560 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	UID functionId;
															#line 567 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	bool failed;
															#line 580 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::vector<ISimulator::ProcessInfo*> killProcArray;
															#line 581 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::vector<AddressExclusion> toKillArray;
															#line 582 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::vector<AddressExclusion> toKillMarkFailedArray;
															#line 583 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	AddressExclusion coordExcl;
															#line 587 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::vector<NetworkAddress> coordinators;
															#line 602 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	int retries;
															#line 604 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	bool safe;
															#line 605 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::set<AddressExclusion> failSet;
															#line 677 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	bool excludeLocalitiesInsteadOfServers;
															#line 678 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::unordered_set<std::string> toKillLocalitiesFailed;
															#line 700 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	std::unordered_set<std::string> toKillLocalities;
															#line 745 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	int cycle;
															#line 746 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	int nQuorum;
															#line 4319 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
};
// This generated class is to be used only via removeAndKill()
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
class RemoveAndKillActor final : public Actor<Void>, public ActorCallback< RemoveAndKillActor, 0, Void >, public ActorCallback< RemoveAndKillActor, 1, Void >, public ActorCallback< RemoveAndKillActor, 2, Void >, public ActorCallback< RemoveAndKillActor, 3, Optional<ClusterConnectionString> >, public ActorCallback< RemoveAndKillActor, 4, Void >, public ActorCallback< RemoveAndKillActor, 5, bool >, public ActorCallback< RemoveAndKillActor, 6, Void >, public ActorCallback< RemoveAndKillActor, 7, Void >, public ActorCallback< RemoveAndKillActor, 8, Void >, public ActorCallback< RemoveAndKillActor, 9, Void >, public ActorCallback< RemoveAndKillActor, 10, Void >, public ActorCallback< RemoveAndKillActor, 11, Void >, public ActorCallback< RemoveAndKillActor, 12, Void >, public ActorCallback< RemoveAndKillActor, 13, CoordinatorsResult >, public FastAllocated<RemoveAndKillActor>, public RemoveAndKillActorState<RemoveAndKillActor> {
															#line 4324 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
public:
	using FastAllocated<RemoveAndKillActor>::operator new;
	using FastAllocated<RemoveAndKillActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15643826204307001344UL, 4094646074824603904UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RemoveAndKillActor, 0, Void >;
friend struct ActorCallback< RemoveAndKillActor, 1, Void >;
friend struct ActorCallback< RemoveAndKillActor, 2, Void >;
friend struct ActorCallback< RemoveAndKillActor, 3, Optional<ClusterConnectionString> >;
friend struct ActorCallback< RemoveAndKillActor, 4, Void >;
friend struct ActorCallback< RemoveAndKillActor, 5, bool >;
friend struct ActorCallback< RemoveAndKillActor, 6, Void >;
friend struct ActorCallback< RemoveAndKillActor, 7, Void >;
friend struct ActorCallback< RemoveAndKillActor, 8, Void >;
friend struct ActorCallback< RemoveAndKillActor, 9, Void >;
friend struct ActorCallback< RemoveAndKillActor, 10, Void >;
friend struct ActorCallback< RemoveAndKillActor, 11, Void >;
friend struct ActorCallback< RemoveAndKillActor, 12, Void >;
friend struct ActorCallback< RemoveAndKillActor, 13, CoordinatorsResult >;
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	RemoveAndKillActor(RemoveServersSafelyWorkload* const& self,Database const& cx,std::set<AddressExclusion> const& toKill,std::set<AddressExclusion>* const& pIncAddrs,bool const& markExcludeAsFailed) 
															#line 4354 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		 : Actor<Void>(),
		   RemoveAndKillActorState<RemoveAndKillActor>(self, cx, toKill, pIncAddrs, markExcludeAsFailed),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("removeAndKill", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9573575084131750400UL, 7240121160811142912UL);
		ActorExecutionContextHelper __helper(static_cast<RemoveAndKillActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("removeAndKill");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("removeAndKill", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RemoveAndKillActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< RemoveAndKillActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< RemoveAndKillActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< RemoveAndKillActor, 3, Optional<ClusterConnectionString> >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< RemoveAndKillActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< RemoveAndKillActor, 5, bool >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< RemoveAndKillActor, 7, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< RemoveAndKillActor, 8, Void >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< RemoveAndKillActor, 9, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< RemoveAndKillActor, 10, Void >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< RemoveAndKillActor, 11, Void >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< RemoveAndKillActor, 12, Void >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< RemoveAndKillActor, 13, CoordinatorsResult >*)0, actor_cancelled()); break;
		}

	}
};
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
[[nodiscard]] static Future<Void> removeAndKill( RemoveServersSafelyWorkload* const& self, Database const& cx, std::set<AddressExclusion> const& toKill, std::set<AddressExclusion>* const& pIncAddrs, bool const& markExcludeAsFailed ) {
															#line 555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	return Future<Void>(new RemoveAndKillActor(self, cx, toKill, pIncAddrs, markExcludeAsFailed));
															#line 4398 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
}

#line 777 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"

	static std::vector<ISimulator::ProcessInfo*> getServers() {
		std::vector<ISimulator::ProcessInfo*> machines;
		std::vector<ISimulator::ProcessInfo*> all = g_simulator->getAllProcesses();
		for (int i = 0; i < all.size(); i++) {
			if (all[i]->name == std::string("Server") && all[i]->isAvailableClass()) {
				machines.push_back(all[i]);
			}
		}
		return machines;
	}

	template <class T>
	static std::set<T> random_subset(std::vector<T> v, int n) {
		std::set<T> subset;
		// No, this isn't efficient!
		deterministicRandom()->randomShuffle(v);
		v.resize(n);
		std::copy(v.begin(), v.end(), std::inserter(subset, subset.end()));
		return subset;
	}

	bool killContainsProcess(AddressExclusion kill, NetworkAddress process) {
		return kill.excludes(process) || (machineProcesses.find(kill) != machineProcesses.end() &&
		                                  machineProcesses[kill].contains(AddressExclusion(process.ip, process.port)));
	}

	// Finds the localities list that can be excluded from the safe killable addresses list.
	// If excluding based on a particular locality of the safe process, kills any other process, that
	// particular locality is not included in the killable localities list.
	std::unordered_set<std::string> getLocalitiesFromAddresses(const std::vector<AddressExclusion>& addresses) {
		std::unordered_map<std::string, int> allLocalitiesCount;
		std::unordered_map<std::string, int> killableLocalitiesCount;
		auto processes = getServers();
		for (const auto& processInfo : processes) {
			std::map<std::string, std::string> localityData = processInfo->locality.getAllData();
			for (const auto& l : localityData) {
				allLocalitiesCount[LocalityData::ExcludeLocalityPrefix.toString() + l.first + ":" + l.second]++;
			}

			AddressExclusion pAddr(processInfo->address.ip, processInfo->address.port);
			if (std::find(addresses.begin(), addresses.end(), pAddr) != addresses.end()) {
				for (const auto& l : localityData) {
					killableLocalitiesCount[LocalityData::ExcludeLocalityPrefix.toString() + l.first + ":" +
					                        l.second]++;
				}
			}
		}

		std::unordered_set<std::string> toKillLocalities;
		for (const auto& l : killableLocalitiesCount) {
			if (l.second == allLocalitiesCount[l.first]) {
				toKillLocalities.insert(l.first);
			}
		}

		for (const auto& processInfo : processes) {
			AddressExclusion pAddr(processInfo->address.ip, processInfo->address.port);
			if (std::find(addresses.begin(), addresses.end(), pAddr) != addresses.end()) {
				std::map<std::string, std::string> localityData = processInfo->locality.getAllData();
				bool found = false;
				for (const auto& l : localityData) {
					if (toKillLocalities.contains(LocalityData::ExcludeLocalityPrefix.toString() + l.first + ":" +
					                              l.second)) {
						found = true;
						break;
					}
				}
				if (!found) {
					return std::unordered_set<std::string>();
				}
			}
		}

		return toKillLocalities;
	}

	// Update the g_simulator processes list with the process ids
	// of the workers, that are generated as part of worker creation.
																#line 4481 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
// This generated class is to be used only via updateProcessIds()
															#line 856 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
template <class UpdateProcessIdsActor>
															#line 856 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
class UpdateProcessIdsActorState {
															#line 4487 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
public:
															#line 856 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	UpdateProcessIdsActorState(Database const& cx) 
															#line 856 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
															#line 856 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		 : cx(cx)
															#line 4494 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
	{
		fdb_probe_actor_create("updateProcessIds", reinterpret_cast<unsigned long>(this));

	}
	~UpdateProcessIdsActorState() 
	{
		fdb_probe_actor_destroy("updateProcessIds", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 857 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			StrictFuture<std::vector<ProcessData>> __when_expr_0 = getWorkers(cx);
															#line 857 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (static_cast<UpdateProcessIdsActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 4511 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<UpdateProcessIdsActor*>(this)->actor_wait_state = 1;
															#line 857 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< UpdateProcessIdsActor, 0, std::vector<ProcessData> >*>(static_cast<UpdateProcessIdsActor*>(this)));
															#line 4516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~UpdateProcessIdsActorState();
		static_cast<UpdateProcessIdsActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<ProcessData> const& workers,int loopDepth) 
	{
															#line 858 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		std::unordered_map<NetworkAddress, int> addressToIndexMap;
															#line 859 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		for(int i = 0;i < workers.size();i++) {
															#line 860 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			addressToIndexMap[workers[i].address] = i;
															#line 4543 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		}
															#line 863 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		std::vector<ISimulator::ProcessInfo*> processes = g_simulator->getAllProcesses();
															#line 864 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		for( auto process : processes ) {
															#line 865 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (addressToIndexMap.find(process->address) != addressToIndexMap.end())
															#line 4551 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			{
															#line 866 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				if (workers[addressToIndexMap[process->address]].locality.processId().present())
															#line 4555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				{
															#line 867 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
					process->locality.set(LocalityData::keyProcessId, workers[addressToIndexMap[process->address]].locality.processId());
															#line 4559 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				}
			}
		}
															#line 873 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (!static_cast<UpdateProcessIdsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateProcessIdsActorState(); static_cast<UpdateProcessIdsActor*>(this)->destroy(); return 0; }
															#line 4565 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		new (&static_cast<UpdateProcessIdsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateProcessIdsActorState();
		static_cast<UpdateProcessIdsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(std::vector<ProcessData> && workers,int loopDepth) 
	{
															#line 858 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		std::unordered_map<NetworkAddress, int> addressToIndexMap;
															#line 859 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		for(int i = 0;i < workers.size();i++) {
															#line 860 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			addressToIndexMap[workers[i].address] = i;
															#line 4581 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		}
															#line 863 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		std::vector<ISimulator::ProcessInfo*> processes = g_simulator->getAllProcesses();
															#line 864 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		for( auto process : processes ) {
															#line 865 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
			if (addressToIndexMap.find(process->address) != addressToIndexMap.end())
															#line 4589 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
			{
															#line 866 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
				if (workers[addressToIndexMap[process->address]].locality.processId().present())
															#line 4593 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				{
															#line 867 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
					process->locality.set(LocalityData::keyProcessId, workers[addressToIndexMap[process->address]].locality.processId());
															#line 4597 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
				}
			}
		}
															#line 873 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
		if (!static_cast<UpdateProcessIdsActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~UpdateProcessIdsActorState(); static_cast<UpdateProcessIdsActor*>(this)->destroy(); return 0; }
															#line 4603 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		new (&static_cast<UpdateProcessIdsActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~UpdateProcessIdsActorState();
		static_cast<UpdateProcessIdsActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(std::vector<ProcessData> const& workers,int loopDepth) 
	{
		loopDepth = a_body1cont1(workers, loopDepth);

		return loopDepth;
	}
	int a_body1when1(std::vector<ProcessData> && workers,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(workers), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<UpdateProcessIdsActor*>(this)->actor_wait_state > 0) static_cast<UpdateProcessIdsActor*>(this)->actor_wait_state = 0;
		static_cast<UpdateProcessIdsActor*>(this)->ActorCallback< UpdateProcessIdsActor, 0, std::vector<ProcessData> >::remove();

	}
	void a_callback_fire(ActorCallback< UpdateProcessIdsActor, 0, std::vector<ProcessData> >*,std::vector<ProcessData> const& value) 
	{
		fdb_probe_actor_enter("updateProcessIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateProcessIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateProcessIds", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< UpdateProcessIdsActor, 0, std::vector<ProcessData> >*,std::vector<ProcessData> && value) 
	{
		fdb_probe_actor_enter("updateProcessIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateProcessIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateProcessIds", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< UpdateProcessIdsActor, 0, std::vector<ProcessData> >*,Error err) 
	{
		fdb_probe_actor_enter("updateProcessIds", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateProcessIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("updateProcessIds", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 856 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	Database cx;
															#line 4688 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
};
// This generated class is to be used only via updateProcessIds()
															#line 856 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
class UpdateProcessIdsActor final : public Actor<Void>, public ActorCallback< UpdateProcessIdsActor, 0, std::vector<ProcessData> >, public FastAllocated<UpdateProcessIdsActor>, public UpdateProcessIdsActorState<UpdateProcessIdsActor> {
															#line 4693 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
public:
	using FastAllocated<UpdateProcessIdsActor>::operator new;
	using FastAllocated<UpdateProcessIdsActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15036412967406513408UL, 13876525908954097664UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< UpdateProcessIdsActor, 0, std::vector<ProcessData> >;
															#line 856 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	UpdateProcessIdsActor(Database const& cx) 
															#line 4710 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
		 : Actor<Void>(),
		   UpdateProcessIdsActorState<UpdateProcessIdsActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("updateProcessIds", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5968864448681772800UL, 15343508188871220992UL);
		ActorExecutionContextHelper __helper(static_cast<UpdateProcessIdsActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("updateProcessIds");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("updateProcessIds", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< UpdateProcessIdsActor, 0, std::vector<ProcessData> >*)0, actor_cancelled()); break;
		}

	}
};
															#line 856 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
[[nodiscard]] static Future<Void> updateProcessIds( Database const& cx ) {
															#line 856 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
	return Future<Void>(new UpdateProcessIdsActor(cx));
															#line 4742 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbserver/workloads/RemoveServersSafely.actor.g.cpp"
}

#line 875 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbserver/workloads/RemoveServersSafely.actor.cpp"
};

WorkloadFactory<RemoveServersSafelyWorkload> RemoveServersSafelyWorkloadFactory;
