#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
/*
 * fdbcli.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "boost/lexical_cast.hpp"
#include "fmt/format.h"
#include "fdbclient/ClusterConnectionFile.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/FDBTypes.h"
#include "fdbclient/IClientApi.h"
#include "fdbclient/MultiVersionTransaction.h"
#include "fdbclient/Status.h"
#include "fdbclient/KeyBackedTypes.actor.h"
#include "fdbclient/StatusClient.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/GlobalConfig.actor.h"
#include "fdbclient/IKnobCollection.h"
#include "fdbclient/NativeAPI.actor.h"
#include "fdbclient/ClusterInterface.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/Schemas.h"
#include "fdbclient/CoordinationInterface.h"
#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/TagThrottle.actor.h"
#include "fdbclient/TenantManagement.actor.h"
#include "fdbclient/Tuple.h"

#include "fdbclient/ThreadSafeTransaction.h"
#include "flow/flow.h"
#include "flow/ApiVersion.h"
#include "flow/ArgParseUtil.h"
#include "flow/DeterministicRandom.h"
#include "flow/FastRef.h"
#include "flow/Platform.h"
#include "flow/SystemMonitor.h"
#include "flow/CodeProbe.h"

#include "flow/TLSConfig.actor.h"
#include "flow/ThreadHelper.actor.h"
#include "SimpleOpt/SimpleOpt.h"

#include "fdbcli/FlowLineNoise.h"
#include "fdbcli/fdbcli.actor.h"

#include <cinttypes>
#include <cstdio>
#include <type_traits>
#include <signal.h>

#ifdef __unixish__
#include <stdio.h>
#include "linenoise/linenoise.h"
#endif

#include "fdbclient/versions.h"
#include "fdbclient/BuildFlags.h"

#include "flow/actorcompiler.h" // This must be the last #include.

/*
 * While we could just use the MultiVersionApi instance directly, this #define allows us to swap in any other IClientApi
 * instance (e.g. from ThreadSafeApi)
 */
#define API ((IClientApi*)MultiVersionApi::api)

extern const char* getSourceVersion();

std::vector<std::string> validOptions;

enum {
	OPT_CONNFILE,
	OPT_DATABASE,
	OPT_HELP,
	OPT_TRACE,
	OPT_TRACE_DIR,
	OPT_LOGGROUP,
	OPT_TIMEOUT,
	OPT_EXEC,
	OPT_NO_STATUS,
	OPT_NO_HINTS,
	OPT_STATUS_FROM_JSON,
	OPT_VERSION,
	OPT_BUILD_FLAGS,
	OPT_TRACE_FORMAT,
	OPT_KNOB,
	OPT_DEBUG_TLS,
	OPT_API_VERSION,
	OPT_MEMORY,
	OPT_USE_FUTURE_PROTOCOL_VERSION
};

CSimpleOpt::SOption g_rgOptions[] = { { OPT_CONNFILE, "-C", SO_REQ_SEP },
	                                  { OPT_CONNFILE, "--cluster-file", SO_REQ_SEP },
	                                  { OPT_DATABASE, "-d", SO_REQ_SEP },
	                                  { OPT_TRACE, "--log", SO_NONE },
	                                  { OPT_TRACE_DIR, "--log-dir", SO_REQ_SEP },
	                                  { OPT_LOGGROUP, "--log-group", SO_REQ_SEP },
	                                  { OPT_TIMEOUT, "--timeout", SO_REQ_SEP },
	                                  { OPT_EXEC, "--exec", SO_REQ_SEP },
	                                  { OPT_NO_STATUS, "--no-status", SO_NONE },
	                                  { OPT_NO_HINTS, "--no-hints", SO_NONE },
	                                  { OPT_HELP, "-?", SO_NONE },
	                                  { OPT_HELP, "-h", SO_NONE },
	                                  { OPT_HELP, "--help", SO_NONE },
	                                  { OPT_STATUS_FROM_JSON, "--status-from-json", SO_REQ_SEP },
	                                  { OPT_VERSION, "--version", SO_NONE },
	                                  { OPT_VERSION, "-v", SO_NONE },
	                                  { OPT_BUILD_FLAGS, "--build-flags", SO_NONE },
	                                  { OPT_TRACE_FORMAT, "--trace-format", SO_REQ_SEP },
	                                  { OPT_KNOB, "--knob-", SO_REQ_SEP },
	                                  { OPT_DEBUG_TLS, "--debug-tls", SO_NONE },
	                                  { OPT_API_VERSION, "--api-version", SO_REQ_SEP },
	                                  { OPT_MEMORY, "--memory", SO_REQ_SEP },
	                                  { OPT_USE_FUTURE_PROTOCOL_VERSION, "--use-future-protocol-version", SO_NONE },
	                                  TLS_OPTION_FLAGS,
	                                  SO_END_OF_OPTIONS };

void printAtCol(const char* text, int col, FILE* stream = stdout) {
	const char* iter = text;
	const char* start = text;
	const char* space = nullptr;

	do {
		iter++;
		if (*iter == '\n' || *iter == ' ' || *iter == '\0')
			space = iter;
		if (*iter == '\n' || *iter == '\0' || (iter - start == col)) {
			if (!space)
				space = iter;
			fprintf(stream, "%.*s\n", (int)(space - start), start);
			start = space;
			if (*start == ' ' || *start == '\n')
				start++;
			space = nullptr;
		}
	} while (*iter);
}

class FdbOptions {
public:
	// Prints an error and throws invalid_option or invalid_option_value if the option could not be set
	void setOption(Reference<ITransaction> tr,
	               StringRef optionStr,
	               bool enabled,
	               Optional<StringRef> arg,
	               bool intrans) {
		auto transactionItr = transactionOptions.legalOptions.find(optionStr.toString());
		if (transactionItr != transactionOptions.legalOptions.end())
			setTransactionOption(tr, transactionItr->second, enabled, arg, intrans);
		else {
			fprintf(stderr,
			        "ERROR: invalid option '%s'. Try `help options' for a list of available options.\n",
			        optionStr.toString().c_str());
			throw invalid_option();
		}
	}

	// Applies all enabled transaction options to the given transaction
	void apply(Reference<ITransaction> tr) {
		for (const auto& [name, value] : transactionOptions.options) {
			tr->setOption(name, value.castTo<StringRef>());
		}
	}

	// Returns true if any options have been set
	bool hasAnyOptionsEnabled() const { return !transactionOptions.options.empty(); }

	// Prints a list of enabled options, along with their parameters (if any)
	void print() const {
		bool found = false;
		found = found || transactionOptions.print();

		if (!found)
			printf("There are no options enabled\n");
	}

	// Returns a vector of the names of all documented options
	std::vector<std::string> getValidOptions() const { return transactionOptions.getValidOptions(); }

	// Prints the help string obtained by invoking `help options'
	void printHelpString() const { transactionOptions.printHelpString(); }

private:
	// Sets a transaction option. If intrans == true, then this option is also applied to the passed in transaction.
	void setTransactionOption(Reference<ITransaction> tr,
	                          FDBTransactionOptions::Option option,
	                          bool enabled,
	                          Optional<StringRef> arg,
	                          bool intrans) {
		// If the parameter type is an int, we will extract into this variable and reference its memory with a StringRef
		int64_t parsedInt = 0;

		if (enabled) {
			auto optionInfo = FDBTransactionOptions::optionInfo.getMustExist(option);
			if (arg.present() != optionInfo.hasParameter) {
				fprintf(stderr, "ERROR: option %s a parameter\n", arg.present() ? "did not expect" : "expected");
				throw invalid_option_value();
			}
			if (arg.present() && optionInfo.paramType == FDBOptionInfo::ParamType::Int) {
				try {
					size_t nextIdx;
					std::string value = arg.get().toString();
					parsedInt = std::stoll(value, &nextIdx);
					if (nextIdx != value.length()) {
						fprintf(
						    stderr, "ERROR: could not parse value `%s' as an integer\n", arg.get().toString().c_str());
						throw invalid_option_value();
					}
					arg = StringRef(reinterpret_cast<uint8_t*>(&parsedInt), 8);
				} catch (std::exception e) {
					fprintf(stderr, "ERROR: could not parse value `%s' as an integer\n", arg.get().toString().c_str());
					throw invalid_option_value();
				}
			}
		}

		if (intrans) {
			tr->setOption(option, arg);
		}

		transactionOptions.setOption(option, enabled, arg.castTo<StringRef>());
	}

	// A group of enabled options (of type T::Option) as well as a legal options map from string to T::Option
	template <class T>
	struct OptionGroup {
		std::map<typename T::Option, Optional<Standalone<StringRef>>> options;
		std::map<std::string, typename T::Option> legalOptions;

		OptionGroup() {}
		OptionGroup(OptionGroup<T>& base)
		  : options(base.options.begin(), base.options.end()), legalOptions(base.legalOptions) {}

		// Enable or disable an option. Returns true if option value changed
		bool setOption(typename T::Option option, bool enabled, Optional<StringRef> arg) {
			auto optionItr = options.find(option);
			if (enabled && (optionItr == options.end() ||
			                Optional<Standalone<StringRef>>(optionItr->second).castTo<StringRef>() != arg)) {
				options[option] = arg.castTo<Standalone<StringRef>>();
				return true;
			} else if (!enabled && optionItr != options.end()) {
				options.erase(optionItr);
				return true;
			}

			return false;
		}

		// Prints a list of all enabled options in this group
		bool print() const {
			bool found = false;

			for (auto itr = legalOptions.begin(); itr != legalOptions.end(); ++itr) {
				auto optionItr = options.find(itr->second);
				if (optionItr != options.end()) {
					if (optionItr->second.present())
						printf("%s: `%s'\n", itr->first.c_str(), formatStringRef(optionItr->second.get()).c_str());
					else
						printf("%s\n", itr->first.c_str());

					found = true;
				}
			}

			return found;
		}

		// Returns true if the specified option is documented
		bool isDocumented(typename T::Option option) const {
			FDBOptionInfo info = T::optionInfo.getMustExist(option);

			std::string deprecatedStr = "Deprecated";
			return !info.comment.empty() && info.comment.substr(0, deprecatedStr.size()) != deprecatedStr;
		}

		// Returns a vector of the names of all documented options
		std::vector<std::string> getValidOptions() const {
			std::vector<std::string> ret;

			for (auto itr = legalOptions.begin(); itr != legalOptions.end(); ++itr)
				if (isDocumented(itr->second))
					ret.push_back(itr->first);

			return ret;
		}

		// Prints a help string for each option in this group. Any options with no comment
		// are excluded from this help string. Lines are wrapped to 80 characters.
		void printHelpString() const {
			for (auto itr = legalOptions.begin(); itr != legalOptions.end(); ++itr) {
				if (isDocumented(itr->second)) {
					FDBOptionInfo info = T::optionInfo.getMustExist(itr->second);
					std::string helpStr = info.name + " - " + info.comment;
					if (info.hasParameter)
						helpStr += " " + info.parameterComment;
					helpStr += "\n";

					printAtCol(helpStr.c_str(), 80);
				}
			}
		}
	};

	OptionGroup<FDBTransactionOptions> transactionOptions;

public:
	FdbOptions() {
		for (auto itr = FDBTransactionOptions::optionInfo.begin(); itr != FDBTransactionOptions::optionInfo.end();
		     ++itr)
			transactionOptions.legalOptions[itr->second.name] = itr->first;
	}

	FdbOptions(FdbOptions& base) : transactionOptions(base.transactionOptions) {}
};

static std::string formatStringRef(StringRef item, bool fullEscaping = false) {
	std::string ret;

	for (int i = 0; i < item.size(); i++) {
		if (fullEscaping && item[i] == '\\')
			ret += "\\\\";
		else if (fullEscaping && item[i] == '"')
			ret += "\\\"";
		else if (fullEscaping && item[i] == ' ')
			ret += format("\\x%02x", item[i]);
		else if (item[i] >= 32 && item[i] < 127)
			ret += item[i];
		else
			ret += format("\\x%02x", item[i]);
	}

	return ret;
}

static std::vector<std::vector<StringRef>> parseLine(std::string& line, bool& err, bool& partial) {
	err = false;
	partial = false;

	bool quoted = false;
	std::vector<StringRef> buf;
	std::vector<std::vector<StringRef>> ret;

	size_t i = line.find_first_not_of(' ');
	size_t offset = i;

	bool forcetoken = false;

	while (i <= line.length()) {
		switch (line[i]) {
		case ';':
			if (!quoted) {
				if (i > offset || (forcetoken && i == offset))
					buf.push_back(StringRef((uint8_t*)(line.data() + offset), i - offset));
				ret.push_back(std::move(buf));
				offset = i = line.find_first_not_of(' ', i + 1);
				forcetoken = false;
			} else
				i++;
			break;
		case '"':
			quoted = !quoted;
			line.erase(i, 1);
			forcetoken = true;
			break;
		case ' ':
		case '\n':
		case '\t':
		case '\r':
			if (!quoted) {
				if (i > offset || (forcetoken && i == offset))
					buf.push_back(StringRef((uint8_t*)(line.data() + offset), i - offset));
				offset = i = line.find_first_not_of(" \n\t\r", i);
				forcetoken = false;
			} else
				i++;
			break;
		case '\\':
			if (i + 2 > line.length()) {
				err = true;
				ret.push_back(std::move(buf));
				return ret;
			}
			switch (line[i + 1]) {
				char ent, save;
			case '"':
			case '\\':
			case ' ':
			case ';':
				line.erase(i, 1);
				break;
			// Handle \uNNNN utf-8 characters from JSON strings but only as a single byte
			// Return an error for a sequence out of range for a single byte
			case 'u': {
				if (i + 6 > line.length()) {
					err = true;
					ret.push_back(std::move(buf));
					return ret;
				}
				char* pEnd;
				save = line[i + 6];
				line[i + 6] = 0;
				unsigned long val = strtoul(line.data() + i + 2, &pEnd, 16);
				ent = char(val);
				if (*pEnd || val > std::numeric_limits<unsigned char>::max()) {
					err = true;
					ret.push_back(std::move(buf));
					return ret;
				}
				line[i + 6] = save;
				line.replace(i, 6, 1, ent);
				break;
			}
			case 'x':
				if (i + 4 > line.length()) {
					err = true;
					ret.push_back(std::move(buf));
					return ret;
				}
				char* pEnd;
				save = line[i + 4];
				line[i + 4] = 0;
				ent = char(strtoul(line.data() + i + 2, &pEnd, 16));
				if (*pEnd) {
					err = true;
					ret.push_back(std::move(buf));
					return ret;
				}
				line[i + 4] = save;
				line.replace(i, 4, 1, ent);
				break;
			default:
				err = true;
				ret.push_back(std::move(buf));
				return ret;
			}
		default:
			i++;
		}
	}

	i -= 1;
	if (i > offset || (forcetoken && i == offset))
		buf.push_back(StringRef((uint8_t*)(line.data() + offset), i - offset));

	ret.push_back(std::move(buf));

	if (quoted)
		partial = true;

	return ret;
}

static void printProgramUsage(const char* name) {
	printf("FoundationDB CLI " FDB_VT_PACKAGE_NAME " (v" FDB_VT_VERSION ")\n"
	       "usage: %s [OPTIONS]\n"
	       "\n",
	       name);
	printf("  -C CONNFILE    The path of a file containing the connection string for the\n"
	       "                 FoundationDB cluster. The default is first the value of the\n"
	       "                 FDB_CLUSTER_FILE environment variable, then `./fdb.cluster',\n"
	       "                 then `%s'.\n",
	       platform::getDefaultClusterFilePath().c_str());
	printf("  --log          Enables trace file logging for the CLI session.\n"
	       "  --log-dir PATH Specifies the output directory for trace files. If\n"
	       "                 unspecified, defaults to the current directory. Has\n"
	       "                 no effect unless --log is specified.\n"
	       "  --log-group LOG_GROUP\n"
	       "                 Sets the LogGroup field with the specified value for all\n"
	       "                 events in the trace output (defaults to `default').\n"
	       "  --trace-format FORMAT\n"
	       "                 Select the format of the log files. xml (the default) and json\n"
	       "                 are supported. Has no effect unless --log is specified.\n"
	       "  --exec CMDS    Immediately executes the semicolon separated CLI commands\n"
	       "                 and then exits.\n"
	       "  --no-status    Disables the initial status check done when starting\n"
	       "                 the CLI.\n"
	       "  --api-version  APIVERSION\n"
	       "                 Specifies the version of the API for the CLI to use.\n" TLS_HELP
	       "  --knob-KNOBNAME KNOBVALUE\n"
	       "                 Changes a knob option. KNOBNAME should be lowercase.\n"
	       "  --debug-tls    Prints the TLS configuration and certificate chain, then exits.\n"
	       "                 Useful in reporting and diagnosing TLS issues.\n"
	       "  --build-flags  Print build information and exit.\n"
	       "  --memory       Resident memory limit of the CLI (defaults to 8GiB).\n"
	       "  --use-future-protocol-version\n"
	       "                 Use the simulated future protocol version to connect to the cluster.\n"
	       "                 This option can be used testing purposes only!\n"
	       "  -v, --version  Print FoundationDB CLI version information and exit.\n"
	       "  -h, --help     Display this help and exit.\n");
}

#define ESCAPINGK "\n\nFor information on escaping keys, type `help escaping'."
#define ESCAPINGKV "\n\nFor information on escaping keys and values, type `help escaping'."

using namespace fdb_cli;
std::map<std::string, CommandHelp>& helpMap = CommandFactory::commands();
std::set<std::string>& hiddenCommands = CommandFactory::hiddenCommands();

void initHelp() {
	helpMap["begin"] =
	    CommandHelp("begin",
	                "begin a new transaction",
	                "By default, the fdbcli operates in autocommit mode. All operations are performed in their own "
	                "transaction, and are automatically committed for you. By explicitly beginning a transaction, "
	                "successive operations are all performed as part of a single transaction.\n\nTo commit the "
	                "transaction, use the commit command. To discard the transaction, use the reset command.");
	helpMap["commit"] = CommandHelp("commit [description]",
	                                "commit the current transaction",
	                                "Any sets or clears executed after the start of the current transaction will be "
	                                "committed to the database. On success, the committed version number is displayed. "
	                                "If commit fails, the error is displayed and the transaction must be retried. The "
	                                "command optionally allows for a description in case the transaction targets the "
	                                "configuration database. If no description is provided in the command, a prompt "
	                                "will be shown asking for a relevant description of the configuration change");
	helpMap["clear"] = CommandHelp(
	    "clear <KEY>",
	    "clear a key from the database",
	    "Clear succeeds even if the specified key is not present, but may fail because of conflicts." ESCAPINGK);
	helpMap["clearrange"] = CommandHelp(
	    "clearrange <BEGINKEY> <ENDKEY>",
	    "clear a range of keys from the database",
	    "All keys between BEGINKEY (inclusive) and ENDKEY (exclusive) are cleared from the database. This command will "
	    "succeed even if the specified range is empty, but may fail because of conflicts." ESCAPINGK);
	helpMap["exit"] = CommandHelp("exit", "exit the CLI", "");
	helpMap["quit"] = CommandHelp();
	helpMap["waitconnected"] = CommandHelp();
	helpMap["waitopen"] = CommandHelp();
	helpMap["sleep"] = CommandHelp("sleep <SECONDS>", "sleep for a period of time", "");
	helpMap["get"] =
	    CommandHelp("get <KEY>",
	                "fetch the value for a given key",
	                "Displays the value of KEY in the database, or `not found' if KEY is not present." ESCAPINGK);
	helpMap["getrange"] =
	    CommandHelp("getrange <BEGINKEY> [ENDKEY] [LIMIT]",
	                "fetch key/value pairs in a range of keys",
	                "Displays up to LIMIT keys and values for keys between BEGINKEY (inclusive) and ENDKEY "
	                "(exclusive). If ENDKEY is omitted, then the range will include all keys starting with BEGINKEY. "
	                "LIMIT defaults to 25 if omitted." ESCAPINGK);
	helpMap["getrangekeys"] = CommandHelp(
	    "getrangekeys <BEGINKEY> [ENDKEY] [LIMIT]",
	    "fetch keys in a range of keys",
	    "Displays up to LIMIT keys for keys between BEGINKEY (inclusive) and ENDKEY (exclusive). If ENDKEY is omitted, "
	    "then the range will include all keys starting with BEGINKEY. LIMIT defaults to 25 if omitted." ESCAPINGK);
	helpMap["getversion"] =
	    CommandHelp("getversion",
	                "Fetch the current read version",
	                "Displays the current read version of the database or currently running transaction.");
	helpMap["quota"] = CommandHelp("quota",
	                               "quota [get <tag> [reserved_throughput|total_throughput|storage] | "
	                               "set <tag> [reserved_throughput|total_throughput|storage] <value> | "
	                               "clear <tag>]",
	                               "Get, modify, or clear the reserved/total throughput quota (in bytes/s) or "
	                               "storage quota (in bytes) for the specified tag.");
	helpMap["reset"] =
	    CommandHelp("reset",
	                "reset the current transaction",
	                "Any sets or clears executed after the start of the active transaction will be discarded.");
	helpMap["rollback"] = CommandHelp("rollback",
	                                  "rolls back the current transaction",
	                                  "The active transaction will be discarded, including any sets or clears executed "
	                                  "since the transaction was started.");
	helpMap["set"] = CommandHelp("set <KEY> <VALUE>",
	                             "set a value for a given key",
	                             "If KEY is not already present in the database, it will be created." ESCAPINGKV);

	helpMap["setknob"] = CommandHelp("setknob <KEY> <VALUE> [CONFIG_CLASS]",
	                                 "updates a knob to specified value",
	                                 "setknob will prompt for a description of the changes" ESCAPINGKV);

	helpMap["getknob"] = CommandHelp(
	    "getknob <KEY> [CONFIG_CLASS]", "gets the value of the specified knob", "CONFIG_CLASS is optional." ESCAPINGK);

	helpMap["clearknob"] = CommandHelp("clearknob <KEY> [CONFIG_CLASS]",
	                                   "clears the value of the specified knob in the configuration database",
	                                   "CONFIG_CLASS is optional." ESCAPINGK);

	helpMap["option"] = CommandHelp(
	    "option <STATE> <OPTION> <ARG>",
	    "enables or disables an option",
	    "If STATE is `on', then the option OPTION will be enabled with optional parameter ARG, if required. If STATE "
	    "is `off', then OPTION will be disabled.\n\nIf there is no active transaction, then the option will be applied "
	    "to all operations as well as all subsequently created transactions (using `begin').\n\nIf there is an active "
	    "transaction (one created with `begin'), then enabled options apply only to that transaction. Options cannot "
	    "be disabled on an active transaction.\n\nCalling `option' with no parameters prints a list of all enabled "
	    "options.\n\nFor information about specific options that can be set, type `help options'.");
	helpMap["help"] = CommandHelp("help [<topic>]", "get help about a topic or command", "");
	helpMap["writemode"] = CommandHelp("writemode <on|off>",
	                                   "enables or disables sets and clears",
	                                   "Setting or clearing keys from the CLI is not recommended.");
	helpMap["usetenant"] =
	    CommandHelp("usetenant [NAME]",
	                "prints or configures the tenant used for transactions",
	                "If no name is given, prints the name of the current active tenant. Otherwise, all commands that "
	                "are used to read or write keys are done inside the tenant with the specified NAME. By default, "
	                "no tenant is configured and operations are performed on the raw key-space. The tenant cannot be "
	                "configured while a transaction started with `begin' is open.");
	helpMap["defaulttenant"] =
	    CommandHelp("defaulttenant",
	                "configures transactions to not use a named tenant",
	                "All commands that are used to read or write keys will be done without a tenant and will operate "
	                "on the raw key-space. This is the default behavior. The tenant cannot be configured while a "
	                "transaction started with `begin' is open.");
}

void printVersion() {
	printf("FoundationDB CLI " FDB_VT_PACKAGE_NAME " (v" FDB_VT_VERSION ")\n");
	printf("source version %s\n", getSourceVersion());
	printf("protocol %" PRIx64 "\n", currentProtocolVersion().version());
}

void printBuildInformation() {
	printf("%s", jsonBuildInformation().c_str());
}

void printHelpOverview() {
	printf("\nList of commands:\n\n");
	for (const auto& [command, help] : helpMap) {
		if (help.short_desc.size())
			printf(" %s:\n      %s\n", command.c_str(), help.short_desc.c_str());
	}
	printf("\nFor information on a specific command, type `help <command>'.");
	printf("\nFor information on escaping keys and values, type `help escaping'.");
	printf("\nFor information on available options, type `help options'.\n\n");
}

void printHelp(StringRef command) {
	auto i = helpMap.find(command.toString());
	if (i != helpMap.end() && i->second.short_desc.size()) {
		printf("\n%s\n\n", i->second.usage.c_str());
		auto cstr = i->second.short_desc.c_str();
		printf("%c%s.\n", toupper(cstr[0]), cstr + 1);
		if (!i->second.long_desc.empty()) {
			printf("\n");
			printAtCol(i->second.long_desc.c_str(), 80);
		}
		printf("\n");
	} else
		printf("I don't know anything about `%s'\n", formatStringRef(command).c_str());
}

int printStatusFromJSON(std::string const& jsonFileName) {
	try {
		json_spirit::mValue value;
		json_spirit::read_string(readFileBytes(jsonFileName, 10000000), value);

		printStatus(value.get_obj(), StatusClient::DETAILED, false, true);

		return 0;
	} catch (std::exception& e) {
		printf("Exception printing status: %s\n", e.what());
		return 1;
	} catch (Error& e) {
		printf("Error printing status: %d %s\n", e.code(), e.what());
		return 2;
	} catch (...) {
		printf("Unknown exception printing status.\n");
		return 3;
	}
}

															#line 681 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
namespace {
// This generated class is to be used only via timeWarning()
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class TimeWarningActor>
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class TimeWarningActorState {
															#line 688 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	TimeWarningActorState(double const& when,const char* const& msg) 
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		 : when(when),
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   msg(msg)
															#line 697 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
	{
		fdb_probe_actor_create("timeWarning", reinterpret_cast<unsigned long>(this));

	}
	~TimeWarningActorState() 
	{
		fdb_probe_actor_destroy("timeWarning", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 680 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(when);
															#line 680 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<TimeWarningActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 714 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TimeWarningActor*>(this)->actor_wait_state = 1;
															#line 680 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TimeWarningActor, 0, Void >*>(static_cast<TimeWarningActor*>(this)));
															#line 719 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TimeWarningActorState();
		static_cast<TimeWarningActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 681 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		fputs(msg, stderr);
															#line 683 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<TimeWarningActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TimeWarningActorState(); static_cast<TimeWarningActor*>(this)->destroy(); return 0; }
															#line 744 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<TimeWarningActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TimeWarningActorState();
		static_cast<TimeWarningActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 681 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		fputs(msg, stderr);
															#line 683 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<TimeWarningActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TimeWarningActorState(); static_cast<TimeWarningActor*>(this)->destroy(); return 0; }
															#line 758 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<TimeWarningActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TimeWarningActorState();
		static_cast<TimeWarningActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TimeWarningActor*>(this)->actor_wait_state > 0) static_cast<TimeWarningActor*>(this)->actor_wait_state = 0;
		static_cast<TimeWarningActor*>(this)->ActorCallback< TimeWarningActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TimeWarningActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("timeWarning", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TimeWarningActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeWarning", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TimeWarningActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("timeWarning", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TimeWarningActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeWarning", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TimeWarningActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("timeWarning", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TimeWarningActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeWarning", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	double when;
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	const char* msg;
															#line 845 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
};
// This generated class is to be used only via timeWarning()
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class TimeWarningActor final : public Actor<Void>, public ActorCallback< TimeWarningActor, 0, Void >, public FastAllocated<TimeWarningActor>, public TimeWarningActorState<TimeWarningActor> {
															#line 850 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
	using FastAllocated<TimeWarningActor>::operator new;
	using FastAllocated<TimeWarningActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13331156647914752768UL, 9283216457067156224UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TimeWarningActor, 0, Void >;
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	TimeWarningActor(double const& when,const char* const& msg) 
															#line 867 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		 : Actor<Void>(),
		   TimeWarningActorState<TimeWarningActor>(when, msg),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("timeWarning", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(1323264741314420480UL, 3686680307599240960UL);
		ActorExecutionContextHelper __helper(static_cast<TimeWarningActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("timeWarning");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("timeWarning", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TimeWarningActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
[[nodiscard]] Future<Void> timeWarning( double const& when, const char* const& msg ) {
															#line 679 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	return Future<Void>(new TimeWarningActor(when, msg));
															#line 900 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
}

#line 685 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"

															#line 905 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
namespace {
// This generated class is to be used only via checkStatus()
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class CheckStatusActor>
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class CheckStatusActorState {
															#line 912 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	CheckStatusActorState(Future<Void> const& f,Reference<IDatabase> const& db,Database const& localDb,bool const& displayDatabaseAvailable = true) 
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		 : f(f),
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   db(db),
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   localDb(localDb),
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   displayDatabaseAvailable(displayDatabaseAvailable)
															#line 925 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
	{
		fdb_probe_actor_create("checkStatus", reinterpret_cast<unsigned long>(this));

	}
	~CheckStatusActorState() 
	{
		fdb_probe_actor_destroy("checkStatus", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 690 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Void> __when_expr_0 = f;
															#line 690 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CheckStatusActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 942 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CheckStatusActor*>(this)->actor_wait_state = 1;
															#line 690 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CheckStatusActor, 0, Void >*>(static_cast<CheckStatusActor*>(this)));
															#line 947 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CheckStatusActorState();
		static_cast<CheckStatusActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 691 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		tr = db->createTransaction();
															#line 692 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		s = StatusObject();
															#line 693 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!tr->isValid())
															#line 974 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 694 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<StatusObject> __when_expr_1 = StatusClient::statusFetcher(localDb);
															#line 694 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CheckStatusActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 980 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CheckStatusActor*>(this)->actor_wait_state = 2;
															#line 694 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckStatusActor, 1, StatusObject >*>(static_cast<CheckStatusActor*>(this)));
															#line 985 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 697 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			statusValueF = tr->get("\xff\xff/status/json"_sr);
															#line 698 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_2 = safeThreadFutureToFuture(statusValueF);
															#line 698 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CheckStatusActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 996 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when2(__when_expr_2.get(), loopDepth); };
			static_cast<CheckStatusActor*>(this)->actor_wait_state = 3;
															#line 698 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckStatusActor, 2, Optional<Value> >*>(static_cast<CheckStatusActor*>(this)));
															#line 1001 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 691 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		tr = db->createTransaction();
															#line 692 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		s = StatusObject();
															#line 693 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!tr->isValid())
															#line 1015 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 694 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<StatusObject> __when_expr_1 = StatusClient::statusFetcher(localDb);
															#line 694 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CheckStatusActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1021 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1cont1when1(__when_expr_1.get(), loopDepth); };
			static_cast<CheckStatusActor*>(this)->actor_wait_state = 2;
															#line 694 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CheckStatusActor, 1, StatusObject >*>(static_cast<CheckStatusActor*>(this)));
															#line 1026 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 697 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			statusValueF = tr->get("\xff\xff/status/json"_sr);
															#line 698 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_2 = safeThreadFutureToFuture(statusValueF);
															#line 698 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CheckStatusActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1037 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont1when2(__when_expr_2.get(), loopDepth); };
			static_cast<CheckStatusActor*>(this)->actor_wait_state = 3;
															#line 698 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CheckStatusActor, 2, Optional<Value> >*>(static_cast<CheckStatusActor*>(this)));
															#line 1042 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CheckStatusActor*>(this)->actor_wait_state > 0) static_cast<CheckStatusActor*>(this)->actor_wait_state = 0;
		static_cast<CheckStatusActor*>(this)->ActorCallback< CheckStatusActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CheckStatusActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("checkStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CheckStatusActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("checkStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CheckStatusActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("checkStatus", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkStatus", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont2(int loopDepth) 
	{
															#line 707 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		printf("\n");
															#line 708 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		printStatus(s, StatusClient::MINIMAL, displayDatabaseAvailable);
															#line 709 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		printf("\n");
															#line 710 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<CheckStatusActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckStatusActorState(); static_cast<CheckStatusActor*>(this)->destroy(); return 0; }
															#line 1133 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<CheckStatusActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CheckStatusActorState();
		static_cast<CheckStatusActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont3(StatusObject const& _s,int loopDepth) 
	{
															#line 695 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		s = _s;
															#line 1145 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont3(StatusObject && _s,int loopDepth) 
	{
															#line 695 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		s = _s;
															#line 1154 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(StatusObject const& _s,int loopDepth) 
	{
		loopDepth = a_body1cont3(_s, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(StatusObject && _s,int loopDepth) 
	{
		loopDepth = a_body1cont3(std::move(_s), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CheckStatusActor*>(this)->actor_wait_state > 0) static_cast<CheckStatusActor*>(this)->actor_wait_state = 0;
		static_cast<CheckStatusActor*>(this)->ActorCallback< CheckStatusActor, 1, StatusObject >::remove();

	}
	void a_callback_fire(ActorCallback< CheckStatusActor, 1, StatusObject >*,StatusObject const& value) 
	{
		fdb_probe_actor_enter("checkStatus", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkStatus", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CheckStatusActor, 1, StatusObject >*,StatusObject && value) 
	{
		fdb_probe_actor_enter("checkStatus", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkStatus", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CheckStatusActor, 1, StatusObject >*,Error err) 
	{
		fdb_probe_actor_enter("checkStatus", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkStatus", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont5(Optional<Value> const& statusValue,int loopDepth) 
	{
															#line 699 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!statusValue.present())
															#line 1238 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 700 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fprintf(stderr, "ERROR: Failed to get status json from the cluster\n");
															#line 701 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (!static_cast<CheckStatusActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckStatusActorState(); static_cast<CheckStatusActor*>(this)->destroy(); return 0; }
															#line 1244 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			new (&static_cast<CheckStatusActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~CheckStatusActorState();
			static_cast<CheckStatusActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 703 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		json_spirit::mValue mv;
															#line 704 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		json_spirit::read_string(statusValue.get().toString(), mv);
															#line 705 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		s = StatusObject(mv.get_obj());
															#line 1256 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont5(Optional<Value> && statusValue,int loopDepth) 
	{
															#line 699 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!statusValue.present())
															#line 1265 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 700 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fprintf(stderr, "ERROR: Failed to get status json from the cluster\n");
															#line 701 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (!static_cast<CheckStatusActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CheckStatusActorState(); static_cast<CheckStatusActor*>(this)->destroy(); return 0; }
															#line 1271 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			new (&static_cast<CheckStatusActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~CheckStatusActorState();
			static_cast<CheckStatusActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 703 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		json_spirit::mValue mv;
															#line 704 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		json_spirit::read_string(statusValue.get().toString(), mv);
															#line 705 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		s = StatusObject(mv.get_obj());
															#line 1283 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(Optional<Value> const& statusValue,int loopDepth) 
	{
		loopDepth = a_body1cont5(statusValue, loopDepth);

		return loopDepth;
	}
	int a_body1cont1when2(Optional<Value> && statusValue,int loopDepth) 
	{
		loopDepth = a_body1cont5(std::move(statusValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CheckStatusActor*>(this)->actor_wait_state > 0) static_cast<CheckStatusActor*>(this)->actor_wait_state = 0;
		static_cast<CheckStatusActor*>(this)->ActorCallback< CheckStatusActor, 2, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< CheckStatusActor, 2, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("checkStatus", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkStatus", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CheckStatusActor, 2, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("checkStatus", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CheckStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkStatus", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CheckStatusActor, 2, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("checkStatus", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CheckStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("checkStatus", reinterpret_cast<unsigned long>(this), 2);

	}
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Future<Void> f;
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Reference<IDatabase> db;
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Database localDb;
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	bool displayDatabaseAvailable;
															#line 691 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Reference<ITransaction> tr;
															#line 692 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	StatusObject s;
															#line 697 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	ThreadFuture<Optional<Value>> statusValueF;
															#line 1377 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
};
// This generated class is to be used only via checkStatus()
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class CheckStatusActor final : public Actor<Void>, public ActorCallback< CheckStatusActor, 0, Void >, public ActorCallback< CheckStatusActor, 1, StatusObject >, public ActorCallback< CheckStatusActor, 2, Optional<Value> >, public FastAllocated<CheckStatusActor>, public CheckStatusActorState<CheckStatusActor> {
															#line 1382 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
	using FastAllocated<CheckStatusActor>::operator new;
	using FastAllocated<CheckStatusActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10945936342331167232UL, 16124623372022266880UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CheckStatusActor, 0, Void >;
friend struct ActorCallback< CheckStatusActor, 1, StatusObject >;
friend struct ActorCallback< CheckStatusActor, 2, Optional<Value> >;
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	CheckStatusActor(Future<Void> const& f,Reference<IDatabase> const& db,Database const& localDb,bool const& displayDatabaseAvailable = true) 
															#line 1401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		 : Actor<Void>(),
		   CheckStatusActorState<CheckStatusActor>(f, db, localDb, displayDatabaseAvailable),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("checkStatus", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6563480658186616320UL, 16474974964592628736UL);
		ActorExecutionContextHelper __helper(static_cast<CheckStatusActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("checkStatus");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("checkStatus", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CheckStatusActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CheckStatusActor, 1, StatusObject >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CheckStatusActor, 2, Optional<Value> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
[[nodiscard]] Future<Void> checkStatus( Future<Void> const& f, Reference<IDatabase> const& db, Database const& localDb, bool const& displayDatabaseAvailable = true ) {
															#line 686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	return Future<Void>(new CheckStatusActor(f, db, localDb, displayDatabaseAvailable));
															#line 1436 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
}

#line 712 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"

															#line 1441 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
namespace {
// This generated class is to be used only via makeInterruptable()
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class T, class MakeInterruptableActor>
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class MakeInterruptableActorState {
															#line 1448 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	MakeInterruptableActorState(Future<T> const& f) 
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		 : f(f)
															#line 1455 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
	{
		fdb_probe_actor_create("makeInterruptable", reinterpret_cast<unsigned long>(this));

	}
	~MakeInterruptableActorState() 
	{
		fdb_probe_actor_destroy("makeInterruptable", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 715 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			Future<Void> interrupt = LineNoise::onKeyboardInterrupt();
															#line 717 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<T> __when_expr_0 = f;
															#line 716 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<MakeInterruptableActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1474 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
															#line 720 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Void> __when_expr_1 = interrupt;
															#line 1478 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1when2(__when_expr_1.get(), loopDepth); };
			static_cast<MakeInterruptableActor*>(this)->actor_wait_state = 1;
															#line 717 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MakeInterruptableActor, 0, T >*>(static_cast<MakeInterruptableActor*>(this)));
															#line 720 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MakeInterruptableActor, 1, Void >*>(static_cast<MakeInterruptableActor*>(this)));
															#line 1485 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MakeInterruptableActorState();
		static_cast<MakeInterruptableActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1when1(T const& t,int loopDepth) 
	{
															#line 718 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<MakeInterruptableActor*>(this)->SAV<T>::futures) { (void)(t); this->~MakeInterruptableActorState(); static_cast<MakeInterruptableActor*>(this)->destroy(); return 0; }
															#line 1508 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<MakeInterruptableActor*>(this)->SAV< T >::value()) T(t);
		this->~MakeInterruptableActorState();
		static_cast<MakeInterruptableActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(T && t,int loopDepth) 
	{
															#line 718 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<MakeInterruptableActor*>(this)->SAV<T>::futures) { (void)(t); this->~MakeInterruptableActorState(); static_cast<MakeInterruptableActor*>(this)->destroy(); return 0; }
															#line 1520 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<MakeInterruptableActor*>(this)->SAV< T >::value()) T(t);
		this->~MakeInterruptableActorState();
		static_cast<MakeInterruptableActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when2(Void const& _,int loopDepth) 
	{
															#line 721 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		f.cancel();
															#line 722 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		return a_body1Catch1(operation_cancelled(), loopDepth);
															#line 1534 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"

		return loopDepth;
	}
	int a_body1when2(Void && _,int loopDepth) 
	{
															#line 721 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		f.cancel();
															#line 722 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		return a_body1Catch1(operation_cancelled(), loopDepth);
															#line 1544 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MakeInterruptableActor*>(this)->actor_wait_state > 0) static_cast<MakeInterruptableActor*>(this)->actor_wait_state = 0;
		static_cast<MakeInterruptableActor*>(this)->ActorCallback< MakeInterruptableActor, 0, T >::remove();
		static_cast<MakeInterruptableActor*>(this)->ActorCallback< MakeInterruptableActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MakeInterruptableActor, 0, T >*,T const& value) 
	{
		fdb_probe_actor_enter("makeInterruptable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakeInterruptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makeInterruptable", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MakeInterruptableActor, 0, T >*,T && value) 
	{
		fdb_probe_actor_enter("makeInterruptable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakeInterruptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makeInterruptable", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MakeInterruptableActor, 0, T >*,Error err) 
	{
		fdb_probe_actor_enter("makeInterruptable", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakeInterruptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makeInterruptable", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MakeInterruptableActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("makeInterruptable", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakeInterruptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makeInterruptable", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MakeInterruptableActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("makeInterruptable", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MakeInterruptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makeInterruptable", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MakeInterruptableActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("makeInterruptable", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MakeInterruptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("makeInterruptable", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Future<T> f;
															#line 1671 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
};
// This generated class is to be used only via makeInterruptable()
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class T>
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class MakeInterruptableActor final : public Actor<T>, public ActorCallback< MakeInterruptableActor<T>, 0, T >, public ActorCallback< MakeInterruptableActor<T>, 1, Void >, public FastAllocated<MakeInterruptableActor<T>>, public MakeInterruptableActorState<T, MakeInterruptableActor<T>> {
															#line 1678 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
	using FastAllocated<MakeInterruptableActor<T>>::operator new;
	using FastAllocated<MakeInterruptableActor<T>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6204446212372090112UL, 13109546991232992768UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<T>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MakeInterruptableActor<T>, 0, T >;
friend struct ActorCallback< MakeInterruptableActor<T>, 1, Void >;
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	MakeInterruptableActor(Future<T> const& f) 
															#line 1696 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		 : Actor<T>(),
		   MakeInterruptableActorState<T, MakeInterruptableActor<T>>(f),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("makeInterruptable", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13224471411932523264UL, 12653261512788980736UL);
		ActorExecutionContextHelper __helper(static_cast<MakeInterruptableActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("makeInterruptable");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("makeInterruptable", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MakeInterruptableActor<T>, 0, T >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class T>
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
[[nodiscard]] Future<T> makeInterruptable( Future<T> const& f ) {
															#line 713 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	return Future<T>(new MakeInterruptableActor<T>(f));
															#line 1731 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
}

#line 726 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"

															#line 1736 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
namespace {
// This generated class is to be used only via commitTransaction()
															#line 727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class CommitTransactionActor>
															#line 727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class CommitTransactionActorState {
															#line 1743 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
															#line 727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	CommitTransactionActorState(Reference<ITransaction> const& tr) 
															#line 727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
															#line 727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		 : tr(tr)
															#line 1750 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
	{
		fdb_probe_actor_create("commitTransaction", reinterpret_cast<unsigned long>(this));

	}
	~CommitTransactionActorState() 
	{
		fdb_probe_actor_destroy("commitTransaction", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 728 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Void> __when_expr_0 = makeInterruptable(safeThreadFutureToFuture(tr->commit()));
															#line 728 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CommitTransactionActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 1767 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CommitTransactionActor*>(this)->actor_wait_state = 1;
															#line 728 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CommitTransactionActor, 0, Void >*>(static_cast<CommitTransactionActor*>(this)));
															#line 1772 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CommitTransactionActorState();
		static_cast<CommitTransactionActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 729 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		auto ver = tr->getCommittedVersion();
															#line 730 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (ver != invalidVersion)
															#line 1797 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 731 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fmt::print("Committed ({})\n", ver);
															#line 1801 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 733 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fmt::print("Nothing to commit\n");
															#line 1807 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
															#line 734 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<CommitTransactionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CommitTransactionActorState(); static_cast<CommitTransactionActor*>(this)->destroy(); return 0; }
															#line 1811 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<CommitTransactionActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CommitTransactionActorState();
		static_cast<CommitTransactionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 729 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		auto ver = tr->getCommittedVersion();
															#line 730 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (ver != invalidVersion)
															#line 1825 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 731 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fmt::print("Committed ({})\n", ver);
															#line 1829 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 733 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fmt::print("Nothing to commit\n");
															#line 1835 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
															#line 734 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<CommitTransactionActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~CommitTransactionActorState(); static_cast<CommitTransactionActor*>(this)->destroy(); return 0; }
															#line 1839 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<CommitTransactionActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~CommitTransactionActorState();
		static_cast<CommitTransactionActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CommitTransactionActor*>(this)->actor_wait_state > 0) static_cast<CommitTransactionActor*>(this)->actor_wait_state = 0;
		static_cast<CommitTransactionActor*>(this)->ActorCallback< CommitTransactionActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CommitTransactionActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("commitTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CommitTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CommitTransactionActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("commitTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CommitTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CommitTransactionActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("commitTransaction", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CommitTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("commitTransaction", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Reference<ITransaction> tr;
															#line 1924 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
};
// This generated class is to be used only via commitTransaction()
															#line 727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class CommitTransactionActor final : public Actor<Void>, public ActorCallback< CommitTransactionActor, 0, Void >, public FastAllocated<CommitTransactionActor>, public CommitTransactionActorState<CommitTransactionActor> {
															#line 1929 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
	using FastAllocated<CommitTransactionActor>::operator new;
	using FastAllocated<CommitTransactionActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15592256090773114112UL, 12800067725496090368UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CommitTransactionActor, 0, Void >;
															#line 727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	CommitTransactionActor(Reference<ITransaction> const& tr) 
															#line 1946 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		 : Actor<Void>(),
		   CommitTransactionActorState<CommitTransactionActor>(tr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("commitTransaction", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16909656787573911808UL, 14603015077843198464UL);
		ActorExecutionContextHelper __helper(static_cast<CommitTransactionActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("commitTransaction");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("commitTransaction", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CommitTransactionActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
[[nodiscard]] Future<Void> commitTransaction( Reference<ITransaction> const& tr ) {
															#line 727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	return Future<Void>(new CommitTransactionActor(tr));
															#line 1979 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
}

#line 736 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"

															#line 1984 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
namespace {
// This generated class is to be used only via createSnapshot()
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class CreateSnapshotActor>
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class CreateSnapshotActorState {
															#line 1991 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	CreateSnapshotActorState(Database const& db,std::vector<StringRef> const& tokens) 
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		 : db(db),
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   tokens(tokens),
															#line 738 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   snapCmd(),
															#line 739 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   snapUID(deterministicRandom()->randomUniqueID())
															#line 2004 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
	{
		fdb_probe_actor_create("createSnapshot", reinterpret_cast<unsigned long>(this));

	}
	~CreateSnapshotActorState() 
	{
		fdb_probe_actor_destroy("createSnapshot", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 740 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			for(int i = 1;i < tokens.size();i++) {
															#line 741 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				snapCmd = snapCmd.withSuffix(tokens[i]);
															#line 742 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (i != tokens.size() - 1)
															#line 2023 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 743 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					snapCmd = snapCmd.withSuffix(" "_sr);
															#line 2027 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
			}
			try {
															#line 747 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				StrictFuture<Void> __when_expr_0 = makeInterruptable(mgmtSnapCreate(db, snapCmd, snapUID));
															#line 747 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (static_cast<CreateSnapshotActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2035 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<CreateSnapshotActor*>(this)->actor_wait_state = 1;
															#line 747 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CreateSnapshotActor, 0, Void >*>(static_cast<CreateSnapshotActor*>(this)));
															#line 2040 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CreateSnapshotActorState();
		static_cast<CreateSnapshotActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 758 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<CreateSnapshotActor*>(this)->SAV<bool>::futures) { (void)(false); this->~CreateSnapshotActorState(); static_cast<CreateSnapshotActor*>(this)->destroy(); return 0; }
															#line 2069 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<CreateSnapshotActor*>(this)->SAV< bool >::value()) bool(false);
		this->~CreateSnapshotActorState();
		static_cast<CreateSnapshotActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 750 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fprintf(stderr, "Snapshot command failed %d (%s)." " Please cleanup any instance level snapshots created with UID %s.\n", e.code(), e.what(), snapUID.toString().c_str());
															#line 756 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (!static_cast<CreateSnapshotActor*>(this)->SAV<bool>::futures) { (void)(true); this->~CreateSnapshotActorState(); static_cast<CreateSnapshotActor*>(this)->destroy(); return 0; }
															#line 2084 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			new (&static_cast<CreateSnapshotActor*>(this)->SAV< bool >::value()) bool(true);
			this->~CreateSnapshotActorState();
			static_cast<CreateSnapshotActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(Void const& _,int loopDepth) 
	{
															#line 748 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		printf("Snapshot command succeeded with UID %s\n", snapUID.toString().c_str());
															#line 2102 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont4(Void && _,int loopDepth) 
	{
															#line 748 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		printf("Snapshot command succeeded with UID %s\n", snapUID.toString().c_str());
															#line 2111 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CreateSnapshotActor*>(this)->actor_wait_state > 0) static_cast<CreateSnapshotActor*>(this)->actor_wait_state = 0;
		static_cast<CreateSnapshotActor*>(this)->ActorCallback< CreateSnapshotActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CreateSnapshotActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("createSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CreateSnapshotActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("createSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CreateSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CreateSnapshotActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("createSnapshot", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CreateSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("createSnapshot", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont6(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Database db;
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	std::vector<StringRef> tokens;
															#line 738 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Standalone<StringRef> snapCmd;
															#line 739 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	UID snapUID;
															#line 2212 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
};
// This generated class is to be used only via createSnapshot()
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class CreateSnapshotActor final : public Actor<bool>, public ActorCallback< CreateSnapshotActor, 0, Void >, public FastAllocated<CreateSnapshotActor>, public CreateSnapshotActorState<CreateSnapshotActor> {
															#line 2217 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
	using FastAllocated<CreateSnapshotActor>::operator new;
	using FastAllocated<CreateSnapshotActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10162353823032867328UL, 2052013179455429376UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<bool>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CreateSnapshotActor, 0, Void >;
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	CreateSnapshotActor(Database const& db,std::vector<StringRef> const& tokens) 
															#line 2234 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		 : Actor<bool>(),
		   CreateSnapshotActorState<CreateSnapshotActor>(db, tokens),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("createSnapshot", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2312050283204205312UL, 2801786450137338112UL);
		ActorExecutionContextHelper __helper(static_cast<CreateSnapshotActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("createSnapshot");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("createSnapshot", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CreateSnapshotActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
[[nodiscard]] Future<bool> createSnapshot( Database const& db, std::vector<StringRef> const& tokens ) {
															#line 737 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	return Future<bool>(new CreateSnapshotActor(db, tokens));
															#line 2267 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
}

#line 760 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"

// TODO: Update the function to get rid of the Database after refactoring
Reference<ITransaction> getTransaction(Reference<IDatabase> db,
                                       Reference<ITenant> tenant,
                                       Reference<ITransaction>& tr,
                                       FdbOptions* options,
                                       bool intrans) {
	// Update "tr" to point to a brand new transaction object when it's not initialized or "intrans" flag is "false",
	// which indicates we need a new transaction object
	if (!tr || !intrans) {
		if (tenant) {
			tr = tenant->createTransaction();
		} else {
			tr = db->createTransaction();
		}
		options->apply(tr);
	}

	return tr;
}

std::string newCompletion(const char* base, const char* name) {
	return format("%s%s ", base, name);
}

void compGenerator(const char* text, bool help, std::vector<std::string>& lc) {
	std::map<std::string, CommandHelp>::const_iterator iter;
	int len = strlen(text);

	const char* helpExtra[] = { "escaping", "options", nullptr };

	const char** he = helpExtra;

	for (auto iter = helpMap.begin(); iter != helpMap.end(); ++iter) {
		const char* name = (*iter).first.c_str();
		if (!strncmp(name, text, len)) {
			lc.push_back(newCompletion(help ? "help " : "", name));
		}
	}

	if (help) {
		while (*he) {
			const char* name = *he;
			he++;
			if (!strncmp(name, text, len))
				lc.push_back(newCompletion("help ", name));
		}
	}
}

void cmdGenerator(const char* text, std::vector<std::string>& lc) {
	compGenerator(text, false, lc);
}

void helpGenerator(const char* text, std::vector<std::string>& lc) {
	compGenerator(text, true, lc);
}

void optionGenerator(const char* text, const char* line, std::vector<std::string>& lc) {
	int len = strlen(text);

	for (auto iter = validOptions.begin(); iter != validOptions.end(); ++iter) {
		const char* name = (*iter).c_str();
		if (!strncmp(name, text, len)) {
			lc.push_back(newCompletion(line, name));
		}
	}
}

namespace fdb_cli {
void arrayGenerator(const char* text, const char* line, const char** options, std::vector<std::string>& lc) {
	const char** iter = options;
	int len = strlen(text);

	while (*iter) {
		const char* name = *iter;
		iter++;
		if (!strncmp(name, text, len)) {
			lc.push_back(newCompletion(line, name));
		}
	}
}
} // namespace fdb_cli

void onOffGenerator(const char* text, const char* line, std::vector<std::string>& lc) {
	const char* opts[] = { "on", "off", nullptr };
	arrayGenerator(text, line, opts, lc);
}

void fdbcliCompCmd(std::string const& text, std::vector<std::string>& lc) {
	bool err, partial;
	std::string whole_line = text;
	auto parsed = parseLine(whole_line, err, partial);
	if (err || partial) // If there was an error, or we are partially through a quoted sequence
		return;

	auto tokens = parsed.back();
	int count = tokens.size();

	// for(int i = 0; i < count; i++) {
	//	printf("Token (%d): `%s'\n", i, tokens[i].toString().c_str());
	// }

	std::string ntext = "";
	std::string base_input = text;

	// If there is a token and the input does not end in a space
	if (count && text.size() > 0 && text[text.size() - 1] != ' ') {
		count--; // Ignore the last token for purposes of later code
		ntext = tokens.back().toString();
		base_input = whole_line.substr(0, whole_line.rfind(ntext));
	}

	// printf("final text (%d tokens): `%s' & `%s'\n", count, base_input.c_str(), ntext.c_str());

	if (!count) {
		cmdGenerator(ntext.c_str(), lc);
		return;
	}

	if (tokencmp(tokens[0], "help") && count == 1) {
		helpGenerator(ntext.c_str(), lc);
		return;
	}

	if (tokencmp(tokens[0], "option")) {
		if (count == 1)
			onOffGenerator(ntext.c_str(), base_input.c_str(), lc);
		if (count == 2)
			optionGenerator(ntext.c_str(), base_input.c_str(), lc);
	}

	if (tokencmp(tokens[0], "writemode") && count == 1) {
		onOffGenerator(ntext.c_str(), base_input.c_str(), lc);
	}

	auto itr = CommandFactory::completionGenerators().find(tokens[0].toString());
	if (itr != CommandFactory::completionGenerators().end()) {
		itr->second(ntext.c_str(), base_input.c_str(), lc, tokens);
	}
}

void LogCommand(std::string line, UID randomID, std::string errMsg) {
	printf("%s\n", errMsg.c_str());
	TraceEvent(SevInfo, "CLICommandLog", randomID).detail("Command", line).detail("Error", errMsg);
}

struct CLIOptions {
	std::string program_name;
	int exit_code = -1;

	std::string commandLine;

	std::string clusterFile;
	bool trace = false;
	std::string traceDir;
	std::string traceFormat;
	std::string logGroup;
	int exit_timeout = 0;
	Optional<std::string> exec;
	bool initialStatusCheck = true;
	bool cliHints = true;
	bool useFutureProtocolVersion = false;
	bool debugTLS = false;
	std::string tlsCertPath;
	std::string tlsKeyPath;
	std::string tlsVerifyPeers;
	std::string tlsCAPath;
	std::string tlsPassword;
	bool tlsDisablePlainTextConnection = false;
	uint64_t memLimit = 8uLL << 30;

	std::vector<std::pair<std::string, std::string>> knobs;

	// api version, using the latest version by default
	int apiVersion = ApiVersion::LATEST_VERSION;

	CLIOptions(int argc, char* argv[]) {
		program_name = argv[0];
		for (int a = 0; a < argc; a++) {
			if (a)
				commandLine += ' ';
			commandLine += argv[a];
		}

		CSimpleOpt args(argc, argv, g_rgOptions, SO_O_HYPHEN_TO_UNDERSCORE);

		while (args.Next()) {
			int ec = processArg(args);
			if (ec != -1) {
				exit_code = ec;
				return;
			}
		}
		if (exit_timeout && !exec.present()) {
			fprintf(stderr, "ERROR: --timeout may only be specified with --exec\n");
			exit_code = FDB_EXIT_ERROR;
			return;
		}
	}

	void setupKnobs() {
		IKnobCollection::setupKnobs(knobs);

		// Reinitialize knobs in order to update knobs that are dependent on explicitly set knobs
		IKnobCollection::getMutableGlobalKnobCollection().initialize(Randomize::False, IsSimulated::False);
	}

	int processArg(CSimpleOpt& args) {
		if (args.LastError() != SO_SUCCESS) {
			printProgramUsage(program_name.c_str());
			return 1;
		}

		switch (args.OptionId()) {
		case OPT_CONNFILE:
			clusterFile = args.OptionArg();
			break;
		case OPT_API_VERSION: {
			char* endptr;
			apiVersion = strtoul((char*)args.OptionArg(), &endptr, 10);
			if (*endptr != '\0') {
				fprintf(stderr, "ERROR: invalid client version %s\n", args.OptionArg());
				return 1;
			} else if (apiVersion < 700 || apiVersion > ApiVersion::LATEST_VERSION) {
				// multi-version fdbcli only available after 7.0
				fprintf(stderr,
				        "ERROR: api version %s is not supported. (Min: 700, Max: %d)\n",
				        args.OptionArg(),
				        ApiVersion::LATEST_VERSION);
				return 1;
			}
			break;
		}
		case OPT_MEMORY: {
			std::string memoryArg(args.OptionArg());
			memLimit = parse_with_suffix(memoryArg, "MiB").orDefault(8uLL << 30);
			break;
		}
		case OPT_TRACE:
			trace = true;
			break;
		case OPT_TRACE_DIR:
			traceDir = args.OptionArg();
			break;
		case OPT_LOGGROUP:
			logGroup = args.OptionArg();
			break;
		case OPT_TIMEOUT: {
			char* endptr;
			exit_timeout = strtoul((char*)args.OptionArg(), &endptr, 10);
			if (*endptr != '\0') {
				fprintf(stderr, "ERROR: invalid timeout %s\n", args.OptionArg());
				return 1;
			}
			break;
		}
		case OPT_EXEC:
			exec = args.OptionArg();
			break;
		case OPT_NO_STATUS:
			initialStatusCheck = false;
			break;
		case OPT_NO_HINTS:
			cliHints = false;
			break;
		case OPT_USE_FUTURE_PROTOCOL_VERSION:
			useFutureProtocolVersion = true;
			break;

		// TLS Options
		case TLSConfig::OPT_TLS_PLUGIN:
			args.OptionArg();
			break;
		case TLSConfig::OPT_TLS_CERTIFICATES:
			tlsCertPath = args.OptionArg();
			break;
		case TLSConfig::OPT_TLS_CA_FILE:
			tlsCAPath = args.OptionArg();
			break;
		case TLSConfig::OPT_TLS_KEY:
			tlsKeyPath = args.OptionArg();
			break;
		case TLSConfig::OPT_TLS_PASSWORD:
			tlsPassword = args.OptionArg();
			break;
		case TLSConfig::OPT_TLS_VERIFY_PEERS:
			tlsVerifyPeers = args.OptionArg();
			break;
		case TLSConfig::OPT_TLS_DISABLE_PLAINTEXT_CONNECTION:
			tlsDisablePlainTextConnection = true;
			break;

		case OPT_HELP:
			printProgramUsage(program_name.c_str());
			return 0;
		case OPT_STATUS_FROM_JSON:
			return printStatusFromJSON(args.OptionArg());
		case OPT_TRACE_FORMAT:
			if (!validateTraceFormat(args.OptionArg())) {
				fprintf(stderr, "WARNING: Unrecognized trace format `%s'\n", args.OptionArg());
			}
			traceFormat = args.OptionArg();
			break;
		case OPT_KNOB: {
			Optional<std::string> knobName = extractPrefixedArgument("--knob", args.OptionSyntax());
			if (!knobName.present()) {
				fprintf(stderr, "ERROR: unable to parse knob option '%s'\n", args.OptionSyntax());
				return FDB_EXIT_ERROR;
			}
			knobs.emplace_back(knobName.get(), args.OptionArg());
			break;
		}
		case OPT_DEBUG_TLS:
			debugTLS = true;
			break;
		case OPT_VERSION:
			printVersion();
			return FDB_EXIT_SUCCESS;
		case OPT_BUILD_FLAGS:
			printBuildInformation();
			return FDB_EXIT_SUCCESS;
		}
		return -1;
	}
};

															#line 2598 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
namespace {
// This generated class is to be used only via stopNetworkAfter()
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class T, class StopNetworkAfterActor>
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class StopNetworkAfterActorState {
															#line 2605 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	StopNetworkAfterActorState(Future<T> const& what) 
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		 : what(what)
															#line 2612 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
	{
		fdb_probe_actor_create("stopNetworkAfter", reinterpret_cast<unsigned long>(this));

	}
	~StopNetworkAfterActorState() 
	{
		fdb_probe_actor_destroy("stopNetworkAfter", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 1090 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				StrictFuture<T> __when_expr_0 = what;
															#line 1090 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (static_cast<StopNetworkAfterActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 2630 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<StopNetworkAfterActor*>(this)->actor_wait_state = 1;
															#line 1090 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< StopNetworkAfterActor, 0, T >*>(static_cast<StopNetworkAfterActor*>(this)));
															#line 2635 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~StopNetworkAfterActorState();
		static_cast<StopNetworkAfterActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& __current_error,int loopDepth=0) 
	{
		try {
															#line 1094 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			API->stopNetwork();
															#line 1095 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			return a_body1Catch1(__current_error, loopDepth);
															#line 2667 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(T const& t,int loopDepth) 
	{
															#line 1091 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		API->stopNetwork();
															#line 1092 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<StopNetworkAfterActor*>(this)->SAV<T>::futures) { (void)(t); this->~StopNetworkAfterActorState(); static_cast<StopNetworkAfterActor*>(this)->destroy(); return 0; }
															#line 2683 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<StopNetworkAfterActor*>(this)->SAV< T >::value()) T(t);
		this->~StopNetworkAfterActorState();
		static_cast<StopNetworkAfterActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont2(T && t,int loopDepth) 
	{
															#line 1091 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		API->stopNetwork();
															#line 1092 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<StopNetworkAfterActor*>(this)->SAV<T>::futures) { (void)(t); this->~StopNetworkAfterActorState(); static_cast<StopNetworkAfterActor*>(this)->destroy(); return 0; }
															#line 2697 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<StopNetworkAfterActor*>(this)->SAV< T >::value()) T(t);
		this->~StopNetworkAfterActorState();
		static_cast<StopNetworkAfterActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(T const& t,int loopDepth) 
	{
		loopDepth = a_body1cont2(t, loopDepth);

		return loopDepth;
	}
	int a_body1when1(T && t,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(t), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<StopNetworkAfterActor*>(this)->actor_wait_state > 0) static_cast<StopNetworkAfterActor*>(this)->actor_wait_state = 0;
		static_cast<StopNetworkAfterActor*>(this)->ActorCallback< StopNetworkAfterActor, 0, T >::remove();

	}
	void a_callback_fire(ActorCallback< StopNetworkAfterActor, 0, T >*,T const& value) 
	{
		fdb_probe_actor_enter("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StopNetworkAfterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< StopNetworkAfterActor, 0, T >*,T && value) 
	{
		fdb_probe_actor_enter("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<StopNetworkAfterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< StopNetworkAfterActor, 0, T >*,Error err) 
	{
		fdb_probe_actor_enter("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<StopNetworkAfterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("stopNetworkAfter", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Future<T> what;
															#line 2782 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
};
// This generated class is to be used only via stopNetworkAfter()
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class T>
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class StopNetworkAfterActor final : public Actor<T>, public ActorCallback< StopNetworkAfterActor<T>, 0, T >, public FastAllocated<StopNetworkAfterActor<T>>, public StopNetworkAfterActorState<T, StopNetworkAfterActor<T>> {
															#line 2789 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
	using FastAllocated<StopNetworkAfterActor<T>>::operator new;
	using FastAllocated<StopNetworkAfterActor<T>>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(13300017793656798720UL, 12835845363623988736UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<T>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< StopNetworkAfterActor<T>, 0, T >;
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	StopNetworkAfterActor(Future<T> const& what) 
															#line 2806 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		 : Actor<T>(),
		   StopNetworkAfterActorState<T, StopNetworkAfterActor<T>>(what),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("stopNetworkAfter", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5440916104915410176UL, 16460411745428950528UL);
		ActorExecutionContextHelper __helper(static_cast<StopNetworkAfterActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("stopNetworkAfter");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("stopNetworkAfter", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< StopNetworkAfterActor<T>, 0, T >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class T>
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
[[nodiscard]] Future<T> stopNetworkAfter( Future<T> const& what ) {
															#line 1087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	return Future<T>(new StopNetworkAfterActor<T>(what));
															#line 2841 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
}

#line 1098 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"

enum TransType { Db = 0, Config, None };

															#line 2848 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
namespace {
// This generated class is to be used only via cli()
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class CliActor>
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class CliActorState {
															#line 2855 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	CliActorState(CLIOptions const& opt,LineNoise* const& plinenoise,Reference<ClusterConnectionFile> const& ccf) 
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		 : opt(opt),
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   plinenoise(plinenoise),
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   ccf(ccf),
															#line 1102 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   linenoise(*plinenoise),
															#line 1103 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   intrans(false),
															#line 1104 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   transtype(TransType::None),
															#line 1105 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   isCommitDesc(false),
															#line 1107 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   localDb(),
															#line 1108 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   db(),
															#line 1109 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   configDb(),
															#line 1110 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   tenant(),
															#line 1111 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   tenantName(),
															#line 1112 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   tenantEntry(),
															#line 1115 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   managementTenant(),
															#line 1117 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   tr(),
															#line 1118 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   config_tr(),
															#line 1119 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   trx(),
															#line 1121 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   writeMode(false),
															#line 1123 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   address_interface(),
															#line 1124 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   storage_interface(),
															#line 1126 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   globalOptions(),
															#line 1127 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   activeOptions(),
															#line 1129 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   options(&globalOptions)
															#line 2906 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
	{
		fdb_probe_actor_create("cli", reinterpret_cast<unsigned long>(this));

	}
	~CliActorState() 
	{
		fdb_probe_actor_destroy("cli", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1133 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			TraceEvent::setNetworkThread();
															#line 2921 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			try {
															#line 1136 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				localDb = Database::createDatabase(ccf, opt.apiVersion, IsInternal::False);
															#line 1137 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (!opt.exec.present())
															#line 2927 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1138 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					printf("Using cluster file `%s'.\n", ccf->getLocation().c_str());
															#line 2931 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
															#line 1140 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				db = API->createDatabase(opt.clusterFile.c_str());
															#line 1141 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				configDb = API->createDatabase(opt.clusterFile.c_str());
															#line 1142 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				configDb->setOption(FDBDatabaseOptions::USE_CONFIG_DATABASE);
															#line 2939 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont4(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~CliActorState();
		static_cast<CliActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1149 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (opt.trace)
															#line 2968 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1150 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			TraceEvent("CLIProgramStart") .setMaxEventLength(12000) .detail("SourceVersion", getSourceVersion()) .detail("Version", FDB_VT_VERSION) .detail("PackageName", FDB_VT_PACKAGE_NAME) .detailf("ActualTime", "%lld", DEBUG_DETERMINISM ? 0 : time(nullptr)) .detail("ClusterFile", ccf->toString()) .detail("ConnectionString", ccf->getConnectionString().toString()) .setMaxFieldLength(10000) .detail("CommandLine", opt.commandLine) .trackLatest("ProgramStart");
															#line 2972 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
															#line 1167 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		;
															#line 2976 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont1loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1144 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fprintf(stderr, "ERROR: %s (%d)\n", e.what(), e.code());
															#line 1145 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("Unable to connect to cluster from `%s'\n", ccf->getLocation().c_str());
															#line 1146 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (!static_cast<CliActor*>(this)->SAV<int>::futures) { (void)(1); this->~CliActorState(); static_cast<CliActor*>(this)->destroy(); return 0; }
															#line 2990 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			new (&static_cast<CliActor*>(this)->SAV< int >::value()) int(1);
			this->~CliActorState();
			static_cast<CliActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont5(int loopDepth) 
	{
															#line 1188 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!opt.exec.present())
															#line 3021 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1189 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (opt.initialStatusCheck)
															#line 3025 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1190 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				Future<Void> checkStatusF = checkStatus(Void(), db, localDb);
															#line 1191 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				StrictFuture<Void> __when_expr_2 = makeInterruptable(success(checkStatusF));
															#line 1191 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 3033 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1cont5when1(__when_expr_2.get(), loopDepth); };
				static_cast<CliActor*>(this)->actor_wait_state = 3;
															#line 1191 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< CliActor, 2, Void >*>(static_cast<CliActor*>(this)));
															#line 3038 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1193 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printf("\n");
															#line 3045 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont8(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont7(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1(int loopDepth) 
	{
		try {
															#line 1169 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			getTransaction(db, managementTenant, tr, options, intrans);
															#line 1170 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 1171 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(3.0) || success(safeThreadFutureToFuture(tr->getReadVersion()));
															#line 1171 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3074 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1cont1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1cont1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 1;
															#line 1171 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< CliActor, 0, Void >*>(static_cast<CliActor*>(this)));
															#line 3079 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1cont1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont5(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1174 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 3114 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1175 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 3118 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
															#line 1177 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (e.code() == error_code_cluster_version_changed)
															#line 3122 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1178 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				StrictFuture<Void> __when_expr_1 = safeThreadFutureToFuture(tr->onError(e));
															#line 1178 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3128 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1cont1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
				static_cast<CliActor*>(this)->actor_wait_state = 2;
															#line 1178 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< CliActor, 1, Void >*>(static_cast<CliActor*>(this)));
															#line 3133 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1181 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				fprintf(stderr, "ERROR: unexpected error %d while initializing the multiversion database\n", e.code());
															#line 1182 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				tr->reset();
															#line 3142 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void const& _,int loopDepth) 
	{
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1cont2(Void && _,int loopDepth) 
	{
		return a_body1cont1break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< CliActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< CliActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1cont1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1cont1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont1loopBody1Catch1cont1(int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1loopBody1Catch1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< CliActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1cont1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< CliActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1cont7(int loopDepth) 
	{
															#line 1200 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		is_error = false;
															#line 1201 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		warn = Future<Void>();
															#line 1202 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		;
															#line 3342 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont7loopHead1(loopDepth);

		return loopDepth;
	}
	int a_body1cont8(int loopDepth) 
	{
															#line 1196 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		printf("Welcome to the fdbcli. For help, type `help'.\n");
															#line 1197 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		validOptions = options->getValidOptions();
															#line 3353 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont7(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont5when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< CliActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< CliActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1cont7loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont7loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1(int loopDepth) 
	{
															#line 1203 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (warn.isValid())
															#line 3456 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1204 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			warn.cancel();
															#line 3460 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
															#line 1206 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		line = std::string();
															#line 1208 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (opt.exec.present())
															#line 3466 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1209 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			line = opt.exec.get();
															#line 3470 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = a_body1cont7loopBody1cont1(loopDepth);
		}
		else
		{
															#line 1211 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Optional<std::string>> __when_expr_3 = linenoise.read("fdb> ");
															#line 1211 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3479 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1when1(__when_expr_3.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 4;
															#line 1211 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< CliActor, 3, Optional<std::string> >*>(static_cast<CliActor*>(this)));
															#line 3484 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1(int loopDepth) 
	{
															#line 1227 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		warn = checkStatus(timeWarning(5.0, "\nWARNING: Long delay (Ctrl-C to interrupt)\n"), db, localDb);
															#line 3494 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		try {
															#line 1230 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			randomID = deterministicRandom()->randomUniqueID();
															#line 1231 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			TraceEvent(SevInfo, "CLICommandLog", randomID).detail("Command", line);
															#line 1233 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			bool malformed, partial;
															#line 1234 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			parsed = parseLine(line, malformed, partial);
															#line 1235 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (malformed)
															#line 3506 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1236 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				LogCommand(line, randomID, "ERROR: malformed escape sequence");
															#line 3510 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
															#line 1237 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (partial)
															#line 3514 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1238 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				LogCommand(line, randomID, "ERROR: unterminated quote");
															#line 3518 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
															#line 1239 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (malformed || partial)
															#line 3522 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1240 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (parsed.size() > 0)
															#line 3526 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1242 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					auto& last_command = parsed.back();
															#line 1243 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					last_command.insert(last_command.begin(), StringRef((const uint8_t*)"parse_error", strlen("parse_error")));
															#line 3532 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
			}
															#line 1248 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			multi = parsed.size() > 1;
															#line 1249 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = false;
															#line 1251 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			iter = std::vector<std::vector<StringRef>>::iterator();
															#line 1252 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			iter = parsed.begin();
															#line 3543 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = a_body1cont7loopBody1cont1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont4(Optional<std::string> const& rawline,int loopDepth) 
	{
															#line 1212 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!rawline.present())
															#line 3558 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1213 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("\n");
															#line 1214 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (!static_cast<CliActor*>(this)->SAV<int>::futures) { (void)(0); this->~CliActorState(); static_cast<CliActor*>(this)->destroy(); return 0; }
															#line 3564 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			new (&static_cast<CliActor*>(this)->SAV< int >::value()) int(0);
			this->~CliActorState();
			static_cast<CliActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1216 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		line = rawline.get();
															#line 1218 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!line.size())
															#line 3574 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
			return a_body1cont7loopHead1(loopDepth); // continue
		}
															#line 1222 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (line.find("writemode") == std::string::npos && line.find("expensive_data_check") == std::string::npos && line.find("unlock") == std::string::npos && line.find("blobrange") == std::string::npos)
															#line 3580 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1224 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			linenoise.historyAdd(line);
															#line 3584 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont4(Optional<std::string> && rawline,int loopDepth) 
	{
															#line 1212 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!rawline.present())
															#line 3594 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1213 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("\n");
															#line 1214 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (!static_cast<CliActor*>(this)->SAV<int>::futures) { (void)(0); this->~CliActorState(); static_cast<CliActor*>(this)->destroy(); return 0; }
															#line 3600 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			new (&static_cast<CliActor*>(this)->SAV< int >::value()) int(0);
			this->~CliActorState();
			static_cast<CliActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1216 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		line = rawline.get();
															#line 1218 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!line.size())
															#line 3610 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
			return a_body1cont7loopHead1(loopDepth); // continue
		}
															#line 1222 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (line.find("writemode") == std::string::npos && line.find("expensive_data_check") == std::string::npos && line.find("unlock") == std::string::npos && line.find("blobrange") == std::string::npos)
															#line 3616 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1224 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			linenoise.historyAdd(line);
															#line 3620 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1when1(Optional<std::string> const& rawline,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont4(rawline, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1when1(Optional<std::string> && rawline,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont4(std::move(rawline), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 3, Optional<std::string> >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 3, Optional<std::string> >*,Optional<std::string> const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< CliActor, 3, Optional<std::string> >*,Optional<std::string> && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1cont7loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< CliActor, 3, Optional<std::string> >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1cont7loopBody1cont9(int loopDepth) 
	{
															#line 2315 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (opt.exec.present())
															#line 3705 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2316 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (!static_cast<CliActor*>(this)->SAV<int>::futures) { (void)(is_error ? 1 : 0); this->~CliActorState(); static_cast<CliActor*>(this)->destroy(); return 0; }
															#line 3709 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			new (&static_cast<CliActor*>(this)->SAV< int >::value()) int(is_error ? 1 : 0);
			this->~CliActorState();
			static_cast<CliActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		if (loopDepth == 0) return a_body1cont7loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2295 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (e.code() == error_code_operation_cancelled)
															#line 3724 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2296 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 3728 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
															#line 2298 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (e.code() == error_code_tenant_name_required)
															#line 3732 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2299 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printAtCol("ERROR: tenant name required. Use the `usetenant' command to select a tenant or enable the " "`RAW_ACCESS' option to read raw keys.", 80, stderr);
															#line 3736 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
			else
			{
															#line 2303 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (e.code() != error_code_actor_cancelled)
															#line 3742 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2304 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					fprintf(stderr, "ERROR: %s (%d)\n", e.what(), e.code());
															#line 3746 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
			}
															#line 2306 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 2307 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (intrans)
															#line 3753 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2308 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printf("Rolling back current transaction\n");
															#line 2309 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				intrans = false;
															#line 2310 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				options = &globalOptions;
															#line 2311 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				options->apply(tr);
															#line 3763 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
			loopDepth = a_body1cont7loopBody1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont10(int loopDepth) 
	{
															#line 2293 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		TraceEvent(SevInfo, "CLICommandLog", randomID).detail("Command", line).detail("IsError", is_error);
															#line 3779 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont7loopBody1cont10cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont7loopBody1cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1(int loopDepth) 
	{
															#line 1252 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!(iter != parsed.end()))
															#line 3795 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
			return a_body1cont7loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1253 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		tokens = *iter;
															#line 1255 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (is_error)
															#line 3803 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1256 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("WARNING: the previous command failed, the remaining commands will not be executed.\n");
															#line 3807 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			return a_body1cont7loopBody1cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1260 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!tokens.size())
															#line 3812 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
			return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
		}
															#line 1263 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "parse_error"))
															#line 3818 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1264 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fprintf(stderr, "ERROR: Command failed to completely parse.\n");
															#line 1265 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() > 1)
															#line 3824 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1266 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				fprintf(stderr, "ERROR: Not running partial or malformed command:");
															#line 1267 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				for(auto t = tokens.begin() + 1;t != tokens.end();++t) {
															#line 1268 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					printf(" %s", formatStringRef(*t, true).c_str());
															#line 3832 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
															#line 1269 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printf("\n");
															#line 3836 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
															#line 1271 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 3840 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
		}
															#line 1275 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (multi)
															#line 3845 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1276 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf(">>>");
															#line 1277 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			for(auto t = tokens.begin();t != tokens.end();++t) {
															#line 1278 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printf(" %s", formatStringRef(*t, true).c_str());
															#line 3853 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
															#line 1279 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("\n");
															#line 3857 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
															#line 1282 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!helpMap.count(tokens[0].toString()) && !hiddenCommands.count(tokens[0].toString()))
															#line 3861 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1283 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fprintf(stderr, "ERROR: Unknown command `%s'. Try `help'?\n", formatStringRef(tokens[0]).c_str());
															#line 1284 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 3867 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
		}
															#line 1288 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "exit") || tokencmp(tokens[0], "quit"))
															#line 3872 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1289 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (!static_cast<CliActor*>(this)->SAV<int>::futures) { (void)(0); this->~CliActorState(); static_cast<CliActor*>(this)->destroy(); return 0; }
															#line 3876 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			new (&static_cast<CliActor*>(this)->SAV< int >::value()) int(0);
			this->~CliActorState();
			static_cast<CliActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 1292 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "help"))
															#line 3884 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1293 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() == 1)
															#line 3888 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1294 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printHelpOverview();
															#line 3892 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
			else
			{
															#line 1295 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (tokens.size() == 2)
															#line 3898 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1296 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (tokencmp(tokens[1], "escaping"))
															#line 3902 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1297 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						printf("\n" "When parsing commands, fdbcli considers a space to delimit individual tokens.\n" "To include a space in a single token, you may either enclose the token in\n" "quotation marks (\"hello world\"), prefix the space with a backslash\n" "(hello\\ world), or encode the space as a hex byte (hello\\x20world).\n" "\n" "To include a literal quotation mark in a token, precede it with a backslash\n" "(\\\"hello\\ world\\\").\n" "\n" "To express a binary value, encode each byte as either\n" "   a) a two-digit hex byte preceded by \\x\n" "   b) a four-digit hex byte in the range of 0x0000-0x00FF preceded by \\u\n" "(e.g. \\x20 or \\u0020 for a space character, or \\x0a\\x00\\x00\\x00 or\n" "\\u000a\\u0000\\u0000\\u0000 for a 32-bit, little-endian representation of\n" "the integer 10.  Any byte can use either syntax, so \\u000a\\x00\\x00\\x00\n" "is also a valid representation of a little-endian value of 10).\n" "\n" "All keys and values are displayed by the fdbcli with non-printable characters\n" "and spaces encoded as two-digit hex bytes.\n\n");
															#line 3906 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 1316 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (tokencmp(tokens[1], "options"))
															#line 3912 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 1317 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							printf("\n" "The following options are available to be set using the `option' command:\n" "\n");
															#line 1320 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							options->printHelpString();
															#line 3918 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						}
						else
						{
															#line 1321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							if (tokencmp(tokens[1], "help"))
															#line 3924 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							{
															#line 1322 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
								printHelpOverview();
															#line 3928 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							}
							else
							{
															#line 1324 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
								printHelp(tokens[1]);
															#line 3934 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							}
						}
					}
				}
				else
				{
															#line 1326 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					printf("Usage: help [topic]\n");
															#line 3943 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
			}
			return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
		}
															#line 1330 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "waitconnected"))
															#line 3950 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1331 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Void> __when_expr_4 = makeInterruptable(localDb->onConnected());
															#line 1331 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3956 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_4.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1when1(__when_expr_4.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 5;
															#line 1331 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< CliActor, 4, Void >*>(static_cast<CliActor*>(this)));
															#line 3961 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont7loopBody1cont10(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1continue1(int loopDepth) 
	{
															#line 1252 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		++iter;
															#line 3988 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont7loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont1(int loopDepth) 
	{
															#line 1335 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "waitopen"))
															#line 3997 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1336 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Void> __when_expr_5 = makeInterruptable(success(safeThreadFutureToFuture( getTransaction(db, managementTenant, tr, options, intrans)->getReadVersion())));
															#line 1336 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4003 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_5.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont1when1(__when_expr_5.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 6;
															#line 1336 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< CliActor, 5, Void >*>(static_cast<CliActor*>(this)));
															#line 4008 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont23(Void const& _,int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont23(Void && _,int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont23(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont23(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont7loopBody1cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< CliActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont7loopBody1cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< CliActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24(int loopDepth) 
	{
															#line 1341 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "sleep"))
															#line 4109 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1342 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() != 2)
															#line 4113 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1343 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1344 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 4119 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont2(loopDepth);
			}
			else
			{
															#line 1346 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				double v;
															#line 1347 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				int n = 0;
															#line 1348 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (sscanf(tokens[1].toString().c_str(), "%lf%n", &v, &n) != 1 || n != tokens[1].size())
															#line 4130 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1349 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					printUsage(tokens[0]);
															#line 1350 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					is_error = true;
															#line 4136 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont4(loopDepth);
				}
				else
				{
															#line 1352 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					StrictFuture<Void> __when_expr_6 = delay(v);
															#line 1352 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4145 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_6.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24when1(__when_expr_6.get(), loopDepth); };
					static_cast<CliActor*>(this)->actor_wait_state = 7;
															#line 1352 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< CliActor, 6, Void >*>(static_cast<CliActor*>(this)));
															#line 4150 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = 0;
				}
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont25(Void const& _,int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont25(Void && _,int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont25(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont25(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7loopBody1cont1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< CliActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7loopBody1cont1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< CliActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont1(int loopDepth) 
	{
															#line 1358 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "status"))
															#line 4253 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1361 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			warn = timeWarning(7.0, "\nWARNING: Long delay (Ctrl-C to interrupt)\n");
															#line 1362 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_7 = makeInterruptable(statusCommandActor(db, localDb, tokens, opt.exec.present()));
															#line 1362 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4261 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont1when1(__when_expr_7.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 8;
															#line 1362 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< CliActor, 7, bool >*>(static_cast<CliActor*>(this)));
															#line 4266 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont8(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont2(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont4(int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont6(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont6(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont6(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont6(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< CliActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< CliActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont8(int loopDepth) 
	{
															#line 1368 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "triggerddteaminfolog"))
															#line 4379 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1369 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Void> __when_expr_8 = success(makeInterruptable(triggerddteaminfologCommandActor(db)));
															#line 1369 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4385 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_8.isReady()) { if (__when_expr_8.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_8.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont8when1(__when_expr_8.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 9;
															#line 1369 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_8.addCallbackAndClear(static_cast<ActorCallback< CliActor, 8, Void >*>(static_cast<CliActor*>(this)));
															#line 4390 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont9(bool const& _result,int loopDepth) 
	{
															#line 1363 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 4404 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1364 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 4408 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont9(bool && _result,int loopDepth) 
	{
															#line 1363 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 4418 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1364 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 4422 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont9(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont9(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 7, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 7, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< CliActor, 7, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< CliActor, 7, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 7);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12(int loopDepth) 
	{
															#line 1373 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "tssq"))
															#line 4507 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1374 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_9 = makeInterruptable(tssqCommandActor(db, tokens));
															#line 1374 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4513 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_9.isReady()) { if (__when_expr_9.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_9.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12when1(__when_expr_9.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 10;
															#line 1374 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_9.addCallbackAndClear(static_cast<ActorCallback< CliActor, 9, bool >*>(static_cast<CliActor*>(this)));
															#line 4518 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont13(Void const& _,int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont13(Void && _,int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont8when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont13(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont8when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont13(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose9() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 8, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 8, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont8when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_fire(ActorCallback< CliActor, 8, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont8when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 8);

	}
	void a_callback_error(ActorCallback< CliActor, 8, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 8);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose9();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 8);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont1(int loopDepth) 
	{
															#line 1380 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "configure"))
															#line 4619 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1381 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_10 = makeInterruptable(configureCommandActor(db, localDb, tokens, &linenoise, warn));
															#line 1381 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4625 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_10.isReady()) { if (__when_expr_10.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_10.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont1when1(__when_expr_10.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 11;
															#line 1381 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_10.addCallbackAndClear(static_cast<ActorCallback< CliActor, 10, bool >*>(static_cast<CliActor*>(this)));
															#line 4630 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont2(bool const& _result,int loopDepth) 
	{
															#line 1375 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 4644 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1376 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 4648 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont2(bool && _result,int loopDepth) 
	{
															#line 1375 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 4658 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1376 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 4662 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont2(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont2(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose10() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 9, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 9, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_fire(ActorCallback< CliActor, 9, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 9);

	}
	void a_callback_error(ActorCallback< CliActor, 9, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 9);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose10();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 9);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont5(int loopDepth) 
	{
															#line 1388 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "fileconfigure"))
															#line 4747 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1389 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() == 2 || (tokens.size() == 3 && (tokens[1] == "new"_sr || tokens[1] == "FORCE"_sr)))
															#line 4751 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1391 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				StrictFuture<bool> __when_expr_11 = makeInterruptable(fileConfigureCommandActor( db, tokens.back().toString(), tokens[1] == "new"_sr, tokens[1] == "FORCE"_sr));
															#line 1391 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4757 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				if (__when_expr_11.isReady()) { if (__when_expr_11.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_11.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont5when1(__when_expr_11.get(), loopDepth); };
				static_cast<CliActor*>(this)->actor_wait_state = 12;
															#line 1391 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				__when_expr_11.addCallbackAndClear(static_cast<ActorCallback< CliActor, 11, bool >*>(static_cast<CliActor*>(this)));
															#line 4762 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = 0;
			}
			else
			{
															#line 1396 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1397 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 4771 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont10(loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont6(bool const& _result,int loopDepth) 
	{
															#line 1383 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 4786 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1384 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 4790 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont6(bool && _result,int loopDepth) 
	{
															#line 1383 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 4800 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1384 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 4804 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont6(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont6(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose11() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 10, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 10, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_fire(ActorCallback< CliActor, 10, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 10);

	}
	void a_callback_error(ActorCallback< CliActor, 10, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 10);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose11();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 10);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont9(int loopDepth) 
	{
															#line 1402 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "coordinators"))
															#line 4889 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1403 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_12 = makeInterruptable(coordinatorsCommandActor(db, tokens));
															#line 1403 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4895 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_12.isReady()) { if (__when_expr_12.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_12.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont9when1(__when_expr_12.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 13;
															#line 1403 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_12.addCallbackAndClear(static_cast<ActorCallback< CliActor, 12, bool >*>(static_cast<CliActor*>(this)));
															#line 4900 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont10(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont11(bool const& _result,int loopDepth) 
	{
															#line 1393 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 4920 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1394 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 4924 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont11(bool && _result,int loopDepth) 
	{
															#line 1393 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 4934 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1394 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 4938 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont5when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont11(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont5when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont11(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose12() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 11, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 11, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_fire(ActorCallback< CliActor, 11, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 11);

	}
	void a_callback_error(ActorCallback< CliActor, 11, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 11);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose12();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 11);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13(int loopDepth) 
	{
															#line 1409 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "exclude"))
															#line 5023 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1410 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_13 = makeInterruptable(excludeCommandActor(db, tokens, warn));
															#line 1410 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5029 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_13.isReady()) { if (__when_expr_13.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_13.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13when1(__when_expr_13.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 14;
															#line 1410 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_13.addCallbackAndClear(static_cast<ActorCallback< CliActor, 13, bool >*>(static_cast<CliActor*>(this)));
															#line 5034 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont14(bool const& _result,int loopDepth) 
	{
															#line 1404 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5048 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1405 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5052 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont14(bool && _result,int loopDepth) 
	{
															#line 1404 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5062 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1405 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5066 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont9when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont14(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont9when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont14(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose13() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 12, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 12, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_fire(ActorCallback< CliActor, 12, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 12);

	}
	void a_callback_error(ActorCallback< CliActor, 12, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 12);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose13();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 12);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont1(int loopDepth) 
	{
															#line 1416 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "include"))
															#line 5151 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1417 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_14 = makeInterruptable(includeCommandActor(db, tokens));
															#line 1417 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5157 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_14.isReady()) { if (__when_expr_14.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_14.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont1when1(__when_expr_14.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 15;
															#line 1417 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_14.addCallbackAndClear(static_cast<ActorCallback< CliActor, 14, bool >*>(static_cast<CliActor*>(this)));
															#line 5162 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont2(bool const& _result,int loopDepth) 
	{
															#line 1411 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5176 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1412 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5180 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont2(bool && _result,int loopDepth) 
	{
															#line 1411 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5190 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1412 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5194 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont2(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont2(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose14() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 13, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 13, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_fire(ActorCallback< CliActor, 13, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 13);

	}
	void a_callback_error(ActorCallback< CliActor, 13, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 13);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose14();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 13);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont5(int loopDepth) 
	{
															#line 1423 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "snapshot"))
															#line 5279 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1424 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_15 = makeInterruptable(snapshotCommandActor(db, tokens));
															#line 1424 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5285 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_15.isReady()) { if (__when_expr_15.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_15.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont5when1(__when_expr_15.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 16;
															#line 1424 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_15.addCallbackAndClear(static_cast<ActorCallback< CliActor, 15, bool >*>(static_cast<CliActor*>(this)));
															#line 5290 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont6(bool const& _result,int loopDepth) 
	{
															#line 1418 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5304 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1419 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5308 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont6(bool && _result,int loopDepth) 
	{
															#line 1418 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5318 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1419 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5322 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont6(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont6(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose15() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 14, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 14, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose15();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_fire(ActorCallback< CliActor, 14, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose15();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 14);

	}
	void a_callback_error(ActorCallback< CliActor, 14, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 14);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose15();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 14);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont9(int loopDepth) 
	{
															#line 1430 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "lock"))
															#line 5407 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1431 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_16 = makeInterruptable(lockCommandActor(db, tokens));
															#line 1431 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5413 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_16.isReady()) { if (__when_expr_16.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_16.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont9when1(__when_expr_16.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 17;
															#line 1431 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_16.addCallbackAndClear(static_cast<ActorCallback< CliActor, 16, bool >*>(static_cast<CliActor*>(this)));
															#line 5418 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont10(bool const& _result,int loopDepth) 
	{
															#line 1425 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5432 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1426 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5436 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont10(bool && _result,int loopDepth) 
	{
															#line 1425 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5446 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1426 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5450 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont5when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont10(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont5when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont10(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose16() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 15, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 15, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose16();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 15);

	}
	void a_callback_fire(ActorCallback< CliActor, 15, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose16();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 15);

	}
	void a_callback_error(ActorCallback< CliActor, 15, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 15);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose16();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 15);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11(int loopDepth) 
	{
															#line 1437 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "changefeed"))
															#line 5535 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1438 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_17 = makeInterruptable(changeFeedCommandActor(localDb, tenantEntry, tokens, warn));
															#line 1438 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5541 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_17.isReady()) { if (__when_expr_17.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_17.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11when1(__when_expr_17.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 18;
															#line 1438 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_17.addCallbackAndClear(static_cast<ActorCallback< CliActor, 17, bool >*>(static_cast<CliActor*>(this)));
															#line 5546 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont12(bool const& _result,int loopDepth) 
	{
															#line 1432 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5560 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1433 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5564 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont12(bool && _result,int loopDepth) 
	{
															#line 1432 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5574 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1433 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5578 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont9when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont12(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont9when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont12(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose17() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 16, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 16, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 16);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose17();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 16);

	}
	void a_callback_fire(ActorCallback< CliActor, 16, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 16);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose17();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 16);

	}
	void a_callback_error(ActorCallback< CliActor, 16, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 16);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose17();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 16);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont1(int loopDepth) 
	{
															#line 1444 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "blobrange"))
															#line 5663 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1445 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_18 = makeInterruptable(blobRangeCommandActor(localDb, tenantEntry, tokens));
															#line 1445 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5669 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_18.isReady()) { if (__when_expr_18.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_18.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont1when1(__when_expr_18.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 19;
															#line 1445 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_18.addCallbackAndClear(static_cast<ActorCallback< CliActor, 18, bool >*>(static_cast<CliActor*>(this)));
															#line 5674 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont2(bool const& _result,int loopDepth) 
	{
															#line 1439 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5688 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1440 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5692 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont2(bool && _result,int loopDepth) 
	{
															#line 1439 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5702 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1440 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5706 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont2(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont2(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose18() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 17, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 17, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 17);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose18();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 17);

	}
	void a_callback_fire(ActorCallback< CliActor, 17, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 17);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose18();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 17);

	}
	void a_callback_error(ActorCallback< CliActor, 17, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 17);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose18();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 17);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont5(int loopDepth) 
	{
															#line 1451 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "blobkey"))
															#line 5791 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1452 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_19 = makeInterruptable(blobKeyCommandActor(localDb, tenantEntry, tokens));
															#line 1452 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5797 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_19.isReady()) { if (__when_expr_19.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_19.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont5when1(__when_expr_19.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 20;
															#line 1452 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_19.addCallbackAndClear(static_cast<ActorCallback< CliActor, 19, bool >*>(static_cast<CliActor*>(this)));
															#line 5802 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont6(bool const& _result,int loopDepth) 
	{
															#line 1446 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5816 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1447 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5820 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont6(bool && _result,int loopDepth) 
	{
															#line 1446 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5830 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1447 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5834 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont6(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont6(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose19() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 18, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 18, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 18);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose19();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 18);

	}
	void a_callback_fire(ActorCallback< CliActor, 18, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 18);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose19();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 18);

	}
	void a_callback_error(ActorCallback< CliActor, 18, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 18);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose19();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 18);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont9(int loopDepth) 
	{
															#line 1458 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "unlock"))
															#line 5919 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1459 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if ((tokens.size() != 2) || (tokens[1].size() != 32) || !std::all_of(tokens[1].begin(), tokens[1].end(), &isxdigit))
															#line 5923 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1461 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1462 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 5929 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont12(loopDepth);
			}
			else
			{
															#line 1464 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				passPhrase = deterministicRandom()->randomAlphaNumeric(10);
															#line 1465 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				warn.cancel();
															#line 1466 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printf("Unlocking the database is a potentially dangerous operation.\n");
															#line 1467 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printf("%s\n", passPhrase.c_str());
															#line 1468 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				fflush(stdout);
															#line 1469 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				StrictFuture<Optional<std::string>> __when_expr_20 = linenoise.read(format("Repeat the above passphrase if you would like to proceed:"));
															#line 1469 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5948 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				if (__when_expr_20.isReady()) { if (__when_expr_20.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_20.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont9when1(__when_expr_20.get(), loopDepth); };
				static_cast<CliActor*>(this)->actor_wait_state = 21;
															#line 1469 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				__when_expr_20.addCallbackAndClear(static_cast<ActorCallback< CliActor, 20, Optional<std::string> >*>(static_cast<CliActor*>(this)));
															#line 5953 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont10(bool const& _result,int loopDepth) 
	{
															#line 1453 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5968 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1454 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5972 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont10(bool && _result,int loopDepth) 
	{
															#line 1453 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 5982 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1454 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 5986 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont5when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont10(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont5when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont10(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose20() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 19, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 19, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 19);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose20();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 19);

	}
	void a_callback_fire(ActorCallback< CliActor, 19, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 19);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose20();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 19);

	}
	void a_callback_error(ActorCallback< CliActor, 19, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 19);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose20();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 19);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11(int loopDepth) 
	{
															#line 1486 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "setclass"))
															#line 6071 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1487 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_22 = makeInterruptable(setClassCommandActor(db, tokens));
															#line 1487 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6077 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_22.isReady()) { if (__when_expr_22.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_22.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11when1(__when_expr_22.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 23;
															#line 1487 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_22.addCallbackAndClear(static_cast<ActorCallback< CliActor, 22, bool >*>(static_cast<CliActor*>(this)));
															#line 6082 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont12(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14(Optional<std::string> const& input,int loopDepth) 
	{
															#line 1471 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		warn = checkStatus(timeWarning(5.0, "\nWARNING: Long delay (Ctrl-C to interrupt)\n"), db, localDb);
															#line 1473 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (input.present() && input.get() == passPhrase)
															#line 6104 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1474 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			UID unlockUID = UID::fromString(tokens[1].toString());
															#line 1475 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_21 = makeInterruptable(unlockDatabaseActor(db, unlockUID));
															#line 1475 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6112 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_21.isReady()) { if (__when_expr_21.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_21.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14when1(__when_expr_21.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 22;
															#line 1475 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_21.addCallbackAndClear(static_cast<ActorCallback< CliActor, 21, bool >*>(static_cast<CliActor*>(this)));
															#line 6117 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1479 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fprintf(stderr, "ERROR: Incorrect passphrase entered.\n");
															#line 1480 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 6126 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14(Optional<std::string> && input,int loopDepth) 
	{
															#line 1471 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		warn = checkStatus(timeWarning(5.0, "\nWARNING: Long delay (Ctrl-C to interrupt)\n"), db, localDb);
															#line 1473 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (input.present() && input.get() == passPhrase)
															#line 6138 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1474 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			UID unlockUID = UID::fromString(tokens[1].toString());
															#line 1475 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_21 = makeInterruptable(unlockDatabaseActor(db, unlockUID));
															#line 1475 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6146 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_21.isReady()) { if (__when_expr_21.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_21.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14when1(__when_expr_21.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 22;
															#line 1475 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_21.addCallbackAndClear(static_cast<ActorCallback< CliActor, 21, bool >*>(static_cast<CliActor*>(this)));
															#line 6151 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1479 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fprintf(stderr, "ERROR: Incorrect passphrase entered.\n");
															#line 1480 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 6160 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont9when1(Optional<std::string> const& input,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14(input, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont9when1(Optional<std::string> && input,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14(std::move(input), loopDepth);

		return loopDepth;
	}
	void a_exitChoose21() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 20, Optional<std::string> >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 20, Optional<std::string> >*,Optional<std::string> const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 20);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose21();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 20);

	}
	void a_callback_fire(ActorCallback< CliActor, 20, Optional<std::string> >*,Optional<std::string> && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 20);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose21();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 20);

	}
	void a_callback_error(ActorCallback< CliActor, 20, Optional<std::string> >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 20);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose21();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 20);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14cont1(int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont12(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14cont2(bool const& _result,int loopDepth) 
	{
															#line 1476 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 6251 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1477 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 6255 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14cont2(bool && _result,int loopDepth) 
	{
															#line 1476 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 6265 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1477 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 6269 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14cont2(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14cont2(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose22() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 21, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 21, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 21);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose22();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 21);

	}
	void a_callback_fire(ActorCallback< CliActor, 21, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 21);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose22();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont14when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 21);

	}
	void a_callback_error(ActorCallback< CliActor, 21, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 21);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose22();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 21);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1(int loopDepth) 
	{
															#line 1493 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "begin"))
															#line 6354 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1494 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() != 1)
															#line 6358 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1495 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1496 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 6364 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
			else
			{
															#line 1497 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (intrans)
															#line 6370 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1498 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					fprintf(stderr, "ERROR: Already in transaction\n");
															#line 1499 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					is_error = true;
															#line 6376 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
				else
				{
															#line 1501 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					activeOptions = FdbOptions(globalOptions);
															#line 1502 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					options = &activeOptions;
															#line 1503 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					intrans = true;
															#line 1504 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					transtype = TransType::None;
															#line 1505 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					getTransaction(db, tenant, tr, options, false);
															#line 1506 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					printf("Transaction started\n");
															#line 6392 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
			}
			return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
		}
															#line 1511 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "commit"))
															#line 6399 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1512 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() > 2)
															#line 6403 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1513 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1514 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 6409 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont11(loopDepth);
			}
			else
			{
															#line 1515 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (!intrans)
															#line 6416 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1516 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					fprintf(stderr, "ERROR: No active transaction\n");
															#line 1517 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					is_error = true;
															#line 6422 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont13(loopDepth);
				}
				else
				{
															#line 1519 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (isCommitDesc && tokens.size() == 1)
															#line 6429 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1521 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						raw = Optional<std::string>();
															#line 1522 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						warn.cancel();
															#line 1523 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						;
															#line 6437 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopHead1(loopDepth);
					}
					else
					{
						loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15(loopDepth);
					}
				}
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont2(bool const& _result,int loopDepth) 
	{
															#line 1488 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 6458 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1489 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 6462 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont2(bool && _result,int loopDepth) 
	{
															#line 1488 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 6472 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1489 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 6476 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont2(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont2(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose23() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 22, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 22, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 22);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose23();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 22);

	}
	void a_callback_fire(ActorCallback< CliActor, 22, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 22);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose23();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 22);

	}
	void a_callback_error(ActorCallback< CliActor, 22, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 22);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose23();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 22);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont5(int loopDepth) 
	{
															#line 1552 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "quota"))
															#line 6561 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1553 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_26 = makeInterruptable(quotaCommandActor(db, tokens));
															#line 1553 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6567 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_26.isReady()) { if (__when_expr_26.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_26.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont5when1(__when_expr_26.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 27;
															#line 1553 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_26.addCallbackAndClear(static_cast<ActorCallback< CliActor, 26, bool >*>(static_cast<CliActor*>(this)));
															#line 6572 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont11(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont13(int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15(int loopDepth) 
	{
															#line 1533 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		warn = checkStatus(timeWarning(5.0, "\nWARNING: Long delay (Ctrl-C to interrupt)\n"), db, localDb);
															#line 1535 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (transtype == TransType::Db)
															#line 6600 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1536 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Void> __when_expr_24 = commitTransaction(tr);
															#line 1536 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6606 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_24.isReady()) { if (__when_expr_24.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_24.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15when1(__when_expr_24.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 25;
															#line 1536 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_24.addCallbackAndClear(static_cast<ActorCallback< CliActor, 24, Void >*>(static_cast<CliActor*>(this)));
															#line 6611 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
															#line 1538 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() > 1)
															#line 6618 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1539 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				config_tr->set("\xff\xff/description"_sr, tokens[1]);
															#line 6622 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
															#line 1541 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Void> __when_expr_25 = commitTransaction(config_tr);
															#line 1541 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 6628 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_25.isReady()) { if (__when_expr_25.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_25.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15when2(__when_expr_25.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 26;
															#line 1541 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_25.addCallbackAndClear(static_cast<ActorCallback< CliActor, 25, Void >*>(static_cast<CliActor*>(this)));
															#line 6633 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont16(int loopDepth) 
	{
															#line 1530 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		std::string line = raw.get();
															#line 1531 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		config_tr->set("\xff\xff/description"_sr, line);
															#line 6645 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopBody1(int loopDepth) 
	{
															#line 1523 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!(!raw.present() || raw.get().empty()))
															#line 6661 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
			return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1524 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		fprintf(stdout, "Please set a description for the change. Description must be non-empty.\n");
															#line 1526 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		StrictFuture<Optional<std::string>> __when_expr_23 = makeInterruptable(linenoise.read("description: "));
															#line 1526 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 6671 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (__when_expr_23.isReady()) { if (__when_expr_23.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_23.getError(), std::max(0, loopDepth - 2)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopBody1when1(__when_expr_23.get(), loopDepth); };
		static_cast<CliActor*>(this)->actor_wait_state = 24;
															#line 1526 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		__when_expr_23.addCallbackAndClear(static_cast<ActorCallback< CliActor, 23, Optional<std::string> >*>(static_cast<CliActor*>(this)));
															#line 6676 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1break1(int loopDepth) 
	{
		try {
			return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont16(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopBody1cont1(int loopDepth) 
	{
															#line 1528 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		raw = rawline;
															#line 6698 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopBody1when1(Optional<std::string> const& __rawline,int loopDepth) 
	{
															#line 1526 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		rawline = __rawline;
															#line 6707 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopBody1when1(Optional<std::string> && __rawline,int loopDepth) 
	{
		rawline = std::move(__rawline);
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose24() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 23, Optional<std::string> >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 23, Optional<std::string> >*,Optional<std::string> const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 23);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose24();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 23);

	}
	void a_callback_fire(ActorCallback< CliActor, 23, Optional<std::string> >*,Optional<std::string> && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 23);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose24();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 23);

	}
	void a_callback_error(ActorCallback< CliActor, 23, Optional<std::string> >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 23);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose24();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 23);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont1(int loopDepth) 
	{
															#line 1543 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		isCommitDesc = false;
															#line 1544 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		intrans = false;
															#line 1545 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		transtype = TransType::None;
															#line 1546 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		options = &globalOptions;
															#line 6792 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont13(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose25() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 24, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 24, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 24);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose25();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 24);

	}
	void a_callback_fire(ActorCallback< CliActor, 24, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 24);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose25();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 24);

	}
	void a_callback_error(ActorCallback< CliActor, 24, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 24);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose25();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 24);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont3(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont3(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15when2(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15when2(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose26() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 25, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 25, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 25);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose26();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15when2(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 25);

	}
	void a_callback_fire(ActorCallback< CliActor, 25, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 25);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose26();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont15when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 25);

	}
	void a_callback_error(ActorCallback< CliActor, 25, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 25);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose26();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 25);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17(int loopDepth) 
	{
															#line 1560 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "reset"))
															#line 6975 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1561 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() != 1)
															#line 6979 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1562 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1563 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 6985 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
			else
			{
															#line 1564 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (!intrans)
															#line 6991 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1565 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					fprintf(stderr, "ERROR: No active transaction\n");
															#line 1566 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					is_error = true;
															#line 6997 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
				else
				{
															#line 1568 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (transtype == TransType::Config)
															#line 7003 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1569 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						config_tr->reset();
															#line 7007 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 1571 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						tr->reset();
															#line 1572 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						activeOptions = FdbOptions(globalOptions);
															#line 1573 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						options = &activeOptions;
															#line 1574 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						options->apply(tr);
															#line 7019 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
															#line 1576 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					isCommitDesc = false;
															#line 1577 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					transtype = TransType::None;
															#line 1578 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					printf("Transaction reset\n");
															#line 7027 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
			}
			return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
		}
															#line 1583 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "rollback"))
															#line 7034 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1584 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() != 1)
															#line 7038 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1585 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1586 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 7044 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
			else
			{
															#line 1587 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (!intrans)
															#line 7050 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1588 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					fprintf(stderr, "ERROR: No active transaction\n");
															#line 1589 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					is_error = true;
															#line 7056 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
				else
				{
															#line 1591 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					intrans = false;
															#line 1592 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					options = &globalOptions;
															#line 1593 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					printf("Transaction rolled back\n");
															#line 7066 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
			}
			return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
		}
															#line 1598 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "get"))
															#line 7073 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1599 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() != 2)
															#line 7077 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1600 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1601 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 7083 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont14(loopDepth);
			}
			else
			{
															#line 1603 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (intrans)
															#line 7090 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1604 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (transtype == TransType::None)
															#line 7094 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1605 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						transtype = TransType::Db;
															#line 7098 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 1606 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (transtype == TransType::Config)
															#line 7104 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 1607 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							fprintf(stderr, "ERROR: Cannot perform get in configuration transaction\n");
															#line 1608 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							is_error = true;
															#line 7110 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
						}
					}
				}
															#line 1612 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				valueF = getTransaction(db, tenant, tr, options, intrans)->get(tokens[1]);
															#line 1614 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				StrictFuture<Optional<Standalone<StringRef>>> __when_expr_27 = makeInterruptable(safeThreadFutureToFuture(valueF));
															#line 1614 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7121 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				if (__when_expr_27.isReady()) { if (__when_expr_27.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_27.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17when1(__when_expr_27.get(), loopDepth); };
				static_cast<CliActor*>(this)->actor_wait_state = 28;
															#line 1614 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				__when_expr_27.addCallbackAndClear(static_cast<ActorCallback< CliActor, 27, Optional<Standalone<StringRef>> >*>(static_cast<CliActor*>(this)));
															#line 7126 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont18(bool const& _result,int loopDepth) 
	{
															#line 1554 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 7141 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 7145 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont18(bool && _result,int loopDepth) 
	{
															#line 1554 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 7155 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1555 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 7159 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont5when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont18(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont5when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont18(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose27() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 26, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 26, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 26);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose27();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 26);

	}
	void a_callback_fire(ActorCallback< CliActor, 26, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 26);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose27();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 26);

	}
	void a_callback_error(ActorCallback< CliActor, 26, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 26);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose27();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 26);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont1(int loopDepth) 
	{
															#line 1624 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "getlocation"))
															#line 7244 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1625 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_28 = makeInterruptable(getLocationCommandActor(localDb, tokens));
															#line 1625 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7250 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_28.isReady()) { if (__when_expr_28.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_28.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont1when1(__when_expr_28.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 29;
															#line 1625 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_28.addCallbackAndClear(static_cast<ActorCallback< CliActor, 28, bool >*>(static_cast<CliActor*>(this)));
															#line 7255 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont14(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont16(Optional<Standalone<StringRef>> const& v,int loopDepth) 
	{
															#line 1616 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (v.present())
															#line 7275 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1617 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("`%s' is `%s'\n", printable(tokens[1]).c_str(), printable(v.get()).c_str());
															#line 7279 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 1619 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("`%s': not found\n", printable(tokens[1]).c_str());
															#line 7285 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont16(Optional<Standalone<StringRef>> && v,int loopDepth) 
	{
															#line 1616 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (v.present())
															#line 7295 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1617 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("`%s' is `%s'\n", printable(tokens[1]).c_str(), printable(v.get()).c_str());
															#line 7299 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 1619 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("`%s': not found\n", printable(tokens[1]).c_str());
															#line 7305 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont14(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17when1(Optional<Standalone<StringRef>> const& v,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont16(v, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17when1(Optional<Standalone<StringRef>> && v,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont16(std::move(v), loopDepth);

		return loopDepth;
	}
	void a_exitChoose28() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 27, Optional<Standalone<StringRef>> >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 27, Optional<Standalone<StringRef>> >*,Optional<Standalone<StringRef>> const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 27);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose28();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 27);

	}
	void a_callback_fire(ActorCallback< CliActor, 27, Optional<Standalone<StringRef>> >*,Optional<Standalone<StringRef>> && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 27);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose28();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 27);

	}
	void a_callback_error(ActorCallback< CliActor, 27, Optional<Standalone<StringRef>> >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 27);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose28();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 27);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21(int loopDepth) 
	{
															#line 1631 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "getall"))
															#line 7390 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1632 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Version> __when_expr_29 = makeInterruptable( safeThreadFutureToFuture(getTransaction(db, tenant, tr, options, intrans)->getReadVersion()));
															#line 1632 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7396 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_29.isReady()) { if (__when_expr_29.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_29.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21when1(__when_expr_29.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 30;
															#line 1632 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_29.addCallbackAndClear(static_cast<ActorCallback< CliActor, 29, Version >*>(static_cast<CliActor*>(this)));
															#line 7401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont22(bool const& _result,int loopDepth) 
	{
															#line 1626 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 7415 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1627 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 7419 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont22(bool && _result,int loopDepth) 
	{
															#line 1626 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 7429 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1627 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 7433 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont22(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont22(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose29() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 28, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 28, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 28);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose29();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 28);

	}
	void a_callback_fire(ActorCallback< CliActor, 28, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 28);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose29();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 28);

	}
	void a_callback_error(ActorCallback< CliActor, 28, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 28);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose29();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 28);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont1(int loopDepth) 
	{
															#line 1640 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "checkall"))
															#line 7518 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1641 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_31 = makeInterruptable(checkallCommandActor(localDb, tokens));
															#line 1641 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7524 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_31.isReady()) { if (__when_expr_31.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_31.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont1when1(__when_expr_31.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 32;
															#line 1641 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_31.addCallbackAndClear(static_cast<ActorCallback< CliActor, 31, bool >*>(static_cast<CliActor*>(this)));
															#line 7529 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont2(Version const& _v,int loopDepth) 
	{
															#line 1634 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		StrictFuture<bool> __when_expr_30 = makeInterruptable(getallCommandActor(localDb, tokens, _v));
															#line 1634 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7545 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (__when_expr_30.isReady()) { if (__when_expr_30.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_30.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont2when1(__when_expr_30.get(), loopDepth); };
		static_cast<CliActor*>(this)->actor_wait_state = 31;
															#line 1634 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		__when_expr_30.addCallbackAndClear(static_cast<ActorCallback< CliActor, 30, bool >*>(static_cast<CliActor*>(this)));
															#line 7550 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont2(Version && _v,int loopDepth) 
	{
															#line 1634 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		StrictFuture<bool> __when_expr_30 = makeInterruptable(getallCommandActor(localDb, tokens, _v));
															#line 1634 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7561 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (__when_expr_30.isReady()) { if (__when_expr_30.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_30.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont2when1(__when_expr_30.get(), loopDepth); };
		static_cast<CliActor*>(this)->actor_wait_state = 31;
															#line 1634 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		__when_expr_30.addCallbackAndClear(static_cast<ActorCallback< CliActor, 30, bool >*>(static_cast<CliActor*>(this)));
															#line 7566 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21when1(Version const& _v,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont2(_v, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21when1(Version && _v,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont2(std::move(_v), loopDepth);

		return loopDepth;
	}
	void a_exitChoose30() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 29, Version >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 29, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 29);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose30();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 29);

	}
	void a_callback_fire(ActorCallback< CliActor, 29, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 29);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose30();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 29);

	}
	void a_callback_error(ActorCallback< CliActor, 29, Version >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 29);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose30();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 29);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont3(bool const& _result,int loopDepth) 
	{
															#line 1635 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 7650 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1636 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 7654 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont3(bool && _result,int loopDepth) 
	{
															#line 1635 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 7664 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1636 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 7668 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont2when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont3(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont2when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont3(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose31() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 30, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 30, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 30);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose31();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 30);

	}
	void a_callback_fire(ActorCallback< CliActor, 30, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 30);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose31();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 30);

	}
	void a_callback_error(ActorCallback< CliActor, 30, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 30);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose31();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 30);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont6(int loopDepth) 
	{
															#line 1647 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "getversion"))
															#line 7753 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1648 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() != 1)
															#line 7757 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1649 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1650 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 7763 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont11(loopDepth);
			}
			else
			{
															#line 1652 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				StrictFuture<Version> __when_expr_32 = makeInterruptable(safeThreadFutureToFuture( getTransaction(db, tenant, tr, options, intrans)->getReadVersion()));
															#line 1652 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7772 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				if (__when_expr_32.isReady()) { if (__when_expr_32.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_32.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont6when1(__when_expr_32.get(), loopDepth); };
				static_cast<CliActor*>(this)->actor_wait_state = 33;
															#line 1652 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				__when_expr_32.addCallbackAndClear(static_cast<ActorCallback< CliActor, 32, Version >*>(static_cast<CliActor*>(this)));
															#line 7777 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont7(bool const& _result,int loopDepth) 
	{
															#line 1642 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 7792 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1643 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 7796 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont7(bool && _result,int loopDepth) 
	{
															#line 1642 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 7806 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1643 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 7810 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont7(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont7(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose32() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 31, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 31, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 31);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose32();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 31);

	}
	void a_callback_fire(ActorCallback< CliActor, 31, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 31);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose32();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 31);

	}
	void a_callback_error(ActorCallback< CliActor, 31, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 31);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose32();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 31);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10(int loopDepth) 
	{
															#line 1659 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "advanceversion"))
															#line 7895 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1660 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_33 = makeInterruptable(advanceVersionCommandActor(db, tokens));
															#line 1660 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 7901 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_33.isReady()) { if (__when_expr_33.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_33.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10when1(__when_expr_33.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 34;
															#line 1660 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_33.addCallbackAndClear(static_cast<ActorCallback< CliActor, 33, bool >*>(static_cast<CliActor*>(this)));
															#line 7906 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont11(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont13(Version const& v,int loopDepth) 
	{
															#line 1654 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		fmt::print("{}\n", v);
															#line 7926 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont13(Version && v,int loopDepth) 
	{
															#line 1654 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		fmt::print("{}\n", v);
															#line 7935 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont11(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont6when1(Version const& v,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont13(v, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont6when1(Version && v,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont13(std::move(v), loopDepth);

		return loopDepth;
	}
	void a_exitChoose33() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 32, Version >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 32, Version >*,Version const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 32);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose33();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 32);

	}
	void a_callback_fire(ActorCallback< CliActor, 32, Version >*,Version && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 32);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose33();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 32);

	}
	void a_callback_error(ActorCallback< CliActor, 32, Version >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 32);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose33();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 32);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont1(int loopDepth) 
	{
															#line 1666 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "versionepoch"))
															#line 8019 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1667 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_34 = makeInterruptable(versionEpochCommandActor(db, localDb, tokens));
															#line 1667 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8025 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_34.isReady()) { if (__when_expr_34.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_34.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont1when1(__when_expr_34.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 35;
															#line 1667 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_34.addCallbackAndClear(static_cast<ActorCallback< CliActor, 34, bool >*>(static_cast<CliActor*>(this)));
															#line 8030 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont2(bool const& _result,int loopDepth) 
	{
															#line 1661 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8044 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1662 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8048 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont2(bool && _result,int loopDepth) 
	{
															#line 1661 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8058 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1662 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8062 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont2(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont2(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose34() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 33, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 33, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 33);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose34();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 33);

	}
	void a_callback_fire(ActorCallback< CliActor, 33, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 33);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose34();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 33);

	}
	void a_callback_error(ActorCallback< CliActor, 33, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 33);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose34();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 33);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont5(int loopDepth) 
	{
															#line 1673 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "kill"))
															#line 8147 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1674 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			getTransaction(db, managementTenant, tr, options, intrans);
															#line 1675 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_35 = makeInterruptable(killCommandActor(db, tr, tokens, &address_interface));
															#line 1675 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8155 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_35.isReady()) { if (__when_expr_35.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_35.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont5when1(__when_expr_35.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 36;
															#line 1675 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_35.addCallbackAndClear(static_cast<ActorCallback< CliActor, 35, bool >*>(static_cast<CliActor*>(this)));
															#line 8160 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont6(bool const& _result,int loopDepth) 
	{
															#line 1668 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8174 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1669 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8178 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont6(bool && _result,int loopDepth) 
	{
															#line 1668 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8188 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1669 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8192 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont6(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont6(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose35() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 34, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 34, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 34);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose35();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 34);

	}
	void a_callback_fire(ActorCallback< CliActor, 34, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 34);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose35();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 34);

	}
	void a_callback_error(ActorCallback< CliActor, 34, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 34);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose35();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 34);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont9(int loopDepth) 
	{
															#line 1681 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "suspend"))
															#line 8277 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1682 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			getTransaction(db, managementTenant, tr, options, intrans);
															#line 1683 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_36 = makeInterruptable(suspendCommandActor(db, tr, tokens, &address_interface));
															#line 1683 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8285 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_36.isReady()) { if (__when_expr_36.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_36.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont9when1(__when_expr_36.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 37;
															#line 1683 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_36.addCallbackAndClear(static_cast<ActorCallback< CliActor, 36, bool >*>(static_cast<CliActor*>(this)));
															#line 8290 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont10(bool const& _result,int loopDepth) 
	{
															#line 1676 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8304 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1677 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8308 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont10(bool && _result,int loopDepth) 
	{
															#line 1676 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8318 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1677 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8322 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont5when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont10(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont5when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont10(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose36() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 35, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 35, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 35);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose36();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 35);

	}
	void a_callback_fire(ActorCallback< CliActor, 35, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 35);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose36();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 35);

	}
	void a_callback_error(ActorCallback< CliActor, 35, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 35);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose36();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 35);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11(int loopDepth) 
	{
															#line 1689 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "audit_storage"))
															#line 8407 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1690 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<UID> __when_expr_37 = makeInterruptable(auditStorageCommandActor(ccf, tokens));
															#line 1690 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8413 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_37.isReady()) { if (__when_expr_37.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_37.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11when1(__when_expr_37.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 38;
															#line 1690 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_37.addCallbackAndClear(static_cast<ActorCallback< CliActor, 37, UID >*>(static_cast<CliActor*>(this)));
															#line 8418 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont12(bool const& _result,int loopDepth) 
	{
															#line 1684 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8432 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1685 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8436 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont12(bool && _result,int loopDepth) 
	{
															#line 1684 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8446 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1685 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8450 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont9when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont12(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont9when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont12(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose37() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 36, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 36, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 36);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose37();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 36);

	}
	void a_callback_fire(ActorCallback< CliActor, 36, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 36);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose37();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 36);

	}
	void a_callback_error(ActorCallback< CliActor, 36, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 36);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose37();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 36);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont1(int loopDepth) 
	{
															#line 1701 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "get_audit_status"))
															#line 8535 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1702 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_38 = makeInterruptable(getAuditStatusCommandActor(localDb, tokens));
															#line 1702 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8541 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_38.isReady()) { if (__when_expr_38.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_38.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont1when1(__when_expr_38.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 39;
															#line 1702 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_38.addCallbackAndClear(static_cast<ActorCallback< CliActor, 38, bool >*>(static_cast<CliActor*>(this)));
															#line 8546 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont2(UID const& auditId,int loopDepth) 
	{
															#line 1691 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!auditId.isValid())
															#line 8560 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1692 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8564 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 1694 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("%s audit: %s\n", tokencmp(tokens[1], "cancel") ? "Cancelled" : "Started", auditId.toString().c_str());
															#line 8570 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont2(UID && auditId,int loopDepth) 
	{
															#line 1691 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!auditId.isValid())
															#line 8580 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1692 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8584 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 1694 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("%s audit: %s\n", tokencmp(tokens[1], "cancel") ? "Cancelled" : "Started", auditId.toString().c_str());
															#line 8590 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11when1(UID const& auditId,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont2(auditId, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11when1(UID && auditId,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont2(std::move(auditId), loopDepth);

		return loopDepth;
	}
	void a_exitChoose38() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 37, UID >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 37, UID >*,UID const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 37);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose38();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 37);

	}
	void a_callback_fire(ActorCallback< CliActor, 37, UID >*,UID && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 37);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose38();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 37);

	}
	void a_callback_error(ActorCallback< CliActor, 37, UID >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 37);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose38();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 37);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont6(int loopDepth) 
	{
															#line 1709 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "location_metadata"))
															#line 8675 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1710 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_39 = makeInterruptable(locationMetadataCommandActor(localDb, tokens));
															#line 1710 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8681 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_39.isReady()) { if (__when_expr_39.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_39.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont6when1(__when_expr_39.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 40;
															#line 1710 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_39.addCallbackAndClear(static_cast<ActorCallback< CliActor, 39, bool >*>(static_cast<CliActor*>(this)));
															#line 8686 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont7(bool const& _result,int loopDepth) 
	{
															#line 1703 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8700 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1704 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8704 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont7(bool && _result,int loopDepth) 
	{
															#line 1703 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8714 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1704 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8718 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont7(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont7(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose39() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 38, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 38, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 38);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose39();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 38);

	}
	void a_callback_fire(ActorCallback< CliActor, 38, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 38);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose39();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 38);

	}
	void a_callback_error(ActorCallback< CliActor, 38, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 38);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose39();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 38);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10(int loopDepth) 
	{
															#line 1717 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "bulkload"))
															#line 8803 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1718 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<UID> __when_expr_40 = makeInterruptable(bulkLoadCommandActor(localDb, tokens));
															#line 1718 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8809 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_40.isReady()) { if (__when_expr_40.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_40.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10when1(__when_expr_40.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 41;
															#line 1718 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_40.addCallbackAndClear(static_cast<ActorCallback< CliActor, 40, UID >*>(static_cast<CliActor*>(this)));
															#line 8814 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont11(bool const& _result,int loopDepth) 
	{
															#line 1711 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8828 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1712 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8832 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont11(bool && _result,int loopDepth) 
	{
															#line 1711 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 8842 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1712 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 8846 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont6when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont11(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont6when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont11(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose40() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 39, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 39, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 39);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose40();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont6when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 39);

	}
	void a_callback_fire(ActorCallback< CliActor, 39, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 39);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose40();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont6when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 39);

	}
	void a_callback_error(ActorCallback< CliActor, 39, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 39);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose40();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 39);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont1(int loopDepth) 
	{
															#line 1725 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "bulkdump"))
															#line 8931 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1726 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<UID> __when_expr_41 = makeInterruptable(bulkDumpCommandActor(localDb, tokens));
															#line 1726 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 8937 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_41.isReady()) { if (__when_expr_41.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_41.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont1when1(__when_expr_41.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 42;
															#line 1726 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_41.addCallbackAndClear(static_cast<ActorCallback< CliActor, 41, UID >*>(static_cast<CliActor*>(this)));
															#line 8942 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont2(UID const& taskId,int loopDepth) 
	{
															#line 1719 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (taskId.isValid())
															#line 8956 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1720 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("Received Job ID: %s\n", taskId.toString().c_str());
															#line 8960 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont2(UID && taskId,int loopDepth) 
	{
															#line 1719 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (taskId.isValid())
															#line 8970 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1720 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("Received Job ID: %s\n", taskId.toString().c_str());
															#line 8974 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10when1(UID const& taskId,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont2(taskId, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10when1(UID && taskId,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont2(std::move(taskId), loopDepth);

		return loopDepth;
	}
	void a_exitChoose41() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 40, UID >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 40, UID >*,UID const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 40);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose41();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 40);

	}
	void a_callback_fire(ActorCallback< CliActor, 40, UID >*,UID && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 40);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose41();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 40);

	}
	void a_callback_error(ActorCallback< CliActor, 40, UID >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 40);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose41();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 40);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont5(int loopDepth) 
	{
															#line 1733 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "force_recovery_with_data_loss"))
															#line 9059 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1734 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_42 = makeInterruptable(forceRecoveryWithDataLossCommandActor(db, tokens));
															#line 1734 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9065 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_42.isReady()) { if (__when_expr_42.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_42.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont5when1(__when_expr_42.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 43;
															#line 1734 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_42.addCallbackAndClear(static_cast<ActorCallback< CliActor, 42, bool >*>(static_cast<CliActor*>(this)));
															#line 9070 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont6(UID const& jobId,int loopDepth) 
	{
															#line 1727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (jobId.isValid())
															#line 9084 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1728 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("Received Job ID: %s\n", jobId.toString().c_str());
															#line 9088 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont6(UID && jobId,int loopDepth) 
	{
															#line 1727 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (jobId.isValid())
															#line 9098 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1728 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("Received Job ID: %s\n", jobId.toString().c_str());
															#line 9102 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont1when1(UID const& jobId,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont6(jobId, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont1when1(UID && jobId,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont6(std::move(jobId), loopDepth);

		return loopDepth;
	}
	void a_exitChoose42() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 41, UID >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 41, UID >*,UID const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 41);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose42();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 41);

	}
	void a_callback_fire(ActorCallback< CliActor, 41, UID >*,UID && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 41);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose42();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 41);

	}
	void a_callback_error(ActorCallback< CliActor, 41, UID >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 41);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose42();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 41);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont9(int loopDepth) 
	{
															#line 1740 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "maintenance"))
															#line 9187 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1741 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_43 = makeInterruptable(maintenanceCommandActor(db, tokens));
															#line 1741 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9193 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_43.isReady()) { if (__when_expr_43.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_43.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont9when1(__when_expr_43.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 44;
															#line 1741 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_43.addCallbackAndClear(static_cast<ActorCallback< CliActor, 43, bool >*>(static_cast<CliActor*>(this)));
															#line 9198 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont10(bool const& _result,int loopDepth) 
	{
															#line 1735 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 9212 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1736 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 9216 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont10(bool && _result,int loopDepth) 
	{
															#line 1735 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 9226 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1736 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 9230 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont5when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont10(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont5when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont10(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose43() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 42, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 42, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 42);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose43();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 42);

	}
	void a_callback_fire(ActorCallback< CliActor, 42, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 42);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose43();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 42);

	}
	void a_callback_error(ActorCallback< CliActor, 42, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 42);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose43();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 42);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11(int loopDepth) 
	{
															#line 1747 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "consistencycheck"))
															#line 9315 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1748 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			getTransaction(db, managementTenant, tr, options, intrans);
															#line 1749 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_44 = makeInterruptable(consistencyCheckCommandActor(tr, tokens, intrans));
															#line 1749 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9323 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_44.isReady()) { if (__when_expr_44.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_44.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11when1(__when_expr_44.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 45;
															#line 1749 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_44.addCallbackAndClear(static_cast<ActorCallback< CliActor, 44, bool >*>(static_cast<CliActor*>(this)));
															#line 9328 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont12(bool const& _result,int loopDepth) 
	{
															#line 1742 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 9342 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1743 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 9346 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont12(bool && _result,int loopDepth) 
	{
															#line 1742 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 9356 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1743 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 9360 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont9when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont12(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont9when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont12(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose44() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 43, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 43, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 43);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose44();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 43);

	}
	void a_callback_fire(ActorCallback< CliActor, 43, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 43);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose44();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 43);

	}
	void a_callback_error(ActorCallback< CliActor, 43, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 43);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose44();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 43);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont1(int loopDepth) 
	{
															#line 1755 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "consistencyscan"))
															#line 9445 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1756 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_45 = makeInterruptable(consistencyScanCommandActor(localDb, tokens));
															#line 1756 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9451 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_45.isReady()) { if (__when_expr_45.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_45.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont1when1(__when_expr_45.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 46;
															#line 1756 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_45.addCallbackAndClear(static_cast<ActorCallback< CliActor, 45, bool >*>(static_cast<CliActor*>(this)));
															#line 9456 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont2(bool const& _result,int loopDepth) 
	{
															#line 1750 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 9470 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1751 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 9474 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont2(bool && _result,int loopDepth) 
	{
															#line 1750 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 9484 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1751 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 9488 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont2(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont2(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose45() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 44, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 44, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 44);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose45();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 44);

	}
	void a_callback_fire(ActorCallback< CliActor, 44, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 44);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose45();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 44);

	}
	void a_callback_error(ActorCallback< CliActor, 44, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 44);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose45();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 44);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont5(int loopDepth) 
	{
															#line 1762 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "profile"))
															#line 9573 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1763 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			getTransaction(db, managementTenant, tr, options, intrans);
															#line 1764 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_46 = makeInterruptable(profileCommandActor(localDb, tr, tokens, intrans));
															#line 1764 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9581 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_46.isReady()) { if (__when_expr_46.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_46.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont5when1(__when_expr_46.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 47;
															#line 1764 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_46.addCallbackAndClear(static_cast<ActorCallback< CliActor, 46, bool >*>(static_cast<CliActor*>(this)));
															#line 9586 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont6(bool const& _result,int loopDepth) 
	{
															#line 1757 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 9600 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1758 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 9604 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont6(bool && _result,int loopDepth) 
	{
															#line 1757 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 9614 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1758 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 9618 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont6(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont6(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose46() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 45, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 45, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 45);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose46();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 45);

	}
	void a_callback_fire(ActorCallback< CliActor, 45, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 45);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose46();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 45);

	}
	void a_callback_error(ActorCallback< CliActor, 45, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 45);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose46();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 45);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont9(int loopDepth) 
	{
															#line 1770 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "expensive_data_check"))
															#line 9703 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1771 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			getTransaction(db, managementTenant, tr, options, intrans);
															#line 1772 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_47 = makeInterruptable(expensiveDataCheckCommandActor(db, tr, tokens, &address_interface));
															#line 1772 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9711 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_47.isReady()) { if (__when_expr_47.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_47.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont9when1(__when_expr_47.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 48;
															#line 1772 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_47.addCallbackAndClear(static_cast<ActorCallback< CliActor, 47, bool >*>(static_cast<CliActor*>(this)));
															#line 9716 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont10(bool const& _result,int loopDepth) 
	{
															#line 1765 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 9730 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1766 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 9734 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont10(bool && _result,int loopDepth) 
	{
															#line 1765 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 9744 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1766 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 9748 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont5when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont10(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont5when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont10(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose47() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 46, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 46, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 46);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose47();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 46);

	}
	void a_callback_fire(ActorCallback< CliActor, 46, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 46);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose47();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 46);

	}
	void a_callback_error(ActorCallback< CliActor, 46, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 46);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose47();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 46);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11(int loopDepth) 
	{
															#line 1779 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "getrange") || tokencmp(tokens[0], "getrangekeys"))
															#line 9833 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1781 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() < 2 || tokens.size() > 4)
															#line 9837 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1782 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1783 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 9843 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont2(loopDepth);
			}
			else
			{
															#line 1785 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				limit = int();
															#line 1786 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				bool valid = true;
															#line 1787 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (intrans)
															#line 9854 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1788 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (transtype == TransType::None)
															#line 9858 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1789 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						transtype = TransType::Db;
															#line 9862 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 1790 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (transtype == TransType::Config)
															#line 9868 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 1791 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							fprintf( stderr, "ERROR: Cannot perform getrange or getrangekeys in configuration transaction\n");
															#line 1794 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							is_error = true;
															#line 9874 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
						}
					}
				}
															#line 1798 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (tokens.size() == 4)
															#line 9881 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1803 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (tokens[3].size() > 9)
															#line 9885 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1804 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						fprintf(stderr, "ERROR: bad limit\n");
															#line 1805 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						is_error = true;
															#line 9891 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
					}
															#line 1808 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					limit = 0;
															#line 1809 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					int place = 1;
															#line 1810 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					for(int i = tokens[3].size();i > 0;i--) {
															#line 1811 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						int val = int(tokens[3][i - 1]) - int('0');
															#line 1812 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (val < 0 || val > 9)
															#line 9904 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 1813 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							valid = false;
															#line 9908 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							break;
						}
															#line 1816 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						limit += val * place;
															#line 1817 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						place *= 10;
															#line 9915 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
															#line 1819 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (!valid)
															#line 9919 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1820 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						fprintf(stderr, "ERROR: bad limit\n");
															#line 1821 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						is_error = true;
															#line 9925 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
					}
				}
				else
				{
															#line 1825 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					limit = 25;
															#line 9933 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
															#line 1828 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				Standalone<StringRef> endKey;
															#line 1829 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (tokens.size() >= 3)
															#line 9939 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1830 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					endKey = tokens[2];
															#line 9943 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
				else
				{
															#line 1831 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (tokens[1].size() == 0)
															#line 9949 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1832 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						endKey = normalKeys.end;
															#line 9953 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 1833 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (tokens[1] == systemKeys.begin)
															#line 9959 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 1834 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							endKey = systemKeys.end;
															#line 9963 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						}
						else
						{
															#line 1835 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							if (tokens[1] >= allKeys.end)
															#line 9969 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							{
															#line 1836 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
								return a_body1cont7loopBody1cont1Catch1(key_outside_legal_range(), std::max(0, loopDepth - 1));
															#line 9973 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							}
							else
							{
															#line 1838 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
								endKey = strinc(tokens[1]);
															#line 9979 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							}
						}
					}
				}
															#line 1841 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				getTransaction(db, tenant, tr, options, intrans);
															#line 1842 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				kvsF = tr->getRange(KeyRangeRef(tokens[1], endKey), limit);
															#line 1843 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				StrictFuture<RangeResult> __when_expr_48 = makeInterruptable(safeThreadFutureToFuture(kvsF));
															#line 1843 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 9992 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				if (__when_expr_48.isReady()) { if (__when_expr_48.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_48.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11when1(__when_expr_48.get(), loopDepth); };
				static_cast<CliActor*>(this)->actor_wait_state = 49;
															#line 1843 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				__when_expr_48.addCallbackAndClear(static_cast<ActorCallback< CliActor, 48, RangeResult >*>(static_cast<CliActor*>(this)));
															#line 9997 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont12(bool const& _result,int loopDepth) 
	{
															#line 1774 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 10012 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1775 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 10016 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont12(bool && _result,int loopDepth) 
	{
															#line 1774 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 10026 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1775 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 10030 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont9when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont12(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont9when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont12(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose48() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 47, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 47, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 47);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose48();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 47);

	}
	void a_callback_fire(ActorCallback< CliActor, 47, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 47);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose48();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 47);

	}
	void a_callback_error(ActorCallback< CliActor, 47, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 47);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose48();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 47);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont1(int loopDepth) 
	{
															#line 1858 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "writemode"))
															#line 10115 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1859 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() != 2)
															#line 10119 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1860 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1861 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 10125 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
			else
			{
															#line 1863 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (tokencmp(tokens[1], "on"))
															#line 10131 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1864 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					writeMode = true;
															#line 10135 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
				else
				{
															#line 1865 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (tokencmp(tokens[1], "off"))
															#line 10141 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1866 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						writeMode = false;
															#line 10145 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 1868 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						printUsage(tokens[0]);
															#line 1869 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						is_error = true;
															#line 10153 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
				}
			}
			return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
		}
															#line 1875 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "set"))
															#line 10161 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1876 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (!writeMode)
															#line 10165 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1877 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				fprintf(stderr, "ERROR: writemode must be enabled to set or clear keys in the database.\n");
															#line 1878 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 10171 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
			}
															#line 1882 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() != 3)
															#line 10176 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1883 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1884 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 10182 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont36(loopDepth);
			}
			else
			{
															#line 1886 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (intrans)
															#line 10189 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1887 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (transtype == TransType::None)
															#line 10193 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1888 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						transtype = TransType::Db;
															#line 10197 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 1889 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (transtype == TransType::Config)
															#line 10203 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 1890 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							fprintf(stderr, "ERROR: Cannot perform set in configuration transaction\n");
															#line 1891 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							is_error = true;
															#line 10209 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
						}
					}
				}
															#line 1895 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				getTransaction(db, tenant, tr, options, intrans);
															#line 1896 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				tr->set(tokens[1], tokens[2]);
															#line 1898 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (!intrans)
															#line 10220 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1899 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					StrictFuture<Void> __when_expr_49 = commitTransaction(tr);
															#line 1899 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10226 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					if (__when_expr_49.isReady()) { if (__when_expr_49.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_49.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont1when1(__when_expr_49.get(), loopDepth); };
					static_cast<CliActor*>(this)->actor_wait_state = 50;
															#line 1899 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					__when_expr_49.addCallbackAndClear(static_cast<ActorCallback< CliActor, 49, Void >*>(static_cast<CliActor*>(this)));
															#line 10231 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont39(loopDepth);
				}
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont2(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont4(RangeResult const& kvs,int loopDepth) 
	{
															#line 1845 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		printf("\nRange limited to %d keys\n", limit);
															#line 1846 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		for(auto iter = kvs.begin();iter < kvs.end();iter++) {
															#line 1847 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokencmp(tokens[0], "getrangekeys"))
															#line 10261 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1848 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printf("`%s'\n", printable((*iter).key).c_str());
															#line 10265 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
			else
			{
															#line 1850 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printf( "`%s' is `%s'\n", printable((*iter).key).c_str(), printable((*iter).value).c_str());
															#line 10271 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
		}
															#line 1853 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		printf("\n");
															#line 10276 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont4(RangeResult && kvs,int loopDepth) 
	{
															#line 1845 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		printf("\nRange limited to %d keys\n", limit);
															#line 1846 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		for(auto iter = kvs.begin();iter < kvs.end();iter++) {
															#line 1847 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokencmp(tokens[0], "getrangekeys"))
															#line 10289 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1848 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printf("`%s'\n", printable((*iter).key).c_str());
															#line 10293 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
			else
			{
															#line 1850 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printf( "`%s' is `%s'\n", printable((*iter).key).c_str(), printable((*iter).value).c_str());
															#line 10299 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
		}
															#line 1853 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		printf("\n");
															#line 10304 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11when1(RangeResult const& kvs,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont4(kvs, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11when1(RangeResult && kvs,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont4(std::move(kvs), loopDepth);

		return loopDepth;
	}
	void a_exitChoose49() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 48, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 48, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 48);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose49();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 48);

	}
	void a_callback_fire(ActorCallback< CliActor, 48, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 48);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose49();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 48);

	}
	void a_callback_error(ActorCallback< CliActor, 48, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 48);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose49();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 48);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28(int loopDepth) 
	{
															#line 1905 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		raw_desc = Optional<std::string>();
															#line 1906 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "setknob"))
															#line 10390 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1907 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() > 4 || tokens.size() < 3)
															#line 10394 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1908 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1909 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 10400 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont2(loopDepth);
			}
			else
			{
															#line 1911 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (intrans)
															#line 10407 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1912 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (transtype == TransType::None)
															#line 10411 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1913 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						transtype = TransType::Config;
															#line 10415 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 1914 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (transtype == TransType::Db)
															#line 10421 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 1915 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							fprintf(stderr, "ERROR: Cannot perform setknob in database transaction\n");
															#line 1916 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							is_error = true;
															#line 1917 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							isCommitDesc = false;
															#line 10429 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
						}
					}
				}
															#line 1921 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				Tuple t;
															#line 1922 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (tokens.size() == 4)
															#line 10438 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1923 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					t.append(tokens[3]);
															#line 10442 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
				else
				{
															#line 1925 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					t.appendNull();
															#line 10448 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
															#line 1927 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				t.append(tokens[1]);
															#line 1928 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				getTransaction(configDb, tenant, config_tr, options, intrans);
															#line 1930 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				config_tr->set(t.pack(), tokens[2]);
															#line 1931 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (!intrans)
															#line 10458 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1933 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					raw_desc.reset();
															#line 1934 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					warn.cancel();
															#line 1935 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					;
															#line 10466 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopHead1(loopDepth);
				}
				else
				{
															#line 1948 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					isCommitDesc = true;
															#line 10473 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont4(loopDepth);
				}
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont36(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont39(int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont36(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont44(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont39(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont44(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont39(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont44(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont44(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose50() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 49, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 49, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 49);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose50();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 49);

	}
	void a_callback_fire(ActorCallback< CliActor, 49, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 49);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose50();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 49);

	}
	void a_callback_error(ActorCallback< CliActor, 49, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 49);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose50();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 49);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont1(int loopDepth) 
	{
															#line 1954 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "getknob"))
															#line 10588 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1955 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() > 3 || tokens.size() < 2)
															#line 10592 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1956 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1957 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 10598 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont15(loopDepth);
			}
			else
			{
															#line 1959 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (intrans)
															#line 10605 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1960 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (transtype == TransType::None)
															#line 10609 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 1961 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						transtype = TransType::Config;
															#line 10613 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 1962 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (transtype == TransType::Db)
															#line 10619 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 1963 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							fprintf(stderr, "ERROR: Cannot perform getknob in database transaction\n");
															#line 1964 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							is_error = true;
															#line 10625 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
						}
					}
				}
															#line 1968 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				Tuple t;
															#line 1969 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (tokens.size() == 2)
															#line 10634 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 1970 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					t.appendNull();
															#line 10638 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
				else
				{
															#line 1972 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					t.append(tokens[2]);
															#line 10644 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
															#line 1974 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				t.append(tokens[1]);
															#line 1975 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				valueF_knob = getTransaction(configDb, tenant, config_tr, options, intrans)->get(t.pack());
															#line 1977 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				StrictFuture<Optional<Standalone<StringRef>>> __when_expr_52 = makeInterruptable(safeThreadFutureToFuture(valueF_knob));
															#line 1977 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10654 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				if (__when_expr_52.isReady()) { if (__when_expr_52.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_52.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont1when1(__when_expr_52.get(), loopDepth); };
				static_cast<CliActor*>(this)->actor_wait_state = 53;
															#line 1977 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				__when_expr_52.addCallbackAndClear(static_cast<ActorCallback< CliActor, 52, Optional<Standalone<StringRef>> >*>(static_cast<CliActor*>(this)));
															#line 10659 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = 0;
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont2(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont4(int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont11(int loopDepth) 
	{
															#line 1942 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		std::string line = raw_desc.get();
															#line 1943 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		config_tr->set("\xff\xff/description"_sr, line);
															#line 1944 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		warn = checkStatus( timeWarning(5.0, "\nWARNING: Long delay (Ctrl-C to interrupt)\n"), db, localDb);
															#line 1946 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		StrictFuture<Void> __when_expr_51 = commitTransaction(config_tr);
															#line 1946 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 10694 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (__when_expr_51.isReady()) { if (__when_expr_51.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_51.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont11when1(__when_expr_51.get(), loopDepth); };
		static_cast<CliActor*>(this)->actor_wait_state = 52;
															#line 1946 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		__when_expr_51.addCallbackAndClear(static_cast<ActorCallback< CliActor, 51, Void >*>(static_cast<CliActor*>(this)));
															#line 10699 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopBody1(int loopDepth) 
	{
															#line 1935 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!(!raw_desc.present() || raw_desc.get().empty()))
															#line 10715 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
			return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1936 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		fprintf(stdout, "Please set a description for the change. Description must be non-empty\n");
															#line 1938 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		StrictFuture<Optional<std::string>> __when_expr_50 = makeInterruptable(linenoise.read("description: "));
															#line 1938 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 10725 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (__when_expr_50.isReady()) { if (__when_expr_50.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_50.getError(), std::max(0, loopDepth - 2)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopBody1when1(__when_expr_50.get(), loopDepth); };
		static_cast<CliActor*>(this)->actor_wait_state = 51;
															#line 1938 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		__when_expr_50.addCallbackAndClear(static_cast<ActorCallback< CliActor, 50, Optional<std::string> >*>(static_cast<CliActor*>(this)));
															#line 10730 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28break1(int loopDepth) 
	{
		try {
			return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont11(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopBody1cont1(Optional<std::string> const& rawline_knob,int loopDepth) 
	{
															#line 1940 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		raw_desc = rawline_knob;
															#line 10752 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopBody1cont1(Optional<std::string> && rawline_knob,int loopDepth) 
	{
															#line 1940 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		raw_desc = rawline_knob;
															#line 10761 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopBody1when1(Optional<std::string> const& rawline_knob,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopBody1cont1(rawline_knob, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopBody1when1(Optional<std::string> && rawline_knob,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopBody1cont1(std::move(rawline_knob), loopDepth);

		return loopDepth;
	}
	void a_exitChoose51() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 50, Optional<std::string> >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 50, Optional<std::string> >*,Optional<std::string> const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 50);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose51();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 50);

	}
	void a_callback_fire(ActorCallback< CliActor, 50, Optional<std::string> >*,Optional<std::string> && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 50);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose51();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 50);

	}
	void a_callback_error(ActorCallback< CliActor, 50, Optional<std::string> >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 50);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose51();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 50);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont11cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont11cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont11cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont11cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose52() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 51, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 51, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 51);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose52();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 51);

	}
	void a_callback_fire(ActorCallback< CliActor, 51, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 51);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose52();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 51);

	}
	void a_callback_error(ActorCallback< CliActor, 51, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 51);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose52();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 51);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14(int loopDepth) 
	{
															#line 1994 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "clearknob"))
															#line 10932 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1995 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() > 3 || tokens.size() < 2)
															#line 10936 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 1996 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 1997 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 10942 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont2(loopDepth);
			}
			else
			{
															#line 1999 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (intrans)
															#line 10949 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2000 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (transtype == TransType::None)
															#line 10953 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 2001 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						transtype = TransType::Config;
															#line 10957 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 2002 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (transtype == TransType::Db)
															#line 10963 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 2003 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							fprintf(stderr, "ERROR: Cannot perform clearknob in database transaction\n");
															#line 2004 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							is_error = true;
															#line 10969 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
						}
					}
				}
															#line 2008 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				Tuple t;
															#line 2009 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (tokens.size() == 2)
															#line 10978 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2010 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					t.appendNull();
															#line 10982 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
				else
				{
															#line 2012 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					t.append(tokens[2]);
															#line 10988 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
															#line 2014 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				t.append(tokens[1]);
															#line 2015 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				getTransaction(configDb, tenant, config_tr, options, intrans);
															#line 2017 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				config_tr->clear(t.pack());
															#line 2018 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (!intrans)
															#line 10998 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2020 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					raw_desc.reset();
															#line 2021 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					warn.cancel();
															#line 2022 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					;
															#line 11006 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopHead1(loopDepth);
				}
				else
				{
															#line 2035 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					isCommitDesc = true;
															#line 11013 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont4(loopDepth);
				}
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont15(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont17(Optional<Standalone<StringRef>> const& v,int loopDepth) 
	{
															#line 1979 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		std::string knob_class = printable(tokens[1]);
															#line 1980 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokens.size() == 3)
															#line 11037 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1981 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			std::string config_class = (" in configuration class " + printable(tokens[2]));
															#line 1982 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			knob_class += config_class;
															#line 11043 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
															#line 1984 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (v.present())
															#line 11047 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1985 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("`%s' is `%s'\n", knob_class.c_str(), Tuple::tupleToString(Tuple::unpack(v.get())).c_str());
															#line 11051 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 1989 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("`%s' is not found\n", knob_class.c_str());
															#line 11057 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont17(Optional<Standalone<StringRef>> && v,int loopDepth) 
	{
															#line 1979 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		std::string knob_class = printable(tokens[1]);
															#line 1980 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokens.size() == 3)
															#line 11069 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1981 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			std::string config_class = (" in configuration class " + printable(tokens[2]));
															#line 1982 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			knob_class += config_class;
															#line 11075 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
															#line 1984 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (v.present())
															#line 11079 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 1985 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("`%s' is `%s'\n", knob_class.c_str(), Tuple::tupleToString(Tuple::unpack(v.get())).c_str());
															#line 11083 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 1989 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("`%s' is not found\n", knob_class.c_str());
															#line 11089 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont1when1(Optional<Standalone<StringRef>> const& v,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont17(v, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont1when1(Optional<Standalone<StringRef>> && v,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont17(std::move(v), loopDepth);

		return loopDepth;
	}
	void a_exitChoose53() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 52, Optional<Standalone<StringRef>> >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 52, Optional<Standalone<StringRef>> >*,Optional<Standalone<StringRef>> const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 52);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose53();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 52);

	}
	void a_callback_fire(ActorCallback< CliActor, 52, Optional<Standalone<StringRef>> >*,Optional<Standalone<StringRef>> && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 52);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose53();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 52);

	}
	void a_callback_error(ActorCallback< CliActor, 52, Optional<Standalone<StringRef>> >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 52);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose53();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 52);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont1(int loopDepth) 
	{
															#line 2041 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "clear"))
															#line 11174 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2042 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (!writeMode)
															#line 11178 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2043 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				fprintf(stderr, "ERROR: writemode must be enabled to set or clear keys in the database.\n");
															#line 2044 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 11184 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
			}
															#line 2048 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() != 2)
															#line 11189 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2049 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 2050 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 11195 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont15(loopDepth);
			}
			else
			{
															#line 2052 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (intrans)
															#line 11202 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2053 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (transtype == TransType::None)
															#line 11206 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 2054 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						transtype = TransType::Db;
															#line 11210 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 2055 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (transtype == TransType::Config)
															#line 11216 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 2056 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							fprintf(stderr, "ERROR: Cannot perform clear in configuration transaction\n");
															#line 2057 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							is_error = true;
															#line 11222 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
						}
					}
				}
															#line 2061 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				getTransaction(db, tenant, tr, options, intrans);
															#line 2062 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				tr->clear(tokens[1]);
															#line 2064 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (!intrans)
															#line 11233 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2065 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					StrictFuture<Void> __when_expr_55 = commitTransaction(tr);
															#line 2065 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11239 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					if (__when_expr_55.isReady()) { if (__when_expr_55.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_55.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont1when1(__when_expr_55.get(), loopDepth); };
					static_cast<CliActor*>(this)->actor_wait_state = 56;
															#line 2065 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					__when_expr_55.addCallbackAndClear(static_cast<ActorCallback< CliActor, 55, Void >*>(static_cast<CliActor*>(this)));
															#line 11244 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont18(loopDepth);
				}
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont2(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont4(int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont11(int loopDepth) 
	{
															#line 2029 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		std::string line = raw_desc.get();
															#line 2030 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		config_tr->set("\xff\xff/description"_sr, line);
															#line 2031 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		warn = checkStatus( timeWarning(5.0, "\nWARNING: Long delay (Ctrl-C to interrupt)\n"), db, localDb);
															#line 2033 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		StrictFuture<Void> __when_expr_54 = commitTransaction(config_tr);
															#line 2033 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11284 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (__when_expr_54.isReady()) { if (__when_expr_54.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_54.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont11when1(__when_expr_54.get(), loopDepth); };
		static_cast<CliActor*>(this)->actor_wait_state = 55;
															#line 2033 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		__when_expr_54.addCallbackAndClear(static_cast<ActorCallback< CliActor, 54, Void >*>(static_cast<CliActor*>(this)));
															#line 11289 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopBody1(int loopDepth) 
	{
															#line 2022 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!(!raw_desc.present() || raw_desc.get().empty()))
															#line 11305 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
			return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 2023 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		fprintf(stdout, "Please set a description for the change. Description must be non-empty\n");
															#line 2025 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		StrictFuture<Optional<std::string>> __when_expr_53 = makeInterruptable(linenoise.read("description: "));
															#line 2025 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 11315 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (__when_expr_53.isReady()) { if (__when_expr_53.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_53.getError(), std::max(0, loopDepth - 2)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopBody1when1(__when_expr_53.get(), loopDepth); };
		static_cast<CliActor*>(this)->actor_wait_state = 54;
															#line 2025 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		__when_expr_53.addCallbackAndClear(static_cast<ActorCallback< CliActor, 53, Optional<std::string> >*>(static_cast<CliActor*>(this)));
															#line 11320 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14break1(int loopDepth) 
	{
		try {
			return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont11(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopBody1cont1(Optional<std::string> const& rawline_knob,int loopDepth) 
	{
															#line 2027 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		raw_desc = rawline_knob;
															#line 11342 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopBody1cont1(Optional<std::string> && rawline_knob,int loopDepth) 
	{
															#line 2027 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		raw_desc = rawline_knob;
															#line 11351 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopBody1when1(Optional<std::string> const& rawline_knob,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopBody1cont1(rawline_knob, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopBody1when1(Optional<std::string> && rawline_knob,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopBody1cont1(std::move(rawline_knob), loopDepth);

		return loopDepth;
	}
	void a_exitChoose54() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 53, Optional<std::string> >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 53, Optional<std::string> >*,Optional<std::string> const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 53);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose54();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 53);

	}
	void a_callback_fire(ActorCallback< CliActor, 53, Optional<std::string> >*,Optional<std::string> && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 53);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose54();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 53);

	}
	void a_callback_error(ActorCallback< CliActor, 53, Optional<std::string> >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 53);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose54();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 53);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont11cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont11cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont11when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont11cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont11when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont11cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose55() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 54, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 54, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 54);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose55();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 54);

	}
	void a_callback_fire(ActorCallback< CliActor, 54, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 54);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose55();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 54);

	}
	void a_callback_error(ActorCallback< CliActor, 54, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 54);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose55();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 54);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14(int loopDepth) 
	{
															#line 2071 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "clearrange"))
															#line 11522 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2072 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (!writeMode)
															#line 11526 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2073 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				fprintf(stderr, "ERROR: writemode must be enabled to set or clear keys in the database.\n");
															#line 2074 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 11532 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
			}
															#line 2078 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() != 3)
															#line 11537 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2079 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 2080 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 11543 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont2(loopDepth);
			}
			else
			{
															#line 2082 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (intrans)
															#line 11550 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2083 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (transtype == TransType::None)
															#line 11554 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 2084 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						transtype = TransType::Db;
															#line 11558 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 2085 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (transtype == TransType::Config)
															#line 11564 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 2086 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							fprintf(stderr, "ERROR: Cannot perform clearrange in configuration transaction\n");
															#line 2087 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							is_error = true;
															#line 11570 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
						}
					}
				}
															#line 2091 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				getTransaction(db, tenant, tr, options, intrans);
															#line 2092 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				tr->clear(KeyRangeRef(tokens[1], tokens[2]));
															#line 2094 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (!intrans)
															#line 11581 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2095 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					StrictFuture<Void> __when_expr_56 = commitTransaction(tr);
															#line 2095 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11587 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					if (__when_expr_56.isReady()) { if (__when_expr_56.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_56.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14when1(__when_expr_56.get(), loopDepth); };
					static_cast<CliActor*>(this)->actor_wait_state = 57;
															#line 2095 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					__when_expr_56.addCallbackAndClear(static_cast<ActorCallback< CliActor, 56, Void >*>(static_cast<CliActor*>(this)));
															#line 11592 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = 0;
				}
				else
				{
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont5(loopDepth);
				}
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont15(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont18(int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont15(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont23(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont23(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont18(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont23(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont23(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose56() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 55, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 55, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 55);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose56();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 55);

	}
	void a_callback_fire(ActorCallback< CliActor, 55, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 55);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose56();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 55);

	}
	void a_callback_error(ActorCallback< CliActor, 55, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 55);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose56();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 55);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont1(int loopDepth) 
	{
															#line 2101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "datadistribution"))
															#line 11711 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2102 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_57 = makeInterruptable(dataDistributionCommandActor(db, tokens));
															#line 2102 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 11717 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_57.isReady()) { if (__when_expr_57.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_57.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont1when1(__when_expr_57.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 58;
															#line 2102 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_57.addCallbackAndClear(static_cast<ActorCallback< CliActor, 57, bool >*>(static_cast<CliActor*>(this)));
															#line 11722 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont2(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont5(int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont10(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont10(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont10(_, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont10(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose57() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 56, Void >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 56, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 56);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose57();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 56);

	}
	void a_callback_fire(ActorCallback< CliActor, 56, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 56);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose57();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 56);

	}
	void a_callback_error(ActorCallback< CliActor, 56, Void >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 56);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose57();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 56);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12(int loopDepth) 
	{
															#line 2108 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "option"))
															#line 11835 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2109 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() == 2 || tokens.size() > 4)
															#line 11839 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2110 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 2111 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 11845 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont2(loopDepth);
			}
			else
			{
															#line 2113 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (tokens.size() == 1)
															#line 11852 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2114 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (options->hasAnyOptionsEnabled())
															#line 11856 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 2115 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						printf("\nCurrently enabled options:\n\n");
															#line 2116 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						options->print();
															#line 2117 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						printf("\n");
															#line 11864 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 2119 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						fprintf(stderr, "There are no options enabled\n");
															#line 11870 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
				}
															#line 2123 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				bool isOn;
															#line 2124 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (tokencmp(tokens[1], "on"))
															#line 11878 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2125 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					isOn = true;
															#line 11882 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
				else
				{
															#line 2126 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (tokencmp(tokens[1], "off"))
															#line 11888 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 2127 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (intrans)
															#line 11892 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 2128 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							fprintf( stderr, "ERROR: Cannot turn option off when using a transaction created with `begin'\n");
															#line 2131 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							is_error = true;
															#line 11898 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
						}
															#line 2134 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (tokens.size() > 3)
															#line 11903 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 2135 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							fprintf(stderr, "ERROR: Cannot specify option argument when turning option off\n");
															#line 2136 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							is_error = true;
															#line 11909 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
							return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
						}
															#line 2140 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						isOn = false;
															#line 11914 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					}
					else
					{
															#line 2142 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						fprintf(stderr, "ERROR: Invalid option state `%s': option must be turned `on' or `off'\n", formatStringRef(tokens[1]).c_str());
															#line 2145 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						is_error = true;
															#line 11922 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
					}
				}
															#line 2149 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				Optional<StringRef> arg = (tokens.size() > 3) ? tokens[3] : Optional<StringRef>();
															#line 11928 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				try {
															#line 2152 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					options->setOption(tr, tokens[2], isOn, arg, intrans);
															#line 2153 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					printf("Option %s for %s\n", isOn ? "enabled" : "disabled", intrans ? "current transaction" : "all transactions");
															#line 11934 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont15(loopDepth);
				}
				catch (Error& error) {
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12Catch1(error, loopDepth);
				} catch (...) {
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12Catch1(unknown_error(), loopDepth);
				}
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont13(bool const& _result,int loopDepth) 
	{
															#line 2103 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 11955 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2104 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 11959 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont13(bool && _result,int loopDepth) 
	{
															#line 2103 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 11969 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2104 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 11973 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont13(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont13(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose58() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 57, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 57, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 57);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose58();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 57);

	}
	void a_callback_fire(ActorCallback< CliActor, 57, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 57);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose58();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 57);

	}
	void a_callback_error(ActorCallback< CliActor, 57, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 57);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose58();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 57);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont1(int loopDepth) 
	{
															#line 2166 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "throttle"))
															#line 12058 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2167 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_58 = makeInterruptable(throttleCommandActor(db, tokens));
															#line 2167 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12064 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_58.isReady()) { if (__when_expr_58.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_58.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont1when1(__when_expr_58.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 59;
															#line 2167 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_58.addCallbackAndClear(static_cast<ActorCallback< CliActor, 58, bool >*>(static_cast<CliActor*>(this)));
															#line 12069 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont2(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont4(int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont2(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2158 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			TraceEvent(SevWarn, "CLISetOptionError").error(e).detail("Option", tokens[2]);
															#line 2159 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12098 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont15(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1cont7loopBody1cont1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17(int loopDepth) 
	{
															#line 2173 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "cache_range"))
															#line 12126 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2174 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_59 = makeInterruptable(cacheRangeCommandActor(db, tokens));
															#line 2174 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12132 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_59.isReady()) { if (__when_expr_59.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_59.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17when1(__when_expr_59.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 60;
															#line 2174 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_59.addCallbackAndClear(static_cast<ActorCallback< CliActor, 59, bool >*>(static_cast<CliActor*>(this)));
															#line 12137 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont18(bool const& _result,int loopDepth) 
	{
															#line 2168 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 12151 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2169 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12155 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont18(bool && _result,int loopDepth) 
	{
															#line 2168 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 12165 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2169 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12169 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont18(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont18(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose59() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 58, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 58, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 58);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose59();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 58);

	}
	void a_callback_fire(ActorCallback< CliActor, 58, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 58);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose59();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 58);

	}
	void a_callback_error(ActorCallback< CliActor, 58, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 58);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose59();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 58);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont1(int loopDepth) 
	{
															#line 2180 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "usetenant"))
															#line 12254 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2181 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() > 2)
															#line 12258 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2182 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 2183 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 12264 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont6(loopDepth);
			}
			else
			{
															#line 2184 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (intrans && tokens.size() == 2)
															#line 12271 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2185 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					fprintf(stderr, "ERROR: Tenant cannot be changed while a transaction is open\n");
															#line 2186 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					is_error = true;
															#line 12277 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont8(loopDepth);
				}
				else
				{
															#line 2187 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					if (tokens.size() == 1)
															#line 12284 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
					{
															#line 2188 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (!tenantName.present())
															#line 12288 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						{
															#line 2189 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							printf("Using the default tenant\n");
															#line 12292 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						}
						else
						{
															#line 2191 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
							printf("Using tenant `%s'\n", printable(tenantName.get()).c_str());
															#line 12298 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						}
						loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont10(loopDepth);
					}
					else
					{
															#line 2194 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						StrictFuture<Optional<TenantMapEntry>> __when_expr_60 = makeInterruptable(TenantAPI::tryGetTenant(db, tokens[1]));
															#line 2194 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12308 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						if (__when_expr_60.isReady()) { if (__when_expr_60.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_60.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont1when1(__when_expr_60.get(), loopDepth); };
						static_cast<CliActor*>(this)->actor_wait_state = 61;
															#line 2194 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
						__when_expr_60.addCallbackAndClear(static_cast<ActorCallback< CliActor, 60, Optional<TenantMapEntry> >*>(static_cast<CliActor*>(this)));
															#line 12313 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
						loopDepth = 0;
					}
				}
			}
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont2(bool const& _result,int loopDepth) 
	{
															#line 2175 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 12330 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2176 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12334 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont2(bool && _result,int loopDepth) 
	{
															#line 2175 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 12344 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2176 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12348 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont2(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont2(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose60() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 59, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 59, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 59);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose60();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 59);

	}
	void a_callback_fire(ActorCallback< CliActor, 59, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 59);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose60();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 59);

	}
	void a_callback_error(ActorCallback< CliActor, 59, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 59);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose60();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 59);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont5(int loopDepth) 
	{
															#line 2210 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "defaulttenant"))
															#line 12433 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2211 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() != 1)
															#line 12437 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2212 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printUsage(tokens[0]);
															#line 2213 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				is_error = true;
															#line 12443 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
			else
			{
															#line 2214 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				if (intrans)
															#line 12449 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				{
															#line 2215 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					fprintf(stderr, "ERROR: Tenant cannot be changed while a transaction is open\n");
															#line 2216 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					is_error = true;
															#line 12455 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
				else
				{
															#line 2218 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					tenant = Reference<ITenant>();
															#line 2219 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					tenantName = Optional<Standalone<StringRef>>();
															#line 2220 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					tenantEntry = Optional<TenantMapEntry>();
															#line 2221 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
					printf("Using the default tenant\n");
															#line 12467 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				}
			}
			return a_body1cont7loopBody1cont1continue1(loopDepth); // continue
		}
															#line 2227 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "tenant"))
															#line 12474 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2228 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_61 = makeInterruptable(tenantCommand(db, tokens));
															#line 2228 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12480 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_61.isReady()) { if (__when_expr_61.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_61.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont5when1(__when_expr_61.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 62;
															#line 2228 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_61.addCallbackAndClear(static_cast<ActorCallback< CliActor, 61, bool >*>(static_cast<CliActor*>(this)));
															#line 12485 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont6(int loopDepth) 
	{
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont8(int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont6(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont10(int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont8(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont14(Optional<TenantMapEntry> const& entry,int loopDepth) 
	{
															#line 2196 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!entry.present())
															#line 12517 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2197 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fprintf(stderr, "ERROR: Tenant `%s' does not exist\n", printable(tokens[1]).c_str());
															#line 2198 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12523 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 2200 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			tenant = db->openTenant(tokens[1]);
															#line 2201 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			tenantName = tokens[1];
															#line 2202 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			tenantEntry = entry;
															#line 2203 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("Using tenant `%s'\n", printable(tokens[1]).c_str());
															#line 12535 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont14(Optional<TenantMapEntry> && entry,int loopDepth) 
	{
															#line 2196 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!entry.present())
															#line 12545 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2197 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			fprintf(stderr, "ERROR: Tenant `%s' does not exist\n", printable(tokens[1]).c_str());
															#line 2198 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12551 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 2200 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			tenant = db->openTenant(tokens[1]);
															#line 2201 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			tenantName = tokens[1];
															#line 2202 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			tenantEntry = entry;
															#line 2203 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			printf("Using tenant `%s'\n", printable(tokens[1]).c_str());
															#line 12563 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont10(loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont1when1(Optional<TenantMapEntry> const& entry,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont14(entry, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont1when1(Optional<TenantMapEntry> && entry,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont14(std::move(entry), loopDepth);

		return loopDepth;
	}
	void a_exitChoose61() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 60, Optional<TenantMapEntry> >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 60, Optional<TenantMapEntry> >*,Optional<TenantMapEntry> const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 60);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose61();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 60);

	}
	void a_callback_fire(ActorCallback< CliActor, 60, Optional<TenantMapEntry> >*,Optional<TenantMapEntry> && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 60);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose61();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 60);

	}
	void a_callback_error(ActorCallback< CliActor, 60, Optional<TenantMapEntry> >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 60);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose61();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 60);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16(int loopDepth) 
	{
															#line 2241 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "createtenant") || tokencmp(tokens[0], "deletetenant") || tokencmp(tokens[0], "listtenants") || tokencmp(tokens[0], "gettenant") || tokencmp(tokens[0], "configuretenant") || tokencmp(tokens[0], "renametenant"))
															#line 12648 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2244 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_62 = makeInterruptable(tenantCommandForwarder(db, tokens));
															#line 2244 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12654 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_62.isReady()) { if (__when_expr_62.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_62.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16when1(__when_expr_62.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 63;
															#line 2244 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_62.addCallbackAndClear(static_cast<ActorCallback< CliActor, 62, bool >*>(static_cast<CliActor*>(this)));
															#line 12659 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont22(bool const& _result,int loopDepth) 
	{
															#line 2229 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 12673 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2230 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12677 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 2231 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() >= 3 && tenantName.present() && tokencmp(tokens[1], "delete") && tokens[2] == tenantName.get())
															#line 12683 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2233 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printAtCol("WARNING: the active tenant was deleted. Use the `usetenant' or `defaulttenant' " "command to choose a new tenant.\n", 80);
															#line 12687 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont22(bool && _result,int loopDepth) 
	{
															#line 2229 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 12698 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2230 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12702 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		else
		{
															#line 2231 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (tokens.size() >= 3 && tenantName.present() && tokencmp(tokens[1], "delete") && tokens[2] == tenantName.get())
															#line 12708 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			{
															#line 2233 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				printAtCol("WARNING: the active tenant was deleted. Use the `usetenant' or `defaulttenant' " "command to choose a new tenant.\n", 80);
															#line 12712 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			}
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont5when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont22(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont5when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont22(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose62() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 61, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 61, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 61);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose62();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 61);

	}
	void a_callback_fire(ActorCallback< CliActor, 61, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 61);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose62();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 61);

	}
	void a_callback_error(ActorCallback< CliActor, 61, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 61);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose62();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 61);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont1(int loopDepth) 
	{
															#line 2251 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "tenantgroup"))
															#line 12798 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2252 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_63 = makeInterruptable(tenantGroupCommand(db, tokens));
															#line 2252 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12804 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_63.isReady()) { if (__when_expr_63.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_63.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont1when1(__when_expr_63.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 64;
															#line 2252 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_63.addCallbackAndClear(static_cast<ActorCallback< CliActor, 63, bool >*>(static_cast<CliActor*>(this)));
															#line 12809 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont2(bool const& _result,int loopDepth) 
	{
															#line 2245 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 12823 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2246 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12827 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont2(bool && _result,int loopDepth) 
	{
															#line 2245 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 12837 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2246 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12841 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont2(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont2(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose63() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 62, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 62, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 62);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose63();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 62);

	}
	void a_callback_fire(ActorCallback< CliActor, 62, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 62);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose63();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 62);

	}
	void a_callback_error(ActorCallback< CliActor, 62, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 62);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose63();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 62);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont5(int loopDepth) 
	{
															#line 2258 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "metacluster"))
															#line 12926 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2259 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_64 = makeInterruptable(metaclusterCommand(db, tokens));
															#line 2259 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 12932 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_64.isReady()) { if (__when_expr_64.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_64.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont5when1(__when_expr_64.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 65;
															#line 2259 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_64.addCallbackAndClear(static_cast<ActorCallback< CliActor, 64, bool >*>(static_cast<CliActor*>(this)));
															#line 12937 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont9(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont6(bool const& _result,int loopDepth) 
	{
															#line 2253 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 12951 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2254 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12955 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont6(bool && _result,int loopDepth) 
	{
															#line 2253 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 12965 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2254 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 12969 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont6(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont6(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose64() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 63, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 63, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 63);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose64();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 63);

	}
	void a_callback_fire(ActorCallback< CliActor, 63, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 63);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose64();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 63);

	}
	void a_callback_error(ActorCallback< CliActor, 63, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 63);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose64();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 63);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont9(int loopDepth) 
	{
															#line 2265 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "idempotencyids"))
															#line 13054 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2266 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_65 = makeInterruptable(idempotencyIdsCommandActor(localDb, tokens));
															#line 2266 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13060 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_65.isReady()) { if (__when_expr_65.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_65.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont9when1(__when_expr_65.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 66;
															#line 2266 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_65.addCallbackAndClear(static_cast<ActorCallback< CliActor, 65, bool >*>(static_cast<CliActor*>(this)));
															#line 13065 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont10(bool const& _result,int loopDepth) 
	{
															#line 2260 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 13079 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2261 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 13083 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont10(bool && _result,int loopDepth) 
	{
															#line 2260 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 13093 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2261 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 13097 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont5when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont10(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont5when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont10(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose65() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 64, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 64, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 64);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose65();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont5when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 64);

	}
	void a_callback_fire(ActorCallback< CliActor, 64, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 64);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose65();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont5when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 64);

	}
	void a_callback_error(ActorCallback< CliActor, 64, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 64);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose65();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 64);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11(int loopDepth) 
	{
															#line 2273 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "hotrange"))
															#line 13182 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2274 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_66 = makeInterruptable(hotRangeCommandActor(localDb, db, tokens, &storage_interface));
															#line 2274 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13188 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_66.isReady()) { if (__when_expr_66.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_66.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11when1(__when_expr_66.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 67;
															#line 2274 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_66.addCallbackAndClear(static_cast<ActorCallback< CliActor, 66, bool >*>(static_cast<CliActor*>(this)));
															#line 13193 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont12(bool const& _result,int loopDepth) 
	{
															#line 2267 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 13207 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2268 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 13211 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont12(bool && _result,int loopDepth) 
	{
															#line 2267 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 13221 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2268 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 13225 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont9when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont12(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont9when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont12(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose66() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 65, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 65, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 65);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose66();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 65);

	}
	void a_callback_fire(ActorCallback< CliActor, 65, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 65);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose66();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 65);

	}
	void a_callback_error(ActorCallback< CliActor, 65, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 65);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose66();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 65);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont1(int loopDepth) 
	{
															#line 2282 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (tokencmp(tokens[0], "rangeconfig"))
															#line 13310 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2283 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<bool> __when_expr_67 = makeInterruptable(rangeConfigCommandActor(localDb, tokens));
															#line 2283 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<CliActor*>(this)->actor_wait_state < 0) return a_body1cont7loopBody1cont1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 13316 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_67.isReady()) { if (__when_expr_67.isError()) return a_body1cont7loopBody1cont1Catch1(__when_expr_67.getError(), std::max(0, loopDepth - 1)); else return a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont1when1(__when_expr_67.get(), loopDepth); };
			static_cast<CliActor*>(this)->actor_wait_state = 68;
															#line 2283 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_67.addCallbackAndClear(static_cast<ActorCallback< CliActor, 67, bool >*>(static_cast<CliActor*>(this)));
															#line 13321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont2(bool const& _result,int loopDepth) 
	{
															#line 2276 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 13335 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2277 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 13339 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont2(bool && _result,int loopDepth) 
	{
															#line 2276 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 13349 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2277 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 13353 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont2(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont2(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose67() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 66, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 66, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 66);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose67();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 66);

	}
	void a_callback_fire(ActorCallback< CliActor, 66, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 66);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose67();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 66);

	}
	void a_callback_error(ActorCallback< CliActor, 66, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 66);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose67();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 66);

	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont5(int loopDepth) 
	{
															#line 2289 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		fprintf(stderr, "ERROR: Unknown command `%s'. Try `help'?\n", formatStringRef(tokens[0]).c_str());
															#line 2290 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		is_error = true;
															#line 1252 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		++iter;
															#line 13442 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (loopDepth == 0) return a_body1cont7loopBody1cont1loopHead1(0);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont6(bool const& _result,int loopDepth) 
	{
															#line 2284 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 13451 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2285 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 13455 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont6(bool && _result,int loopDepth) 
	{
															#line 2284 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!_result)
															#line 13465 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
															#line 2285 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			is_error = true;
															#line 13469 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		}
		return a_body1cont7loopBody1cont1continue1(loopDepth); // continue

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont1when1(bool const& _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont6(_result, loopDepth);

		return loopDepth;
	}
	int a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont1when1(bool && _result,int loopDepth) 
	{
		loopDepth = a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont6(std::move(_result), loopDepth);

		return loopDepth;
	}
	void a_exitChoose68() 
	{
		if (static_cast<CliActor*>(this)->actor_wait_state > 0) static_cast<CliActor*>(this)->actor_wait_state = 0;
		static_cast<CliActor*>(this)->ActorCallback< CliActor, 67, bool >::remove();

	}
	void a_callback_fire(ActorCallback< CliActor, 67, bool >*,bool const& value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 67);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose68();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 67);

	}
	void a_callback_fire(ActorCallback< CliActor, 67, bool >*,bool && value) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 67);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose68();
		try {
			a_body1cont7loopBody1cont1loopBody1cont24cont12cont13cont11cont11cont17cont21cont10cont11cont10cont11cont11cont28cont14cont14cont12cont17cont16cont11cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 67);

	}
	void a_callback_error(ActorCallback< CliActor, 67, bool >*,Error err) 
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), 67);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose68();
		try {
			a_body1cont7loopBody1cont1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1cont7loopBody1cont1Catch1(error, 0);
		} catch (...) {
			a_body1cont7loopBody1cont1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), 67);

	}
	int a_body1cont7loopBody1cont10cont2(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont7loopBody1cont9(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	CLIOptions opt;
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	LineNoise* plinenoise;
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Reference<ClusterConnectionFile> ccf;
															#line 1102 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	LineNoise& linenoise;
															#line 1103 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	bool intrans;
															#line 1104 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	TransType transtype;
															#line 1105 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	bool isCommitDesc;
															#line 1107 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Database localDb;
															#line 1108 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Reference<IDatabase> db;
															#line 1109 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Reference<IDatabase> configDb;
															#line 1110 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Reference<ITenant> tenant;
															#line 1111 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Optional<TenantName> tenantName;
															#line 1112 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Optional<TenantMapEntry> tenantEntry;
															#line 1115 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	const Reference<ITenant> managementTenant;
															#line 1117 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Reference<ITransaction> tr;
															#line 1118 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Reference<ITransaction> config_tr;
															#line 1119 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Transaction trx;
															#line 1121 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	bool writeMode;
															#line 1123 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	std::map<Key, std::pair<Value, ClientLeaderRegInterface>> address_interface;
															#line 1124 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	std::map<std::string, StorageServerInterface> storage_interface;
															#line 1126 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	FdbOptions globalOptions;
															#line 1127 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	FdbOptions activeOptions;
															#line 1129 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	FdbOptions* options;
															#line 1200 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	bool is_error;
															#line 1201 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Future<Void> warn;
															#line 1206 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	std::string line;
															#line 1230 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	UID randomID;
															#line 1234 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	std::vector<std::vector<StringRef>> parsed;
															#line 1248 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	bool multi;
															#line 1251 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	std::vector<std::vector<StringRef>>::iterator iter;
															#line 1253 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	std::vector<StringRef> tokens;
															#line 1464 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	std::string passPhrase;
															#line 1521 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Optional<std::string> raw;
															#line 1526 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Optional<std::string> rawline;
															#line 1612 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	ThreadFuture<Optional<Value>> valueF;
															#line 1785 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	int limit;
															#line 1842 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	ThreadFuture<RangeResult> kvsF;
															#line 1905 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Optional<std::string> raw_desc;
															#line 1975 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	ThreadFuture<Optional<Value>> valueF_knob;
															#line 13641 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
};
// This generated class is to be used only via cli()
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class CliActor final : public Actor<int>, public ActorCallback< CliActor, 0, Void >, public ActorCallback< CliActor, 1, Void >, public ActorCallback< CliActor, 2, Void >, public ActorCallback< CliActor, 3, Optional<std::string> >, public ActorCallback< CliActor, 4, Void >, public ActorCallback< CliActor, 5, Void >, public ActorCallback< CliActor, 6, Void >, public ActorCallback< CliActor, 7, bool >, public ActorCallback< CliActor, 8, Void >, public ActorCallback< CliActor, 9, bool >, public ActorCallback< CliActor, 10, bool >, public ActorCallback< CliActor, 11, bool >, public ActorCallback< CliActor, 12, bool >, public ActorCallback< CliActor, 13, bool >, public ActorCallback< CliActor, 14, bool >, public ActorCallback< CliActor, 15, bool >, public ActorCallback< CliActor, 16, bool >, public ActorCallback< CliActor, 17, bool >, public ActorCallback< CliActor, 18, bool >, public ActorCallback< CliActor, 19, bool >, public ActorCallback< CliActor, 20, Optional<std::string> >, public ActorCallback< CliActor, 21, bool >, public ActorCallback< CliActor, 22, bool >, public ActorCallback< CliActor, 23, Optional<std::string> >, public ActorCallback< CliActor, 24, Void >, public ActorCallback< CliActor, 25, Void >, public ActorCallback< CliActor, 26, bool >, public ActorCallback< CliActor, 27, Optional<Standalone<StringRef>> >, public ActorCallback< CliActor, 28, bool >, public ActorCallback< CliActor, 29, Version >, public ActorCallback< CliActor, 30, bool >, public ActorCallback< CliActor, 31, bool >, public ActorCallback< CliActor, 32, Version >, public ActorCallback< CliActor, 33, bool >, public ActorCallback< CliActor, 34, bool >, public ActorCallback< CliActor, 35, bool >, public ActorCallback< CliActor, 36, bool >, public ActorCallback< CliActor, 37, UID >, public ActorCallback< CliActor, 38, bool >, public ActorCallback< CliActor, 39, bool >, public ActorCallback< CliActor, 40, UID >, public ActorCallback< CliActor, 41, UID >, public ActorCallback< CliActor, 42, bool >, public ActorCallback< CliActor, 43, bool >, public ActorCallback< CliActor, 44, bool >, public ActorCallback< CliActor, 45, bool >, public ActorCallback< CliActor, 46, bool >, public ActorCallback< CliActor, 47, bool >, public ActorCallback< CliActor, 48, RangeResult >, public ActorCallback< CliActor, 49, Void >, public ActorCallback< CliActor, 50, Optional<std::string> >, public ActorCallback< CliActor, 51, Void >, public ActorCallback< CliActor, 52, Optional<Standalone<StringRef>> >, public ActorCallback< CliActor, 53, Optional<std::string> >, public ActorCallback< CliActor, 54, Void >, public ActorCallback< CliActor, 55, Void >, public ActorCallback< CliActor, 56, Void >, public ActorCallback< CliActor, 57, bool >, public ActorCallback< CliActor, 58, bool >, public ActorCallback< CliActor, 59, bool >, public ActorCallback< CliActor, 60, Optional<TenantMapEntry> >, public ActorCallback< CliActor, 61, bool >, public ActorCallback< CliActor, 62, bool >, public ActorCallback< CliActor, 63, bool >, public ActorCallback< CliActor, 64, bool >, public ActorCallback< CliActor, 65, bool >, public ActorCallback< CliActor, 66, bool >, public ActorCallback< CliActor, 67, bool >, public FastAllocated<CliActor>, public CliActorState<CliActor> {
															#line 13646 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
	using FastAllocated<CliActor>::operator new;
	using FastAllocated<CliActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10267675596130443776UL, 5144247705098129152UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< CliActor, 0, Void >;
friend struct ActorCallback< CliActor, 1, Void >;
friend struct ActorCallback< CliActor, 2, Void >;
friend struct ActorCallback< CliActor, 3, Optional<std::string> >;
friend struct ActorCallback< CliActor, 4, Void >;
friend struct ActorCallback< CliActor, 5, Void >;
friend struct ActorCallback< CliActor, 6, Void >;
friend struct ActorCallback< CliActor, 7, bool >;
friend struct ActorCallback< CliActor, 8, Void >;
friend struct ActorCallback< CliActor, 9, bool >;
friend struct ActorCallback< CliActor, 10, bool >;
friend struct ActorCallback< CliActor, 11, bool >;
friend struct ActorCallback< CliActor, 12, bool >;
friend struct ActorCallback< CliActor, 13, bool >;
friend struct ActorCallback< CliActor, 14, bool >;
friend struct ActorCallback< CliActor, 15, bool >;
friend struct ActorCallback< CliActor, 16, bool >;
friend struct ActorCallback< CliActor, 17, bool >;
friend struct ActorCallback< CliActor, 18, bool >;
friend struct ActorCallback< CliActor, 19, bool >;
friend struct ActorCallback< CliActor, 20, Optional<std::string> >;
friend struct ActorCallback< CliActor, 21, bool >;
friend struct ActorCallback< CliActor, 22, bool >;
friend struct ActorCallback< CliActor, 23, Optional<std::string> >;
friend struct ActorCallback< CliActor, 24, Void >;
friend struct ActorCallback< CliActor, 25, Void >;
friend struct ActorCallback< CliActor, 26, bool >;
friend struct ActorCallback< CliActor, 27, Optional<Standalone<StringRef>> >;
friend struct ActorCallback< CliActor, 28, bool >;
friend struct ActorCallback< CliActor, 29, Version >;
friend struct ActorCallback< CliActor, 30, bool >;
friend struct ActorCallback< CliActor, 31, bool >;
friend struct ActorCallback< CliActor, 32, Version >;
friend struct ActorCallback< CliActor, 33, bool >;
friend struct ActorCallback< CliActor, 34, bool >;
friend struct ActorCallback< CliActor, 35, bool >;
friend struct ActorCallback< CliActor, 36, bool >;
friend struct ActorCallback< CliActor, 37, UID >;
friend struct ActorCallback< CliActor, 38, bool >;
friend struct ActorCallback< CliActor, 39, bool >;
friend struct ActorCallback< CliActor, 40, UID >;
friend struct ActorCallback< CliActor, 41, UID >;
friend struct ActorCallback< CliActor, 42, bool >;
friend struct ActorCallback< CliActor, 43, bool >;
friend struct ActorCallback< CliActor, 44, bool >;
friend struct ActorCallback< CliActor, 45, bool >;
friend struct ActorCallback< CliActor, 46, bool >;
friend struct ActorCallback< CliActor, 47, bool >;
friend struct ActorCallback< CliActor, 48, RangeResult >;
friend struct ActorCallback< CliActor, 49, Void >;
friend struct ActorCallback< CliActor, 50, Optional<std::string> >;
friend struct ActorCallback< CliActor, 51, Void >;
friend struct ActorCallback< CliActor, 52, Optional<Standalone<StringRef>> >;
friend struct ActorCallback< CliActor, 53, Optional<std::string> >;
friend struct ActorCallback< CliActor, 54, Void >;
friend struct ActorCallback< CliActor, 55, Void >;
friend struct ActorCallback< CliActor, 56, Void >;
friend struct ActorCallback< CliActor, 57, bool >;
friend struct ActorCallback< CliActor, 58, bool >;
friend struct ActorCallback< CliActor, 59, bool >;
friend struct ActorCallback< CliActor, 60, Optional<TenantMapEntry> >;
friend struct ActorCallback< CliActor, 61, bool >;
friend struct ActorCallback< CliActor, 62, bool >;
friend struct ActorCallback< CliActor, 63, bool >;
friend struct ActorCallback< CliActor, 64, bool >;
friend struct ActorCallback< CliActor, 65, bool >;
friend struct ActorCallback< CliActor, 66, bool >;
friend struct ActorCallback< CliActor, 67, bool >;
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	CliActor(CLIOptions const& opt,LineNoise* const& plinenoise,Reference<ClusterConnectionFile> const& ccf) 
															#line 13730 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		 : Actor<int>(),
		   CliActorState<CliActor>(opt, plinenoise, ccf),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("cli", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13265007209210231040UL, 3873995047876547072UL);
		ActorExecutionContextHelper __helper(static_cast<CliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("cli");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("cli", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< CliActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< CliActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< CliActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< CliActor, 3, Optional<std::string> >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< CliActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< CliActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< CliActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< CliActor, 7, bool >*)0, actor_cancelled()); break;
		case 9: this->a_callback_error((ActorCallback< CliActor, 8, Void >*)0, actor_cancelled()); break;
		case 10: this->a_callback_error((ActorCallback< CliActor, 9, bool >*)0, actor_cancelled()); break;
		case 11: this->a_callback_error((ActorCallback< CliActor, 10, bool >*)0, actor_cancelled()); break;
		case 12: this->a_callback_error((ActorCallback< CliActor, 11, bool >*)0, actor_cancelled()); break;
		case 13: this->a_callback_error((ActorCallback< CliActor, 12, bool >*)0, actor_cancelled()); break;
		case 14: this->a_callback_error((ActorCallback< CliActor, 13, bool >*)0, actor_cancelled()); break;
		case 15: this->a_callback_error((ActorCallback< CliActor, 14, bool >*)0, actor_cancelled()); break;
		case 16: this->a_callback_error((ActorCallback< CliActor, 15, bool >*)0, actor_cancelled()); break;
		case 17: this->a_callback_error((ActorCallback< CliActor, 16, bool >*)0, actor_cancelled()); break;
		case 18: this->a_callback_error((ActorCallback< CliActor, 17, bool >*)0, actor_cancelled()); break;
		case 19: this->a_callback_error((ActorCallback< CliActor, 18, bool >*)0, actor_cancelled()); break;
		case 20: this->a_callback_error((ActorCallback< CliActor, 19, bool >*)0, actor_cancelled()); break;
		case 21: this->a_callback_error((ActorCallback< CliActor, 20, Optional<std::string> >*)0, actor_cancelled()); break;
		case 22: this->a_callback_error((ActorCallback< CliActor, 21, bool >*)0, actor_cancelled()); break;
		case 23: this->a_callback_error((ActorCallback< CliActor, 22, bool >*)0, actor_cancelled()); break;
		case 24: this->a_callback_error((ActorCallback< CliActor, 23, Optional<std::string> >*)0, actor_cancelled()); break;
		case 25: this->a_callback_error((ActorCallback< CliActor, 24, Void >*)0, actor_cancelled()); break;
		case 26: this->a_callback_error((ActorCallback< CliActor, 25, Void >*)0, actor_cancelled()); break;
		case 27: this->a_callback_error((ActorCallback< CliActor, 26, bool >*)0, actor_cancelled()); break;
		case 28: this->a_callback_error((ActorCallback< CliActor, 27, Optional<Standalone<StringRef>> >*)0, actor_cancelled()); break;
		case 29: this->a_callback_error((ActorCallback< CliActor, 28, bool >*)0, actor_cancelled()); break;
		case 30: this->a_callback_error((ActorCallback< CliActor, 29, Version >*)0, actor_cancelled()); break;
		case 31: this->a_callback_error((ActorCallback< CliActor, 30, bool >*)0, actor_cancelled()); break;
		case 32: this->a_callback_error((ActorCallback< CliActor, 31, bool >*)0, actor_cancelled()); break;
		case 33: this->a_callback_error((ActorCallback< CliActor, 32, Version >*)0, actor_cancelled()); break;
		case 34: this->a_callback_error((ActorCallback< CliActor, 33, bool >*)0, actor_cancelled()); break;
		case 35: this->a_callback_error((ActorCallback< CliActor, 34, bool >*)0, actor_cancelled()); break;
		case 36: this->a_callback_error((ActorCallback< CliActor, 35, bool >*)0, actor_cancelled()); break;
		case 37: this->a_callback_error((ActorCallback< CliActor, 36, bool >*)0, actor_cancelled()); break;
		case 38: this->a_callback_error((ActorCallback< CliActor, 37, UID >*)0, actor_cancelled()); break;
		case 39: this->a_callback_error((ActorCallback< CliActor, 38, bool >*)0, actor_cancelled()); break;
		case 40: this->a_callback_error((ActorCallback< CliActor, 39, bool >*)0, actor_cancelled()); break;
		case 41: this->a_callback_error((ActorCallback< CliActor, 40, UID >*)0, actor_cancelled()); break;
		case 42: this->a_callback_error((ActorCallback< CliActor, 41, UID >*)0, actor_cancelled()); break;
		case 43: this->a_callback_error((ActorCallback< CliActor, 42, bool >*)0, actor_cancelled()); break;
		case 44: this->a_callback_error((ActorCallback< CliActor, 43, bool >*)0, actor_cancelled()); break;
		case 45: this->a_callback_error((ActorCallback< CliActor, 44, bool >*)0, actor_cancelled()); break;
		case 46: this->a_callback_error((ActorCallback< CliActor, 45, bool >*)0, actor_cancelled()); break;
		case 47: this->a_callback_error((ActorCallback< CliActor, 46, bool >*)0, actor_cancelled()); break;
		case 48: this->a_callback_error((ActorCallback< CliActor, 47, bool >*)0, actor_cancelled()); break;
		case 49: this->a_callback_error((ActorCallback< CliActor, 48, RangeResult >*)0, actor_cancelled()); break;
		case 50: this->a_callback_error((ActorCallback< CliActor, 49, Void >*)0, actor_cancelled()); break;
		case 51: this->a_callback_error((ActorCallback< CliActor, 50, Optional<std::string> >*)0, actor_cancelled()); break;
		case 52: this->a_callback_error((ActorCallback< CliActor, 51, Void >*)0, actor_cancelled()); break;
		case 53: this->a_callback_error((ActorCallback< CliActor, 52, Optional<Standalone<StringRef>> >*)0, actor_cancelled()); break;
		case 54: this->a_callback_error((ActorCallback< CliActor, 53, Optional<std::string> >*)0, actor_cancelled()); break;
		case 55: this->a_callback_error((ActorCallback< CliActor, 54, Void >*)0, actor_cancelled()); break;
		case 56: this->a_callback_error((ActorCallback< CliActor, 55, Void >*)0, actor_cancelled()); break;
		case 57: this->a_callback_error((ActorCallback< CliActor, 56, Void >*)0, actor_cancelled()); break;
		case 58: this->a_callback_error((ActorCallback< CliActor, 57, bool >*)0, actor_cancelled()); break;
		case 59: this->a_callback_error((ActorCallback< CliActor, 58, bool >*)0, actor_cancelled()); break;
		case 60: this->a_callback_error((ActorCallback< CliActor, 59, bool >*)0, actor_cancelled()); break;
		case 61: this->a_callback_error((ActorCallback< CliActor, 60, Optional<TenantMapEntry> >*)0, actor_cancelled()); break;
		case 62: this->a_callback_error((ActorCallback< CliActor, 61, bool >*)0, actor_cancelled()); break;
		case 63: this->a_callback_error((ActorCallback< CliActor, 62, bool >*)0, actor_cancelled()); break;
		case 64: this->a_callback_error((ActorCallback< CliActor, 63, bool >*)0, actor_cancelled()); break;
		case 65: this->a_callback_error((ActorCallback< CliActor, 64, bool >*)0, actor_cancelled()); break;
		case 66: this->a_callback_error((ActorCallback< CliActor, 65, bool >*)0, actor_cancelled()); break;
		case 67: this->a_callback_error((ActorCallback< CliActor, 66, bool >*)0, actor_cancelled()); break;
		case 68: this->a_callback_error((ActorCallback< CliActor, 67, bool >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
[[nodiscard]] Future<int> cli( CLIOptions const& opt, LineNoise* const& plinenoise, Reference<ClusterConnectionFile> const& ccf ) {
															#line 1101 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	return Future<int>(new CliActor(opt, plinenoise, ccf));
															#line 13830 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
}

#line 2320 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"

															#line 13835 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
namespace {
// This generated class is to be used only via runCli()
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class RunCliActor>
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class RunCliActorState {
															#line 13842 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	RunCliActorState(CLIOptions const& opt,Reference<ClusterConnectionFile> const& ccf) 
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		 : opt(opt),
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   ccf(ccf),
															#line 2322 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   linenoise([](std::string const& line, std::vector<std::string>& completions) { fdbcliCompCmd(line, completions); }, [enabled = opt.cliHints](std::string const& line) -> LineNoise::Hint { if (!enabled) { return LineNoise::Hint(); } bool error = false; bool partial = false; std::string linecopy = line; std::vector<std::vector<StringRef>> parsed = parseLine(linecopy, error, partial); if (parsed.size() == 0 || parsed.back().size() == 0) return LineNoise::Hint(); StringRef command = parsed.back().front(); int finishedParameters = parsed.back().size() + error; if (error && line.back() != '\\') return LineNoise::Hint(std::string(" {malformed escape sequence}"), 90, false); bool inArgument = *(line.end() - 1) != ' '; std::string hintLine = inArgument ? " " : ""; auto itr = CommandFactory::hintGenerators().find(command.toString()); if (itr != CommandFactory::hintGenerators().end()) { std::vector<const char*> hintItems = itr->second(parsed.back(), inArgument); if (hintItems.empty()) { return LineNoise::Hint(); } for (auto item : hintItems) { hintLine = hintLine + item + " "; } } else { auto iter = helpMap.find(command.toString()); if (iter != helpMap.end()) { std::string helpLine = iter->second.usage; std::vector<std::vector<StringRef>> parsedHelp = parseLine(helpLine, error, partial); for (int i = finishedParameters; i < parsedHelp.back().size(); i++) { hintLine = hintLine + parsedHelp.back()[i].toString() + " "; } } else { return LineNoise::Hint(); } } return LineNoise::Hint(hintLine, 90, false); }, 1000, false),
															#line 2374 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		   historyFilename()
															#line 13855 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
	{
		fdb_probe_actor_create("runCli", reinterpret_cast<unsigned long>(this));

	}
	~RunCliActorState() 
	{
		fdb_probe_actor_destroy("runCli", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 2376 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				historyFilename = joinPath(getUserHomeDirectory(), ".fdbcli_history");
															#line 2377 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				linenoise.historyLoad(historyFilename);
															#line 13873 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont3(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~RunCliActorState();
		static_cast<RunCliActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 2385 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		StrictFuture<int> __when_expr_0 = cli(opt, &linenoise, ccf);
															#line 2385 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (static_cast<RunCliActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 13904 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1cont1when1(__when_expr_0.get(), loopDepth); };
		static_cast<RunCliActor*>(this)->actor_wait_state = 1;
															#line 2385 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< RunCliActor, 0, int >*>(static_cast<RunCliActor*>(this)));
															#line 13909 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2379 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			TraceEvent(SevWarnAlways, "ErrorLoadingCliHistory") .error(e) .detail("Filename", historyFilename.empty() ? "<unknown>" : historyFilename) .GetLastError();
															#line 13919 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont3(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont4(int loopDepth) 
	{
															#line 2387 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!historyFilename.empty())
															#line 13947 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		{
			try {
															#line 2389 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
				linenoise.historySave(historyFilename);
															#line 13952 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
				loopDepth = a_body1cont8(loopDepth);
			}
			catch (Error& error) {
				loopDepth = a_body1cont4Catch1(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1cont4Catch1(unknown_error(), loopDepth);
			}
		}
		else
		{
			loopDepth = a_body1cont5(loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont1when1(int const& __result,int loopDepth) 
	{
															#line 2385 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		result = __result;
															#line 13972 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	int a_body1cont1when1(int && __result,int loopDepth) 
	{
		result = std::move(__result);
		loopDepth = a_body1cont4(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<RunCliActor*>(this)->actor_wait_state > 0) static_cast<RunCliActor*>(this)->actor_wait_state = 0;
		static_cast<RunCliActor*>(this)->ActorCallback< RunCliActor, 0, int >::remove();

	}
	void a_callback_fire(ActorCallback< RunCliActor, 0, int >*,int const& value) 
	{
		fdb_probe_actor_enter("runCli", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunCliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runCli", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< RunCliActor, 0, int >*,int && value) 
	{
		fdb_probe_actor_enter("runCli", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<RunCliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runCli", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< RunCliActor, 0, int >*,Error err) 
	{
		fdb_probe_actor_enter("runCli", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<RunCliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("runCli", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
															#line 2398 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<RunCliActor*>(this)->SAV<int>::futures) { (void)(result); this->~RunCliActorState(); static_cast<RunCliActor*>(this)->destroy(); return 0; }
															#line 14051 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<RunCliActor*>(this)->SAV< int >::value()) int(std::move(result)); // state_var_RVO
		this->~RunCliActorState();
		static_cast<RunCliActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont6(int loopDepth) 
	{
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont4Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 2391 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			TraceEvent(SevWarnAlways, "ErrorSavingCliHistory") .error(e) .detail("Filename", historyFilename) .GetLastError();
															#line 14070 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = a_body1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont8(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	CLIOptions opt;
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	Reference<ClusterConnectionFile> ccf;
															#line 2322 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	LineNoise linenoise;
															#line 2374 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	std::string historyFilename;
															#line 2385 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	int result;
															#line 14104 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
};
// This generated class is to be used only via runCli()
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class RunCliActor final : public Actor<int>, public ActorCallback< RunCliActor, 0, int >, public FastAllocated<RunCliActor>, public RunCliActorState<RunCliActor> {
															#line 14109 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
	using FastAllocated<RunCliActor>::operator new;
	using FastAllocated<RunCliActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11945607751746164736UL, 513552915242583808UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<int>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< RunCliActor, 0, int >;
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	RunCliActor(CLIOptions const& opt,Reference<ClusterConnectionFile> const& ccf) 
															#line 14126 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		 : Actor<int>(),
		   RunCliActorState<RunCliActor>(opt, ccf),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("runCli", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(3529787634665039616UL, 6682946001037271808UL);
		ActorExecutionContextHelper __helper(static_cast<RunCliActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("runCli");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("runCli", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< RunCliActor, 0, int >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
[[nodiscard]] Future<int> runCli( CLIOptions const& opt, Reference<ClusterConnectionFile> const& ccf ) {
															#line 2321 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	return Future<int>(new RunCliActor(opt, ccf));
															#line 14159 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
}

#line 2400 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"

															#line 14164 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
namespace {
// This generated class is to be used only via timeExit()
															#line 2401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
template <class TimeExitActor>
															#line 2401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class TimeExitActorState {
															#line 14171 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
															#line 2401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	TimeExitActorState(double const& duration) 
															#line 2401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
															#line 2401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		 : duration(duration)
															#line 14178 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
	{
		fdb_probe_actor_create("timeExit", reinterpret_cast<unsigned long>(this));

	}
	~TimeExitActorState() 
	{
		fdb_probe_actor_destroy("timeExit", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 2402 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			StrictFuture<Void> __when_expr_0 = delay(duration);
															#line 2402 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			if (static_cast<TimeExitActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 14195 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TimeExitActor*>(this)->actor_wait_state = 1;
															#line 2402 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TimeExitActor, 0, Void >*>(static_cast<TimeExitActor*>(this)));
															#line 14200 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TimeExitActorState();
		static_cast<TimeExitActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(Void const& _,int loopDepth) 
	{
															#line 2403 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		fprintf(stderr, "Specified timeout reached -- exiting...\n");
															#line 2404 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<TimeExitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TimeExitActorState(); static_cast<TimeExitActor*>(this)->destroy(); return 0; }
															#line 14225 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<TimeExitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TimeExitActorState();
		static_cast<TimeExitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(Void && _,int loopDepth) 
	{
															#line 2403 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		fprintf(stderr, "Specified timeout reached -- exiting...\n");
															#line 2404 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
		if (!static_cast<TimeExitActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TimeExitActorState(); static_cast<TimeExitActor*>(this)->destroy(); return 0; }
															#line 14239 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		new (&static_cast<TimeExitActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TimeExitActorState();
		static_cast<TimeExitActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TimeExitActor*>(this)->actor_wait_state > 0) static_cast<TimeExitActor*>(this)->actor_wait_state = 0;
		static_cast<TimeExitActor*>(this)->ActorCallback< TimeExitActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TimeExitActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("timeExit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TimeExitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeExit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TimeExitActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("timeExit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TimeExitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeExit", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TimeExitActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("timeExit", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TimeExitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("timeExit", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 2401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	double duration;
															#line 14324 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
};
// This generated class is to be used only via timeExit()
															#line 2401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
class TimeExitActor final : public Actor<Void>, public ActorCallback< TimeExitActor, 0, Void >, public FastAllocated<TimeExitActor>, public TimeExitActorState<TimeExitActor> {
															#line 14329 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
public:
	using FastAllocated<TimeExitActor>::operator new;
	using FastAllocated<TimeExitActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14462429434369659648UL, 6793987304026492672UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TimeExitActor, 0, Void >;
															#line 2401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	TimeExitActor(double const& duration) 
															#line 14346 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
		 : Actor<Void>(),
		   TimeExitActorState<TimeExitActor>(duration),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("timeExit", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(17017769611353615616UL, 11818842507111250432UL);
		ActorExecutionContextHelper __helper(static_cast<TimeExitActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("timeExit");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("timeExit", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TimeExitActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 2401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
[[nodiscard]] Future<Void> timeExit( double const& duration ) {
															#line 2401 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"
	return Future<Void>(new TimeExitActor(duration));
															#line 14379 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/build_output/fdbcli/fdbcli.actor.g.cpp"
}

#line 2406 "/codebuild/output/src2347226878/src/github.com/apple/foundationdb/fdbcli/fdbcli.actor.cpp"

const char* checkTlsConfigAgainstCoordAddrs(const ClusterConnectionString& ccs) {
	// Resolve TLS config and inspect whether any of the certificate, key, ca bytes has been set
	extern TLSConfig tlsConfig;
	auto const loaded = tlsConfig.loadSync();
	const bool tlsConfigured =
	    !loaded.getCertificateBytes().empty() || !loaded.getKeyBytes().empty() || !loaded.getCABytes().empty();
	int tlsAddrs = 0;
	int totalAddrs = 0;
	for (const auto& addr : ccs.coords) {
		if (addr.isTLS())
			tlsAddrs++;
		totalAddrs++;
	}
	for (const auto& host : ccs.hostnames) {
		if (host.isTLS)
			tlsAddrs++;
		totalAddrs++;
	}
	if (!tlsConfigured && tlsAddrs == totalAddrs) {
		return "fdbcli is not configured with TLS, but all of the coordinators have TLS addresses.";
	} else {
		return nullptr;
	}
}

int main(int argc, char** argv) {
	platformInit();
	Error::init();
	std::set_new_handler(&platform::outOfMemory);

	registerCrashHandler();

#ifdef __unixish__
	struct sigaction act;

	// We don't want ctrl-c to quit
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	act.sa_handler = SIG_IGN;
	sigaction(SIGINT, &act, nullptr);
#endif

	CLIOptions opt(argc, argv);
	if (opt.exit_code != -1)
		return opt.exit_code;

	if (opt.trace) {
		if (opt.traceDir.empty())
			setNetworkOption(FDBNetworkOptions::TRACE_ENABLE);
		else
			setNetworkOption(FDBNetworkOptions::TRACE_ENABLE, StringRef(opt.traceDir));

		if (!opt.traceFormat.empty()) {
			setNetworkOption(FDBNetworkOptions::TRACE_FORMAT, StringRef(opt.traceFormat));
		}
		setNetworkOption(FDBNetworkOptions::ENABLE_SLOW_TASK_PROFILING);

		if (!opt.logGroup.empty()) {
			setNetworkOption(FDBNetworkOptions::TRACE_LOG_GROUP, StringRef(opt.logGroup));
		}
	}
	initHelp();

	// deferred TLS options
	if (opt.tlsCertPath.size()) {
		try {
			setNetworkOption(FDBNetworkOptions::TLS_CERT_PATH, opt.tlsCertPath);
		} catch (Error& e) {
			fprintf(stderr, "ERROR: cannot set TLS certificate path to `%s' (%s)\n", opt.tlsCertPath.c_str(), e.what());
			return 1;
		}
	}

	if (opt.tlsCAPath.size()) {
		try {
			setNetworkOption(FDBNetworkOptions::TLS_CA_PATH, opt.tlsCAPath);
		} catch (Error& e) {
			fprintf(stderr, "ERROR: cannot set TLS CA path to `%s' (%s)\n", opt.tlsCAPath.c_str(), e.what());
			return 1;
		}
	}
	if (opt.tlsKeyPath.size()) {
		try {
			if (opt.tlsPassword.size())
				setNetworkOption(FDBNetworkOptions::TLS_PASSWORD, opt.tlsPassword);

			setNetworkOption(FDBNetworkOptions::TLS_KEY_PATH, opt.tlsKeyPath);
		} catch (Error& e) {
			fprintf(stderr, "ERROR: cannot set TLS key path to `%s' (%s)\n", opt.tlsKeyPath.c_str(), e.what());
			return 1;
		}
	}
	if (opt.tlsVerifyPeers.size()) {
		try {
			setNetworkOption(FDBNetworkOptions::TLS_VERIFY_PEERS, opt.tlsVerifyPeers);
		} catch (Error& e) {
			fprintf(
			    stderr, "ERROR: cannot set TLS peer verification to `%s' (%s)\n", opt.tlsVerifyPeers.c_str(), e.what());
			return 1;
		}
	}

	if (opt.tlsDisablePlainTextConnection) {
		try {
			setNetworkOption(FDBNetworkOptions::TLS_DISABLE_PLAINTEXT_CONNECTION);
		} catch (Error& e) {
			fprintf(stderr, "ERROR: cannot disable non-TLS connections (%s)\n", e.what());
			return 1;
		}
	}

	try {
		setNetworkOption(FDBNetworkOptions::DISABLE_CLIENT_STATISTICS_LOGGING);
	} catch (Error& e) {
		fprintf(stderr, "ERROR: cannot disable logging client related information (%s)\n", e.what());
		return 1;
	}

	if (opt.debugTLS) {
		// Backdoor into NativeAPI's tlsConfig, which is where the above network option settings ended up.
		extern TLSConfig tlsConfig;
		printf("TLS Configuration:\n");
		printf("\tCertificate Path: %s\n", tlsConfig.getCertificatePathSync().c_str());
		printf("\tKey Path: %s\n", tlsConfig.getKeyPathSync().c_str());
		printf("\tCA Path: %s\n", tlsConfig.getCAPathSync().c_str());
		printf("\tPlaintext Connection Disable: %s\n", tlsConfig.getDisablePlainTextConnection() ? "true" : "false");
		try {
			LoadedTLSConfig loaded = tlsConfig.loadSync();
			printf("\tPassword: %s\n", loaded.getPassword().empty() ? "Not configured" : "Exists, but redacted");
			printf("\n");
			loaded.print(stdout);
		} catch (Error& e) {
			fprintf(stderr, "ERROR: %s (%d)\n", e.what(), e.code());
			printf("Use --log and look at the trace logs for more detailed information on the failure.\n");
			return 1;
		}
		return 0;
	}

	Reference<ClusterConnectionFile> ccf;
	std::pair<std::string, bool> resolvedClusterFile = ClusterConnectionFile::lookupClusterFileName(opt.clusterFile);

	try {
		ccf = makeReference<ClusterConnectionFile>(resolvedClusterFile.first);
	} catch (Error& e) {
		if (e.code() == error_code_operation_cancelled) {
			throw;
		}
		fprintf(stderr, "%s\n", ClusterConnectionFile::getErrorString(resolvedClusterFile, e).c_str());
		return 1;
	}

	// Make sure that TLS configuration lines up with ":tls" prefix on coordinator addresses
	if (auto errorMsg = checkTlsConfigAgainstCoordAddrs(ccf->getConnectionString())) {
		fprintf(stderr, "ERROR: %s\n", errorMsg);
		return 1;
	}

	try {
		API->selectApiVersion(opt.apiVersion);
		if (opt.useFutureProtocolVersion) {
			API->useFutureProtocolVersion();
		}
		API->setupNetwork();
		opt.setupKnobs();
		if (opt.exit_code != -1) {
			return opt.exit_code;
		}
		Future<Void> memoryUsageMonitor = startMemoryUsageMonitor(opt.memLimit);
		Future<int> cliFuture = runCli(opt, ccf);
		Future<Void> timeoutFuture = opt.exit_timeout ? timeExit(opt.exit_timeout) : Never();
		auto f = stopNetworkAfter(success(cliFuture) || timeoutFuture);
		API->runNetwork();

		if (cliFuture.isReady()) {
			return cliFuture.get();
		} else {
			return 1;
		}
	} catch (Error& e) {
		fprintf(stderr, "ERROR: %s (%d)\n", e.what(), e.code());
		return 1;
	}
}
