#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
/*
 * StorageMetrics.actor.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "flow/UnitTest.h"
#include "fdbserver/StorageMetrics.actor.h"
#include "flow/actorcompiler.h" // This must be the last #include.

CommonStorageCounters::CommonStorageCounters(const std::string& name,
                                             const std::string& id,
                                             const StorageServerMetrics* metrics)
  : cc(name, id), finishedQueries("FinishedQueries", cc), bytesQueried("BytesQueried", cc),
    bytesFetched("BytesFetched", cc), bytesInput("BytesInput", cc), mutationBytes("MutationBytes", cc),
    kvFetched("KVFetched", cc), mutations("Mutations", cc), setMutations("SetMutations", cc),
    clearRangeMutations("ClearRangeMutations", cc), fetchKeyErrors("FetchKeyErrors", cc) {
	if (metrics) {
		specialCounter(cc, "BytesStored", [metrics]() { return metrics->byteSample.getEstimate(allKeys); });
		specialCounter(cc, "BytesReadSampleCount", [metrics]() { return metrics->bytesReadSample.queue.size(); });
		specialCounter(cc, "OpsReadSampleCount", [metrics]() { return metrics->opsReadSample.queue.size(); });
		specialCounter(cc, "BytesWriteSampleCount", [metrics]() { return metrics->bytesWriteSample.queue.size(); });
		specialCounter(cc, "IopsReadSampleCount", [metrics]() { return metrics->iopsSample.queue.size(); });
	}
}

// TODO: update the cost as bytesReadPerKSecond + opsReadPerKSecond * SERVER_KNOBS->EMPTY_READ_PENALTY. The source of
// this model is Redwood will have a constant cost of seek of each read ops then read the actual data.
// As by 71.2.8, bytesReadPerKSecond should be larger than opsReadPerKSecond * SERVER_KNOBS->EMPTY_READ_PENALTY because
// the bytes always round to EMPTY_READ_PENALTY when the returned result size is less than EMPTY_READ_PENALTY. This cost
// is different from what tag throttling use to produce throttling decision.
int64_t StorageMetrics::readLoadKSecond() const {
	return std::max(bytesReadPerKSecond, opsReadPerKSecond * SERVER_KNOBS->EMPTY_READ_PENALTY);
}

int64_t StorageMetricSample::getEstimate(KeyRangeRef keys) const {
	return sample.sumRange(keys.begin, keys.end);
}

KeyRef StorageMetricSample::splitEstimate(KeyRangeRef range, int64_t offset, bool front) const {
	auto fwd_split = sample.index(front ? sample.sumTo(sample.lower_bound(range.begin)) + offset
	                                    : sample.sumTo(sample.lower_bound(range.end)) - offset);

	if (fwd_split == sample.end() || *fwd_split >= range.end)
		return range.end;

	if (!front && *fwd_split <= range.begin)
		return range.begin;

	auto bck_split = fwd_split;

	// Butterfly search - start at midpoint then go in both directions.
	while ((fwd_split != sample.end() && *fwd_split < range.end) ||
	       (bck_split != sample.begin() && *bck_split > range.begin)) {
		if (bck_split != sample.begin() && *bck_split > range.begin) {
			auto it = bck_split;
			bck_split.decrementNonEnd();

			KeyRef split = keyBetween(KeyRangeRef(
			    bck_split != sample.begin() ? std::max<KeyRef>(*bck_split, range.begin) : range.begin, *it));
			if (!front || (getEstimate(KeyRangeRef(range.begin, split)) > 0 &&
			               split.size() <= CLIENT_KNOBS->SPLIT_KEY_SIZE_LIMIT))
				return split;
		}

		if (fwd_split != sample.end() && *fwd_split < range.end) {
			auto it = fwd_split;
			++it;

			KeyRef split =
			    keyBetween(KeyRangeRef(*fwd_split, it != sample.end() ? std::min<KeyRef>(*it, range.end) : range.end));
			if (front ||
			    (getEstimate(KeyRangeRef(split, range.end)) > 0 && split.size() <= CLIENT_KNOBS->SPLIT_KEY_SIZE_LIMIT))
				return split;

			fwd_split = it;
		}
	}

	// If we didn't return above, we didn't find anything.
	TraceEvent(SevWarn, "CannotSplitLastSampleKey").detail("Range", range).detail("Offset", offset);
	return front ? range.end : range.begin;
}

// Get the current estimated metrics for the given keys
StorageMetrics StorageServerMetrics::getMetrics(KeyRangeRef const& keys) const {
	StorageMetrics result;
	result.bytes = byteSample.getEstimate(keys);
	result.bytesWrittenPerKSecond =
	    bytesWriteSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	result.iosPerKSecond = iopsSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;

	result.bytesReadPerKSecond =
	    bytesReadSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	result.opsReadPerKSecond =
	    opsReadSample.getEstimate(keys) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	return result;
}

// Called when metrics should change (IO for a given key)
// Notifies waiting WaitMetricsRequests through waitMetricsMap, and updates metricsAverageQueue and metricsSampleMap
void StorageServerMetrics::notify(const Key& key, StorageMetrics& metrics) {
	ASSERT(metrics.bytes == 0); // ShardNotifyMetrics
	if (g_network->isSimulated()) {
		CODE_PROBE(metrics.bytesWrittenPerKSecond != 0, "ShardNotifyMetrics bytes");
		CODE_PROBE(metrics.iosPerKSecond != 0, "ShardNotifyMetrics ios");
		CODE_PROBE(metrics.bytesReadPerKSecond != 0, "ShardNotifyMetrics bytesRead");
		CODE_PROBE(metrics.opsReadPerKSecond != 0, "ShardNotifyMetrics opsRead");
	}

	double expire = now() + SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL;

	StorageMetrics notifyMetrics;

	if (metrics.bytesWrittenPerKSecond)
		notifyMetrics.bytesWrittenPerKSecond =
		    bytesWriteSample.addAndExpire(key, metrics.bytesWrittenPerKSecond, expire) *
		    SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	if (metrics.iosPerKSecond)
		notifyMetrics.iosPerKSecond = iopsSample.addAndExpire(key, metrics.iosPerKSecond, expire) *
		                              SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	if (metrics.bytesReadPerKSecond)
		notifyMetrics.bytesReadPerKSecond = bytesReadSample.addAndExpire(key, metrics.bytesReadPerKSecond, expire) *
		                                    SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	if (metrics.opsReadPerKSecond) {
		notifyMetrics.opsReadPerKSecond = opsReadSample.addAndExpire(key, metrics.opsReadPerKSecond, expire) *
		                                  SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	}

	if (!notifyMetrics.allZero()) {
		auto& v = waitMetricsMap[key];
		for (int i = 0; i < v.size(); i++) {
			if (g_network->isSimulated()) {
				CODE_PROBE(true, "shard notify metrics");
			}
			// ShardNotifyMetrics
			v[i].send(notifyMetrics);
		}
	}
}

// Due to the fact that read sampling will be called on all reads, use this specialized function to avoid overhead
// around branch misses and unnecessary stack allocation which eventually adds up under heavy load.
void StorageServerMetrics::notifyBytesReadPerKSecond(const Key& key, int64_t in) {
	double expire = now() + SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL;
	int64_t bytesReadPerKSecond =
	    bytesReadSample.addAndExpire(key, in, expire) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	int64_t opsReadPerKSecond =
	    opsReadSample.addAndExpire(key, 1, expire) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;

	if (bytesReadPerKSecond > 0 || opsReadPerKSecond > 0) {
		StorageMetrics notifyMetrics;
		notifyMetrics.bytesReadPerKSecond = bytesReadPerKSecond;
		notifyMetrics.opsReadPerKSecond = opsReadPerKSecond;
		auto& v = waitMetricsMap[key];
		for (int i = 0; i < v.size(); i++) {
			CODE_PROBE(bytesReadPerKSecond > 0, "ShardNotifyMetrics bytesRead");
			CODE_PROBE(opsReadPerKSecond > 0, "ShardNotifyMetrics opsRead");
			v[i].send(notifyMetrics);
		}
	}
}

// Called by StorageServerDisk when the size of a key in byteSample changes, to notify WaitMetricsRequest
// Should not be called for keys past allKeys.end
void StorageServerMetrics::notifyBytes(
    RangeMap<Key, std::vector<PromiseStream<StorageMetrics>>, KeyRangeRef>::iterator shard,
    int64_t bytes) {
	ASSERT(shard.end() <= allKeys.end);

	StorageMetrics notifyMetrics;
	notifyMetrics.bytes = bytes;
	auto size = shard->cvalue().size();
	for (int i = 0; i < size; i++) {
		// fmt::print("NotifyBytes {} {}\n", shard->value().size(), shard->range().toString());
		CODE_PROBE(true, "notifyBytes");
		shard.value()[i].send(notifyMetrics);
	}
}

// Called by StorageServerDisk when the size of a key in byteSample changes, to notify WaitMetricsRequest
void StorageServerMetrics::notifyBytes(const KeyRef& key, int64_t bytes) {
	if (key >= allKeys.end) // Do not notify on changes to internal storage server state
		return;

	notifyBytes(waitMetricsMap.rangeContaining(key), bytes);
}

// Called when a range of keys becomes unassigned (and therefore not readable), to notify waiting
// WaitMetricsRequests (also other types of wait
//   requests in the future?)
void StorageServerMetrics::notifyNotReadable(KeyRangeRef keys) {
	auto rs = waitMetricsMap.intersectingRanges(keys);
	for (auto r = rs.begin(); r != rs.end(); ++r) {
		auto& v = r->value();
		CODE_PROBE(v.size(), "notifyNotReadable() sending errors to intersecting ranges");
		for (int n = 0; n < v.size(); n++)
			v[n].sendError(wrong_shard_server());
	}
}

// Called periodically (~1 sec intervals) to remove older IOs from the averages
// Removes old entries from metricsAverageQueue, updates metricsSampleMap accordingly, and notifies
//   WaitMetricsRequests through waitMetricsMap.
void StorageServerMetrics::poll() {
	{
		StorageMetrics m;
		m.bytesWrittenPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		bytesWriteSample.poll(waitMetricsMap, m);
	}
	{
		StorageMetrics m;
		m.iosPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		iopsSample.poll(waitMetricsMap, m);
	}
	{
		StorageMetrics m;
		m.bytesReadPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		bytesReadSample.poll(waitMetricsMap, m);
	}
	{
		StorageMetrics m;
		m.opsReadPerKSecond = SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
		opsReadSample.poll(waitMetricsMap, m);
	}
	// bytesSample doesn't need polling because we never call addExpire() on it
}

// This function can run on untrusted user data.  We must validate all divisions carefully.
KeyRef StorageServerMetrics::getSplitKey(int64_t remaining,
                                         int64_t estimated,
                                         int64_t limits,
                                         int64_t used,
                                         int64_t infinity,
                                         bool isLastShard,
                                         const StorageMetricSample& sample,
                                         double divisor,
                                         KeyRef const& lastKey,
                                         KeyRef const& key,
                                         bool hasUsed) const {
	ASSERT(remaining >= 0);
	ASSERT(limits > 0);
	ASSERT(divisor > 0);

	if (limits < infinity / 2) {
		int64_t expectedSize;
		if (isLastShard || remaining > estimated) {
			double remaining_divisor = (double(remaining) / limits) + 0.5;
			expectedSize = remaining / remaining_divisor;
		} else {
			// If we are here, then estimated >= remaining >= 0
			double estimated_divisor = (double(estimated) / limits) + 0.5;
			expectedSize = remaining / estimated_divisor;
		}

		if (remaining > expectedSize) {
			// This does the conversion from native units to bytes using the divisor.
			double offset = (expectedSize - used) / divisor;
			if (offset <= 0)
				return hasUsed ? lastKey : key;
			return sample.splitEstimate(
			    KeyRangeRef(lastKey, key),
			    offset * ((1.0 - SERVER_KNOBS->SPLIT_JITTER_AMOUNT) +
			              2 * deterministicRandom()->random01() * SERVER_KNOBS->SPLIT_JITTER_AMOUNT));
		}
	}

	return key;
}

void StorageServerMetrics::splitMetrics(SplitMetricsRequest req) const {
	int minSplitBytes = req.minSplitBytes.present() ? req.minSplitBytes.get() : SERVER_KNOBS->MIN_SHARD_BYTES;
	int minSplitWriteTraffic = SERVER_KNOBS->SHARD_SPLIT_BYTES_PER_KSEC;
	try {
		SplitMetricsReply reply;
		KeyRef lastKey = req.keys.begin;
		StorageMetrics used = req.used;
		StorageMetrics estimated = req.estimated;
		StorageMetrics remaining = getMetrics(req.keys) + used;

		//TraceEvent("SplitMetrics").detail("Begin", req.keys.begin).detail("End", req.keys.end).detail("Remaining", remaining.bytes).detail("Used", used.bytes).detail("MinSplitBytes", minSplitBytes);

		while (true) {
			if (remaining.bytes < 2 * minSplitBytes && (!SERVER_KNOBS->ENABLE_WRITE_BASED_SHARD_SPLIT ||
			                                            remaining.bytesWrittenPerKSecond < minSplitWriteTraffic))
				break;
			KeyRef key = req.keys.end;
			bool hasUsed = used.bytes != 0 || used.bytesWrittenPerKSecond != 0 || used.iosPerKSecond != 0;
			key = getSplitKey(remaining.bytes,
			                  estimated.bytes,
			                  req.limits.bytes,
			                  used.bytes,
			                  req.limits.infinity,
			                  req.isLastShard,
			                  byteSample,
			                  1,
			                  lastKey,
			                  key,
			                  hasUsed);
			if (used.bytes < minSplitBytes)
				key = std::max(
				    key, byteSample.splitEstimate(KeyRangeRef(lastKey, req.keys.end), minSplitBytes - used.bytes));
			key = getSplitKey(remaining.iosPerKSecond,
			                  estimated.iosPerKSecond,
			                  req.limits.iosPerKSecond,
			                  used.iosPerKSecond,
			                  req.limits.infinity,
			                  req.isLastShard,
			                  iopsSample,
			                  SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS,
			                  lastKey,
			                  key,
			                  hasUsed);
			key = getSplitKey(remaining.bytesWrittenPerKSecond,
			                  estimated.bytesWrittenPerKSecond,
			                  req.limits.bytesWrittenPerKSecond,
			                  used.bytesWrittenPerKSecond,
			                  req.limits.infinity,
			                  req.isLastShard,
			                  bytesWriteSample,
			                  SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS,
			                  lastKey,
			                  key,
			                  hasUsed);
			ASSERT(key != lastKey || hasUsed);
			if (key == req.keys.end)
				break;
			reply.splits.push_back_deep(reply.splits.arena(), key);
			if (reply.splits.size() > SERVER_KNOBS->SPLIT_METRICS_MAX_ROWS) {
				reply.more = true;
				break;
			}

			StorageMetrics diff = (getMetrics(KeyRangeRef(lastKey, key)) + used);
			remaining -= diff;
			estimated -= diff;

			used = StorageMetrics();
			lastKey = key;
		}

		reply.used = reply.more ? StorageMetrics() : getMetrics(KeyRangeRef(lastKey, req.keys.end)) + used;
		req.reply.send(reply);
	} catch (Error& e) {
		req.reply.sendError(e);
	}
}

void StorageServerMetrics::getStorageMetrics(GetStorageMetricsRequest req,
                                             StorageBytes sb,
                                             double bytesInputRate,
                                             int64_t versionLag,
                                             double lastUpdate,
                                             int64_t bytesDurable,
                                             int64_t bytesInput,
                                             int ongoingBulkLoadTaskCount) const {
	GetStorageMetricsReply rep;

	// SOMEDAY: make bytes dynamic with hard disk space
	rep.load = getMetrics(allKeys);

	if (sb.free < 1e9) {
		TraceEvent(SevWarn, "PhysicalDiskMetrics")
		    .suppressFor(60.0)
		    .detail("Free", sb.free)
		    .detail("Total", sb.total)
		    .detail("Available", sb.available)
		    .detail("Load", rep.load.bytes);
	}

	rep.available.bytes = sb.available;
	rep.available.iosPerKSecond = 10e6;
	rep.available.bytesWrittenPerKSecond = 100e9;
	rep.available.bytesReadPerKSecond = 100e9;

	rep.capacity.bytes = sb.total;
	rep.capacity.iosPerKSecond = 10e6;
	rep.capacity.bytesWrittenPerKSecond = 100e9;
	rep.capacity.bytesReadPerKSecond = 100e9;

	rep.bytesInputRate = bytesInputRate;

	rep.versionLag = versionLag;
	rep.lastUpdate = lastUpdate;

	rep.bytesDurable = bytesDurable;
	rep.bytesInput = bytesInput;

	rep.ongoingBulkLoadTaskCount = ongoingBulkLoadTaskCount;

	req.reply.send(rep);
}

// Equally split the metrics (specified by splitType) of parentRange into chunkCount and return all the sampled metrics
// (bytes, readBytes and readOps) of each chunk
// NOTE: update unit test "equalDivide" after change
std::vector<ReadHotRangeWithMetrics> StorageServerMetrics::getReadHotRanges(KeyRangeRef parentRange,
                                                                            int chunkCount,
                                                                            uint8_t splitType) const {
	const StorageMetricSample* sampler = nullptr;
	switch (splitType) {
	case ReadHotSubRangeRequest::SplitType::BYTES:
		sampler = &byteSample;
		break;
	case ReadHotSubRangeRequest::SplitType::READ_BYTES:
		sampler = &bytesReadSample;
		break;
	case ReadHotSubRangeRequest::SplitType::READ_OPS:
		sampler = &opsReadSample;
		break;
	default:
		ASSERT(false);
	}

	std::vector<ReadHotRangeWithMetrics> toReturn;
	if (sampler->sample.empty()) {
		return toReturn;
	}

	auto total = sampler->getEstimate(parentRange);
	double splitChunk = std::max(1.0, (double)total / chunkCount);

	KeyRef beginKey = parentRange.begin;
	while (parentRange.contains(beginKey)) {
		auto beginIt = sampler->sample.lower_bound(beginKey);
		if (beginIt == sampler->sample.end()) {
			break;
		}
		auto endIt = sampler->sample.index(sampler->sample.sumTo(beginIt) + splitChunk - 1);
		// because index return x where sumTo(x+1) (that including sample at x) > metrics, we have to forward endIt here
		if (endIt != sampler->sample.end())
			++endIt;

		if (endIt == sampler->sample.end()) {
			KeyRangeRef lastRange(beginKey, parentRange.end);
			toReturn.emplace_back(
			    lastRange,
			    byteSample.getEstimate(lastRange),
			    (double)bytesReadSample.getEstimate(lastRange) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL,
			    (double)opsReadSample.getEstimate(lastRange) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
			break;
		}

		ASSERT_LT(beginKey, *endIt);
		KeyRangeRef range(beginKey, *endIt);
		toReturn.emplace_back(
		    range,
		    byteSample.getEstimate(range),
		    (double)bytesReadSample.getEstimate(range) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL,
		    (double)opsReadSample.getEstimate(range) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);

		beginKey = *endIt;
	}
	return toReturn;
}

// Given a read hot shard, this function will divide the shard into chunks and find those chunks whose
// readBytes/sizeBytes exceeds the `readDensityRatio`. Please make sure to run unit tests
// `StorageMetricsSampleTests.txt` after change made.
std::vector<ReadHotRangeWithMetrics> StorageServerMetrics::_getReadHotRanges(
    KeyRangeRef shard,
    double readDensityRatio,
    int64_t baseChunkSize,
    int64_t minShardReadBandwidthPerKSeconds) const {
	std::vector<ReadHotRangeWithMetrics> toReturn;

	double shardSize = (double)byteSample.getEstimate(shard);
	int64_t shardReadBandwidth = bytesReadSample.getEstimate(shard);
	if (shardReadBandwidth * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS <=
	    minShardReadBandwidthPerKSeconds) {
		return toReturn;
	}
	if (shardSize <= baseChunkSize) {
		// Shard is small, use it as is
		if (bytesReadSample.getEstimate(shard) > (readDensityRatio * shardSize)) {
			toReturn.emplace_back(shard,
			                      bytesReadSample.getEstimate(shard) / shardSize,
			                      bytesReadSample.getEstimate(shard) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
		}
		return toReturn;
	}
	KeyRef beginKey = shard.begin;
	auto endKey =
	    byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + baseChunkSize);
	while (endKey != byteSample.sample.end()) {
		if (*endKey > shard.end) {
			endKey = byteSample.sample.lower_bound(shard.end);
			if (*endKey == beginKey) {
				// No need to increment endKey since otherwise it would stuck here forever.
				break;
			}
		}
		if (*endKey == beginKey) {
			++endKey;
			continue;
		}
		if (bytesReadSample.getEstimate(KeyRangeRef(beginKey, *endKey)) >
		    (readDensityRatio * std::max(baseChunkSize, byteSample.getEstimate(KeyRangeRef(beginKey, *endKey))))) {
			auto range = KeyRangeRef(beginKey, *endKey);
			if (!toReturn.empty() && toReturn.back().keys.end == range.begin) {
				// in case two consecutive chunks both are over the ratio, merge them.
				range = KeyRangeRef(toReturn.back().keys.begin, *endKey);
				toReturn.pop_back();
			}
			toReturn.emplace_back(range,
			                      (double)bytesReadSample.getEstimate(range) /
			                          std::max(baseChunkSize, byteSample.getEstimate(range)),
			                      bytesReadSample.getEstimate(range) / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
		}
		beginKey = *endKey;
		endKey =
		    byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + baseChunkSize);
	}
	return toReturn;
}

int64_t StorageServerMetrics::getHotShards(const KeyRange& range) const {
	auto bytesWrittenPerKSecond =
	    bytesWriteSample.getEstimate(range) * SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL_PER_KSECONDS;
	return bytesWrittenPerKSecond;
}

void StorageServerMetrics::getReadHotRanges(ReadHotSubRangeRequest req) const {
	ReadHotSubRangeReply reply;
	auto _ranges = getReadHotRanges(req.keys, req.chunkCount, req.type);
	reply.readHotRanges = VectorRef(_ranges.data(), _ranges.size());
	req.reply.send(reply);
}

void StorageServerMetrics::getSplitPoints(SplitRangeRequest req, Optional<KeyRef> prefix) const {
	SplitRangeReply reply;
	KeyRangeRef range = req.keys;
	if (prefix.present()) {
		range = range.withPrefix(prefix.get(), req.arena);
	}
	std::vector<KeyRef> points = getSplitPoints(range, req.chunkSize, prefix);

	reply.splitPoints.append_deep(reply.splitPoints.arena(), points.data(), points.size());
	req.reply.send(reply);
}

std::vector<KeyRef> StorageServerMetrics::getSplitPoints(KeyRangeRef range,
                                                         int64_t chunkSize,
                                                         Optional<KeyRef> prefixToRemove) const {
	std::vector<KeyRef> toReturn;
	KeyRef beginKey = range.begin;
	IndexedSet<Key, int64_t>::const_iterator endKey =
	    byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + chunkSize);
	while (endKey != byteSample.sample.end()) {
		if (*endKey > range.end) {
			break;
		}
		if (*endKey == beginKey) {
			++endKey;
			continue;
		}
		KeyRef splitPoint = *endKey;
		if (prefixToRemove.present()) {
			splitPoint = splitPoint.removePrefix(prefixToRemove.get());
		}
		toReturn.push_back(splitPoint);
		beginKey = *endKey;
		endKey = byteSample.sample.index(byteSample.sample.sumTo(byteSample.sample.lower_bound(beginKey)) + chunkSize);
	}
	return toReturn;
}

void StorageServerMetrics::collapse(KeyRangeMap<int>& map, KeyRef const& key) {
	auto range = map.rangeContaining(key);
	if (range == map.ranges().begin() || range == map.ranges().end())
		return;
	int value = range->value();
	auto prev = range;
	--prev;
	if (prev->value() != value)
		return;
	KeyRange keys = KeyRangeRef(prev->begin(), range->end());
	map.insert(keys, value);
}

void StorageServerMetrics::add(KeyRangeMap<int>& map, KeyRangeRef const& keys, int delta) {
	auto rs = map.modify(keys);
	for (auto r = rs.begin(); r != rs.end(); ++r)
		r->value() += delta;
	collapse(map, keys.begin);
	collapse(map, keys.end);
}

// Returns the sampled metric value (possibly 0, possibly increased by the sampling factor)
int64_t TransientStorageMetricSample::addAndExpire(const Key& key, int64_t metric, double expiration) {
	auto x = add(key, metric);
	if (x)
		queue.emplace_back(expiration, std::make_pair(key, -x));
	return x;
}

// FIXME: both versions of erase are broken, because they do not remove items in the queue with will subtract a
// metric from the value sometime in the future
int64_t TransientStorageMetricSample::erase(KeyRef key) {
	auto it = sample.find(key);
	if (it == sample.end())
		return 0;
	int64_t x = sample.getMetric(it);
	sample.erase(it);
	return x;
}

void TransientStorageMetricSample::erase(KeyRangeRef keys) {
	sample.erase(keys.begin, keys.end);
}

bool TransientStorageMetricSample::roll(int64_t metric) const {
	return deterministicRandom()->random01() < (double)metric / metricUnitsPerSample; //< SOMEDAY: Better randomInt64?
}

void TransientStorageMetricSample::poll(KeyRangeMap<std::vector<PromiseStream<StorageMetrics>>>& waitMap,
                                        StorageMetrics metrics) {
	double now = ::now();
	while (queue.size() && queue.front().first <= now) {
		KeyRef key = queue.front().second.first;
		int64_t delta = queue.front().second.second;
		ASSERT(delta != 0);

		auto [m, it] = sample.addMetric(key, delta);
		if (m == 0)
			sample.erase(it);

		StorageMetrics deltaM = metrics * delta;
		auto v = waitMap[key];
		for (int i = 0; i < v.size(); i++) {
			CODE_PROBE(true, "TransientStorageMetricSample poll update");
			v[i].send(deltaM);
		}

		queue.pop_front();
	}
}

void TransientStorageMetricSample::poll() {
	double now = ::now();
	while (queue.size() && queue.front().first <= now) {
		KeyRef key = queue.front().second.first;
		int64_t delta = queue.front().second.second;
		ASSERT(delta != 0);

		auto [m, it] = sample.addMetric(key, delta);
		if (m == 0)
			sample.erase(it);

		queue.pop_front();
	}
}

int64_t TransientStorageMetricSample::add(const Key& key, int64_t metric) {
	if (!metric)
		return 0;
	int64_t mag = metric < 0 ? -metric : metric;

	if (mag < metricUnitsPerSample) {
		if (!roll(mag))
			return 0;
		metric = metric < 0 ? -metricUnitsPerSample : metricUnitsPerSample;
	}

	auto [m, it] = sample.addMetric(key, metric);
	if (m == 0)
		sample.erase(it);

	return metric;
}

															#line 688 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase686()
															#line 686 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase686Actor>
															#line 686 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase686ActorState {
															#line 695 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 686 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase686ActorState(UnitTestParameters const& params) 
															#line 686 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 686 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 702 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase686", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase686ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase686", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 687 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageMetricSample s(1000);
															#line 688 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Apple"_sr, 1000);
															#line 689 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Banana"_sr, 2000);
															#line 690 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Cat"_sr, 1000);
															#line 691 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Cathode"_sr, 1000);
															#line 692 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			s.sample.insert("Dog"_sr, 1000);
															#line 694 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(s.getEstimate(KeyRangeRef("A"_sr, "D"_sr)) == 5000);
															#line 695 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(s.getEstimate(KeyRangeRef("A"_sr, "E"_sr)) == 6000);
															#line 696 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(s.getEstimate(KeyRangeRef("B"_sr, "C"_sr)) == 2000);
															#line 701 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase686Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase686ActorState(); static_cast<FlowTestCase686Actor*>(this)->destroy(); return 0; }
															#line 735 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase686Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase686ActorState();
			static_cast<FlowTestCase686Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase686ActorState();
		static_cast<FlowTestCase686Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 686 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 759 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase686()
															#line 686 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase686Actor final : public Actor<Void>, public FastAllocated<FlowTestCase686Actor>, public FlowTestCase686ActorState<FlowTestCase686Actor> {
															#line 764 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase686Actor>::operator new;
	using FastAllocated<FlowTestCase686Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(12816551934055368448UL, 1623506854664398592UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 686 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase686Actor(UnitTestParameters const& params) 
															#line 780 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase686ActorState<FlowTestCase686Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase686", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16954180358030890496UL, 12427883299091467520UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase686Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase686");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase686", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 686 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase686( UnitTestParameters const& params ) {
															#line 686 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase686Actor(params));
															#line 812 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase686, "/fdbserver/StorageMetricSample/simple")

#line 703 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 818 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase704()
															#line 704 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase704Actor>
															#line 704 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase704ActorState {
															#line 825 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 704 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase704ActorState(UnitTestParameters const& params) 
															#line 704 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 704 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 832 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase704", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase704ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase704", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 706 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 707 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 709 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 200 * sampleUnit);
															#line 710 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 800 * sampleUnit);
															#line 711 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 712 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 713 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 714 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 715 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 716 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 718 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 2000 * sampleUnit, {});
															#line 720 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 1 && t[0] == "Bah"_sr);
															#line 722 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase704Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase704ActorState(); static_cast<FlowTestCase704Actor*>(this)->destroy(); return 0; }
															#line 871 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase704Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase704ActorState();
			static_cast<FlowTestCase704Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase704ActorState();
		static_cast<FlowTestCase704Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 704 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 895 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase704()
															#line 704 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase704Actor final : public Actor<Void>, public FastAllocated<FlowTestCase704Actor>, public FlowTestCase704ActorState<FlowTestCase704Actor> {
															#line 900 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase704Actor>::operator new;
	using FastAllocated<FlowTestCase704Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2747226245629119744UL, 8036751070443620096UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 704 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase704Actor(UnitTestParameters const& params) 
															#line 916 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase704ActorState<FlowTestCase704Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase704", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(16030608593731675648UL, 4844472447775959808UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase704Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase704");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase704", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 704 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase704( UnitTestParameters const& params ) {
															#line 704 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase704Actor(params));
															#line 948 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase704, "/fdbserver/StorageMetricSample/rangeSplitPoints/simple")

#line 724 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 954 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase725()
															#line 725 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase725Actor>
															#line 725 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase725ActorState {
															#line 961 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 725 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase725ActorState(UnitTestParameters const& params) 
															#line 725 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 725 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 968 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase725", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase725ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase725", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 727 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 728 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 730 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 200 * sampleUnit);
															#line 731 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 800 * sampleUnit);
															#line 732 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 733 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 734 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 735 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 736 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 737 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 739 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 600 * sampleUnit, {});
															#line 741 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 3 && t[0] == "Absolute"_sr && t[1] == "Apple"_sr && t[2] == "Bah"_sr);
															#line 743 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase725Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase725ActorState(); static_cast<FlowTestCase725Actor*>(this)->destroy(); return 0; }
															#line 1007 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase725Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase725ActorState();
			static_cast<FlowTestCase725Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase725ActorState();
		static_cast<FlowTestCase725Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 725 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1031 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase725()
															#line 725 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase725Actor final : public Actor<Void>, public FastAllocated<FlowTestCase725Actor>, public FlowTestCase725ActorState<FlowTestCase725Actor> {
															#line 1036 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase725Actor>::operator new;
	using FastAllocated<FlowTestCase725Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2687687484274113280UL, 15752269424401487104UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 725 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase725Actor(UnitTestParameters const& params) 
															#line 1052 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase725ActorState<FlowTestCase725Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase725", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2202044841993097472UL, 15622747507853725952UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase725Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase725");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase725", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 725 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase725( UnitTestParameters const& params ) {
															#line 725 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase725Actor(params));
															#line 1084 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase725, "/fdbserver/StorageMetricSample/rangeSplitPoints/multipleReturnedPoints")

#line 745 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1090 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase746()
															#line 746 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase746Actor>
															#line 746 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase746ActorState {
															#line 1097 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 746 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase746ActorState(UnitTestParameters const& params) 
															#line 746 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 746 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1104 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase746", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase746ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase746", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 748 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 749 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 751 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 200 * sampleUnit);
															#line 752 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 800 * sampleUnit);
															#line 753 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 754 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 755 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 756 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 757 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 758 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 760 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 10000 * sampleUnit, {});
															#line 762 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 0);
															#line 764 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase746Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase746ActorState(); static_cast<FlowTestCase746Actor*>(this)->destroy(); return 0; }
															#line 1143 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase746Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase746ActorState();
			static_cast<FlowTestCase746Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase746ActorState();
		static_cast<FlowTestCase746Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 746 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1167 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase746()
															#line 746 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase746Actor final : public Actor<Void>, public FastAllocated<FlowTestCase746Actor>, public FlowTestCase746ActorState<FlowTestCase746Actor> {
															#line 1172 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase746Actor>::operator new;
	using FastAllocated<FlowTestCase746Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(702771074434463232UL, 7831522698719966464UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 746 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase746Actor(UnitTestParameters const& params) 
															#line 1188 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase746ActorState<FlowTestCase746Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase746", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11576896751573367040UL, 3445149058476419328UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase746Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase746");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase746", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 746 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase746( UnitTestParameters const& params ) {
															#line 746 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase746Actor(params));
															#line 1220 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase746, "/fdbserver/StorageMetricSample/rangeSplitPoints/noneSplitable")

#line 766 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1226 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase767()
															#line 767 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase767Actor>
															#line 767 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase767ActorState {
															#line 1233 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 767 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase767ActorState(UnitTestParameters const& params) 
															#line 767 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 767 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1240 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase767", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase767ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase767", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 769 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 770 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 772 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 773 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 774 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 10 * sampleUnit);
															#line 775 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 776 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 777 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 20 * sampleUnit);
															#line 778 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 10 * sampleUnit);
															#line 779 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 30 * sampleUnit);
															#line 781 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<KeyRef> t = ssm.getSplitPoints(KeyRangeRef("A"_sr, "C"_sr), 1000 * sampleUnit, {});
															#line 783 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 0);
															#line 785 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase767Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase767ActorState(); static_cast<FlowTestCase767Actor*>(this)->destroy(); return 0; }
															#line 1279 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase767Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase767ActorState();
			static_cast<FlowTestCase767Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase767ActorState();
		static_cast<FlowTestCase767Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 767 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1303 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase767()
															#line 767 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase767Actor final : public Actor<Void>, public FastAllocated<FlowTestCase767Actor>, public FlowTestCase767ActorState<FlowTestCase767Actor> {
															#line 1308 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase767Actor>::operator new;
	using FastAllocated<FlowTestCase767Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11809095081622177536UL, 15326265359040200704UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 767 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase767Actor(UnitTestParameters const& params) 
															#line 1324 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase767ActorState<FlowTestCase767Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase767", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(18231450268876123904UL, 6519666288237989888UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase767Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase767");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase767", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 767 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase767( UnitTestParameters const& params ) {
															#line 767 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase767Actor(params));
															#line 1356 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase767, "/fdbserver/StorageMetricSample/rangeSplitPoints/chunkTooLarge")

#line 787 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1362 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase788()
															#line 788 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase788Actor>
															#line 788 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase788ActorState {
															#line 1369 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 788 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase788ActorState(UnitTestParameters const& params) 
															#line 788 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 788 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1376 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase788", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase788ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase788", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 790 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 791 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 793 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 794 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 795 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 796 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 797 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 1000 * sampleUnit);
															#line 799 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 800 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 801 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 802 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 803 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 804 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 805 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 806 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 808 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm._getReadHotRanges(KeyRangeRef("A"_sr, "C"_sr), 2.0, 200 * sampleUnit, 0);
															#line 811 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 1 && (*t.begin()).keys.begin == "Bah"_sr && (*t.begin()).keys.end == "Bob"_sr);
															#line 813 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase788Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase788ActorState(); static_cast<FlowTestCase788Actor*>(this)->destroy(); return 0; }
															#line 1425 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase788Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase788ActorState();
			static_cast<FlowTestCase788Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase788ActorState();
		static_cast<FlowTestCase788Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 788 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1449 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase788()
															#line 788 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase788Actor final : public Actor<Void>, public FastAllocated<FlowTestCase788Actor>, public FlowTestCase788ActorState<FlowTestCase788Actor> {
															#line 1454 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase788Actor>::operator new;
	using FastAllocated<FlowTestCase788Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15965271420228631808UL, 7435085601899237120UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 788 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase788Actor(UnitTestParameters const& params) 
															#line 1470 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase788ActorState<FlowTestCase788Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase788", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12761383513850851840UL, 2718449639609404928UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase788Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase788");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase788", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 788 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase788( UnitTestParameters const& params ) {
															#line 788 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase788Actor(params));
															#line 1502 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase788, "/fdbserver/StorageMetricSample/readHotDetect/simple")

#line 815 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1508 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase816()
															#line 816 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase816Actor>
															#line 816 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase816ActorState {
															#line 1515 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 816 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase816ActorState(UnitTestParameters const& params) 
															#line 816 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 816 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1522 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase816", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase816ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase816", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 818 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 819 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 821 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 822 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 823 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 824 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 825 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 1000 * sampleUnit);
															#line 826 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Final"_sr, 2000 * sampleUnit);
															#line 828 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 829 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 830 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 831 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 832 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 833 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 834 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 835 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 836 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Dah"_sr, 300 * sampleUnit);
															#line 838 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm._getReadHotRanges(KeyRangeRef("A"_sr, "D"_sr), 2.0, 200 * sampleUnit, 0);
															#line 841 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 2 && (*t.begin()).keys.begin == "Bah"_sr && (*t.begin()).keys.end == "Bob"_sr);
															#line 842 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.at(1).keys.begin == "Cat"_sr && t.at(1).keys.end == "Dah"_sr);
															#line 844 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase816Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase816ActorState(); static_cast<FlowTestCase816Actor*>(this)->destroy(); return 0; }
															#line 1577 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase816Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase816ActorState();
			static_cast<FlowTestCase816Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase816ActorState();
		static_cast<FlowTestCase816Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 816 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1601 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase816()
															#line 816 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase816Actor final : public Actor<Void>, public FastAllocated<FlowTestCase816Actor>, public FlowTestCase816ActorState<FlowTestCase816Actor> {
															#line 1606 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase816Actor>::operator new;
	using FastAllocated<FlowTestCase816Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2176629603794611200UL, 13088983444864016640UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 816 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase816Actor(UnitTestParameters const& params) 
															#line 1622 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase816ActorState<FlowTestCase816Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase816", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9142938843537595136UL, 9074228770382408960UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase816Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase816");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase816", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 816 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase816( UnitTestParameters const& params ) {
															#line 816 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase816Actor(params));
															#line 1654 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase816, "/fdbserver/StorageMetricSample/readHotDetect/moreThanOneRange")

#line 846 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1660 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase847()
															#line 847 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase847Actor>
															#line 847 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase847ActorState {
															#line 1667 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 847 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase847ActorState(UnitTestParameters const& params) 
															#line 847 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 847 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1674 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase847", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase847ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase847", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 849 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 850 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 852 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 853 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 854 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Bucket"_sr, 2000 * sampleUnit);
															#line 855 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 856 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 857 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 5000 * sampleUnit);
															#line 858 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Final"_sr, 2000 * sampleUnit);
															#line 860 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20 * sampleUnit);
															#line 861 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80 * sampleUnit);
															#line 862 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 863 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20 * sampleUnit);
															#line 864 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80 * sampleUnit);
															#line 865 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200 * sampleUnit);
															#line 866 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100 * sampleUnit);
															#line 867 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300 * sampleUnit);
															#line 868 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Dah"_sr, 300 * sampleUnit);
															#line 870 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm._getReadHotRanges(KeyRangeRef("A"_sr, "D"_sr), 2.0, 200 * sampleUnit, 0);
															#line 873 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.size() == 2 && (*t.begin()).keys.begin == "Bah"_sr && (*t.begin()).keys.end == "But"_sr);
															#line 874 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT(t.at(1).keys.begin == "Cat"_sr && t.at(1).keys.end == "Dah"_sr);
															#line 876 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase847Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase847ActorState(); static_cast<FlowTestCase847Actor*>(this)->destroy(); return 0; }
															#line 1731 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase847Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase847ActorState();
			static_cast<FlowTestCase847Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase847ActorState();
		static_cast<FlowTestCase847Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 847 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1755 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase847()
															#line 847 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase847Actor final : public Actor<Void>, public FastAllocated<FlowTestCase847Actor>, public FlowTestCase847ActorState<FlowTestCase847Actor> {
															#line 1760 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase847Actor>::operator new;
	using FastAllocated<FlowTestCase847Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(15577303068509677568UL, 2031027049740849664UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 847 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase847Actor(UnitTestParameters const& params) 
															#line 1776 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase847ActorState<FlowTestCase847Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase847", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6782208865185158400UL, 9640981114298156800UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase847Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase847");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase847", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 847 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase847( UnitTestParameters const& params ) {
															#line 847 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase847Actor(params));
															#line 1808 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase847, "/fdbserver/StorageMetricSample/readHotDetect/consecutiveRanges")

#line 878 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"

															#line 1814 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
namespace {
// This generated class is to be used only via flowTestCase879()
															#line 879 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
template <class FlowTestCase879Actor>
															#line 879 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase879ActorState {
															#line 1821 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
															#line 879 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase879ActorState(UnitTestParameters const& params) 
															#line 879 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
															#line 879 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
		 : params(params)
															#line 1828 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
	{
		fdb_probe_actor_create("flowTestCase879", reinterpret_cast<unsigned long>(this));

	}
	~FlowTestCase879ActorState() 
	{
		fdb_probe_actor_destroy("flowTestCase879", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 881 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			int64_t sampleUnit = SERVER_KNOBS->BYTES_READ_UNITS_PER_SAMPLE;
															#line 882 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			StorageServerMetrics ssm;
															#line 886 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Apple"_sr, 1000 * sampleUnit);
															#line 887 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Banana"_sr, 2000 * sampleUnit);
															#line 889 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Bucket"_sr, 2000 * sampleUnit);
															#line 891 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cat"_sr, 1000 * sampleUnit);
															#line 892 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Cathode"_sr, 1000 * sampleUnit);
															#line 894 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Dog"_sr, 5000 * sampleUnit);
															#line 896 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.bytesReadSample.sample.insert("Final"_sr, 2000 * sampleUnit);
															#line 899 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("A"_sr, 20);
															#line 900 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Absolute"_sr, 80);
															#line 901 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Apple"_sr, 1000);
															#line 902 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bah"_sr, 20);
															#line 903 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Banana"_sr, 80);
															#line 904 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Bob"_sr, 200);
															#line 906 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("But"_sr, 100);
															#line 908 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Cat"_sr, 300);
															#line 909 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ssm.byteSample.sample.insert("Dah"_sr, 300);
															#line 912 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			std::vector<ReadHotRangeWithMetrics> t = ssm.getReadHotRanges(KeyRangeRef("Y"_sr, "Z"_sr), 7, ReadHotSubRangeRequest::SplitType::READ_BYTES);
															#line 914 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.size(), 0);
															#line 917 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			t = ssm.getReadHotRanges(KeyRangeRef(""_sr, "\xff"_sr), 7, ReadHotSubRangeRequest::SplitType::READ_BYTES);
															#line 918 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.size(), 5);
															#line 922 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ((*t.begin()).keys.begin, ""_sr);
															#line 924 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ((*t.begin()).keys.end, "Bucket"_sr);
															#line 925 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t[0].bytes, 1400);
															#line 927 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(1).keys.begin, "Bucket"_sr);
															#line 928 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(1).keys.end, "Cat"_sr);
															#line 930 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(2).bytes, 600);
															#line 931 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(3).readBandwidthSec, 5000 * sampleUnit / SERVER_KNOBS->STORAGE_METRICS_AVERAGE_INTERVAL);
															#line 932 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			ASSERT_EQ(t.at(3).bytes, 0);
															#line 933 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
			if (!static_cast<FlowTestCase879Actor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~FlowTestCase879ActorState(); static_cast<FlowTestCase879Actor*>(this)->destroy(); return 0; }
															#line 1903 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
			new (&static_cast<FlowTestCase879Actor*>(this)->SAV< Void >::value()) Void(Void());
			this->~FlowTestCase879ActorState();
			static_cast<FlowTestCase879Actor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~FlowTestCase879ActorState();
		static_cast<FlowTestCase879Actor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
															#line 879 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	UnitTestParameters params;
															#line 1927 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
};
// This generated class is to be used only via flowTestCase879()
															#line 879 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
class FlowTestCase879Actor final : public Actor<Void>, public FastAllocated<FlowTestCase879Actor>, public FlowTestCase879ActorState<FlowTestCase879Actor> {
															#line 1932 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
public:
	using FastAllocated<FlowTestCase879Actor>::operator new;
	using FastAllocated<FlowTestCase879Actor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(14138332577294480384UL, 266686790487736064UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
															#line 879 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	FlowTestCase879Actor(UnitTestParameters const& params) 
															#line 1948 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
		 : Actor<Void>(),
		   FlowTestCase879ActorState<FlowTestCase879Actor>(params),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("flowTestCase879", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(11661159624198825472UL, 15466152518152515584UL);
		ActorExecutionContextHelper __helper(static_cast<FlowTestCase879Actor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("flowTestCase879");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("flowTestCase879", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		}

	}
};
} // namespace
															#line 879 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
static Future<Void> flowTestCase879( UnitTestParameters const& params ) {
															#line 879 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
	return Future<Void>(new FlowTestCase879Actor(params));
															#line 1980 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/build_output/fdbserver/StorageMetrics.actor.g.cpp"
}
ACTOR_TEST_CASE(flowTestCase879, "/fdbserver/StorageMetricSample/readHotDetect/equalDivide")

#line 935 "/codebuild/output/src2680131183/src/github.com/apple/foundationdb/fdbserver/StorageMetrics.actor.cpp"
