#define POST_ACTOR_COMPILER 1
#line 1 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
/*
 * DatabaseContext.cpp
 *
 * This source file is part of the FoundationDB open source project
 *
 * Copyright 2013-2024 Apple Inc. and the FoundationDB project authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// TODO: prune down the list of includes. This was copied from NativeAPI.actor.cpp.
#include "fdbclient/NativeAPI.actor.h"

#include <algorithm>
#include <cstdio>
#include <iterator>
#include <limits>
#include <memory>
#include <regex>
#include <string>
#include <unordered_set>
#include <tuple>
#include <utility>
#include <vector>

#include "boost/algorithm/string.hpp"

#include "fdbclient/Knobs.h"
#include "flow/CodeProbe.h"
#include "fmt/format.h"

#include "fdbclient/FDBOptions.g.h"
#include "fdbclient/FDBTypes.h"
#include "fdbrpc/FailureMonitor.h"
#include "fdbrpc/MultiInterface.h"

#include "fdbclient/ActorLineageProfiler.h"
#include "fdbclient/AnnotateActor.h"
#include "fdbclient/Atomic.h"
#include "fdbclient/ClusterInterface.h"
#include "fdbclient/ClusterConnectionFile.h"
#include "fdbclient/ClusterConnectionMemoryRecord.h"
#include "fdbclient/CoordinationInterface.h"
#include "fdbclient/CommitTransaction.h"
#include "fdbclient/DatabaseContext.h"
#include "fdbclient/GlobalConfig.actor.h"
#include "fdbclient/IKnobCollection.h"
#include "fdbclient/JsonBuilder.h"
#include "fdbclient/KeyBackedTypes.actor.h"
#include "fdbclient/KeyRangeMap.h"
#include "fdbclient/ManagementAPI.actor.h"
#include "fdbclient/NameLineage.h"
#include "fdbclient/CommitProxyInterface.h"
#include "fdbclient/MonitorLeader.h"
#include "fdbclient/MutationList.h"
#include "fdbclient/ParallelStream.actor.h"
#include "fdbclient/ReadYourWrites.h"
#include "fdbclient/SpecialKeySpace.actor.h"
#include "fdbclient/StorageServerInterface.h"
#include "fdbclient/SystemData.h"
#include "fdbclient/TransactionLineage.h"
#include "fdbclient/versions.h"
#include "fdbrpc/WellKnownEndpoints.h"
#include "fdbrpc/LoadBalance.h"
#include "fdbrpc/Net2FileSystem.h"
#include "fdbrpc/simulator.h"
#include "fdbrpc/sim_validation.h"
#include "flow/Arena.h"
#include "flow/ActorCollection.h"
#include "flow/DeterministicRandom.h"
#include "flow/Error.h"
#include "flow/FastRef.h"
#include "flow/GetSourceVersion.h"
#include "flow/IRandom.h"
#include "flow/Trace.h"
#include "flow/ProtocolVersion.h"
#include "flow/flow.h"
#include "flow/genericactors.actor.h"
#include "flow/Knobs.h"
#include "flow/Platform.h"
#include "flow/SystemMonitor.h"
#include "flow/TLSConfig.actor.h"
#include "fdbclient/Tracing.h"
#include "flow/UnitTest.h"
#include "flow/network.h"
#include "flow/serialize.h"

#ifdef ADDRESS_SANITIZER
#include <sanitizer/lsan_interface.h>
#endif

#ifdef WIN32
#define WIN32_LEAN_AND_MEAN
#include <Windows.h>
#undef min
#undef max
#else
#include <time.h>
#endif
#include "flow/actorcompiler.h" // This must be the last #include.

Reference<WatchMetadata> DatabaseContext::getWatchMetadata(KeyRef key) const {
	const auto it = watchMap.find(WatchMapKey(key));
	if (it == watchMap.end())
		return Reference<WatchMetadata>();
	return it->second;
}

void DatabaseContext::setWatchMetadata(Reference<WatchMetadata> metadata) {
	const WatchMapKey key(metadata->parameters->key);
	watchMap[key] = metadata;
	// NOTE Here we do *NOT* update/reset the reference count for the key, see the source code in getWatchFuture.
	// Basically the reference count could be increased, or the same watch is refreshed, or the watch might be cancelled
}

int32_t DatabaseContext::increaseWatchRefCount(KeyRef key, const Version& version) {
	const WatchMapKey mapKey(key);
	watchCounterMap[mapKey].insert(version);
	return watchCounterMap[mapKey].size();
}

int32_t DatabaseContext::decreaseWatchRefCount(KeyRef key, const Version& version) {
	const WatchMapKey mapKey(key);
	auto mapKeyIter = watchCounterMap.find(mapKey);
	if (mapKeyIter == std::end(watchCounterMap)) {
		// Key does not exist. The metadata might be removed by deleteWatchMetadata already.
		return 0;
	}

	auto& versionSet = mapKeyIter->second;
	auto versionIter = versionSet.find(version);

	if (versionIter == std::end(versionSet)) {
		// Version not found, the watch might be cleared before.
		return versionSet.size();
	}
	versionSet.erase(versionIter);

	const auto count = versionSet.size();
	// The metadata might be deleted somewhere else, before calling this decreaseWatchRefCount
	if (auto metadata = getWatchMetadata(key); metadata.isValid() && versionSet.size() == 0) {
		// It is a *must* to cancel the watchFutureSS manually. watchFutureSS waits for watchStorageServerResp, which
		// holds a reference to the metadata. If the ACTOR is not cancelled, it indirectly holds a Future waiting for
		// itself.
		metadata->watchFutureSS.cancel();
		deleteWatchMetadata(key);
	}

	return count;
}

void DatabaseContext::deleteWatchMetadata(KeyRef key, bool removeReferenceCount) {
	const WatchMapKey mapKey(key);
	watchMap.erase(mapKey);
	if (removeReferenceCount) {
		watchCounterMap.erase(mapKey);
	}
}

void DatabaseContext::addTssMapping(StorageServerInterface const& ssi, StorageServerInterface const& tssi) {
	auto result = tssMapping.find(ssi.id());
	// Update tss endpoint mapping if ss isn't in mapping, or the interface it mapped to changed
	if (result == tssMapping.end() ||
	    result->second.getValue.getEndpoint().token.first() != tssi.getValue.getEndpoint().token.first()) {
		Reference<TSSMetrics> metrics;
		if (result == tssMapping.end()) {
			// new TSS pairing
			metrics = makeReference<TSSMetrics>();
			tssMetrics[tssi.id()] = metrics;
			tssMapping[ssi.id()] = tssi;
		} else {
			ASSERT(result->second.id() == tssi.id());
			metrics = tssMetrics[tssi.id()];
			result->second = tssi;
		}

		// data requests duplicated for load and data comparison
		queueModel.updateTssEndpoint(ssi.getValue.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getValue.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getKey.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getKey.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getKeyValues.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getKeyValues.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getMappedKeyValues.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getMappedKeyValues.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getKeyValuesStream.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getKeyValuesStream.getEndpoint(), metrics));

		// non-data requests duplicated for load
		queueModel.updateTssEndpoint(ssi.watchValue.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.watchValue.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.splitMetrics.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.splitMetrics.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getReadHotRanges.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getReadHotRanges.getEndpoint(), metrics));
		queueModel.updateTssEndpoint(ssi.getRangeSplitPoints.getEndpoint().token.first(),
		                             TSSEndpointData(tssi.id(), tssi.getRangeSplitPoints.getEndpoint(), metrics));
	}
}

void DatabaseContext::removeTssMapping(StorageServerInterface const& ssi) {
	auto result = tssMapping.find(ssi.id());
	if (result != tssMapping.end()) {
		tssMetrics.erase(ssi.id());
		tssMapping.erase(result);
		queueModel.removeTssEndpoint(ssi.getValue.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getKey.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getKeyValues.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getMappedKeyValues.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getKeyValuesStream.getEndpoint().token.first());

		queueModel.removeTssEndpoint(ssi.watchValue.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.splitMetrics.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getReadHotRanges.getEndpoint().token.first());
		queueModel.removeTssEndpoint(ssi.getRangeSplitPoints.getEndpoint().token.first());
	}
}

void DatabaseContext::addSSIdTagMapping(const UID& uid, const Tag& tag) {
	ssidTagMapping[uid] = tag;
}

void DatabaseContext::getLatestCommitVersionForSSID(const UID& ssid, Tag& tag, Version& commitVersion) {
	tag = invalidTag;
	commitVersion = invalidVersion;

	auto iter = ssidTagMapping.find(ssid);
	if (iter != ssidTagMapping.end()) {
		tag = iter->second;

		if (ssVersionVectorCache.hasVersion(tag)) {
			commitVersion = ssVersionVectorCache.getVersion(tag);
		}
	}
}

void DatabaseContext::getLatestCommitVersion(const StorageServerInterface& ssi,
                                             Version readVersion,
                                             VersionVector& latestCommitVersion) {
	latestCommitVersion.clear();

	if (ssVersionVectorCache.getMaxVersion() == invalidVersion) {
		return;
	}

	// Error checking (based on the assumption that the read version was not obtained
	// from the client's grv cache).
	if (readVersion > ssVersionVectorCache.getMaxVersion()) {
		TraceEvent(SevError, "ReadVersionExceedsVersionVectorMax")
		    .detail("ReadVersion", readVersion)
		    .detail("VersionVector", ssVersionVectorCache.toString());
		if (g_network->isSimulated()) {
			ASSERT(false);
		} else {
			return; // Do not return a stale commit version in production.
		}
	}

	Tag tag = invalidTag;
	Version commitVersion = invalidVersion;
	getLatestCommitVersionForSSID(ssi.id(), tag, commitVersion);

	if (tag != invalidTag && commitVersion != invalidVersion && commitVersion < readVersion) {
		latestCommitVersion.setVersion(tag, commitVersion);
	}
}

void DatabaseContext::getLatestCommitVersions(const Reference<LocationInfo>& locationInfo,
                                              Reference<TransactionState> info,
                                              VersionVector& latestCommitVersions) {
	latestCommitVersions.clear();

	if (info->readOptions.present() && info->readOptions.get().debugID.present()) {
		g_traceBatch.addEvent(
		    "TransactionDebug", info->readOptions.get().debugID.get().first(), "NativeAPI.getLatestCommitVersions");
	}

	if (!info->readVersionObtainedFromGrvProxy) {
		return;
	}

	if (ssVersionVectorCache.getMaxVersion() == invalidVersion) {
		return;
	}

	if (info->readVersion() > ssVersionVectorCache.getMaxVersion()) {
		if (!CLIENT_KNOBS->FORCE_GRV_CACHE_OFF && !info->options.skipGrvCache && info->options.useGrvCache) {
			return;
		} else {
			TraceEvent(SevError, "GetLatestCommitVersions")
			    .detail("ReadVersion", info->readVersion())
			    .detail("VersionVector", ssVersionVectorCache.toString());
			ASSERT(false);
		}
	}

	std::map<Version, std::set<Tag>> versionMap; // order the versions to be returned
	for (int i = 0; i < locationInfo->locations()->size(); i++) {
		Tag tag = invalidTag;
		Version commitVersion = invalidVersion; // latest commit version
		getLatestCommitVersionForSSID(locationInfo->locations()->getId(i), tag, commitVersion);

		bool updatedVersionMap = false;
		if (tag != invalidTag && commitVersion != invalidVersion && commitVersion < info->readVersion()) {
			updatedVersionMap = true;
			versionMap[commitVersion].insert(tag);
		}

		// Do not log if commitVersion >= readVersion.
		if (!updatedVersionMap && commitVersion == invalidVersion) {
			TraceEvent(SevDebug, "CommitVersionNotFoundForSS")
			    .detail("InSSIDMap", tag != invalidTag ? 1 : 0)
			    .detail("Tag", tag)
			    .detail("CommitVersion", commitVersion)
			    .detail("ReadVersion", info->readVersion())
			    .detail("VersionVector", ssVersionVectorCache.toString())
			    .setMaxEventLength(11000)
			    .setMaxFieldLength(10000);
			++transactionCommitVersionNotFoundForSS;
		}
	}

	// insert the commit versions in the version vector.
	for (auto& iter : versionMap) {
		latestCommitVersions.setVersion(iter.second, iter.first);
	}
}

void updateCachedReadVersionShared(double t, Version v, DatabaseSharedState* p) {
	MutexHolder mutex(p->mutexLock);
	if (v >= p->grvCacheSpace.cachedReadVersion) {
		//TraceEvent(SevDebug, "CacheReadVersionUpdate")
		//    .detail("Version", v)
		//    .detail("CurTime", t)
		//    .detail("LastVersion", p->grvCacheSpace.cachedReadVersion)
		//    .detail("LastTime", p->grvCacheSpace.lastGrvTime);
		p->grvCacheSpace.cachedReadVersion = v;
		if (t > p->grvCacheSpace.lastGrvTime) {
			p->grvCacheSpace.lastGrvTime = t;
		}
	}
}

void DatabaseContext::updateCachedReadVersion(double t, Version v) {
	if (sharedStatePtr) {
		return updateCachedReadVersionShared(t, v, sharedStatePtr);
	}
	if (v >= cachedReadVersion) {
		//TraceEvent(SevDebug, "CachedReadVersionUpdate")
		//    .detail("Version", v)
		//    .detail("GrvStartTime", t)
		//    .detail("LastVersion", cachedReadVersion)
		//    .detail("LastTime", lastGrvTime);
		cachedReadVersion = v;
		// Since the time is based on the start of the request, it's possible that we
		// get a newer version with an older time.
		// (Request started earlier, but was latest to reach the proxy)
		// Only update time when strictly increasing (?)
		if (t > lastGrvTime) {
			lastGrvTime = t;
		}
	}
}

Version DatabaseContext::getCachedReadVersion() {
	if (sharedStatePtr) {
		MutexHolder mutex(sharedStatePtr->mutexLock);
		return sharedStatePtr->grvCacheSpace.cachedReadVersion;
	}
	return cachedReadVersion;
}

double DatabaseContext::getLastGrvTime() {
	if (sharedStatePtr) {
		MutexHolder mutex(sharedStatePtr->mutexLock);
		return sharedStatePtr->grvCacheSpace.lastGrvTime;
	}
	return lastGrvTime;
}

Reference<StorageServerInfo> StorageServerInfo::getInterface(DatabaseContext* cx,
                                                             StorageServerInterface const& ssi,
                                                             LocalityData const& locality) {
	auto it = cx->server_interf.find(ssi.id());
	if (it != cx->server_interf.end()) {
		if (it->second->interf.getValue.getEndpoint().token != ssi.getValue.getEndpoint().token) {
			if (it->second->interf.locality == ssi.locality) {
				// FIXME: load balance holds pointers to individual members of the interface, and this assignment will
				// swap out the object they are
				//       pointing to. This is technically correct, but is very unnatural. We may want to refactor load
				//       balance to take an AsyncVar<Reference<Interface>> so that it is notified when the interface
				//       changes.

				it->second->interf = ssi;
			} else {
				it->second->notifyContextDestroyed();
				Reference<StorageServerInfo> loc(new StorageServerInfo(cx, ssi, locality));
				cx->server_interf[ssi.id()] = loc.getPtr();
				return loc;
			}
		}

		return Reference<StorageServerInfo>::addRef(it->second);
	}

	Reference<StorageServerInfo> loc(new StorageServerInfo(cx, ssi, locality));
	cx->server_interf[ssi.id()] = loc.getPtr();
	return loc;
}

void StorageServerInfo::notifyContextDestroyed() {
	cx = nullptr;
}

StorageServerInfo::~StorageServerInfo() {
	if (cx) {
		auto it = cx->server_interf.find(interf.id());
		if (it != cx->server_interf.end())
			cx->server_interf.erase(it);
		cx = nullptr;
	}
}

void DatabaseContext::validateVersion(Version version) const {
	// Version could be 0 if the INITIALIZE_NEW_DATABASE option is set. In that case, it is illegal to perform any
	// reads. We throw client_invalid_operation because the caller didn't directly set the version, so the
	// version_invalid error might be confusing.
	if (version == 0) {
		throw client_invalid_operation();
	}
	if (switchable && version < minAcceptableReadVersion) {
		CODE_PROBE(true, "Attempted to read a version lower than any this client has seen from the current cluster");
		throw transaction_too_old();
	}

	ASSERT(version > 0 || version == latestVersion);
}

inline HealthMetrics populateHealthMetrics(const HealthMetrics& detailedMetrics, bool detailedOutput) {
	if (detailedOutput) {
		return detailedMetrics;
	} else {
		HealthMetrics result;
		result.update(detailedMetrics, false, false);
		return result;
	}
}

															#line 461 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getHealthMetricsActor()
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
template <class GetHealthMetricsActorActor>
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class GetHealthMetricsActorActorState {
															#line 468 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	GetHealthMetricsActorActorState(DatabaseContext* const& cx,bool const& detailed,bool const& sendDetailedRequest) 
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		 : cx(cx),
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   detailed(detailed),
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   sendDetailedRequest(sendDetailedRequest)
															#line 479 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getHealthMetricsActor", reinterpret_cast<unsigned long>(this));

	}
	~GetHealthMetricsActorActorState() 
	{
		fdb_probe_actor_destroy("getHealthMetricsActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 460 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 494 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetHealthMetricsActorActorState();
		static_cast<GetHealthMetricsActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 462 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_0 = cx->onProxiesChanged();
															#line 461 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<GetHealthMetricsActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 526 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 463 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<GetHealthMetricsReply> __when_expr_1 = basicLoadBalance(cx->getGrvProxies(UseProvisionalProxies::False), &GrvProxyInterface::getHealthMetrics, GetHealthMetricsRequest(sendDetailedRequest));
															#line 530 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<GetHealthMetricsActorActor*>(this)->actor_wait_state = 1;
															#line 462 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetHealthMetricsActorActor, 0, Void >*>(static_cast<GetHealthMetricsActorActor*>(this)));
															#line 463 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >*>(static_cast<GetHealthMetricsActorActor*>(this)));
															#line 537 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(GetHealthMetricsReply const& rep,int loopDepth) 
	{
															#line 466 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->healthMetrics.update(rep.healthMetrics, sendDetailedRequest, true);
															#line 467 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->healthMetricsLastUpdated = now();
															#line 468 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (sendDetailedRequest)
															#line 568 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 469 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			cx->detailedHealthMetricsLastUpdated = now();
															#line 572 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		}
															#line 471 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!static_cast<GetHealthMetricsActorActor*>(this)->SAV<HealthMetrics>::futures) { (void)(populateHealthMetrics(cx->healthMetrics, detailed)); this->~GetHealthMetricsActorActorState(); static_cast<GetHealthMetricsActorActor*>(this)->destroy(); return 0; }
															#line 576 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		new (&static_cast<GetHealthMetricsActorActor*>(this)->SAV< HealthMetrics >::value()) HealthMetrics(populateHealthMetrics(cx->healthMetrics, detailed));
		this->~GetHealthMetricsActorActorState();
		static_cast<GetHealthMetricsActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when2(GetHealthMetricsReply && rep,int loopDepth) 
	{
															#line 466 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->healthMetrics.update(rep.healthMetrics, sendDetailedRequest, true);
															#line 467 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->healthMetricsLastUpdated = now();
															#line 468 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (sendDetailedRequest)
															#line 592 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 469 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			cx->detailedHealthMetricsLastUpdated = now();
															#line 596 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		}
															#line 471 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!static_cast<GetHealthMetricsActorActor*>(this)->SAV<HealthMetrics>::futures) { (void)(populateHealthMetrics(cx->healthMetrics, detailed)); this->~GetHealthMetricsActorActorState(); static_cast<GetHealthMetricsActorActor*>(this)->destroy(); return 0; }
															#line 600 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		new (&static_cast<GetHealthMetricsActorActor*>(this)->SAV< HealthMetrics >::value()) HealthMetrics(populateHealthMetrics(cx->healthMetrics, detailed));
		this->~GetHealthMetricsActorActorState();
		static_cast<GetHealthMetricsActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetHealthMetricsActorActor*>(this)->actor_wait_state > 0) static_cast<GetHealthMetricsActorActor*>(this)->actor_wait_state = 0;
		static_cast<GetHealthMetricsActorActor*>(this)->ActorCallback< GetHealthMetricsActorActor, 0, Void >::remove();
		static_cast<GetHealthMetricsActorActor*>(this)->ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >::remove();

	}
	void a_callback_fire(ActorCallback< GetHealthMetricsActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetHealthMetricsActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetHealthMetricsActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >*,GetHealthMetricsReply const& value) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >*,GetHealthMetricsReply && value) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >*,Error err) 
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	DatabaseContext* cx;
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	bool detailed;
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	bool sendDetailedRequest;
															#line 735 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getHealthMetricsActor()
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class GetHealthMetricsActorActor final : public Actor<HealthMetrics>, public ActorCallback< GetHealthMetricsActorActor, 0, Void >, public ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >, public FastAllocated<GetHealthMetricsActorActor>, public GetHealthMetricsActorActorState<GetHealthMetricsActorActor> {
															#line 740 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetHealthMetricsActorActor>::operator new;
	using FastAllocated<GetHealthMetricsActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7938367474442119936UL, 14916880471160450560UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<HealthMetrics>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetHealthMetricsActorActor, 0, Void >;
friend struct ActorCallback< GetHealthMetricsActorActor, 1, GetHealthMetricsReply >;
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	GetHealthMetricsActorActor(DatabaseContext* const& cx,bool const& detailed,bool const& sendDetailedRequest) 
															#line 758 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		 : Actor<HealthMetrics>(),
		   GetHealthMetricsActorActorState<GetHealthMetricsActorActor>(cx, detailed, sendDetailedRequest),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13925261518677660672UL, 14865277337925754624UL);
		ActorExecutionContextHelper __helper(static_cast<GetHealthMetricsActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getHealthMetricsActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getHealthMetricsActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetHealthMetricsActorActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
[[nodiscard]] static Future<HealthMetrics> getHealthMetricsActor( DatabaseContext* const& cx, bool const& detailed, bool const& sendDetailedRequest ) {
															#line 459 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	return Future<HealthMetrics>(new GetHealthMetricsActorActor(cx, detailed, sendDetailedRequest));
															#line 791 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
}

#line 476 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

Future<HealthMetrics> DatabaseContext::getHealthMetrics(bool detailed = false) {
	if (now() - healthMetricsLastUpdated < CLIENT_KNOBS->AGGREGATE_HEALTH_METRICS_MAX_STALENESS) {
		return populateHealthMetrics(healthMetrics, detailed);
	}
	bool sendDetailedRequest =
	    detailed && now() - detailedHealthMetricsLastUpdated > CLIENT_KNOBS->DETAILED_HEALTH_METRICS_MAX_STALENESS;
	return getHealthMetricsActor(this, detailed, sendDetailedRequest);
}

Future<Optional<HealthMetrics::StorageStats>> DatabaseContext::getStorageStats(const UID& id, double maxStaleness) {
	if (now() - detailedHealthMetricsLastUpdated < maxStaleness) {
		auto it = healthMetrics.storageStats.find(id);
		return it == healthMetrics.storageStats.end() ? Optional<HealthMetrics::StorageStats>() : it->second;
	}

	return map(getHealthMetricsActor(this, true, true), [&id](auto metrics) -> Optional<HealthMetrics::StorageStats> {
		auto it = metrics.storageStats.find(id);
		return it == metrics.storageStats.end() ? Optional<HealthMetrics::StorageStats>() : it->second;
	});
}

// register a special key(s) implementation under the specified module
void DatabaseContext::registerSpecialKeysImpl(SpecialKeySpace::MODULE module,
                                              SpecialKeySpace::IMPLTYPE type,
                                              std::unique_ptr<SpecialKeyRangeReadImpl>&& impl,
                                              int deprecatedVersion) {
	// if deprecated, add the implementation when the api version is less than the deprecated version
	if (deprecatedVersion == -1 || apiVersion.version() < deprecatedVersion) {
		specialKeySpace->registerKeyRange(module, type, impl->getKeyRange(), impl.get());
		specialKeySpaceModules.push_back(std::move(impl));
	}
}

void traceTSSErrors(const char* name, UID tssId, const std::unordered_map<int, uint64_t>& errorsByCode) {
	TraceEvent ev(name, tssId);
	for (auto& it : errorsByCode) {
		ev.detail("E" + std::to_string(it.first), it.second);
	}
}

/*
    For each request type, this will produce
    <Type>Count
    <Type>{SS,TSS}{Mean,P50,P90,P99}
    Example:
    GetValueLatencySSMean
*/
void traceSSOrTSSPercentiles(TraceEvent& ev, const std::string name, DDSketch<double>& sample) {
	ev.detail(name + "Mean", sample.mean());
	// don't log the larger percentiles unless we actually have enough samples to log the accurate percentile instead of
	// the largest sample in this window
	if (sample.getPopulationSize() >= 3) {
		ev.detail(name + "P50", sample.median());
	}
	if (sample.getPopulationSize() >= 10) {
		ev.detail(name + "P90", sample.percentile(0.90));
	}
	if (sample.getPopulationSize() >= 100) {
		ev.detail(name + "P99", sample.percentile(0.99));
	}
}

void traceTSSPercentiles(TraceEvent& ev,
                         const std::string name,
                         DDSketch<double>& ssSample,
                         DDSketch<double>& tssSample) {
	ASSERT(ssSample.getPopulationSize() == tssSample.getPopulationSize());
	ev.detail(name + "Count", ssSample.getPopulationSize());
	if (ssSample.getPopulationSize() > 0) {
		traceSSOrTSSPercentiles(ev, name + "SS", ssSample);
		traceSSOrTSSPercentiles(ev, name + "TSS", tssSample);
	}
}

															#line 870 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via tssLogger()
															#line 551 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
template <class TssLoggerActor>
															#line 551 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class TssLoggerActorState {
															#line 877 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
															#line 551 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	TssLoggerActorState(DatabaseContext* const& cx) 
															#line 551 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
															#line 551 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		 : cx(cx),
															#line 552 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   lastLogged(0)
															#line 886 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("tssLogger", reinterpret_cast<unsigned long>(this));

	}
	~TssLoggerActorState() 
	{
		fdb_probe_actor_destroy("tssLogger", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 553 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 901 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TssLoggerActorState();
		static_cast<TssLoggerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 554 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(CLIENT_KNOBS->TSS_METRICS_LOGGING_INTERVAL, TaskPriority::FlushTrace);
															#line 554 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<TssLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 933 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<TssLoggerActor*>(this)->actor_wait_state = 1;
															#line 554 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TssLoggerActor, 0, Void >*>(static_cast<TssLoggerActor*>(this)));
															#line 938 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 557 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		for( const auto& it : cx->tssMetrics ) {
															#line 558 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (it.second->detailedMismatches.size())
															#line 949 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 559 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				cx->tssMismatchStream.send( std::pair<UID, std::vector<DetailedTSSMismatch>>(it.first, it.second->detailedMismatches));
															#line 953 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 564 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (it.second->ssErrorsByCode.size())
															#line 957 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 565 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				traceTSSErrors("TSS_SSErrors", it.first, it.second->ssErrorsByCode);
															#line 961 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 568 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (it.second->tssErrorsByCode.size())
															#line 965 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 569 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				traceTSSErrors("TSS_TSSErrors", it.first, it.second->tssErrorsByCode);
															#line 969 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 572 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent tssEv("TSSClientMetrics", cx->dbId);
															#line 573 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tssEv.detail("TSSID", it.first) .detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Internal", cx->internal);
															#line 577 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			it.second->cc.logToTraceEvent(tssEv);
															#line 579 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			traceTSSPercentiles(tssEv, "GetValueLatency", it.second->SSgetValueLatency, it.second->TSSgetValueLatency);
															#line 580 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			traceTSSPercentiles( tssEv, "GetKeyValuesLatency", it.second->SSgetKeyValuesLatency, it.second->TSSgetKeyValuesLatency);
															#line 582 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			traceTSSPercentiles(tssEv, "GetKeyLatency", it.second->SSgetKeyLatency, it.second->TSSgetKeyLatency);
															#line 583 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			traceTSSPercentiles(tssEv, "GetMappedKeyValuesLatency", it.second->SSgetMappedKeyValuesLatency, it.second->TSSgetMappedKeyValuesLatency);
															#line 588 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			it.second->clear();
															#line 987 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		}
															#line 591 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		lastLogged = now();
															#line 991 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 557 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		for( const auto& it : cx->tssMetrics ) {
															#line 558 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (it.second->detailedMismatches.size())
															#line 1002 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 559 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				cx->tssMismatchStream.send( std::pair<UID, std::vector<DetailedTSSMismatch>>(it.first, it.second->detailedMismatches));
															#line 1006 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 564 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (it.second->ssErrorsByCode.size())
															#line 1010 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 565 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				traceTSSErrors("TSS_SSErrors", it.first, it.second->ssErrorsByCode);
															#line 1014 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 568 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (it.second->tssErrorsByCode.size())
															#line 1018 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 569 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				traceTSSErrors("TSS_TSSErrors", it.first, it.second->tssErrorsByCode);
															#line 1022 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 572 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent tssEv("TSSClientMetrics", cx->dbId);
															#line 573 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tssEv.detail("TSSID", it.first) .detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Internal", cx->internal);
															#line 577 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			it.second->cc.logToTraceEvent(tssEv);
															#line 579 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			traceTSSPercentiles(tssEv, "GetValueLatency", it.second->SSgetValueLatency, it.second->TSSgetValueLatency);
															#line 580 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			traceTSSPercentiles( tssEv, "GetKeyValuesLatency", it.second->SSgetKeyValuesLatency, it.second->TSSgetKeyValuesLatency);
															#line 582 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			traceTSSPercentiles(tssEv, "GetKeyLatency", it.second->SSgetKeyLatency, it.second->TSSgetKeyLatency);
															#line 583 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			traceTSSPercentiles(tssEv, "GetMappedKeyValuesLatency", it.second->SSgetMappedKeyValuesLatency, it.second->TSSgetMappedKeyValuesLatency);
															#line 588 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			it.second->clear();
															#line 1040 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		}
															#line 591 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		lastLogged = now();
															#line 1044 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TssLoggerActor*>(this)->actor_wait_state > 0) static_cast<TssLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<TssLoggerActor*>(this)->ActorCallback< TssLoggerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TssLoggerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("tssLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TssLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tssLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TssLoggerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("tssLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TssLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tssLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TssLoggerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("tssLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TssLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("tssLogger", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 551 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	DatabaseContext* cx;
															#line 552 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	double lastLogged;
															#line 1128 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
};
// This generated class is to be used only via tssLogger()
															#line 551 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class TssLoggerActor final : public Actor<Void>, public ActorCallback< TssLoggerActor, 0, Void >, public FastAllocated<TssLoggerActor>, public TssLoggerActorState<TssLoggerActor> {
															#line 1133 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
	using FastAllocated<TssLoggerActor>::operator new;
	using FastAllocated<TssLoggerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7494939151537963520UL, 12934904361372836608UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TssLoggerActor, 0, Void >;
															#line 551 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	TssLoggerActor(DatabaseContext* const& cx) 
															#line 1150 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   TssLoggerActorState<TssLoggerActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("tssLogger", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9014176353709562112UL, 14021610819005956096UL);
		ActorExecutionContextHelper __helper(static_cast<TssLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("tssLogger");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("tssLogger", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TssLoggerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 551 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
[[nodiscard]] Future<Void> tssLogger( DatabaseContext* const& cx ) {
															#line 551 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	return Future<Void>(new TssLoggerActor(cx));
															#line 1183 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
}

#line 594 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

															#line 1188 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via databaseLogger()
															#line 595 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
template <class DatabaseLoggerActor>
															#line 595 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class DatabaseLoggerActorState {
															#line 1195 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
															#line 595 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	DatabaseLoggerActorState(DatabaseContext* const& cx) 
															#line 595 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
															#line 595 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		 : cx(cx),
															#line 596 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   lastLogged(0)
															#line 1204 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("databaseLogger", reinterpret_cast<unsigned long>(this));

	}
	~DatabaseLoggerActorState() 
	{
		fdb_probe_actor_destroy("databaseLogger", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 597 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 1219 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DatabaseLoggerActorState();
		static_cast<DatabaseLoggerActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 598 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(CLIENT_KNOBS->SYSTEM_MONITOR_INTERVAL, TaskPriority::FlushTrace);
															#line 598 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<DatabaseLoggerActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1251 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<DatabaseLoggerActor*>(this)->actor_wait_state = 1;
															#line 598 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DatabaseLoggerActor, 0, Void >*>(static_cast<DatabaseLoggerActor*>(this)));
															#line 1256 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 600 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		bool logMetrics = !g_network->isSimulated() || BUGGIFY_WITH_PROB(0.01);
															#line 601 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (logMetrics)
															#line 1267 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 602 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent ev("TransactionMetrics", cx->dbId);
															#line 604 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ev.detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Cluster", cx->getConnectionRecord() ? cx->getConnectionRecord()->getConnectionString().clusterKeyName().toString() : "") .detail("Internal", cx->internal);
															#line 611 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			cx->cc.logToTraceEvent(ev);
															#line 613 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ev.detail("LocationCacheEntryCount", cx->locationCache.size());
															#line 614 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ev.detail("MeanLatency", cx->latencies.mean()) .detail("MedianLatency", cx->latencies.median()) .detail("Latency90", cx->latencies.percentile(0.90)) .detail("Latency98", cx->latencies.percentile(0.98)) .detail("MaxLatency", cx->latencies.max()) .detail("MeanRowReadLatency", cx->readLatencies.mean()) .detail("MedianRowReadLatency", cx->readLatencies.median()) .detail("MaxRowReadLatency", cx->readLatencies.max()) .detail("MeanGRVLatency", cx->GRVLatencies.mean()) .detail("MedianGRVLatency", cx->GRVLatencies.median()) .detail("MaxGRVLatency", cx->GRVLatencies.max()) .detail("MeanCommitLatency", cx->commitLatencies.mean()) .detail("MedianCommitLatency", cx->commitLatencies.median()) .detail("MaxCommitLatency", cx->commitLatencies.max()) .detail("MeanMutationsPerCommit", cx->mutationsPerCommit.mean()) .detail("MedianMutationsPerCommit", cx->mutationsPerCommit.median()) .detail("MaxMutationsPerCommit", cx->mutationsPerCommit.max()) .detail("MeanBytesPerCommit", cx->bytesPerCommit.mean()) .detail("MedianBytesPerCommit", cx->bytesPerCommit.median()) .detail("MaxBytesPerCommit", cx->bytesPerCommit.max()) .detail("NumLocalityCacheEntries", cx->locationCache.size());
															#line 1279 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		}
															#line 637 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->latencies.clear();
															#line 638 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->readLatencies.clear();
															#line 639 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->GRVLatencies.clear();
															#line 640 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->commitLatencies.clear();
															#line 641 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->mutationsPerCommit.clear();
															#line 642 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->bytesPerCommit.clear();
															#line 644 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		lastLogged = now();
															#line 1295 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 600 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		bool logMetrics = !g_network->isSimulated() || BUGGIFY_WITH_PROB(0.01);
															#line 601 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (logMetrics)
															#line 1306 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 602 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent ev("TransactionMetrics", cx->dbId);
															#line 604 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ev.detail("Elapsed", (lastLogged == 0) ? 0 : now() - lastLogged) .detail("Cluster", cx->getConnectionRecord() ? cx->getConnectionRecord()->getConnectionString().clusterKeyName().toString() : "") .detail("Internal", cx->internal);
															#line 611 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			cx->cc.logToTraceEvent(ev);
															#line 613 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ev.detail("LocationCacheEntryCount", cx->locationCache.size());
															#line 614 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ev.detail("MeanLatency", cx->latencies.mean()) .detail("MedianLatency", cx->latencies.median()) .detail("Latency90", cx->latencies.percentile(0.90)) .detail("Latency98", cx->latencies.percentile(0.98)) .detail("MaxLatency", cx->latencies.max()) .detail("MeanRowReadLatency", cx->readLatencies.mean()) .detail("MedianRowReadLatency", cx->readLatencies.median()) .detail("MaxRowReadLatency", cx->readLatencies.max()) .detail("MeanGRVLatency", cx->GRVLatencies.mean()) .detail("MedianGRVLatency", cx->GRVLatencies.median()) .detail("MaxGRVLatency", cx->GRVLatencies.max()) .detail("MeanCommitLatency", cx->commitLatencies.mean()) .detail("MedianCommitLatency", cx->commitLatencies.median()) .detail("MaxCommitLatency", cx->commitLatencies.max()) .detail("MeanMutationsPerCommit", cx->mutationsPerCommit.mean()) .detail("MedianMutationsPerCommit", cx->mutationsPerCommit.median()) .detail("MaxMutationsPerCommit", cx->mutationsPerCommit.max()) .detail("MeanBytesPerCommit", cx->bytesPerCommit.mean()) .detail("MedianBytesPerCommit", cx->bytesPerCommit.median()) .detail("MaxBytesPerCommit", cx->bytesPerCommit.max()) .detail("NumLocalityCacheEntries", cx->locationCache.size());
															#line 1318 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		}
															#line 637 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->latencies.clear();
															#line 638 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->readLatencies.clear();
															#line 639 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->GRVLatencies.clear();
															#line 640 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->commitLatencies.clear();
															#line 641 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->mutationsPerCommit.clear();
															#line 642 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->bytesPerCommit.clear();
															#line 644 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		lastLogged = now();
															#line 1334 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DatabaseLoggerActor*>(this)->actor_wait_state > 0) static_cast<DatabaseLoggerActor*>(this)->actor_wait_state = 0;
		static_cast<DatabaseLoggerActor*>(this)->ActorCallback< DatabaseLoggerActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DatabaseLoggerActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("databaseLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DatabaseLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("databaseLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DatabaseLoggerActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("databaseLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DatabaseLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("databaseLogger", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DatabaseLoggerActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("databaseLogger", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DatabaseLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("databaseLogger", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 595 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	DatabaseContext* cx;
															#line 596 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	double lastLogged;
															#line 1418 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
};
// This generated class is to be used only via databaseLogger()
															#line 595 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class DatabaseLoggerActor final : public Actor<Void>, public ActorCallback< DatabaseLoggerActor, 0, Void >, public FastAllocated<DatabaseLoggerActor>, public DatabaseLoggerActorState<DatabaseLoggerActor> {
															#line 1423 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
	using FastAllocated<DatabaseLoggerActor>::operator new;
	using FastAllocated<DatabaseLoggerActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(11461151301731260928UL, 6357524419572125440UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DatabaseLoggerActor, 0, Void >;
															#line 595 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	DatabaseLoggerActor(DatabaseContext* const& cx) 
															#line 1440 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   DatabaseLoggerActorState<DatabaseLoggerActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("databaseLogger", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(13968243465581786624UL, 631654202351916032UL);
		ActorExecutionContextHelper __helper(static_cast<DatabaseLoggerActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("databaseLogger");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("databaseLogger", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DatabaseLoggerActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 595 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
[[nodiscard]] Future<Void> databaseLogger( DatabaseContext* const& cx ) {
															#line 595 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	return Future<Void>(new DatabaseLoggerActor(cx));
															#line 1473 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
}

#line 647 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

struct TrInfoChunk {
	ValueRef value;
	Key key;
};

static const Key CLIENT_LATENCY_INFO_PREFIX = "client_latency/"_sr;
static const Key CLIENT_LATENCY_INFO_CTR_PREFIX = "client_latency_counter/"_sr;

															#line 1486 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via transactionInfoCommitActor()
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
template <class TransactionInfoCommitActorActor>
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class TransactionInfoCommitActorActorState {
															#line 1493 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	TransactionInfoCommitActorActorState(Transaction* const& tr,std::vector<TrInfoChunk>* const& chunks) 
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		 : tr(tr),
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   chunks(chunks),
															#line 657 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   clientLatencyAtomicCtr(CLIENT_LATENCY_INFO_CTR_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin)),
															#line 658 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   retryCount(0)
															#line 1506 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this));

	}
	~TransactionInfoCommitActorActorState() 
	{
		fdb_probe_actor_destroy("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 659 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 1521 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~TransactionInfoCommitActorActorState();
		static_cast<TransactionInfoCommitActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 661 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->reset();
															#line 662 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 663 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 664 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			vstamp = tr->getVersionstamp();
															#line 665 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			int64_t numCommitBytes = 0;
															#line 666 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			for( auto& chunk : *chunks ) {
															#line 667 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				tr->atomicOp(chunk.key, chunk.value, MutationRef::SetVersionstampedKey);
															#line 668 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				numCommitBytes += chunk.key.size() + chunk.value.size() - 4;
															#line 1566 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 671 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->atomicOp(clientLatencyAtomicCtr, StringRef((uint8_t*)&numCommitBytes, 8), MutationRef::AddValue);
															#line 672 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_0 = tr->commit();
															#line 672 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1574 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state = 1;
															#line 672 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< TransactionInfoCommitActorActor, 0, Void >*>(static_cast<TransactionInfoCommitActorActor*>(this)));
															#line 1579 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 675 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			retryCount++;
															#line 676 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (retryCount == 10)
															#line 1603 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 677 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 1607 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 678 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_1 = tr->onError(e);
															#line 678 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 1613 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_1.get(), loopDepth); };
			static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state = 2;
															#line 678 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< TransactionInfoCommitActorActor, 1, Void >*>(static_cast<TransactionInfoCommitActorActor*>(this)));
															#line 1618 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
															#line 673 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!static_cast<TransactionInfoCommitActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TransactionInfoCommitActorActorState(); static_cast<TransactionInfoCommitActorActor*>(this)->destroy(); return 0; }
															#line 1633 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		new (&static_cast<TransactionInfoCommitActorActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TransactionInfoCommitActorActorState();
		static_cast<TransactionInfoCommitActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
															#line 673 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!static_cast<TransactionInfoCommitActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~TransactionInfoCommitActorActorState(); static_cast<TransactionInfoCommitActorActor*>(this)->destroy(); return 0; }
															#line 1645 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		new (&static_cast<TransactionInfoCommitActorActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~TransactionInfoCommitActorActorState();
		static_cast<TransactionInfoCommitActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state > 0) static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state = 0;
		static_cast<TransactionInfoCommitActorActor*>(this)->ActorCallback< TransactionInfoCommitActorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TransactionInfoCommitActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< TransactionInfoCommitActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< TransactionInfoCommitActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state > 0) static_cast<TransactionInfoCommitActorActor*>(this)->actor_wait_state = 0;
		static_cast<TransactionInfoCommitActorActor*>(this)->ActorCallback< TransactionInfoCommitActorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< TransactionInfoCommitActorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< TransactionInfoCommitActorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< TransactionInfoCommitActorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	Transaction* tr;
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	std::vector<TrInfoChunk>* chunks;
															#line 657 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	const Key clientLatencyAtomicCtr;
															#line 658 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	int retryCount;
															#line 664 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	Future<Standalone<StringRef>> vstamp;
															#line 1825 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
};
// This generated class is to be used only via transactionInfoCommitActor()
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class TransactionInfoCommitActorActor final : public Actor<Void>, public ActorCallback< TransactionInfoCommitActorActor, 0, Void >, public ActorCallback< TransactionInfoCommitActorActor, 1, Void >, public FastAllocated<TransactionInfoCommitActorActor>, public TransactionInfoCommitActorActorState<TransactionInfoCommitActorActor> {
															#line 1830 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
	using FastAllocated<TransactionInfoCommitActorActor>::operator new;
	using FastAllocated<TransactionInfoCommitActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(608273022015797248UL, 968706468587605248UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< TransactionInfoCommitActorActor, 0, Void >;
friend struct ActorCallback< TransactionInfoCommitActorActor, 1, Void >;
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	TransactionInfoCommitActorActor(Transaction* const& tr,std::vector<TrInfoChunk>* const& chunks) 
															#line 1848 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   TransactionInfoCommitActorActorState<TransactionInfoCommitActorActor>(tr, chunks),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(10077237674783052032UL, 3830813435306680320UL);
		ActorExecutionContextHelper __helper(static_cast<TransactionInfoCommitActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("transactionInfoCommitActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("transactionInfoCommitActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< TransactionInfoCommitActorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< TransactionInfoCommitActorActor, 1, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
[[nodiscard]] static Future<Void> transactionInfoCommitActor( Transaction* const& tr, std::vector<TrInfoChunk>* const& chunks ) {
															#line 656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	return Future<Void>(new TransactionInfoCommitActorActor(tr, chunks));
															#line 1882 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
}

#line 682 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

															#line 1887 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via delExcessClntTxnEntriesActor()
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
template <class DelExcessClntTxnEntriesActorActor>
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class DelExcessClntTxnEntriesActorActorState {
															#line 1894 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	DelExcessClntTxnEntriesActorActorState(Transaction* const& tr,int64_t const& clientTxInfoSizeLimit) 
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		 : tr(tr),
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   clientTxInfoSizeLimit(clientTxInfoSizeLimit),
															#line 684 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   clientLatencyName(CLIENT_LATENCY_INFO_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin)),
															#line 685 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   clientLatencyAtomicCtr(CLIENT_LATENCY_INFO_CTR_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin))
															#line 1907 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this));

	}
	~DelExcessClntTxnEntriesActorActorState() 
	{
		fdb_probe_actor_destroy("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 686 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent(SevInfo, "DelExcessClntTxnEntriesCalled").log();
															#line 690 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			retries = 0;
															#line 691 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 1926 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~DelExcessClntTxnEntriesActorActorState();
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
		try {
															#line 693 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->reset();
															#line 694 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 695 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->setOption(FDBTransactionOptions::LOCK_AWARE);
															#line 696 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Optional<Value>> __when_expr_0 = tr->get(KeyRef(clientLatencyAtomicCtr), Snapshot::True);
															#line 696 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 1965 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1loopBody1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 1;
															#line 696 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 1970 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 731 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (e.code() == error_code_actor_cancelled || retries++ >= 10)
															#line 1992 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 732 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 1996 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 735 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr->onError(e);
															#line 735 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2002 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 4;
															#line 735 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 2007 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> const& ctrValue,int loopDepth) 
	{
															#line 697 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!ctrValue.present())
															#line 2022 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 698 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent(SevInfo, "NumClntTxnEntriesNotFound").log();
															#line 699 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (!static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DelExcessClntTxnEntriesActorActorState(); static_cast<DelExcessClntTxnEntriesActorActor*>(this)->destroy(); return 0; }
															#line 2028 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			new (&static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DelExcessClntTxnEntriesActorActorState();
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 701 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		txInfoSize = 0;
															#line 702 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		ASSERT(ctrValue.get().size() == sizeof(int64_t));
															#line 703 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		memcpy(&txInfoSize, ctrValue.get().begin(), ctrValue.get().size());
															#line 704 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (txInfoSize < clientTxInfoSizeLimit)
															#line 2042 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 705 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (!static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DelExcessClntTxnEntriesActorActorState(); static_cast<DelExcessClntTxnEntriesActorActor*>(this)->destroy(); return 0; }
															#line 2046 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			new (&static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DelExcessClntTxnEntriesActorActorState();
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 706 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		int getRangeByteLimit = (txInfoSize - clientTxInfoSizeLimit) < CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT ? (txInfoSize - clientTxInfoSizeLimit) : CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT;
															#line 709 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		GetRangeLimits limit(GetRangeLimits::ROW_LIMIT_UNLIMITED, getRangeByteLimit);
															#line 710 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = tr->getRange(KeyRangeRef(clientLatencyName, strinc(clientLatencyName)), limit);
															#line 710 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2060 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 2;
															#line 710 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 2065 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2(Optional<Value> && ctrValue,int loopDepth) 
	{
															#line 697 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!ctrValue.present())
															#line 2074 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 698 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent(SevInfo, "NumClntTxnEntriesNotFound").log();
															#line 699 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (!static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DelExcessClntTxnEntriesActorActorState(); static_cast<DelExcessClntTxnEntriesActorActor*>(this)->destroy(); return 0; }
															#line 2080 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			new (&static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DelExcessClntTxnEntriesActorActorState();
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 701 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		txInfoSize = 0;
															#line 702 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		ASSERT(ctrValue.get().size() == sizeof(int64_t));
															#line 703 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		memcpy(&txInfoSize, ctrValue.get().begin(), ctrValue.get().size());
															#line 704 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (txInfoSize < clientTxInfoSizeLimit)
															#line 2094 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 705 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (!static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DelExcessClntTxnEntriesActorActorState(); static_cast<DelExcessClntTxnEntriesActorActor*>(this)->destroy(); return 0; }
															#line 2098 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			new (&static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DelExcessClntTxnEntriesActorActorState();
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
															#line 706 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		int getRangeByteLimit = (txInfoSize - clientTxInfoSizeLimit) < CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT ? (txInfoSize - clientTxInfoSizeLimit) : CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT;
															#line 709 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		GetRangeLimits limit(GetRangeLimits::ROW_LIMIT_UNLIMITED, getRangeByteLimit);
															#line 710 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<RangeResult> __when_expr_1 = tr->getRange(KeyRangeRef(clientLatencyName, strinc(clientLatencyName)), limit);
															#line 710 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2112 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1loopBody1Catch1(__when_expr_1.getError(), loopDepth); else return a_body1loopBody1cont2when1(__when_expr_1.get(), loopDepth); };
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 2;
															#line 710 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 2117 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> const& ctrValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(ctrValue, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Optional<Value> && ctrValue,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(ctrValue), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state > 0) static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 0;
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >::remove();

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >*,Optional<Value> const& value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >*,Optional<Value> && value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >*,Error err) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont3(RangeResult const& txEntries,int loopDepth) 
	{
															#line 712 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		numBytesToDel = 0;
															#line 713 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		KeyRef endKey;
															#line 714 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		for( auto& kv : txEntries ) {
															#line 715 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			endKey = kv.key;
															#line 716 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			numBytesToDel += kv.key.size() + kv.value.size();
															#line 717 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (txInfoSize - numBytesToDel <= clientTxInfoSizeLimit)
															#line 2211 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
				break;
			}
		}
															#line 720 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (numBytesToDel)
															#line 2218 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 721 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->clear(KeyRangeRef(txEntries[0].key, strinc(endKey)));
															#line 722 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent(SevInfo, "DeletingExcessCntTxnEntries").detail("BytesToBeDeleted", numBytesToDel);
															#line 723 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			int64_t bytesDel = -numBytesToDel;
															#line 725 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->atomicOp(clientLatencyAtomicCtr, StringRef((uint8_t*)&bytesDel, 8), MutationRef::AddValue);
															#line 726 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr->commit();
															#line 726 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2232 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 3;
															#line 726 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 2237 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont3(RangeResult && txEntries,int loopDepth) 
	{
															#line 712 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		numBytesToDel = 0;
															#line 713 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		KeyRef endKey;
															#line 714 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		for( auto& kv : txEntries ) {
															#line 715 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			endKey = kv.key;
															#line 716 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			numBytesToDel += kv.key.size() + kv.value.size();
															#line 717 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (txInfoSize - numBytesToDel <= clientTxInfoSizeLimit)
															#line 2261 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
				break;
			}
		}
															#line 720 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (numBytesToDel)
															#line 2268 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 721 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->clear(KeyRangeRef(txEntries[0].key, strinc(endKey)));
															#line 722 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent(SevInfo, "DeletingExcessCntTxnEntries").detail("BytesToBeDeleted", numBytesToDel);
															#line 723 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			int64_t bytesDel = -numBytesToDel;
															#line 725 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->atomicOp(clientLatencyAtomicCtr, StringRef((uint8_t*)&bytesDel, 8), MutationRef::AddValue);
															#line 726 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr->commit();
															#line 726 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 2282 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont3when1(__when_expr_2.get(), loopDepth); };
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 3;
															#line 726 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*>(static_cast<DelExcessClntTxnEntriesActorActor*>(this)));
															#line 2287 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont6(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult const& txEntries,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(txEntries, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2when1(RangeResult && txEntries,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(std::move(txEntries), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state > 0) static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 0;
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >::remove();

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*,RangeResult const& value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*,RangeResult && value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*,Error err) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
															#line 728 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (txInfoSize - numBytesToDel <= clientTxInfoSizeLimit)
															#line 2376 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 729 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (!static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~DelExcessClntTxnEntriesActorActorState(); static_cast<DelExcessClntTxnEntriesActorActor*>(this)->destroy(); return 0; }
															#line 2380 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			new (&static_cast<DelExcessClntTxnEntriesActorActor*>(this)->SAV< Void >::value()) Void(Void());
			this->~DelExcessClntTxnEntriesActorActorState();
			static_cast<DelExcessClntTxnEntriesActorActor*>(this)->finishSendAndDelPromiseRef();
			return 0;
		}
		loopDepth = a_body1loopBody1cont12(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont6(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont3when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state > 0) static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 0;
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont3when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont12(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state > 0) static_cast<DelExcessClntTxnEntriesActorActor*>(this)->actor_wait_state = 0;
		static_cast<DelExcessClntTxnEntriesActorActor*>(this)->ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	Transaction* tr;
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	int64_t clientTxInfoSizeLimit;
															#line 684 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	const Key clientLatencyName;
															#line 685 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	const Key clientLatencyAtomicCtr;
															#line 690 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	int retries;
															#line 701 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	int64_t txInfoSize;
															#line 712 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	int64_t numBytesToDel;
															#line 2591 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
};
// This generated class is to be used only via delExcessClntTxnEntriesActor()
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class DelExcessClntTxnEntriesActorActor final : public Actor<Void>, public ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >, public ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >, public ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >, public ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >, public FastAllocated<DelExcessClntTxnEntriesActorActor>, public DelExcessClntTxnEntriesActorActorState<DelExcessClntTxnEntriesActorActor> {
															#line 2596 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
	using FastAllocated<DelExcessClntTxnEntriesActorActor>::operator new;
	using FastAllocated<DelExcessClntTxnEntriesActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(17454751657473048064UL, 8934240984453739008UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >;
friend struct ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >;
friend struct ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >;
friend struct ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >;
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	DelExcessClntTxnEntriesActorActor(Transaction* const& tr,int64_t const& clientTxInfoSizeLimit) 
															#line 2616 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   DelExcessClntTxnEntriesActorActorState<DelExcessClntTxnEntriesActorActor>(tr, clientTxInfoSizeLimit),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(12258001942667996416UL, 1429686287236455680UL);
		ActorExecutionContextHelper __helper(static_cast<DelExcessClntTxnEntriesActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("delExcessClntTxnEntriesActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("delExcessClntTxnEntriesActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< DelExcessClntTxnEntriesActorActor, 0, Optional<Value> >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< DelExcessClntTxnEntriesActorActor, 1, RangeResult >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< DelExcessClntTxnEntriesActorActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< DelExcessClntTxnEntriesActorActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
[[nodiscard]] static Future<Void> delExcessClntTxnEntriesActor( Transaction* const& tr, int64_t const& clientTxInfoSizeLimit ) {
															#line 683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	return Future<Void>(new DelExcessClntTxnEntriesActorActor(tr, clientTxInfoSizeLimit));
															#line 2652 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
}

#line 739 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

// FIXME: explain what "client status" is
// The reason for getting a pointer to DatabaseContext instead of a reference counted object is because reference
// counting will increment reference count for DatabaseContext which holds the future of this actor. This creates a
// cyclic reference and hence this actor and Database object will not be destroyed at all.
															#line 2661 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via clientStatusUpdateActor()
															#line 744 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
template <class ClientStatusUpdateActorActor>
															#line 744 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class ClientStatusUpdateActorActorState {
															#line 2668 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
															#line 744 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	ClientStatusUpdateActorActorState(DatabaseContext* const& cx) 
															#line 744 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
															#line 744 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		 : cx(cx),
															#line 745 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   clientLatencyName(CLIENT_LATENCY_INFO_PREFIX.withPrefix(fdbClientInfoPrefixRange.begin).toString()),
															#line 747 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   tr(),
															#line 748 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   commitQ(),
															#line 749 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   txBytes(0)
															#line 2683 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this));

	}
	~ClientStatusUpdateActorActorState() 
	{
		fdb_probe_actor_destroy("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 751 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 2698 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~ClientStatusUpdateActorActorState();
		static_cast<ClientStatusUpdateActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 754 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_0 = cx->onConnected();
															#line 754 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2730 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 1;
															#line 754 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 0, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 2735 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 758 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_1 = refreshTransaction(cx, &tr);
															#line 758 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2746 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 2;
															#line 758 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 1, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 2751 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 758 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_1 = refreshTransaction(cx, &tr);
															#line 758 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 2762 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.get(), loopDepth); };
		static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 2;
															#line 758 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 1, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 2767 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(Void const& _,int loopDepth) 
	{
		try {
															#line 760 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ASSERT(cx->clientStatusUpdater.outStatusQ.empty());
															#line 761 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			cx->clientStatusUpdater.inStatusQ.swap(cx->clientStatusUpdater.outStatusQ);
															#line 763 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			trChunksQ = std::vector<TrInfoChunk>();
															#line 764 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			for( auto& entry : cx->clientStatusUpdater.outStatusQ ) {
															#line 765 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				auto& bw = entry.second;
															#line 766 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				int64_t value_size_limit = BUGGIFY ? deterministicRandom()->randomInt(1e3, CLIENT_KNOBS->VALUE_SIZE_LIMIT) : CLIENT_KNOBS->VALUE_SIZE_LIMIT;
															#line 769 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				int num_chunks = (bw.getLength() + value_size_limit - 1) / value_size_limit;
															#line 770 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				std::string random_id = deterministicRandom()->randomAlphaNumeric(16);
															#line 771 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				std::string user_provided_id = entry.first.size() ? entry.first + "/" : "";
															#line 772 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				for(int i = 0;i < num_chunks;i++) {
															#line 773 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					TrInfoChunk chunk;
															#line 774 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					BinaryWriter chunkBW(Unversioned());
															#line 775 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					chunkBW << bigEndian32(i + 1) << bigEndian32(num_chunks);
															#line 776 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					chunk.key = KeyRef(clientLatencyName + std::string(10, '\x00') + "/" + random_id + "/" + chunkBW.toValue().toString() + "/" + user_provided_id + std::string(4, '\x00'));
															#line 778 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					int32_t pos = littleEndian32(clientLatencyName.size());
															#line 779 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					memcpy(mutateString(chunk.key) + chunk.key.size() - sizeof(int32_t), &pos, sizeof(int32_t));
															#line 780 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					if (i == num_chunks - 1)
															#line 2884 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
					{
															#line 781 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
						chunk.value = ValueRef(static_cast<uint8_t*>(bw.getData()) + (i * value_size_limit), bw.getLength() - (i * value_size_limit));
															#line 2888 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
					}
					else
					{
															#line 784 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
						chunk.value = ValueRef(static_cast<uint8_t*>(bw.getData()) + (i * value_size_limit), value_size_limit);
															#line 2894 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
					}
															#line 787 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					trChunksQ.push_back(std::move(chunk));
															#line 2898 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
				}
			}
															#line 792 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			dataSizeLimit = BUGGIFY ? deterministicRandom()->randomInt(200e3, 1.5 * CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT) : 0.8 * CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT;
															#line 795 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tracking_iter = trChunksQ.begin();
															#line 796 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ASSERT(commitQ.empty() && (txBytes == 0));
															#line 797 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 2909 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2(Void && _,int loopDepth) 
	{
		try {
															#line 760 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ASSERT(cx->clientStatusUpdater.outStatusQ.empty());
															#line 761 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			cx->clientStatusUpdater.inStatusQ.swap(cx->clientStatusUpdater.outStatusQ);
															#line 763 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			trChunksQ = std::vector<TrInfoChunk>();
															#line 764 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			for( auto& entry : cx->clientStatusUpdater.outStatusQ ) {
															#line 765 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				auto& bw = entry.second;
															#line 766 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				int64_t value_size_limit = BUGGIFY ? deterministicRandom()->randomInt(1e3, CLIENT_KNOBS->VALUE_SIZE_LIMIT) : CLIENT_KNOBS->VALUE_SIZE_LIMIT;
															#line 769 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				int num_chunks = (bw.getLength() + value_size_limit - 1) / value_size_limit;
															#line 770 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				std::string random_id = deterministicRandom()->randomAlphaNumeric(16);
															#line 771 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				std::string user_provided_id = entry.first.size() ? entry.first + "/" : "";
															#line 772 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				for(int i = 0;i < num_chunks;i++) {
															#line 773 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					TrInfoChunk chunk;
															#line 774 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					BinaryWriter chunkBW(Unversioned());
															#line 775 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					chunkBW << bigEndian32(i + 1) << bigEndian32(num_chunks);
															#line 776 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					chunk.key = KeyRef(clientLatencyName + std::string(10, '\x00') + "/" + random_id + "/" + chunkBW.toValue().toString() + "/" + user_provided_id + std::string(4, '\x00'));
															#line 778 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					int32_t pos = littleEndian32(clientLatencyName.size());
															#line 779 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					memcpy(mutateString(chunk.key) + chunk.key.size() - sizeof(int32_t), &pos, sizeof(int32_t));
															#line 780 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					if (i == num_chunks - 1)
															#line 2957 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
					{
															#line 781 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
						chunk.value = ValueRef(static_cast<uint8_t*>(bw.getData()) + (i * value_size_limit), bw.getLength() - (i * value_size_limit));
															#line 2961 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
					}
					else
					{
															#line 784 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
						chunk.value = ValueRef(static_cast<uint8_t*>(bw.getData()) + (i * value_size_limit), value_size_limit);
															#line 2967 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
					}
															#line 787 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
					trChunksQ.push_back(std::move(chunk));
															#line 2971 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
				}
			}
															#line 792 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			dataSizeLimit = BUGGIFY ? deterministicRandom()->randomInt(200e3, 1.5 * CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT) : 0.8 * CLIENT_KNOBS->TRANSACTION_SIZE_LIMIT;
															#line 795 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tracking_iter = trChunksQ.begin();
															#line 796 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ASSERT(commitQ.empty() && (txBytes == 0));
															#line 797 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 2982 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 846 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent(SevWarnAlways, "UnableToWriteClientStatus").error(e);
															#line 847 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 3081 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 848 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				return a_body1Catch1(e, std::max(0, loopDepth - 1));
															#line 3085 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 850 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			cx->clientStatusUpdater.outStatusQ.clear();
															#line 853 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr = Transaction();
															#line 854 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_7 = delay(10.0);
															#line 854 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3095 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_7.isReady()) { if (__when_expr_7.isError()) return a_body1Catch1(__when_expr_7.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2Catch1when1(__when_expr_7.get(), loopDepth); };
			static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 8;
															#line 854 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_7.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 7, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3100 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4(int loopDepth) 
	{
															#line 831 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		cx->clientStatusUpdater.outStatusQ.clear();
															#line 832 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_4 = cx->globalConfig->onInitialized();
															#line 832 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 3119 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_4.isReady()) { if (__when_expr_4.isError()) return a_body1loopBody1cont2Catch1(__when_expr_4.getError(), loopDepth); else return a_body1loopBody1cont4when1(__when_expr_4.get(), loopDepth); };
		static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 5;
															#line 832 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_4.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 4, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3124 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
															#line 798 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		iter = tracking_iter;
															#line 799 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		txBytes = 0;
															#line 800 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		commitQ.clear();
															#line 3144 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		try {
															#line 802 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 3148 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1cont2loopBody1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont4(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 820 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (e.code() == error_code_transaction_too_large)
															#line 3183 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 821 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				dataSizeLimit /= 2;
															#line 822 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				ASSERT(dataSizeLimit >= CLIENT_KNOBS->VALUE_SIZE_LIMIT + CLIENT_KNOBS->KEY_SIZE_LIMIT);
															#line 3189 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
			else
			{
															#line 824 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				TraceEvent(SevWarnAlways, "ClientTrInfoErrorCommit").error(e).detail("TxBytes", txBytes);
															#line 825 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				commitQ.clear();
															#line 826 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				txBytes = 0;
															#line 827 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				return a_body1loopBody1cont2Catch1(e, std::max(0, loopDepth - 1));
															#line 3201 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
			loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1loopBody1cont2Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont2(int loopDepth) 
	{
															#line 813 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!commitQ.empty())
															#line 3217 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 814 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_3 = transactionInfoCommitActor(&tr, &commitQ);
															#line 814 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 3223 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1loopBody1cont2loopBody1Catch1(__when_expr_3.getError(), loopDepth); else return a_body1loopBody1cont2loopBody1cont2when1(__when_expr_3.get(), loopDepth); };
			static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 4;
															#line 814 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 3, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3228 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1(int loopDepth) 
	{
															#line 802 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!(iter != trChunksQ.end()))
															#line 3249 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
			return a_body1loopBody1cont2loopBody1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 803 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (iter->value.size() + iter->key.size() + txBytes > dataSizeLimit)
															#line 3255 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 804 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_2 = transactionInfoCommitActor(&tr, &commitQ);
															#line 804 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2loopBody1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 3261 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont2loopBody1Catch1(__when_expr_2.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont2loopBody1loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 3;
															#line 804 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 2, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3266 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont2loopBody1cont2(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont1(int loopDepth) 
	{
															#line 809 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		commitQ.push_back(*iter);
															#line 810 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		txBytes += iter->value.size() + iter->key.size();
															#line 811 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		++iter;
															#line 3297 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (loopDepth == 0) return a_body1loopBody1cont2loopBody1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont3(Void const& _,int loopDepth) 
	{
															#line 805 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		tracking_iter = iter;
															#line 806 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		commitQ.clear();
															#line 807 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		txBytes = 0;
															#line 3310 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1cont3(Void && _,int loopDepth) 
	{
															#line 805 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		tracking_iter = iter;
															#line 806 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		commitQ.clear();
															#line 807 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		txBytes = 0;
															#line 3323 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont3(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1loopBody1cont3(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont2loopBody1cont3(int loopDepth) 
	{
		return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont4(Void const& _,int loopDepth) 
	{
															#line 815 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		commitQ.clear();
															#line 816 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		txBytes = 0;
															#line 3415 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont4(Void && _,int loopDepth) 
	{
															#line 815 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		commitQ.clear();
															#line 816 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		txBytes = 0;
															#line 3426 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont2when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont4(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont2when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont4(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2loopBody1cont2when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2loopBody1cont2when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 3);

	}
	int a_body1loopBody1cont9(Void const& _,int loopDepth) 
	{
															#line 833 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		double sampleRate = cx->globalConfig->get<double>(fdbClientInfoTxnSampleRate, std::numeric_limits<double>::infinity());
															#line 835 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		double clientSamplingProbability = std::isinf(sampleRate) ? CLIENT_KNOBS->CSI_SAMPLING_PROBABILITY : sampleRate;
															#line 837 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		int64_t sizeLimit = cx->globalConfig->get<int64_t>(fdbClientInfoTxnSizeLimit, -1);
															#line 838 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		int64_t clientTxnInfoSizeLimit = sizeLimit == -1 ? CLIENT_KNOBS->CSI_SIZE_LIMIT : sizeLimit;
															#line 839 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!trChunksQ.empty() && deterministicRandom()->random01() < clientSamplingProbability)
															#line 3518 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 840 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_5 = delExcessClntTxnEntriesActor(&tr, clientTxnInfoSizeLimit);
															#line 840 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 3524 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1cont2Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_5.get(), loopDepth); };
			static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 6;
															#line 840 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 5, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3529 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont9(Void && _,int loopDepth) 
	{
															#line 833 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		double sampleRate = cx->globalConfig->get<double>(fdbClientInfoTxnSampleRate, std::numeric_limits<double>::infinity());
															#line 835 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		double clientSamplingProbability = std::isinf(sampleRate) ? CLIENT_KNOBS->CSI_SAMPLING_PROBABILITY : sampleRate;
															#line 837 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		int64_t sizeLimit = cx->globalConfig->get<int64_t>(fdbClientInfoTxnSizeLimit, -1);
															#line 838 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		int64_t clientTxnInfoSizeLimit = sizeLimit == -1 ? CLIENT_KNOBS->CSI_SIZE_LIMIT : sizeLimit;
															#line 839 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!trChunksQ.empty() && deterministicRandom()->random01() < clientSamplingProbability)
															#line 3551 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 840 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_5 = delExcessClntTxnEntriesActor(&tr, clientTxnInfoSizeLimit);
															#line 840 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 3557 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_5.isReady()) { if (__when_expr_5.isError()) return a_body1loopBody1cont2Catch1(__when_expr_5.getError(), loopDepth); else return a_body1loopBody1cont9when1(__when_expr_5.get(), loopDepth); };
			static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 6;
															#line 840 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_5.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 5, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3562 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		else
		{
			loopDepth = a_body1loopBody1cont10(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont4when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont9(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose5() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 4, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 4, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 4, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont4when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 4, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose5();
		try {
			a_body1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 4);

	}
	int a_body1loopBody1cont10(int loopDepth) 
	{
															#line 843 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		tr = Transaction();
															#line 844 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_6 = delay(CLIENT_KNOBS->CSI_STATUS_DELAY);
															#line 844 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2Catch1(actor_cancelled(), loopDepth);
															#line 3655 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_6.isReady()) { if (__when_expr_6.isError()) return a_body1loopBody1cont2Catch1(__when_expr_6.getError(), loopDepth); else return a_body1loopBody1cont10when1(__when_expr_6.get(), loopDepth); };
		static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 7;
															#line 844 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_6.addCallbackAndClear(static_cast<ActorCallback< ClientStatusUpdateActorActor, 6, Void >*>(static_cast<ClientStatusUpdateActorActor*>(this)));
															#line 3660 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont11(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont9when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont11(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose6() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 5, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 5, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont9when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 5, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont9when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 5, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose6();
		try {
			a_body1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 5);

	}
	int a_body1loopBody1cont10cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont10when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont10cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose7() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 6, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 6, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont10when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 6, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont10when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 6, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose7();
		try {
			a_body1loopBody1cont2Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 6);

	}
	int a_body1loopBody1cont10cont2(int loopDepth) 
	{
		try {
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose8() 
	{
		if (static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state > 0) static_cast<ClientStatusUpdateActorActor*>(this)->actor_wait_state = 0;
		static_cast<ClientStatusUpdateActorActor*>(this)->ActorCallback< ClientStatusUpdateActorActor, 7, Void >::remove();

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 7, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont2Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_fire(ActorCallback< ClientStatusUpdateActorActor, 7, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1loopBody1cont2Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 7);

	}
	void a_callback_error(ActorCallback< ClientStatusUpdateActorActor, 7, Void >*,Error err) 
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 7);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose8();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), 7);

	}
															#line 744 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	DatabaseContext* cx;
															#line 745 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	const std::string clientLatencyName;
															#line 747 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	Transaction tr;
															#line 748 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	std::vector<TrInfoChunk> commitQ;
															#line 749 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	int txBytes;
															#line 763 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	std::vector<TrInfoChunk> trChunksQ;
															#line 792 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	int64_t dataSizeLimit;
															#line 795 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	std::vector<TrInfoChunk>::iterator tracking_iter;
															#line 798 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	std::vector<TrInfoChunk>::iterator iter;
															#line 3957 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
};
// This generated class is to be used only via clientStatusUpdateActor()
															#line 744 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class ClientStatusUpdateActorActor final : public Actor<Void>, public ActorCallback< ClientStatusUpdateActorActor, 0, Void >, public ActorCallback< ClientStatusUpdateActorActor, 1, Void >, public ActorCallback< ClientStatusUpdateActorActor, 2, Void >, public ActorCallback< ClientStatusUpdateActorActor, 3, Void >, public ActorCallback< ClientStatusUpdateActorActor, 4, Void >, public ActorCallback< ClientStatusUpdateActorActor, 5, Void >, public ActorCallback< ClientStatusUpdateActorActor, 6, Void >, public ActorCallback< ClientStatusUpdateActorActor, 7, Void >, public FastAllocated<ClientStatusUpdateActorActor>, public ClientStatusUpdateActorActorState<ClientStatusUpdateActorActor> {
															#line 3962 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
	using FastAllocated<ClientStatusUpdateActorActor>::operator new;
	using FastAllocated<ClientStatusUpdateActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(7370654805766354944UL, 2499834732457588224UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< ClientStatusUpdateActorActor, 0, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 1, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 2, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 3, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 4, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 5, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 6, Void >;
friend struct ActorCallback< ClientStatusUpdateActorActor, 7, Void >;
															#line 744 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	ClientStatusUpdateActorActor(DatabaseContext* const& cx) 
															#line 3986 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   ClientStatusUpdateActorActorState<ClientStatusUpdateActorActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5908733088513914368UL, 12349308750647825408UL);
		ActorExecutionContextHelper __helper(static_cast<ClientStatusUpdateActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("clientStatusUpdateActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("clientStatusUpdateActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 1, Void >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 3, Void >*)0, actor_cancelled()); break;
		case 5: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 4, Void >*)0, actor_cancelled()); break;
		case 6: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 5, Void >*)0, actor_cancelled()); break;
		case 7: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 6, Void >*)0, actor_cancelled()); break;
		case 8: this->a_callback_error((ActorCallback< ClientStatusUpdateActorActor, 7, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 744 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
[[nodiscard]] static Future<Void> clientStatusUpdateActor( DatabaseContext* const& cx ) {
															#line 744 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	return Future<Void>(new ClientStatusUpdateActorActor(cx));
															#line 4026 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
}

#line 858 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

															#line 4031 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via assertFailure()
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
template <class AssertFailureActor>
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class AssertFailureActorState {
															#line 4038 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	AssertFailureActorState(GrvProxyInterface const& remote,Future<ErrorOr<GetReadVersionReply>> const& reply) 
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		 : remote(remote),
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   reply(reply)
															#line 4047 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("assertFailure", reinterpret_cast<unsigned long>(this));

	}
	~AssertFailureActorState() 
	{
		fdb_probe_actor_destroy("assertFailure", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
			try {
															#line 861 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				StrictFuture<ErrorOr<GetReadVersionReply>> __when_expr_0 = reply;
															#line 861 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				if (static_cast<AssertFailureActor*>(this)->actor_wait_state < 0) return a_body1Catch2(actor_cancelled(), loopDepth);
															#line 4065 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
				if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch2(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
				static_cast<AssertFailureActor*>(this)->actor_wait_state = 1;
															#line 861 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >*>(static_cast<AssertFailureActor*>(this)));
															#line 4070 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
				loopDepth = 0;
			}
			catch (Error& error) {
				loopDepth = a_body1Catch2(error, loopDepth);
			} catch (...) {
				loopDepth = a_body1Catch2(unknown_error(), loopDepth);
			}
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~AssertFailureActorState();
		static_cast<AssertFailureActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 875 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!static_cast<AssertFailureActor*>(this)->SAV<Void>::futures) { (void)(Void()); this->~AssertFailureActorState(); static_cast<AssertFailureActor*>(this)->destroy(); return 0; }
															#line 4099 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		new (&static_cast<AssertFailureActor*>(this)->SAV< Void >::value()) Void(Void());
		this->~AssertFailureActorState();
		static_cast<AssertFailureActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1Catch2(const Error& e,int loopDepth=0) 
	{
		try {
															#line 870 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (e.code() == error_code_actor_cancelled)
															#line 4112 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 871 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				return a_body1Catch1(e, loopDepth);
															#line 4116 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1cont2(ErrorOr<GetReadVersionReply> const& res,int loopDepth) 
	{
															#line 862 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!res.isError())
															#line 4132 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 863 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent(SevError, "GotStaleReadVersion") .detail("Remote", remote.getConsistentReadVersion.getEndpoint().addresses.address.toString()) .detail("Provisional", remote.provisional) .detail("ReadVersion", res.get().version);
															#line 867 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ASSERT_WE_THINK(false);
															#line 4138 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1cont2(ErrorOr<GetReadVersionReply> && res,int loopDepth) 
	{
															#line 862 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!res.isError())
															#line 4148 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 863 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent(SevError, "GotStaleReadVersion") .detail("Remote", remote.getConsistentReadVersion.getEndpoint().addresses.address.toString()) .detail("Provisional", remote.provisional) .detail("ReadVersion", res.get().version);
															#line 867 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			ASSERT_WE_THINK(false);
															#line 4154 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1cont5(loopDepth);

		return loopDepth;
	}
	int a_body1when1(ErrorOr<GetReadVersionReply> const& res,int loopDepth) 
	{
		loopDepth = a_body1cont2(res, loopDepth);

		return loopDepth;
	}
	int a_body1when1(ErrorOr<GetReadVersionReply> && res,int loopDepth) 
	{
		loopDepth = a_body1cont2(std::move(res), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<AssertFailureActor*>(this)->actor_wait_state > 0) static_cast<AssertFailureActor*>(this)->actor_wait_state = 0;
		static_cast<AssertFailureActor*>(this)->ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >::remove();

	}
	void a_callback_fire(ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >*,ErrorOr<GetReadVersionReply> const& value) 
	{
		fdb_probe_actor_enter("assertFailure", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AssertFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("assertFailure", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >*,ErrorOr<GetReadVersionReply> && value) 
	{
		fdb_probe_actor_enter("assertFailure", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<AssertFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("assertFailure", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >*,Error err) 
	{
		fdb_probe_actor_enter("assertFailure", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<AssertFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch2(err, 0);
		}
		catch (Error& error) {
			a_body1Catch2(error, 0);
		} catch (...) {
			a_body1Catch2(unknown_error(), 0);
		}
		fdb_probe_actor_exit("assertFailure", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1cont5(int loopDepth) 
	{
		try {
			loopDepth = a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	GrvProxyInterface remote;
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	Future<ErrorOr<GetReadVersionReply>> reply;
															#line 4252 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
};
// This generated class is to be used only via assertFailure()
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class AssertFailureActor final : public Actor<Void>, public ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >, public FastAllocated<AssertFailureActor>, public AssertFailureActorState<AssertFailureActor> {
															#line 4257 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
	using FastAllocated<AssertFailureActor>::operator new;
	using FastAllocated<AssertFailureActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(2708038429462386944UL, 11866021329080923904UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >;
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	AssertFailureActor(GrvProxyInterface const& remote,Future<ErrorOr<GetReadVersionReply>> const& reply) 
															#line 4274 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   AssertFailureActorState<AssertFailureActor>(remote, reply),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("assertFailure", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9499482857341526272UL, 8100088389272420864UL);
		ActorExecutionContextHelper __helper(static_cast<AssertFailureActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("assertFailure");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("assertFailure", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< AssertFailureActor, 0, ErrorOr<GetReadVersionReply> >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
[[nodiscard]] Future<Void> assertFailure( GrvProxyInterface const& remote, Future<ErrorOr<GetReadVersionReply>> const& reply ) {
															#line 859 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	return Future<Void>(new AssertFailureActor(remote, reply));
															#line 4307 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
}

#line 877 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

Future<Void> attemptGRVFromOldProxies(std::vector<GrvProxyInterface> oldProxies,
                                      std::vector<GrvProxyInterface> newProxies) {
	auto debugID = nondeterministicRandom()->randomUniqueID();
	g_traceBatch.addEvent("AttemptGRVFromOldProxyDebug", debugID.first(), "NativeAPI.attemptGRVFromOldProxies.Start");
	Span span("NAPI:VerifyCausalReadRisky"_loc);
	std::vector<Future<Void>> replies;
	replies.reserve(oldProxies.size());
	GetReadVersionRequest req(
	    span.context, 1, TransactionPriority::IMMEDIATE, GetReadVersionRequest::FLAG_CAUSAL_READ_RISKY);
	TraceEvent evt("AttemptGRVFromOldProxies");
	evt.detail("NumOldProxies", oldProxies.size()).detail("NumNewProxies", newProxies.size());
	auto traceProxies = [&](std::vector<GrvProxyInterface>& proxies, std::string const& key) {
		for (int i = 0; i < proxies.size(); ++i) {
			auto k = key + std::to_string(i);
			evt.detail(k.c_str(), proxies[i].id());
		}
	};
	traceProxies(oldProxies, "OldProxy"s);
	traceProxies(newProxies, "NewProxy"s);
	evt.log();
	for (auto& i : oldProxies) {
		req.reply = ReplyPromise<GetReadVersionReply>();
		replies.push_back(assertFailure(i, i.getConsistentReadVersion.tryGetReply(req)));
	}
	return waitForAll(replies);
}

															#line 4339 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via monitorClientDBInfoChange()
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
template <class MonitorClientDBInfoChangeActor>
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class MonitorClientDBInfoChangeActorState {
															#line 4346 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	MonitorClientDBInfoChangeActorState(DatabaseContext* const& cx,Reference<AsyncVar<ClientDBInfo> const> const& clientDBInfo,AsyncTrigger* const& proxiesChangeTrigger) 
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		 : cx(cx),
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   clientDBInfo(clientDBInfo),
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   proxiesChangeTrigger(proxiesChangeTrigger),
															#line 908 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   curCommitProxies(),
															#line 909 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   curGrvProxies(),
															#line 910 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   actors(false),
															#line 911 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   clientDBInfoOnChange(clientDBInfo->onChange())
															#line 4365 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this));

	}
	~MonitorClientDBInfoChangeActorState() 
	{
		fdb_probe_actor_destroy("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 912 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			curCommitProxies = clientDBInfo->get().commitProxies;
															#line 913 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			curGrvProxies = clientDBInfo->get().grvProxies;
															#line 915 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 4384 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~MonitorClientDBInfoChangeActorState();
		static_cast<MonitorClientDBInfoChangeActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 917 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_0 = clientDBInfoOnChange;
															#line 916 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<MonitorClientDBInfoChangeActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4416 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
															#line 941 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_1 = actors.getResult();
															#line 4420 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when2(__when_expr_1.get(), loopDepth); };
		static_cast<MonitorClientDBInfoChangeActor*>(this)->actor_wait_state = 1;
															#line 917 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >*>(static_cast<MonitorClientDBInfoChangeActor*>(this)));
															#line 941 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >*>(static_cast<MonitorClientDBInfoChangeActor*>(this)));
															#line 4427 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
															#line 918 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		clientDBInfoOnChange = clientDBInfo->onChange();
															#line 919 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (clientDBInfo->get().commitProxies != curCommitProxies || clientDBInfo->get().grvProxies != curGrvProxies)
															#line 4444 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 929 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (deterministicRandom()->random01() < cx->verifyCausalReadsProp && !curGrvProxies.empty() && !clientDBInfo->get().grvProxies.empty() && !clientDBInfo->get().grvProxies[0].provisional)
															#line 4448 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 931 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				actors.add(attemptGRVFromOldProxies(curGrvProxies, clientDBInfo->get().grvProxies));
															#line 4452 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 933 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			curCommitProxies = clientDBInfo->get().commitProxies;
															#line 934 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			curGrvProxies = clientDBInfo->get().grvProxies;
															#line 937 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			cx->ssVersionVectorCache.clear();
															#line 938 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			proxiesChangeTrigger->trigger();
															#line 4462 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
															#line 918 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		clientDBInfoOnChange = clientDBInfo->onChange();
															#line 919 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (clientDBInfo->get().commitProxies != curCommitProxies || clientDBInfo->get().grvProxies != curGrvProxies)
															#line 4474 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 929 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (deterministicRandom()->random01() < cx->verifyCausalReadsProp && !curGrvProxies.empty() && !clientDBInfo->get().grvProxies.empty() && !clientDBInfo->get().grvProxies[0].provisional)
															#line 4478 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 931 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				actors.add(attemptGRVFromOldProxies(curGrvProxies, clientDBInfo->get().grvProxies));
															#line 4482 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 933 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			curCommitProxies = clientDBInfo->get().commitProxies;
															#line 934 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			curGrvProxies = clientDBInfo->get().grvProxies;
															#line 937 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			cx->ssVersionVectorCache.clear();
															#line 938 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			proxiesChangeTrigger->trigger();
															#line 4492 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		}
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void const& _,int loopDepth) 
	{
															#line 942 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		UNSTOPPABLE_ASSERT(false);
															#line 4502 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when2(Void && _,int loopDepth) 
	{
															#line 942 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		UNSTOPPABLE_ASSERT(false);
															#line 4511 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont1(loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<MonitorClientDBInfoChangeActor*>(this)->actor_wait_state > 0) static_cast<MonitorClientDBInfoChangeActor*>(this)->actor_wait_state = 0;
		static_cast<MonitorClientDBInfoChangeActor*>(this)->ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >::remove();
		static_cast<MonitorClientDBInfoChangeActor*>(this)->ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >::remove();

	}
	void a_callback_fire(ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when2(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >*,Error err) 
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), 1);

	}
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	DatabaseContext* cx;
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	Reference<AsyncVar<ClientDBInfo> const> clientDBInfo;
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	AsyncTrigger* proxiesChangeTrigger;
															#line 908 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	std::vector<CommitProxyInterface> curCommitProxies;
															#line 909 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	std::vector<GrvProxyInterface> curGrvProxies;
															#line 910 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	ActorCollection actors;
															#line 911 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	Future<Void> clientDBInfoOnChange;
															#line 4651 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
};
// This generated class is to be used only via monitorClientDBInfoChange()
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class MonitorClientDBInfoChangeActor final : public Actor<Void>, public ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >, public ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >, public FastAllocated<MonitorClientDBInfoChangeActor>, public MonitorClientDBInfoChangeActorState<MonitorClientDBInfoChangeActor> {
															#line 4656 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
	using FastAllocated<MonitorClientDBInfoChangeActor>::operator new;
	using FastAllocated<MonitorClientDBInfoChangeActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(492622806113986560UL, 14342123549407049984UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >;
friend struct ActorCallback< MonitorClientDBInfoChangeActor, 1, Void >;
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	MonitorClientDBInfoChangeActor(DatabaseContext* const& cx,Reference<AsyncVar<ClientDBInfo> const> const& clientDBInfo,AsyncTrigger* const& proxiesChangeTrigger) 
															#line 4674 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   MonitorClientDBInfoChangeActorState<MonitorClientDBInfoChangeActor>(cx, clientDBInfo, proxiesChangeTrigger),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(8226933834932854784UL, 6929653946600028160UL);
		ActorExecutionContextHelper __helper(static_cast<MonitorClientDBInfoChangeActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("monitorClientDBInfoChange");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("monitorClientDBInfoChange", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< MonitorClientDBInfoChangeActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
[[nodiscard]] static Future<Void> monitorClientDBInfoChange( DatabaseContext* const& cx, Reference<AsyncVar<ClientDBInfo> const> const& clientDBInfo, AsyncTrigger* const& proxiesChangeTrigger ) {
															#line 905 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	return Future<Void>(new MonitorClientDBInfoChangeActor(cx, clientDBInfo, proxiesChangeTrigger));
															#line 4707 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
}

#line 947 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

void updateLocationCacheWithCaches(DatabaseContext* self,
                                   const std::map<UID, StorageServerInterface>& removed,
                                   const std::map<UID, StorageServerInterface>& added) {
	// TODO: this needs to be more clever in the future
	auto ranges = self->locationCache.ranges();
	for (auto iter = ranges.begin(); iter != ranges.end(); ++iter) {
		if (iter->value() && iter->value()->hasCaches) {
			auto& val = iter->value();
			std::vector<Reference<ReferencedInterface<StorageServerInterface>>> interfaces;
			interfaces.reserve(val->size() - removed.size() + added.size());
			for (int i = 0; i < val->size(); ++i) {
				const auto& interf = (*val)[i];
				if (removed.count(interf->interf.id()) == 0) {
					interfaces.emplace_back(interf);
				}
			}
			for (const auto& p : added) {
				interfaces.push_back(makeReference<ReferencedInterface<StorageServerInterface>>(p.second));
			}
			iter->value() = makeReference<LocationInfo>(interfaces, true);
		}
	}
}

Reference<LocationInfo> addCaches(const Reference<LocationInfo>& loc,
                                  const std::vector<Reference<ReferencedInterface<StorageServerInterface>>>& other) {
	std::vector<Reference<ReferencedInterface<StorageServerInterface>>> interfaces;
	interfaces.reserve(loc->size() + other.size());
	for (int i = 0; i < loc->size(); ++i) {
		interfaces.emplace_back((*loc)[i]);
	}
	interfaces.insert(interfaces.end(), other.begin(), other.end());
	return makeReference<LocationInfo>(interfaces, true);
}

															#line 4747 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via handleTssMismatches()
															#line 983 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
template <class HandleTssMismatchesActor>
															#line 983 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class HandleTssMismatchesActorState {
															#line 4754 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
															#line 983 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	HandleTssMismatchesActorState(DatabaseContext* const& cx) 
															#line 983 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
															#line 983 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		 : cx(cx),
															#line 984 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   tr(),
															#line 985 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   tssMapDB(KeyBackedMap<UID, UID>(tssMappingKeys.begin)),
															#line 986 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   tssMismatchDB(KeyBackedMap<Tuple, std::string>(tssMismatchKeys.begin))
															#line 4767 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("handleTssMismatches", reinterpret_cast<unsigned long>(this));

	}
	~HandleTssMismatchesActorState() 
	{
		fdb_probe_actor_destroy("handleTssMismatches", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 987 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 4782 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HandleTssMismatchesActorState();
		static_cast<HandleTssMismatchesActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 990 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_0 = delay(0);
															#line 990 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4814 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 1;
															#line 990 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HandleTssMismatchesActor, 0, Void >*>(static_cast<HandleTssMismatchesActor*>(this)));
															#line 4819 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
															#line 993 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		auto __when_expr_1 = cx->tssMismatchStream.getFuture();
															#line 4828 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<std::pair<UID, std::vector<DetailedTSSMismatch>>>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<std::pair<UID, std::vector<DetailedTSSMismatch>>>>::value, "invalid type");
															#line 993 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4832 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.pop(), loopDepth); };
		static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 2;
															#line 993 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< HandleTssMismatchesActor, 1, std::pair<UID, std::vector<DetailedTSSMismatch>> >*>(static_cast<HandleTssMismatchesActor*>(this)));
															#line 4837 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
															#line 993 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		auto __when_expr_1 = cx->tssMismatchStream.getFuture();
															#line 4846 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		static_assert(std::is_same<decltype(__when_expr_1), FutureStream<std::pair<UID, std::vector<DetailedTSSMismatch>>>>::value || std::is_same<decltype(__when_expr_1), ThreadFutureStream<std::pair<UID, std::vector<DetailedTSSMismatch>>>>::value, "invalid type");
															#line 993 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 4850 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_1.isReady()) { if (__when_expr_1.isError()) return a_body1Catch1(__when_expr_1.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1cont1when1(__when_expr_1.pop(), loopDepth); };
		static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 2;
															#line 993 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_1.addCallbackAndClear(static_cast<ActorSingleCallback< HandleTssMismatchesActor, 1, std::pair<UID, std::vector<DetailedTSSMismatch>> >*>(static_cast<HandleTssMismatchesActor*>(this)));
															#line 4855 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state > 0) static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 0;
		static_cast<HandleTssMismatchesActor*>(this)->ActorCallback< HandleTssMismatchesActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HandleTssMismatchesActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 0);

	}
	int a_body1loopBody1cont2(int loopDepth) 
	{
															#line 995 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		tssPairID = UID();
															#line 996 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		bool found = false;
															#line 997 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		for( const auto& it : cx->tssMapping ) {
															#line 998 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (it.second.id() == data.first)
															#line 4945 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 999 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				tssPairID = it.first;
															#line 1000 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				found = true;
															#line 4951 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
				break;
			}
		}
															#line 1004 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (found)
															#line 4957 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 1005 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			quarantine = CLIENT_KNOBS->QUARANTINE_TSS_ON_MISMATCH;
															#line 1006 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent(SevWarnAlways, quarantine ? "TSS_QuarantineMismatch" : "TSS_KillMismatch") .detail("TSSID", data.first.toString());
															#line 1008 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			CODE_PROBE(quarantine, "Quarantining TSS because it got mismatch");
															#line 1009 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			CODE_PROBE(!quarantine, "Killing TSS because it got mismatch");
															#line 1011 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr = makeReference<ReadYourWritesTransaction>(Database(Reference<DatabaseContext>::addRef(cx)));
															#line 1012 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tries = 0;
															#line 1013 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 4973 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1cont2loopHead1(loopDepth);
		}
		else
		{
															#line 1047 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			CODE_PROBE(true, "Not handling TSS with mismatch because it's already gone");
															#line 4980 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopBody1cont3(loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(std::pair<UID, std::vector<DetailedTSSMismatch>> const& __data,int loopDepth) 
	{
															#line 993 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		data = __data;
															#line 4990 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont1when1(std::pair<UID, std::vector<DetailedTSSMismatch>> && __data,int loopDepth) 
	{
		data = std::move(__data);
		loopDepth = a_body1loopBody1cont2(loopDepth);

		return loopDepth;
	}
	void a_exitChoose2() 
	{
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state > 0) static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 0;
		static_cast<HandleTssMismatchesActor*>(this)->ActorSingleCallback< HandleTssMismatchesActor, 1, std::pair<UID, std::vector<DetailedTSSMismatch>> >::remove();

	}
	void a_callback_fire(ActorSingleCallback< HandleTssMismatchesActor, 1, std::pair<UID, std::vector<DetailedTSSMismatch>> >*,std::pair<UID, std::vector<DetailedTSSMismatch>> const& value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_fire(ActorSingleCallback< HandleTssMismatchesActor, 1, std::pair<UID, std::vector<DetailedTSSMismatch>> >*,std::pair<UID, std::vector<DetailedTSSMismatch>> && value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1loopBody1cont1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);

	}
	void a_callback_error(ActorSingleCallback< HandleTssMismatchesActor, 1, std::pair<UID, std::vector<DetailedTSSMismatch>> >*,Error err) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose2();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 1);

	}
	int a_body1loopBody1cont3(int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont6(int loopDepth) 
	{
															#line 1045 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		tr = makeReference<ReadYourWritesTransaction>();
															#line 5075 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = a_body1loopBody1cont3(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1cont2loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1(int loopDepth) 
	{
		try {
															#line 1015 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->setOption(FDBTransactionOptions::PRIORITY_SYSTEM_IMMEDIATE);
															#line 1016 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tr->setOption(FDBTransactionOptions::ACCESS_SYSTEM_KEYS);
															#line 1017 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (quarantine)
															#line 5096 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			{
															#line 1018 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				tr->set(tssQuarantineKeyFor(data.first), ""_sr);
															#line 5100 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
			else
			{
															#line 1020 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				tr->clear(serverTagKeyFor(data.first));
															#line 5106 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 1022 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			tssMapDB.erase(tr, tssPairID);
															#line 1024 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			for( const DetailedTSSMismatch& d : data.second ) {
															#line 1026 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
				tssMismatchDB.set(tr, Tuple::makeTuple(data.first.toString(), d.timestamp, d.mismatchId.toString()), d.traceString);
															#line 5114 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			}
															#line 1031 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_2 = tr->commit();
															#line 1031 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state < 0) return a_body1loopBody1cont2loopBody1Catch1(actor_cancelled(), loopDepth);
															#line 5120 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_2.isReady()) { if (__when_expr_2.isError()) return a_body1loopBody1cont2loopBody1Catch1(__when_expr_2.getError(), loopDepth); else return a_body1loopBody1cont2loopBody1when1(__when_expr_2.get(), loopDepth); };
			static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 3;
															#line 1031 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_2.addCallbackAndClear(static_cast<ActorCallback< HandleTssMismatchesActor, 2, Void >*>(static_cast<HandleTssMismatchesActor*>(this)));
															#line 5125 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1loopBody1cont2loopBody1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1loopBody1cont2loopBody1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2break1(int loopDepth) 
	{
		try {
			return a_body1loopBody1cont6(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 1));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 1));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont1(int loopDepth) 
	{
															#line 1037 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		tries++;
															#line 1038 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (tries > 10)
															#line 5155 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
															#line 1040 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			TraceEvent("TSS_MismatchGaveUp").detail("TSSID", data.first.toString());
															#line 5159 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break
		}
		if (loopDepth == 0) return a_body1loopBody1cont2loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1Catch1(const Error& e,int loopDepth=0) 
	{
		try {
															#line 1035 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<Void> __when_expr_3 = tr->onError(e);
															#line 1035 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 2));
															#line 5173 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_3.isReady()) { if (__when_expr_3.isError()) return a_body1Catch1(__when_expr_3.getError(), std::max(0, loopDepth - 2)); else return a_body1loopBody1cont2loopBody1Catch1when1(__when_expr_3.get(), loopDepth); };
			static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 4;
															#line 1035 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_3.addCallbackAndClear(static_cast<ActorCallback< HandleTssMismatchesActor, 3, Void >*>(static_cast<HandleTssMismatchesActor*>(this)));
															#line 5178 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, std::max(0, loopDepth - 2));
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), std::max(0, loopDepth - 2));
		}

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont2(Void const& _,int loopDepth) 
	{
		return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1cont2(Void && _,int loopDepth) 
	{
		return a_body1loopBody1cont2break1(loopDepth==0?0:loopDepth-1); // break

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont2(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont2(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose3() 
	{
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state > 0) static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 0;
		static_cast<HandleTssMismatchesActor*>(this)->ActorCallback< HandleTssMismatchesActor, 2, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 2, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 2, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);

	}
	void a_callback_error(ActorCallback< HandleTssMismatchesActor, 2, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose3();
		try {
			a_body1loopBody1cont2loopBody1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1loopBody1cont2loopBody1Catch1(error, 0);
		} catch (...) {
			a_body1loopBody1cont2loopBody1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 2);

	}
	int a_body1loopBody1cont2loopBody1Catch1cont1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1Catch1cont1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1cont1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1Catch1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1Catch1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1cont2loopBody1Catch1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont2loopBody1Catch1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose4() 
	{
		if (static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state > 0) static_cast<HandleTssMismatchesActor*>(this)->actor_wait_state = 0;
		static_cast<HandleTssMismatchesActor*>(this)->ActorCallback< HandleTssMismatchesActor, 3, Void >::remove();

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 3, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2loopBody1Catch1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_fire(ActorCallback< HandleTssMismatchesActor, 3, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1loopBody1cont2loopBody1Catch1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);

	}
	void a_callback_error(ActorCallback< HandleTssMismatchesActor, 3, Void >*,Error err) 
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose4();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), 3);

	}
															#line 983 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	DatabaseContext* cx;
															#line 984 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	Reference<ReadYourWritesTransaction> tr;
															#line 985 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	KeyBackedMap<UID, UID> tssMapDB;
															#line 986 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	KeyBackedMap<Tuple, std::string> tssMismatchDB;
															#line 993 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	std::pair<UID, std::vector<DetailedTSSMismatch>> data;
															#line 995 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	UID tssPairID;
															#line 1005 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	bool quarantine;
															#line 1012 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	int tries;
															#line 5379 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
};
// This generated class is to be used only via handleTssMismatches()
															#line 983 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class HandleTssMismatchesActor final : public Actor<Void>, public ActorCallback< HandleTssMismatchesActor, 0, Void >, public ActorSingleCallback< HandleTssMismatchesActor, 1, std::pair<UID, std::vector<DetailedTSSMismatch>> >, public ActorCallback< HandleTssMismatchesActor, 2, Void >, public ActorCallback< HandleTssMismatchesActor, 3, Void >, public FastAllocated<HandleTssMismatchesActor>, public HandleTssMismatchesActorState<HandleTssMismatchesActor> {
															#line 5384 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
	using FastAllocated<HandleTssMismatchesActor>::operator new;
	using FastAllocated<HandleTssMismatchesActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(10722857316578785792UL, 10649030141504284160UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<Void>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HandleTssMismatchesActor, 0, Void >;
friend struct ActorSingleCallback< HandleTssMismatchesActor, 1, std::pair<UID, std::vector<DetailedTSSMismatch>> >;
friend struct ActorCallback< HandleTssMismatchesActor, 2, Void >;
friend struct ActorCallback< HandleTssMismatchesActor, 3, Void >;
															#line 983 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	HandleTssMismatchesActor(DatabaseContext* const& cx) 
															#line 5404 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		 : Actor<Void>(),
		   HandleTssMismatchesActorState<HandleTssMismatchesActor>(cx),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("handleTssMismatches", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6231205013535006464UL, 15374262790278653696UL);
		ActorExecutionContextHelper __helper(static_cast<HandleTssMismatchesActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("handleTssMismatches");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("handleTssMismatches", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HandleTssMismatchesActor, 0, Void >*)0, actor_cancelled()); break;
		case 2: this->a_callback_error((ActorSingleCallback< HandleTssMismatchesActor, 1, std::pair<UID, std::vector<DetailedTSSMismatch>> >*)0, actor_cancelled()); break;
		case 3: this->a_callback_error((ActorCallback< HandleTssMismatchesActor, 2, Void >*)0, actor_cancelled()); break;
		case 4: this->a_callback_error((ActorCallback< HandleTssMismatchesActor, 3, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 983 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
[[nodiscard]] static Future<Void> handleTssMismatches( DatabaseContext* const& cx ) {
															#line 983 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	return Future<Void>(new HandleTssMismatchesActor(cx));
															#line 5440 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
}

#line 1051 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

															#line 5445 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
[[nodiscard]] Future<Optional<Value>> getJSON( Database const& db, std::string const& jsonField = "" );

#line 1053 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

struct SingleSpecialKeyImpl : SpecialKeyRangeReadImpl {
	Future<RangeResult> getRange(ReadYourWritesTransaction* ryw,
	                             KeyRangeRef kr,
	                             GetRangeLimits limitsHint) const override {
		ASSERT(kr.contains(k));
		return map(f(ryw), [k = k](Optional<Value> v) {
			RangeResult result;
			if (v.present()) {
				result.push_back_deep(result.arena(), KeyValueRef(k, v.get()));
			}
			return result;
		});
	}

	SingleSpecialKeyImpl(KeyRef k, const std::function<Future<Optional<Value>>(ReadYourWritesTransaction*)>& f)
	  : SpecialKeyRangeReadImpl(singleKeyRange(k)), k(k), f(f) {}

private:
	Key k;
	std::function<Future<Optional<Value>>(ReadYourWritesTransaction*)> f;
};

class HealthMetricsRangeImpl : public SpecialKeyRangeAsyncImpl {
public:
	explicit HealthMetricsRangeImpl(KeyRangeRef kr);
	Future<RangeResult> getRange(ReadYourWritesTransaction* ryw,
	                             KeyRangeRef kr,
	                             GetRangeLimits limitsHint) const override;
};

static RangeResult healthMetricsToKVPairs(const HealthMetrics& metrics, KeyRangeRef kr) {
	RangeResult result;
	if (CLIENT_BUGGIFY)
		return result;
	if (kr.contains("\xff\xff/metrics/health/aggregate"_sr) && metrics.worstStorageDurabilityLag != 0) {
		json_spirit::mObject statsObj;
		statsObj["batch_limited"] = metrics.batchLimited;
		statsObj["tps_limit"] = metrics.tpsLimit;
		statsObj["worst_storage_durability_lag"] = metrics.worstStorageDurabilityLag;
		statsObj["limiting_storage_durability_lag"] = metrics.limitingStorageDurabilityLag;
		statsObj["worst_storage_queue"] = metrics.worstStorageQueue;
		statsObj["limiting_storage_queue"] = metrics.limitingStorageQueue;
		statsObj["worst_log_queue"] = metrics.worstTLogQueue;
		std::string statsString =
		    json_spirit::write_string(json_spirit::mValue(statsObj), json_spirit::Output_options::raw_utf8);
		ValueRef bytes(result.arena(), statsString);
		result.push_back(result.arena(), KeyValueRef("\xff\xff/metrics/health/aggregate"_sr, bytes));
	}
	// tlog stats
	{
		int phase = 0; // Avoid comparing twice per loop iteration
		for (const auto& [uid, logStats] : metrics.tLogQueue) {
			StringRef k{ StringRef(uid.toString()).withPrefix("\xff\xff/metrics/health/log/"_sr, result.arena()) };
			if (phase == 0 && k >= kr.begin) {
				phase = 1;
			}
			if (phase == 1) {
				if (k < kr.end) {
					json_spirit::mObject statsObj;
					statsObj["log_queue"] = logStats;
					std::string statsString =
					    json_spirit::write_string(json_spirit::mValue(statsObj), json_spirit::Output_options::raw_utf8);
					ValueRef bytes(result.arena(), statsString);
					result.push_back(result.arena(), KeyValueRef(k, bytes));
				} else {
					break;
				}
			}
		}
	}
	// Storage stats
	{
		int phase = 0; // Avoid comparing twice per loop iteration
		for (const auto& [uid, storageStats] : metrics.storageStats) {
			StringRef k{ StringRef(uid.toString()).withPrefix("\xff\xff/metrics/health/storage/"_sr, result.arena()) };
			if (phase == 0 && k >= kr.begin) {
				phase = 1;
			}
			if (phase == 1) {
				if (k < kr.end) {
					json_spirit::mObject statsObj;
					statsObj["storage_durability_lag"] = storageStats.storageDurabilityLag;
					statsObj["storage_queue"] = storageStats.storageQueue;
					statsObj["cpu_usage"] = storageStats.cpuUsage;
					statsObj["disk_usage"] = storageStats.diskUsage;
					std::string statsString =
					    json_spirit::write_string(json_spirit::mValue(statsObj), json_spirit::Output_options::raw_utf8);
					ValueRef bytes(result.arena(), statsString);
					result.push_back(result.arena(), KeyValueRef(k, bytes));
				} else {
					break;
				}
			}
		}
	}
	return result;
}

															#line 5548 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via healthMetricsGetRangeActor()
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
template <class HealthMetricsGetRangeActorActor>
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class HealthMetricsGetRangeActorActorState {
															#line 5555 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	HealthMetricsGetRangeActorActorState(ReadYourWritesTransaction* const& ryw,KeyRangeRef const& kr) 
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		 : ryw(ryw),
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		   kr(kr)
															#line 5564 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this));

	}
	~HealthMetricsGetRangeActorActorState() 
	{
		fdb_probe_actor_destroy("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1153 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			StrictFuture<HealthMetrics> __when_expr_0 = ryw->getDatabase()->getHealthMetrics( kr.intersects( KeyRangeRef("\xff\xff/metrics/health/storage/"_sr, "\xff\xff/metrics/health/storage0"_sr)) || kr.intersects(KeyRangeRef("\xff\xff/metrics/health/log/"_sr, "\xff\xff/metrics/health/log0"_sr)));
															#line 1153 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			if (static_cast<HealthMetricsGetRangeActorActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), loopDepth);
															#line 5581 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), loopDepth); else return a_body1when1(__when_expr_0.get(), loopDepth); };
			static_cast<HealthMetricsGetRangeActorActor*>(this)->actor_wait_state = 1;
															#line 1153 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >*>(static_cast<HealthMetricsGetRangeActorActor*>(this)));
															#line 5586 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = 0;
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~HealthMetricsGetRangeActorActorState();
		static_cast<HealthMetricsGetRangeActorActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(HealthMetrics const& metrics,int loopDepth) 
	{
															#line 1157 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!static_cast<HealthMetricsGetRangeActorActor*>(this)->SAV<RangeResult>::futures) { (void)(healthMetricsToKVPairs(metrics, kr)); this->~HealthMetricsGetRangeActorActorState(); static_cast<HealthMetricsGetRangeActorActor*>(this)->destroy(); return 0; }
															#line 5609 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		new (&static_cast<HealthMetricsGetRangeActorActor*>(this)->SAV< RangeResult >::value()) RangeResult(healthMetricsToKVPairs(metrics, kr));
		this->~HealthMetricsGetRangeActorActorState();
		static_cast<HealthMetricsGetRangeActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1cont1(HealthMetrics && metrics,int loopDepth) 
	{
															#line 1157 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!static_cast<HealthMetricsGetRangeActorActor*>(this)->SAV<RangeResult>::futures) { (void)(healthMetricsToKVPairs(metrics, kr)); this->~HealthMetricsGetRangeActorActorState(); static_cast<HealthMetricsGetRangeActorActor*>(this)->destroy(); return 0; }
															#line 5621 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		new (&static_cast<HealthMetricsGetRangeActorActor*>(this)->SAV< RangeResult >::value()) RangeResult(healthMetricsToKVPairs(metrics, kr));
		this->~HealthMetricsGetRangeActorActorState();
		static_cast<HealthMetricsGetRangeActorActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1when1(HealthMetrics const& metrics,int loopDepth) 
	{
		loopDepth = a_body1cont1(metrics, loopDepth);

		return loopDepth;
	}
	int a_body1when1(HealthMetrics && metrics,int loopDepth) 
	{
		loopDepth = a_body1cont1(std::move(metrics), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<HealthMetricsGetRangeActorActor*>(this)->actor_wait_state > 0) static_cast<HealthMetricsGetRangeActorActor*>(this)->actor_wait_state = 0;
		static_cast<HealthMetricsGetRangeActorActor*>(this)->ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >::remove();

	}
	void a_callback_fire(ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >*,HealthMetrics const& value) 
	{
		fdb_probe_actor_enter("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsGetRangeActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >*,HealthMetrics && value) 
	{
		fdb_probe_actor_enter("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsGetRangeActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >*,Error err) 
	{
		fdb_probe_actor_enter("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsGetRangeActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	ReadYourWritesTransaction* ryw;
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	KeyRangeRef kr;
															#line 5708 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
};
// This generated class is to be used only via healthMetricsGetRangeActor()
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class HealthMetricsGetRangeActorActor final : public Actor<RangeResult>, public ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >, public FastAllocated<HealthMetricsGetRangeActorActor>, public HealthMetricsGetRangeActorActorState<HealthMetricsGetRangeActorActor> {
															#line 5713 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
	using FastAllocated<HealthMetricsGetRangeActorActor>::operator new;
	using FastAllocated<HealthMetricsGetRangeActorActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(5581061531928654080UL, 16020947379679506944UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<RangeResult>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >;
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	HealthMetricsGetRangeActorActor(ReadYourWritesTransaction* const& ryw,KeyRangeRef const& kr) 
															#line 5730 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		 : Actor<RangeResult>(),
		   HealthMetricsGetRangeActorActorState<HealthMetricsGetRangeActorActor>(ryw, kr),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(6540468110473879552UL, 10808560399808058880UL);
		ActorExecutionContextHelper __helper(static_cast<HealthMetricsGetRangeActorActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("healthMetricsGetRangeActor");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("healthMetricsGetRangeActor", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< HealthMetricsGetRangeActorActor, 0, HealthMetrics >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
[[nodiscard]] static Future<RangeResult> healthMetricsGetRangeActor( ReadYourWritesTransaction* const& ryw, KeyRangeRef const& kr ) {
															#line 1152 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	return Future<RangeResult>(new HealthMetricsGetRangeActorActor(ryw, kr));
															#line 5763 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
}

#line 1159 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

HealthMetricsRangeImpl::HealthMetricsRangeImpl(KeyRangeRef kr) : SpecialKeyRangeAsyncImpl(kr) {}

Future<RangeResult> HealthMetricsRangeImpl::getRange(ReadYourWritesTransaction* ryw,
                                                     KeyRangeRef kr,
                                                     GetRangeLimits limitsHint) const {
	return healthMetricsGetRangeActor(ryw, kr);
}

															#line 5776 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
namespace {
// This generated class is to be used only via getClusterId()
															#line 1168 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
template <class GetClusterIdActor>
															#line 1168 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class GetClusterIdActorState {
															#line 5783 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
															#line 1168 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	GetClusterIdActorState(Database const& db) 
															#line 1168 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
															#line 1168 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		 : db(db)
															#line 5790 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
	{
		fdb_probe_actor_create("getClusterId", reinterpret_cast<unsigned long>(this));

	}
	~GetClusterIdActorState() 
	{
		fdb_probe_actor_destroy("getClusterId", reinterpret_cast<unsigned long>(this));

	}
	int a_body1(int loopDepth=0) 
	{
		try {
															#line 1169 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
			;
															#line 5805 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
			loopDepth = a_body1loopHead1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1Catch1(Error error,int loopDepth=0) 
	{
		this->~GetClusterIdActorState();
		static_cast<GetClusterIdActor*>(this)->sendErrorAndDelPromiseRef(error);
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1cont1(int loopDepth) 
	{
															#line 1172 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!static_cast<GetClusterIdActor*>(this)->SAV<UID>::futures) { (void)(db->clientInfo->get().clusterId); this->~GetClusterIdActorState(); static_cast<GetClusterIdActor*>(this)->destroy(); return 0; }
															#line 5828 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		new (&static_cast<GetClusterIdActor*>(this)->SAV< UID >::value()) UID(db->clientInfo->get().clusterId);
		this->~GetClusterIdActorState();
		static_cast<GetClusterIdActor*>(this)->finishSendAndDelPromiseRef();
		return 0;

		return loopDepth;
	}
	int a_body1loopHead1(int loopDepth) 
	{
		int oldLoopDepth = ++loopDepth;
		while (loopDepth == oldLoopDepth) loopDepth = a_body1loopBody1(loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1(int loopDepth) 
	{
															#line 1169 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (!(!db->clientInfo->get().clusterId.isValid()))
															#line 5847 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		{
			return a_body1break1(loopDepth==0?0:loopDepth-1); // break
		}
															#line 1170 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		StrictFuture<Void> __when_expr_0 = db->clientInfo->onChange();
															#line 1170 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		if (static_cast<GetClusterIdActor*>(this)->actor_wait_state < 0) return a_body1Catch1(actor_cancelled(), std::max(0, loopDepth - 1));
															#line 5855 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		if (__when_expr_0.isReady()) { if (__when_expr_0.isError()) return a_body1Catch1(__when_expr_0.getError(), std::max(0, loopDepth - 1)); else return a_body1loopBody1when1(__when_expr_0.get(), loopDepth); };
		static_cast<GetClusterIdActor*>(this)->actor_wait_state = 1;
															#line 1170 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
		__when_expr_0.addCallbackAndClear(static_cast<ActorCallback< GetClusterIdActor, 0, Void >*>(static_cast<GetClusterIdActor*>(this)));
															#line 5860 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		loopDepth = 0;

		return loopDepth;
	}
	int a_body1break1(int loopDepth) 
	{
		try {
			return a_body1cont1(loopDepth);
		}
		catch (Error& error) {
			loopDepth = a_body1Catch1(error, loopDepth);
		} catch (...) {
			loopDepth = a_body1Catch1(unknown_error(), loopDepth);
		}

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void const& _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1cont1(Void && _,int loopDepth) 
	{
		if (loopDepth == 0) return a_body1loopHead1(0);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void const& _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(_, loopDepth);

		return loopDepth;
	}
	int a_body1loopBody1when1(Void && _,int loopDepth) 
	{
		loopDepth = a_body1loopBody1cont1(std::move(_), loopDepth);

		return loopDepth;
	}
	void a_exitChoose1() 
	{
		if (static_cast<GetClusterIdActor*>(this)->actor_wait_state > 0) static_cast<GetClusterIdActor*>(this)->actor_wait_state = 0;
		static_cast<GetClusterIdActor*>(this)->ActorCallback< GetClusterIdActor, 0, Void >::remove();

	}
	void a_callback_fire(ActorCallback< GetClusterIdActor, 0, Void >*,Void const& value) 
	{
		fdb_probe_actor_enter("getClusterId", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterIdActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(value, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterId", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_fire(ActorCallback< GetClusterIdActor, 0, Void >*,Void && value) 
	{
		fdb_probe_actor_enter("getClusterId", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(9578512156108914944UL, 107643746863906816UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterIdActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1loopBody1when1(std::move(value), 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterId", reinterpret_cast<unsigned long>(this), 0);

	}
	void a_callback_error(ActorCallback< GetClusterIdActor, 0, Void >*,Error err) 
	{
		fdb_probe_actor_enter("getClusterId", reinterpret_cast<unsigned long>(this), 0);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(5187328219730237440UL, 3113259079076601088UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterIdActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		a_exitChoose1();
		try {
			a_body1Catch1(err, 0);
		}
		catch (Error& error) {
			a_body1Catch1(error, 0);
		} catch (...) {
			a_body1Catch1(unknown_error(), 0);
		}
		fdb_probe_actor_exit("getClusterId", reinterpret_cast<unsigned long>(this), 0);

	}
															#line 1168 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	Database db;
															#line 5967 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
};
// This generated class is to be used only via getClusterId()
															#line 1168 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
class GetClusterIdActor final : public Actor<UID>, public ActorCallback< GetClusterIdActor, 0, Void >, public FastAllocated<GetClusterIdActor>, public GetClusterIdActorState<GetClusterIdActor> {
															#line 5972 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
public:
	using FastAllocated<GetClusterIdActor>::operator new;
	using FastAllocated<GetClusterIdActor>::operator delete;
	static constexpr ActorIdentifier __actorIdentifier = UID(6361677750806306304UL, 10340075671443366656UL);
	ActiveActorHelper activeActorHelper;
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdelete-non-virtual-dtor"
    void destroy() override {
        activeActorHelper.~ActiveActorHelper();
        static_cast<Actor<UID>*>(this)->~Actor();
        operator delete(this);
    }
#pragma clang diagnostic pop
friend struct ActorCallback< GetClusterIdActor, 0, Void >;
															#line 1168 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	GetClusterIdActor(Database const& db) 
															#line 5989 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
		 : Actor<UID>(),
		   GetClusterIdActorState<GetClusterIdActor>(db),
		   activeActorHelper(__actorIdentifier)
	{
		fdb_probe_actor_enter("getClusterId", reinterpret_cast<unsigned long>(this), -1);
		#ifdef WITH_ACAC
		static constexpr ActorBlockIdentifier __identifier = UID(2565818813149026048UL, 5883691562570913024UL);
		ActorExecutionContextHelper __helper(static_cast<GetClusterIdActor*>(this)->activeActorHelper.actorID, __identifier);
		#endif // WITH_ACAC
		#ifdef ENABLE_SAMPLING
		this->lineage.setActorName("getClusterId");
		LineageScope _(&this->lineage);
		#endif
		this->a_body1();
		fdb_probe_actor_exit("getClusterId", reinterpret_cast<unsigned long>(this), -1);

	}
	void cancel() override
	{
		auto wait_state = this->actor_wait_state;
		this->actor_wait_state = -1;
		switch (wait_state) {
		case 1: this->a_callback_error((ActorCallback< GetClusterIdActor, 0, Void >*)0, actor_cancelled()); break;
		}

	}
};
} // namespace
															#line 1168 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
[[nodiscard]] Future<UID> getClusterId( Database const& db ) {
															#line 1168 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"
	return Future<UID>(new GetClusterIdActor(db));
															#line 6022 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/build_output/fdbclient/DatabaseContext.actor.g.cpp.py_gen"
}

#line 1174 "/codebuild/output/src948217385/src/github.com/apple/foundationdb/fdbclient/DatabaseContext.actor.cpp"

void DatabaseContext::initializeSpecialCounters() {
	specialCounter(cc, "OutstandingWatches", [this] { return outstandingWatches; });
	specialCounter(cc, "WatchMapSize", [this] { return watchMap.size(); });
}

DatabaseContext::DatabaseContext(Reference<AsyncVar<Reference<IClusterConnectionRecord>>> connectionRecord,
                                 Reference<AsyncVar<ClientDBInfo>> clientInfo,
                                 Reference<AsyncVar<Optional<ClientLeaderRegInterface>> const> coordinator,
                                 Future<Void> clientInfoMonitor,
                                 TaskPriority taskID,
                                 LocalityData const& clientLocality,
                                 EnableLocalityLoadBalance enableLocalityLoadBalance,
                                 LockAware lockAware,
                                 IsInternal internal,
                                 int _apiVersion,
                                 IsSwitchable switchable)
  : dbId(deterministicRandom()->randomUniqueID()), lockAware(lockAware), switchable(switchable),
    connectionRecord(connectionRecord), proxyProvisional(false), clientLocality(clientLocality),
    enableLocalityLoadBalance(enableLocalityLoadBalance), internal(internal), cc("TransactionMetrics", dbId.toString()),
    transactionReadVersions("ReadVersions", cc), transactionReadVersionsThrottled("ReadVersionsThrottled", cc),
    transactionReadVersionsCompleted("ReadVersionsCompleted", cc),
    transactionReadVersionBatches("ReadVersionBatches", cc),
    transactionBatchReadVersions("BatchPriorityReadVersions", cc),
    transactionDefaultReadVersions("DefaultPriorityReadVersions", cc),
    transactionImmediateReadVersions("ImmediatePriorityReadVersions", cc),
    transactionBatchReadVersionsCompleted("BatchPriorityReadVersionsCompleted", cc),
    transactionDefaultReadVersionsCompleted("DefaultPriorityReadVersionsCompleted", cc),
    transactionImmediateReadVersionsCompleted("ImmediatePriorityReadVersionsCompleted", cc),
    transactionLogicalReads("LogicalUncachedReads", cc), transactionPhysicalReads("PhysicalReadRequests", cc),
    transactionPhysicalReadsCompleted("PhysicalReadRequestsCompleted", cc),
    transactionGetKeyRequests("GetKeyRequests", cc), transactionGetValueRequests("GetValueRequests", cc),
    transactionGetRangeRequests("GetRangeRequests", cc),
    transactionGetMappedRangeRequests("GetMappedRangeRequests", cc),
    transactionGetRangeStreamRequests("GetRangeStreamRequests", cc), transactionWatchRequests("WatchRequests", cc),
    transactionGetAddressesForKeyRequests("GetAddressesForKeyRequests", cc), transactionBytesRead("BytesRead", cc),
    transactionKeysRead("KeysRead", cc), transactionMetadataVersionReads("MetadataVersionReads", cc),
    transactionCommittedMutations("CommittedMutations", cc),
    transactionCommittedMutationBytes("CommittedMutationBytes", cc), transactionSetMutations("SetMutations", cc),
    transactionClearMutations("ClearMutations", cc), transactionAtomicMutations("AtomicMutations", cc),
    transactionsCommitStarted("CommitStarted", cc), transactionsCommitCompleted("CommitCompleted", cc),
    transactionKeyServerLocationRequests("KeyServerLocationRequests", cc),
    transactionKeyServerLocationRequestsCompleted("KeyServerLocationRequestsCompleted", cc),
    transactionStatusRequests("StatusRequests", cc), transactionsTooOld("TooOld", cc),
    transactionsFutureVersions("FutureVersions", cc), transactionsNotCommitted("NotCommitted", cc),
    transactionsMaybeCommitted("MaybeCommitted", cc), transactionsResourceConstrained("ResourceConstrained", cc),
    transactionsProcessBehind("ProcessBehind", cc), transactionsThrottled("Throttled", cc),
    transactionsLockRejected("LockRejected", cc),
    transactionsExpensiveClearCostEstCount("ExpensiveClearCostEstCount", cc),
    transactionGrvFullBatches("NumGrvFullBatches", cc), transactionGrvTimedOutBatches("NumGrvTimedOutBatches", cc),
    transactionCommitVersionNotFoundForSS("CommitVersionNotFoundForSS", cc),

    latencies(), readLatencies(), commitLatencies(), GRVLatencies(), mutationsPerCommit(), bytesPerCommit(),
    outstandingWatches(0), sharedStatePtr(nullptr), lastGrvTime(0.0), cachedReadVersion(0),
    lastRkBatchThrottleTime(0.0), lastRkDefaultThrottleTime(0.0), lastProxyRequestTime(0.0),
    transactionTracingSample(false), taskID(taskID), clientInfo(clientInfo), clientInfoMonitor(clientInfoMonitor),
    coordinator(coordinator), apiVersion(_apiVersion), mvCacheInsertLocation(0), healthMetricsLastUpdated(0),
    detailedHealthMetricsLastUpdated(0), smoothMidShardSize(CLIENT_KNOBS->SHARD_STAT_SMOOTH_AMOUNT),
    specialKeySpace(std::make_unique<SpecialKeySpace>(specialKeys.begin, specialKeys.end, /* test */ false)),
    connectToDatabaseEventCacheHolder(format("ConnectToDatabase/%s", dbId.toString().c_str())) {

	DisabledTraceEvent("DatabaseContextCreated", dbId).backtrace();

	connected = (clientInfo->get().commitProxies.size() && clientInfo->get().grvProxies.size())
	                ? Void()
	                : clientInfo->onChange();

	metadataVersionCache.resize(CLIENT_KNOBS->METADATA_VERSION_CACHE_SIZE);
	maxOutstandingWatches = CLIENT_KNOBS->DEFAULT_MAX_OUTSTANDING_WATCHES;

	snapshotRywEnabled = apiVersion.hasSnapshotRYW() ? 1 : 0;

	logger = databaseLogger(this) && tssLogger(this);
	locationCacheSize = g_network->isSimulated() ? CLIENT_KNOBS->LOCATION_CACHE_EVICTION_SIZE_SIM
	                                             : CLIENT_KNOBS->LOCATION_CACHE_EVICTION_SIZE;

	getValueSubmitted.init("NativeAPI.GetValueSubmitted"_sr);
	getValueCompleted.init("NativeAPI.GetValueCompleted"_sr);

	clientDBInfoMonitor = monitorClientDBInfoChange(this, clientInfo, &proxiesChangeTrigger);
	tssMismatchHandler = handleTssMismatches(this);
	clientStatusUpdater.actor = clientStatusUpdateActor(this);

	smoothMidShardSize.reset(CLIENT_KNOBS->INIT_MID_SHARD_BYTES);
	globalConfig = std::make_unique<GlobalConfig>(this);

	if (apiVersion.version() >= 740) {
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::METRICS,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<FaultToleranceMetricsImpl>(
		        singleKeyRange("fault_tolerance_metrics_json"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::METRICS).begin)));
	}

	if (apiVersion.version() >= 700) {
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::ERRORMSG,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<SingleSpecialKeyImpl>(
		                            SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::ERRORMSG).begin,
		                            [](ReadYourWritesTransaction* ryw) -> Future<Optional<Value>> {
			                            if (ryw->getSpecialKeySpaceErrorMsg().present())
				                            return Optional<Value>(ryw->getSpecialKeySpaceErrorMsg().get());
			                            else
				                            return Optional<Value>();
		                            }));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<ManagementCommandsOptionsImpl>(
		        KeyRangeRef("options/"_sr, "options0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<ExcludeServersRangeImpl>(SpecialKeySpace::getManagementApiCommandRange("exclude")));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<FailedServersRangeImpl>(SpecialKeySpace::getManagementApiCommandRange("failed")));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::MANAGEMENT,
		                        SpecialKeySpace::IMPLTYPE::READWRITE,
		                        std::make_unique<ExcludedLocalitiesRangeImpl>(
		                            SpecialKeySpace::getManagementApiCommandRange("excludedlocality")));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::MANAGEMENT,
		                        SpecialKeySpace::IMPLTYPE::READWRITE,
		                        std::make_unique<FailedLocalitiesRangeImpl>(
		                            SpecialKeySpace::getManagementApiCommandRange("failedlocality")));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<ExclusionInProgressRangeImpl>(
		        KeyRangeRef("in_progress_exclusion/"_sr, "in_progress_exclusion0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::CONFIGURATION,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<ProcessClassRangeImpl>(
		        KeyRangeRef("process/class_type/"_sr, "process/class_type0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::CONFIGURATION).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::CONFIGURATION,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<ProcessClassSourceRangeImpl>(
		        KeyRangeRef("process/class_source/"_sr, "process/class_source0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::CONFIGURATION).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<LockDatabaseImpl>(
		        singleKeyRange("db_locked"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<ConsistencyCheckImpl>(
		        singleKeyRange("consistency_check_suspended"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::GLOBALCONFIG,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<GlobalConfigImpl>(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::GLOBALCONFIG)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::TRACING,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<TracingOptionsImpl>(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::TRACING)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::CONFIGURATION,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<CoordinatorsImpl>(
		        KeyRangeRef("coordinators/"_sr, "coordinators0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::CONFIGURATION).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<CoordinatorsAutoImpl>(
		        singleKeyRange("auto_coordinators"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<AdvanceVersionImpl>(
		        singleKeyRange("min_required_commit_version"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<VersionEpochImpl>(
		        singleKeyRange("version_epoch"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<ClientProfilingImpl>(
		        KeyRangeRef("profiling/"_sr, "profiling0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)),
		    /* deprecated */ ApiVersion::withClientProfilingDeprecated().version());
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<MaintenanceImpl>(
		        KeyRangeRef("maintenance/"_sr, "maintenance0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::MANAGEMENT,
		    SpecialKeySpace::IMPLTYPE::READWRITE,
		    std::make_unique<DataDistributionImpl>(
		        KeyRangeRef("data_distribution/"_sr, "data_distribution0"_sr)
		            .withPrefix(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::MANAGEMENT).begin)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::ACTORLINEAGE,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<ActorLineageImpl>(SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::ACTORLINEAGE)));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::ACTOR_PROFILER_CONF,
		                        SpecialKeySpace::IMPLTYPE::READWRITE,
		                        std::make_unique<ActorProfilerConf>(
		                            SpecialKeySpace::getModuleRange(SpecialKeySpace::MODULE::ACTOR_PROFILER_CONF)));
	}
	if (apiVersion.version() >= 630) {
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::TRANSACTION,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<ConflictingKeysImpl>(conflictingKeysRange));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::TRANSACTION,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<ReadConflictRangeImpl>(readConflictRangeKeysRange));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::TRANSACTION,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<WriteConflictRangeImpl>(writeConflictRangeKeysRange));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::METRICS,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<DDStatsRangeImpl>(ddStatsRange));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::METRICS,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<HealthMetricsRangeImpl>(
		                            KeyRangeRef("\xff\xff/metrics/health/"_sr, "\xff\xff/metrics/health0"_sr)));
		registerSpecialKeysImpl(SpecialKeySpace::MODULE::WORKERINTERFACE,
		                        SpecialKeySpace::IMPLTYPE::READONLY,
		                        std::make_unique<WorkerInterfacesSpecialKeyImpl>(
		                            KeyRangeRef("\xff\xff/worker_interfaces/"_sr, "\xff\xff/worker_interfaces0"_sr)));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::STATUSJSON,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<SingleSpecialKeyImpl>(
		        "\xff\xff/status/json"_sr, [](ReadYourWritesTransaction* ryw) -> Future<Optional<Value>> {
			        if (ryw->getDatabase().getPtr() && ryw->getDatabase()->getConnectionRecord()) {
				        ++ryw->getDatabase()->transactionStatusRequests;
				        return getJSON(ryw->getDatabase());
			        } else {
				        return Optional<Value>();
			        }
		        }));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::CLUSTERFILEPATH,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<SingleSpecialKeyImpl>(
		        "\xff\xff/cluster_file_path"_sr, [](ReadYourWritesTransaction* ryw) -> Future<Optional<Value>> {
			        try {
				        if (ryw->getDatabase().getPtr() && ryw->getDatabase()->getConnectionRecord()) {
					        Optional<Value> output =
					            StringRef(ryw->getDatabase()->getConnectionRecord()->getLocation());
					        return output;
				        }
			        } catch (Error& e) {
				        return e;
			        }
			        return Optional<Value>();
		        }));

		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::CONNECTIONSTRING,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<SingleSpecialKeyImpl>(
		        "\xff\xff/connection_string"_sr, [](ReadYourWritesTransaction* ryw) -> Future<Optional<Value>> {
			        try {
				        if (ryw->getDatabase().getPtr() && ryw->getDatabase()->getConnectionRecord()) {
					        Reference<IClusterConnectionRecord> f = ryw->getDatabase()->getConnectionRecord();
					        Optional<Value> output = StringRef(f->getConnectionString().toString());
					        return output;
				        }
			        } catch (Error& e) {
				        return e;
			        }
			        return Optional<Value>();
		        }));
		registerSpecialKeysImpl(
		    SpecialKeySpace::MODULE::CLUSTERID,
		    SpecialKeySpace::IMPLTYPE::READONLY,
		    std::make_unique<SingleSpecialKeyImpl>(
		        "\xff\xff/cluster_id"_sr, [](ReadYourWritesTransaction* ryw) -> Future<Optional<Value>> {
			        try {
				        if (ryw->getDatabase().getPtr()) {
					        return map(getClusterId(ryw->getDatabase()),
					                   [](UID id) { return Optional<Value>(StringRef(id.toString())); });
				        }
			        } catch (Error& e) {
				        return e;
			        }
			        return Optional<Value>();
		        }));
	}
	throttleExpirer = recurring([this]() { expireThrottles(); }, CLIENT_KNOBS->TAG_THROTTLE_EXPIRATION_INTERVAL);

	if (BUGGIFY) {
		DatabaseContext::debugUseTags = true;
	}

	initializeSpecialCounters();
}

DatabaseContext::DatabaseContext(const Error& err)
  : deferredError(err), internal(IsInternal::False), cc("TransactionMetrics"),
    transactionReadVersions("ReadVersions", cc), transactionReadVersionsThrottled("ReadVersionsThrottled", cc),
    transactionReadVersionsCompleted("ReadVersionsCompleted", cc),
    transactionReadVersionBatches("ReadVersionBatches", cc),
    transactionBatchReadVersions("BatchPriorityReadVersions", cc),
    transactionDefaultReadVersions("DefaultPriorityReadVersions", cc),
    transactionImmediateReadVersions("ImmediatePriorityReadVersions", cc),
    transactionBatchReadVersionsCompleted("BatchPriorityReadVersionsCompleted", cc),
    transactionDefaultReadVersionsCompleted("DefaultPriorityReadVersionsCompleted", cc),
    transactionImmediateReadVersionsCompleted("ImmediatePriorityReadVersionsCompleted", cc),
    transactionLogicalReads("LogicalUncachedReads", cc), transactionPhysicalReads("PhysicalReadRequests", cc),
    transactionPhysicalReadsCompleted("PhysicalReadRequestsCompleted", cc),
    transactionGetKeyRequests("GetKeyRequests", cc), transactionGetValueRequests("GetValueRequests", cc),
    transactionGetRangeRequests("GetRangeRequests", cc),
    transactionGetMappedRangeRequests("GetMappedRangeRequests", cc),
    transactionGetRangeStreamRequests("GetRangeStreamRequests", cc), transactionWatchRequests("WatchRequests", cc),
    transactionGetAddressesForKeyRequests("GetAddressesForKeyRequests", cc), transactionBytesRead("BytesRead", cc),
    transactionKeysRead("KeysRead", cc), transactionMetadataVersionReads("MetadataVersionReads", cc),
    transactionCommittedMutations("CommittedMutations", cc),
    transactionCommittedMutationBytes("CommittedMutationBytes", cc), transactionSetMutations("SetMutations", cc),
    transactionClearMutations("ClearMutations", cc), transactionAtomicMutations("AtomicMutations", cc),
    transactionsCommitStarted("CommitStarted", cc), transactionsCommitCompleted("CommitCompleted", cc),
    transactionKeyServerLocationRequests("KeyServerLocationRequests", cc),
    transactionKeyServerLocationRequestsCompleted("KeyServerLocationRequestsCompleted", cc),
    transactionStatusRequests("StatusRequests", cc), transactionsTooOld("TooOld", cc),
    transactionsFutureVersions("FutureVersions", cc), transactionsNotCommitted("NotCommitted", cc),
    transactionsMaybeCommitted("MaybeCommitted", cc), transactionsResourceConstrained("ResourceConstrained", cc),
    transactionsProcessBehind("ProcessBehind", cc), transactionsThrottled("Throttled", cc),
    transactionsLockRejected("LockRejected", cc),
    transactionsExpensiveClearCostEstCount("ExpensiveClearCostEstCount", cc),
    transactionGrvFullBatches("NumGrvFullBatches", cc), transactionGrvTimedOutBatches("NumGrvTimedOutBatches", cc),
    transactionCommitVersionNotFoundForSS("CommitVersionNotFoundForSS", cc), latencies(), readLatencies(),
    commitLatencies(), GRVLatencies(), mutationsPerCommit(), bytesPerCommit(), sharedStatePtr(nullptr),
    transactionTracingSample(false), smoothMidShardSize(CLIENT_KNOBS->SHARD_STAT_SMOOTH_AMOUNT),
    connectToDatabaseEventCacheHolder(format("ConnectToDatabase/%s", dbId.toString().c_str())), outstandingWatches(0) {
	initializeSpecialCounters();
}

// Static constructor used by server processes to create a DatabaseContext
// For internal (fdbserver) use only
Database DatabaseContext::create(Reference<AsyncVar<ClientDBInfo>> clientInfo,
                                 Future<Void> clientInfoMonitor,
                                 LocalityData clientLocality,
                                 EnableLocalityLoadBalance enableLocalityLoadBalance,
                                 TaskPriority taskID,
                                 LockAware lockAware,
                                 int apiVersion,
                                 IsSwitchable switchable) {
	return Database(new DatabaseContext(Reference<AsyncVar<Reference<IClusterConnectionRecord>>>(),
	                                    clientInfo,
	                                    makeReference<AsyncVar<Optional<ClientLeaderRegInterface>>>(),
	                                    clientInfoMonitor,
	                                    taskID,
	                                    clientLocality,
	                                    enableLocalityLoadBalance,
	                                    lockAware,
	                                    IsInternal::True,
	                                    apiVersion,
	                                    switchable));
}

DatabaseContext::~DatabaseContext() {
	clientDBInfoMonitor.cancel();
	monitorTssInfoChange.cancel();
	tssMismatchHandler.cancel();
	storage = nullptr;

	if (grvUpdateHandler.isValid()) {
		grvUpdateHandler.cancel();
	}
	if (sharedStatePtr) {
		sharedStatePtr->delRef(sharedStatePtr);
	}
	for (auto it = server_interf.begin(); it != server_interf.end(); it = server_interf.erase(it))
		it->second->notifyContextDestroyed();
	ASSERT_ABORT(server_interf.empty());
	locationCache.insert(allKeys, Reference<LocationInfo>());

	DisabledTraceEvent("DatabaseContextDestructed", dbId).backtrace();
}
